      MODULE ageicebc_mod
!***********************************************************************
!  Compute the lateral boundary conditions on the ice age.
!***********************************************************************
      implicit none
      PRIVATE
      PUBLIC ageicebc_tile
      CONTAINS
!
!***********************************************************************
      SUBROUTINE ageicebc (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_ice
      USE mod_stepping
      USE mod_scalars
!
      integer, intent(in) :: ng, tile
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private storage
!  arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J)- and MAX(I,J)-directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL ageicebc_tile (ng, tile,                                     &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    liold(ng), linew(ng), min_h(ng),              &
     &                    ICE(ng) % ui,                                 &
     &                    ICE(ng) % vi,                                 &
     &                    ICE(ng) % hi,                                 &
     &                    ICE(ng) % ageice,                             &
     &                    ICE(ng) % hage)
      RETURN
      END SUBROUTINE ageicebc
!
!***********************************************************************
      SUBROUTINE ageicebc_tile (ng, tile,                               &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           liold, linew, min_h,                   &
     &                           ui, vi, hi, ageice, hage)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_grid
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: liold, linew
      real(r8), intent(in) :: min_h
      real(r8), intent(in)    :: ui(LBi:,LBj:,:)
      real(r8), intent(in)    :: vi(LBi:,LBj:,:)
      real(r8), intent(in)    :: hi(LBi:,LBj:,:)
      real(r8), intent(inout) :: ageice(LBi:,LBj:,:)
      real(r8), intent(inout) :: hage(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j, know
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set time-indices
!-----------------------------------------------------------------------
!
        know=liold
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the western edge.
!-----------------------------------------------------------------------
!
      IF (Istr.eq.1) THEN
!
!  Western edge, closed boundary condition.
!
        DO j=Jstr,Jend
          hage(0,j,linew)=hi(1,j,linew)*ageice(1,j,linew)
          ageice(0,j,linew) = hage(0,j,linew)/                          &
     &         MAX(hi(0,j,linew),1.0E-6_r8)
          hage(0,j,linew)=hage(0,j,linew)*                              &
     &                   GRID(ng)%rmask(0,j)
             ageice(0,j,linew) = hage(0,j,linew)/                       &
     &                       MAX(hi(0,j,linew),1.0E-6_r8)
             IF(hi(0,j,linew).LE.min_h) THEN
                hage(0,j,linew) = 0.0_r8
                ageice(0,j,linew) = 0.0_r8
             END IF
             ageice(0,j,linew)=ageice(0,j,linew)*                       &
     &                   GRID(ng)%rmask(0,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the eastern edge.
!-----------------------------------------------------------------------
!
      IF (Iend.eq.Lm(ng)) THEN
!
!  Eastern edge, closed boundary condition.
!
        DO j=Jstr,Jend
          hage(Lm(ng)+1,j,linew)=hi(Lm(ng),j,linew)*                    &
     &                               ageice(Lm(ng),j,linew)
          hage(Lm(ng)+1,j,linew)=hage(Lm(ng)+1,j,linew)*                &
     &                          GRID(ng)%rmask(Lm(ng)+1,j)
             ageice(Lm(ng)+1,j,linew) = hage(Lm(ng)+1,j,linew)/         &
     &                       MAX(hi(Lm(ng)+1,j,linew),1.0E-6_r8)
             IF(hi(Lm(ng)+1,j,linew).LE.min_h) THEN
                hage(Lm(ng)+1,j,linew) = 0.0_r8
                ageice(Lm(ng)+1,j,linew) = 0.0_r8
             END IF
             ageice(Lm(ng)+1,j,linew)=ageice(Lm(ng)+1,j,linew)*         &
     &                   GRID(ng)%rmask(Lm(ng)+1,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the southern edge.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1) THEN
!
!  Southern edge, closed boundary condition.
!
        DO i=Istr,Iend
          hage(i,0,linew)=hi(i,1,linew)*ageice(i,1,linew)
          hage(i,0,linew)=hage(i,0,linew)*                              &
     &                   GRID(ng)%rmask(i,0)
             ageice(i,0,linew) = hage(i,0,linew)/                       &
     &                       MAX(hi(i,0,linew),1.0E-6_r8)
             IF(hi(i,0,linew).LE.min_h) THEN
                hage(i,0,linew) = 0.0_r8
                ageice(i,0,linew) = 0.0_r8
             END IF
             ageice(i,0,linew)=ageice(i,0,linew)*                       &
     &                   GRID(ng)%rmask(i,0)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the northern edge.
!-----------------------------------------------------------------------
!
      IF (Jend.eq.Mm(ng)) THEN
!
!  Northern edge, closed boundary condition.
!
        DO i=Istr,Iend
          hage(i,Mm(ng)+1,linew)=hi(i,Mm(ng),linew)*                    &
     &                               ageice(i,Mm(ng),linew)
          hage(i,Mm(ng)+1,linew)=hage(i,Mm(ng)+1,linew)*                &
     &                          GRID(ng)%rmask(i,Mm(ng)+1)
             ageice(i,Mm(ng)+1,linew) = hage(i,Mm(ng)+1,linew)/         &
     &                       MAX(hi(i,Mm(ng)+1,linew),1.0E-6_r8)
             IF(hi(i,Mm(ng)+1,linew).LE.min_h) THEN
                hage(i,Mm(ng)+1,linew) = 0.0_r8
                ageice(i,Mm(ng)+1,linew) = 0.0_r8
             END IF
             ageice(i,Mm(ng)+1,linew)=ageice(i,Mm(ng)+1,linew)*         &
     &                   GRID(ng)%rmask(i,Mm(ng)+1)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Boundary corners.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1.and.Istr.eq.1) THEN
        hage(0,0,linew)=0.5_r8*(hage(1,0,linew)+                        &
     &                         hage(0,1,linew))
             ageice(0,0,linew) = hage(0,0,linew)/                       &
     &                       MAX(hi(0,0,linew),1.0E-6_r8)
             IF(hi(0,0,linew).LE.min_h) THEN
                hage(0,0,linew) = 0.0_r8
                ageice(0,0,linew) = 0.0_r8
             END IF
             hage(0,0,linew)=hage(0,0,linew)*                           &
     &                   GRID(ng)%rmask(0,0)
             ageice(0,0,linew)=ageice(0,0,linew)*                       &
     &                   GRID(ng)%rmask(0,0)
      END IF
      IF (Jstr.eq.1.and.Iend.eq.Lm(ng)) THEN
        hage(Lm(ng)+1,0,linew)=0.5_r8*(hage(Lm(ng)+1,1,linew)+          &
     &                                hage(Lm(ng)  ,0,linew))
             ageice(Lm(ng)+1,0,linew) = hage(Lm(ng)+1,0,linew)/         &
     &                       MAX(hi(Lm(ng)+1,0,linew),1.0E-6_r8)
             IF(hi(Lm(ng)+1,0,linew).LE.min_h) THEN
                hage(Lm(ng)+1,0,linew) = 0.0_r8
                ageice(Lm(ng)+1,0,linew) = 0.0_r8
             END IF
             hage(Lm(ng)+1,0,linew)=hage(Lm(ng)+1,0,linew)*             &
     &                   GRID(ng)%rmask(Lm(ng)+1,0)
             ageice(Lm(ng)+1,0,linew)=ageice(Lm(ng)+1,0,linew)*         &
     &                   GRID(ng)%rmask(Lm(ng)+1,0)
      END IF
      IF (Jend.eq.Mm(ng).and.Istr.eq.1) THEN
        hage(0,Mm(ng)+1,linew)=0.5_r8*(hage(0,Mm(ng)  ,linew)+          &
     &                                hage(1,Mm(ng)+1,linew))
             ageice(0,Mm(ng)+1,linew) = hage(0,Mm(ng)+1,linew)/         &
     &                       MAX(hi(0,Mm(ng)+1,linew),1.0E-6_r8)
             IF(hi(0,Mm(ng)+1,linew).LE.min_h) THEN
                hage(0,Mm(ng)+1,linew) = 0.0_r8
                ageice(0,Mm(ng)+1,linew) = 0.0_r8
             END IF
             hage(0,Mm(ng)+1,linew)=hage(0,Mm(ng)+1,linew)*             &
     &                   GRID(ng)%rmask(0,Mm(ng)+1)
             ageice(0,Mm(ng)+1,linew)=ageice(0,Mm(ng)+1,linew)*         &
     &                   GRID(ng)%rmask(0,Mm(ng)+1)
      END IF
      IF (Jend.eq.Mm(ng).and.Iend.eq.Lm(ng)) THEN
        hage(Lm(ng)+1,Mm(ng)+1,linew)=0.5_r8*                           &
     &             (hage(Lm(ng)+1,Mm(ng)  ,linew)+                      &
     &              hage(Lm(ng)  ,Mm(ng)+1,linew))
             ageice(Lm(ng)+1,Mm(ng)+1,linew) =                          &
     &         hage(Lm(ng)+1,Mm(ng)+1,linew)/                           &
     &                       MAX(hi(Lm(ng)+1,Mm(ng)+1,linew),1.0E-6_r8)
             IF(hi(Lm(ng)+1,Mm(ng)+1,linew).LE.min_h) THEN
                hage(Lm(ng)+1,Mm(ng)+1,linew) = 0.0_r8
                ageice(Lm(ng)+1,Mm(ng)+1,linew) = 0.0_r8
             END IF
             hage(Lm(ng)+1,Mm(ng)+1,linew)=                             &
     &             hage(Lm(ng)+1,Mm(ng)+1,linew)*                       &
     &                   GRID(ng)%rmask(Lm(ng)+1,Mm(ng)+1)
      ageice(Lm(ng)+1,Mm(ng)+1,linew)=ageice(Lm(ng)+1,Mm(ng)+1,linew)*  &
     &                   GRID(ng)%rmask(Lm(ng)+1,Mm(ng)+1)
      END IF
      RETURN
      END SUBROUTINE ageicebc_tile
      END MODULE ageicebc_mod
