      MODULE aibc_mod
! 
!***********************************************************************
!  Compute lateral boundary conditions for the ice concentration.
!***********************************************************************
      implicit none
      PRIVATE
      PUBLIC aibc_tile
      CONTAINS
!
!***********************************************************************
      SUBROUTINE aibc (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_ice
      USE mod_stepping
!
      integer, intent(in) :: ng, tile
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private storage
!  arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J)- and MAX(I,J)-directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL aibc_tile (ng, tile,                                         &
     &                LBi, UBi, LBj, UBj,                               &
     &                liold(ng), linew(ng),                             &
     &                ICE(ng) % ui,                                     &
     &                ICE(ng) % vi,                                     &
     &                ICE(ng) % ai)
      RETURN
      END SUBROUTINE aibc
!
!***********************************************************************
      SUBROUTINE aibc_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      liold, linew,                               &
     &                      ui, vi, ai)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_grid
      USE mod_scalars
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: liold, linew
      real(r8), intent(in)    :: ui(LBi:,LBj:,:)
      real(r8), intent(in)    :: vi(LBi:,LBj:,:)
      real(r8), intent(inout) :: ai(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j, know
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set time-indices
!-----------------------------------------------------------------------
!
        know=liold
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the western edge.
!-----------------------------------------------------------------------
!
      IF (Istr.eq.1) THEN
!
!  Western edge, closed boundary condition.
!
        DO j=Jstr,Jend
          ai(0,j,linew)=ai(1,j,linew)
          ai(0,j,linew)=ai(0,j,linew)*                                  &
     &                   GRID(ng)%rmask(0,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the eastern edge.
!-----------------------------------------------------------------------
!
      IF (Iend.eq.Lm(ng)) THEN
!
!  Eastern edge, closed boundary condition.
!
        DO j=Jstr,Jend
          ai(Lm(ng)+1,j,linew)=ai(Lm(ng),j,linew)
          ai(Lm(ng)+1,j,linew)=ai(Lm(ng)+1,j,linew)*                    &
     &                          GRID(ng)%rmask(Lm(ng)+1,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the southern edge.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1) THEN
!
!  Southern edge, closed boundary condition.
!
        DO i=Istr,Iend
          ai(i,0,linew)=ai(i,1,linew)
          ai(i,0,linew)=ai(i,0,linew)*                                  &
     &                   GRID(ng)%rmask(i,0)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the northern edge.
!-----------------------------------------------------------------------
!
      IF (Jend.eq.Mm(ng)) THEN
!
!  Northern edge, closed boundary condition.
!
        DO i=Istr,Iend
          ai(i,Mm(ng)+1,linew)=ai(i,Mm(ng),linew)
          ai(i,Mm(ng)+1,linew)=ai(i,Mm(ng)+1,linew)*                    &
     &                          GRID(ng)%rmask(i,Mm(ng)+1)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Boundary corners.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1.and.Istr.eq.1) THEN
        ai(0,0,linew)=0.5_r8*(ai(1,0,linew)+                            &
     &                         ai(0,1,linew))
      END IF
      IF (Jstr.eq.1.and.Iend.eq.Lm(ng)) THEN
        ai(Lm(ng)+1,0,linew)=0.5_r8*(ai(Lm(ng)+1,1,linew)+              &
     &                                ai(Lm(ng)  ,0,linew))
      END IF
      IF (Jend.eq.Mm(ng).and.Istr.eq.1) THEN
        ai(0,Mm(ng)+1,linew)=0.5_r8*(ai(0,Mm(ng)  ,linew)+              &
     &                                ai(1,Mm(ng)+1,linew))
      END IF
      IF (Jend.eq.Mm(ng).and.Iend.eq.Lm(ng)) THEN
        ai(Lm(ng)+1,Mm(ng)+1,linew)=0.5_r8*                             &
     &             (ai(Lm(ng)+1,Mm(ng)  ,linew)+                        &
     &              ai(Lm(ng)  ,Mm(ng)+1,linew))
      END IF
      RETURN
      END SUBROUTINE aibc_tile
      END MODULE aibc_mod
