      SUBROUTINE inp_par (model)
!
!svn $Id: inp_par.F 966 2009-03-28 00:29:06Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2009 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine reads in input model parameters from standard input.   !
!  It also writes out these parameters to standard output.             !
!                                                                      !
!=======================================================================
!
      USE mod_filter, ONLY: nfil,fcount,nw,dtfil,FIL
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
      USE mod_strings
!
      USE distribute_mod, ONLY : mp_bcasti, mp_bcasts
      USE ran_state, ONLY: ran_seed
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: model
!
!  Local variable declarations.
!
      logical :: Lwrite
      integer :: Itile, Jtile, Nghost, Ntiles, tile
      integer :: Imin, Imax, Jmin, Jmax
      integer :: MaxHaloLenI, MaxHaloLenJ
      integer :: inp, out, itrc, ng, npts, sequence
      real(r8) :: cff
      real(r8), parameter :: spv = 0.0_r8
!
!-----------------------------------------------------------------------
!  Read in and report input model parameters.
!-----------------------------------------------------------------------
!
!  Set input units.
!
      Lwrite=Master
      inp=1
      out=stdout
!
!  Get current data.
!
      IF (Master) CALL get_date (date_str)
      DO ng=1,Ngrids
        CALL mp_bcasts (ng, model, date_str)
      END DO
!
!-----------------------------------------------------------------------
!  Read in physical model input parameters.
!-----------------------------------------------------------------------
!
      IF (Lwrite) WRITE (out,10) version, TRIM(date_str)
 10   FORMAT (/,' Model Input Parameters:  ROMS/TOMS version ',a,/,     &
     &       26x,a,/,1x,77('-'))
!
!  In distributed-memory configurations, the input physical parameters
!  script is opened as a regular file.  It is read and processed by all
!  parallel nodes.  This is to avoid a very complex broadcasting of the
!  input parameters to all nodes.
!
      IF (Master) CALL my_getarg (1,Iname)
!     IF (Master) Iname = 'ocean_nep1.in'
      DO ng=1,Ngrids
        CALL mp_bcasts (ng, model, Iname)
      END DO
      OPEN (inp, FILE=TRIM(Iname), FORM='formatted', STATUS='old',      &
     &      ERR=20)
      GO TO 40
 20   IF (Master) WRITE (stdout,30)
      exit_flag=2
      RETURN
 30   FORMAT (/,' INP_PAR - Unable to open ROMS/TOMS input script ',    &
     &              'file.',                                            &
     &        /,11x,'In distributed-memory applications, the input',    &
     &        /,11x,'script file is processed in parallel. The Unix',   &
     &        /,11x,'routine GETARG is used to get script file name.',  &
     &        /,11x,'For example, in MPI applications make sure that',  &
     &        /,11x,'command line is something like:',/,                &
     &        /,11x,'mpirun -np 4 ocean ocean.in',/,/,11x,'and not',/,  &
     &        /,11x,'mpirun -np 4 ocean < ocean.in',/)
 40   CONTINUE
      CALL read_PhyPar (model, inp, out, Lwrite)
      CALL mp_bcasti (1, model, exit_flag)
      IF (exit_flag.ne.NoError) RETURN
!
!-----------------------------------------------------------------------
!  Set lower and upper bounds indices per domain partition for all
!  nested grids.
!-----------------------------------------------------------------------
!
!  Allocate structure.
!
      IF (.not.allocated(BOUNDS)) THEN
        allocate ( BOUNDS(Ngrids) )
        DO ng=1,Ngrids
          Ntiles=NtileI(ng)*NtileJ(ng)-1
          allocate ( BOUNDS(ng) % tile (-1:Ntiles) )
          allocate ( BOUNDS(ng) % LBi  (-1:Ntiles) )
          allocate ( BOUNDS(ng) % UBi  (-1:Ntiles) )
          allocate ( BOUNDS(ng) % LBj  (-1:Ntiles) )
          allocate ( BOUNDS(ng) % UBj  (-1:Ntiles) )
          allocate ( BOUNDS(ng) % Iend (-1:Ntiles) )
          allocate ( BOUNDS(ng) % Istr (-1:Ntiles) )
          allocate ( BOUNDS(ng) % Jstr (-1:Ntiles) )
          allocate ( BOUNDS(ng) % Jend (-1:Ntiles) )
          allocate ( BOUNDS(ng) % IstrR(-1:Ntiles) )
          allocate ( BOUNDS(ng) % IstrT(-1:Ntiles) )
          allocate ( BOUNDS(ng) % IstrU(-1:Ntiles) )
          allocate ( BOUNDS(ng) % IendR(-1:Ntiles) )
          allocate ( BOUNDS(ng) % IendT(-1:Ntiles) )
          allocate ( BOUNDS(ng) % JstrR(-1:Ntiles) )
          allocate ( BOUNDS(ng) % JstrT(-1:Ntiles) )
          allocate ( BOUNDS(ng) % JstrV(-1:Ntiles) )
          allocate ( BOUNDS(ng) % JendR(-1:Ntiles) )
          allocate ( BOUNDS(ng) % JendT(-1:Ntiles) )
          allocate ( BOUNDS(ng) % Imin (4,0:1,0:Ntiles) )
          allocate ( BOUNDS(ng) % Imax (4,0:1,0:Ntiles) )
          allocate ( BOUNDS(ng) % Jmin (4,0:1,0:Ntiles) )
          allocate ( BOUNDS(ng) % Jmax (4,0:1,0:Ntiles) )
        END DO
      END IF
!
!  Set boundary edge I- or J-indices for each variable type.
!
      DO ng=1,Ngrids
        BOUNDS(ng) % edge(iwest ,p2dvar) = 1
        BOUNDS(ng) % edge(iwest ,r2dvar) = 0
        BOUNDS(ng) % edge(iwest ,u2dvar) = 1
        BOUNDS(ng) % edge(iwest ,v2dvar) = 0
        BOUNDS(ng) % edge(ieast ,p2dvar) = Lm(ng)+1
        BOUNDS(ng) % edge(ieast ,r2dvar) = Lm(ng)+1
        BOUNDS(ng) % edge(ieast ,u2dvar) = Lm(ng)+1
        BOUNDS(ng) % edge(ieast ,v2dvar) = Lm(ng)+1
        BOUNDS(ng) % edge(isouth,p2dvar) = 1
        BOUNDS(ng) % edge(isouth,u2dvar) = 0
        BOUNDS(ng) % edge(isouth,r2dvar) = 0
        BOUNDS(ng) % edge(isouth,v2dvar) = 1
        BOUNDS(ng) % edge(inorth,p2dvar) = Mm(ng)+1
        BOUNDS(ng) % edge(inorth,r2dvar) = Mm(ng)+1
        BOUNDS(ng) % edge(inorth,u2dvar) = Mm(ng)+1
        BOUNDS(ng) % edge(inorth,v2dvar) = Mm(ng)+1
      END DO
!
!  Set tile computational indices and arrays allocation bounds.
!
      Nghost=2
      DO ng=1,Ngrids
        BOUNDS(ng) % LBij = 0
        BOUNDS(ng) % UBij = MAX(Lm(ng)+1,Mm(ng)+1)
        DO tile=-1,NtileI(ng)*NtileJ(ng)-1
          BOUNDS(ng) % tile(tile) = tile
          CALL get_tile (ng, tile, Itile, Jtile,                        &
     &                   BOUNDS(ng) % Istr(tile),                       &
     &                   BOUNDS(ng) % Iend(tile),                       &
     &                   BOUNDS(ng) % Jstr(tile),                       &
     &                   BOUNDS(ng) % Jend(tile),                       &
     &                   BOUNDS(ng) % IstrR(tile),                      &
     &                   BOUNDS(ng) % IstrT(tile),                      &
     &                   BOUNDS(ng) % IstrU(tile),                      &
     &                   BOUNDS(ng) % IendR(tile),                      &
     &                   BOUNDS(ng) % IendT(tile),                      &
     &                   BOUNDS(ng) % JstrR(tile),                      &
     &                   BOUNDS(ng) % JstrT(tile),                      &
     &                   BOUNDS(ng) % JstrV(tile),                      &
     &                   BOUNDS(ng) % JendR(tile),                      &
     &                   BOUNDS(ng) % JendT(tile))
          CALL get_bounds (ng, tile, 0, Nghost, Itile, Jtile,           &
     &                     BOUNDS(ng) % LBi(tile),                      &
     &                     BOUNDS(ng) % UBi(tile),                      &
     &                     BOUNDS(ng) % LBj(tile),                      &
     &                     BOUNDS(ng) % UBj(tile))
        END DO
      END DO
!
!  Set I/O processing minimum (Imin, Jmax) and maximum (Imax, Jmax)
!  indices for non-overlapping (Nghost=0) and overlapping (Nghost>0)
!  tiles for each C-grid type variable.
!
      Nghost=2
      DO ng=1,Ngrids
        DO tile=0,NtileI(ng)*NtileJ(ng)-1
          CALL get_bounds (ng, tile, p2dvar, 0     , Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(1,0,tile),                 &
     &                     BOUNDS(ng) % Imax(1,0,tile),                 &
     &                     BOUNDS(ng) % Jmin(1,0,tile),                 &
     &                     BOUNDS(ng) % Jmax(1,0,tile))
          CALL get_bounds (ng, tile, p2dvar, Nghost, Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(1,1,tile),                 &
     &                     BOUNDS(ng) % Imax(1,1,tile),                 &
     &                     BOUNDS(ng) % Jmin(1,1,tile),                 &
     &                     BOUNDS(ng) % Jmax(1,1,tile))
          CALL get_bounds (ng, tile, r2dvar, 0     , Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(2,0,tile),                 &
     &                     BOUNDS(ng) % Imax(2,0,tile),                 &
     &                     BOUNDS(ng) % Jmin(2,0,tile),                 &
     &                     BOUNDS(ng) % Jmax(2,0,tile))
          CALL get_bounds (ng, tile, r2dvar, Nghost, Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(2,1,tile),                 &
     &                     BOUNDS(ng) % Imax(2,1,tile),                 &
     &                     BOUNDS(ng) % Jmin(2,1,tile),                 &
     &                     BOUNDS(ng) % Jmax(2,1,tile))
          CALL get_bounds (ng, tile, u2dvar, 0     , Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(3,0,tile),                 &
     &                     BOUNDS(ng) % Imax(3,0,tile),                 &
     &                     BOUNDS(ng) % Jmin(3,0,tile),                 &
     &                     BOUNDS(ng) % Jmax(3,0,tile))
          CALL get_bounds (ng, tile, u2dvar, Nghost, Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(3,1,tile),                 &
     &                     BOUNDS(ng) % Imax(3,1,tile),                 &
     &                     BOUNDS(ng) % Jmin(3,1,tile),                 &
     &                     BOUNDS(ng) % Jmax(3,1,tile))
          CALL get_bounds (ng, tile, v2dvar, 0     , Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(4,0,tile),                 &
     &                     BOUNDS(ng) % Imax(4,0,tile),                 &
     &                     BOUNDS(ng) % Jmin(4,0,tile),                 &
     &                     BOUNDS(ng) % Jmax(4,0,tile))
          CALL get_bounds (ng, tile, v2dvar, Nghost, Itile, Jtile,      &
     &                     BOUNDS(ng) % Imin(4,1,tile),                 &
     &                     BOUNDS(ng) % Imax(4,1,tile),                 &
     &                     BOUNDS(ng) % Jmin(4,1,tile),                 &
     &                     BOUNDS(ng) % Jmax(4,1,tile))
        END DO
      END DO
!
!  Set NetCDF IO bounds.
!
      DO ng=1,Ngrids
        CALL get_iobounds (ng)
      END DO
!
!-----------------------------------------------------------------------
!  Check tile partition starting and ending (I,J) indices for illegal
!  domain decomposition parameters NtileI and NtileJ in standard input
!  file.
!-----------------------------------------------------------------------
!
      IF (Master) THEN
        DO ng=1,Ngrids
          WRITE (stdout,50) ng, Lm(ng), Mm(ng), N(ng),                  &
     &                      NtileI(ng), NtileJ(ng)
          Print *,'Number of tracers: ', NT(ng)
          DO tile=0,NtileI(ng)*NtileJ(ng)-1
            npts=(BOUNDS(ng)%Iend(tile)-                                &
     &            BOUNDS(ng)%Istr(tile)+1)*                             &
     &           (BOUNDS(ng)%Jend(tile)-                                &
     &            BOUNDS(ng)%Jstr(tile)+1)*N(ng)
            WRITE (stdout,70) tile,                                     &
     &                        BOUNDS(ng)%Istr(tile),                    &
     &                        BOUNDS(ng)%Iend(tile),                    &
     &                        BOUNDS(ng)%Jstr(tile),                    &
     &                        BOUNDS(ng)%Jend(tile),                    &
     &                        npts
            IF ((BOUNDS(ng)%Iend(tile)-                                 &
     &           BOUNDS(ng)%Istr(tile)+1).lt.2) THEN
              WRITE (stdout,80) ng, 'NtileI = ', NtileI(ng),            &
     &                              'Lm = ', Lm(ng),                    &
     &                              'Istr = ', BOUNDS(ng)%Istr(tile),   &
     &                              '  Iend = ', BOUNDS(ng)%Iend(tile), &
     &                              'NtileI'
              exit_flag=6
              RETURN
            END IF
            IF ((BOUNDS(ng)%Jend(tile)-                                 &
     &           BOUNDS(ng)%Jstr(tile)+1).lt.2) THEN
              WRITE (stdout,80) ng, 'NtileJ = ', NtileJ(ng),            &
     &                              'Mm = ', Mm(ng),                    &
     &                              'Jstr = ', BOUNDS(ng)%Jstr(tile),   &
     &                              '  Jend = ', BOUNDS(ng)%Jend(tile), &
     &                              'NtileJ'
              exit_flag=6
              RETURN
            END IF
          END DO
        END DO
 50     FORMAT (/,' Tile partition information for Grid ',i2.2,':',2x,  &
     &          i4.4,'x',i4.4,'x',i4.4,2x,'tiling: ',i3.3,'x',i3.3,/,/, &
     &          5x,'tile',5x,'Istr',5x,'Iend',5x,'Jstr',5x,'Jend',      &
     &          5x,'Npts',/)
 70     FORMAT (5(5x,i4),2x,i7)
 80     FORMAT (/,' INP_PAR - domain decomposition error in input ',    &
     &                        'script file for grid: ',i2,/,            &
     &          /,11x,'The domain partition parameter, ',a,i3,          &
     &          /,11x,'is incompatible with grid size, ',a,i4,          &
     &          /,11x,'because it yields too small tile, ',a,i3,a,i3,   &
     &          /,11x,'Decrease partition parameter: ',a)
      END IF
      CALL mp_bcasti (1, model, exit_flag)
      IF (exit_flag.ne.NoError) RETURN
!
!  Report tile minimum and maximum fractional grid coordinates.
!
      DO ng=1,Ngrids
        IF (Master) THEN
          WRITE (stdout,90)
          DO tile=0,NtileI(ng)*NtileJ(ng)-1
            WRITE (stdout,100) tile,                                    &
     &                         DOMAIN(ng)%Xmin_rho(tile),               &
     &                         DOMAIN(ng)%Xmax_rho(tile),               &
     &                         DOMAIN(ng)%Ymin_rho(tile),               &
     &                         DOMAIN(ng)%Ymax_rho(tile), 'RHO-points'
          END DO
          WRITE (stdout,'(1x)')
          DO tile=0,NtileI(ng)*NtileJ(ng)-1
            WRITE (stdout,100) tile,                                    &
     &                         DOMAIN(ng)%Xmin_u(tile),                 &
     &                         DOMAIN(ng)%Xmax_u(tile),                 &
     &                         DOMAIN(ng)%Ymin_u(tile),                 &
     &                         DOMAIN(ng)%Ymax_u(tile), '  U-points'
          END DO
          WRITE (stdout,'(1x)')
          DO tile=0,NtileI(ng)*NtileJ(ng)-1
            WRITE (stdout,100) tile,                                    &
     &                         DOMAIN(ng)%Xmin_v(tile),                 &
     &                         DOMAIN(ng)%Xmax_v(tile),                 &
     &                         DOMAIN(ng)%Ymin_v(tile),                 &
     &                         DOMAIN(ng)%Ymax_v(tile), '  V-points'
          END DO
 90       FORMAT (/,' Tile minimum and maximum fractional grid ',       &
     &            'coordinates:',/,                                     &
     &            '   (interior points only)',/,/,                      &
     &            5x,'tile',5x,'Xmin',5x,'Xmax',5x,'Ymin',5x,'Ymax',    &
     &            5x,'grid',/)
 100      FORMAT (5x,i4,4f9.2,2x,a)
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Determine the maximum tile lengths in XI and ETA directions for
!  distributed-memory communications.  Notice that halo size are
!  increased by few points to allow exchanging of private arrays.
!-----------------------------------------------------------------------
!
      Nghost=2
      DO ng=1,Ngrids
        MaxHaloLenI=0
        MaxHaloLenJ=0
        DO tile=0,NtileI(ng)*NtileJ(ng)-1
          Imin=BOUNDS(ng)%LBi(tile)-1
          Imax=BOUNDS(ng)%UBi(tile)+1
          Jmin=BOUNDS(ng)%LBj(tile)-1
          Jmax=BOUNDS(ng)%UBj(tile)+1
          MaxHaloLenI=MAX(MaxHaloLenI,(Imax-Imin+1))
          MaxHaloLenJ=MAX(MaxHaloLenJ,(Jmax-Jmin+1))
        END DO
        HaloSizeI(ng)=Nghost*MaxHaloLenI+6*Nghost
        HaloSizeJ(ng)=Nghost*MaxHaloLenJ+6*Nghost
        TileSide(ng)=MAX(MaxHaloLenI,MaxHaloLenJ)
        TileSize(ng)=MaxHaloLenI*MaxHaloLenJ
        IF (Master) THEN
          WRITE (stdout,110) ng, HaloSizeI(ng), ng, HaloSizeJ(ng),      &
     &                       ng, TileSide(ng),  ng, TileSize(ng)
 110      FORMAT (/,' Maximum halo size in XI and ETA directions:',/,   &
     &            /,'               HaloSizeI(',i1,') = ',i7,           &
     &            /,'               HaloSizeJ(',i1,') = ',i7,           &
     &            /,'                TileSide(',i1,') = ',i7,           &
     &            /,'                TileSize(',i1,') = ',i7,/)
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Read in ice model input parameters.
!-----------------------------------------------------------------------
!
      OPEN (inp, FILE=TRIM(iparnam), FORM='formatted', STATUS='old')
      CALL read_IcePar (inp, out, Lwrite)
!
!-----------------------------------------------------------------------
!  Read in stations input parameters.
!-----------------------------------------------------------------------
!
      OPEN (55, FILE=TRIM(sposnam), FORM='formatted', STATUS='old')
      CALL read_StaPar (model, 55, out, Lwrite)
!
!-----------------------------------------------------------------------
!  Check C-preprocessing options and definitions.
!-----------------------------------------------------------------------
!
      IF (Master) THEN
        CALL checkdefs
        CALL my_flush (out)
      END IF
      CALL mp_bcasti (1, model, exit_flag)
      CALL mp_bcasts (1, model, Coptions)
      IF (exit_flag.ne.NoError) RETURN
!
!-----------------------------------------------------------------------
!  Compute various constants.
!-----------------------------------------------------------------------
!
      gorho0=g/rho0
      DO ng=1,Ngrids
        dtfast(ng)=dt(ng)/REAL(ndtfast(ng),r8)
!
!  Check various parameters for writing time-filtered output
!
        IF (ntsavg(ng).ne.1) THEN
          ntsavg(ng) = 1
          IF (Master) print *, 'Resetting ntsavg to 1'
        END IF
        IF (MOD(dt(ng),1.0_r8).ne.0) THEN
          IF (Master) print *, 'Model timestep must be a whole number'
          exit_flag=2
        END IF
        IF (MOD(dtfil,INT(dt(ng))).ne.0) THEN
          IF (Master) print *,                                          &
     &                'Model timesteps must fall on filter interval'
          exit_flag=2
        END IF
        nfil = nw*dtfil/INT(dt(ng))
        IF (MOD(nfil,navg(ng)).ne.0) THEN
          IF (Master) print *,                                          &
     &                'Must write out in sync with filter length'
          exit_flag=2
        END IF
        IF (nfil/navg(ng).ne.FIL) THEN
          IF (Master) print *, 'Number of filter sums is wrong'
          exit_flag=2
        END IF
        DO itrc=1,FIL
          fcount(itrc) = (1-itrc)*nw/FIL
        END DO
!
!  Hardcode the names of the filter restart files.
!
        filname(1,ng)=TRIM(ADJUSTL('ocean_fil2d.nc'))
        filname(2,ng)=TRIM(ADJUSTL('ocean_filu.nc'))
        filname(3,ng)=TRIM(ADJUSTL('ocean_filv.nc'))
        filname(4,ng)=TRIM(ADJUSTL('ocean_filakt.nc'))
        filname(5,ng)=TRIM(ADJUSTL('ocean_filaks.nc'))
        filname(6,ng)=TRIM(ADJUSTL('ocean_filakv.nc'))
        filname(7,ng)=TRIM(ADJUSTL('ocean_filr.nc'))
        filname(8,ng)=TRIM(ADJUSTL('ocean_filt.nc'))
        filname(9,ng)=TRIM(ADJUSTL('ocean_filw.nc'))
        filname(10,ng)=TRIM(ADJUSTL('ocean_filice.nc'))
!
!  Take the square root of the biharmonic coefficients so it can
!  be applied to each harmonic operator.
!
        visc4(ng)=SQRT(ABS(visc4(ng)))
        tkenu4(ng)=SQRT(ABS(tkenu4(ng)))
!
!  Compute inverse nudging coefficients (1/s) used in various tasks.
!
        IF (Znudg(ng).gt.0.0_r8) THEN
          Znudg(ng)=1.0_r8/(Znudg(ng)*86400.0_r8)
        ELSE
          Znudg(ng)=0.0_r8
        END IF
        IF (M2nudg(ng).gt.0.0_r8) THEN
          M2nudg(ng)=1.0_r8/(M2nudg(ng)*86400.0_r8)
        ELSE
          M2nudg(ng)=0.0_r8
        END IF
        IF (M3nudg(ng).gt.0.0_r8) THEN
          M3nudg(ng)=1.0_r8/(M3nudg(ng)*86400.0_r8)
        ELSE
          M3nudg(ng)=0.0_r8
        END IF
!
!  Convert momentum stresses and tracer flux scales to kinematic
!  Values. Recall, that all the model fluxes are kinematic.
!
        cff=1.0_r8/rho0
        Fscale(idUsms,ng)=cff*Fscale(idUsms,ng)
        Fscale(idVsms,ng)=cff*Fscale(idVsms,ng)
        Fscale(idUbms,ng)=cff*Fscale(idUbms,ng)
        Fscale(idVbms,ng)=cff*Fscale(idVbms,ng)
        Fscale(idUbrs,ng)=cff*Fscale(idUbrs,ng)
        Fscale(idVbrs,ng)=cff*Fscale(idVbrs,ng)
        Fscale(idUbws,ng)=cff*Fscale(idUbws,ng)
        Fscale(idVbws,ng)=cff*Fscale(idVbws,ng)
        Fscale(idUbcs,ng)=cff*Fscale(idUbcs,ng)
        Fscale(idVbcs,ng)=cff*Fscale(idVbcs,ng)
        cff=1.0_r8/(rho0*Cp)
        Fscale(idTsur(itemp),ng)=cff*Fscale(idTsur(itemp),ng)
        Fscale(idTbot(itemp),ng)=cff*Fscale(idTbot(itemp),ng)
        Fscale(idSrad,ng)=cff*Fscale(idSrad,ng)
        Fscale(idLdwn,ng)=cff*Fscale(idLdwn,ng)
        Fscale(idLrad,ng)=cff*Fscale(idLrad,ng)
        Fscale(idLhea,ng)=cff*Fscale(idLhea,ng)
        Fscale(idShea,ng)=cff*Fscale(idShea,ng)
        Fscale(iddQdT,ng)=cff*Fscale(iddQdT,ng)
      END DO
!
!-----------------------------------------------------------------------
!  Initialize random number sequence so we can get identical results
!  everytime that we run the same solution.
!-----------------------------------------------------------------------
!
      sequence=759
      CALL ran_seed (sequence)
      RETURN
      END SUBROUTINE inp_par
      SUBROUTINE read_PhyPar (model, inp, out, Lwrite)
!
!=======================================================================
!                                                                      !
!  This routine reads in physical model input parameters.              !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
      USE mod_strings
!
      implicit none
!
!  Imported variable declarations
!
      logical, intent(in) :: Lwrite
      integer, intent(in) :: model, inp, out
!
!  Local variable declarations.
!
      logical :: inhere
      integer :: Lstr, Npts, Nval, i, itrc, k, ng, status
      integer :: decode_line, load_i, load_l, load_r
      logical, allocatable :: Ltracer(:,:)
      real(r8), allocatable :: Rtracer(:,:)
      real(r8), allocatable :: tracer(:,:)
      real(r8), dimension(100) :: Rval
      character (len=1 ), parameter :: blank = ' '
      character (len=19) :: ref_att
      character (len=40) :: KeyWord
      character (len=160) :: fname, line
      character (len=160), dimension(100) :: Cval
!
!-----------------------------------------------------------------------
!  Read in physical model parameters. Then, load input data into module.
!  Take into account nested grid configurations.
!-----------------------------------------------------------------------
!
      DO WHILE (.TRUE.)
        READ (inp,'(a)',ERR=10,END=20) line
        status=decode_line(line, KeyWord, Nval, Cval, Rval)
        IF (status.gt.0) THEN
          IF (TRIM(KeyWord).eq.'TITLE') THEN
            IF (Nval.eq.1) THEN
              title=TRIM(ADJUSTL(Cval(Nval)))
            ELSE
              WRITE(title,'(a,1x,a)') TRIM(ADJUSTL(title)),             &
     &                                TRIM(ADJUSTL(Cval(Nval)))
            END IF           
          ELSE IF (TRIM(KeyWord).eq.'MyAppCPP') THEN
            DO i=1,LEN(MyAppCPP)
              MyAppCPP(i:i)=blank
            END DO
            MyAppCPP=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'VARNAME') THEN
            DO i=1,LEN(varname)
              varname(i:i)=blank
            END DO
            varname=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'Lm') THEN
            Npts=load_i(Nval, Rval, Ngrids, Lm)
            DO ng=1,Ngrids
              IF (Lm(ng).le.0) THEN
                IF (Master) WRITE (out,300) 'Lm', ng,                   &
     &                                      'must be greater than zero.'
                exit_flag=5
                RETURN
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Mm') THEN
            Npts=load_i(Nval, Rval, Ngrids, Mm)
            DO ng=1,Ngrids
              IF (Mm(ng).le.0) THEN
                IF (Master) WRITE (out,300) 'Mm', ng,                   &
     &                                      'must be greater than zero.'
                exit_flag=5
                RETURN
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'N') THEN
            Npts=load_i(Nval, Rval, Ngrids, N)
            DO ng=1,Ngrids
              IF (N(ng).lt.0) THEN
                IF (Master) WRITE (out,300) 'N', ng,                    &
     &                                      'must be greater than zero.'
                exit_flag=5
                RETURN
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'NAT') THEN
            Npts=load_i(Nval, Rval, 1, NAT)
            IF ((NAT.lt.1).or.(NAT.gt.2)) THEN
              IF (Master) WRITE (out,290) 'NAT = ', NAT,                &
     &                          'make sure that NAT is either 1 or 2.'
              exit_flag=5
              RETURN
            END IF
            IF (NAT.ne.2) THEN
              IF (Master) WRITE (out,290) 'NAT = ', NAT,                &
     &                          'make sure that NAT is equal to 2.'
              exit_flag=5
              RETURN
            END IF
          ELSE IF (TRIM(KeyWord).eq.'NtileI') THEN
            Npts=load_i(Nval, Rval, Ngrids, NtileI)
            NtileX(1:Ngrids)=1
          ELSE IF (TRIM(KeyWord).eq.'NtileJ') THEN
            Npts=load_i(Nval, Rval, Ngrids, NtileJ)
            NtileE(1:Ngrids)=1
            CALL initialize_param
            CALL initialize_scalars
            CALL initialize_ncparam
            IF (.not.allocated(Ltracer)) THEN
              allocate (Ltracer(NAT+NPT,Ngrids))
            END IF
            IF (.not.allocated(Rtracer)) THEN
              allocate (Rtracer(NAT+NPT,Ngrids))
            END IF
            IF (.not.allocated(tracer)) THEN
              allocate (tracer(MT,Ngrids))
            END IF
          ELSE IF (TRIM(KeyWord).eq.'NTIMES') THEN
            Npts=load_i(Nval, Rval, Ngrids, ntimes)
          ELSE IF (TRIM(KeyWord).eq.'DT') THEN
            Npts=load_r(Nval, Rval, Ngrids, dt)
          ELSE IF (TRIM(KeyWord).eq.'NDTFAST') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndtfast)
          ELSE IF (TRIM(KeyWord).eq.'ERstr') THEN
            Npts=load_i(Nval, Rval, 1, ERstr)
          ELSE IF (TRIM(KeyWord).eq.'ERend') THEN
            Npts=load_i(Nval, Rval, 1, ERend)
          ELSE IF (TRIM(KeyWord).eq.'Nouter') THEN
            Npts=load_i(Nval, Rval, 1, Nouter)
          ELSE IF (TRIM(KeyWord).eq.'Ninner') THEN
            Npts=load_i(Nval, Rval, 1, Ninner)
          ELSE IF (TRIM(KeyWord).eq.'Nintervals') THEN
            Npts=load_i(Nval, Rval, 1, Nintervals)
          ELSE IF (TRIM(KeyWord).eq.'NRREC') THEN
            Npts=load_i(Nval, Rval, Ngrids, nrrec)
            DO ng=1,Ngrids
              IF (nrrec(ng).lt.0) THEN
                LastRec(ng)=.TRUE.
              ELSE
                LastRec(ng)=.FALSE.
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'LcycleRST') THEN
            Npts=load_l(Nval, Cval, Ngrids, LcycleRST)
          ELSE IF (TRIM(KeyWord).eq.'NRST') THEN
            Npts=load_i(Nval, Rval, Ngrids, nRST)
          ELSE IF (TRIM(KeyWord).eq.'NSTA') THEN
            Npts=load_i(Nval, Rval, Ngrids, nSTA)
          ELSE IF (TRIM(KeyWord).eq.'NFLT') THEN
            Npts=load_i(Nval, Rval, Ngrids, nFLT)
          ELSE IF (TRIM(KeyWord).eq.'NFISHES') THEN
            Npts=load_i(Nval, Rval, Ngrids, nFISHES)
          ELSE IF (TRIM(KeyWord).eq.'NINFO') THEN
            Npts=load_i(Nval, Rval, Ngrids, ninfo)
          ELSE IF (TRIM(KeyWord).eq.'LDEFOUT') THEN
            Npts=load_l(Nval, Cval, Ngrids, ldefout)
          ELSE IF (TRIM(KeyWord).eq.'NHIS') THEN
            Npts=load_i(Nval, Rval, Ngrids, nHIS)
          ELSE IF (TRIM(KeyWord).eq.'NDEFHIS') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefHIS)
          ELSE IF (TRIM(KeyWord).eq.'NTSAVG') THEN
            Npts=load_i(Nval, Rval, Ngrids, ntsAVG)
          ELSE IF (TRIM(KeyWord).eq.'NAVG') THEN
            Npts=load_i(Nval, Rval, Ngrids, nAVG)
          ELSE IF (TRIM(KeyWord).eq.'NDEFAVG') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefAVG)
          ELSE IF (TRIM(KeyWord).eq.'NTSAVG2') THEN
            Npts=load_i(Nval, Rval, Ngrids, ntsAVG2)
          ELSE IF (TRIM(KeyWord).eq.'NAVG2') THEN
            Npts=load_i(Nval, Rval, Ngrids, nAVG2)
          ELSE IF (TRIM(KeyWord).eq.'NDEFAVG2') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefAVG2)
          ELSE IF (TRIM(KeyWord).eq.'NTSDIA') THEN
            Npts=load_i(Nval, Rval, Ngrids, ntsDIA)
          ELSE IF (TRIM(KeyWord).eq.'NDIA') THEN
            Npts=load_i(Nval, Rval, Ngrids, nDIA)
          ELSE IF (TRIM(KeyWord).eq.'NDEFDIA') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefDIA)
          ELSE IF (TRIM(KeyWord).eq.'LcycleTLM') THEN
            Npts=load_l(Nval, Cval, Ngrids, LcycleTLM)
          ELSE IF (TRIM(KeyWord).eq.'NTLM') THEN
            Npts=load_i(Nval, Rval, Ngrids, nTLM)
          ELSE IF (TRIM(KeyWord).eq.'NDEFTLM') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefTLM)
          ELSE IF (TRIM(KeyWord).eq.'LcycleADJ') THEN
            Npts=load_l(Nval, Cval, Ngrids, LcycleADJ)
          ELSE IF (TRIM(KeyWord).eq.'NADJ') THEN
            Npts=load_i(Nval, Rval, Ngrids, nADJ)
          ELSE IF (TRIM(KeyWord).eq.'NDEFADJ') THEN
            Npts=load_i(Nval, Rval, Ngrids, ndefADJ)
          ELSE IF (TRIM(KeyWord).eq.'NOBC') THEN
            Npts=load_i(Nval, Rval, Ngrids, nOBC)
          ELSE IF (TRIM(KeyWord).eq.'NSFF') THEN
            Npts=load_i(Nval, Rval, Ngrids, nSFF)
          ELSE IF (TRIM(KeyWord).eq.'LrstGST') THEN
            Npts=load_l(Nval, Cval, 1, LrstGST)
          ELSE IF (TRIM(KeyWord).eq.'MaxIterGST') THEN
            Npts=load_i(Nval, Rval, 1, MaxIterGST)
          ELSE IF (TRIM(KeyWord).eq.'NGST') THEN
            Npts=load_i(Nval, Rval, 1, nGST)
          ELSE IF (TRIM(KeyWord).eq.'TNU2') THEN
            Npts=load_r(Nval, Rval, (NAT+NPT)*Ngrids, Rtracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT+NPT
                tnu2(itrc,ng)=Rtracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'TNU4') THEN
            Npts=load_r(Nval, Rval, (NAT+NPT)*Ngrids, Rtracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT+NPT
                tnu4(itrc,ng)=Rtracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'VISC2') THEN
            Npts=load_r(Nval, Rval, Ngrids, visc2)
          ELSE IF (TRIM(KeyWord).eq.'VISC4') THEN
            Npts=load_r(Nval, Rval, Ngrids, visc4)
          ELSE IF (TRIM(KeyWord).eq.'AKT_BAK') THEN
            Npts=load_r(Nval, Rval, (NAT+NPT)*Ngrids, Rtracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT+NPT
                Akt_bak(itrc,ng)=Rtracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'AKV_BAK') THEN
            Npts=load_r(Nval, Rval, Ngrids, Akv_bak)
          ELSE IF (TRIM(KeyWord).eq.'AKK_BAK') THEN
            Npts=load_r(Nval, Rval, Ngrids, Akk_bak)
          ELSE IF (TRIM(KeyWord).eq.'AKP_BAK') THEN
            Npts=load_r(Nval, Rval, Ngrids, Akp_bak)
          ELSE IF (TRIM(KeyWord).eq.'TKENU2') THEN
            Npts=load_r(Nval, Rval, Ngrids, tkenu2)
          ELSE IF (TRIM(KeyWord).eq.'TKENU4') THEN
            Npts=load_r(Nval, Rval, Ngrids, tkenu4)
          ELSE IF (TRIM(KeyWord).eq.'GLS_P') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_p)
          ELSE IF (TRIM(KeyWord).eq.'GLS_M') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_m)
          ELSE IF (TRIM(KeyWord).eq.'GLS_N') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_n)
          ELSE IF (TRIM(KeyWord).eq.'GLS_Kmin') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_Kmin)
          ELSE IF (TRIM(KeyWord).eq.'GLS_Pmin') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_Pmin)
          ELSE IF (TRIM(KeyWord).eq.'GLS_CMU0') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_cmu0)
          ELSE IF (TRIM(KeyWord).eq.'GLS_C1') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_c1)
          ELSE IF (TRIM(KeyWord).eq.'GLS_C2') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_c2)
          ELSE IF (TRIM(KeyWord).eq.'GLS_C3M') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_c3m)
          ELSE IF (TRIM(KeyWord).eq.'GLS_C3P') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_c3p)
          ELSE IF (TRIM(KeyWord).eq.'GLS_SIGK') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_sigk)
          ELSE IF (TRIM(KeyWord).eq.'GLS_SIGP') THEN
            Npts=load_r(Nval, Rval, Ngrids, gls_sigp)
          ELSE IF (TRIM(KeyWord).eq.'CHARNOK_ALPHA') THEN
            Npts=load_r(Nval, Rval, Ngrids, charnok_alpha)
          ELSE IF (TRIM(KeyWord).eq.'ZOS_HSIG_ALPHA') THEN
            Npts=load_r(Nval, Rval, Ngrids, zos_hsig_alpha)
          ELSE IF (TRIM(KeyWord).eq.'SZ_ALPHA') THEN
            Npts=load_r(Nval, Rval, Ngrids, sz_alpha)
          ELSE IF (TRIM(KeyWord).eq.'CRGBAN_CW') THEN
            Npts=load_r(Nval, Rval, Ngrids, crgban_cw)
          ELSE IF (TRIM(KeyWord).eq.'RDRG') THEN
            Npts=load_r(Nval, Rval, Ngrids, rdrg)
          ELSE IF (TRIM(KeyWord).eq.'RDRG2') THEN
            Npts=load_r(Nval, Rval, Ngrids, rdrg2)
          ELSE IF (TRIM(KeyWord).eq.'Zob') THEN
            Npts=load_r(Nval, Rval, Ngrids, Zob)
          ELSE IF (TRIM(KeyWord).eq.'Zos') THEN
            Npts=load_r(Nval, Rval, Ngrids, Zos)
          ELSE IF (TRIM(KeyWord).eq.'BLK_ZQ') THEN
            Npts=load_r(Nval, Rval, Ngrids, blk_ZQ)
          ELSE IF (TRIM(KeyWord).eq.'BLK_ZT') THEN
            Npts=load_r(Nval, Rval, Ngrids, blk_ZT)
          ELSE IF (TRIM(KeyWord).eq.'BLK_ZW') THEN
            Npts=load_r(Nval, Rval, Ngrids, blk_ZW)
          ELSE IF (TRIM(KeyWord).eq.'DCRIT') THEN
            Npts=load_r(Nval, Rval, Ngrids, Dcrit)
          ELSE IF (TRIM(KeyWord).eq.'WTYPE') THEN
            Npts=load_i(Nval, Rval, Ngrids, lmd_Jwt)
          ELSE IF (TRIM(KeyWord).eq.'LEVSFRC') THEN
            Npts=load_i(Nval, Rval, Ngrids, levsfrc)
          ELSE IF (TRIM(KeyWord).eq.'LEVBFRC') THEN
            Npts=load_i(Nval, Rval, Ngrids, levbfrc)
          ELSE IF (TRIM(KeyWord).eq.'Vtransform') THEN
            Npts=load_i(Nval, Rval, Ngrids, Vtransform)
            DO ng=1,Ngrids
              IF ((Vtransform(ng).lt.0).or.                             &
     &            (Vtransform(ng).gt.3)) THEN
                IF (Master) WRITE (out,260) 'Vtransform = ',            &
     &                                      Vtransform(ng),             &
     &                                      'Must be either 1, 2 or 3'
                exit_flag=5
                RETURN
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Vstretching') THEN
            Npts=load_i(Nval, Rval, Ngrids, Vstretching)
            DO ng=1,Ngrids
              IF ((Vstretching(ng).lt.0).or.                            &
     &            (Vstretching(ng).gt.3)) THEN
                IF (Master) WRITE (out,260) 'Vstretching = ',           &
     &                                      Vstretching(ng),            &
     &                                      'Must between 1 and 3'
                exit_flag=5
                RETURN
              END IF
            END DO
          ELSE IF (TRIM(KeyWord).eq.'THETA_S') THEN
            Npts=load_r(Nval, Rval, Ngrids, theta_s)
          ELSE IF (TRIM(KeyWord).eq.'THETA_B') THEN
            Npts=load_r(Nval, Rval, Ngrids, theta_b)
          ELSE IF (TRIM(KeyWord).eq.'TCLINE') THEN
            Npts=load_r(Nval, Rval, Ngrids, Tcline)
            DO ng=1,Ngrids
              hc(ng)=Tcline(ng)
            END DO
          ELSE IF (TRIM(KeyWord).eq.'RHO0') THEN
            Npts=load_r(Nval, Rval, 1, rho0)
          ELSE IF (TRIM(KeyWord).eq.'BVF_BAK') THEN
            Npts=load_r(Nval, Rval, 1, bvf_bak)
          ELSE IF (TRIM(KeyWord).eq.'DSTART') THEN
            Npts=load_r(Nval, Rval, 1, dstart)
          ELSE IF (TRIM(KeyWord).eq.'TIDE_START') THEN
            Npts=load_r(Nval, Rval, 1, tide_start)
          ELSE IF (TRIM(KeyWord).eq.'TIME_REF') THEN
            Npts=load_r(Nval, Rval, 1, time_ref)
            r_text=ref_att(time_ref,r_date)
          ELSE IF (TRIM(KeyWord).eq.'TNUDG') THEN
            Npts=load_r(Nval, Rval, (NAT+NPT)*Ngrids, Rtracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT+NPT
                Tnudg(itrc,ng)=Rtracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'ZNUDG') THEN
            Npts=load_r(Nval, Rval, Ngrids, Znudg)
          ELSE IF (TRIM(KeyWord).eq.'M2NUDG') THEN
            Npts=load_r(Nval, Rval, Ngrids, M2nudg)
          ELSE IF (TRIM(KeyWord).eq.'M3NUDG') THEN
            Npts=load_r(Nval, Rval, Ngrids, M3nudg)
          ELSE IF (TRIM(KeyWord).eq.'OBCFAC') THEN
            Npts=load_r(Nval, Rval, Ngrids, obcfac)
          ELSE IF (TRIM(KeyWord).eq.'R0') THEN
            Npts=load_r(Nval, Rval, Ngrids, R0)
            DO ng=1,Ngrids
              IF (R0(ng).lt.100.0_r8) R0(ng)=R0(ng)+1000.0_r8
            END DO
          ELSE IF (TRIM(KeyWord).eq.'T0') THEN
            Npts=load_r(Nval, Rval, Ngrids, T0)
          ELSE IF (TRIM(KeyWord).eq.'S0') THEN
            Npts=load_r(Nval, Rval, Ngrids, S0)
          ELSE IF (TRIM(KeyWord).eq.'TCOEF') THEN
            Npts=load_r(Nval, Rval, Ngrids, Tcoef)
            DO ng=1,Ngrids
              Tcoef(ng)=ABS(Tcoef(ng))
            END DO
          ELSE IF (TRIM(KeyWord).eq.'SCOEF') THEN
            Npts=load_r(Nval, Rval, Ngrids, Scoef)
            DO ng=1,Ngrids
              Scoef(ng)=ABS(Scoef(ng))
            END DO
          ELSE IF (TRIM(KeyWord).eq.'GAMMA2') THEN
            Npts=load_r(Nval, Rval, Ngrids, gamma2)
          ELSE IF (TRIM(KeyWord).eq.'Hout(idFsur)') THEN
            IF (idFsur.eq.0) THEN
              IF (Master) WRITE (out,280) 'idFsur'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idFsur,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbar)') THEN
            IF (idUbar.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbar'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbar,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbar)') THEN
            IF (idVbar.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbar'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbar,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUvel)') THEN
            IF (idUvel.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUvel'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVvel)') THEN
            IF (idVvel.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVvel'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWvel)') THEN
            IF (idWvel.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWvel'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idOvel)') THEN
            IF (idOvel.eq.0) THEN
              IF (Master) WRITE (out,280) 'idOvel'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idOvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTvar)') THEN
            IF (MAXVAL(idTvar).eq.0) THEN
              IF (Master) WRITE (out,280) 'idTvar'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, NAT*Ngrids, Ltracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT
                i=idTvar(itrc)
                Hout(i,ng)=Ltracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUair)') THEN
            IF (idUair.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUair'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUair,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVair)') THEN
            IF (idVair.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVair'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVair,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUsms)') THEN
            IF (idUsms.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUsms'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUsms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVsms)') THEN
            IF (idVsms.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVsms'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVsms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbms)') THEN
            IF (idUbms.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbms'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbms)') THEN
            IF (idVbms.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbms'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbrs)') THEN
            IF (idUbrs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbrs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbrs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbrs)') THEN
            IF (idVbrs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbrs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbrs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbws)') THEN
            IF (idUbws.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbws'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbws,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbws)') THEN
            IF (idVbws.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbws'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbws,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbcs)') THEN
            IF (idUbcs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbcs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbcs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbcs)') THEN
            IF (idVbcs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbcs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbcs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbot)') THEN
            IF (idUbot.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbot'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbot,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbot)') THEN
            IF (idVbot.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbot'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbot,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUbur)') THEN
            IF (idUbur.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUbur'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUbur,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVbvr)') THEN
            IF (idVbvr.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVbvr'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVbvr,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW2xx)') THEN
            IF (idW2xx.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW2xx'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW2xx,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW2xy)') THEN
            IF (idW2xy.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW2xy'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW2xy,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW2yy)') THEN
            IF (idW2yy.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW2yy'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW2yy,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW3xx)') THEN
            IF (idW3xx.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW3xx'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW3xx,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW3xy)') THEN
            IF (idW3xy.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW3xy'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW3xy,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW3yy)') THEN
            IF (idW3yy.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW3yy'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW3yy,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW3zx)') THEN
            IF (idW3zx.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW3zx'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW3zx,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idW3zy)') THEN
            IF (idW3zy.eq.0) THEN
              IF (Master) WRITE (out,280) 'idW3zy'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idW3zy,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idU2rs)') THEN
            IF (idU2rs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idU2rs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idU2rs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idV2rs)') THEN
            IF (idV2rs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idV2rs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idV2rs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idU2Sd)') THEN
            IF (idU2Sd.eq.0) THEN
              IF (Master) WRITE (out,280) 'idU2Sd'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idU2Sd,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idV2Sd)') THEN
            IF (idV2Sd.eq.0) THEN
              IF (Master) WRITE (out,280) 'idV2Sd'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idV2Sd,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idU3rs)') THEN
            IF (idU3rs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idU3rs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idU3rs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idV3rs)') THEN
            IF (idV3rs.eq.0) THEN
              IF (Master) WRITE (out,280) 'idV3rs'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idV3rs,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idU3Sd)') THEN
            IF (idU3Sd.eq.0) THEN
              IF (Master) WRITE (out,280) 'idU3Sd'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idU3Sd,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idV3Sd)') THEN
            IF (idV3Sd.eq.0) THEN
              IF (Master) WRITE (out,280) 'idV3Sd'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idV3Sd,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWamp)') THEN
            IF (idWamp.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWamp'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWamp,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWlen)') THEN
            IF (idWlen.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWlen'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWlen,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWdir)') THEN
            IF (idWdir.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWdir'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWdir,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTsur)') THEN
            IF (idTsur(itemp).eq.0) THEN
              IF (Master) WRITE (out,280) 'idTsur'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, NAT*Ngrids, Ltracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT
                i=idTsur(itrc)
                Hout(i,ng)=Ltracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Hout(idLhea)') THEN
            IF (idLhea.eq.0) THEN
              IF (Master) WRITE (out,280) 'idLhea'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idLhea,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idShea)') THEN
            IF (idShea.eq.0) THEN
              IF (Master) WRITE (out,280) 'idShea'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idShea,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idLrad)') THEN
            IF (idLrad.eq.0) THEN
              IF (Master) WRITE (out,280) 'idLrad'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idLrad,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSrad)') THEN
            IF (idSrad.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSrad'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSrad,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idevap)') THEN
            IF (idevap.eq.0) THEN
              IF (Master) WRITE (out,280) 'idevap'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idevap,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idrain)') THEN
            IF (idrain.eq.0) THEN
              IF (Master) WRITE (out,280) 'idrain'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idrain,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idDano)') THEN
            IF (idDano.eq.0) THEN
              IF (Master) WRITE (out,280) 'idDano'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idDano,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVvis)') THEN
            IF (idVvis.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVvis'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVvis,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTdif)') THEN
            IF (idTdif.eq.0) THEN
              IF (Master) WRITE (out,280) 'idTdif'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idTdif,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSdif)') THEN
            IF (idSdif.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSdif'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSdif,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idHsbl)') THEN
            IF (idHsbl.eq.0) THEN
              IF (Master) WRITE (out,280) 'idHsbl'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idHsbl,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idHbbl)') THEN
            IF (idHbbl.eq.0) THEN
              IF (Master) WRITE (out,280) 'idHbbl'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idHbbl,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idMtke)') THEN
            IF (idMtke.eq.0) THEN
              IF (Master) WRITE (out,280) 'idMtke'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idMtke,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idMtls)') THEN
            IF (idMtls.eq.0) THEN
              IF (Master) WRITE (out,280) 'idMtls'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idMtls,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idUice)') THEN
            IF (idUice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idUice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idUice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idVice)') THEN
            IF (idVice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idVice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idVice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idAice)') THEN
            IF (idAice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idAice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idAice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idHice)') THEN
            IF (idHice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idHice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idHice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idHsno)') THEN
            IF (idHsno.eq.0) THEN
              IF (Master) WRITE (out,280) 'idHsno'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idHsno,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTice)') THEN
            IF (idTice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idTice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idTice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTimid)') THEN
            IF (idTimid.eq.0) THEN
              IF (Master) WRITE (out,280) 'idTimid'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idTimid,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idTauiw)') THEN
            IF (idTauiw.eq.0) THEN
              IF (Master) WRITE (out,280) 'idTauiw'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idTauiw,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idChuiw)') THEN
            IF (idChuiw.eq.0) THEN
              IF (Master) WRITE (out,280) 'idChuiw'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idChuiw,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSfwat)') THEN
            IF (idSfwat.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSfwat'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSfwat,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idAgeice)') THEN
            IF (idAgeice.eq.0) THEN
              IF (Master) WRITE (out,280) 'idAgeice'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idAgeice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idIomflx)') THEN
            IF (idIomflx.eq.0) THEN
              IF (Master) WRITE (out,280) 'idIomflx'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idIomflx,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSig11)') THEN
            IF (idSig11.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSig11'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSig11,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSig12)') THEN
            IF (idSig12.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSig12'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSig12,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idSig22)') THEN
            IF (idSig22.eq.0) THEN
              IF (Master) WRITE (out,280) 'idSig22'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idSig22,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idT0mk)') THEN
            IF (idT0mk.eq.0) THEN
              IF (Master) WRITE (out,280) 'idT0mk'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idT0mk,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idS0mk)') THEN
            IF (idS0mk.eq.0) THEN
              IF (Master) WRITE (out,280) 'idS0mk'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idS0mk,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWfr)') THEN
            IF (idWfr.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWfr'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWfr,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWai)') THEN
            IF (idWai.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWai'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWai,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWao)') THEN
            IF (idWao.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWao'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWao,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWio)') THEN
            IF (idWio.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWio'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWio,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout(idWro)') THEN
            IF (idWro.eq.0) THEN
              IF (Master) WRITE (out,280) 'idWro'
              exit_flag=5
              RETURN
            END IF
            Npts=load_l(Nval, Cval, Ngrids, Hout(idWro,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idFsur)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idFsur,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idUbar)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idUbar,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idVbar)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idVbar,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idUvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idUvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idVvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idVvel,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idTvar)') THEN
            Npts=load_l(Nval, Cval, NAT*Ngrids, Ltracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT
                i=idTvar(itrc)
                Hout2(i,ng)=Ltracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idUair)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idUair,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idVair)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idVair,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idUsms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idUsms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idVsms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idVsms,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idWamp)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idWamp,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idWlen)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idWlen,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idWdir)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idWdir,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idTsur)') THEN
            Npts=load_l(Nval, Cval, NAT*Ngrids, Ltracer)
            DO ng=1,Ngrids
              DO itrc=1,NAT
                i=idTsur(itrc)
                Hout2(i,ng)=Ltracer(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idLhea)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idLhea,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idShea)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idShea,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idLrad)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idLrad,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idSrad)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idSrad,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idevap)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idevap,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idrain)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idrain,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idDano)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idDano,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idHsbl)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idHsbl,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idHbbl)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idHbbl,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idUice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idUice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idVice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idVice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idAice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idAice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idHice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idHice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idHsno)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idHsno,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idTice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idTice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idTimid)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idTimid,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idTauiw)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idTauiw,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idChuiw)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idChuiw,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idSfwat)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idSfwat,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idAgeice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idAgeice,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idIomflx)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idIomflx,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idSig11)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idSig11,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idSig12)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idSig12,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idSig22)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idSig22,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idT0mk)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idT0mk,:))
          ELSE IF (TRIM(KeyWord).eq.'Hout2(idS0mk)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Hout2(idS0mk,:))
          ELSE IF (TRIM(KeyWord).eq.'NUSER') THEN
            Npts=load_i(Nval, Rval, 1, Nuser)
          ELSE IF (TRIM(KeyWord).eq.'USER') THEN
            Npts=load_r(Nval, Rval, MAX(1,Nuser), user)
          ELSE IF (TRIM(KeyWord).eq.'NC_SHUFFLE') THEN
            Npts=load_i(Nval, Rval, 1, shuffle)
          ELSE IF (TRIM(KeyWord).eq.'NC_DEFLATE') THEN
            Npts=load_i(Nval, Rval, 1, deflate)
          ELSE IF (TRIM(KeyWord).eq.'NC_DLEVEL') THEN
            Npts=load_i(Nval, Rval, 1, deflate_level)
          ELSE IF (TRIM(KeyWord).eq.'GSTNAME') THEN
            DO i=1,LEN(GSTname(Nval))
              GSTname(Nval)(i:i)=blank
            END DO
            GSTname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'RSTNAME') THEN
            DO i=1,LEN(RSTname(Nval))
              RSTname(Nval)(i:i)=blank
            END DO
            RSTname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'HISNAME') THEN
            DO i=1,LEN(HISname(Nval))
              HISname(Nval)(i:i)=blank
              HISbase(Nval)(i:i)=blank
            END DO
            HISname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            HISbase(Nval)=TRIM(ADJUSTL(HISname(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'TLMNAME') THEN
            DO i=1,LEN(TLMname(Nval))
              TLMname(Nval)(i:i)=blank
              TLMbase(Nval)(i:i)=blank
            END DO
            TLMname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            TLMbase(Nval)=TRIM(ADJUSTL(TLMname(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'TLFNAME') THEN
            DO i=1,LEN(TLMname(Nval))
              TLFname(Nval)(i:i)=blank
            END DO
            TLFname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'ADJNAME') THEN
            DO i=1,LEN(ADJname(Nval))
              ADJname(Nval)(i:i)=blank
              ADJbase(Nval)(i:i)=blank
            END DO
            ADJname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            ADJbase(Nval)=TRIM(ADJUSTL(ADJname(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'AVGNAME') THEN
            DO i=1,LEN(AVGname(Nval))
              AVGname(Nval)(i:i)=blank
              AVGbase(Nval)(i:i)=blank
            END DO
            AVGname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            AVGbase(Nval)=TRIM(ADJUSTL(AVGname(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'AVG2NAME') THEN
            DO i=1,LEN(AVG2name(Nval))
              AVG2name(Nval)(i:i)=blank
              AVG2base(Nval)(i:i)=blank
            END DO
            AVG2name(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            AVG2base(Nval)=TRIM(ADJUSTL(AVG2name(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'DIANAME') THEN
            DO i=1,LEN(DIAname(Nval))
              DIAname(Nval)(i:i)=blank
              DIAbase(Nval)(i:i)=blank
            END DO
            DIAname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            DIAbase(Nval)=TRIM(ADJUSTL(DIAname(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'STANAME') THEN
            DO i=1,LEN(STAname(Nval))
              STAname(Nval)(i:i)=blank
            END DO
            STAname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'FLTNAME') THEN
            DO i=1,LEN(FLTname(Nval))
              FLTname(Nval)(i:i)=blank
            END DO
            FLTname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'GRDNAME') THEN
            DO i=1,LEN(GRDname(Nval))
              GRDname(Nval)(i:i)=blank
            END DO
            GRDname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'ININAME') THEN
            DO i=1,LEN(INIname(Nval))
              INIname(Nval)(i:i)=blank
            END DO
            INIname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'IRPNAME') THEN
            DO i=1,LEN(ITLname(Nval))
              IRPname(Nval)(i:i)=blank
            END DO
            IRPname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'ITLNAME') THEN
            DO i=1,LEN(ITLname(Nval))
              ITLname(Nval)(i:i)=blank
            END DO
            ITLname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'IADNAME') THEN
            DO i=1,LEN(IADname(Nval))
              IADname(Nval)(i:i)=blank
            END DO
            IADname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'NFFILES') THEN
            Npts=load_i(Nval, Rval, Ngrids, nFfiles)
            DO ng=1,Ngrids
              IF (nFfiles(ng).le.0) THEN
                IF (Master) WRITE (out,260) 'NFFILES',                  &
     &                            'Must be equal or greater than one.'
                exit_flag=4
                RETURN
              END IF
            END DO                
            Npts=MAXVAL(nFfiles)           
            allocate ( FRCids (Npts,Ngrids) )
            allocate ( FRCname(Npts,Ngrids) )
            FRCids(1:Npts,1:Ngrids)=-1
            DO ng=1,Ngrids
              DO k=1,Npts
                DO i=1,LEN(FRCname(k,ng))
                  FRCname(k,ng)(i:i)=blank
                END DO
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'FRCNAME') THEN
            DO ng=1,Ngrids
              DO i=1,nFfiles(ng)
                IF (Nval.eq.(i+(Ngrids-1)*nFfiles(ng))) THEN
                  FRCname(i,ng)=TRIM(ADJUSTL(Cval(Nval)))
                END IF
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'CLMNAME') THEN
            DO i=1,LEN(CLMname(Nval))
              CLMname(Nval)(i:i)=blank
            END DO
            CLMname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'BRYNAME') THEN
            DO i=1,LEN(BRYname(Nval))
              BRYname(Nval)(i:i)=blank
            END DO
            BRYname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'FWDNAME') THEN
            DO i=1,LEN(FWDname(Nval))
              FWDname(Nval)(i:i)=blank
              FWDbase(Nval)(i:i)=blank
            END DO
            FWDname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
            FWDbase(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'ADSNAME') THEN
            DO i=1,LEN(ADSname(Nval))
              ADSname(Nval)(i:i)=blank
            END DO
            ADSname(Nval)=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'APARNAM') THEN
            DO i=1,LEN(aparnam)
              aparnam(i:i)=blank
            END DO
            aparnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'SPOSNAM') THEN
            DO i=1,LEN(sposnam)
              sposnam(i:i)=blank
            END DO
            sposnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'FPOSNAM') THEN
            DO i=1,LEN(fposnam)
              fposnam(i:i)=blank
            END DO
            fposnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'BPARNAM') THEN
            DO i=1,LEN(bparnam)
              bparnam(i:i)=blank
            END DO
            bparnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'IPARNAM') THEN
            DO i=1,LEN(iparnam)
              iparnam(i:i)=blank
            END DO
            iparnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'SPARNAM') THEN
            DO i=1,LEN(sparnam)
              sparnam(i:i)=blank
            END DO
            sparnam=TRIM(ADJUSTL(Cval(Nval)))
          ELSE IF (TRIM(KeyWord).eq.'USRNAME') THEN
            DO i=1,LEN(USRname)
              USRname(i:i)=blank
            END DO
            USRname=TRIM(ADJUSTL(Cval(Nval)))
          END IF
        END IF
      END DO
  10  IF (Master) WRITE (out,50) line
      exit_flag=4
      RETURN
  20  CLOSE (inp)
!
!  Set switch to create NetCDF file.
!
      DO ng=1,Ngrids
        DO i=1,NV
          IF (Hout(i,ng)) LdefHIS(ng)=.TRUE.
        END DO
        IF (((nrrec(ng).eq.0).and.(nAVG(ng).gt.ntimes(ng))).or.         &
     &      (nAVG(ng).eq.0)) THEN
          LdefAVG(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nAVG2(ng).gt.ntimes(ng))).or.        &
     &      (nAVG2(ng).eq.0)) THEN
          LdefAVG2(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nDIA(ng).gt.ntimes(ng))).or.         &
     &      (nDIA(ng).eq.0)) THEN
          LdefDIA(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nFLT(ng).gt.ntimes(ng))).or.         &
     &      (nFLT(ng).eq.0)) THEN
          LdefFLT(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nFISHES(ng).gt.ntimes(ng))).or.      &
     &      (nFISHES(ng).eq.0)) THEN
          LdefFISH(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nHIS(ng).gt.ntimes(ng))).or.         &
     &      (nHIS(ng).eq.0)) THEN
          LdefHIS(ng)=.FALSE.
        END IF
        IF (((nrrec(ng).eq.0).and.(nRST(ng).gt.ntimes(ng))).or.         &
     &      (nRST(ng).eq.0)) THEN
          LdefRST(ng)=.FALSE.
        END  IF
        IF (((nrrec(ng).eq.0).and.(nSTA(ng).gt.ntimes(ng))).or.         &
     &      (nSTA(ng).eq.0)) THEN
          LdefSTA(ng)=.FALSE.
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Report input parameters.
!-----------------------------------------------------------------------
!
      IF (Lwrite) THEN
        WRITE (out,60) TRIM(title), TRIM(my_os), TRIM(my_cpu),          &
     &                 TRIM(my_fort), TRIM(my_fc), TRIM(my_fflags),     &
     &                 TRIM(Iname), TRIM(svn_url), TRIM(svn_rev),       &
     &                 TRIM(Rdir), TRIM(Hdir), TRIM(Hfile), TRIM(Adir)
        DO ng=1,Ngrids
!
!  Report grid size and domain decomposition.  Check for correct tile
!  decomposition.
!
          WRITE (out,70) ng, Lm(ng), Mm(ng), N(ng), numnodes,           &
     &                   NtileI(ng), NtileJ(ng)
          IF ((NtileI(ng)*NtileJ(ng)).ne.numnodes) THEN
            WRITE (out,80) ng
            exit_flag=6
            RETURN
          END IF
!
!  Report physical parameters.
!
          WRITE (out,110) ng
          WRITE (out,120) ntimes(ng), 'ntimes',                         &
     &          'Number of timesteps for 3-D equations.'
          WRITE (out,140) dt(ng), 'dt',                                 &
     &          'Timestep size (s) for 3-D equations.'
          WRITE (out,130) ndtfast(ng), 'ndtfast',                       &
     &          'Number of timesteps for 2-D equations between',        &
     &          'each 3D timestep.'
          WRITE (out,120) ERstr, 'ERstr',                               &
     &          'Starting ensemble/perturbation run number.'
          WRITE (out,120) ERend, 'ERend',                               &
     &          'Ending ensemble/perturbation run number.'
          WRITE (out,120) nrrec(ng), 'nrrec',                           &
     &          'Number of restart records to read from disk.'
          WRITE (out,170) LcycleRST(ng), 'LcycleRST',                   &
     &          'Switch to recycle time-records in restart file.'
          WRITE (out,130) nRST(ng), 'nRST',                             &
     &          'Number of timesteps between the writing of data',      &
     &          'into restart fields.'
          WRITE (out,130) ninfo(ng), 'ninfo',                           &
     &          'Number of timesteps between print of information',     &
     &          'to standard output.'
          WRITE (out,130) nSTA(ng), 'nSTA',                             &
     &          'Number of timesteps between the writing of data',      &
     &          'the stations file.'
          WRITE (out,170) ldefout(ng), 'ldefout',                       &
     &          'Switch to create a new output NetCDF file(s).'
          WRITE (out,130) nHIS(ng), 'nHIS',                             &
     &          'Number of timesteps between the writing fields',       &
     &          'into history file.'
          IF (ndefHIS(ng).gt.0) THEN
            WRITE (out,130) ndefHIS(ng), 'ndefHIS',                     &
     &            'Number of timesteps between creation of new',        &
     &            'history files.'
          END IF
          WRITE (out,130) ntsAVG(ng), 'ntsAVG',                         &
     &          'Starting timestep for the accumulation of output',     &
     &          'time-averaged data.'
          WRITE (out,130) nAVG(ng), 'nAVG',                             &
     &          'Number of timesteps between the writing of',           &
     &          'time-averaged data into averages file.'
          IF (ndefAVG(ng).gt.0) THEN
            WRITE (out,130) ndefAVG(ng), 'ndefAVG',                     &
     &            'Number of timesteps between creation of new',        &
     &            'time-averaged file.'
          END IF
          WRITE (out,130) ntsAVG2(ng), 'ntsAVG2',                       &
     &          'Starting timestep for the accumulation of output',     &
     &          'secondary time-averaged data.'
          WRITE (out,130) nAVG2(ng), 'nAVG2',                           &
     &          'Number of timesteps between the writing of',           &
     &          'secondary time-averaged data into averages file.'
          IF (ndefAVG2(ng).gt.0) THEN
            WRITE (out,130) ndefAVG2(ng), 'ndefAVG2',                   &
     &            'Number of timesteps between creation of new',        &
     &            'time-averaged file.'
          END IF
          DO itrc=1,NAT+NPT
            WRITE (out,190) tnu2(itrc,ng), 'tnu2', itrc,                &
     &            'Horizontal, harmonic mixing coefficient (m2/s)',     &
     &            'for tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
          END DO
          WRITE (out,210) visc2(ng), 'visc2',                           &
     &          'Horizontal, harmonic mixing coefficient (m2/s)',       &
     &          'for momentum.'
          DO itrc=1,NAT+NPT
            WRITE (out,190) Akt_bak(itrc,ng), 'Akt_bak', itrc,          &
     &            'Background vertical mixing coefficient (m2/s)',      &
     &            'for tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
          END DO
          WRITE (out,210) Akv_bak(ng), 'Akv_bak',                       &
     &          'Background vertical mixing coefficient (m2/s)',        &
     &          'for momentum.'
          WRITE (out,200) rdrg(ng), 'rdrg',                             &
     &          'Linear bottom drag coefficient (m/s).'
          WRITE (out,200) rdrg2(ng), 'rdrg2',                           &
     &          'Quadratic bottom drag coefficient.'
          WRITE (out,200) Zob(ng), 'Zob',                               &
     &          'Bottom roughness (m).'
          WRITE (out,200) blk_ZQ(ng), 'blk_ZQ',                         &
     &          'Height (m) of surface air humidity measurement.'
          IF (blk_ZQ(ng).le.0.0_r8) THEN
            WRITE (out,260) 'blk_ZQ.',                                  &
     &            'It must be greater than zero.'
            exit_flag=5
            RETURN
          END IF
          WRITE (out,200) blk_ZT(ng), 'blk_ZT',                         &
     &          'Height (m) of surface air temperature measurement.'
          IF (blk_ZT(ng).le.0.0_r8) THEN
            WRITE (out,260) 'blk_ZT.',                                  &
     &            'It must be greater than zero.'
            exit_flag=5
            RETURN
          END IF
          WRITE (out,200) blk_ZW(ng), 'blk_ZW',                         &
     &          'Height (m) of surface winds measurement.'
          IF (blk_ZW(ng).le.0.0_r8) THEN
            WRITE (out,260) 'blk_ZW.',                                  &
     &            'It must be greater than zero.'
            exit_flag=5
            RETURN
          END IF
          WRITE (out,120) lmd_Jwt(ng), 'lmd_Jwt',                       &
     &          'Jerlov water type.'
          IF ((lmd_Jwt(ng).lt.1).or.(lmd_Jwt(ng).gt.5)) THEN
            WRITE (out,260) 'lmd_Jwt.',                                 &
     &            'It must between one and five.'
            exit_flag=5
            RETURN
          END IF
          WRITE (out,120) Vtransform(ng), 'Vtransform',                 &
     &          'S-coordinate transformation equation.'
          WRITE (out,120) Vstretching(ng), 'Vstretching',               &
     &          'S-coordinate stretching function.'
          WRITE (out,200) theta_s(ng), 'theta_s',                       &
     &          'S-coordinate surface control parameter.'
          WRITE (out,200) theta_b(ng), 'theta_b',                       &
     &          'S-coordinate bottom  control parameter.'
          WRITE (out,160) Tcline(ng), 'Tcline',                         &
     &          'S-coordinate surface/bottom layer width (m) used',     &
     &          'in vertical coordinate stretching.'
          WRITE (out,140) rho0, 'rho0',                                 &
     &          'Mean density (kg/m3) for Boussinesq approximation.'
          WRITE (out,140) dstart, 'dstart',                             &
     &          'Time-stamp assigned to model initialization (days).'
          WRITE (out,140) tide_start, 'tide_start',                     &
     &          'Reference time origin for tidal forcing (days).'
          WRITE (out,150) time_ref, 'time_ref',                         &
     &          'Reference time for units attribute (yyyymmdd.dd)'
          DO itrc=1,NAT+NPT
            WRITE (out,190) Tnudg(itrc,ng), 'Tnudg', itrc,              &
     &            'Nudging/relaxation time scale (days)',               &
     &            'for tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
          END DO
          WRITE (out,210) Znudg(ng), 'Znudg',                           &
     &          'Nudging/relaxation time scale (days)',                 &
     &          'for free-surface.'
          WRITE (out,210) M2nudg(ng), 'M2nudg',                         &
     &          'Nudging/relaxation time scale (days)',                 &
     &          'for 2D momentum.'
          WRITE (out,210) M3nudg(ng), 'M3nudg',                         &
     &          'Nudging/relaxation time scale (days)',                 &
     &          'for 3D momentum.'
          WRITE (out,210) obcfac(ng), 'obcfac',                         &
     &          'Factor between passive and active',                    &
     &          'open boundary conditions.'
          WRITE (out,140) T0(ng), 'T0',                                 &
     &          'Background potential temperature (C) constant.'
          WRITE (out,140) S0(ng), 'S0',                                 &
     &          'Background salinity (PSU) constant.'
          WRITE (out,160) gamma2(ng), 'gamma2',                         &
     &          'Slipperiness variable: free-slip (1.0) or ',           &
     &          '                     no-slip (-1.0).'
          IF (Hout(idFsur,ng)) WRITE (out,170) Hout(idFsur,ng),         &
     &       'Hout(idFsur)',                                            &
     &       'Write out free-surface.'
          IF (Hout(idUbar,ng)) WRITE (out,170) Hout(idUbar,ng),         &
     &       'Hout(idUbar)',                                            &
     &       'Write out 2D U-momentum component.'
          IF (Hout(idVbar,ng)) WRITE (out,170) Hout(idVbar,ng),         &
     &       'Hout(idVbar)',                                            &
     &       'Write out 2D V-momentum component.'
          IF (Hout(idUvel,ng)) WRITE (out,170) Hout(idUvel,ng),         &
     &       'Hout(idUvel)',                                            &
     &       'Write out 3D U-momentum component.'
          IF (Hout(idVvel,ng)) WRITE (out,170) Hout(idVvel,ng),         &
     &       'Hout(idVvel)',                                            &
     &       'Write out 3D V-momentum component.'
          IF (Hout(idWvel,ng)) WRITE (out,170) Hout(idWvel,ng),         &
     &       'Hout(idWvel)',                                            &
     &       'Write out W-momentum component.'
          IF (Hout(idOvel,ng)) WRITE (out,170) Hout(idOvel,ng),         &
     &       'Hout(idOvel)',                                            &
     &       'Write out omega vertical velocity.'
          DO itrc=1,NAT
            IF (Hout(idTvar(itrc),ng)) WRITE (out,180)                  &
     &          Hout(idTvar(itrc),ng), 'Hout(idTvar)',                  &
     &          'Write out tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
          END DO
          IF (Hout(idUice,ng)) WRITE (out,170) Hout(idUice,ng),         &
     &       'Hout(idUice)',                                            &
     &       'Write out U-component ice velocity.'
          IF (Hout(idVice,ng)) WRITE (out,170) Hout(idVice,ng),         &
     &       'Hout(idVice)',                                            &
     &       'Write out V-component ice velocity.'
          IF (Hout(idAice,ng)) WRITE (out,170) Hout(idAice,ng),         &
     &       'Hout(idAice)',                                            &
     &       'Write out ice concentration.'
          IF (Hout(idHice,ng)) WRITE (out,170) Hout(idHice,ng),         &
     &       'Hout(idHice)',                                            &
     &       'Write out average ice thickness.'
          IF (Hout(idHsno,ng)) WRITE (out,170) Hout(idHsno,ng),         &
     &       'Hout(idHsno)',                                            &
     &       'Write out snow thickness.'
          IF (Hout(idTice,ng)) WRITE (out,170) Hout(idTice,ng),         &
     &       'Hout(idTice)',                                            &
     &       'Write out ice/snow surface temperature.'
          IF (Hout(idTimid,ng)) WRITE (out,170) Hout(idTimid,ng),       &
     &       'Hout(idTimid)',                                           &
     &       'Write out interior ice temperature.'
          IF (Hout(idSfwat,ng)) WRITE (out,170) Hout(idSfwat,ng),       &
     &       'Hout(idSfwat)',                                           &
     &       'Write out surface water (on ice) thickness.'
          IF (Hout(idAgeice,ng)) WRITE (out,170) Hout(idAgeice,ng),     &
     &       'Hout(idAgeice)',                                          &
     &       'Write out ice age.'
          IF (Hout(idIomflx,ng)) WRITE (out,170) Hout(idIomflx,ng),     &
     &       'Hout(idIomflx)',                                          &
     &       'Write out ice-ocean mass flux'
          IF (Hout(idSig11,ng)) WRITE (out,170) Hout(idSig11,ng),       &
     &       'Hout(idSig11)',                                           &
     &       'Write out internal ice stress component 11.'
          IF (Hout(idSig12,ng)) WRITE (out,170) Hout(idSig12,ng),       &
     &       'Hout(idSig12)',                                           &
     &       'Write out internal ice stress component 12.'
          IF (Hout(idSig22,ng)) WRITE (out,170) Hout(idSig22,ng),       &
     &       'Hout(idSig22)',                                           &
     &       'Write out internal ice stress component 22.'
          IF (Hout(idTauiw,ng)) WRITE (out,170) Hout(idTauiw,ng),       &
     &       'Hout(idTauiw)',                                           &
     &       'Write out ice-water friction velocity.'
          IF (Hout(idChuiw,ng)) WRITE (out,170) Hout(idChuiw,ng),       &
     &       'Hout(idChuiw)',                                           &
     &       'Write out ice-water momentum transfer coefficient.'
          IF (Hout(idT0mk,ng)) WRITE (out,170) Hout(idT0mk,ng),         &
     &       'Hout(idT0mk)',                                            &
     &       'Write out temperature of molecular sublayer under ice.'
          IF (Hout(idS0mk,ng)) WRITE (out,170) Hout(idS0mk,ng),         &
     &       'Hout(idS0mk)',                                            &
     &       'Write out salinity of molecular sublayer under ice.'
          IF (Hout(idWfr,ng)) WRITE (out,170) Hout(idWfr,ng),           &
     &       'Hout(idWfr)',                                             &
     &       'Write out frazil ice growth rate.'
          IF (Hout(idWai,ng)) WRITE (out,170) Hout(idWai,ng),           &
     &       'Hout(idWai)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Hout(idWao,ng)) WRITE (out,170) Hout(idWao,ng),           &
     &       'Hout(idWao)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Hout(idWio,ng)) WRITE (out,170) Hout(idWio,ng),           &
     &       'Hout(idWio)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Hout(idWro,ng)) WRITE (out,170) Hout(idWro,ng),           &
     &       'Hout(idWro)',                                             &
     &       'Write out ice melt runoff rate.'
          IF (Hout(idUair,ng)) WRITE (out,170) Hout(idUair,ng),         &
     &       'Hout(idUair)',                                            &
     &       'Write out surface U-winds.'
          IF (Hout(idVair,ng)) WRITE (out,170) Hout(idVair,ng),         &
     &       'Hout(idVair)',                                            &
     &       'Write out surface V-winds.'
          IF (Hout(idUsms,ng)) WRITE (out,170) Hout(idUsms,ng),         &
     &       'Hout(idUsms)',                                            &
     &       'Write out surface U-momentum stress.'
          IF (Hout(idVsms,ng)) WRITE (out,170) Hout(idVsms,ng),         &
     &       'Hout(idVsms)',                                            &
     &       'Write out surface V-momentum stress.'
          IF (Hout(idUbms,ng)) WRITE (out,170) Hout(idUbms,ng),         &
     &       'Hout(idUbms)',                                            &
     &       'Write out bottom U-momentum stress.'
          IF (Hout(idVbms,ng)) WRITE (out,170) Hout(idVbms,ng),         &
     &       'Hout(idVbms)',                                            &
     &       'Write out bottom V-momentum stress.'
          IF (Hout(idWamp,ng)) WRITE (out,170) Hout(idWamp,ng),         &
     &       'Hout(idWamp)',                                            &
     &       'Write out wave height.'
          IF (Hout(idWlen,ng)) WRITE (out,170) Hout(idWlen,ng),         &
     &       'Hout(idWlen)',                                            &
     &       'Write out wave length.'
          IF (Hout(idWdir,ng)) WRITE (out,170) Hout(idWdir,ng),         &
     &       'Hout(idWdir)',                                            &
     &       'Write out wave direction.'
          IF (Hout(idTsur(itemp),ng)) WRITE (out,170)                   &
     &        Hout(idTsur(itemp),ng), 'Hout(idTsur)',                   &
     &       'Write out surface net heat flux.'
          IF (Hout(idTsur(isalt),ng)) WRITE (out,170)                   &
     &        Hout(idTsur(isalt),ng), 'Hout(idTsur)',                   &
     &       'Write out surface net salt flux.'
          IF (Hout(idSrad,ng)) WRITE (out,170) Hout(idSrad,ng),         &
     &       'Hout(idSrad)',                                            &
     &       'Write out shortwave radiation flux.'
          IF (Hout(idLrad,ng)) WRITE (out,170) Hout(idLrad,ng),         &
     &       'Hout(idLrad)',                                            &
     &       'Write out longwave radiation flux.'
          IF (Hout(idLhea,ng)) WRITE (out,170) Hout(idLhea,ng),         &
     &       'Hout(idLhea)',                                            &
     &       'Write out latent heat flux.'
          IF (Hout(idShea,ng)) WRITE (out,170) Hout(idShea,ng),         &
     &       'Hout(idShea)',                                            &
     &       'Write out sensible heat flux.'
          IF (Hout(idevap,ng)) WRITE (out,170) Hout(idevap,ng),         &
     &       'Hout(idevap)',                                            &
     &       'Write out evaporation rate.'
          IF (Hout(idrain,ng)) WRITE (out,170) Hout(idrain,ng),         &
     &       'Hout(idrain)',                                            &
     &       'Write out rain rate.'
          IF (Hout(idDano,ng)) WRITE (out,170) Hout(idDano,ng),         &
     &       'Hout(idDano)',                                            &
     &       'Write out density anomaly.'
          IF (Hout(idVvis,ng)) WRITE (out,170) Hout(idVvis,ng),         &
     &       'Hout(idVvis)',                                            &
     &       'Write out vertical viscosity coefficient.'
          IF (Hout(idTdif,ng)) WRITE (out,170) Hout(idTdif,ng),         &
     &       'Hout(idTdif)',                                            &
     &       'Write out vertical T-diffusion coefficient.'
          IF (Hout(idSdif,ng)) WRITE (out,170) Hout(idSdif,ng),         &
     &       'Hout(idSdif)',                                            &
     &       'Write out vertical S-diffusion coefficient.'
          IF (Hout(idHsbl,ng)) WRITE (out,170) Hout(idHsbl,ng),         &
     &       'Hout(idHsbl)',                                            &
     &       'Write out depth of surface boundary layer.'
          IF (Hout2(idFsur,ng)) WRITE (out,170) Hout2(idFsur,ng),       &
     &       'Hout2(idFsur)',                                           &
     &       'Write out free-surface.'
          IF (Hout2(idUbar,ng)) WRITE (out,170) Hout2(idUbar,ng),       &
     &       'Hout2(idUbar)',                                           &
     &       'Write out 2D U-momentum component.'
          IF (Hout2(idVbar,ng)) WRITE (out,170) Hout2(idVbar,ng),       &
     &       'Hout2(idVbar)',                                           &
     &       'Write out 2D V-momentum component.'
          IF (Hout2(idUvel,ng)) WRITE (out,170) Hout2(idUvel,ng),       &
     &       'Hout2(idUvel)',                                           &
     &       'Write out 3D U-momentum component.'
          IF (Hout2(idVvel,ng)) WRITE (out,170) Hout2(idVvel,ng),       &
     &       'Hout2(idVvel)',                                           &
     &       'Write out 3D V-momentum component.'
          IF (Hout2(idWvel,ng)) WRITE (out,170) Hout2(idWvel,ng),       &
     &       'Hout2(idWvel)',                                           &
     &       'Write out W-momentum component.'
          IF (Hout2(idOvel,ng)) WRITE (out,170) Hout2(idOvel,ng),       &
     &       'Hout2(idOvel)',                                           &
     &       'Write out omega vertical velocity.'
          DO itrc=1,NAT
            IF (Hout2(idTvar(itrc),ng)) WRITE (out,180)                 &
     &          Hout2(idTvar(itrc),ng), 'Hout2(idTvar)',                &
     &          'Write out tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
          END DO
          IF (Hout2(idUice,ng)) WRITE (out,170) Hout2(idUice,ng),       &
     &       'Hout2(idUice)',                                           &
     &       'Write out U-component ice velocity.'
          IF (Hout2(idVice,ng)) WRITE (out,170) Hout2(idVice,ng),       &
     &       'Hout2(idVice)',                                           &
     &       'Write out V-component ice velocity.'
          IF (Hout2(idAice,ng)) WRITE (out,170) Hout2(idAice,ng),       &
     &       'Hout2(idAice)',                                           &
     &       'Write out ice concentration.'
          IF (Hout2(idHice,ng)) WRITE (out,170) Hout2(idHice,ng),       &
     &       'Hout2(idHice)',                                           &
     &       'Write out average ice thickness.'
          IF (Hout2(idHsno,ng)) WRITE (out,170) Hout2(idHsno,ng),       &
     &       'Hout2(idHsno)',                                           &
     &       'Write out snow thickness.'
          IF (Hout2(idTice,ng)) WRITE (out,170) Hout2(idTice,ng),       &
     &       'Hout2(idTice)',                                           &
     &       'Write out ice/snow surface temperature.'
          IF (Hout2(idTimid,ng)) WRITE (out,170) Hout2(idTimid,ng),     &
     &       'Hout2(idTimid)',                                          &
     &       'Write out interior ice temperature.'
          IF (Hout2(idSfwat,ng)) WRITE (out,170) Hout2(idSfwat,ng),     &
     &       'Hout2(idSfwat)',                                          &
     &       'Write out surface water (on ice) thickness.'
          IF (Hout2(idAgeice,ng)) WRITE (out,170) Hout2(idAgeice,ng),   &
     &       'Hout2(idAgeice)',                                         &
     &       'Write out ice age.'
          IF (Hout2(idIomflx,ng)) WRITE (out,170) Hout2(idIomflx,ng),   &
     &       'Hout2(idIomflx)',                                         &
     &       'Write out ice-ocean mass flux'
          IF (Hout2(idSig11,ng)) WRITE (out,170) Hout2(idSig11,ng),     &
     &       'Hout2(idSig11)',                                          &
     &       'Write out internal ice stress component 11.'
          IF (Hout2(idSig12,ng)) WRITE (out,170) Hout2(idSig12,ng),     &
     &       'Hout2(idSig12)',                                          &
     &       'Write out internal ice stress component 12.'
          IF (Hout2(idSig22,ng)) WRITE (out,170) Hout2(idSig22,ng),     &
     &       'Hout2(idSig22)',                                          &
     &       'Write out internal ice stress component 22.'
          IF (Hout2(idTauiw,ng)) WRITE (out,170) Hout2(idTauiw,ng),     &
     &       'Hout2(idTauiw)',                                          &
     &       'Write out ice-water friction velocity.'
          IF (Hout2(idChuiw,ng)) WRITE (out,170) Hout2(idChuiw,ng),     &
     &       'Hout2(idChuiw)',                                          &
     &       'Write out ice-water momentum transfer coefficient.'
          IF (Hout2(idT0mk,ng)) WRITE (out,170) Hout2(idT0mk,ng),       &
     &       'Hout2(idT0mk)',                                           &
     &       'Write out temperature of molecular sublayer under ice.'
          IF (Hout2(idS0mk,ng)) WRITE (out,170) Hout2(idS0mk,ng),       &
     &       'Hout2(idS0mk)',                                           &
     &       'Write out salinity of molecular sublayer under ice.'
          IF (Hout2(idUair,ng)) WRITE (out,170) Hout2(idUair,ng),       &
     &       'Hout2(idUair)',                                           &
     &       'Write out surface U-winds.'
          IF (Hout2(idVair,ng)) WRITE (out,170) Hout2(idVair,ng),       &
     &       'Hout2(idVair)',                                           &
     &       'Write out surface V-winds.'
          IF (Hout2(idUsms,ng)) WRITE (out,170) Hout2(idUsms,ng),       &
     &       'Hout2(idUsms)',                                           &
     &       'Write out surface U-momentum stress.'
          IF (Hout2(idVsms,ng)) WRITE (out,170) Hout2(idVsms,ng),       &
     &       'Hout2(idVsms)',                                           &
     &       'Write out surface V-momentum stress.'
          IF (Hout2(idUbms,ng)) WRITE (out,170) Hout2(idUbms,ng),       &
     &       'Hout2(idUbms)',                                           &
     &       'Write out bottom U-momentum stress.'
          IF (Hout2(idVbms,ng)) WRITE (out,170) Hout2(idVbms,ng),       &
     &       'Hout2(idVbms)',                                           &
     &       'Write out bottom V-momentum stress.'
          IF (Hout2(idWamp,ng)) WRITE (out,170) Hout2(idWamp,ng),       &
     &       'Hout2(idWamp)',                                           &
     &       'Write out wave height.'
          IF (Hout2(idWlen,ng)) WRITE (out,170) Hout2(idWlen,ng),       &
     &       'Hout2(idWlen)',                                           &
     &       'Write out wave length.'
          IF (Hout2(idWdir,ng)) WRITE (out,170) Hout2(idWdir,ng),       &
     &       'Hout2(idWdir)',                                           &
     &       'Write out wave direction.'
          IF (Hout2(idTsur(itemp),ng)) WRITE (out,170)                  &
     &        Hout2(idTsur(itemp),ng), 'Hout2(idTsur)',                 &
     &       'Write out surface net heat flux.'
          IF (Hout2(idTsur(isalt),ng)) WRITE (out,170)                  &
     &        Hout2(idTsur(isalt),ng), 'Hout2(idTsur)',                 &
     &       'Write out surface net salt flux.'
          IF (Hout2(idSrad,ng)) WRITE (out,170) Hout2(idSrad,ng),       &
     &       'Hout2(idSrad)',                                           &
     &       'Write out shortwave radiation flux.'
          IF (Hout2(idLrad,ng)) WRITE (out,170) Hout2(idLrad,ng),       &
     &       'Hout2(idLrad)',                                           &
     &       'Write out longwave radiation flux.'
          IF (Hout2(idLhea,ng)) WRITE (out,170) Hout2(idLhea,ng),       &
     &       'Hout2(idLhea)',                                           &
     &       'Write out latent heat flux.'
          IF (Hout2(idShea,ng)) WRITE (out,170) Hout2(idShea,ng),       &
     &       'Hout2(idShea)',                                           &
     &       'Write out sensible heat flux.'
          IF (Hout2(idevap,ng)) WRITE (out,170) Hout2(idevap,ng),       &
     &       'Hout2(idevap)',                                           &
     &       'Write out evaporation rate.'
          IF (Hout2(idrain,ng)) WRITE (out,170) Hout2(idrain,ng),       &
     &       'Hout2(idrain)',                                           &
     &       'Write out rain rate.'
          IF (Hout2(idDano,ng)) WRITE (out,170) Hout2(idDano,ng),       &
     &       'Hout2(idDano)',                                           &
     &       'Write out density anomaly.'
          IF (Hout2(idVvis,ng)) WRITE (out,170) Hout2(idVvis,ng),       &
     &       'Hout2(idVvis)',                                           &
     &       'Write out vertical viscosity coefficient.'
          IF (Hout2(idTdif,ng)) WRITE (out,170) Hout2(idTdif,ng),       &
     &       'Hout2(idTdif)',                                           &
     &       'Write out vertical T-diffusion coefficient.'
          IF (Hout2(idSdif,ng)) WRITE (out,170) Hout2(idSdif,ng),       &
     &       'Hout2(idSdif)',                                           &
     &       'Write out vertical S-diffusion coefficient.'
          IF (Hout2(idHsbl,ng)) WRITE (out,170) Hout2(idHsbl,ng),       &
     &       'Hout2(idHsbl)',                                           &
     &       'Write out depth of surface boundary layer.'
!
!-----------------------------------------------------------------------
!  Report output/input files and check availability of input files.
!-----------------------------------------------------------------------
!
          WRITE (out,220)
          WRITE (out,230) '           Output Restart File:  ',          &
     &                    TRIM(RSTname(ng))
          IF (LdefHIS(ng)) THEN
            IF (ndefHIS(ng).eq.0) THEN
              WRITE (out,230) '           Output History File:  ',      &
     &                        TRIM(HISname(ng))
            ELSE
              Lstr=LEN_TRIM(HISname(ng))
              WRITE (out,230) '      Prefix for History Files:  ',      &
     &                        HISname(ng)(1:Lstr-3)
            END IF
          END IF
          IF (ndefAVG(ng).eq.0) THEN
            WRITE (out,230) '          Output Averages File:  ',        &
     &                      TRIM(AVGname(ng))
          ELSE
            Lstr=LEN_TRIM(AVGname(ng))
            WRITE (out,230) '     Prefix for Averages Files:  ',        &
     &                      AVGname(ng)(1:Lstr-3)
          END IF
          IF (ndefAVG2(ng).eq.0) THEN
            WRITE (out,230) '          Output Secondary File:  ',       &
     &                      TRIM(AVG2name(ng))
          ELSE
            Lstr=LEN_TRIM(AVG2name(ng))
            WRITE (out,230) '     Prefix for Secondary Files:  ',       &
     &                      AVG2name(ng)(1:Lstr-3)
          END IF
          WRITE (out,230) '          Output Stations File:  ',          &
     &                    TRIM(STAname(ng))
          fname=GRDname(ng)
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '               Input Grid File:  ',          &
     &                    TRIM(fname)
          fname=INIname(ng)
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '  Input Nonlinear Initial File:  ',          &
     &                    TRIM(fname)
          DO i=1,nFfiles(ng)
            fname=FRCname(i,ng)
            INQUIRE (FILE=TRIM(fname), EXIST=inhere)
            IF (.not.inhere) GO TO 30
            WRITE (out,310) '         Input Forcing File ', i, ':  ',   &
     &                      TRIM(fname)
          END DO
          fname=BRYname(ng)
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '           Input Boundary File:  ',          &
     &                    TRIM(fname)
          fname=sposnam
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '        Station positions File:  ',          &
     &                    TRIM(fname)
          fname=iparnam
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '       Ice Model Parameters File:  ',        &
     &                    TRIM(fname)
          fname=varname
          INQUIRE (FILE=TRIM(fname), EXIST=inhere)
          IF (.not.inhere) GO TO 30
          WRITE (out,230) '  IO Variable Information File:  ',          &
     &                    TRIM(fname)
          GO TO 40
  30      IF (Master) WRITE (out,270) TRIM(fname)
          exit_flag=4
          RETURN
  40      CONTINUE
        END DO
        IF (Nuser.gt.0) THEN
          WRITE (out,230) '        Input/Output USER File:  ',          &
     &                    TRIM(USRname)
        END IF
!
!-----------------------------------------------------------------------
!  Report generic USER parameters.
!-----------------------------------------------------------------------
!
        IF (Nuser.gt.0) THEN
          WRITE (out,240)
          DO i=1,Nuser
            WRITE (out,250) user(i), i, i
          END DO
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Rescale active tracer parameters
!-----------------------------------------------------------------------
!
      DO ng=1,Ngrids
        DO itrc=1,NAT+NPT
!
!  Take the square root of the biharmonic coefficients so it can
!  be applied to each harmonic operator.
!
          tnu4(itrc,ng)=SQRT(ABS(tnu4(itrc,ng)))
!
!  Compute inverse nudging coefficients (1/s) used in various tasks.
!
          IF (Tnudg(itrc,ng).gt.0.0_r8) THEN
            Tnudg(itrc,ng)=1.0_r8/(Tnudg(itrc,ng)*86400.0_r8)
          ELSE
            Tnudg(itrc,ng)=0.0_r8
          END IF
        END DO
      END DO
  50  FORMAT (/,' READ_PhyPar - Error while processing line: ',/,a)
  60  FORMAT (/,1x,a,/,                                                 &
     &        /,1x,'Operating system : ',a,                             &
     &        /,1x,'CPU/hardware     : ',a,                             &
     &        /,1x,'Compiler system  : ',a,                             &
     &        /,1x,'Compiler command : ',a,                             &
     &        /,1x,'Compiler flags   : ',a,/,                           &
     &        /,1x,'Input Script  : ',a,/,                              &
     &        /,1x,'SVN Root URL  : ',a,                                &
     &        /,1x,'SVN Revision  : ',a,/,                              &
     &        /,1x,'Local Root    : ',a,                                &
     &        /,1x,'Header Dir    : ',a,                                &
     &        /,1x,'Header file   : ',a,                                &
     &        /,1x,'Analytical Dir: ',a)
  70  FORMAT (/,' Resolution, Grid ',i2.2,': ',i4.4,'x',i4.4,'x',i3.3,  &
     &        ',',2x,'Parallel Nodes: ',i3,',',2x,'Tiling: ',i3.3,      &
     &        'x',i3.3)
  80  FORMAT (/,' ROMS/TOMS: Wrong choice of domain ',i2.2,1x,          &
     &        'partition or number of parallel threads.',               &
     &        /,12x,'NtileI * NtileJ  must be equal to the number of ', &
     &        'parallel nodes.',                                        &
     &        /,12x,'Change -np value to mpirun or',                    &
     &        /,12x,'change domain partition in input script.')
  90  FORMAT (/,' Resolution, Grid ',i2.2,': ',i4.4,'x',i4.4,'x',i3.3,  &
     &        ',',2x,'Parallel Threads: ',i2,',',2x,'Tiling: ',i3.3,    &
     &        'x',i3.3)
 100  FORMAT (/,' ROMS/TOMS: Wrong choice of domain ',i3.3,1x,          &
     &        'partition or number of parallel threads.',               &
     &        /,12x,'NtileI*NtileJ must be a positive multiple of the', &
     &        ' number of threads.',                                    &
     &        /,12x,'Change number of threads (environment variable) ', &
     &        'or',/,12x,'change domain partition in input script.')
 110  FORMAT (/,/,' Physical Parameters, Grid: ',i2.2,                  &
     &        /,  ' =============================',/)  
 120  FORMAT (1x,i10,2x,a,t30,a)
 130  FORMAT (1x,i10,2x,a,t30,a,/,t32,a)
 140  FORMAT (f11.3,2x,a,t30,a)
 150  FORMAT (f11.2,2x,a,t30,a)
 160  FORMAT (f11.3,2x,a,t30,a,/,t32,a)
 170  FORMAT (10x,l1,2x,a,t30,a)
 180  FORMAT (10x,l1,2x,a,t30,a,i2.2,':',1x,a)
 190  FORMAT (1p,e11.4,2x,a,'(',i2.2,')',t30,a,/,t32,a,i2.2,':',1x,a)
 195  FORMAT (1p,e11.4,2x,a,t30,a,i2.2,':',1x,a)
 200  FORMAT (1p,e11.4,2x,a,t30,a)
 210  FORMAT (1p,e11.4,2x,a,t30,a,/,t32,a)
 220  FORMAT (/,' Output/Input Files:',/)
 230  FORMAT (2x,a,a)
 240  FORMAT (/,' Generic User Parameters:',/)
 250  FORMAT (1p,e11.4,2x,'user(',i2.2,')',t30,                         &
     &        'User parameter ',i2.2,'.')
 260  FORMAT (/,' READ_PHYPAR - Invalid input parameter, ',a,/,15x,a)
 270  FORMAT (/,' READ_PHYPAR - could not find input file:  ',a)
 280  FORMAT (/,' READ_PHYPAR - variable info not yet loaded, ', a)
 290  FORMAT (/,' READ_PHYPAR - Invalid dimension parameter, ',a,i4,    &
     &        /,15x,a)
 300  FORMAT (/,' READ_PHYPAR - Invalid dimension parameter, ',a,'(',   &
     &        i2.2,')',/,15x,a)
 310  FORMAT (2x,a,i2.2,a,a)
      RETURN
      END SUBROUTINE read_PhyPar
      SUBROUTINE read_IcePar (inp, out, Lwrite)
!
!=======================================================================
!  Copyright (c) 2002 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine reads in ice model input parameters.                   !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_ncparam
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations
!
      logical, intent(in) :: Lwrite
      integer, intent(in) :: inp, out
!
!  Local variable declarations.
!
      integer :: Lstr, Lval, Npts, Nval, i, ng, itrc, status
      integer :: decode_line, lenstr, load_i, load_l, load_r
      real(r8), dimension(100) :: Rval
      character (len=40) :: KeyWord
      character (len=80) :: line
      character (len=80), dimension(100) :: Cval
!
!-----------------------------------------------------------------------
!  Read in ice model parameters.
!-----------------------------------------------------------------------
!
      DO WHILE (.true.)
        READ (inp,'(a)',ERR=10,END=20) line
        status=decode_line(line, KeyWord, Nval, Cval, Rval)
        IF (status.gt.0) THEN
          IF (TRIM(KeyWord).eq.'Lice') THEN
            Npts=load_l(Nval, Cval, Ngrids, Lice)
          ELSE IF (TRIM(KeyWord).eq.'DTICE') THEN
            Npts=load_r(Nval, Rval, Ngrids, dtice)
          ELSE IF (TRIM(KeyWord).eq.'DTICE_EQ') THEN
            Npts=load_r(Nval, Rval, Ngrids, dtice_eq)
          ELSE IF (TRIM(KeyWord).eq.'nstrs') THEN
            Npts=load_i(Nval, Rval, Ngrids, nstrs)
          ELSE IF (TRIM(KeyWord).eq.'nevp') THEN
            Npts=load_i(Nval, Rval, Ngrids, nevp)
          ELSE IF (TRIM(KeyWord).eq.'rhoice') THEN
            Npts=load_r(Nval, Rval, Ngrids, rhoice)
          ELSE IF (TRIM(KeyWord).eq.'cdiw') THEN
            Npts=load_r(Nval, Rval, Ngrids, cdiw)
          ELSE IF (TRIM(KeyWord).eq.'cdai') THEN
            Npts=load_r(Nval, Rval, Ngrids, cdai)
          ELSE IF (TRIM(KeyWord).eq.'rho_air') THEN
            Npts=load_r(Nval, Rval, Ngrids, rho_air)
          ELSE IF (TRIM(KeyWord).eq.'rhosnow_dry') THEN
            Npts=load_r(Nval, Rval, Ngrids, rhosnow_dry)
          ELSE IF (TRIM(KeyWord).eq.'rhosnow_wet') THEN
            Npts=load_r(Nval, Rval, Ngrids, rhosnow_wet)
          ELSE IF (TRIM(KeyWord).eq.'pstar') THEN
            Npts=load_r(Nval, Rval, Ngrids, pstar)
          ELSE IF (TRIM(KeyWord).eq.'astren') THEN
            Npts=load_r(Nval, Rval, Ngrids, astren)
          ELSE IF (TRIM(KeyWord).eq.'zetamax') THEN
            Npts=load_r(Nval, Rval, Ngrids, zetamax)
          ELSE IF (TRIM(KeyWord).eq.'zetamin') THEN
            Npts=load_r(Nval, Rval, Ngrids, zetamin)
          ELSE IF (TRIM(KeyWord).eq.'ellip_sq') THEN
            Npts=load_r(Nval, Rval, Ngrids, ellip_sq)
          ELSE IF (TRIM(KeyWord).eq.'alphai') THEN
            Npts=load_r(Nval, Rval, Ngrids, alphai)
            do ng=1,Ngrids
               alphai(ng) = alphai(ng)*deg2rad
            enddo
          ELSE IF (TRIM(KeyWord).eq.'tol') THEN
            Npts=load_r(Nval, Rval, 1, tol)
          ELSE IF (TRIM(KeyWord).eq.'min_h') THEN
            Npts=load_r(Nval, Rval, Ngrids, min_h)
          ELSE IF (TRIM(KeyWord).eq.'min_a') THEN
            Npts=load_r(Nval, Rval, Ngrids, min_a)
          ELSE IF (TRIM(KeyWord).eq.'max_a') THEN
            Npts=load_r(Nval, Rval, Ngrids, max_a)
          ELSE IF (TRIM(KeyWord).eq.'stressang') THEN
            Npts=load_r(Nval, Rval, Ngrids, stressang)
            do ng=1,Ngrids
               stressang(ng) = stressang(ng)*deg2rad
            enddo
          ELSE IF (TRIM(KeyWord).eq.'ice_emiss') THEN
            Npts=load_r(Nval, Rval, 1, ice_emiss)
          ELSE IF (TRIM(KeyWord).eq.'spec_heat_air') THEN
            Npts=load_r(Nval, Rval, 1, spec_heat_air)
          ELSE IF (TRIM(KeyWord).eq.'trans_coeff') THEN
            Npts=load_r(Nval, Rval, 1, trans_coeff)
          ELSE IF (TRIM(KeyWord).eq.'sublim_latent_heat') THEN
            Npts=load_r(Nval, Rval, 1, sublim_latent_heat)
          ELSE IF (TRIM(KeyWord).eq.'t0deg') THEN
            Npts=load_r(Nval, Rval, 1, t0deg)
          END IF
        END IF
      END DO
  10  IF (Master) WRITE (out,30) line
      exit_flag=5
      RETURN
  20  CLOSE (inp)
! Set ice time step to ocean time step
     DO ng = 1,Ngrids
        dtice(ng) = dt(ng)
     ENDDO
!
!-----------------------------------------------------------------------
!  Report input parameters.
!-----------------------------------------------------------------------
!
      IF (Lwrite) THEN
        DO ng=1,Ngrids
          IF (Lice(ng)) THEN
            WRITE (out,40) ng
            WRITE(out,*) 'Ice time step = ocean time step'
            WRITE (out,100) dtice(ng), 'DTICE',                         &
     &            'Ice model time step (s).'
            WRITE (out,100) dtice_eq(ng), 'DTICE_EQ',                   &
     &            'Ice drift update (equilibrium) time step (s).'
            WRITE (out,50) nstrs(ng), 'nstrs',                          &
     &            'Number of iterations for nonlinear ice dynamics.'
            WRITE (out,50) nevp(ng), 'nevp',                            &
     &            'Number of elastic steps per plastic step in EVP.'
            WRITE (out,100) rhoice(ng), 'rhoice',                       &
     &            'Density of sea ice (kg/m3).'
            WRITE (out,100) cdiw(ng), 'cdiw',                           &
     &            'Ice-water drag coefficient (nondimensional).'
            WRITE (out,100) cdai(ng), 'cdai',                           &
     &            'Air-ice drag coefficient (nondimensional).'
            WRITE (out,100) rho_air(ng), 'rho_air',                     &
     &            'Air density (kg/m3).'
            WRITE (out,100) rhosnow_dry(ng), 'rhosnow_dry',             &
     &            'Dry snow density (kg/m3).'
            WRITE (out,100) rhosnow_wet(ng), 'rhosnow_wet',             &
     &            'Wet snow density (kg/m3).'
            WRITE (out,100) alphai(ng)*rad2deg, 'alphai',               &
     &            'Mohr-Coulomb stress angle (degrees).'
            WRITE (out,100) min_h(ng), 'min_h',                         &
     &            'Minimum average ice thickness (m).'
            WRITE (out,100) min_a(ng), 'min_a',                         &
     &            'Minimum ice concentration (nondimensional).'
            WRITE (out,100) max_a(ng), 'max_a',                         &
     &            'Maximum ice concentration (nondimensional).'
            WRITE (out,100) stressang(ng)*rad2deg, 'stressang',         &
     &            'Turning angle for ice-water drag (degrees).'
            WRITE (out,100) tol, 'tol',                                 &
     &            'Numerical tolerance in rheology calculations .'
            WRITE (out,100) ice_emiss, 'ice_emiss',                     &
     &            'Ice emissivity.'
            WRITE (out,100) spec_heat_air, 'spec_heat_air',             &
     &            'Specific heat of air.'
            WRITE (out,100) trans_coeff, 'trans_coeff',                 &
     &            'Transfer coefficient.'
            WRITE (out,100) sublim_latent_heat, 'sublim_latent_heat',   &
     &            'Latent_heat of sublimation.'
            WRITE (out,100) t0deg, 't0deg',                             &
     &            'Zero degrees Celsius in degrees Kelvin.'
          END IF
        END DO
      END IF
  30  FORMAT (/,' READ_IcePar - Error while processing line: ',/,a)
  40  FORMAT (/,/,' Ice Parameters, Grid: ',i2.2,                       &
     &        /,' ============================',/)
  50  FORMAT (1x,i10,2x,a,t28,a)
  60  FORMAT (10x,l1,2x,a,t28,a,i2.2,':',1x,a)
  70  FORMAT (f11.3,2x,a,t28,a)
  80  FORMAT (f11.3,2x,a,t28,a,/,t30,a)
  90  FORMAT (1p,e11.4,2x,a,'(',i2.2,')',t28,a,/,t30,a,i2.2,':',1x,a)
 100  FORMAT (1p,e11.4,2x,a,t28,a)
 110  FORMAT (1p,e11.4,2x,a,t28,a,/,t30,a)
 120  FORMAT (10x,l1,2x,a,t28,a)
 280  FORMAT (/,' READ_IcePar - variable info not yet loaded, ', a)
      RETURN
      END SUBROUTINE read_IcePar
      SUBROUTINE read_StaPar (model, inp, out, Lwrite)
!
!=======================================================================
!                                                                      !
!  This routine reads in input station parameters.                     !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations
!
      logical, intent(in) :: Lwrite
      integer, intent(in) :: model, inp, out
!
!  Local variable declarations.
!
      integer :: Mstation, Npts, Nval
      integer :: flag, i, igrid, ista, itrc, ng, status
      integer :: decode_line, load_i, load_l, load_r
      real(r8) :: Xpos, Ypos
      logical, dimension(MT,Ngrids) :: Lsta
      integer, dimension(Ngrids) :: is
      real(r8), dimension(100) :: Rval
      character (len=40) :: KeyWord
      character (len=160) :: line
      character (len=160), dimension(100) :: Cval
!
!-----------------------------------------------------------------------
!  Read in stations parameters.
!-----------------------------------------------------------------------
!
      DO WHILE (.TRUE.)
        READ (inp,'(a)',ERR=20,END=30) line
        status=decode_line(line, KeyWord, Nval, Cval, Rval)
        IF (status.gt.0) THEN
          IF (TRIM(KeyWord).eq.'Lstations') THEN 
            Npts=load_l(Nval, Cval, Ngrids, Lstations)
          ELSE IF (TRIM(KeyWord).eq.'Sout(idFsur)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idFsur,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbar)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbar,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbar)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbar,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUvel,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVvel,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWvel,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idOvel)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idOvel,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTvar)') THEN
            Npts=load_l(Nval, Cval, MT*Ngrids, Lsta)
            DO ng=1,Ngrids
              DO itrc=1,NT(ng)
                Sout(idTvar(itrc),ng)=Lsta(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUair)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUair,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVair)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVair,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUsms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUsms,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVsms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVsms,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbms,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbms)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbms,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbrs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbrs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbrs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbrs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbws)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbws,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbws)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbws,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbcs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbcs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbcs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbcs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbot)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbot,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbot)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbot,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUbur)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUbur,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVbvr)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVbvr,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW2xx)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW2xx,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW2xy)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW2xy,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW2yy)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW2yy,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idU2rs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idU2rs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idV2rs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idV2rs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idU2Sd)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idU2Sd,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idV2Sd)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idV2Sd,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW3xx)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW3xx,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW3xy)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW3xy,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW3yy)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW3yy,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW3zx)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW3zx,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idW3zy)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idW3zy,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idU3rs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idU3rs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idV3rs)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idV3rs,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idU3Sd)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idU3Sd,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idV3Sd)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idV3Sd,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWamp)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWamp,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWlen)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWlen,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWdir)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWdir,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTsur)') THEN
            Npts=load_l(Nval, Cval, NAT*Ngrids, Lsta)
            DO ng=1,Ngrids
              DO itrc=1,NAT
                Sout(idTsur(itrc),ng)=Lsta(itrc,ng)
              END DO
            END DO
          ELSE IF (TRIM(KeyWord).eq.'Sout(idLhea)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idLhea,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idShea)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idShea,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idLrad)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idLrad,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSrad)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSrad,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idevap)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idevap,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idrain)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idrain,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idDano)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idDano,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVvis)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVvis,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTdif)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idTdif,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSdif)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSdif,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idHsbl)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idHsbl,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idHbbl)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idHbbl,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idMtke)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idMtke,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idMtls)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idMtls,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idUice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idUice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idVice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idVice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idAice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idAice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idHice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idHice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idTice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTimid)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idTimid,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSfwat)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSfwat,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idAgeice)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idAgeice,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idIomflx)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idIomflx,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSig11)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSig11,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSig12)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSig12,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idSig22)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idSig22,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWfr)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWfr,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idTauiw)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idTauiw,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idChuiw)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idChuiw,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idT0mk)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idT0mk,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idS0mk)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idS0mk,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWfr)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWfr,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWai)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWai,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWao)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWao,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWio)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWio,1))
          ELSE IF (TRIM(KeyWord).eq.'Sout(idWro)') THEN
            Npts=load_l(Nval, Cval, Ngrids, Sout(idWro,1))
          ELSE IF (TRIM(KeyWord).eq.'NSTATION') THEN
            Npts=load_i(Nval, Rval, Ngrids, Nstation)
          ELSE IF (TRIM(KeyWord).eq.'POS') THEN
            DO ng=1,Ngrids
              allocate ( SCALARS(ng) % Sflag(Nstation(ng)) )            
              allocate ( SCALARS(ng) % SposX(Nstation(ng)) )            
              allocate ( SCALARS(ng) % SposY(Nstation(ng)) )            
            END DO
            is(1:Ngrids)=0
            DO WHILE (.TRUE.)
              READ (inp,*,ERR=10,END=10) igrid, flag, Xpos, Ypos
              ng=MAX(1,ABS(igrid))
              is(ng)=is(ng)+1
              SCALARS(ng)%Sflag(is(ng))=flag
              SCALARS(ng)%SposX(is(ng))=Xpos
              SCALARS(ng)%SposY(is(ng))=Ypos
            END DO
 10         DO ng=1,Ngrids
              IF (Nstation(ng).ne.is(ng)) THEN
                IF (Master) WRITE (out,40) Nstation(ng), is(ng)
                exit_flag=4
                RETURN
              END IF
            END DO
          END IF
        END IF
      END DO
 20   IF (Master) WRITE (out,50) line
      exit_flag=4
      RETURN
 30   CONTINUE
!
!-----------------------------------------------------------------------
!  Report input parameters.
!-----------------------------------------------------------------------
!
      IF (Lwrite) THEN
        DO ng=1,Ngrids
          IF (Lstations(ng)) THEN
            WRITE (out,60) ng
            WRITE (out,70) Nstation(ng), 'Nstation',                    &
     &            'Number of stations to write out into stations file.'
            IF (Sout(idFsur,ng)) WRITE (out,80) Sout(idFsur,ng),        &
     &          'Sout(idFsur)',                                         &
     &          'Write out free-surface.'
            IF (Sout(idUbar,ng)) WRITE (out,80) Sout(idUbar,ng),        &
     &          'Sout(idUbar)',                                         &
     &          'Write out 2D U-momentum component.'
            IF (Sout(idVbar,ng)) WRITE (out,80) Sout(idVbar,ng),        &
     &          'Sout(idVbar)',                                         &
     &          'Write out 2D V-momentum component.'
            IF (Sout(idUvel,ng)) WRITE (out,80) Sout(idUvel,ng),        &
     &          'Sout(idUvel)',                                         &
     &          'Write out 3D U-momentum component.'
            IF (Sout(idVvel,ng)) WRITE (out,80) Sout(idVvel,ng),        &
     &          'Sout(idVvel)',                                         &
     &          'Write out 3D V-momentum component.'
            IF (Sout(idWvel,ng)) WRITE (out,80) Sout(idWvel,ng),        &
     &          'Sout(idWvel)',                                         &
     &          'Write out W-momentum component.'
            IF (Sout(idOvel,ng)) WRITE (out,80) Sout(idOvel,ng),        &
     &          'Sout(idOvel)',                                         &
     &          'Write out omega vertical velocity.'
            DO itrc=1,NT(ng)
              IF (Sout(idTvar(itrc),ng)) WRITE (out,90)                 &
     &            Sout(idTvar(itrc),ng), 'Sout(idTvar)',                &
     &            'Write out tracer ', itrc, TRIM(Vname(1,idTvar(itrc)))
            END DO
            IF (Sout(idUair,ng)) WRITE (out,80) Sout(idUair,ng),        &
     &          'Sout(idUair)',                                         &
     &          'Write out surface U-wind.'
            IF (Sout(idVair,ng)) WRITE (out,80) Sout(idVair,ng),        &
     &          'Sout(idVair)',                                         &
     &          'Write out surface V-wind.'
            IF (Sout(idUsms,ng)) WRITE (out,80) Sout(idUsms,ng),        &
     &          'Sout(idUsms)',                                         &
     &          'Write out surface U-momentum stress.'
            IF (Sout(idVsms,ng)) WRITE (out,80) Sout(idVsms,ng),        &
     &          'Sout(idVsms)',                                         &
     &          'Write out surface V-momentum stress.'
            IF (Sout(idUbms,ng)) WRITE (out,80) Sout(idUbms,ng),        &
     &          'Sout(idUbms)',                                         &
     &          'Write out bottom U-momentum stress.'
            IF (Sout(idVbms,ng)) WRITE (out,80) Sout(idVbms,ng),        &
     &          'Sout(idVbms)',                                         &
     &          'Write out bottom V-momentum stress.'
            IF (Sout(idWamp,ng)) WRITE (out,80) Sout(idWamp,ng),        &
     &         'Sout(idWamp)',                                          &
     &         'Write out wave height.'
            IF (Sout(idWlen,ng)) WRITE (out,80) Sout(idWlen,ng),        &
     &         'Sout(idWlen)',                                          &
     &         'Write out wave length.'
            IF (Sout(idWdir,ng)) WRITE (out,80) Sout(idWdir,ng),        &
     &         'Sout(idWdir)',                                          &
     &         'Write out wave direction.'
            IF (Sout(idTsur(itemp),ng)) WRITE (out,80)                  &
     &          Sout(idTsur(itemp),ng), 'Sout(idTsur)',                 &
     &          'Write out surface net heat flux.'
            IF (Sout(idTsur(isalt),ng)) WRITE (out,80)                  &
     &          Sout(idTsur(isalt),ng), 'Sout(idTsur)',                 &
     &          'Write out surface net salt flux.'
            IF (Sout(idSrad,ng)) WRITE (out,80) Sout(idSrad,ng),        &
     &          'Sout(idSrad)',                                         &
     &          'Write out shortwave radiation flux.'
            IF (Sout(idLrad,ng)) WRITE (out,80) Sout(idLrad,ng),        &
     &          'Sout(idLrad)',                                         &
     &          'Write out longwave radiation flux.'
            IF (Sout(idLhea,ng)) WRITE (out,80) Sout(idLhea,ng),        &
     &          'Sout(idLhea)',                                         &
     &          'Write out latent heat flux.'
            IF (Sout(idShea,ng)) WRITE (out,80) Sout(idShea,ng),        &
     &          'Sout(idShea)',                                         &
     &          'Write out sensible heat flux.'
            IF (Sout(idevap,ng)) WRITE (out,80) Sout(idevap,ng),        &
     &         'Sout(idevap)',                                          &
     &         'Write out evaporation rate.'
            IF (Sout(idrain,ng)) WRITE (out,80) Sout(idrain,ng),        &
     &         'Sout(idrain)',                                          &
     &         'Write out rain rate.'
            IF (Sout(idDano,ng)) WRITE (out,80) Sout(idDano,ng),        &
     &          'Sout(idDano)',                                         &
     &          'Write out density anomaly.'
            IF (Sout(idVvis,ng)) WRITE (out,80) Sout(idVvis,ng),        &
     &          'Sout(idVvis)',                                         &
     &          'Write out vertical viscosity coefficient.'
            IF (Sout(idTdif,ng)) WRITE (out,80) Sout(idTdif,ng),        &
     &          'Sout(idTdif)',                                         &
     &          'Write out vertical T-diffusion coefficient.'
            IF (Sout(idSdif,ng)) WRITE (out,80) Sout(idSdif,ng),        &
     &          'Sout(idSdif)',                                         &
     &          'Write out vertical S-diffusion coefficient.'
            IF (Sout(idHsbl,ng)) WRITE (out,80) Sout(idHsbl,ng),        &
     &          'Sout(idHsbl)',                                         &
     &          'Write out depth of surface boundary layer.'
          IF (Sout(idUice,ng)) WRITE (out,80) Sout(idUice,ng),          &
     &       'Sout(idUice)',                                            &
     &       'Write out U-component ice velocity.'
          IF (Sout(idVice,ng)) WRITE (out,80) Sout(idVice,ng),          &
     &       'Sout(idVice)',                                            &
     &       'Write out V-component ice velocity.'
          IF (Sout(idAice,ng)) WRITE (out,80) Sout(idAice,ng),          &
     &       'Sout(idAice)',                                            &
     &       'Write out ice concentration.'
          IF (Sout(idHice,ng)) WRITE (out,80) Sout(idHice,ng),          &
     &       'Sout(idHice)',                                            &
     &       'Write out average ice thickness.'
          IF (Sout(idHsno,ng)) WRITE (out,80) Sout(idHsno,ng),          &
     &       'Sout(idHsno)',                                            &
     &       'Write out snow thickness.'
          IF (Sout(idTice,ng)) WRITE (out,80) Sout(idTice,ng),          &
     &       'Sout(idTice)',                                            &
     &       'Write out ice/snow surface temperature.'
          IF (Sout(idTimid,ng)) WRITE (out,80) Sout(idTimid,ng),        &
     &       'Sout(idTimid)',                                           &
     &       'Write out interior ice temperature.'
          IF (Sout(idSfwat,ng)) WRITE (out,80) Sout(idSfwat,ng),        &
     &       'Sout(idSfwat)',                                           &
     &       'Write out surface water (on ice) thickness.'
          IF (Sout(idAgeice,ng)) WRITE (out,80) Sout(idAgeice,ng),      &
     &       'Sout(idAgeice)',                                          &
     &       'Write out surface water (on ice) thickness.'
          IF (Sout(idIomflx,ng)) WRITE (out,80) Sout(idIomflx,ng),      &
     &       'Sout(idIomflx)',                                          &
     &       'Write out ice-ocean mass flux.'
          IF (Sout(idSig11,ng)) WRITE (out,80) Sout(idSig11,ng),        &
     &       'Sout(idSig11)',                                           &
     &       'Write out internal ice stress component 11.'
          IF (Sout(idSig12,ng)) WRITE (out,80) Sout(idSig12,ng),        &
     &       'Sout(idSig12)',                                           &
     &       'Write out internal ice stress component 12.'
          IF (Sout(idSig22,ng)) WRITE (out,80) Sout(idSig22,ng),        &
     &       'Sout(idSig22)',                                           &
     &       'Write out internal ice stress component 22.'
          IF (Sout(idTauiw,ng)) WRITE (out,80) Sout(idTauiw,ng),        &
     &       'Sout(idTauiw)',                                           &
     &       'Write out ice-water friction velocity.'
          IF (Sout(idChuiw,ng)) WRITE (out,80) Sout(idChuiw,ng),        &
     &       'Hout(idChuiw)',                                           &
     &       'Write out ice-water momentum transfer coefficient.'
          IF (Sout(idT0mk,ng)) WRITE (out,80) Sout(idT0mk,ng),          &
     &       'Sout(idT0mk)',                                            &
     &       'Write out temperature of molecular sublayer under ice.'
          IF (Sout(idS0mk,ng)) WRITE (out,80) Sout(idS0mk,ng),          &
     &       'Sout(idS0mk)',                                            &
     &       'Write out salinity of molecular sublayer under ice.'
          IF (Sout(idWfr,ng)) WRITE (out,80) Sout(idWfr,ng),            &
     &       'Sout(idWfr)',                                             &
     &       'Write out frazil ice growth rate.'
          IF (Sout(idWai,ng)) WRITE (out,80) Sout(idWai,ng),            &
     &       'Sout(idWai)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Sout(idWao,ng)) WRITE (out,80) Sout(idWao,ng),            &
     &       'Sout(idWao)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Sout(idWio,ng)) WRITE (out,80) Sout(idWio,ng),            &
     &       'Sout(idWio)',                                             &
     &       'Write out ice growth/melt rate.'
          IF (Sout(idWro,ng)) WRITE (out,80) Sout(idWro,ng),            &
     &       'Sout(idWro)',                                             &
     &       'Write out ice melt runoff rate.'
            WRITE (out,*)
            DO i=1,Nstation(ng)
              WRITE (out,100) i, SCALARS(ng)%Sflag(i),                  &
     &                           SCALARS(ng)%SposX(i),                  &
     &                           SCALARS(ng)%SposY(i)
            END DO
          END IF
        END DO
      END IF
  40  FORMAT (/,' READ_StaPar - Inconsistent number of stations, ',     &
     &        'Nstation = ',2i8,/,15x,'change input script values.')  
  50  FORMAT (/,' READ_StaPar - Error while processing line: ',/,a)
  60  FORMAT (/,/,' Stations Parameters, Grid: ',i2.2,                  &
     &        /,  ' =============================',/)  
  70  FORMAT (1x,i10,2x,a,t30,a)
  80  FORMAT (10x,l1,2x,a,t30,a)
  90  FORMAT (10x,l1,2x,a,t30,a,i2.2,':',1x,a)
 100  FORMAT (13x,'Flag and positions for station ',i4.4,':',           &
     &        i3,1x,2f10.4)
 110  FORMAT (/,' READ_StaPAR - variable info not yet loaded, ', a)
      RETURN
      END SUBROUTINE read_StaPar
      FUNCTION decode_line (line_text, KeyWord, Nval, Cval, Rval)
!
!=======================================================================
!                                                                      !
!  This function decodes lines of text from input script files.        !
!                                                                      !
!=======================================================================
!
      USE mod_kinds
!
      implicit none
!
! Imported variable declarations.
!
      character (len=*), intent(in) :: line_text
      character (len=40), intent(inout) :: KeyWord
      integer, intent(inout) :: Nval
      character (len=160), dimension(100), intent(inout) :: Cval
      real(r8), dimension(100), intent(inout) :: Rval
!
! Local variable declarations
!
      logical :: IsString, Kextract, decode, nested
      integer :: Iblank, Icmm, Kstr, Kend, Linp
      integer :: Lend, LenS, Lstr, Lval, Nmul, Schar
      integer :: copies, i, ic, ie, is, j, status
      integer, dimension(20) :: Imul
      integer :: decode_line
      character (len=1 ), parameter :: blank = ' '
      character (len=160) :: Vstring, line, string
!
!------------------------------------------------------------------------
!  Decode input line.
!------------------------------------------------------------------------
!
!  Initialize.
!
      DO i=1,LEN(line)
        line(i:i)=blank
        Vstring(i:i)=blank
        string(i:i)=blank
      END DO
!
!  Get length of "line", remove leading and trailing blanks.
!
      line=TRIM(ADJUSTL(line_text))
      Linp=LEN_TRIM(line)
!
!  If not a blank or comment line [char(33)=!], decode and extract input
!  values.  Find equal sign [char(61)].
!
      status=-1
      nested=.FALSE.
      IF ((Linp.gt.0).and.(line(1:1).ne.CHAR(33))) THEN
        status=1
        Kstr=1
        Kend=INDEX(line,CHAR(61),BACK=.FALSE.)-1
        Lstr=INDEX(line,CHAR(61),BACK=.TRUE.)+1
!
! Determine if KEYWORD is followed by double equal sign (==) indicating
! nested parameter.
!
        IF ((Lstr-Kend).eq.3) nested=.TRUE.
!
! Extract KEYWORD, trim leading and trailing blanks.
!
        Kextract=.FALSE.
        IF (Kend.gt.0) THEN
          Lend=Linp
          KeyWord=line(Kstr:Kend)
          Nval=0
          Kextract=.TRUE.
        ELSE
          Lstr=1
          Lend=Linp
          Kextract=.TRUE.
        END IF
!
! Extract parameter values string.  Remove comments [char(33)=!] or
! continuation symbol [char(92)=\], if any.  Trim leading trailing
! blanks.
!
        IF (Kextract) THEN
          Icmm=INDEX(line,CHAR(33),BACK=.FALSE.)
          IF (Icmm.gt.0) Lend=Icmm-1
          Icmm=INDEX(line,CHAR(92),BACK=.FALSE.)
          IF (Icmm.gt.0) Lend=Icmm-1
          Vstring=ADJUSTL(line(Lstr:Lend))
          Lval=LEN_TRIM(Vstring)
!
! The TITLE KEYWORD is a special one since it can include strings,
! numbers, spaces, and continuation symbol.
!
          IsString=.FALSE.
          IF (TRIM(KeyWord).eq.'TITLE') THEN
            Nval=Nval+1
            Cval(Nval)=Vstring(1:Lval)
            IsString=.TRUE.
          ELSE
!
! Check if there is a multiplication symbol [char(42)=*] in the variable
! string indicating repetition of input values.
!
            Nmul=0
            DO i=1,Lval
              IF (Vstring(i:i).eq.CHAR(42)) THEN
                Nmul=Nmul+1           
                Imul(Nmul)=i
              END IF
            END DO
            ic=1
!
! Check for blank spaces [char(32)=' '] between entries and decode.
!
            is=1
            ie=Lval
            Iblank=0
            decode=.FALSE.
            DO i=1,Lval
              IF (Vstring(i:i).eq.CHAR(32)) THEN
                IF (Vstring(i+1:i+1).ne.CHAR(32)) decode=.TRUE.
                Iblank=i
              ELSE
                ie=i
              ENDIF                                    
              IF (decode.or.(i.eq.Lval)) THEN
                Nval=Nval+1
!
! Processing numeric values.  Check starting character to determine
! if numeric or character values. It is possible to have both when
! processing repetitions via the multiplication symbol.              
!
                Schar=ICHAR(Vstring(is:is))
                IF (((48.le.Schar).and.(Schar.le.57)).or.               &
     &              (Schar.eq.43).or.(Schar.eq.45)) THEN
                  IF ((Nmul.gt.0).and.                                  &
     &                (is.lt.Imul(ic)).and.(Imul(ic).lt.ie)) THEN
                    READ (Vstring(is:Imul(ic)-1),*) copies
                    Schar=ICHAR(Vstring(Imul(ic)+1:Imul(ic)+1))
                    IF ((43.le.Schar).and.(Schar.le.57)) THEN
                      READ (Vstring(Imul(ic)+1:ie),*) Rval(Nval)
                      DO j=1,copies-1
                        Rval(Nval+j)=Rval(Nval)
                      END DO
                    ELSE
                      string=Vstring(Imul(ic)+1:ie)
                      LenS=LEN_TRIM(string)
                      Cval(Nval)=string(1:LenS)
                      DO j=1,copies-1
                        Cval(Nval+j)=Cval(Nval)
                      END DO
                    END IF
                    Nval=Nval+copies-1
                    ic=ic+1
                  ELSE
                    string=Vstring(is:ie)
                    LenS=LEN_TRIM(string)
                    READ (string(1:LenS),*) Rval(Nval)
                  END IF
                ELSE
!
! Processing character values (logicals and strings).
!
                  IF ((Nmul.gt.0).and.                                  &
     &                (is.lt.Imul(ic)).and.(Imul(ic).lt.ie)) THEN
                    READ (Vstring(is:Imul(ic)-1),*) copies
                    Cval(Nval)=Vstring(Imul(ic)+1:ie)
                    DO j=1,copies-1
                      Cval(Nval+j)=Cval(Nval)
                    END DO
                    Nval=Nval+copies-1
                    ic=ic+1
                  ELSE
                    string=Vstring(is:ie)
                    Cval(Nval)=TRIM(ADJUSTL(string))
                  END IF
                  IsString=.TRUE.
                END IF
                is=Iblank+1
                ie=Lval
                decode=.FALSE.
              END IF
            END DO
          END IF
        END IF             
        status=Nval
      END IF
      decode_line=status       
      RETURN
      END FUNCTION decode_line
      FUNCTION load_i (Ninp, Vinp, Nout, Vout)
!
!=======================================================================
!                                                                      !
!  This function loads input values into a requested model integer     !
!  variable.                                                           !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     Ninp       Size of input variable.                               !
!     Vinp       Input values                                          !
!     Nout       Number of output values.                              !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     Vout       Output integer variable.                              !
!     load_i     Number of output values processed.                    !
!                                                                      !
!=======================================================================
!
      USE mod_kinds
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: Ninp, Nout
      real(r8), intent(in) :: Vinp(Ninp)
      integer, intent(out) :: Vout(Nout)
!
!  Local variable declarations.
!
      integer :: i, ic
      integer :: load_i
!
!-----------------------------------------------------------------------
!  Load integer variable with input values.
!-----------------------------------------------------------------------
!
!  If not all values are provided for variable, assume the last value
!  for the rest of the array.
!
      ic=0
      IF (Ninp.le.Nout) THEN
        DO i=1,Ninp
          ic=ic+1
          Vout(i)=INT(Vinp(i))
        END DO
        DO i=Ninp+1,Nout
          ic=ic+1
          Vout(i)=INT(Vinp(Ninp))
        END DO
      ELSE
        DO i=1,Nout
          ic=ic+1
          Vout(i)=INT(Vinp(i))
        END DO
      END IF
      load_i=ic
      RETURN
      END FUNCTION load_i
      FUNCTION load_l (Ninp, Vinp, Nout, Vout)
!
!=======================================================================
!                                                                      !
!  This function loads input values into a requested model logical     !
!  variable.                                                           !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     Ninp       Size of input variable.                               !
!     Vinp       Input values                                          !
!     Nout       Number of output values.                              !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     Vout       Output integer variable.                              !
!     load_l     Number of output values processed.                    !
!                                                                      !
!=======================================================================
!
      USE mod_kinds
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: Ninp, Nout
      character (len=*), intent(in) :: Vinp(Ninp)
      logical, intent(out) :: Vout(Nout)
!
!  Local variable declarations.
!
      integer :: i, ic
      integer :: load_l
!
!-----------------------------------------------------------------------
!  Load integer variable with input values.
!-----------------------------------------------------------------------
!
!  If not all values are provided for variable, assume the last value
!  for the rest of the array.
!
      ic=0
      IF (Ninp.le.Nout) THEN
        DO i=1,Ninp
          ic=ic+1
          IF ((Vinp(i)(1:1).eq.'T').or.(Vinp(i)(1:1).eq.'t')) THEN
            Vout(i)=.TRUE.
          ELSE
            Vout(i)=.FALSE.
          END IF
        END DO
        DO i=Ninp+1,Nout
          ic=ic+1
          IF ((Vinp(Ninp)(1:1).eq.'T').or.(Vinp(Ninp)(1:1).eq.'t')) THEN
            Vout(i)=.TRUE.
          ELSE
            Vout(i)=.FALSE.
          END IF
        END DO
      ELSE
        DO i=1,Nout
          ic=ic+1
          IF ((Vinp(i)(1:1).eq.'T').or.(Vinp(i)(1:1).eq.'t')) THEN
            Vout(i)=.TRUE.
          ELSE
            Vout(i)=.FALSE.
          END IF
        END DO
      END IF
      load_l=ic
      RETURN
      END FUNCTION load_l
      FUNCTION load_r (Ninp, Vinp, Nout, Vout)
!
!=======================================================================
!                                                                      !
!  This function loads input values into a requested model real        !
!  variable.                                                           !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     Ninp       Size of input variable.                               !
!     Vinp       Input values                                          !
!     Nout       Number of output values.                              !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     Vout       Output real variable.                                 !
!     load_r     Number of output values processed.                    !
!                                                                      !
!=======================================================================
!
      USE mod_kinds
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: Ninp, Nout
      real(r8), intent(in) :: Vinp(Ninp)
      real(r8), intent(out) :: Vout(Nout)
!
!  Local variable declarations.
!
      integer :: i, ic
      integer :: load_r
!
!-----------------------------------------------------------------------
!  Load integer variable with input values.
!-----------------------------------------------------------------------
!
!  If not all values are provided for variable, assume the last value
!  for the rest of the array.
!
      ic=0
      IF (Ninp.le.Nout) THEN
        DO i=1,Ninp
          ic=ic+1
          Vout(i)=Vinp(i)
        END DO
        DO i=Ninp+1,Nout
          ic=ic+1
          Vout(i)=Vinp(Ninp)
        END DO
      ELSE
        DO i=1,Nout
          ic=ic+1
          Vout(i)=Vinp(i)
        END DO
      END IF
      load_r=ic
      RETURN
      END FUNCTION load_r
