      MODULE u3dbc_mod
!
!svn $Id: u3dbc_im.F 895 2009-01-12 21:06:20Z kate $
!=======================================================================
!  Copyright (c) 2002-2009 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                           Hernan G. Arango   !
!========================================== Alexander F. Shchepetkin ===
!                                                                      !
!  This subroutine sets lateral boundary conditions for total 3D       !
!  U-velocity.                                                         !
!                                                                      !
!=======================================================================
!
      implicit none
      PRIVATE
      PUBLIC  :: u3dbc_tile
      CONTAINS
!
!***********************************************************************
      SUBROUTINE u3dbc (ng, tile, nout)
!***********************************************************************
!
      USE mod_param
      USE mod_ocean
      USE mod_stepping
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, nout
!
!  Local variable declarations.
!
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private storage
!  arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J)- and MAX(I,J)-directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL u3dbc_tile (ng, tile,                                        &
     &                 LBi, UBi, LBj, UBj, N(ng),                       &
     &                 IminS, ImaxS, JminS, JmaxS,                      &
     &                 nstp(ng), nout,                                  &
     &                 OCEAN(ng) % u)
      RETURN
      END SUBROUTINE u3dbc
!
!***********************************************************************
      SUBROUTINE u3dbc_tile (ng, tile,                                  &
     &                       LBi, UBi, LBj, UBj, UBk,                   &
     &                       IminS, ImaxS, JminS, JmaxS,                &
     &                       nstp, nout,                                &
     &                       u)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_grid
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj, UBk
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: nstp, nout
!
      real(r8), intent(inout) :: u(LBi:,LBj:,:,:)
!
!  Local variable declarations.
!
      integer :: i, j, k
      real(r8), parameter :: eps = 1.0E-20_r8
      real(r8) :: Ce, Cx, cff, dUde, dUdt, dUdx, tau
      real(r8), dimension(IminS:ImaxS,JminS:JmaxS) :: grad
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the western edge.
!-----------------------------------------------------------------------
!
      IF (Istr.eq.1) THEN
!
!  Western edge, implicit upstream radiation condition.
!
        DO k=1,N(ng)
          DO j=Jstr,Jend+1
            grad(Istr  ,j)=u(Istr  ,j  ,k,nstp)-                        &
     &                     u(Istr  ,j-1,k,nstp)
            grad(Istr+1,j)=u(Istr+1,j  ,k,nstp)-                        &
     &                     u(Istr+1,j-1,k,nstp)
          END DO
          DO j=Jstr,Jend
            dUdt=u(Istr+1,j,k,nstp)-u(Istr+1,j,k,nout)
            dUdx=u(Istr+1,j,k,nout)-u(Istr+2,j,k,nout)
            IF ((dUdt*dUdx).lt.0.0_r8) THEN
              tau=M3obc_in(ng,iwest)
            ELSE
              tau=M3obc_out(ng,iwest)
            END IF
            tau=tau*dt(ng)
            IF ((dUdt*dUdx).lt.0.0_r8) dUdt=0.0_r8
            IF ((dUdt*(grad(Istr+1,j)+grad(Istr+1,j+1))).gt.0.0_r8) THEN
              dUde=grad(Istr+1,j  )
            ELSE
              dUde=grad(Istr+1,j+1)
            END IF
            cff=MAX(dUdx*dUdx+dUde*dUde,eps)
            Cx=dUdt*dUdx
            Ce=MIN(cff,MAX(dUdt*dUde,-cff))
            u(Istr,j,k,nout)=(cff*u(Istr  ,j,k,nstp)+                   &
     &                        Cx *u(Istr+1,j,k,nout)-                   &
     &                        MAX(Ce,0.0_r8)*grad(Istr,j  )-            &
     &                        MIN(Ce,0.0_r8)*grad(Istr,j+1))/           &
     &                       (cff+Cx)
            u(Istr,j,k,nout)=u(Istr,j,k,nout)+                          &
     &                       tau*(BOUNDARY(ng)%u_west(j,k)-             &
     &                            u(Istr,j,k,nstp))
            u(Istr,j,k,nout)=u(Istr,j,k,nout)*                          &
     &                       GRID(ng)%umask(Istr,j)
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the eastern edge.
!-----------------------------------------------------------------------
!
      IF (Iend.eq.Lm(ng)) THEN
!
!  Eastern edge, closed boundary condition.
!
        DO k=1,N(ng)
          DO j=Jstr,Jend
            u(Iend+1,j,k,nout)=0.0_r8
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the southern edge.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1) THEN
!
!  Southern edge, implicit upstream radiation condition.
!
        DO k=1,N(ng)
          DO i=IstrU-1,Iend
            grad(i,Jstr-1)=u(i+1,Jstr-1,k,nstp)-                        &
     &                     u(i  ,Jstr-1,k,nstp)
            grad(i,Jstr  )=u(i+1,Jstr  ,k,nstp)-                        &
     &                     u(i  ,Jstr  ,k,nstp)
          END DO
          DO i=IstrU,Iend
            dUdt=u(i,Jstr,k,nstp)-u(i,Jstr  ,k,nout)
            dUde=u(i,Jstr,k,nout)-u(i,Jstr+1,k,nout)
            IF ((dUdt*dUde).lt.0.0_r8) THEN
              tau=M3obc_in(ng,isouth)
            ELSE
              tau=M3obc_out(ng,isouth)
            END IF
            tau=tau*dt(ng)
            IF ((dUdt*dUde).lt.0.0_r8) dUdt=0.0_r8
            IF ((dUdt*(grad(i-1,Jstr)+grad(i,Jstr))).gt.0.0_r8) THEN
              dUdx=grad(i-1,Jstr)
            ELSE
              dUdx=grad(i  ,Jstr)
            END IF
            cff=MAX(dUdx*dUdx+dUde*dUde,eps)
            Cx=MIN(cff,MAX(dUdt*dUdx,-cff))
            Ce=dUdt*dUde
            u(i,Jstr-1,k,nout)=(cff*u(i,Jstr-1,k,nstp)+                 &
     &                          Ce *u(i,Jstr  ,k,nout)-                 &
     &                          MAX(Cx,0.0_r8)*grad(i-1,Jstr-1)-        &
     &                          MIN(Cx,0.0_r8)*grad(i  ,Jstr-1))/       &
     &                         (cff+Ce)
            u(i,Jstr-1,k,nout)=u(i,Jstr-1,k,nout)+                      &
     &                         tau*(BOUNDARY(ng)%u_south(i,k)-          &
     &                              u(i,Jstr-1,k,nstp))
            u(i,Jstr-1,k,nout)=u(i,Jstr-1,k,nout)*                      &
     &                         GRID(ng)%umask(i,Jstr-1)
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the northern edge.
!-----------------------------------------------------------------------
!
      IF (Jend.eq.Mm(ng)) THEN
!
!  Northern edge, closed boundary condition: free slip (gamma2=1)  or
!                                            no   slip (gamma2=-1).
!
        DO k=1,N(ng)
          DO i=Istr,IendR
            u(i,Jend+1,k,nout)=gamma2(ng)*u(i,Jend,k,nout)
            u(i,Jend+1,k,nout)=u(i,Jend+1,k,nout)*                      &
     &                         GRID(ng)%umask(i,Jend+1)
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Boundary corners.
!-----------------------------------------------------------------------
!
      IF ((Jstr.eq.1).and.(Istr.eq.1)) THEN
        DO k=1,N(ng)
          u(Istr,Jstr-1,k,nout)=0.5_r8*(u(Istr+1,Jstr-1,k,nout)+        &
     &                                  u(Istr  ,Jstr  ,k,nout))
        END DO
      END IF
      IF ((Jstr.eq.1).and.(Iend.eq.Lm(ng))) THEN
        DO k=1,N(ng)
          u(Iend+1,Jstr-1,k,nout)=0.5_r8*(u(Iend  ,Jstr-1,k,nout)+      &
     &                                    u(Iend+1,Jstr  ,k,nout))
        END DO
      END IF
      IF ((Jend.eq.Mm(ng)).and.(Istr.eq.1)) THEN
        DO k=1,N(ng)
          u(Istr,Jend+1,k,nout)=0.5_r8*(u(Istr  ,Jend  ,k,nout)+        &
     &                                  u(Istr+1,Jend+1,k,nout))
        END DO
      END IF
      IF ((Jend.eq.Mm(ng)).and.(Iend.eq.Lm(ng))) THEN
        DO k=1,N(ng)
          u(Iend+1,Jend+1,k,nout)=0.5_r8*(u(Iend+1,Jend  ,k,nout)+      &
     &                                    u(Iend  ,Jend+1,k,nout))
        END DO
      END IF
      RETURN
      END SUBROUTINE u3dbc_tile
      END MODULE u3dbc_mod
