      MODULE uibc_mod
!***********************************************************************
!  Compute the lateral boundary conditions on the ice U-velocity.
!***********************************************************************
      implicit none
      PRIVATE
      PUBLIC uibc_tile
      CONTAINS
!
!***********************************************************************
      SUBROUTINE uibc (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_ice
      USE mod_stepping
!
      integer, intent(in) :: ng, tile
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private storage
!  arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J)- and MAX(I,J)-directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL  uibc_tile (ng, tile,                                        &
     &                 LBi, UBi, LBj, UBj,                              &
     &                 liuol(ng), liunw(ng),                            &
     &                 ICE(ng) % ui)
      RETURN
      END SUBROUTINE uibc
!
!***********************************************************************
      SUBROUTINE uibc_tile (ng, tile,                                   &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           liuol, liunw,                          &
     &                           ui)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_grid
      USE mod_scalars
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: liuol, liunw
      real(r8), intent(inout) :: ui(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j, know
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set time-indices
!-----------------------------------------------------------------------
!
        know=liuol
!
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the western edge.
!-----------------------------------------------------------------------
!
      IF (Istr.eq.1) THEN
!
!  Western edge, gradient boundary condition.
!
        DO j=Jstr,Jend
          ui(1,j,liunw)=ui(2,j,liunw)
          ui(1,j,liunw)=ui(1,j,liunw)*                                  &
     &                   GRID(ng)%umask(1,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the eastern edge.
!-----------------------------------------------------------------------
!
      IF (Iend.eq.Lm(ng)) THEN
!
!  Eastern edge, closed boundary condition.
!
        DO j=Jstr,Jend
          ui(Lm(ng)+1,j,liunw)=0.0_r8
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the southern edge.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1) THEN
!
!  Southern edge, gradient boundary condition.
!
        DO i=IstrU,Iend
          ui(i,0,liunw)=ui(i,1,liunw)
          ui(i,0,liunw)=ui(i,0,liunw)*                                  &
     &                   GRID(ng)%umask(i,0)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the northern edge.
!-----------------------------------------------------------------------
!
      IF (Jend.eq.Mm(ng)) THEN
!
!  Northern edge, closed boundary condition: free slip (gamma2=1)  or
!                                            no   slip (gamma2=-1).
!
        DO i=Istr,IendR
          ui(i,Mm(ng)+1,liunw)=gamma2(ng)*ui(i,Mm(ng),liunw)
          ui(i,Mm(ng)+1,liunw)=ui(i,Mm(ng)+1,liunw)*                    &
     &                          GRID(ng)%umask(i,Mm(ng)+1)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Boundary corners.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1.and.Istr.eq.1) THEN
        ui(1,0,liunw)=0.5_r8*(ui(2,0,liunw)+                            &
     &                         ui(1,1,liunw))
      END IF
      IF (Jstr.eq.1.and.Iend.eq.Lm(ng)) THEN
        ui(Lm(ng)+1,0,liunw)=0.5_r8*(ui(Lm(ng)  ,0,liunw)+              &
     &                                ui(Lm(ng)+1,1,liunw))
      END IF
      IF (Jend.eq.Mm(ng).and.Istr.eq.1) THEN
        ui(1,Mm(ng)+1,liunw)=0.5_r8*(ui(2,Mm(ng)+1,liunw)+              &
     &                                ui(1,Mm(ng)  ,liunw))
      END IF
      IF (Jend.eq.Mm(ng).and.Iend.eq.Lm(ng)) THEN
        ui(Lm(ng)+1,Mm(ng)+1,liunw)=0.5_r8*                             &
     &                               (ui(Lm(ng)  ,Mm(ng)+1,liunw)+      &
     &                                ui(Lm(ng)+1,Mm(ng)  ,liunw))
      END IF
      RETURN
      END SUBROUTINE uibc_tile
      END MODULE uibc_mod
