      MODULE zetabc_mod
!
!svn $Id: zetabc.F 895 2009-01-12 21:06:20Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2009 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine sets lateral boundary conditions for free-surface.     !
!                                                                      !
!=======================================================================
!
      implicit none
      PRIVATE
      PUBLIC  :: zetabc_tile
      CONTAINS
!
!***********************************************************************
      SUBROUTINE zetabc (ng, tile, kout)
!***********************************************************************
!
      USE mod_param
      USE mod_ocean
      USE mod_stepping
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, kout
!
!  Local variable declarations.
!
      integer :: IminS, ImaxS, JminS, JmaxS
      integer :: LBi, UBi, LBj, UBj, LBij, UBij
!
!  Set horizontal starting and ending indices for automatic private storage
!  arrays.
!
      IminS=BOUNDS(ng)%Istr(tile)-3
      ImaxS=BOUNDS(ng)%Iend(tile)+3
      JminS=BOUNDS(ng)%Jstr(tile)-3
      JmaxS=BOUNDS(ng)%Jend(tile)+3
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
!  Set array lower and upper bounds for MIN(I,J)- and MAX(I,J)-directions.
!
      LBij=BOUNDS(ng)%LBij
      UBij=BOUNDS(ng)%UBij
!
      CALL zetabc_tile (ng, tile,                                       &
     &                  LBi, UBi, LBj, UBj,                             &
     &                  IminS, ImaxS, JminS, JmaxS,                     &
     &                  krhs(ng), kstp(ng), kout,                       &
     &                  OCEAN(ng) % zeta)
      RETURN
      END SUBROUTINE zetabc
!
!***********************************************************************
      SUBROUTINE zetabc_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        IminS, ImaxS, JminS, JmaxS,               &
     &                        krhs, kstp, kout,                         &
     &                        zeta)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_grid
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: IminS, ImaxS, JminS, JmaxS
      integer, intent(in) :: krhs, kstp, kout
      real(r8), intent(inout) :: zeta(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j, know
      real(r8), parameter :: eps =1.0E-20_r8
      real(r8) :: Ce, Cx
      real(r8) :: cff, cff1, cff2, dt2d, dZde, dZdt, dZdx, tau
      real(r8), dimension(IminS:ImaxS,JminS:JmaxS) :: grad
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set time-indices
!-----------------------------------------------------------------------
!
      IF (iif(ng).eq.1) THEN
        know=krhs
        dt2d=dtfast(ng)
      ELSE IF (PREDICTOR_2D_STEP(ng)) THEN
        know=krhs
        dt2d=2.0_r8*dtfast(ng)
      ELSE
        know=kstp
        dt2d=dtfast(ng)
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the western edge.
!-----------------------------------------------------------------------
!
      IF (Istr.eq.1) THEN
!
!  Western edge, Chapman boundary condition.
!
        DO j=Jstr,Jend
          cff=dt2d*GRID(ng)%pm(Istr,j)
          cff1=SQRT(g*(GRID(ng)%h(Istr,j)+                              &
     &                 zeta(Istr,j,know)))
          Cx=cff*cff1
          cff2=1.0_r8/(1.0_r8+Cx)
          zeta(Istr-1,j,kout)=cff2*(zeta(Istr-1,j,know)+                &
     &                              Cx*zeta(Istr,j,kout))
          zeta(Istr-1,j,kout)=zeta(Istr-1,j,kout)*                      &
     &                        GRID(ng)%rmask(Istr-1,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the eastern edge.
!-----------------------------------------------------------------------
!
      IF (Iend.eq.Lm(ng)) THEN
!
!  Eastern edge, closed boundary condition.
!
        DO j=Jstr,Jend
          zeta(Iend+1,j,kout)=zeta(Iend,j,kout)
          zeta(Iend+1,j,kout)=zeta(Iend+1,j,kout)*                      &
     &                        GRID(ng)%rmask(Iend+1,j)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the southern edge.
!-----------------------------------------------------------------------
!
      IF (Jstr.eq.1) THEN
!
!  Southern edge, Chapman boundary condition.
!
        DO i=Istr,Iend
          cff=dt2d*GRID(ng)%pn(i,Jstr)
          cff1=SQRT(g*(GRID(ng)%h(i,Jstr)+                              &
     &                 zeta(i,Jstr,know)))
          Ce=cff*cff1
          cff2=1.0_r8/(1.0_r8+Ce)
          zeta(i,Jstr-1,kout)=cff2*(zeta(i,Jstr-1,know)+                &
     &                              Ce*zeta(i,Jstr,kout))
          zeta(i,Jstr-1,kout)=zeta(i,Jstr-1,kout)*                      &
     &                        GRID(ng)%rmask(i,Jstr-1)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Lateral boundary conditions at the northern edge.
!-----------------------------------------------------------------------
!
      IF (Jend.eq.Mm(ng)) THEN
!
!  Northern edge, closed boundary condition.
!
        DO i=Istr,Iend
          zeta(i,Jend+1,kout)=zeta(i,Jend,kout)
          zeta(i,Jend+1,kout)=zeta(i,Jend+1,kout)*                      &
     &                        GRID(ng)%rmask(i,Jend+1)
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Boundary corners.
!-----------------------------------------------------------------------
!
      IF ((Jstr.eq.1).and.(Istr.eq.1)) THEN
        zeta(Istr-1,Jstr-1,kout)=0.5_r8*(zeta(Istr  ,Jstr-1,kout)+      &
     &                                   zeta(Istr-1,Jstr  ,kout))
      END IF
      IF ((Jstr.eq.1).and.(Iend.eq.Lm(ng))) THEN
        zeta(Iend+1,Jstr-1,kout)=0.5_r8*(zeta(Iend  ,Jstr-1,kout)+      &
     &                                   zeta(Iend+1,Jstr  ,kout))
      END IF
      IF ((Jend.eq.Mm(ng)).and.(Istr.eq.1)) THEN
        zeta(Istr-1,Jend+1,kout)=0.5_r8*(zeta(Istr-1,Jend  ,kout)+      &
     &                                   zeta(Istr  ,Jend+1,kout))
      END IF
      IF ((Jend.eq.Mm(ng)).and.(Iend.eq.Lm(ng))) THEN
        zeta(Iend+1,Jend+1,kout)=0.5_r8*(zeta(Iend+1,Jend  ,kout)+      &
     &                                   zeta(Iend  ,Jend+1,kout))
      END IF
      RETURN
      END SUBROUTINE zetabc_tile
      END MODULE zetabc_mod
