      SUBROUTINE get_2dfld (ng, model, ifield, ncid, nfiles, fname,     &
     &                      update, LBi, UBi, LBj, UBj, Iout, Irec,     &
     &                      Fmask,                                      &
     &                      Fout)
!
!svn $Id: get_2dfld.F 588 2008-03-21 23:09:01Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2008 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine reads in requested 2D field (point or grided) from     !
!  specified NetCDF file. Forward time processing.                     !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     ifield     Field ID.                                             !
!     ncid       NetCDF file ID.                                       !
!     nfiles     Number of input NetCDF files.                         !
!     fname      NetCDF file name(s).                                  !
!     LBi        I-dimension Lower bound.                              !
!     UBi        I-dimension Upper bound.                              !
!     LBj        J-dimension Lower bound.                              !
!     UBj        J-dimension Upper bound.                              !
!     Iout       Size of the outer dimension,  if any.  Otherwise,     !
!                  Iout must be set to one by the calling program.     !
!     Irec       Number of 2D field records to read.                   !
!     Fmask      Land/Sea mask, if any.                                !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     Fout       Read field.                                           !
!     update     Switch indicating reading of the requested field      !
!                  the current time step.                              !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      logical, intent(out) :: update
      integer, intent(in) :: ng, model, ifield, nfiles, Iout, Irec
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(inout) :: ncid
      character (len=*), intent(in) :: fname(nfiles)
      real(r8), intent(in) :: Fmask(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: Fout(LBi:UBi,LBj:UBj,Iout)
!
!  Local variable declarations.
!
      logical :: Liocycle, Lgrided, Lonerec, got_var, got_time, special
      integer :: Tid, Tindex, Trec, Vid, Vtype
      integer :: dimsiz, i, lend, lfvar, lstr, ltvar, lvar, nrec, nvatts
      integer :: nvd, status, vartype
      integer :: Vsize(4), start(4), total(4)
      integer :: nf_fread2d, nf_fread3d
      real(r8) :: Clength, Fmax, Fmin, Tdelta, Tend
      real(r8) :: Tmax, Tmin, Tmono, Tscale, Tstr, scale
      real(r8) :: Fval(1), Tval(1)
      character (len=45) :: attnam, dimnam
!
!-----------------------------------------------------------------------
!  On first call, inquire about the contents of input NetCDF file.
!-----------------------------------------------------------------------
!
      IF (exit_flag.ne.NoError) RETURN
!
      IF (iic(ng).eq.0) THEN
!
!  Intialize local variables.
!
        Vid=-1
        Tid=-1        
        Liocycle=.FALSE.
        Lgrided=.FALSE.
        Lonerec=.FALSE.
        got_var=.FALSE.
        got_time=.FALSE.
        Vtype=Iinfo(1,ifield,ng)
        lfvar=LEN_TRIM(Vname(1,ifield))
        ltvar=LEN_TRIM(Tname(ifield))
!
!  Inquire about the dimensions and variables. Check for consistency.
!
        IF (InpThread) THEN
          CALL opencdf (ng, nfiles, fname, ncfile, N(ng), ifield, nrec, &
     &                  nvd, Vsize)
          IF (exit_flag.ne.NoError) RETURN
!
!  Scan variable list from input NetCDF and check for requested
!  variables.
!
          DO i=1,nvars
            IF (lfvar.gt.0) THEN
              IF (TRIM(varnam(i)).eq.TRIM(Vname(1,ifield))) THEN
                Vid=i
                got_var=.TRUE.
                IF (nvdims(i).gt.1) Lgrided=.TRUE.
              END IF
            END IF
            IF (ltvar.gt.0) THEN
              IF (TRIM(varnam(i)).eq.TRIM(Tname(ifield))) THEN
                Tid=i
                got_time=.TRUE.
              END IF
            END IF
          END DO
          Linfo(1,ifield,ng)=Lgrided
          Iinfo(2,ifield,ng)=Vid
          Iinfo(3,ifield,ng)=Tid
          Iinfo(4,ifield,ng)=nrec
          Iinfo(5,ifield,ng)=Vsize(1)
          Iinfo(6,ifield,ng)=Vsize(2)
!
!  Terminate execution requested variables are not found.
!
          IF (.not.got_var) THEN
            IF (nfiles.gt.1) THEN
              WRITE (stdout,10) TRIM(Vname(1,ifield)), 'files:'
              DO i=1,nfiles
                WRITE (stdout,'(15x,a)') TRIM(fname(i))
              END DO
            ELSE
              WRITE (stdout,10) TRIM(Vname(1,ifield)), 'file:'
              WRITE (stdout,'(15x,a)') TRIM(ncfile)
            END IF
            exit_flag=2
            RETURN
          END IF
          IF (.not.got_time) THEN
            IF (nfiles.gt.1) THEN
              WRITE (stdout,10) TRIM(Tname(ifield)), 'files:'
              DO i=1,nfiles
                WRITE (stdout,'(15x,a)') TRIM(fname(i))
              END DO
            ELSE
              WRITE (stdout,10) TRIM(Tname(ifield)), 'file:'
              WRITE (stdout,'(15x,a)') TRIM(ncfile)
            END IF
            exit_flag=2
            RETURN
          END IF
!
!  If appropriate, open input NetCDF file for reading.  If processing
!  model forcing (multiple files allowed), check if file for requested
!  field has been already opened and get/save its ID from/to the
!  association table. 
!
          CALL openids (nFfiles(ng), FRCname(1,ng), FRCids(1,ng),       &
     &                  ncfile, ncid)
          IF (ncid.eq.-1) THEN
            status=nf90_open(TRIM(ncfile), nf90_nowrite, ncid)
            IF (status.ne.nf90_noerr) THEN
              WRITE (stdout,20) TRIM(ncfile)
              exit_flag=2
              ioerror=status
              RETURN
            END IF
            CALL openids (nFfiles(ng), FRCname(1,ng), FRCids(1,ng),     &
     &                    ncfile, ncid)
          END IF
          Cinfo(ifield,ng)=TRIM(ncfile)
!
!  If "scale_factor" attribute is present for a variable, the data are
!  to be multiplied by this factor.  Check if only water points are
!  available. Also check if processing special 2D fields with additional
!  dimensions (for example, tides).
!
          IF (got_var) THEN
            status=nf90_inquire_variable(ncid, Vid, varnam(ifield),     &
     &                                   vartype, nvdims(ifield),       &
     &                                   vdims(:,ifield), nvatts)
            IF (status.eq.nf90_noerr) THEN
              DO i=1,nvatts
                status=nf90_inq_attname(ncid, Vid, i, attnam)
                IF (status.eq.nf90_noerr) THEN
                  IF (TRIM(attnam).eq.'scale_factor') THEN
                    status=nf90_get_att(ncid, Vid, TRIM(attnam),        &
     &                                  scale)
                    IF (status.eq.nf90_noerr) THEN
                      Fscale(ifield,ng)=Fscale(ifield,ng)*scale
                    ELSE
                      WRITE (stdout,30) TRIM(attnam)
                      exit_flag=2
                      RETURN
                    END IF
                  ELSE IF (TRIM(attnam).eq.'water_points') THEN
                    Iinfo(1,ifield,ng)=-ABS(Iinfo(1,ifield,ng))
                    Vtype=Iinfo(1,ifield,ng)
                  END IF
                ELSE
                  WRITE (stdout,40) TRIM(varnam(ifield))
                  exit_flag=2
                  RETURN
                END IF
              END DO
!
              special=.FALSE.
              DO i=1,nvdims(ifield)
                status=nf90_inquire_dimension(ncid, vdims(i,ifield),    &
     &                                        dimnam, dimsiz)
                IF (status.eq.nf90_noerr) THEN
                  IF (INDEX(TRIM(dimnam),'period').ne.0) THEN
                    special=.TRUE.
                  END IF                  
                ELSE
                  WRITE (stdout,50) TRIM(varnam(ifield))
                  exit_flag=2
                  RETURN
                END IF
              END DO
              Linfo(4,ifield,ng)=special
            ELSE
              WRITE (stdout,60)
              exit_flag=2
              RETURN
            END IF
          END IF
!
!  Determine initial time record to read and cycling switch.
!
          CALL get_cycle (ng, ncid, Tid, nrec, tdays(ng), Liocycle,     &
     &                    Clength, Trec, Tstr, Tend, Tmin, Tmax, Tscale)
          IF (exit_flag.ne.NoError) RETURN
          Linfo(2,ifield,ng)=Liocycle
          Finfo(1,ifield,ng)=Tmin
          Finfo(2,ifield,ng)=Tmax
          Finfo(3,ifield,ng)=Tstr
          Finfo(4,ifield,ng)=Tend
          Finfo(5,ifield,ng)=Clength
          Finfo(6,ifield,ng)=Tscale
!
!  The strategy here is to create a local, monotonically increasing
!  time variable so the interpolation between snapshots is trivial
!  when cycling forcing fields. Subtract one to time record counter
!  "Trec" to avoid doing special case at initialization.
!
          IF (Irec.eq.1) THEN
            Tindex=Iout
          ELSE
            Tindex=1
          END IF
          IF (Liocycle) THEN
            IF (Trec.eq.nrec) THEN
              IF (tdays(ng).lt.Tmax) THEN
                Tmono=Tstr-Clength
              ELSE
                Tmono=tdays(ng)+(Tstr-Clength)
                IF (Tstr.eq.Tmax) THEN
                  Tmono=Tmono+(Tmin-MOD(tdays(ng)+Tmin,Clength))
                ELSE
                  Tmono=Tmono+(Tstr-MOD(tdays(ng)+Tstr,Clength))
                END IF
              END IF
            ELSE
              IF (tdays(ng).gt.Clength) THEN
                Tmono=tdays(ng)-MOD(tdays(ng)-Tstr,Clength)
              ELSE
                Tmono=Tstr
              END IF
            END IF
          ELSE
            Tmono=Tstr
          END IF
          Tmono=Tmono*day2sec
          Trec=Trec-1
          Iinfo(8,ifield,ng)=Tindex
          Iinfo(9,ifield,ng)=Trec
          Finfo(7,ifield,ng)=Tmono
!
!  Set switch for one time record dataset. In this case, the input
!  data is always the same and time interpolation is not performed.
!
          IF (nrec.eq.1) Lonerec=.TRUE.
          Linfo(3,ifield,ng)=Lonerec
        END IF
        Tindex=Iinfo(8,ifield,ng)
        Vtime(Tindex,ifield,ng)=Finfo(3,ifield,ng)
      END IF         
!
!-----------------------------------------------------------------------
!  Get requested field information from global storage.
!-----------------------------------------------------------------------
!
      Lgrided =Linfo(1,ifield,ng)
      Liocycle=Linfo(2,ifield,ng)
      Lonerec =Linfo(3,ifield,ng)
      special =Linfo(4,ifield,ng)
      Vtype   =Iinfo(1,ifield,ng)
      Vid     =Iinfo(2,ifield,ng)
      Tid     =Iinfo(3,ifield,ng)
      nrec    =Iinfo(4,ifield,ng)
      Vsize(1)=Iinfo(5,ifield,ng)
      Vsize(2)=Iinfo(6,ifield,ng)
      Tindex  =Iinfo(8,ifield,ng)
      Trec    =Iinfo(9,ifield,ng)
      Tmin    =Finfo(1,ifield,ng)
      Tmax    =Finfo(2,ifield,ng)
      Clength =Finfo(5,ifield,ng)
      Tscale  =Finfo(6,ifield,ng)
      Tmono   =Finfo(7,ifield,ng)
      ncfile  =Cinfo(ifield,ng)
!
!-----------------------------------------------------------------------
!  If appropriate, read in new data.
!-----------------------------------------------------------------------
!
      update=.FALSE.
      IF ((Tmono.lt.time(ng)).or.(iic(ng).eq.0).or.                     &
     &    (iic(ng).eq.ntstart(ng))) THEN
        IF (Liocycle) THEN
          Trec=MOD(Trec,nrec)+1
        ELSE
          Trec=Trec+1
        END IF
        Iinfo(9,ifield,ng)=Trec
        IF (Trec.le.nrec) THEN
!
!  Set rolling index for two-time record storage of input data.  If
!  "Iout" is unity, input data is stored in timeless array by the
!  calling program.  If Irec > 1, this routine is used to read a 2D
!  field varying in another non-time dimension.
!
          IF (.not.special.and.(Irec.eq.1)) THEN
            IF (Iout.eq.1) THEN
              Tindex=1
            ELSE
              Tindex=3-Tindex
            END IF
            Iinfo(8,ifield,ng)=Tindex
          END IF
!
!  Read in time coordinate and scale it to day units.
!
          IF (InpThread.and.(Tid.ge.0)) THEN
            start(1)=Trec
            total(1)=1
            status=nf90_get_var(ncid, Tid, Tval, start, total)
            IF (status.ne.nf90_noerr) THEN
              WRITE (stdout,70) TRIM(Tname(ifield)), Trec
              exit_flag=2
              ioerror=status
              RETURN
            END IF
          END IF
          Vtime(Tindex,ifield,ng)=Tval(1)*Tscale
!
!  Read in 2D-grided or point data. Notice for special 2D fields, Vtype
!  is augmented by four indicating reading a 3D field. This rational is
!  used to read fields like tide data.
!
          IF (Vid.ge.0) THEN
            Fmin=0.0_r8
            Fmax=0.0_r8
            IF (Lgrided) THEN
              IF (special) THEN
                Vsize(3)=Irec
                status=nf_fread3d(ng, model, ncid, Vid, 0, Vtype+4,     &
     &                            Vsize, LBi, UBi, LBj, UBj, 1, Irec,   &
     &                            Fscale(ifield,ng), Fmin, Fmax,        &
     &                            Fmask(LBi,LBj),                       &
     &                            Fout(LBi,LBj,1))
                Finfo(8,ifield,ng)=Fmin
                Finfo(9,ifield,ng)=Fmax
              ELSE
                status=nf_fread2d(ng, model, ncid, Vid, Trec, Vtype,    &
     &                            Vsize, LBi, UBi, LBj, UBj,            &
     &                            Fscale(ifield,ng), Fmin, Fmax,        &
     &                            Fmask(LBi,LBj),                       &
     &                            Fout(LBi,LBj,Tindex))
                Finfo(8,ifield,ng)=Fmin
                Finfo(9,ifield,ng)=Fmax
              END IF
            ELSE
              IF (InpThread) THEN
                start(1)=Trec
                total(1)=1
                status=nf90_get_var(ncid, Vid, Fval, start, total)
              END IF
              Fpoint(Tindex,ifield,ng)=Fval(1)*Fscale(ifield,ng)
              Fmin=Fval(1)
              Fmax=Fval(1)
            END IF
            IF (status.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,70) TRIM(Vname(1,ifield)), Trec
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            END IF
            IF (Master) THEN
              IF (special) THEN
                WRITE (stdout,80) TRIM(Vname(2,ifield)), Fmin, Fmax
              ELSE
                lstr=SCAN(ncfile,'/',BACK=.TRUE.)+1
                lend=LEN_TRIM(ncfile)
                lvar=MIN(43,LEN_TRIM(Vname(2,ifield)))
                WRITE (stdout,90) Vname(2,ifield)(1:lvar),              &
     &                            Tval(1)*Tscale, ncfile(lstr:lend),    &
     &                            Trec, Tindex, Tmin, Tmax, Fmin, Fmax
              END IF
            END IF
            update=.TRUE.
          END IF
        END IF
!
!  Increment the local time variable "Tmono" by the interval between
!  snapshots. If the interval is negative, indicating cycling, add in
!  a cycle length.  Load time value (sec) into "Tintrp" which used
!  during interpolation between snapshots.
!
        IF (.not.Lonerec) THEN
          Tdelta=Vtime(Tindex,ifield,ng)-Vtime(3-Tindex,ifield,ng)
          IF (Liocycle.and.(Tdelta.lt.0.0_r8)) THEN
            Tdelta=Tdelta+Clength
          END IF
          Tmono=Tmono+Tdelta*day2sec
          Finfo(7,ifield,ng)=Tmono
          Tintrp(Tindex,ifield,ng)=Tmono
        END IF
      END IF
!
  10  FORMAT (/,' GET_2DFLD   - unable to find requested variable: ',a, &
     &        /,15x,'in input NetCDF ',a)
  20  FORMAT (/,' GET_2DFLD   - unable to open input NetCDF file: ',a)
  30  FORMAT (/,' GET_2DFLD   - error while reading attribute: ',a)
  40  FORMAT (/,' GET_2DFLD   - error while inquiring attributes for',  &
     &        ' variable: ',a)
  50  FORMAT (/,' GET_2DFLD   - error while inquiring dimensions for',  &
     &        ' variable: ',a) 
  60  FORMAT (/,' GET_2DFLD   - cannot inquire about time variable',    &
     &        ' from input NetCDF file.')
  70  FORMAT (/,' GET_2DFLD   - error while reading variable: ',a,2x,   &
     &        ' at TIME index = ',i4)
  80  FORMAT (3x,' GET_2DFLD   - ',a,/,19x,'(Min = ',1p,e15.8,0p,       &
     &        ' Max = ',1p,e15.8,0p,')')
  90  FORMAT (3x,' GET_2DFLD   - ',a,',',t64,'t = ',f12.4,/,19x,        &
     &        '(File: ',a,', Rec=',i4.4,', Index=',i1,')',/,19x,        &
     &        '(Tmin= ', f15.4, ' Tmax= ', f15.4,')',/, 19x,            &
     &        '(Min = ', 1p,e15.8,0p,' Max = ',1p,e15.8,0p,')')
      RETURN
      END SUBROUTINE get_2dfld
