      FUNCTION wrt_info (ng, model, ncid, wrtThread, ncname)
!
!svn $Id: wrt_info.F 537 2008-02-09 02:00:53Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2008 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine writes out information variables into requested        !
!  NetCDF file.                                                        !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng       Nested grid number.                                     !
!     model    Calling model identifier.                               !
!     ncid     NetCDF file ID.                                         !
!     ncname   NetCDF file name.                                       !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     wrt_info   Error flag (integer).                                 !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_biology
      USE mod_grid
      Use mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      logical, intent(in) :: wrtThread
      integer, intent(in) :: ng, model, ncid
      character (len=*), intent(in) :: ncname
!
!  Local variable declarations.
!
      logical :: Cgrid = .TRUE.
      integer :: LBi, UBi, LBj, UBj
      integer :: i, j, k, ibry, ilev, itrc, varid
      integer, dimension(2) :: start, total
      integer :: nf_fwrite2d
      integer :: wrt_info
      real(r8) :: scale
      real(r8), dimension(NT(ng)) :: nudg
      real(r8), dimension(NT(ng),4) :: Tobc
      character (len=1) :: char1
!
      LBi=LBOUND(GRID(ng)%h,DIM=1)
      UBi=UBOUND(GRID(ng)%h,DIM=1)
      LBj=LBOUND(GRID(ng)%h,DIM=2)
      UBj=UBOUND(GRID(ng)%h,DIM=2)
!
!-----------------------------------------------------------------------
!  Write out running parameters.
!-----------------------------------------------------------------------
!
      IF (wrtThread) THEN
!
!  Time stepping parameters.
!
        wrt_info=nf90_inq_varid(ncid, 'ntimes', varid)
        wrt_info=nf90_put_var(ncid, varid, ntimes(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'ntimes', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'ndtfast', varid)
        wrt_info=nf90_put_var(ncid, varid, ndtfast(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'ndtfast', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'dt', varid)
        wrt_info=nf90_put_var(ncid, varid, dt(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'dt', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'dtfast', varid)
        wrt_info=nf90_put_var(ncid, varid, dtfast(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'dtfast', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'dstart', varid)
        wrt_info=nf90_put_var(ncid, varid, dstart)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'dstart', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'nHIS', varid)
        wrt_info=nf90_put_var(ncid, varid, nHIS(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'nHIS', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        IF (ndefHIS(ng).gt.0) THEN
          wrt_info=nf90_inq_varid(ncid, 'ndefHIS', varid)
          wrt_info=nf90_put_var(ncid, varid, ndefHIS(ng))
          IF (wrt_info.ne.nf90_noerr) THEN
            WRITE (stdout,10) 'ndefHIS', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
        wrt_info=nf90_inq_varid(ncid, 'nRST', varid)
        wrt_info=nf90_put_var(ncid, varid, nRST(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'nRST', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'ntsAVG', varid)
        wrt_info=nf90_put_var(ncid, varid, ntsAVG(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'ntsAVG', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'nAVG', varid)
        wrt_info=nf90_put_var(ncid, varid, nAVG(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'nAVG', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        IF (ndefAVG(ng).gt.0) THEN
          wrt_info=nf90_inq_varid(ncid, 'ndefAVG', varid)
          wrt_info=nf90_put_var(ncid, varid, ndefAVG(ng))
          IF (wrt_info.ne.nf90_noerr) THEN
            WRITE (stdout,10) 'ndefAVG', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
!
!  Power-law shape filter parameters for time-averaging of barotropic
!  fields.
!
        wrt_info=nf90_inq_varid(ncid, 'Falpha', varid)
        wrt_info=nf90_put_var(ncid, varid, Falpha)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Falpha', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Fbeta', varid)
        wrt_info=nf90_put_var(ncid, varid, Fbeta)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Fbeta', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Fgamma', varid)
        wrt_info=nf90_put_var(ncid, varid, Fgamma)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Fgamma', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Horizontal mixing coefficients.
!
        wrt_info=nf90_inq_varid(ncid, 'tnu2', varid)
        start(1)=1
        total(1)=NT(ng)
        wrt_info=nf90_put_var(ncid, varid, tnu2(:,ng), start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'tnu2', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Background vertical mixing coefficients.
!
        wrt_info=nf90_inq_varid(ncid, 'Akt_bak', varid)
        start(1)=1
        total(1)=NT(ng)
        wrt_info=nf90_put_var(ncid, varid, Akt_bak(:,ng), start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Akt_bak', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Akv_bak', varid)
        wrt_info=nf90_put_var(ncid, varid, Akv_bak(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Akv_bak', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Drag coefficients.
!
        wrt_info=nf90_inq_varid(ncid, 'rdrg', varid)
        wrt_info=nf90_put_var(ncid, varid, rdrg(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'rdrg', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'rdrg2', varid)
        wrt_info=nf90_put_var(ncid, varid, rdrg2(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'rdrg2', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Zob', varid)
        wrt_info=nf90_put_var(ncid, varid, Zob(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Zob', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Zos', varid)
        wrt_info=nf90_put_var(ncid, varid, Zos(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Zos', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Nudging inverse time scales used in various tasks.
!
        wrt_info=nf90_inq_varid(ncid, 'Znudg', varid)
        wrt_info=nf90_put_var(ncid, varid, Znudg(ng)/sec2day)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Znudg', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'M2nudg', varid)
        wrt_info=nf90_put_var(ncid, varid, M2nudg(ng)/sec2day)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'M2nudg', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'M3nudg', varid)
        wrt_info=nf90_put_var(ncid, varid, M3nudg(ng)/sec2day)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'M3nudg', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Tnudg', varid)
        DO itrc=1,NT(ng)
          nudg(itrc)=Tnudg(itrc,ng)/sec2day
        END DO
        start(1)=1
        total(1)=NT(ng) 
        wrt_info=nf90_put_var(ncid, varid, nudg, start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Tnudg', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Equation of State parameters.
!
        wrt_info=nf90_inq_varid(ncid, 'rho0', varid)
        wrt_info=nf90_put_var(ncid, varid, rho0)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'rho0', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Slipperiness parameters.
!
        wrt_info=nf90_inq_varid(ncid, 'gamma2', varid)
        wrt_info=nf90_put_var(ncid, varid, gamma2(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'gamma2', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!-----------------------------------------------------------------------
!  Write out grid variables.
!-----------------------------------------------------------------------
!
!  Grid type switch.
!
        wrt_info=nf90_inq_varid(ncid, 'spherical', varid)
        WRITE (char1,'(l1)') spherical
        wrt_info=nf90_put_var(ncid, varid, char1)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'spherical', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Domain Length.
!
        wrt_info=nf90_inq_varid(ncid, 'xl', varid)
        wrt_info=nf90_put_var(ncid, varid, xl)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'xl', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'el', varid)
        wrt_info=nf90_put_var(ncid, varid, el)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'el', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  S-coordinate parameters.
!
        wrt_info=nf90_inq_varid(ncid, 'theta_s', varid)
        wrt_info=nf90_put_var(ncid, varid, theta_s(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'theta_s', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'theta_b', varid)
        wrt_info=nf90_put_var(ncid, varid, theta_b(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'theta_b', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Tcline', varid)
        wrt_info=nf90_put_var(ncid, varid, Tcline(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Tcline', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'hc', varid)
        wrt_info=nf90_put_var(ncid, varid, hc(ng))
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'hc', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  S-coordinate non-dimensional independent variables.
!
        wrt_info=nf90_inq_varid(ncid, 's_rho', varid)
        start(1)=1
        total(1)=N(ng)
        wrt_info=nf90_put_var(ncid, varid, SCALARS(ng)%sc_r(:),         &
     &                        start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 's_rho', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 's_w', varid)
        start(1)=1
        total(1)=N(ng)+1
        wrt_info=nf90_put_var(ncid, varid, SCALARS(ng)%sc_w(0:),        &
     &                        start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 's_w', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  S-coordinate non-dimensional stretching curves.
!
        wrt_info=nf90_inq_varid(ncid, 'Cs_r', varid)
        start(1)=1
        total(1)=N(ng)
        wrt_info=nf90_put_var(ncid, varid, SCALARS(ng)%Cs_r(:),         &
     &                        start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Cs_r', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
        wrt_info=nf90_inq_varid(ncid, 'Cs_w', varid)
        start(1)=1
        total(1)=N(ng)+1
        wrt_info=nf90_put_var(ncid, varid, SCALARS(ng)%Cs_w(0:),        &
     &                        start, total)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,10) 'Cs_w', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  User generic parameters.
!
        IF (Nuser.gt.0) THEN
          wrt_info=nf90_inq_varid(ncid, 'user', varid)
          start(1)=1
          total(1)=Nuser
          wrt_info=nf90_put_var(ncid, varid, user, start, total)
          IF (wrt_info.ne.nf90_noerr) THEN
            WRITE (stdout,10) 'user', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
      END IF
      IF (ncid.ne.ncFLTid(ng)) THEN
!
!  Bathymetry.
!
        scale=1.0_r8
        wrt_info=nf90_inq_varid(ncid, 'h', varid)
        IF (ncid.eq.ncSTAid(ng)) THEN
        ELSE
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % rmask(LBi,LBj),               &
     &                         GRID(ng) % h(LBi,LBj))
        END IF
        IF (wrtThread.and.(wrt_info.ne.nf90_noerr)) THEN
          WRITE (stdout,10) 'h', TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
!
!  Coriolis parameter.
!
        IF (ncid.ne.ncSTAid(ng)) THEN
          scale=1.0_r8
          wrt_info=nf90_inq_varid(ncid, 'f', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % rmask(LBi,LBj),               &
     &                         GRID(ng) % f(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'f', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
!
!  Curvilinear transformation metrics.
!
        IF (ncid.ne.ncSTAid(ng)) THEN
          scale=1.0_r8
          wrt_info=nf90_inq_varid(ncid, 'pm', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % rmask(LBi,LBj),               &
     &                         GRID(ng) % pm(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'pm', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'pn', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % rmask(LBi,LBj),               &
     &                         GRID(ng) % pn(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'pn', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
!
!  Grid coordinates of RHO-points.
!
        IF (spherical) THEN
          scale=1.0_r8
          wrt_info=nf90_inq_varid(ncid, 'lon_rho', varid)
          IF (ncid.eq.ncSTAid(ng)) THEN
          ELSE
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % rmask(LBi,LBj),             &
     &                           GRID(ng) % lonr(LBi,LBj))
          END IF
          IF (wrtThread.and.(wrt_info.ne.nf90_noerr)) THEN
            WRITE (stdout,10) 'lon_rho', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'lat_rho', varid)
          IF (ncid.eq.ncSTAid(ng)) THEN
          ELSE
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % rmask(LBi,LBj),             &
     &                           GRID(ng) % latr(LBi,LBj))
          END IF
          IF (wrtThread.and.(wrt_info.ne.nf90_noerr)) THEN
            WRITE (stdout,10) 'lat_rho', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        ELSE
          scale=1.0_r8
          wrt_info=nf90_inq_varid(ncid, 'x_rho', varid)
          IF (ncid.eq.ncSTAid(ng)) THEN
          ELSE
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % rmask(LBi,LBj),             &
     &                           GRID(ng) % xr(LBi,LBj))
          END IF
          IF (wrtThread.and.(wrt_info.ne.nf90_noerr)) THEN
            WRITE (stdout,10) 'x_rho', TRIM(ncname)
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'y_rho', varid)
          IF (ncid.eq.ncSTAid(ng)) THEN
          ELSE
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % rmask(LBi,LBj),             &
     &                           GRID(ng) % yr(LBi,LBj))
          END IF
          IF (wrtThread.and.(wrt_info.ne.nf90_noerr)) THEN
            WRITE (stdout,10) 'y_rho', TRIM(ncname)
            exit_flag=3
            RETURN
          END IF
        END IF
!
!  Grid coordinates of U-points.
!
        IF (spherical) THEN
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lon_u', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, u2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % umask(LBi,LBj),             &
     &                           GRID(ng) % lonu(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lon_u', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lat_u', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, u2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % umask(LBi,LBj),             &
     &                           GRID(ng) % latu(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lat_u', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        ELSE
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'x_u', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, u2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % umask(LBi,LBj),             &
     &                           GRID(ng) % xu(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'x_u', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'y_u', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, u2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % umask(LBi,LBj),             &
     &                           GRID(ng) % yu(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'y_u', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        END IF
!
!  Grid coordinates of V-points.
!
        IF (spherical) THEN
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lon_v', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, v2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % vmask(LBi,LBj),             &
     &                           GRID(ng) % lonv(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lon_v', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lat_v', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, v2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % vmask(LBi,LBj),             &
     &                           GRID(ng) % latv(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lat_v', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        ELSE
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'x_v', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, v2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % vmask(LBi,LBj),             &
     &                           GRID(ng) % xv(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'x_v', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'y_v', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, v2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % vmask(LBi,LBj),             &
     &                           GRID(ng) % yv(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'y_v', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        END IF
!
!  Grid coordinates of PSI-points.
!
        IF (spherical) THEN
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lon_psi', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, p2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % pmask(LBi,LBj),             &
     &                           GRID(ng) % lonp(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lon_p', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'lat_psi', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, p2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % vmask(LBi,LBj),             &
     &                           GRID(ng) % latp(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'lat_p', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        ELSE
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'x_psi', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, p2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % pmask(LBi,LBj),             &
     &                           GRID(ng) % xp(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'x_psi', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
          IF (ncid.ne.ncSTAid(ng)) THEN
            scale=1.0_r8
            wrt_info=nf90_inq_varid(ncid, 'y_psi', varid)
            wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, p2dvar,     &
     &                           LBi, UBi, LBj, UBj, scale,             &
     &                           GRID(ng) % pmask(LBi,LBj),             &
     &                           GRID(ng) % yp(LBi,LBj))
            IF (wrt_info.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,10) 'y_psi', TRIM(ncname)
              END IF
              exit_flag=3
              ioerror=wrt_info
              RETURN
            END IF
          END IF
        END IF
!
!  Masking fields at RHO-, U-, V-points, and PSI-points.
!
        IF (ncid.ne.ncSTAid(ng)) THEN
          scale=1.0_r8
          wrt_info=nf90_inq_varid(ncid, 'mask_rho', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, r2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % rmask(LBi,LBj),               &
     &                         GRID(ng) % rmask(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'mask_rho', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'mask_u', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, u2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % umask(LBi,LBj),               &
     &                         GRID(ng) % umask(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'mask_u', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'mask_v', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, v2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % vmask(LBi,LBj),               &
     &                         GRID(ng) % vmask(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'mask_v', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
          wrt_info=nf90_inq_varid(ncid, 'mask_psi', varid)
          wrt_info=nf_fwrite2d(ng, model, ncid, varid, 0, p2dvar,       &
     &                         LBi, UBi, LBj, UBj, scale,               &
     &                         GRID(ng) % pmask(LBi,LBj),               &
     &                         GRID(ng) % pmask(LBi,LBj))
          IF (wrt_info.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,10) 'mask_psi', TRIM(ncname)
            END IF
            exit_flag=3
            ioerror=wrt_info
            RETURN
          END IF
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Synchronize NetCDF file to disk to allow other processes to access
!  data immediately after it is written.
!-----------------------------------------------------------------------
!
      IF (wrtThread) THEN
        wrt_info=nf90_sync(ncid)
        IF (wrt_info.ne.nf90_noerr) THEN
          WRITE (stdout,20) TRIM(ncname)
          exit_flag=3
          ioerror=wrt_info
          RETURN
        END IF
      END IF
  10  FORMAT (/,' WRT_INFO - error while writing variable: ',a,/,       &
     &        12x,'into NetCDF file: ',a)
  20  FORMAT (/,' WRT_INFO - unable to synchronize NetCDF to ',         &
     &        'disk:',/,12x,a)
      RETURN
      END FUNCTION wrt_info
