      MODULE ice_advect_mod
!
!=======================================================================
!  Copyright (c) 2002 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine computes the advection of the ice tracer fields.       !
!  Currently, the only option is to use the MPDATA advection scheme.   !
!                                                                      !
!=======================================================================
!
      implicit none
      PRIVATE
      PUBLIC ice_advect
      CONTAINS
       SUBROUTINE ice_advect (ng, tile)
!
!*************************************************** W. Paul Budgell ***
!  Copyright (c) 2002 ROMS/TOMS Group                                  !
!************************************************** Hernan G. Arango ***
!                                                                      !
!  This subroutine performs advection of ice scalars using the         !
!  Smolarkiewicz second-order upwind scheme.                           !
!  Reference:                                                          !
!  Smolarkiewicz and Grabowski (1990)                                  !
!***********************************************************************
!
      USE mod_param
      implicit none
      integer, intent(in) :: ng, tile
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
      CALL wclock_on (ng, iNLM, 49)
! ---------------------------------------------------------------------
!  Advect the ice concentration.
! ---------------------------------------------------------------------
      CALL ice_advect_all_tile (ng, tile,                               &
     &                      LBi, UBi, LBj, UBj                          &
     &                      )
      CALL wclock_off (ng, iNLM, 49)
      RETURN
      END SUBROUTINE ice_advect
      SUBROUTINE ice_advect_all_tile (ng, tile,                         &
     &                      LBi, UBi, LBj, UBj                          &
     &                      )
      USE mod_param
      USE mod_grid
      USE mod_ocean
      USE mod_ice
      USE mod_forces
      USE mod_scalars
      USE mod_stepping
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
      USE aibc_mod, ONLY : aibc_tile
      USE hibc_mod, ONLY : hibc_tile
      USE hsnbc_mod, ONLY : hsnbc_tile
      USE tibc_mod, ONLY : tibc_tile
      USE sfwatbc_mod, ONLY : sfwatbc_tile
      USE ageicebc_mod, ONLY : ageicebc_tile
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
!  Local variables
!
      integer :: i, j
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % ai                                &
     &                      )
!
        CALL aibc_tile (ng, tile,                                       &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%ai)
!
! ---------------------------------------------------------------------
!  Advect the ice thickness.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % hi                                &
     &                      )
!
        CALL hibc_tile (ng, tile,                                       &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hi)
!
! ---------------------------------------------------------------------
!  Advect the snow thickness.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % hsn                               &
     &                      )
!
        CALL hsnbc_tile (ng, tile,                                      &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hsn)
!
! ---------------------------------------------------------------------
!  Advect the surface melt water.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % sfwat                             &
     &                      )
!
        CALL sfwatbc_tile (ng, tile,                                    &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     liold(ng), linew(ng),                        &
     &                     ICE(ng)%ui,                                  &
     &                     ICE(ng)%vi,                                  &
     &                     ICE(ng)%sfwat)
!
! ---------------------------------------------------------------------
!  Advect the interior ice temperature.
! ---------------------------------------------------------------------
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          ICE(ng)%enthalpi(i,j,liold(ng)) = ICE(ng)%hi(i,j,liold(ng))*  &
     &                                      ICE(ng)%ti(i,j,liold(ng))
          ICE(ng)%enthalpi(i,j,linew(ng)) = ICE(ng)%hi(i,j,linew(ng))*  &
     &                                      ICE(ng)%ti(i,j,linew(ng))
          IF(ICE(ng)%hi(i,j,liold(ng)).LE.min_h(ng)) THEN
            ICE(ng)%enthalpi(i,j,liold(ng)) = 0.0_r8
            ICE(ng)%ti(i,j,liold(ng)) = 0.0_r8
          END IF
        ENDDO
      ENDDO
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % enthalpi                          &
     &                      )
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          ICE(ng)%ti(i,j,linew(ng)) = ICE(ng)%enthalpi(i,j,linew(ng))/  &
       &                  MAX(ICE(ng)%hi(i,j,linew(ng)),1.0E-6_r8)
          IF (ICE(ng)%hi(i,j,linew(ng)).LE.min_h(ng)) THEN
            ICE(ng)%enthalpi(i,j,linew(ng)) = 0.0_r8
            ICE(ng)%ti(i,j,linew(ng)) = 0.0_r8
          END IF
        ENDDO
      ENDDO
!
        CALL tibc_tile (ng, tile,                                       &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng), min_h(ng),        &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hi,                             &
     &                          ICE(ng)%ti,                             &
     &                          ICE(ng)%enthalpi)
!
! ---------------------------------------------------------------------
!  Advect the ice age.
! ---------------------------------------------------------------------
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
	  ICE(ng)%hage(i,j,liold(ng)) = ICE(ng)%hi(i,j,liold(ng))*             &
     &                       ICE(ng)%ageice(i,j,liold(ng))
	  ICE(ng)%hage(i,j,linew(ng)) = ICE(ng)%hi(i,j,linew(ng))*             &
     &                       ICE(ng)%ageice(i,j,linew(ng))
          IF (ICE(ng)%hi(i,j,liold(ng)).LE.min_h(ng)) THEN
            ICE(ng)%hage(i,j,liold(ng)) = 0.0_r8
            ICE(ng)%ageice(i,j,liold(ng)) = 0.0_r8
          END IF
        ENDDO
      ENDDO
      CALL ice_advect_tile (ng, tile,                                   &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
     &                      GRID(ng) % rmask,                           &
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % hage                              &
     &                      )
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          ICE(ng)%ageice(i,j,linew(ng)) = ICE(ng)%hage(i,j,linew(ng))/  &
     &                  MAX(ICE(ng)%hi(i,j,linew(ng)),1.0E-6_r8)
          IF (ICE(ng)%hi(i,j,linew(ng)).LE.min_h(ng)) THEN
            ICE(ng)%hage(i,j,linew(ng)) = 0.0_r8
            ICE(ng)%ageice(i,j,linew(ng)) = 0.0_r8
          END IF
        ENDDO
      ENDDO
!
        CALL ageicebc_tile (ng, tile,                                   &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng), min_h(ng),        &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hi,                             &
     &                          ICE(ng)%ageice,                         &
     &                          ICE(ng)%hage)
!
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%ai(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%hi(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%hsn(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%sfwat(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%ti(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%enthalpi(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%ageice(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, tile,                               &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%hage(:,:,linew(ng)))
      RETURN
      END SUBROUTINE ice_advect_all_tile
!
!==========================================================================!
      SUBROUTINE ice_advect_tile (ng, tile,                             &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        nrhs, linew, liold, liunw,                &
     &                        rmask,                                    &
     &                        on_u, om_v, omn,                          &
     &                        ui, vi, scr)
!==========================================================================!
      USE mod_param
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: nrhs, linew, liold, liunw
      real(r8), intent(in) :: rmask(LBi:,LBj:)
      real(r8), intent(in) :: on_u(LBi:,LBj:)
      real(r8), intent(in) :: om_v(LBi:,LBj:)
      real(r8), intent(in) :: omn(LBi:,LBj:)
      real(r8), intent(in) :: ui(LBi:,LBj:,:)
      real(r8), intent(in) :: vi(LBi:,LBj:,:)
      real(r8), intent(inout) :: scr(LBi:,LBj:,:)
!
! Local variable definitions
!
      integer :: Imin, Imax, Jmin, Jmax
      integer :: i, j
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: ar
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: aflxu
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: aflxv
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: aif
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: FX
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: FE
      real(r8), parameter :: epsil = 1.0E-15_r8
      real(r8), parameter :: add = 3.0E+3_r8
      real(r8) :: fakt1
      real(r8) :: fakt2
      real(r8) :: aim1
      real(r8) :: ajm1
      real(r8) :: aim1jm1u
      real(r8) :: aim1jm1v
      real(r8) :: ajp1
      real(r8) :: aim1jp1
      real(r8) :: aip1
      real(r8) :: aip1jm1
      real(r8) :: Cu_crss, Cu
      real(r8) :: rateu
      real(r8) :: ratev
      real(r8) :: rateyiu
      real(r8) :: ratexiv
      real(r8) :: ratiou
      real(r8) :: ratiov
      real(r8) :: ratioyiu
      real(r8) :: ratioxiv
      real(r8) :: uiv
      real(r8) :: viu
      real(r8) :: uspeed
      real(r8) :: vspeed
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
      Imin=Istr
      Imax=Iend
      Jmin=Jstr
      Jmax=Jend
!
! upstream:
!
      DO j=Jmin,Jmax
        DO i=Imin,Imax+1
          aflxu(i,j)=on_u(i,j)*                                         &
     &          (max(0.0_r8,ui(i,j,liunw))*scr(i-1,j,liold)             &
     &          +min(0.0_r8,ui(i,j,liunw))*scr(i,j,liold))
        END DO
      END DO
      DO j=Jmin,Jmax+1
        DO i=Imin,Imax
          aflxv(i,j)=om_v(i,j)*                                         &
     &          (max(0.0_r8,vi(i,j,liunw))*scr(i,j-1,liold)             &
     &          +min(0.0_r8,vi(i,j,liunw))*scr(i,j,liold))
!
        END DO
      END DO
!
! step number 1 in mpdata:
!
      DO j=Jmin,Jmax
        DO i=Imin,Imax
!
          ar(i,j)=1.0_r8/omn(i,j)
          aif(i,j)=(scr(i,j,liold)-dtice(ng)*(aflxu(i+1,j)-aflxu(i,j)   &
     &        +aflxv(i,j+1)-aflxv(i,j))*ar(i,j))
          aif(i,j) = aif(i,j)*rmask(i,j)
        END DO
      END DO
!
! set values at the open boundaries
!
!
! mask ???
!
      DO j=Jstr,Jend
        DO i=Istr,Iend
            scr(i,j,linew) = aif(i,j)
        END DO
      END DO
!
      RETURN
      END SUBROUTINE ice_advect_tile
      END MODULE ice_advect_mod
