      MODULE analytical_mod
!
!svn $Id: analytical.F 526 2008-01-29 01:06:18Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2008 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!   PACKAGE:                                                 !
!                                                                      !
!  This package is used to provide various analytical fields to the    !
!  model when appropriate.                                             !
!                                                                      !
!=======================================================================
!
      implicit none
      CONTAINS
      SUBROUTINE ana_biology (ng, tile, model)
!
!=======================================================================
!                                                                      !
!  This routine sets initial conditions for biological tracer fields   !
!  using analytical expressions.                                       !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_ncparam
      USE mod_ocean
      USE mod_grid
!
! Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL ana_biology_tile (ng, tile, model,                           &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       GRID(ng) % z_r,                            &
     &                       GRID(ng) % h,                              &
     &                      OCEAN(ng) % bt,                             &
     &                       OCEAN(ng) % t)
!
! Set analytical header file name used.
!
      IF (Lanafile) THEN
        ANANAME( 1)="ROMS/Functionals/ana_biology.h"
      END IF
      RETURN
      END SUBROUTINE ana_biology
!
!***********************************************************************
      SUBROUTINE ana_biology_tile (ng, tile, model,                     &
     &                             LBi, UBi, LBj, UBj,                  &
     &                             z_r, h,                              &
     &                             bt,                                  &
     &                             t)
!***********************************************************************
!
      USE mod_param
      USE mod_biology
      USE mod_scalars
      USE mod_grid
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
      real(r8), intent(in) :: z_r(LBi:,LBj:,:)
      real(r8), intent(in) :: h(LBi:,LBj:)
        real(r8), intent(inout) :: bt(LBi:,LBj:,:,:,:)
      real(r8), intent(inout) :: t(LBi:,LBj:,:,:,:)
!
!  Local variable declarations.
!
      integer :: i, is, itrc, j, k
      real(r8) :: var1, var2, var3, var4, var5, var6, var7
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1,N(ng)) :: biod
      real(r8), dimension(NT(ng)) :: deepval
      real(r8), dimension(NT(ng)) :: loval
      real(r8) :: FeSurf, FeDeep
      real(r8), parameter :: eps = 1.0E-20_r8
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!=======================================================================
!                                                                      !
!  This routine sets initial conditions for biological tracer fields   !
!  using analytical expressions for line 8 in GOA for GOANPZ.                                       !
!                                                                      !
!=======================================================================
!
!
! Make the curves.  First come parameters that are constant through 
! entire water column.
!
      deepval(iNO3) = 30.0_r8             !was 30.0_r8    
      deepval(iNH4) = eps
      deepval(iPhS) = eps
      deepval(iPhL) = eps
      deepval(iMZS) = eps
      deepval(iMZL) = eps
      deepval(iCop) = 0.00863_r8
      deepval(iNCa) = eps
      deepval(iEup) = 0.08_r8
      deepval(iDet) = eps             !was eps/60.0_r8
      deepval(iJel) = eps 
         do j=JstrR,JendR
            do i=IstrR,IendR
              do k=1,N(ng)
               t(i,j,k,1,iNO3) =09.0_r8  !was 16.0_r8
               t(i,j,k,1,iNH4) =  2.0_r8 !was 2
               t(i,j,k,1,iDet) =  eps
	       t(i,j,k,1,iJel) =  0.0003_r8
               biod(i,j,k) = -1 * ( z_r(i,j,k) + 2.5_r8 )
!               st(i,j,k,1,iProd) =  0.0_r8
!               st(i,j,k,1,iStat1) = 0.0_r8
!               st(i,j,k,1,iStat2) = 0.0_r8
!               st(i,j,k,1,iStat3) = 0.0_r8
!               st(i,j,k,1,iStat4) = 0.0_r8
!               st(i,j,k,1,iStat5) = 0.0_r8
!               st(i,j,k,1,iStat6) = 0.0_r8
!               st(i,j,k,1,iStat7) = 0.0_r8
!               st(i,j,k,1,iStat8) = 0.0_r8
!               st(i,j,k,1,iStat9) = 0.0_r8
            enddo
           do k=1,NBL(ng)
                 bt(i,j,k,1,iBen) = 8000.0_r8    !was eps iBen
		 bt(i,j,k,1,iBenDet) = 000.0_r8    !was eps iBenDet
            enddo
         enddo
      enddo
!
! PS - a combination of 2 curves: 2nd order polynomial curve to create
! a subsurface maximum at ~20 m and an exponentially decreasing curve to
! assure values aren't negative at depth.
!
!5      var1 = 25.58_r8
!5     var2 = -0.250_r8 / (5.0_r8**2)
!5     var3 =  0.008_r8 / (5.0_r8**3)
!5      var4 = 3.82_r8 * 5.0_r8
!5      var5 = 75.0_r8
!5      var6 = var5 - var4 
!5      var7 = var1 + var2*(var6**2) + var3*(var6**3)
!5      do i=IstrR,IendR
!5        do j=JstrR,JendR
!5          do k=1,N(ng)
!5            if ( biod(i,j,k) .le. var5 ) then
!5              var6 = biod(i,j,k) - var4
!5              t(i,j,k,1,iPhS) =                                       &
!5     &          var1 + var2*(var6**2) + var3*(var6**3)
!5            else
!5              t(i,j,k,1,iPhS) =                                       &
!5     &          var7 * exp( ( -1.0_r8*biod(i,j,k) + var5 ) / 5.0_r8 )
!5            endif
!5          enddo
!5        enddo
!5      enddo
!
! PL - exponentially decreasing with depth.
!
       t(i,j,k,1,iPhL) =t(i,j,k,1,iPhS)
      var1 = 8.25_r8
      var2 = 0.322_r8 / 5.0_r8
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=1,N(ng)
             t(i,j,k,1,iPhL) =t(i,j,k,1,iPhS)
            t(i,j,k,1,iPhL) = var1 * exp( -1.0_r8 * var2 * biod(i,j,k) )
	    t(i,j,k,1,iPhS) = t(i,j,k,1,iPhL) 
          enddo
        enddo
      enddo
!
! Microzooplankton, from Howell-Kubler, 1996
! approximated with a straight line and an exponentially decreasing
! curve to assure values aren't negative at depth.  Curves meet ~60m
!
      var1 = 3.1714_r8
      var2 = -0.1865_r8 / 5.0_r8
      var3 = 60.0_r8
      var4 = 0.5_r8
      var5 = var1 + var2 * var3
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=1,N(ng)
            if ( biod(i,j,k) .le. var3 ) then
              t(i,j,k,1,iMZL) = var1 + var2 * biod(i,j,k)
            else
              t(i,j,k,1,iMZL) = var4 +                                  &
     &          ( var5 - var4 ) * exp( ( var3 - biod(i,j,k) ) / 5.0_r8 )
            endif
            t(i,j,k,1,iMZS) = t(i,j,k,1,iMZL)
          enddo
        enddo
      enddo
!
! Pseudocalanus Copepods - from Shelikof data via Shelikof NPZ
! Step at ~ 50 m
!
      var1 = pi / 2.0_r8
!      var2 = 2.876_r8 / pi
       var2 = 0.5_r8 / pi
      var3 = 5.0_r8 / 5.0_r8
      var4 = 9.151_r8 * 5.0_r8
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=1,N(ng)
            var5 = var3 * ( biod(i,j,k) - var4 )
            t(i,j,k,1,iCop) = var1 - var2 * atan( var5 )
          enddo
        enddo
      enddo
!
! Neocalanus, from Shelikof NPZ. Step at ~ 30m
!
!g      var1 = 4.0_r8
!g      var2 = 1.3_r8
!g     var3 = 5.0_r8 / 5.0_r8
!g      var4 = 5.2_r8 * 5.0_r8
!g      var6 =200.0_r8
!g      do i=IstrR,IendR
!g        do j=JstrR,JendR
!g          do k=1,N(ng)
!set neocalanus to zero on the shelf.
!          if ( biod(i,j,k) .le. var6 ) then
!          t(i,j,k,1,iNCa) =0.0_r8
!          else
!g            var5 = var3 * ( biod(i,j,k) - var4 )
!g            t(i,j,k,1,iNCa) = var1 - var2 * atan( var5 )
!           endif
!g          enddo
!g        enddo
!g      enddo
  var1 = pi / 2.0_r8
!      var2 = 2.876_r8 / pi
       var2 = 0.5_r8 / pi
      var3 = 5.0_r8 / 5.0_r8
      var4 = 9.151_r8 * 5.0_r8
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=1,N(ng)
            var5 = var3 * ( biod(i,j,k) - var4 )
!            t(i,j,k,1,iNCa) = var1 - var2 * atan( var5 )
	    t(i,j,k,1,iNCa) =  t(i,j,k,1,iCop)
          enddo
        enddo
      enddo
!
! Euphausiids, Wild guesses. Step at ~ 30m
!
!g      var1 = 1.78_r8
!g      var2 = 0.8_r8
!g      var3 = 5.0_r8 / 5.0_r8
!g      var4 = 5.2_r8 * 5.0_r8
!g      do i=IstrR,IendR
!g        do j=JstrR,JendR
!g          do k=1,N(ng)
!g            var5 = var3 * ( biod(i,j,k) - var4)
!g            t(i,j,k,1,iEup) =  var1 - var2 * atan( var5 )
!g          enddo
!g        enddo
!g      enddo
!
 var1 = pi / 2.0_r8
  var2 = 0.5_r8 / pi
!      var2 = 2.876_r8 / pi
      var3 = 5.0_r8 / 5.0_r8
      var4 = 9.151_r8 * 5.0_r8
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=1,N(ng)
            var5 = var3 * ( biod(i,j,k) - var4 )
!            t(i,j,k,1,iEup) = var1 - var2 * atan( var5 )
             t(i,j,k,1,iEup) = 0.05_r8
          enddo
        enddo
      enddo
! Iron - linear from surface value to value at 100m and increase onshore
      do i=IstrR,IendR
        do j=JstrR,JendR
          var1 = MAX(0._r8,MIN(1._r8,(h(i,j)-Feinh)/(Feoffh-Feinh)))
          FeSurf = Feinlo + var1*(Feofflo-Feinlo)
          FeDeep = Feinhi + var1*(Feoffhi-Feinhi)
          var1 = (FeDeep-FeSurf) / 100._r8
          do k=1,N(ng)
            t(i,j,k,1,iFe) = MIN(FeDeep, FeSurf - z_r(i,j,k)*var1)
          enddo
        enddo
      enddo
!
! Concentrations of everything below 100m - i.e. below
! depths where calculations are performed.  Have linear slope
! between values above and below.
! Iron deep values have already been determined.
!
      do i=IstrR,IendR
        do j=JstrR,JendR
          do k=N(ng),1,-1
            if ( biod(i,j,k) .gt. 120.0_r8 ) then       !was 120
              t(i,j,k,1,iNO3) = deepval(iNO3)
              t(i,j,k,1,iNH4) = deepval(iNH4)
              t(i,j,k,1,iPhS) = deepval(iPhS)
              t(i,j,k,1,iPhL) = deepval(iPhL)
              t(i,j,k,1,iMZS) = deepval(iMZS)
              t(i,j,k,1,iMZL) = deepval(iMZL)
              t(i,j,k,1,iCop) = deepval(iCop)
              t(i,j,k,1,iNCa) = deepval(iNCa)
              t(i,j,k,1,iEup) = deepval(iEup)
              t(i,j,k,1,iDet) = deepval(iDet)
	      t(i,j,k,1,iJel) = deepval(iJel)
!              st(i,j,k,1,iProd) = 0.0_r8
!              st(i,j,k,1,iStat1)  = 0.0_r8
!              st(i,j,k,1,iStat2) = 0.0_r8
!              st(i,j,k,1,iStat3) =0.0_r8
!              st(i,j,k,1,iStat4)  = 0.0_r8
!              st(i,j,k,1,iStat5) = 0.0_r8
!              st(i,j,k,1,iStat6) =0.0_r8
!              st(i,j,k,1,iStat7)  = 0.0_r8
!              st(i,j,k,1,iStat8) = 0.0_r8
!              st(i,j,k,1,iStat9) =0.0_r8
            else if ( biod(i,j,k) .gt. 100.0_r8 .and.                   &
     &                biod(i,j,k) .le. 120.0_r8) then
              var1 = ( 100.0_r8 - biod(i,j,k) ) / ( 100.0_r8-120.0_r8 )
              t(i,j,k,1,iNO3) = loval(iNO3) +                           &
     &                          ( deepval(iNO3) - loval(iNO3) ) * var1
              t(i,j,k,1,iNH4) = loval(iNH4) +                           &
     &                          ( deepval(iNH4) - loval(iNH4) ) * var1
              t(i,j,k,1,iPhS) = loval(iPhS) +                           &
     &                          ( deepval(iPhS) - loval(iPhS) ) * var1
              t(i,j,k,1,iPhL) = loval(iPhL) +                           &
     &                          ( deepval(iPhL) - loval(iPhL) ) * var1
              t(i,j,k,1,iMZS) = loval(iMZS) +                           &
     &                          ( deepval(iMZS) - loval(iMZS) ) * var1
              t(i,j,k,1,iMZL) = loval(iMZL) +                           &
     &                          ( deepval(iMZL) - loval(iMZL) ) * var1
              t(i,j,k,1,iCop) = loval(iCop) +                           &
     &                          ( deepval(iCop) - loval(iCop) ) * var1
              t(i,j,k,1,iNCa) = loval(iNCa) +                           &
     &                          ( deepval(iNCa) - loval(iNCa) ) * var1
              t(i,j,k,1,iEup) = loval(iEup) +                           &
     &                          ( deepval(iEup) - loval(iEup) ) * var1
              t(i,j,k,1,iDet) = loval(iDet) +                           &
     &                          ( deepval(iDet) - loval(iDet) ) * var1
              t(i,j,k,1,iJel) = loval(iJel) +                           &
                                ( deepval(iJel) - loval(iJel) ) * var1
!              st(i,j,k,1,iProd) = 0.0_r8
!              st(i,j,k,1,iStat1) = 0.0_r8 
!              st(i,j,k,1,iStat2) = 0.0_r8
!              st(i,j,k,1,iStat3) = 0.0_r8
!              st(i,j,k,1,iStat4) = 0.0_r8
!              st(i,j,k,1,iStat5) = 0.0_r8
!              st(i,j,k,1,iStat6) = 0.0_r8
!              st(i,j,k,1,iStat7) = 0.0_r8
!              st(i,j,k,1,iStat8) = 0.0_r8
!              st(i,j,k,1,iStat9) = 0.0_r8
            else
              loval(iNO3) = t(i,j,k,1,iNO3)
              loval(iNH4) = t(i,j,k,1,iNH4)
              loval(iPhS) = t(i,j,k,1,iPhS)
              loval(iPhL) = t(i,j,k,1,iPhL)
              loval(iMZS) = t(i,j,k,1,iMZS)
              loval(iMZL) = t(i,j,k,1,iMZL)
              loval(iCop) = t(i,j,k,1,iCop)
              loval(iNCa) = t(i,j,k,1,iNCa)
              loval(iEup) = t(i,j,k,1,iEup)
              loval(iDet) = t(i,j,k,1,iDet)
	      loval(iJel) = t(i,j,k,1,iJel)
!              loval(iProd) = st(i,j,k,1,iProd)
!              loval(iStat1) = st(i,j,k,1,iStat1)
!              loval(iStat2) = st(i,j,k,1,iStat2)
!              loval(iStat3) = st(i,j,k,1,iStat3)
!              loval(iStat4) = st(i,j,k,1,iStat4)
!              loval(iStat5) = st(i,j,k,1,iStat5)
!              loval(iStat6) = st(i,j,k,1,iStat6)
!              loval(iStat7) = st(i,j,k,1,iStat7)
!              loval(iStat8) = st(i,j,k,1,iStat8)
!              loval(iStat9) = st(i,j,k,1,iStat9)
            endif
          enddo
        enddo
      enddo
!
! Check for size, set other time index, and periodic BC's
!
      do i=IstrR,IendR
         do j=JstrR,JendR
            do k=1,N(ng)
               DO is=1,NBT
                  itrc=idbio(is)
                  t(i,j,k,1,itrc) = MAX(t(i,j,k,1,itrc),eps)
                  t(i,j,k,2,itrc) = t(i,j,k,1,itrc)
               enddo
!               DO is=1,MST
!                  itrc=idStat(is)
!                  st(i,j,k,1,itrc) = MAX(st(i,j,k,1,itrc),0.0_r8)
!                  st(i,j,k,2,itrc) = st(i,j,k,1,itrc)
!              enddo
           enddo
        enddo
     enddo
      do i=IstrR,IendR
         do j=JstrR,JendR
            do k=1,NBL(ng)
               DO is=1,NBEN
                itrc=idben(is)
                 bt(i,j,k,1,itrc) = MAX(bt(i,j,k,1,itrc),eps)
                 bt(i,j,k,2,itrc) = bt(i,j,k,1,itrc)
		enddo
            enddo
	  enddo
      enddo
      RETURN
      END SUBROUTINE ana_biology_tile
      SUBROUTINE ana_btflux (ng, tile, model, itrc)
!
!=======================================================================
!                                                                      !
!  This routine sets kinematic bottom flux of tracer type variables    !
!  (tracer units m/s).                                                 !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_forces
      USE mod_ncparam
!
! Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model, itrc
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL ana_btflux_tile (ng, tile, model, itrc,                      &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      FORCES(ng) % btflx)
!
! Set analytical header file name used.
!
      IF (Lanafile) THEN
        ANANAME( 3)="ROMS/Functionals/ana_btflux.h"
      END IF
      RETURN
      END SUBROUTINE ana_btflux
!
!***********************************************************************
      SUBROUTINE ana_btflux_tile (ng, tile, model, itrc,                &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            btflx)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model, itrc
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
      real(r8), intent(inout) :: btflx(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set kinematic bottom heat flux (degC m/s) at horizontal RHO-points.
!-----------------------------------------------------------------------
!
      IF (itrc.eq.itemp) THEN
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          END DO
        END DO
!
!-----------------------------------------------------------------------
!  Set kinematic bottom salt flux (m/s) at horizontal RHO-points,
!  scaling by bottom salinity is done elsewhere.
!-----------------------------------------------------------------------
!
      ELSE IF (itrc.eq.isalt) THEN
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          END DO
        END DO
!
!-----------------------------------------------------------------------
!  Set kinematic bottom flux (T m/s) of passive tracers, if any.
!-----------------------------------------------------------------------
!
      ELSE
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          END DO
        END DO
      END IF
      RETURN
      END SUBROUTINE ana_btflux_tile
      SUBROUTINE ana_m2clima (ng, tile, model)
!
!=======================================================================
!                                                                      !
!  This routine sets analytical 2D momentum climatology fields.        !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_clima
      USE mod_ncparam
!
! Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL ana_m2clima_tile (ng, tile, model,                           &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       CLIMA(ng) % ubarclm,                       &
     &                       CLIMA(ng) % vbarclm)
!
! Set analytical header file name used.
!
      IF (Lanafile) THEN
        ANANAME(11)="Apps/AddBen/ana_m2clima.h"
      END IF
      RETURN
      END SUBROUTINE ana_m2clima
!
!***********************************************************************
      SUBROUTINE ana_m2clima_tile (ng, tile, model,                     &
     &                             LBi, UBi, LBj, UBj,                  &
     &                             ubarclm, vbarclm)
!***********************************************************************
!
      USE mod_param
!
      USE exchange_2d_mod
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
      real(r8), intent(out) :: ubarclm(LBi:,LBj:)
      real(r8), intent(out) :: vbarclm(LBi:,LBj:)
!
!  Local variable declarations.
!
      integer :: i, j
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set 2D momentum climatology.
!-----------------------------------------------------------------------
!
      DO j=JstrR,JendR
        DO i=Istr,IendR
          ubarclm(i,j)=0.0_r8
        END DO
      END DO
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          vbarclm(i,j)=0.0_r8
        END DO
      END DO
      CALL exchange_u2d_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        ubarclm)
      CALL exchange_v2d_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        vbarclm)
      RETURN
      END SUBROUTINE ana_m2clima_tile
      SUBROUTINE ana_nudgcoef (ng, tile, model)
!
!=======================================================================
!                                                                      !
!  This routine set nudging coefficients time-scales (1/s).            !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_ncparam
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
!
!  Local variable declarations.
!
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL ana_nudgcoef_tile (ng, tile, model,                          &
     &                        LBi, UBi, LBj, UBj)
!
! Set analytical header file name used.
!
      IF (Lanafile) THEN
        ANANAME(16)="ROMS/Functionals/ana_nudgcoef.h"
      END IF
      RETURN
      END SUBROUTINE ana_nudgcoef
!
!***********************************************************************
      SUBROUTINE ana_nudgcoef_tile (ng, tile, model,                    &
     &                              LBi, UBi, LBj, UBj)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_boundary
      USE mod_clima
      USE mod_grid
      USE mod_ncparam
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
!  Local variable declarations.
!
      integer :: Iwrk, i, itrc, j
      real(r8) :: cff1, cff2, cff3
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: wrk
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set up nudging towards data time-scale coefficients (1/s).
!-----------------------------------------------------------------------
!
!  Initialize.
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          wrk(i,j)=0.0_r8
        END DO
      END DO
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          CLIMA(ng)%M2nudgcof(i,j)=M2nudg(ng)
        END DO
      END DO
      DO itrc=1,NT(ng)
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            CLIMA(ng)%Tnudgcof(i,j,itrc)=Tnudg(itrc,ng)
          END DO
        END DO
      END DO
!
!-----------------------------------------------------------------------
!  Set nudging coefficients (1/s) for passive/active (outflow/inflow)
!  open boundary conditions.  Weak nudging is expected in passive
!  outflow conditions and strong nudging is expected in active inflow
!  conditions.  Notice that interior nudging coefficient defined
!  above are zero out when boundary condition nudging.  The USER needs
!  to adapt this to his/her application!
!-----------------------------------------------------------------------
!
!  Free-surface nudging coefficients.
!
!
!  2D momentum nudging coefficients.
!
!
!  Tracers nudging coefficients.
!
!
!  3D momentum nudging coefficients.
!
      RETURN
      END SUBROUTINE ana_nudgcoef_tile
      SUBROUTINE ana_stflux (ng, tile, model, itrc)
!
!=======================================================================
!                                                                      !
!  This routine sets kinematic surface flux of tracer type variables   !
!  "stflx" (tracer units m/s) using analytical expressions.            !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_forces
      USE mod_ncparam
!
! Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model, itrc
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL ana_stflux_tile (ng, tile, model, itrc,                      &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      FORCES(ng) % srflx,                         &
     &                      FORCES(ng) % stflx)
!
! Set analytical header file name used.
!
      IF (Lanafile) THEN
        ANANAME(31)="ROMS/Functionals/ana_stflux.h"
      END IF
      RETURN
      END SUBROUTINE ana_stflux
!
!***********************************************************************
      SUBROUTINE ana_stflux_tile (ng, tile, model, itrc,                &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            srflx,                                &
     &                            stflx)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, model, itrc
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
      real(r8), intent(in) :: srflx(LBi:,LBj:)
      real(r8), intent(inout) :: stflx(LBi:,LBj:,:)
!
!  Local variable declarations.
!
      integer :: i, j
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!-----------------------------------------------------------------------
!  Set kinematic surface heat flux (degC m/s) at horizontal
!  RHO-points.
!-----------------------------------------------------------------------
!
      IF (itrc.eq.itemp) THEN
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            stflx(i,j,itrc)=0.0_r8
          END DO
        END DO
!
!-----------------------------------------------------------------------
!  Set kinematic surface freshwater flux (m/s) at horizontal
!  RHO-points, scaling by surface salinity is done in STEP3D.
!-----------------------------------------------------------------------
!
      ELSE IF (itrc.eq.isalt) THEN
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            stflx(i,j,itrc)=0.0_r8
          END DO
        END DO
!
!-----------------------------------------------------------------------
!  Set kinematic surface flux (T m/s) of passive tracers, if any.
!-----------------------------------------------------------------------
!
      ELSE
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            stflx(i,j,itrc)=0.0_r8
          END DO
        END DO
      END IF
      CALL exchange_r2d_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        stflx(:,:,itrc))
      RETURN
      END SUBROUTINE ana_stflux_tile
      END MODULE analytical_mod
