      SUBROUTINE get_state (ng, model, msg, ncname, IniRec, Tindex)
!
!svn $Id: get_state.F 537 2008-02-09 02:00:53Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2008 The ROMS/TOMS Group                         !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine reads in requested model state from specified NetCDF   !
!  file. It is usually used to read initial conditions.                !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     msg        Message index for Mstate.                             !
!     ncname     Nonlinear initial conditions NetCDF file name.        !
!     IniRec     Nonlinear initial conditions time record to read.     !
!     Tindex     State variable time index to intialize.               !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_grid
      USE mod_iounits
      USE mod_mixing
      USE mod_ncparam
      USE mod_netcdf
      USE mod_ocean
      USE mod_scalars
      USE mod_stepping
      USE mod_strings
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, msg, Tindex
      integer, intent(inout) :: IniRec
      character (len=*), intent(in) :: ncname
!
!  Local variable declarations.
!
      logical :: Perfect2D, Perfect3D
      logical, dimension(NV) :: get_var, have_var
      integer :: LBi, UBi, LBj, UBj
      integer :: InpRec, gtype, i, itrc, lstr, lend
      integer :: Nrec, ncINPid, nvatts, nvdim, status, varid, vartype
      integer :: Vsize(4), start(4), total(4), vdimids(5)
      integer :: nf_fread2d
      integer :: nf_fread3d
      integer :: nf_fread4d
      real(r8), parameter :: Fscl = 1.0_r8
      real(r8) :: Fmax, Fmin, Tmax, time_scale
      real(r8) :: INPtime(1)
      character (len=6 ) :: string
      character (len=15) :: attnam, tvarnam
      character (len=40) :: tunits
!
!-----------------------------------------------------------------------
!  Determine variables to read and their availability.
!-----------------------------------------------------------------------
!
!  Set model identification string.
!
      IF (model.eq.iNLM.or.(model.eq.0)) THEN
        string=' NLM: '                       ! nonlinear model, restart
      ELSE IF (model.eq.iTLM) THEN
        string=' TLM: '                       ! tangent linear model
      ELSE IF (model.eq.iRPM) THEN
        string=' RPM: '                       ! representer model
      ELSE IF (model.eq.iADM) THEN
        string=' ADM: '                       ! adjoint model
      ELSE IF (model.eq.5) THEN
        string=' NRM: '                       ! correlations norm factor
      ELSE IF (model.eq.6) THEN
        string=' STD: '                       ! standard deviations
      ELSE IF (model.eq.7) THEN
        string=' FRC: '                       ! impulse forcing
      END IF
!
!  Set switch to process variables for nonlinear model perfect restart.
!
      Perfect2D=.FALSE.
      Perfect3D=.FALSE.
      PerfectRST(ng)=Perfect2d.or.Perfect3d
!
!  Determine variables to read.
!
      CALL checkvars (ng, model, ncname, string, Nrec, NV, tvarnam,     &
     &                get_var, have_var)
!
!  Set Vsize to zero to deactivate interpolation of input data to model
!  grid in "nf_fread2d" and "nf_fread3d".
!
      DO i=1,4
        Vsize(i)=0
      END DO
!
!-----------------------------------------------------------------------
!  Open input NetCDF file and check time variable.
!-----------------------------------------------------------------------
!
      LBi=LBOUND(GRID(ng)%h,DIM=1)
      UBi=UBOUND(GRID(ng)%h,DIM=1)
      LBj=LBOUND(GRID(ng)%h,DIM=2)
      UBj=UBOUND(GRID(ng)%h,DIM=2)
!
!  Open input NetCDF file.
!
      IF (InpThread) THEN
        status=nf90_open(TRIM(ncname), nf90_nowrite, ncINPid)
        IF (status.ne.nf90_noerr) THEN
          WRITE (stdout,20) string, TRIM(ncname)
          exit_flag=2
          ioerror=status
          RETURN
        END IF
!
!  If using the latest time record from input NetCDF file as the
!  initialization record, find time record index to read and read
!  in model time.
!
        status=nf90_inq_varid(ncINPid, TRIM(tvarnam), varid)
        IF (LastRec(ng)) THEN
          Tmax=-1.0_r8
          DO i=1,tsize
            start(1)=i
            total(1)=1
            status=nf90_get_var(ncINPid, varid, INPtime, start, total)
            IF (status.ne.nf90_noerr) THEN
              WRITE (stdout,30) string, TRIM(tvarnam), i, TRIM(ncname)
              exit_flag=2
              ioerror=status
              RETURN
            END IF
            IF (INPtime(1).gt.Tmax) THEN
              Tmax=INPtime(1)
              IniRec=i
            END IF
          END DO
          INPtime(1)=Tmax
          InpRec=IniRec
        ELSE
          IF ((IniRec.ne.0).and.(IniRec.gt.tsize)) THEN
            WRITE (stdout,40) string, IniRec, TRIM(ncname), tsize
            exit_flag=2
            RETURN
          END IF
          IF (IniRec.ne.0) THEN
            InpRec=IniRec
          ELSE
            InpRec=1
          END IF
          start(1)=InpRec
          total(1)=1
          status=nf90_get_var(ncINPid, varid, INPtime, start, total)
          IF (status.ne.nf90_noerr) THEN
            WRITE (stdout,30) string, TRIM(tvarnam), InpRec,            &
     &                        TRIM(ncname)
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
!
!  Inquire time units. Set local time scale.
!
        time_scale=0.0_r8
        status=nf90_inquire_variable(ncINPid, varid, tvarnam, vartype,  &
     &                               nvdim, vdimids, nvatts)
        IF (status.eq.nf90_noerr) THEN
          DO i=1,nvatts
            status=nf90_inq_attname(ncINPid, varid, i, attnam)
            IF (status.eq.nf90_noerr) THEN
              IF (TRIM(attnam).eq.'units') THEN
                status=nf90_get_att(ncINPid, varid, TRIM(attnam),       &
     &                              tunits)
                IF (status.eq.nf90_noerr) THEN
                  IF (tunits(1:3).eq.'day') THEN
                    time_scale=day2sec
                  ELSE IF (tunits(1:6).eq.'second') THEN
                    time_scale=1.0_r8
                  END IF
                ELSE
                  WRITE (stdout,50) string, TRIM(attnam)
                  exit_flag=2
                  ioerror=status
                  RETURN
                END IF
              END IF
            ELSE
              WRITE (stdout,60) string, TRIM(tvarnam)
              exit_flag=2
              RETURN
            END IF
          END DO
        ELSE
          WRITE (stdout,70) string, TRIM(ncname)
          exit_flag=2
          RETURN
        END IF
        INPtime(1)=INPtime(1)*time_scale
      END IF
!
!  Set starting time index and time clock in days.
!
      IF ((model.eq.0).or.(model.eq.iNLM).or.                           &
     &    (model.eq.iTLM).or.(model.eq.iRPM)) THEN
        IF (((model.eq.iTLM).or.(model.eq.iRPM)).and.                   &
     &      (INPtime(1).ne.(dstart*day2sec))) THEN
          INPtime(1)=dstart*day2sec
        END IF
        time(ng)=INPtime(1)
        tdays(ng)=time(ng)*sec2day
        ntstart(ng)=NINT((time(ng)-dstart*day2sec)/dt(ng))+1
        IF (ntstart(ng).lt.1) ntstart(ng)=1
        IF (PerfectRST(ng)) THEN
          ntfirst(ng)=1
        ELSE
          ntfirst(ng)=ntstart(ng)
        END IF
      ELSE IF (model.eq.iADM) THEN
        IF (msg.ne.1) THEN
          time(ng)=INPtime(1)
          tdays(ng)=time(ng)*sec2day
        END IF
        ntstart(ng)=ntimes(ng)+1
        ntend(ng)=1
        ntfirst(ng)=ntend(ng)
      END IF
!
!  Over-write "IniRec" to the actual initial record processed.
!
      IF (model.eq.iNLM) THEN
        IniRec=InpRec
      END IF
!
!  Report information.
!
      lstr=SCAN(ncname,'/',BACK=.TRUE.)+1
      lend=LEN_TRIM(ncname)
      IF (InpThread) THEN
        IF (ERend.gt.ERstr) THEN
          WRITE (stdout,80) string, TRIM(Mstate(msg)),                  &
     &                      INPtime(1)*sec2day, Nrun,                   &
     &                      ncname(lstr:lend), InpRec, Tindex
        ELSE
          WRITE (stdout,90) string, TRIM(Mstate(msg)),                  &
     &                      INPtime(1)*sec2day,                         &
     &                      ncname(lstr:lend), InpRec, Tindex
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Read in nonlinear state variables. If applicable, read in perfect
!  restart variables.
!-----------------------------------------------------------------------
!
      NLM_STATE: IF ((model.eq.iNLM).or.(model.eq.0)) THEN
!
!  Read in nonlinear free-surface (m).
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            IF (Perfect2d) THEN
              gtype=vflag(varid)*r3dvar
            ELSE
              gtype=vflag(varid)*r2dvar
            END IF
          END IF
          IF (Perfect2d) THEN
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, 3,          &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % zeta(LBi,LBj,1))
          ELSE
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % zeta(LBi,LBj,Tindex))
          END IF
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear RHS of free-surface.
!
        IF (get_var(idRzet).and.Perfect2D) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idRzet)), varid)
            gtype=vflag(varid)*r3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, 2,            &
     &                      Fscl, Fmin, Fmax,                           &
     &                      OCEAN(ng) % rzeta(LBi,LBj,1))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idRzet)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idRzet)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 2D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            IF (Perfect2D) THEN
              gtype=vflag(varid)*u3dvar
            ELSE
              gtype=vflag(varid)*u2dvar
            END IF
          END IF
          IF (Perfect2D) THEN
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, 3,          &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % ubar(LBi,LBj,1))
          ELSE
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % ubar(LBi,LBj,Tindex))
          END IF
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear RHS of 2D momentum component in the XI-direction.
!
        IF (get_var(idRu2d).and.Perfect2D) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idRu2d)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, 2,            &
     &                      Fscl, Fmin, Fmax,                           &
     &                      OCEAN(ng) % rubar(LBi,LBj,1))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idRu2d)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idRu2d)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 2D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            IF (Perfect2D) THEN
              gtype=vflag(varid)*v3dvar
            ELSE
              gtype=vflag(varid)*v2dvar
            END IF
          END IF
          IF (Perfect2D) THEN
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, 3,          &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % vbar(LBi,LBj,1))
          ELSE
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % vbar(LBi,LBj,Tindex))
          END IF
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear RHS 2D momentum component in the ETA-direction.
!
        IF (get_var(idRv2d).and.Perfect2D) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idRv2d)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, 2,            &
     &                      Fscl, Fmin, Fmax,                           &
     &                      OCEAN(ng) % rvbar(LBi,LBj,1))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idRv2d)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idRv2d)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 3D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          IF (Perfect3D) THEN
            status=nf_fread4d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng), 1, 2,&
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % u(LBi,LBj,1,1))
          ELSE
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % u(LBi,LBj,1,Tindex))
          END IF
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear RHS of 3D momentum component in the XI-direction.
!
        IF (get_var(idRu3d).and.Perfect3D) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idRu3d)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread4d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng), 1, 2,  &
     &                      Fscl, Fmin, Fmax,                           &
     &                      OCEAN(ng) % ru(LBi,LBj,0,1))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idRu3d)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idRu3d)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 3D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          IF (Perfect3D) THEN
            status=nf_fread4d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng), 1, 2,&
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % v(LBi,LBj,1,1))
          ELSE
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
     &                        OCEAN(ng) % v(LBi,LBj,1,Tindex))
          END IF
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear RHS of 3D momentum component in the ETA-direction.
!
        IF (get_var(idRv3d).and.Perfect3D) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idRv3d)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread4d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng), 1, 2,  &
     &                      Fscl, Fmin, Fmax,                           &
     &                      OCEAN(ng) % rv(LBi,LBj,0,1))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idRv3d)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idRv3d)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear tracer type variables.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf90_inq_varid(ncINPid,                            &
     &                              TRIM(Vname(1,idTvar(itrc))), varid)
              gtype=vflag(varid)*r3dvar
            END IF
            IF (Perfect3D) THEN
              status=nf_fread4d(ng, model, ncINPid, varid, InpRec,      &
     &                          gtype, Vsize,                           &
     &                          LBi, UBi, LBj, UBj, 1, N(ng), 1, 2,     &
     &                          Fscl, Fmin, Fmax,                       &
     &                          OCEAN(ng) % t(LBi,LBj,1,1,itrc))
            ELSE
              status=nf_fread3d(ng, model, ncINPid, varid, InpRec,      &
     &                          gtype, Vsize,                           &
     &                          LBi, UBi, LBj, UBj, 1, N(ng),           &
     &                          Fscl, Fmin, Fmax,                       &
     &                          OCEAN(ng) % t(LBi,LBj,1,Tindex,itrc))
            END IF
            IF (status.ne.nf90_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
!
!  Read in vertical viscosity.
!
        IF (have_var(idVvis)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idVvis)), varid)
            gtype=vflag(varid)*w3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng),        &
     &                      Fscl, Fmin,Fmax,                            &
     &                      MIXING(ng) % AKv(LBi,LBj,0))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvis)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvis)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in temperature vertical diffusion.
!
        IF (have_var(idTdif)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idTdif)), varid)
            gtype=vflag(varid)*w3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng),        &
     &                      Fscl, Fmin,Fmax,                            &
     &                      MIXING(ng) % AKt(LBi,LBj,0,itemp))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idTdif)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idTdif)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in salinity vertical diffusion.
!
        IF (have_var(idSdif)) THEN
          IF (InpThread) THEN
            status=nf90_inq_varid(ncINPid, TRIM(Vname(1,idSdif)), varid)
            gtype=vflag(varid)*w3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng),        &
     &                      Fscl, Fmin,Fmax,                            &
     &                      MIXING(ng) % AKt(LBi,LBj,0,isalt))
          IF (status.ne.nf90_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idSdif)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idSdif)), Fmin, Fmax
            END IF
          END IF
        END IF
      END IF NLM_STATE
!
!-----------------------------------------------------------------------
!  Close input NetCDF file.
!-----------------------------------------------------------------------
!
      status=nf90_close(ncINPid)
!
  20  FORMAT (/,a,'GET_STATE - unable to open input NetCDF file: ',a)
  30  FORMAT (/,a,'GET_STATE - error while reading variable: ',a,2x,    &
     &        'at time record = ',i3,/,18x,'in input NetCDF file: ',a)
  40  FORMAT (/,a,'GET_STATE - requested input time record = ',i3,/,    &
     &        18x,'not found in input NetCDF: ',a,/,                    &
     &        18x,'number of available records = ',i3)
  50  FORMAT (/,a,'GET_STATE - error while reading attribute: ',a)
  60  FORMAT (/,a,'GET_STATE - error while inquiring attributes',       &
     &          ' for variable: ',a)
  70  FORMAT (/,a,'GET_STATE - cannot inquire about time variable',/,   &
     &        18x,'in input NetCDF file: ',a)
  80  FORMAT (/,a,'GET_STATE - ',a,t64,'t = ',f12.4,                    &
     &        /,19x,'(Iter=',i4.4,', File: ',a,', Rec=',i4.4,           &
     &        ', Index=',i1,')')
  90  FORMAT (/,a,'GET_STATE - ',a,t64,'t = ',f12.4,                    &
     &        /,19x,'(File: ',a,', Rec=',i4.4,', Index=',i1,')')
 100  FORMAT (16x,'- ',a,/,19x,'(Min = ',1p,e15.8,                      &
     &        ' Max = ',1p,e15.8,')')
      RETURN
      END SUBROUTINE get_state
