      MODULE set_tides_mod
!
!svn $Id: set_tides.F 588 2008-03-21 23:09:01Z kate $
!================================================== Hernan G. Arango ===
!  Copyright (c) 2002-2008 The ROMS/TOMS Group        Robert Hetland   !
!    Licensed under a MIT/X style license                              !
!    See License_ROMS.txt                                              !
!=======================================================================
!                                                                      !
!  This routine adds tidal elevation (m) and tidal currents (m/s) to   !
!  sea surface height and 2D momentum climatologies, respectively.     !
!                                                                      !
!=======================================================================
!
      implicit none
      PRIVATE
      PUBLIC  :: set_tides
      CONTAINS
!
!***********************************************************************
      SUBROUTINE set_tides (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_boundary
      USE mod_clima
      USE mod_grid
      USE mod_tides
      USE mod_stepping
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
      integer :: Iend, Istr, Jend, Jstr
      integer :: LBi, UBi, LBj, UBj
!
!  Set horizontal starting and ending indices for parallel domain
!  partitions in the XI- and ETA-directions.
!
      Istr=BOUNDS(ng)%Istr(tile)
      Iend=BOUNDS(ng)%Iend(tile)
      Jstr=BOUNDS(ng)%Jstr(tile)
      Jend=BOUNDS(ng)%Jend(tile)
!
!  Determine array lower and upper bounds in the I- and J-directions.
!
      LBi=BOUNDS(ng)%LBi(tile)
      UBi=BOUNDS(ng)%UBi(tile)
      LBj=BOUNDS(ng)%LBj(tile)
      UBj=BOUNDS(ng)%UBj(tile)
!
      CALL wclock_on (ng, iNLM, 11)
      CALL set_tides_tile (ng, tile,                                    &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     GRID(ng) % angler,                           &
     &                     TIDES(ng) % Tperiod,                         &
     &                     TIDES(ng) % UV_Tangle,                       &
     &                     TIDES(ng) % UV_Tphase,                       &
     &                     TIDES(ng) % UV_Tmajor,                       &
     &                     TIDES(ng) % UV_Tminor,                       &
     &                     CLIMA(ng) % ubarclm,                         &
     &                     CLIMA(ng) % vbarclm,                         &
     &                     BOUNDARY(ng) % zeta_east,                    &
     &                     BOUNDARY(ng) % zeta_west,                    &
     &                     BOUNDARY(ng) % zeta_south,                   &
     &                     BOUNDARY(ng) % zeta_north,                   &
     &                     BOUNDARY(ng) % ubar_east,                    &
     &                     BOUNDARY(ng) % vbar_east,                    &
     &                     BOUNDARY(ng) % ubar_west,                    &
     &                     BOUNDARY(ng) % vbar_west,                    &
     &                     BOUNDARY(ng) % ubar_south,                   &
     &                     BOUNDARY(ng) % vbar_south,                   &
     &                     BOUNDARY(ng) % ubar_north,                   &
     &                     BOUNDARY(ng) % vbar_north,                   &
     &                     NTC(ng) )
      CALL wclock_off (ng, iNLM, 11)
      RETURN
      END SUBROUTINE set_tides
!
!***********************************************************************
      SUBROUTINE set_tides_tile (ng, tile,                              &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           angler,                                &
     &                           Tperiod,                               &
     &                           UV_Tangle, UV_Tphase,                  &
     &                           UV_Tmajor, UV_Tminor,                  &
     &                           ubarclm, vbarclm,                      &
     &                           zeta_east, zeta_west,                  &
     &                           zeta_south, zeta_north,                &
     &                           ubar_east, vbar_east,                  &
     &                           ubar_west, vbar_west,                  &
     &                           ubar_south, vbar_south,                &
     &                           ubar_north, vbar_north,                &
     &                           NTC)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
      USE exchange_2d_mod
!
!  Imported variables declarations.
!
      integer, intent(in) :: ng, tile
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: NTC
!
      real(r8), intent(in) :: angler(LBi:,LBj:)
      real(r8), intent(in) :: Tperiod(MTC)
      real(r8), intent(in) :: UV_Tangle(LBi:,LBj:,:)
      real(r8), intent(inout) :: UV_Tmajor(LBi:,LBj:,:)
      real(r8), intent(inout) :: UV_Tminor(LBi:,LBj:,:)
      real(r8), intent(inout) :: UV_Tphase(LBi:,LBj:,:)
      real(r8), intent(inout) :: ubarclm(LBi:,LBj:)
      real(r8), intent(inout) :: vbarclm(LBi:,LBj:)
      real(r8), intent(inout) :: zeta_east(-2:)
      real(r8), intent(inout) :: zeta_west(-2:)
      real(r8), intent(inout) :: zeta_south(-2:)
      real(r8), intent(inout) :: zeta_north(-2:)
      real(r8), intent(inout) :: ubar_east(-2:)
      real(r8), intent(inout) :: vbar_east(-2:)
      real(r8), intent(inout) :: ubar_west(-2:)
      real(r8), intent(inout) :: vbar_west(-2:)
      real(r8), intent(inout) :: ubar_south(-2:)
      real(r8), intent(inout) :: vbar_south(-2:)
      real(r8), intent(inout) :: ubar_north(-2:)
      real(r8), intent(inout) :: vbar_north(-2:)
!
!  Local variables declarations.
!
      logical :: update
      integer :: i, itide, j
      real(r8) :: Cangle, Cphase, Sangle, Sphase
      real(r8) :: angle, cff, phase, omega, ramp, tide_time
      real(r8) :: bry_cor, bry_pgr, bry_str, bry_val
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: Etide
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: Utide
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: Uwrk
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: Vtide
      real(r8), dimension(LBi-1:UBi+1,LBj-1:UBj+1) :: Vwrk
!
!-----------------------------------------------------------------------
!  Set lower and upper tile bounds and staggered variables bounds for
!  this horizontal domain partition.  Notice that if tile=-1, it will
!  set the values for the global grid.
!-----------------------------------------------------------------------
!
      integer :: Istr, IstrR, IstrT, IstrU, Iend, IendR, IendT
      integer :: Jstr, JstrR, JstrT, JstrV, Jend, JendR, JendT
!
      Istr =BOUNDS(ng)%Istr (tile)
      IstrR=BOUNDS(ng)%IstrR(tile)
      IstrT=BOUNDS(ng)%IstrT(tile)
      IstrU=BOUNDS(ng)%IstrU(tile)
      Iend =BOUNDS(ng)%Iend (tile)
      IendR=BOUNDS(ng)%IendR(tile)
      IendT=BOUNDS(ng)%IendT(tile)
      Jstr =BOUNDS(ng)%Jstr (tile)
      JstrR=BOUNDS(ng)%JstrR(tile)
      JstrT=BOUNDS(ng)%JstrT(tile)
      JstrV=BOUNDS(ng)%JstrV(tile)
      Jend =BOUNDS(ng)%Jend (tile)
      JendR=BOUNDS(ng)%JendR(tile)
      JendT=BOUNDS(ng)%JendT(tile)
!
!  Set time-ramping parameter.
!
      ramp=1.0_r8
!
!-----------------------------------------------------------------------
!  Add tidal currents (m/s) to 2D momentum climatologies.
!-----------------------------------------------------------------------
!
      Utide(:,:)=0.0_r8
      Vtide(:,:)=0.0_r8
      cff=2.0_r8*pi*(time(ng)-tide_start*day2sec)
      DO itide=1,NTC
        IF (Tperiod(itide).gt.0.0_r8) THEN
          omega=cff/Tperiod(itide)
          DO j=MIN(JstrR,Jstr-1),JendR
            DO i=MIN(IstrR,Istr-1),IendR
              angle=UV_Tangle(i,j,itide)-angler(i,j)
              Cangle=COS(angle)
              Sangle=SIN(angle)
              phase=omega-UV_Tphase(i,j,itide)
              Cphase=COS(phase)
              Sphase=SIN(phase)
              Uwrk(i,j)=UV_Tmajor(i,j,itide)*Cangle*Cphase-             &
     &                  UV_Tminor(i,j,itide)*Sangle*Sphase
              Vwrk(i,j)=UV_Tmajor(i,j,itide)*Sangle*Cphase+             &
     &                  UV_Tminor(i,j,itide)*Cangle*Sphase
            END DO
          END DO
          DO j=JstrR,JendR
            DO i=Istr,IendR
              Utide(i,j)=Utide(i,j)+                                    &
     &                   ramp*0.5_r8*(Uwrk(i-1,j)+Uwrk(i,j))
            END DO
          END DO
          DO j=Jstr,JendR
            DO i=IstrR,IendR
              Vtide(i,j)=(Vtide(i,j)+                                   &
     &                    ramp*0.5_r8*(Vwrk(i,j-1)+Vwrk(i,j)))
            END DO
          END DO
        END IF
      END DO
!
!  Add sub-tidal forcing and adjust climatology to include tides.
!
      DO j=JstrR,JendR
        DO i=Istr,IendR
          ubarclm(i,j)=ubarclm(i,j)+Utide(i,j)
        END DO
      END DO
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          vbarclm(i,j)=vbarclm(i,j)+Vtide(i,j)
        END DO
      END DO
      CALL exchange_u2d_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        ubarclm)
      CALL exchange_v2d_tile (ng, tile,                                 &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        vbarclm)
!
!  If appropriate, load tidal forcing into boundary arrays.
!
      RETURN
      END SUBROUTINE set_tides_tile
      END MODULE set_tides_mod
