#include "cppdefs.h"
#if defined ADJOINT && !defined SOLVE3D
      SUBROUTINE ad_main2d (ng)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This subroutine is the main driver for adjoint  ROMS/TOMS  when     !
!  configurated as shallow water (barotropic) ocean model only. It     !
!  advances backward the vertically integrated primitive equations     !
!  for a single time step.                                             !               
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
# ifdef FOUR_DVAR
      USE mod_fourdvar
# endif
      USE mod_iounits
      USE mod_scalars
      USE mod_stepping
# ifdef SO_SEMI
      USE mod_storage
# endif
!
# ifdef AIR_OCEAN_NOT_YET
      USE atm_coupler_mod, ONLY : atmos_coupling
# endif
# if defined AD_SENSITIVITY
      USE adsen_force_mod, ONLY : adsen_force
# endif
      USE ad_diag_mod, ONLY : ad_diag
      USE ad_ini_fields_mod, ONLY : ad_ini_fields
# ifdef FOUR_DVAR
      USE ad_misfit_mod, ONLY : ad_misfit
# endif
# ifdef AVERAGES
      USE ad_set_avg_mod, ONLY : ad_set_avg
# endif
# if defined SSH_TIDES || defined UV_TIDES
      USE ad_set_tides_mod, ONLY : ad_set_tides
# endif
      USE ad_set_vbc_mod, ONLY: ad_set_vbc
      USE ad_step2d_mod, ONLY : ad_step2d
# ifdef FLOATS_NOT_YET
      USE ad_step_floats_mod, ONLY : tl_step_floats
# endif
# ifdef ASSIMILATION
!!    USE oi_update_mod, ONLY : oi_update
# endif
# ifdef SO_SEMI
      USE packing_mod, ONLY : so_pack
# endif
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng
!
!  Local variable declarations.
!
      logical:: backward = .TRUE.

      integer :: my_iif, next_indx1, subs, tile, thread
# ifdef SO_SEMI
      integer, save :: SOrec = 0
# endif
# ifdef FLOATS
      integer :: Lend, Lstr, chunk_size
# endif
      integer :: ksav, ktmp

# ifdef FOUR_DVAR
      real(r8) :: HalfDT
# endif
!
!=======================================================================
!  Time-step adjoint vertically integrated equations.
!=======================================================================
!
!  Set time clock.
!
      time(ng)=time(ng)-dt(ng)
      tdays(ng)=time(ng)*sec2day
!
!  Avoid time-stepping if additional delayed IO time-step.
!
      IF (iic(ng).ne.ntstart) THEN

# ifdef FOUR_DVAR 
!
!-----------------------------------------------------------------------
!  If appropriate, read observation and model state at observation
!  locations.  Then, compute adjoint misfit forcing terms.
!-----------------------------------------------------------------------
!
        HalfDT=0.5_r8*dt(ng)
        IF (((iic(ng).eq.ntstart-1).and.                                &
     &       (ObsTime(ng)-HalfDT.eq.time(ng)+HalfDT)).or.               &
     &      (((time(ng)-HalfDT).le.(ObsTime(ng)-HalfDT)).and.           &
     &       ((ObsTime(ng)-HalfDT).lt.(time(ng)+HalfDT)))) THEN
          ProcessObs(ng)=.TRUE.
          CALL obs_read (ng, iADM, backward)
          IF (exit_flag.ne.NoError) RETURN
        ELSE
          ProcessObs(ng)=.FALSE.
        END IF
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile), SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*thread,subs*(thread+1)-1,+1
            CALL ad_misfit (ng, TILE, iADM)
          END DO
        END DO
!$OMP END PARALLEL DO
# endif

# ifdef FLOATS
!
!-----------------------------------------------------------------------
!  Compute Lagrangian drifters trajectories.
!-----------------------------------------------------------------------
!
!  Shift floats time indices.
!
        nfp1(ng)=MOD(nfp1(ng)+1,NFT+1)
        nf(ng)  =MOD(nf(ng)  +1,NFT+1)
        nfm1(ng)=MOD(nfm1(ng)+1,NFT+1)
        nfm2(ng)=MOD(nfm2(ng)+1,NFT+1)
        nfm3(ng)=MOD(nfm3(ng)+1,NFT+1)
!    
!$OMP PARALLEL DO PRIVATE(thread,chunk_size,Lstr,Lend)                  &
!$OMP&            SHARED(ng,numthreads,Nfloats)
        DO thread=0,numthreads-1
          chunk_size=(Nfloats(ng)+numthreads-1)/numthreads
          Lstr=1+thread*chunk_size
          Lend=MIN(Nfloats(ng),Lstr+chunk_size-1)
          CALL ad_step_floats (ng, Lstr, Lend)
        END DO
!$OMP END PARALLEL DO
# endif
!
!-----------------------------------------------------------------------
!  Solve the vertically integrated primitive equations for the
!  free-surface and momentum components.
!-----------------------------------------------------------------------
!
!  Corrector step - Apply 2D time-step corrector scheme.  Notice that
!  ==============   there is not need for a corrector step during the
!  auxiliary (nfast+1) time-step.
!
        my_iif=1
        iif(ng)=my_iif
        nfast(ng)=1
        IF (my_iif.lt.(nfast(ng)+1))then
!$OMP PARALLEL DO PRIVATE(thread,subs,tile), SHARED(ng,numthreads)
          DO thread=0,numthreads-1
            subs=NtileX(ng)*NtileE(ng)/numthreads
            DO tile=subs*thread,subs*(thread+1)-1,+1
              CALL ad_step2d (ng, TILE)
            END DO
          END DO
!$OMP END PARALLEL DO
        END IF
!
!  Set time indices for corrector step.
!
        next_indx1=3-indx1(ng)
        IF (.not.PREDICTOR_2D_STEP(ng)) THEN
          PREDICTOR_2D_STEP(ng)=.TRUE.
          ktmp=knew(ng)
          ksav=kstp(ng)
          knew(ng)=krhs(ng)
          kstp(ng)=ktmp
          krhs(ng)=ksav
        END IF
!
!  Predictor step - Advance barotropic equations using 2D time-step
!  ==============   predictor scheme.
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile), SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*(thread+1)-1,subs*thread,-1
            CALL ad_step2d (ng, TILE)
          END DO
        END DO
!$OMP END PARALLEL DO
!
!  Set time indices for predictor step. The PREDICTOR_2D_STEP switch
!  it is assumed to be false before the first time-step.
!
        IF (PREDICTOR_2D_STEP(ng).and.(iic(ng).ne.ntend)) THEN
          PREDICTOR_2D_STEP(ng)=.FALSE.
          ksav=knew(ng)
          knew(ng)=krhs(ng)
          krhs(ng)=ksav
          iif(ng)=my_iif
        END IF
      END IF

# ifdef SO_SEMI
!
!-----------------------------------------------------------------------
!  If stochastic optimals with respect the seminorm of chosen
!  functional, pack adjoint state surface forcing needed by the
!  dynamical propagator.
!-----------------------------------------------------------------------
!
      IF (MOD(iic(ng),nADJ(ng)).eq.0) THEN
        SOrec=SOrec+1
!$OMP PARALLEL DO PRIVATE(thread,subs,tile,SOrec)                       &
!$OMP&            SHARED(ng,numthreads,Nstr,Nend)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*thread,subs*(thread+1)-1,+1
            CALL so_pack (ng, TILE, Nstr(ng), Nend(ng), SOrec)
          END DO
        END DO
!$OMP END PARALLEL DO
      END IF
# endif
!
!-----------------------------------------------------------------------
!  Set vertical boundary conditions. Process tidal forcing.
!-----------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      DO thread=0,numthreads-1
        subs=NtileX(ng)*NtileE(ng)/numthreads
        DO tile=subs*thread,subs*(thread+1)-1,+1
# if defined SSH_TIDES || defined UV_TIDES
          CALL ad_set_tides (ng, TILE)
# endif
          CALL ad_set_vbc (ng, TILE)
        END DO
      END DO
!$OMP END PARALLEL DO

# ifdef AIR_OCEAN_NOT_YET
!
!-----------------------------------------------------------------------
!  Couple to atmospheric model every "nOCN_steps" timesteps: get
!  air/sea fluxes.
!-----------------------------------------------------------------------
!
      IF (MOD(iic(ng),nOCN_steps).eq.0) THEN
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*(thread+1)-1,subs*thread,-1
            CALL atmos_coupling (ng, TILE)
          END DO
        END DO
!$OMP END PARALLEL DO
      END IF
# endif
!
!-----------------------------------------------------------------------
!  Read in required data, if any, from input NetCDF files.
!-----------------------------------------------------------------------
!
      CALL ad_get_data (ng)
      IF (exit_flag.ne.NoError) RETURN
!
!-----------------------------------------------------------------------
!  Process input data, if any: time interpolate between snapshots.
!  If appropriate, compute and report diagnostics.
!-----------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile), SHARED(ng,numthreads)
      DO thread=0,numthreads-1
        subs=NtileX(ng)*NtileE(ng)/numthreads
        DO tile=subs*thread,subs*(thread+1)-1,+1
          CALL ad_set_data (ng, TILE)
# ifdef AVERAGES
          CALL ad_set_avg (ng, TILE)
# endif
# ifdef DIAGNOSTICS_UV
!!        CALL ad_set_diags (ng, TILE)
# endif
          CALL ad_diag (ng, TILE)
        END DO
      END DO
!$OMP END PARALLEL DO
      IF (exit_flag.ne.NoError) RETURN
!
!-----------------------------------------------------------------------
!  If not a restart, initialize all time levels and compute other
!  initial fields.
!-----------------------------------------------------------------------
!
      IF (iic(ng).eq.ntend) THEN
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*(thread+1)-1,subs*thread,-1
            CALL ad_ini_fields (ng, TILE, iADM)
          END DO
        END DO
!$OMP END PARALLEL DO
      END IF
!
!-----------------------------------------------------------------------
!  If appropriate, write out fields into output NetCDF files.
!-----------------------------------------------------------------------
!
      CALL ad_output (ng)
      IF (exit_flag.ne.NoError) RETURN

# if defined AD_SENSITIVITY
!
!-----------------------------------------------------------------------
!  Add appropriate forcing terms to the adjoint model. The form of the
!  forcing depends on the functional whose sensitivity is required.
!-----------------------------------------------------------------------
!
      IF ((DendS(ng).ge.tdays(ng)).and.(tdays(ng).ge.DstrS(ng))) THEN
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*thread,subs*(thread+1)-1,+1
            CALL adsen_force (ng, TILE)
          END DO
        END DO
!$OMP END PARALLEL DO
      END IF
# endif

      RETURN
      END SUBROUTINE ad_main2d
#else
      SUBROUTINE ad_main2d
      RETURN
      END SUBROUTINE ad_main2d
#endif
