#include "cppdefs.h"
      MODULE ocean_control_mod
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  ROMS/TOMS Strong Constraint 4-Dimensional Variational (4DVar)       !
!            Data Assimilation Driver, Incremental Approach:           !
!                                                                      !
!  This driver is used for strong constraint 4DVar where the only      !
!  errors considered are those for the observations. The model is      !
!  assumed to be perfect.  This is the incremental method and the      !
!  nonlinear, tangent linear and adjoint models are needed.            !
!                                                                      !
!  The misfit (squared difference) between model and observations      !
!  is defined as:                                                      !
!                                                                      !
!         J  = Jb + Jo                                                 !
!                                                                      !
!  where                                                               !
!                                                                      !
!         Jb = transpose(X - Xb) * B^(-1) * (X -Xb)                    !
!                                                                      !
!         Jo = transpose(X - Xo) * O^(-1) * (X -Xo)                    !
!                                                                      !
!         Xb : background state (first guess)                          !
!         Xo : observations                                            !
!         X  : model at observation or background points               !
!         B  : background error covariance                             !
!         O  : observations error covariance (assigned weight)         !
!                                                                      !
!  The routines in this driver control the initialization,  time-      !
!  stepping, and finalization of  ROMS/TOMS  model following ESMF      !
!  conventions:                                                        !
!                                                                      !
!     initialize                                                       !
!     run                                                              !
!     finalize                                                         !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC  :: initialize, run, finalize

      CONTAINS

      SUBROUTINE initialize (first, MyCOMM)
!
!=======================================================================
!                                                                      !
!  This routine allocates and initializes ROMS/TOMS state variables    !
!  and internal and external parameters.                               !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_iounits
      USE mod_scalars
!
#ifdef AIR_OCEAN 
      USE atm_coupler_mod, ONLY : initialize_coupling
#endif
!
!  Imported variable declarations.
!
      logical, intent(inout) :: first

      integer, intent(in), optional :: MyCOMM
!
!  Local variable declarations.
!
      logical :: allocate_vars = .TRUE.

      integer :: ng, thread

#ifdef DISTRIBUTE
!
!-----------------------------------------------------------------------
!  Set distribute-memory (MPI) world communictor.
!-----------------------------------------------------------------------
!
      IF (PRESENT(MyCOMM)) THEN
        OCN_COMM_WORLD=MyCOMM
      ELSE
        OCN_COMM_WORLD=MPI_COMM_WORLD
      END IF
#endif
!
!-----------------------------------------------------------------------
!  On first pass, initialize model parameters a variables for all
!  nested/composed grids.  Notice that the logical switch "first"
!  is used to allow multiple calls to this routine during ensemble
!  configurations.
!-----------------------------------------------------------------------
!
      IF (first) THEN
        first=.FALSE.
!
!  Initialize model internal parameters.
!
        CALL initialize_param
        CALL initialize_parallel
        CALL initialize_scalars
!
!  Initialize wall clocks.
!
        IF (Master) THEN
          WRITE (stdout,10)
 10       FORMAT (' Process Information:',/)
        END IF
        DO ng=1,Ngrids
!$OMP PARALLEL DO PRIVATE(thread) SHARED(ng,numthreads)
          DO thread=0,numthreads-1
            CALL wclock_on (ng, iNLM, 0)
          END DO
!$OMP END PARALLEL DO
        END DO

#ifdef AIR_OCEAN 
!
!  Initialize coupling streams between atmosphere and ocean using the
!  Model Coupling Toolkit (MCT).
!
        CALL initialize_coupling (MyRank)
#endif
!
!  Read in model tunable parameters from standard input.
!
        CALL inp_par (iNLM)
        IF (exit_flag.ne.NoError) THEN
          IF (Master) THEN
            WRITE (stdout,'(/,a,i3,/)') Rerror(exit_flag), exit_flag
          END IF
          RETURN
        END IF
!
!  Allocate and initialize modules variables.
!
        CALL mod_arrays (allocate_vars)
!
!  Allocate and initialize observation arrays.
!
        CALL initialize_fourdvar
      END IF

      RETURN
      END SUBROUTINE initialize

      SUBROUTINE run
!
!=======================================================================
!                                                                      !
!  This routine time-steps ROMS/TOMS nonlinear, tangent linear and     !
!  adjoint models.                                                     !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
      USE mod_stepping
!
      USE ad_convolution_mod, ONLY : ad_convolution
      USE descent_mod, ONLY : descent
      USE ini_adjust_mod, ONLY : ini_adjust
      USE ini_fields_mod, ONLY : ini_fields
      USE mod_ocean, ONLY : initialize_ocean
      USE normalization_mod, ONLY : normalization
      USE tl_convolution_mod, ONLY : tl_convolution
!
!  Local variable declarations.
!
      logical :: converged

      integer :: AdjRec, IniRec, Lsav
      integer :: i, my_iic, ng, subs, tile, thread

      real(r8) :: rate
!
!=======================================================================
!  Run model for all nested grids, if any.
!=======================================================================
!
      NEST_LOOP : DO ng=1,Ngrids
!
!-----------------------------------------------------------------------
!  OUTER LOOP: time-step nonlinear model.
!-----------------------------------------------------------------------
!
!  Initialize relevant parameters.
!
        Lold(ng)=1
        Lnew(ng)=2
        Nrun=1
        IterSD=0
        Ipass=0

        OUTER_LOOP : DO outer=ERstr,ERend
!
!  Initialize nonlinear model with first guess initial conditions.
!
          CALL initial (ng)
          IF (exit_flag.ne.NoError) THEN
            IF (Master) THEN
              WRITE (stdout,10) Rerror(exit_flag), exit_flag
            END IF
            RETURN
          END IF
!
!  If first pass, compute or read in background covariance
!  normalization factors. If computing, write out factor to
!  NetCDF. This is an expensive computation and needs to be
!  computed once for a particular application grid.
!  
          IF (Nrun.eq.1) THEN
            IF (LdefNRM(ng)) THEN
              CALL def_norm (ng)
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(inner,numthreads)
              DO thread=0,numthreads-1
                subs=NtileX(ng)*NtileE(ng)/numthreads
                DO tile=subs*thread,subs*(thread+1)-1
                  CALL normalization (ng, TILE)
                END DO
              END DO
!$OMP END PARALLEL DO
              CALL wrt_norm (ng)
              LdefNRM(ng)=.FALSE.
              LwrtNRM(ng)=.FALSE.
            ELSE
              tNRMindx(ng)=1
              CALL get_state (ng, 5, 5, NRMname(ng), tNRMindx(ng), 1)
#ifdef DISTRIBUTE
              CALL mp_bcasti (ng, iNLM, exit_flag, 1)
#endif
              IF (exit_flag.ne.NoError) RETURN
            END IF
          END IF
!
!  Run nonlinear model. Extact and store nonlinear model values at
!  observation locations.
!
          IF (Master) THEN
            WRITE (stdout,20) 'NL', ntstart, ntend
          END IF

          wrtNLmod(ng)=.TRUE.
          wrtTLmod(ng)=.FALSE.

          time(ng)=time(ng)-dt(ng)

          NL_LOOP : DO my_iic=ntstart,ntend+1

            iic(ng)=my_iic
#ifdef SOLVE3D
            CALL main3d (ng)
#else
            CALL main2d (ng)
#endif
            IF (exit_flag.ne.NoError) THEN
              IF (Master) THEN
                WRITE (stdout,10) Rerror(exit_flag), exit_flag
              END IF  
              RETURN
            END IF

          END DO NL_LOOP

          wrtNLmod(ng)=.FALSE.
          wrtTLmod(ng)=.TRUE.
!
!-----------------------------------------------------------------------
!  INNER LOOP: iterate using tangent linear model increments.
!-----------------------------------------------------------------------
!
          IterSD=0
          CGstepF=CGstepI
          converged=.FALSE.

          INNER_LOOP : DO inner=1,Ninner

            CONVERGE : IF (.not.converged) THEN
!
!  Set cost function convergence switch.
!
              IF (inner.gt.1) THEN
                converged=ABS(FOURDVAR(ng)%CostFunOld(0)-               &
     &                        FOURDVAR(ng)%CostFun(0)).le.              &
     &                    CostFunFac*FOURDVAR(ng)%CostFunOld(0)
              END IF
!
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!  Time-step tangent linear model: compute cost function.
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!
!  If first pass (inner=1, Ipass=1), initialize tangent linear
!  from rest.  Otherwise, initialize with previous tangent linear
!  estimate from descent algorithm (Ipass=2).
! 
              Ipass=1
              CALL tl_initial (ng)
              IF (exit_flag.ne.NoError) THEN
                IF (Master) THEN
                  WRITE (stdout,10) Rerror(exit_flag), exit_flag
                END IF
                RETURN
              END IF
!
!  Run tangent linear model. Compute misfit cost function between model
!  (nonlinear + tangent linear) and observations.
!
              IF (Master) THEN
                WRITE (stdout,20) 'TL', ntstart, ntend
              END IF

              time(ng)=time(ng)-dt(ng)

              TL_LOOP1 : DO my_iic=ntstart,ntend+1

                iic(ng)=my_iic
#ifdef SOLVE3D
                CALL tl_main3d (ng)
#else
                CALL tl_main2d (ng)
#endif
                IF (exit_flag.ne.NoError) THEN
                  IF (Master) THEN
                    WRITE (stdout,10) Rerror(exit_flag), exit_flag
                  END IF
                  RETURN
                END IF

              END DO TL_LOOP1
!
!  Report cost function between model and observations.
!
              IF (Master) THEN        
                IF (Nrun.gt.1) THEN
                  rate=100.0_r8*ABS(FOURDVAR(ng)%CostFun(0)-            &
     &                              FOURDVAR(ng)%CostFunOld(0))/        &
     &                          FOURDVAR(ng)%CostFunOld(0)
                ELSE
                  rate=0.0_r8
                END IF
                WRITE (stdout,30) outer, inner,                         &
     &                            FOURDVAR(ng)%CostFun(0), rate
                DO i=1,NstateVar(ng)
                  IF (FOURDVAR(ng)%CostFun(i).gt.0.0_r8) THEN
                    IF (Nrun.gt.1) THEN
                      rate=100.0_r8*ABS(FOURDVAR(ng)%CostFun(i)-        &
     &                                  FOURDVAR(ng)%CostFunOld(i))/    &
     &                              FOURDVAR(ng)%CostFunOld(i)
                    ELSE
                      rate=0.0_r8
                    END IF
                    IF (i.eq.1) THEN
                      WRITE (stdout,40) outer, inner,                   &
     &                                  FOURDVAR(ng)%CostFun(i),        &
     &                                  TRIM(Vname(1,idSvar(i))), rate
                    ELSE
                      WRITE (stdout,50) outer, inner,                   &
     &                                  FOURDVAR(ng)%CostFun(i),        &
     &                                  TRIM(Vname(1,idSvar(i))), rate
                    END IF
                  END IF
                END DO
              END IF
              DO i=0,NstateVar(ng)
                FOURDVAR(ng)%CostFunOld(i)=FOURDVAR(ng)%CostFun(i)
              END DO
!
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!  Time step adjoint model backwards: compute cost function gradients.
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!
!  Initialize the adjoint model always from rest.
!
              CALL ad_initial (ng)
              IF (exit_flag.ne.NoError) THEN
                IF (Master) THEN
                  WRITE (stdout,10) Rerror(exit_flag), exit_flag
                END IF
                RETURN
              END IF
!
!  Time-step adjoint model backwards. The adjoint model is forced with
!  the adjoint misfit forcing terms between model (nonlinear + tangent
!  linear) and observations.
!
              IF (Master) THEN
                WRITE (stdout,20) 'AD', ntstart, ntend
              END IF

              time(ng)=time(ng)+dt(ng)

              AD_LOOP : DO my_iic=ntstart,ntend,-1

                iic(ng)=my_iic
#ifdef SOLVE3D
                CALL ad_main3d (ng)
#else
                CALL ad_main2d (ng)
#endif
                IF (exit_flag.ne.NoError) THEN
                  IF (Master) THEN
                    WRITE (stdout,10) Rerror(exit_flag), exit_flag
                  END IF
                  RETURN
                END IF

              END DO AD_LOOP
!
!  Clear adjoint arrays.  Is it needed?
!
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile) SHARED(numthreads)
              DO thread=0,numthreads-1
#if defined _OPENMP || defined DISTRIBUTE
                subs=NtileX(ng)*NtileE(ng)/numthreads
#else
                subs=1
#endif
                DO tile=subs*thread,subs*(thread+1)-1
                  CALL initialize_ocean (ng, TILE, iADM)
                END DO
              END DO
!$OMP END PARALLEL DO
!
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!  Descent algorithm: first pass.
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!
!  For efficiency, reinitialize decent algorithm every NSK iterations
!  to avoid searching in directions which have been seached previously. 
!
              IF (MOD(inner-IterSD,NiterSD).eq.0) CGstepF=CGstepI
              IF (Master) THEN
                WRITE (stdout,60) Nrun, Ipass, CGstepF
              END IF
!
!  Read in previous state tangent linear initial conditions and adjoint
!  solution and load it into the appropriate state arrays at descent
!  index "Lold". Also read in latest adjoint solution and load it to
!  descent index "Lnew". In 3D configurations, read latest nonlinear
!  conditions for coupling.
!
              IF (inner.gt.1) THEN
                IF (LcycleADJ(ng)) THEN
                  AdjRec=3-tADJindx(ng)
                ELSE
                  AdjRec=tADJindx(ng)-1
                END IF
                CALL get_state (ng, iTLM, 2, ITLname(ng), tITLindx(ng), &
     &                          Lold(ng))
                CALL get_state (ng, iADM, 3, ADJname(ng), AdjRec,       &
     &                          Lold(ng))
              END IF
#ifdef SOLVE3D
              CALL get_state (ng, iNLM, 1, INIname(ng), tINIindx(ng),   &
     &                        Lnew(ng))
#endif
              CALL get_state (ng, iADM, 4, ADJname(ng), tADJindx(ng),   &
     &                        Lnew(ng))
!
!  Compute new tangent linear initial conditions in V-space using first
!  guess step size, CGstepF.
!
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(inner,CGstepF,numthreads)
              DO thread=0,numthreads-1
                subs=NtileX(ng)*NtileE(ng)/numthreads
                DO tile=subs*thread,subs*(thread+1)-1
                  CALL ad_convolution (ng, TILE, Lnew(ng))
                  CALL descent (ng, TILE, iTLM, inner, CGstepF)
                END DO
              END DO
!$OMP END PARALLEL DO
!
!  Write out first guess tangent linear initial conditions in V-space.
!  This step is needed because of the IO flow. The strategy here is to
!  read data instead of increasing memory with additional state arrays.
!
              CALL tl_wrt_ini (ng, Lnew(ng))
#ifdef DISTRIBUTE
              CALL mp_bcasti (ng, iTLM, exit_flag, 1)
#endif
              IF (exit_flag.ne.NoError) RETURN
!
!  Convolve tangent linear conditions and convert back to physical
!  space.
!
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(inner,CGstepF,numthreads)
              DO thread=0,numthreads-1
                subs=NtileX(ng)*NtileE(ng)/numthreads
                DO tile=subs*thread,subs*(thread+1)-1,+1
                  CALL tl_convolution (ng, TILE, Lnew(ng))
                END DO
              END DO
!$OMP END PARALLEL DO
!
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!  Time step tangent linear model: compute change in cost function
!  due to trial step size.
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!
!  Initialize tangent linear model with just estimated initial
!  conditions.
!             
              Ipass=2
              CALL tl_initial (ng)
              IF (exit_flag.ne.NoError) THEN
                IF (Master) THEN
                  WRITE (stdout,10) Rerror(exit_flag), exit_flag
                END IF
                RETURN
              END IF
!
!  Time-step tangent linear model:  Compute misfit cost function
!  between model (nonlinear + tangent linear) and observations.
!
              IF (Master) THEN
                WRITE (stdout,20) 'TL', ntstart, ntend
              END IF

              time(ng)=time(ng)-dt(ng)

              TL_LOOP2 : DO my_iic=ntstart,ntend+1

                iic(ng)=my_iic
#ifdef SOLVE3D
                CALL tl_main3d (ng)
#else
                CALL tl_main2d (ng)
#endif
                IF (exit_flag.ne.NoError) THEN
                  IF (Master) THEN
                    WRITE (stdout,10) Rerror(exit_flag), exit_flag
                  END IF
                  RETURN
                END IF

              END DO TL_LOOP2
!
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!  Descent algorithm: second pass.
!:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
!
!  Use Derber method (1985) to compute the refined optimum step size
!  for the conjugate gradient algorithm.
!
              CGstepR=CGstepF*(StepSizeT/StepSizeB)
              IF (Master) THEN
                WRITE (stdout,60) Nrun, Ipass, CGstepR
              END IF
!
!  Read in previous tangent linear initial conditions and load it into
!  the appropriate state arrays at descent index "Lold". In 3D
!  configurations, read latest nonlinear conditions for coupling.
!  Reset tangent linear model initial conditions time index to write
!  since initial conditions will be over-written with new adjusted
!  values.
!
              IF (LcycleINI(ng)) THEN
                IniRec=3-tITLindx(ng)
              ELSE
                IniRec=tITLindx(ng)-1
              END IF
              tITLindx(ng)=IniRec
#ifdef SOLVE3D
              CALL get_state (ng, iNLM, 1, INIname(ng), tINIindx(ng),   &
     &                        Lnew(ng))
#endif
              CALL get_state (ng, iTLM, 2, ITLname(ng), IniRec,         &
     &                        Lold(ng))
!
!  Compute new tangent linear initial conditions in V-space using
!  refined step size, CGstepR.
!
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(inner,CGstepR,numthreads)
              DO thread=0,numthreads-1
                subs=NtileX(ng)*NtileE(ng)/numthreads
                DO tile=subs*thread,subs*(thread+1)-1
                  CALL descent (ng, TILE, iTLM, inner, CGstepR)
                END DO
              END DO
!$OMP END PARALLEL DO
!
!  Write out new adjusted tangent linear initial conditions in V-space.
!  Notice that the first guess initial conditions are over written.
!
              CALL tl_wrt_ini (ng, Lnew(ng))
#ifdef DISTRIBUTE
              CALL mp_bcasti (ng, iTLM, exit_flag, 1)
#endif
              IF (exit_flag.ne.NoError) RETURN
!
!  Convolve new adjuted tangent linear conditions and convert back to
!  space.
!
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(inner,CGstepF,numthreads)
              DO thread=0,numthreads-1
                subs=NtileX(ng)*NtileE(ng)/numthreads
                DO tile=subs*thread,subs*(thread+1)-1,+1
                  CALL tl_convolution (ng, TILE, Lnew(ng))
                END DO
              END DO
!$OMP END PARALLEL DO
!
!-----------------------------------------------------------------------
!  Update counters and conjugate gradient step size.
!-----------------------------------------------------------------------
!
              Lsav=Lnew(ng)
              Lnew(ng)=Lold(ng)
              Lold(ng)=Lsav
              CGstepF=CGstepR
              Nrun=Nrun+1

            END IF CONVERGE

          END DO INNER_LOOP
!
!-----------------------------------------------------------------------
!  Clear nonlinear model state variables.
!-----------------------------------------------------------------------
!
          Ipass=0
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile) SHARED(numthreads)
          DO thread=0,numthreads-1
#if defined _OPENMP || defined DISTRIBUTE
            subs=NtileX(ng)*NtileE(ng)/numthreads
#else
            subs=1
#endif
            DO tile=subs*thread,subs*(thread+1)-1
              CALL initialize_ocean (ng, TILE, iNLM)
            END DO
          END DO
!$OMP END PARALLEL DO

        END DO OUTER_LOOP
!
!-----------------------------------------------------------------------
!  Write out final adjusted nonlinear initial conditions.
!-----------------------------------------------------------------------
!
        CALL get_state (ng, iNLM, 1, INIname(ng), tINIindx(ng), 1)
!$OMP PARALLEL DO PRIVATE(ng,thread,subs,tile)                          &
!$OMP&            SHARED(numthreads)
        DO thread=0,numthreads-1
          subs=NtileX(ng)*NtileE(ng)/numthreads
          DO tile=subs*thread,subs*(thread+1)-1
            CALL ini_adjust (ng, TILE, Lold(ng), 1)
            CALL ini_fields (ng, TILE, iNLM)
          END DO
        END DO
!$OMP END PARALLEL DO

        IF (LcycleINI(ng)) THEN
          tINIindx(ng)=1
          NrecINI(ng)=1
        END IF
        CALL wrt_ini (ng, 1)
#ifdef DISTRIBUTE
        CALL mp_bcasti (ng, iNLM, exit_flag, 1)
#endif
        IF (exit_flag.ne.NoError) RETURN

      END DO NEST_LOOP
!
 10   FORMAT (/,a,i3,/)
 20   FORMAT (/,1x,a,1x,'ROMS/TOMS: started time-stepping:',            &
     &        '( TimeSteps: ',i8.8,' - ',i8.8,')',/)
 30   FORMAT (/,' (Outer,Inner) = ','(',i3.3,',',i3.3,')',1x,           &
     &        'Cost Function = ',1p,e16.10,t65,0p,f13.8,' %')
 40   FORMAT (' --------------- ','(',i3.3,',',i3.3,')',1x,             &
     &        'cost function = ',1p,e16.10,1x,a,t65,0p,f13.8,' %')
 50   FORMAT (17x,'(',i3.3,',',i3.3,')',1x,'cost function = ',          &
     &        1p,e16.10,1x,a,t65,0p,f13.8,' %')
 60   FORMAT (/, ' <<<< Descent Algorithm, Iteration = ',i5.5,          &
     &        ', Ipass = ',i1,' >>>>',/,25x,'Step Size = ',1p,e15.8,/)

      RETURN
      END SUBROUTINE run

      SUBROUTINE finalize
!
!=======================================================================
!                                                                      !
!  This routine terminates ROMS/TOMS nonlinear, tangent linear, and    !
!  adjoint models execution.                                           !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
!
!  Local variable declarations.
!
      integer :: ng, thread
!
!-----------------------------------------------------------------------
!  If blowing-up, save latest model state into RESTART NetCDF file.
!-----------------------------------------------------------------------
!
!  If cycling restart records, write solution into record 3.
!
      DO ng=1,Ngrids
        IF (LwrtRST(ng).and.(exit_flag.eq.1)) THEN
          IF (Master) WRITE (stdout,10)
 10       FORMAT (/,' Blowing-up: Saving latest model state into ',     &
     &              ' RESTART file',/)
          IF (LcycleRST(ng).and.(NrecRST(ng).ge.2)) THEN
            tRSTindx(ng)=2
            LcycleRST(ng)=.FALSE.
          END IF
          blowup=exit_flag
          exit_flag=NoError
          CALL wrt_rst (ng)
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Stop model and time profiling clocks.  Close output NetCDF files.
!-----------------------------------------------------------------------
!
!  Stop time clocks.
!
      IF (Master) THEN
        WRITE (stdout,20)
 20     FORMAT (/,'Elapsed CPU time (seconds):',/)
      END IF

      DO ng=1,Ngrids
!$OMP PARALLEL DO PRIVATE(ng,thread) SHARED(numthreads)
        DO thread=0,numthreads-1
          CALL wclock_off (ng, iNLM, 0)
        END DO
!$OMP END PARALLEL DO
      END DO
!
!  Close IO files.
!
      CALL close_io

      RETURN
      END SUBROUTINE finalize

      END MODULE ocean_control_mod
