#include "cppdefs.h"
      MODULE mod_fourdvar

#if defined FOUR_DVAR || defined GRADIENT_CHECK || defined TLM_CHECK
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  Variational data assimilation variables:                            !
!                                                                      !
!  ADmodVal     Adjoint model values at observation locations.         !
!  BackCost     Current background cost function misfit (mean squared  !
!                 difference) between model and background state, Jb.  !
!  CostFun      Previous iteration total cost function (background     !
!                 plus observations), J.                               !    
!  CostGrad     Cost function gradient norm.                           !
!  CostGradOld  Cost function gradient norm from previous iteration.   !
!  CostNorm     Total cost function normalization scales (minimization !
!                 starting value, Jb+Jo).                              !
!  DTsizeH      Horizontal diffusion time-step size for spatial        !
!                 convolutions.                                        !
!  DTsizeV      Vertical diffusion time-step size for spatial          !
!                 convolutions.                                        !
!  NLmodVal     Nonlinear model values at observation locations.       !
!  KhMax        Maximum  horizontal diffusion coefficient.             !
!  KhMin        Minimum  horizontal diffusion coefficient.             !
!  KvMax        Maximum  vertical diffusion coefficient.               !
!  KvMin        Minimum  vertical diffusion coefficient.               !
!  NHsteps      Full number of horizontal diffusion steps for spatial  !
!                 convolutions.                                        !
!  NVsteps      Full number of vertical diffusion steps for spatial    !
!                 convolutions.                                        !
!  ObsCost      Current observation cost function misfit (mean squared !
!                 difference) between model and observations, Jo.      !
!  ObsCount     Current observation counts per state variable.         !
!  ObsErr       Observation error.                                     !
!  ObsScale     Observation scale used during screenning and/or        !
!                 normalization of the observations.                   !
!  ObsType      Observation type identifier.                           !
!  ObsVal       Observation values.                                    !
!  ObsVar       Global observation variance for each state variable.   !
!  TLmodVal     Tangent linear model values at observation locations.  !
!  Vdecay       Covariance vertical decorrelation scale (m).           !
!  Tobs         Observations time (days).                              !
!  Xobs         Observations X-locations (grid coordinates).           !
!  Yobs         Observations Y-locations (grid coordinates).           !
!  Zobs         Observations Z-locations (grid coordinates).           !
# ifdef W4DVAR
!  cg_p         Weak constraint conjugate gradient vector.             !
!  cg_r         Weak constraint conjugate gradient vector,             !
!                 approximation PSI for representer coeffiecients.     !
!  cg_s         Weak constraint conjugate gradient vector,             !
!                 right-hand-side term: R_n*PSI + Cobs*PSI.            !
!  cg_v         Weak constraint conjugate gradient vector.             !
!  cg_x         Weak constraint conjugate gradient vector.             !
!  cg_z         Weak constraint conjugate gradient vector.             !
# endif
!  haveADmod    Logical switch indicating that there is representer    !
!                 coefficients available to process in MODname file.   !
!  haveNLmod    Logical switch indicating that there is nonlinear      !
!                 model data available to process in MODname file.     !
!  haveTLmod    Logical switch indicating that there is tangent        !
!                 model data available to process in MODname file.     !
!                                                                      !
!=======================================================================
!
        USE mod_param
!
        implicit none

        integer,  allocatable :: ObsType(:)

        real(r8), allocatable :: ObsErr(:)
        real(r8), allocatable :: ObsScale(:)
        real(r8), allocatable :: ObsVal(:)
        real(r8), allocatable :: Tobs(:)
        real(r8), allocatable :: Xobs(:)
        real(r8), allocatable :: Yobs(:)
# ifdef SOLVE3D
        real(r8), allocatable :: Zobs(:)
# endif          
        real(r8), allocatable :: ADmodVal(:)
# ifdef S4DVAR
        real(r8), allocatable :: NLmodVal(:,:)
# else
        real(r8), allocatable :: NLmodVal(:)
# endif
# ifdef TLM_OBS
#  ifdef IS4DVAR
        real(r8), allocatable :: TLmodVal(:,:)
#  else
        real(r8), allocatable :: TLmodVal(:)
#  endif
# endif
# ifdef W4DVAR
        real(r8), allocatable :: cg_p(:)
        real(r8), allocatable :: cg_r(:)
        real(r8), allocatable :: cg_s(:)
        real(r8), allocatable :: cg_v(:)
        real(r8), allocatable :: cg_x(:)
        real(r8), allocatable :: cg_z(:)
# endif
!
        TYPE T_FOURDVAR

          integer , pointer :: NobsSurvey(:)
          integer , pointer :: ObsCount(:)

          real(r8), pointer :: BackCost(:)
          real(r8), pointer :: CostFun(:)
          real(r8), pointer :: CostGrad(:)
          real(r8), pointer :: CostGradOld(:)
          real(r8), pointer :: CostNorm(:)
          real(r8), pointer :: ObsCost(:)
          real(r8), pointer :: ObsVar(:)
          real(r8), pointer :: SurveyTime(:)
  
        END TYPE T_FOURDVAR

        TYPE (T_FOURDVAR), allocatable :: FOURDVAR(:)
!
!-----------------------------------------------------------------------
!  Observations parameters.
!-----------------------------------------------------------------------
!
!  Maximum number of observations to process.
!
        integer :: Mobs
!
!  Number of model state variables to process.
!
        integer, dimension(Ngrids) :: NstateVar
!
!  Number of interpolation weights and (I,J,K) indices offsets.
!
# ifdef SOLVE3D
        integer, parameter :: Nweights = 8

        integer, parameter, dimension(Nweights) :: Ioffset =            &
     &                      (/ 0, 1, 0, 1, 0, 1, 0, 1 /)
        integer, parameter, dimension(Nweights) :: Joffset =            &
     &                      (/ 0, 0, 1, 1, 0, 0, 1, 1 /)
        integer, parameter, dimension(Nweights) :: Koffset =            &
     &                      (/ 0, 0, 0, 0, 1, 1, 1, 1 /)
# else
        integer, parameter :: Nweights = 4

        integer, parameter, dimension(Nweights) :: Ioffset =            &
     &                      (/ 0, 1, 0, 1 /)
        integer, parameter, dimension(Nweights) :: Joffset =            &
     &                      (/ 0, 0, 1, 1 /)
# endif
!
!  Size of observation NetCDF file unlimited dimension.
!
        integer, dimension(Ngrids) :: Ndatum         
!
!  Number of observations surveys available.
!
        integer, dimension(Ngrids) :: Nsurvey
!
!  Observation surveys counter.
!
        integer, dimension(Ngrids) :: ObsSurvey
!
!  Current number of observations processed.
!
        integer, dimension(Ngrids) :: Nobs
!
!  Current starting and ending observation file index processed.
!
        integer, dimension(Ngrids) :: NstrObs
        integer, dimension(Ngrids) :: NendObs
!
!  Background error covariance normalization method:
!
!       [0] Exact, very expensive
!       [1] Approximated, randomization
!
        integer, dimension(Ngrids) :: Nmethod
!
!  Random number generation scheme for randomization:
!
!       [0] Intrinsic F90 routine "randon_number"
!       [1] Gaussian distributed deviates, numerical recipes
!
        integer, dimension(Ngrids) :: Rscheme
!
!  Number of iterations in the randomization ensemble used to
!  compute the background error covariance, B, normalization
!  factors. This factors ensure that the diagonal elements of
!  B are equal to unity (Fisher and Coutier, 1995).
!
        integer :: Nrandom = 1000
!
!  Switch to activate the processing of model state at the observation
!  locations.
!
        logical, dimension(Ngrids) :: ProcessObs
!
!  Switch to activate writting of nonlinear model values at
!  observations locations into observations NetCDF file.
!
        logical, dimension(Ngrids) :: wrtNLmod
!
!  Switch to activate writting of representer model values at
!  observations locations into observations NetCDF file.
!
        logical, dimension(Ngrids) :: wrtRPmod
!
!  Switch to activate writting of tangent linear model values at
!  observations locations into observations NetCDF file.
!
        logical, dimension(Ngrids) :: wrtTLmod
!
!  Swiches indicating that there is representer coeffiecients,
!  nonlinear, and tangent linear model data available to process
!  in 4DVAR NetCDF output file (MODname).  At the beeginning,
!  there is not data since this file has been just defined.
!
        logical, dimension(Ngrids) :: haveADmod
        logical, dimension(Ngrids) :: haveNLmod
        logical, dimension(Ngrids) :: haveTLmod
!
!-----------------------------------------------------------------------
!  Spatial convolutions parameters
!-----------------------------------------------------------------------
!
!  Full number of horizontal and vertical diffusion operator step for
!  spatial convolutions.
!
        integer, allocatable :: NHsteps(:)
        integer, allocatable :: NVsteps(:)
!
!  Horizontal and vertical diffusion operator time-step size for
!  spatial convolutions.
!
        real(r8), allocatable :: DTsizeH(:)
        real(r8), allocatable :: DTsizeV(:)
!
!  Minimum and maximum Horizontal and vertical diffusion coefficients
!  used in spatial convolutions.
!
        real(r8), dimension(Ngrids) :: KhMin
        real(r8), dimension(Ngrids) :: KhMax
        real(r8), dimension(Ngrids) :: KvMin
        real(r8), dimension(Ngrids) :: KvMax
!
!-----------------------------------------------------------------------
!  Descent algorithm parameters.
!-----------------------------------------------------------------------
!
!  Number of iteration between each steepest descent step.
!
        integer :: NiterSD
!
!  Iteration of the last steepest descent.
!
        integer :: IterSD
!
!  Conjugate gradient algorithm: [0] Fletcher-Reeves [1] Polak-Riviere
!
        integer :: ICG
!
!  Pass number into the descent algorithm within the same iteration.
!
        integer :: Ipass
!
!  Conjugate gradient alogorithm step size.
!
        real(r8) :: CGstepI              ! initial step size
        real(r8) :: CGstepF              ! first guess step size
        real(r8) :: CGstepR              ! refined step size
!
!  Step size contributions from background and observations
!
        real(r8) :: StepBotBck = 0.0_r8  ! background denominator
        real(r8) :: StepBotObs = 0.0_r8  ! observation denominator
        real(r8) :: StepTopBck = 0.0_r8  ! background numerator
        real(r8) :: StepTopObs = 0.0_r8  ! observation numerator
!
!  Strong constraint conjugate gradient orthogonality tolerance: dot
!  product orthogonality between current and previous search.
!
        real(r8) :: CGtol
!
!  Weak constraint conjugate gradient algorithm convergence criteria.
!
        real(r8) :: CGeps
!
!  Weak contraint conjugate gradient norms.
!
        real(r8) :: cg_gammam1 = 0.0_r8
        real(r8) :: cg_sigmam1 = 0.0_r8
        real(r8) :: cg_rnorm = 0.0_r8
!
!  Cost function percentage change between succesive iterations for
!  convergence.
!
        real(r8) :: CostFunFac
!
!  Brunt-Vaisala expansion polynomial used for cost function
!  normalization.
!
        integer :: npN2
        real(r8), allocatable :: pcoef_N2(:)
!
!------------------------------------------------------------------------
!  Dot product parameters.
!------------------------------------------------------------------------
!
!  Dot product between tangent linear and adjoint vectors.
!
        real(r8) :: DotProduct
        real(r8) :: adDotProduct
!
!  Tangent linear model linearization check dot products.
!
        integer :: ig1count            ! counter for g1 dot product
        integer :: ig2count            ! counter for g2 dot product

        real(r8), dimension(1000) :: g1
        real(r8), dimension(1000) :: g2

      CONTAINS

      SUBROUTINE initialize_fourdvar
!
!=======================================================================
!  Copyright (c) 2004 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine initializes several variables in module "mod_fourdvar" !
!  for all nested grids.                                               !
!                                                                      !
!=======================================================================
!
      USE mod_scalars
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
!
!  Local variable declarations.
!
      integer :: i, ndims, ng, ngatts, recdim, status, varid
      integer :: dimid, dimsiz

      real(r8), parameter :: IniVal = 0.0_r8

      character (len=20) :: dimnam
      character (len=80) :: fname
!
!-----------------------------------------------------------------------
!  Inquire observations NetCDF and determine the maxumum dimension of
!  several observations arrays.
!-----------------------------------------------------------------------
!
      DO ng=1,Ngrids
!
!  Open observation NetCDF.
!
        status=nf_open (TRIM(OBSname(ng)),nf_nowrite,ncOBSid(ng))
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,10) TRIM(OBSname(ng))
          exit_flag=4
          ioerror=status
          RETURN
        END IF
!
!  Inquire about the size of the "datum" unlimitted dimension and
!  "survey" dimension.
!
        Ndatum(ng)=0
        Nsurvey(ng)=0
        status=nf_inq(ncOBSid(ng),ndims,nvars,ngatts,recdim)
        IF (status.eq.nf_noerr) THEN
          DO i=1,ndims
            dimid=i
            status=nf_inq_dim(ncOBSid(ng),dimid,dimnam,dimsiz)
            IF (TRIM(dimnam).eq.'datum') then
              Ndatum(ng)=dimsiz
            ELSE IF (TRIM(dimnam).eq.'survey') THEN
              Nsurvey(ng)=dimsiz
            END IF
          END DO
          IF (Ndatum(ng).eq.0) THEN
            WRITE (stdout,20) 'datum', TRIM(OBSname(ng))
            exit_flag=4
          END IF
          IF (Nsurvey(ng).eq.0) THEN
            WRITE (stdout,20) 'survey', TRIM(OBSname(ng))
            exit_flag=4
          END IF
        ELSE
          WRITE(stdout,30) OBSname(ng)
        END IF
      END DO
!
!  Allocate module structure.
!
      allocate ( FOURDVAR(Ngrids) )
!
!  Allocate vectors to store the number of observations per survey
!  and their times.
!
      DO ng=1,Ngrids

        allocate ( FOURDVAR(ng) % NobsSurvey(Nsurvey(ng)) )
        FOURDVAR(ng) % NobsSurvey = 0

        allocate ( FOURDVAR(ng) % SurveyTime(Nsurvey(ng)) )
        FOURDVAR(ng) % NobsSurvey = IniVal
!
!  Read in number of observations available per survey.
!
        status=nf_inq_varid(ncOBSid(ng),TRIM(Vname(1,idNobs)),varid)
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,40) TRIM(Vname(1,idNobs)), TRIM(OBSname(ng))
          exit_flag=2
          ioerror=status
          RETURN
        END IF
        status=nf_get_vara_int(ncOBSid(ng), varid, 1, Nsurvey(ng),      &
     &                         FOURDVAR(ng)%NobsSurvey)
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,50) TRIM(Vname(1,idNobs)), TRIM(OBSname(ng))
          exit_flag=2
          ioerror=status
          RETURN
        END IF
!
!  Determine maximum size of observation arrays.
!
#ifdef REPRESENTERS
        Mobs=MAXVAL(Ndatum)
#else
        Mobs=0
        DO i=1,Nsurvey(ng)
          Mobs=MAX(Mobs, FOURDVAR(ng)%NobsSurvey(i))
        END DO
#endif
!
!  Close observation NetCDF file.
!
        status=nf_close (ncOBSid(ng))
        ncOBSid(ng)=-1
      END DO
!
!  Allocate and initialize model and observation variables. 
!
      allocate ( ObsType(Mobs) )
      ObsType = 0

      allocate ( ObsErr(Mobs) )
      ObsErr = IniVal

      allocate ( ObsScale(Mobs) )
      ObsScale = IniVal

      allocate ( ObsVal(Mobs) )
      ObsVal = IniVal

      allocate ( Tobs(Mobs) )
      Tobs = IniVal

      allocate ( Xobs(Mobs) )
      Xobs = IniVal
 
      allocate ( Yobs(Mobs) )
      Yobs = IniVal

# ifdef SOLVE3D
      allocate ( Zobs(Mobs) )
      Zobs = IniVal
# endif
 
      allocate ( ADmodVal(Mobs) )
      ADmodVal = IniVal

# ifdef S4DVAR
      allocate ( NLmodVal(Mobs,2) )
# else
      allocate ( NLmodVal(Mobs) )
# endif
      NLmodVal = IniVal

# ifdef TLM_OBS
#  ifdef IS4DVAR
      allocate ( TLmodVal(Mobs,2) )
#  else
      allocate ( TLmodVal(Mobs) )
#  endif
      TLmodVal = IniVal
# endif
# ifdef W4DVAR
!
!  Allocate and initialize weak constraint conjugate gradient vectors.
!
      allocate ( cg_p(Mobs) )
      cg_p = IniVal

      allocate ( cg_r(Mobs) )
      cg_r = IniVal

      allocate ( cg_s(Mobs) )
      cg_s = IniVal

      allocate ( cg_v(Mobs) )
      cg_v = IniVal

      allocate ( cg_x(Mobs) )
      cg_x = IniVal

      allocate ( cg_z(Mobs) )
      cg_z = IniVal
# endif
!
!  Allocate convolution parameters.
!
      allocate ( NHsteps(5+MT) )
      allocate ( NVsteps(5+MT) )
      allocate ( DTsizeH(5+MT) )
      allocate ( DTsizeV(5+MT) )
!
!-----------------------------------------------------------------------
!  Allocate remaining structure variables.
!-----------------------------------------------------------------------
!
      DO ng=1,Ngrids

# ifdef SOLVE3D
        NstateVar(ng)=5+NT(ng)
# else
        NstateVar(ng)=3
# endif

        allocate ( FOURDVAR(ng) % BackCost(0:NstateVar(ng)) )
        FOURDVAR(ng) % BackCost = IniVal

        allocate ( FOURDVAR(ng) % CostFun(0:NstateVar(ng)) )
        FOURDVAR(ng) % CostFun = IniVal

        allocate ( FOURDVAR(ng) % CostGrad(0:NstateVar(ng)) )
        FOURDVAR(ng) % CostGrad = IniVal

        allocate ( FOURDVAR(ng) % CostGradOld(0:NstateVar(ng)) )
        FOURDVAR(ng) % CostGradOld = IniVal

        allocate ( FOURDVAR(ng) % CostNorm(0:NstateVar(ng)) )
        FOURDVAR(ng) % CostNorm = 1.0_r8

        allocate ( FOURDVAR(ng) % ObsCost(0:NstateVar(ng)) )
        FOURDVAR(ng) % ObsCost = IniVal

        allocate ( FOURDVAR(ng) % ObsVar(NstateVar(ng)) )
        FOURDVAR(ng) % ObsVar = IniVal

        allocate ( FOURDVAR(ng) % ObsCount(NstateVar(ng)) )
        FOURDVAR(ng) % ObsCount = 0

      END DO
!
!-----------------------------------------------------------------------
!  Initialize various variables.
!-----------------------------------------------------------------------
!
      DO ng=1,Ngrids
        ProcessObs(ng)=.FALSE.
        wrtNLmod(ng)=.FALSE.
        wrtRPmod(ng)=.FALSE.
        wrtTLmod(ng)=.FALSE.
        KhMin(ng)=1.0_r8
        KhMax(ng)=1.0_r8
        KvMin(ng)=1.0_r8
        KvMax(ng)=1.0_r8
      END DO
!
 10   FORMAT (/,' MOD_FOURDVAR - unable to open input NetCDF file: ',a)
 20   FORMAT (/,' MOD_FOURDVAR - error inquiring dimension: ',a,2x,      &
     &          ' in input NetCDF file: ',a)
 30   FORMAT (/,' MOD_FOURDVAR - unable to inquire size of dimension',   &
     &          ' in variable: ',a)
 40   FORMAT (/,' MOD_FOURDVAR - error inquiring variable ID for: ',a,   &
     &        2x,' in input NetCDF file: ',a)
 50   FORMAT (/,' MOD_FOURDVAR - error while reading variable: ',a,2x,   &
     &          ' in input NetCDF file: ',a)

      RETURN
      END SUBROUTINE initialize_fourdvar
#endif
      END MODULE mod_fourdvar
