#include "cppdefs.h"
      MODULE mod_ncparam
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This MODULE contains all the variables associated with input and    !
!  output  NetCDF  files.  The IO model is very generic and easy to    !
!  change or expand.  The NetCDF files can be in any language.  All    !
!  the IO information is managed using the following variables:        !
!                                                                      !
!  Vname      Input/output variables names and attributes:             !
!               Vname(1,*)  => field variable name.                    !
!               Vname(2,*)  => long-name attribute.                    !
!               Vname(3,*)  => units attribute.                        !
!               Vname(4,*)  => field type attribute.                   !
!               Vname(5,*)  => associated time variable name.          !
!  Tname      Input/output associated time variables names.            !
!                                                                      !
!  Cinfo      Input/output file names associated with each field       !
!                                                                      !
!  Linfo      Input/output fields logical information:                 !
!               Linfo(1,*)  => switch indicating grided data.          !
!               Linfo(2,*)  => switch indicating time cycling.         !
!               Linfo(3,*)  => switch indicating only one-time         !
!                              record available.                       !
!               Linfo(4,*)  => switch indication special record        !
!                              processing (like tides)                 !
!                                                                      !
!  Iinfo      Input/output fields integer information:                 !
!               Iinfo(1,*)  => variable grid type.                     !
!               Iinfo(2,*)  => field variable NetCDF ID.               !
!               Iinfo(3,*)  => associated time variable NetCDF ID.     !
!               Iinfo(4,*)  => number of time records.                 !
!               Iinfo(5,*)  => size of first  spatial dimension.       !
!               Iinfo(6,*)  => size of second spatial dimension.       !
!               Iinfo(7,*)  => size of third  spatial dimension.       !
!               Iinfo(8,*)  => rolling two-time levels index.          !
!               Iinfo(9,*)  => latest processed time record .          !
!                                                                      !
!  Finfo      Input/output field floating-point information:           !
!               Finfo(1,*)  => Starting time (days) of data.           !
!               Finfo(2,*)  => Ending time (days) of data.             !
!               Finfo(3,*)  => Data time lower bound (days) enclosing  !
!                                model starting time.                  !
!               Finfo(4,*)  => Data time upper bound (days) enclosing  !
!                                model starting time.                  !
!               Finfo(5,*)  => length (days) of time cycling.          !
!               Finfo(6,*)  => Scale to convert time to day units.     !
!               Finfo(7,*)  => latest monotonic time (sec).            !
!               Finfo(8,*)  => minimum value for current data.         !
!               Finfo(9,*)  => maximum value for current data.         !
!  Fscale     Scale to convert input data to model units.              !
!  Fpoint     Latest two-time records of input point data.             !
!  Tintrp     Time (sec) of latest field snapshots used for            !
!               interpolation.                                         !
!  Vtime      Latest two-time values of processed input data.          !
!                                                                      !
!=======================================================================
!
        USE mod_param        

        implicit none
!
!  Maximum number of variables in a generic NetCDF file (MV) and
!  maximum number of variables in information arrays (NV).
!
        integer, parameter :: MV = 350
        integer, parameter :: NV = 350
!
!  Input/output grid-type variables.
!
        integer, parameter :: p2dvar = 1         ! 2D PSI-variable
        integer, parameter :: r2dvar = 2         ! 2D RHO-variable
        integer, parameter :: u2dvar = 3         ! 2D U-variable
        integer, parameter :: v2dvar = 4         ! 2D V-variable
        integer, parameter :: p3dvar = 5         ! 3D PSI-variable
        integer, parameter :: r3dvar = 6         ! 3D RHO-variable
        integer, parameter :: u3dvar = 7         ! 3D U-variable
        integer, parameter :: v3dvar = 8         ! 3D V-variable
        integer, parameter :: w3dvar = 9         ! 3D W-variable
        integer, parameter :: b3dvar = 10        ! 3D BED-sediment
!
!  Number of horizontal interior and boundary water points.
!
        integer  :: Nxyp(Ngrids)        ! PSI water points
        integer  :: Nxyr(Ngrids)        ! RHO water points
        integer  :: Nxyu(Ngrids)        ! U water points
        integer  :: Nxyv(Ngrids)        ! V water points
!
!  Number of horizontal interior water points only.
!
        integer  :: NwaterR(Ngrids)     ! RHO water points
        integer  :: NwaterU(Ngrids)     ! U water points
        integer  :: NwaterV(Ngrids)     ! V water points
!
!  Lower and upper bound ranges for RHO-type variables for processing
!  state vector and observations.
!
        integer, dimension(Ngrids) :: rILB
        integer, dimension(Ngrids) :: rIUB
        integer, dimension(Ngrids) :: rJLB
        integer, dimension(Ngrids) :: rJUB
!
!  Lower and upper bound ranges for U-type variables for processing
!  state vector and observations.
!
        integer, dimension(Ngrids) :: uILB
        integer, dimension(Ngrids) :: uIUB
        integer, dimension(Ngrids) :: uJLB
        integer, dimension(Ngrids) :: uJUB
!
!  Lower and upper bound ranges for V-type variables for processing
!  state vector and observations.
!
        integer, dimension(Ngrids) :: vILB
        integer, dimension(Ngrids) :: vIUB
        integer, dimension(Ngrids) :: vJLB
        integer, dimension(Ngrids) :: vJUB
!
!  Switches indicating which variables are written to output files.
!
        logical  :: Hout(NV,Ngrids) ! history file switches
        logical  :: Sout(NV,Ngrids) ! station file switches
!
!  Input/output identification indices.
!
        integer  :: idXgrd = -1   ! XI-grid position
        integer  :: idYgrd = -2   ! ETA-grid position
        integer  :: idZgrd = -3   ! S-grid position
        integer  :: iddpth = -4   ! depth
        integer  :: idglon = -5   ! longitude
        integer  :: idglat = -6   ! latitude

        integer  :: idAice        ! fraction of cell covered by ice
        integer  :: idAKSclm      ! AKS climatology
        integer  :: idCfra        ! cloud fraction
        integer  :: idCG_P        ! conjugate gradient p-vector
        integer  :: idCG_R        ! conjugate gradient r-vector
        integer  :: idCG_S        ! conjugate gradient s-vector
        integer  :: idCG_V        ! conjugate gradient v-vector
        integer  :: idCG_X        ! conjugate gradient x-vector
        integer  :: idCG_Z        ! conjugate gradient z-vector
        integer  :: idDano        ! density anomaly
        integer  :: idDiff(2)     ! temperature and salinity diffusion
        integer  :: iddQdT        ! heat flux sensitivity to SST
        integer  :: idevap        ! evaporation rate
        integer  :: idFsur        ! free-surface
        integer  :: idGhat(2)     ! KPP nonlocal transport
        integer  :: idHbbl        ! depth of bottom boundary layer
        integer  :: idHice        ! depth of ice cover
        integer  :: idHsbl        ! depth of surface boundary layer
        integer  :: idHsno        ! depth of snow cover
        integer  :: idKhor        ! convolution horizontal diffusion
        integer  :: idKver        ! convolution vertical diffusion
        integer  :: idLdwn        ! downwelling longwave radiation flux
        integer  :: idLhea        ! net latent heat flux
        integer  :: idLrad        ! net longwave radiation flux
        integer  :: idMadH        ! ADM interpolation weights
        integer  :: idMtke        ! turbulent kinetic energy
        integer  :: idMtls        ! turbulent length scale
        integer  :: idNLmo        ! NLM at observations locations
        integer  :: idNobs        ! number of observations
        integer  :: idObsD        ! observations depth
        integer  :: idObsT        ! observations time
        integer  :: idObsX        ! observations X-grid location
        integer  :: idObsY        ! observations Y-grid location
        integer  :: idObsZ        ! observations Z-grid location
        integer  :: idOday        ! observations survey time
        integer  :: idOerr        ! observations error
        integer  :: idOtyp        ! observations type
        integer  :: idOval        ! observations value
        integer  :: idOvar        ! observations global variance
        integer  :: idOvel        ! omega vertical velocity
        integer  :: idOclm        ! omega vertical velocity climatology
        integer  :: idQair        ! surface air humidity
        integer  :: idPair        ! surface air pressure
        integer  :: idPbar        ! streamfunction
        integer  :: idRdir        ! river runoff direction
        integer  :: idRepC        ! representer coefficients
        integer  :: idRepo        ! river runoff ETA-positions
        integer  :: idRflg        ! river runoff flag
        integer  :: idRtra        ! river runoff mass transport
        integer  :: idRuct        ! RHS of U-momentum coupling term
        integer  :: idRu2d        ! RHS of 2D U-momentum
        integer  :: idRu3d        ! RHS of total U-momentum
        integer  :: idRvct        ! RHS of V-momentum coupling term
        integer  :: idRv2d        ! RHS of 2D V-momentum
        integer  :: idRv3d        ! RHS of total V-momentum
        integer  :: idRxpo        ! river runoff XI-positions
        integer  :: idRvsh        ! river runoff transport profile
        integer  :: idRzet        ! RHS of free-surface
        integer  :: idrain        ! rainfall rate
        integer  :: idSdif        ! vertical S-diffusion coefficient
        integer  :: idSrad        ! net shortwave radiation flux
        integer  :: idSSHc        ! SSH climatology
        integer  :: idSSHe        ! SSH error variance
        integer  :: idSSHo        ! SSH observations
        integer  :: idSSSc        ! SSS climatology
        integer  :: idSSTc        ! SST climatology
        integer  :: idSSTe        ! SST error variance
        integer  :: idSSTo        ! SST observations
        integer  :: idShea        ! net sensible heat flux
        integer  :: idsfwf        ! surface freswater flux
        integer  :: idTLmo        ! TLM at observation locations
        integer  :: idTair        ! surface air temperature
        integer  :: idTdif        ! vertical T-diffusion coefficient
        integer  :: idTice        ! temperature of ice surface
        integer  :: idtime        ! ocean time
        integer  :: idTper        ! tidal period
        integer  :: idTvan        ! tidal current angle
        integer  :: idTvma        ! maximum tidal current
        integer  :: idTvmi        ! minimum tidal current
        integer  :: idTvph        ! tidal current phase
        integer  :: idTzam        ! tidal elevation amplitude
        integer  :: idTzph        ! tidal elevation phase
        integer  :: idUads        ! 3D U-velocity adjoint sensitivity
        integer  :: idUair        ! surface U-wind
        integer  :: idUbar        ! 2D U-velocity
        integer  :: idUbas        ! 2D U-velocity adjoint sensitivity
        integer  :: idUbcl        ! 2D U-velocity climatology
        integer  :: idUbcs        ! bottom max U-momentum-wave stress
        integer  :: idUbed        ! bed load U-direction
        integer  :: idUbms        ! bottom U-momentum stress
        integer  :: idUbot        ! bed wave orbital U-velocity
        integer  :: idUbrs        ! bottom U-current stress
        integer  :: idUbur        ! bottom U-velocity above bed
        integer  :: idUbws        ! bottom U-wave stress
        integer  :: idUclm        ! 3D U-velocity climatology
        integer  :: idUfx1        ! time averaged U-flux for 2D
        integer  :: idUfx2        ! time averaged U-flux for 3D
        integer  :: idUice        ! ice U-velocity
        integer  :: idUobs        ! 3D U-velocity observations
        integer  :: idUsms        ! surface U-momentum stress
        integer  :: idUsur        ! surface U-velocity observations
        integer  :: idUVer        ! 3D velocity error variance
        integer  :: idUVse        ! surface velocity error variance
        integer  :: idUvel        ! 3D U-velocity
        integer  :: idVads        ! 3D V-velocity adjoint sensitivity
        integer  :: idVair        ! surface V-wind
        integer  :: idVbar        ! 2D V-velocity
        integer  :: idVbas        ! 2D V-velocity adjoint sensitivity
        integer  :: idVbcl        ! 2D V-velocity climatology
        integer  :: idVbcs        ! bottom max V-current-wave stress
        integer  :: idVbed        ! bed load V-direction
        integer  :: idVbms        ! bottom V-momentum stress
        integer  :: idVbot        ! bed wave orbital V-velocity
        integer  :: idVbrs        ! bottom V-current stress
        integer  :: idVbvr        ! bottom V-velocity above bed
        integer  :: idVbws        ! bottom V-wave stress
        integer  :: idVclm        ! 3D V-velocity climatology
        integer  :: idVfx1        ! 2D momentum time-averaged V-flux 
        integer  :: idVfx2        ! 3D momentum time-averaged V-flux  
        integer  :: idVice        ! ice V-velocity
        integer  :: idVmLS        ! vertical mixing length scale
        integer  :: idVmKK        ! Kinetic energy vertical mixing
        integer  :: idVmKP        ! Length scale vertical mixing
        integer  :: idVobs        ! 3D V-velocity observations
        integer  :: idVsms        ! surface V-momentum stress
        integer  :: idVsur        ! surface V-velocity observations
        integer  :: idVvel        ! 3D V-velocity
        integer  :: idVvis        ! vertical viscosity coefficient
        integer  :: idWamp        ! wind-induced wave amplitude
        integer  :: idWdir        ! wind-induced wave direction
        integer  :: idWper        ! wind-induced wave period
        integer  :: idWorb        ! wind-induced wave orbital velocity
        integer  :: idWvel        ! true vertical velocity
        integer  :: idZads        ! Free-surface adjoint sensitivity

        integer  :: idRunoff       ! Surface freshwater runoff rate
        integer  :: idIcec         ! observed (NCEP) ice concentration
        integer  :: idSkt          ! observed (NCEP) skin temperature
        integer  :: idUnms         ! surface U-momentum stress (NCEP)
        integer  :: idVnms         ! surface V-momentum stress (NCEP)

        integer  :: idTimid        ! interior ice temperature
        integer  :: idTauiw        ! ice-water friction velocity
        integer  :: idChuiw        ! ice-water momentum transfer coefficient
        integer  :: idS0mk         ! salinity of molecular sub-layer under ice
        integer  :: idT0mk         ! temperature of molecular sub-layer under ice
        integer  :: idSfwat        ! surface melt water thickness on ice
        integer  :: idIomflx       ! ice-ocean mass flux
        integer  :: idWg2d         ! wind gustiness from NCEP
        integer  :: idCdd          ! momentum transfer coefficient from NCEP
        integer  :: idChd          ! sensible heat trans. coef. from NCEP
        integer  :: idCed          ! latent heat transfer coef. from NCEP
        integer  :: idWg2m         ! wind gustiness from model
        integer  :: idCdm          ! momentum transfer coefficient from model
        integer  :: idChm          ! sensible heat trans. coef. from model
        integer  :: idCem          ! latent heat transfer coef. from model
        integer  :: idRhoa         ! near-surface air density from NCEP
        integer  :: idSig11        ! internal ice stress component 11
        integer  :: idSig22        ! internal ice stress component 22
        integer  :: idSig12        ! internal ice stress component 12

        integer, allocatable :: idRtrc(:)    ! river runoff for tracers
        integer, allocatable :: idTads(:)    ! tracers adjoint sentivity
        integer, allocatable :: idTbot(:)    ! bottom flux for tracers
        integer, allocatable :: idTbry(:,:)  ! tracers boundary
        integer, allocatable :: idTclm(:)    ! tracers climatology
        integer, allocatable :: idTerr(:)    ! tracers error variance
        integer, allocatable :: idTobs(:)    ! tracers observations
        integer, allocatable :: idTsur(:)    ! surface flux for tracers

        integer  :: idU2bc(4)      ! 2D U-velocity boundary conditions
        integer  :: idU3bc(4)      ! 3D U-velocity boundary conditions
        integer  :: idV2bc(4)      ! 2D V-velocity boundary conditions
        integer  :: idV3bc(4)      ! 3D V-velocity boundary conditions
        integer  :: idZbry(4)      ! free-surface boundary conditions
!
!  Time-averaged quadratic terms IDs.
!
        integer  :: idU2av                    ! <ubar*ubar>
        integer  :: idV2av                    ! <vbar*vbar>
        integer  :: idZZav                    ! <zeta*zeta>

        integer  :: idUUav                    ! <u*u>
        integer  :: idUVav                    ! <u*v>
        integer  :: idVVav                    ! <v*v>

        integer  :: idTTav(NAT)               ! <t*t> for active tracers
        integer  :: idUTav(NAT)               ! <u*t> for active tracers
        integer  :: idVTav(NAT)               ! <v*t> for active tracers

 !
 !  Ice variable IDs.
 !
         integer  :: idUibc(4)      ! ice U-velocity boundary conditions
         integer  :: idVibc(4)      ! ice V-velocity boundary conditions
         integer  :: idAibc(4)      ! ice concentration boundary conditions
         integer  :: idHibc(4)      ! ice thickness boundary conditions
         integer  :: idHsnbc(4)     ! snow thickness boundary conditions
         integer  :: idTibc(4)      ! ice interior temp. boundary conditions
         integer  :: idSfwbc(4)     ! surface melt water boundary conditions
         integer  :: idS11bc(4)     ! ice stress Sig11 boundary conditions
         integer  :: idS22bc(4)     ! ice stress Sig22 boundary conditions
         integer  :: idS12bc(4)     ! ice stress Sig12 boundary conditions
!
!  Biological 2D diagnostic variable IDs.
!
        integer, allocatable :: iDbio2(:)     ! 2D biological terms

        integer  :: iCOfx                     ! air-sea CO2 flux
        integer  :: iDNIT                     ! denitrification flux
        integer  :: ipCO2                     ! partial pressure of CO2
        integer  :: iO2fx                     ! air-sea O2 flux
!
!  Biological 3D diagnostic variable IDs.
!
        integer, allocatable :: iDbio3(:)     ! 3D biological terms

        integer  :: iPPro = 1                 ! primary productivity
        integer  :: iNO3u = 2                 ! NO3 uptake
!
!  Tracer/Momentum Diagnostic variable IDs.
!
        integer, allocatable :: idDtrc(:,:)   ! tracers terms
        integer, allocatable :: idDu2d(:)     ! 2D u-momentum terms
        integer, allocatable :: idDv2d(:)     ! 2D v-momentum terms
        integer, allocatable :: idDu3d(:)     ! 3D u-momentum terms
        integer, allocatable :: idDv3d(:)     ! 3D v-momentum terms
!
!  Assimilation state variables indices (order is important).
!
        integer  :: isFsur = 1                ! free-surface
        integer  :: isUbar = 2                ! 2D U-velocity
        integer  :: isVbar = 3                ! 2D V-velocity
        integer  :: isUvel = 4                ! 3D U-velocity
        integer  :: isVvel = 5                ! 3D V-velocity
        integer, allocatable :: isTvar(:)     ! tracers
        integer, allocatable :: idSvar(:)     ! state vector indices
!
!  Stochastic optimals surface forcing IDs.
!
        integer  :: isUstr = 1                ! surface u-stress
        integer  :: isVstr = 2                ! surface v-stress
        integer, allocatable :: isTsur(:)     ! surface tracer flux
!
!  Input/Output NetCDF files IDs.
!
        integer  :: ncADJid(Ngrids)     ! input/output adjoint
        integer  :: ncADSid(Ngrids)     ! input adjoint sensitivity
        integer  :: ncAVGid(Ngrids)     ! output averages
        integer  :: ncBRYid(Ngrids)     ! input boundary conditions
        integer  :: ncCLMid(Ngrids)     ! input climatology
        integer  :: ncDIAid(Ngrids)     ! output diagnostics
#ifdef FILTERED_RST
	integer  :: ncFILid(nfile,Ngrids)     ! output filter
#endif
        integer  :: ncFLTid(Ngrids)     ! output floats
        integer  :: ncFRCid(NV,Ngrids)  ! input forcing
        integer  :: ncFWDid(Ngrids)     ! forward solution
        integer  :: ncGRDid(Ngrids)     ! input grid
        integer  :: ncGSTid(Ngrids)     ! input/output GST restart
        integer  :: ncHISid(Ngrids)     ! output history
        integer  :: ncINIid(Ngrids)     ! input/output NLM initial
        integer  :: ncITLid(Ngrids)     ! input/output TLM initial
        integer  :: ncMODid(Ngrids)     ! output 4DVAR fields
        integer  :: ncNRMid(Ngrids)     ! input/output covariance norm
        integer  :: ncOBSid(Ngrids)     ! input/output observations
        integer  :: ncRSTid(Ngrids)     ! input/output restart
        integer  :: ncSSHid(Ngrids)     ! SSH observations
        integer  :: ncSSTid(Ngrids)     ! SST observations
        integer  :: ncSTAid(Ngrids)     ! output stations
        integer  :: ncTLFid(Ngrids)     ! input/output TLM/RPM impulses
        integer  :: ncTLMid(Ngrids)     ! input/output tangent linear
        integer  :: ncTOBSid(Ngrids)    ! tracer observations
        integer  :: ncVOBSid(Ngrids)    ! currents observations
        integer  :: ncVSURid(Ngrids)    ! surface currents observations

        integer  :: idefADJ(Ngrids)     ! adjoint file creation flag
        integer  :: idefAVG(Ngrids)     ! averages file creation flag
        integer  :: idefDIA(Ngrids)     ! diagnostics file creation flag
        integer  :: idefHIS(Ngrids)     ! history file creation flag
        integer  :: idefTLM(Ngrids)     ! tangent file creation flag
!
!  Output NetCDF variables IDs.
!
        integer, allocatable :: idTvar(:)     ! tracers variables
        integer, allocatable :: adjTid(:,:)   ! adjoint tracers IDs
        integer, allocatable :: avgTid(:,:)   ! averages tracers IDs
	integer, allocatable :: filTid(:,:)   ! filter tracers IDs
        integer, allocatable :: fltTid(:,:)   ! floats tracers IDs
        integer, allocatable :: hisTid(:,:)   ! history tracers IDs
        integer, allocatable :: iniTid(:,:)   ! initial NLM tracers IDs
        integer, allocatable :: itlTid(:,:)   ! initial TLM tracers IDs
        integer, allocatable :: nrmTid(:,:)   ! norm tracers IDs
        integer, allocatable :: obsTid(:,:)   ! observations tracers IDs
        integer, allocatable :: rstTid(:,:)   ! restart tracers IDs
        integer, allocatable :: staTid(:,:)   ! stations tracers IDs
        integer, allocatable :: tlfTid(:,:)   ! TLM impulse tracers IDs
        integer, allocatable :: tlmTid(:,:)   ! tangent tracers IDs

        integer  :: adjVid(NV,Ngrids)    ! adjoint variables IDs
        integer  :: avgVid(NV,Ngrids)    ! averages variables IDs
        integer  :: diaVid(NV,Ngrids)    ! diagnostics variables IDs
	integer  :: filVid(NV,Ngrids)    ! filter variables IDs
        integer  :: fltVid(-6:NV,Ngrids) ! floats variables IDs
        integer  :: hisVid(NV,Ngrids)    ! history variables IDs
        integer  :: iniVid(NV,Ngrids)    ! initial NLM variables IDs
        integer  :: itlVid(NV,Ngrids)    ! initial TLM variables IDs
        integer  :: modVid(NV,Ngrids)    ! 4DVAR variables IDs
        integer  :: nrmVid(NV,Ngrids)    ! norm variables IDs
        integer  :: obsVid(NV,Ngrids)    ! observations variables IDs
        integer  :: rstVid(NV,Ngrids)    ! restart variables IDs
        integer  :: staVid(NV,Ngrids)    ! stations variables IDs
        integer  :: tlfVid(NV,Ngrids)    ! TLM impulse variables IDs
        integer  :: tlmVid(NV,Ngrids)    ! tangent variables IDs

        integer  :: tADJindx(Ngrids)     ! adjoint time record index
        integer  :: tAVGindx(Ngrids)     ! averages time record index
        integer  :: tDIAindx(Ngrids)     ! diagnostics time record index
	integer  :: tFILindx(Ngrids)     ! filter time record index
        integer  :: tFLTindx(Ngrids)     ! floats time record index
        integer  :: tHISindx(Ngrids)     ! history time record index
        integer  :: tINIindx(Ngrids)     ! initial NLM time record index
        integer  :: tITLindx(Ngrids)     ! initial TLM time record index
        integer  :: tNRMindx(Ngrids)     ! norm time record index
        integer  :: tRSTindx(Ngrids)     ! restart time record index
        integer  :: tSTAindx(Ngrids)     ! stations time record index
        integer  :: tTLFindx(Ngrids)     ! TLM impulse time record index
        integer  :: tTLMindx(Ngrids)     ! tangent time record index
!
!  Current NetCDF file information.
!
        integer  :: nvars          ! number of variables
        integer  :: nvdims(MV)     ! number of variable dimensions
        integer  :: tsize          ! size of unlimited dimension
        integer  :: vdims(5,MV)    ! variables dimensions IDs
        integer  :: vflag(MV)      ! variable flag: 1: full field
                                   !               -1: water points only
        character (len=80 ) :: type
!
!  Input/Output information variables.
!
        logical  :: Linfo(4,NV,Ngrids)

        integer  :: Iinfo(9,NV,Ngrids)

        real(r8) :: Finfo(9,NV,Ngrids)
        real(r8) :: Fpoint(2,NV,Ngrids)
        real(r8) :: Fscale(NV,Ngrids)
        real(r8) :: Tintrp(2,NV,Ngrids)

        real(r8), allocatable :: Vtime(:,:,:)

        character (len=5  ) :: version = '3.0  '
        character (len=40 ) :: varnam(MV)
        character (len=44 ) :: date_str
        character (len=46 ) :: Tname(0:NV)
        character (len=80 ) :: Vname(5,0:NV)
        character (len=80 ) :: Cinfo(NV,Ngrids)
        character (len=120) :: history

      CONTAINS

      SUBROUTINE initialize_ncparam
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine allocates and initializes all variables in module      !
!  "mod_ncparam" for all nested grids.                                 !
!                                                                      !
!=======================================================================
!
        USE mod_parallel
        USE mod_iounits
        USE mod_scalars
#if defined SEDIMENT || defined BBL_MODEL
        USE mod_sediment
#endif
!
!  Local variable declarations.
!
        logical :: load, parse

        integer, parameter :: inp = 10
#ifdef DISTRIBUTE
        integer :: Itile, Jtile
#endif
        integer :: Lvar, gtype, i, j, ng, varid

        real(r8) :: scale

        character (len=80), dimension(7) :: Vinfo
!
!-----------------------------------------------------------------------
!  Allocate several indices variables.
!-----------------------------------------------------------------------
!
        allocate ( idRtrc(MT) )
        allocate ( idTbot(MT) )
        allocate ( idTads(MT) )
        allocate ( idTbry(4,MT) )
        allocate ( idTclm(MT) )
        allocate ( idTerr(MT) )
        allocate ( idTobs(MT) )
        allocate ( idTsur(MT) )
        allocate ( idTvar(MT) )
        allocate ( isTvar(MT) )
        allocate ( isTsur(MT) )

        allocate ( adjTid(MT,Ngrids) )
        allocate ( avgTid(MT,Ngrids) )
	allocate ( filTid(MT,Ngrids) )
        allocate ( fltTid(MT,Ngrids) )
        allocate ( hisTid(MT,Ngrids) )
        allocate ( iniTid(MT,Ngrids) )
        allocate ( itlTid(MT,Ngrids) )
        allocate ( nrmTid(MT,Ngrids) )
        allocate ( obsTid(MT,Ngrids) )
        allocate ( rstTid(MT,Ngrids) )
        allocate ( staTid(MT,Ngrids) )
        allocate ( tlfTid(MT,Ngrids) )
        allocate ( tlmTid(MT,Ngrids) )

#if defined BIO_FASHAM && defined DIAGNOSTICS_BIO
        allocate ( iDbio2(NDbio2d) )
        allocate ( iDbio3(NDbio3d) )
#endif

        allocate ( idDtrc(MT,NDT) )
        allocate ( idDu2d(NDM2d) )
        allocate ( idDv2d(NDM2d) )
        allocate ( idDu3d(NDM3d) )
        allocate ( idDv3d(NDM3d) )

        allocate ( idSvar(MAXVAL(NSV)+1) )

#if defined SSH_TIDES || defined UV_TIDES
        allocate ( Vtime(MAX(2,MTC),NV,Ngrids) )
#else
        allocate ( Vtime(2,NV,Ngrids) )
#endif
!
!-----------------------------------------------------------------------
!  Set lower and upper observation processing (I,J) ranges: Either the
!  full grid or only interior points will be considered.
!-----------------------------------------------------------------------
!   
      DO ng=1,Ngrids
# ifdef DISTRIBUTE
        CALL get_bounds (ng, MyRank, r2dvar, 0, Itile, Jtile,           &
     &                   rILB(ng), rIUB(ng), rJLB(ng), rJUB(ng))
#  ifndef FULL_GRID
        IF (Itile.eq.0) THEN
          rILB(ng)=rILB(ng)+1
        END IF
        IF (Itile.eq.(NtileI(ng)-1)) THEN
          rIUB(ng)=rIUB(ng)-1
        END IF
        IF (Jtile.eq.0) THEN
          rJLB(ng)=rJLB(ng)+1
        END IF
        IF (Jtile.eq.(NtileJ(ng)-1)) THEN
          rJUB(ng)=rJUB(ng)-1
        END IF
#  endif
# else
#  ifdef FULL_GRID
        rILB(ng)=0
        rIUB(ng)=Lm(ng)+1
        rJLB(ng)=0
        rJUB(ng)=Mm(ng)+1
#  else
        rILB(ng)=1
        rIUB(ng)=Lm(ng)
        rJLB(ng)=1
        rJUB(ng)=Mm(ng)
#  endif
# endif

# ifdef DISTRIBUTE
        CALL get_bounds (ng, MyRank, u2dvar, 0, Itile, Jtile,           &
     &                   uILB(ng), uIUB(ng), uJLB(ng), uJUB(ng))
#  ifndef FULL_GRID
        IF (Itile.eq.0) THEN
          uILB(ng)=uILB(ng)+1
        END IF
        IF (Itile.eq.(NtileI(ng)-1)) THEN
          uIUB(ng)=uIUB(ng)-1
        END IF
        IF (Jtile.eq.0) THEN
          uJLB(ng)=uJLB(ng)+1
        END IF
        IF (Jtile.eq.(NtileJ(ng)-1)) THEN
          uJUB(ng)=uJUB(ng)-1
        END IF
#  endif
# else
#  ifdef FULL_GRID
        uILB(ng)=1
        uIUB(ng)=Lm(ng)+1
        uJLB(ng)=0
        uJUB(ng)=Mm(ng)+1
#  else
        uILB(ng)=2
        uIUB(ng)=Lm(ng)
        uJLB(ng)=1
        uJUB(ng)=Mm(ng)
#  endif
# endif

# ifdef DISTRIBUTE
        CALL get_bounds (ng, MyRank, v2dvar, 0, Itile, Jtile,           &
     &                   vILB(ng), vIUB(ng), vJLB(ng), vJUB(ng))
#  ifndef FULL_GRID
        IF (Itile.eq.0) THEN
          vILB(ng)=vILB(ng)+1
        END IF
        IF (Itile.eq.(NtileI(ng)-1)) THEN
          vIUB(ng)=vIUB(ng)-1
        END IF
        IF (Jtile.eq.0) THEN
          vJLB(ng)=vJLB(ng)+1
        END IF
        IF (Jtile.eq.(NtileJ(ng)-1)) THEN
          vJUB(ng)=vJUB(ng)-1
        END IF
#  endif
# else
#  ifdef FULL_GRID
        vILB(ng)=0
        vIUB(ng)=Lm(ng)+1
        vJLB(ng)=1
        vJUB(ng)=Mm(ng)+1
#  else
        vILB(ng)=1
        vIUB(ng)=Lm(ng)
        vJLB(ng)=2
        vJUB(ng)=Mm(ng)
#  endif
# endif
      END DO
!
!-----------------------------------------------------------------------
!  Initialize several variables.
!-----------------------------------------------------------------------
!
!  Initialize NetCDF files IDs to closed state.
!
        DO ng=1,Ngrids
          DO i=1,NV
            ncFRCid(i,ng)=-1
          END DO
#ifdef FILTERED_RST
          DO i=1,nfile
	    ncFILid(i,ng)=-1
          END DO
#endif
          ncADJid(ng)=-1
          ncADSid(ng)=-1
          ncAVGid(ng)=-1
          ncBRYid(ng)=-1
          ncCLMid(ng)=-1
          ncDIAid(ng)=-1
          ncFLTid(ng)=-1
          ncGSTid(ng)=-1
          ncFWDid(ng)=-1
          ncGRDid(ng)=-1
          ncHISid(ng)=-1
          ncINIid(ng)=-1
          ncITLid(ng)=-1
          ncMODid(ng)=-1
          ncNRMid(ng)=-1
          ncOBSid(ng)=-1
          ncRSTid(ng)=-1
          ncSSHid(ng)=-1
          ncSSTid(ng)=-1
          ncSTAid(ng)=-1
          ncTLFid(ng)=-1
          ncTLMid(ng)=-1
          ncTOBSid(ng)=-1
          ncVOBSid(ng)=-1
          ncVSURid(ng)=-1

          tADJindx(ng)=0
          tAVGindx(ng)=0
          tDIAindx(ng)=0
	  tFILindx(ng)=0
          tFLTindx(ng)=0
          tHISindx(ng)=0
          tINIindx(ng)=0
          tITLindx(ng)=0
          tNRMindx(ng)=0
          tRSTindx(ng)=0
          tSTAindx(ng)=0
          tTLFindx(ng)=0
          tTLMindx(ng)=0

          idefADJ(ng)=-1
          idefAVG(ng)=-1
          idefDIA(ng)=-1
          idefHIS(ng)=-1
          idefTLM(ng)=-1
        END DO
!
!  Set IDs for tracers state variables.
!
        DO i=1,MT
          isTvar(i)=5+i
        END DO

#ifdef SOLVE3D
!
!  Set IDs for surface tracer flux variables.
!
      DO i=1,NT(ng)
        isTsur(i)=2+i
      END DO
#endif

#if defined BIO_FASHAM && defined DIAGNOSTICS_BIO
!
!  Initialize biology diagnostic indices.
!
        i=0
# ifdef DENITRIFICATION
        iDNIT=i+1
        i=i+1
# endif
# ifdef CARBON
        iCOfx=i+1
        ipCO2=i+2
        i=i+2
# endif
# ifdef OXYGEN
        iO2fx=i+1
# endif
#endif
!
!  Initialize IO information variables.
!
        DO ng=1,Ngrids
          DO i=1,NV
            Linfo(1,i,ng)=.FALSE.
            Linfo(2,i,ng)=.FALSE.
            Linfo(3,i,ng)=.FALSE.
            Linfo(4,i,ng)=.FALSE.
            Hout(i,ng)=.FALSE.
            Sout(i,ng)=.FALSE.
            Iinfo(1,i,ng)=0
            Iinfo(2,i,ng)=-1
            Iinfo(3,i,ng)=-1
            Iinfo(4,i,ng)=0
            Iinfo(5,i,ng)=0
            Iinfo(6,i,ng)=0
            Iinfo(7,i,ng)=0
            Iinfo(8,i,ng)=0
            Iinfo(9,i,ng)=0
            Finfo(1,i,ng)=0.0_r8
            Finfo(2,i,ng)=0.0_r8
            Finfo(3,i,ng)=0.0_r8
            Finfo(5,i,ng)=0.0_r8
            Finfo(6,i,ng)=0.0_r8
            Finfo(7,i,ng)=0.0_r8
            Fscale(i,ng)=1.0_r8
            Fpoint(1,i,ng)=0.0_r8
            Fpoint(2,i,ng)=0.0_r8
            Tintrp(1,i,ng)=0.0_r8
            Tintrp(2,i,ng)=0.0_r8
            Vtime(1,i,ng)=0.0_r8
            Vtime(2,i,ng)=0.0_r8
          END DO
        END DO
!
!-----------------------------------------------------------------------
!  Define names of variables for Input/Output NetCDF files.
!-----------------------------------------------------------------------
!
!  Open input variable information file.
!
        OPEN (inp, FILE=TRIM(varname), FORM='formatted', STATUS='old',  &
     &        ERR=10)
        GOTO 20
  10    IF (Master) WRITE(stdout,50) TRIM(varname)
        STOP
  20    CONTINUE
!
!  Read in variable information.  Ignore blank and comment [char(33)=!]
!  input lines.
!
        varid=0
        DO WHILE (.TRUE.)
          parse=.FALSE.
          READ (inp,*,ERR=30,END=40) Vinfo(1)
          Lvar=LEN_TRIM(Vinfo(1))
          IF ((Lvar.gt.0).and.(Vinfo(1)(1:1).ne.CHAR(33))) THEN
            READ (inp,*,ERR=30) Vinfo(2)
            READ (inp,*,ERR=30) Vinfo(3)
            READ (inp,*,ERR=30) Vinfo(4)
            READ (inp,*,ERR=30) Vinfo(5)
            READ (inp,*,ERR=30) Vinfo(6)
            READ (inp,*,ERR=30) Vinfo(7)
            READ (inp,*,ERR=30) scale
!
!  Determine staggered C-grid variable.
!
            SELECT CASE (TRIM(ADJUSTL(Vinfo(7))))
              CASE ('p2dvar')
                gtype=p2dvar
              CASE ('r2dvar')
                gtype=r2dvar
              CASE ('u2dvar')
                gtype=u2dvar
              CASE ('v2dvar')
                gtype=v2dvar
              CASE ('p3dvar')
                gtype=p3dvar
              CASE ('r3dvar')
                gtype=r3dvar
              CASE ('u3dvar')
                gtype=u3dvar
              CASE ('v3dvar')
                gtype=v3dvar
              CASE ('w3dvar')
                gtype=w3dvar
              CASE ('b3dvar')
                gtype=b3dvar
              CASE DEFAULT
                gtype=0
            END SELECT
!
!  Assign identification indices.
!
            load=.TRUE.
            varid=varid+1
            SELECT CASE (TRIM(ADJUSTL(Vinfo(6))))
              CASE ('idtime')
                idtime=varid
              CASE ('idFsur')
                idFsur=varid
              CASE ('idUbar')
                idUbar=varid
              CASE ('idVbar')
                idVbar=varid
              CASE ('idUvel')
                idUvel=varid
              CASE ('idVvel')
                idVvel=varid
              CASE ('idWvel')
                idWvel=varid
              CASE ('idOvel')
                idOvel=varid
              CASE ('idDano')
                idDano=varid
              CASE ('idTvar(itemp)')
                idTvar(itemp)=varid
              CASE ('idTvar(isalt)')
                idTvar(isalt)=varid
              CASE ('idUsms')
                idUsms=varid
              CASE ('idVsms')
                idVsms=varid
              CASE ('idUbms')
                idUbms=varid
              CASE ('idVbms')
                idVbms=varid
              CASE ('idUbws')
                idUbws=varid
              CASE ('idUbcs')
                idUbcs=varid
              CASE ('idVbws')
                idVbws=varid
              CASE ('idVbcs')
                idVbcs=varid
              CASE ('idTsur(itemp)')
                idTsur(itemp)=varid
              CASE ('iddQdT')
                iddQdT=varid
              CASE ('idsfwf')
                idsfwf=varid
              CASE ('idTsur(isalt)')
                idTsur(isalt)=varid
              CASE ('idTbot(itemp)')
                idTbot(itemp)=varid
              CASE ('idTbot(isalt)')
                idTbot(isalt)=varid
              CASE ('idGhat(itemp)')
                idGhat(itemp)=varid
              CASE ('idGhat(isalt)')
                idGhat(isalt)=varid
              CASE ('idMtke')
                idMtke=varid
              CASE ('idMtls')
                idMtls=varid
              CASE ('idVvis')
                idVvis=varid
              CASE ('idTdif')
                idTdif=varid
                idDiff(itemp)=idTdif 
              CASE ('idSdif')
                idSdif=varid
                idDiff(isalt)=idSdif 
              CASE ('idVmLS')
                idVmLS=varid
              CASE ('idVmKK')
                idVmKK=varid
              CASE ('idVmKP')
                idVmKP=varid
              CASE ('idZbry(iwest)')
                idZbry(iwest)=varid
              CASE ('idZbry(ieast)')
                idZbry(ieast)=varid
              CASE ('idZbry(isouth)')
                idZbry(isouth)=varid
              CASE ('idZbry(inorth)')
                idZbry(inorth)=varid
              CASE ('idU2bc(iwest)')
                idU2bc(iwest)=varid
              CASE ('idU2bc(ieast)')
                idU2bc(ieast)=varid
              CASE ('idU2bc(isouth)')
                idU2bc(isouth)=varid
              CASE ('idU2bc(inorth)')
                idU2bc(inorth)=varid
              CASE ('idV2bc(iwest)')
                idV2bc(iwest)=varid
              CASE ('idV2bc(ieast)')
                idV2bc(ieast)=varid
              CASE ('idV2bc(isouth)')
                idV2bc(isouth)=varid
              CASE ('idV2bc(inorth)')
                idV2bc(inorth)=varid
              CASE ('idU3bc(iwest)')
                idU3bc(iwest)=varid
              CASE ('idU3bc(ieast)')
                idU3bc(ieast)=varid
              CASE ('idU3bc(isouth)')
                idU3bc(isouth)=varid
              CASE ('idU3bc(inorth)')
                idU3bc(inorth)=varid
              CASE ('idV3bc(iwest)')
                idV3bc(iwest)=varid
              CASE ('idV3bc(ieast)')
                idV3bc(ieast)=varid
              CASE ('idV3bc(isouth)')
                idV3bc(isouth)=varid
              CASE ('idV3bc(inorth)')
                idV3bc(inorth)=varid
              CASE ('idTbry(iwest,itemp)')
                idTbry(iwest,itemp)=varid
              CASE ('idTbry(ieast,itemp)')
                idTbry(ieast,itemp)=varid
              CASE ('idTbry(isouth,itemp)')
                idTbry(isouth,itemp)=varid
              CASE ('idTbry(inorth,itemp)')
                idTbry(inorth,itemp)=varid
              CASE ('idTbry(iwest,isalt)')
                idTbry(iwest,isalt)=varid
              CASE ('idTbry(ieast,isalt)')
                idTbry(ieast,isalt)=varid
              CASE ('idTbry(isouth,isalt)')
                idTbry(isouth,isalt)=varid
              CASE ('idTbry(inorth,isalt)')
                idTbry(inorth,isalt)=varid
              CASE ('idAibc(iwest)')
                idAibc(iwest)=varid
              CASE ('idAibc(ieast)')
                idAibc(ieast)=varid
              CASE ('idAibc(isouth)')
                idAibc(isouth)=varid
              CASE ('idAibc(inorth)')
                idAibc(inorth)=varid
              CASE ('idHibc(iwest)')
                idHibc(iwest)=varid
              CASE ('idHibc(ieast)')
                idHibc(ieast)=varid
              CASE ('idHibc(isouth)')
                idHibc(isouth)=varid
              CASE ('idHibc(inorth)')
                idHibc(inorth)=varid
              CASE ('idHsnbc(iwest)')
                idHsnbc(iwest)=varid
              CASE ('idHsnbc(ieast)')
                idHsnbc(ieast)=varid
              CASE ('idHsnbc(isouth)')
                idHsnbc(isouth)=varid
              CASE ('idHsnbc(inorth)')
                idHsnbc(inorth)=varid
              CASE ('idTibc(iwest)')
                idTibc(iwest)=varid
              CASE ('idTibc(ieast)')
                idTibc(ieast)=varid
              CASE ('idTibc(isouth)')
                idTibc(isouth)=varid
              CASE ('idTibc(inorth)')
                idTibc(inorth)=varid
              CASE ('idSfwbc(iwest)')
                idSfwbc(iwest)=varid
              CASE ('idSfwbc(ieast)')
                idSfwbc(ieast)=varid
              CASE ('idSfwbc(isouth)')
                idSfwbc(isouth)=varid
              CASE ('idSfwbc(inorth)')
                idSfwbc(inorth)=varid
              CASE ('idUibc(iwest)')
                idUibc(iwest)=varid
              CASE ('idUibc(ieast)')
                idUibc(ieast)=varid
              CASE ('idUibc(isouth)')
                idUibc(isouth)=varid
              CASE ('idUibc(inorth)')
                idUibc(inorth)=varid
              CASE ('idVibc(iwest)')
                idVibc(iwest)=varid
              CASE ('idVibc(ieast)')
                idVibc(ieast)=varid
              CASE ('idVibc(isouth)')
                idVibc(isouth)=varid
              CASE ('idVibc(inorth)')
                idVibc(inorth)=varid
              CASE ('idS11bc(iwest)')
                idS11bc(iwest)=varid
              CASE ('idS11bc(ieast)')
                idS11bc(ieast)=varid
              CASE ('idS11bc(isouth)')
                idS11bc(isouth)=varid
              CASE ('idS11bc(inorth)')
                idS11bc(inorth)=varid
              CASE ('idS22bc(iwest)')
                idS22bc(iwest)=varid
              CASE ('idS22bc(ieast)')
                idS22bc(ieast)=varid
              CASE ('idS22bc(isouth)')
                idS22bc(isouth)=varid
              CASE ('idS22bc(inorth)')
                idS22bc(inorth)=varid
              CASE ('idS12bc(iwest)')
                idS12bc(iwest)=varid
              CASE ('idS12bc(ieast)')
                idS12bc(ieast)=varid
              CASE ('idS12bc(isouth)')
                idS12bc(isouth)=varid
              CASE ('idS12bc(inorth)')
                idS12bc(inorth)=varid

              CASE ('idPair')
                idPair=varid
              CASE ('idTair')
                idTair=varid
              CASE ('idQair')
                idQair=varid
              CASE ('idCfra')
                idCfra=varid
              CASE ('idSrad')
                idSrad=varid
              CASE ('idLdwn')
                idLdwn=varid
              CASE ('idLrad')
                idLrad=varid
              CASE ('idLhea')
                idLhea=varid
              CASE ('idShea')
                idShea=varid
              CASE ('idrain')
                idrain=varid
              CASE ('idevap')
                idevap=varid
              CASE ('idRunoff')
                idRunoff=varid
              CASE ('idUair')
                idUair=varid
              CASE ('idVair')
                idVair=varid
              CASE ('idWamp')
                idWamp=varid
              CASE ('idWdir')
                idWdir=varid
              CASE ('idWper')
                idWper=varid
              CASE ('idWorb')
                idWorb=varid
              CASE ('idTper')
                idTper=varid
              CASE ('idTzam')
                idTzam=varid
              CASE ('idTzph')
                idTzph=varid
              CASE ('idTvph')
                idTvph=varid
              CASE ('idTvan')
                idTvan=varid
              CASE ('idTvma')
                idTvma=varid
              CASE ('idTvmi')
                idTvmi=varid
              CASE ('idRxpo')
                idRxpo=varid
              CASE ('idRepo')
                idRepo=varid
              CASE ('idRdir')
                idRdir=varid
              CASE ('idRvsh')
                idRvsh=varid
              CASE ('idRtra')
                idRtra=varid
              CASE ('idRflg')
                idRflg=varid
              CASE ('idRtrc(itemp)')
                idRtrc(itemp)=varid
              CASE ('idRtrc(isalt)')
                idRtrc(isalt)=varid
#if defined NPZD_FRANKS || defined NPZD_POWELL
              CASE ('idRtrc(iNO3_)')
                idRtrc(iNO3_)=varid
              CASE ('idRtrc(iPhyt)')
                idRtrc(iPhyt)=varid
              CASE ('idRtrc(iZoop)')
                idRtrc(iZoop)=varid
              CASE ('idRtrc(iSDet)')
                idRtrc(iSDet)=varid
#endif
#ifdef BIO_FASHAM
              CASE ('idRtrc(iNO3_)')
                idRtrc(iNO3_)=varid
              CASE ('idRtrc(iNH4_)')
                idRtrc(iNH4_)=varid
              CASE ('idRtrc(iPhyt)')
                idRtrc(iPhyt)=varid
              CASE ('idRtrc(iZoop)')
                idRtrc(iZoop)=varid
              CASE ('idRtrc(iLDeN)')
                idRtrc(iLDeN)=varid
              CASE ('idRtrc(iSDeN)')
                idRtrc(iSDeN)=varid
              CASE ('idRtrc(iChlo)')
                idRtrc(iChlo)=varid
# ifdef CARBON
              CASE ('idRtrc(iTIC_)')
                idRtrc(iTIC_)=varid
              CASE ('idRtrc(iTAlk)')
                idRtrc(iTAlk)=varid
              CASE ('idRtrc(iLDeC)')
                idRtrc(iLDeC)=varid
              CASE ('idRtrc(iSDeC)')
                idRtrc(iSDeC)=varid
# endif
# ifdef OXYGEN
              CASE ('idRtrc(iOxyg)')
                idRtrc(iOxyg)=varid
# endif
#endif
#ifdef ICE_MODEL
              CASE ('idUice')
                idUice=varid
              CASE ('idVice')
                idVice=varid
              CASE ('idAice')
                idAice=varid
              CASE ('idHice')
                idHice=varid
              CASE ('idHsno')
                idHsno=varid
              CASE ('idTice')
                idTice=varid
              CASE ('idTimid')
                idTimid=varid
              CASE ('idTauiw')
                idTauiw=varid
              CASE ('idChuiw')
                idChuiw=varid
              CASE ('idS0mk')
                idS0mk=varid
              CASE ('idT0mk')
                idT0mk=varid
              CASE ('idSfwat')
                idSfwat=varid
              CASE ('idIomflx')
                idIomflx=varid
              CASE ('idSig11')
                idSig11=varid
              CASE ('idSig22')
                idSig22=varid
              CASE ('idSig12')
                idSig12=varid
#endif
#ifdef SEDIMENT
              CASE ('idRtrc(idmud)')
                load=.TRUE.
              CASE ('idRtrc(isand)')
                load=.TRUE.
#endif
              CASE ('idHsbl')
                idHsbl=varid
              CASE ('idHbbl')
                idHbbl=varid
              CASE ('idUbot')
                idUbot=varid
              CASE ('idVbot')
                idVbot=varid
              CASE ('idUbur')
                idUbur=varid
              CASE ('idVbvr')
                idVbvr=varid
              CASE ('idUbrs')
                idUbrs=varid
              CASE ('idVbrs')
                idVbrs=varid
              CASE ('idSSHc')
                idSSHc=varid
              CASE ('idUbcl')
                idUbcl=varid
              CASE ('idVbcl')
                idVbcl=varid
              CASE ('idUclm')
                idUclm=varid
              CASE ('idVclm')
                idVclm=varid
              CASE ('idOclm')
                idOclm=varid
              CASE ('idAKSclm')
                idAKSclm=varid
              CASE ('idSSSc')
                idSSSc=varid
              CASE ('idSSTc')
                idSSTc=varid
              CASE ('idTclm(itemp)')
                idTclm(itemp)=varid
              CASE ('idTclm(isalt)')
                idTclm(isalt)=varid
#if defined AD_SENSITIVITY || defined SO_SEMI
              CASE ('idZads')
                idZads=varid
              CASE ('idUbas')
                idUbas=varid
              CASE ('idVbas')
                idVbas=varid
              CASE ('idUads')
                idUads=varid
              CASE ('idVads')
                idVads=varid
              CASE ('idTads(itemp)')
                idTads(itemp)=varid
              CASE ('idTads(isalt)')
                idTads(isalt)=varid
#endif
              CASE ('idSSHo')
                idSSHo=varid
              CASE ('idSSHe')
                idSSHe=varid
              CASE ('idUobs')
                idUobs=varid
              CASE ('idVobs')
                idVobs=varid
              CASE ('idUVer')
                idUVer=varid
              CASE ('idUsur')
                idUsur=varid
              CASE ('idVsur')
                idVsur=varid
              CASE ('idUVse')
                idUVse=varid
              CASE ('idSSTo')
                idSSTo=varid
              CASE ('idSSTe')
                idSSTe=varid
              CASE ('idTobs(itemp)')
                idTobs(itemp)=varid
              CASE ('idTerr(itemp)')
                idTerr(itemp)=varid
              CASE ('idTobs(isalt)')
                idTobs(isalt)=varid
              CASE ('idTerr(isalt)')
                idTerr(isalt)=varid
#ifdef AVERAGES_QUADRATIC
              CASE ('idU2av')
                idU2av=varid
              CASE ('idV2av')
                idV2av=varid
              CASE ('idZZav')
                idZZav=varid
#endif
#ifdef SOLVE3D
# ifdef AVERAGES_QUADRATIC
              CASE ('idTTav(itemp)')
                idTTav(itemp)=varid
              CASE ('idUTav(itemp)')
                idUTav(itemp)=varid
              CASE ('idVTav(itemp)')
                idVTav(itemp)=varid
              CASE ('idUUav')
                idUUav=varid
              CASE ('idUVav')
                idUVav=varid
              CASE ('idVVav')
                idVVav=varid
# endif
# ifdef SALINITY
#  ifdef AVERAGES_QUADRATIC
              CASE ('idTTav(isalt)')
                idTTav(isalt)=varid
              CASE ('idUTav(isalt)')
                idUTav(isalt)=varid
              CASE ('idVTav(isalt)')
                idVTav(isalt)=varid
#  endif
# endif
#endif

#ifdef T_PASSIVE
              CASE ('idTvar(inert(i))')
                load=.TRUE.
                parse=.TRUE.
              CASE ('idTbry(iwest,inert(i))')
                parse=.TRUE.
              CASE ('idTbry(ieast,inert(i))')
                parse=.TRUE.
              CASE ('idTbry(isouth,inert(i))')
                parse=.TRUE.
              CASE ('idTbry(inorth,inert(i))')
                parse=.TRUE.
#endif
#if defined SEDIMENT || defined BBL_MODEL
              CASE ('idTvar(idmud(i))')
                load=.TRUE.
              CASE ('idTvar(isand(i))')
                load=.TRUE.                
              CASE ('idfrac')
                load=.TRUE.
              CASE ('idBmas')
                load=.TRUE.
# ifdef BEDLOAD
              CASE ('idUbld')
                load=.TRUE.
              CASE ('idVbld')
                load=.TRUE.
# endif
              CASE ('idSbed(ithck)')
                idSbed(ithck)=varid
              CASE ('idSbed(iaged)')
                idSbed(iaged)=varid
              CASE ('idSbed(iporo)')
                idSbed(iporo)=varid
              CASE ('idSbed(idiff)')
                idSbed(idiff)=varid
              CASE ('idBott(isd50)')
                idBott(isd50)=varid
              CASE ('idBott(idens)')
                idBott(idens)=varid
              CASE ('idBott(iwsed)')
                idBott(iwsed)=varid
              CASE ('idBott(itauc)')
                idBott(itauc)=varid
              CASE ('idBott(irlen)')
                idBott(irlen)=varid
              CASE ('idBott(irhgt)')
                idBott(irhgt)=varid
              CASE ('idBott(ibwav)')
                idBott(ibwav)=varid
              CASE ('idBott(ishgt)')
                idBott(ishgt)=varid
              CASE ('idBott(izNik)')
                idBott(izNik)=varid
              CASE ('idBott(izbio)')
                idBott(izbio)=varid
              CASE ('idBott(izbld)')
                idBott(izbld)=varid
              CASE ('idBott(izbfm)')
                idBott(izbfm)=varid
              CASE ('idBott(izapp)')
                idBott(izapp)=varid
              CASE ('idBott(izwbl)')
                idBott(izwbl)=varid
              CASE ('idBott(izdef)')
                idBott(izdef)=varid
              CASE ('idBott(iactv)')
                idBott(iactv)=varid
#endif
#if defined NPZD_FRANKS || defined NPZD_POWELL
              CASE ('idTvar(iNO3_)')
                idTvar(iNO3_)=varid
              CASE ('idTvar(iPhyt)')
                idTvar(iPhyt)=varid
              CASE ('idTvar(iZoop)')
                idTvar(iZoop)=varid
              CASE ('idTvar(iSDet)')
                idTvar(iSDet)=varid
# if defined AD_SENSITIVITY || defined SO_SEMI
              CASE ('idTads(iNO3_)')
                idTads(iNO3_)=varid
              CASE ('idTads(iPhyt)')
                idTads(iPhyt)=varid
              CASE ('idTads(iZoop)')
                idTads(iZoop)=varid
              CASE ('idTads(iSDet)')
                idTads(iSDet)=varid
# endif
              CASE ('idTbry(iwest,iNO3_)')
                idTbry(iwest,iNO3_)=varid
              CASE ('idTbry(ieast,iNO3_)')
                idTbry(ieast,iNO3_)=varid
              CASE ('idTbry(isouth,iNO3_)')
                idTbry(isouth,iNO3_)=varid
              CASE ('idTbry(inorth,iNO3_)')
                idTbry(inorth,iNO3_)=varid
              CASE ('idTbry(iwest,iPhyt)')
                idTbry(iwest,iPhyt)=varid
              CASE ('idTbry(ieast,iPhyt)')
                idTbry(ieast,iPhyt)=varid
              CASE ('idTbry(isouth,iPhyt)')
                idTbry(isouth,iPhyt)=varid
              CASE ('idTbry(inorth,iPhyt)')
                idTbry(inorth,iPhyt)=varid
              CASE ('idTbry(iwest,iZoop)')
                idTbry(iwest,iZoop)=varid
              CASE ('idTbry(ieast,iZoop)')
                idTbry(ieast,iZoop)=varid
              CASE ('idTbry(isouth,iZoop)')
                idTbry(isouth,iZoop)=varid
              CASE ('idTbry(inorth,iZoop)')
                idTbry(inorth,iZoop)=varid
              CASE ('idTbry(iwest,iSDet)')
                idTbry(iwest,iSDet)=varid
              CASE ('idTbry(ieast,iSDet)')
                idTbry(ieast,iSDet)=varid
              CASE ('idTbry(isouth,iSDet)')
                idTbry(isouth,iSDet)=varid
              CASE ('idTbry(inorth,iSDet)')
                idTbry(inorth,iSDet)=varid
#endif
#ifdef BIO_FASHAM
              CASE ('idTvar(iNO3_)')
                idTvar(iNO3_)=varid
              CASE ('idTvar(iNH4_)')
                idTvar(iNH4_)=varid
              CASE ('idTvar(iPhyt)')
                idTvar(iPhyt)=varid
              CASE ('idTvar(iZoop)')
                idTvar(iZoop)=varid
              CASE ('idTvar(iLDeN)')
                idTvar(iLDeN)=varid
              CASE ('idTvar(iSDeN)')
                idTvar(iSDeN)=varid
              CASE ('idTvar(iChlo)')
                idTvar(iChlo)=varid
# if defined AD_SENSITIVITY || defined SO_SEMI
              CASE ('idTads(iNO3_)')
                idTads(iNO3_)=varid
              CASE ('idTads(iNH4_)')
                idTads(iNH4_)=varid
              CASE ('idTads(iPhyt)')
                idTads(iPhyt)=varid
              CASE ('idTads(iZoop)')
                idTads(iZoop)=varid
              CASE ('idTads(iLDeN)')
                idTads(iLDeN)=varid
              CASE ('idTads(iSDeN)')
                idTads(iSDeN)=varid
              CASE ('idTads(iChlo)')
                idTads(iChlo)=varid
# endif
# ifdef CARBON
              CASE ('idTvar(iTIC_)')
                idTvar(iTIC_)=varid
              CASE ('idTvar(iTAlk)')
                idTvar(iTAlk)=varid
              CASE ('idTvar(iLDeC)')
                idTvar(iLDeC)=varid
              CASE ('idTvar(iSDeC)')
                idTvar(iSDeC)=varid
#  if defined AD_SENSITIVITY || defined SO_SEMI
              CASE ('idTads(iTIC_)')
                idTads(iTIC_)=varid
              CASE ('idTads(iTAlk)')
                idTads(iTAlk)=varid
              CASE ('idTads(iLDeC)')
                idTads(iLDeC)=varid
              CASE ('idTads(iSDeC)')
                idTads(iSDeC)=varid
#  endif
# endif
# ifdef OXYGEN
              CASE ('idTvar(iOxyg)')
                idTvar(iOxyg)=varid
#  if defined AD_SENSITIVITY || defined SO_SEMI
              CASE ('idTads(iOxyg)')
                idTads(iOxyg)=varid
#  endif
# endif
              CASE ('idTbry(iwest,iNO3_)')
                idTbry(iwest,iNO3_)=varid
              CASE ('idTbry(ieast,iNO3_)')
                idTbry(ieast,iNO3_)=varid
              CASE ('idTbry(isouth,iNO3_)')
                idTbry(isouth,iNO3_)=varid
              CASE ('idTbry(inorth,iNO3_)')
                idTbry(inorth,iNO3_)=varid
              CASE ('idTbry(iwest,iNH4_)')
                idTbry(iwest,iNH4_)=varid
              CASE ('idTbry(ieast,iNH4_)')
                idTbry(ieast,iNH4_)=varid
              CASE ('idTbry(isouth,iNH4_)')
                idTbry(isouth,iNH4_)=varid
              CASE ('idTbry(inorth,iNH4_)')
                idTbry(inorth,iNH4_)=varid
              CASE ('idTbry(iwest,iPhyt)')
                idTbry(iwest,iPhyt)=varid
              CASE ('idTbry(ieast,iPhyt)')
                idTbry(ieast,iPhyt)=varid
              CASE ('idTbry(isouth,iPhyt)')
                idTbry(isouth,iPhyt)=varid
              CASE ('idTbry(inorth,iPhyt)')
                idTbry(inorth,iPhyt)=varid
              CASE ('idTbry(iwest,iZoop)')
                idTbry(iwest,iZoop)=varid
              CASE ('idTbry(ieast,iZoop)')
                idTbry(ieast,iZoop)=varid
              CASE ('idTbry(isouth,iZoop)')
                idTbry(isouth,iZoop)=varid
              CASE ('idTbry(inorth,iZoop)')
                idTbry(inorth,iZoop)=varid
              CASE ('idTbry(iwest,iSDeN)')
                idTbry(iwest,iSDeN)=varid
              CASE ('idTbry(ieast,iSDeN)')
                idTbry(ieast,iSDeN)=varid
              CASE ('idTbry(isouth,iSDeN)')
                idTbry(isouth,iSDeN)=varid
              CASE ('idTbry(inorth,iSDeN)')
                idTbry(inorth,iSDeN)=varid
              CASE ('idTbry(iwest,iLDeN)')
                idTbry(iwest,iLDeN)=varid
              CASE ('idTbry(ieast,iLDeN)')
                idTbry(ieast,iLDeN)=varid
              CASE ('idTbry(isouth,iLDeN)')
                idTbry(isouth,iLDeN)=varid
              CASE ('idTbry(inorth,iLDeN)')
                idTbry(inorth,iLDeN)=varid
              CASE ('idTbry(iwest,iChlo)')
                idTbry(iwest,iChlo)=varid
              CASE ('idTbry(ieast,iChlo)')
                idTbry(ieast,iChlo)=varid
              CASE ('idTbry(isouth,iChlo)')
                idTbry(isouth,iChlo)=varid
              CASE ('idTbry(inorth,iChlo)')
                idTbry(inorth,iChlo)=varid
# ifdef CARBON
              CASE ('idTbry(iwest,iSDeC)')
                idTbry(iwest,iSDeC)=varid
              CASE ('idTbry(ieast,iSDeC)')
                idTbry(ieast,iSDeC)=varid
              CASE ('idTbry(isouth,iSDeC)')
                idTbry(isouth,iSDeC)=varid
              CASE ('idTbry(inorth,iSDeC)')
                idTbry(inorth,iSDeC)=varid
              CASE ('idTbry(iwest,iLDeC)')
                idTbry(iwest,iLDeC)=varid
              CASE ('idTbry(ieast,iLDeC)')
                idTbry(ieast,iLDeC)=varid
              CASE ('idTbry(isouth,iLDeC)')
                idTbry(isouth,iLDeC)=varid
              CASE ('idTbry(inorth,iLDeC)')
                idTbry(inorth,iLDeC)=varid
              CASE ('idTbry(iwest,iTIC_)')
                idTbry(iwest,iTIC_)=varid
              CASE ('idTbry(ieast,iTIC_)')
                idTbry(ieast,iTIC_)=varid
              CASE ('idTbry(isouth,iTIC_)')
                idTbry(isouth,iTIC_)=varid
              CASE ('idTbry(inorth,iTIC_)')
                idTbry(inorth,iTIC_)=varid
              CASE ('idTbry(iwest,iTAlk)')
                idTbry(iwest,iTAlk)=varid
              CASE ('idTbry(ieast,iTAlk)')
                idTbry(ieast,iTAlk)=varid
              CASE ('idTbry(isouth,iTAlk)')
                idTbry(isouth,iTAlk)=varid
              CASE ('idTbry(inorth,iTAlk)')
                idTbry(inorth,iTAlk)=varid
# endif
# ifdef OXYGEN
              CASE ('idTbry(iwest,iOxyg)')
                idTbry(iwest,iOxyg)=varid
              CASE ('idTbry(ieast,iOxyg)')
                idTbry(ieast,iOxyg)=varid
              CASE ('idTbry(isouth,iOxyg)')
                idTbry(isouth,iOxyg)=varid
              CASE ('idTbry(inorth,iOxyg)')
                idTbry(inorth,iOxyg)=varid
# endif
# ifdef DIAGNOSTICS_BIO
              CASE ('iDbio2(iDNIT)')
                iDbio2(iDNIT)=varid
#  ifdef CARBON
              CASE ('iDbio2(iCOfx)')
                iDbio2(iCOfx)=varid
              CASE ('iDbio2(ipCO2)')
                iDbio2(ipCO2)=varid
#  endif
#  ifdef OXYGEN
              CASE ('iDbio2(iO2fx)')
                iDbio2(iO2fx)=varid
#  endif
              CASE ('iDbio3(iPPro)')
                iDbio3(iPPro)=varid
              CASE ('iDbio3(iNO3u)')
                iDbio3(iNO3u)=varid
# endif
#endif
#ifdef ECOSIM
              CASE ('idTvar(iNO3_)')
                idTvar(iNO3_)=varid
              CASE ('idTvar(iNH4_)')
                idTvar(iNH4_)=varid
              CASE ('idTvar(iPO4_)')
                idTvar(iPO4_)=varid
              CASE ('idTvar(iFeO_)')
                idTvar(iFeO_)=varid
              CASE ('idTvar(iSiO_)')
                idTvar(iSiO_)=varid
              CASE ('idTvar(iDIC_)')
                idTvar(iDIC_)=varid
              CASE ('idTvar(iBacC)')
                load=.TRUE.
              CASE ('idTvar(iBacF)')
                load=.TRUE.
              CASE ('idTvar(iBacN)')
                load=.TRUE.
              CASE ('idTvar(iBacP)')
                load=.TRUE.
              CASE ('idTvar(iCDMC)')
                load=.TRUE.
              CASE ('idTvar(iDOMC)')
                load=.TRUE.
              CASE ('idTvar(iDOMN)')
                load=.TRUE.
              CASE ('idTvar(iDOMP)')
                load=.TRUE.
              CASE ('idTvar(iFecC)')
                load=.TRUE.
              CASE ('idTvar(iFecF)')
                load=.TRUE.
              CASE ('idTvar(iFecN)')
                load=.TRUE.
              CASE ('idTvar(iFecP)')
                load=.TRUE.
              CASE ('idTvar(iFecS)')
                load=.TRUE.
              CASE ('idTvar(iPhyC)')
                load=.TRUE.
              CASE ('idTvar(iPhyF)')
                load=.TRUE.
              CASE ('idTvar(iPhyN)')
                load=.TRUE.
              CASE ('idTvar(iPhyP)')
                load=.TRUE.
              CASE ('idTvar(iPhyS)')
                load=.TRUE.
              CASE ('idTvar(iPigs)')
                load=.TRUE.
#endif
#ifdef BIO_GOANPZ
              CASE ('idTvar(iNO3)')
                idTvar(iNO3)=varid
              CASE ('idTvar(iNH4)')
                idTvar(iNH4)=varid
              CASE ('idTvar(iPhS)')
                idTvar(iPhS)=varid
              CASE ('idTvar(iPhL)')
                idTvar(iPhL)=varid
              CASE ('idTvar(iMZS)')
                idTvar(iMZS)=varid
              CASE ('idTvar(iMZL)')
                idTvar(iMZL)=varid
              CASE ('idTvar(iCop)')
                idTvar(iCop)=varid
              CASE ('idTvar(iNCa)')
                idTvar(iNCa)=varid
              CASE ('idTvar(iEup)')
                idTvar(iEup)=varid
              CASE ('idTvar(iDet)')
                idTvar(iDet)=varid
# ifdef IRON
              CASE ('idTvar(iFe)')
                idTvar(iFe)=varid
# endif
              CASE ('idTbry(iwest,iNO3)')
                idTbry(iwest,iNO3)=varid
              CASE ('idTbry(ieast,iNO3)')
                idTbry(ieast,iNO3)=varid
              CASE ('idTbry(isouth,iNO3)')
                idTbry(isouth,iNO3)=varid
              CASE ('idTbry(inorth,iNO3)')
                idTbry(inorth,iNO3)=varid
              CASE ('idTbry(iwest,iNH4)')
                idTbry(iwest,iNH4)=varid
              CASE ('idTbry(ieast,iNH4)')
                idTbry(ieast,iNH4)=varid
              CASE ('idTbry(isouth,iNH4)')
                idTbry(isouth,iNH4)=varid
              CASE ('idTbry(inorth,iNH4)')
                idTbry(inorth,iNH4)=varid
              CASE ('idTbry(iwest,iPhS)')
                idTbry(iwest,iPhS)=varid
              CASE ('idTbry(ieast,iPhS)')
                idTbry(ieast,iPhS)=varid
              CASE ('idTbry(isouth,iPhS)')
                idTbry(isouth,iPhS)=varid
              CASE ('idTbry(inorth,iPhS)')
                idTbry(inorth,iPhS)=varid
              CASE ('idTbry(iwest,iPhL)')
                idTbry(iwest,iPhL)=varid
              CASE ('idTbry(ieast,iPhL)')
                idTbry(ieast,iPhL)=varid
              CASE ('idTbry(isouth,iPhL)')
                idTbry(isouth,iPhL)=varid
              CASE ('idTbry(inorth,iPhL)')
                idTbry(inorth,iPhL)=varid
              CASE ('idTbry(iwest,iMZS)')
                idTbry(iwest,iMZS)=varid
              CASE ('idTbry(ieast,iMZS)')
                idTbry(ieast,iMZS)=varid
              CASE ('idTbry(isouth,iMZS)')
                idTbry(isouth,iMZS)=varid
              CASE ('idTbry(inorth,iMZS)')
                idTbry(inorth,iMZS)=varid
              CASE ('idTbry(iwest,iMZL)')
                idTbry(iwest,iMZL)=varid
              CASE ('idTbry(ieast,iMZL)')
                idTbry(ieast,iMZL)=varid
              CASE ('idTbry(isouth,iMZL)')
                idTbry(isouth,iMZL)=varid
              CASE ('idTbry(inorth,iMZL)')
                idTbry(inorth,iMZL)=varid
              CASE ('idTbry(iwest,iCop)')
                idTbry(iwest,iCop)=varid
              CASE ('idTbry(ieast,iCop)')
                idTbry(ieast,iCop)=varid
              CASE ('idTbry(isouth,iCop)')
                idTbry(isouth,iCop)=varid
              CASE ('idTbry(inorth,iCop)')
                idTbry(inorth,iCop)=varid
              CASE ('idTbry(iwest,iNCa)')
                idTbry(iwest,iNCa)=varid
              CASE ('idTbry(ieast,iNCa)')
                idTbry(ieast,iNCa)=varid
              CASE ('idTbry(isouth,iNCa)')
                idTbry(isouth,iNCa)=varid
              CASE ('idTbry(inorth,iNCa)')
                idTbry(inorth,iNCa)=varid
              CASE ('idTbry(iwest,iEup)')
                idTbry(iwest,iEup)=varid
              CASE ('idTbry(ieast,iEup)')
                idTbry(ieast,iEup)=varid
              CASE ('idTbry(isouth,iEup)')
                idTbry(isouth,iEup)=varid
              CASE ('idTbry(inorth,iEup)')
                idTbry(inorth,iEup)=varid
              CASE ('idTbry(iwest,iDet)')
                idTbry(iwest,iDet)=varid
              CASE ('idTbry(ieast,iDet)')
                idTbry(ieast,iDet)=varid
              CASE ('idTbry(isouth,iDet)')
                idTbry(isouth,iDet)=varid
              CASE ('idTbry(inorth,iDet)')
                idTbry(inorth,iDet)=varid
# ifdef IRON
              CASE ('idTbry(iwest,iFe)')
                idTbry(iwest,iFe)=varid
              CASE ('idTbry(ieast,iFe)')
                idTbry(ieast,iFe)=varid
              CASE ('idTbry(isouth,iFe)')
                idTbry(isouth,iFe)=varid
              CASE ('idTbry(inorth,iFe)')
                idTbry(inorth,iFe)=varid
              CASE ('idTclm(iFe)')
                idTclm(iFe)=varid
# endif
#endif
#ifdef POLLOCK_FOOD
              CASE ('idTvar(iNO3)')
                idTvar(iNO3)=varid
              CASE ('idTvar(iPh)')
                idTvar(iPh)=varid
              CASE ('idTvar(iPSEgg)')
                idTvar(iPSegg)=varid
              CASE ('idTvar(iPSNaup)')
                idTvar(iPSNaup)=varid
              CASE ('idTvar(iPSCop)')
                idTvar(iPSCop)=varid
              CASE ('idTvar(iPSAd)')
                idTvar(iPSAd)=varid
              CASE ('idTvar(iNCa)')
                idTvar(iNCa)=varid
              CASE ('idTvar(iEup)')
                idTvar(iEup)=varid
              CASE ('idTbry(iwest,iNO3)')
                idTbry(iwest,iNO3)=varid
              CASE ('idTbry(ieast,iNO3)')
                idTbry(ieast,iNO3)=varid
              CASE ('idTbry(isouth,iNO3)')
                idTbry(isouth,iNO3)=varid
              CASE ('idTbry(inorth,iNO3)')
                idTbry(inorth,iNO3)=varid
              CASE ('idTbry(iwest,iPh)')
                idTbry(iwest,iPh)=varid
              CASE ('idTbry(ieast,iPh)')
                idTbry(ieast,iPh)=varid
              CASE ('idTbry(isouth,iPh)')
                idTbry(isouth,iPh)=varid
              CASE ('idTbry(inorth,iPh)')
                idTbry(inorth,iPh)=varid
              CASE ('idTbry(iwest,iPSEgg)')
                idTbry(iwest,iPSEgg)=varid
              CASE ('idTbry(ieast,iPSEgg)')
                idTbry(ieast,iPSEgg)=varid
              CASE ('idTbry(isouth,iPSEgg)')
                idTbry(isouth,iPSEgg)=varid
              CASE ('idTbry(inorth,iPSEgg)')
                idTbry(inorth,iPSEgg)=varid
              CASE ('idTbry(iwest,iPSNaup)')
                idTbry(iwest,iPSNaup)=varid
              CASE ('idTbry(ieast,iPSNaup)')
                idTbry(ieast,iPSNaup)=varid
              CASE ('idTbry(isouth,iPSNaup)')
                idTbry(isouth,iPSNaup)=varid
              CASE ('idTbry(inorth,iPSNaup)')
                idTbry(inorth,iPSNaup)=varid
              CASE ('idTbry(iwest,iPSCop)')
                idTbry(iwest,iPSCop)=varid
              CASE ('idTbry(ieast,iPSCop)')
                idTbry(ieast,iPSCop)=varid
              CASE ('idTbry(isouth,iPSCop)')
                idTbry(isouth,iPSCop)=varid
              CASE ('idTbry(inorth,iPSCop)')
                idTbry(inorth,iPSCop)=varid
              CASE ('idTbry(iwest,iPSAd)')
                idTbry(iwest,iPSAd)=varid
              CASE ('idTbry(ieast,iPSAd)')
                idTbry(ieast,iPSAd)=varid
              CASE ('idTbry(isouth,iPSAd)')
                idTbry(isouth,iPSAd)=varid
              CASE ('idTbry(inorth,iPSAd)')
                idTbry(inorth,iPSAd)=varid
              CASE ('idTbry(iwest,iNCa)')
                idTbry(iwest,iNCa)=varid
              CASE ('idTbry(ieast,iNCa)')
                idTbry(ieast,iNCa)=varid
              CASE ('idTbry(isouth,iNCa)')
                idTbry(isouth,iNCa)=varid
              CASE ('idTbry(inorth,iNCa)')
                idTbry(inorth,iNCa)=varid
              CASE ('idTbry(iwest,iEup)')
                idTbry(iwest,iEup)=varid
              CASE ('idTbry(ieast,iEup)')
                idTbry(ieast,iEup)=varid
              CASE ('idTbry(isouth,iEup)')
                idTbry(isouth,iEup)=varid
              CASE ('idTbry(inorth,iEup)')
                idTbry(inorth,iEup)=varid
#endif
#ifdef DIAGNOSTICS_UV
              CASE ('idDu2d(M2pgrd)')
                idDu2d(M2pgrd)=varid
              CASE ('idDv2d(M2pgrd)')
                idDv2d(M2pgrd)=varid
              CASE ('idDu2d(M2strs)')
                idDu2d(M2strs)=varid
              CASE ('idDv2d(M2strs)')
                idDv2d(M2strs)=varid
              CASE ('idDu2d(M2rate)')
                idDu2d(M2rate)=varid
              CASE ('idDv2d(M2rate)')
                idDv2d(M2rate)=varid
# ifdef UV_ADV
              CASE ('idDu2d(M2hadv)')
                idDu2d(M2hadv)=varid
              CASE ('idDv2d(M2hadv)')
                idDv2d(M2hadv)=varid
# endif
# ifdef UV_COR
              CASE ('idDu2d(M2fcor)')
                idDu2d(M2fcor)=varid
              CASE ('idDv2d(M2fcor)')
                idDv2d(M2fcor)=varid
# endif
# if defined UV_VIS2 || defined UV_VIS4
              CASE ('idDu2d(M2hvis)')
                idDu2d(M2hvis)=varid
              CASE ('idDv2d(M2hvis)')
                idDv2d(M2hvis)=varid
# endif
# ifdef SOLVE3D
              CASE ('idDu3d(M3pgrd)')
                idDu3d(M3pgrd)=varid
              CASE ('idDv3d(M3pgrd)')
                idDv3d(M3pgrd)=varid
              CASE ('idDu3d(M3vvis)')
                idDu3d(M3vvis)=varid
              CASE ('idDv3d(M3vvis)')
                idDv3d(M3vvis)=varid
              CASE ('idDu3d(M3rate)')
                idDu3d(M3rate)=varid
              CASE ('idDv3d(M3rate)')
                idDv3d(M3rate)=varid
#  ifdef UV_ADV
              CASE ('idDu3d(M3hadv)')
                idDu3d(M3hadv)=varid
              CASE ('idDv3d(M3hadv)')
                idDv3d(M3hadv)=varid
              CASE ('idDu3d(M3vadv)')
                idDu3d(M3vadv)=varid
              CASE ('idDv3d(M3vadv)')
                idDv3d(M3vadv)=varid
#  endif
#  ifdef UV_COR
              CASE ('idDu3d(M3fcor)')
                idDu3d(M3fcor)=varid
              CASE ('idDv3d(M3fcor)')
                idDv3d(M3fcor)=varid
#  endif
#  if defined UV_VIS2 || defined UV_VIS4
              CASE ('idDu3d(M3hvis)')
                idDu3d(M3hvis)=varid
              CASE ('idDv3d(M3hvis)')
                idDv3d(M3hvis)=varid
#  endif
# endif
#endif
#ifdef DIAGNOSTICS_TS
              CASE ('idDtrc(iTrate)')
                load=.TRUE.
              CASE ('idDtrc(iThadv)')
                load=.TRUE.
              CASE ('idDtrc(iTvadv)')
                load=.TRUE.
# if defined TS_DIF2 || defined TS_DIF4
              CASE ('idDtrc(iThdif)')
                load=.TRUE.
# endif
              CASE ('idDtrc(iTvdif)')
                load=.TRUE.
#endif
#if defined FORWARD_READ || defined FORWARD_WRITE
              CASE ('idRuct')
                idRuct=varid
              CASE ('idRu2d')
                idRu2d=varid
              CASE ('idRu3d')
                idRu3d=varid
              CASE ('idRvct')
                idRvct=varid
              CASE ('idRv2d')
                idRv2d=varid
              CASE ('idRv3d')
                idRv3d=varid
              CASE ('idRzet')
                idRzet=varid
              CASE ('idUfx1')
                idUfx1=varid
              CASE ('idUfx2')
                idUfx2=varid
              CASE ('idVfx1')
                idVfx1=varid
              CASE ('idVfx2')
                idVfx2=varid
#endif
#ifdef FOUR_DVAR
# ifdef W4DVAR
              CASE ('idCG_P')
                idCG_P=varid
              CASE ('idCG_R')
                idCG_R=varid
              CASE ('idCG_S')
                idCG_S=varid
              CASE ('idCG_V')
                idCG_V=varid
              CASE ('idCG_X')
                idCG_X=varid
              CASE ('idCG_Z')
                idCG_Z=varid
# endif
              CASE ('idNLmo')
                idNLmo=varid
              CASE ('idNobs')
                idNobs=varid
              CASE ('idObsD')
                idObsD=varid
              CASE ('idObsT')
                idObsT=varid
              CASE ('idObsX')
                idObsX=varid
              CASE ('idObsY')
                idObsY=varid
              CASE ('idObsZ')
                idObsZ=varid
              CASE ('idOday')
                idOday=varid
              CASE ('idOerr')
                idOerr=varid
              CASE ('idOtyp')
                idOtyp=varid
              CASE ('idOval')
                idOval=varid
              CASE ('idOvar')
                idOvar=varid
              CASE ('idKhor')
                idKhor=varid
              CASE ('idKver')
                idKver=varid
# ifdef REPRESENTERS
              CASE ('idRepC')
                idRepC=varid
# endif
              CASE ('idTLmo')
                idTLmo=varid
#endif
              CASE DEFAULT
                load=.FALSE.
            END SELECT
!
!  Load variable data into information arrays.
!
            IF (load) THEN
              load=.FALSE.
              IF (varid.gt.MV) THEN
                WRITE (stdout,60) MV, varid
                STOP
              END IF
              DO i=1,5
                Vname(i,varid)=TRIM(ADJUSTL(Vinfo(i)))
              END DO            
              DO ng=1,Ngrids
                Iinfo(1,varid,ng)=gtype
                Fscale(varid,ng)=scale
              END DO
#ifdef T_PASSIVE
!
!  Adjust information for all inert passive tracers.
!
              IF (parse) THEN
                parse=.FALSE.
                IF (NPT.gt.0) THEN
                  varid=varid-1
                  DO i=1,NPT
                    varid=varid+1
                    SELECT CASE (TRIM(ADJUSTL(Vinfo(6))))
                      CASE ('idTvar(inert(i))')
                        idTvar(inert(i))=varid
                      CASE ('idTbry(iwest,inert(i))')
                        idTbry(iwest,inert(i))=varid
                      CASE ('idTbry(ieast,inert(i))')
                        idTbry(ieast,inert(i))=varid
                      CASE ('idTbry(isouth,inert(i))')
                        idTbry(isouth,inert(i))=varid
                      CASE ('idTbry(inorth,inert(i))')
                        idTbry(inorth,inert(i))=varid
                    END SELECT
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i2.2)')                   &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                    WRITE (Vname(2,varid),'(a,a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', type ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                END IF
              END IF
#endif
#ifdef SEDIMENT
!
!  Adjust information for all sediment tracers.
!
              SELECT CASE (Vinfo(1))
                CASE ('mud_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idTvar(idmud(i))=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('sand_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idTvar(isand(i))=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('mudfrac_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idfrac(i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('sandfrac_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idfrac(NCS+i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('mudmass_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idBmas(i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('sandmass_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idBmas(NCS+i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('bedload_Umud_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idUbld(i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('bedload_Usand_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idUbld(NCS+i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('bedload_Vmud_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idVbld(i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('bedload_Vsand_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idVbld(NCS+i)=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('river_mud_')
                  IF (NCS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NCS
                      varid=varid+1
                      idRtrc(idmud(i))=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
                CASE ('river_sand_')
                  IF (NNS.gt.0) THEN
                    varid=varid-1
                    DO i=1,NNS
                      varid=varid+1
                      idRtrc(isand(i))=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i2.2)')                 &
     &                      TRIM(ADJUSTL(Vinfo(1))), i
                      WRITE (Vname(2,varid),'(a,a,i2.2)')               &
     &                      TRIM(ADJUSTL(Vinfo(2))), ', size class ', i
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END DO
                  END IF
              END SELECT
#endif
#ifdef ECOSIM
!
!  Adjust information for all EcoSim tracers.
!
              SELECT CASE (Vinfo(1))
                CASE ('Bac_C')
                  varid=varid-1
                  DO i=1,Nbac
                    varid=varid+1
                    idTvar(iBacC(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Bac_C', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'bacteria, carbon group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Bac_F')
                  varid=varid-1
                  DO i=1,Nbac
                    varid=varid+1
                    idTvar(iBacF(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Bac_F', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'bacteria, iron group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Bac_N')
                  varid=varid-1
                  DO i=1,Nbac
                    varid=varid+1
                    idTvar(iBacN(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Bac_N', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'bacteria, nitrogen group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Bac_P')
                  varid=varid-1
                  DO i=1,Nbac
                    varid=varid+1
                    idTvar(iBacP(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Bac_P', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'bacteria, phosphorus group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('CDM_C')
                  varid=varid-1
                  DO i=1,Ndom
                    varid=varid+1
                    idTvar(iCDMC(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'CDM_C', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'color degradational matter, carbon group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('DOM_C')
                  varid=varid-1
                  DO i=1,Ndom
                    varid=varid+1
                    idTvar(iDOMC(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'DOM_C', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'dissolved organic matter, carbon group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('DOM_N')
                  varid=varid-1
                  DO i=1,Ndom
                    varid=varid+1
                    idTvar(iDOMN(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'DOM_N', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'dissolved organic matter, nitrogen group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('DOM_P')
                  varid=varid-1
                  DO i=1,Ndom
                    varid=varid+1
                    idTvar(iDOMP(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'DOM_P', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                  'dissolved organic matter, phosphorus group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Fec_C')
                  varid=varid-1
                  DO i=1,Nfec
                    varid=varid+1
                    idTvar(iFecC(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Fec_C', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'fecal matter, carbon group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Fec_F')
                  varid=varid-1
                  DO i=1,Nfec
                    varid=varid+1
                    idTvar(iFecF(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Fec_F', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'fecal matter, iron group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Fec_N')
                  varid=varid-1
                  DO i=1,Nfec
                    varid=varid+1
                    idTvar(iFecN(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Fec_N', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'fecal matter, nitrogen group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Fec_P')
                  varid=varid-1
                  DO i=1,Nfec
                    varid=varid+1
                    idTvar(iFecP(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Fec_P', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'fecal matter, phosphorus group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Fec_S')
                  varid=varid-1
                  DO i=1,Nfec
                    varid=varid+1
                    idTvar(iFecS(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Fec_S', i
                    WRITE (Vname(2,varid),'(a,i1)')                     &
     &                    'fecal matter, silica group ', i
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Phy_C')
                  varid=varid-1
                  DO i=1,Nphy
                    varid=varid+1
                    idTvar(iPhyC(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Phy_C', i
                    WRITE (Vname(2,varid),'(a,a)')                      &
     &                    TRIM(PhyName(i)), ', carbon group'
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Phy_F')
                  varid=varid-1
                  DO i=1,Nphy
                    varid=varid+1
                    idTvar(iPhyF(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Phy_F', i
                    WRITE (Vname(2,varid),'(a,a)')                      &
     &                    TRIM(PhyName(i)), ', iron group'
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Phy_N')
                  varid=varid-1
                  DO i=1,Nphy
                    varid=varid+1
                    idTvar(iPhyN(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Phy_N', i
                    WRITE (Vname(2,varid),'(a,a)')                      &
     &                    TRIM(PhyName(i)), ', nitrogen group'
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Phy_P')
                  varid=varid-1
                  DO i=1,Nphy
                    varid=varid+1
                    idTvar(iPhyP(i))=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,i1)') 'Phy_P', i
                    WRITE (Vname(2,varid),'(a,a)')                      &
     &                    TRIM(PhyName(i)), ', phosphorus group'
                    WRITE (Vname(3,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('Phy_S')
                  j=0
                  DO i=1,Nphy
                    IF (PHY(i).le.2) j=j+1
                  END DO                
                  IF (j.gt.0) varid=varid-1
                  DO i=1,Nphy
                    IF (PHY(i).le.2) THEN
                      varid=varid+1
                      idTvar(iPhyS(i))=varid
                      DO ng=1,Ngrids
                        Fscale(varid,ng)=scale
                        Iinfo(1,varid,ng)=gtype
                      END DO
                      WRITE (Vname(1,varid),'(a,i1)') 'Phy_S', i
                      WRITE (Vname(2,varid),'(a,a)')                    &
     &                      TRIM(PhyName(i)), ', silica group'
                      WRITE (Vname(3,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(3)))
                      WRITE (Vname(4,varid),'(a,a)')                    &
     &                      TRIM(Vname(1,varid)), ', scalar, series'
                      WRITE (Vname(5,varid),'(a)')                      &
     &                      TRIM(ADJUSTL(Vinfo(5)))
                    END IF
                  END DO
                CASE ('Pigs_XY')
                  varid=varid-1
                  DO j=1,Npig 
                    DO i=1,Nphy
                      IF (PIG(PHY(i),j).gt.0) THEN
                        varid=varid+1
                        idTvar(iPigs(i,j))=varid
                        DO ng=1,Ngrids
                          Fscale(varid,ng)=scale
                          Iinfo(1,varid,ng)=gtype
                        END DO
                        WRITE (Vname(1,varid),'(a,i1,i1)') 'Pigs_', i, j
                        WRITE (Vname(2,varid),'(a,a,a)')                &
     &                        TRIM(PhyName(i)), ', ', TRIM(PigName(j))
                        WRITE (Vname(3,varid),'(a)')                    &
     &                        TRIM(ADJUSTL(Vinfo(3)))
                        WRITE (Vname(4,varid),'(a,a)')                  &
     &                        TRIM(Vname(1,varid)), ', scalar, series'
                        WRITE (Vname(5,varid),'(a)')                    &
     &                        TRIM(ADJUSTL(Vinfo(5)))
                      END IF
                    END DO
                  END DO
              END SELECT
#endif
#ifdef DIAGNOSTICS_TS
!
!  Adjust information for tracer diagnostic variables.  This needs to be 
!  done last because it needs all the tracers variable names.
!

              SELECT CASE (Vinfo(1))
                CASE ('_rate')
                  varid=varid-1
                  DO i=1,MT
                    varid=varid+1
                    idDtrc(i,iTrate)=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,a)')                      &
     &                    TRIM(ADJUSTL(Vname(1,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(1)))
                    WRITE (Vname(2,varid),'(a,", ",a)')                 &
     &                    TRIM(ADJUSTL(Vname(2,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(2)))
                    WRITE (Vname(3,varid),'(a,1x,a)')                   &
     &                    TRIM(ADJUSTL(Vname(3,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('_hadv')
                  varid=varid-1
                  DO i=1,MT
                    varid=varid+1
                    idDtrc(i,iThadv)=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,a)')                      &
     &                    TRIM(ADJUSTL(Vname(1,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(1)))
                    WRITE (Vname(2,varid),'(a,", ",a)')                 &
     &                    TRIM(ADJUSTL(Vname(2,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(2)))
                    WRITE (Vname(3,varid),'(a,1x,a)')                   &
     &                    TRIM(ADJUSTL(Vname(3,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
                CASE ('_vadv')
                  varid=varid-1
                  DO i=1,MT
                    varid=varid+1
                    idDtrc(i,iTvadv)=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,a)')                      &
     &                    TRIM(ADJUSTL(Vname(1,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(1)))
                    WRITE (Vname(2,varid),'(a,", ",a)')                 &
     &                    TRIM(ADJUSTL(Vname(2,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(2)))
                    WRITE (Vname(3,varid),'(a,1x,a)')                   &
     &                    TRIM(ADJUSTL(Vname(3,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
# if defined TS_DIF2 || defined TS_DIF4
                CASE ('_hdiff')
                  varid=varid-1
                  DO i=1,MT
                    varid=varid+1
                    idDtrc(i,iThdif)=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,a)')                      &
     &                    TRIM(ADJUSTL(Vname(1,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(1)))
                    WRITE (Vname(2,varid),'(a,", ",a)')                 &
     &                    TRIM(ADJUSTL(Vname(2,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(2)))
                    WRITE (Vname(3,varid),'(a,1x,a)')                   &
     &                    TRIM(ADJUSTL(Vname(3,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                      &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
# endif
                CASE ('_vdiff')
                  varid=varid-1
                  DO i=1,MT
                    varid=varid+1
                    idDtrc(i,iTvdif)=varid
                    DO ng=1,Ngrids
                      Fscale(varid,ng)=scale
                      Iinfo(1,varid,ng)=gtype
                    END DO
                    WRITE (Vname(1,varid),'(a,a)')                      &
     &                    TRIM(ADJUSTL(Vname(1,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(1)))
                    WRITE (Vname(2,varid),'(a,", ",a)')                 &
     &                    TRIM(ADJUSTL(Vname(2,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(2)))
                    WRITE (Vname(3,varid),'(a,1x,a)')                   &
     &                    TRIM(ADJUSTL(Vname(3,idTvar(i)))),            &
     &                    TRIM(ADJUSTL(Vinfo(3)))
                    WRITE (Vname(4,varid),'(a,a)')                      &
     &                    TRIM(Vname(1,varid)), ', scalar, series'
                    WRITE (Vname(5,varid),'(a)')                        &
     &                    TRIM(ADJUSTL(Vinfo(5)))
                  END DO
              END SELECT
#endif
            ELSE
              varid=varid-1
!!            WRITE (stdout,70) TRIM(ADJUSTL(Vinfo(1))),                &
!!   &                          TRIM(ADJUSTL(Vinfo(6)))
            END IF
          END IF
        END DO
        GOTO 40
  30    WRITE (stdout,80) TRIM(ADJUSTL(Vinfo(1)))
        STOP
  40    CLOSE (inp)
!
!
!-----------------------------------------------------------------------
!  Set model state variables indices.
!-----------------------------------------------------------------------
!
        idSvar(isFsur)=idFsur
        idSvar(isUbar)=idUbar
        idSvar(isVbar)=idVbar
#ifdef SOLVE3D
        idSvar(isUvel)=idUvel
        idSvar(isVvel)=idVvel
        DO i=1,MT
          idSvar(isTvar(i))=idTvar(i)
        END DO
#endif   
!
  50    FORMAT (/,' MOD_NCPARAM - Unable to open variable information', &
     &          ' file: ',/,15x,a,/,15x,'Default file is located in',   &
     &          ' source directory.')
  60    FORMAT (/,' MOD_NCPARAM - too small dimension ',                &
     &          'parameter, MV = ',2i5,/,15x,                           &
     &          'change file  mod_ncparam.F  and recompile.')
  70    FORMAT (/,' MOD_NCPARM - Cannot load information for ',         &
     &          'variable: ',a,/,15x,'Need CASE construct for: ',a)
  80    FORMAT (/,' MOD_NCPARM - Error while reading information ',     &
     &          'for variable: ',a)
  90    FORMAT (a,i2.2)
 100    FORMAT (a,a,i2.2)
 110    FORMAT (a)
 120    FORMAT (a,a)

        RETURN
        END SUBROUTINE initialize_ncparam

      END MODULE mod_ncparam
