#include "cppdefs.h"
      MODULE mod_tides
#if defined SSH_TIDES || defined UV_TIDES
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  Tidal Components:                                                   !
!                                                                      !
!  Each of the following arrays has a dimension in tidal components    !
!  classified by period:                                               !
!                                                                      !
!    semi-diurnal:  M2, S2, N2, K2  (12.42, 12.00, 12.66, 11.97h)      !
!         diurnal:  K1, O1, P1, Q1  (23.93, 25.82, 24.07, 26.87h)      !
!                                                                      !
!  and other longer periods. The order of these tidal components is    !
!  irrelevant here.  The number of components to USE is depends on     !
!  the regional application.                                           !
!                                                                      !
!  SSH_Tamp     Tidal elevation amplitude (m) at RHO-points.           !
!  SSH_Tphase   Tidal elevation phase (degrees/360) at RHO-points.     !
!  Tperiod      Tidal period (s).                                      !
!  UV_Tangle    Tidal current angle (radians; counterclockwise         !
!                 from EAST and rotated to curvilinear grid) at        !
!                 RHO-points.                                          !
!  UV_Tmajor    Maximum tidal current: tidal ellipse major axis        !
!                 (m/s) at RHO-points.                                 !
!  UV_Tminor    Minimum tidal current: tidal ellipse minor axis        !
!                 (m/s) at RHO-points.                                 !
!  UV_Tphase    Tidal current phase (degrees/360) at RHO-points.       !
!                                                                      !
!=======================================================================
!
        USE mod_kinds
        USE mod_param

        implicit none

        TYPE T_TIDES

          real(r8), pointer :: Tperiod(:)
# if defined SSH_TIDES
          real(r8), pointer :: SSH_Tamp(:,:,:)
          real(r8), pointer :: SSH_Tphase(:,:,:)
# endif
# if defined UV_TIDES
          real(r8), pointer :: UV_Tangle(:,:,:)
          real(r8), pointer :: UV_Tmajor(:,:,:)
          real(r8), pointer :: UV_Tminor(:,:,:)
          real(r8), pointer :: UV_Tphase(:,:,:)
# endif
        END TYPE T_TIDES

        TYPE (T_TIDES), allocatable :: TIDES(:)

# if defined TIDES_ASTRO
        integer, parameter :: MAX_SAT = 10
        integer  :: doodson(6,MTC)
        real(r8) :: phase_corr(MTC)
        integer  :: num_sat(MTC)
        integer  :: sat_doodson(3,MAX_SAT,MTC)
        real(r8) :: sat_phase_corr(MAX_SAT,MTC)
        real(r8) :: sat_amp(MAX_SAT,MTC)
        real(r8) :: sat_flag(MAX_SAT,MTC)
# endif

      CONTAINS

      SUBROUTINE allocate_tides (ng, LBi, UBi, LBj, UBj)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine allocates all variables in the module for all nested   !
!  grids.                                                              !
!                                                                      !
!=======================================================================
!
      USE mod_param
!
!  Local variable declarations.
!
      integer, intent(in) :: ng, LBi, UBi, LBj, UBj
!
!-----------------------------------------------------------------------
!  Initialize module variables.
!-----------------------------------------------------------------------
!
      IF (ng.eq.1) allocate ( TIDES(Ngrids) )
!
      allocate ( TIDES(ng) % Tperiod(MTC)  )

# if defined SSH_TIDES
      allocate ( TIDES(ng) % SSH_Tamp(LBi:UBi,LBj:UBj,MTC) )
      allocate ( TIDES(ng) % SSH_Tphase(LBi:UBi,LBj:UBj,MTC) )
# endif

# if defined UV_TIDES
      allocate ( TIDES(ng) % UV_Tangle(LBi:UBi,LBj:UBj,MTC) )
      allocate ( TIDES(ng) % UV_Tmajor(LBi:UBi,LBj:UBj,MTC) )
      allocate ( TIDES(ng) % UV_Tminor(LBi:UBi,LBj:UBj,MTC) )
      allocate ( TIDES(ng) % UV_Tphase(LBi:UBi,LBj:UBj,MTC) )
# endif

      RETURN
      END SUBROUTINE allocate_tides

      SUBROUTINE initialize_tides (ng, tile)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine initialize all variables in the module using first     !
!  touch distribution policy. In shared-memory configuration, this     !
!  operation actually performs propagation of the  "shared arrays"     !
!  across the cluster, unless another policy is specified to           !
!  override the default.                                               !
!                                                                      !
!=======================================================================
!
      USE mod_param
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
      integer :: Imin, Imax, Jmin, Jmax
      integer :: i, itide, j

      real(r8), parameter :: IniVal = 0.0_r8
!
# include "tile.h"
!
!  Set array initialization range.
!
#ifdef _OPENMP
      IF (WESTERN_EDGE) THEN
        Imin=LBi
      ELSE
        Imin=Istr
      END IF
      IF (EASTERN_EDGE) THEN
        Imax=UBi
      ELSE
        Imax=Iend
      END IF
      IF (SOUTHERN_EDGE) THEN
        Jmin=LBj
      ELSE
        Jmin=Jstr
      END IF
      IF (NORTHERN_EDGE) THEN
        Jmax=UBj
      ELSE
        Jmax=Jend
      END IF
#else
      Imin=LBi
      Imax=UBi
      Jmin=LBj
      Jmax=UBj
#endif
!
!-----------------------------------------------------------------------
!  Initialize module variables.
!-----------------------------------------------------------------------
!
      IF (SOUTH_WEST_TEST) THEN
        DO itide=1,MTC
          TIDES(ng) % Tperiod(itide) = IniVal
        END DO
      END IF

      DO itide=1,MTC
# if defined SSH_TIDES
        DO j=Jmin,Jmax
          DO i=Imin,Imax
            TIDES(ng) % SSH_Tamp(i,j,itide) = IniVal
            TIDES(ng) % SSH_Tphase(i,j,itide) = IniVal
          END DO
        END DO
# endif
# if defined UV_TIDES
        DO j=Jmin,Jmax
          DO i=Imin,Imax
            TIDES(ng) % UV_Tangle(i,j,itide) = IniVal
            TIDES(ng) % UV_Tmajor(i,j,itide) = IniVal
            TIDES(ng) % UV_Tminor(i,j,itide) = IniVal
            TIDES(ng) % UV_Tphase(i,j,itide) = IniVal
          END DO
        END DO
# endif
      END DO

# if defined TIDES_ASTRO
! Assumes tides are in the order Q1, O1, P1, K1, N2, M2, S2, K2

      doodson = reshape( (/ 1, -2, 0, 1, 0, 0,                          &
     &                      1, -1, 0, 0, 0, 0,                          &
     &                      1, 1, -2, 0, 0, 0,                          &
     &                      1, 1,  0, 0, 0, 0,                          &
     &                      2, -1, 0, 1, 0, 0,                          &
     &                      2, 0,  0, 0, 0, 0,                          &
     &                      2, 2, -2, 0, 0, 0,                          &
     &                      2, 2,  0, 0, 0, 0/), (/ 6,MTC /) )

      phase_corr = (/ -0.25, -0.25, -0.25, -0.75,                       &
     &                             0.0, 0.0, 0.0, 0.0 /)

      num_sat = (/ 10, 8, 6, 10, 4, 9, 3, 5 /)

      sat_doodson(:,:,1) = reshape( (/ -2, -3, 0, -2, -2, 0,            &
     &                                 -1, -2, 0, -1, -1, 0,            &
     &                                 -1,  0, 0,  0, -2, 0,            &
     &                                 -1,  0, 1,  0, -1, 0,            &
     &                                  1,  0, 0,  2,  0, 0 /),         &
     &                               (/ 3,MAX_SAT /) )
      sat_doodson(:,:,2) = reshape( (/ -1, 0, 0, 0, -2, 0,              &
     &                                 0, -1, 0, 1, -1, 0,              &
     &                                 1,  0, 0, 1,  1, 0,              &
     &                                 2,  0, 0, 2,  1, 0,              &
     &                                 -99, -99, -99, -99, -99, -99 /), &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,3) = reshape( (/ 0, -2, 0, 0, -1, 0,              &
     &                                 0,  0, 2, 1,  0, 0,              &
     &                                 2,  0, 0, 2,  1, 0,              &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99 /), &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,4) = reshape( (/ -2, -1, 0, -1, -1, 0,            &
     &                                 -1,  0, 0, -1,  1, 0,            &
     &                                 0, -2, 0,  0, -1, 0,             &
     &                                 0,  1, 0,  0,  2, 0,             &
     &                                 1,  0, 0,  1,  1, 0 /),          &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,5) = reshape( (/ -2, -2, 0, -1,  0, 1,            &
     &                                  0, -2, 0,  0, -1, 0,            &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99 /), &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,6) = reshape( (/ -1, -1, 0, -1,  0, 0,            &
     &                                  0, -2, 0,  0, -1, 0,            &
     &                                  1, -1, 0,  1,  0, 0,            &
     &                                  1,  1, 0,  2,  0, 0,            &
     &                                  2,  1, 0, -99, -99, -99 /),     &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,7) = reshape( (/ 0, -1, 0, 1, 0, 0,               &
     &                                 2,  0, 0, -99, -99, -99,         &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99 /), &
     &                                (/ 3,MAX_SAT /) )
      sat_doodson(:,:,8) = reshape( (/ -1, 0, 0, -1, 1, 0,              &
     &                                 0, -1, 0,  0, 1, 0,              &
     &                                 0,  2, 0, -99, -99, -99,         &
     &                                 -99, -99, -99, -99, -99, -99,    &
     &                                 -99, -99, -99, -99, -99, -99 /), &
     &                                (/ 3,MAX_SAT /) )

      sat_phase_corr = reshape(                                         &
     & (/ 0.5, 0.5, 0.75, 0.75, 0.75, 0.5, 0.0, 0.0, 0.75, 0.5,         &
     &    0.25, 0.5, 0.0, 0.25, 0.75, 0.25, 0.5, 0.5, -99., -99.,       &
     &    0.0, 0.5, 0.5, 0.75, 0.5, 0.5, -99., -99., -99., -99.,        &
     &    0.0, 0.75, 0.25, 0.75, 0.0, 0.5, 0.0, 0.5, 0.25, 0.25,        &
     &    0.5, 0.0, 0.0, 0.5, -99., -99., -99., -99., -99., -99.,       &
     &    0.75, 0.75, 0.0, 0.5, 0.25, 0.75, 0.75, 0.0, 0.0, -99.,       &
     &    0.0, 0.75, 0.0, -99., -99., -99., -99., -99., -99., -99.,     &
     &    0.75, 0.75, 0.5, 0.0, 0.0, -99., -99., -99., -99., -99. /),   &
     &      (/ MAX_SAT,MTC /) )

      sat_amp = reshape(                                                &
     &      (/ 0.0007, 0.0038, 0.0010, 0.0115, 0.0292,                  &
     &         0.0057, 0.0008, 0.1884, 0.0018, 0.0028,                  &
     &         0.0003, 0.0058, 0.1885, 0.0004, 0.0029,                  &
     &         0.0004, 0.0064, 0.0010, 9999.0, 9999.0,                  &
     &         0.0008, 0.0112, 0.0004, 0.0004, 0.0015,                  &
     &         0.0003, 9999.0, 9999.0, 9999.0, 9999.0,                  &
     &         0.0002, 0.0001, 0.0007, 0.0001, 0.0001,                  &
     &         0.0198, 0.1356, 0.0029, 0.0002, 0.0001,                  &
     &         0.0039, 0.0008, 0.0005, 0.0373, 9999.0,                  &
     &         9999.0, 9999.0, 9999.0, 9999.0, 9999.0,                  &
     &         0.0001, 0.0004, 0.0005, 0.0373, 0.0001,                  &
     &         0.0009, 0.0002, 0.0006, 0.0002, 9999.0,                  &
     &         0.0022, 0.0001, 0.0001, 9999.0, 9999.0,                  &
     &         9999.0, 9999.0, 9999.0, 9999.0, 9999.0,                  &
     &         0.0024, 0.0004, 0.0128, 0.2980, 0.0324,                  &
     &         9999.0, 9999.0, 9999.0, 9999.0, 9999.0 /),               &
     &      (/ MAX_SAT,MTC /) )

      sat_flag = reshape(                                               &
     &       (/ 0, 0, 1, 1, 1, 0, 0, 0, 1, 0,                           &
     &          1, 0, 0, 1, 1, 1, 0, 0, -99, -99,                       &
     &          0, 0, 0, 1, 0, 0, -99, -99, -99, -99,                   &
     &          0, 1, 1, 1, 0, 0, 0, 0, 1, 1,                           &
     &          0, 0, 0, 0, -99, -99, -99, -99, -99, -99,               &
     &          2, 2, 0, 0, 2, 2, 2, 0, 0, -99,                         &
     &          0, 2, 0, -99, -99, -99, -99, -99, -99, -99,             &
     &          2, 2, 0, 0, 0, -99, -99, -99, -99, -99 /),              &
     &      (/ MAX_SAT,MTC /) )

# endif
      RETURN
      END SUBROUTINE initialize_tides
#endif

#ifdef TIDES_ASTRO
      SUBROUTINE tide_astro(ttime,vux,fx,xlat)
      USE mod_scalars
      implicit none

      real(r8), intent(in)  :: ttime
      real(r8), intent(in)  :: xlat
      real(r8), intent(out) :: vux(MTC), fx(MTC)

! Local variables
      real(r8) :: slat, hh, d1, tau, dtau, freq(MTC)
      real(r8) :: h, pp, s, p, enp, dh, dpp, ds, dp, dnp
      real(r8) :: uu, twopi, sumc, sums, v, rr, vdbl, uudbl
      integer  :: iv, j, iuu, itide, intdays
!
!***********************************************************************
!*  THIS SUBROUTINE CALCULATES V (ASTRONOMICAL PHASE ARGUMENT), U AND F
!*  (NODAL MODULATION PHASE AND AMPLITUDE CORRECTIONS) FOR ALL CONSTITU-
!*  ENTS.
!***********************************************************************
!*  NTIDAL IS THE NUMBER OF MAIN CONSTITUENTS
!*  NTOTAL IS THE NUMBER OF CONSTITUENTS (MAIN + SHALLOW WATER)
!*  FOR  THE GIVEN TIME KH, THE TABLE OF F AND V+U VALUES IS
!*  CALCULATED FOR ALL THE CONSTITUENTS.
!*     F IS THE NODAL MODULATION ADJUSTMENT FACTOR FOR AMPLITUDE
!*     U IS THE NODAL MODULATION ADJUSTMENT FACTOR FOR PHASE
!*     V IS THE ASTRONOMICAL ARGUMENT ADJUSTMENT FOR PHASE.
!
      slat=SIN(deg2rad*xlat)
!
!***********************************************************************
!*  THE ASTRONOMICAL ARGUMENTS ARE CALCULATED BY LINEAR APPROXIMATION
!*  AT THE MID POINT OF THE ANALYSIS PERIOD.
!
!      S=S0+YEARS*DS
!      H=H0+YEARS*DH
!      P=P0+YEARS*DP
!      ENP=ENP0+YEARS*DNP
!      PP=PP0+YEARS*DPP
!        day number measured from January 0.5 1900 (i.e.,
!        1200 UT December 31, 1899
        d1=ttime*sec2day
      twopi = 2.0_r8*pi
!       call gday(31,12,99,18,kd0)
!       d1=d1-dfloat(kd0)-0.5d0
        d1=d1-693961.5
        call astr(d1,h,pp,s,p,enp,dh,dpp,ds,dp,dnp)
      intdays=int(ttime*sec2day)
      hh=real(ttime-intdays*day2sec,r8)/3600._r8
      tau = hh/24._r8 + h - s
      dtau = 365.d0 + dh - ds
!
!***********************************************************************
!*  ONLY THE FRACTIONAL PART OF A SOLAR DAY NEED BE RETAINED FOR COMPU-
!*  TING THE LUNAR TIME TAU.
!
      DO itide=1,MTC
        freq(itide)=(doodson(1,itide)*dtau+doodson(2,itide)*ds+         &
     &               doodson(3,itide)*dh+doodson(4,itide)*dp+           &
     &               doodson(5,itide)*dnp+doodson(6,itide)*dpp)         &
     &               /(365._r8*24._r8)
        vdbl=doodson(1,itide)*tau+doodson(2,itide)*s+                   &
     &               doodson(3,itide)*h+doodson(4,itide)*p+             &
     &               doodson(5,itide)*enp+doodson(6,itide)*pp+          &
     &               phase_corr(itide)
        iv=vdbl
        iv=(iv/2)*2
        v=vdbl-iv
        sumc=1.
        sums=0.
        DO J=1,num_sat(itide)
!
!***********************************************************************
!*  HERE THE SATELLITE AMPLITUDE RATIO ADJUSTMENT FOR LATITUDE IS MADE
!
          rr=sat_amp(J,itide)
          IF (sat_flag(J,itide) == 1) THEN
            rr=sat_amp(J,itide)*0.36309*(1.-5.*SLAT*SLAT)/SLAT
          ELSE IF (sat_flag(J,itide) == 2) THEN
            rr=sat_amp(J,itide)*2.59808*SLAT
          END IF
          uudbl=sat_doodson(1,J,itide)*p+sat_doodson(2,J,itide)*enp+    &
     &          sat_doodson(3,J,itide)*pp+sat_phase_corr(J,itide)
          iuu=uudbl
          uu=uudbl-iuu
          sumc=sumc+rr*COS(uu*twopi)
          sums=sums+rr*SIN(uu*twopi)
        END DO
        fx(itide)=SQRT(sumc*sumc+sums*sums)
        vux(itide)=v+ATAN2(sums,sumc)/twopi
        v=v-int(v)
        if (v.lt.0.) v=v+1.
        vux(itide)=vux(itide)-int(vux(itide))
        if (vux(itide).lt.0.) vux(itide)=vux(itide)+1.
!       write(6,998) kon(k),v*360.,f(k),360.*(vux(k)-v)
!       write(7,998) kon(k),v*360.,f(k),360.*(vux(k)-v)
!998    format(' ',a5,5x,3f12.5)
      end do
!
      RETURN
      END SUBROUTINE tide_astro
      SUBROUTINE astr(d1,h,pp,s,p,np,dh,dpp,ds,dp,dnp)
!        this subroutine calculates the following five ephermides
!        of the sun and moon
!        h = mean longitude of the sum
!        pp = mean longitude of the solar perigee
!        s = mean longitude of the moon
!        p = mean longitude of the lunar perigee
!        np = negative of the longitude of the mean ascending node
!        and their rates of change.
!        Units for the ephermides are cycles and for their derivatives
!        are cycles/365 days
!        The formulae for calculating this ephermides were taken from
!        pages 98 and 107 of the Explanatory Supplement to the
!        Astronomical Ephermeris and the American Ephermis and
!        Nautical Almanac (1961)
!
        implicit none
      real(r8), intent(in)  :: d1
      real(r8), intent(out) :: h, pp, s, p, np, dh, dpp, ds, dp, dnp
      real(r8)  :: d2, f, f2

        d2=d1*1.d-4
        f=360.d0
        f2=f/365.d0
        h=279.696678d0+.9856473354d0*d1+.00002267d0*d2*d2
        pp=281.220833d0+.0000470684d0*d1+.0000339d0*d2*d2+              &
     &  .00000007d0*d2**3
        s=270.434164d0+13.1763965268d0*d1-.000085d0*d2*d2+              &
     &  .000000039d0*d2**3
        p=334.329556d0+.1114040803d0*d1-.0007739d0*d2*d2-               &
     &  .00000026d0*d2**3
        np=-259.183275d0+.0529539222d0*d1-.0001557d0*d2*d2-             &
     &  .00000005d0*d2**3
        h=h/f
        pp=pp/f
        s=s/f
        p=p/f
        np=np/f
        h=h-int(h)
        pp=pp-int(pp)
        s=s-int(s)
        p=p-int(p)
        np=np-int(np)
        dh=.9856473354d0+2.d-8*.00002267d0*d1
        dpp=.0000470684d0+2.d-8*.0000339d0*d1                           &
     &  +3.d-12*.00000007d0*d1**2
        ds=13.1763965268d0-2.d-8*.000085d0*d1+                          &
     &  3.d-12*.000000039d0*d1**2
        dp=.1114040803d0-2.d-8*.0007739d0*d1-                           &
     &  3.d-12*.00000026d0*d1**2
        dnp=+.0529539222d0-2.d-8*.0001557d0*d1-                         &
     &  3.d-12*.00000005d0*d1**2
        dh=dh/f2
        dpp=dpp/f2
        ds=ds/f2
        dp=dp/f2
        dnp=dnp/f2
        RETURN
      END SUBROUTINE astr
#endif
      END MODULE mod_tides
