#include "cppdefs.h"
      SUBROUTINE biology (ng,tile)
!
!========================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine computes the biological sources and sinks and adds     !
!  then the global biological fields.                                  !
!                                                                      !
!  Sarah Hinckley's GOANPZ Code                                        !
!  Implemented by Craig Lewis (CVL)                                    !
!  Modified by Liz Dobbins and Sarah Hinckley                          !
!                                                                      !
!=======================================================================
!
#define NEWSHADE    /* Use Craig's formulation for self shading in PAR calc 
                       Else use Sarah's self-shading from original NPZ code */
#undef  KODIAK_IRAD /* Generate irradiance with curve matching Kodiak data  
                       Else use shortwave radiation (srflx) as irradiance   */
      USE mod_param
      USE mod_forces
      USE mod_grid
      USE mod_ocean
      USE mod_stepping

      integer, intent(in) :: ng, tile

# include "tile.h"
!
#ifdef PROFILE
      CALL wclock_on (ng, iNLM, 15)
#endif
      CALL biology_tile (ng, Istr, Iend, Jstr, Jend,                    &
     &                   LBi, UBi, LBj, UBj, N(ng), NT(ng),             &
     &                   nnew(ng),                                      &
# ifdef MASKING
     &                   GRID(ng) % rmask,                              &
# endif
     &                   GRID(ng) % Hz,                                 &
     &                   GRID(ng) % z_r,                                &
     &                   GRID(ng) % z_w,                                &
     &                   FORCES(ng) % srflx,                            &
     &                   OCEAN(ng) % t)
#ifdef PROFILE
      CALL wclock_off (ng, iNLM, 15)
#endif
      RETURN
      END SUBROUTINE biology
!
!-----------------------------------------------------------------------
      SUBROUTINE biology_tile (ng, Istr, Iend, Jstr, Jend,              &
     &                         LBi, UBi, LBj, UBj, UBk, UBt,            &
     &                         nnew,                                    &
# ifdef MASKING
     &                         rmask,                                   &
# endif
     &                         Hz, z_r, z_w, srflx, t)
!-----------------------------------------------------------------------
!
      USE mod_param
      USE mod_biology
      USE mod_scalars
      USE mod_ocean
      USE mod_grid
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj, UBk, UBt
      integer, intent(in) :: nnew
!
# ifdef ASSUMED_SHAPE
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hz(LBi:,LBj:,:)
      real(r8), intent(in) :: z_r(LBi:,LBj:,:)
      real(r8), intent(in) :: z_w(LBi:,LBj:,0:)
      real(r8), intent(in) :: srflx(LBi:,LBj:)
      real(r8), intent(inout) :: t(LBi:,LBj:,:,:,:)
# else
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hz(LBi:UBi,LBj:UBj,UBk)
      real(r8), intent(in) :: z_r(LBi:UBi,LBj:UBj,UBk)
      real(r8), intent(in) :: z_w(LBi:UBi,LBj:UBj,0:UBk)
      real(r8), intent(in) :: srflx(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: t(LBi:UBi,LBj:UBj,UBk,3,UBt)
# endif
!
!  Local variable declarations.
!
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: i, j, k, itr, itrmx, itrc
      integer :: Iter
      integer :: iday, month, year

      real(r8) :: cff1, cff2, cff3
      real(r8) :: Drate, Pmax, NOup, NHup
      real(r8) :: dtdays
      real(r8) :: LightLim, NOLim, NHLim, IronLim
      real(r8) :: hour, yday, lat, k_phy, Dl
!      real(r8) :: ToptPhS, TmaxPhS,TempbPhS,TempaPhS
      
      real(r8), dimension(N(ng)) :: NO3, DNO3
      real(r8), dimension(N(ng)) :: NH4, DNH4
      real(r8), dimension(N(ng)) :: PhS, DPhS
      real(r8), dimension(N(ng)) :: PhL, DPhL
      real(r8), dimension(N(ng)) :: MZS, DMZS
      real(r8), dimension(N(ng)) :: MZL, DMZL
      real(r8), dimension(N(ng)) :: Cop, DCop
      real(r8), dimension(N(ng)) :: NCa, DNCa
      real(r8), dimension(N(ng)) :: Eup, DEup
      real(r8), dimension(N(ng)) :: Det, DDet
      real(r8), dimension(N(ng)) :: TempFuncPhS
      real(r8), dimension(N(ng)) :: TempFuncPhL
      real(r8), dimension(N(ng)) :: TempFuncMZS
      real(r8), dimension(N(ng)) :: TempFuncMZL
      real(r8), dimension(N(ng)) :: TempFuncCop
      real(r8), dimension(N(ng)) :: TempFuncNeo
      real(r8), dimension(N(ng)) :: TempFuncEup 
      real(r8), dimension(N(ng)) :: PAR, Temp, HzL, Sal
      real(r8), dimension(0:N(ng)) :: z_wL
      real(r8) :: Sal1, Temp1, Dep1, NH4R, MyPred
      real(r8) :: MaxPAR = 0.0_r8
      real(r8) :: TmaxPhL, TminPhL, CK1_PhL, Topt_PhL, CK4_PhL
      real(r8) :: CTL_PhL, CQ_PhL, CTO_PhL, G1_PhL, G2_PhL
      real(r8) :: L1_PhL, L2_PhL, KA_PhL, KB_PhL, BasalMet, Par1
      real(r8) :: TmaxPhS, TminPhS, CK1_PhS, Topt_PhS, CrChlRatio
      real(r8) :: Nitrif,KtBm_PhS, KtBm_PhL, kfePh, respPh, ccrMin
      real(r8) :: TmaxMZS, TmaxMZL, ktBm_MZS, ktBm_MZL, kfeMZS, kfeMZL
   
#ifdef IRON
      real(r8), dimension(N(ng)) :: Iron, DIron
      real(r8) :: Iron1
#endif
#ifdef DIAPAUSE
      logical :: downward = .false., upward = .false.
#endif
!
      real(r8), parameter :: eps  = 1.0E-20_r8
      real(r8), parameter :: minv = 0.0E-20_r8
!
#include "set_bounds.h"
#ifdef EW_PERIODIC
# define I_RANGE Istr,Iend
#else
# define I_RANGE IstrR,IendR
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr,Jend
#else
# define J_RANGE JstrR,JendR
#endif
!
      CALL caldate (r_date, tdays(ng), year, yday, month, iday, hour)
      dtdays = dt(ng)*sec2day/REAL(BioIter(ng),r8)
      k_phy = k_chl / ccr
!
#ifdef DIAPAUSE
!  Based on date, determine if NCa are going upwards or downwards
      downward = .false.
      upward = .false.
      if ( ( RiseStart.lt.RiseEnd .and.                                 &
     &       yday.ge.RiseStart .and. yday.le.RiseEnd ) .or.             &
     &     ( RiseStart.gt.RiseEnd .and.                                 &
     &     ( yday.ge.RiseStart .or. yday.le.RiseEnd ) ) )  then
        upward = .true.
      else if ( ( SinkStart.lt.SinkEnd .and.                            &
     &            yday.ge.SinkStart .and. yday.le.SinkEnd ) .or.        &
     &          ( SinkStart.gt.SinkEnd .and.                            &
     &          ( yday.ge.SinkStart .or. yday.le.SinkEnd ) ) )  then
        downward = .true.
      endif
#endif
!
! ----------------------------------------------------------------------
! Begin HORIZONTAL INDEX LOOPING
! ----------------------------------------------------------------------
! 
      DO j=J_RANGE
         DO i=I_RANGE
!
!  Extract biological variables from tracer arrays, and 
!  restrict their values to be positive definite.  Removed CVL's
!  conservation of mass correction because conflicted with SPLINES.
!  For ROMS 2.2+, convert from "flux form" to concentrations by
!  dividing by grid cell thickness.
           DO k=1,N(ng)
              NO3(k) = MAX( minv, t(i,j,k,nnew,iNO3)/Hz(i,j,k) )
              NH4(k) = MAX( minv, t(i,j,k,nnew,iNH4)/Hz(i,j,k) )
              PhS(k) = MAX( minv, t(i,j,k,nnew,iPhS)/Hz(i,j,k) )
              PhL(k) = MAX( minv, t(i,j,k,nnew,iPhL)/Hz(i,j,k) )
              MZS(k) = MAX( minv, t(i,j,k,nnew,iMZS)/Hz(i,j,k) )
              MZL(k) = MAX( minv, t(i,j,k,nnew,iMZL)/Hz(i,j,k) )
              Cop(k) = MAX( minv, t(i,j,k,nnew,iCop)/Hz(i,j,k) )
              NCa(k) = MAX( minv, t(i,j,k,nnew,iNCa)/Hz(i,j,k) )
              Eup(k) = MAX( minv, t(i,j,k,nnew,iEup)/Hz(i,j,k) )
              Det(k) = MAX( minv, t(i,j,k,nnew,iDet)/Hz(i,j,k) )
#ifdef IRON
              Iron(k) = MAX( minv, t(i,j,k,nnew,iFe)/Hz(i,j,k) )
#endif      
              DNO3(k) = 0.0_r8
              DNH4(k) = 0.0_r8
              DPhS(k) = 0.0_r8
              DPhL(k) = 0.0_r8
              DMZS(k) = 0.0_r8
              DMZL(k) = 0.0_r8
              DCop(k) = 0.0_r8
              DNCa(k) = 0.0_r8
              DEup(k) = 0.0_r8
              DDet(k) = 0.0_r8
#ifdef IRON
              DIron(k) = 0.0_r8
#endif      
            END DO
!
! ----------------------------------------------------------------------
!  Calculate Day Length and PAR with self-shading
! ----------------------------------------------------------------------
!
!  Calculate Day Length
#if defined ANA_SRFLUX && defined ALBEDO
!  Day Length is already accounted for in ANA_SWRAD so disable correction
            Dl = 24.0_r8
#else
!  Day Length calculation (orig from R. Davis) from latitude and declination.
!  cff2 is Solar declination from Oberhuber (1988) (COADS documentation)
!            lat = 58.00  orig from C code
!            lat = 45.00_r8  test for EPOC
!            lat = 59.12   !Latitude of GAK6
            lat = GRID(ng) % latr(i,j)
            cff1 = 2.0_r8 * pi * ( yday-1.0_r8 ) / 365.0_r8
            cff2 = 0.006918_r8 - 0.399912_r8*cos(cff1)                  &
     &           + 0.070257_r8*sin(cff1) - 0.006758_r8*cos(2*cff1)      &
     &           + 0.000907_r8*sin(2*cff1) - 0.002697_r8*cos(3*cff1)    &
     &           + 0.00148_r8*sin(3*cff1)
            cff3 = lat * pi /180.0_r8
            if ( abs( -tan(cff3)*tan(cff2) ) .le. 1.0_r8 ) then
              cff1 = acos( -tan(cff3)*tan(cff2) ) * 180.0_r8 / pi
              Dl = 2.0_r8 / 15.0_r8 * cff1
            else
              if ( yday.gt.90.0_r8 .and. yday.lt.270.0_r8 ) then 
                Dl = 24.0_r8
              else
                Dl = 0.0_r8
              endif
            endif
#endif
!  Calculate PAR at the surface
#ifndef KODIAK_IRAD
!  For PAR, use Shortwave radiation ( = surface solar irradiance)
!  converted from deg C m/s to E/m2/day
            cff2 = srflx(i,j) * rho0 * Cp * 0.394848_r8
!           print *, 'cff2: ', cff2, 'srflx: ', srflx(i,j)
#else
!  For PAR, Eyeball fit of data from Hinckley's ezeroday.dat (E d-1 m-2)
            cff2 = 41.0_r8 - 35.0_r8                                    &
     &           * COS( ( 12.0_r8 + yday) * 2.0_r8 * pi / 365.0_r8 )
#endif
!  Calculate light decay in the water column
#ifdef NEWSHADE
!  Craig Lewis's version
            PAR(N) = PARfrac * cff2                                     &  
     &               * exp( ( k_ext + k_phy*(PhS(N)+PhL(N)) ) *         &
     &                  ( z_r(i,j,N) -  OCEAN(ng) % zeta(i,j,nnew) ) )
!            print *, 'Par: ', PAR(N), '  ', N
            do k=N(ng)-1,1,-1
              cff1 = k_ext               * ( z_r(i,j,k) - z_r(i,j,k+1) )
              cff2 = k_phy*(PhS(k+1)+PhL(k+1))                          & 
     &                                   * ( z_w(i,j,k) - z_r(i,j,k+1) )
              cff3 = k_phy*(PhS(k)+PhL(k)) * ( z_r(i,j,k) - z_w(i,j,k) )
              PAR(k) = PAR(k+1) * EXP(cff1+cff2+cff3)
!              print *, 'cff2: ', cff2, 'Par: ',PAR(k)
            enddo
#else
!  Version from Sarah's old C code (probably wrong)
            do k=N(ng),1,-1
              cff3 = z_r(i,j,k)+2.5_r8
              if ( cff3 .gt. -71.0_r8 ) then
                cff1 = k_ext + k_chl * ( PhS(k) + PhL(k) ) / ccr
              else 
                cff1 = 0.077_r8
              endif 
              PAR(k) = PARfrac * cff2 * exp( cff1 * cff3 )
            enddo
#endif
!
! ----------------------------------------------------------------------
! Begin BIOITER LOOP
! ----------------------------------------------------------------------
! 
            DO Iter=1,BioIter(ng)
!
               DO k=1,N(ng)
!                 Temp(k) = t(i,j,k,nnew,itemp)
!                 Sal(k) = t(i,j,k,nnew,isalt)
                 ! Flux form?
                 Temp(k) = t(i,j,k,nnew,itemp)/Hz(i,j,k)
                 Sal(k) = t(i,j,k,nnew,isalt)/Hz(i,j,k)
                 HzL(k) = Hz(i,j,k)
               END DO
               DO k=0,N(ng)
                 z_wL(k) = z_w(i,j,k)
               END DO
               LightLim = 1.0_r8
               NOLim = 1.0_r8
               NHLim = 1.0_r8
               IronLim = 1.0_r8
!----------------------------------------------------------------------
!**********************************************************************
!**********************************************************************
!  Uptake by Small Phytoplankton
!  The following code generates a temperature response to respiration
!--------------------------------------------------------------------
               TmaxPhS = 20
               TminPhS = -2
               CK1_PhS = 0.5
               Topt_PhS = 15
               KtBm_PhS = 0.069
!----------------------------------------------------------
! Generate a phytoplanton response relative to depth strata
!----------------------------------------------------------
               do k=1,N(ng)
                 !-----------------------------------
                 !Phytoplanton respiration correction
                 !-----------------------------------
                  Temp1 = Temp(k)
                 !--------------------------------
                 !Arhonditsis temperature function
                 !--------------------------------
                  TempFuncPhS(k) = GetPhytoResp2(Temp1,TmaxPhS,         &
     &              KtBm_PhS)
                 !----------------------------------------
                 !Thornton and Lessem temperature function
                 !----------------------------------------
!                  TempFuncPhS(k) = GetPhytoResp(CK1_PhS,TmaxPhS,        &
!     &                               TminPhS,Topt_PhS,Temp1)
!                  print *, TempFuncPhS(k)
!                 ------------------------
!                 Growth rate computations
!                 ------------------------
                  Drate = DiS * 10.0_r8 ** (DpS * Temp(k) )
                  Pmax = (2.0_r8 ** Drate - 1.0_r8 ) * Dl / 24.0_r8
!                  print *, Drate,' ',PMax,' ',Temp(k),' ',Dl
!                 old LightLim = TANH( alphaPhS * PAR(k) / Pmax / ccr )
!
                  NOLim = NO3(k) * EXP( -psiPhS * NH4(k) )              &
     &                    / ( k1PhS + NO3(k) )
!--------------------------------------------------------------
!  Define Iron Limitation such that it is disabled at high iron 
!  concentration (2 micromol Fe m-3)
!--------------------------------------------------------------
#ifdef IRON
                  IronLim = eps + Iron(k) / (kfePhS + Iron(k))          &
     &                            * (kfePhS + 2._r8) / 2._r8
#endif
!--------------------------------------------------------------
!                  LightLim = 1.0_r8 - EXP( -1.0_r8 *               &
!     &                       alphaPhS * PAR(k) / Pmax / ccr / IronLim )
                  Par1 = Par(k)
                  ccrMin = ccr
                  CrChlRatio = GetCarbChlorRatio(Par1,ccrMin)
                  LightLim = TANH( alphaPhS * PAR(k) / Pmax /           &
     &                 CrChlRatio / IronLim)
                  NOup=PhS(k)*Pmax*LightLim*NOLim*IronLim
                  DNO3(k) = DNO3(k) - xi * NOup * dtdays
!
                  NHLim = NH4(k) / ( k2PhS + NH4(k) )
!                  LightLim = 1.0_r8 - EXP( -1.0_r8 *                    &
!     &                       alphaPhS * PAR(k) / Pmax / ccr )
                  LightLim = TANH( alphaPhL * PAR(k) / Pmax /           &
     &                 CrChlRatio)
                  NHup=PhS(k)*Pmax*LightLim*NHLim
                  DNH4(k) = DNH4(k) - xi * NHup * dtdays
!
                  DPhS(k) = DPhS(k) + ( NOup + NHup ) * dtdays
#ifdef IRON
                  DIron(k) = DIron(k) - FeC * NOup * dtdays
#endif
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iNO3,iPhS) = bflx(iNO3,iPhS) + NOup
                    bflx(iNH4,iPhS) = bflx(iNH4,iPhS) + NHup
                  endif
#endif
               enddo
!********************************************************************
!********************************************************************
!  Uptake by Large Phytoplankton
!---------------------------------------------------
!  Temperature function for respiration Oct 26, 2005
!  Produces the array TempFunc for each depth level
!---------------------------------------------------
               TmaxPhL = 20
               TminPhL = -2
               CK1_PhL = 0.5
               Topt_PhL = 15
               KtBm_PhL = 0.069
               
               do k=1,N(ng)
                 !-----------------------------------
                 !Phytoplanton respiration correction
                 !-----------------------------------
                  Temp1 = Temp(k)
                 !--------------------------------
                 !Arhonditsis temperature function
                 !--------------------------------
                  TempFuncPhL(k) = GetPhytoResp2(Temp1,TmaxPhL,         &
     &              KtBm_PhL)
                 !----------------------------------------
                 !Thornton and Lessem temperature function
                 !----------------------------------------
!                  TempFuncPhL(k) = GetPhytoResp(CK1_PhL,TmaxPhL,        &
!     &                               TminPhL,Topt_PhL,Temp1)
                 !--------------------
                 !Phytoplankton growth
                 !--------------------
                  Drate = DiL * 10.0_r8 ** (DpL * Temp(k) )
                  Pmax = (2.0_r8 ** Drate - 1.0_r8 ) * Dl / 24.0_r8
!                 old LightLim = TANH( alphaPhL * PAR(k) / Pmax / ccr )
!
                  NOLim = NO3(k) * EXP( -psiPhL * NH4(k) )              &
     &                    / ( k1PhL + NO3(k) )
#ifdef IRON
!----------------------------------------------------------------
!  Define Iron Limitation such that it is disabled at the highest  
!  iron concentration (2 micromol Fe m-3)
!----------------------------------------------------------------
                  IronLim = eps + Iron(k) / (kfePhL + Iron(k))          &
     &                           * (kfePhL + 2._r8) / 2._r8
!                 if (k .eq. 30) then
!                   print *, Iron(k)
!                 endif
#endif
!------------------------------------------------------------------
!                  LightLim = 1.0_r8 - EXP( -1.0_r8 *                    &
!     &                 alphaPhL * PAR(k) / Pmax / ccrPhL / IronLim )
                  Par1 = Par(k)
                  ccrMin = ccrPhL
                  CrChlRatio = GetCarbChlorRatio(Par1,ccrMin)
                  LightLim = TANH( alphaPhL * PAR(k) / Pmax /           &
     &                 CrChlRatio / IronLim)
                  NOup = PhL(k) * Pmax * LightLim * NOLim * IronLim
                  DNO3(k) = DNO3(k) - xi * NOup * dtdays
!------------------------------------------------------------------
!                 Print statement to check Par
!                  if (PAR(k) .gt. MaxPAR .AND. k .eq. 30) THEN
!                        print *, 'Par: ', PAR(k), '   ',k
!                        MaxPAR = PAR(k)
!                 end if
!-------------------------------------------------------------------
                  NHLim = NH4(k) / ( k2PhL + NH4(k) )
!                  LightLim = 1.0_r8 - EXP( -1.0_r8 *                   &
!     &                 alphaPhL * PAR(k) / Pmax / ccrPhL )
                  LightLim = TANH( alphaPhL * PAR(k) / Pmax /           &
     &                 CrChlRatio )
                  NHup = PhL(k) * Pmax * LightLim * NHLim
                  DNH4(k) = DNH4(k) - xi * NHup * dtdays
!
                  DPhL(k) = DPhL(k) + ( NOup + NHup ) * dtdays
!----------------------------------------------------------------
!  The following are print statements to check phytoplankton 
!  production calculations
!                  if (DPhL(k) .gt. 0.5) then
!
!                          print *, DPhL(k),' ',PhL(k), ' ',Temp(k)
!                       print *, NOup,' ',NHup,' ',LightLim
!                        print *, Par(k),'*',Pmax,'*',Drate
!                       Print *,Dl,'#',NO3(k),'#',NH4(k),'#',NOLim
!                  endif
!-----------------------------------------------------------------
#ifdef IRON
                  DIron(k) = DIron(k) - FeC * NOup * dtdays
#endif
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iNO3,iPhL) = bflx(iNO3,iPhL) + NOup
                    bflx(iNH4,iPhL) = bflx(iNH4,iPhL) + NHup
                  endif
#endif
               enddo
!----------------------------------------------------------------------
!**********************************************************************
!**********************************************************************
! Grazing by MZS
!--------------------------------------
!Temperature parameters for respiration
!--------------------------------------
               TmaxMZS = 20
               KtBm_MZS = 0.069
               
               do k=1,N(ng)
                 !---------------------------------------
                 !Microzooplankton respiration correction
                 !---------------------------------------
                  Temp1 = Temp(k)
                 !--------------------------------
                 !Arhonditsis temperature function
                 !--------------------------------
                  TempFuncMZS(k) = GetPhytoResp2(Temp1,TmaxMZS,         &
     &              KtBm_MZS)
                 !-----------------------
                 !Microzooplankton growth
                 !-----------------------
                  cff1 = fpPhSMZS * PhS(k)**2                           &
     &                 + fpPhLMZS * PhL(k)**2
                  cff2 = eMZS *                                         &
     &                 ( Q10MZS ** ( (Temp(k)-Q10MZST) / 10.0_r8 ) )    &
     &                 * (MZS(k)) / (fMZS**2 + cff1)
!                 if (yday .gt. 155) print *, MZL(k), '*',               &
!                    MZS(k), '*', cff2, '*', cff1, '*', cff2
!
                  DNH4(k) = DNH4(k) + xi * kMZS * cff1 * cff2 * dtdays
                  DPhS(k) = DPhS(k) - fpPhSMZS * (PhS(k)**2) *          &
     &                  cff2 * dtdays
                  DPhL(k) = DPhL(k) - fpPhLMZS * (PhL(k)**2) *          &
     &                  cff2 * dtdays
                  DMZS(k) = DMZS(k) + gammaMZS * cff1 * cff2 * dtdays
                  DDet(k) = DDet(k) +                                   &
     &                 (1.0_r8 - kMZS - gammaMZS) * cff1 * cff2 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iMZS,iNH4) = bflx(iMZS,iNH4) + kMZS*cff1*cff2
                    bflx(iPhS,iMZS) = bflx(iPhS,iMZS) +                 &
                      fpPhSMZS*PhS(k)*cff2
                    bflx(iPhL,iMZS) = bflx(iPhL,iMZS) +                 &
                      fpPhSMZL*PhL(k)*cff2
                    bflx(iMZS,iDet) = bflx(iMZS,iDet)  +                &
     &                ( 1.0_r8-kMZS-gammaMZS )*cff1*cff2
                  endif
#endif
!
               enddo
!----------------------------------------------------------------------
!**********************************************************************
!**********************************************************************
! Grazing by MZL
!--------------------------------------
!Temperature parameters for respiration
!--------------------------------------
               TmaxMZL = 20
               KtBm_MZL = 0.069
               
               do k=1,N(ng)
                 !---------------------------------------
                 !Microzooplankton respiration correction
                 !---------------------------------------
                  Temp1 = Temp(k)
                 !--------------------------------
                 !Arhonditsis temperature function
                 !--------------------------------
                  TempFuncMZL(k) = GetPhytoResp2(Temp1,TmaxMZL,         &
     &              KtBm_MZL)
                 !-----------------------
                 !Microzooplankton growth
                 !-----------------------
                  cff1 = fpPhSMZL * PhS(k)**2                           &
     &                 + fpPhLMZL * PhL(k)**2                           &
     &                 + fpMZSMZL * MZS(k)**2
                  cff2 = eMZL *                                         &
     &                 ( Q10MZL ** ( (Temp(k)-Q10MZLT) /  5.0_r8 ) )    &
     &                 * MZL(k) / (fMZL**2 + cff1)
!
                  DNH4(k) = DNH4(k) + xi * kMZL * cff1 * cff2 * dtdays
                  DPhS(k) = DPhS(k) - fpPhSMZL * (PhS(k)**2) *          &
     &                cff2 * dtdays
                  DPhL(k) = DPhL(k) - fpPhLMZL * (PhL(k)**2) *          &
     &                cff2 * dtdays
                  DMZS(k) = DMZS(k) - fpMZSMZL * (MZS(k)**2) *          &
     &                cff2 * dtdays
                  DMZL(k) = DMZL(k) + gammaMZL * cff1 * cff2 * dtdays
                  DDet(k) = DDet(k) +                                   &
     &                 (1.0_r8 - kMZL - gammaMZL) * cff1 * cff2 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iMZL,iNH4) = bflx(iMZL,iNH4) + kMZL*cff1*cff2
                    bflx(iPhS,iMZL) = bflx(iPhS,iMZL) +                 &
     &                 fpPhSMZL*PhS(k)*cff2
                    bflx(iPhL,iMZL) = bflx(iPhL,iMZL) +                 &
                      fpPhLMZL*PhL(k)*cff2
                    bflx(iMZS,iMZL) = bflx(iMZS,iMZL) +                 &
     &                fpMZSMZL*MZS(k)*cff2
                    bflx(iMZL,iDet) = bflx(iMZL,iDet)  +                &
     &                ( 1.0_r8-kMZL-gammaMZL )*cff1*cff2
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Grazing and Predation by Cop
!
               do k=1,N(ng)
                 !-------------------------------
                 ! Copepod respiration correction
                 !-------------------------------
                  Temp1 = Temp(k)
                  TempFuncCop(k) = GetCopepodResp(Temp1)
                 !---------------
                 ! Copepod Growth
                 !---------------
                  cff1 = fpPhSCop * PhS(k)**2                           & 
     &                 + fpPhLCop * PhL(k)**2                           &
     &                 + fpMZSCop * MZS(k)**2                           &
     &                 + fpMZLCop * MZL(k)**2
                  cff2 = eCop *                                         &
     &                 ( Q10Cop ** ( (Temp(k)-Q10CopT) / 10.0_r8 ) )    &
     &                 * Cop(k) / (fCop**2 + cff1)
!
                  DCop(k) = DCop(k) + gammaCop * cff1 * cff2 * dtdays
!
                  DPhS(k) = DPhS(k) - fpPhSCop * (PhS(k)**2)            &
     &                    * cff2 * dtdays
                  DPhL(k) = DPhL(k) - fpPhLCop * (PhL(k)**2)            &
     &                    * cff2 * dtdays
                  DMZS(k) = DMZS(k) - fpMZSCop * (MZS(k)**2)            &
     &                    * cff2 * dtdays
                  DMZL(k) = DMZL(k) - fpMZLCop * (MZL(k)**2)            &
     &                    * cff2 * dtdays
!
                  DNH4(k) = DNH4(k) + xi * kCop * cff1 * cff2 * dtdays
                  DDet(k) = DDet(k) +                                   &
     &                 (1.0_r8 - kCop - gammaCop) * cff1 * cff2 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iPhS,iCop)=bflx(iPhS,iCop)+fpPhSCop*PhS(k)*cff2
                    bflx(iPhL,iCop)=bflx(iPhL,iCop)+fpPhLCop*PhL(k)*cff2
                    bflx(iMZS,iCop)=bflx(iMZS,iCop)+fpMZSCop*MZS(k)*cff2
                    bflx(iMZL,iCop)=bflx(iMZL,iCop)+fpMZLCop*MZL(k)*cff2
                    bflx(iCop,iNH4) = bflx(iCop,iNH4) + kCop*cff1*cff2
                    bflx(iCop,iDet) = bflx(iCop,iDet) +                 &
     &                          ( 1.0_r8-kCop-gammaCop )*cff1*cff2
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Grazing and Predation by NCa
!
               do k=1,N(ng)
                 !----------------------------------
                 ! Neocalanus respiration correction
                 !----------------------------------
                  Temp1 = Temp(k)
                  TempFuncNeo(k) = GetCopepodResp(Temp1)                 
                 !------------------
                 ! Neocalanus growth
                 !------------------
                  cff1 = fpPhSNCa * PhS(k)**2                           & 
     &                 + fpPhLNCa * PhL(k)**2                           &
     &                 + fpMZSNCa * MZS(k)**2                           &
     &                 + fpMZLNCa * MZL(k)**2
                  cff2 = eNCa *                                         &
     &                 ( Q10NCa ** ( (Temp(k)-Q10NCaT) / 10.0_r8 ) )    &
     &                 * NCa(k) / (fNCa**2 + cff1)
!
                  DNCa(k) = DNCa(k) + gammaNCa * cff1 * cff2 * dtdays
!
                  DPhS(k) = DPhS(k) - fpPhSNCa * (PhS(k)**2)            &
     &                    * cff2 * dtdays
                  DPhL(k) = DPhL(k) - fpPhLNCa * (PhL(k)**2)            &
     &                    * cff2 * dtdays
                  DMZS(k) = DMZS(k) - fpMZSNCa * (MZS(k)**2)            &
     &                    * cff2 * dtdays
                  DMZL(k) = DMZL(k) - fpMZLNCa * (MZL(k)**2)            &
     &                    * cff2 * dtdays
!
                  DNH4(k) = DNH4(k) + xi * kNCa * cff1 * cff2 * dtdays
                  DDet(k) = DDet(k) +                                   &
     &                 (1.0_r8 - kNCa - gammaNCa) * cff1 * cff2 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iPhS,iNCa)=bflx(iPhS,iNCa)+fpPhSNCa*PhS(k)*cff2
                    bflx(iPhL,iNCa)=bflx(iPhL,iNCa)+fpPhLNCa*PhL(k)*cff2
                    bflx(iMZS,iNCa)=bflx(iMZS,iNCa)+fpMZSNCa*MZS(k)*cff2
                    bflx(iMZL,iNCa)=bflx(iMZL,iNCa)+fpMZLNCa*MZL(k)*cff2
                    bflx(iNCa,iNH4) = bflx(iNCa,iNH4) + kNCa*cff1*cff2
                    bflx(iNCa,iDet) = bflx(iNCa,iDet) +                 &
     &                          ( 1.0_r8-kNCa-gammaNCa )*cff1*cff2
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Grazing and Predation by Eup
!
               do k=1,N(ng)
                 !----------------------------------
                 ! Neocalanus respiration correction
                 !----------------------------------
                  Temp1 = Temp(k)
                  TempFuncEup(k) = GetCopepodResp(Temp1)
                 !-----------------
                 !Euphausiid growth
                 !-----------------                 
                  cff1 = fpPhSEup * PhS(k)**2                           & 
     &                 + fpPhLEup * PhL(k)**2                           &
     &                 + fpMZSEup * MZS(k)**2                           &
     &                 + fpMZLEup * MZL(k)**2                           &
     &                 + fpCopEup * Cop(k)**2
                  cff2 = eEup *                                         &
     &                 ( Q10Eup ** ( (Temp(k)-Q10EupT) / 10.0_r8 ) )    &
     &                 * Eup(k) / (fEup**2 + cff1)
!
                  DEup(k) = DEup(k) + gammaEup * cff1 * cff2 * dtdays
!
                  DPhS(k) = DPhS(k) - fpPhSEup * (PhS(k)**2)            &
     &                    * cff2 * dtdays
                  DPhL(k) = DPhL(k) - fpPhLEup * (PhL(k)**2)            &
     &                    * cff2 * dtdays
                  DMZS(k) = DMZS(k) - fpMZSEup * (MZS(k)**2)            &
     &                    * cff2 * dtdays
                  DMZL(k) = DMZL(k) - fpMZLEup * (MZL(k)**2)            &
     &                    * cff2 * dtdays
                  DCop(k) = DCop(k) - fpCopEup * (Cop(k)**2)            &
     &                    * cff2 * dtdays
!
                  DNH4(k) = DNH4(k) + xi * kEup * cff1 * cff2 * dtdays
                  DDet(k) = DDet(k) +                                   &
     &                 (1.0_r8 - kEup - gammaEup) * cff1 * cff2 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iPhS,iEup)=bflx(iPhS,iEup)+fpPhSEup*PhS(k)*cff2
                    bflx(iPhL,iEup)=bflx(iPhL,iEup)+fpPhLEup*PhL(k)*cff2
                    bflx(iMZS,iEup)=bflx(iMZS,iEup)+fpMZSEup*MZS(k)*cff2
                    bflx(iMZL,iEup)=bflx(iMZL,iEup)+fpMZLEup*MZL(k)*cff2
                    bflx(iCop,iEup)=bflx(iCop,iEup)+fpCopEup*Cop(k)*cff2
                    bflx(iEup,iNH4) = bflx(iEup,iNH4) + kEup*cff1*cff2
                    bflx(iEup,iDet) = bflx(iEup,iDet) +                 &
     &                          ( 1.0_r8-kEup-gammaEup )*cff1*cff2
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Linear Mortality and Senescence Terms
!
               do k=1,N(ng)
                  cff1 = MAX( minmPhS , maxmPhS -                       &
     &                 ( maxmPhS - minmPhS ) * NO3(k) / NcritPhS )
                  cff2 = MAX( minmPhL , maxmPhL -                       &
     &                 ( maxmPhL - minmPhL ) * NO3(k) / NcritPhL )
!
!                  if (NO3(28) .lt. 20) print *, 'cff2: ', cff2,        &
!     &              'NO3: ', NO3(k)
                  DPhS(k) = DPhS(k) - cff1 * PhS(k) * dtdays
                  DPhL(k) = DPhL(k) - cff2 * PhL(k) * dtdays
!
!                  DMZS(k) = DMZS(k) - mMZS * MZS(k) * dtdays
!                  DMZL(k) = DMZL(k) - mMZL * MZL(k) * dtdays
!                  DCop(k) = DCop(k) - mCop * Cop(k) * dtdays
!                  DNCa(k) = DNCa(k) - mNCa * NCa(k) * dtdays
!                  DEup(k) = DEup(k) - mEup * Eup(k) * dtdays
!
                  DDet(k) = DDet(k)                                     &
     &                 + ( cff1 * PhS(k) + cff2 * PhL(k)                &
     &                   + mMZS * MZS(k) + mMZL * MZL(k)                &
     &                   + mCop * Cop(k) + mNCa * NCa(k)                &
     &                   + mEup * Eup(k) ) * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iPhS,iDet)= bflx(iPhS,iDet) + cff1*PhS(k)
                    bflx(iPhL,iDet)= bflx(iPhL,iDet) + cff2*PhL(k)
                    bflx(iMZS,iDet)= bflx(iMZS,iDet) + mMZS*MZS(k)
                    bflx(iMZL,iDet)= bflx(iMZL,iDet) + mMZL*MZL(k)
                    bflx(iCop,iDet)= bflx(iCop,iDet) + mCop*Cop(k)
                    bflx(iNCa,iDet)= bflx(iNCa,iDet) + mNCa*NCa(k)
                    bflx(iEup,iDet)= bflx(iEup,iDet) + mEup*Eup(k)
                  endif
#endif
               enddo
!----------------------------------------------------------------------
!**********************************************************************
!**********************************************************************
! Nonlinear Mortality (closure) Terms
!
               do k=1,N(ng)
                  DMZS(k) = DMZS(k) - mpredMZS*dtdays*MZS(k)**2
                  DMZL(k) = DMZL(k) - mpredMZL*dtdays*MZL(k)**2
                  Sal1 = Sal(k)
                  Temp1 = Temp(k)
                  MyPred = GetMort(Sal1, Temp1, mpredCop)
                  DCop(k) = DCop(k) - MyPred*dtdays*Cop(k)**2
                  DNCa(k) = DNCa(k) - mpredNCa*dtdays*NCa(k)**2
                  DEup(k) = DEup(k) - mpredEup*dtdays*Eup(k)**2
! CVL: This code would be needed for mass conservation, but is not in SH
! CVL: version.  I've included to check for closure.
                  DDet(k) = DDet(k) +                                   &
     &                    (mpredMZS * MZS(k)**2                         &
     &                   + mpredMZL * MZL(k)**2                         &
     &                   + MyPred * Cop(k)**2                           &
     &                   + mpredNCa * NCa(k)**2                         &
     &                   + mpredEup * Eup(k)**2 ) * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iCop,itemp)= bflx(iCop,itemp)+MyPred*Cop(k)**2
                    bflx(iNca,itemp)= bflx(iNCa,itemp)+mpredNCa*NCa(k)**2
                    bflx(iEup,itemp)= bflx(iEup,itemp)+mpredEup*Eup(k)**2
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
#ifdef IRON
! Respiration /////////////////////////////////////////////////////////
               do k=1,N(ng)
                 !---------------------------------------------
                 !Corrects basal metabolism for iron limitation
                 !Small phytoplankton
                 !---------------------------------------------
                  Iron1 = Iron(k)
                  respPh = respPhS
                  kfePh = kfePhS
                  BasalMet = GetBasalMetabolism(respPh,kfePh,Iron1)
                  DPhS(k)=DPhS(k)-TempFuncPhS(k)*BasalMet*dtdays*PhS(k)
                 !---------------------------------------------
                 !Corrects basal metabolism for iron limitation
                 !Large phytoplankton
                 !---------------------------------------------
                  respPh = respPhL
                  kfePh = kfePhL
                  BasalMet = GetBasalMetabolism(respPh,kfePh,Iron1)
                  DPhL(k)=DPhL(k)-TempFuncPhL(k)*BasalMet*dtdays*PhL(k)
                 !--------------------------------------------
                 !Correct basal metabolism for iron limitation
                 !Small microzooplankton (use kfePHS for kfeMZS)
                 !--------------------------------------------
                  respPh = respMZS
                  kfePH = kfePhS
                  BasalMet = GetBasalMetabolism(respPh,kfePh,Iron1)
                  DMZS(k)=DMZS(k)-TempFuncMZS(k)*BasalMet*dtdays*MZS(k)
                 !--------------------------------------------
                 !Correct basal metabolism for iron limitation
                 !Large microzooplankton (use kfePHL for kfeMZL)
                 !--------------------------------------------
                  respPh = respMZL
                  kfePH = kfePhL
                  BasalMet = GetBasalMetabolism(respPh,kfePh,Iron1)
                  DMZL(k)=DMZL(k)-TempFuncMZL(k)*BasalMet*dtdays*MZL(k)

!               print *, '*T*',TempFuncMZL(k),'***',MZL(k),'***',DMZL(k),&
!     &            '*dt*',dtdays 
                  DCop(k) = DCop(k) - TempFuncCop(k)*Cop(k)*dtdays
                  DNCa(k) = DNCa(k) - TempFuncNeo(k)*NCa(k)*dtdays
                  DEup(k) = DEup(k) - TempFuncEup(k)*Eup(k)*dtdays
               enddo
#endif
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Molting
!
               do k=1,N(ng)
                  cff1 = 0.02_r8 / (10.0_r8 - 0.4_r8 * Temp(k))* Eup(k)
                  DDet(k) = DDet(k) + cff1 * dtdays
                  DEup(k) = DEup(k) - cff1 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iEup,iDet)= bflx(iEup,iDet) + cff1
                  endif
#endif
               enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Detrital Remineralization
!
               do k=1,N(ng)
                  cff1 = regen * dgrad * Det(k)
                  !--------------------
                  !Nitrification
                  !--------------------
                  Dep1 = -1 * z_wL(k)
                  NH4R = NH4(k)
                  Temp1 = Temp(k)
                  Nitrif = GetNitrif(Temp1,Dep1,NH4R)
                  DNH4(k) = DNH4(k) - Nitrif * dtdays
                  DNO3(k) = DNO3(k) + Nitrif * dtdays
                  !if (NO3(k) .lt. 5) then
                  !     DNO3(k) = DNO3(k) + 0.5 * dtdays
                  !endif
                  !Print *, Nitrif
                  DNH4(k) = DNH4(k) + xi * cff1 * dtdays
                  DDet(k) = DDet(k) - cff1 * dtdays
#if defined BIOFLUX && defined GAK1D
                  if (i.eq.3.and.j.eq.3) then
                    bflx(iDet,iNH4)= bflx(iDet,iNH4) + cff1
                  endif
#endif
              enddo
!-----------------------------------------------------------------------
!***********************************************************************
!***********************************************************************
! Sinking (code external: CVL)
!
               call BIOSINK(ng,wPhS,PhS,DPhS,HzL,dtdays,z_wL,1.0_r8)
               call BIOSINK(ng,wPhL,PhL,DPhL,HzL,dtdays,z_wL,1.0_r8)
               call BIOSINK(ng,wDet,Det,DDet,HzL,dtdays,z_wL,1.0_r8)
#ifdef DIAPAUSE
               if ( downward ) then
                 call BIOSINK(ng,wNCsink,NCa,DNCa,HzL,dtdays,z_wL,      &
     &             -1*NCmaxz)
               else if ( upward .and. NCa(N(ng)) .lt. 6.0_r8 ) then
                 call BIORISE(ng,wNCrise,NCa,DNCa,HzL,dtdays,z_wL,      &
     &             -1*NCmaxz)
               endif
#endif
!
! ----------------------------------------------------------------------
! END BIOITER LOOP
! ----------------------------------------------------------------------
!
            END DO
! ----------------------------------------------------------------------
!  Update global tracer variables.  For ROMS 2.2+, convert back to
!  "flux form" of tracers
! ----------------------------------------------------------------------
            DO k=1,N(ng)
              t(i,j,k,nnew,iNO3) = (MAX(NO3(k)+DNO3(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iNH4) = (MAX(NH4(k)+DNH4(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iPhS) = (MAX(PhS(k)+DPhS(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iPhL) = (MAX(PhL(k)+DPhL(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iMZS) = (MAX(MZS(k)+DMZS(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iMZL) = (MAX(MZL(k)+DMZL(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iCop) = (MAX(Cop(k)+DCop(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iNCa) = (MAX(NCa(k)+DNCa(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iEup) = (MAX(Eup(k)+DEup(k),eps)) * Hz(i,j,k)
              t(i,j,k,nnew,iDet) = (MAX(Det(k)+DDet(k),eps)) * Hz(i,j,k)
#ifdef IRON
              t(i,j,k,nnew,iFe) = (MAX(Iron(k)+DIron(k),eps)) *Hz(i,j,k)
#endif
            END DO
! ----------------------------------------------------------------------
! End HORIZONTAL INDEX LOOPING
! ----------------------------------------------------------------------
         END DO
      END DO

      RETURN
      END SUBROUTINE biology_tile
!
!================================================ Craig V. W. Lewis ==
!                                                                    !
!  This routine extracts the sinking code from biology.F and allows  !
!  it to be used as an external module for any number of fields      !
!  independently; it's a hack, but better than the current setup     !
!                                                                    !
!  Added 1 argument for diapause calculations:                       !
!    zlimit   - lowest depth for sinking.  -1 = sinks out at         !
!               constant rate                                        !
!  Liz Dobbins 4/15/03                                               !
!=====================================================================
!
      subroutine BIOSINK(ng,wBio,Bio,dBioOut,HzL,dtdays,z_wL,zlimit)
!
      USE mod_param
!
      implicit none
!
      integer, intent(in) :: ng
      real(r8), intent(in) :: wBio
      real(r8), intent(in) :: Bio(N(ng))
      real(r8), intent(inout) :: dBioOut(N(ng))
      real(r8), intent(in) :: HzL(N(ng))
      real(r8), intent(in) :: dtdays
      real(r8), intent(in) :: z_wL(0:N(ng))
      real(r8), intent(in) :: zlimit
!
      integer :: k
      real(r8) :: aL, aR, cff1, cff2, cu
      real(r8) :: FC(0:N(ng)), dBio(0:N(ng)), wBiod(0:N(ng))
!
!
      do k=0,N(ng)
         if ( zlimit .lt. 0 ) then
           if ( z_wL(k) .ge. zlimit ) then
             wBiod(k) = wBio*exp( -1*(z_wL(k)-(zlimit/2))**2 /          & 
     &          (zlimit/2)**2 )
           else
             wBiod(k) = 0.0_r8
           endif
         else
           wBiod(k) = wBio
         endif
      enddo
!
!  Vertical sinking: Vertical advection algorithm based on monotonic,
!  continuous conservative parabolic splines.
!  Construct parabolic splines:  Compute vertical derivatives for
!  "Bio" at W-points.  Neumann boundary conditions are
!  assumed on top and bottom.
!
      FC(0)=0.0_r8
!
      dBio(0)=0.0_r8
      do k=1,N(ng)-1
         cff1=1.0_r8/(2.0_r8*HzL(k+1)+                                  &
     &        HzL(k)*(2.0_r8-FC(k-1)))
         FC(k)=cff1*HzL(k+1)
         dBio(k)=cff1*(6.0_r8*(Bio(k+1)-Bio(k))-                        &
     &        HzL(k)*dBio(k-1))
      enddo
      
      dBio(N(ng))=0.0_r8
      
      do k=N(ng)-1,1,-1
         dBio(k)=dBio(k)-FC(k)*dBio(k+1)
      enddo
!
!  Convert vertical derivatives "dBio" into field values
!  at grid box interfaces assuming parabolic profiles within each
!  grid box.  Restrict these values to lie between bounds determined
!  from box-averaged values of grid boxes adjscent from above and
!  below. This restriction is part of the PPM-like monotonization
!  procedure.
!
      cff1=1.0_r8/3.0_r8
      dBio(0)=Bio(1)            ! -cff*HzL(1)*(dBio(0)+0.5_r8*dBio(1))
      dBio(N(ng))=Bio(N(ng))    ! +cff*HzL(N(ng))*(dBio(N(ng))+0.5_r8*dBio(N(ng)-1))
      do k=2,N(ng)
         dBio(k-1)=Bio(k)-cff1*HzL(k)                                   &
     &        *(0.5_r8*dBio(k)+dBio(k-1))
         dBio(k-1)=MAX(dBio(k-1),MIN(Bio(k-1),Bio(k)))
         dBio(k-1)=MIN(dBio(k-1),MAX(Bio(k-1),Bio(k)))
      enddo
!
!  Convert "dBio" into flux-integrated values;  complete
!  PPM flux limiting.  This procedure starts from assigning left and
!  right (aR,aL) values of the interpolating parabolae, then the
!  monotonicity conditions are checked and aL, aR are modified to fit.
!
      do k=1,N(ng)
         FC(k)=dtdays/HzL(k)
         aR=dBio(k)
         aL=dBio(k-1)
         cff1=(aR-aL)*6.0_r8*(Bio(k)-0.5_r8*(aR+aL))
         cff2=(aR-aL)**2
         if ((aR-Bio(k))*(Bio(k)-aL).lt.0.0_r8) then
            aL=Bio(k)
            aR=Bio(k)
         elseif (cff1.gt.cff2) then
            aL=3.0_r8*Bio(k)-2.0_r8*aR
         elseif (cff1.lt.-cff2) then
            aR=3.0_r8*Bio(k)-2.0_r8*aL
         endif
         cu=wBio*FC(k)
         dBio(k-1)=Bio(k)-(1.0_r8-cu)*(0.5_r8*(aR-aL)-                  &
     &        (0.5_r8*(aR+aL)-Bio(k))*(1.0_r8-2.0_r8*cu))
      enddo
      dBio(N(ng))=0.0_r8
!
! Set change in biological variable.
!
      do k=1,N(ng)
         dBioOut(k) = dBioOut(k) +                                      &
     &        wBiod(k)*FC(k)*dBio(k) - wBiod(k-1)*FC(k)*dBio(k-1)
      enddo
!
      RETURN
      END SUBROUTINE BIOSINK
!=====================================================================
!                                                                    !
!   It's Craig's biosink.F turned upside-down, based on an idea of   !
!   Hal Batchelder's which he used to move NO3 upwards to simulate   !
!   upwelling                                                        !
!                                                                    !
!  Added 1 argument for diapause calculations:                       !
!    zlimit   - lowest depth for sinking.  -1 = constant rate        !
!  Liz Dobbins 4/15/03                                               !
!=====================================================================
!
      subroutine BIORISE(ng,wBio,Bio,dBioOut,HzL,dtdays,z_wL,zlimit)
!
      USE mod_param
!
      implicit none
!
      integer, intent(in) :: ng
      real(r8), intent(in) :: wBio
      real(r8), intent(in) :: Bio(N(ng))
      real(r8), intent(inout) :: dBioOut(N(ng))
      real(r8), intent(in) :: HzL(N(ng))
      real(r8), intent(in) :: dtdays
      real(r8), intent(in) :: z_wL(0:N(ng))
      real(r8), intent(in) :: zlimit
!
      integer :: k
      real(r8) :: aL, aR, cff1, cff2, cu
      real(r8) :: FC(1:N(ng)+1), dBio(1:N(ng)+1), wBiod(1:N(ng)+1)
!
      if ( zlimit .lt. 0 ) then
        do k=1,N(ng)
          if ( z_wL(k-1).ge.zlimit/2) then
            wBiod(k) = wBio*exp( -1*(z_wL(k-1)-(zlimit/2))**2 /          & 
     &          (zlimit/2)**2 )
          else
            wBiod(k) = wBio
          endif
        enddo
        wBiod(N(ng)+1) = 0.0_r8
      else
        do k=1,N(ng)+1
          wBiod(k) = wBio
        enddo
      endif
!
!  Vertical rising: Vertical advection algorithm based on monotonic,
!  continuous conservative parabolic splines.
!  Construct parabolic splines:  Compute vertical derivatives for
!  "Bio" at W-points.  Neumann boundary conditions are
!  assumed on top and bottom.
!
      FC(N(ng)+1)=0.0_r8
!
      dBio(N(ng)+1)=0.0_r8
      do k=N(ng),2,-1
         cff1=1.0_r8/(2.0_r8*HzL(k-1)+                                  &
     &        HzL(k)*(2.0_r8-FC(k+1)))
         FC(k)=cff1*HzL(k-1)
         dBio(k)=cff1*(6.0_r8*(Bio(k-1)-Bio(k))-                        &
     &        HzL(k)*dBio(k+1))
      enddo
      
      dBio(1)=0.0_r8
      
      do k=2,N(ng)
         dBio(k)=dBio(k)-FC(k)*dBio(k-1)
      enddo
!
!  Convert vertical derivatives "dBio" into field values
!  at grid box interfaces assuming parabolic profiles within each
!  grid box.  Restrict these values to lie between bounds determined
!  from box-averaged values of grid boxes adjscent from above and
!  below. This restriction is part of the PPM-like monotonization
!  procedure.
!
      cff1=1.0_r8/3.0_r8
      dBio(N(ng)+1)=Bio(N(ng))  ! -cff*HzL(1)*(dBio(0)+0.5_r8*dBio(1))
      dBio(1)=Bio(1)            ! +cff*HzL(N(ng))*(dBio(N(ng))+0.5_r8*dBio(N(ng)-1))
      do k=N(ng)-1,1,-1
         dBio(k+1)=Bio(k)-cff1*HzL(k)                                   &
     &        *(0.5_r8*dBio(k)+dBio(k+1))
         dBio(k+1)=MAX(dBio(k+1),MIN(Bio(k+1),Bio(k)))
         dBio(k+1)=MIN(dBio(k+1),MAX(Bio(k+1),Bio(k)))
      enddo
!
!  Convert "dBio" into flux-integrated values;  complete
!  PPM flux limiting.  This procedure starts from assigning left and
!  right (aR,aL) values of the interpolating parabolae, then the
!  monotonicity conditions are checked and aL, aR are modified to fit.
!
      do k=N(ng),1,-1
         FC(k)=dtdays/HzL(k)
         aR=dBio(k)
         aL=dBio(k+1)
         cff1=(aR-aL)*6.0_r8*(Bio(k)-0.5_r8*(aR+aL))
         cff2=(aR-aL)**2
         if ((aR-Bio(k))*(Bio(k)-aL).lt.0.0_r8) then
            aL=Bio(k)
            aR=Bio(k)
         elseif (cff1.gt.cff2) then
            aL=3.0_r8*Bio(k)-2.0_r8*aR
         elseif (cff1.lt.-cff2) then
            aR=3.0_r8*Bio(k)-2.0_r8*aL
         endif
         cu=wBio*FC(k)
         dBio(k+1)=Bio(k)-(1.0_r8-cu)*(0.5_r8*(aR-aL)-                  &
     &        (0.5_r8*(aR+aL)-Bio(k))*(1.0_r8-2.0_r8*cu))
      enddo
      dBio(1)=0.0_r8
!
! Set change in biological variable.
!
      do k=1,N(ng)
         dBioOut(k) = dBioOut(k) +                                      &
     &        wBiod(k)*FC(k)*dBio(k) - wBiod(k+1)*FC(k)*dBio(k+1)
      enddo
!
      RETURN
      END SUBROUTINE BIORISE

!====================================================
       Function GetMort(Sal1, Temp1, mpredCop)
!--------------------------------------------------
!Computes a bivariate normal distribution inverted
!for the mortality
!--------------------------------------------------
       USE mod_param
       implicit none
!
       real(r8) :: GetMort
       real(r8) :: Sal1, Temp1
       real(r8) :: mpredCop
                                !--------------------------------------!
       real(r8) :: Sigma12 = 1      !Covariance of salinity and temperature!
       real(r8) :: Sigma11 = 8      !Variance of temperature               !
       real(r8) :: Sigma22 = 0.5    !Variance of salinity                  !
       real(r8) :: TempMean = 15    !Mean temperature                      !
       real(r8) :: SalMean = 31     !Mean salinity                         !
       real(r8) :: MaxFreq = 0.0919 !Maximum frequency value               !
       real(r8) :: Rho12               !Correlation coefficient               !
                                !--------------------------------------!
       real(r8) :: MyPred
       real(r8) :: X1, X2, X3, Freq, Scalor
       !-----------------------------------------------------
       !Compute the correlation coefficient from the variance
       !-----------------------------------------------------
       Rho12 = Sigma12
       Rho12 = Rho12/((Sigma11**0.5)*(Sigma22**0.5))
       !----------------------------------------------
       !Compute the frequency distribution relative to
       !Salinity and temperature
       !----------------------------------------------
       X1 = (Temp1 - TempMean)/(Sigma11**0.5)
       X2 = (Sal1 - SalMean)/(Sigma22**0.5)
       X3 = (X1**2) + (X2**2) - 2*Rho12 * x1 *x2
       X3 = X3/(2*(1 - Rho12**2))
       Freq = 1 - Rho12**2
       Freq = Freq * Sigma11 * Sigma22
       Freq = Freq**0.5
       Freq = 2 * 3.14159 * Freq
       Freq = 1/Freq
       Freq = Freq * exp(-X3)
       !--------------------------------------------------
       !Scale the frequency distribution so that mortality
       !varies between 0.1 and 0.01
       !--------------------------------------------------
       Scalor = 0.9/MaxFreq
       Freq = (Freq * Scalor) + 0.1
       !-------------------------------------------------
       !Invert the frequency distribution so that maximum
       !mortality occurs at the edges of the distribution
       !-------------------------------------------------
       Freq = (1/Freq) * 0.1
       !---------------------------------------------
       !Scale the output to mortality of the copepods
       !---------------------------------------------
       Freq = Freq * mpredCop
       GetMort = Freq
       Return
       End Function GetMort

!=========================================================
        Function GetNitrif(Temp1,Dep1,NH4R)
!-------------------------------------------------------
! Computes the nitrification with respect to temperature
! according to Arhonditsis (2005).  Generates depth
! correction according to Denman (2003).
!-------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetNitrif
                                  !--------------------------------
        real(r8) :: Temp1, Dep1, NH4R    !Passed variables
        real(r8) :: NH4conv  = 14     !mg N/mol N
        real(r8) :: KNH4Nit  = 0.08   !Half Sat Con mg N/m3/day
        real(r8) :: KTNitr   = 0.002  !Temperature responce dec C^2
        real(r8) :: ToptNtr  = 28     !Optimum nitrification temp
        real(r8) :: Zox      = 60     !50% nitrification depth
        real(r8) :: Nexp     = 8      !Exponent to adjust profile shape
                                  !--------------------------------

        real(r8) :: Nitr, DepCor

        Nitr = NH4R/(KNH4Nit + NH4R)
        Nitr = Nitr * exp(-KTNitr*(Temp1 - ToptNtr)**2)
        DepCor = (Dep1**Nexp)/( (Zox**Nexp) + Dep1**Nexp)
        Nitr = (Nitr * DepCor) / NH4conv
        GetNitrif = Nitr
        Return
        End Function GetNitrif

!==================================================================
        Function GetPhytoResp(CK4,CTL,CQ,CTO,Temp1)
!------------------------------------------------------------
!Computes a phytoplankton respiration response to temperature
!From Thornton and Lessem. 1978. Tran. Amer. Fish. Soc. 
! 10(2):284 - 287
!------------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetPhytoResp
                     !----------------------------
                     !Passed variables
        real(r8) :: CK4     !Curve shape parameter
        real(r8) :: CTL     !Maximum temperature response
        real(r8) :: CQ      !Minimum temperature response
        real(r8) :: CTO     !Optimum temperature response
        real(r8) :: Temp1   !Ambient Temperature 
                     !----------------------------

        real(r8) :: G1, G2, CK1, L1, L2, KA, KB
      !--------------------
        CK1 = CK4
        G1 = 1/(CTO - CQ)
        G1 = G1 * log((0.98 * (1 - CK1))/(CK1 * 0.02))
        G2 = 1/(CTL - CTO)
        G2 = G2 * log((0.98 * (1 - CK4))/(CK4 * 0.02))
        L1 = exp(G1 * (Temp1 - CQ))
        L2 = exp(G2 * (CTL - Temp1))
        KA = (CK1 * L1)
        KA = KA/(1 + CK1 * (L1 - 1))
        KB = (CK4 * L2)
        KB = KB/(1 + CK4 * (L2 - 1))
        GetPhytoResp = KA * KB
        Return
        End Function GetPhytoResp

!==============================================================
        Function GetCopepodResp(Temp1)
!--------------------------------------------------------------
! Computes copepod respiration according to Arhonditsis (2005).
!--------------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetCopepodResp
        real(r8) :: Temp1          !Passed variable
        real(r8) :: bm = 0.04   !Basal metabolic rate day**-1
        real(r8) :: ktbm = 0.05 !Temperature response degrees C**-1
        real(r8) :: Tref = 20   !Reference temperature degrees C
        real(r8) :: Resp           !Returned variable

        Resp = bm * exp(ktbm * (Temp1 - Tref))
        GetCopepodResp = Resp
        Return
        End Function GetCopepodResp

!===============================================================
        Function GetPhytoResp2(Temp1,Tref,KbmPh)
!------------------------------------------------------
! Computes the temperature correction for phytoplankton 
! respiration according to Arhonditsis 2005.
!------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetPhytoResp2
        real(r8) :: Temp1      !Temperature, passed
        real(r8) :: Tref       !Reference temperature
        real(r8) :: KbmPh      !Half saturation, temperature

        real(r8) :: Resp       !Returned variable

        Resp = exp(KbmPh * (Temp1 - Tref))
        GetPhytoResp2 = Resp
        Return
        End Function GetPhytoResp2

!=================================================================
        Function GetBasalMetabolism(respPh,kfePh,Iron1)
!---------------------------------------------------------
! Computes an iron correction for the basal metabolism for 
! the phytoplankton respiration calculation
!---------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetBasalMetabolism
        real(r8) :: Iron1     !Iron concentration
        real(r8) :: kfePh     !Half saturation for iron
        real(r8) :: respPh    !Phytoplankton uncorrected basal metabolism
        real(r8) :: BaseMet   !Phytoplankton basal metabolism
         
        BaseMet = Iron1/(kfePh + Iron1)
        BaseMet = BaseMet * ((kfePh +2)/2)
        GetBasalMetabolism = BaseMet * respPh

        Return
        End Function GetBasalMetabolism

!==================================================================
        Function GetCarbChlorRatio(Par1,ccr1)
!----------------------------------------------------------------------
! Computes a carbon chlorphyll ratio based on the optimization function
! in the mat lab code DOS partition at temperature of 10oC
!----------------------------------------------------------------------
        USE mod_param
        implicit none
!
        real(r8) :: GetCarbChlorRatio
        real(r8) :: Par1    !Passed par value
        real(r8) :: ccr1    !Passed minimum carbon chlorophyll ratio
        real(r8) :: CrChlRatio

        CrChlRatio = 2.415_r8 * Par1
        !-------------------------------------------------------
        !If the computed carbon chlorophyll ration is less than
        !the passed minimum value, then return the minimum value
        !-------------------------------------------------------
        if (CrChlRatio .lt. ccr1) CrChlRatio = ccr1
        !----------------------------------------------------
        !If the computed carbon chlorophyll ration is greater 
        !than 400 then return 400
        !----------------------------------------------------
        if (CrChlRatio .gt. 400) CrChlRatio = 400._r8
        
        !if (CrChlratio .gt. 110 ) then 
        !    print *, Par1,'***',ccr1,'***',CrChlRatio
        !endif
        GetCarbChlorRatio = CrChlRatio

        End Function GetCarbChlorRatio
