#include "cppdefs.h"
      MODULE normalization_mod

#ifdef FOUR_DVAR
# ifdef EW_PERIODIC
#  define IR_RANGE Istr-1,Iend+1
#  define IU_RANGE Istr,Iend
#  define IV_RANGE Istr,Iend
# else
#  define IR_RANGE IstrR,IendR
#  define IU_RANGE Istr,IendR
#  define IV_RANGE IstrR,IendR
# endif
# ifdef NS_PERIODIC
#  define JR_RANGE Jstr-1,Jend+1
#  define JU_RANGE Jstr,Jend
#  define JV_RANGE Jstr,Jend
# else
#  define JR_RANGE JstrR,JendR
#  define JU_RANGE JstrR,JendR
#  define JV_RANGE Jstr,JendR
# endif
!
!=======================================================================
!  Copyright (c) ROMS/TOMS Adjoint Group                               !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine computes the background covariance, B, normalization   !
!  factors using the exact or approximated approach.  These  factors   !
!  ensure that the diagonal elements of B are equal to unity. Notice   !
!  that in applications with land/sea masking, it will produce large   !
!  changes in the covariance structures near the boundary.             !
!                                                                      !
!  The exact method is very expensive: the normalization factors are   !
!  computed by perturbing each model grid cell with a delta function   !
!  scaled by the area (2D factors) or volume (3D factors),  and then   !
!  convoluting with the diffusion operator.                            !
!                                                                      !
!  The approximated method is cheaper: the normalization factors are   !
!  computed using the randomization approach of  Fisher and Courtier   !
!  (1995).  The factors are initialized with randon numbers having a   !
!  uniform distribution  (drawn from a population with zero mean and   !
!  unity variance). Then, they scaled the inverse, squared root cell   !
!  area (2D factors) or volume (3D factors) and convoluted with  the   !
!  diffuse operators over a specified number of iterations, Nrandom.   !
!                                                                      !
!  References:                                                         !
!                                                                      !
!    Fisher, M. and. P. Courtier, 1995:  Estimating the covariance     !
!      matrices of analysis and forecast error in variational data     !
!      assimilation, ECMWF Technical Memo N. 220, ECMWF,  Reading,     !
!      UK.                                                             !
!                                                                      !
!      www.ecmwf.int/publications/library/ecpublications/_pdf/tm/      !
!                                                001-300/tm220.pdf     !
!                                                                      !
!    Weaver, A. and P. Courtier, 2001: Correlation modeling on the     !
!      sphere using a generalized diffusion equation, Q.J.R. Meteo.    !
!      Soc, 127, 1815-1846.                                            !
!                                                                      !
!======================================================================!
!
      USE mod_kinds

      implicit none

      PRIVATE
      PUBLIC :: normalization
      PUBLIC :: wrt_norm2d
      PUBLIC :: wrt_norm3d

      CONTAINS
!
!***********************************************************************
      SUBROUTINE normalization (ng, tile, ifac)
!***********************************************************************
!
      USE mod_param
      USE mod_grid
      USE mod_fourdvar
      USE mod_ocean
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile, ifac
!
!  Local variable declarations.
!
# include "tile.h"
!
!  Compute background error covariance normalization factors using
!  the very expensive exact method.
!
      IF (Nmethod(ng).eq.0) THEN
        CALL normalization_tile (ng, Istr, Iend, Jstr, Jend,            &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           ifac,                                  &
     &                           GRID(ng) % pm,                         &
     &                           GRID(ng) % om_r,                       &
     &                           GRID(ng) % om_u,                       &
     &                           GRID(ng) % om_v,                       &
     &                           GRID(ng) % pn,                         &
     &                           GRID(ng) % on_r,                       &
     &                           GRID(ng) % on_u,                       &
     &                           GRID(ng) % on_v,                       &
     &                           GRID(ng) % pmon_p,                     &
     &                           GRID(ng) % pmon_r,                     &
     &                           GRID(ng) % pmon_u,                     &
     &                           GRID(ng) % pnom_p,                     &
     &                           GRID(ng) % pnom_r,                     &
     &                           GRID(ng) % pnom_v,                     &
# ifdef MASKING
     &                           GRID(ng) % rmask,                      &
     &                           GRID(ng) % pmask,                      &
     &                           GRID(ng) % umask,                      &
     &                           GRID(ng) % vmask,                      &
# endif
# ifdef SOLVE3D
     &                           GRID(ng) % h,                          &
#  ifdef ICESHELF
     &                           GRID(ng) % zice,                       &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                           OCEAN(ng) % bed,                       &
     &                           GRID(ng) % bed_thick0,                 &
#  endif
     &                           GRID(ng) % Hz,                         &
     &                           GRID(ng) % z_r,                        &
     &                           GRID(ng) % z_w,                        &
# endif
# ifdef SOLVE3D
     &                           OCEAN(ng) % b_t,                       &
     &                           OCEAN(ng) % b_u,                       &
     &                           OCEAN(ng) % b_v,                       &
# endif
     &                           OCEAN(ng) % b_zeta,                    &
     &                           OCEAN(ng) % b_ubar,                    &
     &                           OCEAN(ng) % b_vbar)
!
!  Compute background error covariance normalization factors using
!  the approximated randomization method.
!
      ELSE IF (Nmethod(ng).eq.1) THEN
        CALL randomization_tile (ng, Istr, Iend, Jstr, Jend,            &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           ifac,                                  &
     &                           GRID(ng) % pm,                         &
     &                           GRID(ng) % om_r,                       &
     &                           GRID(ng) % om_u,                       &
     &                           GRID(ng) % om_v,                       &
     &                           GRID(ng) % pn,                         &
     &                           GRID(ng) % on_r,                       &
     &                           GRID(ng) % on_u,                       &
     &                           GRID(ng) % on_v,                       &
     &                           GRID(ng) % pmon_p,                     &
     &                           GRID(ng) % pmon_r,                     &
     &                           GRID(ng) % pmon_u,                     &
     &                           GRID(ng) % pnom_p,                     &
     &                           GRID(ng) % pnom_r,                     &
     &                           GRID(ng) % pnom_v,                     &
# ifdef MASKING
     &                           GRID(ng) % rmask,                      &
     &                           GRID(ng) % pmask,                      &
     &                           GRID(ng) % umask,                      &
     &                           GRID(ng) % vmask,                      &
# endif
# ifdef SOLVE3D
     &                           GRID(ng) % h,                          &
#  ifdef ICESHELF
     &                           GRID(ng) % zice,                       &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                           OCEAN(ng) % bed,                       &
     &                           GRID(ng) % bed_thick0,                 &
#  endif
     &                           GRID(ng) % Hz,                         &
     &                           GRID(ng) % z_r,                        &
     &                           GRID(ng) % z_w,                        &
# endif
# ifdef SOLVE3D
     &                           OCEAN(ng) % b_t,                       &
     &                           OCEAN(ng) % b_u,                       &
     &                           OCEAN(ng) % b_v,                       &
# endif
     &                           OCEAN(ng) % b_zeta,                    &
     &                           OCEAN(ng) % b_ubar,                    &
     &                           OCEAN(ng) % b_vbar)
      END IF

      RETURN
      END SUBROUTINE normalization
!
!***********************************************************************
      SUBROUTINE normalization_tile (ng, Istr, Iend, Jstr, Jend,        &
     &                               LBi, UBi, LBj, UBj,                &
     &                               ifac,                              &
     &                               pm, om_r, om_u, om_v,              &
     &                               pn, on_r, on_u, on_v,              &
     &                               pmon_p, pmon_r, pmon_u,            &
     &                               pnom_p, pnom_r, pnom_v,            &
# ifdef MASKING
     &                               rmask, pmask, umask, vmask,        &
# endif
# ifdef SOLVE3D
     &                               h,                                 &
#  ifdef ICESHELF
     &                               zice,                              &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                               bed, bed_thick0,                   &
#  endif
     &                               Hz, z_r, z_w,                      &
# endif
# ifdef SOLVE3D
     &                               VnormR, VnormU, VnormV,            &
# endif
     &                               HnormR, HnormU, HnormV)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
      USE bc_2d_mod
      USE conv_2d_mod
# ifdef SOLVE3D
      USE bc_3d_mod
      USE conv_3d_mod
# endif
# ifdef DISTRIBUTE
      USE mp_exchange_mod
# endif
      USE set_depth_mod
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: ifac
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: om_r(LBi:,LBj:)
      real(r8), intent(in) :: om_u(LBi:,LBj:)
      real(r8), intent(in) :: om_v(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: on_r(LBi:,LBj:)
      real(r8), intent(in) :: on_u(LBi:,LBj:)
      real(r8), intent(in) :: on_v(LBi:,LBj:)
      real(r8), intent(in) :: pmon_p(LBi:,LBj:)
      real(r8), intent(in) :: pmon_r(LBi:,LBj:)
      real(r8), intent(in) :: pmon_u(LBi:,LBj:)
      real(r8), intent(in) :: pnom_p(LBi:,LBj:)
      real(r8), intent(in) :: pnom_r(LBi:,LBj:)
      real(r8), intent(in) :: pnom_v(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:,LBj:)
      real(r8), intent(in) :: pmask(LBi:,LBj:)
      real(r8), intent(in) :: umask(LBi:,LBj:)
      real(r8), intent(in) :: vmask(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:,LBj:)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:,LBj:,:,:)
      real(r8), intent(inout):: bed_thick0(LBi:,LBj:)
#   endif
      real(r8), intent(inout) :: h(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:,LBj:,:,:)
      real(r8), intent(out) :: VnormU(LBi:,LBj:,:)
      real(r8), intent(out) :: VnormV(LBi:,LBj:,:)
#  endif
      real(r8), intent(out) :: HnormR(LBi:,LBj:)
      real(r8), intent(out) :: HnormU(LBi:,LBj:)
      real(r8), intent(out) :: HnormV(LBi:,LBj:)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:,LBj:,:)
      real(r8), intent(out) :: z_r(LBi:,LBj:,:)
      real(r8), intent(out) :: z_w(LBi:,LBj:,0:)
#  endif
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_v(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: umask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: vmask(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:UBi,LBj:UBj)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
      real(r8), intent(inout):: bed_thick0(LBi:UBi,LBj:UBj)
#   endif
      real(r8), intent(inout) :: h(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:UBi,LBj:UBj,N(ng),NT(ng))
      real(r8), intent(out) :: VnormU(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: VnormV(LBi:UBi,LBj:UBj,N(ng))
#  endif
      real(r8), intent(out) :: HnormR(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormU(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormV(LBi:UBi,LBj:UBj)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_r(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_w(LBi:UBi,LBj:UBj,0:N(ng))
#  endif
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
# ifdef SOLVE3D
      logical :: Ldiffer, Lsame
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: i, ic, is, j, jc, status
# ifdef SOLVE3D
      integer :: UBt, itrc, k, kc, ntrc
# endif
      integer, dimension(NstateVar(ng)) :: NHsteps
# ifdef SOLVE3D
      integer, dimension(NstateVar(ng)) :: NVsteps
# endif
      real(r8) :: KhMax, cff, compute
# ifdef SOLVE3D
      real(r8) :: KvMax
# endif
      real(r8), dimension(NstateVar(ng)) :: DTsizeH
# ifdef SOLVE3D
      real(r8), dimension(NstateVar(ng)) :: DTsizeV
# endif
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2d
      real(r8), dimension(LBi:UBi,LBj:UBj) :: Hscale
      real(r8), dimension(LBi:UBi,LBj:UBj) :: Kh
# ifdef SOLVE3D
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3d 
      real(r8), dimension(LBi:UBi,LBj:UBj,0:N(ng)) :: Kv
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: Vscale
# endif

# include "set_bounds.h"

# ifdef SOLVE3D
!
!-----------------------------------------------------------------------
!  Compute time invariant depths (use zero free-surface).
!-----------------------------------------------------------------------
!
      DO i=LBi,UBi
        DO j=LBj,UBj
          A2d(i,j)=0.0_r8
        END DO
      END DO

      CALL set_depth_tile (ng, Istr, Iend, Jstr, Jend,                  &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     h,                                           &
#  ifdef ICESHELF
     &                     zice,                                        &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                     bed, bed_thick0,                             &
#  endif
     &                     A2d,                                         &
     &                     Hz, z_r, z_w)
# endif
!
!-----------------------------------------------------------------------
!  Set diffusion operator parameters.
!-----------------------------------------------------------------------
!
!  Set diffusion coefficients.  Notice that an array is used to allow
!  spatial structures in covariance in the future.  For now, set
!  coefficients to unity.
!
      KhMax=0.0_r8
# ifdef SOLVE3D
      KvMax=0.0_r8
# endif
      DO j=LBj,UBj
        DO i=LBi,UBi
          Kh(i,j)=1.0_r8
          KhMax=MAX(KhMax,Kh(i,j))
# ifdef SOLVE3D
          DO k=0,N(ng)
            Kv(i,j,k)=1.0_r8
            KvMax=MAX(KvMax,Kv(i,j,k))
          END DO
# endif
        END DO
      END DO
!
!  Determine time-step size using FTCS stability criteria:
!
!       Kh DTsizeH / MIN(DXmin,DYmin)^2  < Hgamma / 4
!
!       Kv DTsizeH / DZmin^2  < Vgamma / 2
!
!  where a Hgamma and Vgamma are used to scale the time-step below
!  its theoretical limit for stability and accurary.
!
      cff=MIN(DXmin(ng),DYmin(ng))      
      DO is=1,NstateVar(ng)
        DTsizeH(is)=Hgamma*cff*cff/(4.0_r8*KhMax)
# ifdef SOLVE3D
#  ifdef IMPLICIT_VCONV
        DTsizeV(is)=Vgamma*DZmax(ng)*DZmax(ng)/(2.0_r8*KvMax)
#  else
        DTsizeV(is)=Vgamma*DZmin(ng)*DZmin(ng)/(2.0_r8*KvMax)
#  endif
# endif
      END DO
!
!  Determine number of integeration steps as function of the spatial
!  decorrelation scale (Hdecay, Vdecay). Notice that the diffusion
!  arrays of order one.  The IFAC is used to iterate the diffusion
!  operator over the whole or half time-steps. The number of steps
!  are forced to be even.
!
      DO is=1,NstateVar(ng)
        NHsteps(is)=NINT(Hdecay(is,ng)*Hdecay(is,ng)/                   &
     &                   (4.0_r8*KhMax*DTsizeH(is)))
        IF (MOD(NHsteps(is),2).ne.0) THEN
          NHsteps(is)=(NHsteps(is)+1)/ifac
        ELSE
          NHsteps(is)=NHsteps(is)/ifac
        END IF
# ifdef SOLVE3D
        NVsteps(is)=NINT(Vdecay(is,ng)*Vdecay(is,ng)/                   &
     &                   (2.0_r8*KvMax*DTsizeV(is)))
#  ifdef IMPLICIT_VCONV
        NVsteps(is)=MAX(1,NVsteps(is))
#  endif
        IF (MOD(NVsteps(is),2).ne.0) THEN
          NVsteps(is)=(NVsteps(is)+1)/ifac
        ELSE
          NVsteps(is)=NVsteps(is)/ifac
        END IF
# endif 
      END DO
!
!-----------------------------------------------------------------------
!  Compute background covariance, B, normalization factors using the 
!  exact method. It involves computing the filter (convolution)
!  variance at each point independenly.  That is, each point is
!  perturbed with a delta function, scaled by the inverse squared
!  root of the area (2D) or volume (3D), and then convoluted.  
!-----------------------------------------------------------------------
!
!  Set time record index to write in normalization NetCDF file.
!
      tNRMindx(ng)=tNRMindx(ng)+1
      NrecNRM(ng)=NrecNRM(ng)+1
!
!  Write out model time (s).
!
      IF (OutThread) THEN
        status=nf_put_var1_TYPE(ncNRMid(ng), nrmVid(idtime,ng),         &
     &                          tNRMindx(ng), time(ng))
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,10) TRIM(Vname(1,idtime)), tNRMindx(ng)
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  2D norm at RHO-points.
!
      IF (Cnorm(isFsur)) THEN
        IF (Master) THEN
          WRITE (stdout,20)
          WRITE (stdout,30) '2D normalization factors at RHO-points'
          CALL my_flush (stdout)
        END IF
        DO j=JR_RANGE
          DO i=IR_RANGE
            Hscale(i,j)=1.0_r8/(om_r(i,j)*on_r(i,j))
          END DO
        END DO
        DO jc=1,Mm(ng)
          DO ic=1,Lm(ng)
# ifdef MASKING
            compute=0.0_r8
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              IF (rmask(ic,jc).gt.0) compute=1.0_r8
            END IF
#  ifdef DISTRIBUTE
            CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#  endif
# else
            compute=1.0_r8
# endif
            IF (compute.gt.0.0_r8) THEN
              DO j=LBj,UBj
                DO i=LBi,UBi
                  A2d(i,j)=0.0_r8
                END DO
              END DO
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                A2d(ic,jc)=Hscale(ic,jc)
              END IF
              CALL conv_r2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,     &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            NghostPoints,                         &
     &                            NHsteps(isFsur), DTsizeH(isFsur),     &
     &                            Kh,                                   &
     &                            pm, pn, pmon_u, pnom_v,               &
# ifdef MASKING
     &                            rmask, umask, vmask,                  &
# endif
     &                            A2d)
              cff=1.0_r8/SQRT(A2d(ic,jc)) 
            ELSE
              cff=0.0_r8
            END IF
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              HnormR(ic,jc)=cff
            END IF
          END DO
        END DO
        CALL bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormR)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormR)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idFsur,                    &
     &                   ncNRMid(ng), nrmVid(idFsur,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   rmask,                                         &
# endif
     &                   HnormR)
      END IF
!
!  2D norm at U-points.
!
      IF (Cnorm(isUbar)) THEN
        IF (Master) THEN
          WRITE (stdout,30) '2D normalization factors at   U-points'
          CALL my_flush (stdout)
        END IF
        DO j=JU_RANGE
          DO i=IU_RANGE
            Hscale(i,j)=1.0_r8/(om_u(i,j)*on_u(i,j))
          END DO
        END DO
        DO jc=1,Mm(ng)
          DO ic=2,Lm(ng)
# ifdef MASKING
            compute=0.0_r8
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              IF (umask(ic,jc).gt.0) compute=1.0_r8
            END IF
#  ifdef DISTRIBUTE
            CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#  endif
# else
            compute=1.0_r8
# endif
            IF (compute.gt.0.0_r8) THEN
              DO j=LBj,UBj
                DO i=LBi,UBi
                  A2d(i,j)=0.0_r8
                END DO
              END DO
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                A2d(ic,jc)=Hscale(ic,jc)
              END IF
              CALL conv_u2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,     &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            NghostPoints,                         &
     &                            NHsteps(isUbar), DTsizeH(isUbar),     &
     &                            Kh,                                   &
     &                            pm, pn, pmon_r, pnom_p,               &
# ifdef MASKING
     &                            umask, pmask,                         &
# endif
     &                            A2d)
              cff=1.0_r8/SQRT(A2d(ic,jc))
            ELSE
              cff=0.0_r8
            END IF
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              HnormU(ic,jc)=cff
            END IF
          END DO
        END DO
        CALL bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormU)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormU)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idUbar,                    &
     &                   ncNRMid(ng), nrmVid(idUbar,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   umask,                                         &
# endif
     &                   HnormU)
      END IF
!
!  2D norm at V-points.
!
      IF (Cnorm(isVbar)) THEN
        IF (Master) THEN
          WRITE (stdout,30) '2D normalization factors at   V-points'
          CALL my_flush (stdout)
        END IF
        DO j=JV_RANGE
          DO i=IV_RANGE
            Hscale(i,j)=1.0_r8/(om_v(i,j)*on_v(i,j))
          END DO
        END DO
        DO jc=2,Mm(ng)
          DO ic=1,Lm(ng)
# ifdef MASKING
            compute=0.0_r8
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              IF (vmask(ic,jc).gt.0) compute=1.0_r8
            END IF
#  ifdef DISTRIBUTE
            CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#  endif
# else
            compute=1.0_r8
# endif
            IF (compute.gt.0.0_r8) THEN
              DO j=LBj,UBj
                DO i=LBi,UBi
                  A2d(i,j)=0.0_r8
                END DO
              END DO
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                A2d(ic,jc)=Hscale(ic,jc)
              END IF
              CALL conv_v2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,     &
     &                            LBi, UBi, LBj, UBj,                   &
     &                            NghostPoints,                         &
     &                            NHsteps(isVbar), DTsizeH(isVbar),     &
     &                            Kh,                                   &
     &                            pm, pn, pmon_p, pnom_r,               &
# ifdef MASKING
     &                            vmask, pmask,                         &
# endif
     &                            A2d)
              cff=1.0_r8/SQRT(A2d(ic,jc))
            ELSE
              cff=0.0_r8
            END IF
            IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                    &
     &          ((Istr.le.ic).and.(ic.le.Iend))) THEN
              HnormV(ic,jc)=cff
            END IF
          END DO
        END DO
        CALL bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormV)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormV)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idVbar,                    &
     &                   ncNRMid(ng), nrmVid(idVbar,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   vmask,                                         &
# endif
     &                   HnormV)
      END IF

# ifdef SOLVE3D
!
!  3D norm U-points.
!
      IF (Cnorm(isUvel)) THEN
        IF (Master) THEN
          WRITE (stdout,30) '3D normalization factors at   U-points'
          CALL my_flush (stdout)
        END IF
        DO j=JU_RANGE
          DO i=IU_RANGE
            cff=om_u(i,j)*on_u(i,j)*0.5_r8
            DO k=1,N(ng)
              Vscale(i,j,k)=1.0_r8/(cff*(Hz(i-1,j,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        DO kc=1,N(ng)
          DO jc=1,Mm(ng)
            DO ic=2,Lm(ng)
#  ifdef MASKING
              compute=0.0_r8
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                IF (umask(ic,jc).gt.0) compute=1.0_r8
              END IF
#   ifdef DISTRIBUTE
              CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#   endif
#  else
              compute=1.0_r8
#  endif
              IF (compute.gt.0.0_r8) THEN
                DO k=1,N(ng)
                  DO j=LBj,UBj
                    DO i=LBi,UBi
                      A3d(i,j,k)=0.0_r8
                    END DO
                  END DO
                END DO
                IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                &
     &              ((Istr.le.ic).and.(ic.le.Iend))) THEN
                  A3d(ic,jc,kc)=Vscale(ic,jc,kc)
                END IF
                CALL conv_u3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,   &
     &                              LBi, UBi, LBj, UBj, 1, N(ng),       &
     &                              NghostPoints,                       &
     &                              NHsteps(isUvel), NVsteps(isUvel),   &
     &                              DTsizeH(isUvel), DTsizeV(isUvel),   &
     &                              Kh, Kv,                             &
     &                              pm, pn, pmon_r, pnom_p,             &
#  ifdef MASKING
     &                              umask, pmask,                       &
#  endif
     &                              Hz, z_r,                            &
     &                              A3d)
                cff=1.0_r8/SQRT(A3d(ic,jc,kc))
              ELSE
                cff=0.0_r8
              END IF
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                VnormU(ic,jc,kc)=cff
              END IF
            END DO
          END DO
        END DO
        CALL bc_u3d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj, 1, N(ng),                 &
     &                    VnormU)
#  ifdef DISTRIBUTE
        CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      VnormU)
#  endif
        CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, 1, N(ng), idUvel,          &
     &                   ncNRMid(ng), nrmVid(idUvel,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                   umask,                                         &
#  endif
     &                   VnormU)
      END IF
!
!  3D norm at V-points.
!
      IF (Cnorm(isVvel)) THEN
        IF (Master) THEN
          WRITE (stdout,30) '3D normalization factors at   V-points'
          CALL my_flush (stdout)
        END IF
        DO j=JV_RANGE
          DO i=IV_RANGE
            cff=om_v(i,j)*on_v(i,j)*0.5_r8
            DO k=1,N(ng)
              Vscale(i,j,k)=1.0_r8/(cff*(Hz(i,j-1,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        DO kc=1,N(ng)
          DO jc=2,Mm(ng)
            DO ic=1,Lm(ng)
#  ifdef MASKING
              compute=0.0_r8
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                IF (vmask(ic,jc).gt.0) compute=1.0_r8
              END IF
#   ifdef DISTRIBUTE
              CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#   endif
#  else
              compute=1.0_r8
#  endif
              IF (compute.gt.0.0_r8) THEN
                DO k=1,N(ng)
                  DO j=LBj,UBj
                    DO i=LBi,UBi
                      A3d(i,j,k)=0.0_r8
                    END DO
                  END DO
                END DO
                IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                &
     &              ((Istr.le.ic).and.(ic.le.Iend))) THEN
                  A3d(ic,jc,kc)=Vscale(ic,jc,kc)
                END IF
                CALL conv_v3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,   &
     &                              LBi, UBi, LBj, UBj, 1, N(ng),       &
     &                              NghostPoints,                       &
     &                              NHsteps(isVvel), NVsteps(isVvel),   &
     &                              DTsizeH(isVvel), DTsizeV(isVvel),   &
     &                              Kh, Kv,                             &
     &                              pm, pn, pmon_p, pnom_r,             &
#  ifdef MASKING
     &                              vmask, pmask,                       &
#  endif
     &                              Hz, z_r,                            &
     &                              A3d)
                cff=1.0_r8/SQRT(A3d(ic,jc,kc))
              ELSE
                cff=0.0_r8
              END IF
              IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                  &
     &            ((Istr.le.ic).and.(ic.le.Iend))) THEN
                VnormV(ic,jc,kc)=cff
              END IF
            END DO
          END DO
        END DO
        CALL bc_v3d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj, 1, N(ng),                 &
     &                    VnormV)
#  ifdef DISTRIBUTE
        CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      VnormV)
#  endif
        CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, 1, N(ng), idVvel,          &
     &                   ncNRMid(ng), nrmVid(idVvel,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                   vmask,                                         &
#  endif
     &                   VnormV)
      END IF
!
!  3D norm at RHO-points.
!
      IF (Master) THEN
        WRITE (stdout,30) '3D normalization factors at RHO-points'
        CALL my_flush (stdout)
      END IF
!
!  Check if the decorrelation scales for all the tracers are different.
!  If not, just compute the normalization factors for the first tracer
!  and assign the same value to the rest.  Recall that this computation
!  is very expensive.
!
      Ldiffer=.FALSE.
      DO itrc=2,NT(ng)
        IF ((Hdecay(isTvar(itrc  ),ng).ne.                              &
     &       Hdecay(isTvar(itrc-1),ng)).and.                            &
     &      (Vdecay(isTvar(itrc  ),ng).ne.                              &
     &       Vdecay(isTvar(itrc-1),ng))) THEN
          Ldiffer=.TRUE.
        END IF
      END DO
      IF (.not.Ldiffer) THEN
        Lsame=.TRUE.
        UBt=1
      ELSE
        Lsame=.FALSE.
        UBt=NT(ng)
      END IF
!
      DO j=JR_RANGE
        DO i=IR_RANGE
          cff=om_r(i,j)*on_r(i,j)
          DO k=1,N(ng)
            Vscale(i,j,k)=1.0_r8/(cff*Hz(i,j,k))
          END DO
        END DO
      END DO
      DO itrc=1,UBt
        is=isTvar(itrc)
        IF (Cnorm(is)) THEN
          DO kc=1,N(ng)
            DO jc=1,Mm(ng)
              DO ic=1,Lm(ng)
#  ifdef MASKING
                compute=0.0_r8
                IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                &
     &              ((Istr.le.ic).and.(ic.le.Iend))) THEN
                  IF (rmask(ic,jc).gt.0) compute=1.0_r8
                END IF
#   ifdef DISTRIBUTE
                CALL mp_reduce (ng, iNLM, 1, compute, 'SUM')
#   endif
#  else
                compute=1.0_r8
#  endif
                IF (compute.gt.0.0_r8) THEN
                  DO k=1,N(ng)
                    DO j=LBj,UBj
                      DO i=LBi,UBi
                        A3d(i,j,k)=0.0_r8
                      END DO
                    END DO
                  END DO
                  IF (((Jstr.le.jc).and.(jc.le.Jend)).and.              &
     &                ((Istr.le.ic).and.(ic.le.Iend))) THEN
                    A3d(ic,jc,kc)=Vscale(ic,jc,kc)
                  END IF
                  CALL conv_r3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend, &
     &                                LBi, UBi, LBj, UBj, 1, N(ng),     &
     &                                NghostPoints,                     &
     &                                NHsteps(is), NVsteps(is),         &
     &                                DTsizeH(is), DTsizeV(is),         &
     &                                Kh, Kv,                           &
     &                                pm, pn, pmon_u, pnom_v,           &
#  ifdef MASKING
     &                                rmask, umask, vmask,              &
#  endif
     &                                Hz, z_r,                          &
     &                                A3d)
                  cff=1.0_r8/SQRT(A3d(ic,jc,kc))
                ELSE
                  cff=0.0_r8
                END IF
                IF (((Jstr.le.jc).and.(jc.le.Jend)).and.                &
     &              ((Istr.le.ic).and.(ic.le.Iend))) THEN
                  IF (Lsame) THEN
                    DO ntrc=1,NT(ng)
                      VnormR(ic,jc,kc,ntrc)=cff
                    END DO
                  ELSE
                    VnormR(ic,jc,kc,itrc)=cff
                  END IF
                END IF
              END DO
            END DO
          END DO
        END IF
      END DO
      DO itrc=1,NT(ng)
        is=isTvar(itrc)
        IF (Cnorm(is)) THEN
          CALL bc_r3d_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      VnormR(:,:,:,itrc))
#  ifdef DISTRIBUTE
          CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,      &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        NghostPoints, EWperiodic, NSperiodic,     &
     &                        VnormR(:,:,:,itrc))
#  endif
          CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), idTvar(itrc),  &
     &                     ncNRMid(ng), nrmTid(itrc,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                     rmask,                                       &
#  endif
     &                     VnormR(:,:,:,itrc))
        END IF
      END DO
# endif
!
      IF (Master) THEN
        WRITE (stdout,40)
      END IF

 10   FORMAT (/,' NORMALIZATION - error while writing variable: ',a,/,  &
     &        11x,'into normalization NetCDF file for time record: ',i4)
 20   FORMAT (/,' Background Error Covariance Factors: Exact Method',/)
 30   FORMAT (4x,'Computing ',a)
 40   FORMAT (/)

      RETURN
      END SUBROUTINE normalization_tile

!
!***********************************************************************
      SUBROUTINE randomization_tile (ng, Istr, Iend, Jstr, Jend,        &
     &                               LBi, UBi, LBj, UBj,                &
     &                               ifac,                              &
     &                               pm, om_r, om_u, om_v,              &
     &                               pn, on_r, on_u, on_v,              &
     &                               pmon_p, pmon_r, pmon_u,            &
     &                               pnom_p, pnom_r, pnom_v,            &
# ifdef MASKING
     &                               rmask, pmask, umask, vmask,        &
# endif
# ifdef SOLVE3D
     &                               h,                                 &
#  ifdef ICESHELF
     &                               zice,                              &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                               bed, bed_thick0,                   &
#  endif
     &                               Hz, z_r, z_w,                      &
# endif
# ifdef SOLVE3D
     &                               VnormR, VnormU, VnormV,            &
# endif
     &                               HnormR, HnormU, HnormV)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
      USE bc_2d_mod
      USE conv_2d_mod
# ifdef SOLVE3D
      USE bc_3d_mod
      USE conv_3d_mod
# endif
# ifdef DISTRIBUTE
      USE mp_exchange_mod
# endif
      USE set_depth_mod
      USE white_noise_mod, ONLY : white_noise2d
# ifdef SOLVE3D
      USE white_noise_mod, ONLY : white_noise3d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: ifac
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: om_r(LBi:,LBj:)
      real(r8), intent(in) :: om_u(LBi:,LBj:)
      real(r8), intent(in) :: om_v(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: on_r(LBi:,LBj:)
      real(r8), intent(in) :: on_u(LBi:,LBj:)
      real(r8), intent(in) :: on_v(LBi:,LBj:)
      real(r8), intent(in) :: pmon_p(LBi:,LBj:)
      real(r8), intent(in) :: pmon_r(LBi:,LBj:)
      real(r8), intent(in) :: pmon_u(LBi:,LBj:)
      real(r8), intent(in) :: pnom_p(LBi:,LBj:)
      real(r8), intent(in) :: pnom_r(LBi:,LBj:)
      real(r8), intent(in) :: pnom_v(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:,LBj:)
      real(r8), intent(in) :: pmask(LBi:,LBj:)
      real(r8), intent(in) :: umask(LBi:,LBj:)
      real(r8), intent(in) :: vmask(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:,LBj:)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:,LBj:,:,:)
      real(r8), intent(inout):: bed_thick0(LBi:,LBj:)
#   endif
      real(r8), intent(inout) :: h(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:,LBj:,:,:)
      real(r8), intent(out) :: VnormU(LBi:,LBj:,:)
      real(r8), intent(out) :: VnormV(LBi:,LBj:,:)
#  endif
      real(r8), intent(out) :: HnormR(LBi:,LBj:)
      real(r8), intent(out) :: HnormU(LBi:,LBj:)
      real(r8), intent(out) :: HnormV(LBi:,LBj:)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:,LBj:,:)
      real(r8), intent(out) :: z_r(LBi:,LBj:,:)
      real(r8), intent(out) :: z_w(LBi:,LBj:,0:)
#  endif
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_v(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: umask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: vmask(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:UBi,LBj:UBj)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
      real(r8), intent(inout):: bed_thick0(LBi:UBi,LBj:UBj)
#   endif
      real(r8), intent(inout) :: h(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:UBi,LBj:UBj,N(ng),NT(ng))
      real(r8), intent(out) :: VnormU(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: VnormV(LBi:UBi,LBj:UBj,N(ng))
#  endif
      real(r8), intent(out) :: HnormR(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormU(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormV(LBi:UBi,LBj:UBj)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_r(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_w(LBi:UBi,LBj:UBj,0:N(ng))
#  endif
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
# ifdef SOLVE3D
      logical :: Ldiffer, Lsame
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: i, is, iter, j, status
# ifdef SOLVE3D
      integer :: Ubt, itrc, k
# endif
      integer, dimension(NstateVar(ng)) :: NHsteps
# ifdef SOLVE3D
      integer, dimension(NstateVar(ng)) :: NVsteps
# endif
      real(r8) :: KhMax, cff, val
# ifdef SOLVE3D
      real(r8) :: KvMax
# endif
      real(r8), dimension(NstateVar(ng)) :: DTsizeH
# ifdef SOLVE3D
      real(r8), dimension(NstateVar(ng)) :: DTsizeV
# endif
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2d
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2davg
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2dsqr
      real(r8), dimension(LBi:UBi,LBj:UBj) :: Kh
      real(r8), dimension(LBi:UBi,LBj:UBj) :: Hscale
# ifdef SOLVE3D
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3d
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3davg
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3dsqr
      real(r8), dimension(LBi:UBi,LBj:UBj,0:N(ng)) :: Kv
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: Vscale
# endif

# include "set_bounds.h"

# ifdef SOLVE3D
!
!-----------------------------------------------------------------------
!  Compute time invariant depths (use zero free-surface).
!-----------------------------------------------------------------------
!
      DO i=LBi,UBi
        DO j=LBj,UBj
          A2d(i,j)=0.0_r8
        END DO
      END DO

      CALL set_depth_tile (ng, Istr, Iend, Jstr, Jend,                  &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     h,                                           &
#  ifdef ICESHELF
     &                     zice,                                        &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                     bed, bed_thick0,                             &
#  endif
     &                     A2d,                                         &
     &                     Hz, z_r, z_w)
# endif
!
!-----------------------------------------------------------------------
!  Set diffusion operator parameters.
!-----------------------------------------------------------------------
!
!  Set diffusion coefficients.  Notice that an array is used to allow
!  spatial structures in covariance in the future.  For now, set
!  coefficients to unity.
!
      KhMax=0.0_r8
# ifdef SOLVE3D
      KvMax=0.0_r8
# endif
      DO j=LBj,UBj
        DO i=LBi,UBi
          Kh(i,j)=1.0_r8
          KhMax=MAX(KhMax,Kh(i,j))
# ifdef SOLVE3D
          DO k=0,N(ng)
            Kv(i,j,k)=1.0_r8
            KvMax=MAX(KvMax,Kv(i,j,k))
          END DO
# endif
        END DO
      END DO
!
!  Determine time-step size using FTCS stability criteria:
!
!       Kh DTsizeH / MIN(DXmin,DYmin)^2  < Hgamma / 4
!
!       Kv DTsizeH / DZmin^2  < Vgamma / 2
!
!  where a Hgamma and Vgamma are used to scale the time-step below
!  its theoretical limit for stability and accurary.
!
      cff=MIN(DXmin(ng),DYmin(ng))      
      DO is=1,NstateVar(ng)
        DTsizeH(is)=Hgamma*cff*cff/(4.0_r8*KhMax)
# ifdef SOLVE3D
#  ifdef IMPLICIT_VCONV
        DTsizeV(is)=Vgamma*DZmax(ng)*DZmax(ng)/(2.0_r8*KvMax)
#  else
        DTsizeV(is)=Vgamma*DZmin(ng)*DZmin(ng)/(2.0_r8*KvMax)
#  endif
# endif
      END DO
!
!  Determine number of integeration steps as function of the spatial
!  decorrelation scale (Hdecay, Vdecay). Notice that the diffusion
!  arrays of order one.  The IFAC is used to iterate the diffusion
!  operator over the whole or half time-steps. The number of steps
!  are forced to be even.
!
      DO is=1,NstateVar(ng)
        NHsteps(is)=NINT(Hdecay(is,ng)*Hdecay(is,ng)/                   &
     &                   (4.0_r8*KhMax*DTsizeH(is)))
        IF (MOD(NHsteps(is),2).ne.0) THEN
          NHsteps(is)=(NHsteps(is)+1)/ifac
        ELSE
          NHsteps(is)=NHsteps(is)/ifac
        END IF
# ifdef SOLVE3D
        NVsteps(is)=NINT(Vdecay(is,ng)*Vdecay(is,ng)/                   &
     &                   (2.0_r8*KvMax*DTsizeV(is)))
#  ifdef IMPLICIT_VCONV
        NVsteps(is)=MAX(1,NVsteps(is))
#  endif
        IF (MOD(NVsteps(is),2).ne.0) THEN
          NVsteps(is)=(NVsteps(is)+1)/ifac
        ELSE
          NVsteps(is)=NVsteps(is)/ifac
        END IF
# endif 
      END DO
!
!-----------------------------------------------------------------------
!  Compute background covariance, B, normalization factors using the 
!  randomization approach of Fisher and Courtier (1995). These factors
!  ensure that the diagonal elements of B are equal to unity.
!  Notice that in applications with land/sea masking, the boundary
!  conditions will produce large changes in the covariance structures
!  near the boundary.
!
!  Initialize factors with randon numbers ("white-noise") having an
!  uniform distribution (zero mean and unity variance). Then, scale
!  by the inverse squared root area (2D) or volume (3D) and "color"
!  with the diffusion operator. Iterate this step over a specified
!  number of ensamble members, Nrandom.
!-----------------------------------------------------------------------
!
!  Set time record index to write in normalization NetCDF file.
!
      tNRMindx(ng)=tNRMindx(ng)+1
      NrecNRM(ng)=NrecNRM(ng)+1
!
!  Write out model time (s).
!
      IF (OutThread) THEN
        status=nf_put_var1_TYPE(ncNRMid(ng), nrmVid(idtime,ng),         &
     &                          tNRMindx(ng), time(ng))
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,10) TRIM(Vname(1,idtime)), tNRMindx(ng)
          exit_flag=3
          ioerror=status
          RETURN
        END IF
      END IF
!
!  2D norm at RHO-points.
!
      IF (Cnorm(isFsur)) THEN
        cff=1.0_r8/REAL(Nrandom,r8)
        IF (Master) THEN
          WRITE (stdout,20)
          WRITE (stdout,30) '2D normalization factors at RHO-points'
          CALL my_flush (stdout)
        END IF
        DO j=JR_RANGE
          DO i=IR_RANGE
            A2davg(i,j)=0.0_r8
            A2dsqr(i,j)=0.0_r8
            Hscale(i,j)=1.0_r8/(om_r(i,j)*on_r(i,j))
          END DO
        END DO
        DO iter=1,Nrandom
          CALL white_noise2d (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        Rscheme(ng), A2d)
          DO j=JR_RANGE
            DO i=IR_RANGE
              A2d(i,j)=A2d(i,j)*Hscale(i,j)
            END DO
          END DO
          CALL conv_r2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,         &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        NghostPoints,                             &
     &                        NHsteps(isFsur), DTsizeH(isFsur),         &
     &                        Kh,                                       &
     &                        pm, pn, pmon_u, pnom_v,                   &
# ifdef MASKING
     &                        rmask, umask, vmask,                      &
# endif
     &                        A2d)
          DO j=Jstr,Jend
            DO i=Istr,Iend
              A2davg(i,j)=A2davg(i,j)+A2d(i,j)
              A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
            END DO
          END DO
        END DO
        DO j=Jstr,Jend
          DO i=Istr,Iend
            HnormR(i,j)=1.0_r8/                                         &
     &                  SQRT(cff*(A2dsqr(i,j)+                          &
     &                            cff*A2davg(i,j)*A2davg(i,j)))
          END DO
        END DO
        CALL bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormR)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormR)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idFsur,                    &
     &                   ncNRMid(ng), nrmVid(idFsur,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   rmask,                                         &
# endif
     &                   HnormR)
      END IF
!
!  2D norm at U-points.
!
      IF (Cnorm(isUbar)) THEN
        cff=1.0_r8/REAL(Nrandom,r8)
        IF (Master) THEN
          WRITE (stdout,30) '2D normalization factors at   U-points'
          CALL my_flush (stdout)
        END IF
        DO j=JU_RANGE
          DO i=IU_RANGE
            A2davg(i,j)=0.0_r8
            A2dsqr(i,j)=0.0_r8
            Hscale(i,j)=1.0_r8/(om_u(i,j)*on_u(i,j))
          END DO
        END DO
        DO iter=1,Nrandom
          CALL white_noise2d (ng, IstrU, Iend, Jstr, Jend,              &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        Rscheme(ng), A2d)
          DO j=JU_RANGE
            DO i=IU_RANGE
              A2d(i,j)=A2d(i,j)*Hscale(i,j)
            END DO
          END DO
          CALL conv_u2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,         &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        NghostPoints,                             &
     &                        NHsteps(isUbar), DTsizeH(isUbar),         &
     &                        Kh,                                       &
     &                        pm, pn, pmon_r, pnom_p,                   &
#  ifdef MASKING
     &                        umask, pmask,                             &
#  endif
     &                        A2d)
          DO j=Jstr,Jend
            DO i=IstrU,Iend
              A2davg(i,j)=A2davg(i,j)+A2d(i,j)
              A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
            END DO
          END DO
        END DO
        DO j=Jstr,Jend
          DO i=IstrU,Iend
            HnormU(i,j)=1.0_r8/                                         &
     &                  SQRT(cff*(A2dsqr(i,j)+                          &
     &                            cff*A2davg(i,j)*A2davg(i,j)))
          END DO
        END DO
        CALL bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormU)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormU)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idUbar,                    &
     &                   ncNRMid(ng), nrmVid(idUbar,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   umask,                                         &
# endif
     &                   HnormU)
      END IF
!
!  2D norm at V-points.
!
      IF (Cnorm(isVbar)) THEN
        cff=1.0_r8/REAL(Nrandom,r8)
        IF (Master) THEN
          WRITE (stdout,30) '2D normalization factors at   V-points'
          CALL my_flush (stdout)
        END IF
        DO j=JV_RANGE
          DO i=IV_RANGE
            A2davg(i,j)=0.0_r8
            A2dsqr(i,j)=0.0_r8
            Hscale(i,j)=1.0_r8/(om_v(i,j)*on_v(i,j))
          END DO
        END DO
        DO iter=1,Nrandom
          CALL white_noise2d (ng, Istr, Iend, JstrV, Jend,              &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        Rscheme(ng), A2d)
          DO j=JV_RANGE
            DO i=IV_RANGE
              A2d(i,j)=A2d(i,j)*Hscale(i,j)
            END DO
          END DO
          CALL conv_v2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,         &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        NghostPoints,                             &
     &                        NHsteps(isVbar), DTsizeH(isVbar),         &
     &                        Kh,                                       &
     &                        pm, pn, pmon_p, pnom_r,                   &
#  ifdef MASKING
     &                        vmask, pmask,                             &
#  endif
     &                        A2d)
          DO j=JstrV,Jend
            DO i=Istr,Iend
              A2davg(i,j)=A2davg(i,j)+A2d(i,j)
              A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
            END DO
          END DO
        END DO
        DO j=JstrV,Jend
          DO i=Istr,Iend
            HnormV(i,j)=1.0_r8/                                         &
     &                  SQRT(cff*(A2dsqr(i,j)+                          &
     &                            cff*A2davg(i,j)*A2davg(i,j)))
          END DO
        END DO
        CALL bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj,                           &
     &                    HnormV)
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      HnormV)
# endif
        CALL wrt_norm2d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, idVbar,                    &
     &                   ncNRMid(ng), nrmVid(idVbar,ng), tNRMindx(ng),  &
# ifdef MASKING
     &                   vmask,                                         &
# endif
     &                   HnormV)
      END IF

# ifdef SOLVE3D
!
!  3D norm U-points.
!
      IF (Cnorm(isUvel)) THEN
        cff=1.0_r8/REAL(Nrandom,r8)
        IF (Master) THEN
          WRITE (stdout,30) '3D normalization factors at   U-points'
          CALL my_flush (stdout)
        END IF
        DO j=JU_RANGE
          DO i=IU_RANGE
            val=om_u(i,j)*on_u(i,j)*0.5_r8
            DO k=1,N(ng)
              A3davg(i,j,k)=0.0_r8
              A3dsqr(i,j,k)=0.0_r8
              Vscale(i,j,k)=1.0_r8/(val*(Hz(i-1,j,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        DO iter=1,Nrandom
          CALL white_noise3d (ng, IstrU, Iend, Jstr, Jend,              &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        Rscheme(ng), A3d)
          DO k=1,N(ng)
            DO j=JU_RANGE
              DO i=IU_RANGE
                A3d(i,j,k)=A3d(i,j,k)*Vscale(i,j,k)
              END DO
            END DO
          END DO
          CALL conv_u3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,         &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        NghostPoints,                             &
     &                        NHsteps(isUvel), NVsteps(isUvel),         &
     &                        DTsizeH(isUvel), DTsizeV(isUvel),         &
     &                        Kh, Kv,                                   &
     &                        pm, pn, pmon_r, pnom_p,                   &
#  ifdef MASKING
     &                        umask, pmask,                             &
#  endif
     &                        Hz, z_r,                                  &
     &                        A3d)
          DO k=1,N(ng)
            DO j=Jstr,Jend
              DO i=IstrU,Iend
                A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
                A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
              END DO
            END DO
          END DO
        END DO
        DO k=1,N(ng)
          DO j=Jstr,Jend
            DO i=IstrU,Iend
              VnormU(i,j,k)=1.0_r8/                                     &
     &                      SQRT(cff*(A3dsqr(i,j,k)+                    &
     &                                cff*A3davg(i,j,k)*A3davg(i,j,k)))
            END DO
          END DO
        END DO
        CALL bc_u3d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj, 1, N(ng),                 &
     &                    VnormU)
#  ifdef DISTRIBUTE
        CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      VnormU)
#  endif
        CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, 1, N(ng), idUvel,          &
     &                   ncNRMid(ng), nrmVid(idUvel,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                   umask,                                         &
#  endif
     &                   VnormU)
      END IF
!
!  3D norm at V-points.
!
      IF (Cnorm(isVvel)) THEN
        cff=1.0_r8/REAL(Nrandom,r8)
        IF (Master) THEN
          WRITE (stdout,30) '3D normalization factors at   V-points'
          CALL my_flush (stdout)
        END IF
        DO j=JV_RANGE
          DO i=IV_RANGE
            val=om_v(i,j)*on_v(i,j)*0.5_r8
            DO k=1,N(ng)
              A3davg(i,j,k)=0.0_r8
              A3dsqr(i,j,k)=0.0_r8
              Vscale(i,j,k)=1.0_r8/(val*(Hz(i,j-1,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        DO iter=1,Nrandom
          CALL white_noise3d (ng, Istr, Iend, JstrV, Jend,              &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        Rscheme(ng), A3d)
          DO k=1,N(ng)
            DO j=JV_RANGE
              DO i=IV_RANGE
                A3d(i,j,k)=A3d(i,j,k)*Vscale(i,j,k)
              END DO
            END DO
          END DO
          CALL conv_v3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,         &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        NghostPoints,                             &
     &                        NHsteps(isVvel), NVsteps(isVvel),         &
     &                        DTsizeH(isVvel), DTsizeV(isVvel),         &
     &                        Kh, Kv,                                   &
     &                        pm, pn, pmon_p, pnom_r,                   &
#  ifdef MASKING
     &                        vmask, pmask,                             &
#  endif
     &                        Hz, z_r,                                  &
     &                        A3d)
          DO k=1,N(ng)
            DO j=JstrV,Jend
              DO i=Istr,Iend
                A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
                A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
              END DO
            END DO
          END DO
        END DO
        DO k=1,N(ng)
          DO j=JstrV,Jend
            DO i=Istr,Iend
              VnormV(i,j,k)=1.0_r8/                                     &
     &                      SQRT(cff*(A3dsqr(i,j,k)+                    &
     &                                cff*A3davg(i,j,k)*A3davg(i,j,k)))
            END DO
          END DO
        END DO
        CALL bc_v3d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    LBi, UBi, LBj, UBj, 1, N(ng),                 &
     &                    VnormV)
#  ifdef DISTRIBUTE
        CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      VnormV)
#  endif
        CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,              &
     &                   LBi, UBi, LBj, UBj, 1, N(ng), idVvel,          &
     &                   ncNRMid(ng), nrmVid(idVvel,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                   vmask,                                         &
#  endif
     &                   VnormV)
      END IF
!
!  3D norm at RHO-points.
!
      IF (Master) THEN
        WRITE (stdout,30) '3D normalization factors at RHO-points'
        CALL my_flush (stdout)
      END IF
!
!  Check if the decorrelation scales for all the tracers are different.
!  If not, just compute the normalization factors for the first tracer
!  and assign the same value to the rest.  Recall that this computation
!  is very expensive.
!
      Ldiffer=.FALSE.
      DO itrc=2,NT(ng)
        IF ((Hdecay(isTvar(itrc  ),ng).ne.                              &
     &       Hdecay(isTvar(itrc-1),ng)).and.                            &
     &      (Vdecay(isTvar(itrc  ),ng).ne.                              &
     &       Vdecay(isTvar(itrc-1),ng))) THEN
          Ldiffer=.TRUE.
        END IF
      END DO
      IF (.not.Ldiffer) THEN
        Lsame=.TRUE.
        UBt=1
      ELSE
        Lsame=.FALSE.
        UBt=NT(ng)
      END IF
!
      DO j=JR_RANGE
        DO i=IR_RANGE
          val=om_r(i,j)*on_r(i,j)
          DO k=1,N(ng)
            A3davg(i,j,k)=0.0_r8
            A3dsqr(i,j,k)=0.0_r8
            Vscale(i,j,k)=1.0_r8/(val*Hz(i,j,k))
          END DO
        END DO
      END DO
      cff=1.0_r8/REAL(Nrandom,r8)
      DO itrc=1,UBt
        is=isTvar(itrc)
        IF (Cnorm(is)) THEN
          DO iter=1,Nrandom
            CALL white_noise3d (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj, 1, N(ng),           &
     &                          Rscheme(ng), A3d)
            DO k=1,N(ng)
              DO j=JR_RANGE
                DO i=IR_RANGE
                  A3d(i,j,k)=A3d(i,j,k)*Vscale(i,j,k)
                END DO
              END DO
            END DO
            CALL conv_r3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,       &
     &                          LBi, UBi, LBj, UBj, 1, N(ng),           &
     &                          NghostPoints,                           &
     &                          NHsteps(is), NVsteps(is),               &
     &                          DTsizeH(is), DTsizeV(is),               &
     &                          Kh, Kv,                                 &
     &                          pm, pn, pmon_u, pnom_v,                 &
# ifdef MASKING
     &                          rmask, umask, vmask,                    &
# endif
     &                          Hz, z_r,                                &
     &                          A3d)
            DO k=1,N(ng)
              DO j=Jstr,Jend
                DO i=Istr,Iend
                  A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
                  A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
                END DO
              END DO
            END DO
          END DO
          DO k=1,N(ng)
            DO j=Jstr,Jend
              DO i=Istr,Iend
                VnormR(i,j,k,itrc)=1.0_r8/                              &
     &                             SQRT(cff*(A3dsqr(i,j,k)+             &
     &                                       cff*A3davg(i,j,k)*         &
     &                                           A3davg(i,j,k)))
              END DO
            END DO
          END DO
        END IF
      END DO
      IF (Lsame) THEN
        DO itrc=2,NT(ng)
          DO k=1,N(ng)
            DO j=Jstr,Jend
              DO i=Istr,Iend
                VnormR(i,j,k,itrc)=VnormR(i,j,k,1)
              END DO
            END DO
          END DO
        END DO
      END IF
      DO itrc=1,NT(ng)
        is=isTvar(itrc)
        IF (Cnorm(is)) THEN
          CALL bc_r3d_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      VnormR(:,:,:,itrc))
#  ifdef DISTRIBUTE
          CALL mp_exchange3d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,      &
     &                        LBi, UBi, LBj, UBj, 1, N(ng),             &
     &                        NghostPoints, EWperiodic, NSperiodic,     &
     &                        VnormR(:,:,:,itrc))
#  endif
          CALL wrt_norm3d (ng, iNLM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, N(ng), idTvar(itrc),  &
     &                     ncNRMid(ng), nrmTid(itrc,ng), tNRMindx(ng),  &
#  ifdef MASKING
     &                     rmask,                                       &
#  endif
     &                     VnormR(:,:,:,itrc))
        END IF
      END DO
# endif
!
      IF (Master) THEN
        WRITE (stdout,40)
      END IF

 10   FORMAT (/,' RANDOMIZATION - error while writing variable: ',a,/,  &
     &        11x,'into normalization NetCDF file for time record: ',i4)
 20   FORMAT (/,' Error Covariance Factors: Randomization Method',/)
 30   FORMAT (4x,'Computing ',a)
 40   FORMAT (/)

      RETURN
      END SUBROUTINE randomization_tile

!
!***********************************************************************
      SUBROUTINE wrt_norm2d (ng, model, Istr, Iend, Jstr, Jend,         &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       ifield, ncid, ncvarid, tindex,             &
# ifdef MASKING
     &                       Amask,                                     &
# endif
     &                       A)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: ifield, ncid, ncvarid, tindex
!
# ifdef ASSUMED_SHAPE
#  ifdef MASKING
      real(r8), intent(in) :: Amask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: A(LBi:,LBj:)
# else
#  ifdef MASKING
      real(r8), intent(in) :: Amask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
      integer :: gfactor, gtype, status

      integer :: nf_fwrite2d

      real(r8) :: scale
!
!-----------------------------------------------------------------------
!  Write out requested 2D field into normalization NetCDF file. Since
!  the computation of normalization coefficients is a very expensive
!  computation, synchronize NetCDT to disk.
!-----------------------------------------------------------------------
!
      IF (exit_flag.ne.NoError) RETURN
!
!  Set grid type factor to write full (gfactor=1) fields or water
!  points (gfactor=-1) fields only.
!
# if defined WRITE_WATER && defined MASKING
        gfactor=-1
# else
        gfactor=1
# endif
!
!  Write out 2D normalization field.
!
      gtype=gfactor*Iinfo(1,ifield,ng)
      scale=1.0_r8
      status=nf_fwrite2d(ng, model, ncid, ncvarid, tindex, gtype,       &
     &                   LBi, UBi, LBj, UBj, scale,                     &
# ifdef MASKING
     &                   Amask(LBi,LBj),                                &
# endif
     &                   A(LBi,LBj))
      IF (status.ne.nf_noerr) THEN
        IF (Master) THEN
          WRITE (stdout,10) TRIM(Vname(1,ifield)), tindex
        END IF
        exit_flag=3
        ioerror=status
        RETURN
      END IF
!
!  Synchronize normalization NetCDF file to disk to allow other
!  processes to access data immediately after it is written.
!
      IF (OutThread) THEN
        status=nf_sync(ncid)
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,20)
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        WRITE (stdout,30) TRIM(Vname(1,ifield)), tindex
        CALL my_flush (stdout)
      END IF
!
  10  FORMAT (/,' WRT_NORM2D - error while writing variable: ',a,/,11x, &
     &        'into normalization NetCDF file for time record: ',i4)
  20  FORMAT (/,' WRT_NORM2D - unable to synchronize normalization ',   &
     &        'NetCDF file disk.')
  30  FORMAT (7x,'wrote  ',a, t21,'normalization factors into record ', &
     &        i7.7)

      END SUBROUTINE wrt_norm2d

!
!***********************************************************************
      SUBROUTINE wrt_norm3d (ng, model, Istr, Iend, Jstr, Jend,         &
     &                       LBi, UBi, LBj, UBj, LBk, UBk,              &
     &                       ifield, ncid, ncvarid, tindex,             &
# ifdef MASKING
     &                       Amask,                                     &
# endif
     &                       A)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj, LBk, UBk
      integer, intent(in) :: ifield, ncid, ncvarid, tindex
!
# ifdef ASSUMED_SHAPE
#  ifdef MASKING
      real(r8), intent(in) :: Amask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: A(LBi:,LBj:,LBk:)
# else
#  ifdef MASKING
      real(r8), intent(in) :: Amask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: A(LBi:UBi,LBj:UBj,LBk:UBk)
# endif
!
!  Local variable declarations.
!
      integer :: gfactor, gtype, status

      integer :: nf_fwrite3d

      real(r8) :: scale
!
!-----------------------------------------------------------------------
!  Write out requested 3D field into normalization NetCDF file. Since
!  the computation of normalization coefficients is a very expensive
!  computation, synchronize NetCDT to disk.
!-----------------------------------------------------------------------
!
      IF (exit_flag.ne.NoError) RETURN
!
!  Set grid type factor to write full (gfactor=1) fields or water
!  points (gfactor=-1) fields only.
!
# if defined WRITE_WATER && defined MASKING
        gfactor=-1
# else
        gfactor=1
# endif
!
!  Write out 3D normalization field.
!
      gtype=gfactor*Iinfo(1,ifield,ng)
      scale=1.0_r8
      status=nf_fwrite3d(ng, model, ncid, ncvarid, tindex, gtype,       &
     &                   LBi, UBi, LBj, UBj, LBk, UBk, scale,           &
# ifdef MASKING
     &                   Amask(LBi,LBj),                                &
# endif
     &                   A(LBi,LBj,LBk))
      IF (status.ne.nf_noerr) THEN
        IF (Master) THEN
          WRITE (stdout,10) TRIM(Vname(1,ifield)), tindex
        END IF
        exit_flag=3
        ioerror=status
        RETURN
      END IF
!
!  Synchronize normalization NetCDF file to disk to allow other
!  processes to access data immediately after it is written.
!
      IF (OutThread) THEN
        status=nf_sync(ncid)
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,20)
          exit_flag=3
          ioerror=status
          RETURN
        END IF
        WRITE (stdout,30) TRIM(Vname(1,ifield)), tindex
        CALL my_flush (stdout)
      END IF
!
  10  FORMAT (/,' WRT_NORM3D - error while writing variable: ',a,/,11x, &
     &        'into normalization NetCDF file for time record: ',i4)
  20  FORMAT (/,' WRT_NORM3D - unable to synchronize normalization ',   &
     &        'NetCDF file disk.')
  30  FORMAT (7x,'wrote  ',a, t21,'normalization factors into record ', &
     &        i7.7)
      RETURN

      END SUBROUTINE wrt_norm3d

# undef IR_RANGE
# undef JR_RANGE
# undef IU_RANGE
# undef JU_RANGE
# undef IV_RANGE
# undef JV_RANGE

#endif
      END MODULE normalization_mod
