#include "cppdefs.h"

      MODULE ad_bcov_2d_mod

#if defined S4DVAR || defined IS4DVAR || defined W4DVAR
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routines compute the background error covariance for data      !
!  assimilation via a space convolution of the diffusion equation      !
!  for 2D state variables. The diffusion equation is solved using      !
!  an explicit (inefficient) algorithm.                                !
!                                                                      !
!  This routines are used to  compute the isotropic normalization      !
!  factor  Hnorm via a randomization algorithm and to compute the      !
!  the background error covariance.  The norm factora are used to      !
!  ensure that the  diagonal elements of the covariance are equal      !
!  to unity.                                                           !
!                                                                      !
!  For Gaussian (bell-shaped) covariances,  the space convolution      !
!  of the diffusion operator is an efficient way  to estimate the      !
!  finite domain error covariances.  The  number  of  convolution      !
!  steps determined from the values of Hdecay, DTHdif,  and Kdif.      !
!  Here,  Kdif  is a diffusion coefficient,  assumed constant and      !
!  with a value of 1 m2/s. The number of convolution steps Nsteps      !
!  is forced to be an even number because  its associated adjoint      !
!  operator needs to be convoluted for half of the steps only.         !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     Istr       Starting tile index in the I-direction.               !
!     Iend       Ending   tile index in the I-direction.               !
!     Jstr       Starting tile index in the J-direction.               !
!     Jend       Ending   tile index in the J-direction.               !
!     LBi        I-dimension Lower bound.                              !
!     UBi        I-dimension Upper bound.                              !
!     LBj        J-dimension Lower bound.                              !
!     UBj        J-dimension Upper bound.                              !
!     Nghost     Number of ghost points.                               !
!     DTHdif     Time-step for horizontal diffusion convolution (s).   !
!     Hdecay     Horizontal Gaussian decorrelation scale (m).          !
!     Hnorm      Horizontal normalization factors.                     !
!     ad_A       2D adjoint state variable to diffuse.                 !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     ad_A       Diffused 2D adjoint state variable.                   !      
!                                                                      !
!  Routines:                                                           !
!                                                                      !
!    ad_bcov_r2d_tile  Background error covariance at RHO-points       !
!    ad_bcov_u2d_tile  Background error covariance at U-points         !
!    ad_bcov_v2d_tile  Background error covariance at V-points         !
!                                                                      !
!=======================================================================
!
      implicit none

      PUBLIC

      CONTAINS
!
!***********************************************************************
      SUBROUTINE ad_bcov_r2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_u, pnom_v,              &
# ifdef MASKING
     &                             umask, vmask,                        &
# endif
     &                             ad_A)
!***********************************************************************
!
      USE mod_param
!
      USE ad_bc_2d_mod, ONLY: ad_bc_r2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : ad_mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_u(LBi:,LBj:)
      real(r8), intent(in) :: pnom_v(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: umask(LBi:,LBj:)
      real(r8), intent(in) :: vmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: ad_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_v(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: umask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: vmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: ad_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: adfac, cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: ad_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FX

# include "set_bounds.h"
!
      ILB=LBOUND(ad_Awrk,DIM=1)
      IUB=UBOUND(ad_Awrk,DIM=1)
      JLB=LBOUND(ad_Awrk,DIM=2)
      JUB=UBOUND(ad_Awrk,DIM=2)!
!
!-----------------------------------------------------------------------
!  Initialize adjoint private variables.
!-----------------------------------------------------------------------
!
      ad_Awrk(ILB:IUB,JLB:JUB,1:2)=0.0_r8

      ad_FE(ILB:IUB,JLB:JUB)=0.0_r8
      ad_FX(ILB:IUB,JLB:JUB)=0.0_r8
!
!-----------------------------------------------------------------------
!  Adjoint space convolution of the diffusion equation for a 2D state
!  variable at RHO-points.
!-----------------------------------------------------------------------
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))/2
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
      Nold=1
      Nnew=2
!
!  Load adjoint solution.
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
!>        tl_A(i,j)=tl_Awrk(i,j,Nold)
!>
          ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_A(i,j)
          ad_A(i,j)=0.0_r8          
        END DO
      END DO
!
!  Integrate adjoint diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Update integration indices.
!
        Nsav=Nnew
        Nnew=Nold
        Nold=Nsav
!
!  Apply adjoint boundary conditions. If applicable, exchange boundary
!  data.
!
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      tl_Awrk(:,:,Nnew))
!>
        CALL ad_mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,    &
     &                         ILB, IUB, JLB, JUB,                      &
     &                         Nghost, EWperiodic, NSperiodic,          &
     &                         ad_Awrk(:,:,Nnew))
# endif
!>      CALL bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    tl_Awrk(:,:,Nnew))
!>
        CALL ad_bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                &
     &                       ILB, IUB, JLB, JUB,                        &
     &                       ad_Awrk(:,:,Nnew))
!
!  Time-step adjoint harmonic diffusion operator.
!
        DO j=Jstr,Jend
          DO i=Istr,Iend
!>          tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
!>   &                        DTHdif*pm(i,j)*pn(i,j)*                   &
!>   &                        (tl_FX(i+1,j)-tl_FX(i,j)+                 &
!>   &                         tl_FE(i,j+1)-tl_FE(i,j))
!>
            adfac=DTHdif*pm(i,j)*pn(i,j)*ad_Awrk(i,j,Nnew)
            ad_FE(i,j  )=ad_FE(i,j  )-adfac
            ad_FE(i,j+1)=ad_FE(i,j+1)+adfac
            ad_FX(i  ,j)=ad_FX(i  ,j)-adfac
            ad_FX(i+1,j)=ad_FX(i+1,j)+adfac
            ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_Awrk(i,j,Nnew)
            ad_Awrk(i,j,Nnew)=0.0_r8
          END DO
        END DO
!
!  Compute XI- and ETA-components of the adjoint diffusive flux.
!
        DO j=Jstr,Jend+1
          DO i=Istr,Iend
# ifdef MASKING
!>          tl_FE(i,j)=tl_FE(i,j)*vmask(i,j)
!>
            ad_FE(i,j)=ad_FE(i,j)*vmask(i,j)
# endif
!>          tl_FE(i,j)=pnom_v(i,j)*                                     &
!>   &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i,j-1,Nold))
!>
            adfac=pnom_v(i,j)*ad_FE(i,j)
            ad_Awrk(i,j-1,Nold)=ad_Awrk(i,j-1,Nold)-adfac
            ad_Awrk(i,j  ,Nold)=ad_Awrk(i,j  ,Nold)+adfac
            ad_FE(i,j)=0.0_r8
          END DO
        END DO
        DO j=Jstr,Jend
          DO i=Istr,Iend+1
# ifdef MASKING
!>          tl_FX(i,j)=tl_FX(i,j)*umask(i,j)
!>
            ad_FX(i,j)=ad_FX(i,j)*umask(i,j)
# endif
!>          tl_FX(i,j)=pmon_u(i,j)*                                     &
!>   &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i-1,j,Nold))
!>
            adfac=pmon_u(i,j)*ad_FX(i,j)
            ad_Awrk(i-1,j,Nold)=ad_Awrk(i-1,j,Nold)-adfac
            ad_Awrk(i  ,j,Nold)=ad_Awrk(i  ,j,Nold)+adfac
            ad_FX(i,j)=0.0_r8
          END DO
        END DO

      END DO
!
!  Set adjoint initial conditions.
!
      DO j=Jstr-1,Jend+1
        DO i=IstrU-1,Iend+1
!>        tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
!>
          ad_A(i,j)=ad_A(i,j)+ad_Awrk(i,j,Nold)*Hnorm(i,j)
          ad_Awrk(i,j,Nold)=0.0_r8
        END DO
      END DO

      RETURN
      END SUBROUTINE ad_bcov_r2d_tile
!
!***********************************************************************
      SUBROUTINE ad_bcov_u2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_r, pnom_p,              &
# ifdef MASKING
     &                             pmask,                               &
# endif
     &                             ad_A)
!***********************************************************************
!
      USE mod_param
!
      USE ad_bc_2d_mod, ONLY: ad_bc_u2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : ad_mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_r(LBi:,LBj:)
      real(r8), intent(in) :: pnom_p(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: ad_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_p(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: ad_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: adfac, cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: ad_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FX

# include "set_bounds.h"
!
      ILB=LBOUND(ad_Awrk,DIM=1)
      IUB=UBOUND(ad_Awrk,DIM=1)
      JLB=LBOUND(ad_Awrk,DIM=2)
      JUB=UBOUND(ad_Awrk,DIM=2)
!
!-----------------------------------------------------------------------
!  Initialize adjoint private variables.
!-----------------------------------------------------------------------
!
      ad_Awrk(ILB:IUB,JLB:JUB,1:2)=0.0_r8

      ad_FE(ILB:IUB,JLB:JUB)=0.0_r8
      ad_FX(ILB:IUB,JLB:JUB)=0.0_r8
!
!-----------------------------------------------------------------------
!  Adjoint space convolution of the diffusion equation for a 2D state
!  variable at U-points
!-----------------------------------------------------------------------
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))/2
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
      Nold=1
      Nnew=2
!
!  Load adjoint solution.
!
      DO j=JstrR,JendR
        DO i=IstrU-1,IendR
!>        tl_A(i,j)=tl_Awrk(i,j,Nold)
!>
          ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_A(i,j)
          ad_A(i,j)=0.0_r8          
        END DO
      END DO
!
!  Integrate diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Update integration indices.
!
        Nsav=Nnew
        Nnew=Nold
        Nold=Nsav
!
!  Apply adjoint boundary conditions. If applicable, exchange boundary
!  data.
!
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      tl_Awrk(:,:,Nnew))
!>
        CALL ad_mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,    &
     &                         ILB, IUB, JLB, JUB,                      &
     &                         Nghost, EWperiodic, NSperiodic,          &
     &                         ad_Awrk(:,:,Nnew))
# endif
!>      CALL bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    tl_Awrk(:,:,Nnew))
!>
        CALL ad_bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                &
     &                       ILB, IUB, JLB, JUB,                        &
     &                       ad_Awrk(:,:,Nnew))

!
!  Time-step adjoint harmonic diffusion operator.
!
        cff=DTHdif*0.25_r8
        DO j=Jstr,Jend
          DO i=IstrU,Iend
!>          tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
!>   &                        cff*(pm(i-1,j)+pm(i,j))*                  &
!>   &                            (pn(i-1,j)+pn(i,j))*                  &
!>   &                        (tl_FX(i,j)-tl_FX(i-1,j)+                 &
!>   &                         tl_FE(i,j+1)-tl_FE(i,j))
!>
            adfac=cff*(pm(i-1,j)+pm(i,j))*                              &
     &                (pn(i-1,j)+pn(i,j))*ad_Awrk(i,j,Nnew)
            ad_FE(i,j  )=ad_FE(i,j  )-adfac
            ad_FE(i,j+1)=ad_FE(i,j+1)+adfac
            ad_FX(i-1,j)=ad_FX(i-1,j)-adfac
            ad_FX(i  ,j)=ad_FX(i  ,j)+adfac
            ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_Awrk(i,j,Nnew)
            ad_Awrk(i,j,Nnew)=0.0_r8
          END DO
        END DO
!
!  Compute XI- and ETA-components of the adjoint diffusive flux.
!
        DO j=Jstr,Jend+1
          DO i=IstrU,Iend
# ifdef MASKING
!>          tl_FE(i,j)=tl_FE(i,j)*pmask(i,j)
!>
            ad_FE(i,j)=ad_FE(i,j)*pmask(i,j)
# endif
!>          tl_FE(i,j)=pnom_p(i,j)*                                     &
!>   &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i,j-1,Nold))
!>
            adfac=pnom_p(i,j)*ad_FE(i,j)
            ad_Awrk(i,j-1,Nold)=ad_Awrk(i,j-1,Nold)-adfac
            ad_Awrk(i,j  ,Nold)=ad_Awrk(i,j  ,Nold)+adfac
            ad_FE(i,j)=0.0_r8
          END DO
        END DO
        DO j=Jstr,Jend
          DO i=IstrU-1,Iend
!>          tl_FX(i,j)=pmon_r(i,j)*                                     &
!>   &                 (tl_Awrk(i+1,j,Nold)-tl_Awrk(i,j,Nold))
!>
            adfac=pmon_r(i,j)*ad_FX(i,j)
            ad_Awrk(i  ,j,Nold)=ad_Awrk(i  ,j,Nold)-adfac
            ad_Awrk(i+1,j,Nold)=ad_Awrk(i+1,j,Nold)+adfac
            ad_FX(i,j)=0.0_r8
          END DO
        END DO

      END DO
!
!  Set adjoint initial conditions.
!
      DO j=Jstr-1,Jend+1
        DO i=Istr-1,Iend+1
!>        tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
!>
          ad_A(i,j)=ad_A(i,j)+ad_Awrk(i,j,Nold)*Hnorm(i,j)
          ad_Awrk(i,j,Nold)=0.0_r8
        END DO
      END DO

      RETURN
      END SUBROUTINE ad_bcov_u2d_tile
!
!***********************************************************************
      SUBROUTINE ad_bcov_v2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_p, pnom_r,              &
# ifdef MASKING
     &                             pmask,                               &
# endif
     &                             ad_A)
!***********************************************************************
!
      USE mod_param
!
      USE ad_bc_2d_mod, ONLY: ad_bc_v2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : ad_mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_p(LBi:,LBj:)
      real(r8), intent(in) :: pnom_r(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: ad_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_r(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in)  :: pmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: ad_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
! 
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: adfac, cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: ad_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ad_FX

# include "set_bounds.h"
!
      ILB=LBOUND(ad_Awrk,DIM=1)
      IUB=UBOUND(ad_Awrk,DIM=1)
      JLB=LBOUND(ad_Awrk,DIM=2)
      JUB=UBOUND(ad_Awrk,DIM=2)
!
!-----------------------------------------------------------------------
!  Initialize adjoint private variables.
!-----------------------------------------------------------------------
!
      ad_Awrk(ILB:IUB,JLB:JUB,1:2)=0.0_r8

      ad_FE(ILB:IUB,JLB:JUB)=0.0_r8
      ad_FX(ILB:IUB,JLB:JUB)=0.0_r8
!
!-----------------------------------------------------------------------
!  Space convolution of the diffusion equation for a 2D state variable
!  at V-points
!-----------------------------------------------------------------------
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))/2
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
      Nold=1
      Nnew=2
!
!  Load adjoint solution.
!
      DO j=JstrV-1,JendR
        DO i=IstrR,IendR
!>        tl_A(i,j)=tl_Awrk(i,j,Nold)
!>
          ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_A(i,j)
          ad_A(i,j)=0.0_r8
        END DO
      END DO
!
!  Integrate diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Update integration indices.
!
        Nsav=Nnew
        Nnew=Nold
        Nold=Nsav
!
!  Apply adjoint boundary conditions. If applicable, exchange boundary
!  data.
!
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      tl_Awrk(:,:,Nnew))
!>
        CALL ad_mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,    &
     &                         ILB, IUB, JLB, JUB,                      &
     &                         Nghost, EWperiodic, NSperiodic,          &
     &                         ad_Awrk(:,:,Nnew))
# endif
!>      CALL bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    tl_Awrk(:,:,Nnew))
!>
        CALL ad_bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                &
     &                       ILB, IUB, JLB, JUB,                        &
     &                       ad_Awrk(:,:,Nnew))
!
!  Time-step harmonic adjoint diffusion operator.
!
        cff=DTHdif*0.25_r8
        DO j=JstrV,Jend
          DO i=Istr,Iend
!>          tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
!>   &                        cff*(pm(i,j-1)+pm(i,j))*                  &
!>   &                            (pn(i,j-1)+pn(i,j))*                  &
!>   &                        (tl_FX(i+1,j)-tl_FX(i,j)+                 &
!>   &                         tl_FE(i,j)-tl_FE(i,j-1))
!>
            adfac=cff*(pm(i,j-1)+pm(i,j))*                              &
     &                (pn(i,j-1)+pn(i,j))*ad_Awrk(i,j,Nnew)
            ad_FE(i,j-1)=ad_FE(i,j-1)-adfac
            ad_FE(i,j  )=ad_FE(i,j  )+adfac            
            ad_FX(i  ,j)=ad_FX(i  ,j)-adfac
            ad_FX(i+1,j)=ad_FX(i+1,j)+adfac
            ad_Awrk(i,j,Nold)=ad_Awrk(i,j,Nold)+ad_Awrk(i,j,Nnew)
            ad_Awrk(i,j,Nnew)=0.0_r8
          END DO
        END DO
!
!  Compute XI- and ETA-components of the adjoint diffusive flux.
!
        DO j=JstrV-1,Jend
          DO i=Istr,Iend
!>          tl_FE(i,j)=pnom_r(i,j)*                                     &
!>   &                 (tl_Awrk(i,j+1,Nold)-tl_Awrk(i,j,Nold))
!>
            adfac=pnom_r(i,j)*ad_FE(i,j)
            ad_Awrk(i,j  ,Nold)=ad_Awrk(i,j  ,Nold)-adfac
            ad_Awrk(i,j+1,Nold)=ad_Awrk(i,j+1,Nold)+adfac
            ad_FE(i,j)=0.0_r8
          END DO
        END DO
        DO j=JstrV,Jend
          DO i=Istr,Iend+1
# ifdef MASKING
!>          tl_FX(i,j)=tl_FX(i,j)*pmask(i,j)
!>
            ad_FX(i,j)=ad_FX(i,j)*pmask(i,j)
# endif
!>          tl_FX(i,j)=pmon_p(i,j)*                                     &
!>   &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i-1,j,Nold))
!>
            adfac=pmon_p(i,j)*ad_FX(i,j)
            ad_Awrk(i-1,j,Nold)=ad_Awrk(i-1,j,Nold)-adfac
            ad_Awrk(i  ,j,Nold)=ad_Awrk(i  ,j,Nold)+adfac
            ad_FX(i,j)=0.0_r8
          END DO
        END DO

      END DO
!
!  Set adjoint initial conditions.
!
      DO j=JstrV-1,Jend+1
        DO i=Istr-1,Iend+1
!>        tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
!>
          ad_A(i,j)=ad_A(i,j)+ad_Awrk(i,j,Nold)*Hnorm(i,j)
          ad_Awrk(i,j,Nold)=0.0_r8
        END DO
      END DO

      RETURN
      END SUBROUTINE ad_bcov_v2d_tile
#endif
      END MODULE ad_bcov_2d_mod
