#include "cppdefs.h"
      MODULE ad_gather_mod
#ifdef ADJOINT
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine  gathers adjoint data from points outside the tile     !
!  for output purposes.  In parallel applications, it collects the     !
!  data from ghost points and add it to the  appropriate  interior     !
!  point. The adjusted adjoint solution is stored in the nonlinear     !
!  (background) state arrays. They are used as scratch arrays here.    !
!                                                                      !
!  If last adjoint step,  gather the state  contributions from the     !
!  initialization step before writing adjoint output.                  !
!                                                                      !
!  The 3D initialization scheme for the model is as follows:           !
!                                                                      !
!  (1) Set initial conditions to the following state variable at       !
!      time level 1:                                                   !
!                                                                      !
!        zeta(i,j,1)                                                   !
!           u(i,j,k,1)                                                 !
!           v(i,j,k,1)                                                 !
!           t(i,j,k,1,:)                                               !
!                                                                      !
!  (2) Impose boundary conditions of zeta, u, v, t.                    !
!  (3) Since the initial free-surface is known, set time-averaged      !
!      free-surface (Zt_avg1) to that value and compute level          !
!      thicknesses (Hz) with a call to "set_depth".                    !
!  (4) Compute the initial conditions for barotropic momentum,         !
!      ubar(i,j,1) and vbar(i,j,1), using the vertically integrated    !
!      initial u and v.                                                !
!  (5) Note that in "rhs3d", zeta(i,j,1) and zeta(i,j,2) are           !
!      overwritten by "Zt_avg1".                                       !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC ad_gather

      CONTAINS
!
!***********************************************************************
      SUBROUTINE ad_gather (ng, tile)
!***********************************************************************
!
      USE mod_param
# ifdef SOLVE3D
      USE mod_coupling
# endif
      USE mod_grid
      USE mod_ocean
      USE mod_stepping
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
      CALL ad_gather_tile (ng, Istr, Iend, Jstr, Jend,                  &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     kstp(ng), krhs(ng), knew(ng),                &
# ifdef SOLVE3D
     &                     nstp(ng), nnew(ng),                          &
     &                     GRID(ng) % h,                                &
     &                     GRID(ng) % ad_h,                             &
#  ifdef ICESHELF
     &                     GRID(ng) % zice,                             &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                     OCEAN(ng) % ad_bed,                          &
     &                     GRID(ng) % ad_bed_thick0,                    &
#  endif
     &                     COUPLING(ng) % Zt_avg1,                      &
     &                     COUPLING(ng) % ad_Zt_avg1,                   &
     &                     GRID(ng) % ad_Hz,                            &
     &                     GRID(ng) % ad_z_r,                           &
     &                     GRID(ng) % ad_z_w,                           &
     &                     OCEAN(ng) % ad_rho,                          &
     &                     OCEAN(ng) % rho,                             &
     &                     OCEAN(ng) % ad_t,                            &
     &                     OCEAN(ng) % t,                               &
     &                     OCEAN(ng) % ad_u,                            &
     &                     OCEAN(ng) % u,                               &
     &                     OCEAN(ng) % ad_v,                            &
     &                     OCEAN(ng) % v,                               &
#  if defined AD_SENSITIVITY || defined SO_SEMI
     &                     OCEAN(ng) % ad_ubar_sol,                     &
     &                     OCEAN(ng) % ad_vbar_sol,                     &
     &                     OCEAN(ng) % ad_zeta_sol,                     &
#  endif
# endif
     &                     OCEAN(ng) % ad_ubar,                         &
     &                     OCEAN(ng) % ubar,                            &
     &                     OCEAN(ng) % ad_vbar,                         &
     &                     OCEAN(ng) % vbar,                            &
     &                     OCEAN(ng) % ad_zeta,                         &
     &                     OCEAN(ng) % zeta)
      RETURN
      END SUBROUTINE ad_gather
!
!***********************************************************************
      SUBROUTINE ad_gather_tile (ng, Istr, Iend, Jstr, Jend,            &
     &                           LBi, UBi, LBj, UBj,                    &
     &                           kstp, krhs, knew,                      &
# ifdef SOLVE3D
     &                           nstp, nnew,                            &
     &                           h, ad_h,                               &      
#  ifdef ICESHELF
     &                           zice,                                  &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                           ad_bed, ad_bed_thick0,                 &
#  endif
     &                           Zt_avg1, ad_Zt_avg1,                   &
     &                           ad_Hz, ad_z_r, ad_z_w,                 &
     &                           ad_rho, rho,                           &
     &                           ad_t, t,                               &
     &                           ad_u, u,                               &
     &                           ad_v, v,                               &
#  if defined AD_SENSITIVITY || defined SO_SEMI
     &                           ad_ubar_sol, ad_vbar_sol,              &
     &                           ad_zeta_sol,                           &
#  endif
# endif
     &                           ad_ubar, ubar,                         &
     &                           ad_vbar, vbar,                         &
     &                           ad_zeta, zeta)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
# if defined EW_PERIODIC || defined NS_PERIODIC
      USE ad_exchange_2d_mod
#  ifdef SOLVE3D
      USE ad_exchange_2d_mod
#  endif
# endif
# ifdef SOLVE3D
      USE ad_set_depth_mod, ONLY : ad_set_depth_tile
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: kstp, krhs, knew
# ifdef SOLVE3D
      integer, intent(in) :: nstp, nnew
# endif
!
# ifdef ASSUMED_SHAPE
#  ifdef SOLVE3D
      real(r8), intent(in) :: ad_rho(LBi:,LBj:,:)
      real(r8), intent(in) :: ad_t(LBi:,LBj:,:,:,:)
      real(r8), intent(in) :: ad_u(LBi:,LBj:,:,:)
      real(r8), intent(in) :: ad_v(LBi:,LBj:,:,:)
#   if defined AD_SENSITIVITY || defined SO_SEMI
      real(r8), intent(in) :: ad_ubar_sol(LBi:,LBj:)
      real(r8), intent(in) :: ad_vbar_sol(LBi:,LBj:)
      real(r8), intent(in) :: ad_zeta_sol(LBi:,LBj:)
#   endif
#  endif
      real(r8), intent(in) :: ad_ubar(LBi:,LBj:,:)
      real(r8), intent(in) :: ad_vbar(LBi:,LBj:,:)
#  ifdef SOLVE3D
      real(r8), intent(in) :: h(LBi:,LBj:)
      real(r8), intent(in) :: Zt_avg1(LBi:,LBj:)
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:,LBj:)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
       real(r8), intent(in) :: ad_bed(LBi:,LBj:,:,:)
       real(r8), intent(inout) :: ad_bed_thick0(LBi:,LBj:)
#   endif
      real(r8), intent(inout) :: ad_h(LBi:,LBj:)
      real(r8), intent(inout) :: ad_Hz(LBi:,LBj:,:)
      real(r8), intent(inout) :: ad_Zt_avg1(LBi:,LBj:)
      real(r8), intent(inout) :: ad_z_r(LBi:,LBj:,:)
      real(r8), intent(inout) :: ad_z_w(LBi:,LBj:,0:)
#  endif
      real(r8), intent(inout) :: ad_zeta(LBi:,LBj:,:)
      real(r8), intent(inout) :: zeta(LBi:,LBj:,:)
#  ifdef SOLVE3D
      real(r8), intent(out) :: rho(LBi:,LBj:,:)
      real(r8), intent(out) :: t(LBi:,LBj:,:,:,:)
      real(r8), intent(out) :: u(LBi:,LBj:,:,:)
      real(r8), intent(out) :: v(LBi:,LBj:,:,:)
#  endif
      real(r8), intent(out) :: ubar(LBi:,LBj:,:)
      real(r8), intent(out) :: vbar(LBi:,LBj:,:)

# else

#  ifdef SOLVE3D
      real(r8), intent(in) :: ad_rho(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(in) :: ad_t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
      real(r8), intent(in) :: ad_u(LBi:UBi,LBj:UBj,N(ng),2)
      real(r8), intent(in) :: ad_v(LBi:UBi,LBj:UBj,N(ng),2)
#   if defined AD_SENSITIVITY || defined SO_SEMI
      real(r8), intent(in) :: ad_ubar_sol(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: ad_vbar_sol(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: ad_zeta_sol(LBi:UBi,LBj:UBj)
#   endif
#  endif
      real(r8), intent(in) :: ad_ubar(LBi:UBi,LBj:UBj,3)
      real(r8), intent(in) :: ad_vbar(LBi:UBi,LBj:UBj,3)
#  ifdef SOLVE3D
      real(r8), intent(in) :: h(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: Zt_avg1(LBi:UBi,LBj:UBj)
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:UBi,LBj:UBj)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: ad_bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
      real(r8), intent(inout) :: ad_bed_thick0(LBi:UBi,LBj:UBj)
#   endif
      real(r8), intent(inout) :: ad_h(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: ad_Hz(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(inout) :: ad_Zt_avg1(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: ad_z_r(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(inout) :: ad_z_w(LBi:UBi,LBj:UBj,0:N(ng))
#  endif
      real(r8), intent(inout) :: ad_zeta(LBi:UBi,LBj:UBj,3)
      real(r8), intent(inout) :: zeta(LBi:UBi,LBj:UBj,3)
#  ifdef SOLVE3D
      real(r8), intent(out) :: rho(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
      real(r8), intent(out) :: u(LBi:UBi,LBj:UBj,N(ng),2)
      real(r8), intent(out) :: v(LBi:UBi,LBj:UBj,N(ng),2)
#  endif
      real(r8), intent(out) :: ubar(LBi:UBi,LBj:UBj,3)
      real(r8), intent(out) :: vbar(LBi:UBi,LBj:UBj,3)
# endif
!
!  Local variable declarations.
!
      integer, parameter :: Lout = 1

      integer :: i, j
# ifdef SOLVE3D
      integer :: itrc, k
# endif

# ifndef SOLVE3D
!
!-----------------------------------------------------------------------
!  Gather ghost point information, if applicable, and load adjusted
!  adjoint solution into nonlinear (backgound) state arrays.
!-----------------------------------------------------------------------
!
!  Free-surface.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
          zeta(i,j,Lout)=ad_zeta(i,j,kstp)+                             &
     &                   ad_zeta(i,j,krhs)
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     zeta(LBi,LBj,Lout))
#  endif
!
!  2D U-momentum.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
          ubar(i,j,Lout)=ad_ubar(i,j,kstp)+                             &
     &                   ad_ubar(i,j,krhs)
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     ubar(LBi,LBj,Lout))
#  endif
!
!  2D V-momentum.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
          vbar(i,j,Lout)=ad_vbar(i,j,kstp)+                             &
     &                   ad_vbar(i,j,krhs)
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     vbar(LBi,LBj,Lout))
#  endif
# endif
# ifdef SOLVE3D
!
!-----------------------------------------------------------------------
!  If last adjoint step, collect data for adjoint free surface,
!  ad_zeta(:,:,knew), from time averaged array, ad_Zt_avg1(:,:).
!  Then, apply adjoint lateral boundary conditions to all state
!  field associated with initialization.
!-----------------------------------------------------------------------
!
      IF (iic(ng).eq.ntend) THEN
!
!  Compute adjoint model depths and thicknesses.
!
        CALL ad_set_depth_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ad_zeta,                                &
     &                          h, ad_h,                                &
#  ifdef ICESHELF
     &                          zice,                                   &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                          ad_bed, ad_bed_thick0,                  &
#  endif
     &                          Zt_avg1, ad_Zt_avg1,                    &
     &                          ad_Hz, ad_z_r, ad_z_w)

#  if defined EW_PERIODIC || defined NS_PERIODIC
!
!  Gather adjoint periodic boundary contributions for all the state
!  fields associated with the initialization.
!
        CALL ad_exchange_r2d_tile (ng, iADM, Istr, Iend, Jstr, Jend,    &
     &                             LBi, UBi, LBj, UBj,                  &
     &                             NghostPoints,                        &
     &                             ad_zeta(:,:,kstp))
        CALL ad_exchange_r2d_tile (ng, iADM, Istr, Iend, Jstr, Jend,    &
     &                             LBi, UBi, LBj, UBj,                  &
     &                             NghostPoints,                        &
     &                             ad_zeta(:,:,knew))
        CALL ad_exchange_v3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,    &
     &                             LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                             NghostPoints,                        &
     &                             ad_v(:,:,:,nstp))
        CALL ad_exchange_u3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,    &
     &                             LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                             NghostPoints,                        &
     &                             ad_u(:,:,:,nstp))
        DO itrc=1,NT(ng)
          CALL ad_exchange_r3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,  &
     &                               LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                               NghostPoints,                      &
     &                               ad_t(:,:,:,nstp,itrc))
        END DO
#  endif
#  ifdef APPLY_BC
!
!  Gather adjoint lateral boundary contributions for the 3D state
!  fields associated with the initialization.
!
!   HGA:  Check the order here for nstp and nnew.  The F77 code was
!         using only nstp.
!
        CALL ad_v3dbc_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj, N(ng),                  &
     &                      nstp, nnew,                                 &
     &                      ad_v)
        CALL ad_u3dbc_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj, N(ng),                  &
     &                      nstp, nnew,                                 &
     &                      ad_u)
        DO itrc=1,NT(ng)
          CALL ad_t3dbc_tile (ng, Istr, Iend, Jstr, Jend, itrc,         &
     &                        LBi, UBi, LBj, UBj, N(ng), NT(ng),        &
     &                        nstp, nnew,                               &
     &                        ad_t)
        END DO
!
!  Gather adjoint lateral boundary contributions for the 2D state
!  fields associated with the initialisation.
!
!  HGA:  Check the index to update here. The F77 code update first
!        kstp and then knew.

        CALL ad_zetabc_tile (ng, Istr, Iend, Jstr, Jend,                &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       krhs, kstp, kstp,                          &
     &                       zeta, ad_zeta)
        CALL ad_zetabc_tile (ng, Istr, Iend, Jstr, Jend,                &
     &                       LBi, UBi, LBj, UBj,                        &
     &                       krhs, kstp, knew,                          &
     &                       zeta, ad_zeta)
#  endif
      END IF
!
!-----------------------------------------------------------------------
!  Gather ghost point information, if applicable, and load adjusted
!  adjoint solution into nonlinear (backgound) state arrays.
!-----------------------------------------------------------------------
!
!  Free-surface.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
#  if defined AD_SENSITIVITY || defined SO_SEMI
          zeta(i,j,Lout)=ad_zeta_sol(i,j)
#  else
          zeta(i,j,Lout)=ad_zeta(i,j,kstp)+                             &
     &                   ad_zeta(i,j,knew)
#  endif
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     zeta(LBi,LBj,Lout))
#  endif
!
!  2D U-momentum.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
#  if defined AD_SENSITIVITY || defined SO_SEMI
          ubar(i,j,Lout)=ad_ubar_sol(i,j)
#  else
          ubar(i,j,Lout)=ad_ubar(i,j,kstp)+                             &
     &                   ad_ubar(i,j,knew)
#  endif
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     ubar(LBi,LBj,Lout))
#  endif
!
!  2D V-momentum.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
#  if defined AD_SENSITIVITY || defined SO_SEMI
          vbar(i,j,Lout)=ad_vbar_sol(i,j)
#  else
          vbar(i,j,Lout)=ad_vbar(i,j,kstp)+                             &
     &                   ad_vbar(i,j,knew)
#  endif
        END DO
      END DO          
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, 1,                    &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     vbar(LBi,LBj,Lout))
#  endif
!
!  3D U-momentum.
!
      DO k=1,N(ng)
        DO j=LBj,UBj
          DO i=LBi,UBi
            u(i,j,k,Lout)=ad_u(i,j,k,nstp)
          END DO
        END DO
      END DO
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, N(ng),                &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     u(LBi,LBj,1,Lout))
#  endif
!
!  3D V-momentum.
!
      DO k=1,N(ng)
        DO j=LBj,UBj
          DO i=LBi,UBi
            v(i,j,k,Lout)=ad_v(i,j,k,nstp)
          END DO
        END DO
      END DO
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, N(ng),                &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     v(LBi,LBj,1,Lout))
#  endif
!
!  Tracers.
!
      DO itrc=1,NT(ng)
        DO k=1,N(ng)
          DO j=LBj,UBj
            DO i=LBi,UBi
              t(i,j,k,Lout,itrc)=ad_t(i,j,k,nstp,itrc)
            END DO
          END DO
        END DO
#  ifdef DISTRIBUTE
        CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,          &
     &                       LBi, UBi, LBj, UBj, 1, N(ng),              &
     &                       NghostPoints, .FALSE., .FALSE.,            &
     &                       t(LBi,LBj,1,Lout,itrc))
#  endif
      END DO
!
!  Density Anomaly.
!
      DO k=1,N(ng)
        DO j=LBj,UBj
          DO i=LBi,UBi
            rho(i,j,k)=ad_rho(i,j,k)
          END DO
        END DO
      END DO
#  ifdef DISTRIBUTE
      CALL ad_mp_exchange (ng, iADM, Istr, Iend, Jstr, Jend,            &
     &                     LBi, UBi, LBj, UBj, 1, N(ng),                &
     &                     NghostPoints, .FALSE., .FALSE.,              &
     &                     rho(LBi,LBj,1))
#  endif
# endif
# ifdef APPLY_BC
!
!  Apply boundary conditions.
!
          CALL ad_u2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, kstp,                         &
     &                        ad_ubar, ad_vbar, ad_zeta,                & !<=
     &                        ubar, vbar, zeta)
          CALL ad_v2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, kstp,                         &
     &                        ad_ubar, ad_vbar, ad_zeta,                & !<=
     &                        ubar, vbar, zeta)
#  ifdef SOLVE3D
          CALL ad_u2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, knew,                         &
     &                        ad_ubar, ad_vbar, ad_zeta,                & !<=
     &                        ubar, vbar, zeta)
          CALL ad_v2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, knew,                         &
     &                        ad_ubar, ad_vbar, ad_zeta,                & !<=
     &                        ubar, vbar, zeta)
#  else
          CALL ad_u2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, krhs,                         &
     &                        ubar, vbar, zeta,                         &
     &                        ad_ubar, ad_vbar, ad_zeta)
          CALL ad_v2dbc_tile (ng, Istr, Iend, Jstr, Jend,               &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        krhs, kstp, krhs,                         &
     &                        ubar, vbar, zeta,                         &
     &                        ad_ubar, ad_vbar, ad_zeta)
#  endif
# endif

      RETURN
      END SUBROUTINE ad_gather_tile
#endif
      END MODULE ad_gather_mod
