#include "cppdefs.h"
      SUBROUTINE ad_mp_exchange (ng, model, Istr, Iend, Jstr, Jend,     &
     &                           LBi, UBi, LBj, UBj, LBk, UBk,          &
     &                           Nghost, EWperiodic, NSperiodic, A)
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  Adjoint Exchange Communications:                                    !
!                                                                      !
!  This routine updates the tile overlap halo of a  2D or  3D real     !
!  array.  It exchanges the specified number of "ghost-points". In     !
!  order to minimize the number send and receive calls, the ghost-     !
!  points are included in the buffers. Therefore, the order of the     !
!  pack/send/receive/unpack is crucial.                                !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     Istr       Starting tile index in the I-direction.               !
!     Iend       Ending   tile index in the I-direction.               !
!     Jstr       Starting tile index in the J-direction.               !
!     Jend       Ending   tile index in the J-direction.               !
!     LBi        I-dimension Lower bound.                              !
!     UBi        I-dimension Upper bound.                              !
!     LBj        J-dimension Lower bound.                              !
!     UBj        J-dimension Upper bound.                              !
!     LBk        K-dimension Lower bound, if any. Otherwise, a value   !
!                  of one is expected.                                 !
!     LBk        K-dimension Upper bound, if any. Otherwise, a value   !
!                  of one is expected.                                 !
!     UBk        K-dimension Upper bound.                              !
!     Nghost     Number of ghost-points in the halo region.            !
!     EWperiodic Switch indicating EW periodicity exchanges.           !
!     NSperiodic Switch indicating NS periodicity exchanges.           !
!     A          Tile array (2D or 3D) to process.                     !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     A          Updated tile array (2D or 3D).                        !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits 
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      logical, intent(in) :: EWperiodic, NSperiodic

      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj, LBk, UBk, Nghost

      real(r8), intent(inout) :: A(LBi:UBi,LBj:UBj,LBk:UBk)
!
!  Local variable declarations.
!
      logical :: Wexchange, Sexchange, Eexchange, Nexchange

      integer :: i, ikS, ikN, Imin, Imax, Ilen, IKlen, IKsizeS, IKsizeN
      integer :: j, jkW, jkE, Jmin, Jmax, Jlen, JKlen, JKsizeW, JKsizeE
      integer :: k, kc,  m, mc, Klen, Lstr, MyRankI, MyRankJ, rank
      integer :: Wtile, GsendW, GrecvW, Wtag, Werror, Wrequest
      integer :: Stile, GsendS, GrecvS, Stag, Serror, Srequest
      integer :: Etile, GsendE, GrecvE, Etag, Eerror, Erequest
      integer :: Ntile, GsendN, GrecvN, Ntag, Nerror, Nrequest
      integer :: EWsize
      integer :: NSsize
      integer :: Null_Value

      integer, dimension(-1:NtileI(ng),-1:NtileJ(ng)) :: table
# ifdef MPI
      integer, dimension(MPI_STATUS_SIZE,4) :: status
# endif

# if defined EW_PERIODIC || defined NS_PERIODIC
      integer, parameter :: pp = 1
# else
      integer, parameter :: pp = 0
# endif

      real(r8), dimension(TileSize(ng)) :: AsendW
      real(r8), dimension(TileSize(ng)) :: AsendE
      real(r8), dimension(TileSize(ng)) :: AsendS
      real(r8), dimension(TileSize(ng)) :: AsendN

      real(r8), dimension(TileSize(ng)) :: ArecvW
      real(r8), dimension(TileSize(ng)) :: ArecvE
      real(r8), dimension(TileSize(ng)) :: ArecvS
      real(r8), dimension(TileSize(ng)) :: ArecvN

      character (len=80) :: string

# ifdef PROFILE
!
!-----------------------------------------------------------------------
!  Turn on time clocks.
!-----------------------------------------------------------------------
!
      CALL wclock_on (ng, model, 39)
# endif
!
!-----------------------------------------------------------------------
!  Initialize private adjoint communication buffers.
!-----------------------------------------------------------------------
!
!
!-----------------------------------------------------------------------
!  Send and recive tile boundary data.
!-----------------------------------------------------------------------
!
!  Set tile partition table for looking up adjacent processes. Notice
!  that a null value is used in places that data transmition is not
!  required.
!
# if defined MPI
      Null_Value=MPI_PROC_NULL
# else
      Null_Value=-1
# endif
      DO j=-1,NtileJ(ng)
        DO i=-1,NtileI(ng)
          table(i,j)=Null_Value
        END DO
      END DO
      rank=0
      DO j=0,NtileJ(ng)-1
        DO i=0,NtileI(ng)-1
          table(i,j)=rank
          IF (MyRank.eq.rank) THEN
            MyRankI=i
            MyRankJ=j
          END IF
          rank=rank+1            
        END DO
      END DO
!
!  Set boundary communication tags and exchange switches.
!
      Wtag=1
      Stag=2
      Etag=3
      Ntag=4
      Wexchange=.FALSE.
      Sexchange=.FALSE.
      Eexchange=.FALSE.
      Nexchange=.FALSE.
!
!-----------------------------------------------------------------------
!  Process Southern and Northern tile boundary data.
!-----------------------------------------------------------------------
!
!  Determine range and length of the distributed tile boundary segments.
!
      Imin=LBi
      Imax=UBi
      Ilen=Imax-Imin+1
      Klen=UBk-LBk+1
      IKlen=Ilen*Klen
      NSsize=(Nghost+pp)*(UBi-LBi+1)*(UBk-LBk+1)
!
!  Determine the rank of Southern and Northern tiles.  Then, determine
!  the number of ghost-points to send and receive in the South- and
!  North-directions.  This logic only works for two and three ghost
!  points. The number of ghost-points changes when periodic boundary
!  condition are activated.  The periodicity is as follows:
!
!  If two ghost-points:
!
!                      Mm-2  Mm-1  Mm   Mm+1  Mm+2
!                      -2    -1     0    1     2
!
!  If three ghost-points:
!
!                      Mm-2  Mm-1  Mm   Mm+1  Mm+2  Mm+3
!                      -2    -1     0    1     2     3
!
      IF (NSperiodic) THEN
        IF ((table(MyRankI,MyRankJ-1).eq.Null_Value).and.               &
     &      (NtileJ(ng).gt.1)) THEN
          Stile=table(MyRankI,NtileJ(ng)-1)
          Ntile=table(MyRankI,MyRankJ+1)
          GsendS=Nghost
          GsendN=Nghost
# if defined THREE_GHOST
          GrecvS=Nghost
# else
          GrecvS=Nghost+1
# endif
          GrecvN=Nghost
        ELSE IF ((table(MyRankI,MyRankJ+1).eq.Null_Value).and.          &
     &           (NtileJ(ng).gt.1)) then
          Stile=table(MyRankI,MyRankJ-1)
          Ntile=table(MyRankI,0)
          GsendS=Nghost
# if defined THREE_GHOST
          GsendN=Nghost
# else
          GsendN=Nghost+1
# endif
          GrecvS=Nghost
          GrecvN=Nghost
        ELSE
          Stile=table(MyRankI,MyRankJ-1)
          Ntile=table(MyRankI,MyRankJ+1)
          GsendS=Nghost
          GsendN=Nghost
          GrecvS=Nghost
          GrecvN=Nghost
        END IF
      ELSE
        Stile=table(MyRankI,MyRankJ-1)
        Ntile=table(MyRankI,MyRankJ+1)
        GsendS=Nghost
        GsendN=Nghost
        GrecvS=Nghost
        GrecvN=Nghost
      END IF
      IF (Stile.ne.Null_Value) Sexchange=.TRUE.
      IF (Ntile.ne.Null_Value) Nexchange=.TRUE.
!
!-----------------------------------------------------------------------
!  Adjoint of unpacking Northern and Southern segments.
!-----------------------------------------------------------------------
!
      IF (Nexchange) THEN
        IKsizeN=0
        ArecvN=0.0_r8
        AsendN=0.0_r8
        DO m=1,GrecvN
          mc=(m-1)*IKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Ilen
            DO i=Imin,Imax
              IKsizeN=IKsizeN+1
              ikN=1+(i-Imin)+kc+mc
!>            A(i,Jend+m,k)=ArecvN(ikN)
!>
              ArecvN(ikN)=A(i,Jend+m,k)
              A(i,Jend+m,k)=0.0_r8
            END DO
          END DO
        END DO
      END IF
      IF (Sexchange) THEN
        IKsizeS=0
        ArecvS=0.0_r8
        AsendS=0.0_r8
        DO m=GrecvS,1,-1
          mc=(GrecvS-m)*IKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Ilen
            DO i=Imin,Imax
              IKsizeS=IKsizeS+1
              ikS=1+(i-Imin)+kc+mc
!>            A(i,Jstr-m,k)=ArecvS(ikS)
!>
              ArecvS(ikS)=A(i,Jstr-m,k)
              A(i,Jstr-m,k)=0.0_r8
            END DO
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Adjoint of send and receive Southern and Northern segments.
!-----------------------------------------------------------------------
!
# if defined MPI
      IF (Sexchange) THEN
!>      CALL mpi_irecv (ArecvS, NSsize, MP_FLOAT, Stile, Ntag,          &
!>   &                  OCN_COMM_WORLD, Srequest, Serror)
!>
        CALL mpi_irecv (AsendS, NSsize, MP_FLOAT, Stile, Ntag,          &
     &                  OCN_COMM_WORLD, Srequest, Serror)
      END IF
      IF (Nexchange) THEN
!>      CALL mpi_irecv (ArecvN, NSsize, MP_FLOAT, Ntile, Stag,          &
!>   &                  OCN_COMM_WORLD, Nrequest, Nerror)
!>
        CALL mpi_irecv (AsendN, NSsize, MP_FLOAT, Ntile, Stag,          &
     &                  OCN_COMM_WORLD, Nrequest, Nerror)
      END IF
      IF (Sexchange) THEN
!>      CALL mpi_send  (AsendS, IKsizeS, MP_FLOAT, Stile, Stag,         &
!>   &                  OCN_COMM_WORLD, Serror)
!>
        CALL mpi_send  (ArecvS, IKsizeS, MP_FLOAT, Stile, Stag,         &
     &                  OCN_COMM_WORLD, Serror)
      END IF
      IF (Nexchange) THEN
!>      CALL mpi_send  (AsendN, IKsizeN, MP_FLOAT, Ntile, Ntag,         &
!>   &                  OCN_COMM_WORLD, Nerror)
!>
        CALL mpi_send  (ArecvN, IKsizeN, MP_FLOAT, Ntile, Ntag,         &
     &                  OCN_COMM_WORLD, Nerror)
      END IF
# endif
!
!  Adjoint of packing tile boundary data including ghost-points.
!
      IF (Sexchange) THEN            
# ifdef MPI
        CALL mpi_wait (Srequest, status(1,2), Serror)
        IF (Serror.ne.MPI_SUCCESS) THEN
          CALL mpi_error_string (Serror, string, LEN(string), Serror)
          Lstr=LEN_TRIM(string)
          WRITE (stdout,10) 'MPI_SEND/MPI_IRECV (Southern Edge)',       &
     &                      MyRank, Serror, string(1:Lstr)
 10       FORMAT (/,' AD_MP_EXCHANGE - error during ',a,' call,',       &
     &            ' Node = ', i3.3,' Error = ',i3,/,18x,a)
          exit_flag=2
          RETURN
        END IF
# endif
        DO m=1,GsendS
          mc=(m-1)*IKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Ilen
            DO i=Imin,Imax
              ikS=1+(i-Imin)+kc+mc
!>            AsendS(ikS)=A(i,Jstr+m-1,k)
!>
              A(i,Jstr+m-1,k)=A(i,Jstr+m-1,k)+AsendS(ikS)
              AsendS(ikS)=0.0_r8
            END DO
          END DO
        END DO
      END IF
      IF (Nexchange) THEN
# ifdef MPI
        CALL mpi_wait (Nrequest, status(1,4), Nerror)
        IF (Nerror.ne.MPI_SUCCESS) THEN
          CALL mpi_error_string (Nerror, string, LEN(string), Nerror)
          Lstr=LEN_TRIM(string)
          WRITE (stdout,10) 'MPI_SEND/MPI_IRECV (Northern Edge)',       &
     &                      MyRank, Nerror, string(1:Lstr)
          exit_flag=2
          RETURN
        END IF
# endif
        DO m=1,GsendN
          mc=(m-1)*IKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Ilen
            DO i=Imin,Imax
              ikN=1+(i-Imin)+kc+mc
!>            AsendN(ikN)=A(i,Jend-GsendN+m,k)
!>
              A(i,Jend-GsendN+m,k)=A(i,Jend-GsendN+m,k)+AsendN(ikN)
              AsendN(ikN)=0.0_r8
            END DO
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Process Western and Eastern tile boundary data.
!-----------------------------------------------------------------------
!
!  Determine range and length of the distributed tile boundary segments.
!
      Jmin=LBj
      Jmax=UBj
      Jlen=Jmax-Jmin+1
      Klen=UBk-LBk+1
      JKlen=Jlen*Klen
      EWsize=(Nghost+pp)*(UBj-LBj+1)*(UBk-LBk+1)
!
!  Determine the rank of Western and Eastern tiles.  Then, determine
!  the number of ghost-points to send and receive in the West- and
!  East-directions.  This logic only works for two and three ghost
!  points. The number of ghost-points changes when periodic boundary
!  condition are activated.  The periodicity is as follows:
!
!  If two ghost-points:
!
!                      Lm-2  Lm-1  Lm   Lm+1  Lm+2
!                      -2    -1     0    1     2
!
!  If three ghost-points:
!
!                      Lm-2  Lm-1  Lm   Lm+1  Lm+2   Lm+3
!                      -2    -1     0    1     2      3
!
      IF (EWperiodic) THEN
        IF ((table(MyRankI-1,MyRankJ).eq.Null_Value).and.               &
     &      (NtileI(ng).gt.1)) THEN
          Wtile=table(NtileI(ng)-1,MyRankJ)
          Etile=table(MyRankI+1,MyRankJ)
          GsendW=Nghost
          GsendE=Nghost
# if defined THREE_GHOST
          GrecvW=Nghost
# else
          GrecvW=Nghost+1
# endif
          GrecvE=Nghost
        ELSE IF ((table(MyRankI+1,MyRankJ).eq.Null_Value).and.          &
     &           (NtileI(ng).gt.1)) THEN
          Wtile=table(MyRankI-1,MyRankJ)
          Etile=table(0,MyRankJ)
          GsendW=Nghost
# if defined THREE_GHOST
          GsendE=Nghost
# else
          GsendE=Nghost+1
# endif
          GrecvW=Nghost
          GrecvE=Nghost
        ELSE
          Wtile=table(MyRankI-1,MyRankJ)
          Etile=table(MyRankI+1,MyRankJ)
          GsendW=Nghost
          GsendE=Nghost
          GrecvW=Nghost
          GrecvE=Nghost
        END IF
      ELSE
        Wtile=table(MyRankI-1,MyRankJ)
        Etile=table(MyRankI+1,MyRankJ)
        GsendW=Nghost
        GsendE=Nghost
        GrecvW=Nghost
        GrecvE=Nghost
      END IF
      IF (Wtile.ne.Null_Value) Wexchange=.TRUE.
      IF (Etile.ne.Null_Value) Eexchange=.TRUE.
!
!-----------------------------------------------------------------------
!  Adjoint of unpack Eastern and Western segments.
!-----------------------------------------------------------------------
!
      IF (Eexchange) THEN
        JKsizeE=0
        ArecvE=0.0_r8
        AsendE=0.0_r8
        DO m=1,GrecvE
          mc=(m-1)*JKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Jlen
            DO j=Jmin,Jmax
              JKsizeE=JKsizeE+1
              jkE=1+(j-Jmin)+kc+mc
!>            A(Iend+m,j,k)=ArecvE(jkE)
!>
              ArecvE(jkE)=A(Iend+m,j,k)
              A(Iend+m,j,k)=0.0_r8
            END DO
          ENDDO
        END DO
      END IF
      IF (Wexchange) THEN
        JKsizeW=0
        ArecvW=0.0_r8
        AsendW=0.0_r8
        DO m=GrecvW,1,-1
          mc=(GrecvW-m)*JKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Jlen
            DO j=Jmin,Jmax
              JKsizeW=JKsizeW+1
              jkW=1+(j-Jmin)+kc+mc
!>            A(Istr-m,j,k)=ArecvW(jkW)
!>
              ArecvW(jkW)=A(Istr-m,j,k)
              A(Istr-m,j,k)=0.0_r8
            END DO
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Send and receive Western and Eastern segments.
!-----------------------------------------------------------------------
!
# if defined MPI
      IF (Wexchange) THEN
!>      CALL mpi_irecv (ArecvW, EWsize, MP_FLOAT, Wtile, Etag,          &
!>   &                  OCN_COMM_WORLD, Wrequest, Werror)
!>
        CALL mpi_irecv (AsendW, EWsize, MP_FLOAT, Wtile, Etag,          &
     &                  OCN_COMM_WORLD, Wrequest, Werror)
      END IF
      IF (Eexchange) THEN
!>      CALL mpi_irecv (ArecvE, EWsize, MP_FLOAT, Etile, Wtag,          &
!>   &                  OCN_COMM_WORLD, Erequest, Eerror)
!>
        CALL mpi_irecv (AsendE, EWsize, MP_FLOAT, Etile, Wtag,          &
     &                  OCN_COMM_WORLD, Erequest, Eerror)
      END IF
      IF (Wexchange) THEN
!>      CALL mpi_send  (AsendW, JKsizeW, MP_FLOAT, Wtile, Wtag,         &
!>   &                  OCN_COMM_WORLD, Werror)
!>
        CALL mpi_send  (ArecvW, JKsizeW, MP_FLOAT, Wtile, Wtag,         &
     &                  OCN_COMM_WORLD, Werror)
      END IF
      IF (Eexchange) THEN
!>      CALL mpi_send  (AsendE, JKsizeE, MP_FLOAT, Etile, Etag,         &
!>   &                  OCN_COMM_WORLD, Eerror)
!>
        CALL mpi_send  (ArecvE, JKsizeE, MP_FLOAT, Etile, Etag,         &
     &                  OCN_COMM_WORLD, Eerror)
      END IF
# endif
!
!  Adjoint of packing tile boundary data including ghost-points.
!
      IF (Wexchange) THEN
# ifdef MPI
        CALL mpi_wait (Wrequest, status(1,1), Werror)
        IF (Werror.ne.MPI_SUCCESS) THEN
          CALL mpi_error_string (Werror, string, LEN(string), Werror)
          Lstr=LEN_TRIM(string)
          WRITE (stdout,10) 'MPI_SEND/MPI_IRECV (Western Edge)',        &
     &                      MyRank, Werror, string(1:Lstr)
          exit_flag=2
          RETURN
        END IF
# endif
        DO m=1,GsendW
          mc=(m-1)*JKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Jlen
            DO j=Jmin,Jmax
              jkW=1+(j-Jmin)+kc+mc
!>            AsendW(jkW)=A(Istr+m-1,j,k)
!>
              A(Istr+m-1,j,k)=A(Istr+m-1,j,k)+AsendW(jkW)
              AsendW(jkW)=0.0_r8
            END DO
          END DO
        END DO
      END IF
      IF (Eexchange) THEN
# ifdef MPI
        CALL mpi_wait (Erequest, status(1,3), Eerror)
        IF (Eerror.ne.MPI_SUCCESS) THEN
          CALL mpi_error_string (Eerror, string, LEN(string), Eerror)
          Lstr=LEN_TRIM(string)
          WRITE (stdout,10) 'MPI_SEND/MPI_IRECV (Eastern Edge)',        &
     &                      MyRank, Eerror, string(1:Lstr)
          exit_flag=2
          RETURN
        END IF
# endif
        DO m=1,GsendE
          mc=(m-1)*JKlen
          DO k=LBk,UBk
            kc=(k-LBk)*Jlen
            DO j=Jmin,Jmax
              jkE=1+(j-Jmin)+kc+mc
!>            AsendE(jkE)=A(Iend-GsendE+m,j,k)
!>
              A(Iend-GsendE+m,j,k)=A(Iend-GsendE+m,j,k)+AsendE(jkE)
              AsendE(jkE)=0.0_r8
            END DO
          END DO
        END DO
      END IF
# ifdef PROFILE
!
!-----------------------------------------------------------------------
!  Turn off time clocks.
!-----------------------------------------------------------------------
!
      CALL wclock_off (ng, model, 39)
# endif

      RETURN
      END SUBROUTINE ad_mp_exchange
