#include "cppdefs.h"
      MODULE ocean_control_mod
!
!================================================== Hernan G. Arango ===
!  Copyright (c) 2005 ROMS/TOMS Group              Christopher Moore   !
!=======================================================================
!                                                                      !
!  ROMS/TOMS Nonlinear model:                                          !
!                                                                      !
!  This driver executes ROMS/TOMS standard nonlinear model.  It        !
!  controls the initialization, time-stepping, and finalization        !
!  of the nonlinear model execution following ESMF conventions:        !
!                                                                      !
!     initialize                                                       !
!     run                                                              !
!     finalize                                                         !
!                                                                      !
!=======================================================================
!
      implicit none

      PRIVATE
      PUBLIC  :: initialize, run, finalize

      CONTAINS

      SUBROUTINE initialize (first, MyCOMM)
!
!=======================================================================
!                                                                      !
!  This routine allocates and initializes ROMS/TOMS state variables    !
!  and internal and external parameters.                               !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_scalars
!
#ifdef AIR_OCEAN 
      USE atm_coupler_mod, ONLY : initialize_coupling
#endif
!
!  Imported variable declarations.
!
      logical, intent(inout) :: first

      integer, intent(in), optional :: MyCOMM
!
!  Local variable declarations.
!
      logical :: allocate_vars = .TRUE.

      integer :: ng, thread

#ifdef DISTRIBUTE
!
!-----------------------------------------------------------------------
!  Set distribute-memory (MPI) world communictor.
!-----------------------------------------------------------------------
!
      IF (PRESENT(MyCOMM)) THEN
        OCN_COMM_WORLD=MyCOMM
      ELSE
        OCN_COMM_WORLD=MPI_COMM_WORLD
      END IF
#endif
!
!-----------------------------------------------------------------------
!  On first pass, initialize model parameters a variables for all
!  nested/composed grids.  Notice that the logical switch "first"
!  is used to allow multiple calls to this routine during ensemble
!  configurations.
!-----------------------------------------------------------------------
!
      IF (first) THEN
        first=.FALSE.
!
!  Initialize model internal parameters.
!
        CALL initialize_param
        CALL initialize_parallel
        CALL initialize_scalars
!
!  Initialize wall clocks.
!
        IF (Master) THEN
          WRITE (stdout,10)
 10       FORMAT (' Process Information:',/)
        END IF
        DO ng=1,Ngrids
!$OMP PARALLEL DO PRIVATE(thread) SHARED(ng,numthreads)
          DO thread=0,numthreads-1
            CALL wclock_on (ng, iNLM, 0)
          END DO
!$OMP END PARALLEL DO
        END DO

#ifdef AIR_OCEAN 
!
!  Initialize coupling streams between atmosphere and ocean using the
!  Model Coupling Toolkit (MCT).
!
        CALL initialize_coupling (MyRank)
#endif
!
!  Read in model tunable parameters from standard input.
!
        CALL inp_par (iNLM)
        IF (exit_flag.ne.NoError) THEN
          IF (Master) THEN
            WRITE (stdout,'(/,a,i3,/)') Rerror(exit_flag), exit_flag
          END IF
          RETURN
        END IF
!
!  Allocate and initialize modules variables.
!
        CALL mod_arrays (allocate_vars)
      END IF
!
!-----------------------------------------------------------------------
!  Initialize model state variables for all nested/composed grids.
!-----------------------------------------------------------------------
!
      DO ng=1,Ngrids
        CALL initial (ng)
        IF (exit_flag.ne.NoError) THEN
          IF (Master) THEN
            WRITE (stdout,'(/,a,i3,/)') Rerror(exit_flag), exit_flag
          END IF
          RETURN
        END IF
      END DO

      RETURN
      END SUBROUTINE initialize

      SUBROUTINE run
!
!=======================================================================
!                                                                      !
!  This routine time-steps ROMS/TOMS nonlinear model.                  !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_grid
      USE mod_ocean
      USE mod_scalars
!
!  Local variable declarations.
!
      integer :: IniRec, itrc, ng
!
!-----------------------------------------------------------------------
!  Run model for all nested grids, if any.
!-----------------------------------------------------------------------
!
        ng=1
!
!  Get physical non-overlapping horizontal tile bounds.
!
# ifdef DISTRIBUTE
        CALL get_tile (ng, TILE, Itile, Jtile, Istr, Iend, Jstr, Jend)
# else
        Istr=1
        Iend=Lm(ng)
        Jstr=1
        Jend=Mm(ng)
# endif
# include "set_bounds.h"
!
!  Get model state to convolute.
!
      IniRec=1
      CALL get_state (ng, iNLM, 1, HISname(ng), IniRec, 1)

!
!  Convolute 2D diffusion Equation.
!
      CALL bcov_r2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                    LBi, UBi, LBj, UBj, NghostPoints,             &
     &                    DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    GRID(ng) % pmon_u,                            &
     &                    GRID(ng) % pnom_v,                            &
#  ifdef MASKING
     &                    GRID(ng) % umask,                             &
     &                    GRID(ng) % vmask,                             &
#  endif
     &                    OCEAN(ng) % zeta(:,:,1))

      CALL bcov_u2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                    LBi, UBi, LBj, UBj, NghostPoints,             &
     &                    DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    GRID(ng) % pmon_r,                            &
     &                    GRID(ng) % pnom_p,                            &
#  ifdef MASKING
     &                    GRID(ng) % pmask,                             &
#  endif
     &                    OCEAN(ng) % ubar(:,:,1))

      CALL bcov_v2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                    LBi, UBi, LBj, UBj, NghostPoints,             &
     &                    DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    GRID(ng) % pmon_p,                            &
     &                    GRID(ng) % pnom_r,                            &
#  ifdef MASKING
     &                    GRID(ng) % pmask,                             &
#  endif
     &                    OCEAN(ng) % vbar(:,:,1))
!
!  Convolute 2D diffusion Equation.
!
      CALL bcov_u3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                    LBi, UBi, LBj, UBj, 1, N(ng), NghostPoints,   &
     &                    DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    GRID(ng) % pmon_r,                            &
     &                    GRID(ng) % pnom_p,                            &
#  ifdef MASKING
     &                    GRID(ng) % pmask,                             &
#  endif
     &                    GRID(ng) % z_r,                               &
     &                    GRID(ng) % Hz,                                &
     &                    OCEAN(ng) % u(:,:,:,1))


      CALL bcov_u3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                    LBi, UBi, LBj, UBj, 1, N(ng), NghostPoints,   &
     &                    DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                    GRID(ng) % pm,                                &
     &                    GRID(ng) % pn,                                &
     &                    GRID(ng) % pmon_p,                            &
     &                    GRID(ng) % pnom_r,                            &
#  ifdef MASKING
     &                    GRID(ng) % pmask,                             &
#  endif
     &                    GRID(ng) % z_r,                               &
     &                    GRID(ng) % Hz,                                &
     &                    OCEAN(ng) % v(:,:,:,1))

      DO itrc=1,NT(ng)
        CALL bcov_r3d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,             &
     &                      LBi, UBi, LBj, UBj, 1, N(ng), NghostPoints,   &
     &                      DTdif(ng), Ldecay(ng), 1.0_r8,                &
     &                      GRID(ng) % pm,                                &
     &                      GRID(ng) % pn,                                &
     &                      GRID(ng) % pmon_u,                            &
     &                      GRID(ng) % pnom_v,                            &
#  ifdef MASKING
     &                      GRID(ng) % umask,                             &
     &                      GRID(ng) % vmask,                             &
#  endif
     &                      GRID(ng) % z_r,                               &
     &                      GRID(ng) % Hz,                                &
     &                      OCEAN(ng) % t(:,:,:,1,itrc))
!
!  Write out convoluted files.
!
      CALL wrt_his (ng)

      RETURN
      END SUBROUTINE run

      SUBROUTINE finalize
!
!=======================================================================
!                                                                      !
!  This routine terminates ROMS/TOMS nonlinear model execution.        !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
!
!  Local variable declarations.
!
      integer :: ng, thread
!
!-----------------------------------------------------------------------
!  If blowing-up, save latest model state into RESTART NetCDF file.
!-----------------------------------------------------------------------
!
!  If cycling restart records, write solution into the next record.
!
      DO ng=1,Ngrids
        IF (LwrtRST(ng).and.(exit_flag.eq.1)) THEN
          IF (Master) WRITE (stdout,10)
 10       FORMAT (/,' Blowing-up: Saving latest model state into ',     & 
     &              ' RESTART file',/)
          IF (LcycleRST(ng).and.(NrecRST(ng).ge.2)) THEN
            tRSTindx(ng)=2
            LcycleRST(ng)=.FALSE.
          END IF
          blowup=exit_flag
          exit_flag=NoError
          CALL wrt_rst (ng)
        END IF
      END DO
!
!-----------------------------------------------------------------------
!  Stop model and time profiling clocks.  Close output NetCDF files.
!-----------------------------------------------------------------------
!
!  Stop time clocks.
!
      IF (Master) THEN
        WRITE (stdout,20)
 20     FORMAT (/,' Elapsed CPU time (seconds):',/)
      END IF

      DO ng=1,Ngrids
!$OMP PARALLEL DO PRIVATE(thread) SHARED(ng,numthreads)
        DO thread=0,numthreads-1
          CALL wclock_off (ng, iNLM, 0)
        END DO
!$OMP END PARALLEL DO
      END DO
!
!  Close IO files.
!
      CALL close_io

      RETURN
      END SUBROUTINE finalize

      END MODULE ocean_control_mod
