#include "cppdefs.h"
      MODULE impulse_mod
#ifdef IMPULSE
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) ROMS/TOMS Adjoint Group                               !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This subroutine computes impulse forcing for the tangent linear     !
!  models.  The impulse forcing, F,  is computed from snapshots of     !
!  adjoint solution, ad_F:                                             !
!                                                                      !
!  n     0        1        2        3              N-1        N        !
!        |________|________|________|___......._____|_________|        !
!  iic   0      1*nADJ   2*nADJ   3*nADJ      (N-1)*nADJ    N*nADJ     !
!                                                                      !
!  They are only computed at the time-steps where the  solution is     !
!  saved:                                                              !
!                                                                      !
!  At the first time-step, iic=ntdtstr=0, n=0:                         !
!                                                                      !
!        F(0) = A * ad_F(0) + B * ad_F(1)                              !
!                                                                      !
!  at the last time-step, iic=ntend=N*nADJ, n=N:                       !
!                                                                      !
!        F(N) = A * ad_F(N) + B * ad_F(N-1)                            !
!                                                                      !
!  otherwise,                                                          !
!                                                                      !
!        F(n) = B * ad_F(n-1) + C * Ad_F(n) + B * ad_F(n+1)            !
!                                                                      !
!  where,                                                              !
!                                                                      !
!        nADJ = number of time-steps between adjoint snapshots         !
!                                                                      !
!           A = dt * (nADJ + 1) * (2 * nADJ + 1) / (6 * nADJ)          !
!                                                                      !
!           B = dt * (nADJ * nADJ - 1) / (6 * nADJ)                    !
!                                                                      !
!           C = dt * (2 * nADJ * nADJ + 1) / (3 * nADJ)                !
!                                                                      !
!======================================================================!
!
      implicit none

      PRIVATE
      PUBLIC :: impulse

      CONTAINS
!
!***********************************************************************
      SUBROUTINE impulse (ng, tile, FirstPass)
!***********************************************************************
!
      USE mod_param
      USE mod_ocean
!
!  Imported variable declarations.
!
      logical, intent(in) :: FirstPass

      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
      CALL impulse_tile (ng, Istr, Iend, Jstr, Jend,                    &
     &                   LBi, UBi, LBj, UBj,                            &
     &                   FirstPass,                                     &
# ifdef SOLVE3D
     &                   OCEAN(ng) % ad_u,                              &
     &                   OCEAN(ng) % ad_v,                              &
     &                   OCEAN(ng) % ad_t,                              &
     &                   OCEAN(ng) % b_u,                               &
     &                   OCEAN(ng) % b_v,                               &
     &                   OCEAN(ng) % b_t,                               &
# endif
     &                   OCEAN(ng) % ad_ubar,                           &
     &                   OCEAN(ng) % ad_vbar,                           &
     &                   OCEAN(ng) % ad_zeta,                           &
     &                   OCEAN(ng) % b_ubar,                            &
     &                   OCEAN(ng) % b_vbar,                            &
     &                   OCEAN(ng) % b_zeta)
      RETURN
      END SUBROUTINE impulse
!
!***********************************************************************
      SUBROUTINE impulse_tile (ng, Istr, Iend, Jstr, Jend,              &
     &                         LBi, UBi, LBj, UBj,                      &
     &                         FirstPass,                               &
# ifdef SOLVE3D
     &                         ad_u, ad_v, ad_t,                        &
     &                         b_u, b_v, b_t,                           &
# endif
     &                         ad_ubar, ad_vbar, ad_zeta,               &
     &                         b_ubar, b_vbar, b_zeta)
!***********************************************************************
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_scalars
!
!  Imported variable declarations.
!
      logical, intent(in) :: FirstPass

      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
# ifdef ASSUMED_SHAPE
#  ifdef SOLVE3D
      real(r8), intent(in) :: ad_t(LBi:,LBj:,:,:,:)
      real(r8), intent(in) :: ad_u(LBi:,LBj:,:,:)
      real(r8), intent(in) :: ad_v(LBi:,LBj:,:,:)
#  endif
      real(r8), intent(in) :: ad_ubar(LBi:,LBj:,:)
      real(r8), intent(in) :: ad_vbar(LBi:,LBj:,:)
      real(r8), intent(in) :: ad_zeta(LBi:,LBj:,:)
#  ifdef SOLVE3D
      real(r8), intent(out) :: b_t(LBi:,LBj:,:,:)
      real(r8), intent(out) :: b_u(LBi:,LBj:,:)
      real(r8), intent(out) :: b_v(LBi:,LBj:,:)
#  endif
      real(r8), intent(out) :: b_ubar(LBi:,LBj:)
      real(r8), intent(out) :: b_vbar(LBi:,LBj:)
      real(r8), intent(out) :: b_zeta(LBi:,LBj:)
# else
#  ifdef SOLVE3D
      real(r8), intent(in) :: ad_t(LBi:UBi,LBj:UBj,N(ng),3,NT(ng))
      real(r8), intent(in) :: ad_u(LBi:UBi,LBj:UBj,N(ng),2)
      real(r8), intent(in) :: ad_v(LBi:UBi,LBj:UBj,N(ng),2)
#  endif
      real(r8), intent(in) :: ad_ubar(LBi:UBi,LBj:UBj,3)
      real(r8), intent(in) :: ad_vbar(LBi:UBi,LBj:UBj,3)
      real(r8), intent(in) :: ad_zeta(LBi:UBi,LBj:UBj,3)
#  ifdef SOLVE3D
      real(r8), intent(out) :: b_t(LBi:UBi,LBj:UBj,N(ng),NT(ng))
      real(r8), intent(out) :: b_u(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: b_v(LBi:UBi,LBj:UBj,N(ng))
#  endif
      real(r8), intent(out) :: b_ubar(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: b_vbar(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: b_zeta(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: Iold, Inow, Inxt
      integer :: i, j
# ifdef SOLVE3D
      integer :: k, itrc
# endif
      real(r8) :: B, C, cff
!
# include "set_bounds.h"
!
!-----------------------------------------------------------------------
!  Compute impulse forcing terms.
!-----------------------------------------------------------------------
!
      IF (MOD(iic(ng)-1,nADJ(ng)).eq.0) THEN
!
!  Set weight coefficients for adjoint solution snapshots.
!
        cff=REAL(nADJ(ng),r8)
        IF (FirstPass) THEN
          IF (iic(ng).eq.ntstart) THEN
            Iold=IinfoAD(5,idFsur,ng)
            B=(cff+1.0_r8)*(2.0_r8*cff+1.0_r8)/(6.0_r8*cff)
          ELSE IF ((iic(ng)-1).eq.ntend) THEN
            Iold=3-IinfoAD(5,idFsur,ng)
            B=(cff*cff-1.0_r8)/(6.0_r8*cff)
          ELSE
            Iold=3-IinfoAD(5,idFsur,ng)
            B=(cff*cff-1.0_r8)/(6.0_r8*cff)
          END IF
        ELSE
          IF (iic(ng).eq.ntstart) THEN
            Inxt=IinfoAD(5,idFsur,ng)
            Iold=Inxt
            Inow=3-Inxt
            B=0.0_r8
            C=(cff*cff-1.0_r8)/(6.0_r8*cff)
          ELSE IF ((iic(ng)-1).eq.ntend) THEN
            Inxt=IinfoAD(5,idFsur,ng)
            Iold=Inxt
            Inow=3-Inxt
            B=0.0_r8
            C=(cff+1.0_r8)*(2.0_r8*cff+1.0_r8)/(6.0_r8*cff)
          ELSE
            Inxt=IinfoAD(5,idFsur,ng)
            Iold=Inxt
            Inow=3-Inxt
            B=(cff*cff-1.0_r8)/(6.0_r8*cff)
            C=(2.0_r8*cff*cff+1.0_r8)/(3.0_r8*cff)
          END IF
        END IF
!
!  The strategy here is to compute the impulse forcing in two stages
!  to avoid storing in the adjoint solution in tree-time level work
!  arrays.
!
        IF (FirstPass) THEN
!
!  Free-surface impulse forcing term.
!
          DO j=JstrR,JendR
            DO i=IstrR,IendR
              b_zeta(i,j)=B*ad_zeta(i,j,Iold)
            END DO
          END DO
!
!  2D momentum impulse forcing terms.
!
          DO j=JstrR,JendR
            DO i=Istr,IendR
              b_ubar(i,j)=B*ad_ubar(i,j,Iold)
            END DO
          END DO
          DO j=Jstr,JendR
            DO i=IstrR,IendR
              b_vbar(i,j)=B*ad_vbar(i,j,Iold)
            END DO
          END DO
# ifdef SOLVE3D
!
!  3D momentum impulse forcing terms.
!
          DO k=1,N(ng)
            DO j=JstrR,JendR
              DO i=Istr,IendR
                b_u(i,j,k)=B*ad_u(i,j,k,Iold)
              END DO
            END DO
            DO j=Jstr,JendR
              DO i=IstrR,IendR
                b_v(i,j,k)=B*ad_v(i,j,k,Iold)
              END DO
            END DO
          END DO
!
!  Tracer impulse forcing terms.
!
          DO itrc=1,NT(ng)
            DO k=1,N(ng)
              DO j=JstrR,JendR
                DO i=IstrR,IendR
                  b_t(i,j,k,itrc)=B*ad_t(i,j,k,Iold,itrc)
                END DO
              END DO
            END DO
          END DO
# endif
        ELSE
!
!  Free-surface impulse forcing term.
!
          DO j=JstrR,JendR
            DO i=IstrR,IendR
              b_zeta(i,j)=b_zeta(i,j)+                                  &
     &                    C*ad_zeta(i,j,Inow)+                          &
     &                    B*ad_zeta(i,j,Inxt)
            END DO
          END DO
!
!  2D momentum impulse forcing terms.
!
          DO j=JstrR,JendR
            DO i=Istr,IendR
              b_ubar(i,j)=b_ubar(i,j)+                                  &
     &                    C*ad_ubar(i,j,Inow)+                          &
     &                    B*ad_ubar(i,j,Inxt)
            END DO
          END DO
          DO j=Jstr,JendR
            DO i=IstrR,IendR
              b_vbar(i,j)=b_vbar(i,j)+                                  &
     &                    C*ad_vbar(i,j,Inow)+                          &
     &                    B*ad_vbar(i,j,Inxt)
            END DO
          END DO
# ifdef SOLVE3D
!
!  3D momentum impulse forcing terms.
!
          DO k=1,N(ng)
            DO j=JstrR,JendR
              DO i=Istr,IendR
                b_u(i,j,k)=b_u(i,j,k)+                                  &
     &                     C*ad_u(i,j,k,Inow)+                          &
     &                     B*ad_u(i,j,k,Inxt)
              END DO
            END DO
            DO j=Jstr,JendR
              DO i=IstrR,IendR
                b_v(i,j,k)=b_v(i,j,k)+                                  &
     &                     C*ad_v(i,j,k,Inow)+                          &
     &                     B*ad_v(i,j,k,Inxt)
              END DO
            END DO
          END DO
!
!  Tracer impulse forcing terms.
!
          DO itrc=1,NT(ng)
            DO k=1,N(ng)
              DO j=JstrR,JendR
                DO i=IstrR,IendR
                  b_t(i,j,k,itrc)=b_t(i,j,k,itrc)+                      &
     &                            C*ad_t(i,j,k,Inow,itrc)+              &
     &                            B*ad_t(i,j,k,Inxt,itrc)
                END DO
              END DO
            END DO
          END DO
# endif
          IF (MASTER) THEN
            WRITE (stdout,10) tdays, Iold, Inow, Inxt
          END IF
        END IF

      END IF

 10   FORMAT (3x,' IMPULSE     - Computed adjoint impulse forcing,',    &
     &        t64,'t = ',f12.4,/,19x,'(Iold=',i1,' Inow=',i1,           &
     &        ' Inxt=',i1,')')

      RETURN
      END SUBROUTINE impulse_tile
#endif
      END MODULE impulse_mod
