#include "cppdefs.h"
      MODULE randomization_mod

#ifdef FOUR_DVAR
!
!=======================================================================
!  Copyright (c) ROMS/TOMS Adjoint Group                               !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine computes the background covariance, B, normalization   !
!  factors using the  randomization approach of  Fisher and Courtier   !
!  (1995).  These factors ensure that the diagonal elements of B are   !
!  equal to unity. Notice that in applications with land/sea masking,  !
!  it will produce large changes in the  covariance structures  near   !
!  the boundary.                                                       !
!                                                                      !
!======================================================================!
!
      USE mod_kinds

      implicit none

      PRIVATE
      PUBLIC :: randomization

      CONTAINS
!
!***********************************************************************
      SUBROUTINE randomization (ng, tile)
!***********************************************************************
!
      USE mod_param
      USE mod_grid
      USE mod_ocean
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, tile
!
!  Local variable declarations.
!
# include "tile.h"
!
      CALL randomization_tile (ng, Istr, Iend, Jstr, Jend,              &
     &                         LBi, UBi, LBj, UBj,                      &
     &                         GRID(ng) % pm,                           &
     &                         GRID(ng) % om_r,                         &
     &                         GRID(ng) % om_u,                         &
     &                         GRID(ng) % om_v,                         &
     &                         GRID(ng) % pmon_p,                       &
     &                         GRID(ng) % pmon_r,                       &
     &                         GRID(ng) % pmon_u,                       &
     &                         GRID(ng) % pn,                           &
     &                         GRID(ng) % on_r,                         &
     &                         GRID(ng) % on_u,                         &
     &                         GRID(ng) % on_v,                         &
     &                         GRID(ng) % pnom_p,                       &
     &                         GRID(ng) % pnom_r,                       &
     &                         GRID(ng) % pnom_v,                       &
# ifdef MASKING
     &                         GRID(ng) % pmask,                        &
     &                         GRID(ng) % umask,                        &
     &                         GRID(ng) % vmask,                        &
# endif
# ifdef SOLVE3D
     &                         GRID(ng) % h,                            &
#  ifdef ICESHELF
     &                         GRID(ng) % zice,                         &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                         OCEAN(ng) % bed,                         &
     &                         GRID(ng) % bed_thick0,                   &
#  endif
     &                         GRID(ng) % Hz,                           &
     &                         GRID(ng) % z_r,                          &
     &                         GRID(ng) % z_w,                          &
# endif
# ifdef SOLVE3D
     &                         OCEAN(ng) % b_t,                         &
     &                         OCEAN(ng) % b_u,                         &
     &                         OCEAN(ng) % b_v,                         &
# endif
     &                         OCEAN(ng) % b_zeta,                      &
     &                         OCEAN(ng) % b_ubar,                      &
     &                         OCEAN(ng) % b_vbar)

      RETURN
      END SUBROUTINE randomization
!
!***********************************************************************
      SUBROUTINE randomization_tile (ng, Istr, Iend, Jstr, Jend,        &
     &                               LBi, UBi, LBj, UBj,                &
     &                               pm, om_r, om_u, om_v,              &
     &                               pmon_p, pmon_r, pmon_u,            &
     &                               pn, on_r, on_u, on_v,              &
     &                               pnom_p, pnom_r, pnom_v,            &
# ifdef MASKING
     &                               pmask, umask, vmask,               &
# endif
# ifdef SOLVE3D
     &                               h,                                 &
#  ifdef ICESHELF
     &                               zice,                              &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                               bed, bed_thick0,                   &
#  endif
     &                               Hz, z_r, z_w,                      &
# endif
# ifdef SOLVE3D
     &                               VnormR, VnormU, VnormV,            &
# endif
     &                               HnormR, HnormU, HnormV)
!***********************************************************************
!
      USE mod_param
      USE mod_fourdvar
!
      USE bcov_2d_mod
# ifdef SOLVE3D
      USE bcov_3d_mod
# endif
      USE set_depth_mod
      USE white_noise_mod, ONLY : white_noise2d
# ifdef SOLVE3D
      USE white_noise_mod, ONLY : white_noise3d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: om_r(LBi:,LBj:)
      real(r8), intent(in) :: om_u(LBi:,LBj:)
      real(r8), intent(in) :: om_v(LBi:,LBj:)
      real(r8), intent(in) :: pmon_p(LBi:,LBj:)
      real(r8), intent(in) :: pmon_r(LBi:,LBj:)
      real(r8), intent(in) :: pmon_u(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: on_r(LBi:,LBj:)
      real(r8), intent(in) :: on_u(LBi:,LBj:)
      real(r8), intent(in) :: on_v(LBi:,LBj:)
      real(r8), intent(in) :: pnom_p(LBi:,LBj:)
      real(r8), intent(in) :: pnom_r(LBi:,LBj:)
      real(r8), intent(in) :: pnom_v(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:,LBj:)
      real(r8), intent(in) :: umask(LBi:,LBj:)
      real(r8), intent(in) :: vmask(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:,LBj:)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:,LBj:,:,:)
      real(r8), intent(inout):: bed_thick0(LBi:,LBj:)
#   endif
      real(r8), intent(inout) :: h(LBi:,LBj:)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:,LBj:,:,:)
      real(r8), intent(out) :: VnormU(LBi:,LBj:,:)
      real(r8), intent(out) :: VnormV(LBi:,LBj:,:)
#  endif
      real(r8), intent(out) :: HnormR(LBi:,LBj:)
      real(r8), intent(out) :: HnormU(LBi:,LBj:)
      real(r8), intent(out) :: HnormV(LBi:,LBj:)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:,LBj:,:)
      real(r8), intent(out) :: z_r(LBi:,LBj:,:)
      real(r8), intent(out) :: z_w(LBi:,LBj:,0:)
#  endif
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: on_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_v(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: umask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: vmask(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
#   ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:UBi,LBj:UBj)
#   endif
#   if defined SEDIMENT && defined SED_MORPH
      real(r8), intent(in) :: bed(LBi:UBi,LBj:UBj,Nbed,MBEDP)
      real(r8), intent(inout):: bed_thick0(LBi:UBi,LBj:UBj)
#   endif
      real(r8), intent(inout) :: h(LBi:UBi,LBj:UBj)
#  endif
#  ifdef SOLVE3D
      real(r8), intent(out) :: VnormR(LBi:UBi,LBj:UBj,N(ng),NT(ng))
      real(r8), intent(out) :: VnormU(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: VnormV(LBi:UBi,LBj:UBj,N(ng))
#  endif
      real(r8), intent(out) :: HnormR(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormU(LBi:UBi,LBj:UBj)
      real(r8), intent(out) :: HnormV(LBi:UBi,LBj:UBj)
#  ifdef SOLVE3D
      real(r8), intent(out) :: Hz(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_r(LBi:UBi,LBj:UBj,N(ng))
      real(r8), intent(out) :: z_w(LBi:UBi,LBj:UBj,0:N(ng))
#  endif
# endif
!
!  Local variable declarations.
!
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: Rscheme, i, iter, j
# ifdef SOLVE3D
      integer :: itrc, k
# endif
      real(r8) :: cff, val

      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2d
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2davg
      real(r8), dimension(LBi:UBi,LBj:UBj) :: A2dsqr
      real(r8), dimension(LBi:UBi,LBj:UBj) :: Hnorm
# ifdef SOLVE3D
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3d
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3davg
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: A3dsqr
      real(r8), dimension(LBi:UBi,LBj:UBj,1:N(ng)) :: Vnorm
# endif

# include "set_bounds.h"

# ifdef SOLVE3D
!
!-----------------------------------------------------------------------
!  Compute time invariant depths (use zero free-surface).
!-----------------------------------------------------------------------
!
      DO i=LBi,UBi
        DO j=LBj,UBj
          A2d(i,j)=0.0_r8
        END DO
      END DO

      CALL set_depth_tile (ng, Istr, Iend, Jstr, Jend,                  &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     h,                                           &
#  ifdef ICESHELF
     &                     zice,                                        &
#  endif
#  if defined SEDIMENT && defined SED_MORPH
     &                     bed, bed_thick0,                             &
#  endif
     &                     A2d,                                         &
     &                     Hz, z_r, z_w)
# endif
!
!-----------------------------------------------------------------------
!  Compute background covariance, B, normalization factors using the 
!  randomization approach of Fisher and Courtier (1995). These factors
!  ensure that the diagonal elements of B are equal to unity.
!  Notice that in applications with land/sea masking, the boundary
!  conditions will produce large changes in the covariance structures
!  near the boundary.
!
!  Initialize factors with randon numbers ("white-noise") having an
!  uniform distribution (zero mean and unity variance). Then, scale
!  by the cell area (2D) or volume (3D) and "color" with the diffusion
!  operator. Iterate this step over a specified number of ensamble
!  members, Nrandom.
!-----------------------------------------------------------------------
!
      cff=1.0_r8/REAL(Nrandom,r8)
!
!  Set random number generation scheme.
!
      Rscheme=0              !  use F90 intrinsic routine
!     Rscheme=1              !  use Gaussian distributed deviates
!
!  2D norm at RHO-points.
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          A2davg(i,j)=0.0_r8
          A2dsqr(i,j)=0.0_r8
          Hnorm(i,j)=1.0_r8
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise2d (ng, IstrR, IendR, JstrR, JendR,             &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      Rscheme, A2d)
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            A2d(i,j)=A2d(i,j)*SQRT(om_r(i,j)*on_r(i,j))
          END DO
        END DO
        CALL bcov_r2d_tile (ng, iNLM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), Hdecay(ng),                     &
     &                      Hnorm,                                      &
     &                      pm, pn, pmon_u, pnom_v,                     &
# ifdef MASKING
     &                      umask, vmask,                               &
# endif
     &                      A2d)
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            A2davg(i,j)=A2davg(i,j)+A2d(i,j)
            A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
          END DO
        END DO
      END DO
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          HnormR(i,j)=1.0_r8/                                           &
     &                SQRT(cff*(A2dsqr(i,j)-                            &
     &                          cff*A2davg(i,j)*A2davg(i,j)))
        END DO
      END DO
!
!  2D norm at U-points.
!
      DO j=JstrR,JendR
        DO i=Istr,IendR
          A2davg(i,j)=0.0_r8
          A2dsqr(i,j)=0.0_r8
          Hnorm(i,j)=1.0_r8
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise2d (ng, Istr, IendR, JstrR, JendR,              &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      Rscheme, A2d)
        DO j=JstrR,JendR
          DO i=Istr,IendR
            A2d(i,j)=A2d(i,j)*SQRT(om_u(i,j)*on_u(i,j))
          END DO
        END DO
        CALL bcov_u2d_tile (ng, iADM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), Hdecay(ng),                     &
     &                      Hnorm,                                      &
     &                      pm, pn, pmon_r, pnom_p,                     &
#  ifdef MASKING
     &                      pmask,                                      &
#  endif
     &                      A2d)
        DO j=JstrR,JendR
          DO i=Istr,IendR
            A2davg(i,j)=A2davg(i,j)+A2d(i,j)
            A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
          END DO
        END DO
      END DO
      DO j=JstrR,JendR
        DO i=Istr,IendR
          HnormU(i,j)=1.0_r8/                                           &
     &                SQRT(cff*(A2dsqr(i,j)-                            &
     &                          cff*A2davg(i,j)*A2davg(i,j)))
        END DO
      END DO
!
!  2D norm at V-points.
!
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          A2davg(i,j)=0.0_r8
          A2dsqr(i,j)=0.0_r8
          Hnorm(i,j)=1.0_r8
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise2d (ng, IstrR, IendR, Jstr, JendR,              &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      Rscheme, A2d)
        DO j=Jstr,JendR
          DO i=IstrR,IendR
            A2d(i,j)=A2d(i,j)*SQRT(om_v(i,j)*on_v(i,j))
          END DO
        END DO
        CALL bcov_v2d_tile (ng, iADM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), Hdecay(ng),                     &
     &                      Hnorm,                                      &
     &                      pm, pn, pmon_p, pnom_r,                     &
#  ifdef MASKING
     &                      pmask,                                      &
#  endif
     &                      A2d)
        DO j=Jstr,JendR
          DO i=IstrR,IendR
            A2davg(i,j)=A2davg(i,j)+A2d(i,j)
            A2dsqr(i,j)=A2dsqr(i,j)+A2d(i,j)*A2d(i,j)
          END DO
        END DO
      END DO
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          HnormV(i,j)=1.0_r8/                                           &
     &                SQRT(cff*(A2dsqr(i,j)-                            &
     &                          cff*A2davg(i,j)*A2davg(i,j)))
        END DO
      END DO

# ifdef SOLVE3D
!
!  3D norm U-points.
!
      DO j=JstrR,JendR
        DO i=Istr,IendR
          Hnorm(i,j)=1.0_r8
          DO k=1,N(ng)
            A3davg(i,j,k)=0.0_r8
            A3dsqr(i,j,k)=0.0_r8
            Vnorm(i,j,k)=1.0_r8
          END DO
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise3d (ng, Istr, IendR, JstrR, JendR,              &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      Rscheme, A3d)
        DO j=JstrR,JendR
          DO i=Istr,IendR
            DO k=1,N(ng)
              A3d(i,j,k)=A3d(i,j,k)*SQRT(0.5_r8*(Hz(i-1,j,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        CALL bcov_u3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), DTVdif(ng),                     &
     &                      Hdecay(ng), Vdecay(ng),                     &
     &                      Hnorm, Vnorm,                               &
     &                      pm, pn, pmon_r, pnom_p,                     &
#  ifdef MASKING
     &                      pmask,                                      &
#  endif
     &                      Hz, z_r,                                    &
     &                      A3d)
        DO k=1,N(ng)
          DO j=JstrR,JendR
            DO i=Istr,IendR
              A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
              A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
            END DO
          END DO
        END DO
      END DO
      DO k=1,N(ng)
        DO j=JstrR,JendR
          DO i=Istr,IendR
            VnormU(i,j,k)=1.0_r8/                                       &
     &                    SQRT(cff*(A3dsqr(i,j,k)-                      &
     &                              cff*A3davg(i,j,k)*A3davg(i,j,k)))
          END DO
        END DO
      END DO
!
!  3D norm at V-points.
!
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          Hnorm(i,j)=1.0_r8
          DO k=1,N(ng)
            A3davg(i,j,k)=0.0_r8
            A3dsqr(i,j,k)=0.0_r8
            Vnorm(i,j,k)=1.0_r8
          END DO
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise3d (ng, IstrR, IendR, Jstr, JendR,              &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      Rscheme, A3d)
        DO j=JstrR,JendR
          DO i=Istr,IendR
            DO k=1,N(ng)
              A3d(i,j,k)=A3d(i,j,k)*SQRT(0.5_r8*(Hz(i,j-1,k)+Hz(i,j,k)))
            END DO
          END DO
        END DO
        CALL bcov_v3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), DTVdif(ng),                     &
     &                      Hdecay(ng), Vdecay(ng),                     &
     &                      Hnorm, Vnorm,                               &
     &                      pm, pn, pmon_p, pnom_r,                     &
#  ifdef MASKING
     &                      pmask,                                      &
#  endif
     &                      Hz, z_r,                                    &
     &                      A3d)
        DO k=1,N(ng)
          DO j=Jstr,JendR
            DO i=IstrR,IendR
              A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
              A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
            END DO
          END DO
        END DO
      END DO
      DO k=1,N(ng)
        DO j=Jstr,JendR
          DO i=IstrR,IendR
            VnormV(i,j,k)=1.0_r8/                                       &
     &                    SQRT(cff*(A3dsqr(i,j,k)-                      &
     &                              cff*A3davg(i,j,k)*A3davg(i,j,k)))
          END DO
        END DO
      END DO
!
!  3D norm at RHO-points.
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
          Hnorm(i,j)=1.0_r8
          DO k=1,N(ng)
            A3davg(i,j,k)=0.0_r8
            A3dsqr(i,j,k)=0.0_r8
            Vnorm(i,j,k)=1.0_r8
          END DO
        END DO
      END DO
      DO iter=1,Nrandom
        CALL white_noise3d (ng, IstrR, IendR, JstrR, JendR,             &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      Rscheme, A3d)
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            DO k=1,N(ng)
              A3d(i,j,k)=A3d(i,j,k)*SQRT(Hz(i,j,k))
            END DO
          END DO
        END DO
        CALL bcov_r3d_tile (ng, iADM, Istr, Iend, Jstr, Jend,           &
     &                      LBi, UBi, LBj, UBj, 1, N(ng),               &
     &                      NghostPoints,                               &
     &                      DTHdif(ng), DTVdif(ng),                     &
     &                      Hdecay(ng), Vdecay(ng),                     &
     &                      Hnorm, Vnorm,                               &
     &                      pm, pn, pmon_u, pnom_v,                     &
# ifdef MASKING
     &                      umask, vmask,                               &
# endif
     &                      Hz, z_r,                                    &
     &                      A3d)
        DO k=1,N(ng)
          DO j=JstrR,JendR
            DO i=IstrR,IendR
              A3davg(i,j,k)=A3davg(i,j,k)+A3d(i,j,k)
              A3dsqr(i,j,k)=A3dsqr(i,j,k)+A3d(i,j,k)*A3d(i,j,k)
            END DO
          END DO
        END DO
      END DO
      DO k=1,N(ng)
        DO j=JstrR,JendR
          DO i=IstrR,IendR
            val=1.0_r8/SQRT(cff*(A3dsqr(i,j,k)-                         &
     &                           cff*A3davg(i,j,k)*A3davg(i,j,k)))
            DO itrc=1,NT(ng)
              VnormR(i,j,k,itrc)=val
            END DO
          END DO
        END DO
      END DO
# endif

      RETURN
      END SUBROUTINE randomization_tile

#endif
      END MODULE randomization_mod
