#include "cppdefs.h"

      MODULE tl_bcov_2d_mod

#if defined S4DVAR || defined IS4DVAR || defined W4DVAR
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routines compute the background error covariance for data      !
!  assimilation via a space convolution of the diffusion equation      !
!  for 2D state variables. The diffusion equation is solved using      !
!  an explicit (inefficient) algorithm.                                !
!                                                                      !
!  This routines are used to  compute the isotropic normalization      !
!  factor  Hnorm via a randomization algorithm and to compute the      !
!  the background error covariance.  The norm factors are used to      !
!  ensure that the  diagonal elements of the covariance are equal      !
!  to unity.                                                           !
!                                                                      !
!  For Gaussian (bell-shaped) covariances,  the space convolution      !
!  of the diffusion operator is an efficient way  to estimate the      !
!  finite domain error covariances.  The  number  of  convolution      !
!  steps determined from the values of Hdecay, DTHdif,  and Kdif.      !
!  Here,  Kdif  is a diffusion coefficient,  assumed constant and      !
!  with a value of 1 m2/s. The number of convolution steps Nsteps      !
!  is forced to be an even number because  its associated adjoint      !
!  operator needs to be convoluted for half of the steps only.         !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     Istr       Starting tile index in the I-direction.               !
!     Iend       Ending   tile index in the I-direction.               !
!     Jstr       Starting tile index in the J-direction.               !
!     Jend       Ending   tile index in the J-direction.               !
!     LBi        I-dimension Lower bound.                              !
!     UBi        I-dimension Upper bound.                              !
!     LBj        J-dimension Lower bound.                              !
!     UBj        J-dimension Upper bound.                              !
!     Nghost     Number of ghost points.                               !
!     DTHdif     Time-step for horizontal diffusion convolution (s).   !
!     Hdecay     Horizontal Gaussian decorrelation scale (m).          !
!     Hnorm      Horizontal normalization factors.                     !
!     tl_A       2D tangent linear state variable to diffuse.          !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     tl_A       Diffused 2D tangent linear state variable.            !      
!                                                                      !
!  Routines:                                                           !
!                                                                      !
!    tl_bcov_r2d_tile  Background error covariance at RHO-points       !
!    tl_bcov_u2d_tile  Background error covariance at U-points         !
!    tl_bcov_v2d_tile  Background error covariance at V-points         !
!                                                                      !
!=======================================================================
!
      implicit none

      PUBLIC

      CONTAINS
!
!***********************************************************************
      SUBROUTINE tl_bcov_r2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_u, pnom_v,              &
# ifdef MASKING
     &                             umask, vmask,                        &
# endif
     &                             tl_A)
!***********************************************************************
!
      USE mod_param
!
      USE bc_2d_mod, ONLY: bc_r2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_u(LBi:,LBj:)
      real(r8), intent(in) :: pnom_v(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: umask(LBi:,LBj:)
      real(r8), intent(in) :: vmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: tl_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_v(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: umask(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: vmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: tl_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: tl_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FX

# include "set_bounds.h"
!
      ILB=LBOUND(tl_Awrk,DIM=1)
      IUB=UBOUND(tl_Awrk,DIM=1)
      JLB=LBOUND(tl_Awrk,DIM=2)
      JUB=UBOUND(tl_Awrk,DIM=2)
!
!-----------------------------------------------------------------------
!  Space convolution of the diffusion equation for a 2D state variable
!  at RHO-points
!-----------------------------------------------------------------------
!
!  Set integration indices and initial conditions.
!
      Nold=1
      Nnew=2
      DO j=Jstr-1,Jend+1
        DO i=Istr-1,Iend+1
!>        Awrk(i,j,Nold)=A(i,j)*Hnorm(i,j)
!>
          tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
        END DO
      END DO
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
!
!  Integrate diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Compute XI- and ETA-components of diffusive flux.
!
        DO j=Jstr,Jend
          DO i=Istr,Iend+1
!>          FX(i,j)=pmon_u(i,j)*                                        &
!>   &              (Awrk(i,j,Nold)-Awrk(i-1,j,Nold))
!>
            tl_FX(i,j)=pmon_u(i,j)*                                     &
     &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i-1,j,Nold))
# ifdef MASKING
!>          FX(i,j)=FX(i,j)*umask(i,j)
!>
            tl_FX(i,j)=tl_FX(i,j)*umask(i,j)
# endif
          END DO
        END DO
        DO j=Jstr,Jend+1
          DO i=Istr,Iend
!>          FE(i,j)=pnom_v(i,j)*                                        &
!>   &              (Awrk(i,j,Nold)-Awrk(i,j-1,Nold))
!>
            tl_FE(i,j)=pnom_v(i,j)*                                     &
     &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i,j-1,Nold))
# ifdef MASKING
!>          FE(i,j)=FE(i,j)*vmask(i,j)
!>
            tl_FE(i,j)=tl_FE(i,j)*vmask(i,j)
# endif
          END DO
        END DO
!
!  Time-step harmonic diffusion operator.
!
        DO j=Jstr,Jend
          DO i=Istr,Iend
!>          Awrk(i,j,Nnew)=Awrk(i,j,Nold)+                              &
!>   &                     DTHdif*pm(i,j)*pn(i,j)*                      &
!>   &                     (FX(i+1,j)-FX(i,j)+                          &
!>   &                      FE(i,j+1)-FE(i,j))
!>
            tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
     &                        DTHdif*pm(i,j)*pn(i,j)*                   &
     &                        (tl_FX(i+1,j)-tl_FX(i,j)+                 &
     &                         tl_FE(i,j+1)-tl_FE(i,j))
          END DO
        END DO
!
!  Apply boundary conditions. If applicable, exchange boundary data.
!
!>      CALL bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    Awrk(:,:,Nnew))
!>
        CALL bc_r2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    ILB, IUB, JLB, JUB,                           &
     &                    tl_Awrk(:,:,Nnew))
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      Awrk(:,:,Nnew))
!>
        CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
     &                      ILB, IUB, JLB, JUB,                         &
     &                      Nghost, EWperiodic, NSperiodic,             &
     &                      tl_Awrk(:,:,Nnew))
# endif
!
!  Update integration indices.
!
        Nsav=Nold
        Nold=Nnew
        Nnew=Nsav
      END DO
!
!  Load solution.
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
!>        A(i,j)=Awrk(i,j,Nold)
!>
          tl_A(i,j)=tl_Awrk(i,j,Nold)
        END DO
      END DO

      RETURN
      END SUBROUTINE tl_bcov_r2d_tile
!
!***********************************************************************
      SUBROUTINE tl_bcov_u2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_r, pnom_p,              &
# ifdef MASKING
     &                             pmask,                               &
# endif
     &                             tl_A)
!***********************************************************************
!
      USE mod_param
!
      USE bc_2d_mod, ONLY: bc_u2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_r(LBi:,LBj:)
      real(r8), intent(in) :: pnom_p(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: tl_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_r(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_p(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: tl_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: tl_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FX

# include "set_bounds.h"
!
      ILB=LBOUND(tl_Awrk,DIM=1)
      IUB=UBOUND(tl_Awrk,DIM=1)
      JLB=LBOUND(tl_Awrk,DIM=2)
      JUB=UBOUND(tl_Awrk,DIM=2)
!
!-----------------------------------------------------------------------
!  Space convolution of the diffusion equation for a 2D state variable
!  at U-points
!-----------------------------------------------------------------------
!
!  Set integration indices and initial conditions.
!
      Nold=1
      Nnew=2
      DO j=Jstr-1,Jend+1
        DO i=IstrU-1,Iend+1
!>        Awrk(i,j,Nold)=A(i,j)*Hnorm(i,j)
!>
          tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
        END DO
      END DO
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
!
!  Integrate diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Compute XI- and ETA-components of diffusive flux.
!
        DO j=Jstr,Jend
          DO i=IstrU-1,Iend
!>          FX(i,j)=pmon_r(i,j)*                                        &
!>   &              (Awrk(i+1,j,Nold)-Awrk(i,j,Nold))
!>
            tl_FX(i,j)=pmon_r(i,j)*                                     &
     &                 (tl_Awrk(i+1,j,Nold)-tl_Awrk(i,j,Nold))
          END DO
        END DO
        DO j=Jstr,Jend+1
          DO i=IstrU,Iend
!>          FE(i,j)=pnom_p(i,j)*                                        &
!>   &              (Awrk(i,j,Nold)-Awrk(i,j-1,Nold))
!>
            tl_FE(i,j)=pnom_p(i,j)*                                     &
     &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i,j-1,Nold))
# ifdef MASKING
!>          FE(i,j)=FE(i,j)*pmask(i,j)
!>
            tl_FE(i,j)=tl_FE(i,j)*pmask(i,j)
# endif
          END DO
        END DO
!
!  Time-step harmonic diffusion operator.
!
        cff=DTHdif*0.25_r8
        DO j=Jstr,Jend
          DO i=IstrU,Iend
!>          Awrk(i,j,Nnew)=Awrk(i,j,Nold)+                              &
!>   &                     cff*(pm(i-1,j)+pm(i,j))*                     &
!>   &                         (pn(i-1,j)+pn(i,j))*                     &
!>   &                     (FX(i,j)-FX(i-1,j)+                          &
!>   &                      FE(i,j+1)-FE(i,j))
!>
            tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
     &                        cff*(pm(i-1,j)+pm(i,j))*                  &
     &                            (pn(i-1,j)+pn(i,j))*                  &
     &                        (tl_FX(i,j)-tl_FX(i-1,j)+                 &
     &                         tl_FE(i,j+1)-tl_FE(i,j))
          END DO
        END DO
!
!  Apply boundary conditions. If applicable, exchange boundary data.
!
!>      CALL bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    Awrk(:,:,Nnew))
!>
        CALL bc_u2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    ILB, IUB, JLB, JUB,                           &
     &                    tl_Awrk(:,:,Nnew))
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      Awrk(:,:,Nnew))
!>
        CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
     &                      ILB, IUB, JLB, JUB,                         &
     &                      Nghost, EWperiodic, NSperiodic,             &
     &                      tl_Awrk(:,:,Nnew))
# endif
!
!  Update integration indices.
!
        Nsav=Nold
        Nold=Nnew
        Nnew=Nsav
      END DO
!
!  Load solution.
!
      DO j=JstrR,JendR
        DO i=IstrU-1,IendR
!>        A(i,j)=Awrk(i,j,Nold)
!>
          tl_A(i,j)=tl_Awrk(i,j,Nold)
        END DO
      END DO

      RETURN
      END SUBROUTINE tl_bcov_u2d_tile
!
!***********************************************************************
      SUBROUTINE tl_bcov_v2d_tile (ng, model, Istr, Iend, Jstr, Jend,   &
     &                             LBi, UBi, LBj, UBj, Nghost,          &
     &                             DTHdif, Hdecay,                      &
     &                             Hnorm,                               &
     &                             pm, pn, pmon_p, pnom_r,              &
# ifdef MASKING
     &                             pmask,                               &
# endif
     &                             tl_A)
!***********************************************************************
!
      USE mod_param
!
      USE bc_2d_mod, ONLY: bc_v2d_tile
# ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange2d
# endif
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Nghost

      real(r8), intent(in) :: DTHdif, Hdecay
!
# ifdef ASSUMED_SHAPE
      real(r8), intent(in) :: pm(LBi:,LBj:)
      real(r8), intent(in) :: pn(LBi:,LBj:)
      real(r8), intent(in) :: pmon_p(LBi:,LBj:)
      real(r8), intent(in) :: pnom_r(LBi:,LBj:)
#  ifdef MASKING
      real(r8), intent(in) :: pmask(LBi:,LBj:)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:,LBj:)
      real(r8), intent(inout) :: tl_A(LBi:,LBj:)
# else
      real(r8), intent(in) :: pm(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pmon_p(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: pnom_r(LBi:UBi,LBj:UBj)
#  ifdef MASKING
      real(r8), intent(in)  :: pmask(LBi:UBi,LBj:UBj)
#  endif
      real(r8), intent(in) :: Hnorm(LBi:UBi,LBj:UBj)
      real(r8), intent(inout) :: tl_A(LBi:UBi,LBj:UBj)
# endif
!
!  Local variable declarations.
!
# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: ILB, IUB, JLB, JUB
      integer :: Nnew, Nold, Nsav, Nsteps, i, j, step

      real(r8) :: cff

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY,2) :: tl_Awrk

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FE
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: tl_FX

# include "set_bounds.h"
!
      ILB=LBOUND(tl_Awrk,DIM=1)
      IUB=UBOUND(tl_Awrk,DIM=1)
      JLB=LBOUND(tl_Awrk,DIM=2)
      JUB=UBOUND(tl_Awrk,DIM=2)
!
!-----------------------------------------------------------------------
!  Space convolution of the diffusion equation for a 2D state variable
!  at V-points
!-----------------------------------------------------------------------
!
!  Set integration indices and initial conditions.
!
      Nold=1
      Nnew=2
      DO j=JstrV-1,Jend+1
        DO i=Istr-1,Iend+1
!>        Awrk(i,j,Nold)=A(i,j)*Hnorm(i,j)
!>
          tl_Awrk(i,j,Nold)=tl_A(i,j)*Hnorm(i,j)
        END DO
      END DO
!
!  Determine number of integration steps as a function of the spatial
!  decorrelation scale, Hdecay.  A diffusion coefficient of 1 m2/s is
!  assumed.
!
      Nsteps=NINT(Hdecay*Hdecay/(4.0_r8*DTHdif))
      IF (MOD(Nsteps,2).ne.0) Nsteps=Nsteps+1
!
!  Integrate diffusion-type operator.
!
      DO step=1,Nsteps
!
!  Compute XI- and ETA-components of diffusive flux.
!
        DO j=JstrV,Jend
          DO i=Istr,Iend+1
!>          FX(i,j)=pmon_p(i,j)*                                        &
!>   &              (Awrk(i,j,Nold)-Awrk(i-1,j,Nold))
!>
            tl_FX(i,j)=pmon_p(i,j)*                                     &
     &                 (tl_Awrk(i,j,Nold)-tl_Awrk(i-1,j,Nold))

# ifdef MASKING
!>          FX(i,j)=FX(i,j)*pmask(i,j)
!>
            tl_FX(i,j)=tl_FX(i,j)*pmask(i,j)
# endif
          END DO
        END DO
        DO j=JstrV-1,Jend
          DO i=Istr,Iend
!>          FE(i,j)=pnom_r(i,j)*                                        &
!>   &              (Awrk(i,j+1,Nold)-Awrk(i,j,Nold))
!>
            tl_FE(i,j)=pnom_r(i,j)*                                     &
     &                 (tl_Awrk(i,j+1,Nold)-tl_Awrk(i,j,Nold))
          END DO
        END DO
!
!  Time-step harmonic diffusion operator.
!
        cff=DTHdif*0.25_r8
        DO j=JstrV,Jend
          DO i=Istr,Iend
!>          Awrk(i,j,Nnew)=Awrk(i,j,Nold)+                              &
!>   &                     cff*(pm(i,j-1)+pm(i,j))*                     &
!>   &                         (pn(i,j-1)+pn(i,j))*                     &
!>   &                     (FX(i+1,j)-FX(i,j)+                          &
!>   &                      FE(i,j)-FE(i,j-1))
!>
            tl_Awrk(i,j,Nnew)=tl_Awrk(i,j,Nold)+                        &
     &                        cff*(pm(i,j-1)+pm(i,j))*                  &
     &                            (pn(i,j-1)+pn(i,j))*                  &
     &                        (tl_FX(i+1,j)-tl_FX(i,j)+                 &
     &                         tl_FE(i,j)-tl_FE(i,j-1))
          END DO
        END DO
!
!  Apply boundary conditions. If applicable, exchange boundary data.
!
!>      CALL bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
!>   &                    ILB, IUB, JLB, JUB,                           &
!>   &                    Awrk(:,:,Nnew))
!>
        CALL bc_v2d_tile (ng, Istr, Iend, Jstr, Jend,                   &
     &                    ILB, IUB, JLB, JUB,                           &
     &                    tl_Awrk(:,:,Nnew))
# ifdef DISTRIBUTE
!>      CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
!>   &                      ILB, IUB, JLB, JUB,                         &
!>   &                      Nghost, EWperiodic, NSperiodic,             &
!>   &                      Awrk(:,:,Nnew))
!>
        CALL mp_exchange2d (ng, model, 1, Istr, Iend, Jstr, Jend,       &
     &                      ILB, IUB, JLB, JUB,                         &
     &                      Nghost, EWperiodic, NSperiodic,             &
     &                      tl_Awrk(:,:,Nnew))
# endif
!
!  Update integration indices.
!
        Nsav=Nold
        Nold=Nnew
        Nnew=Nsav
      END DO
!
!  Load solution.
!
      DO j=JstrV-1,JendR
        DO i=IstrR,IendR
!>        A(i,j)=Awrk(i,j,Nold)
!>
          tl_A(i,j)=tl_Awrk(i,j,Nold)
        END DO
      END DO

      RETURN
      END SUBROUTINE tl_bcov_v2d_tile
#endif
      END MODULE tl_bcov_2d_mod
