#include "cppdefs.h"
      MODULE utility_mod
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!=======================================================================
!                                                                      !
!  This module contains several all purpuse generic routines:          !
!                                                                      !
!  Routines:                                                           !
!                                                                      !
!    white_noise2d   2D array of random numbers.                       !
!    white_noise3d   2D array of random numbers.                       !
!    gasdev          Gaussian distributed deviates with zero mean      !
!                      and unit deviates.                              !
!    rand1           Get a uniform deviate between 0 and 1.            !
!    nrng            NSWC Gaussian random number generator.            !
!    urng            NSWC uniform random number generator.             !
!                                                                      !
!=======================================================================
!
      USE mod_kinds

      implicit none

      PRIVATE

      PUBLIC :: white_noise2d
# ifdef SOLVE3D
      PUBLIC :: white_noise3d
# endif
      PUBLIC :: nrng

      CONTAINS

      SUBROUTINE white_noise2d (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          Rscheme,                                &
     &                          R)
!
!=======================================================================
!                                                                      !
!  This routine generates a 2D "white noise" array with random         !
!  numbers over specified range.  These numbers  are scaled to         !
!  insure its insure its mean expectation to be  approximately         !
!  zero and a unity variance.                                          !
!                                                                      !
!  Several random number generation schemes are allowed:               !
!                                                                      !
!  Rscheme = 0     Use F90 intrinsic random numbers generator,         !
!                    0 <= R < 1                                        !
!  Rscheme = 1     Random deviates with Gaussian distribution,         !
!                    -1 < R < 1                                        !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Rscheme

      real(r8), intent(out) :: R(LBi:UBi,LBj:UBj)
!
!  Local variable declarations.
!
      integer :: i, j, seed

      real(r8), parameter :: fac = 2.0_r8 * 1.73205080756887720_r8

      real(r8) :: gasdev
!
!-----------------------------------------------------------------------
!  Generate an array of random numbers.
!-----------------------------------------------------------------------
!
!  F90 intrinsic pseudorandom numbers with an uniform distribution
!  over the range 0 <= R < 1.
!
      IF (Rscheme.eq.0) THEN
        CALL random_number (R(LBi:,LBj:))
!
!  Random deviates with a Gaussian (normal) distribuiton over the
!  range from -1 to 1.
!
      ELSE IF (Rscheme.eq.1) THEN
        seed=-2*(MIN(1,Nrun))
        DO i=Jstr,Jend
          DO j=Istr,Iend
            R(i,j)=gasdev(seed)
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Scale (fac=2*SQRT(3)) the random number to insure the expectation
!  mean to be approximately zero, E(R)=0, and the expectation variance
!  to be approximately unity, E(R^2)=1 (Bennett, 2002; book page 72).
!
!  Recall that,
!
!      E(R) = sum[R f(R)]
!
!  where F(R) is the random variable probability distribution.
!
!-----------------------------------------------------------------------
!
      DO j=Jstr,Jend
        DO i=Istr,Iend
          R(i,j)=fac*(R(i,j)-0.5_r8)
        END DO
      END DO

      RETURN
      END SUBROUTINE white_noise2d

#ifdef SOLVE3D
      SUBROUTINE white_noise3d (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj, LBk, UBk,           &
     &                          Rscheme,                                &
     &                          R)
!
!=======================================================================
!                                                                      !
!  This routine generates a 3D "white noise" array with random         !
!  numbers over specified range.  These numbers  are scaled to         !
!  insure its insure its mean expectation to be  approximately         !
!  zero and a unity variance.                                          !
!                                                                      !
!  Several random number generation schemes are allowed:               !
!                                                                      !
!  Rscheme = 0     Use F90 intrinsic random numbers generator,         !
!                   0 <= R < 1                                         !
!  Rscheme = 1     Random deviates with Gaussian distribution,         !
!                   -1 < R < 1                                         !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_scalars
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj, LBk, UBk
      integer, intent(in) :: Rscheme

      real(r8), intent(out) :: R(LBi:UBi,LBj:UBj,LBk:UBk)
!
!  Local variable declarations.
!
      integer :: i, j, k, seed

      real(r8), parameter :: fac = 2.0_r8 * 1.73205080756887720_r8

      real(r8) :: gasdev
!
!-----------------------------------------------------------------------
!  Generate an array of random numbers.
!-----------------------------------------------------------------------
!
!  F90 intrinsic pseudorandom numbers with an uniform distribution
!  over the range 0 <= R < 1.
!
      IF (Rscheme.eq.0) THEN
        CALL random_number (R(LBi:,LBj:,LBk:))
!
!  Random deviates with a Gaussian (normal) distribuiton over the
!  range from -1 to 1.
!
      ELSE IF (Rscheme.eq.1) THEN
        seed=-2*(MIN(1,Nrun))
        DO k=LBk,UBk
          DO i=Jstr,Jend
            DO j=Istr,Iend
              R(i,j,k)=gasdev(seed)
            END DO
          END DO
        END DO
      END IF
!
!-----------------------------------------------------------------------
!  Scale (fac=2*SQRT(3)) the random number to insure the expectation
!  mean to be approximately zero, E(R)=0, and the expectation variance
!  to be approximately unity, E(R^2)=1.
!
!  Recall that,
!
!      E(R) = sum[R f(R)]
!
!  where F(R) is the random variable probability distribution
!
!-----------------------------------------------------------------------
!
      DO k=LBk,UBk
        DO j=Jstr,Jend
          DO i=Istr,Iend
            R(i,j,k)=fac*(R(i,j,k)-0.5_r8)
          END DO
        END DO
      END DO

      RETURN
      END SUBROUTINE white_noise3d
#endif

      FUNCTION gasdev (seed)
!
!=======================================================================
!                                                                      !
!  This function returns a normaly distributed deviate with zero mean  !
!  and unit variance, using RAN1 as the source of uniform deviates.    !
!                                                                      !
!  Reference:                                                          !
!                                                                      !
!    Press, W.H., B.P. Flannery, S.A. Teukolsky, and W.T. Vetterling,  !
!      1989: Numerical Recipes: The Art of Scientific computing,       !
!      p 203.                                                          !
!                                                                      !
!=======================================================================
!
!  Imported variable declarations.
!
      integer, intent(inout) :: seed
!
!  Local variable declarations.
!
      integer, save :: iset = 0

      real(r8), save :: gset

      real(r8) :: fac, r, v1, v2

      real(r8) :: ran1

      real(r8) :: gasdev
!
!-----------------------------------------------------------------------
!  Get a Gaussian distributed deviated with zero mean and unit variance.
!-----------------------------------------------------------------------
!
      IF (iset.eq.0) THEN
!
!  If an extra deviated is not available, pick two uniform numbers in
!  the square extending from -1 to 1 in each direction.
!
        v1=2.0_r8*ran1(seed)-1.0_r8
        v2=2.0_r8*ran1(seed)-1.0_r8
        r=v1*v1+v2*v2
!
!  Check if they are unit circle, and if they are not, try again.
!
        DO WHILE (r.ge.1.0_r8)
          v1=2.0_r8*ran1(seed)-1.0_r8
          v2=2.0_r8*ran1(seed)-1.0_r8
          r=v1*v1+v2*v2
        END DO
!
!  Now make the Box-Muller transformation to get two Gaussian
!  (normal) deviates.  Return one and save the other for next
!  time.
!
        fac=SQRT(-2.0_r8*LOG(r)/r)
        gset=v1*fac
        gasdev=v2*fac
        iset=1
      ELSE
!
!  There is an extra deviate handly, so return it, and unset the
!  seach flag.
!
        gasdev=gset
        iset=0
      END IF

      RETURN
      END FUNCTION gasdev

      FUNCTION ran1 (seed)
!
!=======================================================================
!                                                                      !
!  This function returns a uniform deviate between 0 and 1. Set SEED   !
!  to any negative integer value to initialize  or  reinitialize the   !
!  sequence.                                                           !   
!                                                                      !
!  Reference:                                                          !
!                                                                      !
!    Press, W.H., B.P. Flannery, S.A. Teukolsky, and W.T. Vetterling,  !
!      1989: Numerical Recipes: The Art of Scientific computing,       !
!      p 196.                                                          !
!                                                                      !
!=======================================================================
!
!  Imported variable declarations.
!
      integer, intent(inout) :: seed
!
!  Local variable declarations.
!
      integer, parameter :: m1 = 259200
      integer, parameter :: m2 = 134456
      integer, parameter :: m3 = 243000
      integer, parameter :: ia1 = 7141
      integer, parameter :: ia2 = 8121
      integer, parameter :: ia3 = 4561
      integer, parameter :: ic1 = 54773
      integer, parameter :: ic2 = 28411
      integer, parameter :: ic3 = 51349

      integer, save :: iff = 0

      integer :: i, ix1, ix2, ix3
!
      real(r8), parameter :: rm1 = 1.0_r8 / REAL(m1,r8)
      real(r8), parameter :: rm2 = 1.0_r8 / REAL(m2,r8)
      
      real(r8), dimension(97) :: r

      real(r8) :: ran1
!
!-----------------------------------------------------------------------
!  Get a uniform random deviate between 0 and 1.
!-----------------------------------------------------------------------
!
!  Initialize on first call even if SEED is not negative.
!
      IF ((seed.lt.0).or.(iff.eq.0)) THEN
        iff=1
!
!  Seed teh first routine and use it to seed the second and third.
!
        ix1=MOD(ic1-seed,m1)
        ix1=MOD(ia1*ix1+ic1,m1)
        ix2=MOD(ix1,m2)
        ix1=MOD(ia1*ix1+ic1,m1)
        ix3=MOD(ix1,m3)
!
!  Fill the tabke with sequential uniform deviates generated by the
!  first two routines. Combine low- and high-order pieces.
!
        DO i=1,97
          ix1=MOD(ia1*ix1+ic1,m1)
          ix2=MOD(ia2*ix2+ic2,m2)
          r(i)=(REAL(ix1,r8)+REAL(ix2,r8)*rm2)*rm1
        END DO
        seed=1
      END IF
!
!  Except when initializing, this is where we start. Generate the
!  next number for each sequence.
!
      ix1=MOD(ia1*ix1+ic1,m1)
      ix2=MOD(ia2*ix2+ic2,m2)
      ix3=MOD(ia3*ix3+ic3,m3)
!
!  Use the third sequence to get an integer between 1 and 91.
!
      i=1+(97*ix3)/m3
      IF ((i.gt.97).or.(i.lt.1)) STOP
!
!  Return that table entry, and refill it.
!
      ran1=r(i)
      r(i)=(REAL(ix1,r8)+REAL(ix2,r8)*rm2)*rm1

      RETURN
      END FUNCTION ran1

      SUBROUTINE nrng (ix, a, n, ierr)
!
!=======================================================================
!                                                                      !
!  Gaussian random-number generator from the NSWC Library. It calls    !
!  the NSWC uniform random-number generator, URNG.                     !
!                                                                      !
!  Modernised and included in ROMS by Mark Hadfield, NIWA.             !
!                                                                      !
!=======================================================================
!
!  Imported variable declarations.
!
      integer, intent(in) :: n

      integer, intent(inout) :: ix

      integer, intent(out) :: ierr

#ifdef ASSUMED_SHAPE
      real(r8), intent(out) :: a(:)
#else
      real(r8), intent(out) :: a(n)
#endif
!
!  Local variable declarations.
!
      integer :: i, m

      real(r8), parameter ::pi2 = 6.2831853071796_r8

      real(r8) :: phi, r
      real(r8) :: temp(1)
!
!-----------------------------------------------------------------------
!  Generate Gaussian random numbers.
!-----------------------------------------------------------------------
!
      CALL urng (ix, a, n, ierr)
!
      IF (ierr.ne.0) RETURN
!
      IF (n.gt.1) THEN
        m=n/2
        m=m+m
        DO i=1,m,2
          r=SQRT(-2.0_r8*LOG(a(i)))
          phi=pi2*a(i+1)
          a(i  )=r*COS(phi)
          a(i+1)=r*SIN(phi)
        END DO
         IF (m.eq.n) RETURN
      END IF
!
      CALL urng (ix, temp, 1, ierr)
!
      r=SQRT(-2.0_r8*LOG(a(n)))
!
      a(n)=r*COS(pi2*temp(1))
!
      RETURN
      END SUBROUTINE nrng

      SUBROUTINE urng (ix, x, n, ierr)
!
!=======================================================================
!                                                                      !
!  Uniform random-number generator from the NSWC Library               !
!                                                                      !
!  Uses the recursion ix = ix*a mod p, where 0 < ix < p                !
!                                                                      !
!  Written by Linus Schrage, University of Chicago. Adapted for NSWC   !
!  Library by A. H. Morris. Modernised & included in ROMS by Mark      !
!  Hadfield, NIWA.                                                     !
!                                                                      !
!=======================================================================
!
!  Imported variable declarations.
!
      integer, intent(in) :: n

      integer, intent(inout) :: ix

      integer, intent(out) :: ierr

#ifdef ASSUMED_SHAPE
      real(r8), intent(out) :: x(:)
#else
      real(r8), intent(out) :: x(n)
#endif
!
!  Local variable declarations.
!    
      integer, parameter :: a = 16807          ! 7^5
      integer, parameter :: b15 = 32768        ! 2^15
      integer, parameter :: b16 = 65536        ! 2^16
      integer, parameter :: p = 2147483647     ! 2^31-1

      integer :: fhi, k, l, leftlo, xalo, xhi

      real(r8), parameter :: s = 0.465661E-09_r8
!
!-----------------------------------------------------------------------
!  Generate random numbers.
!-----------------------------------------------------------------------
!
      IF (n.le.0) THEN
        ierr=1
        RETURN
      END IF
      IF ((ix.le.0).or.(ix.ge.p)) THEN
        ierr=2
        RETURN
      END IF
!
      ierr=0
!
      DO l=1,n
!
! Get 15 high order bits of "ix".
!
        xhi=ix/b16
!
! Get 16 lower bits of ix and multiply with "a".
!
        xalo=(ix-xhi*b16)*a
!
! Get 15 high order bits of the product.
!
        leftlo=xalo/b16
!
! Form the 31 highest bits of "a*ix".
!
        fhi=xhi*a+leftlo
!
! Obtain the overflow past the 31st bit of "a*ix".
!
        k=fhi/b15
!
! Assemble all the parts and presubtract "p". The parentheses are
! essential.
!
        ix=(((xalo-leftlo*b16)-p)+(fhi-k*b15)*b16)+k
!
! Add "p" if necessary.
!
        IF (ix.lt.0) ix=ix+p
!
! Rescale "ix", to interpret it as a value between 0 and 1.
! the scale factor "s" is selected to be as near "1/p" as is
! appropriate in order that the floating value for "ix = 1",
! namely "s", be roughly the same distance from 0 as "(p-1)*s"
! is from 1. The current value for "s" assures us that "x(l)"
! is less than 1 for any floating point arithmetic of 6
! or more digits.
!
         x(l)=REAL(ix,r8)*s
      END DO
      RETURN
      END SUBROUTINE urng

      END MODULE utility_mod
