       SUBROUTINE ice_advect (ng, tile)
!
!*************************************************** W. Paul Budgell ***
!  Copyright (c) 2002 ROMS/TOMS Group                                  !
!************************************************** Hernan G. Arango ***
!                                                                      !
!  This subroutine performs advection of ice scalars using the         !
!  Smolarkiewicz second-order upwind scheme.                           !
!  Reference:                                                          !
!  Smolarkiewicz and Grabowski (1990)                                  !
!***********************************************************************
!

      USE mod_param
      USE mod_grid
      USE mod_ocean
      USE mod_ice
      USE mod_forces
      USE mod_stepping

#if defined EW_PERIODIC || defined NS_PERIODIC
      USE exchange_2d_mod, ONLY : exchange_r2d_tile
#endif
#ifdef DISTRIBUTE
      USE mp_exchange_mod, ONLY : mp_exchange2d
#endif
      USE aibc_mod, ONLY : aibc_tile
      USE hibc_mod, ONLY : hibc_tile
      USE hsnbc_mod, ONLY : hsnbc_tile
      USE tibc_mod, ONLY : tibc_tile
      USE sfwatbc_mod, ONLY : sfwatbc_tile

      implicit none

      integer, intent(in) :: ng, tile

# ifdef DISTRIBUTE
#  ifdef EW_PERIODIC
      logical :: EWperiodic=.TRUE.
#  else
      logical :: EWperiodic=.FALSE.
#  endif
#  ifdef NS_PERIODIC
      logical :: NSperiodic=.TRUE.
#  else
      logical :: NSperiodic=.FALSE.
#  endif
# endif
#include "tile.h"

#ifdef PROFILE
      CALL wclock_on (ng, iNLM, 49)
#endif
! ---------------------------------------------------------------------
!  Advect the ice concentration.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
#ifdef MASKING
     &                      GRID(ng) % rmask,                           &
#endif
#ifdef ICESHELF
     &                      GRID(ng) % zice,                            &
#endif
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % ai                                &
     &                      )
!
        CALL aibc_tile (ng, Istr, Iend, Jstr, Jend,                     &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%ai)
!
! ---------------------------------------------------------------------
!  Advect the ice thickness.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
#ifdef MASKING
     &                      GRID(ng) % rmask,                           &
#endif
#ifdef ICESHELF
     &                      GRID(ng) % zice,                            &
#endif
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % hi                                &
     &                      )
!
        CALL hibc_tile (ng, Istr, Iend, Jstr, Jend,                     &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hi)
!
! ---------------------------------------------------------------------
!  Advect the snow thickness.
! ---------------------------------------------------------------------
#ifdef ICE_THERMO
      CALL ice_advect_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
# ifdef MASKING
     &                      GRID(ng) % rmask,                           &
# endif
# ifdef ICESHELF
     &                      GRID(ng) % zice,                            &
# endif
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % hsn                               &
     &                      )
!
        CALL hsnbc_tile (ng, Istr, Iend, Jstr, Jend,                    &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%hsn)
!
! ---------------------------------------------------------------------
!  Advect the surface melt water.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
# ifdef MASKING
     &                      GRID(ng) % rmask,                           &
# endif
# ifdef ICESHELF
     &                      GRID(ng) % zice,                            &
# endif
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % sfwat                             &
     &                      )
!
        CALL sfwatbc_tile (ng, Istr, Iend, Jstr, Jend,                  &
     &                     LBi, UBi, LBj, UBj,                          &
     &                     liold(ng), linew(ng),                        &
     &                     ICE(ng)%ui,                                  &
     &                     ICE(ng)%vi,                                  &
     &                     ICE(ng)%sfwat)
!
! ---------------------------------------------------------------------
!  Advect the interior ice temperature.
! ---------------------------------------------------------------------
      CALL ice_advect_tile (ng, Istr, Iend, Jstr, Jend,                 &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      nrhs(ng), linew(ng), liold(ng), liunw(ng),  &
# ifdef MASKING
     &                      GRID(ng) % rmask,                           &
# endif
# ifdef ICESHELF
     &                      GRID(ng) % zice,                            &
# endif
     &                      GRID(ng) % on_u,                            &
     &                      GRID(ng) % om_v,                            &
     &                      GRID(ng) % omn,                             &
     &                      ICE(ng) % ui,                               &
     &                      ICE(ng) % vi,                               &
     &                      ICE(ng) % ti                                &
     &                      )
!
        CALL tibc_tile (ng, Istr, Iend, Jstr, Jend,                     &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          liold(ng), linew(ng),                   &
     &                          ICE(ng)%ui,                             &
     &                          ICE(ng)%vi,                             &
     &                          ICE(ng)%ti)
!
# if defined EW_PERIODIC || defined NS_PERIODIC
        CALL exchange_r2d_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%ai(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%hi(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%hsn(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%sfwat(:,:,linew(ng)))
        CALL exchange_r2d_tile (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          ICE(ng)%ti(:,:,linew(ng)))
# endif
# ifdef DISTRIBUTE
        CALL mp_exchange2d (ng, iNLM, 4, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      ICE(ng)%ai(:,:,linew(ng)),                  &
     &                      ICE(ng)%hi(:,:,linew(ng)),                  &
     &                      ICE(ng)%hsn(:,:,linew(ng)),                 &
     &                      ICE(ng)%sfwat(:,:,linew(ng)))
        CALL mp_exchange2d (ng, iNLM, 1, Istr, Iend, Jstr, Jend,        &
     &                      LBi, UBi, LBj, UBj,                         &
     &                      NghostPoints, EWperiodic, NSperiodic,       &
     &                      ICE(ng)%ti(:,:,linew(ng)))
# endif
#endif
#ifdef PROFILE
      CALL wclock_off (ng, iNLM, 49)
#endif
      RETURN
      END SUBROUTINE ice_advect
!
!==========================================================================!
      SUBROUTINE ice_advect_tile (ng, Istr, Iend, Jstr, Jend,           &
     &                        LBi, UBi, LBj, UBj,                       &
     &                        nrhs, linew, liold, liunw,                &
#ifdef MASKING
     &                        rmask,                                    &
#endif
#ifdef ICESHELF
     &                        zice,                                     &
#endif
     &                        on_u, om_v, omn,                          &
     &                        ui, vi, scr)
!==========================================================================!

      USE mod_param
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Iend, Istr, Jend, Jstr
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: nrhs, linew, liold, liunw

#ifdef ASSUMED_SHAPE
# ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:,LBj:)
# endif
# ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:,LBj:)
# endif
      real(r8), intent(in) :: on_u(LBi:,LBj:)
      real(r8), intent(in) :: om_v(LBi:,LBj:)
      real(r8), intent(in) :: omn(LBi:,LBj:)
      real(r8), intent(in) :: ui(LBi:,LBj:,:)
      real(r8), intent(in) :: vi(LBi:,LBj:,:)
      real(r8), intent(inout) :: scr(LBi:,LBj:,:)
#else
# ifdef MASKING
      real(r8), intent(in) :: rmask(LBi:UBi,LBj:UBj)
# endif
# ifdef ICESHELF
      real(r8), intent(in) :: zice(LBi:UBi,LBj:UBj)
# endif
      real(r8), intent(in) :: on_u(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: om_v(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: omn(LBi:UBi,LBj:UBj)
      real(r8), intent(in) :: ui(LBi:UBi,LBj:UBj,2)
      real(r8), intent(in) :: vi(LBi:UBi,LBj:UBj,2)
      real(r8), intent(inout) :: scr(LBi:UBi,LBj:UBj,2)
#endif

!
! Local variable definitions
!
      integer :: IstrR, IendR, JstrR, JendR, IstrU, JstrV
      integer :: i, j

      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: ar
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: aflxu
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: aflxv
      real(r8), dimension(PRIVATE_2D_SCRATCH_ARRAY) :: aif

      real(r8), parameter :: epsil = 1.0E-15_r8
      real(r8), parameter :: add = 3.0E+3_r8

      real(r8) :: fakt1
      real(r8) :: fakt2

      real(r8) :: aim1
      real(r8) :: ajm1
      real(r8) :: aim1jm1u
      real(r8) :: aim1jm1v
      real(r8) :: ajp1
      real(r8) :: aim1jp1
      real(r8) :: aip1
      real(r8) :: aip1jm1

      real(r8) :: rateu
      real(r8) :: ratev
      real(r8) :: rateyiu
      real(r8) :: ratexiv
      real(r8) :: ratiou
      real(r8) :: ratiov
      real(r8) :: ratioyiu
      real(r8) :: ratioxiv
      real(r8) :: uiv
      real(r8) :: viu
      real(r8) :: uspeed
      real(r8) :: vspeed

#include "set_bounds.h"

#ifdef EW_PERIODIC
# define I_RANGE Istr-1,Iend+2
#else
# define I_RANGE MAX(Istr-1,0),MIN(Iend+2,Lm(ng)+1)
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr-1,Jend+2
#else
# define J_RANGE MAX(Jstr-1,0),MIN(Jend+2,Mm(ng)+1)
#endif
      do j=J_RANGE
      do i=I_RANGE
        ar(i,j)=1.0_r8/omn(i,j)
      enddo
      enddo
#undef I_RANGE
#undef J_RANGE
!
! upstream:
!
#ifdef EW_PERIODIC
# define I_RANGE Istr-1,Iend+2
#else
# define I_RANGE MAX(Istr-1,1),MIN(Iend+2,Lm(ng)+1)
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr-1,Jend+2
#else
# define J_RANGE MAX(Jstr-1,1),MIN(Jend+2,Mm(ng)+1)
#endif
      do j=J_RANGE
      do i=I_RANGE
!
      aflxu(i,j)=max(0.0_r8,ui(i,j,liunw))*on_u(i,j)*scr(i-1,j,liold)   &
     &          +min(0.0_r8,ui(i,j,liunw))*on_u(i,j)*scr(i,j,liold)
      aflxv(i,j)=max(0.0_r8,vi(i,j,liunw))*om_v(i,j)*scr(i,j-1,liold)   &
     &          +min(0.0_r8,vi(i,j,liunw))*om_v(i,j)*scr(i,j,liold)
!
      enddo
      enddo
#undef I_RANGE
#undef J_RANGE
!
! step number 1 in mpdata:
!
#ifdef EW_PERIODIC
# define I_RANGE Istr-1,Iend+2
#else
# define I_RANGE MAX(Istr-1,1),MIN(Iend+1,Lm(ng))
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr-1,Jend+2
#else
# define J_RANGE MAX(Jstr-1,1),MIN(Jend+1,Mm(ng))
#endif
      do j=J_RANGE
      do i=I_RANGE
!
         aif(i,j)=(scr(i,j,liold)-dtice(ng)*(aflxu(i+1,j)-aflxu(i,j)    &
     &        +aflxv(i,j+1)-aflxv(i,j))*ar(i,j))
#ifdef MASKING
         aif(i,j) = aif(i,j)*rmask(i,j)
#endif
#ifdef ICESHELF
         IF (zice(i,j).ne.0.0_r8) THEN
            aif(i,j) = 0.0_r8
         ENDIF
#endif
!
      enddo
      enddo
#undef I_RANGE
#undef J_RANGE
!
! set values at the open boundaries
!
#define I_RANGE MAX(Istr-1,0),MIN(Iend+1,Lm(ng)+1)
      if (SOUTHERN_EDGE) then
           do i=I_RANGE
             aif(i,Jstr-1)=scr(i,Jstr-1,liold)
           enddo
      endif
        if (NORTHERN_EDGE) then
           do i=I_RANGE
             aif(i,Jend+1)=scr(i,Jend+1,liold)
           enddo
        endif
#undef I_RANGE
#define J_RANGE MAX(Jstr-1,0),MIN(Jend+1,Mm(ng)+1)
        if (WESTERN_EDGE) then
           do j=J_RANGE
             aif(Istr-1,j)=scr(Istr-1,j,liold)
        enddo
        endif
        if (EASTERN_EDGE) then
         do j=J_RANGE
             aif(Iend+1,j)=scr(Iend+1,j,liold)
           enddo
        endif
#undef J_RANGE
!
! mask
!
#ifdef EW_PERIODIC
# define I_RANGE Istr-1,Iend+2
#else
# define I_RANGE MAX(Istr-1,0),MIN(Iend+1,Lm(ng)+1)
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr-1,Jend+2
#else
# define J_RANGE MAX(Jstr-1,0),MIN(Jend+1,Mm(ng)+1)
#endif
#ifdef MASKING
      do j=J_RANGE
      do i=I_RANGE
        aif(i,j)=aif(i,j)*rmask(i,j)
      enddo
      enddo
#endif
#ifdef ICESHELF
      do j=J_RANGE
      do i=I_RANGE
         IF (zice(i,j).ne.0.0_r8) THEN
            aif(i,j) = 0.0_r8
         ENDIF
      enddo
      enddo
#endif

#undef I_RANGE
#undef J_RANGE
!
!
#ifndef ICE_UPWIND
!
!------------ antidiffusion -------------------------
!
# ifdef EW_PERIODIC
#  define I_RANGE Istr-1,Iend+2
# else
#  define I_RANGE MAX(Istr-1,1),MIN(Iend+2,Lm(ng)+1)
# endif
# ifdef NS_PERIODIC
#  define J_RANGE Jstr-1,Jend+2
# else
#  define J_RANGE MAX(Jstr-1,1),MIN(Jend+2,Mm(ng)+1)
# endif
      do j=J_RANGE
      do i=I_RANGE
!
!      if (rmask(i,j).gt.0.0_r8) then
!
      aim1=aif(i-1,j)
      if (rmask(i-1,j).eq.0.0_r8) then   !land to the west
        aim1=aif(i,j)
      end if
      ajm1=aif(i,j-1)
      if (rmask(i,j-1).eq.0.0_r8) then   !land to the south
        ajm1=aif(i,j)
      end if
      aim1jm1u=aif(i-1,j-1)
      aim1jm1v=aif(i-1,j-1)
      if (rmask(i-1,j-1).eq.0.0_r8) then !land to the southwest
        if (rmask(i-1,j).ne.0.0_r8) then
          aim1jm1u=aif(i-1,j)      !only used in da/dy in u-point
        elseif (rmask(i,j-1).ne.0.0_r8) then
          aim1jm1v=aif(i,j-1)      !only used in da/dx in v-point
        end if
!      end if
!
      rateu=(aif(i,j)-aim1)/(aif(i,j)+aim1+epsil)
      ratiou=0.5_r8*dtice(ng)*(pm(i,j)+pm(i-1,j))
      ratev=(aif(i,j)-ajm1)/(aif(i,j)+ajm1+epsil)
      ratiov=0.5_r8*dtice(ng)*(pn(i,j)+pn(i,j-1))
!
      if (NORTHERN_EDGE.and.j.eq.Jend+1) then
        rateyiu=(aim1+aif(i,j)-aim1jm1u-ajm1)                           &
     &         /(aim1+aif(i,j)+aim1jm1u+ajm1+epsil)
        ratioyiu=0.25_r8*dtice(ng)*(pm(i-1,j)+pm(i,j)                   &
     &                             +pm(i-1,j-1)+pm(i,j-1))
        viu=0.5_r8*(vi(i,j,liunw)+vi(i-1,j,liunw))
      else
        ajp1=aif(i,j+1)
        if (rmask(i,j+1).eq.0.0_r8) then  !land to the north
          ajp1=aif(i,j)
        end if
        aim1jp1=aif(i-1,j+1)
        if ((rmask(i-1,j+1).eq.0.0_r8).and.                             &
     &      (rmask(i-1,j).ne.0.0_r8)) then   !land to the northwest
          aim1jp1=aif(i-1,j)        !only used in da/dy in u-point
        end if
!
        rateyiu=(aim1jp1+ajp1-aim1jm1u-ajm1)                            &
     &         /(aim1jp1+ajp1+aim1jm1u+ajm1+epsil)
        ratioyiu=0.125_r8*dtice(ng)*                                    &
     &          (pn(i-1,j+1)+pn(i,j+1)+pn(i-1,j-1)+pn(i,j-1))
        viu=0.25_r8*(vi(i,j,liunw)+vi(i,j+1,liunw)+vi(i-1,j+1,liunw)    &
     &                            +vi(i-1,j,,liunw))
      endif
!
      if (EASTERN_EDGE.and.i.eq.Iend+1) then
        ratexiv=(aif(i,j)+ajm1-aim1-aim1jm1v)                           &
     &         /(aif(i,j)+ajm1+aim1+aim1jm1v+epsil)
        ratioxiv=0.25_r8*dtice(ng)*(pm(i,j)+pm(i,j-1)+pm(i-1,j)         &
     &                             +pm(i-1,j-1))
        uiv=0.5_r8*(ui(i,j,liunw)+uice(i,j-1))
      else
        aip1=aif(i+1,j)
        if (rmask(i+1,j).eq.0.0_r8) then  !land to the east
           aip1=aif(i,j)
        end if
        aip1jm1=aif(i+1,j-1)
        if ((rmask(i+1,j-1).eq.0.0_r8).and.                             &
     &      (rmask(i,j-1).ne.0.0_r8)) then   !land to the southeast
          aip1jm1=aif(i,j-1)        !only used in da/dx in v-point
        end if
!
        ratexiv=(aip1+aip1jm1-aim1-aim1jm1v)                            &
     &         /(aip1+aip1jm1+aim1+aim1jm1v+epsil)
        ratioxiv=0.125_r8*dtice(ng)*                                    &
     &              (pm(i+1,j)+pm(i+1,j-1)+pm(i-1,j)+pm(i-1,j-1))
        uiv=0.25_r8*(ui(i,j,liunw)+ui(i+1,j,liunw)+ui(i+1,j-1,liunw)    &
     &                            +ui(i,j-1),liunw)
      endif
!
      uspeed=((abs(ui(i,j,liunw))-ratiou*(ui(i,j,liunw)*ui(i,j,liunw))) &
     &                                  *rateu)                         &
     &       -(ui(i,j,liunw)*viu*ratioyiu*rateyiu)
      vspeed=((abs(vi(i,j,liunw))-ratiov*(vi(i,j,liunw)*vi(i,j,liunw))) &
     &                                  *ratev)                         &
     &       -(uiv*vi(i,j,liunw)*ratioxiv*ratexiv)
!
      aflxu(i,j)=max(0.0_r8,uspeed)*on_u(i,j)*aim1                      &
     &          +min(0.0_r8,uspeed)*on_u(i,j)*aif(i,j)
      aflxv(i,j)=max(0.0_r8,vspeed)*om_v(i,j)*ajm1                      &
     &          +min(0.0_r8,vspeed)*om_v(i,j)*aif(i,j)

      enddo
      enddo
# undef I_RANGE
# undef J_RANGE
!
! step number 2 in mpdata:
!
# ifdef EW_PERIODIC
#  define I_RANGE Istr-1,Iend+2
# else
#  define I_RANGE MAX(Istr-1,1),MIN(Iend+2,Lm(ng))
# endif
# ifdef NS_PERIODIC
#  define J_RANGE Jstr-1,Jend+2
# else
#  define J_RANGE MAX(Jstr-1,1),MIN(Jend+2,Mm(ng))
# endif
      DO j=J_RANGE
        DO i=I_RANGE
!
          aif(i,j)=(aif(i,j)-dtice(ng)*(aflxu(i+1,j)-aflxu(i,j)           &
     &          +aflxv(i,j+1)-aflxv(i,j))*ar(i,j))
# ifdef MASKING
          aif(i,j) = aif(i,j)*rmask(i,j)
# endif
# ifdef ICESHELF
          IF (zice(i,j).ne.0.0_r8) THEN
            aif(i,j) = 0.0_r8
          END IF
# endif
        END DO
      END DO

#endif /* !ICE_UPWIND */

      DO j=Jstr,Jend
        DO i=Istr,Iend
            scr(i,j,linew) = aif(i,j)
        END DO
      END DO
!
      RETURN
      END SUBROUTINE ice_advect_tile
