#include "cppdefs.h"
      SUBROUTINE get_state (ng, model, msg, ncname, IniRec, Tindex)
!
!=======================================================================
!  Copyright (c) 2005  ROMS/TOMS Adjoint Group                         !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine reads in requested model state from specified NetCDF   !
!  file. It is usually used to read initial conditions.                !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     msg        Message index for Mstate.                             !
!     ncname     Nonlinear initial conditions NetCDF file name.        !
!     IniRec     Nonlinear initial conditions time record to read.     !
!     Tindex     State variable time index to intialize.               !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_grid
      USE mod_iounits
#ifdef FOUR_DVAR
      USE mod_fourdvar
      USE mod_mixing
#endif
      USE mod_ncparam
      USE mod_netcdf
      USE mod_ocean
      USE mod_scalars
#if defined SEDIMENT || defined BBL_MODEL
      USE mod_sediment
#endif
#ifdef ICE_MODEL
      USE mod_ice
#endif
      USE mod_strings
#ifdef FILTERED_RST
      USE mod_filter, ONLY: FIL
#endif
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model, msg, Tindex

      integer, intent(inout) :: IniRec

      character (len=*), intent(in) :: ncname
!
!  Local variable declarations.
!
      logical, dimension(NV) :: get_var, have_var

      integer :: LBi, UBi, LBj, UBj
      integer :: InpRec, gtype, i, itrc, lstr, lend
      integer :: Nrec, ncINPid, nvatts, status, varid, vartype
      integer :: Vsize(4)

      integer :: nf_fread2d
#ifdef SOLVE3D
      integer :: nf_fread3d
#endif

      real(r8), parameter :: Fscl = 1.0_r8

      real(r8) :: INPtime, Fmax, Fmin, Tmax, time_scale

      character (len=6 ) :: string
      character (len=15) :: attnam, tvarnam
      character (len=40) :: tunits
!
!-----------------------------------------------------------------------
!  Determine variables to read and their availability.
!-----------------------------------------------------------------------
!
!  Set model identification string.
!
      IF (model.eq.iNLM) THEN
        string=' NLM: '                     ! nonlinear model
      ELSE IF (model.eq.iTLM) THEN
        string=' TLM: '                     ! tangent linear model
      ELSE IF (model.eq.iRPM) THEN
        string=' RPM: '                     ! representer model
      ELSE IF (model.eq.iADM) THEN
        string=' ADM: '                     ! adjoint model
      ELSE IF (model.eq.5) THEN
        string=' NRM: '                     ! correlations norm factor
      ELSE IF (model.eq.6) THEN
        string=' STD: '                     ! standard deviations
      ELSE IF (model.eq.7) THEN
        string=' FRC: '                     ! impulse forcing
      END IF
!
!  Determine variables to read.
!
      CALL checkvars (ng, model, ncname, string, Nrec, NV, tvarnam,     &
     &                get_var, have_var)
!
!  Set Vsize to zero to deactivate interpolation of input data to model
!  grid in "nf_fread2d" and "nf_fread3d".
!
      DO i=1,4
        Vsize(i)=0
      END DO
!
!-----------------------------------------------------------------------
!  Open input NetCDF file and check time variable.
!-----------------------------------------------------------------------
!
      LBi=LBOUND(GRID(ng)%h,DIM=1)
      UBi=UBOUND(GRID(ng)%h,DIM=1)
      LBj=LBOUND(GRID(ng)%h,DIM=2)
      UBj=UBOUND(GRID(ng)%h,DIM=2)
!
!  Open input NetCDF file.
!
      IF (InpThread) THEN
        status=nf_open(TRIM(ncname), nf_nowrite, ncINPid)
        IF (status.ne.nf_noerr) THEN
          WRITE (stdout,20) string, TRIM(ncname)
          exit_flag=2
          ioerror=status
          RETURN
        END IF
!
!  If using the latest time record from input NetCDF file as the
!  initialization record, find time record index to read and read
!  in model time.
!
        status=nf_inq_varid(ncINPid, TRIM(tvarnam), varid)
        IF (LastRec) THEN
          Tmax=-1.0_r8
          DO i=1,tsize
            status=nf_get_var1_TYPE(ncINPid,varid,i,INPtime)
            IF (status.ne.nf_noerr) THEN
              WRITE (stdout,30) string, TRIM(tvarnam), i, TRIM(ncname)
              exit_flag=2
              ioerror=status
              RETURN
            END IF
            IF (INPtime.gt.Tmax) THEN
              Tmax=INPtime
              IniRec=i
            END IF
          END DO
          INPtime=Tmax
          InpRec=IniRec
        ELSE
          IF ((IniRec.ne.0).and.(IniRec.gt.tsize)) THEN
            WRITE (stdout,40) string, IniRec, TRIM(ncname), tsize
            exit_flag=2
            RETURN
          END IF
          IF (IniRec.ne.0) THEN
            InpRec=IniRec
          ELSE
            InpRec=1
          END IF
          status=nf_get_var1_TYPE(ncINPid, varid, InpRec, INPtime)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,30) string, TRIM(tvarnam), InpRec,            &
     &                        TRIM(ncname)
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
!
!  Inquire time units. Set local time scale.
!
        time_scale=0.0_r8
        status=nf_inq_var(ncINPid, varid, tvarnam, vartype, nvdims,     &
     &                    vdims(1,varid), nvatts)
        IF (status.eq.nf_noerr) THEN
          DO i=1,nvatts
            status=nf_inq_attname(ncINPid, varid, i, attnam)
            IF (status.eq.nf_noerr) THEN
              IF (TRIM(attnam).eq.'units') THEN
                status=nf_get_att_text(ncINPid, varid, TRIM(attnam),    &
     &                                 tunits)
                IF (status.eq.nf_noerr) THEN
                  IF (tunits(1:3).eq.'day') THEN
                    time_scale=day2sec
                  ELSE IF (tunits(1:6).eq.'second') THEN
                    time_scale=1.0_r8
                  END IF
                ELSE
                  WRITE (stdout,50) string, TRIM(attnam)
                  exit_flag=2
                  ioerror=status
                  RETURN
                END IF
              END IF
            ELSE
              WRITE (stdout,60) string, TRIM(tvarnam)
              exit_flag=2
              RETURN
            END IF
          END DO
        ELSE
          WRITE (stdout,70) string, TRIM(ncname)
          exit_flag=2
          RETURN
        END IF
        INPtime=INPtime*time_scale
      END IF
!
!  Set starting time index and time clock in days.
!
#ifdef DISTRIBUTE
      CALL mp_bcastf (ng, model, INPtime, 1)
#endif
      IF ((model.eq.iNLM).or.(model.eq.iTLM).or.(model.eq.iRPM)) THEN
        time(ng)=INPtime
        tdays(ng)=time(ng)*sec2day
        ntstart=NINT((time(ng)-dstart*day2sec)/dt(ng))+1
        IF (ntstart.lt.1) ntstart=1
        ntfirst=ntstart
      ELSE IF (model.eq.iADM) THEN
        IF (msg.ne.1) THEN
          time(ng)=INPtime
          tdays(ng)=time(ng)*sec2day
        END IF
        ntstart=ntimes+1
        ntend=1
        ntfirst=ntend
      END IF
!
!  Over-write "IniRec" to the actual initial record processed.
!
      IF (model.eq.iNLM) THEN
#ifdef DISTRIBUTE
        CALL mp_bcasti (ng, model, InpRec, 1)
#endif
        IniRec=InpRec
      END IF
#ifdef FILTERED
      filtindx = InpRec
#endif
!
!  Report information.
!
        lstr=SCAN(ncname,'/',BACK=.TRUE.)+1
        lend=LEN_TRIM(ncname)
        IF (InpThread) THEN
          IF (ERend.gt.ERstr) THEN
            WRITE (stdout,80) string, TRIM(Mstate(msg)),                &
     &                        INPtime*sec2day, Nrun,                    &
     &                        ncname(lstr:lend), InpRec, Tindex
          ELSE
            WRITE (stdout,90) string, TRIM(Mstate(msg)),                &
     &                        INPtime*sec2day,                          &
     &                        ncname(lstr:lend), InpRec, Tindex
          END IF
        END IF

#ifdef NONLINEAR
!
!-----------------------------------------------------------------------
!  Read in nonlinear state variables.
!-----------------------------------------------------------------------
!
      NLM_STATE: IF (model.eq.iNLM) THEN
!
!  Read in nonlinear free-surface (m).
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % zeta(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 2D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % ubar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 2D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % vbar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
# ifdef SOLVE3D
!
!  Read in nonlinear 3D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % u(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear 3D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % v(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in nonlinear tracer type variables.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % t(LBi,LBj,1,Tindex,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#  ifdef SEDIMENT
!
!  Read in nonlinear sediment fraction of each size class in each bed
!  layer.
!
        DO i=1,NST
          IF (get_var(idfrac(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idfrac(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % bed_frac(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idfrac(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idfrac(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
!  Read in nonlinear sediment mass of each size class in each bed layer.
!
          IF (get_var(idBmas(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBmas(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % bed_mass(LBi,LBj,1,Tindex,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBmas(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBmas(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
!
!  Read in nonlinear sediment properties in each bed layer.
!
        DO i=1,MBEDP
          IF (get_var(idSbed(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSbed(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % bed(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idSbed(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idSbed(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#   ifdef BEDLOAD
!
!  Read in nonlinear sediment fraction of bed load.
!
        DO i=1,NST
          IF (get_var(idUbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*u2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % umask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % bedldu(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idUbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idUbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
          IF (get_var(idVbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*v2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % vmask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % bedldv(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idVbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idVbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#   endif
#  endif

#  if defined SEDIMENT || defined BBL_MODEL
!
!  Read in nonlinear sediment properties in exposed bed layer.
!
        DO i=1,MBOTP
          IF (get_var(idBott(i)).and.have_var(idBott(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBott(i))),    &
     &                            varid)
              gtype=vflag(varid)*r2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % bottom(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBott(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBott(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#  endif
#  ifdef ICE_MODEL
!
!  Read in 2D ice momentum component (m/s) in the XI-direction.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUice)), varid)
          gtype=vflag(varid)*u2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % umask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % ui(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idUice)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idUice)), Fmin, Fmax
          END IF
        END IF
!
!  Read in 2D ice momentum component (m/s) in the ETA-direction.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVice)), varid)
          gtype=vflag(varid)*v2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % vmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % vi(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idVice)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idVice)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice concentration.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idAice)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % ai(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idAice)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idAice)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice average thickness.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idHice)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % hi(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idHice)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idHice)), Fmin, Fmax
          END IF
        END IF
!
!  Read in snow average thickness.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idHsno)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % hsn(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idHsno)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idHsno)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice/snow surface temperature.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTice)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % tis(LBi,LBj))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idTice)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idTice)), Fmin, Fmax
          END IF
        END IF
!
!  Read in surface water thickness (on ice).
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSfwat)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % sfwat(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idSfwat)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idSfwat)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice internal temperature.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTimid)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % ti(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idTimid)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idTimid)), Fmin, Fmax
          END IF
        END IF
!
!  Read in internal ice stress component 11
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSig11)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % sig11(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idSig11)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idSig11)), Fmin, Fmax
          END IF
        END IF
!
!  Read in internal ice stress component 22
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSig22)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % sig22(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idSig22)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idSig22)), Fmin, Fmax
          END IF
        END IF
!
!  Read in internal ice stress component 12
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSig12)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % sig12(LBi,LBj,1))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idSig12)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idSig12)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice-water friction velocity.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTauiw)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
       &                ICE(ng) % utau_iw(LBi,LBj))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idTauiw)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idTauiw)), Fmin, Fmax
          END IF
        END IF
!
!  Read in ice-water momentum transfer coefficient.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idChuiw)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % chu_iw(LBi,LBj))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idChuiw)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idChuiw)), Fmin, Fmax
          END IF
        END IF
!
!  Read in salinity of molecular sub-layer under ice
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idS0mk)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % s0mk(LBi,LBj))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idS0mk)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idS0mk)), Fmin, Fmax
          END IF
        END IF
!
!  Read in temperature of molecular sub-layer under ice
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(Vname(1,idT0mk)), varid)
          gtype=vflag(varid)*r2dvar
        END IF
        status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,     &
     &                Vsize, LBi, UBi, LBj, UBj,                        &
     &                Fscl, Fmin, Fmax,                                 &
#   ifdef MASKING
     &                GRID(ng) % rmask(LBi,LBj),                        &
#   endif
     &                ICE(ng) % t0mk(LBi,LBj))
        IF (status.ne.nf_noerr) THEN
          IF (Master) THEN
            WRITE (stdout,30) string, TRIM(Vname(1,idT0mk)),            &
     &                        InpRec, TRIM(ncname)
          END IF
          exit_flag=2
          ioerror=status
          RETURN
        ELSE
          IF (Master) THEN
            WRITE (stdout,100) TRIM(Vname(2,idT0mk)), Fmin, Fmax
          END IF
        END IF
!
#  endif
# endif
      END IF NLM_STATE
#endif

#if defined TANGENT || defined TL_IOMS
!
!-----------------------------------------------------------------------
!  Read in tangent linear state variables.
!-----------------------------------------------------------------------
!
      TLM_STATE: IF ((model.eq.iTLM).or.(model.eq.iRPM)) THEN
!
!  Read in tangent linear free-surface (m).
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % tl_zeta(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tangent linear 2D momentum component (m/s) in the
!  XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % tl_ubar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tangent linear 2D momentum component (m/s) in the
!  ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % tl_vbar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
# ifdef SOLVE3D
!
!  Read in tangent linear 3D momentum component (m/s) in the
!  XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % tl_u(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tangent linear 3D momentum component (m/s) in the
!  ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % tl_v(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tangent linear tracer type variables.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % tl_t(LBi,LBj,1,Tindex,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF             
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#  ifdef SEDIMENT
!
!  Read in tangent linear sediment fraction of each size class in each
!  bed layer.
!
        DO i=1,NST
          IF (get_var(idfrac(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idfrac(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % tl_bed_frac(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idfrac(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idfrac(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
!  Read in tangent linear sediment mass of each size class in each
!  bed layer.
!
          IF (get_var(idBmas(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBmas(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng)%tl_bed_mass(LBi,LBj,1,Tindex,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBmas(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBmas(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
!
!  Read in tangent linear sediment properties in each bed layer.
!
        DO i=1,MBEDP
          IF (get_var(idSbed(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSbed(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % tl_bed(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idSbed(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idSbed(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#   ifdef BEDLOAD
!
!  Read in tangent linear sediment fraction of bed load.
!
        DO i=1,NST
          IF (get_var(idUbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*u2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % umask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % tl_bedldu(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idUbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idUbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
          IF (get_var(idVbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*v2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % vmask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % tl_bedldv(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idVbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idVbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#   endif
#  endif

#  if defined SEDIMENT || defined BBL_MODEL
!
!  Read in tangent linear sediment properties in exposed bed layer.
!
        DO i=1,MBOTP
          IF (get_var(idBott(i)).and.have_var(idBott(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBott(i))),    &
     &                            varid)
              gtype=vflag(varid)*r2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % tl_bottom(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBott(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBott(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#  endif
# endif
      END IF TLM_STATE
#endif

#ifdef ADJOINT
!
!-----------------------------------------------------------------------
!  Read in adjoint state variables.
!-----------------------------------------------------------------------
!
      ADM_STATE: IF (model.eq.iADM) THEN
!
!  Read in adjoint free-surface (m).
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % ad_zeta(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in adjoint 2D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % ad_ubar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in adjoint 2D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % ad_vbar(LBi,LBj,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
# ifdef SOLVE3D
!
!  Read in adjoint 3D momentum component (m/s) in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % ad_u(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in adjoint 3D momentum component (m/s) in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % ad_v(LBi,LBj,1,Tindex))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in adjoint tracer type variables.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % ad_t(LBi,LBj,1,Tindex,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#  ifdef SEDIMENT
!
!  Read in adjoint sediment fraction of each size class in each bed
!  layer.
!
        DO i=1,NST
          IF (get_var(idfrac(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idfrac(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % ad_bed_frac(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idfrac(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idfrac(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
!  Read in adjoint sediment mass of each size class in each bed layer.
!
          IF (get_var(idBmas(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBmas(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng)%ad_bed_mass(LBi,LBj,1,Tindex,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBmas(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBmas(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
!
!  Read in adjoint sediment properties in each bed layer.
!
        DO i=1,MBEDP
          IF (get_var(idSbed(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idSbed(i))),    &
     &                            varid)
              gtype=vflag(varid)*b3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, Nbed,       &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % ad_bed(LBi,LBj,1,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idSbed(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idSbed(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO

#   ifdef BEDLOAD
!
!  Read in adjoint sediment fraction of bed load.
!
        DO i=1,NST
          IF (get_var(idUbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*u2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % umask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % ad_bedldu(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idUbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idUbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
!
          IF (get_var(idVbld(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbld(i))),    &
     &                            varid)
              gtype=vflag(varid)*v2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#    ifdef MASKING
     &                        GRID(ng) % vmask(LBi,LBj),                &
#    endif
     &                        OCEAN(ng) % ad_bedldv(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idVbld(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idVbld(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#   endif
#  endif

#  if defined SEDIMENT || defined BBL_MODEL
!
!  Read in adjoint sediment properties in exposed bed layer.
!
        DO i=1,MBOTP
          IF (get_var(idBott(i)).and.have_var(idBott(i))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idBott(i))),    &
     &                            varid)
              gtype=vflag(varid)*r2dvar
            END IF
            status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj,                &
     &                        Fscl, Fmin, Fmax,                         &
#   ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#   endif
     &                        OCEAN(ng) % ad_bottom(LBi,LBj,i))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idBott(i))),     &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idBott(i))), Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
#  endif
# endif
      END IF ADM_STATE
#endif

#ifdef FOUR_DVAR
!
!-----------------------------------------------------------------------
!  Read in background-error covariance normalization factors.
!-----------------------------------------------------------------------
!
      NRM_STATE: IF (model.eq.5) THEN
!
!  Read in free-surface norm.
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % b_zeta(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum norm in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % b_ubar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum norm in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % b_vbar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF

# ifdef SOLVE3D
!
!  Read in 3D momentum norm in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % b_u(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 3D momentum norm in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % b_v(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tracer type variables norm.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % b_t(LBi,LBj,1,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
# endif
      END IF NRM_STATE
#endif
#ifdef FILTERED_RST
      IF (nrrec.ne.0) THEN
        DO i=1,FIL
          CALL get_filter (ng, i)
        END DO
      END IF
#endif

#if defined FOUR_DVAR
!
!-----------------------------------------------------------------------
!  Read in backgound-error standard deviation factors.
!-----------------------------------------------------------------------
!
      STD_STATE: IF (model.eq.6) THEN
!
!  Read in free-surface norm.
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % e_zeta(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum norm in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % e_ubar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum norm in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % e_vbar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
# ifdef SOLVE3D
!
!  Read in 3D momentum norm in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % e_u(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 3D momentum norm in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % e_v(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tracer type variables norm.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % e_t(LBi,LBj,1,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
# endif
!
!  Read in convolution horizontal diffusion coefficients.
!
        IF (have_var(idKhor)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idKhor)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, KhMin(ng), KhMax(ng),                 &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      MIXING(ng) % Kh(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idKhor)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idKHor)), KhMin(ng),      &
     &                           KhMax(ng)
            END IF
          END IF
        END IF

# ifdef SOLVE3D
!
!  Read in convolution vertical diffusion coefficient.
!
        IF (have_var(idKver)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idKver)), varid)
            gtype=vflag(varid)*w3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 0, N(ng),        &
     &                      Fscl, KvMin(ng), KvMax(ng),                 &
#  ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
#  endif
     &                      MIXING(ng) % Kv(LBi,LBj,0))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idKver)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idKver)), KvMin(ng),      &
     &                           KvMax(ng)
            END IF
          END IF
        END IF
# endif
      END IF STD_STATE
#endif

#if defined IMPULSE
!
!-----------------------------------------------------------------------
!  Read in adjoint model or tangent linear model impulse forcing terms.
!-----------------------------------------------------------------------
!
      FRC_STATE: IF (model.eq.7) THEN
!
!  Set number of records available.
!
        NrecFrc(ng)=Nrec
!
!  Read in next impulse forcing time to process.
!
        IF (InpThread) THEN
          status=nf_inq_varid(ncINPid, TRIM(tvarnam), varid)
          status=nf_get_var1_TYPE(ncINPid, varid, InpRec, FrcTime(ng))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,30) string, TRIM(tvarnam), InpRec,            &
     &                        TRIM(ncname)
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, FrcTime(ng), 1)
# endif
!
!  Read in free-surface forcing.
!
        IF (get_var(idFsur)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idFsur)), varid)
            gtype=vflag(varid)*r2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % rmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % f_zeta(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idFsur)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idFsur)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum forcing in the XI-direction.
!
        IF (get_var(idUbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUbar)), varid)
            gtype=vflag(varid)*u2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % f_ubar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUbar)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 2D momentum forcing in the ETA-direction.
!
        IF (get_var(idVbar)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVbar)), varid)
            gtype=vflag(varid)*v2dvar
          END IF
          status=nf_fread2d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj,                  &
     &                      Fscl, Fmin, Fmax,                           &
# ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
# endif
     &                      OCEAN(ng) % f_vbar(LBi,LBj))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVbar)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVbar)), Fmin, Fmax
            END IF
          END IF
        END IF
# ifdef SOLVE3D
!
!  Read in 3D momentum forcing in the XI-direction.
!
        IF (get_var(idUvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idUvel)), varid)
            gtype=vflag(varid)*u3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % umask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % f_u(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idUvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idUvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in 3D momentum norm in the ETA-direction.
!
        IF (get_var(idVvel)) THEN
          IF (InpThread) THEN
            status=nf_inq_varid(ncINPid, TRIM(Vname(1,idVvel)), varid)
            gtype=vflag(varid)*v3dvar
          END IF
          status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype,   &
     &                      Vsize, LBi, UBi, LBj, UBj, 1, N(ng),        &
     &                      Fscl, Fmin, Fmax,                           &
#  ifdef MASKING
     &                      GRID(ng) % vmask(LBi,LBj),                  &
#  endif
     &                      OCEAN(ng) % f_v(LBi,LBj,1))
          IF (status.ne.nf_noerr) THEN
            IF (Master) THEN
              WRITE (stdout,30) string, TRIM(Vname(1,idVvel)), InpRec,  &
     &                          TRIM(ncname)
            END IF
            exit_flag=2
            ioerror=status
            RETURN
          ELSE
            IF (Master) THEN
              WRITE (stdout,100) TRIM(Vname(2,idVvel)), Fmin, Fmax
            END IF
          END IF
        END IF
!
!  Read in tracer type variables norm.
!
        DO itrc=1,NT(ng)
          IF (get_var(idTvar(itrc))) THEN
            IF (InpThread) THEN
              status=nf_inq_varid(ncINPid, TRIM(Vname(1,idTvar(itrc))), &
     &                            varid)
              gtype=vflag(varid)*r3dvar
            END IF
            status=nf_fread3d(ng, model, ncINPid, varid, InpRec, gtype, &
     &                        Vsize, LBi, UBi, LBj, UBj, 1, N(ng),      &
     &                        Fscl, Fmin, Fmax,                         &
#  ifdef MASKING
     &                        GRID(ng) % rmask(LBi,LBj),                &
#  endif
     &                        OCEAN(ng) % f_t(LBi,LBj,1,itrc))
            IF (status.ne.nf_noerr) THEN
              IF (Master) THEN
                WRITE (stdout,30) string, TRIM(Vname(1,idTvar(itrc))),  &
     &                            InpRec, TRIM(ncname)
              END IF
              exit_flag=2
              ioerror=status
              RETURN
            ELSE
              IF (Master) THEN
                WRITE (stdout,100) TRIM(Vname(2,idTvar(itrc))),         &
     &                             Fmin, Fmax
              END IF
            END IF
          END IF
        END DO
# endif
      END IF FRC_STATE
#endif
!
!-----------------------------------------------------------------------
!  Close input NetCDF file.
!-----------------------------------------------------------------------
!
      status=nf_close(ncINPid)
!
  20  FORMAT (/,a,'GET_STATE - unable to open input NetCDF file: ',a)
  30  FORMAT (/,a,'GET_STATE - error while reading variable: ',a,2x,    &
     &        'at time record = ',i3,/,17x,'in input NetCDF file: ',a)
  40  FORMAT (/,a,'GET_STATE - requested input time record = ',i3,/,    &
     &        17x,'not found in input NetCDF: ',a,/,                    &
     &        17x,'number of available records = ',i3)
  50  FORMAT (/,a,'GET_STATE - error while reading attribute: ',a)
  60  FORMAT (/,a,'GET_STATE - error while inquiring attributes',       &
     &          ' for variable: ',a)
  70  FORMAT (/,a,'GET_STATE - cannot inquire about time variable',     &
     &          ' in input NetCDF file: ',a)
  80  FORMAT (/,a,'GET_STATE - ',a,t64,'t = ',f12.4,                    &
     &        /,19x,'(Iter=',i4.4,', File: ',a,', Rec=',i4.4,           &
     &        ', Index=',i1,')')
  90  FORMAT (/,a,'GET_STATE - ',a,t64,'t = ',f12.4,                    &
     &        /,19x,'(File: ',a,', Rec=',i4.4,', Index=',i1,')')
 100  FORMAT (16x,'- ',a,/,19x,'(Min = ',1p,e15.8,                      &
     &        ' Max = ',1p,e15.8,')')
      RETURN
      END SUBROUTINE get_state

