#include "cppdefs.h"
#if defined FOUR_DVAR && !defined REPRESENTERS
      SUBROUTINE obs_cost (ng, model)
!
!=================================================== Andrew M. Moore ===
!  Copyright (c) 2005 ROMS/TOMS Adjoint Group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  If minimization first pass, this routine computes the observation   !
!  cost function (Jo) as the misfit (squared difference) between the   !
!  model and observations.                                             !
!                                                                      !
!  If conventional strong contraint 4DVAR:                             !
!                                                                      !
!         Jo = 1/2 transpose(H X - Xo) * O^(-1) * (H X - Xo)           !
!                                                                      !
!  or if incremental strong contraint 4DVAR:                           !
!                                                                      !
!         Jo = 1/2 transpose(H deltaX - d) * O^(-1) * (H deltaX - d)   !
!                                                                      !
!  where                                                               !
!                                                                      !
!          d = Xo - H Xb                                               !
!                                                                      !
!         d  : innovation vector                                       !
!         H  : observation operator (linearized if incremental)        !
!       H Xb : background at observation points previous forecast)     !
!         Xo : observations vector                                     !
!       H X  : nonlinear model at observation points                   !
!  H deltaX  : increment at observation point                          !
!         O  : observations error covariance                           !
!                                                                      !
!  If minimization second pass (m=2), it computes conjugate gradient   !
!  step size numerator and denominator factors due to observations:    !
!                                                                      !
!  If conventional strong contraint 4DVAR:                             !
!                                                                      !
!    StepTopObs = 1/2 * (H X(m-1) - Xo) * O^(-1) * (H X(m) - H X(m-1)) !
!                                                                      !
!    StepBotObs = 1/2 * O^(-1) * (H X(m) - H X(m-1)) ** 2              ! 
!                                                                      !
!  or if incremental strong contraint 4DVAR:                           !
!                                                                      !
!    StepTopObs = 1/2 * (H deltaX(m-1) - d) * O^(-1) *                 !
!                       (H deltaX(m) - H deltaX(m-1))                  !
!                                                                      !
!    StepBotObs = 1/2 * O^(-1) * (H deltaX(m) - H deltaX(m-1)) ** 2    ! 
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, model
!
!  Local variable declarations.
!
      integer :: NSUB, iobs, ivar

      real(r8) ::  cff, my_StepBot, my_StepTop

      real(r8), dimension(0:NstateVar(ng)) :: my_ObsCost

# ifdef DISTRIBUTE
      real(r8), dimension(0:NstateVar(ng)+2) :: buffer
      character (len=3), dimension(0:NstateVar(ng)+2) :: op_handle
# endif
!
!-----------------------------------------------------------------------
!  Compute observation scale (ObsScale). The scale factor is used for
!  screenning  or normalization of the observations. If only screening,
!  the scale is one for good observations and zero for bad observations.
!  If normalizing, the scale is terms of screenning and some specified
!  norm. In incremental 4DVAR only screenning is carried out.
!-----------------------------------------------------------------------
!
      CALL obs_scale (ng, model)
!
!-----------------------------------------------------------------------
!  Compute observation misfit cost function (ObsCost).
!-----------------------------------------------------------------------
!
      DO ivar=0,NstateVar(ng)
        my_ObsCost(ivar)=0.0_r8
      END DO
      IF (Ipass.eq.1) THEN
        DO iobs=1,Nobs(ng)
          ivar=ObsType(iobs)
# if defined S4DVAR
          cff=0.5_r8*ObsScale(iobs)*ObsErr(iobs)*                       &
     &        (NLmodVal(iobs,Ipass)-ObsVal(iobs))**2
# elif defined IS4DVAR
          cff=0.5_r8*ObsScale(iobs)*ObsErr(iobs)*                       &
     &        (NLmodVal(iobs)+TLmodVal(iobs,Ipass)-ObsVal(iobs))**2
# endif
          my_ObsCost(0)=my_ObsCost(0)+cff
          my_ObsCost(ivar)=my_ObsCost(ivar)+cff
        END DO
      END IF

# if defined S4DVAR || defined IS4DVAR
!
!-----------------------------------------------------------------------
!  Compute Conjugate gradient algorithm step size dot products,
!  StepTopObs, StepBotObs.
!-----------------------------------------------------------------------
!
      IF (Ipass.eq.2) THEN
        my_StepTop=0.0_r8
        my_StepBot=0.0_r8
        DO iobs=1,Nobs(ng)
#  if defined S4DVAR
          cff=0.5_r8*ObsScale(iobs)*ObsErr(iobs)
          my_StepTop=my_StepTop+                                        &
     &               cff*                                               &
     &               (NLmodVal(iobs,Ipass-1)-ObsVal(iobs))*             &
     &               (NLmodVal(iobs,Ipass)-NLmodVal(iobs,Ipass-1))
          my_StepBot=my_StepBot+                                        &
     &               cff*                                               &
     &               (NLmodVal(iobs,Ipass)-NLmodVal(iobs,Ipass-1))**2
#  elif defined IS4DVAR
          cff=0.5_r8*ObsScale(iobs)*ObsErr(iobs)
          my_StepTop=my_StepTop+                                        &
     &               cff*                                               &
     &               (NLmodVal(iobs)+TLmodVal(iobs,Ipass-1)-            &
     &                ObsVal(iobs))*                                    &
     &               (TLmodVal(iobs,Ipass)-TLmodVal(iobs,Ipass-1))
          my_StepBot=my_StepBot+                                        &
     &               cff*                                               &
     &               (TLmodVal(iobs,Ipass)-TLmodVal(iobs,Ipass-1))**2
#  endif 
        END DO
      END IF
# endif
!
!-----------------------------------------------------------------------
!  Compute global values.
!-----------------------------------------------------------------------
!
      NSUB=NtileX(ng)*NtileE(ng)
!$OMP CRITICAL (COST_FUN)
      IF (tile_count.eq.0) THEN
        DO ivar=0,NstateVar(ng)
          FOURDVAR(ng)%ObsCost(ivar)=my_ObsCost(ivar)
        END DO
        StepBotObs=my_StepBot
        StepTopObs=my_StepTop
      ELSE
        DO ivar=0,NstateVar(ng)
          FOURDVAR(ng)%ObsCost(ivar)=FOURDVAR(ng)%ObsCost(ivar)+        &
     &                               my_ObsCost(ivar)
        END DO
        StepBotObs=StepBotObs+my_StepBot
        StepTopObs=StepTopObs+my_StepTop
      END IF
      tile_count=tile_count+1
      IF (tile_count.eq.NSUB) THEN
        tile_count=0
#ifdef DISTRIBUTE
        DO ivar=0,NstateVar(ng)
          buffer(ivar)=FOURDVAR(ng)%ObsCost(ivar)
          op_handle(ivar)='SUM'
        END DO
        buffer(NstateVar(ng)+1)=StepBotObs
        buffer(NstateVar(ng)+2)=StepTopObs
        op_handle(NstateVar(ng)+1)='SUM'
        op_handle(NstateVar(ng)+2)='SUM'
        CALL mp_reduce (ng, iNLM, NstateVar(ng)+3, buffer(0:),          &
     &                  op_handle(0:))
        DO ivar=0,NstateVar(ng)
          FOURDVAR(ng)%ObsCost(ivar)=buffer(ivar)
        END DO
        StepBotObs=buffer(NstateVar(ng)+1)
        StepTopObs=buffer(NstateVar(ng)+2)
#endif
      END IF
!$OMP END CRITICAL (COST_FUN)
!
!  If start of minimization, set cost function scales used to report
!  normalized values.
!
      IF ((Nrun.eq.1).and.(Ipass.eq.1)) THEN
        DO ivar=0,NstateVar(ng)
          FOURDVAR(ng)%CostNorm(ivar)=FOURDVAR(ng)%ObsCost(ivar)
        END DO
      END IF

      RETURN
      END SUBROUTINE obs_cost
#else
      SUBROUTINE obs_cost
      RETURN
      END SUBROUTINE obs_cost
#endif
