#include "cppdefs.h"
#ifdef FOUR_DVAR
      SUBROUTINE obs_read (ng, model, backward)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS adjoint group                          !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This subroutine reads in observations data when appropriate from    !
!  observations input NetCDF file.  The observations data is stored    !     
!  for use elsewhere.                                                  ! 
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_fourdvar
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars
!
      implicit none
!
!  Imported variable declarations.
!
      logical, intent(in) :: backward

      integer, intent(in) :: ng, model
!
!  Local variable declarations.
!
      logical :: readNLmod, readTLmod

      integer :: Mstr, Mend
      integer :: ObsSum, i, is, ie, iobs, itrc, status

      integer, dimension(2) :: start, total
!
!---------------------------------------------------------------------
!  Read observation variables needed for interpolating the model
!  state at the observation locations.
!---------------------------------------------------------------------
!
      IF (ProcessObs(ng)) THEN
# if defined S4DVAR
        readNLmod=backward
# elif defined TLM_OBS
        readNLmod=.TRUE.
        readTLmod=.TRUE.
# endif
!
!  Initialize observations processing counters.
!
        DO i=1,NstateVar(ng)
          FOURDVAR(ng)%ObsCount(i)=0
        END DO
        IF (backward) THEN
          ObsSurvey(ng)=ObsSurvey(ng)-1
        ELSE
          ObsSurvey(ng)=ObsSurvey(ng)+1
        END IF
!
!  Set number of observations to process.
!
        Nobs(ng)=FOURDVAR(ng)%NobsSurvey(ObsSurvey(ng))
!
!  Set number of datum to process at current time-step.
!
        IF (backward) THEN
          NendObs(ng)=NstrObs(ng)-1
          NstrObs(ng)=NstrObs(ng)-Nobs(ng)
        ELSE
          NstrObs(ng)=NendObs(ng)+1
          NendObs(ng)=NstrObs(ng)+Nobs(ng)-1
        END IF
        start(1)=NstrObs(ng)
        total(1)=Nobs(ng)
!
!  Set starting index of obervation vectors for reading.  In weak
!  constraint, the entire observation data is loaded. Otherwise,
!  only the observartion for the current time window are loaded
!  and started from vector index one.
!
# ifdef REPRESENTERS
        Mstr=NstrObs(ng)
        Mend=NendObs(ng)
# else
        Mstr=1
        Mend=Nobs(ng)
# endif
!
!  Read in observation type identifier.
!
        IF (InpThread) THEN
          status=nf_get_vara_int(ncOBSid(ng), obsVid(idOtyp,ng),        &
     &                           start, total, ObsType(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idOtyp)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcasti (ng, model, ObsType, Mobs)
# endif
!
!  Read in observation time (days).
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idObsT,ng),       &
     &                            start, total, Tobs(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idObsT)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, Tobs, Mobs)
# endif
!
!  Read in observation X-location (grid units).
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idObsX,ng),       &
     &                            start, total, Xobs(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idObsX)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, Xobs, Mobs)
# endif
!
!  Read in observation Y-location (grid units).
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idObsY,ng),       &
     &                            start, total, Yobs(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idObsY)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, Yobs, Mobs)
# endif

# ifdef SOLVE3D
!
!  Read in observation depth.  If negative, depth is meter. If greater
!  than zero, depth is model level.  The "Zobs" arrays is over-written
!  in "extract_obs3d" with grid units value in order to write grid
!  values into the observation NetCDF file.
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idObsD,ng),       &
     &                            start, total, Zobs(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idObsD)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
#  ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, Zobs, Mobs)
#  endif
# endif
!
!  Read in observation values.
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idOval,ng),       &
     &                            start, total, ObsVal(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idOval)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, ObsVal, Mobs)
# endif
!
# ifdef REPRESENTERS
!  Read in observation error covariance.
# else
!  Read in observation error covariance. To avoid successive divisions,
!  convert to inverse observation error covariance.
# endif
!
        IF (InpThread) THEN
          status=nf_get_vara_TYPE(ncOBSid(ng), obsVid(idOerr,ng),       &
     &                            start, total, ObsErr(Mstr))
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idOerr)), ObsSurvey(ng),     &
     &                        TRIM(OBSname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
# ifndef REPRESENTERS
          DO iobs=1,Nobs(ng)
            ObsErr(iobs)=1.0_r8/ObsErr(iobs)
          END DO
# endif
        END IF
# ifdef DISTRIBUTE
        CALL mp_bcastf (ng, model, ObsErr, Mobs)
# endif
!
!  Read in nonlinear model values at observation locations.
!
        IF (readNLmod.and.haveNLmod(ng)) THEN
          IF (InpThread) THEN
# ifdef S4DVAR
            start(1)=NstrObs(ng)
            total(1)=Nobs(ng)
            start(2)=Ipass
            total(2)=1
            status=nf_get_vara_TYPE(ncMODid(ng), modVid(idNLmo,ng),     &
     &                              start, total, NLmodVal(1,Ipass))
# else
            start(1)=NstrObs(ng)
            total(1)=Nobs(ng)
            status=nf_get_vara_TYPE(ncMODid(ng), modVid(idNLmo,ng),     &
     &                              start, total, NLmodVal(Mstr))
# endif
            IF (status.ne.nf_noerr) THEN
              WRITE (stdout,10) TRIM(Vname(1,idNLmo)), ObsSurvey(ng),   &
     &                          TRIM(MODname(ng))
              exit_flag=2
              ioerror=status
              RETURN
            END IF
          END IF
# ifdef DISTRIBUTE
#  ifdef S4DVAR
          CALL mp_bcastf (ng, model, NLmodVal(:,Ipass), Mobs)
#  else
          CALL mp_bcastf (ng, model, NLmodVal, Mobs)
#  endif
# endif
        END IF

# ifdef TLM_OBS
!
!  If adjoint pass and incremental S4DVAR, read in tangent linear model
!  values at observation locations.
!
#  ifdef IS4DVAR
        IF (readTLmod.and.haveTLmod(ng).and.(Ipass.ne.0)) THEN
#  else
        IF (readTLmod.and.haveTLmod(ng)) THEN
#  endif
          IF (InpThread) THEN
#  ifdef IS4DVAR
            start(1)=NstrObs(ng)
            total(1)=Nobs(ng)
            start(2)=Ipass
            total(2)=1
            status=nf_get_vara_TYPE(ncMODid(ng), modVid(idTLmo,ng),     &
     &                              start, total, TLmodVal(1,Ipass))
#  else
            start(1)=NstrObs(ng)
            total(1)=Nobs(ng)
            status=nf_get_vara_TYPE(ncMODid(ng), modVid(idTLmo,ng),     &
     &                              start, total, TLmodVal(Mstr))
#  endif
            IF (status.ne.nf_noerr) THEN
              WRITE (stdout,10) TRIM(Vname(1,idTLmo)), ObsSurvey(ng),   &
     &                          TRIM(MODname(ng))
              exit_flag=2
              ioerror=status
              RETURN
            END IF
          END IF

#  ifndef REPRESENTERS
!
!  Reset TLM values to zero at the first pass of the inner loop.
!
          IF ((inner.eq.1).and.(Ipass.eq.1)) THEN
            DO iobs=1,Mobs
#   ifdef IS4DVAR
              TLmodVal(iobs,1)=0.0_r8
              TLmodVal(iobs,2)=0.0_r8
#   else
              TLmodVal(iobs)=0.0_r8
#   endif
            END DO
          END IF
#   ifdef DISTRIBUTE
#    ifdef IS4DVAR
          CALL mp_bcastf (ng, model, TLmodVal, 2*Mobs)
#    else
          CALL mp_bcastf (ng, model, TLmodVal, Mobs)
#    endif
#   endif
#  endif
        END IF
# endif
# ifdef IOM_IO
!
!  If multiple executables, read in representer coefficients (or
!  its approximation) and load it in ADmodVal.
!
        IF (backward.and.haveADmod(ng)) THEN
          start(1)=1
          total(1)=Ndatum(ng)
          status=nf_get_vara_TYPE(ncMODid(ng), modVid(idRepC,ng),       &
     &                            start, total, ADmodVal)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(Vname(1,idRepC)), ObsSurvey(ng),     &
     &                        TRIM(MODname(ng))
            exit_flag=2
            ioerror=status
            RETURN
          END IF
        END IF            
# endif
!
!-----------------------------------------------------------------------
!  Set counters for number of observations to process for each state
!  variable.
!-----------------------------------------------------------------------
!
        DO iobs=Mstr,Mend
          IF  (ObsType(iobs).eq.isFsur) THEN
            FOURDVAR(ng)%ObsCount(isFsur)=                              &
     &                           FOURDVAR(ng)%ObsCount(isFsur)+1
          ELSE IF (ObsType(iobs).eq.isUbar) THEN
            FOURDVAR(ng)%ObsCount(isUbar)=                              &
     &                           FOURDVAR(ng)%ObsCount(isUbar)+1
          ELSE IF (ObsType(iobs).eq.isVbar) THEN
            FOURDVAR(ng)%ObsCount(isVbar)=                              &
     &                           FOURDVAR(ng)%ObsCount(isVbar)+1
# ifdef SOLVE3D
          ELSE IF (ObsType(iobs).eq.isUvel) THEN
            FOURDVAR(ng)%ObsCount(isUvel)=                              &
     &                           FOURDVAR(ng)%ObsCount(isUvel)+1
          ELSE IF (ObsType(iobs).eq.isVvel) THEN
            FOURDVAR(ng)%ObsCount(isVvel)=                              &
     &                           FOURDVAR(ng)%ObsCount(isVvel)+1
          ELSE
            DO itrc=1,NT(ng)
              IF (ObsType(iobs).eq.isTvar(itrc)) THEN
                i=isTvar(itrc)
                FOURDVAR(ng)%ObsCount(i)=FOURDVAR(ng)%ObsCount(i)+1
              END IF
            END DO
# endif
          END IF
        END DO
        IF (Master) THEN
          WRITE (stdout,20) ObsTime(ng)*sec2day
          ObsSum=0.0_r8
          is=Mstr
          ie=0
          DO i=1,NstateVar(ng)
            IF (FOURDVAR(ng)%ObsCount(i).gt.0) THEN
              ie=ie+FOURDVAR(ng)%ObsCount(i)
              WRITE (stdout,30) TRIM(Vname(1,idSvar(i))), is, ie,       &
     &                          ie-is+1
              is=ie+1
              ObsSum=ObsSum+FOURDVAR(ng)%ObsCount(i)
            END IF
          END DO
          WRITE (stdout,40) ObsSum
          WRITE (stdout,'(/)')
        END IF
!
!-----------------------------------------------------------------------
!  If applicable, set next observation survey time to process.
!-----------------------------------------------------------------------
!
        IF (backward) THEN
          IF ((ObsSurvey(ng)-1).ge.1) THEN
            ObsTime(ng)=FOURDVAR(ng)%SurveyTime(ObsSurvey(ng)-1)*day2sec
          END IF
        ELSE
          IF ((ObsSurvey(ng)+1).le.Nsurvey(ng)) THEN
            ObsTime(ng)=FOURDVAR(ng)%SurveyTime(ObsSurvey(ng)+1)*day2sec
          END IF
        END IF
      END IF
!
!-----------------------------------------------------------------------
!  Compute observation scale (ObsScale). The scale factor is used
!  for screenning of the observations. This scale is one for good 
!  observations and zero for bad observations.
!-----------------------------------------------------------------------
!
      CALL obs_scale (ng, model)
!
  10  FORMAT (/,' OBS_READ - error while reading variable: ',a,         &
     &        ' at survey record = ',i4,/,                              &
     &        11x,'in input NetCDF file: ',a)
  20  FORMAT (/,' Number of State Observations Processed:',             &
     &        t58,'ObsTime = ',f12.4,/,/,                               &
     &        20x,'Variable',10x,'IstrObs',4x,'IendObs',6x, 'Total',/)
  30  FORMAT (20x,a,t35,3(1x,i10))
  40  FORMAT (/,t58,i10)

      RETURN
      END SUBROUTINE obs_read
#else
      SUBROUTINE obs_read
      RETURN
      END SUBROUTINE obs_read
#endif
