#include "cppdefs.h"
      SUBROUTINE opencdf (ng, nfiles, fname, ncname, nvlev, ifield,     &
     &                    nrec, nvd, Vsize)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine inquires information about requested variable from     !
!  input NetCDF file(s).                                               !
!                                                                      !
!  Input:                                                              !
!                                                                      !
!    ng        Nested grid number.                                     !
!    nfiles    Number of input NetCDF files.                           !
!    fname     Input NetCDF file name(s).                              !
!    ncname    NetCDF file name containing requested variable.         !
!    nvlev     Number of vertical levels to check for consistency.     !
!    ifield    Index of field to inquire for the size of its time      !
!                dimension, if any.                                    !
!                                                                      !
!  Output:                                                             !
!                                                                      !
!     nrec     Number of time records available in requested variable. !
!     nvd      Number of dimension in requested variable.              !
!     Vsize    Size of each dimension in requested variable.           !
!                                                                      !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_scalars
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, nfiles, nvlev, ifield

      integer, intent(out) :: nrec
      integer, intent(out) :: nvd
      integer, intent(out) :: Vsize(4)

      character (len=*), intent(in) :: fname(nfiles)

      character (len=*), intent(out) :: ncname
!
!  Local variable declarations.
!
      logical :: SearchVar, timeatt

      integer :: attype, dimid, dimsiz, i, ifile, j
      integer :: ltvar, ltype, ncid, ndims, ngatts
      integer :: nvatts, recdim, status, varid, vartype

      character (len=20) :: dimnam, attnam
      character (len=45) :: text
!
!-----------------------------------------------------------------------
!  Open input NetCDF file(s). If several input NetCDF files (nfiles>1),
!  scan files until the requested variable is found.
!-----------------------------------------------------------------------
!
      SearchVar=.TRUE.
      DO i=1,4
        Vsize(i)=0
      END DO
      DO ifile=1,nfiles
        IF (SearchVar) THEN
          ncname=fname(ifile)
          status=nf_open (TRIM(ncname),nf_nowrite,ncid)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,10) TRIM(ncname)
            exit_flag=4
            ioerror=status
            RETURN
          END IF
#if !defined OFFLINE_FLOATS && !defined OFFLINE_BIOLOGY
!
!-----------------------------------------------------------------------
!  Inquire and get global "type" attribute.
!-----------------------------------------------------------------------
!
          status=nf_inq_att (ncid,nf_global,'type',attype,ltype)
          IF (status.eq.nf_noerr) THEN
            status=nf_get_att_text (ncid,nf_global,'type',type)
            IF (status.ne.nf_noerr) THEN
              WRITE (stdout,20) 'type (global)', TRIM(ncname)
              exit_flag=4
              ioerror=status
              RETURN
            END IF
          ELSE
            WRITE (stdout,30) 'type (global)', TRIM(ncname)
          END IF
#endif
!
!-----------------------------------------------------------------------
!  Inquire about the dimensions and variables.
!-----------------------------------------------------------------------
!
          recdim=-1
          status=nf_inq(ncid,ndims,nvars,ngatts,recdim)
          IF (status.eq.nf_noerr) THEN
            IF (nvars.gt.MV) THEN
              WRITE (stdout,40) MV, nvars
              exit_flag=4
              RETURN
            END IF
!
!  Inquire about dimensions. If ifield=0, check dimensions between
!  application and NetCDF file for consistency.
!
            nrec=0
            ltvar=0
            IF (ifield.gt.0) THEN
              ltvar=LEN_TRIM(Vname(5,ifield))
              Tname(ifield)=TRIM(Vname(5,ifield))
            END IF
            DO i=1,ndims
              dimid=i
              status=nf_inq_dim(ncid,dimid,dimnam,dimsiz)
              IF (status.ne.nf_noerr) THEN
                WRITE (stdout,50) dimid, TRIM(ncname)
                exit_flag=4
                ioerror=status
                RETURN
              END IF
              IF ((TRIM(dimnam).eq.'xi_rho').or.                        &
     &            (TRIM(dimnam).eq.'xi_v')) THEN
                IF ((dimsiz.ne.Lm(ng)+2).and.(ifield.eq.0)) THEN
                  WRITE (stdout,60) TRIM(dimnam), dimsiz, Lm(ng)+2
                  exit_flag=4
                  RETURN
                END IF
              ELSE IF ((TRIM(dimnam).eq.'xi_u').or.                     &
     &                 (TRIM(dimnam).eq.'xi_psi')) THEN
                IF ((dimsiz.ne.Lm(ng)+1).and.(ifield.eq.0)) THEN
                  WRITE (stdout,60) TRIM(dimnam), dimsiz, Lm(ng)+1
                  exit_flag=4
                  RETURN
                END IF
              ELSE IF ((TRIM(dimnam).eq.'eta_rho').or.                  &
     &                 (TRIM(dimnam).eq.'eta_u')) THEN
                IF ((dimsiz.ne.Mm(ng)+2).and.(ifield.eq.0)) THEN
                  WRITE (stdout,60) TRIM(dimnam), dimsiz, Mm(ng)+2
                  exit_flag=4
                  RETURN
                END IF
              ELSE IF ((TRIM(dimnam).eq.'eta_v').or.                    &
     &                 (TRIM(dimnam).eq.'eta_psi')) THEN
                IF ((dimsiz.ne.Mm(ng)+1).and.(ifield.eq.0)) THEN
                  WRITE (stdout,60) TRIM(dimnam), dimsiz, Mm(ng)+1
                  exit_flag=4
                  RETURN
                END IF
#ifdef SOLVE3D
              ELSE IF (TRIM(dimnam).eq.'s_rho') THEN
                IF ((dimsiz.ne.nvlev).and.(ifield.eq.0)) THEN
                  WRITE (stdout,60) TRIM(dimnam), dimsiz, nvlev
                  exit_flag=4
                  RETURN
                END IF
#endif
              ELSE IF ((ltvar.gt.0).and.                                &
     &                 (TRIM(dimnam).eq.TRIM(Tname(ifield)))) THEN
                nrec=dimsiz
              END IF
            END DO
!
!  Inquire about requested variable. Save its spatial dimensions for
!  reading latter. If "nrec" is zero, it indicates that the requested
!  variable does not have the specified time dimension. Then, check
!  the size of the dimension of its associated time variable. The
!  associated time variable is reset to the value specified in the
!  "time" attribute, if any. Check if only water points are available.
!
            timeatt=.FALSE.
            DO i=1,nvars
              varid=i
              vflag(i)=1
              status=nf_inq_var(ncid,varid,varnam(i),vartype,nvdims(i), &
     &                          vdims(1,i),nvatts)
              IF (status.ne.nf_noerr) THEN
                WRITE (stdout,70) varid, TRIM(ncname)
                exit_flag=4
                ioerror=status
                RETURN
              END IF
              IF ((LEN_TRIM(Vname(1,ifield)).gt.0).and.                 &
     &            (TRIM(varnam(i)).eq.TRIM(Vname(1,ifield)))) THEN
                SearchVar=.FALSE.
                nvd=nvdims(i)
                DO j=1,nvd
                  status=nf_inq_dim (ncid,vdims(j,i),dimnam,dimsiz)
                  Vsize(j)=dimsiz
                  IF (status.ne.nf_noerr) THEN
                    WRITE (stdout,110) TRIM(Tname(ifield))
                    exit_flag=4
                    ioerror=status
                    RETURN
                  END IF
                END DO
                DO j=1,nvatts
                  status=nf_inq_attname(ncid,varid,j,attnam)
                  IF (status.eq.nf_noerr) THEN
                    IF (TRIM(attnam).eq.'water_points'.and.             &
     &                  (nvdims(i).gt.0)) THEN
                      vflag(i)=-1
                    ELSE IF (TRIM(attnam).eq.'time') THEN
                      status=nf_inq_attlen(ncid,varid,TRIM(attnam),     &
     &                                     ltvar)
                      IF (status.ne.nf_noerr) THEN
                        WRITE (stdout,80) TRIM(attnam)
                        exit_flag=4
                        ioerror=status
                        RETURN
                      END IF
                      status=nf_get_att_text(ncid,varid,TRIM(attnam),   &
     &                                       text(1:ltvar))
                      IF (status.ne.nf_noerr) THEN
                        WRITE (stdout,90) TRIM(attnam)
                        exit_flag=4
                        ioerror=status
                        RETURN
                      END IF
                      Tname(ifield)=text(1:ltvar)
                      timeatt=.TRUE.
                    END IF
                  ELSE
                    WRITE (stdout,100) TRIM(varnam(i))
                    exit_flag=4
                    RETURN
                  END IF
                END DO
              END IF
            END DO
            IF (timeatt) THEN
              ltvar=LEN_TRIM(Tname(ifield))
              IF ((ifield.gt.0).and.(nrec.eq.0).and.(ltvar.gt.0)) THEN
                DO i=1,nvars
                  varid=i
                  status=nf_inq_var (ncid,varid,varnam(i),vartype,      &
     &                               nvdims(i),vdims(1,i),nvatts)
                  IF (status.ne.nf_noerr) THEN
                    WRITE (stdout,70) varid, TRIM(ncname)
                    exit_flag=4
                    ioerror=status
                    RETURN
                  END IF
                  IF (TRIM(varnam(i)).eq.TRIM(Tname(ifield))) THEN
                    DO j=1,nvdims(i)
                      status=nf_inq_dim (ncid,vdims(j,i),dimnam,dimsiz)
                      IF (status.ne.nf_noerr) THEN
                        WRITE (stdout,110) TRIM(Tname(ifield))
                        exit_flag=4
                        ioerror=status
                        RETURN
                      END IF
                      IF (INDEX(TRIM(dimnam),'time').ne.0) THEN
                        nrec=dimsiz
                      END IF
                    END DO
                  END IF
                END DO
              END IF
            END IF
          ELSE
            WRITE (stdout,120) TRIM(ncname)
            exit_flag=4
            RETURN
          END IF
!
!-----------------------------------------------------------------------
!  Inquire size of unlimited time record dimension.
!-----------------------------------------------------------------------
!
          tsize=0
          IF (recdim.gt.0) THEN
            status=nf_inq_dim (ncid,recdim,dimnam,tsize)
            IF (status.ne.nf_noerr) THEN
              WRITE (stdout,130) 'time', TRIM(ncname)
              exit_flag=4
              ioerror=status
              RETURN
            END IF
          END IF
!
!  Set number of record to unlimited dimension if unasigned.
!
          IF ((ifield.eq.0).and.(nrec.eq.0)) THEN
            nrec=tsize
          END IF
!
!-----------------------------------------------------------------------
!  Close input NetCDF file.
!-----------------------------------------------------------------------
!
          status=nf_close (ncid)
!
!-----------------------------------------------------------------------
!  If appropriate, scan next NetCDF file.
!-----------------------------------------------------------------------
!
        END IF
      END DO
!
  10  FORMAT (/,' OPENCDF - unable to open input NetCDF file: ',a)
  20  FORMAT (/,' OPENCDF - error while reading attribute: ',a,2x,      &
     &          ' in input NetCDF file: ',a)
  30  FORMAT (/,' OPENCDF - cannot find attribute: ',a,2x,              &
     &          ' in input NetCDF file: ',a)
  40  FORMAT (/,' OPENCDF - too small dimension parameter, MV = ',      &
     &       2i5,/,11x,'change file  mod_ncparam.F  and recompile.')
  50  FORMAT (/,' OPENCDF - error while reading dimension ID: ',i3,2x,  &
     &          ' in input NetCDF file: ',a)
  60  FORMAT (/,' OPENCDF - inconsistent size of dimension: ',a,2x,     &
     &       2i5)
  70  FORMAT (/,' OPENCDF - error while inquiring information for ',    &
     &          ' variable ID: ',i3,2x,' in input NetCDF file: ',a)
  80  FORMAT (/,' OPENCDF - error while inquiring length of',           &
     &          ' attribute: ',a)
  90  FORMAT (/,' OPENCDF - error while reading attribute: ',a)
 100  FORMAT (/,' OPENCDF - error while inquiring attributes for',      &
     &          ' variable: ',a)
 110  FORMAT (/,' OPENCDF - unable to inquire size of time dimension',  &
     &          ' in variable: ',a)
 120  FORMAT (/,' OPENCDF - unable to inquire about contents of',       &
     &          ' input NetCDF file: ',a)
 130  FORMAT (/,' OPENCDF - error inquiring dimension: ',a,2x,          &
     &          ' in input NetCDF file: ',a)
      RETURN
      END SUBROUTINE opencdf

      SUBROUTINE openids (nfiles, fname, fid, ncname, ncid)
!
!=======================================================================
!  Copyright (c) 2005 ROMS/TOMS Group                                  !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This routine determines association between pathnames and NetCDF    !
!  file descriptors. It is used to get the  ID  of an opened NetCDF    !
!  from file name.  This is done to avoid  opening  too  many files    !
!  simultaneously for the same dataset.                                !
!                                                                      !
!  Input:                                                              !
!                                                                      !
!    nfiles    Number of input NetCDF files.                           !
!    fname     NetCDF file name(s).                                    !
!    fid       NetCDF ID(s) associated with fname.                     !
!    ncname    NetCDF file name to check in association table.         !
!    ncid      NetCDF ID to check in association table.                !
!                                                                      !
!  Output:                                                             !
!                                                                      !
!    fid       If updated, it stores the ID associated with fname.     !
!    ncid      If updated, indicates that ncname has been opened       !
!                previously and this is its associated ID.             !
!                                                                      !
!=======================================================================
!
      implicit none
!
!  Imported variable declarations.
!
      integer, intent(in) :: nfiles

      character (len=*), intent(in) :: fname(nfiles)
      character (len=*), intent(in) :: ncname

      integer, intent(inout) :: fid(nfiles)
      integer, intent(inout) :: ncid
!
!  Local variabel declarations.
!
      integer :: i
!
!-----------------------------------------------------------------------
!  Determine association between NetCDF file name and ID descriptor.
!-----------------------------------------------------------------------
!
!  NetCDF file ID descriptors are set to closed state (-1) during
!  initialization.
!
      DO i=1,nfiles
        IF (TRIM(ncname).eq.TRIM(fname(i))) THEN
!
!  NCFILE was just opened. Store its associated ID for future reference.
!
          IF (ncid.ne.-1) fid(i)=ncid
!
!  NCFILE is already open. Get its ID from reference table.
!
          IF ((fid(i).ne.-1).and.(ncid.eq.-1)) ncid=fid(i)
        END IF
      END DO

      RETURN
      END SUBROUTINE openids


#if defined OFFLINE_FLOATS || defined OFFLINE_BIOLOGY
      SUBROUTINE openclms (ng, model, ifield, ncid, nfiles, fname)
!
!=======================================================================
!=======================================================================
!                                                                      !
!  When OFFLINE_FLOATS or BIOLOGY is run with a large 3D grid, it is   !
!  assumed that multiple files will contain the stored output that     !
!  is used.  This subroutines figures out which file in the series     !
!  contains the present timestep.                                      !
!                                                                      !
!  On Input:                                                           !
!                                                                      !
!     ng         Nested grid number.                                   !
!     model      Calling model identifier.                             !
!     ifield     Field ID.                                             !
!     ncid       NetCDF file ID.                                       !
!     nfiles     Number of input NetCDF files.                         !
!     fname      NetCDF file name(s).                                  !
!                                                                      !
!  On Output:                                                          !
!                                                                      !
!     ncid       If updated, indicates that we need the next file in   !
!                the series                                            !
!     fname      If updated, then next filename in the series.         !
!                                                                      !
!=======================================================================
!
      USE mod_param
      USE mod_parallel
      USE mod_iounits
      USE mod_ncparam
      USE mod_netcdf
      USE mod_scalars

      implicit none
!
!  Imported variable declarations.
!
      integer, intent(inout) :: ncid
      integer, intent(in) :: ng, model, ifield, nfiles

      character (len=*), intent(inout) :: fname
!
!  Local variable declarations.
!
      logical :: Liocycle, got_time

      integer :: Tid, Tindex, Trec, Vsize(4)
      integer :: i, lstr, ltvar, nrec, nvd, status, fnumber

      real(r8) :: Clength, Tend, Tmax, Tmin, Tscale, Tstr, Tval

      character (len=80) :: ncfile, fbase
      character (len=4)  :: fshort
!
!  Intialize local variables.
!
      IF (exit_flag.ne.NoError) RETURN
      nrec=0
      Tid=-1        
      Liocycle=.FALSE.
      got_time=.FALSE.
      ltvar=LEN_TRIM(Tname(ifield))

      IF (InpThread) THEN
!
!  Inquire about the dimensions and variables. Check for consistency.
!
        CALL opencdf (ng, nfiles, fname, ncfile, N(ng), ifield, nrec,   &
     &                  nvd, Vsize)
        IF (exit_flag.ne.NoError) RETURN
!
!  Scan variable list from input NetCDF and check time variable.
!
        DO i=1,nvars
          IF (ltvar.gt.0) THEN
            IF (TRIM(varnam(i)).eq.TRIM(Tname(ifield))) THEN
              Tid=i
              got_time=.TRUE.
            END IF
          END IF
        END DO
!
!  Terminate execution requested variables are not found.
!
        IF (.not.got_time) THEN
          WRITE (stdout,10) TRIM(Tname(ifield)), TRIM(ncfile)
          exit_flag=2
          RETURN
        END IF
!
!  If appropriate, open input NetCDF file for reading.  If processing
!  model forcing (multiple files allowed), check if file for requested
!  field has been already opened and get/save its ID from/to the
!  association table. 
!
        CALL openids (nFfiles(ng), FRCname(1,ng), FRCids(1,ng),         &
     &                ncfile, ncid)
        IF (ncid.eq.-1) THEN
          status=nf_open(TRIM(ncfile), nf_nowrite, ncid)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,20) TRIM(ncfile)
            exit_flag=2
            ioerror=status
            RETURN
          END IF
          CALL openids (nFfiles(ng), FRCname(1,ng), FRCids(1,ng),       &
     &                  ncfile, ncid)
        END IF
!
!  If we're at the end of the file, open a new climatology file.
!
        CALL get_cycle (ng, ncid, Tid, nrec, tdays(ng), Liocycle,       &
     &                  Clength, Trec, Tstr, Tend, Tmin, Tmax, Tscale)
        IF (exit_flag.ne.NoError) RETURN
        IF ( tdays(ng) .gt. Tmax ) THEN
!
!  Close current netcdf file, set ncid=-1
!
          status=nf_close(ncid)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,40) TRIM(fname)
            exit_flag=2
            RETURN
          ELSE
            WRITE(stdout,30) TRIM(fname)
            ncid =-1
          END IF
!
!  Increment the file number with some fancy switching between integers 
!  and strings.
!
          lstr = LEN_TRIM(fname)
          fshort = fname((lstr-6):(lstr-3))
          READ(fshort,*) fnumber
          fnumber = fnumber+1
          WRITE (fshort,5) fnumber
   5      FORMAT (i4.4)
          fname = fname(1:(lstr-7))//fshort//'.nc'
!
!  Open the new climatology file
!
          status=nf_open(TRIM(fname), nf_nowrite, ncid)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,20) TRIM(fname)
            exit_flag=2
            RETURN
          ELSE
            WRITE(stdout,50) TRIM(fname)
          END IF
!
!  Set the time index from new climatology file and store it where it will
!  become Trec in get_2dfld, etc.  Note: other Finfo and Info records aren't 
!  updated, either because they aren't needed (OFFLINE will never cycle around 
!  to the beginning of the file's records), or the values from the last file 
!  are good enough.
!
          Trec=0
          Iinfo(9,idSSHc,ng)=Trec
          Iinfo(9,idUbcl,ng)=Trec
          Iinfo(9,idVbcl,ng)=Trec
          DO i=1,NAT
            Iinfo(9,idTclm(i),ng)=Trec
          END DO
          Iinfo(9,idOclm,ng)=Trec
          Iinfo(9,idAKSclm,ng)=Trec
          Iinfo(9,idUclm,ng)=Trec
          Iinfo(9,idVclm,ng)=Trec

          Cinfo(idSSHc,ng)=TRIM(fname)
          Cinfo(idUbcl,ng)=TRIM(fname)
          Cinfo(idVbcl,ng)=TRIM(fname)
          DO i=1,NAT
            Cinfo(idTclm(i),ng)=TRIM(fname)
          END DO
          Cinfo(idOclm,ng)=TRIM(fname)
          Cinfo(idAKSclm,ng)=TRIM(fname)
          Cinfo(idUclm,ng)=TRIM(fname)
          Cinfo(idVclm,ng)=TRIM(fname)

          status=nf_get_var1_TYPE(ncid,Tid,1,Tmin)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,40) TRIM(Vname(1,ifield))
            exit_flag=2
            RETURN
          END IF
          Tmin = Tmin*Tscale
          Finfo(1,idSSHc,ng)=Tmin
          Finfo(1,idUbcl,ng)=Tmin
          Finfo(1,idVbcl,ng)=Tmin
          DO i=1,NAT
            Finfo(1,idTclm(i),ng)=Tmin
          END DO
          Finfo(1,idOclm,ng)=Tmin
          Finfo(1,idAKSclm,ng)=Tmin
          Finfo(1,idUclm,ng)=Tmin
          Finfo(1,idVclm,ng)=Tmin

          status=nf_get_var1_TYPE(ncid,Tid,nrec,Tmax)
          IF (status.ne.nf_noerr) THEN
            WRITE (stdout,40) TRIM(Vname(1,ifield))
            exit_flag=2
            RETURN
          END IF
          Tmax = Tmax*Tscale
          Finfo(2,idSSHc,ng)=Tmax
          Finfo(2,idUbcl,ng)=Tmax
          Finfo(2,idVbcl,ng)=Tmax
          DO i=1,NAT
            Finfo(2,idTclm(i),ng)=Tmax
          END DO
          Finfo(2,idOclm,ng)=Tmax
          Finfo(2,idAKSclm,ng)=Tmax
          Finfo(2,idUclm,ng)=Tmax
          Finfo(2,idVclm,ng)=Tmax
        END IF
      ENDIF

  10  FORMAT (/,' OPENCLMS    - unable to find requested variable: ',a, &
     &        /,15x,'in input NetCDF file: ',a)
  20  FORMAT (/,' OPENCLMS    - unable to open input NetCDF file: ',a)
  30  FORMAT (' OPENCLMS    - closing input NetCDF file: ',a)
  40  FORMAT (/,' OPENCLMS    - unable to close input NetCDF file: ',a)
  50  FORMAT (' OPENCLMS    - opening input NetCDF file: ',a)
      RETURN
      END SUBROUTINE openclms
#endif
