#include "cppdefs.h"
      MODULE white_noise_mod
!
!=======================================================================
!                                                                      !
!  These routines generates "white noise" arrays with random numbers   !
!  over specified range.  These random numbers are scaled to insure,   !
!  approximately, zero mean expectation and unit variance.             !
!                                                                      !
!  Several random number generation schemes are allowed:               !
!                                                                      !
!  Rscheme = 0     Use F90 intrinsic random numbers generator,         !
!                    0 <= R < 1                                        !
!  Rscheme = 1     Random deviates with Gaussian distribution,         !
!                    -1 < R < 1                                        !
!                                                                      !
!  Routines:                                                           !
!                                                                      !
!  white_noise2d   Random numbers for 2D arrays                        !
!  white_noise3d   Random numbers for 3D arrays                        !
!                                                                      !
!=======================================================================
!
      USE mod_kinds

      implicit none

      PRIVATE

      PUBLIC :: white_noise1d
      PUBLIC :: white_noise2d
# ifdef SOLVE3D
      PUBLIC :: white_noise3d
# endif

      CONTAINS
!
!***********************************************************************
      SUBROUTINE white_noise1d (ng, Istr, Iend, LBi, UBi,               &
     &                          Rscheme,                                &
     &                          R)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
      USE nrutil, ONLY : gasdev
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Istr, Iend
      integer, intent(in) :: LBi, UBi
      integer, intent(in) :: Rscheme

      real(r8), intent(out) :: R(LBi:UBi)
!
!  Local variable declarations.
!
      integer :: i, ic

      real(r8), parameter :: fac = 2.0_r8 * 1.73205080756887720_r8

      real(r8), dimension((UBi-LBi+1)) :: random
!
!-----------------------------------------------------------------------
!  Generate an array of random numbers.
!-----------------------------------------------------------------------
!
!  Initialize output array.
!
      DO i=LBi,UBi
        R(i)=0.0_r8
      END DO
!
!  F90 intrinsic pseudorandom numbers with an uniform distribution
!  over the range 0 <= R < 1.
!
      IF (Rscheme.eq.0) THEN
        CALL random_number (R(LBi:))
!
!  Scale (fac=2*SQRT(3)) the random number to insure the expectation
!  mean to be approximately zero, E(R)=0, and the expectation variance
!  to be approximately unity, E(R^2)=1 (Bennett, 2002; book page 72).
!
!  Recall that,
!
!      E(R) = sum[R f(R)]
!
!  where F(R) is the random variable probability distribution.
!
        DO i=Istr,Iend
          R(i)=fac*(R(i)-0.5_r8)
        END DO
!
!  Random deviates with a Gaussian (normal) distribuiton over the
!  range from -1 to 1.
!
      ELSE IF (Rscheme.eq.1) THEN
        CALL gasdev (random)
        ic=0
        DO i=Istr,Iend
          ic=ic+1
          R(i)=random(ic)
        END DO
      END IF

      RETURN
      END SUBROUTINE white_noise1d

!
!***********************************************************************
      SUBROUTINE white_noise2d (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj,                     &
     &                          Rscheme,                                &
     &                          R)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
      USE nrutil, ONLY : gasdev
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj
      integer, intent(in) :: Rscheme

      real(r8), intent(out) :: R(LBi:UBi,LBj:UBj)
!
!  Local variable declarations.
!
      integer :: i, ic, j

      real(r8), parameter :: fac = 2.0_r8 * 1.73205080756887720_r8

      real(r8), dimension((UBi-LBi+1)*(UBj-LBj+1)) :: random
!
!-----------------------------------------------------------------------
!  Generate an array of random numbers.
!-----------------------------------------------------------------------
!
!  Initialize output array.
!
      DO j=LBj,UBj
        DO i=LBi,UBi
          R(i,j)=0.0_r8
        END DO
      END DO
!
!  F90 intrinsic pseudorandom numbers with an uniform distribution
!  over the range 0 <= R < 1.
!
      IF (Rscheme.eq.0) THEN
        CALL random_number (R(LBi:,LBj:))
!
!  Scale (fac=2*SQRT(3)) the random number to insure the expectation
!  mean to be approximately zero, E(R)=0, and the expectation variance
!  to be approximately unity, E(R^2)=1 (Bennett, 2002; book page 72).
!
!  Recall that,
!
!      E(R) = sum[R f(R)]
!
!  where F(R) is the random variable probability distribution.
!
        DO j=Jstr,Jend
          DO i=Istr,Iend
            R(i,j)=fac*(R(i,j)-0.5_r8)
          END DO
        END DO
!
!  Random deviates with a Gaussian (normal) distribuiton with zero
!  mean and unit variance.
!
      ELSE IF (Rscheme.eq.1) THEN
        CALL gasdev (random)
        ic=0
        DO j=Jstr,Jend
          DO i=Istr,Iend
            ic=ic+1
            R(i,j)=random(ic)
          END DO
        END DO
      END IF

      RETURN
      END SUBROUTINE white_noise2d

#ifdef SOLVE3D
!
!***********************************************************************
      SUBROUTINE white_noise3d (ng, Istr, Iend, Jstr, Jend,             &
     &                          LBi, UBi, LBj, UBj, LBk, UBk,           &
     &                          Rscheme,                                &
     &                          R)
!***********************************************************************
!
      USE mod_param
      USE mod_scalars
!
      USE nrutil, ONLY : gasdev
!
!  Imported variable declarations.
!
      integer, intent(in) :: ng, Istr, Iend, Jstr, Jend
      integer, intent(in) :: LBi, UBi, LBj, UBj, LBk, UBk
      integer, intent(in) :: Rscheme

      real(r8), intent(out) :: R(LBi:UBi,LBj:UBj,LBk:UBk)
!
!  Local variable declarations.
!
      integer :: i, ic, j, k

      real(r8), parameter :: fac = 2.0_r8 * 1.73205080756887720_r8

      real(r8), dimension((UBi-LBi+1)*(UBj-LBj+1)*(UBk-LBk+1)) :: random
!
!-----------------------------------------------------------------------
!  Generate an array of random numbers.
!-----------------------------------------------------------------------
!
!  Initialize output array.
!
      DO k=LBk,UBk
        DO j=LBj,UBj
          DO i=LBi,UBi
            R(i,j,k)=0.0_r8
          END DO
        END DO
      END DO
!
!  F90 intrinsic pseudorandom numbers with an uniform distribution
!  over the range 0 <= R < 1.
!
      IF (Rscheme.eq.0) THEN
        CALL random_number (R(LBi:,LBj:,LBk:))
!
!  Scale (fac=2*SQRT(3)) the random number to insure the expectation
!  mean to be approximately zero, E(R)=0, and the expectation variance
!  to be approximately unity, E(R^2)=1.
!
!  Recall that,
!
!      E(R) = sum[R f(R)]
!
!  where F(R) is the random variable probability distribution
!
        DO k=LBk,UBk
          DO j=Jstr,Jend
            DO i=Istr,Iend
              R(i,j,k)=fac*(R(i,j,k)-0.5_r8)
            END DO
          END DO
        END DO
!
!  Random deviates with a Gaussian (normal) distribution with zero
!  mean and unit variance.
!
      ELSE IF (Rscheme.eq.1) THEN
        CALL gasdev (random)
        ic=0
        DO k=LBk,UBk
          DO j=Jstr,Jend
            DO i=Istr,Iend
              ic=ic+1
              R(i,j,k)=random(ic)
            END DO
          END DO
        END DO
      END IF

      RETURN
      END SUBROUTINE white_noise3d
#endif

      END MODULE white_noise_mod
