/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.util.cache;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Formatter;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.util.CancelTask;
import ucar.nc2.util.cache.FileCacheable;
import ucar.nc2.util.cache.FileFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
@ThreadSafe
public class FileCache {
    private static Logger log = LoggerFactory.getLogger(FileCache.class);
    private static ScheduledExecutorService exec;
    private static boolean debugPrint;
    private static boolean debugCleanup;
    private final int softLimit;
    private final int minElements;
    private final int hardLimit;
    private final ConcurrentHashMap<Object, CacheElement> cache;
    private final ConcurrentHashMap<Object, CacheElement.CacheFile> files;
    private final AtomicInteger counter = new AtomicInteger();
    private final AtomicBoolean hasScheduled = new AtomicBoolean(false);
    private final AtomicBoolean disabled = new AtomicBoolean(false);
    private final AtomicInteger cleanups = new AtomicInteger();
    private final AtomicInteger hits = new AtomicInteger();
    private final AtomicInteger miss = new AtomicInteger();

    public static void shutdown() {
        if (exec != null) {
            exec.shutdown();
        }
        exec = null;
    }

    public FileCache(int minElementsInMemory, int maxElementsInMemory, int period) {
        this(minElementsInMemory, maxElementsInMemory, -1, period);
    }

    public FileCache(int minElementsInMemory, int softLimit, int hardLimit, int period) {
        this.minElements = minElementsInMemory;
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
        this.cache = new ConcurrentHashMap(2 * softLimit, 0.75f, 8);
        this.files = new ConcurrentHashMap(4 * softLimit, 0.75f, 8);
        if (period > 0) {
            if (exec == null) {
                exec = Executors.newSingleThreadScheduledExecutor();
            }
            exec.scheduleAtFixedRate(new CleanupTask(), period, period, TimeUnit.SECONDS);
        }
    }

    public void disable() {
        this.disabled.set(true);
        this.clearCache(true);
    }

    public void enable() {
        this.disabled.set(false);
    }

    public FileCacheable acquire(FileFactory factory, String location, CancelTask cancelTask) throws IOException {
        return this.acquire(factory, location, location, -1, cancelTask, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public FileCacheable acquire(FileFactory factory, Object hashKey, String location, int buffer_size, CancelTask cancelTask, Object spiObject) throws IOException {
        int count;
        CacheElement elem;
        FileCacheable ncfile;
        if (null == hashKey) {
            hashKey = location;
        }
        if ((ncfile = this.acquireCacheOnly(hashKey)) != null) {
            this.hits.incrementAndGet();
            return ncfile;
        }
        this.miss.incrementAndGet();
        ncfile = factory.open(location, buffer_size, cancelTask, spiObject);
        if (log.isDebugEnabled()) {
            log.debug("FileCache.acquire " + hashKey + " " + ncfile.getLocation());
        }
        if (debugPrint) {
            System.out.println("  FileCache.acquire " + hashKey + " " + ncfile.getLocation());
        }
        if (cancelTask != null && cancelTask.isCancel()) {
            if (ncfile != null) {
                ncfile.close();
            }
            return null;
        }
        if (this.disabled.get()) {
            return ncfile;
        }
        Object object = this.cache;
        synchronized (object) {
            elem = this.cache.get(hashKey);
            if (elem == null) {
                this.cache.put(hashKey, new CacheElement(ncfile, hashKey));
            }
        }
        if (elem != null) {
            object = elem;
            synchronized (object) {
                elem.addFile(ncfile);
            }
        }
        if ((count = this.counter.incrementAndGet()) > this.hardLimit && this.hardLimit > 0) {
            if (debugCleanup) {
                System.out.println("CleanupTask due to hard limit time=" + new Date().getTime());
            }
            this.hasScheduled.getAndSet(true);
            this.cleanup(this.hardLimit);
        } else if (count > this.softLimit && this.hasScheduled.compareAndSet(false, true)) {
            exec.schedule(new CleanupTask(), 100L, TimeUnit.MILLISECONDS);
            if (debugCleanup) {
                System.out.println("CleanupTask scheduled due to soft limit time=" + new Date().getTime());
            }
        }
        return ncfile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileCacheable acquireCacheOnly(Object hashKey) {
        if (this.disabled.get()) {
            return null;
        }
        FileCacheable ncfile = null;
        CacheElement elem = this.cache.get(hashKey);
        if (elem != null) {
            CacheElement cacheElement = elem;
            synchronized (cacheElement) {
                for (CacheElement.CacheFile file : elem.list) {
                    if (!file.isLocked.compareAndSet(false, true)) continue;
                    ncfile = file.ncfile;
                    break;
                }
            }
        }
        if (ncfile != null) {
            try {
                ncfile.sync();
                if (log.isDebugEnabled()) {
                    log.debug("FileCache.aquire from cache " + hashKey + " " + ncfile.getLocation());
                }
                if (debugPrint) {
                    System.out.println("  FileCache.aquire from cache " + hashKey + " " + ncfile.getLocation());
                }
            }
            catch (IOException e) {
                log.error("FileCache.synch failed on " + ncfile.getLocation() + " " + e.getMessage());
            }
        }
        return ncfile;
    }

    public void release(FileCacheable ncfile) throws IOException {
        if (ncfile == null) {
            return;
        }
        if (this.disabled.get()) {
            ncfile.setFileCache(null);
            ncfile.close();
            return;
        }
        CacheElement.CacheFile file = this.files.get(ncfile);
        if (file != null) {
            if (!file.isLocked.get()) {
                log.warn("FileCache.release " + ncfile.getLocation() + " not locked");
            }
            file.lastAccessed = System.currentTimeMillis();
            ++file.countAccessed;
            file.isLocked.set(false);
            if (log.isDebugEnabled()) {
                log.debug("FileCache.release " + ncfile.getLocation());
            }
            if (debugPrint) {
                System.out.println("  FileCache.release " + ncfile.getLocation());
            }
            return;
        }
        throw new IOException("FileCache.release does not have file in cache = " + ncfile.getLocation());
    }

    Map<Object, CacheElement> getCache() {
        return this.cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void clearCache(boolean force) {
        ArrayList<Object> deleteList = new ArrayList<Object>(2 * this.cache.size());
        if (force) {
            this.cache.clear();
            deleteList.addAll(this.files.values());
            this.files.clear();
        } else {
            Object file;
            Iterator<CacheElement.CacheFile> iter = this.files.values().iterator();
            while (iter.hasNext()) {
                file = iter.next();
                if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
                ((CacheElement.CacheFile)file).remove();
                deleteList.add(file);
                iter.remove();
            }
            file = this.cache;
            synchronized (file) {
                Iterator<CacheElement> i$ = this.cache.values().iterator();
                while (i$.hasNext()) {
                    CacheElement elem;
                    CacheElement cacheElement = elem = i$.next();
                    synchronized (cacheElement) {
                        if (elem.list.size() == 0) {
                            this.cache.remove(elem.hashKey);
                        }
                    }
                }
            }
        }
        for (Object file : deleteList) {
            if (force && ((CacheElement.CacheFile)file).isLocked.get()) {
                log.warn("FileCache force close locked file= " + file);
            }
            this.counter.decrementAndGet();
            try {
                ((CacheElement.CacheFile)file).ncfile.setFileCache(null);
                ((CacheElement.CacheFile)file).ncfile.close();
                ((CacheElement.CacheFile)file).ncfile = null;
            }
            catch (IOException e) {
                log.error("FileCache.close failed on " + file);
            }
        }
        log.debug("*FileCache.clearCache force= " + force + " deleted= " + deleteList.size() + " left=" + this.counter.get());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void showCache(Formatter format) {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.counter.get());
        Iterator<CacheElement> i$ = this.cache.values().iterator();
        while (i$.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = i$.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        for (CacheElement.CacheFile file : allFiles) {
            format.format("  %s\n", file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<String> showCache() {
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.counter.get());
        Iterator<CacheElement> i$ = this.cache.values().iterator();
        while (i$.hasNext()) {
            CacheElement elem;
            CacheElement cacheElement = elem = i$.next();
            synchronized (cacheElement) {
                allFiles.addAll(elem.list);
            }
        }
        Collections.sort(allFiles);
        ArrayList<String> result = new ArrayList<String>(allFiles.size());
        for (CacheElement.CacheFile file : allFiles) {
            result.add(file.toString());
        }
        return result;
    }

    public void showStats(Formatter format) {
        format.format("  hits= %d miss= %d nfiles= %d elems= %d\n", this.hits.get(), this.miss.get(), this.counter.get(), this.cache.values().size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void cleanup(int maxElements) {
        Object file;
        if (this.disabled.get()) {
            return;
        }
        int size = this.counter.get();
        if (size <= this.minElements) {
            return;
        }
        log.debug("FileCache.cleanup started at " + new Date() + " for cleanup maxElements=" + maxElements);
        if (debugCleanup) {
            System.out.println("FileCache.cleanup started at " + new Date().getTime() + " for cleanup maxElements=" + maxElements);
        }
        this.cleanups.incrementAndGet();
        ArrayList<CacheElement.CacheFile> allFiles = new ArrayList<CacheElement.CacheFile>(this.counter.get());
        for (CacheElement.CacheFile file2 : this.files.values()) {
            if (file2.isLocked.get()) continue;
            allFiles.add(file2);
        }
        Collections.sort(allFiles);
        int need2delete = size - this.minElements;
        int minDelete = size - maxElements;
        ArrayList<Object> deleteList = new ArrayList<Object>(need2delete);
        int count = 0;
        Iterator iter = allFiles.iterator();
        while (iter.hasNext() && count < need2delete) {
            file = (CacheElement.CacheFile)iter.next();
            if (!((CacheElement.CacheFile)file).isLocked.compareAndSet(false, true)) continue;
            ((CacheElement.CacheFile)file).remove();
            deleteList.add(file);
            ++count;
        }
        if (count < minDelete) {
            log.warn("FileCache.cleanup couldnt removed enough to keep under the maximum= " + maxElements + " due to locked files; currently at = " + (size - count));
            if (debugCleanup) {
                System.out.println("FileCache.cleanup couldnt removed enough to keep under the maximum= " + maxElements + " due to locked files; currently at = " + (size - count));
            }
        }
        file = this.cache;
        synchronized (file) {
            Iterator<CacheElement> i$ = this.cache.values().iterator();
            while (i$.hasNext()) {
                CacheElement elem;
                CacheElement cacheElement = elem = i$.next();
                synchronized (cacheElement) {
                    if (elem.list.size() == 0) {
                        this.cache.remove(elem.hashKey);
                    }
                }
            }
        }
        long start = System.currentTimeMillis();
        for (CacheElement.CacheFile cacheFile : deleteList) {
            this.counter.decrementAndGet();
            this.files.remove(cacheFile.ncfile);
            try {
                cacheFile.ncfile.setFileCache(null);
                cacheFile.ncfile.close();
                cacheFile.ncfile = null;
            }
            catch (IOException e) {
                log.error("FileCache.close failed on " + cacheFile.getCacheName());
            }
        }
        long took = System.currentTimeMillis() - start;
        log.debug("FileCache.cleanup had= " + size + " removed= " + deleteList.size() + " took=" + took + " msec");
        if (debugCleanup) {
            System.out.println("FileCache.cleanup had= " + size + " removed= " + deleteList.size() + " took=" + took + " msec");
        }
        this.hasScheduled.set(false);
    }

    static {
        debugPrint = false;
        debugCleanup = false;
    }

    private class CleanupTask
    implements Runnable {
        private CleanupTask() {
        }

        public void run() {
            FileCache.this.cleanup(FileCache.this.softLimit);
        }
    }

    class CacheElement {
        @GuardedBy(value="this")
        List<CacheFile> list = new LinkedList<CacheFile>();
        final Object hashKey;

        CacheElement(FileCacheable ncfile, Object hashKey) {
            this.hashKey = hashKey;
            CacheFile file = new CacheFile(ncfile);
            this.list.add(file);
            FileCache.this.files.put(ncfile, file);
            if (log.isDebugEnabled()) {
                log.debug("CacheElement add to cache " + hashKey);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        CacheFile addFile(FileCacheable ncfile) {
            CacheFile file = new CacheFile(ncfile);
            CacheElement cacheElement = this;
            synchronized (cacheElement) {
                this.list.add(file);
            }
            FileCache.this.files.put(ncfile, file);
            return file;
        }

        public String toString() {
            return this.hashKey + " count=" + this.list.size();
        }

        /*
         * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
         */
        class CacheFile
        implements Comparable<CacheFile> {
            FileCacheable ncfile;
            final AtomicBoolean isLocked = new AtomicBoolean(true);
            int countAccessed = 1;
            long lastAccessed = 0L;

            private CacheFile(FileCacheable ncfile) {
                this.ncfile = ncfile;
                this.lastAccessed = System.currentTimeMillis();
                ncfile.setFileCache(FileCache.this);
                if (log.isDebugEnabled()) {
                    log.debug("FileCache add to cache " + CacheElement.this.hashKey);
                }
                if (debugPrint) {
                    System.out.println("  FileCache add to cache " + CacheElement.this.hashKey);
                }
            }

            String getCacheName() {
                return this.ncfile.getLocation();
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            void remove() {
                CacheElement cacheElement = CacheElement.this;
                synchronized (cacheElement) {
                    CacheElement.this.list.remove(this);
                }
                if (log.isDebugEnabled()) {
                    log.debug("FileCache.remove " + this.ncfile.getLocation());
                }
                if (debugPrint) {
                    System.out.println("  FileCache.remove " + this.ncfile.getLocation());
                }
            }

            public String toString() {
                return this.isLocked + " " + CacheElement.this.hashKey + " " + this.countAccessed + " " + new Date(this.lastAccessed);
            }

            @Override
            public int compareTo(CacheFile o) {
                return (int)(this.lastAccessed - o.lastAccessed);
            }
        }
    }
}

