/**
 * DapperBackendService
 */
package gov.noaa.pmel.tmap.las.service.dapper;

import com.cohort.util.Calendar2;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.pointdata.Table;

import gov.noaa.pmel.tmap.las.exception.LASException;
import gov.noaa.pmel.tmap.las.jdom.JDOMUtils;
import gov.noaa.pmel.tmap.las.jdom.LASDapperBackendRequest;
import gov.noaa.pmel.tmap.las.jdom.LASBackendResponse;
import gov.noaa.pmel.tmap.las.service.BackendService;

import java.io.File;
import java.io.IOException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.jdom.JDOMException;

/**
 * This class represents an LAS BackendService (a provider of data), in this case, a Dapper
 * server (http://www.epic.noaa.gov/epic/software/dapper/). 
 *
 * @author Roland Schweitzer
 * @author Bob Simons (bob.simons@noaa.gov)
 *
 */
public class DapperBackendService extends BackendService {
    private static Logger log = LogManager.getLogger(DapperBackendService.class.getName());

    /**
     * This processes the backendRequestXML and returns the requested data.
     *
     * @param dapperBackendRequestXML the information from an LASDapperBackendRequest.xml file.
     * @param cacheFileName [I'm not sure what this is for. 
     *     It may be a name that this method can used for a cacheFile, if one is needed.]
     *     Currently, this is not used.
     * @return the results of the request (from lasBackendResponse.toString()).
     *    This indicates either success and failure.
     */
    public String getProduct(String dapperBackendRequestXML, String cacheFileName) 
            throws Exception, LASException, IOException, JDOMException {       

        //convert the backendRequestXML to an lasBackendRequest
        LASDapperBackendRequest dapperBackendRequest = new LASDapperBackendRequest();      
        JDOMUtils.XML2JDOM(dapperBackendRequestXML, dapperBackendRequest);        
        
        //report logging level only for "debug" and "trace" levels
        String debug = dapperBackendRequest.getProperty("las", "debug");      
        setLogLevel(debug);
        log.debug("Logging set to " + log.getEffectiveLevel().toString() + " for " + log.getName());
        
        //get the lasBackendResponse
        DapperTool tool = new DapperTool();
        LASBackendResponse lasBackendResponse = tool.run(dapperBackendRequest);
        return lasBackendResponse.toString();
    }

    /** This is the directory where the test results files will be placed. */
    public static String testResultsDirectory = "C:\\";

    /** This is the file name for the test results file (should match the name in the xml info). */
    public static String testResultsNcName = "TestDapperBackendService.nc";

    /** 
     * This is the LASDapperBackendRequest.xml info that is used to test this class via main(null). 
     * The tags which are used by this method (either required or optional) are marked below.
     * Other tags may be used by other parts of LAS.
     *
     * <p>This sample url should be generated by this xml.
     *   This url doesn't work in a browser, only via opendap library:
     * url=http://las.pfeg.noaa.gov/dods/ndbc/all_noaa_time_series.cdp?location.LON,location.LAT,
     * location.DEPTH,location.profile.TIME,location.profile.WSPD1,location.profile.BAR1
     * &location.LON>=235.45001&location.LON<=235.47&location.LAT>=40.77&location.LAT<=40.789997
     * &location.profile.TIME>=1072915199999&location.profile.TIME<=1072918800001
     */
    public static String testRequestXml = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<backend_request>\n" +
" <region ID=\"region_0\">\n" +              //required
"   <x_lo>235.45001</x_lo>\n" +              //all lo and hi tags are optional
"   <x_hi>235.47</x_hi>\n" +
"   <y_lo>40.77</y_lo>\n" +
"   <y_hi>40.789997</y_hi>\n" +
"   <t_lo>01-Jan-2004 00:00:00</t_lo>\n" +
"   <t_hi>01-Jan-2004 01:00:00</t_hi>\n" +
" </region>\n" +
" <properties>\n" +
"   <property_group type=\"las\">\n" +
"     <property>\n" +
"       <name>debug</name>\n" +
"       <value>debug</value>\n" +            //required, a log4j logging level
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"dapper\">\n" +
"     <property>\n" +
"       <name>service_action</name>\n" +
"       <value>dapper</value>\n" +
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"operation\">\n" +
"     <property>\n" +
"       <name>service</name>\n" +
"       <value>dapper</value>\n" +
"     </property>\n" +
"     <property>\n" +
"       <name>service_action</name>\n" +  //required
"       <value>dapper</value>\n" +        //required
"     </property>\n" +
"     <property>\n" +
"       <name>name</name>\n" +
"       <value>Database Extraction</value>\n" +
"     </property>\n" +
"     <property>\n" +
"       <name>ID</name>\n" +
"       <value>DapperExtract</value>\n" +
"     </property>\n" +
"     <property>\n" +
"       <name>key</name>\n" +
"       <value>F9B6C974354E316D391E061962284986</value>\n" +
"     </property>\n" +
"   </property_group>\n" +
" </properties>\n" +
" <dataObjects>\n" +
"   <data url=\"http://las.pfeg.noaa.gov/dods/ndbc/all_noaa_time_series.cdp\" \n" +  //required (especially 'url')
  "var=\"location.profile.WSPD1\" title=\"Wind Speed\" xpath=\"/lasdata/datasets/ndbcMet/variables/ndbcMet_wspd1\" \n" +
  "dataset_name=\"NOAA Time Series\" dataset_ID=\"ndbcMet\" units=\"m/sec\" name=\"Wind Speed\" ID=\"ndbcMet_ev\" \n" +
  "points=\"xyzt\" intervals=\"xyzt\" grid_type=\"scattered\">\n" +
"     <region IDREF=\"region_0\" />\n" +
"     <properties>\n" +
"       <property_group type=\"database_access\">\n" +
"         <property>\n" +                           //required? DapperTool always returns positive down
"           <name>positive</name>\n" +
"           <value>down</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>time</name>\n" +                  //required
"           <value>location.profile.TIME</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>depth_units</name>\n" +           //required? DapperTool always returns depth in meters
"           <value>meters</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>db_table</name>\n" +
"           <value>all_noaa_time_series.cdp</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>depth</name>\n" +                 //required
"           <value>location.DEPTH</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>db_title</name>\n" +
"           <value>NOAA Time Series</value>\n" +
"         </property>\n" +
//"         <property>\n" +
//"           <name>missing</name>\n" +               //not used?
//"           <value>NaN</value>\n" +
//"         </property>\n" +
"         <property>\n" +
"           <name>lon_domain</name>\n" +            //required
"           <value>0:360</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>time_units</name>\n" +            //required
"           <value>msec since 1970-01-01 00:00:00</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>longitude</name>\n" +             //required
"           <value>location.LON</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>db_name</name>\n" +
"           <value>NOAA Time Series</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>db_server</name>\n" +
"           <value>PFEG Server</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>time_type</name>\n" +             //required
"           <value>double</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>db_type</name>\n" +
"           <value>dapper</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>latitude</name>\n" +              //required
"           <value>location.LAT</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"ui\">\n" +
"         <property>\n" +
"           <name>default</name>\n" +
"           <value>file:ui.xml#nwioos_demo</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"product_server\">\n" +
"         <property>\n" +
"           <name>ui_timeout</name>\n" +
"           <value>1000</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>ps_timeout</name>\n" +
"           <value>3600</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>use_cache</name>\n" +
"           <value>false</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"     </properties>\n" +
"   </data>\n" +

//Test getting other variables via other <data> objects.
//!!!This BackendService only looks at 'var'. Everything else is assumed to be the same!
"   <data url=\"http://las.pfeg.noaa.gov/dods/ndbc/all_noaa_time_series.cdp\" \n" +  //required (especially 'url')
  "var=\"location.profile.BAR1\" title=\"Sea Level Pressure\" xpath=\"/lasdata/datasets/ndbcMet/variables/ndbcMet_bar1\" \n" +
  "dataset_name=\"NOAA Time Series\" dataset_ID=\"ndbcMet\" units=\"hPa\" name=\"Sea Level Pressure\" ID=\"ndbcMet_bar1\" \n" +
  "points=\"xyzt\" intervals=\"xyzt\" grid_type=\"scattered\">\n" +
"     <region IDREF=\"region_0\" />\n" +
"   </data>\n" +

  " </dataObjects>\n" +
" <response ID=\"DBExtractResponse\">\n" +
"   <result type=\"debug\" ID=\"db_debug\" \n" +   //recommended
  "file=\"" + testResultsDirectory + "F9B6C974354E316D391E061962284986_db_debug.txt\" \n" +
  "index=\"0\" />\n" +
"   <result type=\"netCDF\" ID=\"netcdf\" \n" +    //results file; required 
  "file=\"" + testResultsDirectory + testResultsNcName + "\" \n" +
  "index=\"1\" />\n" +
" </response>\n" +
"</backend_request>\n";

    /**
     * This runs DapperBackendService with the xml file you specify, or with
     * a standard test xml file.
     * This method prints getProduct's result to System.out.
     * This puts the results in a file called testResultsDirectory + testResultsName.
     *
     * @param args If present, args[0] should be the name of a LASDapperBackendRequest.xml file.
     *   If args==null or args.length==0, this runs a standard test.
     * @throws Exception if there is an error
     */
    public static void main(String args[]) throws Exception {

        //get the requestXml
        String2.log("\n*** DapperBackendService");
        long time = System.currentTimeMillis();
        String requestXml = testRequestXml;
        boolean doStandardTest = true;
        if (args != null && args.length > 0) {
            doStandardTest = false;
            String results[] = String2.readFromFile(args[0]);
            if (results[0].equals(""))
                requestXml = results[1];
            else throw new Exception(results[0]);
        }

        //run DapperBackendService
        DapperBackendService service = new DapperBackendService();
        String resultXml = service.getProduct(requestXml, null);
        System.out.println("resultXml=\n" + resultXml); 

        //check the results of the standard test
        if (doStandardTest) {
            //read the .nc file
            float lon = 235.460007f; //exact values from just get LON and LAT values available
            float lat = 40.779999f;
            long time1 = Calendar2.newGCalendarZulu(2004, 1, 1).getTimeInMillis();
            long time2 = time1 + Calendar2.MILLIS_PER_HOUR;
            Table table = new Table();
            table.readFlatNc(testResultsDirectory + testResultsNcName, null, 1);
            String2.log(table.toString());
            Test.ensureEqual(table.nColumns(), 6, "");
            Test.ensureEqual(table.nRows(), 2, "");
            Test.ensureEqual(table.getColumnName(1), "LON", "");  //note that LON and LAT are dataset order, 
            Test.ensureEqual(table.getColumnName(0), "LAT", "");  //not in order I requested!
            Test.ensureEqual(table.getColumnName(2), "DEPTH", "");
            Test.ensureEqual(table.getColumnName(3), "TIME", "");
            Test.ensureEqual(table.getColumnName(4), "WSPD1", "");
            Test.ensureEqual(table.getColumnName(5), "BAR1", "");
            Test.ensureEqual(table.getColumn(0).getElementTypeString(), "float", "");
            Test.ensureEqual(table.getColumn(3).getElementTypeString(), "double", "");
            Test.ensureEqual(table.getColumn(4).getElementTypeString(), "float", "");
            Test.ensureEqual(table.getColumn(5).getElementTypeString(), "float", "");
            Test.ensureEqual(table.globalAttributes().getString("Conventions"), 
                "LAS Intermediate netCDF File, Unidata Observation Dataset v1.0", "");  //cf? coards? epic in-situ?
            Test.ensureEqual(table.globalAttributes().get("lat_range").toString(), "32.43, 42.75", "");
            Test.ensureEqual(table.getFloatData(1, 0), lon, ""); //again, LON and LAT are reversed column order
            Test.ensureEqual(table.getFloatData(1, 1), lon, "");
            Test.ensureEqual(table.getFloatData(0, 0), lat, "");
            Test.ensureEqual(table.getFloatData(0, 1), lat, "");
            //outer attributes...
            Test.ensureEqual(table.columnAttributes(0).getString("units"), "degrees_north", ""); 
            Test.ensureEqual(table.columnAttributes(0).getString("long_name"), "Latitude", "");
            Test.ensureEqual(table.columnAttributes(0).getDouble("missing_value"), Double.NaN, "");
            Test.ensureEqual(table.columnAttributes(0).getString("axis"), "Y", "");
            //inner attributes...
            Test.ensureEqual(table.columnAttributes(2).getString("_CoordinateAxisType"), "Depth", "");
            Test.ensureEqual(table.columnAttributes(2).getString("axis"), "Z", "");
            Test.ensureEqual(table.columnAttributes(2).getString("long_name"), "DEPTH", "");
            Test.ensureEqual(table.columnAttributes(2).getString("positive"), "down", "");
            Test.ensureEqual(table.columnAttributes(2).getString("units"), "meters", ""); 

            Test.ensureEqual(table.columnAttributes(3).getString("_CoordinateAxisType"), "Time", "");
            Test.ensureEqual(table.columnAttributes(3).getString("axis"), "T", "");
            Test.ensureEqual(table.columnAttributes(3).getString("long_name"), "Time", "");
            Test.ensureEqual(table.columnAttributes(3).getString("time_origin"), "01-Jan-1970 00:00:00", ""); 
            Test.ensureEqual(table.columnAttributes(3).getString("units"), "msec since 1970-01-01 00:00:00", ""); //GMT?

            Test.ensureEqual(table.getDoubleData(3, 0), time1, "");
            Test.ensureEqual(table.getDoubleData(3, 1), time2, "");

            //data col 1:  wspd1
            Test.ensureEqual(table.columnAttributes(4).getString("units"), "m s^-1", "");  //units are from datasource! ok?
            Test.ensureEqual(table.columnAttributes(4).getString("long_name"), "wind speed", ""); 
            Test.ensureEqual(table.getFloatData(4, 0), 2.0f, ""); //wspd1
            Test.ensureEqual(table.getFloatData(4, 1), 5.9f, "");
            //data col 2: bar
            Test.ensureEqual(table.columnAttributes(5).getString("units"), "hpa", ""); //units are from datasource! ok?
            Test.ensureEqual(table.columnAttributes(5).getString("long_name"), "sea level pressure", ""); 
            Test.ensureEqual(table.getFloatData(5, 0), 1010.7f, ""); //bar
            Test.ensureEqual(table.getFloatData(5, 1), 1010.4f, "");

//Need test of request that returns no data. Is table as desired?
//Need test of xLo > xHi, so split lon request.
        }

        //it is possible to pass the standard test but get an error code back from the service
        Test.ensureTrue(resultXml.indexOf("type=\"error\"") < 0, 
            String2.ERROR + " in resultXml=\n" + resultXml);

        String2.log("  DapperBackendService finished successfully. TIME=" +
            (System.currentTimeMillis() - time));
    }
}
