/**
 * TabledapBackendService
 */
package gov.noaa.pmel.tmap.las.service.tabledap;

import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.pointdata.Table;

import gov.noaa.pmel.tmap.las.exception.LASException;
import gov.noaa.pmel.tmap.las.jdom.JDOMUtils;
import gov.noaa.pmel.tmap.las.jdom.LASBackendRequest;
import gov.noaa.pmel.tmap.las.jdom.LASBackendResponse;
import gov.noaa.pmel.tmap.las.service.BackendService;

import java.io.File;
import java.io.IOException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.jdom.JDOMException;

/**
 * This class represents an LAS BackendService (a provider of data), in this case, 
 * an ERDDAP Tabledap dataset 
 * (http://coastwatch.pfel.noaa.gov/erddap/tabledap/index.html). 
 *
 * @author Roland Schweitzer
 * @author Bob Simons (bob.simons@noaa.gov)
 *
 */
public class TabledapBackendService extends BackendService {
    private static Logger log = LogManager.getLogger(TabledapBackendService.class.getName());

    /**
     * This processes the backendRequestXML and returns the requested data.
     *
     * @param backendRequestXML the information from an LASBackendRequest.xml file.
     * @param cacheFileName [I'm not sure what this is for. 
     *     It may be a name that this method can used for a cacheFile, if one is needed.]
     *     Currently, this is not used.
     * @return the results of the request (from lasBackendResponse.toString()).
     *    This indicates either success and failure.
     */
    public String getProduct(String backendRequestXML, String cacheFileName) 
            throws Exception, LASException, IOException, JDOMException {       

        //convert the backendRequestXML to an lasBackendRequest
        LASBackendRequest backendRequest = new LASBackendRequest();      
        JDOMUtils.XML2JDOM(backendRequestXML, backendRequest);        
        
        //report logging level only for "debug" and "trace" levels
        String tLevel = backendRequest.getProperty("las", "debug");      
        setLogLevel(tLevel);
        log.debug("Logging set to " + log.getEffectiveLevel().toString() + " for " + log.getName());
        
        //get the lasBackendResponse
        TabledapTool tool = new TabledapTool();
        LASBackendResponse lasBackendResponse = tool.run(backendRequest);
        return lasBackendResponse.toString();
    }

    /** This is the directory where the test results files will be placed. */
    public static String testResultsDirectory = "C:\\";

    /** This is the file name for the test results file (should match the name in the xml info). */
    public static String testResultsNcName = "TestTabledapBackendService.nc";

    /** 
     * This is the LASBackendRequest.xml info that is used to test this class via main(null). 
     * The tags which are used by this method (either required or optional) are marked below.
     * Other tags may be used by other parts of LAS.
     *
     * <p>This sample url should be generated by this xml.
http://coastwatch.pfel.noaa.gov/erddap/tabledap/pmelTao?
longitude,latitude,time,sea_surface_temperature,air_temperature&longitude>264.9&longitude<265.1&latitude>4.9&latitude<5.0&time>2007-01-01&time<2007-02-01
     */
    public static String testRequestXml = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<backend_request>\n" +
" <region ID=\"region_0\">\n" +              //region_0 is required
"   <x_lo>264.9</x_lo>\n" +                  //all lo and hi tags are optional
"   <x_hi>265.1</x_hi>\n" +                  //tao station lon lat change every week, so don't ask for exact value
"   <y_lo>4.9</y_lo>\n" +
"   <y_hi>5.0</y_hi>\n" +
"   <t_lo>01-Jan-2007 00:00:00</t_lo>\n" +
"   <t_hi>01-Feb-2007 00:00:00</t_hi>\n" +
" </region>\n" +
" <properties>\n" +
"   <property_group type=\"las\">\n" +
"     <property>\n" +
"       <name>debug</name>\n" +
"       <value>debug</value>\n" +            //required, a log4j logging level  
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"tabledap\">\n" +
"     <property>\n" +
"       <name>service_action</name>\n" +
"       <value>tabledap</value>\n" +
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"operation\">\n" +
"     <property>\n" +
"       <name>service</name>\n" +
"       <value>ERD Server</value>\n" +  //required, see .../JavaSource/resources/tabledap/TabledapBackendConfig.xml
"     </property>\n" +                     
"     <property>\n" +
"       <name>service_action</name>\n" +  //required
"       <value>tabledap</value>\n" +        //required
"     </property>\n" +
"     <property>\n" +  //what is this?
"       <name>name</name>\n" +
"       <value>Database Extraction</value>\n" +
"     </property>\n" +
"     <property>\n" +  //what is this?
"       <name>ID</name>\n" +
"       <value>TabledapExtract</value>\n" +
"     </property>\n" +
"     <property>\n" +
"       <name>key</name>\n" +
"       <value>F9B6C974354E316D391E061962284986</value>\n" +  //for this request
"     </property>\n" +
"   </property_group>\n" +
" </properties>\n" +
" <dataObjects>\n" +
//        url is from service name from .../JavaSource/resources/tabledap/TabledapBackendConfig.xml
"   <data url=\"http://coastwatch.pfel.noaa.gov/erddap/tabledap/\" \n" +  //required (especially 'url')
  "var=\"sea_surface_temperature\" title=\"Sea Surface Temperature\" \n" +
  "xpath=\"/lasdata/datasets/pmelTao/variables/sea_surface_temperature\" \n" +  //???
  "dataset_name=\"pmelTao\" dataset_ID=\"pmelTao\" units=\"degree_C\" \n" + //???
  "name=\"Sea Surface Temperature\" ID=\"pmelTao\" \n" +  //???
  "points=\"xyzt\" intervals=\"xyzt\" grid_type=\"scattered\">\n" + //???
"     <region IDREF=\"region_0\" />\n" +
"     <properties>\n" +
"       <property_group type=\"tabledap_access\">\n" +
"         <property>\n" +                           
"           <name>id</name>\n" +
"           <value>pmelTao</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>longitude</name>\n" +      //lon,lat,alt,time indicate if the variables exist and should be requested. 
"           <value>longitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>lon_domain</name>\n" +            
"           <value>0:360</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>latitude</name>\n" +       
"           <value>latitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>altitude</name>\n" +            
"           <value>altitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>time</name>\n" +          
"           <value>time</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"ui\">\n" +
"         <property>\n" +
"           <name>default</name>\n" +
"           <value>file:ui.xml#tao_demo</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"product_server\">\n" +
"         <property>\n" +
"           <name>ui_timeout</name>\n" +
"           <value>120000</value>\n" +  //seconds    ignored here
"         </property>\n" +
"         <property>\n" +
"           <name>ps_timeout</name>\n" +
"           <value>120000</value>\n" +  //seconds    ignored here
"         </property>\n" +
"         <property>\n" +
"           <name>use_cache</name>\n" +
"           <value>false</value>\n" +  //ignored here
"         </property>\n" +
"       </property_group>\n" +
"     </properties>\n" +
"   </data>\n" +

//Test getting other variables via other <data> objects.
//!!!This BackendService only looks at 'var'. Everything else is assumed to be the same!
"   <data url=\"http://coastwatch.pfel.noaa.gov/erddap/tabledap/\" \n" +  
  "var=\"air_temperature\" title=\"Air Temperature\" \n" +   //var is required
  "xpath=\"/lasdata/datasets/pmelTao/variables/air_temperature\" \n" +  //???
  "dataset_name=\"pmelTao\" dataset_ID=\"pmelTao\" units=\"degree_C\" \n" + //???
  "name=\"Air Temperature\" ID=\"pmelTao\" \n" +  //???
  "points=\"xyzt\" intervals=\"xyzt\" grid_type=\"scattered\">\n" + //???
"     <region IDREF=\"region_0\" />\n" +  //required to refer to same region
"   </data>\n" +

  " </dataObjects>\n" +
" <response ID=\"DBExtractResponse\">\n" +
"   <result type=\"debug\" ID=\"db_debug\" \n" +   //used if provided, but not required
  "file=\"" + testResultsDirectory + "F9B6C974354E316D391E061962284986_db_debug.txt\" \n" +
  "index=\"0\" />\n" +
"   <result type=\"netCDF\" ID=\"netcdf\" \n" +    //results file; required 
  "file=\"" + testResultsDirectory + testResultsNcName + "\" \n" +
  "index=\"1\" />\n" +
" </response>\n" +
"</backend_request>\n";


    /** 
     * A test for 2008-01-09.
     *
     * <p>This sample url should be generated by this xml.
http://coastwatch.pfel.noaa.gov/erddap/tabledap/cscWT?longitude,latitude,altitude,time,sea_water_temperature
&latitude>=24.0&latitude<=39.0&time>=2007-
01-01T00:00:00&time<=2008-01-09T23:00:00&longitude>=-98.0&longitude<=-70.0 
    */
    public static String testRequestXml20080109 = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<backend_request>\n" +
" <region ID=\"region_0\">\n" +              //region_0 is required
"   <x_lo>-98</x_lo>\n" +                  //all lo and hi tags are optional
"   <x_hi>-70</x_hi>\n" +                  //tao station lon lat change every week, so don't ask for exact value
"   <y_lo>24</y_lo>\n" +
"   <y_hi>39</y_hi>\n" +
"   <t_lo>01-Jan-2007 00:00:00</t_lo>\n" +
"   <t_hi>09-Jan-2008 23:00:00</t_hi>\n" +
" </region>\n" +
" <properties>\n" +
"   <property_group type=\"las\">\n" +
"     <property>\n" +
"       <name>debug</name>\n" +
"       <value>debug</value>\n" +            //required, a log4j logging level  
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"tabledap\">\n" +
"     <property>\n" +
"       <name>service_action</name>\n" +
"       <value>tabledap</value>\n" +
"     </property>\n" +
"   </property_group>\n" +
"   <property_group type=\"operation\">\n" +
"     <property>\n" +
"       <name>service</name>\n" +
"       <value>ERD Server</value>\n" +  //required, see .../JavaSource/resources/tabledap/TabledapBackendConfig.xml
"     </property>\n" +                     
"     <property>\n" +
"       <name>service_action</name>\n" +  //required
"       <value>tabledap</value>\n" +        //required
"     </property>\n" +
"     <property>\n" +  //what is this?
"       <name>name</name>\n" +
"       <value>Database Extraction</value>\n" +
"     </property>\n" +
"     <property>\n" +  //what is this?
"       <name>ID</name>\n" +
"       <value>TabledapExtract</value>\n" +
"     </property>\n" +
"     <property>\n" +
"       <name>key</name>\n" +
"       <value>F9B6C974354E316D391E061962284986</value>\n" +  //for this request
"     </property>\n" +
"   </property_group>\n" +
" </properties>\n" +
" <dataObjects>\n" +
//        url is from service name from .../JavaSource/resources/tabledap/TabledapBackendConfig.xml
"   <data url=\"http://coastwatch.pfel.noaa.gov/erddap/tabledap/\" \n" +  //required (especially 'url')
  "var=\"sea_water_temperature\" title=\"Sea Water Temperature\" \n" +
  "xpath=\"/lasdata/datasets/cscWT/variables/sea_water_temperature\" \n" +  //???
  "dataset_name=\"cscWT\" dataset_ID=\"cscWT\" units=\"degree_C\" \n" + //???
  "name=\"Sea Water Temperature\" ID=\"cscWT\" \n" +  //???
  "points=\"xyzt\" intervals=\"xyzt\" grid_type=\"scattered\">\n" + //???
"     <region IDREF=\"region_0\" />\n" +
"     <properties>\n" +
"       <property_group type=\"tabledap_access\">\n" +
"         <property>\n" +                           
"           <name>id</name>\n" +
"           <value>cscWT</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>longitude</name>\n" +      //lon,lat,alt,time indicate if the variables exist and should be requested. 
"           <value>longitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>lon_domain</name>\n" +            
"           <value>-180:180</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>latitude</name>\n" +       
"           <value>latitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>altitude</name>\n" +            
"           <value>altitude</value>\n" +
"         </property>\n" +
"         <property>\n" +
"           <name>time</name>\n" +          
"           <value>time</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"ui\">\n" +
"         <property>\n" +
"           <name>default</name>\n" +
"           <value>file:ui.xml#tao_demo</value>\n" +
"         </property>\n" +
"       </property_group>\n" +
"       <property_group type=\"product_server\">\n" +
"         <property>\n" +
"           <name>ui_timeout</name>\n" +
"           <value>120000</value>\n" +  //seconds    ignored here
"         </property>\n" +
"         <property>\n" +
"           <name>ps_timeout</name>\n" +
"           <value>120000</value>\n" +  //seconds    ignored here
"         </property>\n" +
"         <property>\n" +
"           <name>use_cache</name>\n" +
"           <value>false</value>\n" +  //ignored here
"         </property>\n" +
"       </property_group>\n" +
"     </properties>\n" +
"   </data>\n" +

  " </dataObjects>\n" +
" <response ID=\"DBExtractResponse\">\n" +
"   <result type=\"debug\" ID=\"db_debug\" \n" +   //used if provided, but not required
  "file=\"" + testResultsDirectory + "F9B6C974354E316D391E061962284986_db_debug.txt\" \n" +
  "index=\"0\" />\n" +
"   <result type=\"netCDF\" ID=\"netcdf\" \n" +    //results file; required 
  "file=\"" + testResultsDirectory + testResultsNcName + "\" \n" +
  "index=\"1\" />\n" +
" </response>\n" +
"</backend_request>\n";

    /**
     * This runs TabledapBackendService with the xml file you specify, or with
     * a standard test xml file.
     * This method prints getProduct's result to System.out.
     * This puts the results in a file called testResultsDirectory + testResultsName.
     *
     * @param args If present, args[0] should be the name of a LASBackendRequest.xml file.
     *   If args==null or args.length==0, this runs a standard test.
     * @throws Exception if there is an error
     */
    public static void main(String args[]) throws Exception {

//once in a while: generate dataset descriptions in (svn)/conf/example/tabledap_erd.xml   
//GenerateTabledapXml.test();
//if (true) System.exit(0);

//one time thing:
//args = new String[]{"c:/programs/las/roland_tabledap_request.xml"};

        //get the requestXml
        String2.log("\n*** TabledapBackendService");
        long time = System.currentTimeMillis();
        String requestXml = testRequestXml;
        //String requestXml = testRequestXml20080109; //just to look at error it throws
        boolean doStandardTest = true;
        if (args != null && args.length > 0) {
            doStandardTest = false;
            String results[] = String2.readFromFile(args[0]);
            if (results[0].equals(""))
                requestXml = results[1];
            else throw new Exception(results[0]);
        }
        if (doStandardTest) 
            File2.delete(testResultsDirectory + testResultsNcName);

        //run TabledapBackendService
        TabledapBackendService service = new TabledapBackendService();
        String resultXml = service.getProduct(requestXml, null);
        System.out.println("resultXml=\n" + resultXml); 

        //check the results of the standard test
        if (doStandardTest) {
            //read the .nc file
            //tao lon and lat values change every week, so don't check for exact values
            double time1 = Calendar2.isoStringToEpochSeconds("2007-01-01T12:00:00");
            double time2 = Calendar2.isoStringToEpochSeconds("2007-01-02T12:00:00");
            Table table = new Table();
            table.readFlatNc(testResultsDirectory + testResultsNcName, null, 1);
            String2.log(table.toString());
            Test.ensureEqual(table.nColumns(), 6, "");
            Test.ensureEqual(table.nRows(), 31, "");
            Test.ensureEqual(table.getColumnName(0), "longitude", "");  
            Test.ensureEqual(table.getColumnName(1), "latitude", ""); 
            Test.ensureEqual(table.getColumnName(2), "altitude", "");
            Test.ensureEqual(table.getColumnName(3), "time", "");
            Test.ensureEqual(table.getColumnName(4), "sea_surface_temperature", "");
            Test.ensureEqual(table.getColumnName(5), "air_temperature", "");
            Test.ensureEqual(table.getColumn(0).getElementTypeString(), "float", "");
            Test.ensureEqual(table.getColumn(3).getElementTypeString(), "double", "");
            Test.ensureEqual(table.getColumn(4).getElementTypeString(), "float", "");
            Test.ensureEqual(table.getColumn(5).getElementTypeString(), "float", "");
            Test.ensureEqual(table.globalAttributes().getString("Conventions"), //added in TabledapTool
                "LAS Intermediate netCDF File, Unidata Observation Dataset v1.0", "");  //cf? coards? epic in-situ?
            Test.ensureEqual(table.globalAttributes().get("cdm_data_type").toString(), "Station", ""); //from erddap
            Test.ensureEqual(table.globalAttributes().getString("title"), //from erddap
                "TAO Array Data from the PMEL DAPPER Server", "");
            //lat attributes...
            Test.ensureEqual(table.columnAttributes(1).getString("units"), "degrees_north", ""); 
            Test.ensureEqual(table.columnAttributes(1).getString("long_name"), "Latitude", "");
            Test.ensureEqual(table.columnAttributes(1).getString("missing_value"), null, "");
            Test.ensureEqual(table.columnAttributes(1).getString("axis"), "Y", "");
            //alt attributes...
            Test.ensureEqual(table.columnAttributes(2).getString("_CoordinateAxisType"), "Height", "");
            Test.ensureEqual(table.columnAttributes(2).getString("axis"), "Z", "");
            Test.ensureEqual(table.columnAttributes(2).getString("long_name"), "Altitude", "");
            Test.ensureEqual(table.columnAttributes(2).getString("positive"), "up", "");
            Test.ensureEqual(table.columnAttributes(2).getString("units"), "m", ""); 

            Test.ensureEqual(table.columnAttributes(3).getString("_CoordinateAxisType"), "Time", "");
            Test.ensureEqual(table.columnAttributes(3).getString("axis"), "T", "");
            Test.ensureEqual(table.columnAttributes(3).getString("long_name"), "Time", "");
            Test.ensureEqual(table.columnAttributes(3).getString("time_origin"), "01-JAN-1970 00:00:00", ""); 
            Test.ensureEqual(table.columnAttributes(3).getString("units"), "seconds since 1970-01-01T00:00:00Z", ""); 

            Test.ensureEqual(table.getDoubleData(3, 0), time1, "");
            Test.ensureEqual(table.getDoubleData(3, 1), time2, "");

            //col4 sst
            Test.ensureEqual(table.columnAttributes(4).getString("units"), "degree_C", "");  
            Test.ensureEqual(table.columnAttributes(4).getString("long_name"), "Sea Surface Temperature", ""); 
            Test.ensureEqual(table.getFloatData(4, 0), 27.56f, ""); 
            Test.ensureEqual(table.getFloatData(4, 1), 27.54f, "");
            //col5 air temp
            Test.ensureEqual(table.columnAttributes(5).getString("units"), "degree_C", "");
            Test.ensureEqual(table.columnAttributes(5).getString("long_name"), "Air Temperature", ""); 
            Test.ensureEqual(table.getFloatData(5, 0), 25.33f, ""); 
            Test.ensureEqual(table.getFloatData(5, 1), 24.23f, "");

//Need test of request that returns no data. Is table as desired?
//Need test of xLo > xHi, so split lon request.
        }

        //it is possible to pass the standard test but get an error code back from the service
        Test.ensureTrue(resultXml.indexOf("type=\"error\"") < 0, 
            String2.ERROR + " in resultXml=\n" + resultXml);

        String2.log("  TabledapBackendService finished successfully. TIME=" +
            (System.currentTimeMillis() - time));
    }
}
