/* This file is part of the EMA project and is 
 * Copyright (c) 2005 Robert Alten Simons (info@cohort.com).
 * See the MIT/X-like license in LICENSE.txt.
 * For more information visit www.cohort.com or contact info@cohort.com.
 */
package com.cohort.array;

import com.cohort.util.*;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.List;

/**
 * PrimitiveArray defines the methods to be implemented by various XxxArray classes
 * for 1D arrays of primitives with methods like ArrayLists's methods.
 *
 * <p> Primitive Arrays for integer types support the idea of a NaN or missing value,
 * by designating their MAX_VALUE as the missing value. This is consistent with JGOFS
 * ("the server will set the missing value field to the largest value 
 * possible for specified type.",
 * http://www.opendap.org/server/install-html/install_22.html). 
 * This has the convenient side effect that missing values sort high
 * (as to NaNs for floats and doubles).
 */  
public abstract class PrimitiveArray {

    /** The number of active values (which may be different from the array's capacity). */
    protected int size = 0;

    /**
     * The constants identify the items in the array returned by getStats.
     */
    public final static int STATS_N = 0, STATS_MIN = 1, STATS_MAX = 2, STATS_SUM = 3; 

    //these are carefully ordered from smallest to largest data type
    public final static int TYPE_INDEX_BYTE = 0;
    public final static int TYPE_INDEX_SHORT = 1;
    public final static int TYPE_INDEX_CHAR = 2;
    public final static int TYPE_INDEX_INT = 3;
    public final static int TYPE_INDEX_LONG = 4;
    public final static int TYPE_INDEX_FLOAT = 5;
    public final static int TYPE_INDEX_DOUBLE = 6;
    public final static int TYPE_INDEX_STRING = 7;     


    /** 
     * This returns a PrimitiveArray wrapped around a String[] or array of primitives.
     *
     * @param o a char[][], String[] or primitive[] (e.g., int[])
     * @return a PrimitiveArray which (at least initially) uses the array for data storage.
     */
    public static PrimitiveArray factory(Object o) {
        if (o instanceof char[][]) {
            char[][] car = (char[][])o;
            int nStrings = car.length;
            StringArray sa = new StringArray(nStrings, false);
            for (int i = 0; i < nStrings; i++) {
                String s = new String(car[i]);
                int po0 = s.indexOf('\u0000');
                if (po0 >= 0)
                    s = s.substring(0, po0);
                sa.add(s);
            }
            return sa;
        }
        if (o instanceof double[]) return new DoubleArray((double[])o);
        if (o instanceof float[])  return new FloatArray((float[])o);
        if (o instanceof long[])   return new LongArray((long[])o);
        if (o instanceof int[])    return new IntArray((int[])o);
        if (o instanceof short[])  return new ShortArray((short[])o);
        if (o instanceof byte[])   return new ByteArray((byte[])o);
        if (o instanceof char[])   return new CharArray((char[])o);
        if (o instanceof String[]) return new StringArray((String[])o);
        if (o instanceof Object[]) {
            Object oar[] = (Object[])o;
            int n = oar.length;
            StringArray sa = new StringArray(n, false);
            for (int i = 0; i < n; i++)
                sa.add(oar[i].toString());
            return sa;
        }

        throw new IllegalArgumentException(String2.ERROR + 
            " in PrimitiveArray.factory: unexpected object type: " + o.toString());
    }

    /**
     * This makes a new object which is a copy of this object.
     *
     * @return a new object, with the same elements, but with a 
     *    capacity equal to 'size'.
     */
    abstract public Object clone();

    /**
     * This returns a PrimitiveArray of a specified type and capacity.
     *
     * @param elementType e.g., float.class
     * @param capacity
     * @param active if true, size will be set to capacity, else size = 0.
     * @return a PrimitiveArray
     */
    public static PrimitiveArray factory(Class elementType, int capacity, boolean active) {
        if (elementType == double.class) return new DoubleArray(capacity, active);
        if (elementType == float.class)  return new FloatArray(capacity, active);
        if (elementType == long.class)   return new LongArray(capacity, active);
        if (elementType == int.class)    return new IntArray(capacity, active);
        if (elementType == short.class)  return new ShortArray(capacity, active);
        if (elementType == byte.class)   return new ByteArray(capacity, active);
        if (elementType == char.class)   return new CharArray(capacity, active);
        if (elementType == String.class) return new StringArray(capacity, active);

        throw new IllegalArgumentException(String2.ERROR + 
            " in PrimitiveArray.factory: unexpected elementType: " + elementType);
    }

    /** 
     * This returns a PrimitiveArray with size constantValues.
     *
     * @param elementType e.g., float.class
     * @param size the desired number of elements
     * @param constantValue the value for all of the elements (e.g., "1.6").
     *    For numeric elementTypes, constantValue is parsed.
     *    For char, the first character is used (e.g., "ab" -&gt; 'a' -&gt; 97).
     * @return a PrimitiveArray with size constantValues.
     */
    public static PrimitiveArray factory(Class elementType, int size, String constantValue) {
        Test.ensureNotNull(constantValue, 
            String2.ERROR + " in PrimitiveArray.factory: constantValue is null.");

        if (elementType == double.class) {
            double ar[] = new double[size];
            Arrays.fill(ar, String2.parseDouble(constantValue)); //does to/from String bruise it?
            return new DoubleArray(ar);
        }
        if (elementType == float.class) {
            float ar[] = new float[size];
            Arrays.fill(ar, String2.parseFloat(constantValue)); //does to/from String bruise it?
            return new FloatArray(ar);
        }
        if (elementType == long.class) {
            long ar[] = new long[size];
            Arrays.fill(ar, String2.parseLong(constantValue)); 
            return new LongArray(ar);
        }  
        if (elementType == int.class) {
            int ar[] = new int[size];
            Arrays.fill(ar, String2.parseInt(constantValue)); 
            return new IntArray(ar);
        }
        if (elementType == short.class) {
            short ar[] = new short[size];
            Arrays.fill(ar, Math2.roundToShort(String2.parseInt(constantValue)));
            return new ShortArray(ar);
        }
        if (elementType == byte.class) {
            byte ar[] = new byte[size];
            Arrays.fill(ar, Math2.roundToByte(String2.parseInt(constantValue))); 
            return new ByteArray(ar);
        }
        if (elementType == char.class) {
            char ar[] = new char[size];
            Arrays.fill(ar, constantValue.charAt(0)); 
            return new CharArray(ar);
        }
        if (elementType == String.class) {
            String ar[] = new String[size];
            Arrays.fill(ar, constantValue); //does to/from String bruise it?
            return new StringArray(ar);
        }
        throw new IllegalArgumentException(String2.ERROR + 
            " in PrimitiveArray.factory: unexpected elementType: " + elementType);
    }

    /**
     * This returns a PrimitiveArray of the specified type from the comma-separated values.
     *
     * @param elementType e.g., float.class or String.class
     * @param csv For elementType=String.class, individual values with interior commas
     *    must be completely enclosed in double quotes with interior double
     *    quotes converted to 2 double quotes. For String values without interior commas,
     *    you don't have to double quote the whole value.    
     * @return a PrimitiveArray
     */
    public static PrimitiveArray csvFactory(Class elementType, String csv) {
        if (elementType == String.class) 
            return StringArray.fromCSV(csv);

        String[] sa = String2.split(csv, ',');
        int n = sa.length;
        PrimitiveArray pa = factory(elementType, n, false);
        for (int i = 0; i < n; i++)
            pa.addString(sa[i]);
        return pa;
    }

    /**
     * This returns a PrimitiveArray of the specified type from the comma-separated values.
     *
     * @param elementType e.g., float.class or String.class
     * @param ssv For elementType=String.class, individual values with interior spaces or commas
     *    must be completely enclosed in double quotes with interior double
     *    quotes converted to 2 double quotes. For String values without interior spaces or commas,
     *    you don't have to double quote the whole value.    
     * @return a PrimitiveArray
     */
    public static PrimitiveArray ssvFactory(Class elementType, String ssv) {
        StringArray sa = StringArray.wordsAndQuotedPhrases(ssv);
        if (elementType == String.class) 
            return sa;
        int n = sa.size();
        PrimitiveArray pa = factory(elementType, n, false);
        for (int i = 0; i < n; i++)
            pa.addString(sa.get(i));
        return pa;
    }

    /**
     * Return the number of elements in the array.
     *
     * @return the number of elements in the array.
     */
    public int size() {
        return size;
    }

    /**
     * This sets size to 0.
     * XxxArrays with objects override this to set the no-longer-accessible 
     * elements to null.
     *
     */
    public void clear() {
        size = 0;
    }

    /**
     * This returns the class (e.g., float.class or String.class) 
     * of the element type.
     *
     * @return the class (e.g., float.class) of the element type.
     */
    abstract public Class getElementType();

    /**
     * This returns the string form (e.g., "float" or "String") 
     * of the element type.
     *
     * @return the string form (e.g., "float" or "String") 
     * of the element type.
     */
    public String getElementTypeString() {
        return elementTypeToString(getElementType());
    }

    /**
     * This converts an element type (e.g., float.class) to a String (e.g., float).
     *
     * @param type an element type (e.g., float.class)
     * @return the string representation of the element type (e.g., float)
     * @throws exception if not one of the PrimitiveArray types
     */
    public static String elementTypeToString(Class type) {
        if (type == double.class) return "double";
        if (type == float.class)  return "float";
        if (type == long.class)   return "long";
        if (type == int.class)    return "int";
        if (type == short.class)  return "short";
        if (type == byte.class)   return "byte";
        if (type == char.class)   return "char";
        if (type == String.class) return "String";
        throw new IllegalArgumentException(
            "PrimitiveArray.elementTypeToString unsupported type: "  + type.toString());
    }

    /**
     * This converts an element type String (e.g., "float") to an element type (e.g., float.class).
     *
     * @param type an element type string (e.g., "float")
     * @return the corresponding element type (e.g., float.class)
     * @throws exception if not one of the PrimitiveArray types
     */
    public static Class elementStringToType(String type) {
        if (type.equals("double")) return double.class;
        if (type.equals("float"))  return float.class;
        if (type.equals("long"))   return long.class;
        if (type.equals("int"))    return int.class;
        if (type.equals("short"))  return short.class;
        if (type.equals("byte"))   return byte.class;
        if (type.equals("char"))   return char.class;
        if (type.equals("String")) return String.class;
        throw new IllegalArgumentException("PrimitiveArray.elementStringToType unsupported type: " + type);
    }

    /**
     * This indicates the number of bytes per element of the given type.
     * The value for "String" isn't a constant, so this returns 20.
     *
     * @param type an element type string (e.g., "float")
     * @return the corresponding number of bytes
     * @throws exception if not one of the PrimitiveArray types
     */
    public static int elementSize(String type) {
        if (type.equals("double")) return 8;
        if (type.equals("float"))  return 4;
        if (type.equals("long"))   return 8;
        if (type.equals("int"))    return 4;
        if (type.equals("short"))  return 2;
        if (type.equals("byte"))   return 1;
        if (type.equals("char"))   return 2;
        if (type.equals("String")) return 20;
        throw new IllegalArgumentException("PrimitiveArray.sizeOf unsupported type: " + type);
    }

    /**
     * This indicates the number of bytes per element of the given type.
     * The value for String.class isn't a constant, so this returns 20.
     *
     * @param type an element type (e.g., float.class)
     * @return the corresponding number of bytes
     * @throws exception if not one of the PrimitiveArray types
     */
    public static int elementSize(Class type) {
        return elementSize(elementTypeToString(type));
    }

    /**
     * This returns the number of bytes per element for this PrimitiveArray.
     * The value for "String" isn't a constant, so this returns 20.
     *
     * @return the number of bytes per element for this PrimitiveArray.
     * The value for "String" isn't a constant, so this returns 20.
     */
    public int elementSize() {
        return elementSize(getElementTypeString());
    }


    /** 
     * This returns the recommended sql data type for this PrimitiveArray.
     * See 
     * <ul>
     * <li> For Strings this is difficult, because it is hard to know what max length
     *   might be in the future.  See stringLengthFactor below.
     *   (PostgresQL supports varchar without a length, or TEXT,
     *   but it isn't a SQL standard.)
     * <li> This doesn't deal with sql DATE or TIMESTAMP, since I often store 
     *   seconds since epoch in a DoubleArray.
     * <li> Not all SQL types are universally supported (e.g., BIGINT for LongArray).
     *   See http://www.techonthenet.com/sql/datatypes.php .
     * <li> For safety, ByteArray returns SMALLINT (not TINYINT,
     *   which isn't universally supported, e.g., postgresql).
     * <ul>
     *
     * @param stringLengthFactor for StringArrays, this is the factor (typically 1.5)  
     *   to be multiplied by the current max string length (then rounded up to 
     *   a multiple of 10, but only if stringLengthFactor &gt; 1) 
     *   to estimate the varchar length.  
     * @return the recommended sql type as a string e.g., varchar(40)
     */
    public String getSqlTypeString(double stringLengthFactor) {
        Class type = getElementType();
        if (type == double.class) return "double precision";
        if (type == float.class)  return "real";  //postgres treats "float" as double precision
        if (type == long.class)   return "bigint"; //not universally supported (pgsql does support it)
        if (type == int.class)    return "integer";
        if (type == short.class)  return "smallint";
        if (type == byte.class)   return "smallint"; //not TINYINT, not universally supported (even pgsql)
        if (type == char.class)   return "char(1)";
        if (type == String.class) {
            StringArray sa = (StringArray)this;
            int max = Math.max(1, sa.maxStringLength());
            if (stringLengthFactor > 1) {
                max = Math2.roundToInt(max * stringLengthFactor);
                max = Math2.hiDiv(max, 10) * 10;            
            }
            //postgresql doesn't use longvarchar and allows varchar's max to be very large
            return "varchar(" + max + ")";  
        }
        throw new IllegalArgumentException(String2.ERROR + 
            " in PrimitiveArray.getSqlTypeString: unexpected type: " + type.toString());
    }

    /** 
     * This returns the suggested elementType for the given java.sql.Types.
     * This conversion is not standardized across databases (see 
     * http://www.onlamp.com/pub/a/onlamp/2001/09/13/aboutSQL.html?page=last).
     * But choices below are fairly safe.
     * I can't find a table to link java.sql.Types constants to Postgres types.
     * See postgresql types at
     * http://www.postgresql.org/docs/8.2/static/datatype-numeric.html
     *
     * @param sqlType  a java.sql.Types constant
     * @return a PrimitiveArray of the suggested type.
     *   Basically, numeric types return numeric PrimitiveArrays;
     *     other other types return StringArray.
     *   Bit and Boolean return ByteArray.
     *   Date, Time and Timestamp return a StringArray.
     *   If the type is unexpected, this returns StringArray.
     */
    public static PrimitiveArray sqlFactory(int sqlType) {

        //see recommended types in table at
        //  http://java.sun.com/j2se/1.5.0/docs/guide/jdbc/getstart/resultset.html
        //see JDBC API Tutorial book, pg 1087
        if (sqlType == Types.BIT ||      //PrimitiveArray doesn't have a separate BooleanArray
            sqlType == Types.BOOLEAN ||  //PrimitiveArray doesn't have a separate BooleanArray
            sqlType == Types.TINYINT)   return new ByteArray();
        if (sqlType == Types.SMALLINT)  return new ShortArray();
        if (sqlType == Types.INTEGER)   return new IntArray();
        if (sqlType == Types.BIGINT)    return new LongArray();
        if (sqlType == Types.REAL)      return new FloatArray();
        if (sqlType == Types.FLOAT ||   //a 64 bit value(!)
            sqlType == Types.DOUBLE ||
            sqlType == Types.DECIMAL ||  //DECIMAL == NUMERIC; infinite precision!!!
            sqlType == Types.NUMERIC)   return new DoubleArray();
        //if (sqlType == Types.DATE ||
        //    sqlType == Types.TIMESTAMP ||
        //    sqlType == Types.CHAR ||
        //    sqlType == Types.LONGVARCHAR ||
        //    sqlType == Types.VARCHAR ||
        //    sqlType == Types.CLOB ||
        //    sqlType == Types.DATALINK ||
        //    sqlType == Types.REF ||
        //    sqlType == Types.TIME ||     //or convert to time in 0001-01-01?
        //       true //What the heck!!!!! just get everything else as String, too.
             return new StringArray(); 
    }

    /**
     * This indicates if a given type (e.g., float.class) can be contained in a long.
     *
     * @param type an element type (e.g., float.class)
     * @return true if the given type (e.g., float.class) can be contained in a long.
     * @throws exception if not one of the PrimitiveArray types
     */
    public static boolean isIntegerType(Class type) {
        return 
            type == long.class ||
            type == int.class ||
            type == short.class ||
            type == byte.class ||
            type == char.class;
    }

    /**
     * This returns for missing value for a given element type (e.g., byte.class).
     *
     * @param type an element type (e.g., byte.class)
     * @return the string representation of the element type (e.g., Byte.MAX_VALUE).
     *   Note that the mv for float is Float.NaN, but it gets converted
     *   to Double.NaN when returned by this method.
     *   StringArray supports several incoming missing values, but
     *   "" is used as the outgoing missing value.
     */
    public static double getMissingValue(Class type) {
        if (type == double.class) return Double.NaN;
        if (type == float.class)  return Double.NaN;
        if (type == long.class)   return Long.MAX_VALUE;
        if (type == int.class)    return Integer.MAX_VALUE;
        if (type == short.class)  return Short.MAX_VALUE;
        if (type == byte.class)   return Byte.MAX_VALUE;
        if (type == char.class)   return Character.MAX_VALUE;
        if (type == String.class) return Double.NaN;
        return Double.NaN;
    }

    /**
     * This returns the type index (e.g., TYPE_INDEX_INT) of the element type.
     *
     * @return the type index (e.g., TYPE_INDEX_INT) of the element type.
     */
    abstract public int getElementTypeIndex();

    /**
     * This adds an element to the array at the specified index.
     *
     * @param index 0..
     * @param value the value, as a String.
     */
    abstract public void addString(int index, String value);

    /**
     * This adds an element to the array.
     *
     * @param value the value, as a String.
     */
    abstract public void addString(String value);

    /**
     * This adds n Strings to the array.
     *
     * @param n the number of times 'value' should be added
     * @param value the value, as a String.
     */
    abstract public void addNStrings(int n, String value);

    /**
     * This adds an element to the array.
     *
     * @param value the value, as a float.
     */
    abstract public void addFloat(float value);

    /**
     * This adds an element to the array.
     *
     * @param value the value, as a Double.
     */
    abstract public void addDouble(double value);

    /**
     * This adds n doubles to the array.
     *
     * @param n the number of times 'value' should be added
     * @param value the value, as a double.
     */
    abstract public void addNDoubles(int n, double value);

    /**
     * This adds an element to the array.
     *
     * @param value the value, as an int.
     */
    abstract public void addInt(int value);

    /**
     * This removes the specified element.
     *
     * @param index the element to be removed, 0 ... size-1
     * @throws Exception if trouble.
     */
    abstract public void remove(int index);

    /**
     * This removes the specified range of elements.
     *
     * @param from the first element to be removed, 0 ... size
     * @param to one after the last element to be removed, from ... size
     * @throws Exception if trouble.
     */
    abstract public void removeRange(int from, int to);

    /**
     * Moves elements 'first' through 'last' (inclusive)
     *   to 'destination'.
     *
     * @param first  the first to be move
     * @param last  (exclusive)
     * @param destination the destination, can't be in the range 'first+1..last-1'.
     * @throws Exception if trouble
     */
    abstract public void move(int first, int last, int destination);

    /**
     * This just keeps the rows for the 'true' values in the bitset.
     * Rows that aren't kept are removed.
     * The resulting PrimitiveArray is compacted (i.e., it has a smaller size()).
     *
     * @param bitset
     */
    abstract public void justKeep(BitSet bitset);

    /**
     * This ensures that the capacity is at least 'minCapacity'.
     *
     * @param minCapacity the minimum acceptable capacity
     */
    abstract public void ensureCapacity(int minCapacity);

    /**
     * This returns a primitive[] (perhaps 'array') which has 'size' 
     * elements.
     *
     * @return a primitive[] (perhaps 'array') which has 'size' elements.
     */
    abstract public Object toObjectArray();

    /**
     * This returns a double[] which has 'size' elements.
     *
     * @return a double[] which has 'size' elements.
     */
    abstract public double[] toDoubleArray();

    /**
     * This returns a String[] which has 'size' elements.
     *
     * @return a String[] which has 'size' elements.
     */
    abstract public String[] toStringArray();

    /**
     * Return a value from the array as an int.
     * Floating point values are rounded.
     * 
     * @param index the index number 0 ... size-1
     * @return the value as an int. String values are parsed
     *   with String2.parseInt and so may return Integer.MAX_VALUE.
     * @throws Exception if trouble.
     */
    abstract public int getInt(int index);

    /**
     * Set a value in the array as an int.
     * 
     * @param index the index number 0 .. size-1
     * @param i the value. For numeric PrimitiveArray's, it is narrowed 
     *   if needed by methods like Math2.narrowToByte(d).
     * @throws Exception if trouble.
     */
    abstract public void setInt(int index, int i);

    /**
     * Return a value from the array as a float.
     * 
     * @param index the index number 0 ... size-1
     * @return the value as a float. String values are parsed
     *   with String2.parseFloat and so may return Float.NaN.
     * @throws Exception if trouble.
     */
    abstract public float getFloat(int index);

    /**
     * Set a value in the array as a float.
     * 
     * @param index the index number 0 ... size-1
     * @param d the value. For numeric PrimitiveArray's, it is narrowed 
     *   if needed by methods like Math2.roundToInt(d).
     * @throws Exception if trouble.
     */
    abstract public void setFloat(int index, float d);

    /**
     * Return a value from the array as a double.
     * FloatArray converts float to double in a simplistic way.
     * 
     * @param index the index number 0 ... size-1
     * @return the value as a double. String values are parsed
     *   with String2.parseDouble and so may return Double.NaN.
     * @throws Exception if trouble.
     */
    abstract public double getDouble(int index);

    /**
     * Return a value from the array as a double.
     * FloatArray converts float to double via Math2.floatToDouble.
     * 
     * @param index the index number 0 ... size-1
     * @return the value as a double. String values are parsed
     *   with String2.parseDouble and so may return Double.NaN.
     * @throws Exception if trouble.
     */
    public double getNiceDouble(int index) {
        return getDouble(index);
    }

    /**
     * Set a value in the array as a double.
     * 
     * @param index the index number 0 ... size-1
     * @param d the value. For numeric PrimitiveArray's, it is narrowed 
     *   if needed by methods like Math2.roundToInt(d).
     * @throws Exception if trouble.
     */
    abstract public void setDouble(int index, double d);

    /**
     * Return a value from the array as a String.
     * 
     * @param index the index number 0 ... size-1 
     * @return For numeric types, this returns ("" + ar[index]), or "" if NaN or infinity.
     * @throws Exception if trouble.
     */
    abstract public String getString(int index);

    /**
     * Set a value in the array as a String.
     * 
     * @param index the index number 0 ... size-1 
     * @param s the value. For numeric PrimitiveArray's, it is parsed
     *   with String2.parse and narrowed if needed by methods like
     *   Math2.roundToInt(d).
     * @throws Exception if trouble.
     */
    abstract public void setString(int index, String s);


    /**
     * This finds the first instance of 'lookFor' starting at index 'startIndex'.
     *
     * @param lookFor the value to be looked for
     * @param startIndex 0 ... size-1
     * @return the index where 'lookFor' is found, or -1 if not found.
     */
    abstract public int indexOf(String lookFor, int startIndex);

    /**
     * This finds the first instance of 'lookFor' starting at index 0.
     *
     * @param lookFor the value to be looked for
     * @return the index where 'lookFor' is found, or -1 if not found.
     */
    public int indexOf(String lookFor) {return indexOf(lookFor, 0); }

    /**
     * This finds the last instance of 'lookFor' starting at index 'startIndex'.
     *
     * @param lookFor the value to be looked for
     * @param startIndex 0 ... size-1
     * @return the index where 'lookFor' is found, or -1 if not found.
     */
    abstract public int lastIndexOf(String lookFor, int startIndex);

    /**
     * This finds the last instance of 'lookFor' starting at index size() - 1.
     *
     * @param lookFor the value to be looked for
     * @return the index where 'lookFor' is found, or -1 if not found.
     */
    public int lastIndexOf(String lookFor) {return lastIndexOf(lookFor, size() - 1); }

    /**
     * If size != capacity, this makes a new 'array' of size 'size'
     * so capacity will equal size.
     */
    abstract public void trimToSize();

    /** 
     * This converts the elements into a comma-separated String.
     *
     * @return the comma-separated String representation of o
     */
    abstract public String toString();

    /**
     * This returns a JSON-style comma-separate-value list of the elements.
     * StringArray overwrites this so Strings are encoded via String2.toJson.
     *
     * @return a csv string of the elements.
     */
    public String toJsonCsvString() {
        return toString();
    }

    /** 
     * This sorts the elements in ascending order.
     * To get the elements in reverse order, just read from the end of the list
     * to the beginning.
     */
    abstract public void sort();

    /**
     * This calculates min, max, and nValid for the values in this 
     * PrimitiveArray.
     * Each data type has its own missing value (e.g., Byte.MAX_VALUE).
     *
     * @return a double[] with 
     *    dar[STATS_N] containing the number of valid values.
     *    dar[STATS_MIN] containing the minimum value, and
     *    dar[STATS_MAX] containing the maximum value.
     *    dar[STATS_SUM] containing the sum of the values.
     *    If n is 0, min and max will be Double.NaN, and sum will be 0.
     */
    public double[] calculateStats() {
        long time = System.currentTimeMillis();

        int n = 0;
        double min = Double.MAX_VALUE;
        double max = -Double.MAX_VALUE; //not Double.MIN_VALUE
        double sum = 0;

        for (int i = 0; i < size; i++) {
            double d = getDouble(i);
            if (Math2.isFinite(d)) { 
                n++;
                min = Math.min(min, d);
                max = Math.max(max, d);
                sum += d;
            }
        }

        if (n == 0) {
            min = Double.NaN;
            max = Double.NaN;
        }
        return new double[]{n, min, max, sum};
    }


    /**
     * This compares the values in row1 and row2 for SortComparator,
     * and returns a negative integer, zero, or a positive integer if the 
     * value at index1 is less than, equal to, or greater than 
     * the value at index2.  Think (ar[index1] - ar[index2]).
     *
     * @param index1 an index number 0 ... size-1
     * @param index2 an index number 0 ... size-1
     * @return  a negative integer, zero, or a positive integer if the 
     * value at index1 is less than, equal to, or greater than 
     * the value at index2.
     */
    abstract public int compare(int index1, int index2);

    /**
     * This copies the value in row 'from' to row 'to'.
     * This does not check that 'from' and 'to' are valid;
     * the caller should be careful.
     *
     * @param from an index number 0 ... size-1
     * @param to an index number 0 ... size-1
     */
    abstract public void copy(int from, int to);

    /**
     * This reorders the values in 'array' based on rank.
     *
     * @param rank is an Integer[] with values (0 ... size-1) 
     * which points to the row number for a row with a specific 
     * rank (e.g., rank[0] is the row number of the first item 
     * in the sorted list, rank[1] is the row number of the
     * second item in the sorted list, ...).
     */
    abstract public void reorder(int rank[]);

    /**
     * This writes 'size' elements to a DataOutputStream.
     *
     * @param dos the DataOutputStream
     * @return the number of bytes used per element (for Strings, this is
     *    the size of one of the strings, not others, and so is useless;
     *    for other types the value is consistent).
     *    But if size=0, this returns 0.
     * @throws Exception if trouble
     */
    public int writeDos(DataOutputStream dos) throws Exception {
        //ByteArray overwrites this
        int nb = 0;
        for (int i = 0; i < size; i++)
            nb = writeDos(dos, i);
        return nb;
    }

    /**
     * This writes one element to a DataOutputStream.
     *
     * @param dos the DataOutputStream
     * @param i the index of the element to be written
     * @return the number of bytes used for this element
     *    (for Strings, this varies; for others it is consistent)
     * @throws Exception if trouble
     */
    abstract public int writeDos(DataOutputStream dos, int i) throws Exception;

    /**
     * This reads/adds n elements from a DataInputStream.
     *
     * @param dis the DataInputStream
     * @param n the number of elements to be read/added
     * @throws Exception if trouble
     */
    abstract public void readDis(DataInputStream dis, int n) throws Exception;

    /**
     * This writes all the data to a DataOutputStream in the
     * DODS Array format (see www.opendap.org DAP 2.0 standard, section 7.3.2.1).
     * See also the XDR standard (http://tools.ietf.org/html/rfc4506#section-4.11).
     * ByteArray, ShortArray, StringArray override this.
     * ???Does CharArray need to override this???
     *
     * @param dos
     */
    public void externalizeForDODS(DataOutputStream dos) throws Exception {
        dos.writeInt(size);
        dos.writeInt(size); //yes, a second time
        writeDos(dos);
    }

    /**
     * This writes one element to a DataOutputStream in the
     * DODS Atomic-type format (see www.opendap.org DAP 2.0 standard, section 7.3.2).
     * See also the XDR standard (http://tools.ietf.org/html/rfc4506#section-4.11).
     * ByteArray, ShortArray, StringArray override this.
     * ???Does CharArray need to override this???
     *
     * @param dos
     * @param i the index of the element to be written
     */
    public void externalizeForDODS(DataOutputStream dos, int i) throws Exception {
        writeDos(dos, i);
    }

    /**
     * This reads/appends same-type values to this PrimitiveArray from a DODS DataInputStream,
     * and is thus the complement of externalizeForDODS.
     *
     * @param dis
     * @throws IOException if trouble
     */
    public abstract void internalizeFromDODS(DataInputStream dis) throws java.io.IOException;

    /**
     * This reads one numeric value from a randomAccessFile.
     * This doesn't support StringArray (for which you need nBytesPer)
     * and in which you generally wouldn't be storing numeric values.
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array (nBytes)
     * @param index the index of the desired value (0..)
     * @return the requested value as a double
     * @throws Exception if trouble
     */
    public static double rafReadDouble(RandomAccessFile raf, Class type, 
        long start, long index) throws Exception {

        if (type == byte.class)   return ByteArray.rafReadDouble(  raf, start, index);
        if (type == char.class)   return CharArray.rafReadDouble(  raf, start, index);
        if (type == double.class) return DoubleArray.rafReadDouble(raf, start, index);
        if (type == float.class)  return FloatArray.rafReadDouble( raf, start, index);
        if (type == int.class)    return IntArray.rafReadDouble(   raf, start, index);
        if (type == long.class)   return LongArray.rafReadDouble(  raf, start, index);
        if (type == short.class)  return ShortArray.rafReadDouble( raf, start, index);
        //if (type == String.class) return ByteArray.rafReadDouble(raf, start, index, nBytesPer);
        throw new Exception("PrimitiveArray.rafReadDouble type '" + type + "' not supported.");
    }

    /**
     * This writes one numeric value to a randomAccessFile at the current position.
     * This doesn't support StringArray (for which you need nBytesPer)
     * and in which you generally wouldn't be storing numeric values.
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param value the value which will be converted to 'type' and then stored
     * @throws Exception if trouble
     */
    public static void rafWriteDouble(RandomAccessFile raf, Class type, 
        double value) throws Exception {

        if (type == byte.class)   {raf.writeByte(Math2.roundToByte(value));       return;}
        if (type == char.class)   {raf.writeChar(Math2.roundToChar(value));       return;}     
        if (type == double.class) {raf.writeDouble(value);                        return;}
        if (type == float.class)  {raf.writeFloat(Math2.doubleToFloatNaN(value)); return;}
        if (type == int.class)    {raf.writeInt(Math2.roundToInt(value));         return;}
        if (type == long.class)   {raf.writeLong(Math2.roundToLong(value));       return;}
        if (type == short.class)  {raf.writeShort(Math2.roundToShort(value));     return;}
        //if (type == String.class) {ByteArray.rafWriteDouble(raf, start, index, nBytesPer); return;}
        throw new Exception("PrimitiveArray.rafWriteDouble type '" + type + "' not supported.");
    }

    /**
     * This writes one numeric value to a randomAccessFile.
     * This doesn't support StringArray (for which you need nBytesPer)
     * and in which you generally wouldn't be storing numeric values.
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array (nBytes)
     * @param index the index of the value (0..)
     * @param value the value which will be converted to 'type' and then stored
     * @throws Exception if trouble
     */
    public static void rafWriteDouble(RandomAccessFile raf, Class type, 
        long start, long index, double value) throws Exception {

        if (type == byte.class)   {ByteArray.rafWriteDouble(  raf, start, index, value); return;}
        if (type == char.class)   {CharArray.rafWriteDouble(  raf, start, index, value); return;}
        if (type == double.class) {DoubleArray.rafWriteDouble(raf, start, index, value); return;}
        if (type == float.class)  {FloatArray.rafWriteDouble( raf, start, index, value); return;}
        if (type == int.class)    {IntArray.rafWriteDouble(   raf, start, index, value); return;}
        if (type == long.class)   {LongArray.rafWriteDouble(  raf, start, index, value); return;}
        if (type == short.class)  {ShortArray.rafWriteDouble( raf, start, index, value); return;}
        //if (type == String.class) {ByteArray.rafWriteDouble(raf, start, index, nBytesPer); return;}
        throw new Exception("PrimitiveArray.rafWriteDouble type '" + type + "' not supported.");
    }

    /**
     * This tests if the other object is of the same type and has equal values.
     *
     * @param other 
     * @return "" if equal, or message if not.
     *     other=null throws an exception.
     */
    abstract public String testEquals(Object other);

    /**
     * This tests if the other PrimitiveArray has almost equal values.
     * If both are integer types or String types, this is an exact test (and says null==null is true).
     * If either are double types, this tests almostEqual9() (and says NaN==NaN is true).
     * If either are float types, this tests almostEqual5() (and says NaN==NaN is true).
     *
     * @param other 
     * @return "" if almost equal, or message if not.
     *     other=null throws an exception.
     */
    public String almostEqual(PrimitiveArray other) {
        String msg = "The other primitiveArray has a different ";
        if (size != other.size())
            return msg + "size (" + size + " != " + other.size() + ")";
        
        if (this instanceof StringArray ||
            other instanceof StringArray) {
            for (int i = 0; i < size; i++) {
                String s1 = getString(i);
                String s2 = other.getString(i);
                if (s1 == null && s2 == null) {
                } else if (s1 != null && s2 != null && s1.equals(s2)) {
                } else {
                    return msg + "value #" + i + " (\"" + s1 + "\" != \"" + s2 + "\")";
                }
            }
            return "";
        }

        if (this instanceof DoubleArray || this instanceof LongArray ||
            other instanceof DoubleArray || other instanceof LongArray) {
            for (int i = 0; i < size; i++)
                if (!Test.equal(getDouble(i), other.getDouble(i)))  //this says NaN==NaN is true
                    return msg + "value #" + i + " (" + getDouble(i) + " != " + other.getDouble(i) + ")";
            return "";
        }

        if (this instanceof FloatArray ||
            other instanceof FloatArray) {
            for (int i = 0; i < size; i++)
                if (!Test.equal(getFloat(i), other.getFloat(i))) //this says NaN==NaN is true
                    return msg + "value #" + i + " (" + getFloat(i) + " != " + other.getFloat(i) + ")";
            return "";
        }

        for (int i = 0; i < size; i++)
            if (getInt(i) != other.getInt(i))
                return msg + "value #" + i + " (" + getInt(i) + " != " + other.getInt(i) + ")";
        return "";
    }
        

    /**
     * Given a sorted PrimitiveArray, stored to a randomAccessFile,
     * this finds the index of an instance of the value 
     * (not necessarily the first or last instance)
     * (or -index-1 where it should be inserted).
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of an instance of the value 
     *     (not necessarily the first or last instance)
     *     (or -index-1 where it should be inserted, with extremes of
     *     -lowPo-1 and -(highPo+1)-1).
     * @throws Exception if trouble
     */
    public static long rafBinarySearch(RandomAccessFile raf, Class type,
        long start, long lowPo, long highPo, double value) throws Exception {
        
        //ensure lowPo <= highPo
        //lowPo == highPo is handled by the following two chunks of code
        Test.ensureTrue(lowPo <= highPo,  
            String2.ERROR + "in PrimitiveArray.rafBinarySearch: lowPo > highPo.");
        
        double tValue = rafReadDouble(raf, type, start, lowPo);
        //String2.log("rafBinarySearch value=" + value + " po=" + lowPo + " tValue=" + tValue);
        if (tValue == value)
            return lowPo;
        if (tValue > value)
            return -lowPo - 1;

        tValue = rafReadDouble(raf, type, start, highPo);
        //String2.log("rafBinarySearch value=" + value + " po=" + highPo + " tValue=" + tValue);
        if (tValue == value)
            return highPo;
        if (tValue < value)
            return -(highPo+1) - 1;

        //repeatedly look at midpoint
        //If no match, this always ends with highPo - lowPo = 1
        //  and desired value would be in between them.
        while (highPo - lowPo > 1) {
            long midPo = (highPo + lowPo) / 2;
            tValue = rafReadDouble(raf, type, start, midPo);
            //String2.log("rafBinarySearch value=" + value + " po=" + midPo + " tValue=" + tValue);
            if (tValue == value)
                return midPo;
            if (tValue < value) 
                lowPo = midPo;
            else highPo = midPo;
        }

        //not found
        return -highPo - 1;
    }
        
    /**
     * Given a sorted PrimitiveArray, stored to a randomAccessFile,
     * this finds the index of the first element >= value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element @gt;= value 
     *     (or highPo + 1, if there are none)
     * @throws Exception if trouble
     */
    public static long rafFirstGE(RandomAccessFile raf, Class type,
        long start, long lowPo, long highPo, double value) throws Exception {

        if (lowPo > highPo)
            return highPo + 1;
        long po = rafBinarySearch(raf, type, start, lowPo, highPo, value);

        //an exact match? find the first exact match
        if (po >= 0) {
            while (po > lowPo && rafReadDouble(raf, type, start, po - 1) == value)
                po--;
            return po;
        }

        //no exact match? return the binary search po
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        return -po -1;
    }

    /**
     * Given a sorted PrimitiveArray, stored to a randomAccessFile,
     * this finds the index of the first element &gt; or almostEqual5 to value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element &gt; or Math2.almostEqual5 to value 
     *     (or highPo + 1, if there are none)
     * @throws Exception if trouble
     */
    public static long rafFirstGAE5(RandomAccessFile raf, Class type,
        long start, long lowPo, long highPo, double value) throws Exception {

        if (lowPo > highPo)
            return highPo + 1;

        long po = rafBinarySearch(raf, type, start, lowPo, highPo, value);

        //no exact match? return the binary search po
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        if (po < 0)
            po = -po -1;

        //find the first GAE
        while (po > lowPo && Math2.almostEqual(5, rafReadDouble(raf, type, start, po - 1), value))
            po--;

        return po;
    }

    /**
     * Given a sorted PrimitiveArray, stored to a randomAccessFile,
     * this finds the index of the last element &lt;= value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element &lt;= value 
     *     (or -1, if there are none)
     * @throws Exception if trouble
     */
    public static long rafLastLE(RandomAccessFile raf, Class type,
        long start, long lowPo, long highPo, double value) throws Exception {

        if (lowPo > highPo)
            return -1;
        long po = rafBinarySearch(raf, type, start, lowPo, highPo, value);

        //an exact match? find the first exact match
        if (po >= 0) {
            while (po < highPo && rafReadDouble(raf, type, start, po + 1) == value)
                po++;
            return po;
        }

        //no exact match? return binary search po -1
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        return -po -1 -1;
    }

    /**
     * Given a sorted PrimitiveArray, stored to a randomAccessFile,
     * this finds the index of the last element &lt; or almostEqual5 to value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param raf the RandomAccessFile
     * @param type the element type of the original PrimitiveArray
     * @param start the raf offset of the start of the array
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element &lt; or Math2.almostEqual5 to value 
     *     (or -1, if there are none)
     * @throws Exception if trouble
     */
    public static long rafLastLAE5(RandomAccessFile raf, Class type,
        long start, long lowPo, long highPo, double value) throws Exception {

        if (lowPo > highPo)
            return -1;
        long po = rafBinarySearch(raf, type, start, lowPo, highPo, value);

        //no exact match? return previous value (binary search po -1)
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        if (po < 0)
            po = -po -1 -1;

        //look for last almost equal value
        while (po < highPo && Math2.almostEqual(5, rafReadDouble(raf, type, start, po + 1), value))
            po++;

        return po;
    }

        
    /**
     * Given a sorted PrimitiveArray,
     * this finds the index of an instance of the value 
     * (not necessarily the first or last instance)
     * (or -index-1 where it should be inserted).
     *
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually (size - 1)
     * @param value the value you are searching for
     * @return the index of an instance of the value 
     *     (not necessarily the first or last instance)
     *     (or -index-1 where it should be inserted, with extremes of
     *     -lowPo-1 and -(highPo+1)-1).
     * @throws Exception if lowPo > highPo.
     */
    public int binarySearch(int lowPo, int highPo, double value) {
        
        //ensure lowPo <= highPo
        //lowPo == highPo is handled by the following two chunks of code
        Test.ensureTrue(lowPo <= highPo, 
            String2.ERROR + " in PrimitiveArray.binarySearch: lowPo > highPo.");
        
        double tValue = getDouble(lowPo);
        if (tValue == value)
            return lowPo;
        if (tValue > value)
            return -lowPo - 1;

        tValue = getDouble(highPo);
        if (tValue == value)
            return highPo;
        if (tValue < value)
            return -(highPo+1) - 1;

        //repeatedly look at midpoint
        //If no match, this always ends with highPo - lowPo = 1
        //  and desired value would be in between them.
        while (highPo - lowPo > 1) {
            int midPo = (highPo + lowPo) / 2;
            tValue = getDouble(midPo);
            if (tValue == value)
                return midPo;
            if (tValue < value) 
                lowPo = midPo;
            else highPo = midPo;
        }

        //not found
        return -highPo - 1;
    }
        
    /**
     * Given a sorted PrimitiveArray,
     * this finds the index of the first element &gt;= value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually (size - 1)
     * @param value the value you are searching for
     * @return the index of the first element &gt;= value 
     *     (or highPo + 1, if there are none)
     */
    public int binaryFindFirstGE(int lowPo, int highPo, double value) {

        if (lowPo > highPo)
            return highPo + 1;
         
        int po = binarySearch(lowPo, highPo, value);

        //an exact match? find the first exact match
        if (po >= 0) {
            while (po > lowPo && getDouble(po - 1) == value)
                po--;
            return po;
        }

        //no exact match? return the binary search po
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        return -po -1;
    }

    /**
     * Given a sorted PrimitiveArray,
     * this finds the index of the first element &gt; or almostEqual5 to value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually (size - 1)
     * @param value the value you are searching for
     * @return the index of the first element &gt; or Math2.almostEqual5 to value 
     *     (or highPo + 1, if there are none)
     */
    public int binaryFindFirstGAE5(int lowPo, int highPo, double value) {

        if (lowPo > highPo)
            return highPo + 1;
        int po = binarySearch(lowPo, highPo, value);

        //no exact match? start at high and work back
        //the inverse of -x-1 is -x-1 !
        if (po < 0) 
            po = -po -1;

        //find the first match
        while (po > lowPo && Math2.almostEqual(5, getDouble(po - 1), value))
            po--;

        return po;

    }

    /**
     * Given a sorted PrimitiveArray,
     * this finds the index of the last element &lt;= value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element &lt;= value 
     *     (or -1, if there are none)
     */
    public int binaryFindLastLE(int lowPo, int highPo, double value) {

        if (lowPo > highPo)
            return -1;
        int po = binarySearch(lowPo, highPo, value);

        //an exact match? find the first exact match
        if (po >= 0) {
            while (po < highPo && getDouble(po + 1) == value)
                po++;
            return po;
        }

        //no exact match? return binary search po -1
        //thus returning a positive number
        //the inverse of -x-1 is -x-1 !
        return -po -1 -1;
    }


    /**
     * Given a sorted PrimitiveArray,
     * this finds the index of the last element &lt; or almostEqual5 to value. 
     *
     * <p>If firstGE &gt; lastLE, there are no matching elements (because
     * the requested range is less than or greater than all the values,
     * or between two adjacent values).
     *
     * @param lowPo the low index to start with, usually 0
     * @param highPo the high index to start with, usually 
     *  (originalPrimitiveArray.size() - 1)
     * @param value the value you are searching for
     * @return the index of the first element &lt; or Math2.almostEqual5 to value 
     *     (or -1, if there are none)
     */
    public int binaryFindLastLAE5(int lowPo, int highPo, double value) {

        if (lowPo > highPo)
            return -1;
        int po = binarySearch(lowPo, highPo, value);

        //no exact match? start at lower and work forward
        //the inverse of -x-1 is -x-1 !
        if (po < 0)
            po = -po -1 -1;

        //find the first match
        while (po < highPo && Math2.almostEqual(5, getDouble(po + 1), value))
            po++;

        return po;
    }


    /**
     * Find the closest element to x in an ascending sorted array.
     * If there are duplicates, any may be returned.
     *
     * @param x
     * @return the index of the index of the element closest to x.
     *   If x is NaN, this returns -1.
     */
    public int binaryFindClosest(double x) {
        if (Double.isNaN(x))
            return -1;
        int i = binarySearch(0, size - 1, x);
        if (i >= 0)
            return i; //success, exact match

        //insertionPoint at end point?
        int insertionPoint = -i - 1;  //0.. dar.length
        if (insertionPoint == 0) 
            return 0;
        if (insertionPoint >= size)
            return size - 1;

        //insertionPoint between 2 points 
        if (Math.abs(getDouble(insertionPoint - 1) - x) <
            Math.abs(getDouble(insertionPoint) - x))
             return insertionPoint - 1;
        else return insertionPoint;
    }

    /**
     * Find the closest element to x in an array (regardless of if sorted or not).
     * If there are duplicates, any may be returned (i.e., for now, not specified).
     *
     * @param x
     * @return the index of the index of the element closest to x.
     *   If x is NaN, this returns -1.
     */
    public int linearFindClosest(double x) {
        if (Double.isNaN(x))
            return -1;
        double diff = Double.MAX_VALUE;
        int which = -1;
        for (int i = 0; i < size; i++) {
            double tDiff = Math.abs(getDouble(i) - x);
            if (tDiff < diff) {
                diff = tDiff;
                which = i;
            }
        }
        return which;
    }



    /**
     * This returns an PrimitiveArray which is the simplist possible 
     * type of PrimitiveArray which can accurately hold the data.
     *
     * <p>If this primitiveArray is a StringArray, then null, ".", "", and "NaN" are allowable 
     *   missing values for conversion to numeric types.
     *
     * @return the simpliest possible PrimitiveArray (possibly this PrimitiveArray) 
     *     (although not CharArray).
     */
    public PrimitiveArray simplify() {
        int type = 0; //the current, simplest possible type
        int n = size();
        boolean isStringArray = this instanceof StringArray;
        double dar[] = new double[n];
        for (int i = 0; i < n; i++) {
            double d = getDouble(i);
            dar[i] = d; //set this before s.equals tests
            if (isStringArray) {
                String s = getString(i);
                if (s == null || s.equals(".") || s.equals("") || s.equals("NaN"))
                    continue;
                //if a String is found (not an acceptable "NaN" string above), return original array
                if (Double.isNaN(d))
                    return this; //it's already a StringArray
            }
            //all types allow NaN
            if (Double.isNaN(d))
                continue;

            //assume column contains only bytes
            //if not true, work way up: short -> int -> long -> float -> double -> String
            if (type == 0) { //byte
                if (d != Math.rint(d) || d < Byte.MIN_VALUE || d > Byte.MAX_VALUE) {
                    type++;
                    if (this instanceof CharArray || this instanceof ShortArray)
                        return this; //don't continue; it would just check that a ShortArray contains shorts
                }
            }
            if (type == 1) { //short
                if (d != Math.rint(d) || d < Short.MIN_VALUE || d > Short.MAX_VALUE) {
                    type++;
                    if (this instanceof IntArray)
                        return this;  //don't continue; it would just check that an IntArray contains ints
                }
            }
            if (type == 2) { //int
                if (d != Math.rint(d) || d < Integer.MIN_VALUE || d > Integer.MAX_VALUE) {
                    type++;
                    if (this instanceof LongArray)
                        return this; //don't continue; it would just check that a LongArray contains long
                }
            }
            if (type == 3) { //long
                if (d != Math.round(d) || d < Long.MIN_VALUE || d > Long.MAX_VALUE) {
                    type++;
                    if (this instanceof FloatArray)
                        return this;  //don't continue; it would just check that a FloatArray contains floats
                }
            }
            if (type == 4) { //float
                if (d < -Float.MAX_VALUE || d > Float.MAX_VALUE || d != Math2.niceDouble(d, 7)) {
                    type++;
                    if (this instanceof DoubleArray)
                        return this;  //don't continue; it would just check that a DoubletArray contains doubles
                }
            }
            //otherwise it is a valid double
        }

        //make array of simplified type
        if (type == 0) {
            byte[] array = new byte[n];
            for (int i = 0; i < n; i++) 
                array[i] = Math2.roundToByte(dar[i]);
            return new ByteArray(array);
        }
        if (type == 1) {
            short[] array = new short[n];
            for (int i = 0; i < n; i++)
                array[i] = Math2.roundToShort(dar[i]);
            return new ShortArray(array);
        }
        if (type == 2) {
            int[] array = new int[n];
            for (int i = 0; i < n; i++)
                array[i] = Math2.roundToInt(dar[i]);
            return new IntArray(array);
        }
        if (type == 3) {
            long[] array = new long[n];
            for (int i = 0; i < n; i++)
                array[i] = Math2.roundToLong(dar[i]);
            return new LongArray(array);
        }
        if (type == 4) {
            float[] array = new float[n];
            for (int i = 0; i < n; i++)
                array[i] = (float)dar[i];
            return new FloatArray(array);
        }
        if (type == 5) {
            return new DoubleArray(dar);
        }
        throw new IllegalArgumentException(String2.ERROR + 
            " in PrimitiveArray.simplify: unknown type (" + type + ").");
    }

    /**
     * This appends the data in another primitiveArray to the current data.
     * WARNING: information may be lost from incoming primitiveArray this
     * primitiveArray is of a simpler type.
     *
     * @param primitiveArray primitiveArray must be a narrower data type,
     *  or the data will be rounded.
     */
    abstract public void append(PrimitiveArray primitiveArray);
    
    /**
     * Given table[], keys[], and ascending[],
     * this creates an int[] with the ranks the rows of the table. 
     *
     * <p>This sort is stable: equal elements will not be reordered as a result of the sort.
     *
     * @param table a List of PrimitiveArrays 
     * @param keys an array of the key column numbers 
     *    (each is 0..nColumns-1, the first key is the most important)
     *    which are used to determine the sort order
     * @param ascending an array of booleans corresponding to the keys
     *    indicating if the arrays are to be sorted by a given key in 
     *    ascending or descending order.
     * @return an int[] with values (0 ... size-1) 
     *   which points to the row number for a row with a specific 
     *   rank (e.g., rank[0] is the row number of the first item 
     *   in the sorted list, rank[1] is the row number of the
     *   second item in the sorted list, ...).
     */
    public static int[] rank(List table, int keys[], boolean[] ascending) {

        //create the comparator and the rowArray with pointer to specific rows
        RowComparator comparator = new RowComparator(table, keys, ascending);
        int n = ((PrimitiveArray)table.get(0)).size();
        Integer rowArray[] = new Integer[n];
        for (int i = 0; i < n; i++)
            rowArray[i] = new Integer(i);

        //sort the rows
        Arrays.sort(rowArray, comparator);   //this is "stable"
        //String2.log("rank results: " + String2.toCSVString(integerArray));

        //create the int[] 
        int newArray[] = new int[n];
        for (int i = 0; i < n; i++)
            newArray[i] = rowArray[i].intValue();

        return newArray;
    }

    /**
     * Given a List of PrimitiveArrays, which represents a table of data,
     * this sorts all of the PrimitiveArrays in the table based on the keys and
     * ascending values.
     *
     * <p>This sort is stable: equal elements will not be reordered as a result of the sort.
     *
     * @param table a List of PrimitiveArray[]
     * @param keys an array of the key column numbers 
     *    (each is 0..nColumns-1, the first key is the most important)
     *    which are used to determine the sort order
     * @param ascending an array of booleans corresponding to the keys
     *    indicating if the arrays are to be sorted by a given key in 
     *    ascending or descending order.
     */
    public static void sort(List table, 
        int keys[], boolean[] ascending) {

        //rank the rows
        int ranks[] = rank(table, keys, ascending);

        //reorder the columns
        for (int col = 0; col < table.size(); col++) {
            ((PrimitiveArray)table.get(col)).reorder(ranks);
        }
    }

    /**
     * Given a List of PrimitiveArrays, which represents a table of data,
     * this copies the values from one row to another (without affecting
     * any other rows).
     *
     * @param table a List of PrimitiveArray
     * @param from the 'from' row
     * @param to the 'to' row
     */
    public static void copyRow(List table, int from, int to) {
        int nColumns = table.size();
        for (int col = 0; col < nColumns; col++) 
            ((PrimitiveArray)table.get(col)).copy(from, to);
    }

    /**
     * Given a (presumably) sorted PrimitiveArray List, which represents a table of data,
     * this looks for adjacent identical rows of data and removes the duplicates.
     *
     * @param table a List of PrimitiveArray
     * @return the number of duplicates removed
     */
    public static int removeDuplicates(List table) {

        int nRows = ((PrimitiveArray)table.get(0)).size();
        if (nRows <= 1) 
            return 0;
        int nColumns = table.size();
        int po = 1; //row 0 is unique
        for (int row = 1; row < nRows; row++) { //start at 1; compare to previous row
            //does it equal row above?
            boolean equal = true;
            for (int col = 0; col < nColumns; col++) {
                if (((PrimitiveArray)table.get(col)).compare(row - 1, row) != 0) {
                    equal = false;
                    break;
                }
            }
            if (!equal) {
                //no? copy row 'row' to row 'po'
                if (row != po) 
                    for (int col = 0; col < nColumns; col++) 
                        ((PrimitiveArray)table.get(col)).copy(row, po);
                po++;
            }
        }

        //remove the stuff at the end
        for (int col = 0; col < nColumns; col++) 
            ((PrimitiveArray)table.get(col)).removeRange(po, nRows);

        return nRows - po;
    }

    /**
     * This removes rows in which the value in 'column' are less than
     * the value in the previous row.
     * Rows with values of NaN or bigger than 1e300 are also removed.
     * !!!Trouble: one erroneous big value will cause all subsequent valid values to be tossed.
     *
     * @param table a List of PrimitiveArray
     * @param column the column which should be ascending
     * @return the number of rows removed
     */
    public static int ensureAscending(List table, int column) {

        PrimitiveArray columnPA = (PrimitiveArray)table.get(column);
        int nRows = columnPA.size();
        int nColumns = table.size();
        int nGood = 0; 
        double lastGood = -Double.MAX_VALUE;
        for (int row = 0; row < nRows; row++) { 
            //is this a good row?
            double d = columnPA.getDouble(row);
            if (Math2.isFinite(d) && d < 1e300 && d >= lastGood) {
                //copy row 'row' to row 'nGood'
                if (row != nGood) 
                    for (int col = 0; col < nColumns; col++) 
                        ((PrimitiveArray)table.get(col)).copy(row, nGood);
                nGood++;
                lastGood = d;
            } else {
            }
        }

        //remove the stuff at the end
        for (int col = 0; col < nColumns; col++) 
            ((PrimitiveArray)table.get(col)).removeRange(nGood, nRows);

        int nRemoved = nRows - nGood;
        if (nRemoved > 0) 
            String2.log("PrimitveArray.ensureAscending nRowsRemoved=" + 
                nRemoved + " lastGood=" + lastGood);

        return nRemoved;
    }

    /**
     * Given two PrimitivesArray[]'s (representing two tables of data
     * with the same number columns), 
     * this appends table2 to the end of table1.
     *
     * <p>The columns types may be different. If table2's column is narrower,
     * the data is simply appended to table1. If table2's column is wider,
     * a new wider table1 is made before appending table2's data.
     * table1 and its columns will be affected by this method; table1 may
     * contain different PrimitiveArrays at the end.
     * table2 and its columns will not be changed by this method.
     *
     * @param table1 a List of PrimitiveArrays; it will contain the resulting
     *    table, perhaps containing some different PrimitiveArrays,
     *    always containing all the data from table1 and table2.
     * @param table2 a List of PrimitiveArrays
     */
    public static void append(List table1, List table2) {

        String errorInMethod = String2.ERROR + " in PrimitiveArray.append:\n";
        Test.ensureEqual(table1.size(), table2.size(), 
            errorInMethod + "the tables have a different number of columns.");

        //ensure that columns in table1 have data types at least as wide as table2
        //for (int i = 0; i < table1.length; i++)
        //    Test.ensureEqual(table1[i].getElementType(), table2[i].getElementType(), 
        //        errorInMethod + "the column #" + i + "'s have different element types.");        

        //append table2 to the end of table1
        for (int col = 0; col < table1.size(); col++) {
            //if needed, make a new wider PrimitiveArray in table1
            if (((PrimitiveArray)table2.get(col)).getElementTypeIndex() > 
                ((PrimitiveArray)table1.get(col)).getElementTypeIndex()) {
                PrimitiveArray oldTable1Col = ((PrimitiveArray)table1.get(col));
                PrimitiveArray newTable1Col = factory(((PrimitiveArray)table2.get(col)).getElementType(), 
                    oldTable1Col.size() + ((PrimitiveArray)table2.get(col)).size(), false);
                newTable1Col.append(oldTable1Col);
                table1.set(col, newTable1Col);
            }

            //append the data from table2 to table1
            ((PrimitiveArray)table1.get(col)).append(((PrimitiveArray)table2.get(col)));
        }

    }

    /**
     * Given two PrimitivesArray[]'s (representing two tables of data
     * with the same number columns), 
     * this merges table2 and table1 according to the sort order
     * defined by keys and ascending.
     * The columns types may be different; if table2's column is wider,
     * its data will be narrowed when it is appended.
     * table2 will not be changed by this method.
     *
     * @param table1 a List of PrimitiveArrays; it will contain the resulting
     *    table, perhaps containing some different PrimitiveArrays,
     *    always containing all the data from table1 and table2.
     * @param table2 a List of PrimitiveArrays
     * @param keys an array of the key column numbers 
     *    (each is 0..nColumns-1, the first key is the most important)
     *    which are used to determine the sort order
     * @param ascending an array of booleans corresponding to the keys
     *    indicating if the arrays are to be sorted by a given key in 
     *    ascending or descending order.
     * @param removeDuplicates specifies if completely identical rows should
     *    be removed.
     */
    public static void merge(List table1, List table2,
        int[] keys, boolean ascending[], boolean removeDuplicates) {
        //the current approach is quick, fun, and easy, but uses lots of memory
        //future: if needed, this could be done in a more space-saving way:
        //   sort each table, then merge table2 into table1, 
        //   but would have to be careful to avoid slowness from inserting rows
        //   of table2 into table1, or space lost to copying to a 3rd table.

        append(table1, table2);//this is fast (although uses lots of memory)
        sort(table1, keys, ascending);  //this is fast 
        if (removeDuplicates)
            removeDuplicates(table1);   //this is fast
    }


    /**
     * For all values, this multiplies by scale and then adds addOffset.
     * Calculations are done as doubles then, if necessary, rounded and stored.
     *
     * @param scale
     * @param addOffset
     */
    public void scaleAddOffset(double scale, double addOffset) {
        if (scale == 1 && addOffset == 0)
            return;
        for (int i = 0; i < size; i++)
            setDouble(i, getDouble(i) * scale + addOffset); //NaNs remain NaNs
    }
     
    /**
     * For all values, this adds addOffset then multiplies by scale.
     * Calculations are done as doubles then, if necessary, rounded and stored.
     *
     * @param scale
     * @param addOffset
     */
    public void addOffsetScale(double addOffset, double scale) {
        if (scale == 1 && addOffset == 0)
            return;
        for (int i = 0; i < size; i++)
            setDouble(i, (getDouble(i) + addOffset) * scale); //NaNs remain NaNs
    }
     
    /**
     * This returns a new (always) PrimitiveArray of type elementType
     * which has had the scale and addOffset values applied.
     * Calculations are done as doubles then, if necessary, rounded and stored.
     *
     * @param elementType 
     * @param scale
     * @param addOffset
     * @return a new (always) PrimitiveArray
     */
    public PrimitiveArray scaleAddOffset(Class elementType, double scale, double addOffset) {
        PrimitiveArray pa = factory(elementType, size, true);
        for (int i = 0; i < size; i++)
            pa.setDouble(i, getDouble(i) * scale + addOffset); //NaNs remain NaNs
        return pa;
    }
     
    /**
     * This returns a new (always) PrimitiveArray of type elementType
     * which has had the addOffset and scale values applied.
     * Calculations are done as doubles then, if necessary, rounded and stored.
     *
     * @param elementType 
     * @param addOffset
     * @param scale
     * @return a new (always) PrimitiveArray
     */
    public PrimitiveArray addOffsetScale(Class elementType, double addOffset, double scale) {
        PrimitiveArray pa = factory(elementType, size, true);
        for (int i = 0; i < size; i++)
            pa.setDouble(i, (getDouble(i) + addOffset) * scale); //NaNs remain NaNs
        return pa;
    }
     
    /**
     * This populates 'indices' with the indices (ranks) of the values in this PrimitiveArray
     * (ties get the same index). For example, 10,10,25,3 returns 1,1,2,0.
     *
     * @param indices the intArray that will capture the indices of the values 
     *  (ties get the same index). For example, 10,10,25,3 returns 1,1,2,0.
     * @return a PrimitveArray (the same type as this class) with the unique values, sorted
     */
    public abstract PrimitiveArray makeIndices(IntArray indices);

    /**
     * This changes all instances of the first value to the second value.
     * Note that, e.g., ByteArray.switchFromTo("127", "") will correctly
     * detect that 127=127 and do nothing.
     *
     * @param from the original value (use "" for standard missingValue)
     * @param to   the new value (use "" for standard missingValue)
     * @return the number of values switched
     */
    public abstract int switchFromTo(String from, String to);

    /**
     * If the primitiveArray has fake _FillValue and/or missing_values (e.g., -9999999),
     * those values are converted to PrimitiveArray-style missing values 
     * (NaN, or MAX_VALUE for integer types).
     *
     * @param fakeFillValue (e.g., -9999999) from colAttributes.getDouble("_FillValue"); use NaN if none
     * @param fakeMissingValue (e.g., -9999999) from colAttributes.getDouble("missing_value"); use NaN if none
     */
    public void convertToStandardMissingValues(double fakeFillValue, double fakeMissingValue) {
        //do nothing to String columns
        if (this instanceof StringArray)
            return;

        //is _FillValue used?    switch data to standard mv
        //String2.log("Table.convertToStandardMissingValues col=" + column + " fillValue=" + fillValue);
        if (!Double.isNaN(fakeFillValue))
            switchFromTo("" + fakeFillValue, "");

        //is missing_value used?    switch data to standard mv
        //String2.log("Table.convertToStandardMissingValues col=" + column + " missingValue=" + missingValue);
        if (!Double.isNaN(fakeMissingValue) && fakeMissingValue != fakeFillValue) //if fakeFillValue==NaN   2nd clause always true (good)
            switchFromTo("" + fakeMissingValue, "");
    }

    /**
     * For FloatArray and DoubleArray, this changes all standard 
     * missing values (NaN's) to fakeMissingValues.
     *
     * @param fakeMissingValue
     * @return the number of values switched
     */
    public int switchNaNToFakeMissingValue(double fakeMissingValue) {
        if (Math2.isFinite(fakeMissingValue) &&
//???why just FloatArray and DoubleArray???
            (this instanceof FloatArray || this instanceof DoubleArray))
            return switchFromTo("", "" + fakeMissingValue);
        return 0;
    }

    /**
     * For FloatArray and DoubleArray, this changes all fakeMissingValues
     * to standard missing values (NaN's).
     *
     * @param fakeMissingValue
     */
    public void switchFakeMissingValueToNaN(double fakeMissingValue) {
        if (Math2.isFinite(fakeMissingValue) &&
//???why just FloatArray and DoubleArray???
            (this instanceof FloatArray || this instanceof DoubleArray))
            switchFromTo("" + fakeMissingValue, "");
    }

    /**
     * This calls calculateStats and returns a diagnostic string with n, min, max.
     *
     */
    public String statsString() {
        double[] stats = calculateStats();
        return "n=" + String2.right(String2.genEFormat6(stats[STATS_N]), 7) + 
            " nNaN=" + String2.right(String2.genEFormat6(size() - stats[STATS_N]), 7) + 
            " min=" + String2.right(String2.genEFormat6(stats[STATS_MIN]), 15) + 
            " max=" + String2.right(String2.genEFormat6(stats[STATS_MAX]), 15);
    }

    /**
     * This tests if the values in the array are sorted in ascending order (ties are ok).
     * This details of this test are geared toward determining if the 
     * values are suitable for binarySearch.
     *
     * @return "" if the values in the array are sorted in ascending order;
     *   or an error message if not (i.e., if descending or unordered).
     *   If size is 0 or 1 (non-missing value), this returns "".
     */
    public abstract String isAscending();

    /**
     * This tests if the values in the array are sorted in descending order (ties are ok).
     *
     * @return "" if the values in the array are sorted in descending order;
     *   or an error message if not (i.e., if ascending or unordered).
     *   If size is 0 or 1 (non-missing value), this returns "".
     */
    public abstract String isDescending();

    /**
     * This tests for adjacent tied values and returns the index of the first tied value.
     * Adjacent NaNs are treated as ties.
     *
     * @return the index of the first tied value (or -1 if none).
     */
    public abstract int firstTie();

    /**
     * This compares this PrimitiveArray's values to anothers, string representation by string representation, 
     * and returns the first index where different.
     * this.get(i)=null and other.get(i)==null is treated as same value.
     *  
     * @param other
     * @return index where first different (or -1 if same).
     *    Note that the index may equal the size of this or the other primitiveArray.
     */
    public int diffIndex(PrimitiveArray other) {
        int i = 0;
        int otherSize = other.size();

        while (true) {
            if (i == size && size == otherSize)
                return -1;
            if (i == size ||
                i == otherSize) 
                return i;
            String s  = getString(i);
            String so = other.getString(i);
            if (s  == null && so != null) return i;
            if (so == null && s  != null) return i;
            if (s != null && so != null && !s.equals(so))
                return i;
            i++;
        }

        //you could do a double test if both pa's were numeric
        //but tests with inifinity and nan are awkward and time consuming
        //so string test is pretty good approach.
    }

    /**
     * This compares this PrimitiveArray's values to anothers, string representation by string representation, 
     * and returns String indicating where different (or "" if not different).
     * this.get(i)=null and other.get(i)==null is treated as same value.
     *  
     * @param other (or old)
     * @return String indicating where different (or "" if not different).
     */
    public String diffString(PrimitiveArray other) {
        int diffi = diffIndex(other);
        if (diffi == -1)
            return "";
        String s1 = diffi == size? null : getString(diffi);
        String s2 = diffi == other.size()? null : other.getString(diffi);
        return 
            "  old index #" + diffi + "=" + s2 + ",\n" +
            "  new index #" + diffi + "=" + s1 + ".";
    }

    /**
     * This compares this PrimitiveArray's values to anothers, string representation by string representation, 
     * and throws Exception if different.
     * this.get(i)=null and other.get(i)==null is treated as same value.
     *  
     * @param other
     * @throws Exception if different
     */
    public void diff(PrimitiveArray other) throws Exception {
        int diffi = diffIndex(other);
        if (diffi == -1)
            return;
        String s1 = diffi == size? null : getString(diffi);
        String s2 = diffi == other.size()? null : other.getString(diffi);
        Test.ensureEqual(s1, s2, "The PrimitiveArrays differ at index=" + diffi);
    }

    /**
     * This tests if the values in the array are evenly spaced (ascending or descending).
     *
     * @return "" if the values in the array are evenly spaced;
     *   or an error message if not.
     *   If size is 0 or 1, this returns "".
     */
    public abstract String isEvenlySpaced();

    /** This returns the minimum value that can be held by this class. */
    public abstract String minValue();

    /** This returns the maximum value that can be held by this class. */
    public abstract String maxValue();

    /**
     * This finds the number of non-missing values, and the index of the min and
     *    max value.
     *
     * @return int[3], [0]=the number of non-missing values, 
     *    [1]=index of min value (if tie, index of last found; -1 if all mv),
     *    [2]=index of max value (if tie, index of last found; -1 if all mv).
     */
    public abstract int[] getNMinMaxIndex();

    /**
     * Given nHave values and stride, this returns the actual number of points that will be found.
     *
     * @param nHave the size of the array (or  end-start+1)
     * @param stride  (must be >= 1)
     * @return the actual number of points that will be found.
     */
    public static int strideWillFind(int nHave, int stride) {
        return 1 + (nHave - 1) / stride;
    }

    /**
     * This makes a new subset of this PrimitiveArray based on startIndex, stride,
     * and stopIndex.
     *
     * @param startIndex must be a valid index
     * @param stride   must be at least 1
     * @param stopIndex (inclusive) must be &gt;= startIndex and &lt; size.
     * @return a new PrimitiveArray with the desired subset.
     */
    public abstract PrimitiveArray subset(int startIndex, int stride, int stopIndex);

    /**
     * This tests the methods of this class.
     *
     * @throws Exception if trouble.
     */
    public static void test() throws Throwable {
        String2.log("*** Testing PrimitiveArray");


        //test factory 
        PrimitiveArray pa;
        Test.ensureEqual(factory(new byte[]{1}).getElementType(), byte.class, "");
        Test.ensureEqual(factory(new char[]{1}).getElementType(), char.class, "");
        Test.ensureEqual(factory(new short[]{1}).getElementType(), short.class, "");
        Test.ensureEqual(factory(new int[]{1}).getElementType(), int.class, "");
        Test.ensureEqual(factory(new long[]{1}).getElementType(), long.class, "");
        Test.ensureEqual(factory(new float[]{1}).getElementType(), float.class, "");
        Test.ensureEqual(factory(new double[]{1}).getElementType(), double.class, "");
        Test.ensureEqual(factory(new String[]{"1"}).getElementType(), String.class, "");

        Test.ensureEqual(factory(byte.class, 1, true).getElementType(), byte.class, "");
        Test.ensureEqual(factory(char.class, 1, true).getElementType(), char.class, "");
        Test.ensureEqual(factory(short.class, 1, true).getElementType(), short.class, "");
        Test.ensureEqual(factory(int.class, 1, true).getElementType(), int.class, "");
        Test.ensureEqual(factory(long.class, 1, true).getElementType(), long.class, "");
        Test.ensureEqual(factory(float.class, 1, true).getElementType(), float.class, "");
        pa = factory(double.class, 1, true);
        Test.ensureEqual(pa.getElementType(), double.class, "");
        Test.ensureEqual(pa.getDouble(0), 0, "");
        pa = factory(String.class, 1, true);
        Test.ensureEqual(pa.getElementType(), String.class, "");
        Test.ensureEqual(pa.getString(0), "", "");

        Test.ensureEqual(factory(byte.class,   1, "10").toString(), "10", "");
        Test.ensureEqual(factory(char.class,   2, "abc").toString(), "97, 97", "");
        Test.ensureEqual(factory(short.class,  3, "30").toString(), "30, 30, 30", "");
        Test.ensureEqual(factory(int.class,    4, "40").toString(), "40, 40, 40, 40", "");
        Test.ensureEqual(factory(long.class,   5, "50").toString(), "50, 50, 50, 50, 50", "");
        Test.ensureEqual(factory(float.class,  6, "60").toString(), "60.0, 60.0, 60.0, 60.0, 60.0, 60.0", "");
        Test.ensureEqual(factory(double.class, 7, "70").toString(), "70.0, 70.0, 70.0, 70.0, 70.0, 70.0, 70.0", "");
        Test.ensureEqual(factory(String.class, 8, "ab").toString(), "ab, ab, ab, ab, ab, ab, ab, ab", "");

        //test simplify
        pa = new StringArray(new String[]{"-127", "126", ".", "NaN", null});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof ByteArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -127, "");
        Test.ensureEqual(pa.getDouble(1), 126, "");
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");
        Test.ensureEqual(pa.getDouble(3), Double.NaN, "");
        Test.ensureEqual(pa.getDouble(4), Double.NaN, "");

        //pa = new StringArray(new String[]{"0", "65534", "."});
        //pa = pa.simplify();
        //Test.ensureTrue(pa instanceof CharArray, "elementType=" + pa.getElementType());
        //Test.ensureEqual(pa.getDouble(0), 0, "");
        //Test.ensureEqual(pa.getDouble(1), 65534, "");
        //Test.ensureEqual(pa.getDouble(2), Character.MAX_VALUE, "");

        pa = new StringArray(new String[]{"-32767", "32766", "."});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof ShortArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -32767, "");
        Test.ensureEqual(pa.getDouble(1), 32766, "");
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");

        pa = new StringArray(new String[]{"-2000000000", "2000000000", "."});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof IntArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -2000000000, "");
        Test.ensureEqual(pa.getDouble(1), 2000000000, "");
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");

        pa = new StringArray(new String[]{"-2000000000000000", "2000000000000000", "."});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof LongArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -2000000000000000L, "");
        Test.ensureEqual(pa.getDouble(1), 2000000000000000L, "");
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");

        pa = new StringArray(new String[]{"-1e33", "1e33", "."});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof FloatArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -1e33f, ""); //'f' bruises it
        Test.ensureEqual(pa.getDouble(1), 1e33f, "");  //'f' bruises it
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");

        pa = new StringArray(new String[]{"-1e307", "1e307", "."});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof DoubleArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getDouble(0), -1e307, "");
        Test.ensureEqual(pa.getDouble(1), 1e307, "");
        Test.ensureEqual(pa.getDouble(2), Double.NaN, "");

        pa = new StringArray(new String[]{".", "123", "4b"});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof StringArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getString(0), ".", "");
        Test.ensureEqual(pa.getString(1), "123", "");
        Test.ensureEqual(pa.getString(2), "4b", "");

        pa = new DoubleArray(new double[]{Double.NaN, 123.4, 12});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof FloatArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getFloat(0), Float.NaN, "");
        Test.ensureEqual(pa.getFloat(1), 123.4f, "");
        Test.ensureEqual(pa.getFloat(2), 12f, "");

        pa = new DoubleArray(new double[]{Double.NaN, 100000, 12});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof IntArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getInt(0), Integer.MAX_VALUE, "");
        Test.ensureEqual(pa.getInt(1), 100000, "");
        Test.ensureEqual(pa.getInt(2), 12, "");

        pa = new DoubleArray(new double[]{Double.NaN, 100, 12});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof ByteArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getInt(0), Integer.MAX_VALUE, "");
        Test.ensureEqual(pa.getInt(1), 100, "");
        Test.ensureEqual(pa.getInt(2), 12, "");

        pa = new IntArray(new int[]{Integer.MAX_VALUE, 100, 12});
        pa = pa.simplify();
        Test.ensureTrue(pa instanceof ByteArray, "elementType=" + pa.getElementType());
        Test.ensureEqual(pa.getInt(0), Integer.MAX_VALUE, "");
        Test.ensureEqual(pa.getInt(1), 100, "");
        Test.ensureEqual(pa.getInt(2), 12, "");

        //test rank
        ByteArray arByte     = new ByteArray(   new byte[]  {0, 100, 50, 110});
        FloatArray arFloat   = new FloatArray(  new float[] {1, 3, 3, -5});
        DoubleArray arDouble = new DoubleArray( new double[]{17, 1e300, 3, 0});
        StringArray arString = new StringArray( new String[]{"a", "abe", "A", "ABE"});
        ArrayList table = String2.toArrayList(new Object[]{arByte, arFloat, arDouble, arString});
        Test.ensureEqual(rank(table, new int[]{0}, new boolean[]{true}), //ascending
            new int[]{0, 2, 1, 3}, "");
        Test.ensureEqual(rank(table, new int[]{0}, new boolean[]{false}), //descending
            new int[]{3, 1, 2, 0}, "");
        Test.ensureEqual(rank(table, new int[]{1}, new boolean[]{true}), //ties
            new int[]{3, 0, 1, 2}, "");
        Test.ensureEqual(rank(table, new int[]{2}, new boolean[]{true}),
            new int[]{3, 2, 0, 1}, "");
        Test.ensureEqual(rank(table, new int[]{3}, new boolean[]{true}),
            new int[]{2, 3, 0, 1}, "");
        Test.ensureEqual(rank(table, new int[]{1, 0}, new boolean[]{true, true}), //tie, a/ascending
            new int[]{3, 0, 2, 1}, "");
        Test.ensureEqual(rank(table, new int[]{1, 0}, new boolean[]{true, false}), //tie, a/descending
            new int[]{3, 0, 1, 2}, "");
        Test.ensureEqual(arByte.getElementType(), byte.class, "");
        Test.ensureEqual(arFloat.getElementType(), float.class, "");
        Test.ensureEqual(arDouble.getElementType(), double.class, "");
        Test.ensureEqual(arString.getElementType(), String.class, "");

        Test.ensureEqual(arByte.getElementTypeString(), "byte", "");
        Test.ensureEqual(arFloat.getElementTypeString(), "float", "");
        Test.ensureEqual(arDouble.getElementTypeString(), "double", "");
        Test.ensureEqual(arString.getElementTypeString(), "String", "");

        //test sort  result = {3, 0, 2, 1});
        sort(table, new int[]{1, 0}, new boolean[]{true, true}); //tie, a/ascending
        Test.ensureEqual(arByte.array,   new byte[]{  110,   0,   50,  100},   "");
        Test.ensureEqual(arFloat.array,  new float[]{ -5,    1,   3,   3},     "");
        Test.ensureEqual(arDouble.array, new double[]{0,     17,  3,   1e300}, "");
        Test.ensureEqual(arString.array, new String[]{"ABE", "a", "A", "abe"}, "");

        //test sort  result = {3, 0, 1, 2});
        sort(table, new int[]{1, 0}, new boolean[]{true, false}); //tie, a/descending
        Test.ensureEqual(arByte.array,   new byte[]{  110,   0,   100,   50},  "");
        Test.ensureEqual(arFloat.array,  new float[]{ -5,    1,   3,     3},   "");
        Test.ensureEqual(arDouble.array, new double[]{0,     17,  1e300, 3},   "");
        Test.ensureEqual(arString.array, new String[]{"ABE", "a", "abe", "A"}, "");

        //test removeDuplicates
        IntArray arInt3a = new IntArray(new int[]{1,5,5,7,7,7});
        IntArray arInt3b = new IntArray(new int[]{2,6,6,8,8,8});
        ArrayList table3 = String2.toArrayList(new Object[]{arInt3a, arInt3b});
        removeDuplicates(table3);
        Test.ensureEqual(arInt3a.toString(), "1, 5, 7", "");
        Test.ensureEqual(arInt3b.toString(), "2, 6, 8", "");

        //test merge  (which tests append and sort)
        ByteArray arByte2     = new ByteArray(   new byte[]  {5,   15,  50,   25});
        FloatArray arFloat2   = new FloatArray(  new float[] {4,   14,   3,   24});
        IntArray arInt2       = new IntArray(    new int[]{   3,   13,   3,   1}); //test: narrower than arDouble
        StringArray arString2 = new StringArray( new String[]{"b", "aa", "A", "c"});
        ArrayList table2 = String2.toArrayList(new Object[]{arByte2, arFloat2, arInt2, arString2});
        merge(table2, table, new int[]{1, 0}, new boolean[]{true, true}, false);
        Test.ensureEqual(((PrimitiveArray)table2.get(0)).toDoubleArray(), new double[]{110,   0,   50,  50,100,    5,  15,  25},   "");
        Test.ensureEqual(((PrimitiveArray)table2.get(1)).toDoubleArray(), new double[]{-5,    1,   3,   3,   3,    4,  14,  24},   "");
        Test.ensureEqual(((PrimitiveArray)table2.get(2)).toDoubleArray(), new double[]{0,     17,  3,   3, 1e300,  3,  13,  1},    "");
        Test.ensureEqual(((PrimitiveArray)table2.get(3)).toStringArray(), new String[]{"ABE", "a", "A", "A", "abe","b","aa","c"},  "");

        merge(table2, table, new int[]{1, 0}, new boolean[]{true, false}, true);
        Test.ensureEqual(((PrimitiveArray)table2.get(0)).toDoubleArray(), new double[]{110,   0,   100,  50,   5,   15,   25},   "");
        Test.ensureEqual(((PrimitiveArray)table2.get(1)).toDoubleArray(), new double[]{-5,    1,   3,    3,    4,   14,   24},   "");
        Test.ensureEqual(((PrimitiveArray)table2.get(2)).toDoubleArray(), new double[]{0,     17, 1e300, 3,    3,   13,   1},    "");
        Test.ensureEqual(((PrimitiveArray)table2.get(3)).toStringArray(), new String[]{"ABE", "a", "abe","A",  "b","aa","c"},    "");

        

        //** test speed
        int n = 10000000;
        long time1 = System.currentTimeMillis();
        int iar[] = new int[]{13,24,56};
        int sum = 0;
        for (int i = 0; i < n; i++)
            sum += iar[2];
        time1 = System.currentTimeMillis() - time1;

        long time2 = System.currentTimeMillis();
        IntArray ia = new IntArray(new int[]{13,24,56});
        sum = 0;
        for (int i = 0; i < n; i++)
            sum += ia.get(2);
        time2 = System.currentTimeMillis() - time2;

        String2.log("[] time=" + time1 + " IntArray time=" + time2);


        //** raf tests
        ByteArray   bar = new ByteArray(new byte[]{2,4,6,6,6,8});
        CharArray   car = new CharArray(new char[]{'\u0002','\u0004','\u0006','\u0006','\u0006','\u0008'});
        DoubleArray dar = new DoubleArray(new double[]{2,4,6,6,6,8});
        FloatArray  far = new FloatArray(new float[]{2,4,6,6,6,8});
        IntArray    Iar = new IntArray(new int[]{2,4,6,6,6,8});
        LongArray   lar = new LongArray(new long[]{2,4,6,6,6,8});
        ShortArray  sar = new ShortArray(new short[]{2,4,6,6,6,8});
        StringArray Sar = new StringArray(new String[]{"22","4444","666666","666666","666666","88888888"});

        Test.ensureEqual(bar.indexOf("6"), 2, "");
        Test.ensureEqual(car.indexOf("6"), 2, "");
        Test.ensureEqual(dar.indexOf("6"), 2, "");
        Test.ensureEqual(far.indexOf("6"), 2, "");
        Test.ensureEqual(Iar.indexOf("6"), 2, "");
        Test.ensureEqual(lar.indexOf("6"), 2, "");
        Test.ensureEqual(sar.indexOf("6"), 2, "");
        Test.ensureEqual(Sar.indexOf("666666"), 2, "");

        Test.ensureEqual(bar.indexOf("a"), -1, "");
        Test.ensureEqual(car.indexOf("a"), -1, "");
        Test.ensureEqual(dar.indexOf("a"), -1, "");
        Test.ensureEqual(far.indexOf("a"), -1, "");
        Test.ensureEqual(Iar.indexOf("a"), -1, "");
        Test.ensureEqual(lar.indexOf("a"), -1, "");
        Test.ensureEqual(sar.indexOf("a"), -1, "");
        Test.ensureEqual(Sar.indexOf("a"), -1, "");

        Test.ensureEqual(bar.indexOf("6", 3), 3, "");
        Test.ensureEqual(car.indexOf("6", 3), 3, "");
        Test.ensureEqual(dar.indexOf("6", 3), 3, "");
        Test.ensureEqual(far.indexOf("6", 3), 3, "");
        Test.ensureEqual(Iar.indexOf("6", 3), 3, "");
        Test.ensureEqual(lar.indexOf("6", 3), 3, "");
        Test.ensureEqual(sar.indexOf("6", 3), 3, "");
        Test.ensureEqual(Sar.indexOf("666666", 3), 3, "");

        Test.ensureEqual(bar.lastIndexOf("6"), 4, "");
        Test.ensureEqual(car.lastIndexOf("6"), 4, "");
        Test.ensureEqual(dar.lastIndexOf("6"), 4, "");
        Test.ensureEqual(far.lastIndexOf("6"), 4, "");
        Test.ensureEqual(Iar.lastIndexOf("6"), 4, "");
        Test.ensureEqual(lar.lastIndexOf("6"), 4, "");
        Test.ensureEqual(sar.lastIndexOf("6"), 4, "");
        Test.ensureEqual(Sar.lastIndexOf("666666"), 4, "");

        Test.ensureEqual(bar.lastIndexOf("a"), -1, "");
        Test.ensureEqual(car.lastIndexOf("a"), -1, "");
        Test.ensureEqual(dar.lastIndexOf("a"), -1, "");
        Test.ensureEqual(far.lastIndexOf("a"), -1, "");
        Test.ensureEqual(Iar.lastIndexOf("a"), -1, "");
        Test.ensureEqual(lar.lastIndexOf("a"), -1, "");
        Test.ensureEqual(sar.lastIndexOf("a"), -1, "");
        Test.ensureEqual(Sar.lastIndexOf("a"), -1, "");

        Test.ensureEqual(bar.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(car.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(dar.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(far.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(Iar.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(lar.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(sar.lastIndexOf("6", 3), 3, "");
        Test.ensureEqual(Sar.lastIndexOf("666666", 3), 3, "");

        //raf test2
        String raf2Name = File2.getSystemTempDirectory() + "PrimitiveArrayRaf2Test.bin";
        String2.log("raf2Name=" + raf2Name);
        File2.delete(raf2Name);
        Test.ensureEqual(File2.isFile(raf2Name), false, "");

        RandomAccessFile raf2 = new RandomAccessFile(raf2Name, "rw");
        long bStart = raf2.getFilePointer();
        rafWriteDouble(raf2, byte.class, 1.0);
        rafWriteDouble(raf2, byte.class, Double.NaN);
        long cStart = raf2.getFilePointer();
        rafWriteDouble(raf2, char.class, 2.0);
        rafWriteDouble(raf2, char.class, Double.NaN);
        long dStart = raf2.getFilePointer();
        rafWriteDouble(raf2, double.class, 3.0);
        rafWriteDouble(raf2, double.class, Double.NaN);
        long fStart = raf2.getFilePointer();
        rafWriteDouble(raf2, float.class, 4.0);
        rafWriteDouble(raf2, float.class, Double.NaN);
        long iStart = raf2.getFilePointer();
        rafWriteDouble(raf2, int.class, 5.0);
        rafWriteDouble(raf2, int.class, Double.NaN);
        long lStart = raf2.getFilePointer();
        rafWriteDouble(raf2, long.class, 6.0);
        rafWriteDouble(raf2, long.class, Double.NaN);
        long sStart = raf2.getFilePointer();
        rafWriteDouble(raf2, short.class, 7.0);
        rafWriteDouble(raf2, short.class, Double.NaN);
        //read in reverse order
        Test.ensureEqual(rafReadDouble(raf2, short.class,  sStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, short.class,  sStart, 0), 7.0, "");

        Test.ensureEqual(rafReadDouble(raf2, long.class,   lStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, long.class,   lStart, 0), 6.0, "");

        Test.ensureEqual(rafReadDouble(raf2, int.class,    iStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, int.class,    iStart, 0), 5.0, "");

        Test.ensureEqual(rafReadDouble(raf2, float.class,  fStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, float.class,  fStart, 0), 4.0, "");

        Test.ensureEqual(rafReadDouble(raf2, double.class, dStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, double.class, dStart, 0), 3.0, "");

        Test.ensureEqual(rafReadDouble(raf2, char.class,   cStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, char.class,   cStart, 0), 2.0, "");

        Test.ensureEqual(rafReadDouble(raf2, byte.class,   bStart, 1), Double.NaN, "");
        Test.ensureEqual(rafReadDouble(raf2, byte.class,   bStart, 0), 1.0, "");

        raf2.close();

        //raf test
        String rafName = File2.getSystemTempDirectory() + "PrimitiveArrayRafTest.bin";
        String2.log("rafName=" + rafName);
        File2.delete(rafName);
        Test.ensureEqual(File2.isFile(rafName), false, "");

        DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(
            new FileOutputStream(rafName)));
        long barStart = 0;
        long carStart = barStart + 6*bar.writeDos(dos);
        long darStart = carStart + 6*car.writeDos(dos);
        long farStart = darStart + 6*dar.writeDos(dos);
        long IarStart = farStart + 6*far.writeDos(dos);
        long larStart = IarStart + 6*Iar.writeDos(dos);
        long sarStart = larStart + 6*lar.writeDos(dos);
        long SarStart = sarStart + 6*sar.writeDos(dos);
        Test.ensureEqual(Sar.writeDos(dos), 10, "");
        int nBytesPerS = 9;
        //String2.log(File2.hexDump(dosName, 500));

        dos.close();

        //test rafReadDouble 
        RandomAccessFile raf = new RandomAccessFile(rafName, "rw");
        Test.ensureEqual(rafReadDouble(raf, byte.class,   barStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, byte.class,   barStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, char.class,   carStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, char.class,   carStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, double.class, darStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, double.class, darStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, float.class,  farStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, float.class,  farStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, int.class,    IarStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, int.class,    IarStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, long.class,   larStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, long.class,   larStart, 5), 8, "");
        Test.ensureEqual(rafReadDouble(raf, short.class,  sarStart, 0), 2, "");
        Test.ensureEqual(rafReadDouble(raf, short.class,  sarStart, 5), 8, "");
        //Test.ensureEqual(StringArray.rafReadString(raf,   SarStart, 0, nBytesPerS), "22", "");
        //Test.ensureEqual(StringArray.rafReadString(raf,   SarStart, 5, nBytesPerS), "88888888", "");

        //test rafBinarySearch
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 2), 0, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 4), 1, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 6), 2, ""); //2,3,4
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 8), 5, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 1), -1, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 3), -2, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 5), -3, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 7), -6, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 5, 9), -7, "");

        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 2), 0, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 4), 1, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 6), 4, ""); //any of 2,3,4
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 1), -1, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 3), -2, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 5), -3, "");
        Test.ensureEqual(rafBinarySearch(raf, float.class, farStart, 0, 4, 7), -6, "");

        //test rafFirstGE
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 2), 0, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 4), 1, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 6), 2, ""); //first
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 8), 5, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 1), 0, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 3), 1, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 5), 2, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 7), 5, "");
        Test.ensureEqual(rafFirstGE(raf, float.class, farStart, 0, 5, 9), 6, "");

        //test rafFirstGAE5
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 2), 0, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 2.0000001), 0, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 1.9999999), 0, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 4), 1, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 6), 2, ""); //first
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 6.0000001), 2, ""); 
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 5.9999999), 2, ""); 
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 8), 5, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 1), 0, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 3), 1, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 3.0000001), 1, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 2.9999999), 1, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 5), 2, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 7), 5, "");
        Test.ensureEqual(rafFirstGAE5(raf, float.class, farStart, 0, 5, 9), 6, "");

        //test rafLastLE
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 2), 0, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 4), 1, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 6), 4, ""); //last
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 8), 5, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 1), -1, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 3), 0, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 5), 1, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 7), 4, "");
        Test.ensureEqual(rafLastLE(raf, float.class, farStart, 0, 5, 9), 5, "");

        //test rafLastLAE5
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 2), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 2.0000001), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 1.9999999), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 4), 1, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 6), 4, ""); //last
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 6.0000001), 4, ""); 
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 5.9999999), 4, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 8), 5, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 1), -1, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 3), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 3.0000001), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 2.9999999), 0, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 5), 1, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 7), 4, "");
        Test.ensureEqual(rafLastLAE5(raf, float.class, farStart, 0, 5, 9), 5, "");
        raf.close();

        //test binarySearch
        //FloatArray  far = new FloatArray(new float[]{2,4,6,6,6,8});
        Test.ensureEqual(far.binarySearch(0, 5, 2), 0, "");
        Test.ensureEqual(far.binarySearch(0, 5, 4), 1, "");
        Test.ensureEqual(far.binarySearch(0, 5, 6), 2, ""); //2,3,4
        Test.ensureEqual(far.binarySearch(0, 5, 8), 5, "");
        Test.ensureEqual(far.binarySearch(0, 5, 1), -1, "");
        Test.ensureEqual(far.binarySearch(0, 5, 3), -2, "");
        Test.ensureEqual(far.binarySearch(0, 5, 5), -3, "");
        Test.ensureEqual(far.binarySearch(0, 5, 7), -6, "");
        Test.ensureEqual(far.binarySearch(0, 5, 9), -7, "");

        Test.ensureEqual(far.binarySearch(0, 4, 2), 0, "");
        Test.ensureEqual(far.binarySearch(0, 4, 4), 1, "");
        Test.ensureEqual(far.binarySearch(0, 4, 6), 4, ""); //any of 2,3,4
        Test.ensureEqual(far.binarySearch(0, 4, 1), -1, "");
        Test.ensureEqual(far.binarySearch(0, 4, 3), -2, "");
        Test.ensureEqual(far.binarySearch(0, 4, 5), -3, "");
        Test.ensureEqual(far.binarySearch(0, 4, 7), -6, "");

        //test binaryFindFirstGE
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 2), 0, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 4), 1, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 6), 2, ""); //first
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 8), 5, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 1), 0, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 3), 1, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 5), 2, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 7), 5, "");
        Test.ensureEqual(far.binaryFindFirstGE(0, 5, 9), 6, "");

        //test binaryFindFirstGAE5
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 2), 0, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 2.0000001), 0, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 1.9999999), 0, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 4), 1, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 6), 2, ""); //first
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 6.0000001), 2, ""); 
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 5.9999999), 2, ""); 
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 8), 5, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 1), 0, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 3), 1, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 3.0000001), 1, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 2.9999999), 1, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 5), 2, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 7), 5, "");
        Test.ensureEqual(far.binaryFindFirstGAE5(0, 5, 9), 6, "");

        //test binaryFindLastLE
        //FloatArray  far = new FloatArray(new float[]{2,4,6,6,6,8});
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 2), 0, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 4), 1, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 6), 4, ""); //last
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 8), 5, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 1), -1, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 3), 0, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 5), 1, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 7), 4, "");
        Test.ensureEqual(far.binaryFindLastLE(0, 5, 9), 5, "");

        //test binaryFindLastLAE5
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 2), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 2.0000001), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 1.9999999), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 4), 1, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 6), 4, ""); //last
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 6.0000001), 4, ""); 
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 5.9999999), 4, ""); 
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 8), 5, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 1), -1, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 3), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 3.0000001), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 2.9999999), 0, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 5), 1, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 7), 4, "");
        Test.ensureEqual(far.binaryFindLastLAE5(0, 5, 9), 5, "");

        //test binaryFindClosest
        //FloatArray  far = new FloatArray(new float[]{2,4,6,6,6,8});
        Test.ensureEqual(far.binaryFindClosest(2),   0, "");
        Test.ensureEqual(far.binaryFindClosest(2.1), 0, "");
        Test.ensureEqual(far.binaryFindClosest(1.9), 0, "");
        Test.ensureEqual(far.binaryFindClosest(4),   1, "");
        Test.ensureEqual(far.binaryFindClosest(6),   2, ""); //by chance
        Test.ensureEqual(far.binaryFindClosest(5.9), 2, ""); 
        Test.ensureEqual(far.binaryFindClosest(6.1), 4, ""); //since between 6 and 8
        Test.ensureEqual(far.binaryFindClosest(8),   5, "");
        Test.ensureEqual(far.binaryFindClosest(1),   0, "");
        Test.ensureEqual(far.binaryFindClosest(2.9), 0, "");
        Test.ensureEqual(far.binaryFindClosest(3.1), 1, "");
        Test.ensureEqual(far.binaryFindClosest(5.1), 2, "");
        Test.ensureEqual(far.binaryFindClosest(7.1), 5, "");
        Test.ensureEqual(far.binaryFindClosest(9),   5, "");

        //test linearFindClosest
        //FloatArray  far = new FloatArray(new float[]{2,4,6,6,6,8});
        Test.ensureEqual(far.linearFindClosest(2),   0, "");
        Test.ensureEqual(far.linearFindClosest(2.1), 0, "");
        Test.ensureEqual(far.linearFindClosest(1.9), 0, "");
        Test.ensureEqual(far.linearFindClosest(4),   1, "");
        Test.ensureEqual(far.linearFindClosest(6),   2, ""); //unspecified
        Test.ensureEqual(far.linearFindClosest(5.9), 2, ""); //unspecified
        Test.ensureEqual(far.linearFindClosest(6.1), 2, ""); //unspecified
        Test.ensureEqual(far.linearFindClosest(8),   5, "");
        Test.ensureEqual(far.linearFindClosest(1),   0, "");
        Test.ensureEqual(far.linearFindClosest(2.9), 0, "");
        Test.ensureEqual(far.linearFindClosest(3.1), 1, "");
        Test.ensureEqual(far.linearFindClosest(5.1), 2, ""); //unspecified
        Test.ensureEqual(far.linearFindClosest(7.1), 5, "");
        Test.ensureEqual(far.linearFindClosest(9),   5, "");

        //strideWillFind
        Test.ensureEqual(strideWillFind(5, 1), 5, "");
        Test.ensureEqual(strideWillFind(5, 2), 3, "");
        Test.ensureEqual(strideWillFind(5, 3), 2, "");
        Test.ensureEqual(strideWillFind(5, 4), 2, "");
        Test.ensureEqual(strideWillFind(5, 5), 1, "");

        //scaleAddOffset
        ia = new IntArray(new int[]{0,1,2,3,Integer.MAX_VALUE});
        ia.scaleAddOffset(1.5, 10);
        Test.ensureEqual(ia.toString(), "10, 12, 13, 15, 2147483647", "");

        String2.log("PrimitiveArray.test finished successfully.");
    }

    /**
     * This runs test.
     */
    public static void main(String args[]) throws Throwable {
        test();
    }

}

