/* This file is Copyright (c) 2005 Robert Alten Simons (info@cohort.com).
 * See the MIT/X-like license in LICENSE.txt.
 * For more information visit www.cohort.com or contact info@cohort.com.
 */
package com.cohort.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PushbackInputStream;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.Set;
import java.util.Vector;

/**
 * A class with static String methods that add to native String methods.
 * All are static methods. 
 */
public class String2 {

    /** the source code version number. */
    public static final double version = 1.000;

    /**
     * ERROR is a constant so that it will be consistent, so that one can 
     * search for it in output files.
     * This is the original definition, referenced by many other classes.
     */
    public final static String ERROR = "ERROR";

    //public static Logger log = Logger.getLogger("com.cohort.util");
    private static boolean logToSystemOut = false;
    private static boolean logToSystemErr = true;
    private static BufferedWriter logFile;
    private static String logFileName;
    private static StringBuffer logStringBuffer;
    private static int logMaxSize;
    private static long logFileSize;

    /**
     * This returns the line separator from
     *  <code>System.getProperty("line.separator");</code>
     */
    public static String lineSeparator = System.getProperty("line.separator");

    /** Returns true if the current Operating System is Windows. */
    public static boolean OSIsWindows =
        System.getProperty("os.name").toLowerCase().indexOf("windows") >= 0;
    /** Returns true if the current Operating System is Linux. */
    public static boolean OSIsLinux =
        System.getProperty("os.name").toLowerCase().indexOf("linux") >= 0;
    /** Returns true if the current Operating System is Mac OS X. */
    public static boolean OSIsMacOSX =
        System.getProperty("mrj.version") != null;

    private static DecimalFormat genStdFormat6 = new DecimalFormat("0.######");
    private static DecimalFormat genEngFormat6 = new DecimalFormat("##0.#####E0");
    private static DecimalFormat genExpFormat6 = new DecimalFormat("0.######E0");
    private static DecimalFormat genStdFormat10 = new DecimalFormat("0.##########");
    private static DecimalFormat genEngFormat10 = new DecimalFormat("##0.#########E0");
    private static DecimalFormat genExpFormat10 = new DecimalFormat("0.##########E0");

    private static String classPath; //lazy creation by getClassPath


    /**
     * This returns the string which sorts higher.
     * null sorts low.
     *
     * @param s1
     * @param s2
     * @return the string which sorts higher.
     */
    public static String max(String s1, String s2) {
        if (s1 == null)
            return s2;
        if (s2 == null)
            return s1;
        return s1.compareTo(s2) >= 0? s1 : s2;
    }

    /**
     * This returns the string which sorts lower.
     * null sorts low.
     *
     * @param s1
     * @param s2
     * @return the string which sorts lower.
     */
    public static String min(String s1, String s2) {
        if (s1 == null)
            return s1;
        if (s2 == null)
            return s2;
        return s1.compareTo(s2) < 0? s1 : s2;
    }

    /**
     * This makes a new String of specified length, filled with ch.
     * For safety, if length>=1000000, it returns "".
     * 
     * @param ch the character to fill the string
     * @param length the length of the string
     * @return a String 'length' long, filled with ch.
     *    If length < 0 or >= 1000000, this returns "".
     */
    public static String makeString(char ch, int length) {
        if ((length < 0) || (length >= 1000000))
            return "";

        char[] car = new char[length];
        java.util.Arrays.fill(car, ch);
        return new String(car);
    }

    /**
     * Returns a String 'length' long, with 's' right-justified  
     * (using spaces as the added characters) within the resulting String.
     * If s is already longer, then there will be no change.
     * 
     * @param s is the string to be right-justified.
     * @param length is desired length of the resulting string.
     * @return 's' right-justified to make the result 'length' long.
     */
    public static String right(String s, int length) {
        int toAdd = length - s.length();

        if (toAdd <= 0)
            return s;
        else
            return makeString(' ', toAdd).concat(s);
    }

    /**
     * Returns a String 'length' long, with 's' left-justified  
     * (using spaces as the added characters) within the resulting String.  
     * If s is already longer, then there will be no change.
     * 
     * @param s is the string to be left-justified.
     * @param length is desired length of the resulting string.
     * @return 's' left-justified to make the result 'length' long.
     */
    public static String left(String s, int length) {
        int toAdd = length - s.length();

        if (toAdd <= 0)
            return s;
        else
            return s.concat(makeString(' ', toAdd));
    }

    /**
     * Returns a String 'length' long, with 's' centered  
     * (using spaces as the added characters) within the resulting String.  
     * If s is already longer, then there will be no change.
     * 
     * @param s is the string to be centered.
     * @param length is desired length of the resulting string.
     * @return 's' centered to make the result 'length' long.
     */
    public static String center(String s, int length) {
        int toAdd = length - s.length();

        if (toAdd <= 0)
            return s;
        else
            return makeString(' ', toAdd / 2) + s
            + makeString(' ', toAdd - (toAdd / 2));
    }

    /**
     * This returns a string no more than max characters long, throwing away the excess.
     * If you want to keep the whole string and just insert newlines periodically, 
     * use noLongLines() instead.
     *
     * @param s
     * @param max
     * @return s (if it is short) or the first max characters of s
     */
    public static String noLongerThan(String s, int max) {
        if (s == null)
            return "";
        if (s.length() <= max)  
            return s;
        return s.substring(0, max);
    }

    /**
     * This converts non-isPrintable characters to "[#]".
     * \\n generates both [10] and a newline character.
     *
     * @param s the string
     * @return s, but with non-32..126 characters replaced by [#].
     *    The result ends with "[end]".
     *    null returns "[null][end]".
     */
    public static String annotatedString(String s) {
        if (s == null) 
            return "[null][end]";
        StringBuffer buffer = new StringBuffer();
        int sLength = s.length();

        for (int i = 0; i < sLength; i++) {
            char ch = s.charAt(i);

            if (ch >= 32 && ch <= 126) {
                buffer.append(ch);
            } else {
                buffer.append("[" + ((int) ch) + "]");
                if (ch == '\n') 
                    buffer.append('\n');
            }
        }

        buffer.append("[end]");
        return buffer.toString();
    }


    /**
     * This determines the number of initial characters that match.
     * 
     * @param s1
     * @param s2
     * @return the number of characters that are the same at the start
     *   of both strings.
     */
    public static int getNMatchingCharacters(String s1, String s2) {
        int minLength = Math.min(s1.length(), s2.length());
        for (int i = 0; i < minLength; i++)
            if (s1.charAt(i) != s2.charAt(i))
                return i;
        return minLength;
    }

    /**
     * Finds the first instance of s at or after fromIndex (0.. ) in sb.
     *
     * @param sb a StringBuffer
     * @param s the String you want to find
     * @param fromIndex the index number of the position to start the search
     * @return The starting position of s. If not found, it returns -1.
     */
    public static int indexOf(StringBuffer sb, String s, int fromIndex) {
        int sLength = s.length();
        if (sLength == 0)
            return -1;

        char ch = s.charAt(0);
        int index = Math.max(fromIndex, 0);
        int tSize = sb.length() - sLength + 1; //no point in searching last few char
        while (index < tSize) {
            if (sb.charAt(index) == ch) {
                int nCharsMatched = 1;
                while ((nCharsMatched < sLength)
                        && (sb.charAt(index + nCharsMatched) == s.charAt(nCharsMatched)))
                    nCharsMatched++;
                if (nCharsMatched == sLength)
                    return index;
            }

            index++;
        }

        return -1;
    }

    /**
     * This returns the first section of s (starting at fromIndex) 
     * which matches regex.
     *
     * @param s the source String
     * @param regex the regular expression, see java.util.regex.Pattern.
     * @param fromIndex the starting index in s
     * @return the section of s which matches regex, or null if not found
     * @throws Exception if trouble
     */
    public static String extractRegex(String s, String regex, int fromIndex) {
        Pattern p = Pattern.compile(regex);
        Matcher m = p.matcher(s);
        if (m.find(fromIndex)) 
            return s.substring(m.start(), m.end());
        return null; 
    }

    /**
     * This returns all the sections of s that match regex.
     * It assumes that the extracted parts don't overlap.
     *
     * @param s the source String
     * @param regex the regular expression, see java.util.regex.Pattern.
     *    Note that you often want to use the "reluctant" qualifiers
     *    which match as few chars as possible (e.g., ??, *?, +?)
     *    not the "greedy"  qualifiers
     *    which match as many chars as possible (e.g., ?, *, +).
     * @return a String[] with all the matching sections of s (or String[0] if none)
     * @throws Exception if trouble
     */
    public static String[] extractAllRegexes(String s, String regex) {
        ArrayList al = new ArrayList();
        Pattern p = Pattern.compile(regex);
        Matcher m = p.matcher(s);
        int fromIndex = 0;
        while (m.find(fromIndex)) {
            al.add(s.substring(m.start(), m.end()));
            fromIndex = m.end();
        }
        return toStringArray(al.toArray());
    }

    /**
     * Finds the first instance of i at or after fromIndex (0.. ) in iArray.
     *
     * @param iArray
     * @param i the int you want to find
     * @param fromIndex the index number of the position to start the search
     * @return The first instance of i. If not found, it returns -1.
     */
    public static int indexOf(int[] iArray, int i, int fromIndex) {
        int iArrayLength = iArray.length;
        if (iArrayLength == 0)
            return -1;

        int index = Math.max(fromIndex, 0);
        while (index < iArrayLength) {
            if (iArray[index] == i) 
                return index;
            index++;
        }

        return -1;
    }

    /**
     * Finds the first instance of i in iArray.
     *
     * @param iArray
     * @param i the int you want to find
     * @return The first instance of i. If not found, it returns -1.
     */
    public static int indexOf(int[] iArray, int i) {
        return indexOf(iArray, i, 0);
    }

    /**
     * Finds the first instance of d at or after fromIndex (0.. ) in dArray
     * (tested with Math2.almostEqual5).
     *
     * @param dArray
     * @param d the double you want to find
     * @param fromIndex the index number of the position to start the search
     * @return The first instance of d. If not found, it returns -1.
     */
    public static int indexOf(double[] dArray, double d, int fromIndex) {
        int dArrayLength = dArray.length;
        if (dArrayLength == 0)
            return -1;

        int index = Math.max(fromIndex, 0);
        while (index < dArrayLength) {
            if (Math2.almostEqual(5, dArray[index], d)) 
                return index;
            index++;
        }

        return -1;
    }

    /**
     * Finds the first instance of d in dArray
     * (tested with Math2.almostEqual5).
     *
     * @param dArray
     * @param d the double you want to find
     * @return The first instance of d. If not found, it returns -1.
     */
    public static int indexOf(double[] dArray, double d) {
        return indexOf(dArray, d, 0);
    }

    /**
     * This reads the text contents of the specified file.
     * This assumes the file uses the default character encoding.
     * 
     * <P>This method uses try/catch to ensure that all possible
     * exceptions are caught and returned as the error String
     * (throwable.toString()).
     * 
     * <P>This method is generally appropriate for small and medium-sized
     * files. For very large files or files that need additional processing,
     * it may be more efficient to write a custom method to
     * read the file line-by-line, processing as it goes.
     *
     * @param fileName is the (usually canonical) path (dir+name) for the file
     * @return a String array with two strings.
     *     Using a String array gets around Java's limitation of
     *         only returning one value from a method.
     *     String #0 is an error String (or "" if no error).
     *     String #1 has the contents of the file
     *         (with any end-of-line characters converted to \n).
     *     If the error String is not "", String #1
     *         may not have all the contents of the file.
     *     ***This ensures that the last character in the file (if any) is \n.
     *     This behavior varies from other implementations of readFromFile.
     */
    public static String[] readFromFile(String fileName) {

        //declare the BufferedReader variable
        //declare the results variable: String results[] = {"", ""}; 
        //BufferedReader and results are declared outside try/catch so 
        //that they can be accessed from within either try/catch block.
        BufferedReader bufferedReader = null;
        String results[] = {"", ""};
        int errorIndex = 0;
        int contentsIndex = 1;

        try {
            //open the file
            //This uses a bufferedReader wrapped around a FileReader.
            //To deal with problems in multithreaded apps 
            //(when deleting and renaming files, for an instant no file with that name exists),
            //try for five seconds if necessary.
            FileReader fr = null;
            int maxAttempt = 3;
            for (int attempt = 1; attempt <= maxAttempt; attempt++) {
                try {
                    fr = new FileReader(fileName);
                } catch (Exception e) {
                    if (attempt == maxAttempt) {
                        log(ERROR + ": String2.readFromFile was unable to read " + fileName);
                        throw e;
                    } else {
                        log("WARNING #" + attempt + 
                            ": String2.readFromFile is having trouble. It will try again to read " + 
                            fileName);
                        if (attempt == 1) Math2.gc(1000);
                        else Math2.sleep(1000);
                    }
                }
            }
            bufferedReader = new BufferedReader(fr);
                         
            //get the text from the file
            //This uses bufferedReader.readLine() to repeatedly
            //read lines from the file and thus can handle various 
            //end-of-line characters.
            //The lines (with \n added at the end) are added to a 
            //stringBuffer.
            StringBuffer sb = new StringBuffer();
            String s = bufferedReader.readLine();
            while (s != null) { //null = end-of-file
                sb.append(s);
                sb.append('\n');
                s = bufferedReader.readLine();
                }

            //save the contents as results[1]
            results[contentsIndex] = sb.toString();

        } catch (Exception e) {
            results[errorIndex] = MustBe.throwable("fileName=" + fileName, e);
        }

        //close the bufferedReader
        try {
            if (bufferedReader != null) 
                bufferedReader.close();
        } catch (Exception e) {
            if (results[errorIndex].length() == 0)
                results[errorIndex] = e.toString(); 
            //else ignore the error (the first one is more important)
        }

        //return results
        return results;
    }

    /*
    Here is a skeleton for more direct control of reading text from a file: 
        try {
            BufferedReader bufferedReader = 
                new BufferedReader(new FileReader(fileName));                      
            String s;
            while ((s = bufferedReader.readLine()) != null) { //null = end-of-file
                //do something with s
                //for example, split at whitespace: String fields[] = s.split("\\s+"); //s = whitespace regex
                }

            bufferedReader.close();
        } catch (Exception e) {
            System.err.println(error + "while reading file '" + filename + "':\n" + e);
            e.printStackTrace(System.err);
        }
    */

    /**
     * This saves some text in a file named fileName.
     * This uses the default character encoding.
     * 
     * <P>This method uses try/catch to ensure that all possible
     * exceptions are caught and returned as the error String
     * (throwable.toString()).
     *
     * <P>This method is generally appropriate for small and medium-sized
     * files. For very large files or files that need additional processing,
     * it may be more efficient to write a custom method to
     * read the file line-by-line, processing as it goes.
     *
     * @param fileName is the (usually canonical) path (dir+name) for the file
     * @param contents has the text that will be written to the file.
     *     contents must use \n as the end-of-line marker.
     *     Currently, this method purposely does not convert \n to the 
     *     operating-system-appropriate end-of-line characters when writing 
     *     to the file (see lineSeparator).
     * @return an error message (or "" if no error).
     */
    public static String writeToFile(String fileName, String contents) {
        return lowWriteToFile(fileName, contents, "\n", false);
    }

    /**
     * This is like writeToFile, but it appends the text if the file already 
     * exists. If the file doesn't exist, it makes a new file. 
     */
    public static String appendFile(String fileName, String contents) {
        return lowWriteToFile(fileName, contents, "\n", true);
    }

    /**
     * This provides servies to writeToFile and appendFile. 
     * If ther is an error and !append, the partial file is deleted.
     *
     * @param fileName is the (usually canonical) path (dir+name) for the file
     * @param contents has the text that will be written to the file.
     *     contents must use \n as the end-of-line marker.
     *     Currently, this method purposely does not convert \n to the 
     *     operating-system-appropriate end-of-line characters when writing 
     *     to the file (see lineSeparator).
     * @param lineSeparator is the desired lineSeparator for the outgoing file.
     * @param append if you want to append any existing fileName;
     *   otherwise any existing file is deleted first.
     * @return an error message (or "" if no error).
     */
    private static String lowWriteToFile(String fileName, String contents, 
        String lineSeparator, boolean append) {
        
        //bufferedWriter and error are declared outside try/catch so 
        //that they can be accessed from within either try/catch block.
        BufferedWriter bufferedWriter = null;
        String error = "";

        try {
            //open the file
            //This uses a BufferedWriter wrapped around a FileWriter
            //to write the information to the file.
            bufferedWriter = new BufferedWriter(new FileWriter(fileName, append));
                         
            //convert \n to operating-system-specific lineSeparator
            if (!lineSeparator.equals("\n"))
                contents = replaceAll(contents, "\n", lineSeparator);
                //since the first String is a regex, you can use "[\\n]" too

            //write the text to the file
            bufferedWriter.write(contents);

            //test speed
            //int start = 0;
            //while (start < contents.length()) {
            //    bufferedWriter.write(contents.substring(start, Math.min(start+39, contents.length())));
            //    start += 39;
            //}

        } catch (Exception e) {
            error = e.toString();
        }

        //make sure bufferedWriter is closed
        try {
            if (bufferedWriter != null) {
                bufferedWriter.close();

            }
        } catch (Exception e) {
            if (error.length() == 0)
                error = e.toString(); 
            //else ignore the error (the first one is more important)
        }

        //and delete partial file if error and not appending
        if (error.length() > 0 && !append)
            File2.delete(fileName);

        return error;
    }

    /**
     * A string of Java info (version, vendor).
     */
    public static String javaInfo() {
        String javaVersion = System.getProperty("java.version");
        String mrjVersion = System.getProperty("mrj.version");
        mrjVersion = (mrjVersion == null) ? "" : (" (mrj=" + mrjVersion + ")");
        return "Java " + javaVersion + mrjVersion + " ("
            + System.getProperty("java.vendor") + ") on "
            + System.getProperty("os.name") + " ("
            + System.getProperty("os.version")
            + ").";
    }

    /**
     * This includes hiASCII/ISO Latin 1 but not extensive unicode characters.
     * Letters are A..Z, a..z, and #192..#255 (except #215 and #247).
     * For unicode characters, see Java Lang Spec pg 14.
     *
     * @param c a char
     * @return true if c is a letter
     */
    public static final boolean isLetter(int c) {
        //return (((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))
        //|| ((c >= '\u00c0') && (c <= '\u00FF') && (c != '\u00d7')
        //&& (c != '\u00f7')));
        if (c <  'A') return false;
        if (c <= 'Z') return true;
        if (c <  'a') return false;
        if (c <= 'z') return true;
        if (c <  '\u00c0') return false;
        if (c == '\u00d7') return false;
        if (c <= '\u00FF') return true;
        return false;
    }

    /**
     * First letters for identifiers (e.g., variable names, method names) can be
     * all isLetter()'s plus $ and _.
     *
     * @param c a char
     * @return true if c is a valid character for the first character if a Java ID
     */
    public static final boolean isIDFirstLetter(int c) {
        if (c == '_') return true;
        if (c == '$') return true;
        return isLetter(c);
    }

    /**
     * 0..9, a..f, A..F
     * Hex numbers are 0x followed by hexDigits.
     *
     * @param c a char
     * @return true if c is a valid hex digit
     */
    public static final boolean isHexDigit(int c) {
        //return (((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'f'))
        //|| ((c >= 'A') && (c <= 'F')));
        if (c <  '0') return false;
        if (c <= '9') return true;
        if (c <  'A') return false;
        if (c <= 'F') return true;
        if (c <  'a') return false;
        if (c <= 'f') return true;
        return false;
    }

    /**
     * 0..9.
     * Non-Latin numeric characters are not included (see Java Lang Spec pg 14).
     *
     * @param c a char
     * @return true if c is a digit
     */
    public static final boolean isDigit(int c) {
        return ((c >= '0') && (c <= '9'));
    }

    /**
     * Determines if the character is a digit or a letter.
     *
     * @param c a char
     * @return true if c is a letter or a digit
     */
    public static final boolean isDigitLetter(int c) {
        return isLetter(c) || isDigit(c);
    }

    /**
     * Whitespace characters are u0001 .. ' '.
     * Java just considers a few of these (sp HT FF) as white space,
     *  see the Java Lang Specification.
     * u0000 is not whitespace.  Some methods count on this fact.
     *
     * @param c a char
     * @return true if c is a whitespace character
     */
    public static final boolean isWhite(int c) {
        return (c >= '\u0001') && (c <= ' ');
    }

    /**
     * This indicates if ch is printable with System.err.println() and
     *   Graphics.drawString(); hence, it is a subset of 0..255.
     * <UL>
     * <LI> This is used, for example, to limit characters entering CoText.
     * <LI> Currently, this accepts the ch if
     *   <TT>(ch>=32 && ch<127) || (ch>=161 && ch<=255)</TT>.
     * <LI> tab(#9) is not included.  It should be caught separately
     *   and dealt with (expand to spaces?).  The problem is that
     *   tabs are printed with a wide box (non-character symbol)
     *   in Windows Courier font.
     *   Thus, they mess up the positioning of characters in CoText.
     * <LI> newline is not included.  It should be caught separately
     *   and dealt with.
     * <LI> This requires further study into all standard fonts on all
     *   platforms to see if other characters can be accepted.
     * </UL>
     *
     * @param ch a char
     * @return true if ch is a printable character
     */
    public static final boolean isPrintable(int ch) {
        //return (ch>=32 && ch<127) || (ch>=161 && ch<=255);  //was 160
        if (ch <   32) return false;
        if (ch <= 126) return true;  //was 127 
        if (ch <  161) return false; //was 160
        if (ch <= 255) return true;
        return false;
    }

    /**
     * This returns the string with all non-isPrintable characters removed.
     *
     * @param s
     * @return s with all the non-isPrintable characters removed
     */
    public static String justPrintable(String s) {
        StringBuffer sb = new StringBuffer();
        int n = s.length();
        for (int i = 0; i < n; i++) {
            char ch = s.charAt(i);
            if (isPrintable(ch))  
                sb.append(ch);
        }
        return sb.toString();
    }

    /** This crudely converts 160.. 255 to plainASCII characters which look similar. */
    public final static String plainASCII = 
        " !cLoY|% ca<--r-" +
        "0+23'uP.,'o>424?" +
        "AAAAAAACEEEEIIII" +
        "DNOOOOOxOUUUUYpB" +
        "aaaaaaaceeeeiiii" +
        "onooooo:ouuuuypy";

    /**
     * This converts the string to plain ascii (0..127).
     * Diacritics are stripped off high ASCII characters.
     * Some high ASCII characters are crudely converted to look-alike characters.
     * Other characters become spaces.
     * The result will be the same length as s.
     *
     * @param s
     * @return the string converted to plain ascii (0..127).
     */
    public static String modifyToBeASCII(String s) {
        StringBuffer sb = new StringBuffer(s);
        int n = s.length();
        for (int i = 0; i < n; i++) {
            char ch = sb.charAt(i);
            if (ch <= 127) {}
            else if (ch >= 160 && ch <= 255) sb.setCharAt(i, plainASCII.charAt(ch - 160));
            else sb.setCharAt(i, ' ');
        }
        return sb.toString();
    }

    /**
     * A description of file-name-safe characters.
     */
    public final static String fileNameSafeDescription = "(A-Z, a-z, 0-9, _, -, or .)";

    /**
     * This indicates if ch is a file-name-safe character (A-Z, a-z, 0-9, _, -, or .).
     *
     * @param ch
     * @return true if ch is a file-name-safe character (A-Z, a-z, 0-9, _, -, .).
     */
    public static boolean isFileNameSafe(char ch) {
        //return (ch >= 'A' && ch <= 'Z') ||                
        //       (ch >= 'a' && ch <= 'z') ||                
        //       (ch >= '0' && ch <= '9') ||
        //        ch == '-' || ch == '_' || ch == '.';
        if (ch == '.' || ch == '-') return true;
        if (ch <  '0') return false;
        if (ch <= '9') return true;
        if (ch <  'A') return false;
        if (ch <= 'Z') return true;
        if (ch == '_') return true;
        if (ch <  'a') return false;
        if (ch <= 'z') return true;
        return false;
    }

    /**
     * This indicates if 'email' is a valid email address.
     *
     * @param email a possible email address
     * @return true if 'email' is a valid email address.
     */
    public static boolean isEmailAddress(String email) {
        if (email == null || email.length() == 0)
            return false;

        //regex from http://www.regular-expressions.info/email.html 
        //(with a-z added instead of using case-insensitive regex)
        //(This isn't perfect, but it is probably good enough.)
        return email.matches("[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}");
    }

    /**
     * This indicates if s has just file-name-safe characters (0-9, A-Z, a-z, _, -, .).
     *
     * @param s a string, usually a file name
     * @return true if s has just file-name-safe characters (0-9, A-Z, a-z, _, -, .).
     *    It returns false if s is null or "".
     */
    public static boolean isFileNameSafe(String s) {
        if (s == null || s.length() == 0)
            return false;
        for (int i = 0; i < s.length(); i++)
            if (!isFileNameSafe(s.charAt(i)))
                return false;
        return true;
    }

    /**
     * This returns the string with just file-name-safe characters (0-9, A-Z, a-z, _, -, .).
     * This is different from EDStatic.makeFileNameSafe --
     *   this emphasizes readability, not avoiding losing information.
     * Spaces and '/' are converted to '_'.
     * Non-safe characters are removed.
     * See posix fully portable file names at http://en.wikipedia.org/wiki/Filename .
     * See javadocs for java.net.URLEncoder, which describes valid characters
     *  (but deals with encoding, whereas this method alters or removes).
     * The result may be shorter than s.
     *
     * @param s  If s is null, this returns "_null".
     *    If s is "", this returns "_".
     * @return s with all of the non-fileNameSafe characters removed or changed
     */
    public static String modifyToBeFileNameSafe(String s) {
        if (s == null)
            return "_null";
        int n = s.length();
        if (n == 0)
            return "_";
        s = modifyToBeASCII(s);
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < n; i++) {
            char ch = s.charAt(i);
            if (ch == '_' || ch <= 32 || ch == '/') {
                if (sb.length() > 0 && sb.charAt(sb.length() - 1) != '_')
                    sb.append('_');
            } else if (isFileNameSafe(ch)) {
                sb.append(ch);
            }
            //others are removed
        }

        return sb.toString();
    }

    /**
     * Replaces all occurences of <TT>oldS</TT> in sb with <TT>newS</TT>.
     * If <TT>oldS</TT> occurs inside <TT>newS</TT>, it won't be replaced
     *   recursively (obviously).
     * 
     * @param sb the stringBuffer
     * @param oldS the string to be searched for
     * @param newS the string to replace oldS
     */
    public static void replaceAll(StringBuffer sb, String oldS, String newS) {
        int po = sb.indexOf(oldS);
        int oldSL = oldS.length();
        int newSL = newS.length();
        if (oldSL == 0 || po < 0) return;
        StringBuffer sb2 = new StringBuffer();
        int base = 0;
        while (po >= 0) {
            sb2.append(sb.substring(base, po));
            sb2.append(newS);
            base = po + oldSL;
            po = sb.indexOf(oldS, base);
        }
        sb2.append(sb.substring(base));
        sb.setLength(0);
        sb.append(sb2);
    }

    /**
     * Returns a string where all occurences of <TT>oldS</TT> have
     *   been replaced with <TT>newS</TT>.
     * If <TT>oldS</TT> occurs inside <TT>newS</TT>, it won't be replaced
     *   recursively (obviously).
     *
     * @param s the main string
     * @param oldS the string to be searched for
     * @param newS the string to replace oldS
     * @return a modified version of s, with newS in place of all the olds.
     *   Throws exception if s is null.
     */
    public static String replaceAll(String s, String oldS, String newS) {
        int po = s.indexOf(oldS);
        if (po < 0)
            return s;

        StringBuffer sb = new StringBuffer(s);
        replaceAll(sb, oldS, newS);
        return sb.toString();
    }

    /**
     * Returns a string where all cases of more than one space are 
     * replaced by one space.  The string is also trim'd to remove
     * leading and trailing spaces.
     *
     * @param s 
     * @return s, but with the spaces combined
     *    (or null if s is null)
     */
    public static String combineSpaces(String s) {
        if (s == null)
            return null;
        StringBuffer sb = new StringBuffer(s.trim());
        for (int i = sb.length() - 1; i > 0; i--) {
            //look for a last space
            if (sb.charAt(i) == ' ') {
                //look for its first space
                int first = i;
                while (first > 0 && sb.charAt(first - 1) == ' ')
                    first--;
                if (first < i) {
                    sb.delete(first, i); //leave the i space in place
                    i = first - 1; //we know first-1 isn't a space
                } else i--;  //we know i-1 isn't a space
            }
        }
        return sb.toString();
    }

    /**
     * Returns a string where all occurences of <TT>oldCh</TT> have
     *   been replaced with <TT>newCh</TT>.
     * This doesn't throw exceptions if bad values.
     */
    public static String replaceAll(String s, char oldCh, char newCh) {
        int po = s.indexOf(oldCh);
        if (po < 0)
            return s;

        int sLength = s.length();
        StringBuffer buffer = new StringBuffer(s);

        for (int i = po; i < sLength; i++)
            if (buffer.charAt(i) == oldCh)
                buffer.setCharAt(i, newCh);

        return buffer.toString();
    }

    /**
     * This adds 0's to the left of the string until there are <TT>nDigits</TT>
     *   to the left of the decimal point (or nDigits total if there isn't
     *   a decimal point).
     * If the number is too big, nothing is added or taken away.
     *
     * @param number a positive number.  This doesn't handle negative numbers.
     * @param nDigits the desired number of digits to the left of the decimal point
     * (or total, if no decimal point)
     * @return the number, left-padded with 0's so there are nDigits to 
     * the left of the decimal point
     */
    public static String zeroPad(String number, int nDigits) {
        int decimal = number.indexOf(".");
        if (decimal < 0)
            decimal = number.length();

        int toAdd = nDigits - decimal;
        if (toAdd <= 0)
            return number;

        return makeString('0', toAdd).concat(number);
    }

    /**
     * This makes a JSON version of a string 
     * (\\, \f, \n, \r, \t and \" are escaped with a backslash character
     * and double quotes are added before and after).
     * null is returned as null.
     *
     * @param s
     */
    public static String toJson(String s) {
        if (s == null)
            return "null";
        StringBuffer sb = new StringBuffer("\"");
        int sLength = s.length();
        for (int i = 0; i < sLength; i++) {
            char ch = s.charAt(i);
            if (ch < 32) {
                if      (ch == '\f') sb.append("\\f");
                else if (ch == '\n') sb.append("\\n");
                else if (ch == '\r') sb.append("\\r");
                else if (ch == '\t') sb.append("\\t");
                //else lose it
            } else if (ch == '\\') {sb.append("\\\\");
            } else if (ch == '\"') {sb.append("\\\"");
            } else {sb.append(ch);
            }
        }
        sb.append('\"');
        return sb.toString();
    }

   
    /**
     * This returns the unJSON version of a JSON string 
     * (surrounding "'s (if any) are removed and \\, \f, \n, \r, \t and \" are unescaped).
     * null and "null" are returned as null.
     *
     * @param s
     * @return the decoded string
     */
    public static String fromJson(String s) {
        if (s == null || s.equals("null"))
            return null;
        if (s.length() >= 2 && s.charAt(0) == '"' && s.charAt(s.length() - 1) == '"')
            s = s.substring(1, s.length() - 1);
        StringBuffer sb = new StringBuffer();
        int sLength = s.length();
        int po = 0;
        while (po < sLength) {
            char ch = s.charAt(po);
            if (ch == '\\') {
                if (po == sLength - 1) 
                    po--;  //so reread \ and treat as \\
                po++; 
                ch = s.charAt(po);
                if      (ch == 'f') sb.append('\f');
                else if (ch == 'n') sb.append('\n');
                else if (ch == 'r') sb.append('\r');
                else if (ch == 't') sb.append('\t');
                else if (ch == '\\')sb.append('\\');
                else if (ch == '"') sb.append('\"');
                else { 
                    //this shouldn't happen
                    //before 2009-02-27, this tossed the \    is this the best solution?
                    sb.append('\\'); 
                    sb.append(ch); 
                } 
            } else {
                sb.append(ch);
            }
            po++;
        }
        return sb.toString();
    }
    
    /**
     * This takes a multi-line string (with \\r, \\n, \\r\\n line separators)
     *   and converts it into an ArrayList strings.
     * <ul>
     * <li> Only isPrintable characters are saved.
     * <li> Each line separator generates another line.  So if last char
     *   is a line separator, it generates a blank line at the end.
     * <li> If s is "", this still generates 1 string ("").
     * </ul>
     *
     * @param s the string with internal line separators
     * @return an arrayList with the separate lines of text
     */
    public static ArrayList multiLineStringToArrayList(String s) {
        char endOfLineChar = s.indexOf('\n') >= 0? '\n' : '\r';
        int sLength = s.length();
        ArrayList arrayList = new ArrayList(); //this is local, so okay if not threadsafe
        StringBuffer oneLine = new StringBuffer();
        char ch;
        for (int po = 0; po < sLength; po++) {
            ch = s.charAt(po);
            if (isPrintable(ch)) 
                oneLine.append(ch);
            //else if (ch=='\t') oneLine.add(' ');
            else if (ch == endOfLineChar) {
                arrayList.add(oneLine.toString());
                oneLine.setLength(0);
            }
          //other characters ignored
          }
        arrayList.add(oneLine.toString());
        return arrayList;
    }

    /**
     * This creates an ArrayList with the objects from the enumeration.
     *
     * @param e an enumeration
     * @return arrayList with the objects from the enumeration
     */
    public static ArrayList toArrayList(Enumeration e) {
        ArrayList al = new ArrayList();
        synchronized (e) {
            while (e.hasMoreElements()) 
                al.add(e.nextElement());
        }
        return al;
    }

    /**
     * This creates an ArrayList from an Object[].
     *
     * @param objectArray an Object[]
     * @return arrayList with the objects
     */
    public static ArrayList toArrayList(Object objectArray[]) {
        int n = objectArray.length;
        ArrayList al = new ArrayList(n);
        for (int i = 0; i < n; i++)
            al.add(objectArray[i]);
        return al;
    }

    /**
     * This returns the standard Help : About message.
     *
     * @return the standard Help : About message
     */
    public static String standardHelpAboutMessage() {
        return
            "This program includes open source com.cohort classes (version " + 
                version + "),\n" +
            "Copyright(c) 2004 - 2007, CoHort Software.\n" +
            "For more information, visit www.cohort.com.\n" +
            "\n" + 
            "This program is using\n" + 
            javaInfo();
    }


    /**
     * This replaces "{0}", "{1}", and "{2}" in msg with s0, s1, s2.
     *
     * @param msg a string which may contain "{0}", "{1}", and/or "{2}".
     * @param s0 the first substitution string. If null, that substitution
     *     won't be attempted.
     * @param s1 the second substitution string. If null, that substitution
     *     won't be attempted.
     * @param s2 the third substitution string. If null, that substitution
     *     won't be attempted.
     * @return the modified msg
     */
    public static String substitute(String msg, String s0, String s1, String s2) {
        StringBuffer msgSB = new StringBuffer(msg);
        if (s0 != null) 
            replaceAll(msgSB, "{0}", s0); 
        if (s1 != null) 
            replaceAll(msgSB, "{1}", s1); 
        if (s2 != null) 
            replaceAll(msgSB, "{2}", s2); 
        return msgSB.toString();
    }

    /**
     * Generates a comma-separated-value string.  
     *
     * @param en an enumeration of objects
     * @return a CSV String with the values with ", " after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toCSVString(Enumeration en) {
        return toSVString(toArrayList(en).toArray(), ", ", false);
    }

    /**
     * Generates a comma-separated-value string.  
     *
     * @param al an arrayList of objects
     * @return a CSV String with the values with ", " after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toCSVString(ArrayList al) {
        return toSVString(al.toArray(), ", ", false);
    }

    /**
     * Generates a comma-separated-value string.  
     *
     * @param vec a vector of objects
     * @return a CSV String with the values with ", " after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toCSVString(Vector v) {
        return toSVString(v.toArray(), ", ", false);
    }

    /**
     * Generates a comma-separated-value string.  
     *
     * @param ar an array of objects
     * @return a CSV String with the values with ", " after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toCSVString(Object ar[]) {
        return toSVString(ar, ", ", false);
    }

    /**
     * Generates a space-separated-value string.  
     *
     * @param ar an array of objects
     *    (for an ArrayList or Vector, use o.toArray())
     * @return a SSV String with the values with " " after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toSSVString(Object ar[]) {
        return toSVString(ar, " ", false);
    }

    /**
     * Generates a tab-separated-value string.  
     *
     * @param ar an array of objects
     *    (for an ArrayList or Vector, use o.toArray())
     * @return a TSV String with the values with "\t" after
     *    all but the last value.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toTSVString(Object ar[]) {
        return toSVString(ar, "\t", false);
    }

    /**
     * Generates a newline-separated string.
     *
     * @param ar an array of objects
     *    (for an ArrayList or Vector, use o.toArray())
     * @return a String with the values, 
     *    with a '\n' after each value, even the last.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toNewlineString(Object ar[]) {
        return toSVString(ar, "\n", true);
    }

    /**
     * This is used at a low level to generate a 
     * 'separator'-separated-value string (without newlines) 
     * with the element.toString()'s from the array.
     *
     * @param ar an array of objects
     *    (for an ArrayList or Vector, use o.toArray())
     * @param separator the separator string
     * @param finalSeparator if true, a separator will be added to the 
     *    end of the resulting string (if it isn't "").
     * @return a separator-separated-value String.
     *    Returns null if ar is null.
     *    null elements are represented as "[null]".
     */
    public static String toSVString(Object ar[], String separator, boolean finalSeparator) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            if (i > 0)
                sb.append(separator);
            Object o = ar[i];
            sb.append(o == null? "[null]": o.toString());
        }
        if (finalSeparator && n > 0)
            sb.append(separator);
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of bytes
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(byte ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            if (i > 0)
                sb.append(", ");
            sb.append(ar[i]);
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     * (chars are treated as unsigned shorts).
     *
     * @param ar an array of char
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(char ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            if (i > 0)
                sb.append(", ");
            sb.append((int)ar[i]);
        }
        return sb.toString();
    }

    /**
     * This generates a hexadecimal CSV String from the array.
     * Negative numbers are twos compliment, e.g., -4 -> 0xfc.
     *
     * @param ar an array of bytes
     * @return a CSV String (or null if ar is null)
     */
    public static String toHexCSVString(byte ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            if (i > 0)
                sb.append(", ");
            String s = Integer.toHexString(ar[i]);
            if (s.length() == 8 && s.startsWith("ffffff"))
                s = s.substring(6);
            sb.append("0x" + s);
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of shorts
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(short ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a hexadecimal CSV String from the array.
     * Negative numbers are twos compliment, e.g., -4 -> 0xfffc.
     *
     * @param ar an array of short
     * @return a CSV String (or null if ar is null)
     */
    public static String toHexCSVString(short ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            String s = Integer.toHexString(ar[i]);
            if (s.length() == 8 && s.startsWith("ffff"))
                s = s.substring(4);
            sb.append("0x" + s);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of ints
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(int ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a hexadecimal CSV String from the array.
     * Negative numbers are twos compliment, e.g., -4 -> 0xfffffffc.
     *
     * @param ar an array of ints
     * @return a CSV String (or null if ar is null)
     */
    public static String toHexCSVString(int ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append("0x" + Integer.toHexString(ar[i]));
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of longs
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(long ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of float
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(float ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a CSV String from the array.
     *
     * @param ar an array of double
     * @return a CSV String (or null if ar is null)
     */
    public static String toCSVString(double ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            if (i < n - 1)
                sb.append(", ");
        }
        return sb.toString();
    }

    /**
     * This generates a newline-separated (always '\n') String from the array.
     *
     * @param ar an array of ints
     * @return a newline-separated String (or null if ar is null)
     */
    public static String toNewlineString(int ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            sb.append('\n');
        }
        return sb.toString();
    }

    /**
     * This generates a newline-separated (always '\n') String from the array.
     *
     * @param ar an array of double
     * @return a newline-separated String (or null if ar is null)
     */
    public static String toNewlineString(double ar[]) {
        if (ar == null) 
            return null;
        StringBuffer sb = new StringBuffer();
        int n = ar.length;
        for (int i = 0; i < n; i++) {
            sb.append(ar[i]);
            sb.append('\n');
        }
        return sb.toString();
    }

    /**
     * This converts an ArrayList of Strings into a String[].
     * If you have an ArrayList or a Vector, use arrayList.toArray().
     *
     * @param aa
     * @return the corresponding String[] by calling toString() for each object
     */
    public static String[] toStringArray(Object aa[]) {
        if (aa == null)
            return null;
        int n = aa.length;
        String sa[] = new String[n];
        for (int i = 0; i < n; i++) {
            Object o = aa[i];
            sa[i] = o == null? (String)o : o.toString();
        }
        return sa;
    }

    /**
     * Add the items in the array (if any) to the arrayList.
     *
     * @param arrayList
     * @param ar the items to be added
     */
    public static void add(ArrayList arrayList, Object ar[]) {
        if (arrayList == null || ar == null) 
            return;
        for (int i = 0; i < ar.length; i++)
            arrayList.add(ar[i]);
    }

    /**
     * This displays the contents of a bitSet as a String.
     *
     * @param bitSet
     * @return the corresponding String (the 'true' bits, comma separated)
     */
    public static String toString(BitSet bitSet) {
        if (bitSet == null)
            return null;
        StringBuffer sb = new StringBuffer();

        String separator = "";
        int i = bitSet.nextSetBit(0);
        while (i >= 0) {
            sb.append(separator + i);
            separator = ", ";
            i = bitSet.nextSetBit(i + 1);
        }
        return sb.toString();
    }

    /**
     * This displays the contents of a map as a String.
     *
     * @param map
     * @return the corresponding String, with one entry on each line 
     *    (<key> = <value>)
     */
    public static String toString(Map map) {
        if (map == null)
            return null;
        StringBuffer sb = new StringBuffer();

        //synchronize so protected from changes in other threads
        synchronized (map) { 
            Set entrySet = map.entrySet();
            Iterator it = entrySet.iterator();
            while (it.hasNext()) {
                Map.Entry me = (Map.Entry)it.next();
                sb.append(me.getKey() + " = " + me.getValue() + "\n");
            }
        }
        return sb.toString();
    }

    /**
     * From an arrayList which alternates attributeName (a String) and 
     * attributeValue (an object), this generates a String with 
     * "    <name>=<value>" on each line.
     * If arrayList == null, this returns "    [null]\n".
     *
     * @param arrayList 
     * @return the desired string representation
     */
    public static String alternateToString(ArrayList arrayList) {
        if (arrayList == null)
            return "    [null]\n";
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < arrayList.size(); i += 2) {
            sb.append("    ");
            sb.append(arrayList.get(i).toString());
            sb.append('=');
            sb.append(arrayToCSVString(arrayList.get(i+1)));
            sb.append('\n');
        }
        return sb.toString();
    }


    /**
     * From an arrayList which alternates attributeName (a String) and 
     * attributeValue (an object), this an array of attributeNames.
     * If arrayList == null, this returns "    [null]\n".
     *
     * @param arrayList 
     * @return the attributeNames in the arrayList
     */
    public static String[] alternateGetNames(ArrayList arrayList) {
        if (arrayList == null)
            return null;
        int n = arrayList.size();
        String[] sar = new String[n / 2];
        int i2 = 0;
        for (int i = 0; i < n / 2; i++) {
            sar[i] = arrayList.get(i2).toString();
            i2 += 2;
        }
        return sar;
    }
    /**
     * From an arrayList which alternates attributeName (a String) and 
     * attributeValue (an object), this returns the attributeValue
     * associated with the supplied attributeName.
     * If array == null or there is no matching value, this returns null.
     *
     * @param arrayList 
     * @param attributeName
     * @return the associated value
     */
    public static Object alternateGetValue(ArrayList arrayList, String attributeName) {
        if (arrayList == null)
            return null;
        for (int i = 0; i < arrayList.size(); i += 2) {
            if (arrayList.get(i).toString().equals(attributeName))
                return arrayList.get(i + 1);
        }
        return null;
    }

    /**
     * Given an arrayList which alternates attributeName (a String) and 
     * attributeValue (an object), this either removes the attribute
     * (if value == null), adds the attribute and value (if it isn't in the list),
     * or changes the value (if the attriubte is in the list).
     *
     * @param arrayList 
     * @param attributeName
     * @param value the value associated with the attributeName
     * @return the previous value for the attribute (or null)
     * @throws Exception of trouble (e.g., if arrayList is null)
     */
    public static Object alternateSetValue(ArrayList arrayList, 
            String attributeName, Object value) {
        Test.ensureNotNull(arrayList, ERROR + " in String2.alternateSetValue:\narrayList is null.");
        for (int i = 0; i < arrayList.size(); i += 2) {
            if (arrayList.get(i).toString().equals(attributeName)) {
                Object oldValue = arrayList.get(i + 1);
                if (value == null) {
                    arrayList.remove(i + 1); //order of removal is important
                    arrayList.remove(i);
                }
                else arrayList.set(i+1, value);
                return oldValue;
            }
        }

        //attributeName not found? 
        if (value == null)
            return null;
        else {
            //add it
            arrayList.add(attributeName);
            arrayList.add(value);
            return null;
        }
    }

    /**
     * This returns a nice String representation of the attribute value
     * (which should be a String or an array of primitives).
     *
     * @param value
     * @return a nice String representation
     */
    public static String arrayToCSVString(Object value) {
        if (value instanceof byte[]) return String2.toCSVString((byte[])value);
        if (value instanceof char[]) return String2.toCSVString((char[])value);
        if (value instanceof short[]) return String2.toCSVString((short[])value);
        if (value instanceof int[]) return String2.toCSVString((int[])value);
        if (value instanceof long[]) return String2.toCSVString((long[])value);
        if (value instanceof float[]) return String2.toCSVString((float[])value);
        if (value instanceof double[]) return String2.toCSVString((double[])value);
        return value.toString();
    }

    /**
     * This extracts the lower 8 bits of each char to form a
     * byte array.
     *
     * @param s a String
     * @return the corresponding byte[] (or null if s is null)
     */
    public static byte[] toByteArray(String s) {
        if (s == null)
            return null;
        byte[] ba = new byte[s.length()];
        for (int i = 0; i < s.length(); i++)
            ba[i] = (byte)s.charAt(i);
        return ba;
    }

    /**
     * This extracts the lower 8 bits of each char to form a
     * byte array.
     *
     * @param sb a StringBuffer
     * @return the corresponding byte[] (or null if s is null)
     */
    public static byte[] toByteArray(StringBuffer sb) {
        if (sb == null)
            return null;
        byte[] ba = new byte[sb.length()];
        for (int i = 0; i < sb.length(); i++)
            ba[i] = (byte)sb.charAt(i);
        return ba;
    }

    /**
     * This creates a String which displays the bytes in hex, 16 per line.
     *
     * @param byteArray   perhaps from toByteArray(s)
     * @return the hex dump of the bytes (or null if byteArray is null).
     *    Each line will be 71 chars long (char#71 will be newline).
     */
    public static String hexDump(byte[] byteArray) {
        StringBuffer printable = new StringBuffer();
        StringBuffer sb = new StringBuffer();
        int i;
        for (i = 0; i < byteArray.length; i++) {
            int data = byteArray[i] & 255;
            sb.append(zeroPad(Integer.toHexString(data), 2) + " ");
            printable.append(data >= 32 && data <= 126? (char)data: ' '); 
            if (i % 8 == 7) 
                sb.append("  ");
            if (i % 16 == 15) {
                sb.append(printable + " |\n");
                printable.setLength(0);
            }

        }
        if (byteArray.length % 16 != 0) {
            sb.append(printable);
            sb.append(makeString(' ', 69 - sb.length() % 71));
            sb.append("|\n");
        }

        return sb.toString();
    }

    /**
     * This finds the first element in Object[] 
     * where the ar[i].toString value equals to s.
     *
     * @param ar the array of Objects
     * @param s the String to be found
     * @return the element number of ar which is equal to s (or -1 if ar is null, or s is null or not found)
     */
    public static int indexOf(Object[] ar, String s) {
        return indexOf(ar, s, 0);
    }

    /**
     * This finds the first element in Object[]  (starting at element startAt)
     * where the ar[i].toString value equals s.
     *
     * @param ar the array of Objects
     * @param s the String to be found
     * @param startAt the first element of ar to be checked.
     *    If startAt < 0, this starts with startAt = 0.
     *    If startAt >= ar.length, this returns -1.
     * @return the element number of ar which is equal to s (or -1 if ar is null, or s is null or not found)
     */
    public static int indexOf(Object[] ar, String s, int startAt) {
        if (ar == null || s == null)
            return -1;
        int n = ar.length;
        for (int i = Math.max(0, startAt); i < n; i++)
            if (ar[i] != null && s.equals(ar[i].toString()))  
                return i;
        return -1;
    }

    /**
     * This finds the first element in Object[] 
     * where ar[i].toString().toLowerCase() equals to s.toLowerCase().
     *
     * @param ar the array of Objects
     * @param s the String to be found
     * @return the element number of ar which is equal to s (or -1 if s is null or not found)
     */
    public static int caseInsensitiveIndexOf(Object[] ar, String s) {
        if (ar == null || s == null)
            return -1;
        int n = ar.length;
        s = s.toLowerCase();
        for (int i = 0; i < n; i++)
            if (ar[i] != null && s.equals(ar[i].toString().toLowerCase()))  
                return i;
        return -1;
    }

    /**
     * This finds the first element in Object[] 
     * where the ar[i].toString value contains the substring s.
     *
     * @param ar the array of objects
     * @param s the String to be found
     * @return the element number of ar which is equal to s (or -1 if not found)
     */
    public static int lineContaining(Object[] ar, String s) {
        return lineContaining(ar, s, 0);
    }

    /**
     * This finds the first element in Object[] (starting at element startAt)
     * where the ar[i].toString value contains the substring s.
     *
     * @param ar the array of objects
     * @param s the String to be found
     * @param startAt the first element of ar to be checked.
     *    If startAt < 0, this starts with startAt = 0.
     * @return the element number of ar which is equal to s (or -1 if not found)
     */
    public static int lineContaining(Object[] ar, String s, int startAt) {
        if (ar == null || s == null)
            return -1;
        int n = ar.length;
        for (int i = Math.max(0, startAt); i < n; i++)
            if (ar[i] != null && ar[i].toString().indexOf(s) >= 0) 
                return i;
        return -1;
    }

    /**
     * This returns the first element in Object[] (starting at element 0)
     * where the ar[i].toString value starts with s.
     *
     * @param ar the array of objects
     * @param s the String to be found
     * @return the first element ar (as a String) which starts with s (or null if not found)
     */
    public static String stringStartsWith(Object[] ar, String s) {
        int i = lineStartsWith(ar, s, 0);
        return i < 0? null : ar[i].toString();
    }

    /**
     * This finds the first element in Object[] (starting at element 0)
     * where the ar[i].toString value starts with s.
     *
     * @param ar the array of objects
     * @param s the String to be found
     * @return the element number of ar which starts with s (or -1 if not found)
     */
    public static int lineStartsWith(Object[] ar, String s) {
        return lineStartsWith(ar, s, 0);
    }

    /**
     * This finds the first element in Object[] (starting at element startAt)
     * where the ar[i].toString value starts with s.
     *
     * @param ar the array of objects
     * @param s the String to be found
     * @param startAt the first element of ar to be checked.
     *    If startAt < 0, this starts with startAt = 0.
     * @return the element number of ar which is equal to s (or -1 if not found)
     */
    public static int lineStartsWith(Object[] ar, String s, int startAt) {
        if (ar == null || s == null)
            return -1;
        int n = ar.length;
        for (int i = Math.max(0, startAt); i < n; i++)
            if (ar[i] != null && ar[i].toString().startsWith(s)) 
                return i;
        return -1;
    }

    /* *  NOT ACTIVE.     
     * This replaces the current log.
     * The default log prints to System.err.
     * Use the logger by calling, e.g., String2.log.fine(msg) or
     *   String2.log.log(level, msg);
     *
     * @param mainClassName e.g., "gov.noaa.pfel.coastwatch.CWDataBrowser", 
     *   sets up a separate logger related to the current program
     * @param fullFileName the name for the log file (or null or "" for System.err).
     *   Append .%g to the end of the file name to create and rotate through a series of 
     *   up to 10 log files, each with up to 1MB of messages.
     * @param append determines whether log info should be appended
     *   or overwrite previous log files.
     * @param defaultLevel e.g., java.util.logging.Level.FINER.
     *    Note that this can be changed any time with String2.log.setLevel(level).
     */
    /*public static void setUpLog(String mainClassName, String fullFileName,
            boolean append, Level initialLevel) throws Exception {
        log = Logger.getLogger(mainClassName);
        if (fullFileName.length() > 0) {
            Handler har[] = log.getHandlers();
            for (int i = 0; i < har.length; i++)
                log.removeHandler(har[i]);
            FileHandler fh = fullFileName.endsWith("%g")? 
                new FileHandler(fullFileName, 1000000, 10, append) :
                new FileHandler(fullFileName, 1000000, 1,  append);
            fh.setFormatter(new RawFormatter()); //replace the default XMLFormatter
            log.addHandler(fh);
        }
        log.setLevel(initialLevel);
    }*/

    /**
     * This tells Commons Logging to use com.cohort.util.String2Log.
     * I use this at the beginning of my programs 
     * (TestAll, NetCheck, Browser, ConvertTable, DoubleCenterGrids)
     * to route Commons Logging requests through String2Log.
     * !!!Don't use this in lower level methods as it will hijack the 
     * parent program's (e.g., Armstrong's) logger setup.
     *
     * param level a String2Log.XXX_LEVEL constant (or -1 to leave unchanged,
     *     default=String2Log.WARN_LEVEL)
     */
    public static void setupCommonsLogging(int level) {
        //By setting this property, I specify that String2LogFactory
        //  will be used to generate logFactories.
        //  (It makes one String2Log, which sends all messages to String2.log.)
        System.setProperty("org.apache.commons.logging.LogFactory", 
            "com.cohort.util.String2LogFactory");
        if (level >= 0) {
            System.setProperty("com.cohort.util.String2Log.level", "" + level);
        } else {
            if (System.getProperty("com.cohort.util.String2Log.level") == null)
                System.setProperty("com.cohort.util.String2Log.level", 
                    "" + String2Log.WARN_LEVEL);
        }

        //this dummy variable ensures String2LogFactory gets compiled 
        String2LogFactory string2LogFactory;

    }

    /**
     * This changes the log system set up.
     * The default log prints to System.err.
     * Use the logger by calling String2.log(msg); 
     *
     * @param tLogToSystemOut indicates if info should be printed to System.out (default = false).
     * @param tLogToSystemErr indicates if info should be printed to System.err (default = true).
     * @param fullFileName the name for the log file (or "" for none).
     * @param logToStringBuffer specifies if the logged info should also
     *   be saved in a stringBuffer (see getLogStringBuffer)
     * @param append If a previous log file of the same name exists,
     *   and/or if a logStringBuffer exists,
     *   this determines whether a new log file should be created
     *   or whether info should be appended to the old file.
     * @param maxSize determines the approximate max size of the log file
     *   and/or logStringBuffer.
     *   When maxSize is reached, the current log file is copied to 
     *   fullFileName.previous, and a new fullFileName is created.
     *   When maxSize is reached, the first half of the
     *   logStringBuffer is deleted.
     *   Specify 0 for no limit to the size.    
     */
    public static synchronized void setupLog(
            boolean tLogToSystemOut, boolean tLogToSystemErr,
            String fullFileName, boolean logToStringBuffer, 
            boolean append, int maxSize) 
            throws Exception {

        if (logFileName != null && logFileName.equals(fullFileName))
            return;

        logToSystemOut = tLogToSystemOut;
        logToSystemErr = tLogToSystemErr;

        if (!append)
            logFileSize = 0;
        logMaxSize = maxSize;

        //close the old file
        closeLogFile();

        //stringBufferToo?
        if (logToStringBuffer) {
            if (append && logStringBuffer != null) {
                //use existing logStringBuffer
            } else logStringBuffer = new StringBuffer();
        } else logStringBuffer = null;

        //if no file name, return
        if (fullFileName.length() == 0)
            return;

        //open the file
        //This uses a BufferedWriter wrapped around a FileWriter
        //to write the information to the file.
        logFileName = fullFileName;
        logFile = new BufferedWriter(new FileWriter(fullFileName, append));
        logFileSize = (new File(fullFileName)).length();
    }

    /**
     * This closes the log file (if it exists and is open).
     * It is best if a crashing program calls this.
     * It seems like Java should handle this if program crashes,
     *   but it doesn't seem to (at least in Windows XP Pro).
     */
    public static void closeLogFile() {
        if (logFile != null) {
            try {
                logFile.close();
                logFile = null;
                logFileName = null;
            } catch (Exception e) {
                //do nothing
            }
        }
    }

    /**
     * This writes the specified message (with \n as line separator) to the log file,
     * appending \n at the end.
     * This will not throw an exception.
     *
     * @param message the message
     */
    public static synchronized void log(String message) {
        logNoNewline(message + "\n");
    }

    /**
     * This writes the specified message (with \n as line separator) to the log file
     * without appending \n at the end.
     * This will not throw an exception.
     *
     * @param message the message
     */
    public static synchronized void logNoNewline(String message) {
        try {
            //print message with \n's to logStringBuffer
            if (logStringBuffer != null) {
                if (logStringBuffer.length() > logMaxSize && logMaxSize > 0)
                    logStringBuffer.delete(0, logMaxSize / 2);
                logStringBuffer.append(message);
            }

            //write to system.out or logFile 
            if (!lineSeparator.equals("\n"))
                message = replaceAll(message, "\n", lineSeparator);
            if (logToSystemOut)
                System.out.print(message);
            if (logToSystemErr)
                System.err.print(message);
            if (logFile != null) {
                //is file too big?
                if (logFileSize > logMaxSize && logMaxSize > 0) {
                    logFile.close();
                    File2.rename(logFileName, logFileName + ".previous");
                    logFile = new BufferedWriter(new FileWriter(logFileName));
                    logFileSize = 0;
                }

                //write the message to the file
                logFile.write(message);
                logFile.flush(); //so file is always up-to-date if trouble
                logFileSize += message.length(); 

            }
        } catch (Exception e) {
            //eek! what should I do?
        }
    }

    /**
     * This returns the logStringBuffer object.
     * Then you can get call sb.toString() to get the contents,
     * setLength(0) to clear it, etc.
     *
     * @return the logStringBuffer (or null if none)
     */
    public static synchronized StringBuffer getLogStringBuffer() {
        return logStringBuffer;
    }

    /**
     * This splits the string at the specified character.
     * Leading and trailing whitespace is removed.
     * A missing final strings is treated as "" (not discarded as with String.split).
     * 
     * @param s a string with 0 or more separator chatacters
     * @param separator
     * @return an ArrayList of strings.
     *   s=null returns null.
     *   s="" returns ArrayList with one value: "".
     */
    public static ArrayList splitToArrayList(String s, char separator) {
        return splitToArrayList(s, separator, true);
    }

    /**
     * This splits the string at the specified character.
     * A missing final strings is treated as "" (not discarded as with String.split).
     * 
     * @param s a string with 0 or more separator chatacters
     * @param separator
     * @return an ArrayList of strings.
     *   s=null returns null.
     *   s="" returns ArrayList with one value: "".
     */
    public static ArrayList splitToArrayList(String s, char separator, boolean trim) {
        if (s == null) 
            return null;

        //go through the string looking for separators
        ArrayList al = new ArrayList();
        int sLength = s.length();
        int start = 0;
        //log("split line=" + annotatedString(s));
        for (int index = 0; index < sLength; index++) {
            if (s.charAt(index) == separator) {
                String ts = s.substring(start, index);
                if (trim) ts = ts.trim();
                al.add(ts);
                start = index + 1;
            }
        }

        //add the final substring
        String ts = s.substring(start, sLength); //start == sLength? "" : s.substring(start, sLength);
        if (trim) ts = ts.trim();
        al.add(ts);
        //log("al.size=" + al.size() + "\n");
        return al;
    }

    /**
     * This splits the string at the specified character.
     * Leading and trailing spaces are removed.
     * A missing final string is treated as "" (not discarded as with String.split).
     * 
     * @param s a string with 0 or more separator chatacters
     * @param separator
     * @return a String[] with the strings.
     *   s=null returns null.
     *   s="" returns String[1]{""}.
     */
    public static String[] split(String s, char separator) {
        ArrayList al = splitToArrayList(s, separator, true);
        if (al == null)
            return null;
        return toStringArray(al.toArray());
    }

    /**
     * This splits the string at the specified character.
     * A missing final string is treated as "" (not discarded as with String.split).
     * 
     * @param s a string with 0 or more separator chatacters
     * @param separator
     * @return a String[] with the strings.
     *   s=null returns null.
     *   s="" returns String[1]{""}.
     */
    public static String[] splitNoTrim(String s, char separator) {
        ArrayList al = splitToArrayList(s, separator, false);
        if (al == null)
            return null;
        return toStringArray(al.toArray());
    }

    /**
     * This converts an Object[] (for example, where objects are Strings or 
     *  Integers) into an int[].
     *
     * @param oar an Object[]
     * @return the corresponding int[]  (invalid values are converted to Integer.MAX_VALUE).
     *   oar=null returns null.
     */
    public static int[] toIntArray(Object oar[]) {
        if (oar == null)
            return null;
        int n = oar.length;
        int ia[] = new int[n];
        for (int i = 0; i < n; i++)
            ia[i] = parseInt(oar[i].toString());
        return ia;
    }

    /**
     * This converts an Object[] (for example, where objects are Strings or 
     *  Floats) into a float[].
     *
     * @param oar an Object[]
     * @return the corresponding float[] (invalid values are converted to Float.NaN).
     *   oar=null returns null.
     */
    public static float[] toFloatArray(Object oar[]) {
        if (oar == null)
            return null;
        int n = oar.length;
        float fa[] = new float[n];
        for (int i = 0; i < n; i++)
            fa[i] = parseFloat(oar[i].toString());
        return fa;
    }

    /**
     * This converts an Object[] (for example, where objects are Strings or 
     *  Doubles) into a double[].
     *
     * @param oar an Object[]
     * @return the corresponding double[] (invalid values are converted to Double.NaN).
     *   oar=null returns null.
     */
    public static double[] toDoubleArray(Object oar[]) {
        if (oar == null)
            return null;
        int n = oar.length;
        double da[] = new double[n];
        for (int i = 0; i < n; i++)
            da[i] = parseDouble(oar[i].toString());
        return da;
    }

    /**
     * This converts an ArrayList with Integers into an int[].
     *
     * @param al an Object[]
     * @return the corresponding int[]  (invalid values are converted to Integer.MAX_VALUE).
     *   al=null returns null.
     */
    public static int[] toIntArray(ArrayList al) {
        if (al == null)
            return null;
        int n = al.size();
        int ia[] = new int[n];
        for (int i = 0; i < n; i++)
            ia[i] = ((Integer)al.get(i)).intValue();
        return ia;
    }

    /**
     * This converts an ArrayList with Floats into a float[].
     *
     * @param al an Object[]
     * @return the corresponding float[] (invalid values are converted to Float.NaN).
     *   al=null returns null.
     */
    public static float[] toFloatArray(ArrayList al) {
        if (al == null)
            return null;
        int n = al.size();
        float fa[] = new float[n];
        for (int i = 0; i < n; i++)
            fa[i] = ((Float)al.get(i)).floatValue();
        return fa;
    }

    /**
     * This converts an ArrayList with Doubles into a double[].
     *
     * @param al an Object[]
     * @return the corresponding double[] (invalid values are converted to Double.NaN).
     *   al=null returns null.
     */
    public static double[] toDoubleArray(ArrayList al) {
        if (al == null)
            return null;
        int n = al.size();
        double da[] = new double[n];
        for (int i = 0; i < n; i++)
            da[i] = ((Double)al.get(i)).doubleValue();
        return da;
    }

    /**
     * This returns an int[] with just the non-Integer.MAX_VALUE values from the original 
     * array.
     *
     * @param iar is an int[]
     * @return a new int[] with just the non-Integer.MAX_VALUE values.
     *   iar=null returns null.
     */
    public static int[] justFiniteValues(int iar[]) {
        if (iar == null)
            return null;
        int n = iar.length;
        int nFinite = 0;
        int ia[] = new int[n];
        for (int i = 0; i < n; i++)
            if (iar[i] < Integer.MAX_VALUE)
                ia[nFinite++] = iar[i];

        //copy to a new array
        int iaf[] = new int[nFinite];
        System.arraycopy(ia, 0, iaf, 0, nFinite);
        return iaf;
    }

    /**
     * This returns a double[] with just the finite values from the original 
     * array.
     *
     * @param dar is a double[]
     * @return a new double[] with just finite values.
     *   dar=null returns null.
     */
    public static double[] justFiniteValues(double dar[]) {
        if (dar == null)
            return null;
        int n = dar.length;
        int nFinite = 0;
        double da[] = new double[n];
        for (int i = 0; i < n; i++)
            if (Math2.isFinite(dar[i]))
                da[nFinite++] = dar[i];

        //copy to a new array
        double daf[] = new double[nFinite];
        System.arraycopy(da, 0, daf, 0, nFinite);
        return daf;
    }

    /** 
     * This returns a String[] with just non-null strings
     * from the original array.
     *
     * @param sar is a String[]
     * @return a new String[] with just non-null strings.
     *   sar=null returns null.
     */
    public static String[] removeNull(String sar[]) {
        if (sar == null)
            return null;
        int n = sar.length;
        int nValid = 0;
        String sa[] = new String[n];
        for (int i = 0; i < n; i++)
            if (sar[i] != null)
                sa[nValid++] = sar[i];

        //copy to a new array
        String sa2[] = new String[nValid];
        System.arraycopy(sa, 0, sa2, 0, nValid);
        return sa2;
    }

    /** 
     * This returns a String[] with just non-null and non-"" strings
     * from the original array.
     *
     * @param sar is a String[]
     * @return a new String[] with just non-null and non-"" strings.
     *   sar=null returns null.
     */
    public static String[] removeNullOrEmpty(String sar[]) {
        if (sar == null)
            return null;
        int n = sar.length;
        int nValid = 0;
        String sa[] = new String[n];
        for (int i = 0; i < n; i++)
            if (sar[i] != null && sar[i].length() > 0)
                sa[nValid++] = sar[i];

        //copy to a new array
        String sa2[] = new String[nValid];
        System.arraycopy(sa, 0, sa2, 0, nValid);
        return sa2;
    }

    /**
     * This converts a comma-separated-value String into an int[].
     * Invalid values are converted to Integer.MAX_VALUE.
     *
     * @param csv the comma-separated-value String.
     * @return the corresponding int[].
     *    csv=null returns null.
     *    csv="" is converted to int[1]{Integer.MAX_VALUE}.
     */
    public static int[] csvToIntArray(String csv) {
        return toIntArray(split(csv, ','));      
    }

    /**
     * This converts a comma-separated-value String into a double[].
     * Invalid values are converted to Double.NaN.
     *
     * @param csv the comma-separated-value String
     * @return the corresponding double[].
     *    csv=null returns null.
     *    csv="" is converted to double[1]{Double.NAN}.
     */
    public static double[] csvToDoubleArray(String csv) {
        return toDoubleArray(split(csv, ','));      
    }

    /**
     * This converts a string to a boolean.
     * 
     * @param s the string
     * @return false if s is "false", "f", or "0". Case and leading/trailing
     *   spaces don't matter.  All other values (and null) are treated as true.
     */
    public static boolean parseBoolean(String s) {
        if (s == null)
            return true;
        s = s.toLowerCase().trim();
        return !(s.equals("false") || s.equals("f") || s.equals("0"));
    }

    /**
     * Convert a string to an int.
     * Leading or trailing spaces are automatically removed.
     * This accepts hexadecimal integers starting with "0x".
     * Floating point numbers are rounded.
     * This won't throw an exception if the number isn't formatted right.
     * To make a string from an int, see Integer.toHexString or Integer.toString(i,radix);
     *
     * @param s is the String representation of a number.
     * @return the int value from the String 
     *    (or Integer.MAX_VALUE if error).
     */
    public static int parseInt(String s) {
        return Math2.roundToInt(parseDouble(s));
    }

    /**
     * Convert a string to a double.
     * Leading or trailing spaces are automatically removed.
     * This accepts hexadecimal integers starting with "0x".
     * This won't throw an exception if the number isn't formatted right.
     *
     * @param s is the String representation of a number.
     * @return the double value from the String (a finite value,
     *   Double.POSITIVE_INFINITY, Double.NEGATIVE_INFINITY, 
     *   or Double.NaN if error).
     */
    public static double parseDouble(String s) {
        try {
            if (s.startsWith("0x")) 
                return Integer.parseInt(s.substring(2), 16);
            return Double.parseDouble(s.trim());
        } catch (Exception e) {
            return Double.NaN;
        }
    }

    /**
     * Convert a string to an int, with rounding.
     * Leading or trailing spaces are automatically removed.
     * This won't throw an exception if the number isn't formatted right.
     *
     * @param s is the String representation of a number.
     * @return the int value from the String (or Double.NaN if error).
     */
    public static double roundingParseInt(String s) {
        return Math2.roundToInt(parseDouble(s));
    }

    /** 
     * This converts String representation of a long. 
     * Leading or trailing spaces are automatically removed.
     *
     * @param s a String representation of a long value
     * @return a long (or Long.MAX_VALUE if trouble).
     */
    public static long parseLong(String s) {
        try {
            if (s.startsWith("0x"))
                return Long.parseLong(s.substring(2), 16);
            return Long.parseLong(s.trim());
        } catch (Exception e) {
            return Long.MAX_VALUE;
        }
    }

    /**
     * Parse as a float with either "." or "," as the decimal point.
     * Leading or trailing spaces are automatically removed.
     *
     * @param s a String representing a float value (e.g., 1234.5 or 1234,5 or 
     *     1.234e3 1,234e3)
     * @return the corresponding float (or Float.NaN if not properly formatted)
     */
    public static float parseFloat(String s) {
        try {
            s = s.replace(',', '.');
            return Float.parseFloat(s.trim());
        } catch (Exception e) {
            return Float.NaN;
        }
    }

    /**
     * This converts a multiple-space-separated string into a String[] of separate tokens.
     * Double quoted tokens may have internal spaces.
     *
     * @param s the space-separated string
     * @return String[] of tokens (or null if s is null)
     */
    public static String[] tokenize(String s) {
        if (s == null)
            return null;

        ArrayList arrayList = new ArrayList();
        int sLength = s.length();
        int index = 0; //next char to be read
        //eat spaces
        while (index < sLength && s.charAt(index) == ' ') 
            index++;
        //repeatedly get tokens
        while (index < sLength) {
            //grab a token
            int start = index;
            int stop;
            //does it start with quotes?
            if (s.charAt(index) == '"') {
                index++; //skip the quotes
                start++;
                while (index < sLength && s.charAt(index) != '"') 
                    index++;
                stop = index; //if end of string and no closing quotes, it's a silent error
                index++; //skip the quotes
            } else {
                while (index < sLength && s.charAt(index) != ' ') 
                    index++;
                stop = index;
            }
            arrayList.add(s.substring(start, stop));

            //eat spaces
            while (index < sLength && s.charAt(index) == ' ') 
                index++;
        }

        return toStringArray(arrayList.toArray());
    }

    /** The size of the int[] needed for distribute() and getDistributionStatistics(). */
    public static int DistributionSize = 22;

    private static int BinMax[] = new int[]{0, 1, 2, 5, 10, 20, 50, 100, 200, 500,
        1000, 2000, 5000, 10000, 20000, //1,2,5,10,20 seconds
        60000, 120000, 300000, 600000, 1200000, 3600000, //1,2,5,10,20,60 minutes
        Integer.MAX_VALUE};

    /**
     * Put aTime into one of the distribution bins.
     * @param aTime 
     * @param distribution an int[DistributionSize] holding the counts of aTimes in 
     *   different categories
     */
    public static void distribute(long aTime, int[] distribution) {
        //catch really long times (greater than Integer.MAX_VALUE)
        if (aTime > 3600000) { distribution[21]++; return; }   //1hr   

        //safe to convert to int, and find appropriate bin
        int iTime = (int)aTime;
        for (int bin = 0; bin < DistributionSize; bin++) {
            if (iTime <= BinMax[bin]) {
                distribution[bin]++;
                return;
            }
        }
    }

    /**
     * Get the number of values in the distribution.
     *
     * @param distribution an int[DistributionSize] holding the counts of aTimes in 
     *   different categories
     * @return the number of values in the distribution.
     */
    public static int getDistributionN(int[] distribution) {
        //calculate n
        int n = 0;
        for (int bin = 0; bin < DistributionSize; bin++)
            n += distribution[bin];
        return n;
    }

    /**
     * Get the approximate median of the distribution.
     * See Sokal and Rohlf, Biometry, Box 4.1, pg 45.
     *
     * @param distribution an int[DistributionSize] holding the counts of aTimes in 
     *   different categories
     * @param n from getDistributionN
     * @return the approximate median of the distribution.
     */
    public static int getDistributionMedian(int[] distribution, int n) {
        double n2 = n / 2.0;

        if (n > 0) {
            //handle bin 0
            int cum = distribution[0];
            if (cum >= n2)
                return 0;

            for (int bin = 1; bin < DistributionSize; bin++) {  //bin 0 handled above
                if (distribution[bin] > 0) {
                    int tCum = cum + distribution[bin];
                    if (cum <= n2 && tCum >= n2) 
                        return Math2.roundToInt(
                            BinMax[bin-1] + ((n2-cum+0.0)/distribution[bin]) * (BinMax[bin] - BinMax[bin-1]));
                    cum = tCum;
                }
            }
        }
        return -1; //trouble
    }

    /**
     * Generate brief statistics for a distribution.
     * @param distribution an int[DistributionSize] holding the counts of aTimes in 
     *   different categories
     * @return the statistics
     */
    public static String getBriefDistributionStatistics(int[] distribution) {
        int n = getDistributionN(distribution);
        String s = "n =" + right("" + n, 9);
        if (n == 0) 
            return s;
        int median = getDistributionMedian(distribution, n);
        return s + ",  median ~=" + right("" + median, 9) + " ms";
    }

    /**
     * Generate statistics for a distribution.
     * @param distribution an int[DistributionSize] holding the counts of aTimes in 
     *   different categories
     * @return the statistics
     */
    public static String getDistributionStatistics(int[] distribution) {
        int n = getDistributionN(distribution);
        String s = 
            "    " + getBriefDistributionStatistics(distribution) + "\n";

        if (n == 0)
            return s;

        return 
            s + 
            "    0 ms:      " + right("" + distribution[0], 10) + "\n" +
            "    1 ms:      " + right("" + distribution[1], 10) + "\n" +
            "    2 ms:      " + right("" + distribution[2], 10) + "\n" +
            "    <= 5 ms:   " + right("" + distribution[3], 10) + "\n" +
            "    <= 10 ms:  " + right("" + distribution[4], 10) + "\n" +
            "    <= 20 ms:  " + right("" + distribution[5], 10) + "\n" +
            "    <= 50 ms:  " + right("" + distribution[6], 10) + "\n" +
            "    <= 100 ms: " + right("" + distribution[7], 10) + "\n" +
            "    <= 200 ms: " + right("" + distribution[8], 10) + "\n" +
            "    <= 500 ms: " + right("" + distribution[9], 10) + "\n" +
            "    <= 1 s:    " + right("" + distribution[10], 10) + "\n" +
            "    <= 2 s:    " + right("" + distribution[11], 10) + "\n" +
            "    <= 5 s:    " + right("" + distribution[12], 10) + "\n" +
            "    <= 10 s:   " + right("" + distribution[13], 10) + "\n" +
            "    <= 20 s:   " + right("" + distribution[14], 10) + "\n" +
            "    <= 1 min:  " + right("" + distribution[15], 10) + "\n" +
            "    <= 2 min:  " + right("" + distribution[16], 10) + "\n" +
            "    <= 5 min:  " + right("" + distribution[17], 10) + "\n" +
            "    <= 10 min: " + right("" + distribution[18], 10) + "\n" +
            "    <= 20 min: " + right("" + distribution[19], 10) + "\n" +
            "    <= 1 hr:   " + right("" + distribution[20], 10) + "\n" +
            "    >  1 hr:   " + right("" + distribution[21], 10) + "\n";
    }

    /**
     * If lines in this stringBuffer are >=maxLength characters, this inserts "\n"+spaces at the
     * previous non-DigitLetter + DigitLetter; or if none, this inserts "\n"+spaces at maxLength.
     * Useful keywords for searching for this method: longer, longest, noLongerThan.
     *
     * @param sb a StringBuffer with multiple lines, separated by \n's
     * @param maxLength the maximum line length allowed
     * @param spaces the string to be inserted after the inserted newline, e.g., "    " 
     * @return (for convenience) the same stringBuffer, but with no long lines
     */
    public static StringBuffer noLongLines(StringBuffer sb, int maxLength, String spaces) {
        int maxLength2 = maxLength / 2;
        int count = 0;
        for (int i = 0; i < sb.length(); i++) { //don't precalculate sb.size(); inserting \n changes it
           if (sb.charAt(i) == '\n') count = 0;
           else {
               count++; 
               if (count >= maxLength) {
                   int oi = i;
                   while (count > maxLength2) {
                       if (!isDigitLetter(sb.charAt(i-1)) && sb.charAt(i-1) != '(' && isDigitLetter(sb.charAt(i))) {
                           sb.insert(i, "\n" + spaces);
                           count = 0; //signal success
                           break;
                       }
                       count--;
                       i--;
                   }
                   if (count > 0) { //newline not inserted above
                       i = oi;
                       sb.insert(i, "\n" + spaces);
                       count = 0;
                   }
               }
           }
        }
        return sb;
/*        int count = 0;
        int maxLength10 = maxLength - 10;
        for (int i = 0; i < sb.length(); i++) { //don't precalculate sb.size(); inserting \n changes it
           if (sb.charAt(i) == '\n') count = 0;
           else {
               count++; 
               if (count >= maxLength10) {
                   char chm1 = sb.charAt(i-1);
                   if ((!isDigitLetter(chm1) && chm1 != '(' && isDigitLetter(sb.charAt(i))) ||
                       count >= maxLength) {
                       sb.insert(i, "\n" + spaces);
                       count = 0;
                   }
               }
           }
        }
        return sb;*/
    }

    /**
     * If lines in this stringBuffer are >=maxLength characters, this inserts "\n"+spaces at the
     * previous non-DigitLetter + DigitLetter; or if none, this inserts "\n"+spaces at maxLength.
     * Useful keywords for searching for this method: longer, longest, noLongerThan.
     *
     * @param s a String with multiple lines, separated by \n's
     * @param maxLength the maximum line length allowed
     * @param spaces the string to be inserted after the inserted newline, e.g., "    " 
     * @return (for convenience) the same String, but with no long lines
     */
    public static String noLongLines(String s, int maxLength, String spaces) {
        return noLongLines(new StringBuffer(s), maxLength, spaces).toString();
    }

    /**
     * This reads an ASCII file line by line (with any common end-of-line characters), 
     * does a simple (not regex) search and replace on each line, 
     * and saves the lines in another file (with String2.lineSeparator's).
     *
     * @param fullInFileName the full name of the input file
     * @param fullOutFileName the full name of the output file
     * @param search  a plain text string to search for
     * @param replace  a plain text string to replace any instances of <search>
     * @throws Exception if any trouble
     */
    public static void simpleSearchAndReplace(String fullInFileName,
        String fullOutFileName, String search, String replace) 
        throws Exception {
         
        BufferedReader bufferedReader = new BufferedReader(new FileReader(fullInFileName));
        BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(fullOutFileName));
                         
        //get the text from the file
        //This uses bufferedReader.readLine() to repeatedly
        //read lines from the file and thus can handle various 
        //end-of-line characters.
        String s = bufferedReader.readLine();
        while (s != null) { //null = end-of-file
            bufferedWriter.write(String2.replaceAll(s, search, replace));
            bufferedWriter.write(String2.lineSeparator);
            s = bufferedReader.readLine();
        }

        bufferedReader.close();
        bufferedWriter.close();

    }

    /**
     * This reads an ASCII file line by line (with any common end-of-line characters), 
     * does a regex search and replace on each line, 
     * and saves the lines in another file (with String2.lineSeparator's).
     *
     * @param fullInFileName the full name of the input file
     * @param fullOutFileName the full name of the output file
     * @param search  a regex to search for
     * @param replace  a plain text string to replace any instances of <search>
     * @throws Exception if any trouble
     */
    public static void regexSearchAndReplace(String fullInFileName,
        String fullOutFileName, String search, String replace) 
        throws Exception {
         
        BufferedReader bufferedReader = new BufferedReader(new FileReader(fullInFileName));
        BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(fullOutFileName));
                         
        //get the text from the file
        //This uses bufferedReader.readLine() to repeatedly
        //read lines from the file and thus can handle various 
        //end-of-line characters.
        String s = bufferedReader.readLine();
        while (s != null) { //null = end-of-file
            bufferedWriter.write(s.replaceAll(search, replace));
            bufferedWriter.write(String2.lineSeparator);
            s = bufferedReader.readLine();
        }

        bufferedReader.close();
        bufferedWriter.close();

    }


    /**
     * This returns a string with the keys and values of the Hashtable (sorted by the keys).
     *
     * @param hashtable (keys and values are objects with good toString methods)
     * @return a string with the sorted keys and values ("key1: value1\nkey2: value2\n")
     */
    public static String getKeysAndValuesString(Hashtable hashtable) {
        ArrayList al = new ArrayList();

        //synchronize so protected from changes in other threads
        synchronized (hashtable) { 
            Enumeration keys = hashtable.keys();
            while (keys.hasMoreElements()) {
                String key = (String)keys.nextElement();
                al.add(key + ": " + hashtable.get(key));
            }
        }
        Collections.sort(al);
        return toNewlineString(al.toArray());
    }

    /**
     * This returns the number formatted with up to 6 digits to the left and right of
     * the decimal and trailing decimal 0's removed.  
     * If abs(d) &lt; 0.0999995 or abs(d) &gt;= 999999.9999995, the number is displayed
     * in scientific notation (e.g., 8.954321E-5).
     * Thus the maximum length should be 14 characters (-123456.123456).
     * 0 returns "0"
     * NaN returns "NaN".
     * Double.POSITIVE_INFINITY returns "Infinity".
     * Double.NEGATIVE_INFINITY returns "-Infinity".
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genEFormat6(double d) {

        //!finite
        if (!Math2.isFinite(d))
            return "" + d;

        //almost 0
        if (Math2.almost0(d))
            return "0";

        //close to 0 
        //String2.log("genEFormat test " + (d*1000) + " " + Math.rint(d*1000));
        if (Math.abs(d) < 0.0999995 &&  
            !Math2.almostEqual(6, d * 10000, Math.rint(d * 10000))) {     //leave .0021 as .0021, but display .00023 as 2.3e-4
            synchronized(genExpFormat6) {
                return genExpFormat6.format(d);
            }
        }

        //large int
        if (Math.abs(d) < 1e13 && d == Math.rint(d))
            return "" + Math2.roundToLong(d);

        //>10e6
        if (Math.abs(d) >= 999999.9999995) {
            synchronized(genExpFormat6) {
                return genExpFormat6.format(d);
            }
        }

        synchronized(genStdFormat6) {
            return genStdFormat6.format(d);
        }
    }

    /**
     * This returns the number formatted with up to 10 digits to the left and right of
     * the decimal and trailing decimal 0's removed.  
     * If abs(d) &lt; 0.09999999995 or abs(d) &gt;= 999999.99999999995, the number is displayed
     * in scientific notation (e.g., 8.9544680321E-5).
     * Thus the maximum length should be 18 characters (-123456.1234567898).
     * 0 returns "0"
     * NaN returns "NaN".
     * Double.POSITIVE_INFINITY returns "Infinity".
     * Double.NEGATIVE_INFINITY returns "-Infinity".
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genEFormat10(double d) {

        //!finite
        if (!Math2.isFinite(d))
            return "" + d;

        //almost 0
        if (Math2.almost0(d))
            return "0";

        //close to 0 and many sig digits
        //String2.log("genEFormat test " + (d*1000) + " " + Math.rint(d*1000));
        if (Math.abs(d) < 0.09999999995 &&     
            !Math2.almostEqual(9, d * 1000000, Math.rint(d * 1000000))) {     //leave .0021 as .0021, but display .00023 as 2.3e-4
            synchronized(genExpFormat10) {
                return genExpFormat10.format(d);
            }
        }

        //large int
        if (Math.abs(d) < 1e13 && d == Math.rint(d)) //rint only catches 9 digits(?)
            return "" + Math2.roundToLong(d);

        //>10e6
        if (Math.abs(d) >= 999999.99999999995) {
            synchronized(genExpFormat10) {
                return genExpFormat10.format(d);
            }
        }

        synchronized(genStdFormat10) {
            return genStdFormat10.format(d);
        }
    }

    /**
     * This is like genEFormat6, but the scientific notation format
     * is, e.g., 8.954321x10^-5.
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genX10Format6(double d) {
        return replaceAll(genEFormat6(d), "E", "x10^");
    }

    /**
     * This is like genEFormat10, but the scientific notation format
     * is, e.g., 8.9509484321x10^-5.
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genX10Format10(double d) {
        return replaceAll(genEFormat10(d), "E", "x10^");
    }

    /**
     * This is like genEFormat6, but the scientific notation format
     * is, e.g., 8.954321x10<sup>-5</sup>.
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genHTMLFormat6(double d) {
        String s = genEFormat6(d);
        int po = s.indexOf('E');
        if (po >= 0) 
            s = replaceAll(genEFormat6(d), "E", "x10<sup>") + "</sup>";
        return s;
    }

    /**
     * This is like genEFormat10, but the scientific notation format
     * is, e.g., 8.9509244321x10<sup>-5</sup>.
     *
     * @param d a number
     * @return the number converted to a string
     */
    public static String genHTMLFormat10(double d) {
        String s = genEFormat10(d);
        int po = s.indexOf('E');
        if (po >= 0) 
            s = replaceAll(genEFormat10(d), "E", "x10<sup>") + "</sup>";
        return s;
    }

    /**
     * This removes white space characters at the beginning and end of a StringBuffer.
     *
     * @param sb a stringBuffer
     * @return the same pointer to the stringBuffer
     */
    public static StringBuffer trim(StringBuffer sb) {
        int po = 0;
        while (po < sb.length() && isWhite(sb.charAt(po))) po++;
        sb.delete(0, po);

        po = sb.length();
        while (po > 0 && isWhite(sb.charAt(po - 1))) po--;
        sb.delete(po, sb.length());
        return sb;
    }

    /**
     * This returns the directory that is the classpath for the source
     * code files (with forward slashes and a trailing slash, 
     * e.g., c:/programs/tomcat/webapps/cwexperimental/WEB-INF/classes/.
     *
     * @return directory that is the classpath for the source
     *     code files 
     * @throws Exception if trouble
     */
    public static String getClassPath() {
        if (classPath == null) {
            String find = "/com/cohort/util/String2.class";
            //use this.getClass(), not ClassLoader.getSystemResource (which fails in Tomcat)
            classPath = String2.class.getResource(find).getFile();
            classPath = replaceAll(classPath, '\\', '/');
            int po = classPath.indexOf(find);
            classPath = classPath.substring(0, po + 1);

            //on windows, remove the troublesome leading "/"
            if (OSIsWindows && classPath.length() > 2 && 
                classPath.charAt(0) == '/' && classPath.charAt(2) == ':')
                classPath = classPath.substring(1);
        }

        return classPath;
    }

    /**
     * On the command line, this prompts the user a String.
     *
     * @param prompt
     * @return the String the user entered
     * @throws Exception if trouble
     */
    public static String getStringFromSystemIn(String prompt) throws Exception{
        System.out.print(prompt);
        BufferedReader inReader = new BufferedReader(new InputStreamReader(System.in));
        return inReader.readLine();
    }

    /**
     * On the command line, this prompts the user a String (which is
     * not echoed to the screen, so is suitable for passwords).
     * This is slighly modified from 
     * http://java.sun.com/developer/technicalArticles/Security/pwordmask/ .
     *
     * @param prompt
     * @return the String the user entered
     * @throws IOException if trouble
     */
/*    public static String getPasswordFromSystemIn(String prompt) throws Exception {
        System.out.print(prompt);
        StringBuffer sb = new StringBuffer();
        while (true) {
            while (System.in.available() == 0) {
                Math2.sleep(1);
                System.out.print("\b*");
            }                    
            int ch = System.in.read();
            if (ch <= 0) 
                continue;
            if (ch == '\n') 
                return sb.toString();
            sb.append((char)ch);
        }
    }
*/
    public static final String getPasswordFromSystemIn(String prompt) throws Exception {
        InputStream in = System.in; //bob added, instead of parameter

        MaskingThread maskingthread = new MaskingThread(prompt);
        Thread thread = new Thread(maskingthread);
        thread.start();
        

        char[] lineBuffer;
        char[] buf;
        int i;

        buf = lineBuffer = new char[128];

        int room = buf.length;
        int offset = 0;
        int c;
        
        try { //bob added
            loop:   while (true) {
                c = in.read();
                if (c == -1 || c == '\n')
                    break loop;
                if (c == '\r') {
                    int c2 = in.read();
                    if ((c2 != '\n') && (c2 != -1)) {
                        if (!(in instanceof PushbackInputStream)) {
                            in = new PushbackInputStream(in);
                        }
                        ((PushbackInputStream)in).unread(c2);
                    } else {
                        break loop;
                    }
                }

                //if not caught and 'break loop' above...
                if (--room < 0) {
                    buf = new char[offset + 128];
                    room = buf.length - offset - 1;
                    System.arraycopy(lineBuffer, 0, buf, 0, offset);
                    Arrays.fill(lineBuffer, ' ');
                    lineBuffer = buf;
                }
                buf[offset++] = (char) c;
            }
        } catch (Exception e) {
        }
        maskingthread.stopMasking();
        if (offset == 0) {
           return ""; //bob changed from null
        }
        char[] ret = new char[offset];
        System.arraycopy(buf, 0, ret, 0, offset);
        Arrays.fill(buf, ' ');
        return new String(ret); //bob added; originally it returned char[]
    }

    /**
     * Find the last element which is <= s in an ascending sorted array.
     *
     * @param sar an ascending sorted String[] which may have duplicate values
     * @param s
     * @return the index of the last element which is <= s in an ascending sorted array.
     *   If s is null or s < the smallest element, this returns -1  (no element is appropriate).
     *   If s > the largest element, this returns sar.length-1.
     */
    public static int binaryFindLastLE(String[] sar, String s) {
        if (s == null) 
            return -1;
        int i = Arrays.binarySearch(sar, s);

        //an exact match; look for duplicates
        if (i >= 0) {
            while (i < sar.length - 1 && sar[i + 1].compareTo(s) <= 0)
                i++;
            return i; 
        }

        int insertionPoint = -i - 1;  //0.. sar.length
        return insertionPoint - 1;
    }

    /**
     * Find the first element which is >= s in an ascending sorted array.
     *
     * @param sar an ascending sorted String[] which currently may not have duplicate values
     * @param s
     * @return the index of the first element which is >= s in an ascending sorted array.
     *   If s < the smallest element, this returns 0.
     *   If s is null or s > the largest element, this returns sar.length (no element is appropriate).
     */
    public static int binaryFindFirstGE(String[] sar, String s) {
        if (s == null) 
            return sar.length;
        int i = Arrays.binarySearch(sar, s);

        //an exact match; look for duplicates
        if (i >= 0) {
            while (i > 0 && sar[i - 1].compareTo(s) >= 0)
                i--;
            return i; 
        }

        return -i - 1;  //the insertion point,  0.. sar.length
    }

    /**
     * Find the closest element to s in an ascending sorted array.
     *
     * @param sar an ascending sorted String[].
     *   It the array has duplicates and s equals one of them,
     *   it isn't specified which duplicate's index will be returned.
     * @param s
     * @return the index of the element closest to s.
     *   If s is null, this returns -1.
     */
    public static int binaryFindClosest(String[] sar, String s) {
        if (s == null)
            return -1;
        int i = Arrays.binarySearch(sar, s);
        if (i >= 0)
            return i; //success

        //insertionPoint at end point?
        int insertionPoint = -i - 1;  //0.. sar.length
        if (insertionPoint == 0) 
            return 0;
        if (insertionPoint >= sar.length)
            return sar.length - 1;

        //insertionPoint between 2 points 
        //do they differ at a different position?
        //make all the same length
        int preIndex = insertionPoint - 1;
        int postIndex = insertionPoint;
        String pre  = sar[preIndex];
        String post = sar[postIndex];
        int longest = Math.max(s.length(), Math.max(pre.length(), post.length()));
        String ts = s + makeString(' ', longest - s.length());
        pre  += makeString(' ', longest - pre.length());
        post += makeString(' ', longest - post.length());
        for (i = 0; i < longest; i++) {
            char ch = ts.charAt(i);
            char preCh = pre.charAt(i);
            char postCh = post.charAt(i);
            if (preCh == ch && postCh != ch) return preIndex;
            if (preCh != ch && postCh == ch) return postIndex;
            if (preCh != ch && postCh != ch) {
                //which one is closer
                return Math.abs(preCh - ch) < Math.abs(postCh - ch)?
                    preIndex : postIndex;
            }
        }
        //shouldn't all be equal
        return preIndex;
    }

    /**
     * This returns the index of the first non-utf-8 character.
     * Currently, valid characters are #32 - #126, #160+.
     *
     * @param s
     * @param alsoOK a string with characters (e.g., \n, \t) which are also valid
     * @return the index of the first non-utf-8 character, or -1 if all valid.
     */
    public static int findInvalidUtf8(String s, String alsoOK) {
        int n = s.length();
        for (int i = 0; i < n; i++) {
            char ch = s.charAt(i);
            if (alsoOK.indexOf(ch) >= 0)
                continue;
            if (ch < 32) 
                return i;
            if (ch <= 126)
                continue;
            if (ch <= 159)
                return i;
            //160+ is valid
        }
        return -1;
    }

    /**
     * This returns the UTF-8 encoding of the string (or null if trouble).
     * The inverse of this is utf8ToString.
     */
    public static byte[] getUTF8Bytes(String s) {
        try {
            return s.getBytes("UTF-8");             
        } catch (Exception e) {
            String2.log(ERROR + " in String2.getUTF8Bytes: " + e.toString());
            return null;
        }
    }

    /**
     * This returns a string from the UTF-8 encoded byte[] (or null if trouble).
     * The inverse of this is getUTF8Bytes.
     */
    public static String utf8ToString(byte[] bar) {
        try {
            return new String(bar, "UTF-8");             
        } catch (Exception e) {
            String2.log(ERROR + " in String2.utf8ToString: " + e.toString());
            return null;
        }
    }

    /**
     * This creates the jump table (int[256]) for a given 'find' stringUtf8
     * of use by indexOf(byte[], byte[], jumpTable[]) below.
     * Each entry in the result is: how far for indexOf to jump endPo forward for any given s[endPo] byte.
     *
     * @param find the byte array to be found
     * @return jump table (int[256]) for a given 'find' stringUtf8
     */
    public static int[] makeJumpTable(byte[] find) {
        //work forwards so last found instance of a letter is most important
        //   s = Two times nine.
        //find = nine
        //First test will compare find's 'e' and s's ' '
        //Not a match, so jump jump[' '] positions forward.
        int findLength = find.length;
        int jump[] = new int[256];
        Arrays.fill(jump, findLength);
        for (int po = 0; po < findLength; po++)
            jump[find[po] & 0xFF] = findLength - 1 - po;   //make b 0..255
        return jump;
    }


    /**
     * Return the first index of 'find' in s (or -1 if not found).
     * Idea: since a full text search entails looking for a few 'find' strings
     *   inside any of nDatasets long searchStrings (that don't change often),
     *   and since we don't really care about exact index, just relative index,
     *   it would be nice to store searchStrings as byte[]
     *   (1/2 the memory use and simpler search).
     *   So encode 'find' and searchString as UTF-8 via byte[] find.getBytes(utf8Charset).
     *   Then we can do Boyer-Moore-like search for first indexOf.
     *   This can speed up the searches ~3.5X in good conditions (assuming setup is amortized).
     *
     * @param s the long string to be search, stored utf8.
     * @param find the short string to be found, stored utf 8.
     * @param jumpTable from makeJumpTable
     * @return the first index of 'find' in s (or -1 if not found).
     */
    public static int indexOf(byte[] s, byte[] find, int jumpTable[]) {
        //future: is jump table for second character jumpTable[s[endPo]]-1 IFF that value isn't <=0?

        //see algorithm in makeJumpTable
        int findLength = find.length;
        int sLength = s.length;
        if (findLength == 0)  return 0;
        if (sLength == 0)     return -1;
        int findLength1 = findLength - 1;
        int endPo = findLength1;
        byte lastFindByte = find[findLength1]; 

        //if findLength is 1, do simple search
        if (findLength == 1) {
            int po = -1;
            while (++po < sLength) {
                if (s[po] == lastFindByte)
                    return po;  
            }
            return -1;
        }

        //Boyer-Moore-like search
        whileBlock:
        while (endPo < sLength) {
            byte b = s[endPo];

            //last bytes don't match? jump
            if (b != lastFindByte) {
                endPo += jumpTable[b & 0xFF]; //make b 0..255
                continue;
            }

            //last bytes do match: try to match all of 'find'
            int countBack = 1;
            do {  //we know find is at least 2 long
                if (s[endPo - countBack] == find[findLength1 - countBack]) {
                   countBack++;
                } else {
                    endPo += 1;
                    continue whileBlock;
                }
            } while (countBack < findLength); 

            //found it!
            return endPo - findLength1;
        }
        return -1;
    } 

    /**
     * Given two strings with internal newlines, oldS and newS, this a message
     * indicating where they differ.
     * 
     * @param oldS  
     * @param newS
     * @return a message indicating where they differ, or "" if there is no difference.
     */
    public static String differentLine(String oldS, String newS) {
        if (oldS == null) return "(There is no old version.)";
        if (newS == null) return "(There is no new version.)";
        int oldLength = oldS.length();
        int newLength = newS.length();
        int newlinePo = -1;
        int line = 1;
        int n = Math.min(oldLength, newLength);
        int po = 0;
        while (po < n && oldS.charAt(po) == newS.charAt(po)) {
            if (oldS.charAt(po) == '\n') {newlinePo = po; line++;}
            po++;
        }
        if (po == oldLength && po == newLength)
            return "";
        int oldEnd = newlinePo + 1;
        int newEnd = newlinePo + 1;
        while (oldEnd < oldLength && oldS.charAt(oldEnd) != '\n') oldEnd++;
        while (newEnd < newLength && newS.charAt(newEnd) != '\n') newEnd++;
        return 
            "  old line #" + line + "=\"" + oldS.substring(newlinePo + 1, oldEnd) + "\",\n" +
            "  new line #" + line + "=\"" + newS.substring(newlinePo + 1, newEnd) + "\".";
    }

    /* *
     * This makes a medium-deep clone of an ArrayList by calling clone() of
     * each element of the ArrayList.
     *
     * @param oldArrayList
     * @param newArrayList  If oldArrayList is null, this returns null.
     *    Elements of oldArrayList can be null.
     */
    /* I couldn't make this compile. clone throws an odd exception.
    public ArrayList clone(ArrayList oldArrayList) {
        if (oldArrayList == null)
            return (ArrayList)null;

        ArrayList newArrayList = new ArrayList();
        int n = oldArrayList.size();
        for (int i = 0; i < n; i++) {
            Object o = oldArrayList.get(i);
            try {
                if (o != null) o = o.clone();
            } catch (Exception e) {
            }
            newArrayList.add(o);
        }
        return newArrayList;
    } */

} //End of String2 class.
