/* 
 * CCUser Copyright 2005, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch;

import com.cohort.array.ByteArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.ema.*;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import com.lowagie.text.PageSize;

import gov.noaa.pfel.coastwatch.griddata.*;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.*;
import gov.noaa.pfel.coastwatch.util.*;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.RenderingHints; 
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.GregorianCalendar;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/**
 * This is a collection of the things unique to a user's CencoosCurrents session.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2005-09-12
 */
public class CCUser extends User {


    private EmaClass emaClass;
    private CurrentsScreen currentsScreen;
    private EmaSelect  edit;
    private EmaButton  submitForm, resetAll, back;
    final static boolean displayErrorMessages = false;//an EMA setting; always false
    final static int CURRENTS_SCREEN_INDEX = 0;
    private String imageStoreRootDirectory; //null if not used
    private String clickOnMapForDrifterModel;
    private String missingDataCausedModelToStop;
    private String vectorPalette;
    private String vectorScale;
    private double vectorColorBarMin;
    private double vectorColorBarMax;
    private int vectorNSections;
    private boolean vectorContinuous;

    /** 
     * The constructor for CCUser. 
     */
    public CCUser(OneOf oneOf, Shared shared, HttpSession session, boolean doTally) {
        super(oneOf, shared, session, doTally);

        //first thing: change data set names in shared
        customizeGridDataSetNames();

        String errorInMethod = String2.ERROR + " in CCUser constructor:\n";
        emaClass = new EmaClass(oneOf.fullClassName(), oneOf.emaRB2(), oneOf.classRB2());
        int editOption = 0;
        currentsScreen = new CurrentsScreen(editOption++, oneOf, shared, emaClass, doTally);

        //addAttribute(new EmaLabel(this, "instructions"));
        emaClass.addAttribute(edit        = new EmaSelect(emaClass, "edit"));
        emaClass.addAttribute(submitForm  = new EmaButton(emaClass, "submitForm"));
        emaClass.addAttribute(resetAll    = new EmaButton(emaClass, "resetAll"));
        emaClass.addAttribute(back        = new EmaButton(emaClass, "back"));

        Test.ensureNotNull(edit.getLabel(),       errorInMethod + "edit.label is null.");
        Test.ensureNotNull(submitForm.getLabel(), errorInMethod + "submitForm.label is null.");
        Test.ensureNotNull(resetAll.getLabel(),   errorInMethod + "resetAll.label is null.");
        Test.ensureNotNull(back.getLabel(),       errorInMethod + "back.label is null.");

        //addDefaults
        emaClass.addDefaultsToSession(session);

        //imageStoreRootDirectory
        imageStoreRootDirectory = oneOf.classRB2().getString("imageStoreRootDirectory", null);
        clickOnMapForDrifterModel = oneOf.classRB2().getString("clickOnMapForDrifterModel", null);
        missingDataCausedModelToStop = oneOf.classRB2().getString("missingDataCausedModelToStop", null);
        Test.ensureNotNull(clickOnMapForDrifterModel, errorInMethod + "clickOnMapForDrifterModel is not found in CencoosCurrents.properties.");
        Test.ensureNotNull(missingDataCausedModelToStop, errorInMethod + "missingDataCausedModelToStop is not found in CencoosCurrents.properties.");
        if (imageStoreRootDirectory != null) {
            if (!imageStoreRootDirectory.endsWith("/"))
                imageStoreRootDirectory += "/";
            Test.ensureTrue(File2.isDirectory(imageStoreRootDirectory), 
                errorInMethod + "imageStoreRootDirectory doesn't exist: " + imageStoreRootDirectory);
        }

        //vector settings
        vectorPalette     = oneOf.classRB2().getString("vector.palette", null);
        vectorScale       = oneOf.classRB2().getString("vector.scale", null);
        vectorColorBarMin = oneOf.classRB2().getDouble("vector.colorBarMin", Double.NaN);
        vectorColorBarMax = oneOf.classRB2().getDouble("vector.colorBarMax", Double.NaN);
        vectorNSections   = oneOf.classRB2().getInt("vector.nSections", -1);
        vectorContinuous  = oneOf.classRB2().getBoolean("vector.continuous", false);


    }


    /**
     * This resets the Shared info for this user.
     * Because this is handled by one method (with one value passed in), 
     * it has the effect of synchronizing everything done within it.
     *
     * @param shared the new Shared object
     */
    public void setShared(Shared shared) {
        //first thing: change data set names in shared
        customizeGridDataSetNames();

        this.shared = shared;
        currentsScreen.setShared(shared);
    }

    /** This customizes the grid dataset names so they are more human-friendly. 
     *  Yes, this is very kludgey.
     */
    private void customizeGridDataSetNames() {
        for (int i = OneOf.N_DUMMY_GRID_DATASETS; i < shared.activeGridDataSets().size(); i++) {  //2.. because 0=(None) 1=Bath
            //figure out the new name
            String option = shared.activeGridDataSetOptions()[i]; 
            if (option.startsWith("Chlorophyll-a, Aqua MODIS"))
                option = "MODIS Aqua Chlorophyll-a Concentration";
            else if (option.startsWith("SST, NOAA POES AVHRR"))
                option = "AVHRR Sea Surface Temperature";
            else if (option.startsWith("SST, NOAA GOES"))
                option = "GOES Sea Surface Temperature";
            //else String2.log(String2.ERROR + " in CCUser.customizeGridDataSetNames: unrecognized option: " + option);

            //make the changes
            shared.activeGridDataSetOptions()[i] = option;
            GridDataSet gds = (GridDataSet)shared.activeGridDataSets().get(i);
            gds.option = option;
            gds.boldTitle = option;
        }

    }

    
    /** This returns the user's emaClass object. */
    public EmaClass emaClass() {return emaClass;}

    /** This returns the user's currentsScreen object. */
    public CurrentsScreen currentsScreen() {return currentsScreen;}

    /** This returns the user's edit object. */
    public EmaSelect edit() {return edit;}

    /** This returns the user's submitForm object. */
    public EmaButton submitForm() {return submitForm;}

    /** This returns the user's emaButton object. */
    public EmaButton resetAll() {return resetAll;}

    /** This returns the user's back object. */
    public EmaButton back() {return back;}


    /**
     * This returns the number of requests made in this user session.
     *
     * @param session usually created with request.getSession()
     * @return the number of requests made in this user session
     *    (or -1 if defaults haven't even been set)
     */
    public int getNRequestsThisSession(HttpSession session) {
        return emaClass.getNRequestsThisSession(session);
    }

    /**
     * This handles a "request" from a user, storing incoming attributes
     * as session values.
     * This updates totalNRequests, totalProcessingTime, maxProcessingTime.
     *
     * @param request 
     * @return true if all the values on the form are valid
     */
    public boolean processRequest(HttpServletRequest request) throws Exception {   
        resetLastAccessTime();

        //call standard processRequest
        boolean result = emaClass().processRequest(request);

        //was submitter a request to reset all settings for this client?
        String submitter = emaClass().getSubmitterButtonName(request);
        if (submitter.equals(resetAll().getName())) {
            emaClass().addDefaultsToSession(session);
        }
        
        return result;
    }

    /** 
     * This does most of the work (validate each screen, generate html,
     * and create the files).
     *
     * @param request is a request from a user
     * @param StringBuffer htmlSB
     * @return true if successful (no exceptions thrown)
     */
    public boolean getHTMLForm(HttpServletRequest request, StringBuffer htmlSB, long startTime) {
        boolean succeeded = true;

        try {
            if (oneOf.verbose()) 
                String2.log("\n************ getHTMLForm");

            IntObject step      = new IntObject(1); //the next step number for the user
            IntObject rowNumber = new IntObject(0);
         
            //show (show = CURRENTS_SCREEN_INDEX, but equals -1 if a getXxx button pressed)
            //  Thus, show reflects which screen is shown to the user.
            int show = CURRENTS_SCREEN_INDEX;
            String submitter = emaClass().getSubmitterButtonName(request);
            if (submitter.length() > 0) { 
                if (currentsScreen.submitterIsAGetButton(submitter)) 
                    show = -1; 
            }

            //start the form
            if (show >= 0) {
                //the start of the HTML form and the HTML Table
                htmlSB.append(emaClass().getStartOfHTMLForm());

            }

            //validate currentsScreen  (and if visible, add html to htmlSB)
            currentsScreen.validate(session, show, step, rowNumber, htmlSB);

            //get 'warning'
            String warning = getWarning();

            //make the imageFileName
            //Note: I had trouble with long psgif file names (>126 char). ImageMagick would just return a black .gif. E.g.
            //LATsstaS1day_20050425_x-135_X-113_y30_Y50_PRainbow_Linear_8_28_CLATsstaS1day20050425_4.0_8.0_13.0_16.0_0xFF0000_VOQNux10S1day20050415.gif
            //So I replaced the less important (and potentially long) palette info with hashcodes
            //(which return the same code for the same data -- so images can be cached).  
            //And I removed nice but not necessary underscores. 
            //4/26/05
            String imageFileName = currentsScreen.startImageFileName + OneOf.imageExtension;

            //update requestedGridFilesMap
            if (doTally) 
                oneOf.tally().add("Most Requested .gif Files:", imageFileName);

            //use SGT to make the .gif map (if it doesn't already exist)
            if (File2.touch(oneOf.fullPublicDirectory() + imageFileName)) {
                if (oneOf.verbose()) 
                    String2.log("reusing .gif publicDir: " + imageFileName);

            } else if (!oneOf.displayDiagnosticInfo() && 
                       imageStoreRootDirectory != null &&
                       File2.touch(imageStoreDirectory() + imageFileName)) {
                File2.copy(imageStoreDirectory() + imageFileName,
                     oneOf.fullPublicDirectory() + imageFileName);
                if (oneOf.verbose()) 
                    String2.log("reusing image from imageStoreDir: " + imageFileName);

            } else {

                if (oneOf.verbose()) 
                    String2.log("makeMap new imageFileName=" + imageFileName);

                //make the image
                BufferedImage bufferedImage = 
                    SgtUtil.getBufferedImage(currentsScreen.imageWidth, currentsScreen.imageHeight);
                Graphics2D g2d = (Graphics2D)bufferedImage.getGraphics();

                //makeMap
                makeMap(
                    oneOf.lowResLogoImageFile(), g2d,  
                    0, 0, currentsScreen.imageWidth, currentsScreen.imageHeight,
                    1);    //standard font size

                //warning
                if (warning != null) 
                    AttributedString2.drawHtmlText(g2d, warning, currentsScreen.imageWidth / 2, 10, 
                        oneOf.fontFamily(), 10, Color.red, 1);

                //save image
                SgtUtil.saveImage(bufferedImage, oneOf.fullPublicDirectory() + imageFileName);

                //store a copy in imageStoreDirectory
                if (!oneOf.displayDiagnosticInfo() &&
                    imageStoreRootDirectory != null)
                    File2.copy(oneOf.fullPublicDirectory() + imageFileName,
                                     imageStoreDirectory() + imageFileName);
            }
 
            //User clicked on map. User wants drifter model.
            //get x,y from query
            String query = request.getQueryString();
            String2.log("Query: " + query);
            int drifterX = Integer.MAX_VALUE;
            int drifterY = Integer.MAX_VALUE;
            if (query != null && query.matches("\\d+,\\d+")) { //digits,digits
                int po = query.indexOf(',');
                drifterX = String2.parseInt(query.substring(0, po)); 
                drifterY = String2.parseInt(query.substring(po + 1));
                String2.log("  drifter x=" + drifterX + " y=" + drifterY);
            }

            //generate the animated gif?  
            if (submitter.equals(currentsScreen.viewAnimation.getName())) 
                imageFileName = createAnimatedGif(session, htmlSB, startTime, 
                    String2.parseInt(currentsScreen.animationNValue));
            else if (currentsScreen.plotVectorData && 
                     drifterX < Integer.MAX_VALUE && drifterY < Integer.MAX_VALUE) 
                imageFileName = createDrifterAnimatedGif(session, htmlSB, 
                    startTime, drifterX, drifterY, 
                    "non-interpolating".equals(currentsScreen.interpolatingValue),
                    String2.parseInt(currentsScreen.modelNValue)); 

            //end of table, end of form; display the map image
            if (show >= 0) {
                endOfFormShowImage(session, htmlSB, imageFileName, startTime);
            }

           //deal with show < 0
           /*if (show < 0) {

                //deal with 'get' requests
                boolean submitterHandled = true;
                if () {
                }  else if (currentsScreen.submitterIsAGetButton(submitter)) {
                    //backButtonForm is useful in several situations.
                    //The link to go back to CWBrowser.jsp often lead to jumble of previous values. [why?]
                    //String oBack = "o back to editing the map.";
                    //htmlSB.append("<p>Then, <a href=\"CWBrowser.jsp\" title=\"G" + oBack + "\">g" + oBack + "</a>\n");
                    //(The browser's Back button goes to the previous page which is fine.)
                    //So use a Back button:
                    String backButtonForm =
                        emaClass.getStartOfHTMLForm() +
                        "<tr>\n" +
                        "  <td>" +
                            back.getLabel() + "\n" +
                            back.getControl(back.getValue(session)) + "\n" +
                        "  </td>\n" +
                        "  <td>&nbsp;</td>\n" + //to take up horizontal space
                        "</tr>\n" +
                        emaClass.getEndOfHTMLForm(startTime, "") + "\n";

                    currentsScreen.respondToSubmitter(submitter, htmlSB, backButtonForm);
                } else submitterHandled = false;
                if (!submitterHandled) 
                    throw new Exception("Unexpected \"submitter\": " + submitter);

            } //end of show < 0
            */

        } catch (Exception e) {
            succeeded = false;

            //display the error message to the user and print to log 
            String tError = MustBe.throwableToString(e);
            String2.log(tError);

            htmlSB.setLength(0);
            htmlSB.append( 
                oneOf.errorMessage1() + "\n" +
                e.toString() + "\n" +
                oneOf.errorMessage2() + "\n" +
                emaClass.getStartOfHTMLForm() +
                  "<tr>\n" +
                  "  <td>" +
                      //The link to go back to CWBrowser.jsp often lead to jumble of previous values. [why?]
                      //String GoBack = "Go back to editing the map.";
                      //"<a href=\"CWBrowser.jsp\" title=\"" + GoBack + "\">" + GoBack + "</a>" +
                      //(The browser's Back button goes to the previous page which is fine.)
                      //So use a Back button:
                      back.getLabel() + "\n" +
                      back.getControl(back.getValue(session)) + "\n" +
                      "<br>(" + resetAll.getLabel() + "\n" +
                      resetAll.getControl(resetAll.getValue(session)) + 
                          "&nbsp;)</td>\n" +
                  "  <td>&nbsp;</td>\n" + //to take up horizontal space
                  "</tr>\n" +
                emaClass.getEndOfHTMLForm(startTime, "") + "\n" +
                "<p>&nbsp;<hr>\n" +
                "<pre>Details:\n" +
                "\n");

            //add the actual diagnostic info  
            StringBuffer logSB = String2.getLogStringBuffer();
            if (logSB == null) 
                logSB = new StringBuffer("[logStringBuffer is null.]");
            else String2.noLongLines(logSB, 100, "    "); //ccbrowser's form expands if diagnostics are wide!
            htmlSB.append(logSB);
            htmlSB.append("\n" + tError); //always in htmlSB
            htmlSB.append("</pre>\n"); 
            //clear the String2.logStringBuffer
            logSB.setLength(0); 

            //send email
            oneOf.email(oneOf.emailEverythingTo(), 
                OneOf.ERROR + " in " + oneOf.shortClassName(), htmlSB.toString());

        } //end of 'catch'

        //display diagnostic information during development
        if (oneOf.verbose()) String2.log("************ getHTMLForm done. TOTAL TIME=" + 
            (System.currentTimeMillis() - startTime));
        StringBuffer logSB = String2.getLogStringBuffer();
        if (oneOf.displayDiagnosticInfo()) {
            htmlSB.append("<p>Diagnostic Info = <pre>");
            if (logSB == null)
                logSB.append("[logStringBuffer is null.]");
            else String2.noLongLines(logSB, 100, "    "); //ccbrowser's form expands if diagnostics are wide!
            htmlSB.append(logSB);
            htmlSB.append("</pre>\n");        
        }

        //clear the String2.logStringBuffer
        if (logSB != null) {
            logSB.setLength(0); 
        }

        return succeeded;

    }

    /**
     * This determines the directory where the image should be stored,
     * based on the current user selections. 
     * If this directory doesn't exist, this creates it.
     * 
     * @return the directory where the image should be stored
     * @throws Exception (e.g., if imageStoreRootDirectory is null)
     */
    private String imageStoreDirectory() throws Exception {
        String errorInMethod = String2.ERROR + " in UserCC.imageStoreDirectory:\n";
        Test.ensureNotNull(imageStoreRootDirectory, 
            errorInMethod + "imageStoreRootDirectory is null.");

        String gridInternalName = currentsScreen.gridInternalName;
        String2.log("vectorDataSetIndex=" + currentsScreen.vectorDataSetIndex +
            " xVectorDataSet==null?" + (currentsScreen.xVectorDataSet==null));
        String vectorInternalName = currentsScreen.xVectorDataSet == null?
            "none" : currentsScreen.xVectorDataSet.internalName;
        String timePeriodName = currentsScreen.timePeriodValue == null?
            "none" : TimePeriods.getInFileName(currentsScreen.timePeriodValue);
        String dir = 
            imageStoreRootDirectory +
            String2.replaceAll(currentsScreen.regionValue, " ", "") + "/" + //e.g., MontereyBay
            vectorInternalName + "/" + //e.g., LCMusfc
            gridInternalName + "/" + //e.g., LATssta
            timePeriodName + "/";
        
        //ensure dir exists
        if (!File2.isDirectory(dir)) {
            File d = new File(dir);
            Test.ensureTrue(d.mkdirs(), errorInMethod + "unable to create " + dir);
        }    

        return dir;
    }

    /**
     * This adds the end of the form to htmlSB and shows the image.
     *
     * @param session
     * @param htmlSB
     * @param imageFileName the imageFileName without the directory, with the extension
     * @param startTime
     */
    public void endOfFormShowImage(HttpSession session, StringBuffer htmlSB, 
        String imageFileName, long startTime) {

        //display submitForm button
        //This isn't beginning of row, just set the color.
        emaClass().setBeginRow("<tr bgcolor=\"#" + oneOf.backgroundColor(3) + "\">");
        //old way
        //but 'noscript' must be between td /td tags to be HTML compliant. 
        htmlSB.append("    <noscript>\n" + 
            submitForm.getTableEntry(submitForm.getValue(session), displayErrorMessages) +
            "    </noscript>\n");
        //new way
        //This causes thin empty row in table.
        //htmlSB.append(
        //    "    " + emaClass.getBeginRow() + "\n" + 
        //    "      <td><noscript>" + submitForm.getLabel() + "&nbsp;</noscript></td>\n" +
        //    "      <td><noscript>" + submitForm.getControl(submitForm().getValue(session)) + "</noscript></td>\n" +
        //    "    " + emaClass.getEndRow() + "\n");

        //end of form
        htmlSB.append(emaClass().getEndOfHTMLForm(startTime, ""));

        //display the map image    is <a href> so ismap works
        htmlSB.append(
            "<a href=\"" + emaClass.getUrl() + "\">" +
                "<img src=\"" + OneOf.PUBLIC_DIRECTORY + imageFileName + "\"\n" + 
            "  title=\"" + clickOnMapForDrifterModel + "\"\n" +
            "  alt=\"" + oneOf.hereIsAlt() + "\"\n" + 
            "  onclick=\"pleaseWait();\"\n" + 
            "  ismap=\"ismap\" width=\"" + currentsScreen.imageWidth + 
                "\" height=\"" + currentsScreen.imageHeight + "\"></a>\n");
    }

    /**
     * This determines the level of timePeriod sychronization and generates
     * the 'warning' message.
     *
     * @return the warning message  (or null if there is no warning)
     */
    private String getWarning() {

        //determine the level of timePeriod sychronization
        long startEndCount[] = {Long.MIN_VALUE, Long.MAX_VALUE, 0, Long.MAX_VALUE, Long.MIN_VALUE};
        boolean timePeriodsPerfectlySynchronized = true;
        if (currentsScreen.plotGridData) {
            timePeriodsPerfectlySynchronized = adjustStartEndCount(
                startEndCount, 
                currentsScreen.gridStartCalendar.getTimeInMillis(),
                currentsScreen.gridEndCalendar.getTimeInMillis(),
                timePeriodsPerfectlySynchronized);
        }
        if (currentsScreen.plotVectorData) {
            timePeriodsPerfectlySynchronized = adjustStartEndCount(
               startEndCount, 
                currentsScreen.vectorStartCalendar.getTimeInMillis(),
                currentsScreen.vectorEndCalendar.getTimeInMillis(),
                timePeriodsPerfectlySynchronized);
        }
        //note that stationVectors share vectorStartCalendar and endCalendar.
        //Not nearest time. If no data found, then no data plotted.
        
        boolean timePeriodsDisjoint = startEndCount[2] >= 2 && //at least 2 data sets will be plotted
            startEndCount[1] < startEndCount[0]; //end time < start time    (not =, since pass data will be =
        //special case: if total time range <=59 minutes,
        //  the time periods are considered the same
        if (startEndCount[2] >= 2 && //at least 2 data sets will be plotted
            startEndCount[4] - startEndCount[3] <= 59 * Calendar2.MILLIS_PER_MINUTE) {
            timePeriodsDisjoint = false;
            timePeriodsPerfectlySynchronized = true;
        }
        String warning = null;
        if (timePeriodsDisjoint) 
            warning = oneOf.warningTimePeriodsDisjoint();
        else if (!timePeriodsPerfectlySynchronized)
            warning = oneOf.warningTimePeriodsDifferent();
        return warning;
    }

    /**
     * This adjusts latestStartEarliestEndCount.
     *
     * @param latestStartEarliestEndCount the long[5] with the 
     *    0=latestStart, 1=earliestEnd, 2=count, 3=earliestStart, 4=latestEnd
     * @param tStart in millis since epoch
     * @param tEnd in millis since epoch
     * @param timePeriodsPerfectlySynchronized
     * @return false if timePeriodsPerfectlySychronized was false or tStart!=latestStart || tEnd!=earliestEnd.
     */
    private boolean adjustStartEndCount(long[] latestStartEarliestEndCount,
        long tStart, long tEnd, boolean timePeriodsPerfectlySynchronized) {
        latestStartEarliestEndCount[0] = Math.max(latestStartEarliestEndCount[0], tStart);
        latestStartEarliestEndCount[1] = Math.min(latestStartEarliestEndCount[1], tEnd);
        latestStartEarliestEndCount[2]++;
        latestStartEarliestEndCount[3] = Math.min(latestStartEarliestEndCount[3], tStart);
        latestStartEarliestEndCount[4] = Math.max(latestStartEarliestEndCount[4], tEnd);
        if (latestStartEarliestEndCount[2] > 1 && 
            (tStart != latestStartEarliestEndCount[0] || tEnd != latestStartEarliestEndCount[1]))
            timePeriodsPerfectlySynchronized = false;
        //String2.log("adjustStartEndCount tStart=" + Calendar2.epochSecondsToIsoStringT(tStart/1000) +
        //    " tEnd=" + Calendar2.epochSecondsToIsoStringT(tEnd/1000) +
        //    "\n latestStart=" + Calendar2.epochSecondsToIsoStringT(latestStartEarliestEndCount[0]/1000) +
        //    " earliestEnd=" + Calendar2.epochSecondsToIsoStringT(latestStartEarliestEndCount[1]/1000) +
        //    " perfect=" + timePeriodsPerfectlySynchronized +
        //    "\n earliestStart=" + Calendar2.epochSecondsToIsoStringT(latestStartEarliestEndCount[3]/1000) +
        //    " latestEnd=" + Calendar2.epochSecondsToIsoStringT(latestStartEarliestEndCount[4]/1000));
        return timePeriodsPerfectlySynchronized;
    }

    /**
     * This creates an animated gif.
     *
     * @param session
     * @param htmlSB
     * @param startTime
     * @param nHoursBack (e.g., 24)
     * @return the name of the animated gif file (without directory, but with .gif)
     * @throws Exception if trouble
     */
    private String createAnimatedGif(HttpSession session, StringBuffer htmlSB, 
            long startTime, int nHoursBack) throws Exception {
        String errorInMethod = OneOf.ERROR + " in createAnimatedGif:\n";
        long time = System.currentTimeMillis();
        int delay = 100; //usually 100; in 1/100th's second   200 for testing
        int show = -1;
        IntObject step = new IntObject(1);
        IntObject rowNumber = new IntObject(0);
        if (doTally) 
            oneOf.tally().add("Animated Gifs Displayed:", "Standard");

        //generate the animatedGifFileName
        String animatedGifFileName = 
            "Animate" + nHoursBack + "_" + currentsScreen.startImageFileName + ".gif"; //always .gif

        //already exists?
        if (File2.touch(oneOf.fullPublicDirectory() + animatedGifFileName)) {
            if (oneOf.verbose()) 
                String2.log("\nANIMATED GIF reusing: " + animatedGifFileName);
            return animatedGifFileName;
        } else {
            if (oneOf.verbose())
                String2.log("\nANIMATED GIF creating: " + animatedGifFileName);
        }

        //generate the magic number  so ImageMagick regex can find the files
        int randomInt = Math2.random(Integer.MAX_VALUE);

        String centeredTimeValue = currentsScreen.timeValue;
        double currentSeconds = Calendar2.isoStringToEpochSeconds(centeredTimeValue) - //throws exception if trouble
            Calendar2.SECONDS_PER_HOUR * nHoursBack;

        //generate special images's for last nFrames time periods (including current)
        StringArray specialImageNames = new StringArray();
        String originalVectorDataSetValue = currentsScreen.vectorDataSetValue;
        try { //so can clean up if exception thrown
            for (int frame = 0; frame <= nHoursBack; frame++) {

                //set desired timeValue
                String currentSecondsString = Calendar2.epochSecondsToIsoStringSpace(currentSeconds);
                currentsScreen.time.setValue(session, currentSecondsString);
                String2.log("createAnimatedGif frame#" + frame + " for " + currentSecondsString);

                //process that (to synchronize other things)
                currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
                step.i = 1;
                rowNumber.i = 0;
                currentsScreen.validate(session, show, step, rowNumber, htmlSB);

                //check if desired time was matched exactly 
                if (!currentsScreen.timeValue.equals(currentSecondsString)) {
                    //don't display any vector data
                    currentsScreen.vectorDataSet.setValue(session, OneOf.NO_DATA);
                    step.i = 1;
                    rowNumber.i = 0;
                    currentsScreen.validate(session, show, step, rowNumber, htmlSB);
                }

                //generate the frame's image
                String imageFileName = currentsScreen.startImageFileName + OneOf.imageExtension;

                //update tally
                if (doTally) 
                    oneOf.tally().add("Most Requested " + OneOf.imageExtension + " Files:", imageFileName);

                //use SGT to make the image (if it doesn't already exist)
                if (File2.touch(oneOf.fullPublicDirectory() + imageFileName)) {
                    if (oneOf.verbose()) 
                        String2.log("  reusing .gif publicDir: " + imageFileName);

                } else if (!oneOf.displayDiagnosticInfo() && 
                           imageStoreRootDirectory != null &&
                           File2.touch(imageStoreDirectory() + imageFileName)) {
                    File2.copy(imageStoreDirectory() + imageFileName,
                         oneOf.fullPublicDirectory() + imageFileName);
                    if (oneOf.verbose()) 
                        String2.log("reusing .gif imageStoreDir: " + imageFileName);

                } else {
                    if (oneOf.verbose()) 
                        String2.log("  makeMap new imageFileName=" + imageFileName);

                    //make the image
                    BufferedImage bufferedImage = 
                        SgtUtil.getBufferedImage(currentsScreen.imageWidth, currentsScreen.imageHeight);
                    Graphics2D g2d = (Graphics2D)bufferedImage.getGraphics();

                    //makeMap
                    makeMap(
                        oneOf.lowResLogoImageFile(), g2d,  
                        0, 0, currentsScreen.imageWidth, currentsScreen.imageHeight,
                        1);    //standard font size

                    //get the 'warning' message
                    String warning = getWarning();
                    if (warning != null) 
                        AttributedString2.drawHtmlText(g2d, warning, currentsScreen.imageWidth / 2, 10, 
                            oneOf.fontFamily(), 10, Color.red, 1);

                    //save image
                    SgtUtil.saveImage(bufferedImage, oneOf.fullPublicDirectory() + imageFileName);

                    //store a copy in imageStoreDirectory
                    if (!oneOf.displayDiagnosticInfo() &&
                        imageStoreRootDirectory != null) 
                        File2.copy(oneOf.fullPublicDirectory() + imageFileName,
                                         imageStoreDirectory() + imageFileName);
                }

                //make a copy of the gif under a special name  (so ImageMagick regex can find them in order)
                String specialName = oneOf.fullPrivateDirectory() + 
                    randomInt + String2.zeroPad("" + frame, 3) + //3 zeroPadded digits so they will be sorted by frame number
                    imageFileName;
                File2.copy(
                    oneOf.fullPublicDirectory() + imageFileName,
                    specialName);
                specialImageNames.add(specialName);


                //advance currentSeconds
                currentSeconds += Calendar2.SECONDS_PER_HOUR;
            }

            //ensure back to the original values
            currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
            currentsScreen.time.setValue(session, centeredTimeValue);

            //process that (to synchronize other things)
            step.i = 1;
            rowNumber.i = 0;
            currentsScreen.validate(session, -1, step, rowNumber, htmlSB); //-1=don't show

            //generate the animated Gif with ImageMagick
            long imTime = System.currentTimeMillis();
            SSR.dosOrCShell("convert -delay " + delay + " " +
                "-loop 1 " + //defines how many times the animation should loop
                oneOf.fullPrivateDirectory() + randomInt + "*" + OneOf.imageExtension + " " + 
                oneOf.fullPublicDirectory() + animatedGifFileName, 120);  
            if (oneOf.verbose()) 
                String2.log("  animated gif generated in time=" + (System.currentTimeMillis() - imTime) + " ms");

            //delete the specially named gif files
            for (int i = 0; i < specialImageNames.size(); i++)
                File2.delete(specialImageNames.get(i));

            //done
            if (oneOf.verbose()) 
                String2.log("  createAnimatedGif done. TOTAL TIME=" + (System.currentTimeMillis() - time) + " ms");
        } catch (Exception e) {
            String2.log(errorInMethod + MustBe.throwableToString(e));

            //delete the specially named gif files
            for (int i = 0; i < specialImageNames.size(); i++)
                File2.delete(specialImageNames.get(i));

            //delete the animatedGif file
            File2.delete(oneOf.fullPublicDirectory() + animatedGifFileName);  

            //go back to the original values
            currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
            currentsScreen.time.setValue(session, centeredTimeValue);

            //process that (to synchronize other things)
            step.i = 1;
            rowNumber.i = 0;
            currentsScreen.validate(session, -1, step, rowNumber, htmlSB); //-1=don't show

            throw e;
        }

        return animatedGifFileName;
    }

    /**
     * This creates an animated gif with drifters from a model.
     *
     * @param session
     * @param htmlSB
     * @param startTime
     * @param pixelX the drifter's initial x location
     * @param pixelY the drifter's initial y location
     * @param strictModel If true, data must be available or the drifter disappears.
     *    If false, nearest neighbor data is allowed.
     * @param nHoursBack e.g., 24
     * @return the name of the animated gif file (without directory, but with .gif extension)
     * @throws Exception if trouble
     */
    private String createDrifterAnimatedGif(HttpSession session, StringBuffer htmlSB, 
            long startTime, int pixelX, int pixelY, boolean strictModel, 
            int nHoursBack) throws Exception {

        String2.log("\ncreateDrifterAnimatedGif");
        String errorInMethod = OneOf.ERROR + " in createDrifterAnimatedGif:\n";
        long time = System.currentTimeMillis();
        int delay = 100; //usually 100; in 1/100th's second   200 for testing
        int show = -1;
        IntObject step = new IntObject(1);
        IntObject rowNumber = new IntObject(0);
        double minX = currentsScreen.minX;
        double maxX = currentsScreen.maxX;
        double minY = currentsScreen.minY; 
        double maxY = currentsScreen.maxY;
        boolean makeLonPM180 = !DataHelper.lonNeedsToBe0360(minX, maxX);
        if (doTally) 
            oneOf.tally().add("Animated Gifs Displayed:", 
                strictModel? "Non-Interpolating Drifter Model" :
                    "Interpolating Drifter Model");

        //generate the animatedGifFileName
        String animatedGifFileName = 
            "Drifter" + nHoursBack + "_x" + pixelX + "y" + pixelY + "_" + 
            (strictModel? "N" : "I") + 
            "_" + currentsScreen.startImageFileName + ".gif"; //always .gif

        //already exists?
        if (File2.touch(oneOf.fullPublicDirectory() + animatedGifFileName)) {
            if (oneOf.verbose()) 
                String2.log("  reusing: " + animatedGifFileName);
            return animatedGifFileName;
        } else {
            if (oneOf.verbose())
                String2.log("  creating: " + animatedGifFileName);
        }

        //generate one map just to find out pixel location of x and y axis on image
        //make an image
        BufferedImage bufferedImage = 
            SgtUtil.getBufferedImage(currentsScreen.imageWidth, currentsScreen.imageHeight);
        Graphics2D g2d = (Graphics2D)bufferedImage.getGraphics();
        //make a map
        ArrayList mapResults = makeMap(
            oneOf.lowResLogoImageFile(), g2d,  
            0, 0, currentsScreen.imageWidth, currentsScreen.imageHeight,
            1);    //standard font size
        IntArray graphLocation = (IntArray)mapResults.get(6);
        int originXPixel = graphLocation.get(0);
        int endXPixel = graphLocation.get(1);
        int originYPixel = graphLocation.get(2);
        int endYPixel = graphLocation.get(3);
        double lon0 = minX + ((maxX - minX) * (pixelX - originXPixel)) / (endXPixel - originXPixel);
        double lat0 = minY + ((maxY - minY) * (pixelY - originYPixel)) / (endYPixel - originYPixel);
        String originalVectorDataSetValue = currentsScreen.vectorDataSetValue;
        String2.log("  drifter lon0=" + lon0 + " lat0=" + lat0);

        //generate the magic number  so ImageMagick regex can find the files
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //make the land mask grid  
        //first get xVectorDataSet range and resolution by making a grid
        Grid landMask = currentsScreen.xVectorDataSet.makeGrid(
            currentsScreen.timePeriodValue, currentsScreen.timeValue,          
            minX, maxX, minY, maxY, Integer.MAX_VALUE, Integer.MAX_VALUE);

        //then make a bathymetry Grid based on that info  (etopo2 docs say: land is >= 0,  ocean is < 0)
        int nLon = landMask.lon.length;
        int nLat = landMask.lat.length;
        landMask = SgtMap.createBathymetryGrid(oneOf.fullPrivateDirectory(),
            landMask.lon[0], landMask.lon[nLon - 1],
            landMask.lat[0], landMask.lat[nLat - 1],
            nLon, nLat);
        //String2.log("landMask minX=" + landMask.lon[0] + " maxX=" + landMask.lon[nLon - 1] + 
        //    " minY=" + landMask.lat[0] + " minY=" + landMask.lat[nLat - 1]);

        //run the drifter model
        DrifterModel.verbose = true;
        String[] activeVectorTimeOptions = (String[])currentsScreen.activeVectorTimePeriodTimes.get(
            currentsScreen.timePeriodIndex);
        Table modelResults = DrifterModel.run(strictModel, oneOf.fullPrivateDirectory(), 
            lon0, lat0, 0.01, 
            makeLonPM180, landMask,
            currentsScreen.xVectorDataSet, currentsScreen.yVectorDataSet, 
            currentsScreen.timePeriodValue, //used to figure out which type of data to use from x/yVectorDataSet 
            currentsScreen.timeValue,
            activeVectorTimeOptions,
            nHoursBack); 
        int modelRow = 0;

        //create currentSeconds  (nHoursBack)
        String centeredTimeValue = currentsScreen.timeValue;
        double currentSeconds = Calendar2.isoStringToEpochSeconds(centeredTimeValue) - //throws exception if trouble
            Calendar2.SECONDS_PER_HOUR * nHoursBack;

        //generate special .gif's for last nHoursBack
        String siblingColor[] = 
            //see ImageMagick's list of colors via "convert -list color"
            //{"orange", "orange", "orange", "orange", "orange", "orange", "orange", "orange", "red"};
            {"orange", "yellow", "YellowGreen", "green", "cyan", "blue", "purple", "brown", "red"};
        StringArray specialImageNames = new StringArray();
        try { //so can clean up if exception thrown
            StringBuffer siblingSB[] = new StringBuffer[9]; //to hold the cumulative path info
            int tx[] = new int[9];
            int ty[] = new int[9];
            Arrays.fill(tx, Integer.MAX_VALUE); //important, so initial otx is mv
            Arrays.fill(ty, Integer.MAX_VALUE);
            for (int frame = 0; frame <= nHoursBack; frame++) {

                //set desired timeValue
                String currentSecondsString = Calendar2.epochSecondsToIsoStringSpace(currentSeconds);
                currentsScreen.time.setValue(session, currentSecondsString);
                String2.log("createDrifterAnimatedGif frame#" + frame + " for " + currentSecondsString);

                //process that (to synchronize other things)
                currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
                step.i = 1;
                rowNumber.i = 0;
                currentsScreen.validate(session, show, step, rowNumber, htmlSB);

                //check if desired time was matched exactly 
                if (!currentsScreen.timeValue.equals(currentSecondsString)) {
                    //don't display any vector data
                    currentsScreen.vectorDataSet.setValue(session, OneOf.NO_DATA);
                    step.i = 1;
                    rowNumber.i = 0;
                    currentsScreen.validate(session, show, step, rowNumber, htmlSB);
                }

                //generate the frame's image
                String imageFileName = currentsScreen.startImageFileName + OneOf.imageExtension;

                //use SGT to make the image (without drifters) (if it doesn't already exist)
                if (File2.touch(oneOf.fullPublicDirectory() + imageFileName)) {
                    if (oneOf.verbose()) 
                        String2.log("  reusing .gif publicDir: " + imageFileName);

                } else if (!oneOf.displayDiagnosticInfo() && 
                          imageStoreRootDirectory != null &&
                       File2.touch(imageStoreDirectory() + imageFileName)) {
                    File2.copy(imageStoreDirectory() + imageFileName,
                               oneOf.fullPublicDirectory() + imageFileName);
                    if (oneOf.verbose()) 
                        String2.log("reusing .gif imageStoreDir: " + imageFileName);

                } else {
                    if (oneOf.verbose()) 
                        String2.log("  makeMap new imageFileName=" + imageFileName);

                    //make the image
                    bufferedImage = 
                        SgtUtil.getBufferedImage(currentsScreen.imageWidth, currentsScreen.imageHeight);
                    g2d = (Graphics2D)bufferedImage.getGraphics();

                    //makeMap
                    makeMap(
                        oneOf.lowResLogoImageFile(), g2d,  
                        0, 0, currentsScreen.imageWidth, currentsScreen.imageHeight,
                        1);    //standard font size

                    //get the 'warning' message
                    String warning = getWarning();
                    if (warning != null) 
                        AttributedString2.drawHtmlText(g2d, warning, currentsScreen.imageWidth / 2, 10, 
                            oneOf.fontFamily(), 10, Color.red, 1);

                    //saveAsGif
                    SgtUtil.saveImage(bufferedImage, oneOf.fullPublicDirectory() + imageFileName);

                    //store a copy in imageStoreDirectory
                    if (!oneOf.displayDiagnosticInfo() &&
                        imageStoreRootDirectory != null) 
                        File2.copy(oneOf.fullPublicDirectory() + imageFileName,
                            imageStoreDirectory() + imageFileName);
                }

                //add drifters to gif and 
                //  make a copy of the image under a special name  (so ImageMagick regex can find them in order)
                String oldName = oneOf.fullPublicDirectory() + imageFileName;
                String specialName = oneOf.fullPrivateDirectory() + 
                    randomInt + String2.zeroPad("" + frame, 3) + //3 zeroPadded digits so they will be sorted by frame number
                    imageFileName;
                specialImageNames.add(specialName);


                //draw all the tracer lines  (including calculating current tx ty for each sibling)
                StringBuffer sb = new StringBuffer("convert -linewidth 1 ");
                int nLiveSiblings = 0;
                for (int sibling = 0; sibling < 9; sibling++) {
                    int otx = tx[sibling];
                    int oty = ty[sibling];
                    double tLon = modelResults.getDoubleData(0, modelRow);
                    double tLat = modelResults.getDoubleData(1, modelRow);
                    tx[sibling] = Math2.roundToInt(originXPixel + ((endXPixel - originXPixel) * (tLon - minX)) / (maxX - minX));
                    ty[sibling] = Math2.roundToInt(originYPixel + ((endYPixel - originYPixel) * (tLat - minY)) / (maxY - minY));
                    double tSeconds = modelResults.getDoubleData(2, modelRow);
                    if (tSeconds != currentSeconds) 
                        Test.ensureEqual(tSeconds, currentSeconds, 
                            errorInMethod + "Model time and image time don't match.");
                    int tSibling = modelResults.getIntData(3, modelRow);
                    if (tSibling != sibling) 
                        Test.ensureEqual(tSibling, sibling, 
                            errorInMethod + "Model sibling# and image sibling# don't match.");
                    String2.log("frame=" + frame + " sib=" + sibling + " row=" + modelRow + 
                        " tLon=" + tLon + " tLat=" + tLat + " x=" + tx[sibling] + " y=" + ty[sibling]);
                    modelRow++;

                    if (tx[sibling] < Integer.MAX_VALUE && ty[sibling] < Integer.MAX_VALUE) {
                        nLiveSiblings++;

                        //add a line segment to siblingSB?
                        if (otx < Integer.MAX_VALUE && oty < Integer.MAX_VALUE) {
                            //draw the tracer line segment to last frame's tx,ty
                            if (siblingSB[sibling] == null) 
                                siblingSB[sibling] = new StringBuffer("-stroke " + siblingColor[sibling] + " ");
                            siblingSB[sibling].append(                                
                                "-draw 'line " + otx + "," + oty + " " + 
                                tx[sibling] + "," + ty[sibling] + "' ");
                        }

                    }

                    //whether sibling still alive or not, if siblingSB has any info, draw it on this frame
                    if (siblingSB[sibling] != null)
                        sb.append(siblingSB[sibling]);

                }
                if (nLiveSiblings == 0) 
                    sb.append("-stroke none -fill red -draw 'text 30,40 \"" + 
                        missingDataCausedModelToStop + "\"' ");

                //draw all the drifter rectangles
                for (int sibling = 0; sibling < 9; sibling++) {
                    if (tx[sibling] < Integer.MAX_VALUE && ty[sibling] < Integer.MAX_VALUE) {
                        sb.append("-fill " + siblingColor[sibling] + " ");
                        sb.append("-stroke " + siblingColor[sibling] + " ");
                        sb.append("-draw 'rectangle " + 
                            (tx[sibling]-2) + "," + (ty[sibling]-2) + " " + 
                            (tx[sibling]+2) + "," + (ty[sibling]+2) + "' ");
                    }
                }
                //String2.log("sb=" + sb);
                SSR.dosOrCShell(sb.toString() + oldName + " " + specialName, 20);  

                //advance currentSeconds
                currentSeconds += Calendar2.SECONDS_PER_HOUR;
            }

            //ensure back to the original values
            currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
            currentsScreen.time.setValue(session, centeredTimeValue);

            //process that (to synchronize other things)
            step.i = 1;
            rowNumber.i = 0;
            currentsScreen.validate(session, -1, step, rowNumber, htmlSB); //-1=don't show

            //generate the animated Gif with ImageMagick
            long imTime = System.currentTimeMillis();
            SSR.dosOrCShell("convert -delay " + delay + " " +
                "-loop 1 " + //defines how many times the animation should loop
                oneOf.fullPrivateDirectory() + randomInt + "*" + OneOf.imageExtension + " " + 
                oneOf.fullPublicDirectory() + animatedGifFileName, 120);  
            if (oneOf.verbose()) 
                String2.log("  animated gif generated in " + (System.currentTimeMillis() - imTime) + " ms");

            //delete the specially named gif files
            for (int i = 0; i < specialImageNames.size(); i++)
                File2.delete(specialImageNames.get(i));

            //done
            if (oneOf.verbose()) 
                String2.log("  createDrifterAnimatedGif done. TOTAL TIME=" + (System.currentTimeMillis() - time) + " ms");
        } catch (Exception e) {
            String2.log(errorInMethod + MustBe.throwableToString(e));

            //delete the specially named gif files
            for (int i = 0; i < specialImageNames.size(); i++)
                File2.delete(specialImageNames.get(i));

            //delete the animatedGif file
            File2.delete(oneOf.fullPublicDirectory() + animatedGifFileName);  

            //go back to the original values
            currentsScreen.vectorDataSet.setValue(session, originalVectorDataSetValue);
            currentsScreen.time.setValue(session, centeredTimeValue);

            //process that (to synchronize other things)
            step.i = 1;
            rowNumber.i = 0;
            currentsScreen.validate(session, -1, step, rowNumber, htmlSB); //-1=don't show

            throw e;
        }
        return animatedGifFileName;
    }

        
        

    /**
     * This uses SgtMap to make a map.
     *
     * @param logoFileName
     * @param g2D
     * @param ulx the upperleft x pixel for the active area
     * @param uly the upperleft y pixel for the active area
     * @param areaWidth the width of the active area (graph, labels, legend)
     * @param areaHeight the height of the active area (graph, labels, legend)
     * @param fontScale the relative font scale (1 = normal)
     * @return ArrayList with info about where the GraphDataLayer markers were plotted 
     *   (for generating the user map on the image:
     *   0=IntArray minX, 1=IntArray maxX, 2=IntArray minY, 3=IntArray maxY,
     *   4=IntArray rowNumber 5=IntArray whichPointScreen(0,1,2,...)),
     *   and pixel location of graph 6=IntArray originX,endX,originY,endY.
     *    For 0..5, if no graphDataLayers or no visible stations, these will exist but have size()=0.
     * @throws Exception if trouble
     */
    public ArrayList makeMap(String logoFileName, Graphics2D g2D, int ulx, int uly, 
        int areaWidth, int areaHeight, double fontScale) 
        throws Exception {

        if (oneOf.verbose()) String2.log("\n////******** CCUser.makeMap");
        long time = System.currentTimeMillis();
        double minX = currentsScreen.minX;
        double maxX = currentsScreen.maxX;
        double minY = currentsScreen.minY; 
        double maxY = currentsScreen.maxY;

        int graphArea[] = SgtMap.predictGraphSize(fontScale, areaWidth, areaHeight,
            minX, maxX, minY, maxY);
        int graphWidth = graphArea[0];
        int graphHeight = graphArea[1];

        //add the vector graphDataLayer
        ArrayList graphDataLayers = new ArrayList();
        if (currentsScreen.plotVectorData) {
            String standardVector = oneOf.vectorInfo()[currentsScreen.vectorAbsoluteDataSetIndex][OneOf.VISize];
            CompoundColorMap ccm = new CompoundColorMap(
                oneOf.fullPaletteDirectory(), 
                vectorPalette, vectorScale, vectorColorBarMin, vectorColorBarMax,
                vectorNSections, vectorContinuous, oneOf.fullPrivateDirectory());

            //Further reduce graphWidth and Height by /5 (nice round number)
            //  since vectors are always sparse.
            //This reduces nPoints significantly (1/25), 
            //but leaves flexibility of decimation process in sgtMap.makeMap(),
            //which typically wants ~1/20.
            String tUnits = oneOf.vectorInfo()[currentsScreen.vectorAbsoluteDataSetIndex][OneOf.VIUnits];
            graphDataLayers.add(new GraphDataLayer(-1, 
                -1, -1, -1, -1, -1, GraphDataLayer.DRAW_GRID_VECTORS, false, false,
                "", tUnits, //x,y axis title
                oneOf.vectorInfo()[currentsScreen.vectorAbsoluteDataSetIndex][OneOf.VIBoldTitle],
                "(" + tUnits + ") " + currentsScreen.vectorLegendTime, //title2
                currentsScreen.xVectorDataSet.courtesy.length() == 0? 
                    null : "Data courtesy of " + currentsScreen.xVectorDataSet.courtesy, 
                null, //title4
                null,
                currentsScreen.getXGrid(minX, maxX, minY, maxY, graphWidth/5, graphHeight/5), 
                currentsScreen.getYGrid(minX, maxX, minY, maxY, graphWidth/5, graphHeight/5), 
                ccm, Color.green,
                GraphDataLayer.MARKER_TYPE_NONE, 0, 
                String2.parseDouble(standardVector), // standardVector=e.g. 10m/s 
                -1));
        }

        if (currentsScreen.stationVectorGraphDataLayer != null)
            graphDataLayers.add(currentsScreen.stationVectorGraphDataLayer);
        Grid gridGrid = currentsScreen.getGridGrid(minX, maxX, minY, maxY, graphWidth, graphHeight);

        ArrayList results = SgtMap.makeMap(
            SgtUtil.LEGEND_BELOW,
            oneOf.legendTitle1(),
            oneOf.legendTitle2(),
            oneOf.fullContextDirectory() + "images/", 
            logoFileName,
            minX, maxX, minY, maxY,
            Browser.drawLandAsMask,

            currentsScreen.plotGridData || currentsScreen.plotBathymetryData, 
            gridGrid, 
            1, //scaleFactor
            currentsScreen.gridAltScaleFactor,
            currentsScreen.gridAltOffset,
            currentsScreen.fullDataSetCptName, 
            currentsScreen.gridBoldTitle,
            SgtUtil.getNewTitle2(
                currentsScreen.gridUnitsValue,
                currentsScreen.gridLegendTime, 
                ""),
            currentsScreen.gridCourtesy == null || currentsScreen.gridCourtesy.length() == 0? 
                null : "Data courtesy of " + currentsScreen.gridCourtesy,
            "",

            false, //contourScreen.plotData,
            null, //contourGrid,
            1, //scaleFactor
            1,
            0,
            "1000", 
            Color.black,
            null,
            null,
            null,
            null,
            null,

            graphDataLayers,
            g2D,
            ulx, uly, areaWidth, areaHeight,
            0,   //no coastline adjustment
            fontScale); 

        if (oneOf.verbose()) String2.log("\\\\\\\\******** CCUser.makeMap done. TOTAL TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
        return results;
    }



}
