/* 
 * CurrentsScreen Copyright 2005, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch;

import com.cohort.array.StringArray;
import com.cohort.ema.*;
import com.cohort.util.Calendar2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.*;
import gov.noaa.pfel.coastwatch.pointdata.*;
import gov.noaa.pfel.coastwatch.sgt.CompoundColorMap;
import gov.noaa.pfel.coastwatch.sgt.GraphDataLayer;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.util.IntObject;
import gov.noaa.pfel.coastwatch.util.SSR;

import java.awt.Color;
import java.util.GregorianCalendar;
import java.util.Vector;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/**
 * This handles the user interface for the Current screen (the only
 * screen for CCBrowser).
 * Each web page request will generate a new instance of this.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2005-09-12
 */
public class CurrentsScreen extends Screen {
    public final static int imageSizeIndex = 1;

    public EmaSelect  region, dataSet, vectorDataSet, timePeriod,  //dataSet is for gridData
        animationN, modelN, interpolating; 
    public EmaDateTimeSelect2 time;
    public EmaButton  viewAnimation, viewLastDrifterAnimation;

    //set by the constructor
    public String regionLabel, dataSetLabel, vectorDataSetLabel, 
        timePeriodLabel, timeLabel,
        animationNLabel, animationNPre, animationNPost, viewAnimationWarning, 
        interpolatingLabel, interpolatingPre, modelNPre, modelNPost,
        gridGetLabel, vectorGetLabel;
    public int dataSetShowN;
    public int imageWidth, imageHeight;
    public String regionInfo[][];
    public String pointVectorName;  //e.g. "Currents, Near Real Time (MBARI)"
    public double pointVectorMinDepth, pointVectorMaxDepth;
    public String pointVectorPalette;
    public String pointVectorScale;
    public double pointVectorColorBarMin;
    public double pointVectorColorBarMax;
    public int    pointVectorNSections;
    public boolean pointVectorContinuous;

    //set by validate
    String regionValue;
    double minX, maxX, minY, maxY;
    String cWESNString;  

    public boolean plotGridData;
    public boolean plotBathymetryData;
    public String gridInternalName;
    public String gridBoldTitle;
    public String gridCourtesy;
    public double gridAltScaleFactor, gridAltOffset;    

    public String dataSetValue;
    public int dataSetIndex;
    private GridDataSet gridDataSet;
    public GregorianCalendar gridStartCalendar, gridEndCalendar;
    public String gridLegendTime;
//    public String gridImageResBaseFileName; //the start of the file name (without min/max/X/Y or width height or .grd)

    public boolean plotVectorData;
    public String vectorDataSetValue;
    public int vectorDataSetIndex;
    public int vectorAbsoluteDataSetIndex;
    public GridDataSet xVectorDataSet, yVectorDataSet;
    public GregorianCalendar vectorStartCalendar, vectorEndCalendar;
//    public String xImageResBaseFileName; //the start of the file name (without min/max/X/Y or width height or .grd)
//    public String yImageResBaseFileName; //the start of the file name (without min/max/X/Y or width height or .grd)
    public boolean xDataAccessAllowed, yDataAccessAllowed;
    public String vectorLegendTime;
    
    public int timePeriodIndex;
    public Vector activeVectorTimePeriodTimes; 
    public String timePeriodValue;

    public String timeValue;
    public int timeIndex;

    public String gridTimePeriodValue;
    public String gridTimeValue;
    public int    gridUnitsIndex;
    public String gridUnitsValue;
    public boolean gridDataAccessAllowed;
    public String paletteValue;
    public String scaleValue;
    public String paletteMinValue;
    public String paletteMaxValue;
    public String fullDataSetCptName;

    public String animationNValue;
    public String modelNValue;    
    public String interpolatingValue;

    public GraphDataLayer stationVectorGraphDataLayer;

    public String startImageFileName;




    /** 
     * The constructor for CurrentsScreen. 
     *
     * @param editOption the edit option number for this screen
     * @param oneOf the oneOf class
     * @param shared the shared class 
     * @param emaClass the emaClass to which this belongs
     * @param doTally indicates if activity of this user should be tallied
     *     (see dontTallyIPAddresses in properties file)
     */
    public CurrentsScreen(int editOption, OneOf oneOf, Shared shared, EmaClass emaClass, boolean doTally) {
        this.editOption = editOption;
        this.oneOf = oneOf;
        this.shared = shared;
        this.emaClass = emaClass;
        this.doTally = doTally;
        ResourceBundle2 classRB2 = oneOf.classRB2(); 
        String errorInMethod = String2.ERROR + " in CurrentsScreen(): ";

        //add the components to the emaClass
        //The names (e.g., "dataSet") are not prefixed by "currents" because there is only one screen in CCBrowser.
        emaClass.addAttribute(region          = new EmaSelect(emaClass, "region")); 
        emaClass.addAttribute(dataSet         = new EmaSelect(emaClass, "dataSet")); 
        emaClass.addAttribute(vectorDataSet   = new EmaSelect(emaClass, "vectorDataSet")); 
        emaClass.addAttribute(timePeriod      = new EmaSelect(emaClass, "timePeriod"));
        emaClass.addAttribute(time            = new EmaDateTimeSelect2(emaClass, "time"));
        emaClass.addAttribute(animationN      = new EmaSelect(emaClass, "animationN")); 
        emaClass.addAttribute(viewAnimation   = new EmaButton(emaClass, "viewAnimation")); 
        emaClass.addAttribute(interpolating   = new EmaSelect(emaClass, "interpolating")); 
        emaClass.addAttribute(modelN          = new EmaSelect(emaClass, "modelN")); 
        //emaClass.addAttribute(viewLastDrifterAnimation = new EmaButton(emaClass, "viewLastDrifterAnimation"));

        //get all the labels 
        regionLabel          = classRB2.getNotNullString("region.label", errorInMethod);
        dataSetLabel         = classRB2.getNotNullString("dataSet.label", errorInMethod);
        vectorDataSetLabel   = classRB2.getNotNullString("vectorDataSet.label", errorInMethod);
        timePeriodLabel      = classRB2.getNotNullString("timePeriod.label", errorInMethod);
        timeLabel            = classRB2.getNotNullString("time.label", errorInMethod);
        animationNLabel      = classRB2.getNotNullString("animationN.label", errorInMethod);
        animationNPre        = classRB2.getNotNullString("animationNPre", errorInMethod);
        animationNPost       = classRB2.getNotNullString("animationNPost", errorInMethod);
        viewAnimationWarning = classRB2.getNotNullString("viewAnimationWarning", errorInMethod);
        interpolatingLabel   = classRB2.getNotNullString("interpolating.label", errorInMethod);
        interpolatingPre     = classRB2.getNotNullString("interpolatingPre", errorInMethod);
        modelNPre            = classRB2.getNotNullString("modelNPre", errorInMethod);
        modelNPost           = classRB2.getNotNullString("modelNPost", errorInMethod);
        gridGetLabel         = classRB2.getNotNullString("gridGet.label", errorInMethod);
        vectorGetLabel       = classRB2.getNotNullString("gridVectorGet.label", errorInMethod);

        pointVectorName        = classRB2.getString("pointVector.name", null); //may be null
        pointVectorMinDepth    = classRB2.getDouble("pointVector.minDepth", 21.5);
        pointVectorMaxDepth    = classRB2.getDouble("pointVector.maxDepth", 23.5);
        pointVectorPalette     = classRB2.getString("pointVector.palette", "Rainbow");
        pointVectorScale       = classRB2.getString("pointVector.scale", "Linear");
        pointVectorColorBarMin = classRB2.getDouble( "pointVector.colorBarMin", 0);
        pointVectorColorBarMax = classRB2.getDouble( "pointVector.colorBarMax", 1);
        pointVectorNSections   = classRB2.getInt(    "pointVector.nSections", 5);
        pointVectorContinuous  = classRB2.getBoolean("pointVector.continuous", false);

        //read oneOf things
        dataSetShowN = oneOf.classRB2().getInt("dataSetShowN", 1000000);

        imageWidth  = oneOf.imageWidths()[imageSizeIndex];
        imageHeight = oneOf.imageHeights()[imageSizeIndex];

        regionInfo = oneOf.regionInfo();
    }

    /**
     * This validates the user's choices for this screen, and, if htmlSB isn't null,
     * adds the html code for the screen to htmlSB.
     *
     * @param session the user's session
     * @param showEditOption  =0 for CurrentsScreen or &lt;0 for responses to ViewAnimation buttons 
     * @param step the next step number on the form (e.g., 1), 2), 3), ...).
     * @param rowNumber the number of the next row in the html form.
     * @param htmlSB the StringBuffer collecting the html to be sent to the users.
     */
    public void validate(HttpSession session, int showEditOption, IntObject step, 
            IntObject rowNumber, StringBuffer htmlSB) throws Exception {

        //use the latest info
        //restrict the activeGridDataSetOptions to not show the vector component datasets
        String originalAGDSO[] = shared.activeGridDataSetOptions();
        int tempN = originalAGDSO.length;
        int showN = Math.min(tempN, dataSetShowN);
        String activeGridDataSetOptions[] = new String[showN];
        System.arraycopy(originalAGDSO, 0, activeGridDataSetOptions, 0, showN);
        dataSet.setOptions(activeGridDataSetOptions); 
        vectorDataSet.setOptions(shared.activeVectorOptions());

        //grid (satellite) dataSet
        dataSetValue = dataSet.getValue(session);
        //String2.log("dataSet inSession=" + dataSetValue + 
        //    " default=" + dataSet.getDefaultValue() +
        //    "\n  options=" + String2.toCSVString(shared.activeGridDataSetOptions()));
        dataSetIndex = dataSet.indexOf(dataSetValue);
        if (dataSetIndex < 0) {
            dataSetIndex = 0; //none
            dataSetValue = dataSet.getOption(dataSetIndex);
            dataSet.setValue(session, dataSetValue);
        }
        if (oneOf.verbose()) 
            String2.log("currentsScreen dataSetValue = " + dataSetValue);
        if (showEditOption == editOption) 
            addTableEntry(dataSet, dataSetLabel, dataSetValue, rowNumber, step, htmlSB);

        //vectorDataSet   
        //Ideally, None option wouldn't be shown.
        //  But drifter animation needs None option if time is not exact match.
        xVectorDataSet = null;
        yVectorDataSet = null;
        vectorDataSetValue = vectorDataSet.getValue(session);
        String2.log("  vectorDataSet inSession=" + vectorDataSetValue +
            " default=" + vectorDataSet.getDefaultValue() 
            //+ "\n  options=" + String2.toCSVString(shared.activeVectorOptions())
            );
        vectorDataSetIndex = vectorDataSet.indexOf(vectorDataSetValue);
        if (vectorDataSetIndex < 0) {
            vectorDataSetValue = vectorDataSet.getDefaultValue();
            vectorDataSetIndex = vectorDataSet.indexOf(vectorDataSetValue);
            if (vectorDataSetIndex < 0) {
                vectorDataSetIndex = 0;
                vectorDataSetValue = vectorDataSet.getOption(vectorDataSetIndex);
            }
            vectorDataSet.setValue(session, vectorDataSetValue);
        }
        plotVectorData = vectorDataSetIndex > 0;
        if (oneOf.verbose()) 
            String2.log("  vectorDataSetValue = " + vectorDataSetValue + " index=" + vectorDataSetIndex +
               "  xIndexes=" + String2.toCSVString(shared.activeVectorXDataSetIndexes()));
        if (showEditOption == editOption) 
            addTableEntry(vectorDataSet, vectorDataSetLabel, vectorDataSetValue, rowNumber, step, htmlSB);
        vectorAbsoluteDataSetIndex = String2.indexOf(oneOf.vectorOptions(), vectorDataSetValue); //this 'absolute' is okay
        if (plotVectorData) {
            xVectorDataSet = (GridDataSet)shared.activeGridDataSets().get(shared.activeVectorXDataSetIndexes()[vectorDataSetIndex]);
            yVectorDataSet = (GridDataSet)shared.activeGridDataSets().get(shared.activeVectorYDataSetIndexes()[vectorDataSetIndex]);
            //String2.log("currentsScreen xVectorDataSet=" + xVectorDataSet.internalName);
        }
        
        //region   is set silently based on the vectorDataset
        regionValue = null;
        if (xVectorDataSet == null) {}
        else if (xVectorDataSet.internalName.substring(1, 3).equals("CA")) regionValue = "Ano Nuevo";
        else if (xVectorDataSet.internalName.substring(1, 3).equals("CB")) regionValue = "Bodega Bay";
        else if (xVectorDataSet.internalName.substring(1, 3).equals("C1")) regionValue = "Golden Gate";
        else if (xVectorDataSet.internalName.substring(1, 3).equals("CG")) regionValue = "Golden Gate";
        else if (xVectorDataSet.internalName.substring(1, 3).equals("CM")) regionValue = "Monterey Bay";
        else if (xVectorDataSet.internalName.substring(1, 3).equals("CS")) regionValue = "San Francisco Bay";
        //else no change
        if (regionValue != null) region.setValue(session, regionValue);

        //regular region stuff 
        int regionIndex = region.indexOf(region.getValue(session));
        if (regionIndex < 0) {
            regionIndex = region.indexOf(region.getDefaultValue());
            region.setValue(session, region.getDefaultValue());            
        }
        regionValue = region.getValue(session);
        minX = String2.parseDouble(regionInfo[regionIndex][1]);
        maxX = String2.parseDouble(regionInfo[regionIndex][2]);
        minY = String2.parseDouble(regionInfo[regionIndex][3]);
        maxY = String2.parseDouble(regionInfo[regionIndex][4]);
        cWESNString = FileNameUtility.makeWESNString(minX, maxX, minY, maxY);  
        //String2.log("cWESNString=" + cWESNString);
        if (oneOf.verbose()) 
            String2.log("  region = " + regionValue);

        //tally 
        if (doTally) {
            oneOf.tally().add("Gif Size Usage:", oneOf.imageSizeOptions()[imageSizeIndex]);
            oneOf.tally().add("Region Usage:", regionValue);
            oneOf.tally().add("Vector Data Set Usage:", vectorDataSetValue);
        }

        //timePeriod   options are for vector/currents data, not grid
        timePeriodIndex = -1;
        activeVectorTimePeriodTimes  = null; 
        timePeriodValue = null;
        if (plotVectorData) { 

            Object oar[] = (Object[])shared.activeVectorContents().get(vectorDataSetIndex);
            String[] activeVectorTimePeriodOptions = (String[])oar[0];
            String[] activeVectorTimePeriodTitles  = (String[])oar[1];
            activeVectorTimePeriodTimes            = (Vector)oar[2]; 

            //kludge:
            if (activeVectorTimePeriodOptions[0] == "1 observation") {
                activeVectorTimePeriodOptions[0] = "hourly";
                activeVectorTimePeriodTitles[1] = "Get the data from one hourly observation.";
            }
            timePeriod.setOptions(activeVectorTimePeriodOptions);
            timePeriod.setTitles(activeVectorTimePeriodTitles);
            timePeriodIndex = TimePeriods.closestTimePeriod(
                timePeriod.getValue(session), activeVectorTimePeriodOptions);
            timePeriodValue = activeVectorTimePeriodOptions[timePeriodIndex];
            timePeriod.setValue(session, timePeriodValue);
            if (oneOf.verbose())
                String2.log("  timePeriodValue = " + timePeriodValue);
            if (showEditOption == editOption && activeVectorTimePeriodOptions.length > 1) 
                addTableEntry(timePeriod, timePeriodLabel, timePeriodValue, rowNumber, step, htmlSB);
        }

        //time  options are for vector/currents data, not grid
        timeValue = null;
        timeIndex = -1;
        vectorStartCalendar = null;
        vectorEndCalendar = null;
//        xImageResBaseFileName = null;
//        yImageResBaseFileName = null;
        xDataAccessAllowed = false;
        yDataAccessAllowed = false;
        vectorLegendTime = null;
        if (plotVectorData) {
            String[] activeVectorTimeOptions = (String[])activeVectorTimePeriodTimes.get(timePeriodIndex);
            time.setOptions(activeVectorTimeOptions);
            timeValue = time.getValue(session);
            timeIndex = Calendar2.binaryFindClosest(activeVectorTimeOptions, 
                String2.replaceAll(timeValue, "T", " "));
            timeValue = activeVectorTimeOptions[timeIndex];
            time.setValue(session, timeValue);
            if (oneOf.verbose())
                String2.log("  timeValue = " + timeValue);
            if (showEditOption == editOption) 
                addTableEntry(time, timeLabel, timeValue, rowNumber, step, htmlSB);
            vectorStartCalendar = TimePeriods.getStartCalendar(timePeriodValue, timeValue, "");
            vectorEndCalendar   = TimePeriods.getEndCalendar(timePeriodValue, timeValue, "");

            //make fileNames
            //FILE_NAME_RELATED_CODE
            //kludge:
            if (xVectorDataSet.activeTimePeriodOptions[0] == "1 observation") {
                xVectorDataSet.activeTimePeriodOptions[0] = "hourly";
            }
            if (yVectorDataSet.activeTimePeriodOptions[0] == "1 observation") {
                yVectorDataSet.activeTimePeriodOptions[0] = "hourly";
            }
            int xTimePeriodIndex = String2.indexOf(xVectorDataSet.activeTimePeriodOptions, timePeriodValue);
            int yTimePeriodIndex = String2.indexOf(yVectorDataSet.activeTimePeriodOptions, timePeriodValue);
/*            xImageResBaseFileName = FileNameUtility.makeBaseName(
                xVectorDataSet.internalName,  //e.g., LATssta
                'S', //e.g., S=Standard units A=Alt units; base file is always 'S' 
                timePeriodValue, //e.g., 1 day
                timeValue);         
            yImageResBaseFileName = FileNameUtility.makeBaseName(
                yVectorDataSet.internalName, //e.g., LATssta
                'S', //e.g., S=Standard units A=Alt units; base file is always 'S'
                timePeriodValue, //e.g., 1 day
                timeValue);         
*/
            xDataAccessAllowed = xVectorDataSet.dataAccessAllowedCentered(timePeriodValue, 
                timeValue);
            yDataAccessAllowed = yVectorDataSet.dataAccessAllowedCentered(timePeriodValue, 
                timeValue);

            //make vectorLegendTime String 
            vectorLegendTime = xVectorDataSet.getLegendTime(timePeriodValue, timeValue);

        }

        //set gridDataSet, gridTimePeriod, gridTime, gridUnits, palette, paletteMin, paletteMax, scale
        //plotVectorData must be true for plotGridData to be true,
        //  since grid uses some vector info (e.g., time Period, time)
        plotGridData = dataSetIndex > 1 && plotVectorData; 
        plotBathymetryData = dataSetIndex == 1; 
        gridDataSet        = null;
        int gridTimePeriodIndex= -1;
        gridTimePeriodValue= null;
        gridTimeValue      = null;
        gridUnitsIndex     = -1;
        gridUnitsValue     = null;
        gridStartCalendar  = null;
        gridEndCalendar    = null;
        paletteValue       = null;
        scaleValue         = null;
        paletteMinValue    = null;
        paletteMaxValue    = null;
        fullDataSetCptName = null;
        startImageFileName= null; 
//        gridImageResBaseFileName = null;
        gridLegendTime = null;
        gridDataAccessAllowed  = false;
        gridInternalName = "None";
        gridBoldTitle = null;
        gridAltScaleFactor = 1; 
        gridAltOffset = 0;    
        if (plotGridData) { 
            gridDataSet = (GridDataSet)shared.activeGridDataSets().get(dataSetIndex);

            //synchronize timePeriod
            gridTimePeriodIndex = TimePeriods.closestTimePeriod( //synchronize
                timePeriodValue, gridDataSet.activeTimePeriodOptions);
            gridTimePeriodValue = gridDataSet.activeTimePeriodOptions[gridTimePeriodIndex];

            //synchronize time
            String gridTimeOptions[] = (String[])gridDataSet.activeTimePeriodTimes.get(gridTimePeriodIndex);
            int gridTimeIndex = Calendar2.binaryFindClosest( 
                gridTimeOptions, String2.replaceAll(timeValue, "T", " "));
            gridTimeValue = gridTimeOptions[gridTimeIndex];

            //calculate start/endCalendar
            gridStartCalendar = TimePeriods.getStartCalendar(gridTimePeriodValue, gridTimeValue, "");
            gridEndCalendar   = TimePeriods.getEndCalendar(gridTimePeriodValue, gridTimeValue, "");

            //get other default values
            gridUnitsIndex     = gridDataSet.defaultUnits == 'A'? 1 : 0;
            gridUnitsValue     = gridDataSet.unitsOptions[gridUnitsIndex];
            fullDataSetCptName = CompoundColorMap.makeCPT(
                oneOf.fullPaletteDirectory(), 
                //dave suggested grayscale, but it is to similar to mv (clouds), so use BlackBlueWhite
                "LightBlueWhite", //"BlackBlueWhite", //the default palette is: gridDataSet.palette, 
                gridDataSet.paletteScale, 
                String2.parseDouble(gridUnitsIndex <= 0? gridDataSet.paletteMin : gridDataSet.altPaletteMin), 
                String2.parseDouble(gridUnitsIndex <= 0? gridDataSet.paletteMax : gridDataSet.altPaletteMax),
                -1, false, oneOf.fullPrivateDirectory());

            //generate file names 
            //make file names easily parsed for specific values
            //FILE_NAME_RELATED_CODE
            String gridBaseFileName = FileNameUtility.makeBaseName(
                gridDataSet.internalName, //e.g., LATssta
                gridUnitsIndex == 1? 'A' : 'S', //e.g., A=Alternate units S=Standard units
                gridTimePeriodValue, //e.g., 1 day
                gridTimeValue);         
/*            gridImageResBaseFileName = FileNameUtility.makeBaseName(
                gridDataSet.internalName, //e.g., LATssta
                'S', //e.g., A=Alternate units S=Standard units //gridImageResFileName is always 'S' units
                gridTimePeriodValue, //e.g., 1 day
                gridTimeValue); 
*/
            startImageFileName = "_G" +
                Math2.reduceHashCode((gridBaseFileName + imageSizeIndex + cWESNString + fullDataSetCptName).hashCode()); //hashCode will be same for same info
            gridDataAccessAllowed = gridDataSet.dataAccessAllowedCentered(gridTimePeriodValue, 
                gridTimeValue);

            //make legendTime String 
            gridLegendTime = gridDataSet.getLegendTime(gridTimePeriodValue, gridTimeValue);

            gridInternalName = gridDataSet.internalName;
            gridBoldTitle = gridDataSet.boldTitle;
            gridCourtesy = gridDataSet.courtesy;
            if (gridUnitsIndex == 1) {
                gridAltScaleFactor = gridDataSet.altScaleFactor;
                gridAltOffset = gridDataSet.altOffset;    
            }
        } else if (plotBathymetryData) {
            //plotBathymetryData
            fullDataSetCptName = SgtMap.bathymetryCptFullName;
            String gridBaseFileName = SgtMap.BATHYMETRY_7NAME;         
            startImageFileName = "_G" +
                Math2.reduceHashCode((gridBaseFileName + cWESNString + imageSizeIndex).hashCode()); //hashCode will be same for same info
            gridDataAccessAllowed = true;
            gridLegendTime = "";

            gridInternalName = SgtMap.BATHYMETRY_7NAME;
            gridBoldTitle = SgtMap.BATHYMETRY_BOLD_TITLE + " (" + SgtMap.BATHYMETRY_UNITS + ")";
            gridCourtesy = SgtMap.BATHYMETRY_COURTESY;
            gridUnitsIndex = 0;
            gridUnitsValue = ""; //appended to title above
        } else {
            //!plotGridData and !plotBathymetryData
            startImageFileName = "_G" + Math2.reduceHashCode((imageSizeIndex + cWESNString).hashCode());
        } 

        //add vector info to startImageFileName
        if (plotVectorData) {
            String tVTime = Calendar2.removeSpacesDashesColons(timeValue);
            String xBaseFileName = 
                xVectorDataSet.internalName +  //e.g., LATssta
                "S" + //e.g., S=Standard units A=Alt units; base file is always 'S' 
                TimePeriods.getInFileName(timePeriodValue); //e.g., 1 day
                //"_" + tVTime;         
            startImageFileName = tVTime +  //so time is 1st thing and visible
                startImageFileName + "_V" + Math2.reduceHashCode(xBaseFileName.hashCode());
        }

        //add the mbari station currents pointVector graphDataLayer  
        stationVectorGraphDataLayer = null;
        boolean plotStationVectorData = false;
        if (plotVectorData && //otherwise, time not valid
            regionValue.equals("Monterey Bay") && pointVectorName != null &&
            pointVectorName.length() > 0) {
            //which option equals pointVectorName (e.g., "Currents, Near Real Time (MBARI)")?
            int whichPV = -1;
            for (int i = 0; i < shared.activePointVectorOptions().length; i++) {
                if (shared.activePointVectorOptions()[i].equals(pointVectorName)) {
                    whichPV = i;
                    break;
                }
            }
            if (whichPV == -1) {
                String2.log("WARNING: pointVectorName=" + pointVectorName + 
                    " not found.\n options=" + String2.toCSVString(shared.activePointVectorOptions()));
            } else {

                int absoluteWhichPV = shared.activePointVectorOriginalIndex(whichPV);

                PointDataSet uPointVectorDataSet = (PointDataSet)shared.activePointDataSets().get(shared.activePointVectorXDataSetIndexes()[whichPV]);
                PointDataSet vPointVectorDataSet = (PointDataSet)shared.activePointDataSets().get(shared.activePointVectorYDataSetIndexes()[whichPV]);

                //make the colorbar            
                String standardVector = oneOf.pointVectorInfo()[absoluteWhichPV][OneOf.PVISize];
                String fullPVCptName = CompoundColorMap.makeCPT(
                    oneOf.fullPaletteDirectory(), 
                    pointVectorPalette, pointVectorScale, 
                    pointVectorColorBarMin, pointVectorColorBarMax, 
                    pointVectorNSections, pointVectorContinuous, oneOf.fullPrivateDirectory());

                //get the data
                Table pvTable = PointVectors.makeAveragedTimeSeries(
                    uPointVectorDataSet, vPointVectorDataSet, minX, maxX,
                    minY, maxY, pointVectorMinDepth, pointVectorMaxDepth,
                    timeValue, timeValue, timePeriodValue); 
                if (oneOf.verbose()) String2.log("pointVectors nRows=" + pvTable.nRows() + 
                    " absoluteWhichPV=" + absoluteWhichPV + " for" + 
                    "\n  uPointVectorDataSet=" + uPointVectorDataSet +
                    "\n  vPointVectorDataSet=" + vPointVectorDataSet + 
                    "\n  minX=" + minX + 
                    " maxX=" + maxX +
                    " minY=" + minY +
                    " maxY=" + maxY +
                    "\n  pointVectorMinDepth=" + pointVectorMinDepth + 
                    " pointVectorMaxDepth=" + pointVectorMaxDepth +
                    "\n  timeValue" + timeValue +
                    " timePeriodValue=" + timePeriodValue); 
                if (pvTable.nRows() > 0) {
                    plotStationVectorData = true;

                    //modify startImageFileName
                    String pointVectorInfo = 
                        PointVectors.getAveragedTimeSeriesName(
                            oneOf.pointVectorInfo()[absoluteWhichPV][OneOf.PVIInternalName],
                            'S', minX, maxX, minY, maxY, 
                            pointVectorMinDepth, pointVectorMaxDepth,
                            timeValue, timeValue, timePeriodValue);
                    startImageFileName += "_P" +
                        Math2.reduceHashCode(pointVectorInfo.hashCode()); //hashCode will be same for same info
                    //String2.log("postHash startImageFileName=" + startImageFileName);

                    //make the graphDataLayer      
                    String pointVectorLegendTime = 
                        TimePeriods.getLegendTime(timePeriodValue, timeValue);
                    stationVectorGraphDataLayer = new GraphDataLayer(
                        -2, //-2 id's stationVector info
                        0, 1, 5, 6, -1, GraphDataLayer.DRAW_POINT_VECTORS, false, false,
                        "", oneOf.pointVectorInfo()[absoluteWhichPV][OneOf.PVIUnits], //x,y axis title
                        oneOf.pointVectorInfo()[absoluteWhichPV][OneOf.PVIBoldTitle],
                        //was: use dataset units (not vector units with std length) 
                        //  because dave wants colorbar range different from std length
                        //uPointVectorDataSet.unitsOptions[0], 
                        pointVectorLegendTime + "  " +
                            "Depth = " + 
                            (pointVectorMinDepth == pointVectorMaxDepth? "" :
                                String2.genEFormat6(pointVectorMinDepth) + " to ") +
                            String2.genEFormat6(pointVectorMaxDepth) + " meters.", //title2
                        uPointVectorDataSet.courtesy.length() == 0? 
                            null : "Data courtesy of " + uPointVectorDataSet.courtesy, //Point vector data
                        null, //title4
                        pvTable, null, null,
                        null, Color.cyan, //new CompoundColorMap(fullPVCptName), null,
                        GraphDataLayer.MARKER_TYPE_NONE, 0, 
                        String2.parseDouble(standardVector), // standardVector=e.g. 10m/s 
                        -1);
                }
            }
        }

        //display the animation options
        animationNValue = null;
        modelNValue = null;
        interpolatingValue = null;
        if (plotVectorData) {
            //animation
            animationNValue = animationN.getValue(session);
            int animationNIndex = animationN.indexOf(animationNValue);
            if (animationNIndex < 0) {
                animationNValue = animationN.getDefaultValue();
                animationN.setValue(session, animationNValue);
            }
            String2.log("  animationN value=" + animationNValue);
            if (showEditOption == editOption) {
                animationN.setLabel(String2.substitute(animationNLabel, "" + (step.i++), null, null));
                htmlSB.append(
                    "    " + oneOf.getBeginRowTag(Math2.odd(rowNumber.i++)) + "\n" + 
                    "      <td>" + animationN.getLabel() + "&nbsp;</td>\n" +
                    //width=99% is what keeps all of 2nd column of table to the left
                    "      <td width=\"99%\">" + animationNPre + " " +
                        animationN.getControl(animationN.getValue(session)) + 
                        " " + animationNPost + 
                        " " + viewAnimation.getControl(viewAnimation.getValue(session)) +
                        viewAnimationWarning + "</td>\n" +
                    "    " + emaClass.getEndRow() + "\n");
            }

            //view model info
            modelNValue = modelN.getValue(session);
            int modelNIndex = modelN.indexOf(modelNValue);
            if (modelNIndex < 0) {
                modelNValue = modelN.getDefaultValue();
                modelN.setValue(session, modelNValue);
            }
            String2.log("  modelN value=" + modelNValue);

            interpolatingValue = interpolating.getDefaultValue(); //for now: always non-interpolating
            int interpolatingIndex = interpolating.indexOf(interpolatingValue);
            interpolating.setValue(session, interpolatingValue);
            //interpolatingValue = interpolating.getValue(session);
            //int interpolatingIndex = interpolating.indexOf(interpolatingValue);
            //if (interpolatingIndex < 0) {
            //    interpolatingValue = interpolating.getDefaultValue();
            //    interpolatingIndex = interpolating.indexOf(interpolatingValue);
            //    interpolating.setValue(session, interpolatingValue);
            //}
            if (showEditOption == editOption) {
                interpolating.setLabel(String2.substitute(interpolatingLabel, "" + (step.i++), null, null));
                htmlSB.append(
                    "    " + oneOf.getBeginRowTag(Math2.odd(rowNumber.i++)) + "\n" + 
                    "      <td>" + interpolating.getLabel() + "&nbsp;</td>\n" +
                    "      <td>" + 
                        //interpolatingPre +
                        //interpolating.getControl(interpolating.getValue(session)) +
                        //modelNPre + 
                        "of the previous " +
                        modelN.getControl(modelN.getValue(session)) + 
                        modelNPost + "</td>\n" +
                    "    " + emaClass.getEndRow() + "\n");

                if (plotGridData)   //e.g., satellite data
                    GridScreen.showGetGridDataLinks(oneOf,  
                        htmlSB, rowNumber, step, 
                        gridGetLabel, gridDataSet, gridTimePeriodIndex, 
                        gridTimeValue, gridDataAccessAllowed, 
                        minX, maxX, minY, maxY);

                if (plotBathymetryData)  
                    GridScreen.showGetBathymetryDataLinks(oneOf, 
                        htmlSB, rowNumber, step, gridGetLabel, 
                        minX, maxX, minY, maxY);

                if (plotVectorData) { //e.g., HFRadar gridded vector data
                    int xTimePeriodIndex = String2.indexOf(xVectorDataSet.activeTimePeriodOptions, timePeriodValue);
                    VectorScreen.showGetGridVectorDataLinks(oneOf, session, 
                        htmlSB, rowNumber, step, vectorGetLabel, 
                        oneOf.vectorInfo()[vectorAbsoluteDataSetIndex][OneOf.PVIInternalName],
                        xVectorDataSet, yVectorDataSet, xTimePeriodIndex, timeValue,
                        xDataAccessAllowed && yDataAccessAllowed, emaClass, 
                        minX, maxX, minY, maxY, false);
                }

            }

        }

    }

    /**
     * This actually generates the satellite data's image-resolution grid,
     * so it should be called after validate() if/when you actually need to use the data.
     *
     * @param minLon
     * @param maxLon
     * @param minLat
     * @param maxLat
     * @param imageWidth  from oneOf.imageWidths()[imageSizeIndex]
     * @param imageHeight
     * @return the requested grid, as best it can
     * @throws Exception if trouble
     */
    public Grid getGridGrid(double minLon, double maxLon, 
            double minLat, double maxLat, int imageWidth, int imageHeight) throws Exception { 
        try {
            if (plotGridData) {
                return gridDataSet.makeGrid(
                    gridTimePeriodValue, gridTimeValue, 
                    minLon, maxLon, minLat, maxLat, imageWidth, imageHeight);
            } else if (plotBathymetryData) {
                return SgtMap.createBathymetryGrid(oneOf.fullPrivateDirectory(),
                    minLon, maxLon, minLat, maxLat, imageWidth, imageHeight);
            } else return null;
        } catch (Exception e) {
            String error = String2.ERROR + " in CurrentsScreen.getGrid:\n" + MustBe.throwableToString(e);
            String2.log(error);
            if (error.indexOf(DataHelper.THERE_IS_NO_DATA) < 0)
                oneOf.email(oneOf.emailEverythingTo(), 
                    OneOf.ERROR + " in " + oneOf.shortClassName(), error);
            plotGridData = false;
            plotBathymetryData = false;
            return null;
        }
    }

    /**
     * This actually generates the x image-resolution grid,
     * so it should be called after validate() if/when you actually need to use the data.
     *
     * @param minLon
     * @param maxLon
     * @param minLat
     * @param maxLat
     * @param imageWidth  from oneOf.imageWidths()[imageSizeIndex]
     * @param imageHeight
     * @return the requested grid, as best it can
     * @throws Exception if trouble
     */
    public Grid getXGrid(double minLon, double maxLon, 
            double minLat, double maxLat, int imageWidth, int imageHeight) throws Exception { 
        if (!plotVectorData)
            return null;
        return xVectorDataSet.makeGrid(timePeriodValue, timeValue, 
            minLon, maxLon, minLat, maxLat, imageWidth, imageHeight);
    }

    /**
     * This actually generates the y image-resolution grid,
     * so it should be called after validate() if/when you actually need to use the data.
     *
     * @param minLon
     * @param maxLon
     * @param minLat
     * @param maxLat
     * @param imageWidth  from oneOf.imageWidths()[imageSizeIndex]
     * @param imageHeight
     * @return the requested grid, as best it can
     * @throws Exception if trouble
     */
    public Grid getYGrid(double minLon, double maxLon, 
            double minLat, double maxLat, int imageWidth, int imageHeight) 
            throws Exception { 
        if (!plotVectorData)
            return null;
        return yVectorDataSet.makeGrid(timePeriodValue, timeValue, 
            minLon, maxLon, minLat, maxLat, imageWidth, imageHeight);
    }

    /**
     * This determines if the submitter's name matches the name of 
     * a Get button (e.g., getAsc).
     *
     * @param submitter
     * @return true if the submitter's name matches the name of 
     *    a Get button (e.g., getAsc); else false. 
     */
    public boolean submitterIsAGetButton(String submitter) {
        //viewAnimation handled differently
        return false;
    }

    /**
     * This responds to a submitter button, preparing the file and the 
     * html.
     *
     * @param submitter the name of the submitter button
     * @param htmlSB the StringBuffer which is collecting the html
     * @param backButtonForm the html for a form with just a back button
     * @param throws Exception if trouble
     */
    public void respondToSubmitter(String submitter, StringBuffer htmlSB, 
            String backButtonForm) throws Exception {

    }

    /**
     * NOT ACTIVELY USED SINCE CACHE NOT USED ANY MORE. 
     * This is a custom program to call CCBrowser to generate/cache lots of images.
     * 
     */
    public static void main(String args[]) throws Exception {
        //typical url=http://coastwatch.pfeg.noaa.gov/cwexperimental/CencoosCurrents.jsp?
        //  dataSet=GOES%20SST&timePeriod=25%20hour&time=2006-03-02%2008:00:00
        String currentTimeString = 
            Calendar2.getCurrentISODateTimeStringZulu().substring(0, 13) + ":00:00"; //to nearest hour
        double currentTime = Calendar2.isoStringToEpochSeconds(currentTimeString) - //throws exception if trouble
            Calendar2.SECONDS_PER_DAY;  //go back one day

        //standard settings
        String baseUrl = "http://coastwatch.pfeg.noaa.gov/coastwatch/CencoosCurrents.jsp";
        String dataSets[] = {OneOf.NO_DATA, SgtMap.BATHYMETRY_BOLD_TITLE, "SST%201.25%20km", "GOES%20SST", "OSU%20Aqua%20Chl-%3Ci%3Ea%3C%2Fi%3E"}; 
        String timePeriods[] = {"hourly", "25%20hour", "33%20hour"};
        String startSecondsString = "2006-01-07 00:00:00";
        String endSecondsString =   "2006-04-26 00:00:00";
        int firstDS = 0, lastDS = dataSets.length - 1;
        int firstTP = 0, lastTP = timePeriods.length - 1;
        int sleepMillis = 1000;
        String2.log("CurrentsScreen.main baseUrl=" + baseUrl);

        //customize the settings
        //firstDS = 0;
        //firstTP = 0;
        //startSecondsString = "2006-04-05 00:00:00";
        endSecondsString = "2006-04-03 11:00:00";

        //generate all of the images   starting with current time
        double startSeconds = Calendar2.isoStringToEpochSeconds(startSecondsString); //throws exception if trouble
        double endSeconds =   Calendar2.isoStringToEpochSeconds(endSecondsString);   //throws exception if trouble
        double seconds = endSeconds;
        while (seconds >= startSeconds) {
            String timeUrl = Calendar2.epochSecondsToIsoStringT(seconds);
            timeUrl = String2.replaceAll(timeUrl, "T", " "); //2006-03-02 08:00:00
            for (int ds = firstDS; ds <= lastDS; ds++) {
                for (int tp = firstTP; tp <= lastTP; tp++) {
                    String endUrl = "?dataSet=" + dataSets[ds] +
                        "&timePeriod=" + timePeriods[tp] + "&time=" + timeUrl;
                    String2.log(endUrl);
                    SSR.getUrlResponse(baseUrl + endUrl);
                    Math2.incgc(sleepMillis);
                }
            }       
            seconds -= Calendar2.SECONDS_PER_HOUR;
        }
    }

}
