/* 
 * DrifterModel Copyright 2006, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch;

import com.cohort.array.*;
import com.cohort.ema.EmaDateTimeSelect2;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.FileNameUtility;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.griddata.GridDataSet;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.RegexFilenameFilter;
import gov.noaa.pfel.coastwatch.util.SSR;
import gov.noaa.pfel.coastwatch.util.StringObject;

import java.util.Arrays;


/**
 * This class simulates a drifter's movement, buffeted by the
 * x and y datasets.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2006-04-10
 *
 */
public class DrifterModel  {

    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;

    public static int nearestNeighborLimit = 5; //in every direction

    /** "ERROR" is defined so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR; 

    /**
     * Test if lon,lat is out of the bathymetryMask's range.
     * @param lon 
     * @param lat
     * @param lonIndex the index of the lon in bathymetryMask.lon
     * @param latIndex the index of the lat in bathymetryMask.lat
     * @param bathymetryMask the grid with bathymetry data 
     */
    private static boolean outOfRange(double lon, double lat, int lonIndex, int latIndex, Grid bathymetryMask) {
        int nLon = bathymetryMask.lon.length;
        int nLat = bathymetryMask.lat.length;
        boolean outOfRange = 
            lonIndex < 0 || lonIndex >= nLon ||
            (lonIndex == 0 && bathymetryMask.lon[0] - lon > bathymetryMask.lonSpacing / 2) ||
            (lonIndex == nLon - 1 && lon - bathymetryMask.lon[nLon - 1] > bathymetryMask.lonSpacing / 2);
        if (outOfRange) {
            String2.log("!!!OutOfRange:" +
                " lon=" + lon + " nLon=" + nLon + " lon0=" + bathymetryMask.lon[0] + 
                " lonn=" + bathymetryMask.lon[nLon-1] + " lonSp=" + bathymetryMask.lonSpacing);
            return true;
        }

        outOfRange = 
            latIndex < 0 || latIndex >= nLat ||
            (latIndex == 0 && bathymetryMask.lat[0] - lat > bathymetryMask.latSpacing / 2) ||
            (latIndex == nLat - 1 && lat - bathymetryMask.lat[nLat - 1] > bathymetryMask.latSpacing / 2);
        if (outOfRange) { 
            String2.log("!!!OutOfRange:" +
                " lat=" + lat + " nLat=" + nLat + " lat0=" + bathymetryMask.lat[0] + 
                " latn=" + bathymetryMask.lat[nLat-1] + " latSp=" + bathymetryMask.latSpacing); 
            return true;
        }

        return false;
    }

    /**
     * Test if lon,lat is over land (within the bathymetryMask range, but over land).
     * (etopo2 docs say: land is &gt;= 0,  ocean is &lt; 0.)
     */
    private static boolean overLand(int lonIndex, int latIndex, Grid bathymetryMask) {
        return lonIndex >= 0 && lonIndex < bathymetryMask.lon.length &&
               latIndex >= 0 && latIndex < bathymetryMask.lat.length &&
            bathymetryMask.getData(lonIndex, latIndex) >= 0; //getData() will throw exception if lon/lat out of range
    }

    /**
     * This class simulates a drifter's movement, buffeted by the
     * x and y datasets.
     * <p>The model simply moves the drifter by the amount specified
     *   in a given frame of the x and y gridDataSet (adjusted for the time
     *   between frames of the data set). 
     *   The movement is done after a given time point, so when you watch
     *   the animation you see the vector that will act on the drifter.
     * <p>If !strictModel, at each time point, if there is no
     *   data (at all) for the current time, the drifters maintain their 
     *   previous x/yVelocity.
     * <p>If !strictModel, the model fills gaps in the gridDataSets by simply searching 
     *   for valid points in ever larger squares around a point.
     *   If there is no data in the +-5 square, the model just uses the
     *   previous velocity. If there is no data in the +-5 square 
     *   for the initial time period, the drifters don't move.
     * <p>If strictModel, if any needed data isn't available, the
     *   drifter disappears.
     * <p>If drifter goes beyond the dataset's range, it disappears.
     *
     * @param strictModel If true, data must be available or the drifter disappears.
     *    If false, nearest neighbor data is allowed.
     * @param tempDir a directory to hold the temp files
     * @param lon0 the initial lon of the drifter
     * @param lat0 the initial lat of the drifter
     * @param offset the initial distance from the main drifter to the siblings, 
     *    in degrees, e.g., 0.01.
     *    Yes this is only technically correct at the equator, but
     *    it will look right on a geographic-projection map.
     * @param makePM180 true if lon0 and grids should work in +/-180;
     *    if false, they work in 0..360.
     * @param bathymetryMask is a grid with the same xy range (not necessary the same resolution) 
     *    as the x/yGridDataSets with bathymetry information.
     *    If a drifter goes over land, it stops moving.
     * @param xVectorDataSet Holds the information about the original x-component data. 
     * @param yVectorDataSet Holds the information about the original y-component data. 
     * @param timePeriod the timePeriod you want to work on (e.g., "1 Observation").
     *    It must be one of the options for the given x/yVectorDataSet.
     *    It is used to figure out which type of data to use from x/yVectorDataSet
     *    See TimePeriods.
     * @param endTime the dateTime you want to end at (an ISO date/time e.g., "2006-01-03T12:00:00").
     *     It must be one of the options for the given time period for the yVectorDataSet.
     * @param activeVectorDateOptions a list of the valid shared dates for xVector and yVectorDataSets
     * @param nHoursBack This determines the start time for the model
     * @return a results Table: 4 columns: lon, lat, secondsSince1970-01-01Z, drifterNumber; 
     *   nFrames*9 rows: #0..#7 the siblings and #8 main drifter. 
     *   The siblings are initially at 45 degree compass points
     *   relative to the main drifter, starting at 0 (true north).
     * @throws Exception if trouble (e.g., if endTime is invalid)
     */
    public static Table run(boolean strictModel,
        String tempDir, double lon0, double lat0, double offset, 
        boolean makeLonPM180, Grid bathymetryMask,
        GridDataSet xVectorDataSet, GridDataSet yVectorDataSet, 
        String timePeriod, String endTime, String activeVectorDateOptions[], 
        int nHoursBack)
        throws Exception {

//SOON: make it so that if x/yVelocity are not known, the siblings also move away
//  from the main drifter.

        if (verbose) String2.log("DrifterModel.run lon0=" + lon0 + 
            " lat0=" + lat0 + " timePeriod=" + timePeriod + 
            " end=" + endTime + " nHoursBack=" + nHoursBack); 
        String errorInMethod = ERROR + " in DrifterModel.run:\n";
        long time = System.currentTimeMillis();
        long readDataTime = 0;

        //find the timePeriodIndexes
        int xTimePeriodIndex = String2.indexOf(xVectorDataSet.activeTimePeriodOptions, timePeriod);
        int yTimePeriodIndex = String2.indexOf(yVectorDataSet.activeTimePeriodOptions, timePeriod);
        if (xTimePeriodIndex < 0) 
            Test.error(errorInMethod + "timePeriod=" + timePeriod + 
                " not found in xVectorDataSet.activeTimePeriodOptions:\n" + 
                String2.toCSVString(xVectorDataSet.activeTimePeriodOptions));
        if (yTimePeriodIndex < 0) 
            Test.error(errorInMethod + "timePeriod=" + timePeriod + 
                " not found in yVectorDataSet.activeTimePeriodOptions:\n" + 
                String2.toCSVString(yVectorDataSet.activeTimePeriodOptions));

        //get the relevant dateOptions
        String[] xDateOptions = (String[])xVectorDataSet.activeTimePeriodTimes.get(xTimePeriodIndex);
        String[] yDateOptions = (String[])yVectorDataSet.activeTimePeriodTimes.get(yTimePeriodIndex);

        //create currentSeconds  (nHoursBack)
        double currentSeconds = Calendar2.isoStringToEpochSeconds(endTime) - //throws exception if trouble
            Calendar2.SECONDS_PER_HOUR * nHoursBack;

        //Calculate the conversion factor for meters (the vector data is m/s) to 
        //  latLonDegrees.
        //Eeeek! Is this too crude?
        //http://www.hypernews.org/HyperNews/get/trails/SAR/291/1.html says
        //"A degree of latitude is 60 nautical miles, or 69.04 statute miles ...
        //35 degrees North, a degree of longitude is 56.73 statute miles", etc
        double degreesNorth[] =         {    0,    30,    35,    40,    45,    50,    55,    60};
        double milesPerXDegreeArray[] = {69.04, 59.96, 56.73, 53.06, 49.00, 44.55, 39.77, 34.67};
        double milesPerXDegree = milesPerXDegreeArray[Math2.binaryFindClosest(degreesNorth, Math.abs(lat0))];
        double milesPerYDegree = 69.04;
        double xDegreesPerMeter = (1 / Math2.mPerMile) / milesPerXDegree;  
        double yDegreesPerMeter = (1 / Math2.mPerMile) / milesPerYDegree; 

        //set up the results table
        int nDrifters = 9;
        DoubleArray xPA = new DoubleArray();
        DoubleArray yPA = new DoubleArray();
        DoubleArray timePA = new DoubleArray();
        IntArray drifterPA = new IntArray();
        Table resultsTable = new Table();
        resultsTable.addColumn("x", xPA);
        resultsTable.addColumn("y", yPA);
        resultsTable.addColumn("time", timePA);
        resultsTable.addColumn("drifter", drifterPA);

        //set up the velocity data structures
        //All velocities are m/s
        double defaultXVelocity = 0;  //default = don't move
        double defaultYVelocity = 0;
        double xVelocity[] = new double[nDrifters];
        double yVelocity[] = new double[nDrifters];
        Arrays.fill(xVelocity, defaultXVelocity);
        Arrays.fill(yVelocity, defaultYVelocity);

        //set overland
        boolean overLand[] = new boolean[nDrifters];
        Arrays.fill(overLand, false);

        //set the base file names    FILE_NAME_RELATED_CODE
        String xBaseName = xVectorDataSet.internalName + "S" + TimePeriods.getInFileName(timePeriod);
        String yBaseName = yVectorDataSet.internalName + "S" + TimePeriods.getInFileName(timePeriod);

        //deal with lonPM180
        double dataMinLon = makeLonPM180? -180 : 0;
        double dataMaxLon = makeLonPM180? 180  : 360;

        //run the simulation
        double getNearestResults[] = new double[2];
        boolean someValid = true;
        for (int frame = 0; frame <= nHoursBack; frame++) { 
            String2.log("frame=" + frame);

            if (frame == 0) {
                //set initial drifter locations
                double sin45 = 0.7071 * offset;                
                xPA.add(lon0);           yPA.add(lat0 + offset); timePA.add(currentSeconds); drifterPA.add(0); //N
                xPA.add(lon0 + sin45);   yPA.add(lat0 + sin45);  timePA.add(currentSeconds); drifterPA.add(1);
                xPA.add(lon0 + offset);  yPA.add(lat0);          timePA.add(currentSeconds); drifterPA.add(2); //E
                xPA.add(lon0 + sin45);   yPA.add(lat0 - sin45);  timePA.add(currentSeconds); drifterPA.add(3);
                xPA.add(lon0);           yPA.add(lat0 - offset); timePA.add(currentSeconds); drifterPA.add(4); //S
                xPA.add(lon0 - sin45);   yPA.add(lat0 - sin45);  timePA.add(currentSeconds); drifterPA.add(5);
                xPA.add(lon0 - offset);  yPA.add(lat0);          timePA.add(currentSeconds); drifterPA.add(6); //W
                xPA.add(lon0 - sin45);   yPA.add(lat0 + sin45);  timePA.add(currentSeconds); drifterPA.add(7);
                xPA.add(lon0);           yPA.add(lat0);          timePA.add(currentSeconds); drifterPA.add(8); //main drifter last

                //check individually for out-of-range and overLand
                for (int drifter = 0; drifter < nDrifters; drifter++) {
                    int index = xPA.size() - nDrifters + drifter;
                    double tx = xPA.get(index);
                    double ty = yPA.get(index);
                    int lonIndex = Math2.binaryFindClosest(bathymetryMask.lon, tx);
                    int latIndex = Math2.binaryFindClosest(bathymetryMask.lat, ty);
                    if (outOfRange(tx, ty, lonIndex, latIndex, bathymetryMask)) {
                        xPA.set(index, Double.NaN);
                        yPA.set(index, Double.NaN);
                    } else if (overLand(lonIndex, latIndex, bathymetryMask)) {
                        overLand[drifter] = true;
                    }
                    //String2.log("frame=0 drifter=" + drifter + " index=" + index +
                    //    " tx=" + tx + " ty=" + ty + 
                    //    " xPA=" + xPA.get(index) + " yPA=" + yPA.get(index));
                }

            } else {

                //move the drifters based on the previous velocity
                someValid = false;
                for (int drifter = 0; drifter < nDrifters; drifter++) {

                    //move the drifter
                    double tx = xPA.get(xPA.size() - nDrifters);  //x/yPA.size increases with each step
                    double ty = yPA.get(yPA.size() - nDrifters);
                    if (Double.isNaN(tx) || overLand[drifter]) {
                        //don't move the drifter at all
                        if (verbose && drifter == 8)
                            String2.log("drifter 8: x=" + tx + " y=" + ty + " is unchanged");
                    } else {
                        double otx = tx;
                        double oty = ty;
                        tx += Calendar2.SECONDS_PER_HOUR * xVelocity[drifter] * xDegreesPerMeter;
                        ty += Calendar2.SECONDS_PER_HOUR * yVelocity[drifter] * yDegreesPerMeter;

                        if (verbose) 
                            String2.log(
                                "frame=" + frame + " drifter=" + drifter +  
                                " x=" + otx + " xVel=" + xVelocity[drifter] + " xDeg/m=" + xDegreesPerMeter + " tx=" + tx + 
                                " y=" + oty + " yVel=" + yVelocity[drifter] + " yDeg/m=" + yDegreesPerMeter + " ty=" + ty);

                        //is it out of range or over land now?
                        int lonIndex = Math2.binaryFindClosest(bathymetryMask.lon, tx);
                        int latIndex = Math2.binaryFindClosest(bathymetryMask.lat, ty);
                        if (outOfRange(tx, ty, lonIndex, latIndex, bathymetryMask)) {
                            tx = Double.NaN;
                            ty = Double.NaN;
                        } else if (overLand(lonIndex, latIndex, bathymetryMask)) {
                            overLand[drifter] = true;
                        } else someValid = true;
                    }

                    //store the info
                    xPA.add(tx);           
                    yPA.add(ty); 
                    timePA.add(currentSeconds); 
                    drifterPA.add(drifter); 
                }
            }

            //are we done yet?
            if (frame == nHoursBack) { 
                if (verbose)
                    String2.log("DrifterModel.run finished successfully time=" + 
                        (System.currentTimeMillis() - time) + 
                        " ms; readDataTime=" + readDataTime);
                return resultsTable;
            }

            //figure out which xGrid and yGrid data
            String currentDateString = Calendar2.epochSecondsToIsoStringSpace(currentSeconds);
            int vectorIndex = 
                String2.indexOf(activeVectorDateOptions, currentDateString);
                //not: Calendar2.binaryFindClosest(activeVectorDateOptions, currentDateString);

            //load the xGrid and yGrid data
            Grid xGrid = null;
            Grid yGrid = null;
            if (vectorIndex < 0 || !someValid) { 
                //occurs if strictModel and no vector data for currentDateString 
                //leave xGrid and yGrid = null
            } else {
                String dateString = activeVectorDateOptions[vectorIndex];
                readDataTime -= System.currentTimeMillis();
                xGrid = xVectorDataSet.makeGrid(timePeriod, dateString,          
                    dataMinLon, dataMaxLon, -90, 90, 
                    Integer.MAX_VALUE, Integer.MAX_VALUE);
                yGrid = yVectorDataSet.makeGrid(timePeriod, dateString,          
                    dataMinLon, dataMaxLon, -90, 90, 
                    Integer.MAX_VALUE, Integer.MAX_VALUE);
                Test.ensureEqual(xGrid.lon, yGrid.lon,
                    errorInMethod + "xGrid and yGrid lon arrays are different.");
                Test.ensureEqual(xGrid.lat, yGrid.lat,
                    errorInMethod + "xGrid and yGrid lat arrays are different.");
                readDataTime += System.currentTimeMillis();
            }

            //get the new velocity for each drifter
            for (int drifter = 0; drifter < nDrifters; drifter++) {
                //no need to do anything?
                double tx = xPA.get(xPA.size() - nDrifters + drifter);
                double ty = yPA.get(yPA.size() - nDrifters + drifter);
                if (overLand[drifter] || Double.isNaN(tx))
                    continue;

                if (xGrid == null || yGrid == null) {
                    if (strictModel) {
                        xVelocity[drifter] = Double.NaN;
                        yVelocity[drifter] = Double.NaN; 
                    } else {
                        //x/yVelocity unchanged
                    }
                } else {
                    //find the closest latIndex and lonIndex
                    int lonIndex = Math2.binaryFindClosest(xGrid.lon, tx);
                    int latIndex = Math2.binaryFindClosest(xGrid.lat, ty);

                    //find the nearest velocity
                    getNearestVelocity(strictModel? 0 : nearestNeighborLimit, 
                        lonIndex, latIndex, xGrid, yGrid, getNearestResults);
                    //String2.log("drifter=" + drifter + " lonI=" + lonIndex + 
                    //    " latI=" + latIndex + " xVel=" + getNearestResults[0]); 
                    if (strictModel) {
                        xVelocity[drifter] = getNearestResults[0];  //may be NaN
                        yVelocity[drifter] = getNearestResults[1];
                    } else if (!Double.isNaN(getNearestResults[0])) { //if NaN, no change
                        xVelocity[drifter] = getNearestResults[0];    //if !NaN, change
                        yVelocity[drifter] = getNearestResults[1];
                    }
                }
            }

            //advance currentSeconds
            currentSeconds += Calendar2.SECONDS_PER_HOUR;
        }

        //return   but real exit is in middle of loop
        return resultsTable;

    }
    
    /**
     * This gets a velocity value for lonIndex,latIndex in from xGrid and yGrid,
     * by searching for nearest neighbors.
     *
     * @param maxStep Use 0 for strictModel; otherwise 5(?).
     * @param lonIndex
     * @param latIndex
     * @param xGrid
     * @param yGrid
     * @param results a double[2] to catch the results: [0]=xVelocity and [1]=yVelocity.
     *    If there are no data values within maxStep, this returns 2 Double.NaN's.
     * @throws Exception if trouble
     */
    public static void getNearestVelocity(int maxStep, int lonIndex, int latIndex,
        Grid xGrid, Grid yGrid, double results[]) {

        //handle simple case: data is available at lonIndex,latIndex
        results[0] = xGrid.getData(lonIndex, latIndex);
        results[1] = yGrid.getData(lonIndex, latIndex);
        if (maxStep == 0 || 
            (!Double.isNaN(results[0]) && !Double.isNaN(results[1]))) 
            return;

        //look in ever larger squares around the original lonIndex,latIndex
        int nLon = xGrid.lon.length;
        int nLat = xGrid.lat.length;
        for (int step = 1; step < maxStep; step++) {
            //String2.log("getNearestVelocity step=" + step);

            //collect all the x and y values around the square
            double xSum = 0;
            double ySum = 0;
            int count = 0;

            //look at top and bottom of square
            for (int x = -step; x <= step; x++) {
                int tLonIndex = lonIndex + x;
                if (tLonIndex < 0 || tLonIndex >= nLon)
                    continue;
                for (int y = -step; y <= step; y += step + step) {
                    int tLatIndex = latIndex + y;
                    if (tLatIndex < 0 || tLatIndex >= nLat)
                        continue;
                    double tx = xGrid.getData(tLonIndex, tLatIndex);
                    if (!Double.isNaN(tx)) {
                        double ty = yGrid.getData(tLonIndex, tLatIndex);
                        if (!Double.isNaN(ty)) {
                            xSum += tx;
                            ySum += ty;
                            count++;
                        }
                    }
                }
            }

            //look at sides of square
            for (int y = -step + 1; y < step; y++) {  //+1, < because don't want to do corners
                int tLatIndex = latIndex + y;
                if (tLatIndex < 0 || tLatIndex >= nLat)
                    continue;
                for (int x = -step; x <= step; x += step + step) {
                    int tLonIndex = lonIndex + x;
                    if (tLonIndex < 0 || tLonIndex >= nLon)
                        continue;
                    double tx = xGrid.getData(tLonIndex, tLatIndex);
                    if (!Double.isNaN(tx)) {
                        double ty = yGrid.getData(tLonIndex, tLatIndex);
                        if (!Double.isNaN(ty)) {
                            xSum += tx;
                            ySum += ty;
                            count++;
                        }
                    }
                }
            }

            //if count>0, we're done
            if (count > 0) {
                //just do simple mean because all found points are ~equidistant from original point
                results[0] = xSum / count;
                results[1] = ySum / count;
                return;
            }
        }

        //fail
        results[0] = Double.NaN;
        results[1] = Double.NaN;
        return;
    }




    /**
     * A main method -- used to run this class.
     *
     * @throws Exception if trouble
     */
    //public static void main(String args[]) throws Exception {


    //}


}
