/* 
 * Projects Copyright 2005, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch;

import com.cohort.array.*;
import com.cohort.util.*;

/** The Java DAP classes.  */
import dods.dap.*;

import gov.noaa.pfel.coastwatch.griddata.*;
import gov.noaa.pfel.coastwatch.hdf.*;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.*;

import java.io.FileWriter;
import java.io.Writer;
import java.util.GregorianCalendar;
import java.util.List;

import org.codehaus.janino.ExpressionEvaluator;

/**
 * Get netcdf-X.X.XX.jar from http://www.unidata.ucar.edu/software/netcdf-java/index.htm
 * and copy it to <context>/WEB-INF/lib renamed as netcdf-latest.jar.
 * Get slf4j-jdk14.jar from 
 * ftp://ftp.unidata.ucar.edu/pub/netcdf-java/slf4j-jdk14.jar
 * and copy it to <context>/WEB-INF/lib.
 * Put both of these .jar files in the classpath for the compiler and for Java.
 */
import ucar.nc2.*;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dods.*;
import ucar.nc2.util.*;
import ucar.ma2.*;



/**
 * This class has static methods special, one-time projects.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2006-12-11
 */
public class Projects  {

    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;


    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR; 

    public final static String stationColumnName = "ID";


    /** A special project for Roy. 
     * This extracts/converts the Channel Islands data in
     * c:/temp/kushner/KFM_Temperature.txt
     * and SiteLocations.txt into separate 4D .nc files with metadata.
     * 12/11/06
     * I put the results in \\Xserve\pfel_share\PERSON_PERSON\BobSimons\ChannelIslands
     * for Roy. 12/13/06
     * Email from David_Kushner@nps.gov 12/12/06
<pre>
1) I note from the email below that you don't want the exact lat/lon and
depth locations made public.  OK.  Is it sufficient for me to simply use
the degree and minute values, and ignore the seconds values?

YES

2) Given that the lat and lon will be crude, can I use the Depth as is? YES
Or, how should I obscure it? NO NEED TO IF.

3) It seems that many of the minute values for the SiteLocations are 0.
More than one would normally expect by chance.  Are the minute values
correct?  Or can you provide me with an updated SiteLocations file (.mdb
or ASCII files are fine)?

YES, IT IS A FORMAT ISSUE IN OUR DATA SET, THAT I SHOULD FIX.  0 = 00 AND 2
EQUALS 02.  LET ME KNOW IF THAT DOES NOT MAKE SENSE.

4) Are the times recorded in the file UTC times? Or are they local
times? If they are local times, did you use clock time (so that the
times may be standard time or daylight savings time)?

THEY ARE PACIFIC STANDARD TIME SET TO THE COMPUTER I USE TO SET THE
LOGGERS.  WE RETRIEVE THEM ONCE PER YEAR SO I ASSUME THEY ARE CORRECTED FOR
DAYLIGHT SAVINGS TIME, BUT I NEVER REALLY THOUGHT ABOUT THIS.

[in another email he says]
Wow, good question.  They are local PST.  However, I will just presume that
they are daylight savings time, but never thought about it.  We retrieved
the data only once per year in the summer so I never thought to check that.

[Bob interprets this as: all times are Pacific Daylight Savings Time, 
e.g., 7 hours earlier than GMT,
since he used his computer's clock in the summer to set the logger's clocks.]

5) Is it okay if the courtesy information appears as "Channel Islands
National Park, National Park Service"?

I WILL CHECK ON THIS, BUT I THINK IT SHOULD HAVE BOTH:  :COURTESY OF THE
NATIONAL PARK SERVICE, CHANNEL ISLANDS NATIONAL PARK.

[in the 12/12/2006 email he says:]
We have been usuing Onset computer Corp. temperature loggers.  However, we
have been usuing these since the company began selling them and the models
have changed three times.  In the early years we used their HoboTemp tm
loggers, these had little memory which is why we recorded temperature every
4-5 hours.  Then they developed the StowAway loggers with more memory and
went to one hour intervals.  There was another version of StowAway before
we then switched to the newer Tidbit (these are completely waterproof,
yeah!) loggers several years ago.  It often took sevaral years to swap out
models since we only visit the sites once per year so it is hard to get an
exact date for each site of which model was used when, though I do have
this info on raw data sheets.  All the temperature loggers are supposed to
be accurate to +- 0.2C.  For many of the past 10 years I have installed two
loggers in the same underwater housing at most of the sites and have
compared the data.  In nearly all cases the loggers have been within
specifications, so I do have much confidence in the loggers to +-0.2 C.  In
the early years I was even testing them periodically in an ice bath.

</pre>
     */ 
    public static void channelIslands() throws Exception {
        String2.log("\n*** Projects.channelIslands");

        //read SiteLocations.txt 
        //IslandName\tSiteName\tLatDegrees\tLatMinutes\tLatSeconds\tLatDir\tLongDegrees\tLongMinutes\tLongSeconds\tLongDir\tDepth (meters)
        //Anacapa\tAdmiral's Reef\t34\t00\t200\tN\t119\t25\t520\tW\t16
        //Anacapa\tBlack Sea Bass Reef\t34\t00\t756\tN\t119\t23\t351\tW\t17
        Table site = new Table();
        StringArray siteID = new StringArray();
        FloatArray siteLon = new FloatArray();
        FloatArray siteLat = new FloatArray();
        IntArray siteDepth = new IntArray();
        site.addColumn("ID", siteID);
        site.addColumn("LON", siteLon);
        site.addColumn("LAT", siteLat);
        site.addColumn("DEPTH", siteDepth);
        int siteNRows;
        {
            //read the site location info
            Table tSite = new Table();
            tSite.readASCII("c:/temp/kushner/Site_Locations.txt", 0, 1, null, null, null, null);

            //test that first row's data is as expected
            Test.ensureEqual(tSite.getStringData(0, 0), "Anacapa", "");
            Test.ensureEqual(tSite.getStringData(1, 0), "Admiral's Reef", "");
            Test.ensureEqual(tSite.getDoubleData(2, 0), 34, "");  //lat
            Test.ensureEqual(tSite.getDoubleData(3, 0), 0, "");
            Test.ensureEqual(tSite.getDoubleData(4, 0), 200, "");
            Test.ensureEqual(tSite.getStringData(5, 0), "N", "");
            Test.ensureEqual(tSite.getDoubleData(6, 0), 119, ""); //lon
            Test.ensureEqual(tSite.getDoubleData(7, 0), 25, "");
            Test.ensureEqual(tSite.getDoubleData(8, 0), 520, "");
            Test.ensureEqual(tSite.getStringData(9, 0), "W", "");
            Test.ensureEqual(tSite.getDoubleData(10, 0), 16, "");

            //fill siteID, siteLat, siteLon, siteDepth  in new site table
            siteNRows = tSite.nRows();
            for (int row = 0; row < siteNRows; row++) {
                siteID.add(tSite.getStringData(0, row) + " (" + tSite.getStringData(1, row) + ")");

                //lat
                double d = tSite.getDoubleData(2, row) + 
                    tSite.getDoubleData(3, row) / 60.0;
                    //12/01/2006 email: Kushner says don't make exact location public
                    //so I say: ignore "seconds" column (which is actually milli-minutes;
                    //his 12/12/2006 email says "The .351[sic 351] is .351 X 60 (seconds) = 21.06 seconds.").
                    //+ tSite.getDoubleData(4, row) / 60000.0
                siteLat.addDouble(d);

                //lon
                d = tSite.getDoubleData(6, row) + 
                    tSite.getDoubleData(7, row) / 60.0;
                    //+ tSite.getDoubleData(8, row) / 60000.0 
                if (tSite.getStringData(9, 0).equals("W"))
                    d = -d;                 
                siteLon.addDouble(d);

                //but I am using depth value as is
                siteDepth.add(tSite.getIntData(10, row));
            }

            //sort by ID
            site.sort(new int[]{0}, new boolean[]{true});
            String2.log("site table=\n" + site.toString("row", siteNRows));
        }

        //read c:/temp/kushner/KFM_Temperature.txt which has all the temperature data
        //IslandName\tSiteName\tDate\tTemperatureC
        //Anacapa\tAdmiral's Reef\t8/26/1995 14:49:00\t18.47
        //Anacapa\tAdmiral's Reef\t8/25/1995 14:49:00\t18.79
        Table temp = new Table();
        temp.allowRaggedRightInReadASCII = true; //it has some missing temp values
        temp.readASCII("c:/temp/kushner/KFM_Temperature.txt", 0, 1, null, null, null, null);

        //ensure that first row's data is as expected
        Test.ensureEqual(temp.getStringData(0, 0), "Anacapa", "");
        Test.ensureEqual(temp.getStringData(1, 0), "Admiral's Reef", "");
        Test.ensureEqual(temp.getStringData(2, 0), "8/26/1995 14:49:00", "");
        Test.ensureEqual(temp.getFloatData( 3, 0), 18.47f, "");
        int tempNRows = temp.nRows();

        //convert temp table's dateTime to epochSeconds
        {
            StringArray oldTimePA = (StringArray)temp.getColumn(2);
            DoubleArray newTimePA = new DoubleArray();
            for (int row = 0; row < tempNRows; row++) {
                double sec = Calendar2.gcToEpochSeconds(
                    Calendar2.parseUSSlash24Zulu(temp.getStringData(2, row))); //throws Exception
                if (sec < 100 || sec > 2e9)
                    String2.log("row=" + row + " sec=" + sec + " Unexpected time=" + temp.getStringData(2, row));
                newTimePA.add(sec);
            }
            temp.setColumn(2, newTimePA);
        }

        //sort by  island, site, time
        String2.log("pre-sort n=" + temp.nRows() + " temp table=\n" + temp.toString("row", 5));
        temp.sort(new int[]{0, 1, 2}, new boolean[] {true, true, true});

        //go through rows of temp, saving separate station files
        String oldIsland, oldSite, oldID, newIsland = "", newSite = "", newID = "";
        Attributes attrib;
        int oldPo, newPo = -1;
        int nStationsCreated = 0;
        Table station = new Table();   //x,y,z,t,id,temperature
        station.addColumn(0, "LON", new FloatArray()); //x
        station.addColumn(1, "LAT", new FloatArray()); //y
        station.addColumn(2, "DEPTH", new IntArray()); //z
        station.addColumn(3, "TIME", new DoubleArray()); //t

        StringArray stationPA = new StringArray();
        Attributes stationAttributes = new Attributes();
        station.addColumn(4, stationColumnName, stationPA, stationAttributes); //id
        stationAttributes.set("long_name", "Station Identifier");
        //stationAttributes.set("units", DataHelper.UNITLESS);

        station.addColumn(5, "Temperature", new FloatArray()); //temperature
        attrib = station.columnAttributes(5);
        attrib.set("long_name", "Sea Temperature");
        attrib.set("standard_name", "sea_water_temperature");
        attrib.set("units", "degree_C");

        for (int row = 0; row <= tempNRows; row++) { //yes, =n since comparing current to previous row
            oldIsland = newIsland;
            oldSite = newSite;
            oldID = newID;
            oldPo = newPo;
            if (row < tempNRows) {
                newIsland = temp.getStringData(0, row);
                newSite = temp.getStringData(1, row);             
                newID = newIsland + " (" + newSite + ")";
                if (newID.equals(oldID)) {
                    newPo = oldPo;
                } else {
                    newPo = siteID.indexOf(newID, 0);
                    Test.ensureNotEqual(newPo, -1, "tID not found: " + newID);
                    String2.log("newID=" + newID);
                }
            }

            //save the station file
            if (row == tempNRows || 
                (row > 0 && !newID.equals(oldID))) {
                String tOldIsland = String2.replaceAll(oldIsland, " ", "");
                String tOldSite = String2.replaceAll(oldSite, " ", "");
                tOldSite = String2.replaceAll(tOldSite, "'", "");
                String tempID = "KFMTemperature_" + tOldIsland + "_" + tOldSite;

                //setAttributes
                station.setAttributes(0, 1, 2, 3, //x,y,z,t
                    "Sea Temperature (Channel Islands, " + oldIsland + ", " + oldSite + ")", //boldTitle
                    "Station", //cdmDataType
                    DataHelper.ERD_CREATOR_EMAIL, //"Roy.Mendelssohn@noaa.gov", //creatorEmail
                    DataHelper.ERD_CREATOR_NAME,  //"NOAA NMFS SWFSC ERD",  //creatorName
                    DataHelper.ERD_CREATOR_URL,   //"http://www.pfel.noaa.gov", //creatorUrl
                    DataHelper.ERD_PROJECT,       
                    tempID, //id
                    "GCMD Science Keywords", //keywordsVocabulary,
                    "EARTH SCIENCE > Oceans > Ocean Temperature > Water Temperature", //keywords

                    //references   from 2006-12-19 email from Kushner
                    "Channel Islands National Parks Inventory and Monitoring information: " +
                        "http://nature.nps.gov/im/units/medn . " +
                        "Kelp Forest Monitoring Protocols: " +
                        "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                    //summary  from 2006-12-19 email from Kushner
                    "The subtidal temperature data taken at Channel Islands National " +
                        "Park's Kelp Forest Monitoring Programs permanent monitoring sites.  Since " +
                        "1993, remote temperature loggers manufactured by Onset Computer Corporation " +
                        "were deployed at each site approximately 10-20 cm from the bottom in a " +
                        "underwater housing.  Since 1993, three models of temperature loggers " +
                        "(HoboTemp (tm), StowAway (R) and Tidbit(R) )were used to collect " +
                        "temperature data every 1-5 hours depending on the model used.",
                    //my old summary
                    //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                    //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                    //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                    //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                    //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                    "Channel Islands National Park, National Park Service", 
                    null); //timeLongName     use default long name) {

                //add the National Park Service disclaimer from 2006-12-19 email
                String license = station.globalAttributes().getString("license");
                license += "  National Park Service Disclaimer: " +
                    "The National Park Service shall not be held liable for " +
                    "improper or incorrect use of the data described and/or contained " +
                    "herein. These data and related graphics are not legal documents and " +
                    "are not intended to be used as such. The information contained in " +
                    "these data is dynamic and may change over time. The data are not " +
                    "better than the original sources from which they were derived. It is " +
                    "the responsibility of the data user to use the data appropriately and " +
                    "consistent within the limitation of geospatial data in general and " +
                    "these data in particular. The related graphics are intended to aid " +
                    "the data user in acquiring relevant data; it is not appropriate to " +
                    "use the related graphics as data. The National Park Service gives no " +
                    "warranty, expressed or implied, as to the accuracy, reliability, or " +
                    "completeness of these data. It is strongly recommended that these " +
                    "data are directly acquired from an NPS server and not indirectly " +
                    "through other sources which may have changed the data in some way. " +
                    "Although these data have been processed successfully on computer " +
                    "systems at the National Park Service, no warranty expressed or " +
                    "implied is made regarding the utility of the data on other systems " +
                    "for general or scientific purposes, nor shall the act of distribution " +
                    "constitute any such warranty. This disclaimer applies both to " +
                    "individual use of the data and aggregate use with other data.";
                station.globalAttributes().set("license", license);

                //save the station table
                station.saveAs4DNcWithStringVariable("c:/temp/kushner/" + tempID + ".nc", 
                    0, 1, 2, 3, 4);  //x,y,z,t, stringVariable=ID
                nStationsCreated++;

                //see what results are like 
                if (nStationsCreated == 1) {
                    Table tTable = new Table();
                    tTable.read4DNc("c:/temp/kushner/" + tempID + ".nc", null, 0, stationColumnName, 4);
                    String2.log("\nstation0=\n" + tTable.toString("row", 3));
                    //from site ascii file
                    //Anacapa	Admiral's Reef	34	00	200	N	119	25	520	W	16
                    //data from original ascii file
                    //Anacapa	Admiral's Reef	8/26/1995 14:49:00	18.47 //first Admiral's Reef in file
                    //Anacapa	Admiral's Reef	8/25/1995 17:49:00	18.95 //random
                    //Anacapa	Admiral's Reef	5/18/1998 11:58:00	14.27 //last  in file
                    DoubleArray ttimepa = (DoubleArray)tTable.getColumn(3);
                    double adjustTime = 7 * Calendar2.SECONDS_PER_HOUR; 
                    int trow;   
                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("1995-08-26 14:49:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 18.47f, "");

                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("1995-08-25 17:49:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 18.95f, "");

                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("1998-05-18 11:58:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 14.27f, "");
                }

                //set station table to 0 rows
                station.removeRows(0, station.nRows());
            }

            //add a row of data to station table
            if (row < tempNRows) {
                station.addFloatData(0, siteLon.get(newPo)); //x
                station.addFloatData(1, siteLat.get(newPo)); //y
                station.addIntData(2, siteDepth.get(newPo)); //z
                //t is adjusted from Pacific Daylight Savings Time to UTC (they are 7 hours ahead of PDST)
                //see metadata "summary" regarding this conversion.
                station.addDoubleData(3, temp.getDoubleData(2, row) + 7 * Calendar2.SECONDS_PER_HOUR); //t
                station.addStringData(4, newID); //id
                station.addFloatData(5, temp.getFloatData(3, row)); //temperature
            }
        }
        Test.ensureEqual(nStationsCreated, siteNRows, "nStationsCreated != siteNRows");

        String2.log("\n*** Projects.channelIslands finished successfully.");
    }

    /** 
     * Makes c:/temp/kfm200801/KFMSpeciesName.tab which has
     * a table of species number, species name in original files, species name in my files.
     * I remove the blank second line (no units) manually.
     * I will put the results in otter /u00/bob/kfm2008/ which mimics my c:/temp/kfm200801 .
     * for Lynn. 
     */ 
    public static void kfmSpeciesNameConversion200801() throws Exception {
        String2.log("\n*** Projects.kfmSpeciesNameConversion200801");

        //*** read the site location info
        //test that first row's data is as expected
        Table spp = new Table();
        spp.readASCII("c:/temp/kfm200801/KFMSpeciesName.tab", 
            0, 1, null, null, null, null);
        Test.ensureEqual(spp.getColumnName(0), "Species", "");
        Test.ensureEqual(spp.getColumnName(1), "Species Name", "");
        spp.removeColumns(2, spp.nColumns());
        StringArray oldNames = (StringArray)spp.getColumn(1);
        StringArray newNames = new StringArray();
        spp.addColumn(2, "New Name", newNames);

        int nRows = spp.nRows();
        for (int row = 0; row < nRows; row++) 
            newNames.add(convertSpeciesName(oldNames.get(row)));
        spp.saveAsTabbedASCII("c:/temp/kfm200801/KFMSpeciesNewName.tab");
     }

    
    /** 
     * This is the Jan 2008 revision of channelIslands().
     * This extracts/converts the Channel Islands data in
     * c:/temp/kfm200801/KFMHourlyTemperature.tab
     * and c:/temp/kfm200801/KFM_Site_Info.tsv into separate 4D .nc files with metadata.
     * 2008-01-15
     * I will put the results in otter /u00/bob/kfm2008/ which mimics my c:/temp/kfm200801 .
     */ 
    public static void kfmTemperature200801() throws Exception {
        String2.log("\n*** Projects.kfmTemperature200801");
        File2.deleteAllFiles("c:/temp/kfm200801/KFMTemperature/");

        //*** read the site location info
        //test that first row's data is as expected
        //Island	SiteName	Lat	Lon	Depth (meters)
        //San Miguel	Wyckoff Ledge	34.0166666666667	-120.383333333333	13
        Table site = new Table();
        site.readASCII("c:/temp/kfm200801/KFM_Site_Info.tsv", //copied from last year
            0, 1, null, null, null, null);
        Test.ensureEqual(site.getColumnName(0), "Island", "");
        Test.ensureEqual(site.getColumnName(1), "SiteName", "");
        Test.ensureEqual(site.getColumnName(2), "Lat", "");
        Test.ensureEqual(site.getColumnName(3), "Lon", "");
        Test.ensureEqual(site.getColumnName(4), "Depth (meters)", "");
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(site.getStringData(1, 0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.016666666f, "");  //lat   to nearest minute
        Test.ensureEqual(site.getFloatData(3, 0), -120.38333333f, "");       //to nearest minute  
        Test.ensureEqual(site.getDoubleData(4, 0), 13, "");
        int siteNRows = site.nRows();
        String2.log(site.toString());

        //*** read c:/temp/kfm200801/KFMHourlyTemperature.tab which has all the temperature data
        //test that first row's data is as expected
        //SiteNumber	IslandName	SiteName	Date	Time	TemperatureC
        //11	Anacapa	Admiral's Reef	8/26/1993 0:00:00	12/30/1899 10:35:00	18.00
        Table temp = new Table();
        temp.allowRaggedRightInReadASCII = true; //it has some missing temp values
        temp.readASCII("c:/temp/kfm200801/KFMHourlyTemperature.tab", 0, 1, null, null, null, null);
        temp.removeColumn(0); //now columns shifted down one
        //Test.ensureEqual(temp.getColumnName(0), "SiteNumber", "");
        Test.ensureEqual(temp.getColumnName(0), "IslandName", "");
        Test.ensureEqual(temp.getColumnName(1), "SiteName", "");
        Test.ensureEqual(temp.getColumnName(2), "Date", "");
        Test.ensureEqual(temp.getColumnName(3), "Time", "");
        Test.ensureEqual(temp.getColumnName(4), "TemperatureC", "");
        //Test.ensureEqual(temp.getFloatData(0, 0), 11, "");
        Test.ensureEqual(temp.getStringData(0, 0), "Anacapa", "");
        Test.ensureEqual(temp.getStringData(1, 0), "Admiral's Reef", "");
        Test.ensureEqual(temp.getStringData(2, 0), "8/26/1993 0:00:00", "");
        Test.ensureEqual(temp.getStringData(3, 0), "12/30/1899 10:35:00", "");
        Test.ensureEqual(temp.getFloatData( 4, 0), 18f, "");
        int tempNRows = temp.nRows();

        //convert temp table's IslandName+SiteName->ID; date+time->epochSeconds
        {
            StringArray oldIslandName = (StringArray)temp.getColumn(0);
            StringArray oldSiteName   = (StringArray)temp.getColumn(1);
            StringArray oldDatePA     = (StringArray)temp.getColumn(2);
            StringArray oldTimePA     = (StringArray)temp.getColumn(3);
            StringArray newID = new StringArray();
            DoubleArray newTimePA = new DoubleArray();
            for (int row = 0; row < tempNRows; row++) {
                String tID = oldIslandName.get(row) + " (" + oldSiteName.get(row) + ")";
                newID.add(tID);

                String tDate = oldDatePA.get(row);
                String tTime = oldTimePA.get(row);
                int po1 = tDate.indexOf(' ');
                int po2 = tTime.indexOf(' ');
                String dt = tDate.substring(0, po1) + tTime.substring(po2);
                double sec = Calendar2.gcToEpochSeconds(
                    Calendar2.parseUSSlash24Zulu(dt)); //throws Exception
                if (sec < 100 || sec > 2e9)
                    String2.log("row=" + row + " Unexpected sec=" + sec + " from date=" + tDate + " time=" + tTime);
                newTimePA.add(sec);
            }
            temp.removeColumn(3);
            temp.removeColumn(2);
            temp.removeColumn(1);
            temp.removeColumn(0);
            temp.addColumn(0, "Time", newTimePA, new Attributes());
            temp.addColumn(1, stationColumnName, newID, new Attributes());
            //col 2 is temperature
        }

        //sort by  ID, Time
        String2.log("pre-sort n=" + temp.nRows() + " temp table=\n" + temp.toString("row", 5));
        temp.sort(new int[]{1, 0}, new boolean[] {true, true});

        //go through rows of temp, saving separate station files
        int nStationsCreated = 0;
        Table station = new Table();   //x,y,z,t,id,temperature
        station.addColumn(0, "LON", new FloatArray()); //x
        station.addColumn(1, "LAT", new FloatArray()); //y
        station.addColumn(2, "DEPTH", new IntArray()); //z

        DoubleArray stationTime = new DoubleArray();
        station.addColumn(3, "TIME", stationTime); 

        StringArray stationID = new StringArray();
        Attributes stationAttributes = new Attributes();
        stationAttributes.set("long_name", "Station Identifier");
        //stationAttributes.set("units", DataHelper.UNITLESS);
        station.addColumn(4, stationColumnName, stationID, stationAttributes); 

        FloatArray stationTemperature = new FloatArray();
        station.addColumn(5, "Temperature", stationTemperature); 
        Attributes attrib = station.columnAttributes(5);
        attrib.set("long_name", "Sea Temperature");
        attrib.set("standard_name", "sea_water_temperature");
        attrib.set("units", "degree_C");

        int first = 0;
        String oldID, newID = null;
        for (int row = 0; row <= tempNRows; row++) { //yes, =n since comparing current to previous row
            oldID = newID;
            if (row < tempNRows) 
                newID = temp.getStringData(1, row);

            //save the station file?
            if (row == tempNRows || 
                (row > 0 && !newID.equals(oldID))) { //station ID has changed

                //find the matching site
                String oldIsland = null, oldSite = null;
                int siteRow = 0;
                while (siteRow < siteNRows) {
                    oldIsland = site.getStringData(0, siteRow);
                    oldSite = site.getStringData(1, siteRow);
                    String tID = oldIsland + " (" + oldSite + ")";
                    if (tID.equals(oldID))
                        break;
                    siteRow++;
                }
                Test.ensureNotEqual(siteRow, siteNRows, "");
                String tempID = "KFMTemperature_" + oldIsland + "_" + oldSite;
                tempID = String2.replaceAll(tempID, "'", "");
                tempID = String2.replaceAll(tempID, " ", "_");

                //add multiple copies of lon, lat, depth
                int tn = stationTime.size();
                String2.log("tn=" + tn);
                ((FloatArray)station.getColumn(0)).addN(tn, site.getFloatData(3, siteRow)); //x
                ((FloatArray)station.getColumn(1)).addN(tn, site.getFloatData(2, siteRow)); //y
                ((IntArray)  station.getColumn(2)).addN(tn, site.getIntData(  4, siteRow)); //z

                //setAttributes
                station.setAttributes(0, 1, 2, 3, //x,y,z,t
                    "Sea Temperature (Kelp Forest Monitoring, Channel Islands)", //boldTitle
                    "Station", //cdmDataType
                    DataHelper.ERD_CREATOR_EMAIL, //"Roy.Mendelssohn@noaa.gov", //creatorEmail
                    DataHelper.ERD_CREATOR_NAME,  //"NOAA NMFS SWFSC ERD",  //creatorName
                    DataHelper.ERD_CREATOR_URL,   //"http://www.pfel.noaa.gov", //creatorUrl
                    DataHelper.ERD_PROJECT,       
                    tempID, //id
                    "GCMD Science Keywords", //keywordsVocabulary,
                    "EARTH SCIENCE > Oceans > Ocean Temperature > Water Temperature", //keywords

                    //references   from 2006-12-19 email from Kushner
                    "Channel Islands National Parks Inventory and Monitoring information: " +
                        //was "http://www.nature.nps.gov/im/units/chis/ " +
                        "http://nature.nps.gov/im/units/medn . " +
                        "Kelp Forest Monitoring Protocols: " +
                        "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                    //summary  from 2006-12-19 email from Kushner
                    "The subtidal temperature data taken at Channel Islands National " +
                        "Park's Kelp Forest Monitoring Programs permanent monitoring sites.  Since " +
                        "1993, remote temperature loggers manufactured by Onset Computer Corporation " +
                        "were deployed at each site approximately 10-20 cm from the bottom in a " +
                        "underwater housing.  Since 1993, three models of temperature loggers " +
                        "(HoboTemp (tm), StowAway (R) and Tidbit(R) )were used to collect " +
                        "temperature data every 1-5 hours depending on the model used.",
                    //my old summary
                    //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                    //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                    //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                    //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                    //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                    "Channel Islands National Park, National Park Service", 
                    null); //timeLongName     use default long name) {

                //add the National Park Service disclaimer from 2006-12-19 email
                String license = station.globalAttributes().getString("license");
                license += "  National Park Service Disclaimer: " +
                    "The National Park Service shall not be held liable for " +
                    "improper or incorrect use of the data described and/or contained " +
                    "herein. These data and related graphics are not legal documents and " +
                    "are not intended to be used as such. The information contained in " +
                    "these data is dynamic and may change over time. The data are not " +
                    "better than the original sources from which they were derived. It is " +
                    "the responsibility of the data user to use the data appropriately and " +
                    "consistent within the limitation of geospatial data in general and " +
                    "these data in particular. The related graphics are intended to aid " +
                    "the data user in acquiring relevant data; it is not appropriate to " +
                    "use the related graphics as data. The National Park Service gives no " +
                    "warranty, expressed or implied, as to the accuracy, reliability, or " +
                    "completeness of these data. It is strongly recommended that these " +
                    "data are directly acquired from an NPS server and not indirectly " +
                    "through other sources which may have changed the data in some way. " +
                    "Although these data have been processed successfully on computer " +
                    "systems at the National Park Service, no warranty expressed or " +
                    "implied is made regarding the utility of the data on other systems " +
                    "for general or scientific purposes, nor shall the act of distribution " +
                    "constitute any such warranty. This disclaimer applies both to " +
                    "individual use of the data and aggregate use with other data.";
                station.globalAttributes().set("license", license);

                //save the station table
                station.ensureValid();
                station.saveAs4DNcWithStringVariable("c:/temp/kfm200801/KFMTemperature/" + tempID + ".nc", 
                    0, 1, 2, 3, 4);  //x,y,z,t, stringVariable=ID
                nStationsCreated++;

                //see what results are like 
                if (nStationsCreated == 1) {
                    Table tTable = new Table();
                    tTable.read4DNc("c:/temp/kfm200801/KFMTemperature/" + tempID + ".nc", null, 0, stationColumnName, 4);
                    String2.log("\nstation0=\n" + tTable.toString("row", 3));
                    //from site ascii file
                    //Anacapa	Admiral's Reef	34	00	200	N	119	25	520	W	16
                    //data from original ascii file
                    //11	Anacapa	Admiral's Reef	8/26/1993 0:00:00	12/30/1899 10:35:00	18.00 //first line
                    //11	Anacapa	Admiral's Reef	11/3/2001 0:00:00	12/30/1899 16:12:00	15.88 //random
                    //11	Anacapa	Admiral's Reef	7/13/2007 0:00:00	12/30/1899 11:25:00	16.89 //last line
                    DoubleArray ttimepa = (DoubleArray)tTable.getColumn(3);
                    double adjustTime = 7 * Calendar2.SECONDS_PER_HOUR; 
                    int trow;   
                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("1993-08-26 10:35:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 18f, "");

                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("2001-11-03 16:12:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 15.88f, "");

                    trow = ttimepa.indexOf("" + (Calendar2.isoStringToEpochSeconds("2007-07-13 11:25:00") + adjustTime), 0);
                    Test.ensureNotEqual(trow, -1, "");
                    Test.ensureEqual(tTable.getFloatData(0, trow), -119f - 25f/60f, "");
                    Test.ensureEqual(tTable.getFloatData(1, trow), 34f, "");
                    Test.ensureEqual(tTable.getFloatData(2, trow), 16f, "");
                    Test.ensureEqual(tTable.getStringData(4, trow), "Anacapa (Admiral's Reef)", "");
                    Test.ensureEqual(tTable.getFloatData(5, trow), 16.89f, "");
                }

                //set station table to 0 rows
                station.removeRows(0, station.nRows());
            }

            //add a row of data to station table
            if (row < tempNRows) {
                //t is adjusted from Pacific Daylight Savings Time to UTC (they are 7 hours ahead of PDST)
                //see metadata "summary" regarding this conversion.
                stationTime.add(temp.getDoubleData(0, row) + 7 * Calendar2.SECONDS_PER_HOUR); //t
                stationID.add(newID);
                stationTemperature.add(temp.getFloatData(2, row));
            }
        }
        String2.log("siteNRows=" + siteNRows);
        String2.log("nStationsCreated=" + nStationsCreated);
        String2.log("\n*** Projects.kfmTemperature200801 finished successfully.");
    }

    public static String convertSpeciesName(String oldName) {
        String newName = oldName;
        int sp2Po = newName.indexOf("  ");
        if (sp2Po > 0) newName = newName.substring(0, sp2Po);
        int pPo = newName.indexOf("(");
        if (pPo > 0)   newName = newName.substring(0, pPo);
        newName = String2.replaceAll(newName, ".", "");
        newName = String2.replaceAll(newName, ",", "");
        newName = String2.replaceAll(newName, ' ', '_');
        return newName;
    }

    /**
     * KFM - This is the second group of files from Kushner (see c:/temp/kfm, 
     * processed starting 2007-03-27). This is biological data.
     * <ul>
     * <li>I converted the .mdb file to separate tab-separated value files,
     *    one per table, in MS Access (see instructions below --
     *    or with &lt;65,000 rows just use clipboard to move data) with extension .tab.
     *   <p>In the future: to export Access tables to a format we like (tab separated ASCII files):
     *   <ul>
     *   <li>In Access, select the specified table
     *     <ul>
     *     <li>Choose "File : Export"
     *     <li>Pick the table
     *     <li>Choose "Save as type: Text Files"
     *     <li>Change the file's extension to .tab
     *     <li>Don't check "Save formatted"
     *     <li>Click "Export All"
     *     </ul>
     *   <li>Check "Delimited"
     *   <li>Click "Next"
     *   <li>Check "Tab"
     *   <li>Check "Include Field Names on First Row"
     *   <li>Choose "Text Qualifier: none"
     *   <li>Click "Next"
     *   <li>Click "Finish"
     *   </ul>
     * <li>In CoStat, I removed extraneous columns from .tab files and save as .tsv so 
     *   <br> data .tsv files are: col 1)siteName, 2) year, 3+) data columns
     *   <br> site .tsv file is: col 1)island, 2) siteName, 
     *       3) lat(deg. N), 4) lon(deg. E), 5) depth(m)
     *   <br>(lat and lon are just degree and minute info -- as before, at Kushner's
     *     request, no seconds data.
     * <li> In this method, load site table, and process a data file
     *    (convert siteName to Lon, Lat, Depth) and save as .nc file with metadata.
     * </ul>
     *
     * <p>In 2007-03-29 email, David Kushner said in response to my questions:
     * <pre>

I'm working on converting the latest KFM data files to another format
and adding metadata. It's going pretty well. I have a few questions...

***
I presume that I should obscure the exact locations of the stations by
just releasing the degrees and minutes (not seconds) for the lat and lon
values, as before.  If that has changed, please let me know.
      YES, THAT WORKS WELL FOR NOW AND APPRECIATE YOUR FORESIGHT ON THIS
***
Where does the Species number come from?  Is this your internal
numbering system -- a number that you have assigned to each species? Or,
are they standardized numbers from some standard numbering system?
      THEY ARE OUR INTERNAL NUMBER SYSTEM.  THEY DON'T CHANGE BUT THE
SPECIES NAMES DO WAY TOO OFTEN.  I ONLY INCLUDED THESE THINKING THEY MAY BE
USEFUL FOR YOU TO ORGANIZE THE DATA, BUT IF NOT NO WORRIES...  THEY MAY
COME IN HANDY TO HAVE LINKED IF WE UPDATE THE DATA EVERY YEAR AND THE NAMES
CHANGE.  PERHAPS I SHOULD SEND YOU A TABLE THAT HAS THE SPECIES CODE,
SPECIES NAME, AND COMMON NAME AND YOU CAN LINK THEM THIS WAY WHICH IS WHAT
WE DO.  WHATEVER YOU THINK IS BEST.

***
Is the reference information the same as for the temperature data?  If
not, please let me know how you want it to appear. The information
currently is:  YES, THE SAME.  WE ARE WORKING ON THE NPS WEB SITE NOW AND
THOSE LINKS MAY CHANGE, BUT I WILL NOTIFY YOU IF THEY DO.


Channel Islands National Parks Inventory and Monitoring information:
http://www.nature.nps.gov/im/units/chis/
(changing in early 2007 to http://nature.nps.gov/im/units/medn ).
Kelp Forest Monitoring Protocols:
http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf

.

***
Is the "courtesy" information still:
"Channel Islands National Park, National Park Service"?

[I take his no-comment as tacit acceptance.]
***
Should the license/disclaimer still be:
"National Park Service Disclaimer: " +
"The National Park Service shall not be held liable for " +
"improper or incorrect use of the data described and/or contained " +
"herein. These data and related graphics are not legal documents and " +
"are not intended to be used as such. The information contained in " +
"these data is dynamic and may change over time. The data are not " +
"better than the original sources from which they were derived. It is "  +
"the responsibility of the data user to use the data appropriately and " +
"consistent within the limitation of geospatial data in general and " +
"these data in particular. The related graphics are intended to aid " +
"the data user in acquiring relevant data; it is not appropriate to " +
"use the related graphics as data. The National Park Service gives no " +
"warranty, expressed or implied, as to the accuracy, reliability, or " +
"completeness of these data. It is strongly recommended that these " +
"data are directly acquired from an NPS server and not indirectly " +
"through other sources which may have changed the data in some way. " +
"Although these data have been processed successfully on computer " +
"systems at the National Park Service, no warranty expressed or " +
"implied is made regarding the utility of the data on other systems " +
"for general or scientific purposes, nor shall the act of distribution "  +
"constitute any such warranty. This disclaimer applies both to " +
"individual use of the data and aggregate use with other data.";

[I take his no-comment as tacit acceptance.]
</pre>
     * <p>I added the following info to the KFM_Site_Info.tsv file based
     *   on info in emails from Kushner:
     *   <br> San Miguel, Miracle Mile, 34.0166666666667, -120.4, 10
     *   <br>That station didn't have a temperature logger, so there wasn't
     *   an entry in the station file, but there is
     *   biological data for the station.
     * @throws Exception if trouble
     */ 
    public static void kfmBiological() throws Exception {
        String2.log("\n*** Projects.kfmBiological");

        //'_' in tsvNames will be converted to ' ' for tempID below
        //Station .nc files will be stored in subdirectories named tsvName.
        String tsvDir = "c:/temp/kfm/";
        //order not important
        String tsvNames[] = {"KFM_5m_Quadrat", "KFM_Quadrat", "KFM_Random_Point", "KFM_Transect"};

        //read KFM_Site_Info.tsv: col 0)island e.g., "Anacapa", 1=siteID (e.g., Admiral's Reef)", 
        //  2) lat(deg. N), 3) lon(deg. E), 4) depth(m)
        Table site = new Table();
        site.readASCII(tsvDir + "KFM_Site_Info.tsv", 
            0, 1, //col names on row 0, data on row 1
            null, null, null, null);
        StringArray sitePa = (StringArray)site.getColumn(1);
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(sitePa.get(0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.0166666666667f, ""); //lat, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(3, 0), -120.383333333333f, "");  //lon, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(4, 0), 13, "");   //depth

        //go through the source tab-separated-value files
        StringArray missingSites = new StringArray();
        for (int tsvI = 0; tsvI < tsvNames.length; tsvI++) {
            String tsvName = tsvNames[tsvI];
            String2.log("processing " + tsvDir + tsvName + ".tsv");

            //empty the results directory
            File2.deleteAllFiles(tsvDir + tsvName + "/");

            //read datafile into a table
            //col 0)siteName, e.g., "Admiral's Reef", 1) year 2+) data columns
            Table data = new Table();
            data.readASCII(tsvDir + tsvName + ".tsv", 
                //row 0=namesLine, 1=firstDataLine, nulls: limitations
                0, 1, null, null, null, null); 
            int dataNRows = data.nRows();
           
            //create x,y,z,t,id columns  (numeric coordinate columns are all doubles)
            DoubleArray xPa = new DoubleArray(dataNRows, false);
            DoubleArray yPa = new DoubleArray(dataNRows, false);
            DoubleArray zPa = new DoubleArray(dataNRows, false);
            DoubleArray tPa = new DoubleArray(dataNRows, false);
            StringArray idPa = new StringArray(dataNRows, false);
            for (int row = 0; row < dataNRows; row++) {
                String tSiteName = data.getStringData(0, row);
                int siteRow = sitePa.indexOf(tSiteName);
                if (siteRow == -1) {
                    int tpo = missingSites.indexOf(tSiteName);
                    if (tpo == -1) missingSites.add(tSiteName);
                    siteRow = 0; //clumsy, but lets me collect all the missing sites
                }

                xPa.add(site.getNiceDoubleData(3, siteRow));
                yPa.add(site.getNiceDoubleData(2, siteRow));
                zPa.add(site.getNiceDoubleData(4, siteRow));
                //they are just year #'s. no time zone issues.
                //Times are vague (may to oct), so assign to July 1 (middle of year).
                String tYear = data.getStringData(1, row);
                Test.ensureEqual(tYear.length(), 4, "Unexpected year=" + tYear + " on row=" + row);
                tPa.add(Calendar2.isoStringToEpochSeconds(tYear + "-07-01"));  
                idPa.add(site.getStringData(0, siteRow) + " (" + tSiteName + ")");
            }

            //put x,y,z,t,id columns in place
            data.removeColumn(0);
            data.addColumn(0, "LON",   xPa, new Attributes());
            data.addColumn(1, "LAT",   yPa, new Attributes());
            data.addColumn(2, "DEPTH", zPa, new Attributes());
            data.addColumn(3, "TIME",  tPa, new Attributes());
            data.addColumn(4, "ID",    idPa, new Attributes());
            data.columnAttributes(4).set("long_name", "Station Identifier");
            //data.columnAttributes(4).set("units", DataHelper.UNITLESS);

            //remove the year column
            Test.ensureEqual(data.getColumnName(5), "Year", "Unexpected col 5 name.");
            data.removeColumn(5);

            //add metadata for data columns
            //standardNames from http://cf-pcmdi.llnl.gov/documents/cf-standard-names/2/cf-standard-name-table.html
            //none seem relevant here
            for (int col = 5; col < data.nColumns(); col++) {
                String pm2 = " per square meter";  //must be all lowercase
                String colName = data.getColumnName(col);
                if (colName.toLowerCase().endsWith(pm2)) {
                    colName = colName.substring(0, colName.length() - pm2.length());
                    data.setColumnName(col, colName); 
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "m-2");
                } else if (colName.equals("Species")) {
                    data.columnAttributes(col).set("long_name", "Species");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("SpeciesName")) {
                    data.columnAttributes(col).set("long_name", "Species Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("CommonName")) {
                    data.columnAttributes(col).set("long_name", "Common Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Percent Cover")) {
                    data.columnAttributes(col).set("long_name", colName);
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else {
                    Test.error("Unexpected column name=" + colName);
                }
                //data.columnAttributes(col).set("long_name", "Sea Temperature");
                //data.columnAttributes(col).set("standard_name", "sea_water_temperature");
            }

            //summaries are verbatim (except for the first few words) 
            //from c:\content\kushner\NOAA Web page KFM protocol descriptions.doc
            //from Kushner's 2007-04-11 email.
            String summary = null; 
            if (tsvName.equals("KFM_5m_Quadrat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance of selected rare, clumped, sedentary indicator species. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The original measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tsvName.equals("KFM_Quadrat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance (density) of relatively abundant selected sedentary indicator species. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tsvName.equals("KFM_Random_Point")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has estimates of substrate composition and percent cover of selected algal and invertebrate taxa. " + 
    "The summary data presented here represents the mean percent cover of the indicator species at the site. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tsvName.equals("KFM_Transect")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance and distribution of rare and clumped organisms not adequately sampled by quadrats. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else Test.error("Unexpected tsvName=" + tsvName);

            //sort by id, x,y,z,t
            data.sort(new int[]{4,0,1,2,3}, new boolean[]{true, true, true, true, true});
            int sppCol = 6;
String2.log("sppCol name = " + data.getColumnName(sppCol));
            int commonCol = 7;
String2.log("commonCol name = " + data.getColumnName(commonCol));
            int dataCol = 8;
String2.log("dataCol name = " + data.getColumnName(dataCol));

            //find unique spp
            IntArray sppIndices = new IntArray();
            StringArray uniqueSpp = (StringArray)data.getColumn(sppCol).makeIndices(sppIndices);
            int nUniqueSpp = uniqueSpp.size();
            for (int sp = 0; sp < nUniqueSpp; sp++) {
                String newName = convertSpeciesName(uniqueSpp.get(sp));
                newName = String2.replaceAll(newName + " " + data.getColumnName(dataCol), ' ', '_');
                uniqueSpp.set(sp, newName);
            }
String2.log("uniqueSpp = " + uniqueSpp);

            //find unique years
            IntArray yearIndices = new IntArray();
            PrimitiveArray uniqueYear = data.getColumn(3).makeIndices(yearIndices);
            int nUniqueYear = uniqueYear.size();
String2.log("uniqueYear = " + uniqueYear);
            
            //make a separate file for each station
            int startRow = 0;
            for (int row = 1; row <= dataNRows; row++) {  //yes 1..=
                //id changed?
                if (row == dataNRows || //test this first
                    !data.getStringData(4, startRow).equals(data.getStringData(4, row))) {
                    
                    //make stationTable x,y,z(constant), t, col for each sp
                    Table stationTable = new Table();
                    data.globalAttributes().copyTo(stationTable.globalAttributes());
                    for (int col = 0; col < 5; col++) {
                        stationTable.addColumn(col, data.getColumnName(col), 
                            PrimitiveArray.factory(data.getColumn(col).getElementType(), dataCol, false), 
                            (Attributes)data.columnAttributes(col).clone());
                    }
                    for (int col = 0; col < nUniqueSpp; col++) {
                        stationTable.addColumn(5 + col, uniqueSpp.get(col), 
                            new FloatArray(), 
                            (Attributes)data.columnAttributes(dataCol).clone());
                        stationTable.columnAttributes(5 + col).set("long_name",
                            String2.replaceAll(uniqueSpp.get(col), '_', ' '));
                        int rowWithThisSpp = sppIndices.indexOf("" + col);
                        stationTable.columnAttributes(5 + col).set("comment",
                            "Common name: " + data.getStringData(commonCol, rowWithThisSpp));
                    }

                    //fill the stationTable with axis info and blanks
                    for (int tRow = 0; tRow < nUniqueYear; tRow++) {
                        //x,y,z,t,id
                        stationTable.getColumn(0).addDouble(data.getDoubleData(0, startRow));
                        stationTable.getColumn(1).addDouble(data.getDoubleData(1, startRow));
                        stationTable.getColumn(2).addDouble(data.getDoubleData(2, startRow));
                        stationTable.getColumn(3).addDouble(uniqueYear.getDouble(tRow));
                        stationTable.getColumn(4).addString(data.getStringData(4, startRow));
                        //spp
                        for (int col = 0; col < nUniqueSpp; col++) 
                            stationTable.getColumn(5 + col).addFloat(Float.NaN);
                    }

                    //fill the stationTable with data
                    for (int tRow = startRow; tRow < row; tRow++) {
                        float d = data.getFloatData(dataCol, tRow);
                        if (d < -9000)
                            Test.error("d=" + d + " is < -9000.");
                        stationTable.setFloatData(5 + sppIndices.get(tRow), 
                            yearIndices.get(tRow), d);
                    }

                    //setAttributes
                    String id = data.getStringData(4, startRow); //e.g., "San Miguel (Wyckoff Ledge)"
                    int pPo = id.indexOf('(');
                    Test.ensureNotEqual(pPo, -1, "'(' in id=" + id);
                    String island = id.substring(0, pPo - 1);
                    String station = id.substring(pPo + 1, id.length() - 1);
                    stationTable.setAttributes(0, 1, 2, 3, //x,y,z,t
                        String2.replaceAll(tsvName, '_', ' ') + " (Channel Islands)", //bold title
                            //don't make specific to this station; when aggregated, just one boldTitle will be used
                            //", " + island + ", " + station + ")", 
                        "Station", //cdmDataType
                        DataHelper.ERD_CREATOR_EMAIL,
                        DataHelper.ERD_CREATOR_NAME, 
                        DataHelper.ERD_CREATOR_URL,   
                        DataHelper.ERD_PROJECT,       
                        tsvName, //id    //don't make specific to this station; when aggregated, just one id will be used
                        "GCMD Science Keywords", //keywordsVocabulary,
                        //see http://gcmd.gsfc.nasa.gov/Resources/valids/gcmd_parameters.html
                        //there are plands and invertebrates, so habitat seems closest keyword
                        "EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat", //keywords

                        //references   from 2006-12-19 email from Kushner
                        "Channel Islands National Parks Inventory and Monitoring information: " +
                            "http://nature.nps.gov/im/units/medn . " +
                            "Kelp Forest Monitoring Protocols: " +
                            "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                        //summary  from 2006-12-19 email from Kushner
                        summary,
                        //my old summary
                        //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                        //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                        //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                        //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                        //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                        "Channel Islands National Park, National Park Service", 
                        null); //timeLongName     use default long name

                    //add the National Park Service disclaimer from 2006-12-19 email
                    String license = stationTable.globalAttributes().getString("license") +
                        "  National Park Service Disclaimer: " +
                        "The National Park Service shall not be held liable for " +
                        "improper or incorrect use of the data described and/or contained " +
                        "herein. These data and related graphics are not legal documents and " +
                        "are not intended to be used as such. The information contained in " +
                        "these data is dynamic and may change over time. The data are not " +
                        "better than the original sources from which they were derived. It is " +
                        "the responsibility of the data user to use the data appropriately and " +
                        "consistent within the limitation of geospatial data in general and " +
                        "these data in particular. The related graphics are intended to aid " +
                        "the data user in acquiring relevant data; it is not appropriate to " +
                        "use the related graphics as data. The National Park Service gives no " +
                        "warranty, expressed or implied, as to the accuracy, reliability, or " +
                        "completeness of these data. It is strongly recommended that these " +
                        "data are directly acquired from an NPS server and not indirectly " +
                        "through other sources which may have changed the data in some way. " +
                        "Although these data have been processed successfully on computer " +
                        "systems at the National Park Service, no warranty expressed or " +
                        "implied is made regarding the utility of the data on other systems " +
                        "for general or scientific purposes, nor shall the act of distribution " +
                        "constitute any such warranty. This disclaimer applies both to " +
                        "individual use of the data and aggregate use with other data.";
                    stationTable.globalAttributes().set("license", license);

                    //fix up the attributes
                    stationTable.globalAttributes().set("acknowledgement",
                        stationTable.globalAttributes().getString("acknowledgement") + ", " +
                        "Channel Islands National Park, National Park Service");

                    //review the table
                    if (tsvI == 0 && (startRow == 0 || row == dataNRows)) {
                        String2.log(stationTable.toString("row", 100));
                        String2.getStringFromSystemIn("Check if the file (above) is ok. Press Enter to continue ->");
                    }
                    String2.log("  startRow=" + startRow + " end=" + (row-1) + " island=" + island + " station=" + station); 

                    //save the data table    
                    String tFileName = tsvDir + tsvName + "/" + tsvName + "_" + 
                        String2.replaceAll(island, " ", "") + "_" + 
                        String2.replaceAll(station, " ", "") + ".nc";
                    tFileName = String2.replaceAll(tFileName, ' ', '_');
                    tFileName = String2.replaceAll(tFileName, "'", "");
                    stationTable.saveAs4DNcWithStringVariable(tFileName,0,1,2,3,4); 

                    startRow = row;
                }
            }
        }
        if (missingSites.size() > 0) {
            String2.log("\n*** Projects.kfmBiological FAILED. Missing sites=" + missingSites);
        } else {
            String2.log("\n*** Projects.kfmBiological finished successfully.");
        }
    }

    /**
     * This is the 4th processing (Jan 2008) of data from Kushner/KFM project.
     *
     * <p>This year I didn't convert .tab to .tsv in CoStat and rearrange columns.
     * I just rearranged the columns below.
     *
     * <p>In KFM1mQuadrat.tab line 2158, I replaced the missing stdDev and stdErr with NaNs.
     * <br>and line 3719: 8	Santa Cruz	Pelican Bay	11007.00	Parastichopus parvimensis	Warty sea cucumber	1983	NaN	NaN	NaN
     *
     * I will put the results in otter /u00/bob/kfm2008/ which mimics my c:/temp/kfm200801 .
     */
    public static void kfmBiological200801() throws Exception {
        String2.log("\n*** Projects.kfmBiological200801");

        //'_' in tabNames will be converted to ' ' for tempID below
        //Station .nc files will be stored in subdirectories named tabName.
        String tabDir = "c:/temp/kfm200801/";
        //order not important
        String tabNames[] = {"KFM5mQuadrat", "KFM1mQuadrat", "KFMRandomPointContact", "KFMBandTransect"};
        String titles[] = {"Survey, 5m Quadrat", "Survey, 1m Quadrat", "Survey, Random Point Contact", "Survey, Band Transect"};

        //read KFM_Site_Info.tsv: col 0)island e.g., "Anacapa", 1=siteID (e.g., Admiral's Reef)", 
        //  2) lat(deg. N), 3) lon(deg. E), 4) depth(m)
        Table site = new Table();
        site.readASCII(tabDir + "KFM_Site_Info.tsv",    //read siteInfo
            0, 1, //col names on row 0, data on row 1
            null, null, null, null);
        StringArray sitePa = (StringArray)site.getColumn(1);
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(sitePa.get(0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.0166666666667f, ""); //lat, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(3, 0), -120.383333333333f, "");  //lon, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(4, 0), 13, "");   //depth

        //go through the source tab-separated-value files
        StringArray missingSites = new StringArray();
        for (int tabI = 0; tabI < tabNames.length; tabI++) {
            String tabName = tabNames[tabI];
            String title = titles[tabI];
            String2.log("processing " + tabDir + tabName + ".tab");

            //empty the results directory
            File2.deleteAllFiles(tabDir + tabName + "/");

            //read datafile into a table
            //initially 0=SiteNumber 1=Island 2=SiteName 3=Species 4=SpeciesName
            //5=CommonName 6=Year 7=...Per Square Meter [8=StdDev] [9=Std Error]
            Table data = new Table();
            data.readASCII(tabDir + tabName + ".tab", 
                //row 0=namesLine, 1=firstDataLine, nulls: limitations
                0, 1, null, null, null, null); 
            int dataNRows = data.nRows();
           
            //create x,y,z,t,id columns  (numeric coordinate columns are all doubles)
            DoubleArray xPa = new DoubleArray(dataNRows, false);
            DoubleArray yPa = new DoubleArray(dataNRows, false);
            DoubleArray zPa = new DoubleArray(dataNRows, false);
            DoubleArray tPa = new DoubleArray(dataNRows, false);
            StringArray idPa = new StringArray(dataNRows, false);
            for (int row = 0; row < dataNRows; row++) {
                String tSiteName = data.getStringData(2, row);
                int siteRow = sitePa.indexOf(tSiteName);
                if (siteRow == -1) {
                    int tpo = missingSites.indexOf(tSiteName);
                    if (tpo == -1) missingSites.add(tSiteName);
                    siteRow = 0; //clumsy, but lets me collect all the missing sites
                }

                xPa.add(site.getNiceDoubleData(3, siteRow));
                yPa.add(site.getNiceDoubleData(2, siteRow));
                zPa.add(site.getNiceDoubleData(4, siteRow));
                //they are just year #'s. no time zone issues.
                //Times are vague (may to oct), so assign to July 1 (middle of year).
                String tYear = data.getStringData(6, row);
                Test.ensureEqual(tYear.length(), 4, "Unexpected year=" + tYear + " on row=" + row);
                tPa.add(Calendar2.isoStringToEpochSeconds(tYear + "-07-01"));  
                idPa.add(site.getStringData(0, siteRow) + " (" + tSiteName + ")");
            }

            //put x,y,z,t,id columns in place
            data.removeColumn(6); //Year
            data.removeColumn(2); //SiteName
            data.removeColumn(1); //Island
            data.removeColumn(0); //SiteNumber
            data.addColumn(0, "LON",   xPa, new Attributes());
            data.addColumn(1, "LAT",   yPa, new Attributes());
            data.addColumn(2, "DEPTH", zPa, new Attributes());
            data.addColumn(3, "TIME",  tPa, new Attributes());
            data.addColumn(4, "ID",    idPa, new Attributes());
            data.columnAttributes(4).set("long_name", "Station Identifier");
            //data.columnAttributes(4).set("units", DataHelper.UNITLESS);
            Test.ensureEqual(data.getColumnName(5), "Species", "");
            Test.ensureEqual(data.getColumnName(6), "SpeciesName", "");
            Test.ensureEqual(data.getColumnName(7), "CommonName", "");
            String newDataName = null, newDataUnits = null;
            String ts = data.getColumnName(8);
            if (ts.endsWith(" Per Square Meter")) {
                newDataName = ts.substring(0, ts.length() - " Per Square Meter".length());
                newDataUnits = "m-2";
            } else if (ts.endsWith(" Percent Cover")) {
                newDataName = ts.substring(0, ts.length() - " Percent Cover".length());
                newDataUnits = "percent cover";
            } else 
                Test.error(ts);
            Test.ensureEqual(data.getColumnName(9), "StdDev", "");
            Test.ensureEqual(data.getColumnName(10), "Std Error", "");
            int sppCol = 6;       
            int commonCol = 7;    
            int dataCol = 8;      
            int sdCol = 9;        
            int seCol = 10;       
            String2.log("newDataName=" + newDataName + " newDataUnits=" + newDataUnits);

            //add metadata for data columns
            //standardNames from http://cf-pcmdi.llnl.gov/documents/cf-standard-names/2/cf-standard-name-table.html
            //none seem relevant here
            for (int col = 5; col < data.nColumns(); col++) {
                String pm2 = " per square meter";  //must be all lowercase
                String colName = data.getColumnName(col);
                if (colName.toLowerCase().endsWith(pm2)) {
                    colName = colName.substring(0, colName.length() - pm2.length());
                    data.setColumnName(col, colName); 
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "m-2");
                } else if (colName.equals("Species")) {
                    data.columnAttributes(col).set("long_name", "Species");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("SpeciesName")) {
                    data.columnAttributes(col).set("long_name", "Species Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("CommonName")) {
                    data.columnAttributes(col).set("long_name", "Common Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Mean Percent Cover")) {
                    data.setColumnName(col, "Mean");
                    data.columnAttributes(col).set("long_name", "Mean");
                    data.columnAttributes(col).set("units", "percent cover");
                } else if (colName.equals("StdDev")) {
                    data.columnAttributes(col).set("long_name", "Standard Deviation");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Std Error")) {
                    data.setColumnName(col, "StdErr");
                    data.columnAttributes(col).set("long_name", "Standard Error");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else {
                    Test.error("Unexpected column name=" + colName);
                }
            }

            //summaries are verbatim (except for the first few words) 
            //from c:\content\kushner\NOAA Web page KFM protocol descriptions.doc
            //from Kushner's 2007-04-11 email.
            String summary = null; 
            if (tabName.equals("KFM5mQuadrat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance of selected rare, clumped, sedentary indicator species. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The original measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tabName.equals("KFM1mQuadrat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance (density) of relatively abundant selected sedentary indicator species. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tabName.equals("KFMRandomPointContact")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has estimates of substrate composition and percent cover of selected algal and invertebrate taxa. " + 
    "The summary data presented here represents the mean percent cover of the indicator species at the site. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else if (tabName.equals("KFMBandTransect")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance and distribution of rare and clumped organisms not adequately sampled by quadrats. " + 
    "The summary data presented here represents the mean density per square meter. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The actual measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else Test.error("Unexpected tabName=" + tabName);

            //sort by id, x,y,z,t
            data.sort(new int[]{4,0,1,2,3}, new boolean[]{true, true, true, true, true});

            //find unique spp
            IntArray sppIndices = new IntArray();
            StringArray uniqueSpp = (StringArray)data.getColumn(sppCol).makeIndices(sppIndices);
            int nUniqueSpp = uniqueSpp.size();
            for (int sp = 0; sp < nUniqueSpp; sp++) 
                uniqueSpp.set(sp, convertSpeciesName(uniqueSpp.get(sp)));
String2.log("uniqueSpp = " + uniqueSpp);

            //find unique years
            IntArray yearIndices = new IntArray();
            PrimitiveArray uniqueYear = data.getColumn(3).makeIndices(yearIndices);
            int nUniqueYear = uniqueYear.size();
String2.log("uniqueYear = " + uniqueYear);
            
            //make a separate file for each station
            int startRow = 0;
            for (int row = 1; row <= dataNRows; row++) {  //yes 1..=
                //id changed?
                if (row == dataNRows || //test this first
                    !data.getStringData(4, startRow).equals(data.getStringData(4, row))) {
                    
                    //make stationTable x,y,z(constant), t, col for each sp
                    Table stationTable = new Table();
                    data.globalAttributes().copyTo(stationTable.globalAttributes());
                    for (int col = 0; col < 5; col++) {
                        stationTable.addColumn(col, data.getColumnName(col), 
                            PrimitiveArray.factory(data.getColumn(col).getElementType(), dataCol, false), 
                            (Attributes)data.columnAttributes(col).clone());
                    }
                    //add data columns
                    for (int col = 0; col < nUniqueSpp; col++) {

                        String ulDataName = String2.replaceAll(newDataName, ' ', '_');
                        stationTable.addColumn(5 + col * 3 + 0, uniqueSpp.get(col) + "_" + ulDataName, 
                            new FloatArray(), 
                            (Attributes)data.columnAttributes(dataCol).clone());
                        stationTable.addColumn(5 + col * 3 + 1, uniqueSpp.get(col) + "_StdDev", 
                            new FloatArray(), 
                            (Attributes)data.columnAttributes(sdCol).clone());
                        stationTable.addColumn(5 + col * 3 + 2, uniqueSpp.get(col) + "_StdErr", 
                            new FloatArray(), 
                            (Attributes)data.columnAttributes(seCol).clone());

                        stationTable.columnAttributes(5 + col * 3 + 0).set("long_name",
                            String2.replaceAll(uniqueSpp.get(col) + "_" + newDataName, '_', ' '));
                        stationTable.columnAttributes(5 + col * 3 + 1).set("long_name",
                            String2.replaceAll(uniqueSpp.get(col) + "_StdDev", '_', ' '));
                        stationTable.columnAttributes(5 + col * 3 + 2).set("long_name",
                            String2.replaceAll(uniqueSpp.get(col) + "_StdErr", '_', ' '));

                        int rowWithThisSpp = sppIndices.indexOf("" + col);
                        stationTable.columnAttributes(5 + col * 3 + 0).set("comment",
                            "Common name: " + data.getStringData(commonCol, rowWithThisSpp));

                    }

                    //fill the stationTable with axis info and blanks
                    for (int tRow = 0; tRow < nUniqueYear; tRow++) {
                        //x,y,z,t,id
                        stationTable.getColumn(0).addDouble(data.getDoubleData(0, startRow));
                        stationTable.getColumn(1).addDouble(data.getDoubleData(1, startRow));
                        stationTable.getColumn(2).addDouble(data.getDoubleData(2, startRow));
                        stationTable.getColumn(3).addDouble(uniqueYear.getDouble(tRow));
                        stationTable.getColumn(4).addString(data.getStringData(4, startRow));
                        //spp
                        for (int col = 0; col < nUniqueSpp; col++) {
                            stationTable.getColumn(5 + col * 3 + 0).addFloat(Float.NaN);
                            stationTable.getColumn(5 + col * 3 + 1).addFloat(Float.NaN);
                            stationTable.getColumn(5 + col * 3 + 2).addFloat(Float.NaN);
                        }
                    }

                    //fill the stationTable with data
                    for (int tRow = startRow; tRow < row; tRow++) {
                        float tm = data.getFloatData(dataCol, tRow);
                        float td = data.getFloatData(sdCol, tRow);
                        float te = data.getFloatData(seCol, tRow);
                        if (tm < -9000)
                            Test.error("tm=" + tm + " is < -9000.");
                        stationTable.setFloatData(5 + sppIndices.get(tRow) * 3 + 0, yearIndices.get(tRow), tm);
                        stationTable.setFloatData(5 + sppIndices.get(tRow) * 3 + 1, yearIndices.get(tRow), td);
                        stationTable.setFloatData(5 + sppIndices.get(tRow) * 3 + 2, yearIndices.get(tRow), te);
                    }

                    //setAttributes
                    String id = data.getStringData(4, startRow); //e.g., "San Miguel (Wyckoff Ledge)"
                    int pPo = id.indexOf('(');
                    Test.ensureNotEqual(pPo, -1, "'(' in id=" + id);
                    String island = id.substring(0, pPo - 1);
                    String station = id.substring(pPo + 1, id.length() - 1);
                    stationTable.setAttributes(0, 1, 2, 3, //x,y,z,t
                        title + " (Kelp Forest Monitoring, Channel Islands)", //bold title
                            //don't make specific to this station; when aggregated, just one boldTitle will be used
                            //", " + island + ", " + station + ")", 
                        "Station", //cdmDataType
                        DataHelper.ERD_CREATOR_EMAIL,
                        DataHelper.ERD_CREATOR_NAME, 
                        DataHelper.ERD_CREATOR_URL,   
                        DataHelper.ERD_PROJECT,       
                        tabName, //id    //don't make specific to this station; when aggregated, just one id will be used
                        "GCMD Science Keywords", //keywordsVocabulary,
                        //see http://gcmd.gsfc.nasa.gov/Resources/valids/gcmd_parameters.html
                        //there are plands and invertebrates, so habitat seems closest keyword
                        "EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat", //keywords

                        //references   from 2006-12-19 email from Kushner
                        "Channel Islands National Parks Inventory and Monitoring information: " +
                            "http://nature.nps.gov/im/units/medn . " +
                            "Kelp Forest Monitoring Protocols: " +
                            "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                        //summary  from 2006-12-19 email from Kushner
                        summary,
                        //my old summary
                        //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                        //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                        //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                        //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                        //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                        "Channel Islands National Park, National Park Service", 
                        null); //timeLongName     use default long name

                    //add the National Park Service disclaimer from 2006-12-19 email
                    String license = stationTable.globalAttributes().getString("license") +
                        "  National Park Service Disclaimer: " +
                        "The National Park Service shall not be held liable for " +
                        "improper or incorrect use of the data described and/or contained " +
                        "herein. These data and related graphics are not legal documents and " +
                        "are not intended to be used as such. The information contained in " +
                        "these data is dynamic and may change over time. The data are not " +
                        "better than the original sources from which they were derived. It is " +
                        "the responsibility of the data user to use the data appropriately and " +
                        "consistent within the limitation of geospatial data in general and " +
                        "these data in particular. The related graphics are intended to aid " +
                        "the data user in acquiring relevant data; it is not appropriate to " +
                        "use the related graphics as data. The National Park Service gives no " +
                        "warranty, expressed or implied, as to the accuracy, reliability, or " +
                        "completeness of these data. It is strongly recommended that these " +
                        "data are directly acquired from an NPS server and not indirectly " +
                        "through other sources which may have changed the data in some way. " +
                        "Although these data have been processed successfully on computer " +
                        "systems at the National Park Service, no warranty expressed or " +
                        "implied is made regarding the utility of the data on other systems " +
                        "for general or scientific purposes, nor shall the act of distribution " +
                        "constitute any such warranty. This disclaimer applies both to " +
                        "individual use of the data and aggregate use with other data.";
                    stationTable.globalAttributes().set("license", license);

                    //fix up the attributes
                    stationTable.globalAttributes().set("acknowledgement",
                        stationTable.globalAttributes().getString("acknowledgement") + ", " +
                        "Channel Islands National Park, National Park Service");

                    //review the table
                    if (tabI == 0 && (startRow == 0 || row == dataNRows)) {
                        String2.log(stationTable.toString("row", 100));
                        String2.getStringFromSystemIn("Check if the file (above) is ok. Press Enter to continue ->");
                    }
                    String2.log("  startRow=" + startRow + " end=" + (row-1) + " island=" + island + " station=" + station); 

                    //save the data table    
                    String tFileName = tabDir + tabName + "/" + tabName + "_" + 
                        String2.replaceAll(island, " ", "") + "_" + 
                        String2.replaceAll(station, " ", "") + ".nc";
                    tFileName = String2.replaceAll(tFileName, ' ', '_');
                    tFileName = String2.replaceAll(tFileName, "'", "");
                    stationTable.saveAs4DNcWithStringVariable(tFileName,0,1,2,3,4); 

                    startRow = row;
                }
            }
        }
        if (missingSites.size() > 0) {
            String2.log("\n*** Projects.kfmBiological200801 FAILED. Missing sites=" + missingSites);
        } else {
            String2.log("\n*** Projects.kfmBiological200801 finished successfully.");
        }
    }

    /**
     * This is the 4th processing (Jan 2008) of data from Kushner/KFM project.
     *
     * I will put the results in otter /u00/bob/kfm2008/ which mimics my c:/temp/kfm200801 .
     */
    public static void kfmFishTransect200801() throws Exception {
        String2.log("\n*** Projects.kfmFishTransect200801");

        //Station .nc files will be stored in subdirectories named tabName.
        String tabDir = "c:/temp/kfm200801/";
        //order not important
        String tabNames[] = {"KFMFishTransect"};
        String boldTitles[] = {"Fish Survey, Transect"};

        //read KFM_Site_Info.tsv: col 0)island e.g., "Anacapa", 1=siteID (e.g., Admiral's Reef)", 
        //  2) lat(deg. N), 3) lon(deg. E), 4) depth(m)
        Table site = new Table();
        site.readASCII(tabDir + "KFM_Site_Info.tsv",    //read siteInfo
            0, 1, //col names on row 0, data on row 1
            null, null, null, null);
        StringArray sitePa = (StringArray)site.getColumn(1);
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(sitePa.get(0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.0166666666667f, ""); //lat, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(3, 0), -120.383333333333f, "");  //lon, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(4, 0), 13, "");   //depth

        //go through the source tab-separated-value files
        StringArray missingSites = new StringArray();
        for (int tabI = 0; tabI < tabNames.length; tabI++) {
            String tabName = tabNames[tabI];
            String2.log("processing " + tabDir + tabName + ".tab");

            //empty the results directory
            File2.deleteAllFiles(tabDir + tabName + "/");

            //read datafile into a table
            Table data = new Table();
            data.readASCII(tabDir + tabName + ".tab", 
                //row 0=namesLine, 1=firstDataLine, nulls: limitations
                0, 1, null, null, null, null); 
            int dataNRows = data.nRows();
            Test.ensureEqual(
                String2.toCSVString(data.getColumnNames()),
                "Year, IslandName, SiteName, Date, " +                            //0,1,2,3
                    "Species, Species Name, Adult/Juvenile/sex, " +               //4,5,6
                    "CommonName, Transect, Number fish per 100mX2mX30m transect", //7,8,9
                "");
          
            //create x,y,z,t,id columns  (numeric coordinate columns are all doubles)
            DoubleArray xPa = new DoubleArray(dataNRows, false);
            DoubleArray yPa = new DoubleArray(dataNRows, false);
            DoubleArray zPa = new DoubleArray(dataNRows, false);
            DoubleArray tPa = new DoubleArray(dataNRows, false);
            StringArray idPa = new StringArray(dataNRows, false);
            for (int row = 0; row < dataNRows; row++) {
                String tSiteName = data.getStringData(2, row);
                int siteRow = sitePa.indexOf(tSiteName);
                if (siteRow == -1) {
                    int tpo = missingSites.indexOf(tSiteName);
                    if (tpo == -1) missingSites.add(tSiteName);
                    siteRow = 0; //clumsy, but lets me collect all the missing sites
                }

                xPa.add(site.getNiceDoubleData(3, siteRow));
                yPa.add(site.getNiceDoubleData(2, siteRow));
                zPa.add(site.getNiceDoubleData(4, siteRow));
                double sec = Calendar2.gcToEpochSeconds(
                    Calendar2.parseUSSlash24Zulu(data.getStringData(3, row))); //throws Exception
                if (sec < 100 || sec > 2e9)
                    String2.log("row=" + row + " sec=" + sec + " Unexpected time=" + data.getStringData(3, row));
                tPa.add(sec);
                idPa.add(site.getStringData(0, siteRow) + " (" + tSiteName + ")");

                //combine SpeciesName, Adult/Juvenile/sex
                String tsp = data.getStringData(5, row);
                int sp2Po = tsp.indexOf("  "); //precedes description I don't keep
                if (sp2Po > 0) tsp = tsp.substring(0, sp2Po);
                int pPo = tsp.indexOf("("); //precedes description I don't keep
                if (pPo > 0)   tsp = tsp.substring(0, pPo); 
                tsp += "_" + data.getStringData(6, row); 
                tsp = String2.replaceAll(tsp, ".", "");
                tsp = String2.replaceAll(tsp, ",", "");
                tsp = String2.replaceAll(tsp, ' ', '_');
                data.setStringData(5, row, tsp);

                //ensure transect always 1
                Test.ensureEqual(data.getDoubleData(8, row), 1, "");
            }

            //put x,y,z,t,id columns in place
            //    "Year, IslandName, SiteName, Date, " +                        //0,1,2,3
            //    "Species, Species Name, Adult/Juvenile/sex, " +               //4,5,6
            //    "CommonName, Transect, Number fish per 100mX2mX30m transect", //7,8,9
            data.removeColumn(8); //Transect (always 1)
            data.removeColumn(6); //Adult
            data.removeColumn(4); //Species
            data.removeColumn(3); //Date
            data.removeColumn(2); //SiteName
            data.removeColumn(1); //Island
            data.removeColumn(0); //Year
            data.addColumn(0, "LON",   xPa, new Attributes());
            data.addColumn(1, "LAT",   yPa, new Attributes());
            data.addColumn(2, "DEPTH", zPa, new Attributes());
            data.addColumn(3, "TIME",  tPa, new Attributes());
            data.addColumn(4, "ID",    idPa, new Attributes());
            data.columnAttributes(4).set("long_name", "Station Identifier");
            //data.columnAttributes(4).set("units", DataHelper.UNITLESS);
            //now
            //    "LON, LAT, DEPTH, TIME, ID, " +                     //0,1,2,3,4
            //    "Species Name + Adult/Juvenile/sex, " +             //5
            //    CommonName, Number fish per 100mX2mX30m transect",  //6,7

            //add metadata for data columns
            //standardNames from http://cf-pcmdi.llnl.gov/documents/cf-standard-names/2/cf-standard-name-table.html
            //none seem relevant here
//Year	IslandName	SiteName	Date	Species	Species Name	Adult/Juvenile/sex	CommonName	Transect	Number fish per 100mX2mX30m transect
//1985	Anacapa	Admiral's Reef	8/30/1985 0:00:00	14001.00	Chromis punctipinnis	 Adult	Blacksmith Adult	1	224
            for (int col = 5; col < data.nColumns(); col++) {
                String colName = data.getColumnName(col);
                if (colName.equals("Species Name")) {
                    data.setColumnName(col, "SpeciesName");
                    data.columnAttributes(col).set("long_name", "Species Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("CommonName")) {
                    //no need for metadata; just used as comment
                } else if (colName.equals("Number fish per 100mX2mX30m transect")) {
                    data.setColumnName(col, "NumberOfFish");
                    data.columnAttributes(col).set("long_name", "Number of fish per 100mX2mX30m transect");
                    data.columnAttributes(col).set("units", "per 100mX2mX30m transect");
                } else {
                    Test.error("Unexpected column name=" + colName);
                }
            }

            //summaries are verbatim (except for the first few words) 
            //from c:\content\kushner\NOAA Web page KFM protocol descriptions.doc
            //from Kushner's 2007-04-11 email.
            String summary = null; 
            if (tabName.equals("KFMFishTransect")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the abundance of fish species. " + 
    "The original measurements were taken at various depths, " +
    "so the Depth data in this file is the depth of the station's temperature logger, which is a typical depth." ;
            else Test.error("Unexpected tabName=" + tabName);

            //sort by id, x,y,z,t
            data.sort(new int[]{4,0,1,2,3}, new boolean[]{true, true, true, true, true});
            int sppCol = 5;
String2.log("sppCol name = " + data.getColumnName(sppCol));
            int comCol = 6;
String2.log("comCol name = " + data.getColumnName(comCol));
            int dataCol = 7;
String2.log("dataCol name = " + data.getColumnName(dataCol));

            //find unique spp
            IntArray sppIndices = new IntArray();
            StringArray uniqueSpp = (StringArray)data.getColumn(sppCol).makeIndices(sppIndices);
            int nUniqueSpp = uniqueSpp.size();
String2.log("uniqueSpp = " + uniqueSpp);
            
            //make a separate file for each station
            int startRow = 0;
            DoubleArray uniqueTimes = new DoubleArray();
            for (int row = 0; row <= dataNRows; row++) {  //yes 0..=
                //id changed?
                if (row == dataNRows || //test this first
                    !data.getStringData(4, startRow).equals(data.getStringData(4, row))) {
                    
                    //make stationTable x,y,z(constant), t, col for each sp
                    Table stationTable = new Table();
                    data.globalAttributes().copyTo(stationTable.globalAttributes());
                    for (int col = 0; col < 5; col++) {
                        stationTable.addColumn(col, data.getColumnName(col), 
                            PrimitiveArray.factory(data.getColumn(col).getElementType(), dataCol, false), 
                            (Attributes)data.columnAttributes(col).clone());
                    }
                    for (int col = 0; col < nUniqueSpp; col++) {
                        stationTable.addColumn(5 + col, uniqueSpp.get(col), 
                            new IntArray(), 
                            (Attributes)data.columnAttributes(dataCol).clone());
                        stationTable.columnAttributes(5 + col).set("long_name",
                            "Number of " + String2.replaceAll(uniqueSpp.get(col), '_', ' '));
                        int rowWithThisSpp = sppIndices.indexOf("" + col);
                        stationTable.columnAttributes(5 + col).set("comment",
                            "Common name: " + data.getStringData(comCol, rowWithThisSpp));
                    }

                    //fill the stationTable with axis info and blanks
                    int nUniqueTimes = uniqueTimes.size();
                    uniqueTimes.sort();
                    for (int tRow = 0; tRow < nUniqueTimes; tRow++) {
                        //x,y,z,t,id
                        stationTable.getColumn(0).addDouble(data.getDoubleData(0, startRow));
                        stationTable.getColumn(1).addDouble(data.getDoubleData(1, startRow));
                        stationTable.getColumn(2).addDouble(data.getDoubleData(2, startRow));
                        stationTable.getColumn(3).addDouble(uniqueTimes.get(tRow));
                        stationTable.getColumn(4).addString(data.getStringData(4, startRow));
                        //spp
                        for (int col = 0; col < nUniqueSpp; col++) 
                            stationTable.getColumn(5 + col).addInt(Integer.MAX_VALUE);
                    }

                    //fill the stationTable with data
                    for (int tRow = startRow; tRow < row; tRow++) {
                        int stationRow = uniqueTimes.indexOf(tPa.get(tRow), 0);
                        int d = data.getIntData(dataCol, tRow);
                        if (d < 0)
                            Test.error("d=" + d + " is < 0.");
                        stationTable.setIntData(5 + sppIndices.get(tRow), stationRow, d);
                    }

                    //setAttributes
                    String id = data.getStringData(4, startRow); //e.g., "San Miguel (Wyckoff Ledge)"
                    int pPo = id.indexOf('(');
                    Test.ensureNotEqual(pPo, -1, "'(' in id=" + id);
                    String island = id.substring(0, pPo - 1);
                    String station = id.substring(pPo + 1, id.length() - 1);
                    stationTable.setAttributes(0, 1, 2, 3, //x,y,z,t
                        boldTitles[tabI] + " (Channel Islands)", //bold title
                            //don't make specific to this station; when aggregated, just one boldTitle will be used
                            //", " + island + ", " + station + ")", 
                        "Station", //cdmDataType
                        DataHelper.ERD_CREATOR_EMAIL,
                        DataHelper.ERD_CREATOR_NAME, 
                        DataHelper.ERD_CREATOR_URL,   
                        DataHelper.ERD_PROJECT,       
                        tabName, //id    //don't make specific to this station; when aggregated, just one id will be used
                        "GCMD Science Keywords", //keywordsVocabulary,
                        //see http://gcmd.gsfc.nasa.gov/Resources/valids/gcmd_parameters.html
                        //there are plands and invertebrates, so habitat seems closest keyword
                        "EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat", //keywords

                        //references   from 2006-12-19 email from Kushner
                        "Channel Islands National Parks Inventory and Monitoring information: " +
                            "http://nature.nps.gov/im/units/medn . " +
                            "Kelp Forest Monitoring Protocols: " +
                            "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                        //summary  from 2006-12-19 email from Kushner
                        summary,
                        //my old summary
                        //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                        //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                        //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                        //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                        //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                        "Channel Islands National Park, National Park Service", 
                        null); //timeLongName     use default long name
 
                    //add the National Park Service disclaimer from 2006-12-19 email
                    String license = stationTable.globalAttributes().getString("license") +
                        "  National Park Service Disclaimer: " +
                        "The National Park Service shall not be held liable for " +
                        "improper or incorrect use of the data described and/or contained " +
                        "herein. These data and related graphics are not legal documents and " +
                        "are not intended to be used as such. The information contained in " +
                        "these data is dynamic and may change over time. The data are not " +
                        "better than the original sources from which they were derived. It is " +
                        "the responsibility of the data user to use the data appropriately and " +
                        "consistent within the limitation of geospatial data in general and " +
                        "these data in particular. The related graphics are intended to aid " +
                        "the data user in acquiring relevant data; it is not appropriate to " +
                        "use the related graphics as data. The National Park Service gives no " +
                        "warranty, expressed or implied, as to the accuracy, reliability, or " +
                        "completeness of these data. It is strongly recommended that these " +
                        "data are directly acquired from an NPS server and not indirectly " +
                        "through other sources which may have changed the data in some way. " +
                        "Although these data have been processed successfully on computer " +
                        "systems at the National Park Service, no warranty expressed or " +
                        "implied is made regarding the utility of the data on other systems " +
                        "for general or scientific purposes, nor shall the act of distribution " +
                        "constitute any such warranty. This disclaimer applies both to " +
                        "individual use of the data and aggregate use with other data.";
                    stationTable.globalAttributes().set("license", license);

                    //fix up the attributes
                    stationTable.globalAttributes().set("acknowledgement",
                        stationTable.globalAttributes().getString("acknowledgement") + ", " +
                        "Channel Islands National Park, National Park Service");

                    //review the table
                    if (tabI == 0 && (startRow == 0 || row == dataNRows)) {
                        String2.log(stationTable.toString("row", 100));
                        String2.getStringFromSystemIn("Check if the file (above) is ok. Press Enter to continue ->");
                    }
                    String2.log("  startRow=" + startRow + " end=" + (row-1) + " island=" + island + " station=" + station); 

                    //do tests that look for known data
//1985	Anacapa	Admiral's Reef	8/30/1985 0:00:00	14003.00	Oxyjulis californica	 Adult	Seorita Adult	1	15
//2005	Anacapa	Admiral's Reef	8/22/2005 0:00:00	14003.00	Oxyjulis californica	 Adult	Seorita Adult	1	38
                    if (island.equals("Anacapa") && station.equals("Admiral's Reef")) {
                        int nRows = stationTable.nRows();
                        int testCol = stationTable.findColumnNumber("Oxyjulis_californica_Adult");
                        Test.ensureNotEqual(testCol, -1, "testCol");
                        double testTime = Calendar2.isoStringToEpochSeconds("1985-08-30");
                        int testRow = uniqueTimes.indexOf(testTime, 0);
                        Test.ensureNotEqual(testRow, -1, "testRow");
                        Test.ensureEqual(stationTable.getIntData(testCol, testRow), 15, "");

                        testTime = Calendar2.isoStringToEpochSeconds("2005-08-22");
                        testRow = uniqueTimes.indexOf(testTime, 0);
                        Test.ensureEqual(stationTable.getIntData(testCol, testRow), 38, "");
                        String2.log("passed Anacapa Admiral's Reef tests");
                    }

//2002	San Miguel	Wyckoff Ledge	9/26/2002 0:00:00	14005.00	Sebastes mystinus	 Adult	Blue rockfish Adult	1	8
//2003	San Miguel	Wyckoff Ledge	9/9/2003 0:00:00	14005.00	Sebastes mystinus	 Adult	Blue rockfish Adult	1	1
                    if (island.equals("San Miguel") && station.equals("Wyckoff Ledge")) {
                        int nRows = stationTable.nRows();
                        int testCol = stationTable.findColumnNumber("Sebastes_mystinus_Adult");
                        Test.ensureNotEqual(testCol, -1, "testCol");
                        double testTime = Calendar2.isoStringToEpochSeconds("2002-09-26");
                        int testRow = uniqueTimes.indexOf(testTime, 0);
                        Test.ensureNotEqual(testRow, -1, "testRow");
                        Test.ensureEqual(stationTable.getIntData(testCol, testRow), 8, "");

                        testTime = Calendar2.isoStringToEpochSeconds("2003-09-09");
                        testRow = uniqueTimes.indexOf(testTime, 0);
                        Test.ensureEqual(stationTable.getIntData(testCol, testRow), 1, "");
                        String2.log("passed San Miguel Syckoff Ledge tests");
                    }


                    //save the data table    
                    String tFileName = tabDir + tabName + "/" + tabName + "_" + 
                        String2.replaceAll(island, " ", "") + "_" + 
                        String2.replaceAll(station, " ", "") + ".nc";
                    tFileName = String2.replaceAll(tFileName, ' ', '_');
                    tFileName = String2.replaceAll(tFileName, "'", "");
                    stationTable.saveAs4DNcWithStringVariable(tFileName,0,1,2,3,4); 

                    startRow = row;

                    //clear uniqueTimes
                    uniqueTimes.clear();
                }
            
                //add time to uniqueTimes (for this station)?
                if (row < dataNRows) {
                    int po = uniqueTimes.indexOf(tPa.get(row), 0);
                    if (po < 0) 
                        uniqueTimes.add(tPa.get(row));
                }

            }
        }
        if (missingSites.size() > 0) {
            String2.log("\n*** Projects.kfmFishTransect200801 FAILED. Missing sites=" + missingSites);
        } else {
            String2.log("\n*** Projects.kfmFishTransect200801 finished successfully.");
        }
    }

    /**
     * KFM3 - This is the third group of files from Kushner (see c:/temp/kfm, 
     * processed starting 2007-06-26). This is biological data.
     *
     * Same procedure outline as kfmBiological, above.
     *
     * @throws Exception if trouble
     */ 
    public static void kfm3() throws Exception {
        String2.log("\n*** Projects.kfm3");

        //'_' in tsvNames will be converted to ' ' for tempID below
        //Station .nc files will be stored in subdirectories named tsvName.
        String tsvDir = "c:/temp/kfm3/";
        //order not important
        String tsvNames[] = {"KFM_SizeFrequencyGorgoniansAndStylaster", 
            "KFM_SizeFrequencyMacrocystis", "KFM_SizeFrequencyNaturalHabitat"};

        //read KFM_Site_Info.tsv: col 0)island e.g., "Anacapa", 1=siteID (e.g., Admiral's Reef)", 
        //  2) lat(deg. N), 3) lon(deg. E), 4) depth(m)
        Table site = new Table();
        site.readASCII("c:/temp/kfm/KFM_Site_Info.tsv", 
            0, 1, //col names on row 0, data on row 1
            null, null, null, null);
        StringArray sitePa = (StringArray)site.getColumn(1);
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(sitePa.get(0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.0166666666667f, ""); //lat, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(3, 0), -120.383333333333f, "");  //lon, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(4, 0), 13, "");   //depth

        //go through the source tab-separated-value files
        StringArray missingSites = new StringArray();
        for (int tsvI = 0; tsvI < tsvNames.length; tsvI++) {
            String tsvName = tsvNames[tsvI];
            String2.log("processing " + tsvDir + tsvName + ".tsv");

            //empty the results directory
            File2.deleteAllFiles(tsvDir + tsvName + "/");

            //read datafile into a table
            //col 0)siteName, e.g., "Admiral's Reef", 1) year 2+) data columns
            Table data = new Table();
            data.allowRaggedRightInReadASCII = true;  //macrocystis file has last col missing
            data.readASCII(tsvDir + tsvName + ".tsv", 
                //row 0=namesLine, 1=firstDataLine, nulls: limitations
                0, 1, null, null, null, null); 
            int dataNRows = data.nRows();
           
            //create x,y,z,t,id columns  (numeric coordinate columns are all doubles)
            DoubleArray xPa = new DoubleArray(dataNRows, false);
            DoubleArray yPa = new DoubleArray(dataNRows, false);
            DoubleArray zPa = new DoubleArray(dataNRows, false);
            DoubleArray tPa = new DoubleArray(dataNRows, false);
            StringArray idPa = new StringArray(dataNRows, false);
            for (int row = 0; row < dataNRows; row++) {
                String tSiteName = data.getStringData(0, row);
                int siteRow = sitePa.indexOf(tSiteName);
                if (siteRow == -1) {
                    int tpo = missingSites.indexOf(tSiteName);
                    if (tpo == -1) missingSites.add(tSiteName);
                    siteRow = 0; //clumsy, but lets me collect all the missing sites
                }

                xPa.add(site.getNiceDoubleData(3, siteRow));
                yPa.add(site.getNiceDoubleData(2, siteRow));
                zPa.add(site.getNiceDoubleData(4, siteRow));
                //they are just year #'s. no time zone issues.
                //Times are vague (may to oct), so assign to July 1 (middle of year).
                String tYear = data.getStringData(1, row);
                Test.ensureEqual(tYear.length(), 4, "Unexpected year=" + tYear + " on row=" + row);
                tPa.add(Calendar2.isoStringToEpochSeconds(tYear + "-07-01"));  
                idPa.add(site.getStringData(0, siteRow) + " (" + tSiteName + ")");
            }

            //put x,y,z,t,id columns in place
            data.removeColumn(0);
            data.addColumn(0, "LON",   xPa, new Attributes());
            data.addColumn(1, "LAT",   yPa, new Attributes());
            data.addColumn(2, "DEPTH", zPa, new Attributes());
            data.addColumn(3, "TIME",  tPa, new Attributes());
            data.addColumn(4, "ID",    idPa, new Attributes());
            data.columnAttributes(4).set("long_name", "Station Identifier");
            //data.columnAttributes(4).set("units", DataHelper.UNITLESS);

            //remove the year column
            Test.ensureEqual(data.getColumnName(5), "Year", "Unexpected col 5 name.");
            data.removeColumn(5);

            //add metadata for data columns
            //standardNames from http://cf-pcmdi.llnl.gov/documents/cf-standard-names/2/cf-standard-name-table.html
            //none seem relevant here
            for (int col = 5; col < data.nColumns(); col++) {
                String colName = data.getColumnName(col);
                if (colName.equals("Species")) {
                    data.columnAttributes(col).set("long_name", "Species");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("SpeciesName")) {
                    data.columnAttributes(col).set("long_name", "Species Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("CommonName")) {
                    data.columnAttributes(col).set("long_name", "Common Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Marker")) {
                    data.columnAttributes(col).set("long_name", colName);
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Stipes")) {
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "count");
                } else if (colName.equals("Height") || //in Macrocystis and Gorg&Styl
                           colName.equals("Width")) {
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "cm");
                } else if (colName.equals("Size")) {  //in NaturalHabitat
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "mm");
                } else {
                    Test.error("Unexpected column name=" + colName);
                }
                //data.columnAttributes(col).set("long_name", "Sea Temperature");
                //data.columnAttributes(col).set("standard_name", "sea_water_temperature");
            }

            //summaries are verbatim (except for the first few words) 
            //from c:\content\kushner\NOAA Web page KFM protocol descriptions.doc
            //from Kushner's 2007-04-11 email.
            String summary = null; 
            if (tsvName.equals("KFM_SizeFrequencyGorgoniansAndStylaster")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of corals at selected locations in the Channel Islands National Park. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else if (tsvName.equals("KFM_SizeFrequencyMacrocystis")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of kelp at selected locations in the Channel Islands National Park. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else if (tsvName.equals("KFM_SizeFrequencyNaturalHabitat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of selected animal species at selected locations in the Channel Islands National Park. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else Test.error("Unexpected tsvName=" + tsvName);

            //sort by id, x,y,z,t, spp#
            data.sort(new int[]{4,0,1,2,3,5}, new boolean[]{true, true, true, true, true, true});
            int sppCol = 6;
String2.log("sppCol name = " + data.getColumnName(sppCol));

            //make a separate file for each station
            int startRow = 0;
            for (int row = 1; row <= dataNRows; row++) {  //yes 1..=
                //id changed?
                if (row == dataNRows || //test this first
                    !data.getStringData(4, startRow).equals(data.getStringData(4, row))) {
                    
                    //make stationTable x,y,z(constant), t, col for each sp
                    int tnRows = row - startRow;
                    Table stationTable = new Table();
                    data.globalAttributes().copyTo(stationTable.globalAttributes());
                    for (int col = 0; col < data.nColumns(); col++) {
                        PrimitiveArray oldColumn = data.getColumn(col);
                        Class elementType = oldColumn.getElementType();
                        PrimitiveArray newColumn = 
                            PrimitiveArray.factory(elementType, tnRows, false);
                        stationTable.addColumn(col, data.getColumnName(col), 
                            newColumn, 
                            (Attributes)data.columnAttributes(col).clone());

                        //fill the stationTable with data
                        boolean isString = elementType == String.class;
                        for (int tRow = startRow; tRow < row; tRow++) {
                            if (isString)
                                newColumn.addString(oldColumn.getString(tRow));
                            else newColumn.addDouble(oldColumn.getDouble(tRow));
                        }
                    }

                    //setAttributes
                    String id = data.getStringData(4, startRow); //e.g., "San Miguel (Wyckoff Ledge)"
                    int pPo = id.indexOf('(');
                    Test.ensureNotEqual(pPo, -1, "'(' in id=" + id);
                    String island = id.substring(0, pPo - 1);
                    String station = id.substring(pPo + 1, id.length() - 1);
                    stationTable.setAttributes(0, 1, 2, 3, //x,y,z,t
                        String2.replaceAll(tsvName, '_', ' ') + " (Channel Islands)", //bold title
                            //don't make specific to this station; when aggregated, just one boldTitle will be used
                            //", " + island + ", " + station + ")", 
                        "Station", //cdmDataType
                        DataHelper.ERD_CREATOR_EMAIL,
                        DataHelper.ERD_CREATOR_NAME, 
                        DataHelper.ERD_CREATOR_URL,   
                        DataHelper.ERD_PROJECT,       
                        tsvName, //id    //don't make specific to this station; when aggregated, just one id will be used
                        "GCMD Science Keywords", //keywordsVocabulary,
                        //see http://gcmd.gsfc.nasa.gov/Resources/valids/gcmd_parameters.html
                        //there are plands and invertebrates, so habitat seems closest keyword
                        "EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat", //keywords

                        //references   from 2006-12-19 email from Kushner
                        "Channel Islands National Parks Inventory and Monitoring information: " +
                            "http://nature.nps.gov/im/units/medn . " +
                            "Kelp Forest Monitoring Protocols: " +
                            "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                        //summary  from 2006-12-19 email from Kushner
                        summary,
                        //my old summary
                        //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                        //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                        //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                        //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                        //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                        "Channel Islands National Park, National Park Service", 
                        null); //timeLongName     use default long name

                    //add the National Park Service disclaimer from 2006-12-19 email
                    String license = stationTable.globalAttributes().getString("license") +
                        "  National Park Service Disclaimer: " +
                        "The National Park Service shall not be held liable for " +
                        "improper or incorrect use of the data described and/or contained " +
                        "herein. These data and related graphics are not legal documents and " +
                        "are not intended to be used as such. The information contained in " +
                        "these data is dynamic and may change over time. The data are not " +
                        "better than the original sources from which they were derived. It is " +
                        "the responsibility of the data user to use the data appropriately and " +
                        "consistent within the limitation of geospatial data in general and " +
                        "these data in particular. The related graphics are intended to aid " +
                        "the data user in acquiring relevant data; it is not appropriate to " +
                        "use the related graphics as data. The National Park Service gives no " +
                        "warranty, expressed or implied, as to the accuracy, reliability, or " +
                        "completeness of these data. It is strongly recommended that these " +
                        "data are directly acquired from an NPS server and not indirectly " +
                        "through other sources which may have changed the data in some way. " +
                        "Although these data have been processed successfully on computer " +
                        "systems at the National Park Service, no warranty expressed or " +
                        "implied is made regarding the utility of the data on other systems " +
                        "for general or scientific purposes, nor shall the act of distribution " +
                        "constitute any such warranty. This disclaimer applies both to " +
                        "individual use of the data and aggregate use with other data.";
                    stationTable.globalAttributes().set("license", license);

                    //fix up the attributes
                    stationTable.globalAttributes().set("acknowledgement",
                        stationTable.globalAttributes().getString("acknowledgement") + ", " +
                        "Channel Islands National Park, National Park Service");

                    //review the table
                    String2.log("  startRow=" + startRow + " end=" + (row-1) + 
                        " island=" + island + " station=" + station); 

                    //save the data table    
                    String tFileName = tsvDir + tsvName + "/" + tsvName + "_" + 
                        String2.replaceAll(island, " ", "") + "_" + 
                        String2.replaceAll(station, " ", "") + ".nc";
                    tFileName = String2.replaceAll(tFileName, ' ', '_');
                    tFileName = String2.replaceAll(tFileName, "'", "");
                    stationTable.saveAsFlatNc(tFileName, "row"); 
                    //if (startRow == 0 || row == data.nRows()) {
                    //    String2.log("\n  table:" + tFileName + "\n" + stationTable);
                    //    String2.getStringFromSystemIn(                            
                    //        "Check if the file (above) is ok. Press Enter to continue ->");
                    //}

                    startRow = row;
                }
            }
        }
        if (missingSites.size() > 0) {
            String2.log("\n*** Projects.kfm3 FAILED. Missing sites=" + missingSites);
        } else {
            String2.log("\n*** Projects.kfm3 finished successfully.");
        }
    }

    /**
     * KFM3 - This is the third group of files from Kushner (see c:/temp/kfm, 
     * processed starting 2007-06-26). This is biological data.
     *
     * Same procedure outline as kfmBiological, above.
     *
     * I will put the results in otter /u00/bob/kfm2008/ which mimics my c:/temp/kfm200801 .
     *
     * @throws Exception if trouble
     */ 
    public static void kfmSizeFrequency200801() throws Exception {
        String2.log("\n*** Projects.kfmSizeFrequency200801");

        //'_' in tabNames will be converted to ' ' for tempID below
        //Station .nc files will be stored in subdirectories named tabName.
        String tabDir = "c:/temp/kfm200801/";
        //order not important
        String tabNames[] = {
            //"KFMSizeFrequencyGorgoniansAndStylaster", 
            //"KFMSizeFrequencyMacrocystis", 
            "KFMSizeFrequencyNaturalHabitat"};
        String boldTitles[] = {
            //"Size and Frequency of Gorgonians And Stylaster", 
            //"Size and Frequency of Macrocystis", 
            "Size and Frequency, Natural Habitat"};

        //read KFM_Site_Info.tsv: col 0)island e.g., "Anacapa", 1=siteID (e.g., Admiral's Reef)", 
        //  2) lat(deg. N), 3) lon(deg. E), 4) depth(m)
        //Island	SiteName	Lat	Lon	Depth (meters)
        //San Miguel	Wyckoff Ledge	34.0166666666667	-120.383333333333	13
        Table site = new Table();
        site.readASCII(tabDir + "KFM_Site_Info.tsv", //copied from last year
            0, 1, //col names on row 0, data on row 1
            null, null, null, null);
        StringArray sitePa = (StringArray)site.getColumn(1);
        Test.ensureEqual(site.getStringData(0, 0), "San Miguel", "");
        Test.ensureEqual(sitePa.get(0), "Wyckoff Ledge", "");
        Test.ensureEqual(site.getFloatData(2, 0), 34.0166666666667f, ""); //lat, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(3, 0), -120.383333333333f, "");  //lon, !!! already rounded to nearest minute at Kushner's request
        Test.ensureEqual(site.getFloatData(4, 0), 13, "");   //depth

        //go through the source tab-separated-value files
        StringArray missingSites = new StringArray();
        for (int tabI = 0; tabI < tabNames.length; tabI++) {
            String tabName = tabNames[tabI];
            String boldTitle = boldTitles[tabI];
            String2.log("processing " + tabDir + tabName + ".tab");

            //empty the results directory
            File2.deleteAllFiles(tabDir + tabName + "/");

            //read datafile into a table
            //0=SiteNumber	1=IslandName	2=SiteName	3=Year	4=Species	5=Species Name	6=CommonName	7=Size mm
            //1	San Miguel	Wyckoff Ledge	1985	9002.00	Haliotis rufescens	Red abalone	210
            Table data = new Table();
            data.allowRaggedRightInReadASCII = true;  //macrocystis file has last col missing
            data.readASCII(tabDir + tabName + ".tab", 
                //row 0=namesLine, 1=firstDataLine, nulls: limitations
                0, 1, null, null, null, null); 
            int dataNRows = data.nRows();
            Test.ensureEqual(data.getColumnName(0), "SiteNumber", "");
            Test.ensureEqual(data.getColumnName(1), "IslandName", "");
            Test.ensureEqual(data.getColumnName(2), "SiteName", "");
            Test.ensureEqual(data.getColumnName(3), "Year", "");
            Test.ensureEqual(data.getColumnName(4), "Species", "");
            Test.ensureEqual(data.getColumnName(5), "Species Name", "");
            Test.ensureEqual(data.getColumnName(6), "CommonName", "");
            Test.ensureEqual(data.getColumnName(7), "Size mm", "");
           
            //create x,y,z,t,id columns  (numeric coordinate columns are all doubles)
            DoubleArray xPa = new DoubleArray(dataNRows, false);
            DoubleArray yPa = new DoubleArray(dataNRows, false);
            DoubleArray zPa = new DoubleArray(dataNRows, false);
            DoubleArray tPa = new DoubleArray(dataNRows, false);
            StringArray idPa = new StringArray(dataNRows, false);
            for (int row = 0; row < dataNRows; row++) {
                String tSiteName = data.getStringData(2, row);
                int siteRow = sitePa.indexOf(tSiteName);
                if (siteRow == -1) {
                    int tpo = missingSites.indexOf(tSiteName);
                    if (tpo == -1) missingSites.add(tSiteName);
                    siteRow = 0; //clumsy, but lets me collect all the missing sites
                }

                xPa.add(site.getNiceDoubleData(3, siteRow));
                yPa.add(site.getNiceDoubleData(2, siteRow));
                zPa.add(site.getNiceDoubleData(4, siteRow));
                //they are just year #'s. no time zone issues.
                //Times are vague (may to oct), so assign to July 1 (middle of year).
                String tYear = data.getStringData(3, row);
                Test.ensureEqual(tYear.length(), 4, "Unexpected year=" + tYear + " on row=" + row);
                tPa.add(Calendar2.isoStringToEpochSeconds(tYear + "-07-01"));  
                idPa.add(site.getStringData(0, siteRow) + " (" + tSiteName + ")");
            }

            //remove the year, siteName, islandName, site# column
            data.removeColumn(3);
            data.removeColumn(2);
            data.removeColumn(1);
            data.removeColumn(0);

            //put x,y,z,t,id columns in place
            data.removeColumn(0);
            data.addColumn(0, "LON",   xPa, new Attributes());
            data.addColumn(1, "LAT",   yPa, new Attributes());
            data.addColumn(2, "DEPTH", zPa, new Attributes());
            data.addColumn(3, "TIME",  tPa, new Attributes());
            data.addColumn(4, "ID",    idPa, new Attributes());
            data.columnAttributes(4).set("long_name", "Station Identifier");
            //data.columnAttributes(4).set("units", DataHelper.UNITLESS);

            //add metadata for data columns
            //standardNames from http://cf-pcmdi.llnl.gov/documents/cf-standard-names/2/cf-standard-name-table.html
            //none seem relevant here
            for (int col = 5; col < data.nColumns(); col++) {
                String colName = data.getColumnName(col);
                if (colName.equals("Species")) {
                    data.columnAttributes(col).set("long_name", "Species");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Species Name")) {
                    data.columnAttributes(col).set("long_name", "Species Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("CommonName")) {
                    data.columnAttributes(col).set("long_name", "Common Name");
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Marker")) {
                    data.columnAttributes(col).set("long_name", colName);
                    //data.columnAttributes(col).set("units", DataHelper.UNITLESS);
                } else if (colName.equals("Stipes")) {
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "count");
                } else if (colName.equals("Height") || //in Macrocystis and Gorg&Styl
                           colName.equals("Width")) {
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "cm");
                } else if (colName.equals("Size mm")) {  //in NaturalHabitat
                    data.setColumnName(col, "Size");
                    data.columnAttributes(col).set("long_name", colName);
                    data.columnAttributes(col).set("units", "mm");
                } else {
                    Test.error("Unexpected column name=" + colName);
                }
                //data.columnAttributes(col).set("long_name", "Sea Temperature");
                //data.columnAttributes(col).set("standard_name", "sea_water_temperature");
            }

            //summaries are verbatim (except for the first few words) 
            //from c:\content\kushner\NOAA Web page KFM protocol descriptions.doc
            //from Kushner's 2007-04-11 email.
            String summary = null; 
            if (tabName.equals("KFMSizeFrequencyGorgoniansAndStylaster")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of corals at selected locations. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else if (tabName.equals("KFMSizeFrequencyMacrocystis")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of kelp at selected locations. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else if (tabName.equals("KFMSizeFrequencyNaturalHabitat")) summary = 
    "This dataset from the Channel Islands National Park's Kelp Forest Monitoring Program has measurements of the size of selected animal species at selected locations. " + 
    "Sampling is conducted annually between the months of May-October, " +
    "so the Time data in this file is July 1 of each year (a nominal value). " +
    "The size frequency measurements were taken within 10 meters of the " +  //depth description from Kushner's 6/29/2007 email
    "transect line at each site.  Depths at the site vary some, but we describe " +
    "the depth of the site along the transect line where that station's " +
    "temperature logger is located, a typical depth for the site.";
            else Test.error("Unexpected tabName=" + tabName);

            //sort by id, x,y,z,t, spp#
            data.sort(new int[]{4,0,1,2,3,5}, new boolean[]{true, true, true, true, true, true});
            int sppCol = 6;
String2.log("sppCol name = " + data.getColumnName(sppCol));

            //make a separate file for each station
            int startRow = 0;
            for (int row = 1; row <= dataNRows; row++) {  //yes 1..=
                //id changed?
                if (row == dataNRows || //test this first
                    !data.getStringData(4, startRow).equals(data.getStringData(4, row))) {
                    
                    //make stationTable x,y,z(constant), t, col for each sp
                    int tnRows = row - startRow;
                    Table stationTable = new Table();
                    data.globalAttributes().copyTo(stationTable.globalAttributes());
                    for (int col = 0; col < data.nColumns(); col++) {
                        PrimitiveArray oldColumn = data.getColumn(col);
                        Class elementType = oldColumn.getElementType();
                        PrimitiveArray newColumn = 
                            PrimitiveArray.factory(elementType, tnRows, false);
                        stationTable.addColumn(col, data.getColumnName(col), 
                            newColumn, 
                            (Attributes)data.columnAttributes(col).clone());

                        //fill the stationTable with data
                        boolean isString = elementType == String.class;
                        for (int tRow = startRow; tRow < row; tRow++) {
                            if (isString)
                                newColumn.addString(oldColumn.getString(tRow));
                            else newColumn.addDouble(oldColumn.getDouble(tRow));
                        }
                    }

                    //setAttributes
                    String id = data.getStringData(4, startRow); //e.g., "San Miguel (Wyckoff Ledge)"
                    int pPo = id.indexOf('(');
                    Test.ensureNotEqual(pPo, -1, "'(' in id=" + id);
                    String island = id.substring(0, pPo - 1);
                    String station = id.substring(pPo + 1, id.length() - 1);
                    stationTable.setAttributes(0, 1, 2, 3, //x,y,z,t
                        boldTitle + " (Kelp Forest Monitoring, Channel Islands)", //bold title
                            //don't make specific to this station; when aggregated, just one boldTitle will be used
                            //", " + island + ", " + station + ")", 
                        "Station", //cdmDataType
                        DataHelper.ERD_CREATOR_EMAIL,
                        DataHelper.ERD_CREATOR_NAME, 
                        DataHelper.ERD_CREATOR_URL,   
                        DataHelper.ERD_PROJECT,       
                        tabName, //id    //don't make specific to this station; when aggregated, just one id will be used
                        "GCMD Science Keywords", //keywordsVocabulary,
                        //see http://gcmd.gsfc.nasa.gov/Resources/valids/gcmd_parameters.html
                        //there are plands and invertebrates, so habitat seems closest keyword
                        "EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat", //keywords

                        //references   from 2006-12-19 email from Kushner
                        "Channel Islands National Parks Inventory and Monitoring information: " +
                            "http://nature.nps.gov/im/units/medn . " +
                            "Kelp Forest Monitoring Protocols: " +
                            "http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .",

                        //summary  from 2006-12-19 email from Kushner
                        summary,
                        //my old summary
                        //"Temperatures were recorded by David Kushner (David_Kushner@nps.gov) " + 
                        //    "using Onset Computer Corp. temperature loggers, accurate to +/- 0.2 C. The raw time values " +
                        //    "(Pacific Daylight Savings Time) were converted to Zulu time by adding 7 hours and then stored in this file. " +
                        //    "LAT and LON values were stored without seconds values to obscure the station's exact location.",

                        //courtesy, see 2006-12-12 email, but order reversed to current in 2006-12-19 email from Kushner
                        "Channel Islands National Park, National Park Service", 
                        null); //timeLongName     use default long name

                    //add the National Park Service disclaimer from 2006-12-19 email
                    String license = stationTable.globalAttributes().getString("license") +
                        "  National Park Service Disclaimer: " +
                        "The National Park Service shall not be held liable for " +
                        "improper or incorrect use of the data described and/or contained " +
                        "herein. These data and related graphics are not legal documents and " +
                        "are not intended to be used as such. The information contained in " +
                        "these data is dynamic and may change over time. The data are not " +
                        "better than the original sources from which they were derived. It is " +
                        "the responsibility of the data user to use the data appropriately and " +
                        "consistent within the limitation of geospatial data in general and " +
                        "these data in particular. The related graphics are intended to aid " +
                        "the data user in acquiring relevant data; it is not appropriate to " +
                        "use the related graphics as data. The National Park Service gives no " +
                        "warranty, expressed or implied, as to the accuracy, reliability, or " +
                        "completeness of these data. It is strongly recommended that these " +
                        "data are directly acquired from an NPS server and not indirectly " +
                        "through other sources which may have changed the data in some way. " +
                        "Although these data have been processed successfully on computer " +
                        "systems at the National Park Service, no warranty expressed or " +
                        "implied is made regarding the utility of the data on other systems " +
                        "for general or scientific purposes, nor shall the act of distribution " +
                        "constitute any such warranty. This disclaimer applies both to " +
                        "individual use of the data and aggregate use with other data.";
                    stationTable.globalAttributes().set("license", license);

                    //fix up the attributes
                    stationTable.globalAttributes().set("acknowledgement",
                        stationTable.globalAttributes().getString("acknowledgement") + ", " +
                        "Channel Islands National Park, National Park Service");

                    //review the table
                    String2.log("  startRow=" + startRow + " end=" + (row-1) + 
                        " island=" + island + " station=" + station); 

                    //save the data table    
                    String tFileName = tabDir + tabName + "/" + tabName + "_" + 
                        String2.replaceAll(island, " ", "") + "_" + 
                        String2.replaceAll(station, " ", "") + ".nc";
                    tFileName = String2.replaceAll(tFileName, ' ', '_');
                    tFileName = String2.replaceAll(tFileName, "'", "");
                    stationTable.saveAsFlatNc(tFileName, "row"); 
                    //if (startRow == 0 || row == data.nRows()) {
                    //    String2.log("\n  table:" + tFileName + "\n" + stationTable);
                    //    String2.getStringFromSystemIn(                            
                    //        "Check if the file (above) is ok. Press Enter to continue ->");
                    //}

                    startRow = row;
                }
            }
        }
        if (missingSites.size() > 0) {
            String2.log("\n*** Projects.kfmSizeFrequency200801 FAILED. Missing sites=" + missingSites);
        } else {
            String2.log("\n*** Projects.kfmSizeFrequency200801 finished successfully.");
        }
    }

    /** 
     * A special project for David Kushner. 
     * This extracts data from 5 of my standard ndbc buoy files in the format Kushner wants.
     * 12/06
     */ 
    public static void kushner() throws Exception {
        int id[] = {46023, 46025, 46053, 46054, 46063, 46069};
        int nId = id.length;
        for (int idi = 0; idi < nId; idi++) { 
            String2.log("\nID=" + id[idi]);
            //original names
              ///* 0*/"LON", "LAT", "DEPTH", "TIME", "ID", //use names that Lynn used in file that worked
              ///* 5*/"WD", "WSPD", "GST", "WVHT", "DPD", //most common name in ndbc files
              ///*10*/"APD", "MWD", "BAR", "ATMP", "WTMP", 
              ///*15*/"DEWP", "VIS", "PTDY", "TIDE", "WSPU", 
              ///*20*/"WSPV"};
            ///* 0*/"degrees_east", "degrees_north", "m", Calendar2.SECONDS_SINCE_1970, DataHelper.UNITLESS, 
            ///* 5*/"degrees_true", "m s-1", "m s-1", "m", "s", 
            ///*10*/"s", "degrees_true", "hPa", "degree_C", "degree_C", 
            ///*15*/"degree_C", "km","hPa", "m", "m s-1", 
            ///*20*/"m s-1"};
            String desiredColumns[] = {"TIME", "WD", "WSPD", "GST",  "WVHT", "DPD",  "APD",  "ATMP", "WTMP"};
            Table table = new Table();
            table.read4DNc("c:/temp/kushner/NDBC_" + id[idi] + "_met.nc", null, 1, stationColumnName, 4);
            String2.log("colNames=" + String2.toCSVString(table.getColumnNames()));

            //pluck out desired columns
            //wants:
            //TIME\tWDIR\tWSPD\tGST\tWVHT\tDPD\tAPD\tATMP\tWTMP
            //12/1/2006\t15:00:00\t260\t9.7\t11.7\t2.3\t16\t4.4\t60.4\t61
            Table tTable = new Table();
            for (int col = 0; col < desiredColumns.length; col++) 
                tTable.addColumn(col, desiredColumns[col], table.getColumn(table.findColumnNumber(desiredColumns[col])));
            table = tTable;
            Test.ensureEqual(table.getColumnNames(), desiredColumns, "");
          
            //populate newTime
            StringArray newTime = new StringArray();        
            DoubleArray oldTime = (DoubleArray)table.getColumn(0);
            int nRows = oldTime.size();
            for (int row = 0; row < nRows; row++) {
                //US slash-style date, 24 hour time
                double d = oldTime.get(row);
                Test.ensureNotEqual(d, Double.NaN, "");
                GregorianCalendar gc = Calendar2.epochSecondsToGc(d);
                newTime.add(Calendar2.formatAsUSSlash24(gc));
            }

            //insert the new time column
            table.setColumn(0, newTime);

            //write out the file
            table.saveAsTabbedASCII("c:/temp/kushner/NDBC_" + id[idi] + "_met.asc"); 
            String2.log(table.toString("row", 5));

        }
    }

    /**
     * This is the controlling program for finding all the .nc files
     * in subdirectories or url and writing catalog.xml-style info to fileName.
     *
     * @throws Exception if trouble
     */
    public static void ssc() throws Exception {
        String url = "http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/"; 
        String fileName = "c:/temp/ssc.xml";

        Writer writer = new FileWriter(fileName);
        oneSsc(writer,
            "http://biloxi-bay.ssc.hpc.msstate.edu",
            "/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/"); // ANO001/
        writer.close();
        String2.log(String2.readFromFile(fileName)[1]);
        String2.log("fileName=" + fileName);

    }
    /**
     * This is called by ssc to process the ssc information at the baseUrl+url:
     * either handling the referenced .nc files or
     * calling oneSsc recursively to handle subdirectories.
     *
     * @param writer 
     * @param baseUrl e.g., http://biloxi-bay.ssc.hpc.msstate.edu
     * @param url e.g., /dods-bin/dapnav_main.py/WCOS/nmsp/wcos/ANO001/2005/  
     *    (already percentEncoded as needed)
     */
    public static void oneSsc(Writer writer, String baseUrl, String url) throws Exception { 
        String2.log("oneSsc entering " + url);
        String sar[] = SSR.getUrlResponse(baseUrl + url);
        int line = 0;
        //read to "parent directory" line
        while (line < sar.length && sar[line].indexOf("parent directory") < 0)
            line++;
        line++;
        if (line == sar.length)
            throw new Exception("No 'parent directory' found\n" + baseUrl + url + 
                "\n" + String2.toNewlineString(sar));

        String aStart = "<a href=\"";
        while (true) {

            //are we done with this file?
            if (line >= sar.length || sar[line].startsWith("</pre>")) {
                String2.log("  oneSsc leaving " + url);
                return;
            }

            //is it a file?    sample is one line...
            //ANO001_021MTBD000R00_20050617.nc        
            //[<a href="/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/ANO001/2005/
            //ANO001_021MTBD000R00_20050617.nc">File information</a>] 
            //[<a href="http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/pyt.sh/WCOS/nmsp/wcos/ANO001/2005/ANO001_021MTBD000R00_20050617.nc">OPeNDAP direct</a>] 
            //Thu Feb 23 13:48:30 2006          627120 
            String s = sar[line];
            //String2.log("  line#" + line + "=" + s.substring(0, Math.min(70, s.length())));
            int po;
            int po2 = s.indexOf("\">OPeNDAP direct</a>");
            if (po2 > 0) {
                po = s.lastIndexOf(aStart, po2);
                s = s.substring(po + aStart.length(), po2);
                String2.log("  opendap=" + s);
                /*
                //get nTimes
                //isn't there a better way to get nTimes?
                String dds = SSR.getUrlResponseString(s + ".dds");
                po = dds.indexOf("Time = ");
                po2 = dds.indexOf("]", po);
                int nTimes = String2.parseInt(dds.substring(po + 7, po2));
                if (nTimes == Integer.MAX_VALUE)
                    throw new Exception("Unexpected Time from dds=\n" + dds);
                DConnect dConnect = new DConnect(s, true, 1, 1);
                double beginTime = OpendapHelper.getDoubleArray(dConnect, "?Time[0]")[0]; 
                double endTime   = OpendapHelper.getDoubleArray(dConnect, "?Time[" + (nTimes-1) + "]")[0]; 
                String2.log("  Time n=" + nTimes + " begin=" + beginTime + " end=" + endTime);
                */
                writeSsc(writer, s);
                //return;  //to just see first file, during development 

            } else if (s.startsWith(aStart)) {
                //is it a dir?
                //from http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/ANO001/2005/
                //<a href="/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/ANO001/2005/">2005/  </a>  Sat May 27 22:19:30 2006            4096 
                po = 0;
                po2 = s.indexOf("\">");
                s = s.substring(po + aStart.length(), po2);
                oneSsc(writer, baseUrl, s);
            } else {
                throw new Exception("Unexpected line:\n" + s);
            }

            line++;
        }
    }

    public static String lastNcName = null;

    /**
     * This is called by oneSsc to write the catalog.xml-style xml related
     * to ncUrl to the writer.
     *
     * @param writer
     * @param ncUrl e.g., http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/dapnav_main.py/WCOS/nmsp/wcos/ANO001/2005/
     *     ANO001_021MTBD000R00_20050617.nc
     * @throws Exception if trouble
     */
    public static void writeSsc(Writer writer, String ncUrl) throws Exception {
        String nameExt = File2.getNameAndExtension(ncUrl);
        if (nameExt.length() != 32 || !nameExt.endsWith(".nc") || nameExt.charAt(17) != 'R' ||
            nameExt.charAt(6) != '_')
            throw new Exception("Improper format for ncUrl=" + ncUrl);
        String name6 = nameExt.substring(0, 6);
        int    depth = String2.parseInt(nameExt.substring(14, 17));
        String year  = nameExt.substring(21, 25);

        String lastName6 = lastNcName == null? "" : lastNcName.substring(0, 6);
        int    lastDepth = lastNcName == null? -99999 : String2.parseInt(lastNcName.substring(14, 17));
        String lastYear  = lastNcName == null? "" : lastNcName.substring(21, 25);

        //end previous inner dataset?
        if (!name6.equals(lastName6) || !year.equals(lastYear) || depth != lastDepth) {
            //end this dataset
            if (lastNcName != null)
                writer.write(
                "        </aggregation>\n" +
                "      </netcdf>\n" +
                "    </dataset>\n" +
                "\n");
        }

        //outer dataset  (change in name6 or year)
        if (!name6.equals(lastName6) || !year.equals(lastYear)) {
            //end previous outer dataset
            if (lastNcName != null)
                writer.write(
                "  </dataset>\n" +
                "\n");

            //start new outer dataset
            writer.write(
                "  <dataset name=\"WCOS [?Point Ano Nuevo CA] (" + name6 + ") " + year + "\">\n");
        }

        //start the inner dataset?
        if (!name6.equals(lastName6) || !year.equals(lastYear) || depth != lastDepth) {
                writer.write(
                "    <dataset name=\"Measurement at " + depth + "m\" " +
                    "ID=\"WCOS/temp/" + year + "_" + name6 + "_" + depth + "m\" " +
                    "urlPath=\"WCOS/temp/" + year + "_" + name6 + "_" + depth + "m\">\n" +
                "      <serviceName>ncdods</serviceName>\n" +
                "      <!-- <serviceName>wcs</serviceName> -->\n" +
                "      <netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\">\n" +
                "        <variable name=\"Time\" shape=\"Time\" type=\"double\">\n" +
                "          <attribute name=\"units\" value=\"seconds since 1970-01-01 00:00:00\"/>\n" +
                "          <attribute name=\"_CoordinateAxisType\" value=\"Time\" />\n" +
                "        </variable>\n" +
                "        <aggregation dimName=\"Time\" type=\"joinExisting\">\n" +
                "          <variableAgg name=\"T\" />\n");
        }

        //add the file to the aggregation
        writer.write(
                "          <netcdf location=\"" + ncUrl + "\" />\n");


        lastNcName = nameExt;
    }

    /**
     * This adds metadata and a time dimension to SODA data files.
     * See http://www.atmos.umd.edu/~ocean/
     * ncdump of sample source file
<pre>
netcdf SODA_1.4.2_200112 {
dimensions:
        lon = 720 ;
        lat = 330 ;
        depth = 40 ;
        time = UNLIMITED ; // (1 currently)
variables:
        float temp(depth, lat, lon) ;
                temp:long_name = "TEMPERATURE" ;
                temp:units = "deg. C" ;
                temp:_FillValue = -9.99e+033f ;
        float salt(depth, lat, lon) ;
                salt:long_name = "SALINITY(ppt)" ;
                salt:units = "frac. by wt. less" ;
                salt:_FillValue = -9.99e+033f ;
        float u(depth, lat, lon) ;
                u:long_name = "ZONAL VELOCITY" ;
                u:units = "cm/sec" ;
                u:_FillValue = -9.99e+033f ;
        float v(depth, lat, lon) ;
                v:long_name = "MERIDIONAL VELOCITY" ;
                v:units = "cm/sec" ;
                v:_FillValue = -9.99e+033f ;
        float taux(lat, lon) ;
                taux:long_name = "TAU X" ;
                taux:units = "dynes/cm^2" ;
                taux:_FillValue = -9.99e+033f ;
        float tauy(lat, lon) ;
                tauy:long_name = "TAU Y" ;
                tauy:units = "dynes/cm^2" ;
                tauy:_FillValue = -9.99e+033f ;
        float ssh(lat, lon) ;
                ssh:long_name = "SEA LEVEL HEIGHT" ;
                ssh:units = "cm" ;
                ssh:_FillValue = -9.99e+033f ;
        double lon(lon) ;
                lon:units = "degrees_east" ;
        double lat(lat) ;
                lat:units = "degrees_north" ;
        double depth(depth) ;
                depth:units = "meters" ;
                depth:positive = "down" ;
        float time(time) ;
                time:units = "months" ;

// global attributes:
                :title = "SODA - POP 1.4.2 Assimilation TAMU/UMD" ;
}
</pre>
     *
     * @param sodaVersion e.g., 1.4.2
     * @param oldDir e.g., F:/SODA_1.4.2/
     * @param newDir e.g., F:/SODA_1.4.2nc/
     * @throws Exception if trouble
     */
    public static void soda(String sodaVersion, String oldDir, String newDir) throws Exception {

        //get a list of files
        String tempDir = "c:/temp/";
        String[] fileNames = RegexFilenameFilter.list(oldDir, "(.+cdf|.+cdf.gz)");
        NetcdfFile oldFile = null;
        NetcdfFileWriteable newFile = null;
        try {

            //for each file
            for (int fn = 0; fn < fileNames.length; fn++) { 
            //for (int fn = 0; fn < 1; fn++) { 
                String2.log("converting " + fileNames[fn]); //e.g., SODA_1.4.2_195806.cdf
    
                int po = fileNames[fn].lastIndexOf('_');
                int year  = String2.parseInt(fileNames[fn].substring(po + 1, po + 5));
                int month = String2.parseInt(fileNames[fn].substring(po + 5, po + 7)); //1..
                int months = (year - 1950) * 12 + month - 1;
//if (year < 2007) continue;
//if (year == 2007 && month < 2) continue;
                String2.log("  year=" + year + " month=" + month + " monthsSinceJan1950=" + months);

                //if .gz, make a temp file
                String cdfDir = oldDir;
                String cdfName = oldDir + fileNames[fn];
                boolean gzipped = cdfName.endsWith(".gz");
                if (gzipped) {
                    SSR.unGzip(oldDir + fileNames[fn], tempDir, true, 90);
                    cdfDir = tempDir;
                    cdfName = fileNames[fn].substring(0, fileNames[fn].length() - 3);
                }                    

                if (fn == 0) String2.log("\noldFile=" + NcHelper.dumpString(cdfDir + cdfName, false) + "\n");

                //open the old file
                oldFile = NcHelper.openFile(cdfDir + cdfName);

                //open the new file
                String newName = cdfName.substring(0, cdfName.length() - 3) + "nc";
                newFile = NetcdfFileWriteable.createNew(newDir + newName);

                //find old dimensions
                Dimension oldTimeDimension  = oldFile.findDimension("time");
                Dimension oldDepthDimension = oldFile.findDimension("depth");
                Dimension oldLatDimension   = oldFile.findDimension("lat");
                Dimension oldLonDimension   = oldFile.findDimension("lon");

                //find variables
                List<Variable> vars = oldFile.getVariables();

                //create the dimensions
                Dimension timeDimension  = newFile.addDimension("time", 1);
                Dimension depthDimension = newFile.addDimension("depth", oldDepthDimension.getLength());
                Dimension latDimension   = newFile.addDimension("lat",   oldLatDimension.getLength());
                Dimension lonDimension   = newFile.addDimension("lon",   oldLonDimension.getLength());

                //define each variable
                double minLon = Double.NaN, maxLon = Double.NaN, lonSpacing = Double.NaN;
                double minLat = Double.NaN, maxLat = Double.NaN, latSpacing = Double.NaN;
                double minDepth = Double.NaN, maxDepth = Double.NaN;

                for (int v = 0; v < vars.size(); v++) {
                    Variable var = vars.get(v);
                    String varName = var.getName();
                    Attributes atts = new Attributes(); 
                    NcHelper.getVariableAttributes(var, atts);
                    Dimension dimensions[];
                    DataType dataType = var.getDataType();

                    //if lon 
                    if (varName.equals("lon")) {
                        dimensions = new Dimension[1];
                        dimensions[0] = var.getDimension(0);

                        PrimitiveArray pa = NcHelper.getPrimitiveArray(var);
                        minLon = pa.getDouble(0);
                        maxLon = pa.getDouble(pa.size() - 1);
                        if (pa.isEvenlySpaced().length() == 0)
                            lonSpacing = (maxLon - minLon) / (pa.size() - 1);

                        atts.add("_CoordinateAxisType", "Lon");
                        atts.add("actual_range", new DoubleArray(new double[]{minLon, maxLon}));
                        atts.add("axis", "X");
                        atts.add("coordsys", "geographic");
                        atts.add("long_name", "Longitude");
                        atts.add("standard_name", "longitude");
                        atts.add("units", "degrees_east");

                    //if lat 
                    } else if (varName.equals("lat")) {
                        dimensions = new Dimension[1];
                        dimensions[0] = var.getDimension(0);

                        PrimitiveArray pa = NcHelper.getPrimitiveArray(var);
                        minLat = pa.getDouble(0);
                        maxLat = pa.getDouble(pa.size() - 1);
                        if (pa.isEvenlySpaced().length() == 0)
                            latSpacing = (maxLat - minLat) / (pa.size() - 1);

                        atts.add("_CoordinateAxisType", "Lat");
                        atts.add("actual_range", new DoubleArray(new double[]{minLat, maxLat}));
                        atts.add("axis", "Y");
                        atts.add("coordsys", "geographic");
                        atts.add("long_name", "Latitude");
                        atts.add("standard_name", "latitude");
                        atts.add("units", "degrees_north");                    

                    //if depth
                    } else if (varName.equals("depth")) {
                        dimensions = new Dimension[1];
                        dimensions[0] = var.getDimension(0);

                        PrimitiveArray pa = NcHelper.getPrimitiveArray(var);
                        minDepth = pa.getDouble(0);
                        maxDepth = pa.getDouble(pa.size() - 1);

                        atts.add("_CoordinateAxisType", "Height");
                        atts.add("_CoordinateZisPositive", "down");
                        atts.add("actual_range", new DoubleArray(new double[]{minDepth, maxDepth}));
                        atts.add("axis", "Z");
                        atts.add("long_name", "Depth");
                        atts.add("positive", "down");
                        atts.add("standard_name", "depth");
                        atts.add("units", "m");

                    //if time
                    } else if (varName.equals("time")) {
                        dimensions = new Dimension[1];
                        dimensions[0] = timeDimension;

                        dataType = DataType.INT; //the only var that changes dataType

                        //ensure time size == 1;
                        PrimitiveArray pa = NcHelper.getPrimitiveArray(var);
                        if (pa.size() != 1)
                            throw new Exception("time size=" + pa.size() + "\n" + pa);

                        atts.add("_CoordinateAxisType", "Time");
                        atts.add("axis", "T");
                        atts.add("long_name", "Time");
                        atts.add("standard_name", "time");
                        atts.add("time_origin", "15-JAN-1950 00:00:00");
                        atts.add("units", "months since 1950-01-15T00:00:00Z");                    

                    //other variables
                    } else {

                        //add time dimension
                        int rank = var.getRank();
                        dimensions = new Dimension[rank + 1];
                        dimensions[0] = timeDimension;
                        for (int r = 0; r < rank; r++)
                            dimensions[r + 1] = var.getDimension(r);

                        atts.add("missing_value", atts.getFloat("_FillValue"));
                        if (varName.equals("temp")) {
                            Test.ensureEqual(atts.getString("units"), "deg. C", "");
                            atts.add("long_name", "Sea Water Temperature");
                            atts.add("standard_name", "sea_water_temperature");
                            atts.add("units", "degree_C");

                        } else if (varName.equals("salt")) {
                            if (atts.getString("units").equals("frac. by wt. less")) {
                                //atts.add("units", "frac. by wt. less"); //???
                            } else if (atts.getString("units").equals("g/kg")) {
                                atts.add("units", "g kg-1"); //???
                            } else {
                                Test.error("Unexpected salt units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Salinity");
                            atts.add("standard_name", "sea_water_salinity");

                        } else if (varName.equals("u")) {
                            if (atts.getString("units").equals("cm/sec")) {
                                atts.add("units", "cm s-1");
                            } else if (atts.getString("units").equals("m/sec")) {
                                atts.add("units", "m s-1");
                            } else {
                                Test.error("Unexpected u units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Zonal Velocity");
                            atts.add("standard_name", "sea_water_x_velocity");

                        } else if (varName.equals("v")) {
                            if (atts.getString("units").equals("cm/sec")) {
                                atts.add("units", "cm s-1");
                            } else if (atts.getString("units").equals("m/sec")) {
                                atts.add("units", "m s-1");
                            } else {
                                Test.error("Unexpected v units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Meridional Velocity");
                            atts.add("standard_name", "sea_water_y_velocity");

                        } else if (varName.equals("w")) {
                            if (atts.getString("units").equals("cm/sec")) {
                                atts.add("units", "cm s-1");
                            } else if (atts.getString("units").equals("m/sec")) {
                                atts.add("units", "m s-1");
                            } else {
                                Test.error("Unexpected w units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Vertical Velocity");
                            atts.add("standard_name", "sea_water_z_velocity"); //not offical standard name, but direct extension of x and y


                        } else if (varName.equals("utrans")) {
                            if (atts.getString("units").equals("degC/sec")) {
                                atts.add("units", "degree_C s-1");
                            } else {
                                Test.error("Unexpected utrans units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Zonal Temperature Transport");  //TEMP-> Temperature???
                            atts.add("standard_name", "eastward_ocean_heat_transport"); //???

                        } else if (varName.equals("vtrans")) {
                            if (atts.getString("units").equals("degC/sec")) {
                                atts.add("units", "degree_C s-1");
                            } else {
                                Test.error("Unexpected vtrans units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Meridional Temperature Transport"); //TEMP-> Temperature???
                            atts.add("standard_name", "northward_ocean_heat_transport"); //???

                        } else if (varName.equals("hflx")) {
                            if (atts.getString("units").equals("watts/m^2")) {
                                atts.add("units", "watt m-2");
                            } else {
                                Test.error("Unexpected hflx units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Surface Heat Flux");  
                            atts.add("standard_name", "surface_downward_heat_flux_in_sea_water"); //???

                        } else if (varName.equals("wflx")) {
                            if (atts.getString("units").equals("m/year")) {
                                atts.add("units", "m year-1");
                            } else {
                                Test.error("Unexpected wflx units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Surface Water Flux");
                            atts.add("standard_name", "surface_downward_water_flux"); //???

                        } else if (varName.equals("CFC11")) {
                            if (atts.getString("units").equals("mmol/m**3")) {
                                atts.add("units", "mmole m-3");
                            } else {
                                Test.error("Unexpected vtrans units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "CFC11 Concentration"); 
                            atts.add("standard_name", "mole_concentration"); //not standard!!! but close

                        } else if (varName.equals("taux")) {
                            if (atts.getString("units").equals("dynes/cm^2")) {
                                atts.add("units", "dynes cm-2");  //convert to Pa???
                            } else if (atts.getString("units").equals("N/m^2")) {
                                atts.add("units", "N m-2");
                            } else {
                                Test.error("Unexpected taux units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Zonal Wind Stress"); //???
                            atts.add("standard_name", "surface_downward_eastward_stress"); //???

                        } else if (varName.equals("tauy")) {
                            if (atts.getString("units").equals("dynes/cm^2")) {
                                atts.add("units", "dynes cm-2");  //convert to Pa???
                            } else if (atts.getString("units").equals("N/m^2")) {
                                atts.add("units", "N m-2");
                            } else {
                                Test.error("Unexpected tauy units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Meridional Wind Stress"); //???
                            atts.add("standard_name", "surface_downward_northward_stress"); //???

                        } else if (varName.equals("ssh")) {
                            if (atts.getString("units").equals("cm")) {
                            } else if (atts.getString("units").equals("m")) {
                            } else {
                                Test.error("Unexpected ssh units=" + atts.getString("units"));
                            }
                            atts.add("long_name", "Sea Surface Height");
                            atts.add("standard_name", "sea_surface_height_above_geoid"); //???
                        } else {
                            throw new Exception("Unexpected varName=" + varName);
                        }
                    }

                    //define var in new file
                    newFile.addVariable(varName, dataType, dimensions); 
                    String attNames[] = atts.getNames();
                    for (int i = 0; i < attNames.length; i++) 
                        newFile.addVariableAttribute(varName, 
                            NcHelper.getAttribute(attNames[i], atts.get(attNames[i])));
                }

                //define GLOBAL metadata 
                Attributes gatts = new Attributes();
                String cDate = Calendar2.getCurrentISODateStringZulu();
                NcHelper.getGlobalAttributes(oldFile, gatts);
                gatts.add("acknowledgement", "NSF, NASA, NOAA"); //from http://www.atmos.umd.edu/~ocean/reanalysis.pdf
                gatts.add("cdm_data_type", "Grid");
                gatts.add("composite", "true");
                gatts.add("contributor_name", "World Ocean Atlas, Expendable Bathythermograph Archive, " +
                    "TOGA-TAO thermistor array, Soviet SECTIONS tropical program, " +
                    "and Satellite altimetry from Geosat, ERS/1 and TOPEX/Poseidon.");
                gatts.add("contributor_role", "source data");
                gatts.add("Conventions", "COARDS, CF-1.0, Unidata Dataset Discovery v1.0");
                gatts.add("creator_email", "carton@umd.edu");
                gatts.add("creator_name", "SODA");
                gatts.add("creator_url", "http://www.atmos.umd.edu/~ocean/");
                gatts.add("date_created", cDate);
                gatts.add("date_issued", cDate);
                gatts.add("Easternmost_Easting", maxLon);
                gatts.add("geospatial_lat_max", maxLat);
                gatts.add("geospatial_lat_min", minLat);
                if (!Double.isNaN(latSpacing)) gatts.add("geospatial_lat_resolution", latSpacing);
                gatts.add("geospatial_lat_units", "degrees_north");
                gatts.add("geospatial_lon_max", maxLon);
                gatts.add("geospatial_lon_min", minLon);
                if (!Double.isNaN(lonSpacing)) gatts.add("geospatial_lon_resolution", lonSpacing);
                gatts.add("geospatial_lon_units", "degrees_east");
                gatts.add("geospatial_vertical_max", maxDepth);
                gatts.add("geospatial_vertical_min", minDepth);
                gatts.add("geospatial_vertical_positive", "down");
                gatts.add("geospatial_vertical_units", "m");
                gatts.add("history", "http://dsrs.atmos.umd.edu/\n" +
                    cDate + " NOAA SWFSC ERD added metadata and time dimension");
                gatts.add("infoUrl", "http://www.atmos.umd.edu/~ocean/");
                gatts.add("institution", "TAMU/UMD"); //from title
                gatts.add("keywords", "EARTH SCIENCE > Oceans > Ocean Temperature > Water Temperature");
                gatts.add("keywords_vocabulary", "GCMD Science Keywords");
                gatts.add("license", "The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Creator, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.");
                gatts.add("naming_authority", "SODA");
                gatts.add("Northernmost_Northing", maxLat);
                //gatts.add("origin", "TAMU/UMD"); cwhdf attribute, see institution instead
                gatts.add("processing_level", "4 (model)");
                gatts.add("project", "SODA (http://www.atmos.umd.edu/~ocean/)");
                gatts.add("projection", "geographic");
                gatts.add("projection_type", "mapped");
                gatts.add("references", //from http://www.met.rdg.ac.uk/~swr02ldc/SODA.html
                    "Carton, J. A., Chepurin, G., Cao, X. H. and Giese, B. (2000). " +
                    "A Simple Ocean Data Assimilation analysis of the global upper ocean 1950-95. " +
                    "Part I: Methodology. Journal of Physical Oceanography, 30, 2, pp294-309. " +
                    "Carton, J. A., Chepurin, G. and Cao, X. H. (2000). A Simple Ocean Data " +
                    "Assimilation analysis of the global upper ocean 1950-95. Part II: Results. " +
                    "Journal of Physical Oceanography, 30, 2, pp311-326. " +
                    "See also http://www.atmos.umd.edu/~ocean/reanalysis.pdf .");
                //gatts.add("satellite", "POES");   cwhdf attribute, not appropriate here
                //gatts.add("sensor", "AVHRR GAC"); cwhdf attribute, not appropriate here
                gatts.add("source", "model; SODA " + sodaVersion);
                gatts.add("Southernmost_Northing", minLat);
                gatts.add("standard_name_vocabulary", FileNameUtility.getStandardNameVocabulary());
                gatts.add("summary", 
                    "Simple Ocean Data Assimilation (SODA) version " + sodaVersion + " - A reanalysis of ocean climate. " +
                    //from http://www.met.rdg.ac.uk/~swr02ldc/SODA.html
                    "SODA uses the GFDL modular ocean model version 2.2. The model is forced by observed " +
                    "surface wind stresses from the COADS data set (from 1958 to 1992) and from NCEP (after 1992). " +
                    "Note that the wind stresses were detrended before use due to inconsistencies with " +
                    "observed sea level pressure trends. The model is also constrained by constant assimilation " +
                    "of observed temperatures, salinities, and altimetry using an optimal data assimilation " +
                    "technique. The observed data comes from: " +
                    "1) The World Ocean Atlas 1994 which contains ocean temperatures and salinities from " +
                    "mechanical bathythermographs, expendable bathythermographs and conductivity-temperature-depth probes. " +
                    "2) The expendable bathythermograph archive " +
                    "3) The TOGA-TAO thermistor array " +
                    "4) The Soviet SECTIONS tropical program " +
                    "5) Satellite altimetry from Geosat, ERS/1 and TOPEX/Poseidon. \n" +
                    //from http://www.atmos.umd.edu/~ocean/history.html
                    "We are now exploring an eddy-permitting reanalysis based on the Parallel Ocean Program " +
                    "POP-1.4 model with 40 levels in the vertical and a 0.4x0.25 degree displaced pole grid " +
                    "(25 km resolution in the western North Atlantic).  The first version of this we will release " +
                    "is SODA1.2, a reanalysis driven by ERA-40 winds covering the period 1958-2001 (extended " +
                    "to the current year using available altimetry). ");
                //has title
                gatts.add("Westernmost_Easting", minLon);
                String gattNames[] = gatts.getNames();
                for (int i = 0; i < gattNames.length; i++) 
                    newFile.addGlobalAttribute(
                        NcHelper.getAttribute(gattNames[i], gatts.get(gattNames[i])));
            
                //leave define mode
                newFile.create();

                //write data for each variable
                for (int v = 0; v < vars.size(); v++) {
                    Variable var = vars.get(v);
                    String name = var.getName();

                    //if lon, lat, depth
                    if (name.equals("lon") || name.equals("lat") || name.equals("depth")) {
                        //just read it and write it unchanged
                        newFile.write(name, var.read());

                    //if time
                    } else if (name.equals("time")) {
                        //just read it and write it unchanged
                        newFile.write(name, NcHelper.get1DArray(new int[]{months}));

                    //if other variables
                    } else {
                        //read it
                        Array array = var.read();
                        //add time dimension
                        int oldShape[] = array.getShape();
                        int newShape[] = new int[oldShape.length + 1];
                        newShape[0] = 1;
                        System.arraycopy(oldShape, 0, newShape, 1, oldShape.length);
                        array = array.reshape(newShape);
                        //write it
                        newFile.write(name, array);
                    }
                }

                oldFile.close(); oldFile = null;
                newFile.close(); newFile = null;

                if (gzipped) 
                    File2.delete(tempDir + cdfName);

                String2.log("newFile=" + NcHelper.dumpString(newDir + newName, false));

            } //end file loop

            String2.log("Projects.soda finished converting " + oldDir + " successfully.");

        } catch (Exception e) {
            try {oldFile.close();} catch (Exception e2) {}
            try {newFile.close();} catch (Exception e2) {}
            String2.log(MustBe.throwableToString(e));
        }
    }

    /**
     * This is a test of reading a coastwatch .hdf file (as they distribute)
     * with the new netcdf-java 4.0 library.
     * With the hope of making a thredds iosp for these files (see 
     * http://www.unidata.ucar.edu/software/netcdf-java/tutorial/IOSPoverview.html ).
     *
     * @throws Exception if trouble
     */
    public static void testHdf4() throws Exception {
        /*  
        //one time - change ucar to ucar4
        String[] list = RegexFilenameFilter.recursiveFullNameList(
            "c:/programs/tomcat/webapps/cwexperimental/WEB-INF/classes/ucar4", ".*\\.java", false);
        for (int i = 0; i < list.length; i++) {
            String content[] = String2.readFromFile(list[i]);
            if (content[0].length() == 0) {
                String2.log("processing " + list[i]);
                content[1] = String2.replaceAll(content[1], "ucar.", "ucar4.");
                String2.writeToFile(list[i], content[1]);
            } else {
                String2.log(content[0]);
            } 
        }
        // */


        /* 
        //ensure compiled:
        ucar4.nc2.NetcdfFile nc4; //changed to encourage compilation of iosp classes
        ucar4.nc2.iosp.netcdf3.SPFactory spf;

        String fileName = "c:/temp/2008_062_0118_n15_ax.hdf";
        //String fileName = "c:/temp/MODSCW_P2008063_C3_1750_1755_1930_CB05_closest_chlora.hdf";

        //trying to read with my code fails 
        //SdsReader.verbose = true;
        //SdsReader sr = new SdsReader(fileName);

        //the netcdf-java way
        ucar4.nc2.NetcdfFile nc = ucar4.nc2.NetcdfFile.open(fileName);
        try {
            String2.log(nc.toString());
            ucar4.nc2.Variable v = nc.findVariable("avhrr_ch1");
            ucar4.ma2.Array a = v.read("0:100:10,0:100:10");   //start:end:stride,start:end:stride
            String2.log(a.toString());
        } finally {
            nc.close();
        }
        // */
    }

    /** A test of Janino.
     *
     * Compile the expression once; relatively slow.
     * <li><a href="http://www.janino.net/">Janino</a> is a Java compiler
     *    which is useful for compiling and then evaluating expressions at runtime
     *    (Copyright (c) 2001-2007, Arno Unkrig, All rights reserved; license: 
     *    <a href="http://www.janino.net/javadoc/org/codehaus/janino/doc-files/new_bsd_license.txt">BSD</a>).
     */
    public static void testJanino() throws Exception {
        if (true) {
            //Janino
            ExpressionEvaluator ee = new ExpressionEvaluator(
                "c > d ? c : d",                     // expression
                int.class,                           // expressionType
                new String[] { "c", "d" },           // parameterNames
                new Class[] { int.class, int.class } // parameterTypes
            );
            Integer res = (Integer) ee.evaluate(
                new Object[] {          // parameterValues
                    new Integer(10),
                    new Integer(11),
                }
            );
            System.out.println("res = " + res);
        }
        if (true) {
            ExpressionEvaluator ee = new ExpressionEvaluator(
                "import com.cohort.util.Calendar2; Calendar2.isoStringToEpochSeconds(a[0] + \"T\" + a[1])",    // expression
                double.class,                 // expressionType
                new String[] {"a" },          // array of parameterNames
                new Class[] {String[].class } // array of parameterTypes, e.g., int.class, or String[].class
            );

            // Evaluate it with varying parameter values; very fast.
            String2.log("result=" + 
                (Double)ee.evaluate(new Object[] {   // parameterValues
                    new String[]{"1970-01-02", "12:00:00"}})
                );
        }
    }

    /**
     * A test of getting Grids from .nc files.
     *
     */
    public static void testGetNcGrids() throws Exception {
        if (true) { 
            //getGrids test
            //NetcdfDataset ncd = NetcdfDataset.openDataset("c:/temp/cwsamples/MODSCW_P2008045_P2008105_D61_GM05_closest_chlora.nc");
            NetcdfDataset ncd = NetcdfDataset.openDataset("c:/temp/cwsamples/MODSCW_P2008073_2010_D61_P2007351_P2008046_GM03_closest_R667ANOMALY.nc");
            try {
                List list = ncd.getCoordinateSystems();
                System.out.println("nCoordSystems=" + list.size());
                System.out.println("coordinateSystem0=" + list.get(0));
                ucar.nc2.dt.grid.GridDataset gd = new ucar.nc2.dt.grid.GridDataset(ncd);
                try {
                    List gdList = gd.getGrids();
                    System.out.println("nGrids=" + gdList.size());
                    System.out.println("grid=" + gdList.get(0));
                } finally {
                    gd.close();
                }
            } finally {
                ncd.close();
            }
        }
        //Grid.testHDF(new FileNameUtility("gov.noaa.pfel.coastwatch.CWBrowser"), false);
        //"C:/programs/tomcat/webapps/cwexperimental/WEB-INF/classes/gov/noaa/pfel/coastwatch/griddata/OQNux10S1day_20050712_x-135_X-105_y22_Y50Test.hdf";
        //File2.copy("c:/TestDapperBackendService.nc", "c:/zztop/testFileCopy.nc");
        //String2.log(" -5%4=" + (-5%4) + " 5%-4=" + (5%-4) + " -5%-4=" + (-5%-4));
        if (false) {
            //ucar.nc2.util.DebugFlags.set("DODS/serverCall", true);
            //NetcdfFile nc = NetcdfDataset.openFile("http://dapper.pmel.noaa.gov/dapper/epic/tao_time_series.cdp", null);
            //NetcdfFile nc = NetcdfDataset.openFile("http://coastwatch.pfeg.noaa.gov/erddap2/tabledap/cwwcNDBCMet", null);
            NetcdfFile nc = NetcdfDataset.openFile("http://127.0.0.1:8080/cwexperimental/tabledap/cwwcNDBCMet", null);
            //NetcdfFile nc = NetcdfDataset.openFile("http://192.168.31.27:8080/thredds/dodsC/satellite/cwtest/aqua/modis/chlora/D1", null);
            String2.log(nc.toString());
            nc.close();
        }
    }

    /** Test wobbly lon and lat values in AGssta. */
    public static void testWobblyLonLat() throws Exception {
        //test of "wobbly" lat and lon values in AGssta 14day
        Grid grid = new Grid();
        //grid.readGrd("c:/temp/roy/AG2006009_2006022_ssta_westus.grd",
        grid.readNetCDF("c:/temp/roy/AG2006009_2006022_ssta.nc", null);
        String2.log("lon=" + String2.toCSVString(grid.lon) + "\nlat=" + String2.toCSVString(grid.lat));
        int nLon = grid.lon.length;
        double maxLonDif = 0;
        double dLon[] = new double[nLon];
        for (int i = 0; i < nLon; i++) {
            dLon[i] = (float)grid.lon[i];
            if (i > 0)
                maxLonDif = Math.max(maxLonDif, Math.abs(dLon[i]-dLon[i-1] -.1));
        }
        int nLat = grid.lat.length;
        double maxLatDif = 0;
        for (int i = 1; i < nLat; i++) 
            maxLatDif = Math.max(maxLatDif, Math.abs(grid.lat[i]-grid.lat[i-1]));
        String2.log("maxLonDif=" + maxLonDif + " maxLatDif = " + maxLatDif);
            //+ "\ndLon=" + String2.toCSVString(dLon));
    }

    /** A tunnel test for ERDDAP. */
    public static void erddapTunnelTest() throws Exception {
        String url = "http://coastwatch.pfeg.noaa.gov/erddap/griddap/erdCMsfc";
        String varName = "eastCurrent";
        SSR.genericTunnelTest(1000, url + ".csv", varName); 
        SSR.genericTunnelTest(1000, url + ".nc",  varName); 
        SSR.genericTunnelTest(1000, url + ".csv", varName); 
        SSR.genericTunnelTest(1000, url + ".nc",  varName); 
    }

    /** A test of Opendap availability. 
     *
     * @param maxSeconds specifies the delay between tests: a random number between 0 and maxSeconds.
     * @param thredds if true, the requests include lon and lat.
     *     If false, they include longitude and latitude.
     */
    public static void testOpendapAvailability(String opendapBaseUrl, String varName,
        int nIterations, int maxSeconds, boolean thredds) throws Exception {

        boolean tVerbose = true;
        String2.log("testOpendapAvailablity nIterations=" + nIterations + ", maxSeconds=" + maxSeconds +
            "\nopendapBaseUrl=" + opendapBaseUrl);
        int nFailures = 0;
        long cumTime = 0;
        long clockTime = System.currentTimeMillis();
        int distribution[] = new int[String2.DistributionSize];

        for (int i = 0; i < nIterations; i++) {
            //get dataset info
            String response1="", response2="", response3="", response4="", response5="", response6="";
            long time = System.currentTimeMillis(), 
                tTime1=-1, tTime2=-1, tTime3=-1, tTime4=-1, tTime5=-1, tTime6=-1;
            try {
                tTime1 = System.currentTimeMillis();
                response1 = SSR.getUrlResponseString(opendapBaseUrl + ".das");
                tTime1 = System.currentTimeMillis() - tTime1;

                tTime2 = System.currentTimeMillis();
                response2 = SSR.getUrlResponseString(opendapBaseUrl + ".dds");
                tTime2 = System.currentTimeMillis() - tTime2;

                tTime3 = System.currentTimeMillis();
                response3 = SSR.getUrlResponseString(opendapBaseUrl + ".asc?time");      
                tTime3 = System.currentTimeMillis() - tTime3;

                tTime4 = System.currentTimeMillis();
                response4 = SSR.getUrlResponseString(opendapBaseUrl + ".asc?lat" + (thredds? "" : "itude"));      
                tTime4 = System.currentTimeMillis() - tTime4;

                tTime5 = System.currentTimeMillis();
                response5 = SSR.getUrlResponseString(opendapBaseUrl + ".asc?lon" + (thredds? "" : "gitude"));      
                tTime5 = System.currentTimeMillis() - tTime5;

                tTime6 = System.currentTimeMillis();
                response6 = SSR.getUrlResponseString(opendapBaseUrl + ".asc?" + varName + 
                    "[" + Math2.random(100) + "][0]" +
                    "[" + Math2.random(20) + "]" +
                    "[" + Math2.random(20) + "]");      
                tTime6 = System.currentTimeMillis() - tTime6;

                time = System.currentTimeMillis() - time;
                cumTime += time;
                String2.distribute(time, distribution);
            } catch (Exception e) {
                time = -1;
                nFailures++;
                String2.log(MustBe.throwableToString(e));
            }

            //outside of timings 
            if (i == 0) String2.log("\n***r1\n" + response1 + "\n***r2\n" + response2 + 
                "\n***r3\n" + response3 + "\n***r4\n" + response4 + "\n***r5\n" + response5 + 
                "\n***r6\n" + response6);

            //wait 0 - maxMinutes
            int sec = Math2.random(maxSeconds);
            if (i== 0) String2.log("\niter   response ms   .das   .dds   time    lat    lon  datum   sleep sec");
            String2.log(
                String2.right("" + i, 4) + 
                String2.right("" + time, 14) + 
                String2.right("" + tTime1, 7) + 
                String2.right("" + tTime2, 7) + 
                String2.right("" + tTime3, 7) + 
                String2.right("" + tTime4, 7) + 
                String2.right("" + tTime5, 7) + 
                String2.right("" + tTime6, 7) + 
                String2.right("" + sec, 12));
            Math2.sleep(sec * 1000);
        }
        String2.log("\ntestOpendapAvailablity finished nIterations=" + nIterations + ", maxSeconds=" + maxSeconds +
            "\nopendapBaseUrl=" + opendapBaseUrl +
            "\nclockTime=" + Calendar2.elapsedTimeString(System.currentTimeMillis() - clockTime) +
            "\ncumulativeTime=" + (cumTime/1000) + "sec, nFailures=" + nFailures + 
            "\nresponseMsDistribution:\n" +
            String2.getDistributionStatistics(distribution));
    }

        

}

