/* 
 * Boundaries Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch.sgt;

import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.StringArray;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pmel.sgt.*;
import gov.noaa.pmel.sgt.dm.*;

import java.io.File;
import java.io.*;
import java.util.ArrayList;


/**
 * This class has methods related to National and State Boundaries.
 */
public class Boundaries  {

    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR;

    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;

    /**
     * Set this to true (by calling reallyVerbose=true in your program, not but changing the code here)
     * if you want lots and lots of diagnostic messages sent to String2.log.
     */
    public static boolean reallyVerbose = false;

    public final static String REF_DIRECTORY = SSR.getContextDirectory() + "WEB-INF/ref/";

    /** 
     * The nationalBoundary and stateBoundary files must be in the refDirectory. 
     *    "gshhs_?.b" (?=f|h|i|l|c) files. 
     *    The files are from the GSHHS project
     *    (http://www.ngdc.noaa.gov/mgg/shorelines/gshhs.html).
     *    landMaskDir should have slash at end.
     */
    public String directory = SSR.getContextDirectory() + "WEB-INF/ref/";

    /**
     * Since boundary SGTLines are time-consuming to contruct,
     *   this caches the last cacheSize used SgtLines.
     * <br>Memory for each cached GP (CWBrowser typical use) is 2KB to 
     *    500KB (whole world, crude), (20KB is typical).
     * <br>Suggested cacheSize is nPredefinedRegions + 5 (remember that 0-360 regions
     *   are different from +/-180 regions).
     */
    public final static int CACHE_SIZE = 50;
    private StringArray cachedNames = new StringArray();
    private ArrayList cachedSgtLines = new ArrayList();
    private int nSuccesses = 0;
    private int nTossed = 0;
    private int totalKB = 0;

    /** This identifies the subclass as National or State. */
    private String id = "";

    // national and State Boundaries are nulls until needed
    // public access via getNationalBoundary...
    // I got state and national boundaries from GMT:
    // In general: pscoast -Df -N1 -M -Rwest/east/south/north >! file
    // I used:     pscoast -Df -N1 -M -R-180/180/-90/90 >! nationalBoundariesf.asc
    // where "f"ull can be changed to "h"igh, "i"ntermediate, "l"ow, "c"rude.
    // where "N1" (national) can be changed to "N2" (state)
    //   e.g., String2.getContextDirectory() + "WEB-INF/ref/nationalBoundariesf.asc"
    // The resulting files use the GMT_FORMAT
    // The source files must be in fullRefDirectory.
    //??? Are there newer national boundary files (not showing USSR)???
    private String[] fileNames;

    public final static String NATIONAL_FILE_NAMES[] = {
        "nationalBoundariesf.asc", "nationalBoundariesh.asc", "nationalBoundariesi.asc", 
        "nationalBoundariesl.asc", "nationalBoundariesc.asc"};
    //??? Why are there state boundaries for all of Americas, but not Australia? Mistake on my part???
    public final static String STATE_FILE_NAMES[] = {
        "stateBoundariesf.asc", "stateBoundariesh.asc", "stateBoundariesi.asc", 
        "stateBoundariesl.asc", "stateBoundariesc.asc"};

    public final static int MATLAB_FORMAT = 0;
    public final static int GMT_FORMAT = 1;
    private int fileFormat = GMT_FORMAT;  

    /**
     * The constructor.
     *
     * @param id e.g., "National" or "State"
     * @param directory the directory with the fileNames
     * @param fileNames the 5 file names
     * @param fileFormat   MATLAB_FORMAT or GMT_FORMAT
     */
    public Boundaries(String id, String directory, String fileNames[], int fileFormat) {
        this.id = id;
        this.directory = directory;
        this.fileNames = fileNames;
        this.fileFormat = fileFormat;
    }

    /**
     * This gets the SGTLine with the relevant boundary information.
     *
     * @param resolution 0='f'ull, 1='h'igh, 2='i'ntermediate, 3='l'ow, 4='c'rude.
     * @param west 0..360 or +/-180, 
     * @param east 0..360 or +/-180
     * @param south +/-90
     * @param north +/-90
     * @return a SgtLine with the requested boundaries.
     * @throws exception if trouble
     */
    public synchronized SGTLine getSgtLine( 
        int resolution, double west, double east, 
        double south, double north) throws Exception {

        //"synchronized" so changes to cache and creation of new SGTLines occur atomically.

        String cachedName = resolution + 
            "W" + String2.genEFormat10(west) +
            "E" + String2.genEFormat10(east) +
            "S" + String2.genEFormat10(south) +
            "N" + String2.genEFormat10(north);
        if (reallyVerbose) String2.log("  Boundaries.getSgtLine " + id + " request=" + cachedName);
        long time = System.currentTimeMillis();

        //*** is SGTLine in cache?
        int po = cachedNames.indexOf(cachedName);
        if (po >= 0) {
            //yes, it is in cache
                
            //remove from cache
            cachedNames.remove(po);
            SGTLine sgtLine = (SGTLine)cachedSgtLines.remove(po);

            //reinsert at top of cache
            cachedNames.add(0, cachedName);
            cachedSgtLines.add(0, sgtLine);

            //return gp
            nSuccesses++;
            if (reallyVerbose) String2.log("    Boundaries.getSgtLine done (already in cache). nSuccesses=" + 
                nSuccesses + " nTossed=" + nTossed);
            return sgtLine;
        }

        //*** else need to make SgtLine
        SGTLine sgtLine = readSgtLine(
            directory + fileNames[resolution], fileFormat,
            west, east, south, north);

        //cache full?
        boolean tSuccess = false;
        if (cachedNames.size() == CACHE_SIZE) {
            //yes, throw away oldest gp
            cachedNames.remove(CACHE_SIZE - 1);
            cachedSgtLines.remove(CACHE_SIZE - 1);
            nTossed++;
        } else {
            tSuccess = true;  //if cache wasn't full, treat as success
            nSuccesses++;
        }
        int thisKB = (sgtLine.getXArray().length * 16) / 1024;  //16 because x,y, both doubles
        totalKB += thisKB;

        //reinsert new path at top of cache
        cachedNames.add(0, cachedName);
        cachedSgtLines.add(0, sgtLine);

        //return sgtLine
        if (reallyVerbose) String2.log(
            "    Boundaries.getSgtLine done (created SGTLine) success=" + tSuccess + " size=" + thisKB + "KB" +
            "\n      nSuccesses=" + nSuccesses + " nTossed=" + nTossed + 
            " totalKB=" + totalKB + ", TOTAL TIME=" + (System.currentTimeMillis() - time));
        return sgtLine;

    }

    /**
     * This creates an SGTLine from the line (e.g., boundary) 
     * info in an SGTLine object.
     *
     * <pre>
     * I got Matlab data from http://rimmer.ngdc.noaa.gov/mgg/coast/getcoast.html
     *   NOAA Digital Vector Shoreline is highest resolution but is US only
     *   so get World Vector Shoreline:
     *     x range: -135   -105
     *     y range: 22 50
     *     database: World Vector Shoreline
     *     zip file
     *     Preview GMT plot
     *     saved in WEB-INF/ref/shorelineW-135E-105S22N50.zip and unzipped to 
     *              WEB-INF/ref/shorelineW-135E-105S22N50.dat 
     *     WVS Info: http://rimmer.ngdc.noaa.gov/mgg/coast/wvs.html
     * 
     * I got GMT data from our GMT installation:
     *   In general: pscoast -Df -N2 -M -Rwest/east/south/north > file
     *   I used (for national, state, and coast boundaries):
     *     pscoast -Df -N1 -N2 -W -M -R-135/-105/22/50 > boundariesW-135E-105S22N50.asc
     *   See GMT docs (http://gmt.soest.hawaii.edu/) for pscoast
     * </pre>
     *
     * The inspiration for this was in gov.noaa.pmel.sgt.demo.TAOMap.java (see old code).
     * 
     * @param fullFileName the full name of the Matlab format source file
     * @param format MATLAB_FORMAT or GMT_FORMAT.
     *   MATLAB_FORMAT: an ASCII file where each line has either 
     *     "<lon>[tab]<lat>" or "nan[space]nan".
     *   GMT_FORMAT: an ASCII file where each line has either 
     *     "<lon>[tab]<lat>" or starts with "#" or ">".
     *   <p>The input file can be lonPm180 or 0..360.
     * @param minX 0..360 or +/-180
     * @param maxX 0..360 or +/-180 
     * @param minY
     * @param maxY
     * @return an SGTLine object
     * @throws Exception if trouble
     */
    public static SGTLine readSgtLine(String fullFileName, int format, 
            double minX, double maxX, 
            double minY, double maxY) throws Exception {

        //if (reallyVerbose) String2.log("    readSGTLine");

        boolean lonPM180 = minX < 0;
        BufferedReader bufferedReader = new BufferedReader(new FileReader(fullFileName));
        DoubleArray lat = new DoubleArray(); 
        DoubleArray lon = new DoubleArray();
        DoubleArray tempLat = new DoubleArray(); 
        DoubleArray tempLon = new DoubleArray();
        String startGapLine1 = format == MATLAB_FORMAT? "nan nan" : ">";
        String startGapLine2 = format == MATLAB_FORMAT? "nan nan" : "#";
        String s = bufferedReader.readLine();
        int nObjects = 0;
        while (s != null) { //null = end-of-file
            if (s.startsWith(startGapLine1) ||
                s.startsWith(startGapLine2)) {
                //try to add this subpath
                int tn = GSHHS.reduce(tempLat.size(), tempLon.array, tempLat.array, minX, maxX, minY, maxY);
                tempLat.removeRange(tn, tempLat.size());
                tempLon.removeRange(tn, tempLon.size());
                if (tn > 0) {
                    lon.append(tempLon);
                    lat.append(tempLat);
                    lon.add(Double.NaN);
                    lat.add(Double.NaN);
                    nObjects++;
                }
                tempLon.clear();
                tempLat.clear();
            } else {
                //each line: x\ty 
                String[] items = String2.split(s, '\t');
                if (items.length == 2) {
                    double tLon = String2.parseDouble(items[0]);
                    double tLat = String2.parseDouble(items[1]);
                    if (lonPM180) {
                        if (tLon >= 180) tLon -= 360;
                    } else {
                        if (tLon < 0) tLon += 360;
                    }
                    //cut lines going from one edge of world to the other
                    if (tempLon.size() > 0 && Math.abs(tLon - tempLon.get(tempLon.size() - 1)) > 50.0) {
                        //try to add this subpath
                        int tn = GSHHS.reduce(tempLat.size(), tempLon.array, tempLat.array, minX, maxX, minY, maxY);
                        tempLat.removeRange(tn, tempLat.size());
                        tempLon.removeRange(tn, tempLon.size());
                        if (tn > 0) {
                            lon.append(tempLon);
                            lat.append(tempLat);
                            lon.add(Double.NaN);
                            lat.add(Double.NaN);
                            nObjects++;
                        }
                        tempLon.clear();
                        tempLat.clear();
                    }
                    tempLon.add(tLon);
                    tempLat.add(tLat);
                } else String2.log("ERROR at readSGTLine items.length!=2 (" + items.length + ") s=" + s);
            }
            s = bufferedReader.readLine();
        }
        bufferedReader.close();
        if (reallyVerbose) String2.log("    Boundaries.readSgtLine nObjects=" + nObjects);        
        int n = lon.size();
        double lonVal[] = new double[n];
        double latVal[] = new double[n];
        System.arraycopy(lon.array, 0, lonVal, 0, n);
        System.arraycopy(lat.array, 0, latVal, 0, n);
         
        SimpleLine line = new SimpleLine(lonVal, latVal, "Boundary");
        line.setXMetaData(new SGTMetaData("Longitude", "degrees_E", false, true));
        line.setYMetaData(new SGTMetaData("Latitude",  "degrees_N", false, false));

        return line;
    }


    /** This returns a stats string for Boundaries. */
    public String statsString() {
        return 
            id + " Boundaries: nCached=" + cachedNames.size() + " of " + CACHE_SIZE +
            ", nSuccesses=" + nSuccesses + ", nTossed=" + nTossed + 
            ", totalKB=" + totalKB;
    }

    /** This is a convenience method to construct a national boundaries object. */
    public static Boundaries getNationalBoundaries() {
        return new Boundaries("National", 
            REF_DIRECTORY, Boundaries.NATIONAL_FILE_NAMES, Boundaries.GMT_FORMAT);
    }

    /** This is a convenience method to construct a national boundaries object. */
    public static Boundaries getStateBoundaries() {
        return new Boundaries("State", 
            REF_DIRECTORY, Boundaries.STATE_FILE_NAMES, Boundaries.GMT_FORMAT);
    }

    /**
     * This runs a unit test.
     */
    public static void test() throws Exception {
        verbose = true;

        String2.log("\n*** Boundaries.test");

        Boundaries nationalBoundaries = Boundaries.getNationalBoundaries();
        Boundaries stateBoundaries = Boundaries.getStateBoundaries();
        SGTLine sgtLine;
        
        //*** national
        //force creation of new file
        sgtLine = nationalBoundaries.getSgtLine(2, -135, -105, 22, 50);

        //read cached version
        long time = System.currentTimeMillis();
        sgtLine = nationalBoundaries.getSgtLine(2, -135, -105, 22, 50);
        time = System.currentTimeMillis() - time;

        //is it the same  (is SGTLine.equals a deep test? probably not)

        //test speed
        Test.ensureTrue(time < 20, "time=" + time); 


        //*** state
        //force creation of new file
        sgtLine = stateBoundaries.getSgtLine(2, -135, -105, 22, 50);

        //read cached version
        time = System.currentTimeMillis();
        sgtLine = stateBoundaries.getSgtLine(2, -135, -105, 22, 50);
        time = System.currentTimeMillis() - time;

        //is it the same  (is SGTLine.equals a deep test? probably not)

        //test speed
        Test.ensureTrue(time < 20, "time=" + time); 

        String2.log(nationalBoundaries.statsString() + "\n" +
            stateBoundaries.statsString());

    }

}
