/* 
 * GSHHS Copyright 2005, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch.sgt;

import com.cohort.array.IntArray;
import com.cohort.array.StringArray;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.util.SSR;

import java.awt.geom.GeneralPath;
import java.io.File;
import java.io.*;
import java.util.ArrayList;
import javax.imageio.ImageIO;


/**
 * This class has methods related to GSHHS -
 * A Global Self-consistent, Hierarchical, High-resolution Shoreline Database
 * created by the GMT authors Wessel and Smith 
 * (see http://www.ngdc.noaa.gov/mgg/shorelines/gshhs.html and 
 * http://www.ngdc.noaa.gov/mgg/shorelines/data/gshhs/version1.3/readmegshhs.txt).
 */
public class GSHHS  {

    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR;

    /** The characters of the RESOLUTIONS string represent the 5 resolutions
     * in order of decreasing accuracy: "fhilc": 'f'ull, 'h'igh, 'i'ntermediate, 'l'ow, 'c'rude.
     */
    public final static String RESOLUTIONS = "fhilc"; //'f'ull, 'h'igh, 'i'ntermediate, 'l'ow, 'c'rude.


    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;

    /**
     * Set this to true (by calling reallyVerbose=true in your program, not but changing the code here)
     * if you want lots and lots of diagnostic messages sent to String2.log.
     */
    public static boolean reallyVerbose = false;

    /** 
     * The GSHHS files must be in the refDirectory. 
     *    "gshhs_?.b" (?=f|h|i|l|c) files. 
     *    The files are from the GSHHS project
     *    (http://www.ngdc.noaa.gov/mgg/shorelines/gshhs.html).
     *    landMaskDir should have slash at end.
     */
    public static String gshhsDirectory = SSR.getContextDirectory() + "WEB-INF/ref/";

    /**
     * Since GeneralPaths are time-consuming to contruct,
     *   getGeneralPath caches the last CACHE_SIZE used GeneralPaths.
     * <br>Memory for each cached GP (CWBrowser typical use) is 2KB to 
     *    500KB (whole world, crude), (20KB is typical).
     * <br>Note that I originally tried to cache the float lat lons to cache files.
     * <br>Reading/writing the files is fast, but creating the GeneralPath is slow, 
     *   so I switched to caching the GeneralPaths in memory.
     * <br>Suggested CACHE_SIZE is nPredefinedRegions + 5 (remember that 0-360 regions
     *   are different from +/-180 regions).
     */
    public final static int CACHE_SIZE = 50;
    private static StringArray cachedNames = new StringArray();
    private static ArrayList cachedGeneralPaths = new ArrayList();
    private static int nSuccesses = 0;
    private static int nTossed = 0;
    private static int totalKB = 0;


    /**
     * This gets the GeneralPath with the relevant shoreline information.
     *
     * @param resolution 'f'ull, 'h'igh, 'i'ntermediate, 'l'ow, 'c'rude.
     * @param desiredLevel determines the highest level of data included:
     *      1=land, 2=lake, 3=islandInLake, 4=pondInIslandInLake
     * @param westDeg 0..360 or +/-180, 
     * @param eastDeg 0..360 or +/-180
     * @param southDeg +/-90
     * @param northDeg +/-90
     * @param addAntarcticCorners If true, corners are added to the 
     *    antarctic polygon (for y=-90), so the polygon plots correctly.
     *    This is true for most projections, but not some polar projections.
     * @return a GeneralPath with the requested polygons, with lat and lon
     *    stored as ints (in millionths of a degree).
     * @throws exception if trouble
     */
    public synchronized static GeneralPath getGeneralPath(  
        char resolution, int desiredLevel, double westDeg, double eastDeg, 
        double southDeg, double northDeg, boolean addAntarcticCorners) throws Exception {

        //"synchronized" so changes to cache and creation of new GeneralPaths occur atomically.

        String cachedName = "GSHHS" + resolution + desiredLevel +
            "W" + String2.genEFormat10(westDeg) +
            "E" + String2.genEFormat10(eastDeg) +
            "S" + String2.genEFormat10(southDeg) +
            "N" + String2.genEFormat10(northDeg) +
            "A" + (addAntarcticCorners? "1" : "0");
        if (reallyVerbose) String2.log("  GSHHS.getGeneralPath request=" + cachedName);
        long time = System.currentTimeMillis();

        //*** is GeneralPath in cache?
        int po = cachedNames.indexOf(cachedName);
        if (po >= 0) {
            //yes, it is in cache
                
            //remove from cache
            cachedNames.remove(po);
            GeneralPath gp = (GeneralPath)cachedGeneralPaths.remove(po);

            //reinsert at top of cache
            cachedNames.add(0, cachedName);
            cachedGeneralPaths.add(0, gp);

            //return gp
            nSuccesses++;
            if (verbose) String2.log("    GSHHS.getGeneralPath done (already in cache). nSuccesses=" + 
                nSuccesses + " nTossed=" + nTossed);
            return gp;
        }

        //*** else need to make GeneralPath
        GeneralPath path = new GeneralPath(GeneralPath.WIND_EVEN_ODD); //winding rule is important
        IntArray lon = new IntArray();
        IntArray lat = new IntArray();
        getPathInfo(gshhsDirectory, resolution, desiredLevel, westDeg, eastDeg, 
            southDeg, northDeg, addAntarcticCorners, lon, lat);

        //fill path object 
        int n = lon.size();
        int lonAr[] = lon.array;
        int latAr[] = lat.array;
        int nObjects = 0;
        for (int i = 0; i < n; i++) {
            if (lonAr[i] == Integer.MAX_VALUE) {
                i++; //move to next point
                path.moveTo(lonAr[i], latAr[i]); 
                nObjects++;
            } else {
                path.lineTo(lonAr[i], latAr[i]);
            }
        }

        //cache full?
        boolean tSuccess = false;
        if (cachedNames.size() == CACHE_SIZE) {
            //yes, throw away oldest gp
            cachedNames.remove(CACHE_SIZE - 1);
            cachedGeneralPaths.remove(CACHE_SIZE - 1);
            nTossed++;
        } else {
            tSuccess = true;  //if cache wasn't full, treat as success
            nSuccesses++;
        }
        totalKB += (n * 8) / 1024;

        //reinsert new path at top of cache
        cachedNames.add(0, cachedName);
        cachedGeneralPaths.add(0, path);

        //return gp
        if (reallyVerbose) String2.log("    GSHHS.getGeneralPath done (created GP), success=" + tSuccess + 
            " nSuccesses=" + nSuccesses + " nTossed=" + nTossed + 
            "\n    nObjects=" + nObjects + " size=" + ((n * 8) / 1024) +
            "KB, totalKB=" + totalKB + ", TOTAL TIME=" + (System.currentTimeMillis() - time));
        return path;

    }

    /** This returns a stats string for GSHHS. */
    public static String statsString() {
        return 
            "GSHHS: nCached=" + cachedNames.size() + " of " + CACHE_SIZE +
            ", nSuccesses=" + nSuccesses + ", nTossed=" + nTossed + 
            ", totalKB=" + totalKB;
    }

    /**
     * This actually reads the GSHHS files and populates lon and lat with info for a GeneralPath.
     * This has nothing to do with the cache system.
     *
     * @param gshhsDir the directory with the gshhs_[fhilc].b data files, with a slash at the end.
     * @param resolution 'f'ull, 'h'igh, 'i'ntermediate, 'l'ow, 'c'rude.
     * @param desiredLevel 1=land, 2=lake, 3=islandInLake, 4=pondInIslandInLake
     * @param westDeg 0..360 or +/-180, 
     * @param eastDeg 0..360 or +/-180
     * @param southDeg +/-90
     * @param northDeg +/-90
     * @param addAntarcticCorners If true, corners are added to the 
     *    antarctic polygon (for y=-90), so the polygon plots correctly.
     *    This is true for most projections, but not some polar projections.
     * @param lon the IntArray which will receive the lon values
     *    (Integer.MAX_VALUE) precedes each moveTo value.
     *    The values are in the appropriate range (0 - 360e6, or +/-180e6).
     * @param lat the IntArray which will receive the lat values
     *    (Integer.MAX_VALUE) precedes each moveTo value.
     * @throws exception if trouble
     */
    public static void getPathInfo(String gshhsDir, 
        char resolution, int desiredLevel, double westDeg, double eastDeg, 
        double southDeg, double northDeg, boolean addAntarcticCorners,
        IntArray lon, IntArray lat) throws Exception {

        long time = System.currentTimeMillis();
        lon.clear();
        lat.clear();

        int desiredWest  = (int)(westDeg  * 1000000);
        int desiredEast  = (int)(eastDeg  * 1000000);
        int desiredSouth = (int)(southDeg * 1000000);
        int desiredNorth = (int)(northDeg * 1000000);

        //the files work in lon 0..360, so adjust if desired
        //The adjustment is crude: either keep original values
        //(generally 0..360, but not always) or subtract 360
        //(so generally -360..360).
        boolean lonPM180 = westDeg < 0;
        int intShift = 360 * 1000000;

        //open the file
        //String2.log(File2.hexDump(dir + "gshhs_" + resolution + ".b", 10000));
        DataInputStream dis = new DataInputStream(new BufferedInputStream(
            new FileInputStream(gshhsDir + "gshhs_" + resolution + ".b")));

        //read the records
        //the xArrays and yArrays grow as needed
        int xArray[] = new int[1];
        int yArray[] = new int[1];
        int xArray2[] = new int[1];
        int yArray2[] = new int[1];
        boolean aMsgDisplayed = false;
        boolean gMsgDisplayed = false;
        while (dis.available() > 0) {
            //read the header
            int id        = dis.readInt();   
            int n         = dis.readInt();   
            int level     = dis.readInt(); 
            int west      = dis.readInt();  
            int east      = dis.readInt();   
            int south     = dis.readInt();   
            int north     = dis.readInt();  
            int area      = dis.readInt();   
            int greenwich = dis.readShort(); //1 == it crosses greenwich
            int source    = dis.readShort(); 

            //tests show greenwich objects have a negative west bound
            //(even though <0 lon values are stored +360)
            //if (!gMsgDisplayed && greenwich == 1) {
            //    String2.log("greenwich n=" + n + " west=" + west + " east=" + east + " south=" + south + " north=" + north);
            //    gMsgDisplayed = true;
            //}

            //tests show antarctic object has 
            //bounds (degrees) are west=0 east=360 south=-90 north=-63
            //if (!aMsgDisplayed && south < -60000000) {
            //    String2.log("antartic n=" + n + " west=" + west + " east=" + east + " south=" + south + " north=" + north);
            //    aMsgDisplayed = true;
            //}

            //Do the tests for the 3 possible independent uses of this data.
            //Note that often 2 of the 3 are true.
            //can I use this object with standard coordinates?
            boolean useStandard = 
                level <=  desiredLevel &&
                west  < desiredEast &&
                east  > desiredWest &&
                south < desiredNorth &&
                north > desiredSouth;
            
            //can I use this object with coordinates shifted left (e.g., pm 180)?
            boolean useShiftedLeft = 
                lonPM180 &&
                level <=  desiredLevel &&
                west-intShift < desiredEast &&
                east-intShift > desiredWest &&
                south < desiredNorth &&
                north > desiredSouth;
            
            //can I use this object with coordinates shifted right (for greenwich==1 objects when !pm180)?
            boolean useShiftedRight = 
                level <=  desiredLevel &&
                west+intShift < desiredEast &&
                east+intShift > desiredWest &&
                south < desiredNorth &&
                north > desiredSouth;
            
            //can I use the object?   
            if (useStandard || useShiftedLeft || useShiftedRight) {
                int cShift = 0;
                if (useShiftedLeft) cShift = -intShift;
                else if (useShiftedRight) cShift = intShift;

                //read the data
                if (n > xArray.length) {
                    xArray = new int[n + 4];  //+4 for addAntarticCorners
                    yArray = new int[n + 4];
                }
                for (int i = 0; i < n; i++) {
                    xArray[i] = dis.readInt();
                    yArray[i] = dis.readInt();
                }  

                //for addAntarcticCorners, insert points at corners of map.
                //antarctic object bounds (degrees) are west=0 east=360 south=-90 north=-63
                //search for lon=0
                if (south == -90000000) { //catches antarctic polygon
                    //this shows first x=360, x decreases to 0, then jumps to 360 (last point)
                    //String2.log("antarctic n=" + n + " x[0]=" + xArray[0] + 
                    //    " x[1]=" + xArray[1] + " x[n-2]=" + xArray[n-2] + 
                    //    " x[n-1]=" + xArray[n-1]);
                    if (desiredWest < 0 && desiredEast > 0) {
                        //Desired is e.g. -180 to 180. 
                        //To avoid seam in bad place, manually shift 1/2 of it left.
                        useShiftedLeft = false; 
                        for (int i = 0; i < n; i++) {
                            if (xArray[i] >= 180000000) { //in practice there is no 180000000 point
                                xArray[i] -= 360000000;
                                if (i < n-1 && xArray[i+1] < 180000000) {
                                    //We're crossing x=180, where we want the seam.
                                    //These tests show x=180 isn't first or last point
                                    //  and prev x is almost 0, next x is 360
                                    //String2.log("antarctic x crosses 180 at x[" + 
                                    //    i + "]=" + xArray[i] + " x[i+1]=" + xArray[i+1]);

                                    if (addAntarcticCorners) {
                                        //add the 4 antarctic corners  
                                        //this puts a seam at x=-180 ... x=180
                                        System.arraycopy(xArray, i + 1, xArray, i + 5, n - (i+1));
                                        System.arraycopy(yArray, i + 1, yArray, i + 5, n - (i+1));
                                        xArray[i + 1] = -180000000; yArray[i + 1] = yArray[i];
                                        xArray[i + 2] = -180000000; yArray[i + 2] = -90000000;
                                        xArray[i + 3] =  180000000; yArray[i + 3] = -90000000;
                                        xArray[i + 4] =  180000000; yArray[i + 4] = yArray[i];
                                        i += 4;
                                        n += 4;
                                    }
                                }
                            }
                        }
                    } else {
                        //probably just useStandard (e.g., 0..360)    
                        //  or useLeft (limited range e.g., -130.. -120)
                        if (addAntarcticCorners) {
                            //this puts a seam at x=0 ... x=360
                            for (int i = 0; i < n; i++) {
                                if (xArray[i] == 0) {
                                    //these tests show x=0 isn't first or last point
                                    //  and prev x is almost 0, next x is 360
                                    //String2.log("antarctic x=0 at i=" + i + " n=" + n);
                                    //if (i > 0) String2.log("  x[i-1]=" + xArray[i-1]);
                                    //if (i < n-1) String2.log("  x[i+1]=" + xArray[i+1]);

                                    //add the 2 antarctic corners  
                                    System.arraycopy(xArray, i + 1, xArray, i + 3, n - (i+1));
                                    System.arraycopy(yArray, i + 1, yArray, i + 3, n - (i+1));
                                    xArray[i + 1] =         0; yArray[i + 1] = -90000000;
                                    xArray[i + 2] = 360000000; yArray[i + 2] = -90000000;
                                    n += 2;
                                    break;
                                }
                            }
                        }
                    }
                }

                //if polygon crosses greenwhich, x's < 0 are stored +360 degrees
                //see http://www.ngdc.noaa.gov/mgg/shorelines/data/gshhs/version1.2/gshhs.c
                if (greenwich == 1) {
                    for (int i = 0; i < n; i++) 
                        if (xArray[i] > east) xArray[i] -= intShift;
                }  
                

                //useShiftedLeft   (this is independent of useShiftedRight and useStandard)
                if (useShiftedLeft) {

                    //copy the data into x/yArray2's  
                    //so data is undisturbed for useShiftedRight and useStandard
                    if (n > xArray2.length) {
                        xArray2 = new int[n];
                        yArray2 = new int[n];
                    }
                    System.arraycopy(xArray, 0, xArray2, 0, n);
                    System.arraycopy(yArray, 0, yArray2, 0, n);

                    //reduce and draw
                    int tn = reduce(n, xArray2, yArray2, 
                        desiredWest + intShift, desiredEast + intShift, //shifting desired is faster than shifting all x,y
                        desiredSouth, desiredNorth);
                    if (tn > 0) {
                        lon.add(Integer.MAX_VALUE); //indicates moveTo next point
                        lat.add(Integer.MAX_VALUE);
                        for (int i = 0; i < tn; i++) {
                            lon.add(xArray2[i] - intShift);
                            lat.add(yArray2[i]);
                        }
                    }
                }

                //useShiftedRight   (this is independent of useShiftedLeft and useStandard)
                if (useShiftedRight) {
                    //copy the data into x/yArray2's
                    //so data is undisturbed for useShiftedRight and useStandard
                    if (n > xArray2.length) {
                        xArray2 = new int[n];
                        yArray2 = new int[n];
                    }
                    System.arraycopy(xArray, 0, xArray2, 0, n);
                    System.arraycopy(yArray, 0, yArray2, 0, n);

                    //reduce and draw
                    int tn = reduce(n, xArray2, yArray2, 
                        desiredWest - intShift, desiredEast - intShift, //shifting desired is faster than shifting all x,y
                        desiredSouth, desiredNorth);
                    if (tn > 0) {
                        lon.add(Integer.MAX_VALUE); //indicates moveTo next point
                        lat.add(Integer.MAX_VALUE);
                        for (int i = 0; i < tn; i++) {
                            lon.add(xArray2[i] + intShift);
                            lat.add(yArray2[i]);
                        }
                    }
                }

                //useStandard   (this is independent of useShiftedLeft and useShiftedRight)
                if (useStandard) {
                    //reduce and draw
                    int tn = reduce(n, xArray, yArray, 
                        desiredWest, desiredEast, 
                        desiredSouth, desiredNorth);
                    if (tn > 0) {
                        //add to lon and lat
                        lon.add(Integer.MAX_VALUE); //indicates moveTo next point
                        lat.add(Integer.MAX_VALUE);
                        for (int i = 0; i < tn; i++) {
                            lon.add(xArray[i]);
                            lat.add(yArray[i]);
                        }
                    }
                }

            } else {
                //skip over the data
                long remain = 8 * n;
                while (remain > 0) 
                    remain -= dis.skip(remain);
            }

        }
        if (verbose) String2.log("  GSHHS.getPathInfo done. TIME=" + (System.currentTimeMillis() - time));
    }

    /**
     * This reduces the points outside of the desired bounds.
     * This is simpler than clipping but serves my purpose well --
     * far fewer points to store and draw.
     *
     * @param n the number of active points in xa and ya (at least 2).
     *    They must all be non-NaN.
     * @param xa the array with x values
     * @param ya the array with y values
     * @param west   degrees * 1000000, in range (0 - 360 or +-180) appropriate for xa
     * @param east   degrees * 1000000, in range (0 - 360 or +-180) appropriate for xa
     * @param south  degrees * 1000000
     * @param north  degrees * 1000000
     * @return n the new active number of points in the arrays (may be 0)
     */
    public static int reduce(int n, int x[], int y[], 
        int west, int east, int south, int north) {
 
        //algorithm relies on: 
        //for a series of points which are out of bounds in the same direction,
        //  only the first and last points need to be kept.
        //But corners are a problem, so always permanentely save first and last 
        //  out in given direction.
        if (n < 2) return n;
        int tn = 2;  //temp n good points  
        int i = 2;  //next point to look at
        while (i < n) {
            //look for sequences of points all to beyond one border
            boolean caughtSequence = false;
            if (x[tn-2] < west  && x[tn-1] < west  && x[i] < west)  {
                i++;
                while (i < n && x[i] < west) 
                    i++;
                caughtSequence = true;
            } else if (x[tn-2] > east  && x[tn-1] > east  && x[i] > east)  {
                i++;
                while (i < n && x[i] > east) 
                    i++;
                caughtSequence = true;
            } else if (y[tn-2] < south && y[tn-1] < south && y[i] < south)  {
                i++;
                while (i < n && y[i] < south) 
                    i++;
                caughtSequence = true;
            } else if (y[tn-2] > north && y[tn-1] > north && y[i] > north)  {
                i++;
                while (i < n && y[i] > north) 
                    i++;
                caughtSequence = true;
            } 

            if (caughtSequence) {
                //save the point one back
                x[tn] = x[i-1];
                y[tn] = y[i-1];
                tn++;
            } 
            
            //always save this point
            if (i < n) {
                x[tn] = x[i];
                y[tn] = y[i];
                tn++;
                i++;
            }
        }
        //if (n > 1000) String2.log("GSHHS.reduce n=" + n + " newN=" + (n2+1)); 

        //are the remaining points out of range?
        if (tn <= 4) {
            for (i = 0; i < tn; i++) if (x[i] >= west) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (x[i] <= east) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (y[i] >= south) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (y[i] <= north) break; 
            if (i == tn) return 0;
        }
        return tn;
    }

    /**
     * This reduces the points outside of the desired bounds.
     * This is simpler than clipping but serves my purpose well --
     * far fewer points to store and draw.
     *
     * @param n the number of active points in xa and ya (at least 2).
     *    They must all be non-NaN. 
     * @param xa the array with x values
     * @param ya the array with y values
     * @param west
     * @param east
     * @param south
     * @param north
     * @return n the new active number of points in the arrays (may be 0)
     */
    public static int reduce(int n, double x[], double y[], 
        double west, double east, double south, double north) {
 
        //algorithm relies on: 
        //for a series of points which are out of bounds in the same direction,
        //  only the first and last points need to be kept.
        //But corners are a problem, so always permanentely save first and last 
        //  out in given direction.
        if (n < 2) return n;
        int tn = 2;  //temp n good points
        int i = 2;  //next point to look at
        while (i < n) {
            //look for sequences of points all to beyond one border
            boolean caughtSequence = false;
            if (x[tn-2] < west  && x[tn-1] < west  && x[i] < west)  {
                i++;
                while (i < n && x[i] < west) 
                    i++;
                caughtSequence = true;
            } else if (x[tn-2] > east  && x[tn-1] > east  && x[i] > east)  {
                i++;
                while (i < n && x[i] > east) 
                    i++;
                caughtSequence = true;
            } else if (y[tn-2] < south && y[tn-1] < south && y[i] < south)  {
                i++;
                while (i < n && y[i] < south) 
                    i++;
                caughtSequence = true;
            } else if (y[tn-2] > north && y[tn-1] > north && y[i] > north)  {
                i++;
                while (i < n && y[i] > north) 
                    i++;
                caughtSequence = true;
            } 

            if (caughtSequence) {
                //save the point one back
                x[tn] = x[i-1];
                y[tn] = y[i-1];
                tn++;
            } 
            
            //always save this point
            if (i < n) {
                x[tn] = x[i];
                y[tn] = y[i];
                tn++;
                i++;
            }
        }
        //if (n > 1000) String2.log("GSHHS.reduce n=" + n + " newN=" + (n2+1)); 

        //are the remaining points out of range?
        if (tn <= 4) {
            for (i = 0; i < tn; i++) if (x[i] >= west) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (x[i] <= east) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (y[i] >= south) break; 
            if (i == tn) return 0;
            for (i = 0; i < tn; i++) if (y[i] <= north) break; 
            if (i == tn) return 0;
        }
        return tn;
    }


    /**
     * This runs a unit test.
     */
    public static void test() throws Exception {
        verbose = true;

        //force creation of new file
        GeneralPath gp1 = getGeneralPath('h', 1, -135, -105, 22, 50, true);

        //read cached version
        long time = System.currentTimeMillis();
        GeneralPath gp2 = getGeneralPath('h', 1, -135, -105, 22, 50, true);
        time = System.currentTimeMillis() - time;

        //is it the same  (is GeneralPath.equals a deep test? probably not)

        //test speed
        Test.ensureTrue(time < 20, "time=" + time); 


    }


}
