/* 
 * SgtGraph Copyright 2006, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch.sgt;

import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.DoubleObject;
import com.cohort.util.File2;
import com.cohort.util.Image2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.AttributedString2;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pmel.sgt.*;
import gov.noaa.pmel.sgt.demo.*;
import gov.noaa.pmel.sgt.dm.*;
import gov.noaa.pmel.util.*;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;
import java.awt.RenderingHints; 
import java.awt.Shape;
import java.io.File;
import java.io.*;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import javax.imageio.ImageIO;


/**
 * This class holds a SGT graph.
 * A note about coordinates:
 *   <ul>
 *   <li> Graph - uses "user" coordinates (e.g., lat and lon).
 *   <li> Layer - uses "physical" coordinates (doubles, 0,0 at lower left).
 *   <li> JPane - uses "device" coordinates (ints, 0,0 at upper left).
 *   </ul>
 */
public class SgtGraph  {


    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR;

    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;

    /**
     * Set this to true (by calling reallyVerbose=true in your program, not but changing the code here)
     * if you want lots and lots of diagnostic messages sent to String2.log.
     */
    public static boolean reallyVerbose = false;

    private String fontFamily;
    public double defaultAxisLabelHeight = SgtUtil.DEFAULT_AXIS_LABEL_HEIGHT; 
    public double defaultLabelHeight = SgtUtil.DEFAULT_LABEL_HEIGHT; 
    public double majorLabelRatio = 1.25; //axisTitleHeight/axisLabelHeight; 1.25 matches SGT

    public static Color backgroundColor = new Color(0xCCCCFF); 
    public int widenOnePoint = 1;  //pixels
    private final static String testImageExtension = ".png"; //was/could be ".gif"

    /**
     * Constructor.
     * This throws exception if trouble
     *
     * @param fontFamily the name of the font family (a safe choice is "SansSerif")
     */
    public SgtGraph(String fontFamily) {

        this.fontFamily = fontFamily;

        //create the font  (ensure the fontFamily is available)
        SgtUtil.getFont(fontFamily);

    }
    
    private static int rint(double d) {return Math2.roundToInt(d); }
     

    /**
     * This uses SGT to plot data on a graph.
     * Strings should be "" if not needed.
     * If there is no data, the graph and legend are still drawn, but
     * the graph has "No Data" in the center.
     *
     * @param xAxisTitle  null = none
     * @param yAxisTitle  null = none
     * @param legendPosition must be SgtUtil.LEGEND_BELOW (SgtUtil.LEGEND_RIGHT currently not supported)
     * @param legendTitle1 the first line of the legend
     * @param legendTitle2 the second line of the legend
     * @param imageDir the directory with the logo file 
     * @param logoImageFile the logo image file in the imageDir (should be square image) 
     *    (currently, must be png, gif, jpg, or bmp)
     *    (currently noaa-simple-40.gif for lowRes),
     *    or null for none.
     * @param minX the min value for the X axis. Use Double.NaN to tell makeGraph to set it.
     *    If xIsTimeAxis, specify minX in epochSeconds.
     * @param maxX the max value for the X axis. Use Double.NaN to tell makeGraph to set it.
     *    If xIsTimeAxis, specify minX in epochSeconds.
     * @param minY the min value for the Y axis. Use Double.NaN to tell makeGraph to set it.
     *    If xIsTimeAxis, specify minX in epochSeconds. 
     * @param maxY the max value for the Y axis. Use Double.NaN to tell makeGraph to set it.
     *    If xIsTimeAxis, specify minX in epochSeconds. 
     * @param xIsTimeAxis   note that SGT doesn't allow x and y to be time axes
     * @param yIsTimeAxis
     * @param graphDataLayers an ArrayList of GraphDataLayers with the data to be plotted; 
     *    the first column in the table is treated as the X data, 
     *    the second column as the y data.
     *    If you want variable-color filled markers, specify a colorMap
     *    and choose a "filled" marker type.
     * @param g2 the graphics2D object to be used (the image background color should
     *    already have been drawn)
     * @param baseULXPixel defines area to be used, in pixels 
     * @param baseULYPixel defines area to be used, in pixels 
     * @param imageWidthPixels defines the area to be used, in pixels 
     * @param imageHeightPixels defines the area to be used, in pixels 
     * @param graphWidthOverHeight the desired shape of the graph 
     *     (or NaN or 0 to fill the available space)
     * @param fontScale relative to 1=normalHeight
     * @throws Exception
     */
    public void makeGraph(String xAxisTitle, String yAxisTitle,
        int legendPosition, String legendTitle1, String legendTitle2,
        String imageDir, String logoImageFile,
        double minX, double maxX, double minY, double maxY, 
        boolean xIsTimeAxis, boolean yIsTimeAxis,
        ArrayList graphDataLayers,
        Graphics2D g2,
        int baseULXPixel, int baseULYPixel,
        int imageWidthPixels, int imageHeightPixels,
        double graphWidthOverHeight,
        double fontScale
        //, String customFileName
        ) throws Exception {

        //Coordinates in SGT:
        //   Graph - 'U'ser coordinates      (graph's axes' coordinates)
        //   Layer - 'P'hysical coordinates  (e.g., psuedo-inches, 0,0 is lower left)
        //   JPane - 'D'evice coordinates    (pixels, 0,0 is upper left)

        //set the clip region
        g2.setClip(baseULXPixel, baseULYPixel, imageWidthPixels, imageHeightPixels);
        try {
            if (verbose) String2.log("\n{{ SgtGraph.makeGraph "); // + Math2.memoryString());
            long startTime = System.currentTimeMillis();
            long setupTime = System.currentTimeMillis();

            if (xIsTimeAxis && yIsTimeAxis) {
                //Test.error(String2.ERROR + " in SgtGraph.makeGraph: SGT doesn't allow x and y axes to be time axes.");
                //kludge: just make y not a time axis
                yIsTimeAxis = false;
            }
            double axisLabelHeight = fontScale * defaultAxisLabelHeight;
            double labelHeight     = fontScale * defaultLabelHeight;

            //read the data from the files
            Table tables[] = new Table[graphDataLayers.size()];
            Grid  grids[]  = new Grid[graphDataLayers.size()];
            boolean useTableData[] = new boolean[graphDataLayers.size()];
            boolean someUseGridData = false;
            for (int i = 0; i < graphDataLayers.size(); i++) {
                GraphDataLayer gdl = (GraphDataLayer)graphDataLayers.get(i);
                if (reallyVerbose) String2.log("  graphDataLayer" + i + "=" + gdl);
                tables[i] = gdl.table;
                grids[i] = gdl.grid1;
                useTableData[i] = tables[i] != null;
                if (!useTableData[i]) 
                    someUseGridData = true;
                //String2.log("SgtGraph table" + i + "=" + tables[i].toString("row", 20));
            }
            
            //if minX maxX not specified, calculate x axis ranges
            if (Math2.isFinite(minX) && Math2.isFinite(maxX)) {
            } else {
                minX = Double.MAX_VALUE;
                maxX = -Double.MAX_VALUE;
                for (int i = 0; i < tables.length; i++) {
                    GraphDataLayer gdl = (GraphDataLayer)graphDataLayers.get(i);
                    double xStats[] = useTableData[i]?
                        tables[i].getColumn(gdl.v1).calculateStats() :
                        (new DoubleArray(grids[i].lon)).calculateStats();
                    if (reallyVerbose) String2.log("  table=" + i + 
                        " minX=" + xStats[PrimitiveArray.STATS_MIN] + 
                        " maxX=" + xStats[PrimitiveArray.STATS_MAX]);
                    if (xStats[PrimitiveArray.STATS_N] > 0) {
                        minX = Math.min(minX, xStats[PrimitiveArray.STATS_MIN]);
                        maxX = Math.max(maxX, xStats[PrimitiveArray.STATS_MAX]);
                    }
                }
                if (minX == Double.MAX_VALUE) { 
                    //no data, draw fake axis
                    minX = 0; maxX = 1;
                    xIsTimeAxis = false;
                } else if (someUseGridData) {
                    //use current x range unless no data
                } else if (xIsTimeAxis) {
                    double r20 = (maxX - minX) / 20;
                    minX -= r20;
                    maxX += r20;                    
                } else {
                    double xLowHigh[] = Math2.suggestLowHigh(minX, maxX);
                    minX = xLowHigh[0];
                    maxX = xLowHigh[1];
                }
            }
            if (minX == maxX) { 
                //no data variation
                if (xIsTimeAxis) { //do +/-an hour
                    minX -= Calendar2.SECONDS_PER_HOUR;
                    maxX += Calendar2.SECONDS_PER_HOUR;
                } else {
                    double xLowHigh[] = Math2.suggestLowHigh(minX, maxX);
                    minX = xLowHigh[0];
                    maxX = xLowHigh[1];
                }
            }

            //are any of the gdl's vector STICKS?  
            boolean sticksGraph = false;
            int sticksGraphNRows = -1;
            double minData = Double.MAX_VALUE;
            double maxData = -Double.MAX_VALUE;
            for (int spli = 0; spli < graphDataLayers.size(); spli++) {
                GraphDataLayer gdl = (GraphDataLayer)graphDataLayers.get(spli);
                if (gdl.draw == GraphDataLayer.DRAW_STICKS) {
                    sticksGraph = true;
                    sticksGraphNRows = tables[spli].nRows();

                    //calculate v variable's stats
                    double vStats[] = tables[spli].getColumn(gdl.v3).calculateStats();
                    if (reallyVerbose) String2.log("  table=" + spli + 
                        " minData=" + vStats[PrimitiveArray.STATS_MIN] + 
                        " maxData=" + vStats[PrimitiveArray.STATS_MAX]);
                    if (vStats[PrimitiveArray.STATS_N] > 0) {
                        minData = Math.min(minData, vStats[PrimitiveArray.STATS_MIN]);
                        maxData = Math.max(maxData, vStats[PrimitiveArray.STATS_MAX]);
                    }
                }
            }

            //if minY maxY not specified, calculate y axis ranges
            if (!Math2.isFinite(minY) || !Math2.isFinite(maxY)) {
                minY = Double.MAX_VALUE;
                maxY = -Double.MAX_VALUE;
                for (int i = 0; i < tables.length; i++) {
                    GraphDataLayer gdl = (GraphDataLayer)graphDataLayers.get(i);
                    double yStats[] = useTableData[i]?
                        tables[i].getColumn(gdl.v2).calculateStats() :
                        (new DoubleArray(grids[i].lat)).calculateStats();
                    if (reallyVerbose) String2.log("  table=" + i + " minY=" + yStats[1] + " maxY=" + yStats[2]);
                    if (yStats[PrimitiveArray.STATS_N] > 0) {
                        minY = Math.min(minY, yStats[PrimitiveArray.STATS_MIN]);
                        maxY = Math.max(maxY, yStats[PrimitiveArray.STATS_MAX]);
                    }
                }
                if (minY == Double.MAX_VALUE) { 
                    //no data, draw fake axes
                    minY = 0; maxY = 0;
                    yIsTimeAxis = false;
                } else if (someUseGridData) {
                    //use exact minY maxY
                } else if (yIsTimeAxis) {
                    double r20 = (maxY - minY) / 20;
                    minY -= r20;
                    maxY += r20;                    
                } else {
                    //get suggestedLowHigh for y
                    double yLowHigh[] = Math2.suggestLowHigh(minY, maxY);
                    minY = yLowHigh[0];
                    maxY = yLowHigh[1];
                }

                //if any gdl's are vector STICKS, include minData,maxData and make y range symmetrical
                //(only need longest u or v component, not longest stick)
                //(technically, just need to encompass longest v component,
                //  but there may be stick and non-stick layers)
                if (sticksGraph) {
                    maxY = Math.max(Math.abs(minY), Math.abs(maxY));  //u component
                    if (maxY == 0) 
                        maxY = 0.1;
                    if (minData != Double.MAX_VALUE) {  //v component
                        maxY = Math.max(maxY, Math.abs(minData));
                        maxY = Math.max(maxY, Math.abs(maxData));
                    }
                    minY = -maxY;
                }
            }
            if (minY == maxY) { 
                double yLowHigh[] = Math2.suggestLowHigh(minY, maxY);
                minY = yLowHigh[0];
                maxY = yLowHigh[1];
            }

            //whether yMin yMax predefined or not, if sticksGraph  ensure maxY=-minY 
            //rendering algorithm below depends on it
            if (sticksGraph) {
                //make room for sticks at end points
                //  just less than 1 month; any more looks like no data for those months
                //--nice idea, but it looks like there is no data for that section of time
                //just do it for 1month data
                double xRange = maxX - minX;
                if (sticksGraphNRows >= 2 && 
                    xRange / (sticksGraphNRows - 1) > 29 * Calendar2.SECONDS_PER_DAY) { //x is in seconds
                    minX -= 29 * Calendar2.SECONDS_PER_DAY;  //almost a month
                    maxX += 29 * Calendar2.SECONDS_PER_DAY;
                }

                //special calculation of minY maxY
                maxY = Math.max(Math.abs(minY), Math.abs(maxY));  //u component
                minY = -maxY;
            }
            double scaleXIfTime = xIsTimeAxis? 1000 : 1; //s to ms
            double scaleYIfTime = yIsTimeAxis? 1000 : 1; //s to ms
            if (reallyVerbose) String2.log("  sticksGraph=" + sticksGraph + 
                "\n    minX=" + (xIsTimeAxis? Calendar2.epochSecondsToIsoStringT(minX) : "" + minX) + 
                     " maxX=" + (xIsTimeAxis? Calendar2.epochSecondsToIsoStringT(maxX) : "" + maxX) + 
                "\n    minY=" + (yIsTimeAxis? Calendar2.epochSecondsToIsoStringT(minY) : "" + minY) + 
                     " maxY=" + (yIsTimeAxis? Calendar2.epochSecondsToIsoStringT(maxY) : "" + maxY)); 


            //figure out the params needed to make the graph
            String error = "";
            if (minX > maxX) {double d = minX; minX = maxX; maxX = d;}
            if (minY > maxY) {double d = minY; minY = maxY; maxY = d;}
            double xRange = maxX - minX; 
            double yRange = maxY - minY; 
            int rangeScaler = 1;
            boolean narrowXGraph = false; 
            if      (imageWidthPixels <= 300) {
                rangeScaler *= 2; //adjust for small graphs, e.g., 250  
                narrowXGraph = true;
            } else if (imageWidthPixels >= 600) rangeScaler /= 1; //adjust for big graphs e.g., 700
            if      (fontScale >= 3)    rangeScaler *= 4;  //aim at fontScale 4 
            else if (fontScale >= 1.5)  rangeScaler *= 2;  //aim at fontScale 2
            else if (fontScale >= 0.75) ;                  //aim at fontScale 1
            else if (fontScale >= 0.37) rangeScaler /= 2;  //aim at fontScale 0.5
            else                        rangeScaler /= 4;  //aim at fontScale 0.25
            double xDivisions[] = Math2.suggestDivisions(xRange * rangeScaler);
            double yDivisions[] = Math2.suggestDivisions(yRange * rangeScaler); 

            //define sizes
            double dpi = 100; //dots per inch
            double imageWidthInches  = imageWidthPixels  / dpi;  
            double imageHeightInches = imageHeightPixels / dpi;
            double betweenGraphAndLegend = fontScale * .25;
            int labelHeightPixels = Math2.roundToInt(labelHeight * dpi);
            double betweenGraphAndColorBar  = fontScale * .25;

            //set legend location and size (in pixels)   for LEGEND_RIGHT
            //standard length of vector (and other samples) in user units (e.g., inches)
            double legendSampleSizeInches = 0.22; //Don't change this (unless make other changes re vector length on graph)
            int legendSampleSize = Math2.roundToInt(legendSampleSizeInches * dpi); 
            int legendBoxWidth  = Math2.roundToInt(fontScale * 1.4 * dpi); //1.4inches 
            int legendBoxHeight = imageHeightPixels;  
            int legendBoxULX = baseULXPixel + imageWidthPixels - legendBoxWidth; 
            int legendBoxULY = baseULYPixel;
            int legendInsideBorder = Math2.roundToInt(fontScale * 0.1 * dpi); 

            //deal with LEGEND_BELOW   
            boolean narrowLegend = imageWidthPixels < Math2.roundToInt(300 / fontScale);
            int narrowLegend1 = narrowLegend? 1 : 0;
            if (legendPosition == SgtUtil.LEGEND_BELOW) {
                double legendLineCount = 
                    (legendTitle1 == null && legendTitle2 == null)? -1 : 1; //for legend title   //???needs adjustment for larger font size
                for (int i = 0; i < graphDataLayers.size(); i++) 
                    legendLineCount += ((GraphDataLayer)graphDataLayers.get(i)).legendLineCount(narrowLegend);
                //String2.log("legendLineCount=" + legendLineCount);
                legendBoxWidth = imageWidthPixels;  
                legendBoxHeight = (int)(legendLineCount * labelHeightPixels) + 
                    2 * legendInsideBorder;  
                legendBoxULX = baseULXPixel;
                legendBoxULY = baseULYPixel + imageHeightPixels - legendBoxHeight;
            }
            //determine appropriate axis lengths to best fill available space
            double graphULX = fontScale * 0.37;   //relative to baseULXYPixel
            double graphULY = fontScale * 0.2;
            double graphBottomY = fontScale * 0.37;
            double graphWidth = imageWidthInches - graphULX - legendBoxWidth/dpi -
                betweenGraphAndLegend;
            double graphHeight = imageHeightInches - graphBottomY - graphULY;
            if (legendPosition == SgtUtil.LEGEND_BELOW) {
                graphWidth = imageWidthInches - graphULX - betweenGraphAndLegend;
                graphHeight = imageHeightInches - graphBottomY - graphULY - legendBoxHeight/dpi;
            }

            //adjust graph width or height
            if (Double.isNaN(graphWidthOverHeight) || graphWidthOverHeight < 0) {
                //leave it (as big as possible)
            } else if (graphWidth / graphHeight < graphWidthOverHeight) {
                //graph is too tall -- reduce the height
                double newGraphHeight = graphWidth / graphWidthOverHeight;
                //String2.log("graph is too tall  oldHt=" + graphHeight + " newHt=" + newGraphHeight);
                double diff = graphHeight - newGraphHeight; //a positive number
                graphHeight = newGraphHeight;
                if (legendPosition == SgtUtil.LEGEND_BELOW) {
                    graphBottomY += diff;
                    legendBoxULY -= Math2.roundToInt(diff * dpi);
                } else {
                    graphBottomY += diff / 2;
                    graphULY += diff / 2;
                }
            } else { 
                //graph is too wide -- reduce the width
                double newGraphWidth = graphWidthOverHeight * graphHeight;
                //String2.log("graph is too wide  oldWidth=" + graphWidth + " newWidth=" + newGraphWidth);
                double diff = graphWidth - newGraphWidth; //a positive number
                graphWidth = newGraphWidth;
                if (legendPosition == SgtUtil.LEGEND_BELOW) {
                    graphULX += diff / 2;
                } else {
                    graphULX += diff;
                    legendBoxULX -= Math2.roundToInt(diff * dpi);
                }
            } 
            //String2.log("graphULX=" + graphULX + " ULY=" + graphULY + " width=" + graphWidth +
            //    " height=" + graphHeight + " bottomY=" + graphBottomY + 
            //    "\n  widthPixels=" + graphWidthPixels + " heightPixels=" + graphHeightPixels);

            //legendTextX and Y
            int legendTextX = legendPosition == SgtUtil.LEGEND_BELOW?
                legendBoxULX + legendSampleSize + 2 * legendInsideBorder :  //leftX    
                legendBoxULX + legendBoxWidth / 2; //centerX 
            int legendTextY = legendBoxULY + legendInsideBorder + labelHeightPixels;   
            //String2.log("SgtGraph baseULXPixel=" + baseULXPixel +  " baseULYPixel=" + baseULYPixel +
            //    "  imageWidth=" + imageWidthPixels + " imageHeight=" + imageHeightPixels +
            //    "\n  legend boxULX=" + legendBoxULX + " boxULY=" + legendBoxULY + 
            //    " boxWidth=" + legendBoxWidth + " boxHeight=" + legendBoxHeight +
            //    "\n  textX=" + legendTextX + " textY=" + legendTextY +
            //    " insideBorder=" + legendInsideBorder + " labelHeightPixels=" + labelHeightPixels);

            //create the label font
            Font labelFont = new Font(fontFamily, Font.PLAIN, 10); //Font.ITALIC

            //SgtUtil.drawHtmlText needs non-text antialiasing ON
            Object originalAntialiasing = 
                g2.getRenderingHint(RenderingHints.KEY_ANTIALIASING); 
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                RenderingHints.VALUE_ANTIALIAS_ON); 

            //draw legend basics
            if (true) {
                //box for legend    
                g2.setColor(new Color(0xFFFFCC));       
                g2.fillRect(legendBoxULX, legendBoxULY, legendBoxWidth - 1, legendBoxHeight - 1);
                g2.setColor(Color.black);
                g2.drawRect(legendBoxULX, legendBoxULY, legendBoxWidth - 1, legendBoxHeight - 1);

                //legend titles
                if (legendTitle1 == null && legendTitle2 == null) {
                    //don't draw the legend title
                } else {
                    if (legendPosition == SgtUtil.LEGEND_BELOW) {
                        //draw LEGEND_BELOW
                        legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                            0, fontFamily, labelHeightPixels * 3 / 2, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle1 + " " + 
                                legendTitle2) + "</color></b>");
                        legendTextY += labelHeightPixels / 2;
                    } else {
                        //draw LEGEND_RIGHT
                        int tx = legendBoxULX + legendInsideBorder;
                        legendTextY = SgtUtil.drawHtmlText(g2, tx, legendTextY, 
                            0, fontFamily, labelHeightPixels * 5 / 4, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle1) + "</color></b>");
                        legendTextY = SgtUtil.drawHtmlText(g2, tx, legendTextY, 
                            0, fontFamily, labelHeightPixels * 5 / 4, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle2) + "</color></b>");
                        legendTextY += labelHeightPixels * 3 / 2;
                    }

                    //draw the logo
                    if (logoImageFile != null && File2.isFile(imageDir + logoImageFile)) {
                        long logoTime = System.currentTimeMillis();
                        BufferedImage bi2 = ImageIO.read(new File(imageDir + logoImageFile));

                        //g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                        //                    RenderingHints.VALUE_INTERPOLATION_BICUBIC);
                        //                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
                        //                    RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
                        //draw LEGEND_RIGHT
                        int ulx = baseULXPixel + (int)((imageWidthInches - fontScale * 0.45)*dpi);
                        int uly = baseULYPixel + (int)(fontScale * 0.05 * dpi);
                        int tSize = (int)(fontScale * 40);
                        if (legendPosition == SgtUtil.LEGEND_BELOW) {
                            //draw LEGEND_BELOW
                            ulx = legendBoxULX + legendSampleSize / 2;
                            uly = legendBoxULY + legendInsideBorder / 2;
                            tSize = (int)(fontScale * 20);
                        }
                        g2.drawImage(bi2, ulx, uly, tSize, tSize, null); //null=ImageObserver 
                        //if (verbose) String2.log("  draw logo time=" + 
                        //    (System.currentTimeMillis() - logoTime));
                    }
                }
            }

            //set up the Projection Cartesian  
            int graphX1 = baseULXPixel + rint(graphULX * dpi);
            int graphY2 = baseULYPixel + rint(graphULY * dpi);
            int graphX2 = graphX1 + rint(graphWidth * dpi); 
            int graphY1 = graphY2 + rint(graphHeight * dpi);
            int graphWidthPixels  = graphX2 - graphX1;
            int graphHeightPixels = graphY1 - graphY2;
            //String2.log("    graphX1=" + graphX1 + " X2=" + graphX2 + 
            //    " graphY1=" + graphY1 + " Y2=" + graphY2);
            boolean narrowYGraph = graphHeightPixels < 250; 
            if (reallyVerbose) String2.log(
                "  graphWidth=" + graphWidthPixels + " narrowX=" + narrowXGraph +
                " graphHeight=" + graphHeightPixels + " narrowY=" + narrowYGraph);
            CartesianProjection cp = new CartesianProjection(
                minX * scaleXIfTime, maxX * scaleXIfTime, 
                minY * scaleYIfTime, maxY * scaleYIfTime, 
                graphX1, graphX2, graphY1, graphY2);
            DoubleObject dox1 = new DoubleObject(0);
            DoubleObject doy1 = new DoubleObject(0);
            DoubleObject dox2 = new DoubleObject(0);
            DoubleObject doy2 = new DoubleObject(0);

            //draw the graph background color 
            g2.setColor(backgroundColor);
            g2.fillRect(graphX1, graphY2, graphWidthPixels + 1, graphHeightPixels + 1); //+1 since SGT sometimes draws larger rect

             //create the pane
            JPane jPane = new JPane("", new java.awt.Dimension(
                baseULXPixel + imageWidthPixels, 
                baseULYPixel + imageHeightPixels));
            jPane.setLayout(new StackedLayout());
            StringArray layerNames = new StringArray();

            //create the common graph parts
            //graph's physical location (start, end, delta); delta is ignored
            Range2D xPhysRange = new Range2D(
                baseULXPixel/dpi + graphULX, 
                baseULXPixel/dpi + graphULX + graphWidth,  1); 
            Range2D yPhysRange = new Range2D(
                (legendPosition == SgtUtil.LEGEND_BELOW? legendBoxHeight/dpi : 0) + graphBottomY, 
                imageHeightInches - graphULY, 1); 
            Dimension2D layerDimension2D = new Dimension2D(
                baseULXPixel/dpi + imageWidthInches, 
                baseULYPixel/dpi + imageHeightInches);

            //redefine some graph parts for this graph with SoT objects -- real min max
            SoTRange xUserRange = xIsTimeAxis?
                (SoTRange)new SoTRange.Time((long)(minX * scaleXIfTime), (long)(maxX * scaleXIfTime)):
                (SoTRange)new SoTRange.Double(minX, maxX, xDivisions[0]);
            SoTRange yUserRange = yIsTimeAxis?
                (SoTRange)new SoTRange.Time((long)(minY * scaleYIfTime), (long)(maxY * scaleYIfTime)):
                (SoTRange)new SoTRange.Double(minY, maxY, yDivisions[0]);
            gov.noaa.pmel.sgt.LinearTransform xt = 
                new gov.noaa.pmel.sgt.LinearTransform(xPhysRange, xUserRange);
            gov.noaa.pmel.sgt.LinearTransform yt = 
                new gov.noaa.pmel.sgt.LinearTransform(yPhysRange, yUserRange);
            SoTPoint origin2 = new SoTPoint(
                xIsTimeAxis? (SoTValue)new SoTValue.Time((long)(minX * scaleXIfTime)) : (SoTValue)new SoTValue.Double(minX), 
                yIsTimeAxis? (SoTValue)new SoTValue.Time((long)(minY * scaleYIfTime)) : (SoTValue)new SoTValue.Double(minY));

            //draw the point layers 
            int nTotalValid = 0;
            for (int gdli = 0; gdli < graphDataLayers.size(); gdli++) {

                long gdlTime = System.currentTimeMillis();
             
                //prepare to plot the data              
                GraphDataLayer gdl = (GraphDataLayer)graphDataLayers.get(gdli);              
                int tMarkerSize = rint(gdl.markerSize * fontScale); 
                boolean drawMarkers = gdl.draw == GraphDataLayer.DRAW_MARKERS;
                boolean drawLines = gdl.draw == GraphDataLayer.DRAW_LINES;
                boolean drawMarkersAndLines = gdl.draw == GraphDataLayer.DRAW_MARKERS_AND_LINES;
                boolean drawSticks = gdl.draw == GraphDataLayer.DRAW_STICKS;               
                boolean drawColoredSurface = gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE ||
                    gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE_AND_CONTOUR_LINES;
                boolean drawContourLines = gdl.draw == GraphDataLayer.DRAW_CONTOUR_LINES ||
                    gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE_AND_CONTOUR_LINES;

                //clip to graph region
                g2.setClip(graphX1, graphY2, graphWidthPixels, graphHeightPixels);                        

                //plot tableData
                if (useTableData[gdli]) {
                    PrimitiveArray xPA = tables[gdli].getColumn(gdl.v1);
                    PrimitiveArray yPA = tables[gdli].getColumn(gdl.v2);
                    PrimitiveArray vPA = gdl.v3 >= 0? 
                        tables[gdli].getColumn(gdl.v3) : null;
                    //String2.log("vPA is null? " + (vPA==null) + " colorMap=" + gdl.colorMap);
                    //first single pt is wide. all others a dot.
                    int tWidenOnePoint = drawLines? widenOnePoint : 0; //0 if markers involved
                    tMarkerSize = rint(gdl.markerSize * fontScale); 
                    int n = Math.min(xPA.size(), yPA.size());
                    int gpState = 0; //0=needsMoveTo (0 pts), 1=needsLineTo (1pt), 2=last was lineTo (2+ pts)
                    int nValid = 0;
                    int itx = 0, ity = 0;
                    //String2.log("SgtGraph: draw point layer " + gdli);

                    //draw the mean line first (so underneath data) 
                    if (gdl.regressionType == GraphDataLayer.REGRESS_MEAN) {
                    
                        //calculate the mean of y
                        double[] stats = yPA.calculateStats();
                        double statsN = stats[PrimitiveArray.STATS_N]; 
                        if (statsN > 0) {
                            double mean = stats[PrimitiveArray.STATS_SUM] / statsN;
                            if (drawSticks) {
                                //for sticks, mean is "component mean"
                                //It is calculated from uMean and vMean (not from mean of vector lengths).
                                //This matches method used to plot "average" vector on the graph.
                                //And this matches how NDBC calculates the "average" for a given hour
                                // (see http://www.ndbc.noaa.gov/measdes.shtml#stdmet).
                                stats = vPA.calculateStats();
                                statsN = stats[PrimitiveArray.STATS_N]; 
                                if (statsN > 0) {
                                    double vMean = stats[PrimitiveArray.STATS_SUM] / statsN;
                                    mean = Math.sqrt(mean*mean + vMean*vMean);
                                }
                            }
                            if (reallyVerbose) String2.log("  drawing mean=" + mean + 
                                " n=" + stats[PrimitiveArray.STATS_N] + 
                                " min=" + stats[PrimitiveArray.STATS_MIN] + 
                                " max=" + stats[PrimitiveArray.STATS_MAX]);
                            if (statsN > 0 && //check again, it may have changed
                                mean > minY && mean < maxY && 
                                //don't let mean line obscure no variability of data 
                                stats[PrimitiveArray.STATS_MIN] != stats[PrimitiveArray.STATS_MAX]) {
                                    
                                //draw the line
                                g2.setColor(SgtUtil.whiter(SgtUtil.whiter(SgtUtil.whiter(gdl.lineColor))));
                                cp.graphToDevice(minX * scaleXIfTime, mean * scaleYIfTime, dox1, doy1);
                                cp.graphToDevice(maxX * scaleXIfTime, mean * scaleYIfTime, dox2, doy2);
                                g2.drawLine(rint(dox1.d), rint(doy1.d),
                                    rint(dox2.d), rint(doy2.d));
                            }
                        }
                    }


                    //drawSticks
                    if (drawSticks) {
                        g2.setColor(gdl.lineColor);
                        for (int ni = 0; ni < n; ni++) {
                            double dtx = xPA.getDouble(ni); 
                            double dty = yPA.getDouble(ni);

                            //for sticksGraph, maxY=-minY (set above), 
                            double dtv = vPA.getDouble(ni);
                            if (Double.isNaN(dtx) || Double.isNaN(dty) || Double.isNaN(dtv))
                                continue;
                            nValid++;

                            //draw line from base to tip
                            cp.graphToDevice(dtx * scaleXIfTime, 0, dox1, doy1);
                            g2.drawLine(
                                rint(dox1.d), rint(doy1.d),
                                rint(dox1.d + cp.graphToDeviceYDistance(dty)), //u component
                                //move in x direction, the distance you would have moved up
                                rint(doy1.d - cp.graphToDeviceYDistance(dtv))); //v component  "-" to move up 
                        }
                    }

                    //drawMarkers || drawLines || drawMarkersAndLines
                    if (drawMarkers || drawLines || drawMarkersAndLines) {
                        g2.setColor(gdl.lineColor);
                        IntArray markerXs = new IntArray();
                        IntArray markerYs = new IntArray();
                        ArrayList markerInteriorColors = new ArrayList();
                        if (reallyVerbose) String2.log("  draw=" + GraphDataLayer.DRAW_NAMES[gdl.draw] + " n=" + n);
                        long accumTime = System.currentTimeMillis();
                        int oitx, oity;
                        for (int ni = 0; ni < n; ni++) {
                            double dtx = xPA.getDouble(ni);  
                            double dty = yPA.getDouble(ni);
                            double dtv = vPA == null? Double.NaN : vPA.getDouble(ni);

                            if (Double.isNaN(dtx) || Double.isNaN(dty)
                                //I think valid dtx dty with invalid dtv is hollow point
                                //|| (gpl.colorMap != null && Double.isNaN(dtv))
                                ) {
                                if (gpState == 1) { //gpState==1 only occurs if drawing lines
                                    //just one point (moveTo left hanging)? draw a mini horizontal line to right and left
                                    //itx, ity will still have last valid point
                                    g2.drawLine(itx - tWidenOnePoint, ity, itx + tWidenOnePoint, ity);
                                    tWidenOnePoint = 0;
                                }
                                gpState = 0;
                                continue;
                            }

                            //point is valid.  get int coordinates of point
                            nValid++;
                            cp.graphToDevice(dtx * scaleXIfTime, dty * scaleYIfTime, dox1, doy1);
                            oitx = itx;
                            oity = ity;
                            itx = rint(dox1.d);
                            ity = rint(doy1.d);

                            //draw line
                            if (drawLines || drawMarkersAndLines) {
                                if (gpState == 0) {
                                    gpState = 1;
                                } else {
                                    g2.drawLine(oitx, oity, itx, ity);
                                    gpState = 2;
                                }
                            }

                            //drawing markers? store location
                            if (drawMarkers || drawMarkersAndLines) {
                                markerXs.add(itx);
                                markerYs.add(ity);
                                markerInteriorColors.add(
                                    gdl.colorMap == null? gdl.lineColor : gdl.colorMap.getColor(dtv));
                            }
                        }
                        if (reallyVerbose) String2.log("  accum time=" + (System.currentTimeMillis() - accumTime));

                        if (gpState == 1) { //gpState==1 only occurs if drawing lines
                            //just one point (moveTo left hanging)? draw a mini horizontal line to right and left
                            g2.drawLine(itx - tWidenOnePoint, ity, itx + tWidenOnePoint, ity);
                            tWidenOnePoint = 0;
                            gpState = 0;
                            //if (verbose) String2.log("  gpState=1! drawing final point");
                        }

                        //draw markers on top of lines
                        int nMarkerXs = markerXs.size();
                        if (nMarkerXs > 0) {
                            long markerTime = System.currentTimeMillis();
                            for (int i = 0; i < nMarkerXs; i++) {
                                drawMarker(g2, gdl.markerType, tMarkerSize, 
                                    markerXs.array[i], markerYs.array[i],
                                    (Color)markerInteriorColors.get(i), gdl.lineColor);
                            }
                            if (reallyVerbose) String2.log("  draw markers n=" + nMarkerXs + 
                                " time=" + (System.currentTimeMillis() - markerTime));
                        }

                    }

                    //update nTotalValid
                    nTotalValid += nValid;
                } else {

                    if (scaleXIfTime != 1) {
                        double ar[] = gdl.grid1.lon;
                        int tn = ar.length;
                        for (int i = 0; i < tn; i++) 
                            ar[i] *= scaleXIfTime;
                    }
                    if (scaleYIfTime != 1) {
                        double ar[] = gdl.grid1.lat;
                        int tn = ar.length;
                        for (int i = 0; i < tn; i++) 
                            ar[i] *= scaleYIfTime;
                    }


                    //useGridData
                    if (drawColoredSurface) {
                        if (reallyVerbose) String2.log("  drawColoredSurface: " + gdl);
                        CompoundColorMap colorMap = (CompoundColorMap)gdl.colorMap;
                        CartesianGraph graph = new CartesianGraph("", xt, yt);
                        Layer layer = new Layer("coloredSurface", layerDimension2D);
                        layerNames.add(layer.getId());
                        jPane.add(layer);      //calls layer.setPane(this);
                        layer.setGraph(graph); //calls graph.setLayer(this);
                        //graph.setClip(minX * scaleXIfTime, maxX * scaleXIfTime,
                        //              minY * scaleXIfTime, maxY * scaleYIfTime);
                        //graph.setClipping(true);

                        //get the Grid
                        SimpleGrid simpleGrid = new SimpleGrid(gdl.grid1.data, 
                            gdl.grid1.lon, gdl.grid1.lat, ""); //title

                        //temp
                        if (reallyVerbose) {
                            gdl.grid1.calculateStats();
                            String2.log("  grid data min=" + gdl.grid1.minData + 
                                "  max=" + gdl.grid1.maxData + 
                                "  n=" + gdl.grid1.nValidPoints +
                                "\n  grid x min=" + gdl.grid1.lon[0] + " max=" + gdl.grid1.lon[gdl.grid1.lon.length - 1] +
                                "\n  grid y min=" + gdl.grid1.lat[0] + " max=" + gdl.grid1.lat[gdl.grid1.lat.length - 1]);
                        }

                        //assign the data 
                        graph.setData(simpleGrid, new GridAttribute(GridAttribute.RASTER, colorMap)); 

                        if (gdl.boldTitle == null) {
                        } else if (legendPosition == SgtUtil.LEGEND_BELOW) {
                            //draw LEGEND_BELOW
                            //add a horizontal colorBar
                            legendTextY += labelHeightPixels; 
                            CompoundColorMapLayerChild ccmLayerChild = 
                                new CompoundColorMapLayerChild("", colorMap);
                            ccmLayerChild.setRectangle( //leftX,upperY(when rotated),width,height
                                layer.getXDtoP(legendTextX), layer.getYDtoP(legendTextY), 
                                imageWidthInches - (2 * legendInsideBorder + legendSampleSize)/dpi - 
                                    betweenGraphAndColorBar, 
                                fontScale * 0.15); 
                            ccmLayerChild.setLabelFont(labelFont);
                            ccmLayerChild.setLabelHeightP(axisLabelHeight);
                            ccmLayerChild.setTicLength(fontScale * 0.02);
                            layer.addChild(ccmLayerChild);
                            legendTextY += 3 * labelHeightPixels; 

                            //add legend text
                            legendTextY = SgtUtil.belowLegendText(g2, narrowLegend, legendTextX, legendTextY,
                                fontFamily, labelHeightPixels, gdl.boldTitle, 
                                gdl.title2, gdl.title3, gdl.title4);
                        } else {
                            //draw LEGEND_RIGHT    //NO LONGER UP-TO-DATE
                            //box for colorBar
                            /*g2.setColor(new Color(0xFFFFCC));       
                            g2.fillRect(colorBarBoxLeftX, baseULYPixel, 
                                colorBarBoxWidth - 1, imageHeightPixels - 1); 
                            g2.setColor(Color.black);
                            g2.drawRect(colorBarBoxLeftX, baseULYPixel, 
                                colorBarBoxWidth - 1, imageHeightPixels - 1); 

                            //add a vertical colorBar
                            CompoundColorMapLayerChild ccmLayerChild = 
                                new CompoundColorMapLayerChild("", colorMap);
                            int bottomStuff = legendInsideBorder + 3 * labelHeightPixels;
                            ccmLayerChild.setRectangle( //leftX,lowerY,width,height
                                layer.getXDtoP(colorBarBoxLeftX + legendInsideBorder), 
                                layer.getYDtoP(baseULYPixel + imageHeightPixels - bottomStuff), 
                                fontScale * 0.2, //inches
                                (imageHeightPixels - legendInsideBorder - labelHeightPixels/2 - bottomStuff)/dpi); 
                            ccmLayerChild.setLabelFont(labelFont);
                            ccmLayerChild.setLabelHeightP(axisLabelHeight);
                            ccmLayerChild.setTicLength(fontScale * 0.02);
                            layer.addChild(ccmLayerChild);

                            //add text in the colorBarBox
                            if (verbose) String2.log("  baseULY=" + baseULYPixel + 
                                " imageHeightPixels=" + imageHeightPixels +
                                " inside=" + legendInsideBorder +
                                " labelHeightPixels=" + labelHeightPixels); 
                            int ty = baseULYPixel + imageHeightPixels - legendInsideBorder - labelHeightPixels;
                            ty = SgtUtil.drawHtmlText(g2, colorBarBoxLeftX + colorBarBoxWidth / 2, 
                                ty, 1, fontFamily, labelHeightPixels, false,
                                "<b>" + SgtUtil.encodeAsHtml(gdl.boldTitle) + "</b>");
                            //SgtUtil.drawHtmlText(g2, colorBarBoxLeftX + colorBarBoxWidth / 2, 
                            //    ty, 1, fontFamily, labelHeightPixels, false, SgtUtil.encodeAsHtml(gridUnits));

                            //add legend text
                            legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                1, fontFamily, labelHeightPixels, false, 
                                "<b>" + SgtUtil.encodeAsHtml(gdl.boldTitle) + "</b>");
                            legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                1, fontFamily, labelHeightPixels, false, SgtUtil.encodeAsHtml(gridUnits));
                            legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                1, fontFamily, labelHeightPixels, false, 
                                SgtUtil.encodeAsHtml(gridTitle2));                    
                            legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                1, fontFamily, labelHeightPixels, false, 
                                SgtUtil.encodeAsHtml(gridDate));                    
                            legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                1, fontFamily, labelHeightPixels, true, 
                                SgtUtil.encodeAsHtml(gridCourtesy));                    
                            */
                        }
                    }
                    if (drawContourLines) {
                        CartesianGraph graph = new CartesianGraph("", xt, yt);
                        Layer layer = new Layer("contourLines", layerDimension2D);
                        layerNames.add(layer.getId());
                        jPane.add(layer);      //calls layer.setPane(this);
                        layer.setGraph(graph); //calls graph.setLayer(this);
                        //graph.setClip(minX * scaleXIfTime, maxX * scaleXIfTime,
                        //              minY * scaleXIfTime, maxY * scaleYIfTime);
                        //graph.setClipping(true);

                        //get the Grid
                        SimpleGrid simpleGrid = new SimpleGrid(gdl.grid1.data, 
                            gdl.grid1.lon, gdl.grid1.lat, ""); //title
                        gdl.grid1.calculateStats(); //so grid.minData maxData is correct
                        double gridMinData = gdl.grid1.minData;
                        double gridMaxData = gdl.grid1.maxData;
                        if (reallyVerbose)
                            String2.log("  contour minData=" + String2.genEFormat6(gridMinData) + 
                                " maxData=" + String2.genEFormat10(gridMaxData));

                        //assign the data 
                        //get color levels from colorMap
                        CompoundColorMap colorMap = (CompoundColorMap)gdl.colorMap;
                        int nLevels = colorMap.rangeLow.length + 1;
                        double levels[] = new double[nLevels];
                        System.arraycopy(colorMap.rangeLow, 0, levels, 0, nLevels - 1);
                        levels[nLevels - 1] = colorMap.rangeHigh[nLevels - 1];
                        DecimalFormat format = new DecimalFormat("#0.######");
                        ContourLevels contourLevels = new ContourLevels();
                        for (int i = 0; i < levels.length; i++) {
                            ContourLineAttribute contourLineAttribute = new ContourLineAttribute();
                            contourLineAttribute.setColor(gdl.lineColor);
                            contourLineAttribute.setLabelColor(gdl.lineColor);
                            contourLineAttribute.setLabelHeightP(fontScale * 0.15);
                            contourLineAttribute.setLabelFormat("%g"); //this seems to be active
                            contourLineAttribute.setLabelText(format.format(levels[i])); //this seems to be ignored
                            contourLevels.addLevel(levels[i], contourLineAttribute);
                        }
                        graph.setData(simpleGrid, new GridAttribute(contourLevels));
                        if (reallyVerbose) 
                            String2.log("  contour levels = " + String2.toCSVString(levels));

                        //add legend text
                        //don't draw legend if gdl.draw = DRAW_COLORED_SURFACE_AND_CONTOUR_LINES
                        if (gdl.boldTitle != null && gdl.draw == GraphDataLayer.DRAW_CONTOUR_LINES) { 
                            if (legendPosition == SgtUtil.LEGEND_BELOW) {
                                //draw LEGEND_BELOW
                                g2.setColor(gdl.lineColor);
                                g2.drawLine(
                                    legendTextX - legendSampleSize - legendInsideBorder, 
                                    legendTextY - labelHeightPixels/2, 
                                    legendTextX - legendInsideBorder, 
                                    legendTextY - labelHeightPixels/2);

                                //add legend text
                                legendTextY = SgtUtil.belowLegendText(g2, narrowLegend, legendTextX, legendTextY,
                                    fontFamily, labelHeightPixels, gdl.boldTitle, 
                                    gdl.title2, gdl.title3, gdl.title4);
                            } else {
                                //draw LEGEND_RIGHT
                                /*g2.setColor(contourColor);
                                g2.drawLine(
                                    legendTextX - legendSampleSize/2, 
                                    legendTextY - labelHeightPixels*7/8, 
                                    legendTextX + legendSampleSize/2, 
                                    legendTextY - labelHeightPixels*7/8);

                                legendTextY += labelHeightPixels/2; //for demo line
                                legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                    1, fontFamily, labelHeightPixels, false, 
                                        "<b>" + SgtUtil.encodeAsHtml(contourBoldTitle) + "</b>");
                                legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                    1, fontFamily, labelHeightPixels, false, SgtUtil.encodeAsHtml(contourUnits));
                                legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                    1, fontFamily, labelHeightPixels, false, 
                                    SgtUtil.encodeAsHtml(contourTitle2));
                                legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                    1, fontFamily, labelHeightPixels, false, 
                                    SgtUtil.encodeAsHtml(contourDate));
                                legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                                    1, fontFamily, labelHeightPixels, true, 
                                    SgtUtil.encodeAsHtml(contourCourtesy));
                                */
                            }
                        }
                    }
                }

                //change clip back to full area
                g2.setClip(baseULXPixel, baseULYPixel, imageWidthPixels, imageHeightPixels);

                 //draw legend for this GraphDataLayer
                if (gdl.boldTitle == null) {
                    //no legend entry for this gdl
                } else if (legendPosition == SgtUtil.LEGEND_BELOW) {

                    //draw LEGEND_BELOW
                    g2.setColor(gdl.lineColor);

                    //draw colorMap (do first since colorMap shifts other things down)
                    if ((drawMarkers || drawMarkersAndLines) && gdl.colorMap != null) {
                        //draw the color bar
                        CartesianGraph graph = new CartesianGraph("colorbar" + gdli, xt, yt);
                        Layer layer = new Layer("colorbar" + gdli, layerDimension2D);
                        layerNames.add(layer.getId());
                        jPane.add(layer);      //calls layer.setPane(this);
                        layer.setGraph(graph); //calls graph.setLayer(this);

                        legendTextY += labelHeightPixels; 
                        CompoundColorMapLayerChild lc = 
                            new CompoundColorMapLayerChild("", (CompoundColorMap)gdl.colorMap);
                        lc.setRectangle( //leftX,upperY(when rotated),width,height
                            layer.getXDtoP(legendTextX), layer.getYDtoP(legendTextY), 
                            imageWidthInches - (2 * legendInsideBorder + legendSampleSize)/dpi - 
                                betweenGraphAndColorBar, 
                            fontScale * 0.15); 
                        lc.setLabelFont(labelFont);
                        lc.setLabelHeightP(axisLabelHeight);
                        lc.setTicLength(fontScale * 0.02);
                        layer.addChild(lc);
                        legendTextY += 3 * labelHeightPixels; 
                    }
                        
                    //draw a line
                    if (drawLines || drawMarkersAndLines || drawSticks) {  
                        g2.drawLine(
                            legendTextX - legendSampleSize - legendInsideBorder, 
                            legendTextY - labelHeightPixels/2, 
                            legendTextX - legendInsideBorder, 
                            legendTextY - labelHeightPixels/2);
                    }

                    //then draw marker
                    if (drawMarkers || drawMarkersAndLines) {
                        int tx = legendTextX - legendInsideBorder - legendSampleSize / 2;
                        int ty = legendTextY - labelHeightPixels/2;
                        drawMarker(g2, gdl.markerType, tMarkerSize, tx, ty, 
                            gdl.colorMap == null? gdl.lineColor : 
                                gdl.colorMap.getColor((gdl.colorMap.getRange().start + gdl.colorMap.getRange().end) / 2), 
                            gdl.lineColor);
                    } 
                    //draw legend text
                    legendTextY = SgtUtil.belowLegendText(g2, narrowLegend, legendTextX, legendTextY,
                        fontFamily, labelHeightPixels, 
                        gdl.boldTitle, gdl.title2, gdl.title3, gdl.title4);
                } else {
                    //draw LEGEND_RIGHT
                    g2.setColor(gdl.lineColor);

                    //draw a line
                    if (drawLines || drawMarkersAndLines || drawSticks) {
                        g2.drawLine(
                            legendTextX - legendSampleSize/2, 
                            legendTextY - labelHeightPixels*7/8, 
                            legendTextX + legendSampleSize/2, 
                            legendTextY - labelHeightPixels*7/8);
                        legendTextY += labelHeightPixels / 2; //for demo line
                    }

                    //draw a marker
                    if (drawMarkers || drawMarkersAndLines) {
                        int tx = legendTextX;
                        int ty = legendTextY - labelHeightPixels; 
                        drawMarker(g2, gdl.markerType, tMarkerSize, tx, ty, 
                            gdl.lineColor, gdl.lineColor);
                        legendTextY += labelHeightPixels;
                    } 
                    
                    //point legend text
                    legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                        1, fontFamily, labelHeightPixels, false, 
                        "<b>" + SgtUtil.encodeAsHtml(gdl.boldTitle) + "</b>");
                    legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                        1, fontFamily, labelHeightPixels, false, 
                        SgtUtil.encodeAsHtml(gdl.title2));
                    legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                        1, fontFamily, labelHeightPixels, true, 
                        SgtUtil.encodeAsHtml(gdl.title3));
                    legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                        1, fontFamily, labelHeightPixels, true, 
                        SgtUtil.encodeAsHtml(gdl.title4));
                }
                if (reallyVerbose)
                    String2.log("  graphDataLayer" + gdli + " time=" +
                        (System.currentTimeMillis() - gdlTime));
            }

            if (reallyVerbose) String2.log("  set up the graph time=" + 
                (System.currentTimeMillis() - setupTime));

            //*** set up graph with the AXIS LINES 
            //Drawing once avoids anti-aliasing problems when axis labels drawn 2+ times.
            //Draw this last, so axis lines drawn over data at the edges.
            //String2.log("SgtGraph.makeGraph before jPane.draw: " + Math2.memoryString());
            long drawGraphTime = System.currentTimeMillis();

            if (true) {
                CartesianGraph graph = new CartesianGraph("", xt, yt);
                Layer layer = new Layer("axis", layerDimension2D);
                layerNames.add(layer.getId());
                jPane.add(layer);      //calls layer.setPane(this);
                layer.setGraph(graph); //calls graph.setLayer(this);
                //no clipping needed
                //DegreeMinuteFormatter dmf = new DegreeMinuteFormatter(); 

                //create the x axis
                Axis xAxis;
                if (xIsTimeAxis) {
                    //TimeAxis
                    TimeAxis timeAxis = new TimeAxis(TimeAxis.AUTO);
                    xAxis = timeAxis;
                    //timeAxis.setRangeU(xUserRange);
                    //timeAxis.setLocationU(origin);
                } else {
                    //plainAxis
                    PlainAxis2 plainAxis = new PlainAxis2(new GenEFormatter());
                    xAxis = plainAxis;
                    //plainAxis.setRangeU(xUserRange);
                    //plainAxis.setLocationU(origin);
                    int nXSmallTics = Math2.roundToInt(xDivisions[0] / xDivisions[1]) - 1;
                    plainAxis.setNumberSmallTics(nXSmallTics); 
                    plainAxis.setLabelInterval(1);
                    //plainAxis.setLabelFormat("%g");
                    if (xAxisTitle != null && xAxisTitle.length() > 0) {
                        SGLabel xTitle = new SGLabel("", xAxisTitle, new Point2D.Double(0, 0));
                        xTitle.setHeightP(majorLabelRatio * axisLabelHeight);
                        xTitle.setFont(labelFont);
                        plainAxis.setTitle(xTitle);
                    }
                }
                xAxis.setRangeU(xUserRange);
                xAxis.setLocationU(origin2); //exception thrown here if x and y are time axes
                xAxis.setLabelFont(labelFont); 
                xAxis.setLabelHeightP(axisLabelHeight);
                xAxis.setSmallTicHeightP(fontScale * .02);
                xAxis.setLargeTicHeightP(fontScale * .05);
                graph.addXAxis(xAxis);
                if (narrowXGraph && xIsTimeAxis) {
                    TimeAxis timeAxis = (TimeAxis)xAxis;
                    int interval = timeAxis.getMinorLabelInterval();
                    if (reallyVerbose) String2.log("  x timeAxis interval (now doubled) was " + interval);
                    timeAxis.setMinorLabelInterval(interval * 2);
                }

                //create the y axes
                Axis yAxis;
                if (yIsTimeAxis) {
                    //TimeAxis
                    TimeAxis timeAxis = new TimeAxis(TimeAxis.AUTO);
                    yAxis = timeAxis;
                    //timeAxis.setRangeU(yUserRange);
                    //timeAxis.setLocationU(origin);
                } else {
                    //plainAxis
                    PlainAxis2 plainAxis = new PlainAxis2(new GenEFormatter());
                    yAxis = plainAxis;
                    //plainAxis.setRangeU(yUserRange);
                    //plainAxis.setLocationU(origin);
                    int nYSmallTics = Math2.roundToInt(yDivisions[0] / yDivisions[1]) - 1;
                    plainAxis.setNumberSmallTics(nYSmallTics); 
                    plainAxis.setLabelInterval(1);
                    //plainAxis.setLabelFormat("%g");
                    if (yAxisTitle != null && yAxisTitle.length() > 0) {
                        SGLabel yTitle = new SGLabel("", yAxisTitle, new Point2D.Double(0, 0));
                        yTitle.setHeightP(majorLabelRatio * axisLabelHeight);
                        yTitle.setFont(labelFont);
                        plainAxis.setTitle(yTitle);
                    }
                }
                yAxis.setRangeU(yUserRange);
                yAxis.setLocationU(origin2);
                yAxis.setLabelFont(labelFont); 
                yAxis.setLabelHeightP(axisLabelHeight);
                yAxis.setSmallTicHeightP(fontScale * .02);
                yAxis.setLargeTicHeightP(fontScale * .05);
                graph.addYAxis(yAxis);
                if (narrowYGraph && yIsTimeAxis) {
                    TimeAxis timeAxis = (TimeAxis)yAxis;
                    int interval = timeAxis.getMinorLabelInterval();
                    if (reallyVerbose) String2.log("  y timeAxis interval (now doubled) was " + interval);
                    timeAxis.setMinorLabelInterval(interval * 2);
                }
            }

            //actually draw the graph
            jPane.draw(g2);  //comment out for memory leak tests

            //avoid memory leak in sgt
            //Problem: JPane and all subcomponents don't seem to be garbage-collected
            //  as one would expect (since they are created and used only in this  
            //  method).  
            //Note that the parts all have references to each other
            //  (e.g., JPane keeps track of Layers and Layers know their JPane,
            //  and similarly Layers have links to/from Graphs and 
            //  Graphs have links to/from Renderers(SimpleGrid + Attribute).
            //  It is all one big, bidirectionally-linked blob.
            //Possible Cause 1: There could be so many links (and cross-links?)
            //  than the gc can't release any of it.
            //Possible Cause 2: There could be a link to the JPane (or some other
            //  part of the graph) held externally 
            //  (like, but not, the mouse event listeners in JPane and Pane, 
            //  or jPane.removeNotify(), or Swing?)
            //  which is preventing the blob from being gc'd.
            //  I looked for, but never found, any actual cause like this.
            //Solution: deal with possible cause 1: I manually removed links 
            //  between parts of the JPane, Layers, Graphs, and parts of the graph.
            //  I really thought culprit was cause 2, but the success of this
            //  solution is evidence that cause 1 is the culprit.
            //  For maps, bathymetry and boundary CartesianGraphs still not being gc'd; 
            //    but they aren't created by this class, so no known lingering
            //    problems.
            for (int i = 0; i < layerNames.size(); i++) {
                try {
                    Layer layer = jPane.getLayer(layerNames.get(i)); 
                    if (layer != null) {
                        CartesianGraph graph = (CartesianGraph)layer.getGraph(); 
                        if (graph != null) {
                            //remove the graph's parts
                            graph.setData((SimpleGrid)null, (GridAttribute)null);  //has huge effect
                            graph.xClipRange_ = null; //must be made public
                            graph.yClipRange_ = null; //must be made public
                            graph.setXTransform((AxisTransform)null); //requires small change to allow null
                            graph.setYTransform((AxisTransform)null); //requires small change to allow null
                            graph.xAxis_ = null; //must be made public (removeAllXAxes wasn't enough)
                            graph.yAxis_ = null; //must be made public (removeAllYAxes wasn't enough)

                            //break the links to/from layer
                            graph.setLayer((Layer)null); //graph.setLayer must be made public
                            layer.setGraph((CartesianGraph)null); //requires small change to setGraph to allow null
                        }
                        layer.removeAllChildren();  //has big effect (all the SGLabels) 
                        layer.setPane((JPane)null); //requires small change to setPane to allow null
                    }
                } catch (Exception e) {
                    String2.log(MustBe.throwable("SgtGraph.makeGraph removing graph's parts", e));
                }
            }
            jPane.removeAll();  

            if (reallyVerbose) {
                String2.log("  draw the graph time=" + 
                    (System.currentTimeMillis() - drawGraphTime));
                //Math2.gc(50); //outside of timing system
                //String2.log("SgtGraph.makeGraph after jPane.draw: " + Math2.memoryString());
                //String2.log("SgtGraph.makeGraph after gc: " + Math2.memoryString());
            }

            //draw the topX and rightY axis lines  (and main X Y if just lines)
            g2.setColor(Color.black);
            g2.drawLine(graphX1, graphY2, graphX2, graphY2);
            g2.drawLine(graphX2, graphY2, graphX2, graphY1);
            //if (!drawAxes) {
            //    g2.drawLine(graphX1, graphY1, graphX2, graphY1);
            //    g2.drawLine(graphX1, graphY1, graphX1, graphY2);
            //}
            if (!someUseGridData && nTotalValid == 0) {
                g2.setFont(fontScale == 1? labelFont : labelFont.deriveFont(labelFont.getSize2D() * (float)fontScale));
                g2.drawString("No Data", 
                    (graphX1 + graphX2) / 2 - Math2.roundToInt(17 * fontScale), 
                    (graphY1 + graphY2) / 2);
            }

            //turn off antialiasing           
            if (originalAntialiasing != null)
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                    originalAntialiasing); 

            //display time to makeGraph
            if (verbose) String2.log("}} SgtGraph.makeGraph done. TOTAL TIME=" + 
                (System.currentTimeMillis() - startTime) + "\n");
        } finally {
            g2.setClip(null);
        }
    }

    /**
     * This draws the requested marker at the requested position. 
     * g2.setColor must have been already used.
     * 
     * @param g2d 
     * @param markerType
     * @param markerSize2 the marker size in pixels (actually Graphics2D units)
     * @param x the center x
     * @param y the center y
     * @param interiorColor  use null if inactive, or to get hollow "filled" marker.
     *    For non-filled markers, this color is used as the line color. 
     * @param lineColor
     */
    public static void drawMarker(Graphics2D g2d, int markerType, int markerSize, 
        int x, int y, Color interiorColor, Color lineColor) {

        if (markerType <= GraphDataLayer.MARKER_TYPE_NONE) 
            return;

        //allow markers of any size (simple use of /2 forces to act like only even sizes)
        int m2 = markerSize / 2;
        int ulx = x - m2;
        int uly = y - m2;
        
        if (markerType == GraphDataLayer.MARKER_TYPE_SQUARE) {
            int xa[] = {ulx, ulx, ulx + markerSize, ulx + markerSize};
            int ya[] = {uly, uly + markerSize, uly + markerSize, uly};
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawPolygon(xa, ya, 4);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_FILLED_SQUARE) {
            int xa[] = {ulx, ulx, ulx + markerSize, ulx + markerSize};
            int ya[] = {uly, uly + markerSize, uly + markerSize, uly};
            if (interiorColor != null) {
                g2d.setColor(interiorColor);
                g2d.fillPolygon(xa, ya, 4);
            }
            g2d.setColor(lineColor);
            g2d.drawPolygon(xa, ya, 4);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_CIRCLE) {
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawOval(ulx, uly, markerSize, markerSize);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_FILLED_CIRCLE) {
            if (interiorColor != null) {
                g2d.setColor(interiorColor);
                g2d.fillOval(ulx, uly, markerSize, markerSize); 
            }
            g2d.setColor(lineColor);
            g2d.drawOval(ulx, uly, markerSize, markerSize); 
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_UP_TRIANGLE) {
            int m21 = m2 + 1; //to make the size look same as others
            int xa[] = {x - m21, x,       x + m21}; //ensure symmetrical
            int ya[] = {y + m21, y - m21, y + m21};
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawPolygon(xa, ya, 3);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_FILLED_UP_TRIANGLE) {
            int m21 = m2 + 1; //to make the size look same as others
            int xa[] = {x - m21, x,       x + m21}; //ensure symmetrical
            int ya[] = {y + m21, y - m21, y + m21};
            if (interiorColor != null) {
                g2d.setColor(interiorColor);
                g2d.fillPolygon(xa, ya, 3);
            }
            g2d.setColor(lineColor);
            g2d.drawPolygon(xa, ya, 3);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_PLUS) {
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawLine(ulx, y, x + m2, y); //ensure symmetrical
            g2d.drawLine(x, uly, x, y + m2);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_X) {
            m2 = (markerSize - 1) / 2;  //-1 to make similar wt
            ulx = x - m2;
            uly = y - m2;
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawLine(ulx, uly, ulx + markerSize, uly + markerSize);
            g2d.drawLine(ulx, uly + markerSize, ulx + markerSize, uly);
            return;
        }
        if (markerType == GraphDataLayer.MARKER_TYPE_DOT) {
            g2d.setColor(interiorColor == null? lineColor : interiorColor);
            g2d.drawLine(x, y, x+1, y);
            return;
        }
    }


    /**
     * This uses SGT to make an image with a horizontal legend for one GraphDataLayer.
     *
     * @param xAxisTitle  null = none
     * @param yAxisTitle  null = none
     * @param legendPosition must be SgtUtil.LEGEND_BELOW (SgtUtil.LEGEND_RIGHT currently not supported)
     * @param legendTitle1 the first line of the legend
     * @param legendTitle2 the second line of the legend
     * @param imageDir the directory with the logo file 
     * @param logoImageFile the logo image file in the imageDir (should be square image) 
     *    (currently, must be png, gif, jpg, or bmp)
     *    (currently noaa-simple-40.gif for lowRes),
     *    or null for none.
     * @param graphDataLayer
     * @param g2 the graphics2D object to be used (the background need not 
     *    already have been drawn)
     * @param imageWidthPixels defines the image width, in pixels 
     * @param imageHeightPixels defines the image height, in pixels 
     * @param fontScale relative to 1=normalHeight
     * @throws Exception
     */
    public void makeLegend(String xAxisTitle, String yAxisTitle,
        int legendPosition, String legendTitle1, String legendTitle2,
        String imageDir, String logoImageFile,
        GraphDataLayer gdl,
        Graphics2D g2,
        int imageWidthPixels, int imageHeightPixels,
        double fontScale
        ) throws Exception {

        //Coordinates in SGT:
        //   Graph - 'U'ser coordinates      (graph's axes' coordinates)
        //   Layer - 'P'hysical coordinates  (e.g., psuedo-inches, 0,0 is lower left)
        //   JPane - 'D'evice coordinates    (pixels, 0,0 is upper left)

        //set the clip region
        g2.setClip(0, 0, imageWidthPixels, imageHeightPixels);
        try {
            if (reallyVerbose) String2.log("\n{{ SgtGraph.makeLegend "); // + Math2.memoryString());
            long startTime = System.currentTimeMillis();
            long setupTime = System.currentTimeMillis();

            double axisLabelHeight = fontScale * defaultAxisLabelHeight;
            double labelHeight     = fontScale * defaultLabelHeight;

            boolean sticksGraph = gdl.draw == GraphDataLayer.DRAW_STICKS;

            String error = "";
            boolean narrowXGraph = imageWidthPixels <= 300;

            //define sizes
            double dpi = 100; //dots per inch
            double imageWidthInches  = imageWidthPixels  / dpi;  
            double imageHeightInches = imageHeightPixels / dpi;
            int labelHeightPixels = Math2.roundToInt(labelHeight * dpi);
            double betweenGraphAndColorBar  = fontScale * .25;

            //set legend location and size (in pixels)  
            //standard length of vector (and other samples) in user units (e.g., inches)
            double legendSampleSizeInches = 0.22; //Don't change this (unless make other changes re vector length on graph)
            int legendSampleSize = Math2.roundToInt(legendSampleSizeInches * dpi); 
            int legendInsideBorder = Math2.roundToInt(fontScale * 0.1 * dpi); 
            boolean narrowLegend = imageWidthPixels < Math2.roundToInt(300 / fontScale);
            int narrowLegend1 = narrowLegend? 1 : 0;
            double legendLineCount = 
                (legendTitle1 == null && legendTitle2 == null)? -1 : 1; //for legend title   //???needs adjustment for larger font size
            legendLineCount += gdl.legendLineCount(narrowLegend);
                //String2.log("legendLineCount=" + legendLineCount);
            int legendBoxULX = 0;
            int legendBoxULY = 0;

            //legendTextX and Y
            int legendTextX = legendBoxULX + legendSampleSize + 2 * legendInsideBorder; 
            int legendTextY = legendBoxULY + legendInsideBorder + labelHeightPixels;   
            //String2.log("SgtGraph baseULXPixel=" + baseULXPixel +  " baseULYPixel=" + baseULYPixel +
            //    "  imageWidth=" + imageWidthPixels + " imageHeight=" + imageHeightPixels +
            //    "\n  legend boxULX=" + legendBoxULX + " boxULY=" + legendBoxULY + 
            //    "\n  textX=" + legendTextX + " textY=" + legendTextY +
            //    " insideBorder=" + legendInsideBorder + " labelHeightPixels=" + labelHeightPixels);

            //create the label font
            Font labelFont = new Font(fontFamily, Font.PLAIN, 10); //Font.ITALIC

            //SgtUtil.drawHtmlText needs non-text antialiasing ON
            Object originalAntialiasing = 
                g2.getRenderingHint(RenderingHints.KEY_ANTIALIASING); 
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                RenderingHints.VALUE_ANTIALIAS_ON); 

            //draw legend basics
            if (true) {
                //box for legend    
                g2.setColor(new Color(0xFFFFCC));       
                g2.fillRect(legendBoxULX, legendBoxULY, imageWidthPixels - 1, imageHeightPixels - 1);
                g2.setColor(Color.black);
                g2.drawRect(legendBoxULX, legendBoxULY, imageWidthPixels - 1, imageHeightPixels - 1);

                //legend titles
                if (legendTitle1 == null && legendTitle2 == null) {
                    //don't draw the legend title
                } else {
                    if (legendPosition == SgtUtil.LEGEND_BELOW) {
                        //draw LEGEND_BELOW
                        legendTextY = SgtUtil.drawHtmlText(g2, legendTextX, legendTextY, 
                            0, fontFamily, labelHeightPixels * 3 / 2, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle1 + " " + 
                                legendTitle2) + "</color></b>");
                        legendTextY += labelHeightPixels / 2;
                    } else {
                        //draw LEGEND_RIGHT
                        int tx = legendBoxULX + legendInsideBorder;
                        legendTextY = SgtUtil.drawHtmlText(g2, tx, legendTextY, 
                            0, fontFamily, labelHeightPixels * 5 / 4, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle1) + "</color></b>");
                        legendTextY = SgtUtil.drawHtmlText(g2, tx, legendTextY, 
                            0, fontFamily, labelHeightPixels * 5 / 4, false, 
                            "<b><color=#2600aa>" + SgtUtil.encodeAsHtml(legendTitle2) + "</color></b>");
                        legendTextY += labelHeightPixels * 3 / 2;
                    }

                    //draw the logo
                    if (logoImageFile != null && File2.isFile(imageDir + logoImageFile)) {
                        long logoTime = System.currentTimeMillis();
                        BufferedImage bi2 = ImageIO.read(new File(imageDir + logoImageFile));

                        //g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                        //                    RenderingHints.VALUE_INTERPOLATION_BICUBIC);
                        //                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
                        //                    RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
                        int ulx = legendBoxULX + legendSampleSize / 2;
                        int uly = legendBoxULY + legendInsideBorder / 2;
                        int tSize = (int)(fontScale * 20);
                        g2.drawImage(bi2, ulx, uly, tSize, tSize, null); //null=ImageObserver 
                        //if (verbose) String2.log("  draw logo time=" + 
                        //    (System.currentTimeMillis() - logoTime));
                    }
                }
            }

            //done?
            if (gdl.boldTitle == null) 
                return; 

             //create the pane
            JPane jPane = new JPane("", new java.awt.Dimension(
                imageWidthPixels, imageHeightPixels));
            jPane.setLayout(new StackedLayout());
            StringArray layerNames = new StringArray();
            Dimension2D layerDimension2D = new Dimension2D(imageWidthInches, imageHeightInches);

            
            //prepare to plot the data              
            int tMarkerSize = rint(gdl.markerSize * fontScale); 
            boolean drawMarkers = gdl.draw == GraphDataLayer.DRAW_MARKERS;
            boolean drawLines = gdl.draw == GraphDataLayer.DRAW_LINES;
            boolean drawMarkersAndLines = gdl.draw == GraphDataLayer.DRAW_MARKERS_AND_LINES;
            boolean drawSticks = gdl.draw == GraphDataLayer.DRAW_STICKS;               
            boolean drawColoredSurface = gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE ||
                gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE_AND_CONTOUR_LINES;
            boolean drawContourLines = gdl.draw == GraphDataLayer.DRAW_CONTOUR_LINES ||
                gdl.draw == GraphDataLayer.DRAW_COLORED_SURFACE_AND_CONTOUR_LINES;

            //useGridData
            if (drawColoredSurface) {
                if (reallyVerbose) String2.log("  drawColoredSurface: " + gdl);
                CompoundColorMap colorMap = (CompoundColorMap)gdl.colorMap;
                Layer layer = new Layer("coloredSurface", layerDimension2D);
                layerNames.add(layer.getId());
                jPane.add(layer);      //calls layer.setPane(this);

                //add a horizontal colorBar
                legendTextY += labelHeightPixels; 
                CompoundColorMapLayerChild ccmLayerChild = 
                    new CompoundColorMapLayerChild("", colorMap);
                ccmLayerChild.setRectangle( //leftX,upperY(when rotated),width,height
                    layer.getXDtoP(legendTextX), layer.getYDtoP(legendTextY), 
                    imageWidthInches - (2 * legendInsideBorder + legendSampleSize)/dpi - 
                        betweenGraphAndColorBar, 
                    fontScale * 0.15); 
                ccmLayerChild.setLabelFont(labelFont);
                ccmLayerChild.setLabelHeightP(axisLabelHeight);
                ccmLayerChild.setTicLength(fontScale * 0.02);
                layer.addChild(ccmLayerChild);
                legendTextY += 3 * labelHeightPixels; 
            }

            //draw colorMap (do first since colorMap shifts other things down)
            if ((drawMarkers || drawMarkersAndLines) && gdl.colorMap != null) {
                //draw the color bar
                Layer layer = new Layer("colorbar", layerDimension2D);
                layerNames.add(layer.getId());
                jPane.add(layer);      //calls layer.setPane(this);

                legendTextY += labelHeightPixels; 
                CompoundColorMapLayerChild lc = 
                    new CompoundColorMapLayerChild("", (CompoundColorMap)gdl.colorMap);
                lc.setRectangle( //leftX,upperY(when rotated),width,height
                    layer.getXDtoP(legendTextX), layer.getYDtoP(legendTextY), 
                    imageWidthInches - (2 * legendInsideBorder + legendSampleSize)/dpi - 
                        betweenGraphAndColorBar, 
                    fontScale * 0.15); 
                lc.setLabelFont(labelFont);
                lc.setLabelHeightP(axisLabelHeight);
                lc.setTicLength(fontScale * 0.02);
                layer.addChild(lc);
                legendTextY += 3 * labelHeightPixels; 
            }
                
            //draw a line
            //don't draw line if gdl.draw = DRAW_COLORED_SURFACE_AND_CONTOUR_LINES
            if (gdl.draw == GraphDataLayer.DRAW_CONTOUR_LINES || 
                drawLines || drawMarkersAndLines || drawSticks) {  
                g2.setColor(gdl.lineColor);
                g2.drawLine(
                    legendTextX - legendSampleSize - legendInsideBorder, 
                    legendTextY - labelHeightPixels/2, 
                    legendTextX - legendInsideBorder, 
                    legendTextY - labelHeightPixels/2);
            }

            //draw marker
            if (drawMarkers || drawMarkersAndLines) {
                int tx = legendTextX - legendInsideBorder - legendSampleSize / 2;
                int ty = legendTextY - labelHeightPixels/2;
                g2.setColor(gdl.lineColor);
                drawMarker(g2, gdl.markerType, tMarkerSize, tx, ty, 
                    gdl.colorMap == null? gdl.lineColor : 
                        gdl.colorMap.getColor((gdl.colorMap.getRange().start + gdl.colorMap.getRange().end) / 2), 
                    gdl.lineColor);
            } 

            //draw legend text
            g2.setColor(gdl.lineColor);
            legendTextY = SgtUtil.belowLegendText(g2, narrowLegend, legendTextX, legendTextY,
                fontFamily, labelHeightPixels, 
                gdl.boldTitle, gdl.title2, gdl.title3, gdl.title4);

            //actually draw the graph
            jPane.draw(g2);  //comment out for memory leak tests

            //avoid memory leak in sgt
            //Problem: JPane and all subcomponents don't seem to be garbage-collected
            //  as one would expect (since they are created and used only in this  
            //  method).  
            //Note that the parts all have references to each other
            //  (e.g., JPane keeps track of Layers and Layers know their JPane,
            //  and similarly Layers have links to/from Graphs and 
            //  Graphs have links to/from Renderers(SimpleGrid + Attribute).
            //  It is all one big, bidirectionally-linked blob.
            //Possible Cause 1: There could be so many links (and cross-links?)
            //  than the gc can't release any of it.
            //Possible Cause 2: There could be a link to the JPane (or some other
            //  part of the graph) held externally 
            //  (like, but not, the mouse event listeners in JPane and Pane, 
            //  or jPane.removeNotify(), or Swing?)
            //  which is preventing the blob from being gc'd.
            //  I looked for, but never found, any actual cause like this.
            //Solution: deal with possible cause 1: I manually removed links 
            //  between parts of the JPane, Layers, Graphs, and parts of the graph.
            //  I really thought culprit was cause 2, but the success of this
            //  solution is evidence that cause 1 is the culprit.
            //  For maps, bathymetry and boundary CartesianGraphs still not being gc'd; 
            //    but they aren't created by this class, so no known lingering
            //    problems.
            for (int i = 0; i < layerNames.size(); i++) {
                try {
                    Layer layer = jPane.getLayer(layerNames.get(i)); 
                    if (layer != null) {
                        layer.removeAllChildren();  //has big effect (all the SGLabels) 
                        layer.setPane((JPane)null); //requires small change to setPane to allow null
                    }
                } catch (Exception e) {
                    String2.log(MustBe.throwable("SgtGraph.makeGraph removing graph's parts", e));
                }
            }
            jPane.removeAll();  

            //turn off antialiasing           
            if (originalAntialiasing != null)
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                    originalAntialiasing); 

            //display time to makeGraph
            if (verbose) String2.log("}} SgtGraph.makeLegend done. TOTAL TIME=" + 
                (System.currentTimeMillis() - startTime) + "\n");
        } finally {
            g2.setClip(null);
        }
    }

    /** This tests SgtGraph. */ 
    public static void test() throws Exception {
        verbose = true;
        reallyVerbose = true;
        PathCartesianRenderer.verbose = true;
        PathCartesianRenderer.reallyVerbose = true;
        //AttributedString2.verbose = true;
        long time = System.currentTimeMillis();
        String tempDir = SSR.getTempDirectory();
        SgtGraph sgtGraph = new SgtGraph("Bitstream Vera Sans"); //"SansSerif" is safe choice
        String imageDir = SSR.getContextDirectory() + "images/";

        int width = 800;  //2 graphs wide
        int height = 900;  //3 graphs high

        //graph 1: make a data file with data
        PrimitiveArray xCol1 = PrimitiveArray.factory(new double[]{1.500e9, 1.501e9, 1.502e9, 1.503e9});
        PrimitiveArray yCol1 = PrimitiveArray.factory(new double[]{1.1, 2.9, Double.NaN, 2.3});
        Table table1 = new Table();
        table1.addColumn("X", xCol1);
        table1.addColumn("Y", yCol1);

        //graph 2: make a data file with data
        PrimitiveArray xCol2 = PrimitiveArray.factory(new double[]{0,1,2,3}); //comment out for another test
        //PrimitiveArray xCol2 = PrimitiveArray.factory(new double[]{Double.NaN, Double.NaN, Double.NaN, Double.NaN}); //comment out for another test
        PrimitiveArray yCol2 = PrimitiveArray.factory(new double[]{Double.NaN, Double.NaN, Double.NaN, Double.NaN}); //comment out for another test
        Table table2 = new Table();
        table2.addColumn("X", xCol2);
        table2.addColumn("Y", yCol2);

        //graph 3: make a data file with data
        PrimitiveArray xCol3 = PrimitiveArray.factory(new double[]{1.500e9, 1.501e9, 1.502e9, 1.503e9, 1.504e9, 1.505e9, 1.506e9});
        PrimitiveArray uCol3 = PrimitiveArray.factory(new double[]{1,2,3,4,5,6,7}); 
        PrimitiveArray vCol3 = PrimitiveArray.factory(new double[]{-3,-2,-1,0,1,2,3}); 
        Table table3 = new Table();
        table3.addColumn("X", xCol3);
        table3.addColumn("U", uCol3);
        table3.addColumn("V", vCol3);
        
        //graph 5: make a data file with two time columns
        PrimitiveArray xCol5 = PrimitiveArray.factory(new double[]{1.500e9, 1.501e9, 1.502e9, 1.503e9, 1.504e9, 1.505e9, 1.506e9});
        PrimitiveArray yCol5 = PrimitiveArray.factory(new double[]{1.500e9, 1.504e9, 1.501e9, 1.500e9, 1.503e9, 1.502e9, 1.506e9});
        Table table5 = new Table();
        table5.addColumn("X", xCol5);
        table5.addColumn("Y", yCol5);
        
        //graph 1: make a graphDataLayer with data for a time series line
        GraphDataLayer graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            0, 1, -1, -1, -1, //x,y,  others unused
            GraphDataLayer.DRAW_MARKERS_AND_LINES, true, false,
            "Time", "Y Axis Title", //x,yAxisTitle  for now, always std units 
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "This is title4.",
            table1, null, null,
            null, new java.awt.Color(0x0099FF),
            GraphDataLayer.MARKER_TYPE_PLUS, GraphDataLayer.MARKER_SIZE_SMALL,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers1 = new ArrayList();
        graphDataLayers1.add(graphDataLayer);

        //graph 1: plus 10 random points of each marker type
        int lastMarker = 9;
        for (int i = 1; i <= lastMarker; i++) {
            //make a data file with data
            PrimitiveArray txCol = new DoubleArray();
            PrimitiveArray tyCol = new DoubleArray();
            for (int j = 0; j < 10; j++) {
                txCol.addDouble(1.500e9 + Math.random() * .0025e9);
                tyCol.addDouble(1.5 + Math.random());
            }
            Table ttable = new Table();
            ttable.addColumn("X", txCol);
            ttable.addColumn("Y", tyCol);

            graphDataLayer = new GraphDataLayer(
                -1, //which pointScreen
                0, 1, -1, -1, -1, //x,y,  others unused
                GraphDataLayer.DRAW_MARKERS, 
                true, false, //x,yIsTimeAxis
                "Time", "Y Axis Title", //x,yAxisTitle for now, always std units 
                null, //no bold title, so no legend entry                
                "title2", "title3", "title4", //will be ignored
                ttable, null, null,
                null, new java.awt.Color(0x0099FF),
                i, GraphDataLayer.MARKER_SIZE_SMALL,
                0, //vectorStandard
                GraphDataLayer.REGRESS_NONE);
            graphDataLayers1.add(graphDataLayer);
        }
        
        //graph 2: make a graphDataLayer with no data
        graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            0, 1, -1, -1, -1, //x,y, others unused
            GraphDataLayer.DRAW_MARKERS, 
            false, false, //x,yIsTimeAxis
            "X Axis Title", "Y Axis Title", //for now, always std units
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "title4",
            table2, null, null, 
            null, new java.awt.Color(0x0099FF),
            GraphDataLayer.MARKER_TYPE_FILLED_CIRCLE, GraphDataLayer.MARKER_SIZE_MEDIUM,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers2 = new ArrayList();
        graphDataLayers2.add(graphDataLayer);

        //graph 3: make a graphDataLayer with data for a sticks graph
        graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            0, 1, 2, -1, -1, //x,u,v
            GraphDataLayer.DRAW_STICKS, 
            true, false, //x,yIsTimeAxis
            "Time", "Y Axis Title", //for now, always std units
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "title4",
            table3, null, null,
            null, new java.awt.Color(0xFF9900),
            GraphDataLayer.MARKER_TYPE_NONE, 0,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers3 = new ArrayList();
        graphDataLayers3.add(graphDataLayer);

        //graph 4: make a graphDataLayer with data for a time series line
        graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            0, 1, -1, -1, -1, //x,y,  others unused
            GraphDataLayer.DRAW_MARKERS_AND_LINES, 
            true, false, //x,yIsTimeAxis
            "Time", "Y Axis Title", //for now, always std units
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "title4",
            table1, null, null,
            null, new java.awt.Color(0x0099FF),
            GraphDataLayer.MARKER_TYPE_PLUS, GraphDataLayer.MARKER_SIZE_MEDIUM,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers4 = new ArrayList();
        graphDataLayers4.add(graphDataLayer);

        //graph 4: plus 10 random points of each marker type
        for (int i = 1; i <= lastMarker; i++) {
            //make a data file with data
            PrimitiveArray txCol = new DoubleArray();
            PrimitiveArray tyCol = new DoubleArray();
            for (int j = 0; j < 10; j++) {
                txCol.addDouble(1.500e9 + Math.random() * .0025e9);
                tyCol.addDouble(1.5 + Math.random());
            }
            Table ttable = new Table();
            ttable.addColumn("X", txCol);
            ttable.addColumn("Y", tyCol);

            graphDataLayer = new GraphDataLayer(
                -1, //which pointScreen
                0, 1, -1, -1, -1, //x,y,  others unused
                GraphDataLayer.DRAW_MARKERS, 
                true, false,//x,yIsTimeAxis
                "Time", "Y Axis Title", //for now, always std units
                null, //no bold title, so no legend entry
                "title2", "title3", "title4", //will be ignored
                ttable, null, null,
                null, new java.awt.Color(0x0099FF),
                i, GraphDataLayer.MARKER_SIZE_MEDIUM,
                0, //vectorStandard
                GraphDataLayer.REGRESS_NONE);
            graphDataLayers4.add(graphDataLayer);
        }

        //graph 5: make a graphDataLayer with data for a time:time graph
        graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            0, 1, -1, -1, -1, //x,y
            GraphDataLayer.DRAW_MARKERS, 
            true, false,  //x,yIsTimeAxis
            "Time", "Y Axis Title", //for now, always std units
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "title4",
            table5, null, null,
            null, new java.awt.Color(0xFF0000),
            GraphDataLayer.MARKER_TYPE_FILLED_SQUARE, GraphDataLayer.MARKER_SIZE_SMALL,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers5 = new ArrayList();
        graphDataLayers5.add(graphDataLayer);

        //graph 6: make a graphDataLayer with data for a x=data, y=time line
        graphDataLayer = new GraphDataLayer(
            -1, //which pointScreen
            1, 0, -1, -1, -1, //x,y,  others unused         //x and y swapped from graph 1
            GraphDataLayer.DRAW_MARKERS_AND_LINES, false, true,
            "Time", "Y Axis Title", //x,yAxisTitle  for now, always std units 
            "This is the really, really, extra long and informative bold title.",             
            "This is a really, really, extra long and informative title2.", 
            "This is a really, really, extra long and informative title3.",
            "title4",
            table1, null, null,
            null, new java.awt.Color(0x0099FF),
            GraphDataLayer.MARKER_TYPE_PLUS, GraphDataLayer.MARKER_SIZE_SMALL,
            0, //vectorStandard
            GraphDataLayer.REGRESS_MEAN);
        ArrayList graphDataLayers6 = new ArrayList();
        graphDataLayers6.add(graphDataLayer);

        //draw the graph with data
        BufferedImage bufferedImage = SgtUtil.getBufferedImage(width, height);
        Graphics2D g2 = (Graphics2D)bufferedImage.getGraphics();

        //graph 1
        String2.log("Graph 1");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 1,", "x is TimeAxis",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            true, false, //x/yIsTimeAxis,
            graphDataLayers1,
            g2, 0, 0, //upperLeft
            width/2, height/3, 2, //graph width/height
            1); //fontScale
        
        //graph 2
        String2.log("Graph 2");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 2,", "no time, no data.",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            false, false, //x/yIsTimeAxis,
            graphDataLayers2,
            g2, width/2 + 10, 0, //upper Right
            width/2, height/3, 2, //graph width/height
            1.5); //fontScale

        //graph 3
        String2.log("Graph 3");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 3,", "stick graph",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            true, false, //x/yIsTimeAxis,
            graphDataLayers3,
            g2, 0, height/3, //mid Left
            width/2 - 10, height/3, 2, //graph width/height
            1); //fontScale

        //graph 4
        String2.log("Graph 4");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 4,", "x is TimeAxis",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            true, false, //x/yIsTimeAxis,
            graphDataLayers4,
            g2, width/2 + 10, height/3, //mid Right
            width/3, height/3, 2, //graph width/height
            1); //fontScale

        //graph 5
        String2.log("Graph 5");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 5,", "2 time axis! y->not",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            true, true, //x/yIsTimeAxis,
            graphDataLayers5,
            g2, 0, height*2/3, //low left
            width/3, height/3, 2, //graph width/height
            1); //fontScale

        //graph 6
        String2.log("Graph 6");
        sgtGraph.makeGraph("xAxisTitle", "yAxisTitle",
            SgtUtil.LEGEND_BELOW, "Graph 6,", "y is TimeAxis",
            imageDir, "noaa20.gif",
            Double.NaN, Double.NaN, Double.NaN, Double.NaN, //predefined min/maxX/Y
            false, true, //x/yIsTimeAxis,
            graphDataLayers6,
            g2, width/2, height*2/3, //low right
            width/2, height/3, 2, //graph width/height
            1); //fontScale


        //save image
        String fileName = tempDir + "SgtGraphTest" + testImageExtension;
        SgtUtil.saveImage(bufferedImage, fileName);

        //view it
        SSR.displayInBrowser("file://" + fileName);
        Math2.sleep(2000);
        //String2.getStringFromSystemIn("Press ^C to stop or Enter to continue..."); 
        
        //delete files        
        File2.delete(fileName);

        //done
        for (int i = 0; i < 5; i++)
            Math2.gc(100); //all should be garbage collected now
        String2.log("time=" + (System.currentTimeMillis() - time) + " ms\n" +
            Math2.memoryString());
    } 


}
