/* 
 * SgtUtil Copyright 2005, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.coastwatch.sgt;

import com.cohort.array.DoubleArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Image2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

//from itext-1.3.1.jar:
import com.lowagie.text.Document;
//import com.lowagie.text.DocumentException;
//import com.lowagie.text.FontFactory;
import com.lowagie.text.PageSize;
//import com.lowagie.text.pdf.DefaultFontMapper;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfTemplate;
import com.lowagie.text.pdf.PdfWriter;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.util.AttributedString2;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pmel.sgt.*;
import gov.noaa.pmel.sgt.demo.*;
import gov.noaa.pmel.sgt.dm.*;
import gov.noaa.pmel.util.*;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.geom.GeneralPath;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.RenderingHints; 
import java.io.File;
import java.io.*;
import java.text.DecimalFormat;
import java.util.ArrayList;
import javax.imageio.ImageIO;


/**
 * This class has utilities for SgtMap and SgtGraph.
 * A note about coordinates:
 *   <ul>
 *   <li> Graph - uses "user" coordinates (e.g., lat and lon).
 *   <li> Layer - uses "physical" coordinates (doubles, 0,0 at lower left).
 *   <li> JPane - uses "device" coordinates (ints, 0,0 at upper left).
 *   </ul>
 */
public class SgtUtil  {


    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR;

    /**
     * Set this to true (by calling verbose=true in your program, not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false;

    /** For the legend position. */
    public final static int LEGEND_RIGHT = 0;
    public final static int LEGEND_BELOW = 1;

    public final static com.lowagie.text.Rectangle PDF_LANDSCAPE = PageSize.LETTER.rotate();
    public final static com.lowagie.text.Rectangle PDF_PORTRAIT  = PageSize.LETTER;

    public static final double DEFAULT_AXIS_LABEL_HEIGHT = 0.12; 
    public static final double DEFAULT_LABEL_HEIGHT = 0.09; //in the legend    .08 causes problems with 'w' 'm'...

    public static final Color TRANSPARENT = new Color(0, 0, 0, 0); //4th 0 is alpha value   //Hmmm, it may not be this simple

    public static int NARROW_LEGEND_N_CHARS = 38;

    /**
     * This creates a font and throws exception if font family not available
     *
     * @param fontFamily
     * @throws Exception if fontFamily not available
     */
    public static Font getFont(String fontFamily) {
        //minor or major failures return a default font ("Dialog"!)
        Font font = new Font(fontFamily, Font.PLAIN, 10); //Font.ITALIC
        if (!font.getFamily().equals(fontFamily)) 
            Test.error(ERROR + " in SgtUtil.getFont: " + fontFamily + " not available.\n" +
                String2.javaInfo() + "\n" +
                "Fonts available: " + 
                GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames());
        return font;
    }



    /**
     * This draws the standard legend text for a BELOW legend.
     *
     * @param g2
     * @param narrowLegend true if legend must be narrow (6 lines); 
     *    false for standard legend (3 lines)
     * @param legentTextX
     * @param legendTextY
     * @param fontFamily
     * @param labelHeightPixels
     * @param boldTitle must be valid  (if null, nothing will be drawn, and 
     *     return value will be legendTextY unchanged)
     * @param title2 may be null or ""
     * @param title3 may be null or ""
     * @param title4 may be null or ""
     * @return the new legendTextY (adjusted so there is a gap after
     *   the current text).
     */
    public static int belowLegendText(Graphics2D g2, boolean narrowLegend, 
        int legendTextX, int legendTextY,
        String fontFamily, int labelHeightPixels, 
        String boldTitle, String title2, String title3, String title4) {
        
        //String2.log("belowLegendText boldTitle=" + boldTitle);
        if (boldTitle == null) 
            return legendTextY;
        
        //clean up info strings (encodeAsXML, ensure not null)
        if (title2 == null) title2 = "";
        if (title3 == null) title3 = "";
        if (title4 == null) title4 = "";
        boldTitle = encodeAsHtml(boldTitle);
        title2    = encodeAsHtml(title2);
        title3    = encodeAsHtml(title3);
        title4    = encodeAsHtml(title4);

        StringArray stringArray = new StringArray();
        int limit = narrowLegend? NARROW_LEGEND_N_CHARS : 1000;
        
        //gather the strings
        splitLine(limit - 4, stringArray, "<b>", boldTitle, "</b>");  //-4 for bold
        if (title2.length() > 0) splitLine(limit, stringArray, "", title2, "");
        if (title3.length() > 0) splitLine(limit, stringArray, "", title3, "");
        if (title4.length() > 0) splitLine(limit, stringArray, "", title4, "");

        //draw the lines
        int stringArraySize = stringArray.size();
        for (int i = 0; i < stringArraySize; i++)
            legendTextY = drawHtmlText(g2, legendTextX, legendTextY, 
                0, fontFamily, labelHeightPixels, i == stringArraySize - 1, stringArray.get(i));

        return legendTextY;
    }

    /**
     * This is creates "(units) date, title2", and deals with nulls and ""'s.
     *
     * @param units  e.g., m s^-1
     * @param date  e.g., "1998-02-28 14:00:00"
     * @param title2  e.g., "Horizontal line is mean."
     */
    public static String getNewTitle2(String units, String date, String title2) {
        StringBuffer sb = new StringBuffer();
        if (units != null && units.length() > 0)
            sb.append("(" + units + ") ");
        if (date != null && date.length() > 0)
            sb.append(date + " ");
        if (title2 != null)
            sb.append(title2);
        return sb.toString();
    }

    /**
     * If the line is short, this adds the line to StringArray.
     * If the line is long, this splits the line in 2 and
     * adds both to StringArray.
     *
     * @param limit is the maximum number of characters per line
     * @param stringArray to capture the parts of s
     * @param prefix is added to the beginning of each line (e.g., &lt;b&gt;).
     * @param s the string to be split (if needed)
     * @param suffix is added to the end of each line (e.g., &lt;/b&gt;).
     */
    private static void splitLine(int limit, StringArray stringArray, 
        String prefix, String s, String suffix) {
        int sLength = s.length();
        if (sLength <= limit) {
            stringArray.add(prefix + s + suffix);
            return;
        }

        //find the last space before limit
        int po = limit - 1;
        while (po > limit / 2 && s.charAt(po) != ' ')
            po--;

        //fail? find the last non digitLetter before limit
        if (po == limit / 2) {
            po = limit - 1;
            while (po > limit/2 && String2.isDigitLetter(s.charAt(po)))
                po--;
        }

        //fail? just split it
        if (po == limit / 2) 
            po = limit - 1;

        //add the strings
        stringArray.add(prefix + s.substring(0, po + 1) + suffix);
        stringArray.add(prefix + s.substring(po + 1)    + suffix);       
    }



    /**
     * drawHtmlText draws simple HTML text to g2d.
     * drawHtmlText benefits greatly from setting non-text antialising ON:
     * <TT>g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
     *         RenderingHints.VALUE_ANTIALIAS_ON); </TT>
     *
     * @param g2d
     * @param x the base x for the text  (in pixels)
     * @param y the base y for the text  (in pixels)
     * @param hAlign one of SGLabel.LEFT|CENTER|RIGHT (0=left 1=center 2=right).
     * @param fontFamily
     * @param labelHeight (in pixels)
     * @param extraGapBelow adds an extra labelHeight to the returned yAdjusted
     *   (even if htmlText is null or "")
     * @param htmlText  Unless modified by a <color=#ffffff> tag, the text will be black.
     * @return y adjusted to prepare for the text below this text
     */
    public static int drawHtmlText(Graphics2D g2d, int x, int y, 
            int hAlign, String fontFamily, int labelHeight, boolean extraGapBelow, String htmlText) {

        if (htmlText == null || htmlText.length() == 0)
            return y + (extraGapBelow? labelHeight : 0);

//quick fix red affecting whole string?
//htmlText= "<color=#000000> " + htmlText;
        AttributedString2.drawHtmlText(g2d, htmlText, x, y, 
            fontFamily, labelHeight, Color.black, hAlign);
        return y + labelHeight + (extraGapBelow? labelHeight : 0);    
    }


    /**
     * This is a special version of XML.encodeAsXML that  
     * displays any occurence of "EXPERIMENTAL PRODUCT" or "EXPERIMENTAL" in red.
     * @param plainText
     * @return htmlText
     */
    public static String encodeAsHtml(String plainText) {
        if (plainText == null || plainText.length() == 0)
            return "";
        int po = plainText.indexOf("EXPERIMENTAL PRODUCT");
        if (po >= 0)
             return XML.encodeAsXML(plainText.substring(0, po)) + 
                "<color=#ff0000>EXPERIMENTAL PRODUCT</color>" +
                XML.encodeAsXML(plainText.substring(po + 20));

        po = plainText.indexOf("EXPERIMENTAL");
        if (po >= 0)
             return XML.encodeAsXML(plainText.substring(0, po)) + 
                "<color=#ff0000>EXPERIMENTAL</color>" +
                XML.encodeAsXML(plainText.substring(po + 12));

        return XML.encodeAsXML(plainText);
    }

    /**
     * This makes a new bufferedImage suitable for SgtMap.makeMap or SgtGraph.makeGraph.
     * The background is white.
     *
     * @param gifWidth
     * @param gifHeight
     * @return a bufferedImage of the requested size
     * @throws Exception if trouble
     */
    public static BufferedImage getBufferedImage(int gifWidth, int gifHeight) {

        //  Work with BufferedImage requires the following line be added to
        //  beginning of startup.sh:
        //    export JAVA_OPTS=-Djava.awt.headless=true
        BufferedImage bi = new BufferedImage(gifWidth, gifHeight, BufferedImage.TYPE_INT_RGB); 
        Graphics g = bi.getGraphics(); 
        Graphics2D g2 = (Graphics2D)g;
        g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, 
            RenderingHints.VALUE_FRACTIONALMETRICS_ON); 
        g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, 
            RenderingHints.VALUE_TEXT_ANTIALIAS_ON); 
        g2.setColor(Color.white);                //I'm not sure why necessary, but it is
        g2.fillRect(0, 0, gifWidth, gifHeight);  //I'm not sure why necessary, but it is

        return bi;
    }

    /**
     * This reads a image using Java's ImageIO routines.
     *
     * 
     * @param fullName with directory and extension
     * @return a BufferedImage
     * @throws Exception if trouble
     */
    public static BufferedImage readImage(String fullName) throws Exception {
        return ImageIO.read(new File(fullName));
    }

    /**
     * Saves an image as a non-transparent .gif or .png based on the fullImageName's extension.
     * This will overwrite an existing file.
     * Gif's are saved with ImageMagick's convert (which does great color reduction).
     *
     * @param bi
     * @param fullName with directory and extension
     * @throws Exception if trouble
     */
    public static void saveImage(BufferedImage bi, String fullName) throws Exception {
        String shortName = fullName.substring(0, fullName.length() - 4); //currently, all extensions are 4 char
        if (fullName.endsWith(".gif"))
            saveAsGif(bi, shortName);
        else if (fullName.endsWith(".png"))
            saveAsPng(bi, shortName);
        //else if (fullName.endsWith(".jpg"))
        //    saveAsJpg(bi, shortName);
        else Test.error(String2.ERROR + " in SgtUtil.saveImage: " +
            "Unsupported image type for fileName=" + fullName);
    }

    /**
     * Saves an image as a gif.
     * Currently this uses ImageMagick's "convert" (Windows or Linux) because it 
     * does the best job at color reduction (and is fast and is cross-platform).
     * This will overwrite an existing file.
     *
     * @param bi
     * @param fullGifName but without the .gif at the end
     * @throws Exception if trouble
     */
    public static void saveAsGif(BufferedImage bi, String fullGifName) throws Exception {

        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //save as .bmp     (note: doesn't support transparent pixels)
        long time = System.currentTimeMillis(); 
        if (verbose) String2.log("SgtUtil.saveAsGif"); 
        ImageIO.write(bi, "bmp", new File(fullGifName + randomInt + ".bmp"));
        if (verbose) String2.log("  make .bmp done. time=" + 
            (System.currentTimeMillis() - time));

        //"convert" to .gif 
        SSR.dosOrCShell("convert " + 
            fullGifName + randomInt + ".bmp" + " " + 
            fullGifName + randomInt + ".gif", 30);  
        File2.delete(fullGifName + randomInt + ".bmp");

        //try fancy color reduction algorithms
        //Image2.saveAsGif(Image2.reduceTo216Colors(bi), fullGifName + randomInt + ".gif");

        //try dithering 
        //Image2.saveAsGif216(bi, fullGifName + randomInt + ".gif", true);

        //last step: rename to final gif name
        File2.rename(fullGifName + randomInt + ".gif", fullGifName + ".gif");

        if (verbose) String2.log("SgtUtil.saveAsGif done. TOTAL TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * Saves an image as a gif.
     * Currently this uses ImageMagick's "convert" (Windows or Linux) because it 
     * does the best job at color reduction (and is fast and is cross-platform).
     * This will overwrite an existing file.
     *
     * @param bi
     * @param transparent the color to be made transparent
     * @param fullGifName but without the .gif at the end
     * @throws Exception if trouble
     */
    public static void saveAsTransparentGif(BufferedImage bi, Color transparent, 
        String fullGifName) throws Exception {

        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //convert transparent color to be transparent
        long time = System.currentTimeMillis();
        Image image = Image2.makeImageBackgroundTransparent(bi, 
           transparent, 10000);

        //convert image back to bufferedImage
        bi = new BufferedImage(bi.getWidth(), bi.getHeight(), BufferedImage.TYPE_INT_ARGB);
        Graphics g = bi.getGraphics();
        g.drawImage(image, 0, 0, bi.getWidth(), bi.getHeight(), null);
        image = null; //encourage garbage collection
        Math2.gc(100);
        
        //save as png
        int random = Math2.random(Integer.MAX_VALUE);
        ImageIO.write(bi, "png", new File(fullGifName + randomInt + ".png"));

        //"convert" to .gif 
        SSR.dosOrCShell("convert " + 
            fullGifName + randomInt + ".png" + " " + 
            fullGifName + randomInt + ".gif", 30);  
        File2.delete(fullGifName + randomInt + ".png");

        //try fancy color reduction algorithms
        //Image2.saveAsGif(Image2.reduceTo216Colors(bi), fullGifName + randomInt + ".gif");

        //try dithering 
        //Image2.saveAsGif216(bi, fullGifName + randomInt + ".gif", true);

        //last step: rename to final gif name
        File2.rename(fullGifName + randomInt + ".gif", fullGifName + ".gif");

        if (verbose) String2.log("SgtUtil.saveAsTransparentGif TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * Saves an image as a png.
     * This will overwrite an existing file.
     *
     * @param bi
     * @param fullPngName but without the .png at the end
     * @throws Exception if trouble
     */
    public static void saveAsPng(BufferedImage bi, 
        String fullPngName) throws Exception {
        saveAsTransparentPng(bi, null, fullPngName);
    }


    /**
     * Saves an image as a png.
     * This will overwrite an existing file.
     *
     * @param bi
     * @param transparent the color to be made transparent  (or null if none)
     * @param fullPngName but without the .png at the end
     * @throws Exception if trouble
     */
    public static void saveAsTransparentPng(BufferedImage bi, Color transparent, 
        String fullPngName) throws Exception {

        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //create fileOutputStream
        BufferedOutputStream bos = new BufferedOutputStream(
            new FileOutputStream(fullPngName + randomInt + ".png"));

        //save the image
        saveAsTransparentPng(bi, transparent, bos);
        bos.close();

        //last step: rename to final Png name
        File2.rename(fullPngName + randomInt + ".png", fullPngName + ".png");

    }

    /**
     * Saves an image as a png.
     * This will overwrite an existing file.
     *
     * @param bi
     * @param outputStream
     * @throws Exception if trouble
     */
    public static void saveAsPng(BufferedImage bi, OutputStream outputStream) 
        throws Exception {
        saveAsTransparentPng(bi, null, outputStream);
    }

    /**
     * Saves an image as a png.
     * This will overwrite an existing file.
     *
     * @param bi
     * @param transparent the color to be made transparent  (or null if none)
     * @param outputStream  (it is flushed at the end)
     * @throws Exception if trouble
     */
    public static void saveAsTransparentPng(BufferedImage bi, Color transparent, 
        OutputStream outputStream) throws Exception {

        //convert transparent color to be transparent
        long time = System.currentTimeMillis();
        if (transparent != null) {
            Image image = Image2.makeImageBackgroundTransparent(bi, 
                transparent, 10000);

            //convert image back to bufferedImage
            bi = new BufferedImage(bi.getWidth(), bi.getHeight(), BufferedImage.TYPE_INT_ARGB);
            Graphics g = bi.getGraphics();
            g.drawImage(image, 0, 0, bi.getWidth(), bi.getHeight(), null);
        }
        
        //save as png
        ImageIO.write(bi, "png", outputStream);
        outputStream.flush();

        if (verbose) String2.log("SgtUtil.saveAsPng TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * This creates a file to capture the pdf output generated by calls to 
     * graphics2D (e.g., use makeMap).
     * This will overwrite an existing file.
     * 
     * @param pageSize e.g, PageSize.LETTER or PageSize.LETTER.rotate() (or A4, or, ...)
     * @param width the bounding box width, in 1/144ths of an inch
     * @param height the bounding box height, in 1/144ths of an inch
     * @param fullFileName (with the extension .pdf)
     * @return an object[] with 0=g2D, 1=document, 2=pdfContentByte, 3=pdfTemplate
     * @throws Exception if trouble
     */
    public static Object[] createPdf(com.lowagie.text.Rectangle pageSize,
            int bbWidth, int bbHeight, String fullFileName) throws Exception {
        return createPdf(pageSize, bbWidth, bbHeight, new FileOutputStream(fullFileName));
    }

    /**
     * This creates a file to capture the pdf output generated by calls to 
     * graphics2D (e.g., use makeMap).
     * This will overwrite an existing file.
     * 
     * @param pageSize e.g, PageSize.LETTER or PageSize.LETTER.rotate() (or A4, or, ...)
     * @param width the bounding box width, in 1/144ths of an inch
     * @param height the bounding box height, in 1/144ths of an inch
     * @param outputStream
     * @return an object[] with 0=g2D, 1=document, 2=pdfContentByte, 3=pdfTemplate
     * @throws Exception if trouble
     */
    public static Object[] createPdf(com.lowagie.text.Rectangle pageSize,
            int bbWidth, int bbHeight, OutputStream outputStream) throws Exception {
        //currently, this uses itext
        //see the sample program:
        //  file://localhost/C:/programs/iText/examples/com/lowagie/examples/directcontent/graphics2D/G2D.java
        //Document.compress = false; //for test purposes only
        Document document = new Document(pageSize);
        document.addCreationDate();
        document.addCreator("gov.noaa.pfel.coastwatch.SgtUtil.createPdf");

        document.setPageSize(pageSize);
        PdfWriter writer = PdfWriter.getInstance(document, outputStream);
        document.open();
        
        //create contentByte and template and Graphics2D objects
        PdfContentByte pdfContentByte = writer.getDirectContent();
        PdfTemplate pdfTemplate = pdfContentByte.createTemplate(bbWidth, bbHeight);
        Graphics2D g2D = pdfTemplate.createGraphics(bbWidth, bbHeight); 

        return new Object[]{g2D, document, pdfContentByte, pdfTemplate};
    }

    /**
     * This closes the pdf file created by createPDF, after you have
     * written things to g2D.
     *
     * @param oar the object[] returned from createPdf
     * @throwsException if trouble
     */
    public static void closePdf(Object oar[]) throws Exception {
        Graphics2D g2D = (Graphics2D)oar[0];
        Document document = (Document)oar[1];
        PdfContentByte pdfContentByte = (PdfContentByte)oar[2];
        PdfTemplate pdfTemplate = (PdfTemplate)oar[3];

        g2D.dispose();

        //center it
        if (verbose) String2.log("SgtUtil.closePdf" +
            " left="   + document.left()   + " right=" + document.right() + 
            " bottom=" + document.bottom() + " top="   + document.top()   + 
            " template.width="  + pdfTemplate.getWidth() + " template.height=" + pdfTemplate.getHeight());
        //x device = ax user + by user + e
        //y device = cx user + dy user + f
        pdfContentByte.addTemplate(pdfTemplate, //a,b,c,d,e,f      //x,y location in points 
            0.5f, 0, 0, 0.5f, 
            document.left()   + (document.right() - document.left()   - pdfTemplate.getWidth()/2)  / 2, 
            document.bottom() + (document.top()   - document.bottom() - pdfTemplate.getHeight()/2) / 2);

        /*
        //if boundingBox is small, center it
        //if boundingBox is large, shrink and center it
        //document.left/right/top/bottom include 1/2" margins
        float xScale = (document.right() - document.left())   / pdfTemplate.getWidth();   
        float yScale = (document.top()   - document.bottom()) / pdfTemplate.getHeight();  
        float scale = Math.min(Math.min(xScale, yScale), 1);
        float xSize = pdfTemplate.getWidth()  / scale;
        float ySize = pdfTemplate.getHeight() / scale;
        //x device = ax user + by user + e
        //y device = cx user + dy user + f
        pdfContentByte.addTemplate(pdfTemplate, //a,b,c,d,e,f
            scale, 0, 0, scale, 
            document.left()   + (document.right() - document.left()   - xSize) / 2,
            document.bottom() + (document.top()   - document.bottom() - ySize) / 2); 
        */

        document.close();
    }

    /**
     * This returns a whiter color than c.
     *
     * @param color
     * @return a whiter color than c
     */
    public static Color whiter(Color color) {
        int r = color.getRed();
        int g = color.getGreen();
        int b = color.getBlue();
        return new Color(
            r + (255 - r) / 4,    //little changes close to 255 have big effect
            g + (255 - g) / 4, 
            b + (255 - b) / 4);
    }

    /**
     * This returns a blacker color than c.
     *
     * @param color
     * @return a blacker color than c
     */
    public static Color blacker(Color color) {
        int r = color.getRed();
        int g = color.getGreen();
        int b = color.getBlue();
        return new Color(
            Math.max(0, r - (255 - r) / 4), //little changes close to 255 have big effect
            Math.max(0, g - (255 - g) / 4), 
            Math.max(0, b - (255 - b) / 4));
    }

    /** 
     * The default palette (aka color bar) range ([0]=min, [1]=max). 
     * The values are also suitable for the axis range on a graph.
     * 
     * @param dataMin the raw minimum value of the data
     * @param dataMax the raw maximum value of the data
     * @return the default palette (aka color bar) range ([0]=min, [1]=max).
     */
    public static double[] suggestPaletteRange(double dataMin, double dataMax) {

        double lowHigh[] = Math2.suggestLowHigh(dataMin, dataMax);

        //log axis?
        if (suggestPaletteScale(dataMin, dataMax).equals("Log")) { //yes, use dataMin,dataMax,  not lowHigh
            lowHigh[0] = Math2.suggestLowHigh(dataMin, 2*dataMin)[0]; //trick to get nice suggested min>0
            return lowHigh;
        }

        //axis is linear
        //suggest symmetric around 0 (symbolized by BlueWhiteRed)?
        if (suggestPalette(dataMin, dataMax).equals("BlueWhiteRed")) { //yes, use dataMin,dataMax,  not lowHigh
            double rangeMax = Math.max(-lowHigh[0], lowHigh[1]);
            lowHigh[0] = -rangeMax;
            lowHigh[1] = rangeMax;
        }

        //standard Rainbow Linear
        return lowHigh;
    }

    /** 
     * The name of the suggested palette (aka color bar), e.g., Rainbow or BlueWhiteRed. 
     * Must be one of the palettes available to PointDataSets in the browser.
     *
     * @param min the raw minimum value of the data (preferred) or the refined minimum value for the palette
     * @param max the raw maximum value of the data (preferred) or the refined maximum value for the palette
     * @return the name of the suggested palette (aka color bar), e.g., Rainbow. 
     *    "BlueWhiteRed" is suggested if the palette should be centered on 0.
     */
    public static String suggestPalette(double min, double max) {
        if (min < 0 && max > 0 &&
            -min/max >= .5 &&
            -min/max <= 2)
            return "BlueWhiteRed";
        if (min >= 0 && min < max / 5)
            return "WhiteRedBlack";
        return "Rainbow";
    } 

    /** 
     * The name of the suggested palette scale, e.g., Linear or Log. 
     *
     * @param min the raw minimum value of the data (preferred) or the refined minimum value for the palette
     * @param max the raw maximum value of the data (preferred) or the refined maximum value for the palette
     * @return the name of the suggested palette (aka color bar) scale, e.g., Linear or Log. 
     */
    public static String suggestPaletteScale(double min, double max) {
        if (min > 0 && min < 1 && 
            max / min > 100) 
            return "Log";
        return "Linear"; 
    }

    /** This tests SgtUtil. */ 
    public static void test() throws Exception {
        //test splitLine
        String2.log("\n*** SgtUtil.test");
        StringArray sa = new StringArray();

        //wide  
        sa.clear();
        splitLine(1000, sa, "<b>", "This is a test of splitline.", "</b>");
        Test.ensureEqual(sa.size(), 1, "");
        Test.ensureEqual(sa.get(0), "<b>This is a test of splitline.</b>", "");

        //narrow 
        sa.clear();
        splitLine(12, sa, "<b>", "This is a test of splitline.", "</b>");
        Test.ensureEqual(sa.size(), 2, "");
        Test.ensureEqual(sa.get(0), "<b>This is a </b>", "");
        Test.ensureEqual(sa.get(1), "<b>test of splitline.</b>", "");

        //narrow and can't split, so chop at limit
        sa.clear();
        splitLine(12, sa, "<b>", "This1is2a3test4of5splitline.", "</b>");
        Test.ensureEqual(sa.size(), 2, "");
        Test.ensureEqual(sa.get(0), "<b>This1is2a3te</b>", "");
        Test.ensureEqual(sa.get(1), "<b>st4of5splitline.</b>", "");

        //test suggestPaletteRange
        Test.ensureEqual(suggestPaletteRange(.3, 8.9), new double[]{0, 10}, ""); //typical Rainbow Linear
        Test.ensureEqual(suggestPaletteRange(.11, 890), new double[]{.1, 1000}, ""); //typical Rainbow Log
        Test.ensureEqual(suggestPaletteRange(-7, 8), new double[]{-10, 10}, ""); //typical BlueWhiteRed Linear symmetric

        //test suggestPalette
        Test.ensureEqual(suggestPalette(.3, 8.9), "WhiteRedBlack", ""); //small positive, large positive
        Test.ensureEqual(suggestPalette(300, 890), "Rainbow", ""); //typical Rainbow Log
        Test.ensureEqual(suggestPalette(-7, 8), "BlueWhiteRed", ""); //typical BlueWhiteRed Linear symmetric

        //test suggestPaletteScale
        Test.ensureEqual(suggestPaletteScale(.3, 8.9), "Linear", ""); //typical Rainbow Linear
        Test.ensureEqual(suggestPaletteScale(.11, 890), "Log", ""); //typical Rainbow Log
        Test.ensureEqual(suggestPaletteScale(-7, 8), "Linear", ""); //typical BlueWhiteRed Linear symmetric
    } 


//*** Junk Yard *******
            //create the colorbar for the legend
            /*ColorKey colorKey = new ColorKey(new Point2D.Double(4.5, 3), //location 
                new Dimension2D(0.25, 2.5), //size
                ColorKey.TOP, ColorKey.LEFT); //valign, halign
            colorKey.setOrientation(ColorKey.VERTICAL);
            colorKey.setBorderStyle(ColorKey.NO_BORDER);
            colorKey.setColorMap(colorMap);
            Ruler ruler = colorKey.getRuler();
            ruler.setLabelFont(labelFont);
            ruler.setLabelHeightP(0.15);
            ruler.setLabelInterval(2); //temp
            ruler.setLargeTicHeightP(0.04);
            ruler.setRangeU(colorMap.getRange());
            String2.log("colorMap start=" + colorMap.getRange().start + " end=" +
                colorMap.getRange().end + " delta=" + colorMap.getRange().delta);
            layer.addChild(colorKey);
            */

}
