/*
 * ERDDAP Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap;

import com.cohort.array.Attributes;
import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.CompoundColorMap;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.sgt.SgtUtil;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.dataset.*;
import gov.noaa.pfel.erddap.util.*;
import gov.noaa.pfel.erddap.variable.EDV;
import gov.noaa.pfel.erddap.variable.EDVGridAxis;
import gov.noaa.pfel.erddap.variable.EDVTimeGridAxis;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.verisign.joid.consumer.OpenIdFilter;

/**
 * ERDDAP is NOAA NMFS SWFSC ERD's Data Access Program, 
 * a Java servlet which serves gridded and tabular data
 * in common data file formats (e.g., ASCII, .dods, .mat, .nc)
 * and image file formats (e.g., .pdf and .png). 
 *
 * <p> This works like an OPeNDAP DAP-style server conforming to the
 * DAP 2.0 spec (see the Documentation section at www.opendap.org). 
 *
 * <p>The authentication method is set by the authentication tag in setup.xml.
 * See its use below and in EDStatic.
 *
 * <p>Authorization is specified by roles tags and accessibleTo tags in datasets.xml.
 * <br>If a user isn't authorized to use a dataset, then EDStatic.listPrivateDatasets 
 *    determines whether the dataset appears on lists of datasets (e.g., categorize or search).
 * <br>If a user isn't authorized to use a dataset and requests info about that
 *    dataset, EDStatic.redirectToLogin is called.
 * <br>These policies are enforced by checking edd.isAccessibleTo results from 
 *    gridDatasetHashMap and tableDatasetHashMap
 *    (notably also via gridDatasetIDs, tableDatasetIDs, allDatasetIDs).
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-20
 */
public class Erddap extends HttpServlet {

    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR; 

    /**
     * Set this to true (by calling verbose=true in your program, 
     * not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false; 

    /**
     * Set this to true (by calling reallyVerbose=true in your program, 
     * not but changing the code here)
     * if you want lots and lots of diagnostic messages sent to String2.log.
     */
    public static boolean reallyVerbose = false; 

    /** This identifies the dods server/version that this mimics. */
    public static String dapVersion = "DAP/2.0";   //???
    public static String serverVersion = "dods/3.7"; //this is what thredds replies
      //drds at http://oceanwatch.pfeg.noaa.gov/opendap/GLOBEC/GLOBEC_bottle.ver replies "DODS/3.2"
      //both reply with server version, neither replies with coreVersion
      //spec says #.#.#, but Gallagher says #.# is fine.

    /** The programmatic/computer access to Erddap services are available as 
     * all of the plainFileTypes. 
     * All plainFileTypes must be valid EDDTable.dataFileTypeNames.
     * If added a new type, also add to sendPlainTable below.
     */
    public static String plainFileTypes[] = {".csv", ".htmlTable", ".json", ".mat", ".tsv", ".xhtml"};

    public final static int WMS_MAX_LAYERS = 16; //arbitrary
    public final static int WMS_MAX_WIDTH = 2048; //arbitrary
    public final static int WMS_MAX_HEIGHT = 2048; //arbitrary
    public final static char WMS_SEPARATOR = ':'; //separates datasetID and variable name (not a valid interior char)

    // ************** END OF STATIC VARIABLES *****************************

    protected RunLoadDatasets runLoadDatasets;
    public int todaysNRequests, totalNRequests;
    public String lastReportDate = "";

    /** Set by loadDatasets. */
    /** datasetHashMaps are read from many threads and written to by loadDatasets, 
     * so need to synchronize these maps.
     * grid/tableDatasetHashMap are key=datasetID value=edd.
     */
    public Map gridDatasetHashMap = Collections.synchronizedMap(new HashMap());
    public Map tableDatasetHashMap = Collections.synchronizedMap(new HashMap());
    /** The RSS info: key=datasetId, value=utf8 byte[] of rss xml */
    public Map rssHashMap = Collections.synchronizedMap(new HashMap()); 
    public Map categoryInfo = Collections.synchronizedMap(new HashMap()); 
    public Map failedLogins = Collections.synchronizedMap(new HashMap()); 
    public long lastClearedFailedLogins = System.currentTimeMillis();


    /** Used by slideSorter. */
    protected Map slideInfo = Collections.synchronizedMap(new HashMap()); 

    /**
     * The constructor.
     *
     * <p> This needs to find the content/erddap directory.
     * It may be a defined environment variable ("erddapContent"),
     * but is usually a subdir of <tomcat> (e.g., usr/local/tomcat/content/erddap/).
     *
     * <p>This redirects logging messages to the log.txt file in bigParentDirectory 
     * (specified in <tomcat>/content/erddap/setup.xml) or to a CommonsLogging file.
     * This is appropriate for use as a web service. 
     * If you are using Erddap within a Java program and you want
     * to redirect diagnostic and error messages back to System.out, use
     * String2.setupLog(true, false, "", false, false, 1000000);
     * after calling this.
     *
     * @throws Throwable if trouble
     */
    public Erddap() throws Throwable {
        String2.log("\n\\\\\\\\**** Start Erddap constructor");
        long constructorMillis = System.currentTimeMillis();

        //make new catInfo with first level hashMaps
        int nCat = EDStatic.categoryAttributes.length;
        for (int cat = 0; cat < nCat; cat++) 
            categoryInfo.put(EDStatic.categoryAttributes[cat], new HashMap());

        //start RunLoadDatasets
        runLoadDatasets = new RunLoadDatasets(this);
        EDStatic.runningThreads.put("runLoadDatasets", runLoadDatasets); 
        runLoadDatasets.start(); 

        //done
        String2.log("\n\\\\\\\\**** Erddap constructor finished. TIME=" +
            (System.currentTimeMillis() - constructorMillis));
    }

    /**
     * destroy() is called by Tomcat whenever the servlet is removed from service.
     * See example at http://classes.eclab.byu.edu/462/demos/PrimeSearcher.java
     *
     * <p> Erddap doesn't overwrite HttpServlet.init(servletConfig), but it could if need be. 
     * runLoadDatasets is created by the Erddap constructor.
     */
    public void destroy() {
        EDStatic.destroy();
    }

    /**
     * This returns a StringArray with all the datasetIDs for all of the grid datasets.
     *
     * @param sorted if true, the resulting StringArray is sorted (ignoring case)
     * @return a StringArray with all the datasetIDs for all of the grid datasets.
     */
    public StringArray gridDatasetIDs(boolean sorted) {
        StringArray sa  = new StringArray(gridDatasetHashMap.keySet().iterator());
        if (sorted) sa.sortIgnoreCase();
        return sa;
    }
    
    /**
     * This returns a StringArray with all the datasetIDs for all of the table datasets.
     *
     * @param sorted if true, the resulting StringArray is sorted (ignoring case)
     * @return a StringArray with all the datasetIDs for all of the table datasets.
     */
    public StringArray tableDatasetIDs(boolean sorted) {
        StringArray sa  = new StringArray(tableDatasetHashMap.keySet().iterator());
        if (sorted) sa.sortIgnoreCase();
        return sa;
    }
    
    /**
     * This returns a StringArray with all the datasetIDs for all of the datasets.
     *
     * @param sorted if true, the resulting StringArray is sorted (ignoring case)
     * @return a StringArray with all the datasetIDs for all of the datasets.
     */
    public StringArray allDatasetIDs(boolean sorted) {
        StringArray sa  = new StringArray(gridDatasetHashMap.keySet().iterator());
        sa.append(new StringArray(tableDatasetHashMap.keySet().iterator()));
        if (sorted) sa.sortIgnoreCase();
        return sa;
    }

    /**
     * This returns the category values (sorted) for a given category attribute.
     * 
     * @param attribute e.g., "institution"
     * @return the category values for a given category attribute (or empty StringArray if none).
     */
    public StringArray categoryInfo(String attribute) {
        HashMap hm = (HashMap)categoryInfo.get(attribute);
        if (hm == null)
            return new StringArray();
        StringArray sa  = new StringArray(hm.keySet().iterator());
        sa.sortIgnoreCase();
        return sa;
    }
    
    /**
     * This returns the datasetIDs (sorted) for a given category value for a given category attribute.
     * 
     * @param attribute e.g., "institution"
     * @param value e.g., "NOAA_NDBC"
     * @return the datasetIDs for a given category value for a given category 
     *    attribute (or empty StringArray if none).
     */
    public StringArray categoryInfo(String attribute, String value) {
        HashMap hm = (HashMap)categoryInfo.get(attribute);
        if (hm == null)
            return new StringArray();
        HashSet hs = (HashSet)hm.get(value);
        if (hs == null)
            return new StringArray();
        StringArray sa  = new StringArray(hs.iterator());
        sa.sortIgnoreCase();
        return sa;
    }

    /**
     * This responds to a "post" request from the user by extending HttpServlet's doPost
     * and passing the request to doGet.
     *
     * @param request 
     * @param response
     */
    public void doPost(HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        doGet(request, response);
    }

    /** 
     * This responds to a "get" request from the user by extending HttpServlet's doGet.
     * Mostly, this just identifies the protocol (e.g., "tabledap") in the requestUrl
     * (right after the warName) and calls doGet&lt;Protocol&gt; to handle
     * the request. That allows Erddap to work like a DAP server, or a WCS server,
     * or a ....
     *
     * @param request
     * @param response
     * @throws ServletException, IOException
     */
    public void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException {

        long doGetTime = System.currentTimeMillis();
        todaysNRequests++;
        int requestNumber = totalNRequests++;

        try {

            //get loggedInAs
            String loggedInAs = EDStatic.getLoggedInAs(request);
            EDStatic.tally.add("Requester Is Logged In (since startup)", "" + (loggedInAs != null));
            EDStatic.tally.add("Requester Is Logged In (last 24 hours)", "" + (loggedInAs != null));

            //String2.log("requestURL=" + request.getRequestURL());
            String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
            String requestUrl = request.getRequestURI();  //post EDD.baseUrl, pre "?"

            //get requester's ip addresses (x-forwarded-for)
            //getRemoteHost(); returns our proxy server (never changes)
            //For privacy reasons, don't tally full individual IP address; the 4th ip number is removed.
            String ipAddress = request.getHeader("x-forwarded-for");  
            if (ipAddress == null) {
                ipAddress = "";
            } else {
                //if csv, get last part
                //see http://en.wikipedia.org/wiki/X-Forwarded-For
                int cPo = ipAddress.lastIndexOf(',');
                if (cPo >= 0)
                    ipAddress = ipAddress.substring(cPo + 1);
            }
            ipAddress = ipAddress.trim();
            if (ipAddress.length() == 0)
                ipAddress = "(unknownIPAddress)";

            //get userQuery
            String userQuery = request.getQueryString(); //may be null;  leave encoded
            if (userQuery == null)
                userQuery = "";
            String2.log("{{{{#" + requestNumber + " " +
                Calendar2.getCurrentISODateTimeStringLocal() + " " + 
                (loggedInAs == null? "(notLoggedIn)" : loggedInAs) + " " +
                ipAddress + " " +
                requestUrl + (userQuery.length() == 0? "" : "?" + userQuery));

            //refuse request? e.g., to fend of a Denial of Service attack or an overzealous web robot
            int periodPo = ipAddress.lastIndexOf('.'); //to make #.#.#.* test below
            if (EDStatic.requestBlacklist != null &&
                (EDStatic.requestBlacklist.contains(ipAddress) ||
                 (periodPo >= 0 && EDStatic.requestBlacklist.contains(ipAddress.substring(0, periodPo+1) + "*")))) {
                //use full ipAddress, to help id user                //odd capitilization sorts better
                EDStatic.tally.add("Requester's IP Address (Blocked) (since last Major LoadDatasets)", ipAddress);
                EDStatic.tally.add("Requester's IP Address (Blocked) (since last daily report)", ipAddress);
                EDStatic.tally.add("Requester's IP Address (Blocked) (since startup)", ipAddress);
                String2.log("Requester is on the datasets.xml requestBlacklist.");
                response.sendError(HttpServletResponse.SC_FORBIDDEN); //a.k.a. Error 403
                return;
            }

            //tally ipAddress                                    //odd capitilization sorts better
            EDStatic.tally.add("Requester's IP Address (Allowed) (since last Major LoadDatasets)", ipAddress);
            EDStatic.tally.add("Requester's IP Address (Allowed) (since last daily report)", ipAddress);
            EDStatic.tally.add("Requester's IP Address (Allowed) (since startup)", ipAddress);

            //requestUrl should start with /erddap/
            //deal with /erddap
            //??? '\' on windows computers??? or '/' since it isn't a real directory?
            if (!requestUrl.startsWith("/" + EDStatic.warName + "/")) {
                response.sendRedirect(tErddapUrl + "/index.html");
                return;
            }
            int protocolStart = EDStatic.warName.length() + 2;            

            //get protocol (e.g., "griddap" or "tabledap")
            int protocolEnd = requestUrl.indexOf("/", protocolStart);
            if (protocolEnd < 0)
                protocolEnd = requestUrl.length();
            String protocol = requestUrl.substring(protocolStart, protocolEnd);
            String endOfRequest = requestUrl.substring(protocolStart);
            if (reallyVerbose) String2.log("  protocol=" + protocol);

            //Pass the query to the requested protocol or web page.
            //Be as restrictive as possible (so resourceNotFound can be caught below, if possible).
            if (protocol.equals("griddap") ||
                protocol.equals("tabledap")) {
                doDap(request, response, loggedInAs, protocol, protocolEnd + 1, userQuery);
            /*} else if (protocol.equals("sos")) {
                doSos(request, response, protocolEnd + 1); */
            } else if (protocol.equals("wms")) {
                doWms(request, response, loggedInAs, protocolEnd + 1, userQuery);
            } else if (endOfRequest.equals("") || endOfRequest.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/index.html");
            } else if (protocol.startsWith("index.")) {
                doIndex(request, response, loggedInAs);
            } else if (protocol.equals("download") ||
                       protocol.equals("images") ||
                       protocol.equals("public")) {
                doTransfer(request, response, protocol, protocolEnd + 1);
            } else if (protocol.equals("rss")) {
                doRss(request, response, protocol, protocolEnd + 1);
            } else if (protocol.equals("search")) {
                doSearch(request, response, loggedInAs, protocol, protocolEnd + 1, userQuery);
            } else if (protocol.equals("categorize")) {
                doCategorize(request, response, loggedInAs, protocol, protocolEnd + 1, userQuery);
            } else if (protocol.equals("info")) {
                doInfo(request, response, loggedInAs, protocol, protocolEnd + 1);
            //} else if (protocol.equals("generateDatasetsXml")) {
            //    doGenerateDatasetsXml(request, response, loggedInAs, 
            //        protocol, protocolEnd + 1, userQuery);
            } else if (endOfRequest.equals("information.html")) {
                doInformationHtml(request, response, loggedInAs);
            } else if (endOfRequest.equals("login.html")) {
                doLogin(request, response, loggedInAs);
            } else if (endOfRequest.equals("logout.html")) {
                doLogout(request, response, loggedInAs);
            } else if (endOfRequest.equals("rest.html")) {
                doRestHtml(request, response, loggedInAs);
            } else if (endOfRequest.equals("setDatasetFlag.txt")) {
                doSetDatasetFlag(request, response, userQuery);
            } else if (endOfRequest.equals("sitemap.xml")) {
                doSitemap(request, response);
            } else if (endOfRequest.equals("slidesorter.html")) {
                doSlideSorter(request, response, loggedInAs, userQuery);
            } else if (endOfRequest.equals("status.html")) {
                doStatus(request, response, loggedInAs);
            } else if (protocol.equals("subscriptions")) {
                doSubscriptions(request, response, loggedInAs, ipAddress, endOfRequest, 
                    protocol, protocolEnd + 1, userQuery);
            } else {
                sendResourceNotFoundError(request, response, "");
            }
            
            //tally
            EDStatic.tally.add("Protocol (since startup)", protocol);
            EDStatic.tally.add("Protocol (last 24 hours)", protocol);

            //clear the String2.logStringBuffer
            StringBuffer logSB = String2.getLogStringBuffer();
            if (logSB != null) 
                logSB.setLength(0); 

            long responseTime = System.currentTimeMillis() - doGetTime;
            String2.distribute(responseTime, EDStatic.responseTimesDistributionLoadDatasets);
            String2.distribute(responseTime, EDStatic.responseTimesDistribution24);
            String2.distribute(responseTime, EDStatic.responseTimesDistributionTotal);
            if (verbose) String2.log("}}}}#" + requestNumber + " SUCCESS. TIME=" + responseTime + "\n");

        } catch (Throwable t) {

            try {
                String message = MustBe.throwableToString(t);
                
                //Don't email common, unimportant exceptions   e.g., ClientAbortException
                //Are there others I don't need to see?
                if (message.indexOf("ClientAbortException") >= 0) {
                    String2.log("#" + requestNumber + " Error: ClientAbortException");

                } else if (message.indexOf(DataHelper.THERE_IS_NO_DATA) >= 0) {
                    String2.log("#" + requestNumber + " " + message);

                } else {
                    String q = request.getQueryString(); //not decoded
                    message = "#" + requestNumber + " Error for url=" + request.getRequestURI() +
                        (q == null || q.length() == 0? "" : "?" + q) + 
                        "\nerror=" + message;
                    String2.log(message);
                    if (reallyVerbose) //should this be just 'verbose', or a separate setting?
                        EDStatic.email(EDStatic.emailEverythingTo, 
                            String2.ERROR, 
                            message);
                }

                //"failure" includes clientAbort and there is no data
                long responseTime = System.currentTimeMillis() - doGetTime;
                String2.distribute(responseTime, EDStatic.failureTimesDistributionLoadDatasets);
                String2.distribute(responseTime, EDStatic.failureTimesDistribution24);
                String2.distribute(responseTime, EDStatic.failureTimesDistributionTotal);
                if (verbose) String2.log("}}}}#" + requestNumber + " FAILURE. TIME=" + responseTime + "\n");

                //???if "ClientAbortException", should ERDDAP not send error code below?
            } catch (Throwable t2) {
                String2.log("Error while handling error:\n" + MustBe.throwableToString(t2));
            }

            //if sendErrorCode fails because response.isCommitted(), it throws ServletException
            sendErrorCode(request, response, t); 
        }

    }

    /** 
     * This responds to an /erddap/index.xxx request
     *
     * @param request
     * @param response
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @throws ServletException, IOException
     */
    public void doIndex(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDD.baseUrl, pre "?"

        //plain file types  
        for (int pft = 0; pft < plainFileTypes.length; pft++) { 

            //index.pft  - return a list of resources
            if (requestUrl.equals("/" + EDStatic.warName + "/index" + plainFileTypes[pft])) {

                String fileTypeName = File2.getExtension(requestUrl);
                EDStatic.tally.add("Main Resources List (since startup)", fileTypeName);
                EDStatic.tally.add("Main Resources List (last 24 hours)", fileTypeName);
                Table table = new Table();
                StringArray resourceCol = new StringArray();
                StringArray urlCol = new StringArray();
                table.addColumn("Resource", resourceCol);
                table.addColumn("URL", urlCol);
                String resources[] = {"info", "search", "categorize", "griddap", "tabledap", "wms"}; 
                for (int r = 0; r < resources.length; r++) {
                    resourceCol.add(resources[r]);
                    urlCol.add(tErddapUrl + "/" + resources[r] + "/index" + fileTypeName +
                        (resources[r].equals("search")? "?searchFor=" : ""));
                }
                sendPlainTable(request, response, table, "Resources", fileTypeName);
                return;
            }
        }

        //only thing left should be erddap/index.html request
        if (!requestUrl.equals("/" + EDStatic.warName + "/index.html")) {
            sendResourceNotFoundError(request, response, "");
            return;
        }

        //display main erddap index.html page 
        EDStatic.tally.add("Home Page (since startup)", ".html");
        EDStatic.tally.add("Home Page (last 24 hours)", ".html");
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Home Page", out); 
        try {
            //set up the table
            String tdString = " align=\"left\" valign=\"top\">\n";
            writer.write("<table width=\"100%\" border=\"0\" cellspacing=\"12\" cellpadding=\"0\">\n" +
                "<tr>\n<td width=\"60%\"" + tdString);


            //*** left column: theShortDescription
            writer.write(EDStatic.theShortDescriptionHtml(tErddapUrl));

            //thin vertical line between text columns
            writer.write(
                "</td>\n" + 
                "<td width=\"1\" bgcolor=\"#bbbbbb\"><br></td>\n" + //thin vertical line
                "<td" + tdString); //unspecified width will be the remainder

            //*** the right column: Get Started with ERDDAP
            writer.write(
                "<h2><a name=\"GetStarted\">" + EDStatic.getStartedHtml + "</a></h2>\n" +
                EDStatic.getStarted2Html + "\n" +
                "<ul>");

            //display /info link with list of all datasets
            writer.write(
                "\n<li><h3><a href=\"" + tErddapUrl + "/info/index.html\">" +
                EDStatic.viewAllDatasetsHtml + "</a></h3>\n");

            //display a search form
            writer.write("\n<li>");
            writeSearchFormHtml(tErddapUrl, writer, 3, "");

            //display categorize options
            writer.write("\n<li>");
            writeCategorizeOptionsHtml1(tErddapUrl, writer, true);

            //display protocol links
            String protSee = " title=\"" + EDStatic.protocolClick + " ";
            writer.write(
                "\n<li>" +
                "<h3>" + EDStatic.protocolSearchHtml + "</h3>\n" +
                EDStatic.protocolSearch2Html +
                "<br>&nbsp;\n" +
                //"<table style=\"border: 1px solid #AA6\" border=\"0\" bgcolor=\"" + EDStatic.tableBGColor + "\"" + 
                //"  cellspacing=\"0\" cellpadding=\"2\">\n" +
                "<table class=\"erd\" bgcolor=\"" + EDStatic.tableBGColor + "\" cellspacing=\"0\">\n" +
                "  <tr><th>Protocol</th><th>Description</th></tr>\n" +
                "  <tr>\n" +
                "    <td><a href=\"" + tErddapUrl + "/griddap/index.html\"" + 
                    protSee + "griddap protocol.\">griddap</a> </td>\n" +
                "    <td>" + EDStatic.EDDGridDapDescription + "</td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><a href=\"" + tErddapUrl + "/tabledap/index.html\"" + 
                    protSee + "tabledap protocol.\">tabledap</a></td>\n" +
                "    <td>" + EDStatic.EDDTableDapDescription + "</td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><a href=\"" + tErddapUrl + "/wms/index.html\"" + 
                    protSee + "WMS protocol.\">WMS</a></td>\n" +
                "    <td>" + EDStatic.wmsDescriptionHtml + "</td>\n" +
                "  </tr>\n" +
                "</table>\n");  

            //end of search/protocol options list
            writer.write("\n</ul>\n");

            //end of table
            writer.write("</td>\n</tr>\n</table>\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end
        endHtmlWriter(out, writer, tErddapUrl, false);
    }

    /**
     * This responds by sending out the "Information" Html page (EDStatic.theLongDescriptionHtml).
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doInformationHtml(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs) throws Throwable {        

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);        
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Information", out);
        try {
            writer.write(EDStatic.youAreHere(tErddapUrl, "Information"));
            writer.write(EDStatic.theLongDescriptionHtml(tErddapUrl));
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        endHtmlWriter(out, writer, tErddapUrl, false);
    }

    /** This is used by doLogin to add a failed login attempt to failedLogins */
    public void loginFailed(String user) {
        if (verbose) String2.log("loginFailed " + user);
        EDStatic.tally.add("Log in failed (since startup)", user);
        EDStatic.tally.add("Log in failed (last 24 hours)", user);
        int ia[] = (int[])failedLogins.get(user);
        boolean wasNull = ia == null;
        if (wasNull)
            ia = new int[]{0,0};
        //update the count of recent failed logins
        ia[0]++;  
        //update the minute of the last failed login
        ia[1] = Math2.roundToInt(System.currentTimeMillis() / Calendar2.MILLIS_PER_MINUTE);  
        if (wasNull)
            failedLogins.put(user, ia);
    }

    /** This is used by doLogin when a users successfully logs in 
     *(to remove failed login attempts from failedLogins) */
    public void loginSucceeded(String user) {
        if (verbose) String2.log("loginSucceeded " + user);
        EDStatic.tally.add("Log in succeeded (since startup)", user);
        EDStatic.tally.add("Log in succeeded (last 24 hours)", user);
        //erase any info about failed logins
        failedLogins.remove(user);

        //clear failedLogins ~ every ~48.3 hours (just larger than 48 hours (2880 min), 
        //  so it occurs at different times of day)
        //this prevents failedLogins from accumulating never-used-again userNames
        //at worst, someone who just failed 3 times now appears to have failed 0 times; no big deal
        //but do it after a success, not a failure, so even that is less likely
        if (lastClearedFailedLogins + 2897L * Calendar2.MILLIS_PER_MINUTE < System.currentTimeMillis()) {
            if (verbose) String2.log("clearing failedLogins (done every few days)");
            lastClearedFailedLogins = System.currentTimeMillis();
            failedLogins.clear();
        }

    }

    /** This returns the number of minutes until the user can try to log in 
     * again (0 = now, 15 is max temporarily locked out).
     */
    public int minutesUntilLoginAttempt(String user) {
        int ia[] = (int[])failedLogins.get(user);

        //no recent attempt?
        if (ia == null)
            return 0;

        //greater than 15 minutes since last attempt?
        int minutesSince = Math2.roundToInt(System.currentTimeMillis() / Calendar2.MILLIS_PER_MINUTE - ia[1]);
        int minutesToGo = Math.max(0, 15 - minutesSince);
        if (minutesToGo == 0) { 
            failedLogins.remove(user); //erase any info about failed logins
            return 0;
        }

        //allow login if <3 recent failures
        if (ia[0] < 3) {
            return 0;
        } else {
            EDStatic.tally.add("Log in attempt blocked temporarily (since startup)", user);
            EDStatic.tally.add("Log in attempt blocked temporarily (last 24 hours)", user);
            if (verbose) String2.log("minutesUntilLoginAttempt=" + minutesToGo + " " + user);
            return minutesToGo;
        }
    }

    
    
    /**
     * This responds by prompting the user to login (e.g., login.html).
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doLogin(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String loginUrl = EDStatic.erddapHttpsUrl + "/login.html";
        String userQuery = request.getQueryString(); //may be null;  leave encoded
        String message = request.getParameter("message");
        String redMessage = message == null? "" :
            "<font color='red'><pre>" + 
            XML.encodeAsXML(message) +  //encoding is important to avoid security problems (HTML injection)
            "</pre></font>\n";                   
        String problemsLoggingIn = 
            "<p>&nbsp;<hr noshade><p><b>Problems?</b>\n" +
            "<p>If you have problems logging in:\n" +
            "<ul>\n" +
            "<li>Make sure that you log in with the exact same &info; that you gave to the ERDDAP administrator.\n" +
            "<li>Make sure your browser is set to allow\n" +
            "  <a href=\"http://en.wikipedia.org/wiki/HTTP_cookie\">cookies</a>:\n" +
            "  <ul>\n" +
            "  <li>In Internet Explorer, use \"Tools : Internet Options : Privacy\"\n" +
            "  <li>In Firefox, use \"Tools : Options : Privacy\"\n" +
            "  <li>In Opera, use \"Tools : Quick Preferences\"\n" +
            "  <li>In Safari, use \"Safari : Preferences : Security\"\n" +
            "  </ul>\n" +
            "  If you are making ERDDAP requests from a computer program (other than a browser),\n" +
            "  cookies are hard to work with. Sorry.\n" +
            "<li>Contact the administrator of this ERDDAP: " + EDStatic.adminContact() + ".\n" +
            "</ul>\n" +
            "\n";
        String problemsAfterLoggedIn = 
            "<p>&nbsp;<hr noshade><p><b>Problems and Solutions</b>\n" +

            "<p><b>Problem: On some web pages, ERDDAP says you aren't logged in;\n" +
            "  but when you try to log in, ERDDAP says you are already logged in.</b>\n" +
            "<br>Solution: URLs that start with \"http:\" are for users who aren't logged in.\n" +
            "<br>&nbsp; &nbsp; If/when you are logged in, change the URL to use \"https:\". Then, ERDDAP will see that you are logged in.\n" +

            "<p><b>Problem: You are logged in, but your private datasets aren't visible.</b>\n" +
            "<br>Solutions:\n" +
            "<ul>\n" +
            "<li>For any web page that doesn't seem right, click on your browser's Refresh button.\n" +
            "  <br>Your browser may have cached web pages from before you logged in.\n" +
            "&second;" +
            "<li>If you have multiple accounts, make sure you are logged into the appropriate one to see the desired datasets.\n" +
            "<li>Contact the administrator of this ERDDAP: " + EDStatic.adminContact() + ".\n" +
            "  <br>It is possible that the datasets are not currently visible or that you don't have permission to access them.\n" +
            "</ul>\n" +
            "\n";
        String publicAccess = "(You don't have to be logged in to access ERDDAP's publicly available datasets.)\n";
        String attemptBlocked1 = ERROR + ": Login attempts for ";
        String attemptBlocked2 = " are being temporarily blocked after 3 failed attempts.\nWait ";
        String attemptBlocked3 = " minutes, then try again.";

        //if authentication is active ...
        if (!EDStatic.authentication.equals("")) {
            //if request was sent to http:, redirect to https:
            String actualUrl = request.getRequestURL().toString();
            if (EDStatic.baseHttpsUrl != null && EDStatic.baseHttpsUrl.startsWith("https://") && //EDStatic ensures this is true
                !actualUrl.startsWith("https://")) {
                //hopefully this won't happen much
                //hopefully headers and other info won't be lost in the redirect
                response.sendRedirect(loginUrl +
                    (userQuery == null || userQuery.length() == 0? "" : "?" + userQuery));
                return;            
            }
        }


        //*** BASIC
        /*
        if (EDStatic.authentication.equals("basic")) {

            //this is based on the example code in Java Servlet Programming, pg 238

            //since login is external, there is no way to limit login attempts to 3 tries in 15 minutes

            //write the html for the form 
            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "Log In", out);
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, "Log In"));

                //show message from EDStatic.redirectToLogin (which redirects to here) or logout.html
                writer.write(redMessage);           

                writer.write("<p>This ERDDAP is configured to let you log in by entering your User Name and Password.\n");

                if (loggedInAs == null) {
                    //I don't think this can happen; users must be logged in to see this page
                    writer.write(
                    "<p><b>Something is wrong!</b> Your browser should have asked you to log in to see this web page!\n" +
                    "<br>Tell the ERDDAP administrator to check the &lt;tomcat&gt;/conf/web.xml file.\n" +
                    "<p>" + publicAccess);

                } else {
                    //tell user he is logged in
                    writer.write("<p>You are logged in as <b>" + loggedInAs + "</b>\n" +
                        "(<a href=\"" + EDStatic.erddapHttpsUrl + "/logout.html\">log out</a>)\n");
                }
            
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }
        */


        //*** CUSTOM
        if (EDStatic.authentication.equals("custom")) {

            //is user trying to log in?
            String user = request.getParameter("user");
            String password = request.getParameter("password");
            if (loggedInAs == null &&   //can't log in if already logged in
                user != null && user.length() > 0 && password != null) {
                int minutesUntilLoginAttempt = minutesUntilLoginAttempt(user);
                if (minutesUntilLoginAttempt > 0) {
                    response.sendRedirect(loginUrl + "?message=" + 
                        SSR.minimalPercentEncode(attemptBlocked1 + user + attemptBlocked2 + 
                            minutesUntilLoginAttempt + attemptBlocked3));
                    return;
                }
                try {
                    if (EDStatic.doesPasswordMatch(user, password)) {
                        //valid login
                        HttpSession session = request.getSession(); //make one if one doesn't exist
                        //it is stored on server.  user doesn't have access, so can't spoof it
                        //  (except by guessing the sessionID number (a long) and storing a cookie with it?)
                        session.setAttribute("loggedInAs:" + EDStatic.warName, user);  
//??? should I create/add a ticket number to session so it can't be spoofed???
                        loginSucceeded(user);
                        response.sendRedirect(loginUrl);
                        return;
                    } else {
                        //invalid login;  if currently logged in, logout
                        HttpSession session = request.getSession(false); //don't make one if one doesn't exist
                        if (session != null) {
                            session.removeAttribute("loggedInAs:" + EDStatic.warName);
                            session.invalidate();
                        }
                        loginFailed(user);
                        response.sendRedirect(loginUrl + "?message=" + SSR.minimalPercentEncode(
                            ERROR + ": Login failed: Invalid User Name and/or Password."));
                        return;
                    }
                } catch (Throwable t) {
                    response.sendRedirect(loginUrl +
                        "?message=" + ERROR + ": Login failed: " + 
                        SSR.minimalPercentEncode(MustBe.getShortErrorMessage(t)));
                    return;
                }
            }

            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "Log In", out);
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, "Log In"));

                //show message from EDStatic.redirectToLogin (which redirects to here) or logout.html
                writer.write(redMessage);           

                writer.write("<p>This ERDDAP is configured to let you log in by entering your User Name and Password.\n");

                if (loggedInAs == null) {
                    //show the login form
                    writer.write(
                    "<p>You are not logged in.\n" +
                    publicAccess +
                    //use POST, not GET, so that form params aren't in url (and so browser history, etc.)
                    "<form action=\"login.html\" method=\"post\" id=\"login_form\">\n" +  
                    "<p><b>Please log in:</b>\n" +
                    "<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
                    "  <tr>\n" +
                    "    <td>User Name: </td>\n" +
                    "    <td><input type=\"text\" size=\"30\" value=\"\" name=\"user\" id=\"user\"/></td>\n" +
                    "  </tr>\n" +
                    "  <tr>\n" +
                    "    <td>Password: </td>\n" + 
                    "    <td><input type=\"password\" size=\"20\" value=\"\" name=\"password\" id=\"password\"/>\n" +
                    "      <input type=\"submit\" value=\"Login\"/></td>\n" +
                    "  </tr>\n" +
                    "</table>\n" +
                    "</form>\n" +
                    "\n" +
                    String2.replaceAll(problemsLoggingIn, "&info;", "User Name and Password"));               

                } else {
                    //tell user he is already logged in
                    writer.write("<p>You are logged in as <b>" + loggedInAs + "</b>\n" +
                        "(<a href=\"" + EDStatic.erddapHttpsUrl + "/logout.html\">log out</a>)\n" +
                        String2.replaceAll(problemsAfterLoggedIn, "&second;", ""));
                }
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }


        //*** OpenID
        if (EDStatic.authentication.equals("openid")) {

            //this is based on the example code at http://joid.googlecode.com/svn/trunk/examples/server/login.jsp

            //check if user is requesting to signin, before writing content
            String oid = request.getParameter("openid_url");
            if (loggedInAs != null) {
                loginSucceeded(loggedInAs); //this over-counts successes (any time logged-in user visits login page)
            }
            if (loggedInAs == null &&   //can't log in if already logged in
                request.getParameter("signin") != null && oid != null && oid.length() > 0) {

                //first thing: normalize oid
                if (!oid.startsWith("http")) 
                    oid = "http://" + oid;

                //check if loginAttempt is allowed  AFTER oid has been normalized
                int minutesUntilLoginAttempt = minutesUntilLoginAttempt(oid);
                if (minutesUntilLoginAttempt > 0) {
                    response.sendRedirect(loginUrl + "?message=" + 
                        SSR.minimalPercentEncode(attemptBlocked1 + oid + attemptBlocked2 + 
                            minutesUntilLoginAttempt + attemptBlocked3));
                    return;
                }
                try {
                    String returnTo = loginUrl;

                    //tally as if login failed -- AFTER oid has been normalized
                    //this assumes it will fail (a success will initially be counted as a failure)
                    loginFailed(oid); 

                    //Future: read about trust realms: http://openid.net/specs/openid-authentication-2_0.html#realms
                    //Maybe this could be used to authorize a group of erddaps, e.g., https://*.pfeg.noaa.gov:8443
                    //But I think it is just an informative string for the user.
                    String trustRealm = EDStatic.erddapHttpsUrl; //i.e., logging into all of this erddap  (was loginUrl;)
                    String s = OpenIdFilter.joid().getAuthUrl(oid, returnTo, trustRealm);
                    String2.log("redirect to " + s);
                    response.sendRedirect(s);
                    return;
                } catch (Throwable t) {
                    response.sendRedirect(loginUrl +
                        "?message=" + ERROR + ": Login failed: " + 
                        SSR.minimalPercentEncode(MustBe.getShortErrorMessage(t)));
                    return;
                }
            }
         
            //write the html for the openID info and form
            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "OpenID Log In", out);
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl,
                    "<img align=\"bottom\" src=\"" + //align=middle looks bad on safari
                    EDStatic.imageDirUrl + "openid.png\" alt=\"OpenID\"/>OpenID Log In"));

                //show message from EDStatic.redirectToLogin (which redirects to here) or logout.html
                writer.write(redMessage);           

                //OpenID info
                writer.write(
                    "<p><a href=\"http://openid.net/\">OpenID</a> is an open standard that lets you\n" +
                    "log in with your password at one web site\n" +
                    "<br>and then log in without your password at many other web sites, including ERDDAP.\n");

                if (loggedInAs == null) {
                    //show the login form
                    writer.write(
                    "<p>You are not logged in.\n" + 
                    publicAccess +
                    "<script type=\"text/javascript\">\n" +
                    "  function submitForm(url){\n" +
                    "    document.getElementById(\"openid_url\").value = url;\n" +
                    "    document.getElementById(\"openid_form\").submit();\n" +
                    "  }\n" +
                    "</script>\n" +
                    //they use POST, not GET, probably so that form params aren't in url (and so browser history, etc.)
                    "<form action=\"login.html\" method=\"post\" id=\"openid_form\">\n" +  
                    "  <input type=\"hidden\" name=\"signin\" value=\"true\"/>\n" +
                    "  <p><b>Log in with your OpenID URL:</b>\n" +
                    "  <input type=\"text\" size=\"30\" value=\"\" name=\"openid_url\" id=\"openid_url\"/>\n" +
                    "  <input type=\"submit\" value=\"Login\"/>\n" +
                    "  <br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<small>For example: <tt>http://yourId.myopenid.com/</tt></small>\n" +
                    "</form>\n" +
                    "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Or log in to your existing (non-OpenID) account at \n" +
                    "<img align=\"middle\" src=\"http://l.yimg.com/us.yimg.com/i/ydn/openid-signin-yellow.png\" \n" +
                        "alt=\"Sign in with Yahoo\" onclick=\"submitForm('http://www.yahoo.com');\"/>\n" +
                    "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Or create an OpenID URL at\n" + 
                    "<a href=\"http://www.myopenid.com/\" target=\"_blank\">MyOpenID</a> (recommended), \n" +
                    "<a href=\"https://pip.verisignlabs.com/\" target=\"_blank\">Verisign</a>, or\n" +
                    "<a href=\"https://myvidoop.com/\" target=\"_blank\">Vidoop</a>. (They are all free!)\n" +
                    "\n" +
                    String2.replaceAll(problemsLoggingIn, "&info;", "OpenID URL"));

                } else {
                    //tell user he is already logged in
                    String s = String2.replaceAll(problemsAfterLoggedIn, "&second;", 
                        "<li>Make sure that you logged in with the exact same OpenID URL that you gave to the ERDDAP administrator.\n");
                    writer.write("<p>You are logged in as <b>" + loggedInAs + "</b>\n" +
                        "(<a href=\"" + EDStatic.erddapHttpsUrl + "/logout.html\">log out</a>)\n" + 
                        s);

                }
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }

        //*** Other
        //response.sendError(HttpServletResponse.SC_UNAUTHORIZED, 
        //    "This ERDDAP is not set up to let users log in.");
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Log In", out);
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, "Log In") +
                redMessage +
                "<p>This ERDDAP is not configured to let users log in.\n");       
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        endHtmlWriter(out, writer, tErddapUrl, false);
        return;
    }

    /**
     * This responds to a logout.html request.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doLogout(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {

        String loginUrl = EDStatic.erddapHttpsUrl + "/login.html";

        try {       
            //user wasn't logged in?
            String youWerentLoggedIn = "You weren't logged in.";
            String encodedYouWerentLoggedIn = "?message=" + 
                SSR.minimalPercentEncode(youWerentLoggedIn);
            if (loggedInAs == null && !EDStatic.authentication.equals("basic")) {
                //user wasn't logged in
                response.sendRedirect(loginUrl + encodedYouWerentLoggedIn);
                return;
            }

            //user was logged in
            HttpSession session = request.getSession(false); //false = don't make a session if none currently
            String successMessage = 
                "You have successfully logged out.\n" +
                "For increased security (especially if this is a shared computer), you can:\n" +
                " * Clear your browser's cache:\n" +
                "   * In Internet Explorer, use \"Tools : Delete Browsing History : Delete Files\"\n" +
                "   * In Firefox, use \"Tools : Clear Private Data\"\n" +
                "   * In Opera, use \"Tools : Delete Private Data\"\n" +
                "   * In Safari, use \"Safari : Empty Cache\"\n";
            String encodedSuccessMessage = "?message=" + SSR.minimalPercentEncode(successMessage);

            //*** BASIC
            /*
            if (EDStatic.authentication.equals("basic")) {
                if (session != null) {
                    //!!!I don't think this works!!!
                    ArrayList al = String2.toArrayList(session.getAttributeNames());
                    for (int i = 0; i < al.size(); i++)
                        session.removeAttribute(al.get(i).toString());
                    session.invalidate();
                }
                EDStatic.tally.add("Log out (since startup)", "success");
                EDStatic.tally.add("Log out (last 24 hours)", "success");

                //show the log out web page.   
                //Don't return to login.html, which triggers logging in again.
                String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, "Log Out", out);
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, "Log Out"));
                    if (loggedInAs == null) { 
                        //never was logged in 
                        writer.write(youWerentLoggedIn);
                    } else {
                        //still logged in?
                        loggedInAs = EDStatic.getLoggedInAs(request);
                        if (loggedInAs == null) {
                            //successfully logged out
                            String s = String2.replaceAll(successMessage, "\n", "\n<br>");
                            s = String2.replaceAll(successMessage, "   ", " &nbsp; ");
                            writer.write(s);       
                        } else {
                            //couldn't log user out!
                            writer.write(
                                "ERDDAP is having trouble logging you out.\n" +
                                "<br>To log out, please close your browser.\n");
                        }
                    }
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }
                endHtmlWriter(out, writer, tErddapUrl, false);                
                return;
            }
            */

            //*** CUSTOM
            if (EDStatic.authentication.equals("custom")) {
                if (session != null) { //should always be !null
                    session.removeAttribute("loggedInAs:" + EDStatic.warName);
                    session.invalidate();
                    EDStatic.tally.add("Log out (since startup)", "success");
                    EDStatic.tally.add("Log out (last 24 hours)", "success");
                }
                response.sendRedirect(loginUrl + encodedSuccessMessage);
                return;
            }

            //*** OpenID
            if (EDStatic.authentication.equals("openid")) {
                if (session != null) {  //should always be !null
                    OpenIdFilter.logout(session);
                    session.removeAttribute("user");
                    session.invalidate();
                    EDStatic.tally.add("Log out (since startup)", "success");
                    EDStatic.tally.add("Log out (last 24 hours)", "success");
                }
                response.sendRedirect(loginUrl + encodedSuccessMessage +
                    SSR.minimalPercentEncode(" * Log out from your OpenID provider."));
                return;
            }

            //*** Other    (shouldn't get here)
            response.sendRedirect(loginUrl + encodedYouWerentLoggedIn);
            return;

        } catch (Throwable t) {
            response.sendRedirect(loginUrl + 
                "?message=" + SSR.minimalPercentEncode(MustBe.getShortErrorMessage(t)));
            return;
        }
    }

    /**
     * This responds to a request for status.html.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doStatus(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Status", out);
        try {
            int nGridDatasets = gridDatasetHashMap.size();
            int nTableDatasets = tableDatasetHashMap.size();
            writer.write(
                EDStatic.youAreHere(tErddapUrl, "Status") +
                "<pre>");
            StringBuffer sb = new StringBuffer();
            EDStatic.addIntroStatistics(sb);

            //append number of active threads
            String traces = MustBe.allStackTraces(true, true);
            int po = traces.indexOf('\n');
            if (po > 0)
                sb.append(traces.substring(0, po + 1));

            sb.append(Math2.memoryString() + " " + Math2.xmxMemoryString() + "\n\n");
            EDStatic.addCommonStatistics(sb);
            sb.append(traces);
            writer.write(XML.encodeAsXML(sb.toString()));
            writer.write("</pre>");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        endHtmlWriter(out, writer, tErddapUrl, false);
    }

    /**
     * This responds by sending out the "Computer Programs"/REST information Html page.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doRestHtml(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Computer Programs", out);
        try {
            String htmlQueryUrl = tErddapUrl + "/search/index.html?searchFor=temperature";
            String jsonQueryUrl = tErddapUrl + "/search/index.json?searchFor=temperature";
            writer.write(
                EDStatic.youAreHere(tErddapUrl, "Computer Programs") +
                "<h2><a name=\"WebService\">Accessing</a> ERDDAP as a Web Service</h2>\n" +
                "ERDDAP is both:\n" +
                "<ul>\n" +
                "<li><a href=\"http://en.wikipedia.org/wiki/Web_application\">A web application</a> \n" +
                " &ndash; a web site that humans with browsers can use (in this case, to get data, graphs, and information about datasets).\n" +
                "<li><a href=\"http://en.wikipedia.org/wiki/Web_service\">A web service</a> \n" +
                " &ndash; a web site that computer programs can use (in this case, to get data, graphs, and information about datasets).\n" +
                "</ul>\n" +
                "For every ERDDAP feature that you as a human with a browser can use\n" +
                "<br>(for example, <a href=\"" + htmlQueryUrl + "\">Full Text Search for datasets</a>),\n" +
                "<br>there is an almost identical feature that is designed to be easy for computer programs to access\n" +
                "<br>(for example, <a href=\"" + jsonQueryUrl + "\">Full Text Search for datasets which returns the results\n" +
                "  as some other file type</a>, like <a href=\"http://www.json.org/\">JSON</a>),\n" +
                "<br>These features can be used by computer programs or scripts that you write.\n" +
                "<br>And they can be used to build other web applications or web services on top of ERDDAP \n" +
                "<br>(making ERDDAP do most of the work!).\n" +
                "<br>So if you have an idea for a better interface to the data the ERDDAP serves,\n" +
                "<br>we encourage you to build your own web application or web service, and use ERDDAP as the foundation.\n" +
                "<br>Your system can get data, graphs, and other information from our ERDDAP server, or you can \n" +
                //setup always from coastwatch's erddap 
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/download/setup.html\">set up your own ERDDAP server</a>.\n" + 
                "\n" +
                "<p><b>Requests</b>\n" +
                "<br>Requests for interface information from ERDDAP (for example, search results) \n" +
                "  use the web's universal standard for requests:\n" +
                "<a href=\"http://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html#sec9.3\">HTTP GET</a>.\n" +
                "<br>This is the same mechanism that your browser uses when you fill out a form on a web page \n" +
                "  and click on <tt>Submit</tt>.\n" +
                "<br>To use HTTP GET, you generate a specially formed URL (perhaps with a query) \n" +
                "  and send it to the web with HTTP GET.\n" +
                "<br>You can form these URLs by hand and enter them in the address textfield of your browser\n" +
                "<br>(for example, <a href=\"" + jsonQueryUrl + "\">" + jsonQueryUrl + "</a>),\n" +
                "<br>or a computer program can be written to create the URLs, submit them, and get the responses.\n" +
                "<br><a name=\"PercentEncode\">Often,</a> the computer program will need to \n" +
                "  <a href=\"http://en.wikipedia.org/wiki/Percent-encoding\">percent encode</a> the URL\n" +
                "<br>(for example, replace \" \" in a query value with \"%20\", for example, \"searchFor=temperature%20wind%20speed\").\n" +
                "<br>(Browsers do this for you automatically when you fill out a form on a web page and click on Submit.)\n" +
                "<br>Programming languages have tools to do this (for example, see \n" +
                "  <a href=\"http://java.sun.com/j2se/1.4.2/docs/api/java/net/URLEncoder.html\">java.net.URLEncoder</a>).\n" +
                "<br>HTTP GET was chosen because it is simple, it works, \n" +
                "<br>it is universally supported (in browsers, computer languages, operating system tools, etc),\n" +
                "<br>and it is a foundation of\n" +
                "  <a href=\"http://en.wikipedia.org/wiki/Representational_State_Transfer\">REST</a> and\n" +
                "  <a href=\"http://www.crummy.com/writing/RESTful-Web-Services/\">ROA</a>.\n" +
                "<br>These URLs are great because they completely define a given request,\n" +
                "<br>and you can bookmark them in your browser, write them in your notes, email them to a friend, ...\n" +
                "\n" +
                "<p>" + OutputStreamFromHttpResponse.acceptEncodingHtml +
                "\n" +
                "<p><b>Responses</b>\n" +
                "<br>Although humans using browsers want to receive interface results \n" +
                "  (for example, search results) as HTML documents,\n" +
                "<br>computer programs often prefer to get results in simple, easily parsed, \n" +
                "  less verbose documents.\n" +
                "<br>ERDDAP can return interface results as a table of data in these common, \n" +
                "  computer-program friendly, file types:\n" +
                "<ul>\n");
            for (int pft = 0; pft < plainFileTypes.length; pft++) {
                int tIndex = String2.indexOf(EDDTable.dataFileTypeNames, plainFileTypes[pft]);
                if (tIndex >= 0) 
                    writer.write("<li>" + plainFileTypes[pft] + " - " +
                        EDDTable.dataFileTypeDescriptions[tIndex] + " (" +
                        "<a href=\"" + XML.encodeAsXML(EDDTable.dataFileTypeInfo[tIndex]) + 
                        "\">more info</a>)");
            }
            writer.write(
                "</ul>\n" +
                "The first row of the table has the column names.\n" +
                "<br>In some types of files, the second row has the column's Java-style data type \n" +
                "  (e.g., \"String\" or \"float\").\n" +
                "<br>Subsequent rows of the table have the results.\n" +
                "\n" +
                "<p>.csv and .tsv issues:<ul>\n" +
                "<li>If a datum in a .csv file has internal double quotes or commas, \n" +
                "  ERDDAP follows the .csv specification strictly: it puts double quotes around the datum \n" +
                "  and doubles the internal double quotes.\n" +
                "<li>If a datum in a .csv or .tsv file has internal newline characters, \n" +
                "  ERDDAP converts the newline characters to character #166 (&brvbar;). This is non-standard.\n" +
                "</ul>\n" +
                "\n" +
                "<p><a name=\"jsonp\">Requests</a> for .json files may now include an optional\n" +
                "<a href=\"http://niryariv.wordpress.com/2009/05/05/jsonp-quickly/\">jsonp</a> request\n" +
                "<br>by adding \"&amp;.jsonp=<i>functionName</i>\" to the end of the query.\n" +
                "<br>Basically, this just tells ERDDAP to add \"<i>functionName</i>(\" to the beginning of the response \n" +
                "<br>and \")\" to the end of the response.\n" +
                "<br>If originally there was no query, leave off the \"&amp;\" in your query.\n" +
                "\n" + 
                "<p>Remember that these are the file types ERDDAP can use to respond to \n" +
                "  user-interface types of requests (for example, search requests).\n" +
                "<br>ERDDAP supports a different set of file types for the actual data \n" +
                "  (for example, satellite and buoy data) requests (see \n" +
                "  <a href=\"" + tErddapUrl + "/griddap/index.html\">griddap</a> and\n" +
                "  <a href=\"" + tErddapUrl + "/tabledap/index.html\">tabledap</a>).\n" +
                "\n" +

                "<p><b>Access URLs</b>\n" +
                "<br>" + EDStatic.ProgramName + " has these URL access points for computer programs:\n" +
                "<ul>\n" +
                "<li>The list of the main resource access URLs can be returned as a\n" +
                    plainLinkExamples(tErddapUrl, "/index", "") +
                "<li>The current list of all datasets can be returned as a " + plainLinkExamples(tErddapUrl, "/info/index", "") + 
                "<li>Info about a specific data set (the list of variables and their attributes) \n" +
                "  is returned by a call to\n" + 
                    "<br>" + tErddapUrl + "/info/&lt;datasetID&gt;/index&lt;fileType&gt; , \n" +
                    "<br>for example, a " + plainLinkExamples(tErddapUrl, "/info/" + EDStatic.EDDGridIdExample + "/index", "") + 
                "<li>The results of full text searches for datasets can be returned as a (for example)\n" +
                    plainLinkExamples(tErddapUrl, "/search/index", "?searchFor=temperature") +
                "  <br>(You probably need to \n" +
                    "<a href=\"http://en.wikipedia.org/wiki/Percent-encoding\">percent-encode</a>\n" +
                    "the values in the query (e.g., \"searchFor=temperature%20wind%20speed\").\n" +
                "<li>The list of categoryAttributes can be returned as a\n" +
                    plainLinkExamples(tErddapUrl, "/categorize/index", "") +
                "<li>The list of categories for a specific categoryAttribute can be returned as a (for example)\n" +
                    plainLinkExamples(tErddapUrl, "/categorize/standard_name/index", "") +
                "<li>The list of datasets in a specific category can be returned as a (for example)\n" +
                    plainLinkExamples(tErddapUrl, "/categorize/standard_name/time/index", ""));
            int tDasIndex = String2.indexOf(EDDTable.dataFileTypeNames, ".das");
            int tDdsIndex = String2.indexOf(EDDTable.dataFileTypeNames, ".dds");
            writer.write(
                "<li>The current list of datasets which can be accessed by the various protocols can be returned as\n" +
                "  <ul>\n" +
                "  <li>For griddap: a\n" +
                    plainLinkExamples(tErddapUrl, "/griddap/index", "") +
                "  <li>For tabledap: a\n" +
                    plainLinkExamples(tErddapUrl, "/tabledap/index", "") +
                "  <li>For WMS: a\n" +
                    plainLinkExamples(tErddapUrl, "/wms/index", "") +
                "  </ul>\n" +
                "<li>The Data Access Forms are just simple web pages to generate URLs which request data \n" +
                "  (e.g., satellite and buoy data).\n" +
                "  <br>The Make A Graph pages are just simple web pages to generate URLs \n" +
                "  which request graphs of a subset of the data.\n" +
                "  <br>Your program can generate these URLs directly. For more information, read about\n" +
                "    <a href=\"" + tErddapUrl + "/griddap/documentation.html\">griddap</a>,\n" +
                "    <a href=\"" + tErddapUrl + "/tabledap/documentation.html\">tabledap</a>, and\n" +
                "    <a href=\"" + tErddapUrl + "/wms/documentation.html\">WMS</a>.\n" +
                "  <br>griddap and tabledap can return data in many common file formats. \n" +
                "  See their documentation for details.\n" +
                "  <br>Note that griddap and tabledap can return\n" +
                "    <a href=\"" + XML.encodeAsXML(EDDTable.dataFileTypeInfo[tDdsIndex]) + "\">.dds</a>\n" +
                "    files with the dataset's structure, including variable names.\n" + 
                "  <br>Note that griddap and tabledap can return\n" +
                "    <a href=\"" + XML.encodeAsXML(EDDTable.dataFileTypeInfo[tDasIndex]) + "\">.das</a>\n" +
                "    files with all of the dataset's \n" +
                "    <a href=\"http://en.wikipedia.org/wiki/Metadata\">metadata</a>.\n" +
                "  <br>Or, you can get variable names and metadata via a call to\n" + 
                    "<br>" + tErddapUrl + "/info/&lt;datasetID&gt;/index&lt;fileType&gt; , \n" +
                    "<br>for example, a " + plainLinkExamples(tErddapUrl, "/info/" + EDStatic.EDDGridIdExample + "/index", "") + 
                "<li>ERDDAP offers \n" +
                "    <a href=\"" + tErddapUrl + "/information.html#subscriptions\">email/URL and RSS subscriptions</a>,\n" +
                "    so your computer program can be notified whenever a dataset changes.\n" +
                "<li>Need other types of information? \n" +
                "   <br>The links above are just the first batch of links to provide access to information from ERDDAP.\n" +
                "   <br>If you have suggestions for additional links, \n" + 
                "      contact <tt>bob dot simons at noaa dot gov</tt>.\n" +
                "</ul>\n");

            //within a Java program
            writer.write(
                "<h2><a name=\"JavaPrograms\">Using</a> ERDDAP as a Data Source within Your Java Program</h2>\n" +
                "As described above, since Java programs can access data available on the web, \n" +
                "<br>you can write a Java program that accesses data from any publicly accessible ERDDAP installation.\n" +
                "\n" +
                "<p>Or, since ERDDAP is an all-open source program, you can also set up your own copy of ERDDAP\n" +
                "<br>on your own server (internal or public) to serve your own data.\n" +
                "<br>Any Java programs that you write can get data from that copy of ERDDAP.\n" +
                "\n" +
                "<p>Or, since ERDDAP is an all-open source, all-Java program, \n" +
                "<br>you can also set up your own copy of ERDDAP on your own computer within your Java program.\n" +
                "<br>Then you can access ERDDAP's services from within your program\n" +
                "<br>(for example, doing full-text searches for datasources or creating data files \n" +
                "  from remote data sources) .\n" +
                //setup.html always from coastwatch's erddap
                "<br>See <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/download/setup.html\">Set Up Your Own ERDDAP</a>,\n" +
                    "especially the JavaDocs for the gov.noaa.pfel.erddap.Erddap class.\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        endHtmlWriter(out, writer, tErddapUrl, false);

    }

    /**
     * This responds by sending out the sitemap.xml file.
     * <br>See http://www.sitemaps.org/protocol.php
     * <br>This uses the startupDate as the lastmod date.
     * <br>This uses changefreq=monthly. Datasets may change in small ways (e.g., near-real-time data), 
     *   but that doesn't affect the metadata that search engines are interested in.
     */
    public void doSitemap(HttpServletRequest request, HttpServletResponse response) throws Throwable {

        //always use plain EDStatic.erddapUrl
        String pre = 
            "<url>\n" +
            "<loc>" + EDStatic.erddapUrl + "/";
        String basicPost =
            "</loc>\n" +
            "<lastmod>" + EDStatic.startupLocalDateTime.substring(0,10) + "</lastmod>\n" +
            "<changefreq>monthly</changefreq>\n" +
            "<priority>";
        //highPriority
        String postHigh = basicPost + "0.7</priority>\n" +  
            "</url>\n" +
            "\n";
        //medPriority
        String postMed = basicPost + "0.5</priority>\n" +  //0.5 is the default
            "</url>\n" +
            "\n";
        //lowPriority
        String postLow = basicPost + "0.3</priority>\n" +
            "</url>\n" +
            "\n";

        //beginning
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, "sitemap", ".xml", ".xml");
        OutputStream out = outSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out, "UTF-8");
        writer.write(
            "<?xml version='1.0' encoding='UTF-8'?>\n" +
            //this is their simple example
            "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n" +
            //this is what they recommend to validate it, but it doesn't validate for me ('urlset' not defined)
            //"<urlset xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" +
            //"    xsi:schemaLocation=\"http://www.sitemaps.org/schemas/sitemap/0.9\"\n" +
            //"    url=\"http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd\"\n" +
            //"    xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n" +
            "\n");

        //write the individual urls
        //don't include the admin pages that all link to ERD's erddap
        //don't include setDatasetFlag.txt, setup.html, setupDatasetsXml.html, status.html, 
        writer.write(pre); writer.write("categorize/index.html");             writer.write(postMed);
        writer.write(pre); writer.write("griddap/documentation.html");        writer.write(postHigh);
        writer.write(pre); writer.write("griddap/index.html");                writer.write(postHigh);
        writer.write(pre); writer.write("images/embed.html");                 writer.write(postHigh);
        writer.write(pre); writer.write("images/gadgets/GoogleGadgets.html"); writer.write(postHigh);
        writer.write(pre); writer.write("index.html");                        writer.write(postHigh);
        writer.write(pre); writer.write("info/index.html");                   writer.write(postHigh); 
        writer.write(pre); writer.write("information.html");                  writer.write(postHigh);
        writer.write(pre); writer.write("rest.html");                         writer.write(postHigh);
        writer.write(pre); writer.write("search/index.html");                 writer.write(postHigh); 
        writer.write(pre); writer.write("slidesorter.html");                  writer.write(postHigh);
        writer.write(pre); writer.write("subscriptions/index.html");          writer.write(postHigh); 
        writer.write(pre); writer.write("subscriptions/add.html");            writer.write(postMed); 
        writer.write(pre); writer.write("subscriptions/validate.html");       writer.write(postMed);
        writer.write(pre); writer.write("subscriptions/list.html");           writer.write(postMed); 
        writer.write(pre); writer.write("subscriptions/remove.html");         writer.write(postMed);
        writer.write(pre); writer.write("tabledap/documentation.html");       writer.write(postHigh);
        writer.write(pre); writer.write("tabledap/index.html");               writer.write(postHigh);
        writer.write(pre); writer.write("wms/documentation.html");            writer.write(postHigh);
        writer.write(pre); writer.write("wms/index.html");                    writer.write(postHigh);

        //special links only for ERD's erddap
        if (EDStatic.baseUrl.equals("http://coastwatch.pfeg.noaa.gov")) {
            writer.write(pre); writer.write("download/grids.html");                 writer.write(postHigh);
            writer.write(pre); writer.write("download/setup.html");                 writer.write(postHigh);
            writer.write(pre); writer.write("download/setupDatasetsXml.html");      writer.write(postHigh);
        }

        //write the dataset .html and .graph urls
        StringArray sa = gridDatasetIDs(true);
        int n = sa.size();
        String gPre = pre + "griddap/";
        String iPre = pre + "info/";
        String tPre = pre + "tabledap/";
        String wPre = pre + "wms/";
        for (int i = 0; i < n; i++) {
            //don't inlude index/datasetID, .das, .dds; better that people go to .html or .graph
            String dsi = sa.get(i);
            writer.write(gPre); writer.write(dsi); writer.write(".html");        writer.write(postMed);    
            writer.write(gPre); writer.write(dsi); writer.write(".graph");       writer.write(postMed);
            writer.write(iPre); writer.write(dsi); writer.write("/index.html");  writer.write(postLow);
            EDD edd = (EDD)gridDatasetHashMap.get(dsi);
            if (edd != null && edd.accessibleViaWMS()) {
                writer.write(wPre); writer.write(dsi); writer.write("/index.html");  writer.write(postLow);
            }
        }

        sa = tableDatasetIDs(true);
        n = sa.size();
        for (int i = 0; i < n; i++) {
            String dsi = sa.get(i);
            writer.write(tPre); writer.write(dsi); writer.write(".html");        writer.write(postMed);
            writer.write(tPre); writer.write(dsi); writer.write(".graph");       writer.write(postMed);
            writer.write(iPre); writer.write(dsi); writer.write("/index.html");  writer.write(postLow);
            //EDDTable currently don't do wms
        }

        //write the category urls
        for (int ca1 = 0; ca1 < EDStatic.categoryAttributes.length; ca1++) {
            String ca1s = EDStatic.categoryAttributes[ca1];
            StringArray cats = categoryInfo(ca1s);
            int nCats = cats.size();
            String cPre = pre + "categorize/" + ca1s + "/";
            writer.write(cPre); writer.write("index.html"); writer.write(postMed);
            for (int ca2 = 0; ca2 < nCats; ca2++) {
                writer.write(cPre); writer.write(cats.get(ca2)); writer.write("/index.html"); writer.write(postMed);
            }
        }

        //end
        writer.write(
            "</urlset>\n"); 
        writer.close(); //it flushes, it closes 'out'
    }

    /**
     * This is used to generate examples for the plainFileTypes in the method above.
     * 
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param relativeUrl e.g., "/griddap/index"
     * @return a string with a series of html links to information about the plainFileTypes
     */
    protected String plainLinkExamples(String tErddapUrl,
        String relativeUrl, String query) throws Throwable {

        StringBuffer sb = new StringBuffer();
        int n = plainFileTypes.length;
        for (int pft = 0; pft < n; pft++) {
            sb.append(
                "    <a href=\"" + tErddapUrl + relativeUrl + plainFileTypes[pft] + query + "\">" + 
                plainFileTypes[pft] + "</a>");
            if (pft <= n - 3) sb.append(",\n");
            if (pft == n - 2) sb.append(", or\n");
            if (pft == n - 1) sb.append(" document.\n");
        }
        return sb.toString();
    }

    /**
     * Process a grid or table OPeNDAP DAP-style request.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    ("griddap" or "tabledap") in the requestUrl
     * @param userDapQuery  post "?".  Still percentEncoded.
     */
    public void doDap(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs,
        String protocol, int datasetIDStartsAt, String userDapQuery) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);       
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String fileTypeName = "";
        try {

            String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
                requestUrl.substring(datasetIDStartsAt);

            //respond to a documentation.html request
            if (endOfRequestUrl.equals("documentation.html")) {

                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, protocol + " Documentation", out); 
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, protocol, "Documentation"));
                    if (protocol.equals("griddap"))       EDDGrid.writeGeneralDapHtmlInstructions(tErddapUrl, writer, true);
                    else if (protocol.equals("tabledap")) EDDTable.writeGeneralDapHtmlInstructions(tErddapUrl, writer, true);
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }
                endHtmlWriter(out, writer, tErddapUrl, false);
                return;
            }

            //first, always set the standard DAP response header info
            standardDapHeader(response);

            //redirect to index.html
            if (endOfRequestUrl.equals("") ||
                endOfRequestUrl.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/" + protocol + "/index.html");
                return;
            }      
            
            //respond to a version request (see opendap spec section 7.2.5)
            if (endOfRequestUrl.equals("version") ||
                endOfRequestUrl.startsWith("version.") ||
                endOfRequestUrl.endsWith(".ver")) {

                //write version response 
                //DAP 2.0 7.1.1 says version requests DON'T include content-description header.
                OutputStreamSource outSource = new OutputStreamFromHttpResponse(
                    request, response, "version", //fileName is not used
                    ".txt", ".txt");
                OutputStream out = outSource.outputStream("ISO-8859-1");
                Writer writer = new OutputStreamWriter(out, "ISO-8859-1");
                writer.write( 
                    "Core Version: " + dapVersion + OpendapHelper.EOL + //see EOL definition for comments
                    "Server Version: " + serverVersion + OpendapHelper.EOL); 

                //DODSServlet always does this if successful     done automatically?
                //response.setStatus(HttpServletResponse.SC_OK);

                //essential
                writer.flush();
                if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
                out.close(); 

                return;
            }

            //respond to a help request  (see opendap spec section 7.2.6)
            //Note that lack of fileType (which opendap spec says should lead to help) 
            //  is handled elsewhere with error message and help 
            //  (which seems appropriate and mimics other dap servers)
            if (endOfRequestUrl.equals("help") ||
                endOfRequestUrl.startsWith("help.") ||                
                endOfRequestUrl.endsWith(".help")) {

                //write help response 
                //DAP 2.0 7.1.1 says help requests DON'T include content-description header.
                OutputStreamSource outputStreamSource = 
                    new OutputStreamFromHttpResponse(request, response, 
                        "help", ".html", ".html");
                OutputStream out = outputStreamSource.outputStream("ISO-8859-1");
                Writer writer = new OutputStreamWriter(
                    //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit
                    out, "ISO-8859-1");
                writer.write(EDStatic.startHeadHtml(tErddapUrl, protocol + " Help"));
                writer.write(EDStatic.standardHead);
                writer.write("\n</head>\n");
                writer.write(EDStatic.startBodyHtml(loggedInAs));
                writer.write("\n");
                writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl));     
                writer.write(EDStatic.youAreHere(tErddapUrl, protocol, "Help"));
                writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
                try {
                    if (protocol.equals("griddap")) 
                        EDDGrid.writeGeneralDapHtmlInstructions(tErddapUrl, writer, true); //true=complete
                    if (protocol.equals("tabledap")) 
                        EDDTable.writeGeneralDapHtmlInstructions(tErddapUrl, writer, true); //true=complete

                    writer.write("<br>&nbsp;\n" +
                        "<br><small>" + EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small>\n");
                    if (EDStatic.displayDiagnosticInfo) 
                        EDStatic.writeDiagnosticInfoHtml(writer);
                    writer.write(EDStatic.endBodyHtml(tErddapUrl));
                    writer.write("\n</html>\n");
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }
                //essential
                writer.flush();
                if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
                out.close(); 

                return;
            }

            //get the datasetID and requested fileType
            int dotPo = endOfRequestUrl.lastIndexOf('.');
            if (dotPo < 0) 
                throw new SimpleException("URL error: " +
                    "No file type (e.g., .html) was specified after the datasetID.");

            String id = endOfRequestUrl.substring(0, dotPo);
            fileTypeName = endOfRequestUrl.substring(dotPo);
            if (reallyVerbose) String2.log("  id=" + id + "\n  fileTypeName=" + fileTypeName);

            //respond to xxx/index request
            //show list of 'protocol'-supported datasets in .html file
            if (id.equals("index")) {
                sendDatasetList(request, response, loggedInAs, protocol, fileTypeName);
                return;
            }

            //get the dataset
            EDD dataset = protocol.equals("griddap")? 
                (EDD)gridDatasetHashMap.get(id) : 
                (EDD)tableDatasetHashMap.get(id);
            if (dataset == null) {
                sendResourceNotFoundError(request, response, 
                    "Currently unknown datasetID=" + id);
                return;
            }
            if (!dataset.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
                EDStatic.redirectToLogin(loggedInAs, response, id);
                return;
            }

            EDStatic.tally.add(protocol + " DatasetID (since startup)", id);
            EDStatic.tally.add(protocol + " DatasetID (last 24 hours)", id);
            EDStatic.tally.add(protocol + " File Type (since startup)", fileTypeName);
            EDStatic.tally.add(protocol + " File Type (last 24 hours)", fileTypeName);
            String fileName = dataset.suggestFileName(userDapQuery, fileTypeName);
            String extension = dataset.fileTypeExtension(fileTypeName);
            if (reallyVerbose) String2.log("  fileName=" + fileName + "\n  extension=" + extension);

            //if remote ERDDAP dataset, forward request
            //Note that .html and .graph are handled locally so links on web pages 
            //  are for this server and the reponses can be handled quickly.
            String tqs = request.getQueryString();  //still encoded
            if (tqs == null) tqs = "";
            if (tqs.length() > 0) tqs = "?" + tqs;
            if (dataset instanceof EDDGridFromErddap &&
                (!fileTypeName.equals(".html") && !fileTypeName.equals(".graph"))) {
                String tUrl = ((EDDGridFromErddap)dataset).getNextSourceErddapUrl() + fileTypeName;
                if (verbose) String2.log("redirected to " + tUrl + tqs);
                response.sendRedirect(tUrl + tqs);  //response.encodeRedirectURL(tUrl)); //not necessary to add sessionInfo to url
                //javax.servlet.RequestDispatcher dispatcher = getServletContext().getRequestDispatcher(tUrl);
                //dispatcher.redirect(request,response);
                return;
            }
            if (dataset instanceof EDDTableFromErddap &&
                (!fileTypeName.equals(".html") && !fileTypeName.equals(".graph"))) {
                String tUrl = ((EDDTableFromErddap)dataset).getNextSourceErddapUrl() + fileTypeName;
                if (verbose) String2.log("redirected to " + tUrl + tqs);
                response.sendRedirect(tUrl + tqs);  
                return;
            }

            OutputStreamSource outputStreamSource = 
                new OutputStreamFromHttpResponse(request, response, 
                    fileName, fileTypeName, extension);

            //*** tell the dataset to send the data
            try {
                dataset.respondToDapQuery(request, loggedInAs, requestUrl, userDapQuery, 
                    outputStreamSource, 
                    EDStatic.fullCacheDirectory + dataset.datasetID() + "/", fileName, //dir is created by EDD.ensureValid
                    fileTypeName);
            } catch (WaitThenTryAgainException wttae) {
                String2.log("!!ERDDAP caught WaitThenTryAgainException");
                //is response committed?
                if (response.isCommitted()) {
                    String2.log("but the response is already committed. So rethrowing the error.");
                    throw wttae;
                }

                //wait up to 30 seconds for dataset to reload
                int waitSeconds = 30;
                for (int sec = 0; sec < waitSeconds; sec++) {
                    //sleep for a second
                    Math2.sleep(1000); 

                    //has the dataset finished reloading?
                    EDD dataset2 = protocol.equals("griddap")? 
                        (EDD)gridDatasetHashMap.get(id) : 
                        (EDD)tableDatasetHashMap.get(id);
                    if (dataset2 != null && dataset != dataset2) { //yes, simplistic !=,  not !equals
                        //yes! ask dataset2 to respond to the query

                        //does user still have access to the dataset?
                        if (!dataset2.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
                            EDStatic.redirectToLogin(loggedInAs, response, id);
                            return;
                        }

                        try {
                            //note that this will fail if the previous reponse is already committed
                            dataset2.respondToDapQuery(request, loggedInAs,
                                requestUrl, userDapQuery, outputStreamSource, 
                                EDStatic.fullCacheDirectory + dataset.datasetID() + "/", fileName, //dir is created by EDD.ensureValid
                                fileTypeName);
                            String2.log("!!ERDDAP successfully used dataset2 to respond to the request.");
                            break; //success! jump out of for(sec) loop
                        } catch (Throwable t) {
                            String2.log("!!!!ERDDAP caught Exception while trying WaitThenTryAgainException:\n" +
                                MustBe.throwableToString(t));
                            throw wttae; //throw original error
                        }
                    }

                    //if the dataset didn't reload after waitSeconds, throw the original error
                    if (sec == waitSeconds - 1)
                        throw wttae;
                }
            }

            //DODSServlet always does this if successful     //???is it done by default???
            //response.setStatus(HttpServletResponse.SC_OK);

            OutputStream out = outputStreamSource.outputStream("");
            if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
            out.close(); //essential, to end compression
            return;

        } catch (Throwable t) {

            //deal with the DAP error

            //catch errors after the response has begun
            if (neededToSendErrorCode(request, response, t))
                return;

            //display dap error message in a web page
            boolean isDapType = 
                (fileTypeName.equals(".asc")  ||
                 fileTypeName.equals(".das")  || 
                 fileTypeName.equals(".dds")  || 
                 fileTypeName.equals(".dods") ||
                 fileTypeName.equals(".html"));

            if (isDapType) {
                OutputStreamSource outSource = new OutputStreamFromHttpResponse(
                    request, response, "error", //fileName is not used
                    fileTypeName, fileTypeName);
                OutputStream out = outSource.outputStream("ISO-8859-1");
                Writer writer = new OutputStreamWriter(out, "ISO-8859-1");

                //see DAP 2.0, 7.2.4  for error structure               
                //see dods.dap.DODSException for codes.  I don't know (here), so use 3=Malformed Expr
                String error = MustBe.getShortErrorMessage(t);
                writer.write("Error {\n" +
                    "  code = 3 ;\n" +
                    "  message = \"" +
                        String2.replaceAll(error, "\"", "\\\"") + //see DAP appendix A, quoted-string    
                        "\" ;\n" +
                    "} ; "); //thredds has final ";"; spec doesn't

                //essential
                writer.flush();
                if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
                out.close(); 
            } else { 
                //other file types, e.g., .csv, .json, .mat,
                throw t;
            }
        }
    }


    /**
     * This sends the list of graph or table datasets
     *
     * @param request
     * @param response
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param protocol   must be "griddap", "tabledap", or "wms"
     * @param fileTypeName e.g., .html or .json
     * throws Throwable if trouble
     */
    public void sendDatasetList(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String protocol, String fileTypeName) throws Throwable {

        //is it a valid fileTypeName?
        int pft = String2.indexOf(plainFileTypes, fileTypeName);
        if (pft < 0 && !fileTypeName.equals(".html")) {
            sendResourceNotFoundError(request, response, "Unsupported fileType=" + fileTypeName);
            return;
        }

        //gather the datasetIDs and descriptions
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        StringArray ids;
        String description;  
        if      (protocol.equals("griddap")) {
            ids = gridDatasetIDs(true);
            description = EDDGrid.longDapDescriptionHtml;
        } else if (protocol.equals("tabledap")) {
            ids = tableDatasetIDs(true);
            description = EDDTable.longDapDescriptionHtml;
        } else if (protocol.equals("wms")) {
            ids = new StringArray();
            StringArray gids = gridDatasetIDs(true);
            for (int gi = 0; gi < gids.size(); gi++) {
                EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(gids.get(gi));
                if (eddGrid != null && //if just deleted
                    eddGrid.accessibleViaWMS()) 
                    ids.add(eddGrid.datasetID());
            }
            description = EDStatic.wmsLongDescriptionHtml;
        } else {
            sendResourceNotFoundError(request, response, "Unknown protocol=" + protocol);
            return;
        }
        
        //handle plainFileTypes   
        if (pft >= 0) {
            //make the plain table with the dataset list
            Table table = makePlainDatasetTable(loggedInAs, ids, true, fileTypeName);  
            sendPlainTable(request, response, table, protocol, fileTypeName);
            return;
        }


        //make the html table with the dataset list
        Table table = makeHtmlDatasetTable(loggedInAs, ids, true);  

        //display start of web page
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "List of " + protocol + " Datasets", out); 
        try {
            writer.write(EDStatic.youAreHere(tErddapUrl, protocol));
            String uProtocol = protocol.equals("wms")? "WMS" : protocol;

            writer.write("<h2>Description</h2>\n");
            writer.write(description); 
            writer.write(
                "\n<br>For details, see ERDDAP's <a href=\"" + tErddapUrl + "/" + protocol +  
                    "/documentation.html\">" + uProtocol + " Documentation</a>.\n");

            if (table.nRows() == 0) {
                writer.write("\n<p><b>" + EDStatic.noDatasetWith + " protocol = \"" + protocol + "\"</b>");
            } else {
                writer.write("\n<h2>Datasets Which Can Be Accessed via " + uProtocol + "</h2>\n" 
                    //+ EDStatic.clickAccessHtml + "\n" +
                    //"<br>&nbsp;\n"
                    );
                table.saveAsHtmlTable(writer, EDStatic.tableBGColor, 1, false, -1, false, false);        
                writer.write("<p>" + table.nRows() + " " + EDStatic.nDatasetsListed + "\n");
            }
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }


    /**
     * Process a SOS request -- NOT YET FINISHED.
     * This SOS service is intended to simulate the IOOS DIF SOS service (datasetID=ndbcSOS).
     * For IOOS DIF schemas, see http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/ .
     * O&M document(?) says that query names are case insensitive, but query values are case sensitive.
     * Background info: http://www.opengeospatial.org/projects/groups/sensorweb     
     *
     * <p>This assumes request was for /erddap/sos.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    ("sos") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     */
    public void doSos(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, int datasetIDStartsAt, String userQuery) throws Throwable {

/*
This isn't finished!   Reference server (ndbcSOS) is in flux and ...
Interesting IOOS DIF info c:/programs/sos/EncodingIOOSv0.6.0Observations.doc
Returning GetCapabilities is roughly done, but not bad.
Returning DescribeSensor isn't at all done.
Returning GetObservation isn't at all done.
I looked into GetObservation, but got stuck on 
   <om:result>
      <ioos:Composite gml:id="CurrentsPointTimeSeriesDataObservations" 
        recDef="http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.0/dataRecordDefinitions/CurrentsPointTimeSeriesDataRecordDefinition.xml">
  Can I substitute a different type of result?
     (I prefer the Oostethys block response.)
  Must I use an IOOS DIF predefined recDef, or can I make my own?
    If it has to be a predefined (by IOOS DIF) schema, does this invalidate
       my approach of having ERDDAP convert data from different sources
       made available via a SOS service?
    Can I pre-make some multi-phenomena response types to cover all cases?
    Can I make response types on the fly?
  See c:/programs/sos/ndbcSosResponse.xml
  I decided to wait until this is an official project, then ask Jeff Dlb for suggestions...
Roy says:
  If you want to ask, ask Jeff DLb, otherwise let it drop.  
  it is Jeff's baby though, so be thoughtful in how you word it.  
  Right now, the client side is probably more important for us, as is WMS.
  Jeff (author of WMS spec!) is at Jeff.deLaBeaujardiere@noaa.gov 
*/

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);

        //catch other reponses outside of try/catch  (so errors handled in doGet)
        if (endOfRequestUrl.equals("") || endOfRequestUrl.equals("index.htm")) {
            response.sendRedirect(tErddapUrl + "/sos/index.html");
            return;
        }
        if (endOfRequestUrl.equals("index.html")) {
            doSosHtml(request, response, loggedInAs);
            return;
        }
        if (!endOfRequestUrl.equals("request"))
            sendResourceNotFoundError(request, response, "");
        
        try {

            //parse userQuery  e.g., ?service=SOS&request=GetCapabilities
            HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase

            //must be service=SOS     but I don't require it
            String tService = queryMap.get("service"); 
            //if (tService == null || !tService.equals("SOS"))
            //    throw new SimpleException("Query error: service='" + tService + "' must be 'SOS'."); 

            //deal with different request=
            String tRequest = queryMap.get("request");
            if (tRequest == null)
                tRequest = "";

            if (tRequest.equals("GetCapabilities")) {
                //e.g., ?service=SOS&request=GetCapabilities
                writeSosCapabilities(request, response, loggedInAs); 
                return;

            } else if (
                tRequest.equals("GetObservation") ||
                tRequest.equals("DescribeSensor")) {
                throw new SimpleException("Query error: request='" + tRequest + "' is not yet supported."); 

            } else {
                throw new SimpleException("Query error: request='" + tRequest + "' isn't supported."); 
            }

        } catch (Throwable t) {

            //deal with the SOS error
            //catch errors after the response has begun
            if (neededToSendErrorCode(request, response, t))
                return;

            OutputStreamSource outSource = new OutputStreamFromHttpResponse(
                request, response, "error", //fileName is not used
                ".xml", ".xml");
            OutputStream out = outSource.outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8");

            //for now, mimic oostethys  (ndbcSOS often doesn't throw exceptions)
            String error = MustBe.getShortErrorMessage(t);
            writer.write(
                "<?xml version=\"1.0\"?>\n" +
                "<ExceptionReport xmlns=\"http://www.opengis.net/ows\" \n" +
                "  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" \n" +
                "  xsi:schemaLocation=\"http://www.opengis.net/ows owsExceptionReport.xsd\" \n" +
                "  version=\"1.0.0\" language=\"en\">\n" +
                //???locator= ???
                //???where is list of exceptionCodes???
                "  <Exception locator=\"service\" exceptionCode=\"OperationNotSupported\">\n" +
                "    <ExceptionText>" + XML.encodeAsXML(error) + "</ExceptionText>\n" +
                "  </Exception>\n" +
                "</ExceptionReport>\n");

            //essential
            writer.flush();
            if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
            out.close(); 

        }
    }


    /**
     * This responds by sending out "ERDDAP's SOS Service" Html page.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doSosHtml(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "SOS Information", out);
        try {
            String getCapabilities = tErddapUrl + "/sos/request?service=SOS&amp;request=GetCapabilities";
            String sosExample = tErddapUrl + "/sos/request?NotYetFinished"; //EDStatic.sosExample;
            writer.write(
                EDStatic.youAreHere(tErddapUrl, "Sensor Observation Service (SOS)") +
                "<b>If you have SOS client software, you can point it to this ERDDAP installation's SOS at \n" +
                tErddapUrl + "/sos/request .</b>\n" +
                "\n" +
                "<h2>Overview</h2>\n" +
                "In addition to making data available via \n" +
                "<a href=\"" + tErddapUrl + "/griddap/index.html\">gridddap</a> and \n" +
                "<a href=\"" + tErddapUrl + "/tabledap/index.html\">tabledap</a>,\n" + 
                "ERDDAP makes some data available via ERDDAP's Sensor Observation Service (SOS) web service.\n" +
                "\n" +
                "<p>SOS is an <a href=\"http://www.opengeospatial.org/\">Open Geospatial Consortium (OGC)</a>\n" +
                "standard which partially defines a \"standard web service interface for requesting, filtering, \n" +
                "and retrieving observations and sensor system information. \n" +
                "This is the intermediary between a client and an observation repository \n" +
                "or near real-time sensor channel.\"\n" +
                "SOS is part of the OGC's \n" +
                "  <a href=\"http://www.opengeospatial.org/projects/groups/sensorweb\">Sensor Web Enablement (SWE) project</a>.\n" +
                "Detailed information about SOS can be found at the SWE web site.\n" +
                "\n" +
                "<p>ERDDAP's SOS service is intended be compatible with the \n" +
                "<a href=\"http://sdf.ndbc.noaa.gov/sos/\">IOOS DIF SOS service</a>, which uses the \n" +
                "<a href=\"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/\">IOOS DIF SOS schema</a>.\n" +
                "\n" +
                "<p>SOS clients send HTTP GET requests (specially formed URLs) to the SOS service and get XML responses.\n" +
                "\n" +
                "<h2>Examples</h2>\n" +
                "<ul>\n" +
                "<li>You can request the list of capabilities via \n" +
                "  <a href=\"" + getCapabilities + "\">" + getCapabilities + "</a>.\n" +
                "<li>A sample data request is \n" +
                "  <a href=\"" + sosExample + "\">" + sosExample + "</a>.\n" +
                "</ul>\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        endHtmlWriter(out, writer, tErddapUrl, false);

    }

    /**
     * Return the SOS capabilities xml for doSos.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void writeSosCapabilities(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {

        //return capabilities xml
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, "Capabilities", ".xml", ".xml");
        OutputStream out = outSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out, "UTF-8");
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String sosUrl = tErddapUrl + "/sos";
        writer.write(
//???this was patterned after ndbcSOS capabilities 2008-07-20
"<?xml version=\"1.0\"?>\n" +
"<Capabilities xmlns:gml=\"http://www.opengis.net/gml/\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
"  xmlns:swe=\"http://www.opengis.net/swe/1.0.1\" xmlns:om=\"http://www.opengis.net/om/1.0\" \n" +
"  xmlns=\"http://www.opengis.net/sos/1.0\" xmlns:sos=\"http://www.opengis.net/sos/1.0\" \n" +
"  xmlns:ows=\"http://www.opengis.net/ows/1.1\" xmlns:ogc=\"http://www.opengis.net/ogc\" \n" +
"  xmlns:tml=\"http://www.opengis.net/tml\" xmlns:sml=\"http://www.opengis.net/sensorML/1.0.1\" \n" +
"  xmlns:myorg=\"http://www.myorg.org/features\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" \n" +
"  version=\"1.0.0\">\n" +
"<!-- This is a PROTOTYPE.  The information in this response is NOT complete. -->\n");

        //ServiceIdentification
        writer.write(
"  <ows:ServiceIdentification>\n" +
"    <ows:Title>" + XML.encodeAsXML(EDStatic.sosTitle) + "</ows:Title>\n" +
"    <ows:Abstract>" + XML.encodeAsXML(EDStatic.sosAbstract) + "</ows:Abstract>\n" +
"    <ows:Keywords>\n");
            for (int i = 0; i < EDStatic.sosKeywords.length; i++)
                writer.write(
"      <ows:Keyword>" + XML.encodeAsXML(EDStatic.sosKeywords[i]) + "</ows:Keyword>\n");
            writer.write(
"    </ows:Keywords>\n" +
"    <ows:ServiceType codeSpace=\"http://opengeospatial.net\">OGC:SOS</ows:ServiceType>\n" +
"    <ows:ServiceTypeVersion>0.0.0</ows:ServiceTypeVersion>\n" +  //???really???
"    <ows:Fees>" + XML.encodeAsXML(EDStatic.sosFees) + "</ows:Fees>\n" +
"    <ows:AccessConstraints>" + XML.encodeAsXML(EDStatic.sosAccessConstraints) + "</ows:AccessConstraints>\n" +
"  </ows:ServiceIdentification>\n");

        //ServiceProvider 
        writer.write(
"  <ows:ServiceProvider>\n" +
"    <ows:ProviderName>" + XML.encodeAsXML(EDStatic.adminInstitution) + "</ows:ProviderName>\n" +
"    <ows:ProviderSite xlink:href=\"" + tErddapUrl + "\"/>\n" +
"    <ows:ServiceContact>\n" +
"      <ows:IndividualName>" + XML.encodeAsXML(EDStatic.adminIndividualName) + "</ows:IndividualName>\n" +
"      <ows:ContactInfo>\n" +
"        <ows:Phone>\n" +
"          <ows:Voice>" + XML.encodeAsXML(EDStatic.adminPhone) + "</ows:Voice>\n" +
"        </ows:Phone>\n" +
"        <ows:Address>\n" +
"          <ows:DeliveryPoint>" + XML.encodeAsXML(EDStatic.adminAddress) + "</ows:DeliveryPoint>\n" +
"          <ows:City>" + XML.encodeAsXML(EDStatic.adminCity) + "</ows:City>\n" +
"          <ows:AdministrativeArea>" + XML.encodeAsXML(EDStatic.adminStateOrProvince) + "</ows:AdministrativeArea>\n" +
"          <ows:PostalCode>" + XML.encodeAsXML(EDStatic.adminPostalCode) + "</ows:PostalCode>\n" +
"          <ows:Country>" + XML.encodeAsXML(EDStatic.adminCountry) + "</ows:Country>\n" +
"          <ows:ElectronicMailAddress>" + XML.encodeAsXML(EDStatic.adminEmail) + "</ows:ElectronicMailAddress>\n" +
"        </ows:Address>\n" +
"      </ows:ContactInfo>\n" +
"    </ows:ServiceContact>\n" +
"  </ows:ServiceProvider>\n");

        //OperationsMetadaata 
        writer.write(
"  <ows:OperationsMetadata>\n" +
"    <ows:Operation name=\"GetCapabilities\">\n" +
"      <ows:DCP>\n" +
"        <ows:HTTP>\n" +
"          <ows:Get xlink:href=\"" + sosUrl + "\"/>\n" +
//"          <ows:Post xlink:href=\"" + sosUrl + "\"/>\n" +  //support Post???
"        </ows:HTTP>\n" +
"      </ows:DCP>\n" +
"      <ows:Parameter name=\"Sections\">\n" + //the sections of this document; allow request sections???
"        <ows:AllowedValues>\n" +
"          <ows:Value>ServiceIdentification</ows:Value>\n" +
"          <ows:Value>ServiceProvider</ows:Value>\n" +
"          <ows:Value>OperationsMetadata</ows:Value>\n" +
"          <ows:Value>Contents</ows:Value>\n" +
"          <ows:Value>Filter_Capabilities</ows:Value>\n" +  //???where?
"          <ows:Value>All</ows:Value>\n" +
"        </ows:AllowedValues>\n" +
"      </ows:Parameter>\n" +
"    </ows:Operation>\n" +
"    <ows:Operation name=\"GetObservation\">\n" +
"      <ows:DCP>\n" +
"        <ows:HTTP>\n" +
"          <ows:Get xlink:href=\"" + sosUrl + "\"/>\n" +
//"          <ows:Post xlink:href=\"" + sosUrl + "\"/>\n" +  //allow???
"        </ows:HTTP>\n" +
"      </ows:DCP>\n" +
"    </ows:Operation>\n" +
"    <ows:Operation name=\"DescribeSensor\">\n" +
"      <ows:DCP>\n" +
"        <ows:HTTP>\n" +
"          <ows:Get xlink:href=\"" + sosUrl + "\"/>\n" +
//"          <ows:Post xlink:href=\"" + sosUrl + "\"/>\n" +  //allow???
"        </ows:HTTP>\n" +
"      </ows:DCP>\n" +
"    </ows:Operation>\n" +
"    <ows:Parameter name=\"service\">\n" +
"      <ows:AllowedValues>\n" +
"        <ows:Value>SOS</ows:Value>\n" +
"      </ows:AllowedValues>\n" +
"    </ows:Parameter>\n" +
"    <ows:Parameter name=\"version\">\n" +
"      <ows:AllowedValues>\n" +
"        <ows:Value>1.0.0</ows:Value>\n" +  
"      </ows:AllowedValues>\n" +
"    </ows:Parameter>\n" +
"  </ows:OperationsMetadata>\n");

        //Contents 
        writer.write(
"  <Contents>\n" +
"    <ObservationOfferingList>\n");

        Object tableDatasets[] = tableDatasetHashMap.values().toArray();
        String roles[] = EDStatic.getRoles(loggedInAs);
        for (int ds = 0; ds < tableDatasets.length; ds++) {
            EDDTable eddTable = (EDDTable)tableDatasets[ds];
            if (!EDStatic.listPrivateDatasets && !eddTable.isAccessibleTo(roles))
                continue;
            int nOfferings = eddTable.sosNOfferings();
            for (int offering = 0; offering < nOfferings; offering++) { //e.g. 100 stations
                String datasetID = eddTable.datasetID();
                String offeringName = eddTable.sosOfferingName(offering);
                String obsProps[] = eddTable.sosObservedProperties(); //ERDDAP assumes all the same(!)
                String gmlName = "urn:x-noaa:def:station:" +  //x-noaa -> x-erd???  station/trajectory/profile...???
                    datasetID + ":" + offeringName; //ndbcSOS had ndbc:41012
                double minTime = eddTable.sosMinTime(offering);
                double maxTime = eddTable.sosMaxTime(offering);
                String minTimeString = Double.isNaN(minTime)?  
                    " indeterminatePosition=\"unknown\">" :
                    ">" + Calendar2.epochSecondsToIsoStringT(minTime) + "Z";
                String maxTimeString = Double.isNaN(maxTime)?  
                    " indeterminatePosition=\"unknown\">" :
                    ">" + Calendar2.epochSecondsToIsoStringT(maxTime) + "Z";

                writer.write(
"      <ObservationOffering gml:id=\"" + //ndbc started id with "offering-", but is seem unnecessarily verbose
            datasetID + offeringName + "\">\n" +  //???separate with dot, dash, _ or ':'?
"        <gml:name>" + gmlName + "</gml:name>\n" +  
"        <gml:srsName>EPSG:4326</gml:srsName>\n" +
"        <gml:boundedBy>\n" +
"          <gml:Envelope srsName=\"EPSG:4326\">\n" +
"            <gml:lowerCorner>" + eddTable.sosMinLatitude( offering) + " " + 
                                  eddTable.sosMinLongitude(offering) + "</gml:lowerCorner>\n" +
"            <gml:upperCorner>" + eddTable.sosMaxLatitude( offering) + " " + 
                                  eddTable.sosMaxLongitude(offering) + "</gml:upperCorner>\n" +
"          </gml:Envelope>\n" +
"        </gml:boundedBy>\n" +
"        <time>\n" +
"          <gmlTimePeriod>\n" +
"            <gml:beginPosition" + minTimeString + "</gml:beginPosition>\n" +
"            <gml:endPosition"   + maxTimeString + "</gml:endPosition>\n" +
"          </gmlTimePeriod>\n" +
"        </time>\n" +
"        <procedure xlink:href=\"urn:x-noaa:def:sensor:" +  //x-noaa -> x-erd???    sensor???
                    datasetID + ":" + offeringName + ":1\"/>\n");  //:1???

                for (int op = 0; op < obsProps.length; op++) { //e.g. currents and temperature
                    writer.write(
"        <observedProperty xlink:href=\"" + obsProps[op] + "\"/>\n");
                }

                writer.write(
"        <featureOfInterest xlink:href=\"" + gmlName + "\"/>\n" +
"        <responseFormat>text/xml;schema=\"ioos/0.6.0\"</responseFormat>\n" + //???why 2 responseFormat?
"        <responseFormat>application/ioos+xml;version=0.6.0</responseFormat>\n" +  
"        <responseMode>inline</responseMode>\n" +
"      </ObservationOffering>\n");
            }
        }

        writer.write(
"    </ObservationOfferingList>\n" +
"  </Contents>\n");

        //end of getCapabilities
        writer.write(
"</Capabilities>\n");

        //essential
        writer.flush();
        if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
        out.close(); 
    }

    /**
     * Direct a WMS request to proper handler.
     *
     * <p>This assumes request was for /erddap/wms
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    ("wms") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     */
    public void doWms(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, int datasetIDStartsAt, String userQuery) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        int slashPo = endOfRequestUrl.indexOf('/'); //between datasetID/endEnd
        if (slashPo < 0) slashPo = endOfRequestUrl.length();
        String tDatasetID = endOfRequestUrl.substring(0, slashPo);
        String endEnd = slashPo >= endOfRequestUrl.length()? "" : 
            endOfRequestUrl.substring(slashPo + 1);

        //catch other reponses outside of try/catch  (so errors handled in doGet)
        if (endOfRequestUrl.equals("") || endOfRequestUrl.equals("index.htm")) {
            response.sendRedirect(tErddapUrl + "/wms/index.html");
            return;
        }
        if (endEnd.length() == 0 && endOfRequestUrl.startsWith("index.")) {
            sendDatasetList(request, response, loggedInAs, "wms", endOfRequestUrl.substring(5)); 
            return;
        }
        if (endOfRequestUrl.equals("documentation.html")) {
            doWmsDocumentation(request, response, loggedInAs);
            return;
        }
        if (endOfRequestUrl.equals("openlayers110.html")) { 
            doWmsOpenLayers(request, response, loggedInAs, "1.1.0", EDStatic.wmsSampleDatasetID);
            return;
        }
        if (endOfRequestUrl.equals("openlayers111.html")) { 
            doWmsOpenLayers(request, response, loggedInAs, "1.1.1", EDStatic.wmsSampleDatasetID);
            return;
        }
        if (endOfRequestUrl.equals("openlayers130.html")) { 
            doWmsOpenLayers(request, response, loggedInAs, "1.3.0", EDStatic.wmsSampleDatasetID);
            return;
        }

        if (endOfRequestUrl.equals("request")) {
            doWmsRequest(request, response, loggedInAs, "", userQuery); //all datasets
            return;
        }
        
        //for a specific dataset
        EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(tDatasetID);
        if (eddGrid != null) {
            if (!eddGrid.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
                EDStatic.redirectToLogin(loggedInAs, response, tDatasetID);
                return;
            }

            //request is for /wms/datasetID/...
            if (endEnd.equals("") || endEnd.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/wms/index.html");
                return;
            }

            if (endEnd.equals("index.html")) {
                doWmsOpenLayers(request, response, loggedInAs, "1.3.0", tDatasetID);
                return;
            }
            if (endEnd.equals("request")) {
                doWmsRequest(request, response, loggedInAs, tDatasetID, userQuery); 
                return;
            }

            //error
            sendResourceNotFoundError(request, response, "");
        } 

        //error
        sendResourceNotFoundError(request, response, "");
    }

    /**
     * This handles a request for the /wms/request or /wms/datasetID/request -- a real WMS service request.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param tDatasetID   an EDDGrid datasetID or "" (for all EDDGrids)
     * @param userQuery post '?', still percentEncoded, may be null.
     */
    public void doWmsRequest(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String tDatasetID, String userQuery) throws Throwable {

        try {

            //parse userQuery  e.g., ?service=WMS&request=GetCapabilities
            HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase

            //must be service=WMS     but I don't require it
            String tService = queryMap.get("service");
            //if (tService == null || !tService.equals("WMS"))
            //    throw new SimpleException("Query error: service='" + tService + "' must be 'WMS'."); 

            //deal with different request=
            String tRequest = queryMap.get("request");
            if (tRequest == null)
                tRequest = "";

            //e.g., ?service=WMS&request=GetCapabilities
            if (tRequest.equals("GetCapabilities")) {
                doWmsGetCapabilities(request, response, loggedInAs, tDatasetID, queryMap); 
                return;
            }
            
            if (tRequest.equals("GetMap")) {
                doWmsGetMap(request, response, loggedInAs, queryMap); 
                return;
            }

            //if (tRequest.equals("GetFeatureInfo")) { //optional, not yet supported

            throw new SimpleException("Query error: request='" + tRequest + "' isn't supported."); 

        } catch (Throwable t) {

            String2.log("  doWms caught Exception:\n" + MustBe.throwableToString(t));

            //catch errors after the response has begun
            if (neededToSendErrorCode(request, response, t))
                return;

            //send out WMS XML error
            OutputStreamSource outSource = new OutputStreamFromHttpResponse(
                request, response, "error", //fileName is not used
                ".xml", ".xml");
            OutputStream out = outSource.outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8");

            //see WMS 1.3.0 spec, section H.2
            String error = MustBe.getShortErrorMessage(t);
            writer.write(
"<?xml version='1.0' encoding=\"UTF-8\"?>\n" +
"<ServiceExceptionReport version=\"1.3.0\"\n" +
"  xmlns=\"http://www.opengis.net/ogc\"\n" +
"  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" +
"  xsi:schemaLocation=\"http://www.opengis.net/ogc http://schemas.opengis.net/wms/1.3.0/exceptions_1_3_0.xsd\">\n" +
"  <ServiceException" + // code=\"InvalidUpdateSequence\"    ???list of codes
//security: encodeAsXml important to prevent xml injection
">" + XML.encodeAsXML(error) + "</ServiceException>\n" + 
"</ServiceExceptionReport>\n");

            //essential
            writer.flush();
            if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
            out.close(); 

        }
    }


    /**
     * This responds by sending out the WMS html documentation page (long description).
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     */
    public void doWmsDocumentation(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs) throws Throwable {
       
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String e0 = tErddapUrl + "/wms/" + EDStatic.wmsSampleDatasetID + "/request?"; 
        String ec = "service=WMS&amp;request=GetCapabilities&amp;version=";
        String e1 = "service=WMS&amp;version="; 
        String e2 = "&amp;request=GetMap&amp;bbox=" + EDStatic.wmsSampleBBox +
                    "&amp;"; //needs c or s
        //this section of code is in 2 places
        int bbox[] = String2.toIntArray(String2.split(EDStatic.wmsSampleBBox, ',')); 
        int tHeight = Math2.roundToInt(((bbox[3] - bbox[1]) * 360) / Math.max(1, bbox[2] - bbox[0]));
        tHeight = Math2.minMaxDef(10, 600, 180, tHeight);
        String e2b = "rs=EPSG:4326&amp;width=360&amp;height=" + tHeight + 
            "&amp;bgcolor=0x808080&amp;layers=";
        //Land,erdBAssta5day:sst,Coastlines,Nations
        String e3 = EDStatic.wmsSampleDatasetID + WMS_SEPARATOR + EDStatic.wmsSampleVariable;
        String e4 = "&amp;styles=&amp;format=image/png";
        String et = "&amp;transparent=TRUE";

        String tWmsGetCapabilities110    = e0 + ec + "1.1.0";
        String tWmsGetCapabilities111    = e0 + ec + "1.1.1";
        String tWmsGetCapabilities130    = e0 + ec + "1.3.0";
        String tWmsOpaqueExample110      = e0 + e1 + "1.1.0" + e2 + "s" + e2b + "Land," + e3 + ",Coastlines,Nations" + e4;
        String tWmsOpaqueExample111      = e0 + e1 + "1.1.1" + e2 + "s" + e2b + "Land," + e3 + ",Coastlines,Nations" + e4;
        String tWmsOpaqueExample130      = e0 + e1 + "1.3.0" + e2 + "c" + e2b + "Land," + e3 + ",Coastlines,Nations" + e4;
        String tWmsTransparentExample110 = e0 + e1 + "1.1.0" + e2 + "s" + e2b + e3 + e4 + et;
        String tWmsTransparentExample111 = e0 + e1 + "1.1.1" + e2 + "s" + e2b + e3 + e4 + et;
        String tWmsTransparentExample130 = e0 + e1 + "1.3.0" + e2 + "c" + e2b + e3 + e4 + et;

        //What is WMS?   (generic) 
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "WMS Documentation", out);
        try {
            String likeThis = "<a href=\"" + tErddapUrl + "/wms/" + EDStatic.wmsSampleDatasetID + 
                     "/index.html\">like this</a>";
            String makeAGraphRef = "<a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/embed.html\">Make A Graph</a>\n";
            String datasetListRef = "<br>See the <a href=\"" + tErddapUrl + 
                "/wms/index.html\">list of datasets available via WMS</a> at this ERDDAP installation.\n";
            String makeAGraphListRef =
                "  <br>See the <a href=\"" + tErddapUrl + 
                   "/info/index.html\">list of datasets with Make A Graph</a> at this ERDDAP installation.\n";

            writer.write(
                //see almost identical documentation at ...
                EDStatic.youAreHere(tErddapUrl, "wms", "Documentation") +
                EDStatic.wmsLongDescriptionHtml + "\n" +
                datasetListRef +
                //"<p>\n" +
                "<h2>Three Ways to Make Maps with WMS</h2>\n" +
                "<ol>\n" +
                "<li> <b>In theory, anyone can download, install, and use WMS client software.</b>\n" +
                "  <br>Some clients are: \n" +
                "    <a href=\"http://www.esri.com/software/arcgis/\">ArcGIS</a>,\n" +
                "    <a href=\"http://mapserver.refractions.net/phpwms/phpwms-cvs/\">Refractions PHP WMS Client</a>, and\n" +
                "    <a href=\"http://udig.refractions.net//\">uDig</a>. \n" +
                "  <br>To make these work, you would install the software on your computer.\n" +
                "  <br>Then, you would enter the URL of the WMS service into the client.\n" +
                //arcGis required WMS 1.1.1 (1.1.0 and 1.3.0 didn't work)
                "  <br>For example, in ArcGIS (not yet fully working because it doesn't handle time!), use\n" +
                "  <br>\"Arc Catalog : Add Service : Arc Catalog Servers Folder : GIS Servers : Add WMS Server\".\n" +
                "  <br>In ERDDAP, each dataset has its own WMS service, which is located at\n" +
                "  <br>&nbsp; &nbsp; " + tErddapUrl + "/wms/<i>datasetID</i>/request?\n" +  
                "  <br>&nbsp; &nbsp; For example: <b>" + e0 + "</b>\n" +  
                "  <br>(Some WMS client programs don't want the <b>?</b> at the end of that URL.)\n" +
                datasetListRef +
                "  <p><b>In practice,</b> we haven't found any WMS clients that properly handle dimensions\n" +
                "  <br>other than longitude and latitude (e.g., time), a feature which is specified by the WMS\n" +
                "  <br>specification and which is utilized by most datasets in ERDDAP's WMS servers.\n" +
                "  <br>You may find that using a dataset's " + makeAGraphRef + 
                "     form and selecting the .kml file type\n" +
                "  <br>(an OGC standard) to load images into <a href=\"http://earth.google.com/\">Google Earth</a> provides\n" +            
                "    a good (non-WMS) map client.\n" +
                makeAGraphListRef +
                "  <br>&nbsp;\n" +
                "<li> <b>Web page authors can embed a WMS client in a web page.</b>\n" +
                "  <br>For example, ERDDAP uses \n" +
                "    <a href=\"http://openlayers.org\">OpenLayers</a>, \n" +  
                "    which is a very versatile WMS client, for the WMS page for \n" +
                "    each ERDDAP dataset \n" +
                "    (" + likeThis + ").\n" +  
                datasetListRef +
                "  <br>OpenLayers doesn't automatically deal with dimensions\n" +
                "    other than longitude and latitude (e.g., time),\n" +            
                "  <br>so you will have to write JavaScript (or other scripting code) to do that.\n" +
                "  <br>(Adventurous JavaScript programmers can look at the Source Code from a web page " + likeThis + ".)\n" + 
                "  <br>&nbsp;\n" +
                "<li> <b>A person with a browser or a computer program can generate special GetMap URLs\n" +
                "  and view/use the resulting image file.</b>\n" +
                "  <br><b>Opaque example:</b> <a href=\"" + tWmsOpaqueExample130 + "\">" + 
                                                            tWmsOpaqueExample130 + "</a>\n" +
                "  <br><b>Transparent example:</b> <a href=\"" + tWmsTransparentExample130 + "\">" + 
                                                                 tWmsTransparentExample130 + "</a>\n" +
                datasetListRef +
                "  <br><b>See the details below.</b>\n" +
                "  <p><b>In practice, it is probably easier and more versatile to use a dataset's\n" +
                "    " + makeAGraphRef + " form</b>\n" +
                "  <br>than to use WMS for this purpose.\n" +
                makeAGraphListRef +
                "</ol>\n" +
                "\n");

            writer.write(
                "<p>&nbsp;\n" +
                "<h2><a name=\"details\">Forming GetMap URLs</a></h2>\n" +
                //"<p>In ERDDAP's WMS, all data variables in grid datasets that use\n" +
                //"  longitude and latitude dimensions are available via WMS.\n" +
                "  A person with a browser or a computer program can generate a special URL to request a map.\n" + 
                "  <br>The URL must be in the form\n" +
                "  <br>&nbsp;&nbsp;&nbsp;" + tErddapUrl + "/wms/<i>datasetID</i>/request?<i>query</i> " +
                "  <br>The query for a WMS GetMap request consists of several <i>parameterName=value</i>, separated by '&amp;'.\n" +
                "  <br>For example,\n" +
                "  <br>&nbsp; &nbsp; <a href=\"" + tWmsOpaqueExample130 + "\">" + 
                                                   tWmsOpaqueExample130 + "</a>\n" +
                "  <br>Parameter names are case insensitive.\n" +  //WMS 1.3.0 spec section 6.8.1
                "  <br>Parameter values are case sensitive.\n" +
                "  <br>The various <i>parameterName=value</i> pairs can be in any order in the GetMap query.\n" +
                "  <br>The <a name=\"parameters\">parameter</a> options for the GetMap request are:\n" +
                "<p><table class=\"erd\" cellspacing=\"0\">\n" +
                "  <tr>\n" +
                "    <th>Request parameter</th>\n" +
                "    <th>Mandatory/<br>Optional</th>\n" +
                "    <th>Description</th>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>VERSION=<i>version</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Request version.\n" +
                "      <br>Currently, ERDDAP's WMS supports \"1.1.0\", \"1.1.1\", and \"1.3.0\".\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>REQUEST=GetMap</td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Request name.</td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>LAYERS=<i>layer_list</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Comma-separated list of one or more map layers.\n" +
                "        <br>Layers are drawn in the order they occur in the list.\n" +
                "        <br>Currently in ERDDAP's WMS, the layer names from datasets are named <i>datasetID</i>" + 
                    WMS_SEPARATOR + "<i>variableName</i> .\n" +
                "        <br>In ERDDAP's WMS, there are three layers not based on ERDDAP datasets:\n" +
                "        <br> * \"Land\" may be drawn BEFORE (as an under layer) or AFTER (as a land mask) layers from grid datasets.\n" +
                "        <br> * \"Coastlines\" should usually be drawn AFTER layers from grid datasets.\n" +  
                "        <br> * \"Nations\" draws political boundaries. This should usually be drawn AFTER layers from grid datasets.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>STYLES=<i>style_list</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Comma-separated list of one rendering style per requested layer.\n" +
                "      <br>Currently in ERDDAP's WMS, the only style offered for each layer is the default style,\n" +
                "      <br>which is specified via \"\".\n" +
                "      <br>For example, if you request 3 layers, you can use \"STYLES=,,\".\n" +
                "      <br>Or, even easier, you can request the default style for all layers via \"STYLES=\".\n" + 
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td nowrap>1.1.0: SRS=<i>namespace:identifier</i>" +
                           "<br>1.1.1: SRS=<i>namespace:identifier</i>" +
                           "<br>1.3.0: CRS=<i>namespace:identifier</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Coordinate reference system.\n" +
                "        <br>Currently in ERDDAP's WMS 1.1.0, the only valid SRS is EPSG:4326.\n" +
                "        <br>Currently in ERDDAP's WMS 1.1.1, the only valid SRS is EPSG:4326.\n" +
                "        <br>Currently in ERDDAP's WMS 1.3.0, the only valid CRS's are CRS:84 and EPSG:4326,\n" +
                "        <br>All of those options support longitude from -180 to 180 and latitude -90 to 90.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>BBOX=<i>minx,miny,maxx,maxy</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Bounding box corners (lower left, upper right) in CRS units.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>WIDTH=<i>output_width</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Width in pixels of map picture.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>HEIGHT=<i>output_height</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Height in pixels of map picture.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>FORMAT=<i>output_format</i></td>\n" +
                "    <td align=\"center\">M</td>\n" +
                "    <td>Output format of map.\n" +
                "      <br>Currently in ERDDAP's WMS, only image/png is valid.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>TRANSPARENT=TRUE|FALSE</td>\n" +
                "    <td align=\"center\">O</td>\n" +
                "    <td>Background transparency of map (default=FALSE).\n" +
                "      <br>If TRUE, any part of the image using the BGColor will be made transparent.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>BGCOLOR=<i>color_value</i></td>\n" +
                "    <td align=\"center\">O</td>\n" +
                "    <td>Hexadecimal 0xRRGGBB color value for the background color (default=0xFFFFFF, white).\n" +
                "      <br>If Transparent=TRUE, we recommend BGCOLOR=0x808080, since white is in some color palettes.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>EXCEPTIONS=<i>exception_format</i></td>\n" +
                "    <td align=\"center\">O</td>\n" +
                "    <td>The format in which exceptions are to be reported by the WMS (the default is the xml option).\n" +
                "      <br>Currently, ERDDAP's WMS 1.1.0 and 1.1.1 supports \"application/vnd.ogc.se_xml\", " +
                "         \"application/vnd.ogc.se_blank\" (a blank image) and\n" +
                "         \"application/vnd.ogc.se_inimage\" (the error in an image).\n" +
                "      <br>Currently, ERDDAP's WMS 1.3.0 supports \"XML\", \"BLANK\" (a blank image)," +
                "         and \"INIMAGE\" (the error in an image).\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>TIME=<i>time</i></td>\n" +
                "    <td align=\"center\">O</td>\n" +
                "    <td>Time value of layer desired, specified in ISO8601 format: yyyy-MM-ddTHH:mm:ssZ .\n" +
                "      <br>Currently in ERDDAP's WMS, you can only specify one time value per request.\n" +
                "      <br>In ERDDAP's WMS, the value nearest to the value you specify (if between min and max) will be used.\n" +
                "      <br>In ERDDAP's WMS, the default value is the last value in the dataset's 1D time array.\n" +
                "      <br>In ERDDAP's WMS, \"current\" is not supported.\n" +
                "      <br>To get the last value (whether it is recent or not), just don't specify a value (use the default).\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>ELEVATION=<i>elevation</i></td>\n" +
                "    <td align=\"center\">O</td>\n" +
                "    <td>Elevation of layer desired.\n" +
                "      <br>Currently in ERDDAP's WMS, you can only specify one elevation value per request.\n" +
                "      <br>In ERDDAP's WMS, this is used for the altitude dimension (if any). (in meters, positive=up)\n" +
                "      <br>In ERDDAP's WMS, the value nearest to the value you specify (if between min and max) will be used.\n" +
                "      <br>Currently in ERDDAP's WMS, the default value is the last value in the dataset's 1D altitude array.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td>dim_<i>name</i>=<i>value</i></td>\n" + //see WMS 1.3.0 spec section C.3.5
                "    <td align=\"center\">O</td>\n" +
                "    <td>Value of other dimensions as appropriate.\n" +
                "      <br>Currently in ERDDAP's WMS, you can only specify one value per dimension per request.\n" +
                "      <br>In ERDDAP's WMS, this is used for the non-time, non-altitude dimensions.\n" +
                "      <br>The name of a dimension will be \"dim_\" plus the dataset's name for the dimension, for example \"dim_model\".\n" +
                "      <br>In ERDDAP's WMS, the value nearest to the value you specify (if between min and max) will be used.\n" +
                "      <br>Currently in ERDDAP's WMS, the default value is the last value in the dimension's 1D array.\n" +
                "    </td>\n" +
                "  </tr>\n" +
                "</table>\n" +
                "<p>(Revised from Table 8 of the WMS 1.3.0 specification)\n" +
                "\n");

            //notes
            writer.write(
                "<h3><a name=\"notes\">Notes</a></h3>\n" +
                "<ul>\n" +            
                "<li><b>Grid data layers:</b> In ERDDAP's WMS, all data variables in grid datasets that use\n" +
                "  longitude and latitude dimensions are available via WMS.\n" +
                "  Each such variable is available as a WMS layer, with the name <i>datasetID</i>" + 
                    WMS_SEPARATOR + "<i>variableName</i>.\n" +
                "  Each such layer is transparent (i.e., data values are represented as a range of colors\n" +
                "    and missing values are represented by transparent pixels).\n" +
                "<li><b>Table data layers:</b> Currently in ERDDAP's WMS, data variables in table datasets are not available via WMS.\n" +
                "<li><b>Dimensions:</b> A consequence of the WMS design is that the TIME, ELEVATION, and other \n" +
                "    dimension values that you specify in a GetMap request apply to all of the layers.\n" +
                "    There is no way to specify different values for different layers.\n" +
                //"<li><b>Longitude:</b> The supported CRS values only support longitude values from -180 to 180.\n" +
                //"   <br>But some ERDDAP datasets have longitude values 0 to 360.\n" +
                //"   <br>Currently in ERDDAP's WMS, those datasets are only available from longitude 0 to 180 in WMS.\n" +
                "<li><b>Strict?</b> The table above specifies how a client should form a GetMap request.\n" +
                "   <br>In practice, ERDDAP's WMS tries to be as lenient as possible when processing GetMap requests, \n" +
                "   <br>since many current clients don't follow the specification.\n" +
                "   <br>However, if you are forming GetMap URLs, we encourage you to try to follow the specification.\n" +
                "<li><b>Why are there separate WMS servers for each dataset?</b> Because the GetCapabilities document \n" +
                "   <br>lists all values of all dimensions for each dataset, the information for each dataset \n" +
                "   <br>can be voluminous (easily 300 KB). If all the datasets (currently ~200 at the ERDDAP main site)\n" +
                "   <br>were to be included in one WMS, the resulting GetCapabilities document would be huge\n" +
                "   <br>(~60 MB) which would take a long time to download (causing many people think something was wrong\n" +
                "   <br>and give up) and would overwhelm most clients.\n" +
                //"   <br>However, a WMS server with all of this ERDDAP's datasets does exist.  You can access it at\n" +
                //"   <br>" + tErddapUrl + "/wms/request?\n" + 
                "</ul>\n");

            writer.write(
                //1.3.0 examples
                "<h2><a name=\"examples\">Examples</a></h2>\n" +
                "<p>ERDDAP is compatible with the current <b>WMS 1.3.0</b> standard.\n" +
                "<table class=\"erd\" cellspacing=\"0\">\n" +
                "  <tr>\n" +
                "    <td><b> GetCapabilities </b></td>\n" +
                "    <td><a href=\"" + tWmsGetCapabilities130 + "\">" + 
                                       tWmsGetCapabilities130 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (opaque) </td>\n" +
                "    <td><a href=\"" + tWmsOpaqueExample130 + "\">" + 
                                       tWmsOpaqueExample130 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (transparent) </td>\n" +
                "    <td><a href=\"" + tWmsTransparentExample130 + "\">" + 
                                       tWmsTransparentExample130 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td nowrap><b> In <a href=\"http://openlayers.org\">OpenLayers</a> </b></td> \n" +
                "    <td><a href=\"" + tErddapUrl + "/wms/openlayers130.html\">OpenLayers Example (WMS 1.3.0)</a></td>\n" +  
                "  </tr>\n" +
                "</table>\n" +
                "\n" +

                //1.1.1 examples
                "<br>&nbsp;\n" +
                "<p><a name=\"examples111\">ERDDAP</a> is also compatible with the older\n" +
                "<b>WMS 1.1.1</b> standard, which may be needed when working with older client software.\n" +
                "<table class=\"erd\" cellspacing=\"0\">\n" +
                "  <tr>\n" +
                "    <td><b> GetCapabilities </b></td>\n" +
                "    <td><a href=\"" + tWmsGetCapabilities111 + "\">" + 
                                       tWmsGetCapabilities111 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (opaque) </td>\n" +
                "    <td><a href=\"" + tWmsOpaqueExample111 + "\">" + 
                                       tWmsOpaqueExample111 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (transparent) </td>\n" +
                "    <td><a href=\"" + tWmsTransparentExample111 + "\">" + 
                                       tWmsTransparentExample111 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td nowrap><b> In <a href=\"http://openlayers.org\">OpenLayers</a> </b></td> \n" +
                "    <td><a href=\"" + tErddapUrl + "/wms/openlayers111.html\">OpenLayers Example (WMS 1.1.1)</a></td>\n" +  
                "  </tr>\n" +
                "</table>\n" +
                "\n" +

                //1.1.0 examples
                "<br>&nbsp;\n" +
                "<p><a name=\"examples110\">ERDDAP</a> is also compatible with the older\n" +
                "<b>WMS 1.1.0</b> standard, which may be needed when working with older client software.\n" +
                "<table class=\"erd\" cellspacing=\"0\">\n" +
                "  <tr>\n" +
                "    <td><b> GetCapabilities </b></td>\n" +
                "    <td><a href=\"" + tWmsGetCapabilities110 + "\">" + 
                                       tWmsGetCapabilities110 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (opaque) </td>\n" +
                "    <td><a href=\"" + tWmsOpaqueExample110 + "\">" + 
                                       tWmsOpaqueExample110 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td><b> GetMap </b><br> (transparent) </td>\n" +
                "    <td><a href=\"" + tWmsTransparentExample110 + "\">" + 
                                       tWmsTransparentExample110 + "</a></td>\n" +
                "  </tr>\n" +
                "  <tr>\n" +
                "    <td nowrap><b> In <a href=\"http://openlayers.org\">OpenLayers</a> </b></td> \n" +
                "    <td><a href=\"" + tErddapUrl + "/wms/openlayers110.html\">OpenLayers Example (WMS 1.1.0)</a></td>\n" +  
                "  </tr>\n" +
                "</table>\n" +
                "\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        endHtmlWriter(out, writer, tErddapUrl, false);

    }

    /**
     * Respond to WMS GetMap request for doWms.
     *
     * <p>If the request is from one dataset's wms and it's an EDDGridFromErddap, redirect to remote erddap.
     *  Currently, all requests are from one dataset's wms.
     *
     * <p>Similarly, if request if from one dataset's wms, 
     *   this method can cache results in separate dataset directories.
     *   (Which is good, because dataset's cache is emptied when dataset reloaded.)
     *   Otherwise, it uses EDStatic.fullWmsCacheDirectory.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param queryMap has name=value from the url query string.
     *    names are toLowerCase. values are original values.
     */
    public void doWmsGetMap(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, HashMap<String, String> queryMap) throws Throwable {

        String userQuery = request.getQueryString(); //post "?", still encoded, may be null
        if (userQuery == null)
            userQuery = "";
        String fileName = "wms_" + Math2.reduceHashCode(userQuery.hashCode()); //no extension

        int width = -1, height = -1, bgColori = 0xFFFFFF;
        String format = null, fileTypeName = null, exceptions = null;
        boolean transparent = false;
        OutputStreamSource outputStreamSource = null;
        OutputStream outputStream = null;
        try {

            //find mainDatasetID  (if request is from one dataset's wms)
            //Currently, all requests are from one dataset's wms.
            // http://coastwatch.pfeg.noaa.gov/erddap/wms/erdBAssta5day/request?service=.....
            String[] requestParts = String2.split(request.getRequestURI(), '/');  //post EDD.baseUrl, pre "?"
            int wmsPart = String2.indexOf(requestParts, "wms");
            String mainDatasetID = null;
            if (wmsPart >= 0 && wmsPart == requestParts.length - 3) { //it exists, and there are two more parts
                mainDatasetID = requestParts[wmsPart + 1];
                EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(mainDatasetID);
                if (eddGrid == null) {
                    mainDatasetID = null; //something else is going on, e.g., wms for all dataset's together
                } else if (!eddGrid.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
                    EDStatic.redirectToLogin(loggedInAs, response, mainDatasetID);
                    return;
                } else if (eddGrid instanceof EDDGridFromErddap) {
                    //Redirect to remote erddap if request is from one dataset's wms and it's an EDDGridFromErddap.
                    //tUrl e.g., http://coastwatch.pfeg.noaa.gov/erddap/griddap/erdBAssta5day
                    String tUrl = ((EDDGridFromErddap)eddGrid).getNextSourceErddapUrl();
                    int gPo = tUrl.indexOf("/griddap/");
                    if (gPo >= 0) {
                        String rDatasetID = tUrl.substring(gPo + "/griddap/".length()); //rDatasetID is at end of tUrl
                        tUrl = String2.replaceAll(tUrl, "/griddap/", "/wms/");
                        StringBuffer etUrl = new StringBuffer(tUrl + "/request?");

                        //change request's layers=mainDatasetID:var,mainDatasetID:var 
                        //              to layers=rDatasetID:var,rDatasetID:var
                        if (userQuery != null) {
                            String qParts[] = EDD.getUserQueryParts(userQuery); //always at least 1 part (may be "")
                            for (int qpi = 0; qpi < qParts.length; qpi++) {
                                if (qpi > 0) etUrl.append('&');

                                //this use of replaceAll is perhaps not perfect but very close...
                                //percentDecode parts, so WMS_SEPARATOR and other chars won't be encoded
                                String part = qParts[qpi]; 
                                int epo = part.indexOf('=');
                                if (epo >= 0) {
                                    part = String2.replaceAll(part, 
                                        "=" + mainDatasetID + WMS_SEPARATOR, 
                                        "=" + rDatasetID    + WMS_SEPARATOR);
                                    part = String2.replaceAll(part, 
                                        "," + mainDatasetID + WMS_SEPARATOR, 
                                        "," + rDatasetID    + WMS_SEPARATOR);
                                    //encodedKey = encodedValue
                                    etUrl.append(SSR.minimalPercentEncode(part.substring(0, epo)) + "=" +
                                                 SSR.minimalPercentEncode(part.substring(epo + 1)));
                                } else {
                                    etUrl.append(qParts[qpi]);
                                }
                            }
                        }
                        if (verbose) String2.log("doWmsGetMap redirected to\n" + etUrl.toString()); 
                        response.sendRedirect(etUrl.toString()); 
                        return;
                    } else String2.log("Internal Error: \"/griddap/\" should have been in " +
                        "EDDGridFromErddap.getNextSourceErddapUrl()=" + tUrl + " .");
                }
            }
            EDStatic.tally.add("WMS doWmsGetMap (last 24 hours)", mainDatasetID);
            EDStatic.tally.add("WMS doWmsGetMap (since startup)", mainDatasetID);
            if (mainDatasetID != null) 
                fileName = mainDatasetID + "_" + fileName;
            String cacheDir = mainDatasetID == null? EDStatic.fullWmsCacheDirectory :
                EDStatic.fullCacheDirectory + mainDatasetID + "/";            
            if (reallyVerbose) String2.log("doWmsGetMap cacheDir=" + cacheDir);

            //*** get required values   see wms spec, section 7.3.2   queryMap names are toLowerCase
            String tVersion     = queryMap.get("version");
            if (tVersion == null)
                tVersion = "1.3.0";
            if (!tVersion.equals("1.1.0") && !tVersion.equals("1.1.1") && 
                !tVersion.equals("1.3.0"))
                throw new SimpleException("Query error: VERSION=" + tVersion + 
                    " must be '1.1.0', '1.1.1', or '1.3.0'.");

            String layersCsv    = queryMap.get("layers");
            String stylesCsv    = queryMap.get("styles");
            String crs          = queryMap.get("crs");
            if (crs == null) 
                crs             = queryMap.get("srs");
            String bboxCsv      = queryMap.get("bbox");
            width               = String2.parseInt(queryMap.get("width"));
            height              = String2.parseInt(queryMap.get("height"));
            format              = queryMap.get("format");

            //optional values
            String tTransparent = queryMap.get("transparent"); 
            String tBgColor     = queryMap.get("bgcolor");
            exceptions          = queryMap.get("exceptions");
            //+ dimensions   time=, elevation=, ...=  handled below

            //*** validate parameters
            transparent = tTransparent == null? false : 
                String2.parseBoolean(tTransparent);  //e.g., "false"

            bgColori = tBgColor == null || tBgColor.length() != 8 || !tBgColor.startsWith("0x")? 
                0xFFFFFF :
                String2.parseInt(tBgColor); //e.g., "0xFFFFFF"
            if (bgColori == Integer.MAX_VALUE)
                bgColori = 0xFFFFFF;


            //*** throw exceptions related to throwing exceptions 
            //(until exception, width, height, and format are valid, fall back to XML format)

            //convert exceptions to latest format
            String oExceptions = exceptions;
            if (exceptions == null)
                exceptions = "XML";
            if      (exceptions.equals("application/vnd.ogc.se_xml"))     exceptions = "XML";
            else if (exceptions.equals("application/vnd.ogc.se_blank"))   exceptions = "BLANK";
            else if (exceptions.equals("application/vnd.ogc.se_inimage")) exceptions = "INIMAGE";
            if (!exceptions.equals("XML") && 
                !exceptions.equals("BLANK") &&
                !exceptions.equals("INIMAGE")) {
                exceptions = "XML"; //fall back
                if (tVersion.equals("1.1.0") || tVersion.equals("1.1.1"))
                    throw new SimpleException("Query error: EXCEPTIONS=" + oExceptions + 
                        " must be one of 'application/vnd.ogc.se_xml', 'application/vnd.ogc.se_blank', " +
                        "or 'application/vnd.ogc.se_inimage'.");  
                else //1.3.0+
                    throw new SimpleException("Query error: EXCEPTIONS=" + oExceptions + 
                        " must be one of 'XML', 'BLANK', or 'INIMAGE'.");  
            }

            if (width < 2 || width > WMS_MAX_WIDTH) {
                exceptions = "XML"; //fall back
                throw new SimpleException("Query error: WIDTH=" + width + 
                    " must be between 2 and " + WMS_MAX_WIDTH + ".");
            }
            if (height < 2 || height > WMS_MAX_HEIGHT) {
                exceptions = "XML"; //fall back
                throw new SimpleException("Query error: HEIGHT=" + height + 
                    " must be between 2 and " + WMS_MAX_HEIGHT + ".");
            }
            if (format == null || !format.toLowerCase().equals("image/png")) {
                exceptions = "XML"; //fall back
                throw new SimpleException("Query error: FORMAT=" + format +
                    " must be image/png.");
            }
            format = format.toLowerCase();
            fileTypeName = ".png"; 
            String extension = fileTypeName;  //here, not in other situations

            //is the image in the cache?
            if (File2.isFile(cacheDir + fileName + extension)) { //don't 'touch' it
                //write out the image
                outputStreamSource = new OutputStreamFromHttpResponse(request, response, 
                    fileName, fileTypeName, extension);
                doTransfer(request, response, cacheDir, "_wms/", 
                    fileName + extension, outputStreamSource.outputStream("")); 
                return;
            }

            //*** throw Warnings/Exceptions for other params?   (try to be lenient)
            //layers
            String layers[];
            if (layersCsv == null) {
                layers = new String[]{""};
                //it is required and so should be an Exception, 
                //but http://mapserver.refractions.net/phpwms/phpwms-cvs/ (?) doesn't send it sometimes,
                //so treat null as all defaults
                String2.log("WARNING: In the WMS query, LAYERS wasn't specified: " + userQuery);
            } else {
                layers = String2.split(layersCsv, ',');
            }
            if (layers.length > WMS_MAX_LAYERS)
                throw new SimpleException("Query error: the number of LAYERS=" + layers.length +
                    " must not be more than " + WMS_MAX_LAYERS + "."); //should be 1.., but allow 0
            //layers.length is at least 1, but it may be ""

            //Styles,  see WMS 1.3.0 section 7.2.4.6.5 and 7.3.3.4
            if (stylesCsv == null) {
                stylesCsv = "";
                //it is required and so should be an Exception, 
                //but http://mapserver.refractions.net/phpwms/phpwms-cvs/ doesn't send it,
                //so treat null as all defaults
                String2.log("WARNING: In the WMS query, STYLES wasn't specified: " + userQuery);
            }
            if (stylesCsv.length() == 0) //shorthand for all defaults
                stylesCsv = String2.makeString(',', layers.length - 1);
            String styles[] = String2.split(stylesCsv, ',');
            if (layers.length != styles.length)
                throw new SimpleException("Query error: the number of STYLES=" + styles.length +
                    " must equal the number of LAYERS=" + layers.length + ".");

            //CRS or SRS must be present  
            if (crs == null || crs.length() == 0)   //be lenient: default to CRS:84
                crs = "CRS:84";
            if (crs == null || 
                (!crs.equals("CRS:84") && !crs.equals("EPSG:4326"))) 
                throw new SimpleException("Query error: " + 
                    (tVersion.equals("1.1.0") || 
                     tVersion.equals("1.1.1")? 
                    "SRS=" + crs + " must be EPSG:4326." :
                    "SRS=" + crs + " must be EPSG:4326 or CRS:84."));

            //BBOX = minx,miny,maxx,maxy   see wms 1.3.0 spec section 7.3.3.6            
            if (bboxCsv == null || bboxCsv.length() == 0)
                throw new SimpleException("Query error: BBOX must be specified.");
                //bboxCsv = "-180,-90,180,90";  //be lenient, default to full range
            double bbox[] = String2.toDoubleArray(String2.split(bboxCsv, ','));
            if (bbox.length != 4)
                throw new SimpleException("Query error: BBOX length=" + bbox.length + " must be 4.");
            double minx = bbox[0];
            double miny = bbox[1];
            double maxx = bbox[2];
            double maxy = bbox[3];
            if (!Math2.isFinite(minx) || !Math2.isFinite(miny) ||
                !Math2.isFinite(maxx) || !Math2.isFinite(maxy))
                throw new SimpleException("Query error: invalid number in BBOX=" + bboxCsv + ".");
            if (minx >= maxx)
                throw new SimpleException("Query error: BBOX minx=" + minx + " must be < maxx=" + maxx + ".");
            if (miny >= maxy)
                throw new SimpleException("Query error: BBOX miny=" + miny + " must be < maxy=" + maxy + ".");

            //*** params are basically ok; try to make the map
            //make the image
            BufferedImage bufferedImage = new BufferedImage(width, height, 
                BufferedImage.TYPE_INT_ARGB); //I need opacity "A"
            Graphics g = bufferedImage.getGraphics(); 
            Graphics2D g2 = (Graphics2D)g;
            Color bgColor = new Color(0xFF000000 | bgColori); //0xFF000000 makes it opaque
            g.setColor(bgColor);    
            g.fillRect(0, 0, width, height);  

            //add the layers
            String roles[] = EDStatic.getRoles(loggedInAs);
            LAYER:
            for (int layeri = 0; layeri < layers.length; layeri++) {

                //***deal with non-data layers
                if (layers[layeri].equals(""))
                    continue; 
                if (layers[layeri].equals("Land") || 
                    layers[layeri].equals("LandMask") || 
                    layers[layeri].equals("Coastlines") || 
                    layers[layeri].equals("Nations")) {
                    SgtMap.makeCleanMap(minx, maxx, miny, maxy, 
                        false,
                        null, 1, 1, 0, null,
                        layers[layeri].equals("Land") || 
                        layers[layeri].equals("LandMask"), //no need to draw it twice; no distinction here
                        layers[layeri].equals("Coastlines"), 
                        layers[layeri].equals("Nations"),
                        g2, 0, 0, width, height);  
                    continue;
                }

                //*** deal with grid data
                int spo = layers[layeri].indexOf(WMS_SEPARATOR);
                if (spo <= 0 || spo >= layers[layeri].length() - 1)
                    throw new SimpleException("Query error: LAYER=" + layers[layeri] + 
                        " is invalid (invalid separator position).");
                String datasetID = layers[layeri].substring(0, spo);
                String destVar = layers[layeri].substring(spo + 1);
                EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(datasetID);
                if (eddGrid == null)
                    throw new SimpleException("Query error: LAYER=" + layers[layeri] + 
                        " is invalid (dataset not found).");
                if (!eddGrid.isAccessibleTo(roles)) { //listPrivateDatasets doesn't apply
                    EDStatic.redirectToLogin(loggedInAs, response, datasetID);
                    return;
                }
                if (!eddGrid.accessibleViaWMS())
                    throw new SimpleException("Query error: LAYER=" + layers[layeri] + 
                        " is invalid (not accessible via WMS).");
                int dvi = String2.indexOf(eddGrid.dataVariableDestinationNames(), destVar);
                if (dvi < 0)
                    throw new SimpleException("Query error: LAYER=" + layers[layeri] + 
                        " is invalid (variable not found).");
                EDV tDataVariable = eddGrid.dataVariables()[dvi];
                if (!tDataVariable.hasColorBarMinMax())
                    throw new SimpleException("Query error: LAYER=" + layers[layeri] + 
                        " is invalid (variable doesn't have valid colorBarMinimum/Maximum).");

                //style  (currently just the default)
                if (!styles[layeri].equals("") && 
                    !styles[layeri].toLowerCase().equals("default")) { //nonstandard?  but allow it
                    throw new SimpleException("Query error: for LAYER=" + layers[layeri] + 
                        ", STYLE=" + styles[layeri] + " is invalid (must be \"\").");
                }

                //get other dimension info
                EDVGridAxis ava[] = eddGrid.axisVariables();
                StringBuffer tQuery = new StringBuffer(destVar);
                for (int avi = 0; avi < ava.length; avi++) {
                    EDVGridAxis av = ava[avi];
                    if (avi == eddGrid.lonIndex()) {
                        if (maxx <= av.destinationMin() ||
                            minx >= av.destinationMax()) {
                            if (reallyVerbose) String2.log("  layer=" + layeri + " rejected because request is out of lon range.");
                            continue LAYER;
                        }
                        int first = av.destinationToClosestSourceIndex(minx);
                        int last = av.destinationToClosestSourceIndex(maxx);
                        if (first > last) {int ti = first; first = last; last = ti;}
                        int stride = DataHelper.findStride(last - first + 1, width);
                        tQuery.append("[" + first + ":" + stride + ":" + last + "]");
                        continue;
                    }

                    if (avi == eddGrid.latIndex()) {
                        if (maxy <= av.destinationMin() ||
                            miny >= av.destinationMax()) {
                            if (reallyVerbose) String2.log("  layer=" + layeri + " rejected because request is out of lat range.");
                            continue LAYER;
                        }
                        int first = av.destinationToClosestSourceIndex(miny);
                        int last = av.destinationToClosestSourceIndex(maxy);
                        if (first > last) {int ti = first; first = last; last = ti;}
                        int stride = DataHelper.findStride(last - first + 1, height);
                        tQuery.append("[" + first + ":" + stride + ":" + last + "]");
                        continue;
                    }

                    //all other axes
                    String tAvName = 
                        avi == eddGrid.altIndex()? "elevation" :
                        avi == eddGrid.timeIndex()? "time" : 
                            "dim_" + ava[avi].destinationName().toLowerCase(); //make it case-insensitive for queryMap.get
                    String tValueS = queryMap.get(tAvName);
                    if (tValueS == null)
                        //default is always the last value
                        tQuery.append("[" + (ava[avi].sourceValues().size() - 1) + "]");
                    else {
                        double tValueD = av.destinationToDouble(tValueS); //needed in particular for iso time -> epoch seconds
                        if (Double.isNaN(tValueD) ||
                            tValueD < av.destinationCoarseMin() ||
                            tValueD > av.destinationCoarseMax()) {
                            if (reallyVerbose) String2.log("  layer=" + layeri + " rejected because tValueD=" + tValueD + 
                                " for " + tAvName);
                            continue LAYER;
                        }
                        int first = av.destinationToClosestSourceIndex(tValueD);
                        tQuery.append("[" + first + "]");
                    }
                }

                //get the data
                GridDataAccessor gda = new GridDataAccessor(
                    eddGrid, 
                    "/erddap/griddap/" + datasetID + ".dods", tQuery.toString(), 
                    false, //Grid needs column-major order
                    true); //convertToNaN
                long requestNL = gda.totalIndex().size();
                int nBytesPerElement = 8;
                if (requestNL * nBytesPerElement >= Integer.MAX_VALUE)
                    throw new SimpleException("The data request is too large.");
                int requestN = (int)requestNL;
                EDStatic.ensureMemoryAvailable(requestN * nBytesPerElement, "doWmsGetMap"); 
                Grid grid = new Grid();
                grid.data = new double[requestN];
                int po = 0;
                while (gda.increment()) 
                    grid.data[po++] = gda.getDataValueAsDouble(0);
                grid.lon = gda.axisValues(eddGrid.lonIndex()).toDoubleArray();
                grid.lat = gda.axisValues(eddGrid.latIndex()).toDoubleArray(); 

                //make the palette
                //I checked hasColorBarMinMax above.
                //Note that EDV checks validity of values.
                double minData = tDataVariable.combinedAttributes().getDouble("colorBarMinimum"); 
                double maxData = tDataVariable.combinedAttributes().getDouble("colorBarMaximum"); 
                String palette = tDataVariable.combinedAttributes().getString("colorBarPalette"); 
                if (String2.indexOf(EDV.VALID_PALETTES, palette) < 0)
                    palette = Math2.almostEqual(3, -minData, maxData)? "BlueWhiteRed" : "Rainbow"; 
                boolean paletteContinuous = String2.parseBoolean( //defaults to true
                    tDataVariable.combinedAttributes().getString("colorBarContinuous")); 
                String scale = tDataVariable.combinedAttributes().getString("colorBarScale"); 
                if (String2.indexOf(EDV.VALID_SCALES, scale) < 0)
                    scale = "Linear";
                String cptFullName = CompoundColorMap.makeCPT(EDStatic.fullPaletteDirectory, 
                    palette, scale, minData, maxData, -1, paletteContinuous, 
                    EDStatic.fullCptCacheDirectory);

                //draw the data on the map
                //for now, just cartesian  -- BEWARE: it may be stretched!
                SgtMap.makeCleanMap( 
                    minx, maxx, miny, maxy, 
                    false,
                    grid, 1, 1, 0, cptFullName, 
                    false, false, false,
                    g2, 0, 0, width, height); 

            }

            //save image as file in cache dir
            //(It saves as temp file, then renames if ok.)
            SgtUtil.saveAsTransparentPng(bufferedImage, 
                transparent? bgColor : null, 
                cacheDir + fileName); 

            //copy image from file to client
            if (reallyVerbose) String2.log("  image created. copying to client: " + fileName + extension);
            outputStreamSource = new OutputStreamFromHttpResponse(request, response, 
                fileName, fileTypeName, extension);
            doTransfer(request, response, cacheDir, "_wms/", 
                fileName + extension, outputStreamSource.outputStream("")); 

        } catch (Throwable t) {
            //deal with the WMS error  
            //exceptions in this block fall to error handling in doWms

            //catch errors after the response has begun
            if (neededToSendErrorCode(request, response, t))
                return;

            if (exceptions == null)
                exceptions = "XML";

            //send INIMAGE or BLANK response   
            //see wms 1.3.0 spec sections 6.9, 6.10 and 7.3.3.11
            if ((width > 0 && width <= WMS_MAX_WIDTH &&
                 height > 0 && height <= WMS_MAX_HEIGHT &&
                 format != null) &&
                (format.equals("image/png")) &&
                (exceptions.equals("INIMAGE") || exceptions.equals("BLANK"))) {

                //since handled here 
                String msg = MustBe.getShortErrorMessage(t);
                String2.log("  doWms caught Exception (sending " + exceptions + "):\n" + 
                    MustBe.throwableToString(t)); //log full message with stack trace

                //make image
                BufferedImage bufferedImage = new BufferedImage(width, height, 
                    BufferedImage.TYPE_INT_ARGB); //I need opacity "A"
                Graphics g = bufferedImage.getGraphics(); 
                Color bgColor = new Color(0xFF000000 | bgColori); //0xFF000000 makes it opaque
                g.setColor(bgColor);
                g.fillRect(0, 0, width, height);  

                //write exception in image   (if not THERE_IS_NO_DATA)
                if (exceptions.equals("INIMAGE") &&
                    msg.indexOf(DataHelper.THERE_IS_NO_DATA) < 0) {
                    int tHeight = 12; //pixels high
                    msg = String2.noLongLines(msg, (width * 10 / 6) / tHeight, "    ");
                    String lines[] = msg.split("\\n"); //not String2.split which trims
                    g.setColor(Color.black);
                    g.setFont(new Font(EDStatic.fontFamily, Font.PLAIN, tHeight));
                    int ty = tHeight * 2;
                    for (int i = 0; i < lines.length; i++) {
                        g.drawString(lines[i], tHeight, ty);
                        ty += tHeight + 2;
                    }                    
                } //else BLANK

                //send image to client  (don't cache it)
                
                //if (format.equals("image/png")) { //currently, just .png
                    fileTypeName = ".png";
                //}
                String extension = fileTypeName;  //here, not in other situations
                if (outputStreamSource == null)
                    outputStreamSource = 
                        new OutputStreamFromHttpResponse(request, response, 
                            fileName, fileTypeName, extension);
                if (outputStream == null)
                    outputStream = outputStreamSource.outputStream("");
                SgtUtil.saveAsTransparentPng(bufferedImage, 
                    transparent? bgColor : null, 
                    outputStream); 

                return;
            } 
            
            //fall back to XML Exception in doWMS   rethrow t, so it is caught by doWms XML exception handler
            throw t;
        }

    }

    /**
     * Respond to WMS GetCapabilities request for doWms.
     * To become a Layer, a grid variable must use evenly-spaced longitude and latitude variables.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param tDatasetID  a specific dataset or "" for all datasets
     * @param queryMap should have lowercase'd names
     */
    public void doWmsGetCapabilities(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String tDatasetID, HashMap<String, String> queryMap) throws Throwable {

        //make sure version is unspecified (latest), 1.1.0, 1.1.1, or 1.3.0.
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String tVersion = queryMap.get("version");
        if (tVersion == null)
            tVersion = "1.3.0";
        if (!tVersion.equals("1.1.0") &&
            !tVersion.equals("1.1.1") &&
            !tVersion.equals("1.3.0"))
            throw new SimpleException("In an ERDDAP WMS getCapabilities query, VERSION=" + tVersion + " is not supported.\n");
        String qm = tVersion.equals("1.1.0") || 
                    tVersion.equals("1.1.1")? "" : "?";  //default for 1.3.0+
        String sc = tVersion.equals("1.1.0") || 
                    tVersion.equals("1.1.1")? "S" : "C";  //default for 1.3.0+
        EDStatic.tally.add("WMS doWmsGetCapabilities (last 24 hours)", tDatasetID);
        EDStatic.tally.add("WMS doWmsGetCapabilities (since startup)", tDatasetID);

        //return capabilities xml
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, "Capabilities", ".xml", ".xml");
        OutputStream out = outSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out, "UTF-8");
        String wmsUrl = tErddapUrl + "/wms/request";
        //see the WMS 1.1.0, 1.1.1, and 1.3.0 specification for details 
        //This based example in Annex H.
        if (tVersion.equals("1.1.0"))
            writer.write(
"<?xml version='1.0' encoding=\"UTF-8\" standalone=\"no\" ?>\n" +
"<!DOCTYPE WMT_MS_Capabilities SYSTEM\n" +
"  \"http://schemas.opengis.net/wms/1.1.0/capabilities_1_1_0.dtd\" \n" +
" [\n" +
" <!ELEMENT VendorSpecificCapabilities EMPTY>\n" +
" ]>  <!-- end of DOCTYPE declaration -->\n" +
"<WMT_MS_Capabilities version=\"1.1.0\">\n" +
"  <Service>\n" +
"    <Name>GetMap</Name>\n");  
        else if (tVersion.equals("1.1.1"))
            writer.write(
"<?xml version='1.0' encoding=\"UTF-8\" standalone=\"no\" ?>\n" +
"<!DOCTYPE WMT_MS_Capabilities SYSTEM\n" +
"  \"http://schemas.opengis.net/wms/1.1.1/capabilities_1_1_1.dtd\" \n" +
" [\n" +
" <!ELEMENT VendorSpecificCapabilities EMPTY>\n" +
" ]>  <!-- end of DOCTYPE declaration -->\n" +
"<WMT_MS_Capabilities version=\"1.1.1\">\n" +
"  <Service>\n" +
"    <Name>OGC:WMS</Name>\n");  
        else if (tVersion.equals("1.3.0"))
            writer.write(
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
//not yet supported: optional updatesequence parameter
"<WMS_Capabilities version=\"1.3.0\" xmlns=\"http://www.opengis.net/wms\"\n" +
"    xmlns:xlink=\"http://www.w3.org/1999/xlink\"\n" +
"    xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" +
"    xsi:schemaLocation=\"http://www.opengis.net/wms http://schemas.opengis.net/wms/1.3.0/capabilities_1_3_0.xsd\">\n" +
"  <Service>\n" +
"    <Name>WMS</Name>\n");

        writer.write(
"    <Title>" + XML.encodeAsXML(EDStatic.wmsTitle) + "</Title>\n" +
"    <Abstract>" + XML.encodeAsXML(EDStatic.wmsAbstract) + "</Abstract>\n" + 
"    <KeywordList>\n");
        for (int i = 0; i < EDStatic.wmsKeywords.length; i++) 
            writer.write(
"      <Keyword>" + XML.encodeAsXML(EDStatic.wmsKeywords[i]) + "</Keyword>\n");
        writer.write(
"    </KeywordList>\n" +
"    <!--Top-level address of service or service provider-->\n" +
"    <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:type=\"simple\"\n" +
"       xlink:href=\"" + tErddapUrl + "\" />\n" +
"    <ContactInformation>\n" +
"      <ContactPersonPrimary>\n" +
"        <ContactPerson>" + XML.encodeAsXML(EDStatic.adminIndividualName) + "</ContactPerson>\n" +
"        <ContactOrganization>" + XML.encodeAsXML(EDStatic.adminInstitution) + "</ContactOrganization>\n" +
"      </ContactPersonPrimary>\n" +
"      <ContactPosition>" + XML.encodeAsXML(EDStatic.adminPosition) + "</ContactPosition>\n" +
"      <ContactAddress>\n" +
"        <AddressType>postal</AddressType>\n" +
"        <Address>" + XML.encodeAsXML(EDStatic.adminAddress) + "</Address>\n" +
"        <City>" + XML.encodeAsXML(EDStatic.adminCity) + "</City>\n" +
"        <StateOrProvince>" + XML.encodeAsXML(EDStatic.adminStateOrProvince) + "</StateOrProvince>\n" +
"        <PostCode>" + XML.encodeAsXML(EDStatic.adminPostalCode) + "</PostCode>\n" +
"        <Country>" + XML.encodeAsXML(EDStatic.adminCountry) + "</Country>\n" +
"      </ContactAddress>\n" +
"      <ContactVoiceTelephone>" + XML.encodeAsXML(EDStatic.adminPhone) + "</ContactVoiceTelephone>\n" +
"      <ContactElectronicMailAddress>" + XML.encodeAsXML(EDStatic.adminEmail) + "</ContactElectronicMailAddress>\n" +
"    </ContactInformation>\n" +
"    <Fees>" + XML.encodeAsXML(EDStatic.wmsFees) + "</Fees>\n" +
"    <AccessConstraints>" + XML.encodeAsXML(EDStatic.wmsAccessConstraints) + "</AccessConstraints>\n" +

        (tVersion.equals("1.1.0") || 
         tVersion.equals("1.1.1")? "" :
"    <LayerLimit>" + WMS_MAX_LAYERS + "</LayerLimit>\n" +  
"    <MaxWidth>" + WMS_MAX_WIDTH + "</MaxWidth>\n" +  
"    <MaxHeight>" + WMS_MAX_HEIGHT + "</MaxHeight>\n") +  

"  </Service>\n");

        //Capability
        writer.write(
"  <Capability>\n" +
"    <Request>\n" +
"      <GetCapabilities>\n" +
"        <Format>" + (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? 
            "application/vnd.ogc.wms_xml" : 
            "text/xml") + 
          "</Format>\n" +
"        <DCPType>\n" +
"          <HTTP>\n" +
"            <Get>\n" +
"              <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
"                xlink:type=\"simple\" \n" +
"                xlink:href=\"" + wmsUrl + qm + "\" />\n" +
"            </Get>\n" +
"          </HTTP>\n" +
"        </DCPType>\n" +
"      </GetCapabilities>\n" +
"      <GetMap>\n" +
"        <Format>image/png</Format>\n" +
//"        <Format>image/jpeg</Format>\n" +
"        <DCPType>\n" +
"          <HTTP>\n" +
"            <Get>\n" +
"              <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
"                xlink:type=\"simple\" \n" +
"                xlink:href=\"" + wmsUrl + qm + "\" />\n" +
"            </Get>\n" +
"          </HTTP>\n" +
"        </DCPType>\n" +
"      </GetMap>\n" +
/* GetFeatureInfo is optional; not currently supported.  (1.1.0, 1.1.1 and 1.3.0 vary)
"      <GetFeatureInfo>\n" +
"        <Format>text/xml</Format>\n" +
"        <Format>text/plain</Format>\n" +
"        <Format>text/html</Format>\n" +
"        <DCPType>\n" +
"          <HTTP>\n" +
"            <Get>\n" +
"              <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
"                xlink:type=\"simple\" \n" +
"                xlink:href=\"" + wmsUrl + qm + "\" />\n" +
"            </Get>\n" +
"          </HTTP>\n" +
"        </DCPType>\n" +
"      </GetFeatureInfo>\n" +
*/
"    </Request>\n" +
"    <Exception>\n");
if (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")) 
    writer.write(
"      <Format>application/vnd.ogc.se_xml</Format>\n" +
"      <Format>application/vnd.ogc.se_inimage</Format>\n" +  
"      <Format>application/vnd.ogc.se_blank</Format>\n" +
"    </Exception>\n");
else writer.write(
"      <Format>XML</Format>\n" +
"      <Format>INIMAGE</Format>\n" +  
"      <Format>BLANK</Format>\n" +
"    </Exception>\n");

if (tVersion.equals("1.1.0") || tVersion.equals("1.1.1"))
    writer.write(
"    <VendorSpecificCapabilities />\n");

//*** start the outer layer
writer.write(
"    <Layer>\n" + 
"      <Title>" + XML.encodeAsXML(EDStatic.wmsTitle) + "</Title>\n");
//?Authority
//?huge bounding box?
//CRS   both CRS:84 and EPSG:4326 are +-180, +-90;     ???other CRSs?
//(tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? "" : "      <CRS>CRS:84</CRS>\n") +
//"      <" + sc + "RS>EPSG:4326</" + sc + "RS>\n" +

        //get the datasetID StringArray
        StringArray idsa;
        if (tDatasetID.equals("")) 
            idsa = gridDatasetIDs(true);
        else { 
            idsa = new StringArray();
            idsa.add(tDatasetID);
        }

        //*** describe a Layer for each wms-able data variable in each grid dataset
        //Elements must occur in proper sequence
        boolean firstDataset = true;
        boolean pm180 = true;
        String roles[] = EDStatic.getRoles(loggedInAs);
        for (int idi = 0; idi < idsa.size(); idi++) {
            String ids = idsa.get(idi);
            EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(ids);
            if (eddGrid == null) 
                continue; //just deleted
            if (!EDStatic.listPrivateDatasets && !eddGrid.isAccessibleTo(roles)) 
                continue;
            if (!eddGrid.accessibleViaWMS())
                continue; //no appropriate lat lon data or no WMS-able vars
            int loni = eddGrid.lonIndex();
            int lati = eddGrid.latIndex();
            EDVGridAxis avs[] = eddGrid.axisVariables();


            //EEEEK!!!! CRS:84 and EPSG:4326 want lon -180 to 180, but many erddap datasets are 0 to 360.
            //That seems to be ok.   But still limit x to -180 to 360.
            //pre 2009-02-11 was limit x to +/-180.
            double safeMinX = Math.max(-180, avs[loni].destinationMin());
            double safeMinY = Math.max( -90, avs[lati].destinationMin());
            double safeMaxX = Math.min( 360, avs[loni].destinationMax());
            double safeMaxY = Math.min(  90, avs[lati].destinationMax());

            //*** firstDataset, describe the LandMask non-data layers 
            if (firstDataset) {
                firstDataset = false;
                pm180 = idsa.size() > 1? true :  //if lots of datasets, stick to pm180
                    safeMaxX < 181; //crude
                addWmsNonDataLayer(writer, tVersion, 0, 0, pm180); 
            }

            //1 outer Layer for each dataset
            //Elements are in order of elements described in spec.
            writer.write(
       "      <Layer>\n" +
       "        <Title>" + XML.encodeAsXML(eddGrid.title()) + "</Title>\n" +

            //?optional Abstract and KeywordList

            //Style: WMS 1.3.0 section 7.2.4.6.5 says "If only a single style is available, 
            //that style is known as the "default" style and need not be advertised by the server."
            //See also 7.3.3.4.
            //I'll go with that. It's simple.
            //???separate out different palettes?
       //"      <Style>\n" +
       //         //example: Default, Transparent    features use specific colors, e.g., LightBlue, Brown
       //"        <Name>Transparent</Name>\n" +
       //"        <Title>Transparent</Title>\n" +
       //"      </Style>\n" +

       //CRS   both CRS:84 and EPSG:4326 are +-180, +-90;     ???other CRSs?

       (tVersion.equals("1.1.0")? "        <SRS>EPSG:4326</SRS>\n" : // >1? space separate them
        tVersion.equals("1.1.1")? "        <SRS>EPSG:4326</SRS>\n" : // >1? use separate tags
            "        <CRS>CRS:84</CRS>\n" +
            "        <CRS>EPSG:4326</CRS>\n") +

       (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? 
       "        <LatLonBoundingBox " +
                   "minx=\"" + safeMinX + "\" " +
                   "miny=\"" + safeMinY + "\" " +
                   "maxx=\"" + safeMaxX + "\" " +
                   "maxy=\"" + safeMaxY + "\" " +
                   "/>\n" :
       "        <EX_GeographicBoundingBox>\n" + 
                   //EEEEK!!!! CRS:84 and EPSG:4326 want lon -180 to 180, but many erddap datasets are 0 to 360.
                   //That seems to be ok.   But still limit x to -180 to 360.
                   //pre 2009-02-11 was limit x to +/-180.
       "          <westBoundLongitude>" + safeMinX + "</westBoundLongitude>\n" +
       "          <eastBoundLongitude>" + safeMaxX + "</eastBoundLongitude>\n" +
       "          <southBoundLatitude>" + safeMinY + "</southBoundLatitude>\n" +
       "          <northBoundLatitude>" + safeMaxY + "</northBoundLatitude>\n" +
       "        </EX_GeographicBoundingBox>\n") +

       "        <BoundingBox " + sc + "RS=\"EPSG:4326\" " +
                "minx=\"" + safeMinX + "\" " +
                "miny=\"" + safeMinY + "\" " +
                "maxx=\"" + safeMaxX + "\" " +
                "maxy=\"" + safeMaxY + "\" " +
                (avs[loni].isEvenlySpaced()? "resx=\"" + avs[loni].averageSpacing() + "\" " : "") +
                (avs[lati].isEvenlySpaced()? "resy=\"" + avs[lati].averageSpacing() + "\" " : "") +
                "/>\n");

            //???AuthorityURL

            //?optional MinScaleDenominator and MaxScaleDenominator

            //for 1.1.0 and 1.1.1, make a <Dimension> for each non-lat lon dimension
            // so all <Dimension> elements are together
            if (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")) {
                for (int avi = 0; avi < avs.length; avi++) {
                    if (avi == loni || avi == lati)
                        continue;
                    EDVGridAxis av = avs[avi];
                    String avName = av.destinationName();
                    String avUnits = av.units() == null? "" : av.units(); //"" is required by spec if not known (C.2)
                    //required by spec (C.2)
                    if (avi == eddGrid.timeIndex()) {
                        avName = "time";      
                        avUnits = "ISO8601"; 
                    }
                    if (avi == eddGrid.altIndex())  {
                        avName = "elevation"; 
                        //???is CRS:88 the most appropriate  (see spec 6.7.5 and B.6)
                        //"EPSG:5030" means "meters above the WGS84 ellipsoid."
                        avUnits = "EPSG:5030"; //here just 1.1.0 or 1.1.1
                    }

                    writer.write(
           "        <Dimension name=\"" + av.destinationName() + "\" " +
                        //???units are supposed to be from Unified Code for Units of Measure (UCUM)
                        "units=\"" + avUnits + "\" />\n");
                }
            }


            //the values for each non-lat lon dimension   
            //  for 1.3.0, make a <Dimension>
            //  for 1.1.0 and 1.1.1, make a <Extent> 
            for (int avi = 0; avi < avs.length; avi++) {
                if (avi == loni || avi == lati)
                    continue;
                EDVGridAxis av = avs[avi];
                String avName = av.destinationName();
                String avUnits = av.units() == null? "" : av.units(); //"" is required by spec if not known (C.2)
                String unitSymbol = "";
                //required by spec (C.2)
                if (avi == eddGrid.timeIndex()) {
                    avName = "time";      
                    avUnits = "ISO8601"; 
                }
                if (avi == eddGrid.altIndex())  {
                    avName = "elevation"; 
                    //???is CRS:88 the most appropriate  (see spec 6.7.5 and B.6)
                    //"EPSG:5030" means "meters above the WGS84 ellipsoid."
                    avUnits = tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? "EPSG:5030" : "CRS:88"; 
                    unitSymbol = "unitSymbol=\"m\" "; 
                }

                if (tVersion.equals("1.1.0")) writer.write(
       "        <Extent name=\"" + av.destinationName() + "\" ");
//???nearestValue is important --- validator doesn't like it!!! should be allowed in 1.1.0!!!
//It is described in OGC 01-047r2, section C.3
//  but if I look in 1.1.0 GetCapabilities DTD from http://schemas.opengis.net/wms/1.1.0/capabilities_1_1_0.dtd
//  and look at definition of Extent, there is no mention of multipleValues or nearestValue.
//2008-08-22 I sent email to revisions@opengis.org asking about it
//                    "multipleValues=\"0\" " +  //don't allow request for multiple values    
//                    "nearestValue=\"1\" ");   //do find nearest value                      

                else if (tVersion.equals("1.1.1")) writer.write(
       "        <Extent name=\"" + av.destinationName() + "\" " +
                    "multipleValues=\"0\" " +  //don't allow request for multiple values    
                    "nearestValue=\"1\" ");   //do find nearest value                      

                else writer.write( //1.3.0+
       "        <Dimension name=\"" + av.destinationName() + "\" " +
                    //???units are supposed to be from Unified Code for Units of Measure (UCUM)
                    "units=\"" + avUnits + "\" " +
                    unitSymbol +
                    "multipleValues=\"0\" " +  //don't allow request for multiple values    
                    "nearestValue=\"1\" ");   //do find nearest value                       

                writer.write(
                    "default=\"" + av.destinationToString(av.lastDestinationValue()) +  "\" " + //default is last value
                    //!!!currently, no support for "current" since grid av doesn't have that info to identify if relevant
                    //???or just always use last value is "current"???
                    ">");

                 //extent value(s)
                 if (avi != eddGrid.timeIndex() && av.destinationMin() == av.destinationMax()) {
                     // single numeric value  or iso time
                     writer.write(av.destinationMinString()); 
                 } else if (avi != eddGrid.timeIndex() && av.isEvenlySpaced()) {
                     //non-time min/max/spacing     
                     writer.write(av.destinationMinString() + "/" + 
                         av.destinationMaxString() + "/" + Math.abs(av.averageSpacing()));
                 //} else if (avi == eddGrid.timeIndex() && av.isEvenlySpaced()) {
                     //time min/max/spacing (time always done via iso strings)
                     //!!??? For time, express averageSpacing as ISO time interval, e.g., P1D
                     //Forming them is a little complex, so defer doing it.
                 } else {
                     //csv values   (times as iso8601)
                     writer.write(String2.toCSVString(av.destinationStringValues().toStringArray()));
                 }

                if (tVersion.equals("1.1.0") || tVersion.equals("1.1.1"))
                    writer.write("</Extent>\n");
                else //1.3.0+
                    writer.write("</Dimension>\n");
            }

            //?optional MetadataURL   needs to be in standard format (e.g., fgdc)

            writer.write(
       "        <Attribution>\n" +
       "          <Title>" + XML.encodeAsXML(eddGrid.institution()) + "</Title> \n" +
       "          <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
       "            xlink:type=\"simple\" \n" +
       "            xlink:href=\"" + XML.encodeAsXML(eddGrid.infoUrl()) + "\" />\n" +
                //LogoURL
       "        </Attribution>\n");

            //?optional Identifier and AuthorityURL
            //?optional FeatureListURL
            //?optional DataURL (tied to a MIME type)
            //?optional LegendURL

/*
            //gather all of the av destinationStringValues
            StringArray avDsv[] = new StringArray[avs.length];
            for (int avi = 0; avi < avs.length; avi++) {
                if (avi == loni || avi == lati)
                    continue;
                avDsv[avi] = avs[avi].destinationStringValues();
            }

*/

            //an inner Layer for each dataVariable
            String dvNames[] = eddGrid.dataVariableDestinationNames();
            for (int dvi = 0; dvi < dvNames.length; dvi++) {
                if (!eddGrid.dataVariables()[dvi].hasColorBarMinMax())
                    continue;
                writer.write(
       "        <Layer opaque=\"1\" >\n" + //see 7.2.4.7.1  use opaque for grid data, non for table data
       "          <Name>" + XML.encodeAsXML(ids + WMS_SEPARATOR + dvNames[dvi]) + "</Name>\n" +
       "          <Title>" + XML.encodeAsXML(eddGrid.title() + " - " + dvNames[dvi]) + "</Title>\n");
/*

                //make a sublayer for each index combination  !!!???          
                NDimensionalIndex ndi = new NDimensionalIndex( shape[]);
                int current[] = ndi.getCurrent();
                StringBuffer dims = new StringBuffer();
                while (ndi.increment()) {
                    //make the dims string, e.g., !time:2006-08-23T12:00:00Z!elevation:0
                    dims.setLength(0);
                    for (int avi = 0; avi < avs.length; avi++) {
                        if (avi == loni || avi == lati)
                            continue;
                        dims.append(WMS_SEPARATOR +  ...currentavDsv[avi] = avs[avi].destinationStringValues();
                    }
                    writer.write
                        "<Layer opaque=\"1\" >\n" + //see 7.2.4.7.1  use opaque for grid data, non for table data
                        "  <Name>" + XML.encodeAsXML(ids + WMS_SEPARATOR + dvNames[dvi]) + dims + "</Name>\n" +
                        "  <Title>" + XML.encodeAsXML(eddGrid.title() + " - " + dvNames[dvi]) + dims + "</Title>\n");
                        "        </Layer>\n");
 
                }

*/
                writer.write(
       "        </Layer>\n");
            }

            //end of the dataset's layer
            writer.write(
       "      </Layer>\n");        
        }

        //*** describe the non-data layers   Land, Coastlines, Nations
        addWmsNonDataLayer(writer, tVersion, 0, 2, pm180); 

        //*** end of the outer layer
        writer.write(
       "    </Layer>\n");        

        writer.write(
       "  </Capability>\n" +
       (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? 
            "</WMT_MS_Capabilities>\n" : 
            "</WMS_Capabilities>\n"));

        //essential
        writer.flush();
        if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
        out.close(); 
    }


    /** 
     * Add a non-data layer to the writer's GetCapabilities:  Land/LandMask, Coastlines, Nations
     */
    private static void addWmsNonDataLayer(Writer writer, String tVersion, 
        int first, int last, boolean pm180) throws Throwable {

//Elements must occur in proper sequence
        String firstName = first == last && first == 0? "Land" : "LandMask";
        String sc = tVersion.equals("1.1.0") || 
                    tVersion.equals("1.1.1")? "S" : "C";  //default for 1.3.0+
        double safeMinX = pm180? -180 : 0;
        double safeMaxX = pm180?  180 : 360;

        for (int layeri = first; layeri <= last; layeri++) {
            writer.write(
"      <Layer" +
     (tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? "" : 
         " opaque=\"" + (layeri == 0? 1 : 0) + "\"") + //see 7.2.4.7.1  use opaque for coverages
     ">\n" + 
"        <Name>" + (layeri == 0? firstName : layeri == 1? "Coastlines" : "Nations") + "</Name>\n" +
"        <Title>" + (layeri == 0? firstName : layeri == 1? "Coastlines" : "National Boundaries") + "</Title>\n" +
//?optional Abstract and KeywordList
//don't have to define style if just one

//CRS   both CRS:84 and EPSG:4326 are +-180, +-90;     ???other CRSs?
(tVersion.equals("1.1.0")? "        <SRS>EPSG:4326</SRS>\n" : // >1? space separate them
 tVersion.equals("1.1.1")? "        <SRS>EPSG:4326</SRS>\n" : // >1? use separate tags
     "        <CRS>CRS:84</CRS>\n" +
     "        <CRS>EPSG:4326</CRS>\n") +

(tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? 
"        <LatLonBoundingBox minx=\"" + safeMinX + "\" miny=\"-90\" maxx=\"" + safeMaxX + "\" maxy=\"90\" />\n" :

"        <EX_GeographicBoundingBox>\n" + 
"          <westBoundLongitude>" + safeMinX + "</westBoundLongitude>\n" +
"          <eastBoundLongitude>" + safeMaxX + "</eastBoundLongitude>\n" +
"          <southBoundLatitude>-90</southBoundLatitude>\n" +
"          <northBoundLatitude>90</northBoundLatitude>\n" +
"        </EX_GeographicBoundingBox>\n") +

"        <BoundingBox " + sc + "RS=\"EPSG:4326\" minx=\"" + safeMinX + 
      "\" miny=\"-90\" maxx=\"" + safeMaxX + "\" maxy=\"90\" />\n" +

//???AuthorityURL
//?optional MinScaleDenominator and MaxScaleDenominator
//?optional MetadataURL   needs to be in standard format (e.g., fgdc)

"        <Attribution>\n" +
"          <Title>" + XML.encodeAsXML(
    layeri < 2? "NOAA NGDC GSHHS" : "pscoast in GMT"
    ) + "</Title> \n" +
"          <OnlineResource xmlns:xlink=\"http://www.w3.org/1999/xlink\" \n" +
"            xlink:type=\"simple\" \n" +
"            xlink:href=\"" + XML.encodeAsXML(
    layeri < 2? "http://www.ngdc.noaa.gov/mgg/shorelines/gshhs.html" : 
                "http://gmt.soest.hawaii.edu/"
    ) + "\" />\n" +
         //LogoURL
"        </Attribution>\n" +

//?optional Identifier and AuthorityURL
//?optional FeatureListURL
//?optional DataURL (tied to a MIME type)
//?optional LegendURL

"      </Layer>\n");        
        }
    }


    /**
     * This responds by sending out the /wms/datasetID/index.html (or 111 or 130) page.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param tVersion the WMS version to use: "1.1.0", "1.1.1" or "1.3.0"
     * @param tDatasetID  currently must be an EDDGrid datasetID, e.g., erdBAssta5day   
     */
    public void doWmsOpenLayers(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String tVersion, String tDatasetID) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        if (!tVersion.equals("1.1.0") &&
            !tVersion.equals("1.1.1") &&
            !tVersion.equals("1.3.0"))
            throw new SimpleException("WMS version=" + tVersion + " must be " +
                "1.1.0, 1.1.1, or 1.3.0.");            
        EDStatic.tally.add("WMS doWmsOpenLayers (last 24 hours)", tDatasetID);
        EDStatic.tally.add("WMS doWmsOpenLayers (since startup)", tDatasetID);

        String csrs = tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? "srs" : "crs";
        String exceptions = tVersion.equals("1.1.0") || tVersion.equals("1.1.1")? 
            "" :  //default is ok for 1.1.0 and 1.1.1
            "exceptions:'INIMAGE', "; 

        EDDGrid eddGrid = (EDDGrid)gridDatasetHashMap.get(tDatasetID);
        if (eddGrid == null)
            sendResourceNotFoundError(request, response, 
                "Currently, datasetID=" + tDatasetID + " isn't available.");
        if (!eddGrid.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
            EDStatic.redirectToLogin(loggedInAs, response, tDatasetID);
            return;
        }
        int loni = eddGrid.lonIndex();
        int lati = eddGrid.latIndex();
        int alti = eddGrid.altIndex();
        int timei = eddGrid.timeIndex();
        if (loni < 0 || lati < 0) 
            throw new SimpleException("datasetID=" + tDatasetID + 
                " doesn't have longitude and latitude dimensions.");            
        if (!eddGrid.accessibleViaWMS())
            throw new SimpleException("datasetID=" + tDatasetID + 
                " isn't accessible via WMS (no lon/lat variables or colorBar attributes?).");            

        EDVGridAxis gaa[] = eddGrid.axisVariables();
        EDV dva[] = eddGrid.dataVariables();
        String options[][] = new String[gaa.length][];
        String tgaNames[] = new String[gaa.length];
        for (int gai = 0; gai < gaa.length; gai++) {
            if (gai == loni || gai == lati)
                continue;
            options[gai] = gaa[gai].destinationStringValues().toStringArray();
            tgaNames[gai] = gai == alti? "elevation" :
                gai == timei? "time" : "dim_" + gaa[gai].destinationName();
        }
        String baseUrl = tErddapUrl + "/wms/" + tDatasetID;
        String requestUrl = baseUrl + "/request";
      
        String varNames[] = eddGrid.dataVariableDestinationNames();
        int nVars = varNames.length;

        double minX = gaa[loni].destinationMin();
        double maxX = gaa[loni].destinationMax();
        double minY = gaa[lati].destinationMin();
        double maxY = gaa[lati].destinationMax();
        double xRange = maxX - minX;
        double yRange = maxY - minY;
        double centerX = (minX + maxX) / 2;
        boolean pm180 = centerX < 90;
        StringBuffer scripts = new StringBuffer();
        scripts.append(
            //documentation http://dev.openlayers.org/releases/OpenLayers-2.6/doc/apidocs/files/OpenLayers-js.html
            //I tried to get http://www.openlayers.org/api/OpenLayers.js 
            //   and re-serve it so Expires header can be added by ERDDAP,
            //   but it didn't find images for controls.
            "<script type=\"text/javascript\" src=\"http://www.openlayers.org/api/OpenLayers.js\"></script>\n" +
            "<script type=\"text/javascript\">\n" +
            "  var map; var vLayer=new Array();\n" +
            "  function init(){\n" +
            "    var options = {\n" +
            "      minResolution: \"auto\",\n" +
            "      minExtent: new OpenLayers.Bounds(-1, -1, 1, 1),\n" +
            "      maxResolution: \"auto\",\n" +
            "      maxExtent: new OpenLayers.Bounds(" + 
                //put buffer space around data
                Math.max(pm180? -180 : 0,  minX - xRange/8) + ", " + 
                Math.max(-90,              minY - yRange/8) + ", " +
                Math.min(pm180? 180 : 360, maxX + xRange/8) + ", " + 
                Math.min( 90,              maxY + yRange/8) + ") };\n" +
            "    map = new OpenLayers.Map('map', options);\n" +
            "\n" +
            //"    var ol_wms = new OpenLayers.Layer.WMS( \"OpenLayers WMS\",\n" +
            //"        \"http://labs.metacarta.com/wms/vmap0?\", {layers: 'basic'} );\n" +
            //e.g., ?service=WMS&version=1.3.0&request=GetMap&bbox=0,-75,180,75&crs=EPSG:4326&width=360&height=300
            //    &bgcolor=0x808080&layers=Land,erdBAssta5day:sst,Coastlines,Nations&styles=&format=image/png
            //"    <!-- for OpenLayers, always use 'srs'; never 'crs' -->\n" +
            "    var Land = new OpenLayers.Layer.WMS( \"Land\",\n" +
            "        \"" + requestUrl + "?\", \n" +
            "        {" + exceptions + "version:'" + tVersion + "', srs:'EPSG:4326', \n" +
            "          layers:'Land', bgcolor:'0x808080', format:'image/png'} );\n" +
            //2009-06-22 this isn't working, see http://onearth.jpl.nasa.gov/
            //  Their server is overwhelmed. They added an extension to wms for tiled images.
            //  But I can't make OpenLayers support extensions.
            //  For now, remove it.
            //"\n" +
            //"    var jplLayer = new OpenLayers.Layer.WMS( \"NASA Global Mosaic\",\n" +
            //"        \"http://t1.hypercube.telascience.org/cgi-bin/landsat7\", \n" +
            //"        {layers: \"landsat7\"});\n" +
            //"    jplLayer.setVisibility(false);\n" +
            "\n");

        int nLayers = 0;
        for (int dv = 0; dv < nVars; dv++) {
            if (!dva[dv].hasColorBarMinMax())
                continue;
            scripts.append(
            //"        ia_wms = new OpenLayers.Layer.WMS(\"Nexrad\"," +
            //    "\"http://mesonet.agron.iastate.edu/cgi-bin/wms/nexrad/n0r.cgi?\",{layers:\"nexrad-n0r-wmst\"," +
            //    "transparent:true,format:'image/png',time:\"2005-08-29T13:00:00Z\"});\n" +
            "    vLayer[" + nLayers + "] = new OpenLayers.Layer.WMS( \"" + 
                         tDatasetID + WMS_SEPARATOR + varNames[dv] + "\",\n" +
            "        \"" + requestUrl + "?\", \n" +
            "        {" + exceptions + "version:'" + tVersion + "', srs:'EPSG:4326', " +
                         "layers:'" + tDatasetID + WMS_SEPARATOR + varNames[dv] + "', \n" +
            "        ");
            for (int gai = 0; gai < gaa.length; gai++) {
                if (gai == loni || gai == lati)
                    continue;
                scripts.append(
                    tgaNames[gai] + ":'" + 
                    options[gai][options[gai].length - 1] + "', "); //!!!trouble if internal '
            }
            scripts.append("\n" +
            "        transparent:'true', bgcolor:'0x808080', format:'image/png'} );\n" +
            "    vLayer[" + nLayers + "].isBaseLayer=false;\n" +
            "    vLayer[" + nLayers + "].setVisibility(" + (nLayers == 0) + ");\n" +
            "\n");
            nLayers++;
        }

        scripts.append(
            "    var LandMask = new OpenLayers.Layer.WMS( \"Land Mask\",\n" +
            "        \"" + requestUrl + "?\", \n" +
            "        {" + exceptions + "version:'" + tVersion + "', srs:'EPSG:4326', layers:'LandMask', \n" +
            "          bgcolor:'0x808080', format:'image/png', transparent:'true'} );\n" +
            "    LandMask.isBaseLayer=false;\n" +
            "    LandMask.setVisibility(false);\n" +
            "\n" +
            "    var Coastlines = new OpenLayers.Layer.WMS( \"Coastlines\",\n" +
            "        \"" + requestUrl + "?\", \n" +
            "        {" + exceptions + "version:'" + tVersion + "', srs:'EPSG:4326', layers:'Coastlines', \n" +
            "          bgcolor:'0x808080', format:'image/png', transparent:'true'} );\n" +
            "    Coastlines.isBaseLayer=false;\n" +
            "\n" +
            "    var Nations = new OpenLayers.Layer.WMS( \"National Boundaries\",\n" +
            "        \"" + requestUrl + "?\", \n" +
            "        {" + exceptions + "version:'" + tVersion + "', srs:'EPSG:4326', layers:'Nations', \n" +
            "          bgcolor:'0x808080', format:'image/png', transparent:'true'} );\n" +
            "    Nations.isBaseLayer=false;\n" +
            "\n");

        scripts.append(
            "    map.addLayers([Land"); //, jplLayer");
        for (int v = 0; v < nLayers; v++) 
            scripts.append(", vLayer[" + v + "]");

        scripts.append(", LandMask, Coastlines, Nations]);\n" +  
            "    map.addControl(new OpenLayers.Control.LayerSwitcher());\n" +
            "    map.zoomToMaxExtent();\n" +
            "  }\n");

        for (int gai = 0; gai < gaa.length; gai++) {
            if (gai == loni || gai == lati || options[gai].length <= 1)
                continue;
            scripts.append(
            "  function update" + tgaNames[gai] + "() {\n" +
            "    t = document.f1." + tgaNames[gai] + ".options[document.f1." + tgaNames[gai] + ".selectedIndex].text; \n" +            
            "    for (v=0; v<" + nLayers + "; v++)\n" + 
            "      vLayer[v].mergeNewParams({'" + tgaNames[gai] + "':t});\n" +
            "  }\n");
        }

        scripts.append(
            "</script>\n");

        //*** html head
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = new OutputStreamWriter(out);
        writer.write(EDStatic.startHeadHtml(tErddapUrl, "WMS for " + eddGrid.title()));
        writer.write(EDStatic.standardHead);
        writer.write("\n" + eddGrid.rssHeadLink());
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
        writer.write(
            "</head>\n");

        //*** html body
        String tBody = String2.replaceAll(EDStatic.startBodyHtml(loggedInAs), "<body", "<body onLoad=\"init()\"");
        String makeAGraphRef = "<a href=\"" + tErddapUrl + "/griddap/" + tDatasetID + ".graph\">Make A Graph</a>";
        writer.write(
            tBody + "\n" +
            HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl) +
            EDStatic.youAreHere(tErddapUrl, "wms", tDatasetID));
        try {
            String queryString = request.getQueryString();
            if (queryString == null)
                queryString = "";
            eddGrid.writeHtmlDatasetInfo(loggedInAs, writer, true, true, queryString, "");
            writer.write(
                "<p>\n" +
                "<form name=\"f1\" action=\"\">\n" +
                "<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
                "  <tr>\n" +
                "    <td colspan=\"2\">This web page is using <a href=\"http://openlayers.org\">OpenLayers</a> \n" +  
                "        to display maps which are created\n" +
                "        on-the-fly by ERDDAP's Web Map Server (WMS) version " + tVersion + ".\n" +
                "      <br>The controls on the left of the map are for navigation.\n" +
                "        The control on the right manages the layers.\n" +
                "      <br>You can select different values for the data variable's dimension(s):</td>\n" +
                "  </tr>\n" 
                );

            //a select widget for each axis (but not for lon or lat)
            for (int gai = 0; gai < gaa.length; gai++) {
                if (gai == loni || gai == lati)
                    continue;
                int olmo = options[gai].length - 1;
                writer.write(   
                "  <tr align=\"left\">\n" +
                "    <td>" + gaa[gai].destinationName() + ":&nbsp;</td>\n" + 
                "    <td width=\"95%\" align=\"left\">");

                //one value: display it
                if (olmo <= 0) {
                    writer.write(
                        options[gai][olmo] + //numeric or time so don't need XML.encodeAsXML
                        "</td>\n");
                    continue;
                }

                //many values: select
                writer.write(
                    "<select name=\"" + tgaNames[gai] + "\" size=\"1\" title=\"\" " +
                    "onChange=\"update" + tgaNames[gai] + "()\" >\n");
                for (int i = 0; i <= olmo; i++)
                    writer.write("<option" + 
                        (i == olmo? " selected=\"selected\"" : "") +
                        ">" + options[gai][i] + "</option>\n"); //numeric or time so don't need XML.encodeAsXML
                writer.write(
                "</select>\n");

                String axisSelectedIndex = "document.f1." + tgaNames[gai] + ".selectedIndex"; //var name can't have internal "." or "-"
                writer.write(
                "<img src=\"" + EDStatic.imageDirUrl + "arrowLL.gif\"  \n" +
                "  title=\"Select the first item.\"   alt=\"&larr;\" \n" +
                "  onMouseUp=\"" + axisSelectedIndex + "=0; update" + tgaNames[gai] + "();\" >\n" +
                "<img src=\"" + EDStatic.imageDirUrl + "minus.gif\"\n" +
                "  title=\"Select the previous item.\"   alt=\"-\" \n" +
                "  onMouseUp=\"" + axisSelectedIndex + "=Math.max(0, " +
                   axisSelectedIndex + "-1); update" + tgaNames[gai] + "();\" >\n" +
                "<img src=\"" + EDStatic.imageDirUrl + "plus.gif\"  \n" +
                "  title=\"Select the next item.\"   alt=\"+\" \n" +
                "  onMouseUp=\"" + axisSelectedIndex + "=Math.min(" + olmo + 
                   ", " + axisSelectedIndex + "+1); update" + tgaNames[gai] + "();\" >\n" +
                "<img src=\"" + EDStatic.imageDirUrl + "arrowRR.gif\" \n" +
                "  title=\"Select the last item.\"   alt=\"&rarr;\" \n" +
                "  onMouseUp=\"" + axisSelectedIndex + "=" + olmo + "; update" + tgaNames[gai] + "();\" >\n");

                writer.write(
                "    </td>\n" +
                "  </tr>\n");
            } //end of gai loop

            writer.write(
                "</table>\n" +
                "</form>\n" +
                "\n" +
                "<p><div style=\"width:600px; height:300px\" id=\"map\"></div>\n" +
                "\n");
            writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better

            //*** What is WMS?
            String e0 = tErddapUrl + "/wms/" + EDStatic.wmsSampleDatasetID + "/request?";
            String ec = "service=WMS&amp;request=GetCapabilities&amp;version=";
            String e1 = "service=WMS&amp;version="; 
            //this section of code is in 2 places
            int bbox[] = String2.toIntArray(String2.split(EDStatic.wmsSampleBBox, ',')); 
            int tHeight = Math2.roundToInt(((bbox[3] - bbox[1]) * 360) / Math.max(1, bbox[2] - bbox[0]));
            tHeight = Math2.minMaxDef(10, 600, 180, tHeight);
            String e2 = "&amp;request=GetMap&amp;bbox=" + EDStatic.wmsSampleBBox +
                        "&amp;" + csrs + "=EPSG:4326&amp;width=360&amp;height=" + tHeight + 
                        "&amp;bgcolor=0x808080&amp;layers=";
            //Land,erdBAssta5day:sst,Coastlines,Nations
            String e3 = EDStatic.wmsSampleDatasetID + WMS_SEPARATOR + EDStatic.wmsSampleVariable;
            String e4 = "&amp;styles=&amp;format=image/png";
            String et = "&amp;transparent=TRUE";

            String tWmsOpaqueExample      = e0 + e1 + tVersion + e2 + "Land," + e3 + ",Coastlines,Nations" + e4;
            String tWmsTransparentExample = e0 + e1 + tVersion + e2 +           e3 + e4 + et;
            String datasetListRef = 
                "<br>See the\n" +
                "  <a href=\"" + tErddapUrl + "/wms/index.html\">list \n" +
                "    of datasets available via WMS</a> at this ERDDAP installation.\n";
            String makeAGraphListRef =
                "  <br>See the\n" +
                "    <a href=\"" + tErddapUrl + "/info/index.html\">list \n" +
                "      of datasets with Make A Graph</a> at this ERDDAP installation.\n";

            //What is WMS?   (for tDatasetID) 
            //!!!see the almost identical documentation above
            writer.write(
                "<h2><a name=\"description\">What</a> is WMS?</h2>\n" +
                EDStatic.wmsLongDescriptionHtml + "\n" +
                datasetListRef +
                "\n" +
                "<h2>Three Ways to Make Maps with WMS</h2>\n" +
                "<ol>\n" +
                "<li> <b>In theory, anyone can download, install, and use WMS client software.</b>\n" +
                "  <br>Some clients are: \n" +
                "    <a href=\"http://www.esri.com/software/arcgis/\">ArcGIS</a>,\n" +
                "    <a href=\"http://mapserver.refractions.net/phpwms/phpwms-cvs/\">Refractions PHP WMS Client</a>, and\n" +
                "    <a href=\"http://udig.refractions.net//\">uDig</a>. \n" +
                "  <br>To make a client work, you would install the software on your computer.\n" +
                "  <br>Then, you would enter the URL of the WMS service into the client.\n" +
                "  <br>For example, in ArcGIS (not yet fully working because it doesn't handle time!), use\n" +
                "  <br>\"Arc Catalog : Add Service : Arc Catalog Servers Folder : GIS Servers : Add WMS Server\".\n" +
                "  <br>In ERDDAP, this dataset has its own WMS service, which is located at\n" +
                "  <br>&nbsp; &nbsp; <b>" + tErddapUrl + "/wms/" + tDatasetID + "/request?</b>\n" +  
                "  <br>(Some WMS client programs don't want the <b>?</b> at the end of that URL.)\n" +
                datasetListRef +
                "  <p><b>In practice,</b> we haven't found any WMS clients that properly handle dimensions\n" +
                "  <br>other than longitude and latitude (e.g., time), a feature which is specified by the WMS\n" +
                "  <br>specification and which is utilized by most datasets in ERDDAP's WMS servers.\n" +
                "  <br>You may find that using\n" +
                makeAGraphRef + "\n" +
                "    and selecting the .kml file type (an OGC standard)\n" +
                "  <br>to load images into <a href=\"http://earth.google.com/\">Google Earth</a> provides\n" +            
                "     a good (non-WMS) map client.\n" +
                makeAGraphListRef +
                "  <br>&nbsp;\n" +
                "<li> <b>Web page authors can embed a WMS client in a web page.</b>\n" +
                "  <br>For the map above, ERDDAP is using \n" +
                "    <a href=\"http://openlayers.org\">OpenLayers</a>, \n" +  
                "    which is a very versatile WMS client.\n" +
                "  <br>OpenLayers doesn't automatically deal with dimensions\n" +
                "    other than longitude and latitude (e.g., time),\n" +            
                "  <br>so you will have to write JavaScript (or other scripting code) to do that.\n" +
                "  <br>(Adventurous JavaScript programmers can look at the Souce Code for this web page.)\n" + 
                "  <br>&nbsp;\n" +
                "<li> <b>A person with a browser or a computer program can generate special GetMap URLs\n" +
                "  and view/use the resulting image file.</b>\n" +
                "  <br><b>Opaque example:</b> <a href=\"" + tWmsOpaqueExample + "\">" + 
                                                            tWmsOpaqueExample + "</a>\n" +
                "  <br><b>Transparent example:</b> <a href=\"" + tWmsTransparentExample + "\">" + 
                                                                 tWmsTransparentExample + "</a>\n" +
                datasetListRef +
                "  <br><b>For more information, see ERDDAP's \n" +
                "    <a href=\"" +tErddapUrl + "/wms/documentation.html\">WMS Documentation</a> .</b>\n" +
                "  <p><b>In practice, it is probably easier and more versatile to use this dataset's\n" +
                "    " + makeAGraphRef + " form</b>\n" +
                "  <br>than to use WMS for this purpose.\n" +
                makeAGraphListRef +
                "</ol>\n" +
                "\n");
            
            //"<p>\"<a href=\"http://openlayers.org\">OpenLayers</a> makes it easy to put a dynamic map in any web page.\n" +
            //    "It can display map tiles and markers loaded from any source.\n" +
            //    "OpenLayers is completely free, Open Source JavaScript, released under a BSD-style License. ...\n" +
            //    "OpenLayers is a pure JavaScript library for displaying map data in most modern web browsers,\n" +
            //    "with no server-side dependencies. OpenLayers implements a (still-developing) JavaScript API\n" +
            //    "for building rich web-based geographic applications, similar to the Google Maps and \n" +
            //    "MSN Virtual Earth APIs, with one important difference -- OpenLayers is Free Software, \n" +
            //    "developed for and by the Open Source software community.\" (from the OpenLayers website) \n");

            writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
            writer.write(scripts.toString());
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        endHtmlWriter(out, writer, tErddapUrl, false);

    }

    /** 
     * This responds to a user's requst for a file in the (psuedo)'protocol' (e.g., images) 
     * directory.
     * This works with files in subdirectories of 'protocol'.
     * <p>The problem is that web.xml transfers all requests to [url]/erddap/*
     * to this servlet. There is no way to allow requests for files in e.g. /images
     * to be handled by Tomcat. So handle them here by doing a simple file transfer.
     *
     * @param protocol here is 'download', 'images', or 'public'
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (e.g., "images") in the requestUrl
     */
    public void doTransfer(HttpServletRequest request, HttpServletResponse response,
        String protocol, int datasetIDStartsAt) throws Throwable {

        String requestUrl = request.getRequestURI();  // /erddap/images/QuestionMark.jpg
        String dir = EDStatic.contextDirectory + protocol + "/";
        String fileNameAndExt = requestUrl.length() <= datasetIDStartsAt? "" : 
            requestUrl.substring(datasetIDStartsAt);
        if (!File2.isFile(dir + fileNameAndExt)) {
            sendResourceNotFoundError(request, response, "");
            return;
        }
        String ext = File2.getExtension(fileNameAndExt);
        String fileName = fileNameAndExt.substring(0, fileNameAndExt.length() - ext.length()); 
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, fileName, ext, ext); 
        //characterEncoding not relevant for binary files
        String charEncoding = 
            ext.equals(".asc") || ext.equals(".csv") || 
            ext.equals(".htm") || ext.equals(".html") || 
            ext.equals(".js")  || ext.equals(".json") || ext.equals(".kml") || 
            ext.equals(".pdf") || ext.equals(".tsv") || 
            ext.equals(".txt") || ext.equals(".xml")? 
            "UTF-8" : //an assumption, the most universal solution
            "";

        //Set expires header for things that don't change often.
        //See "High Performance Web Sites" Steve Souders, Ch 3.
        if (protocol.equals("images")) { 
            //&& fileName.indexOf('/') == -1) {   //file not in a subdirectory
            //&& (ext.equals(".gif") || ext.equals(".jpg") || ext.equals(".js") || ext.equals(".png"))) {
            
            GregorianCalendar gc = Calendar2.newGCalendarZulu();
            int nDays = 7; //one week gets most of benefit and few problems
            gc.add(Calendar2.DATE, nDays); 
            String expires = Calendar2.formatAsRFC822GMT(gc);
            if (reallyVerbose) String2.log("  setting expires=" + expires + " header");
     		response.setHeader("Cache-Control", "PUBLIC, max-age=" + 
                (nDays * Calendar2.SECONDS_PER_DAY) + ", must-revalidate");
			response.setHeader("Expires", expires);
        }

        doTransfer(request, response, dir, protocol + "/", fileNameAndExt, 
            outSource.outputStream(charEncoding)); 
    }

    /** 
     * This is the lower level version of doTransfer.
     *
     * @param localDir the actual hard disk directory, ending in '/'
     * @param webDir the apparent directory, ending in '/' (e.g., "public/"),
     *    for error message only
     * @param fileNameAndExt e.g., wms_29847362839.png
     *    (although it can be e.g., /subdir/wms_29847362839.png)
     */
    public void doTransfer(HttpServletRequest request, HttpServletResponse response,
            String localDir, String webDir, String fileNameAndExt, 
            OutputStream outputStream) throws Throwable {
        if (verbose) String2.log("doTransfer " + localDir + fileNameAndExt);

        //To deal with problems in multithreaded apps 
        //(when deleting and renaming files, for an instant no file with that name exists),
        int maxAttempts = 3;
        for (int attempt = 1; attempt <= maxAttempts; attempt++) {
            if (File2.isFile(localDir + fileNameAndExt)) {                
                //ok, copy it
                File2.copy(localDir + fileNameAndExt, outputStream);
                outputStream.close();
                return;
            }

            String2.log("WARNING #" + attempt + 
                ": ERDDAP.doTransfer is having trouble. It will try again to transfer " + 
                localDir + fileNameAndExt);
            if (attempt == maxAttempts) {
                //failure
                String2.log("Error: Unable to transfer " + localDir + fileNameAndExt); //localDir
                throw new SimpleException("Error: Unable to transfer " + webDir + fileNameAndExt); //webDir
            } else if (attempt == 1) {
                Math2.gc(1000);  //in File2.delete: gc works better than sleep
            } else {
                Math2.sleep(1000);  //but no need to call gc more than once
            }
        }

    }

    /** 
     * This responds to a user's requst for an rss feed.
     * <br>The login/authentication system does not apply to RSS.
     * <br>The RSS information is always available, for all datasets, to anyone.
     * <br>(This is not ideal.  But otherwise, a user would need to be logged in 
     *   all of the time so that the RSS reader could read the information.)
     * <br>But, since private datasets that aren't accessible aren't advertised, their RSS links are not advertised either.
     *
     * @param protocol here is always 'rss'
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *   in the requestUrl
     */
    public void doRss(HttpServletRequest request, HttpServletResponse response,
        String protocol, int datasetIDStartsAt) throws Throwable {

        String requestUrl = request.getRequestURI();  // /erddap/images/QuestionMark.jpg
        String nameAndExt = requestUrl.length() <= datasetIDStartsAt? "" : 
            requestUrl.substring(datasetIDStartsAt); //should be <datasetID>.rss
        if (!nameAndExt.endsWith(".rss")) {
            sendResourceNotFoundError(request, response, "Invalid name");
            return;
        }
        String name = nameAndExt.substring(0, nameAndExt.length() - 4);
        EDStatic.tally.add("RSS (last 24 hours)", name);
        EDStatic.tally.add("RSS (since startup)", name);

        byte rssAr[] = name.length() == 0? null : (byte[])rssHashMap.get(name);
        if (rssAr == null) {
            sendResourceNotFoundError(request, response, "Currently, there is no RSS feed for that name");
            return;
        }
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, name, ".rss", ".rss"); 
        OutputStream outputStream = outSource.outputStream("UTF-8"); 
        outputStream.write(rssAr);
        outputStream.close();
    }


    /**
     * This responds to a setDatasetFlag.txt request.
     *
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doSetDatasetFlag(HttpServletRequest request, HttpServletResponse response, 
        String userQuery) throws Throwable {
        //see also EDD.flagUrl()

        //generate text response
        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, "setDatasetFlag", ".txt", ".txt");
        OutputStream out = outSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out); 

        //look at the request
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //false so names are case insensitive
        String datasetID = queryMap.get("datasetid"); //lowercase name
        String flagKey = queryMap.get("flagkey"); //lowercase name
        //isFileNameSafe is doubly useful: it ensures datasetID could be a dataseID, 
        //  and it ensures file of this name can be created
        String message;
        if (datasetID == null || datasetID.length() == 0 ||
            flagKey == null   || flagKey.length() == 0) {
            message = ERROR + ": Incomplete request.";
        } else if (!String2.isFileNameSafe(datasetID)) {
            message = ERROR + ": Invalid datasetID.";
        } else if (!EDD.flagKey(datasetID).equals(flagKey)) {
            message = ERROR + ": Invalid flagKey.";
        } else {
            //It's ok if it isn't an existing edd.  An inactive dataset is a valid one to flag.
            //And ok of it isn't even in datasets.xml.  Unknown files are removed.
            EDStatic.tally.add("SetDatasetFlag (since startup)", datasetID);
            EDStatic.tally.add("SetDatasetFlag (last 24 hours)", datasetID);
            String2.writeToFile(EDStatic.fullResetFlagDirectory + datasetID, datasetID);
            message = "SUCCESS: The flag has been set.";
        }

        writer.write(message);
        if (verbose) String2.log(message);

        //end        
        writer.flush(); //essential
        out.close();         
    }


    /**
     * This responds to a slidesorter.html request.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doSlideSorter(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String userQuery) throws Throwable {

        //constants 
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        int unsupportedContent = -1;
        int imageContent = 0;
        int htmlContent = 1;
        String formName = "f1";
        String dFormName = "document." + formName;
        int border = 20;
        int gap = 10;
        String borderPx = "\"" + border + "px\"";
        String gapPx = "\"" + gap + "px\"";
        int defaultContentWidth = 360;
        String bgColor = "#ccccff";
        int connTimeout = 120000; //ms
        String bePatientAlt = "alt=\"Be patient. Or, check the URL. Or, press 'Submit' again.\" ";
        String ssInstructionsHtml = 
            "<big><b>ERDDAP's Slide Sorter</b></big> lets you build a personal web page that displays the content from many other \n" +
            "<br>web pages, each in it's own, draggable slide.\n" +
            "<br>&nbsp;\n" +
            "<br><b>To add a slide:</b> Fill out the 'Add a Slide' form.\n" +
            "<br>For ERDDAP grid datasets, you will usually change the date in the URL to <tt>last</tt>\n" +
            "<br>so the graph always shows the latest data.\n" +
            "<br>'file://' URL's won't work because of security restrictions.\n" +
            "<br>&nbsp;\n" +
            "<br><b>To move a slide:</b> Drag the slide with your mouse.\n" +                
            "<br>&nbsp;\n" +
            "<br><b><font color='red'>WARNING!!!</font> Your slides will be lost when you close this browser window, unless you:</b><ol>\n" +
            "<li>Scroll to the top of this document, then click on any <tt>Submit</tt> button so this document is up-to-date.\n" +
            "<li>Use your browser's <tt>File : Save As</tt> to save this document to your hard drive.\n" +
            "<li>Optional: Use your browser's <tt>File : Open</tt> to open the copy on your hard drive and bookmark it.\n" +
            "</ol>\n";

        //DON'T use GET-style params, use POST-style (request.getParameter)  

        //get info from document
        int nSlides = String2.parseInt(request.getParameter("nSlides"));
        int scrollX = String2.parseInt(request.getParameter("scrollX"));
        int scrollY = String2.parseInt(request.getParameter("scrollY"));
        if (nSlides < 0 || nSlides > 250)   nSlides = 250;   //for all of these, consider mv-> MAX_VALUE
        if (scrollX < 0 || scrollX > 10000) scrollX = 0; 
        if (scrollY < 0 || scrollY > 10000) scrollY = 0; 

        //generate html response
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Slide Sorter", out); 
        try {
            writer.write(HtmlWidgets.dragDropScript(EDStatic.imageDirUrl));
            writer.write(EDStatic.youAreHereWithHelp(tErddapUrl, "Slide Sorter", ssInstructionsHtml)); 

            //begin form
            HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
            widgets.enterTextSubmitsForm = false; 
            writer.write(widgets.beginForm(formName, "POST", //POST, not GET, because there may be lots of text   >1000 char
                tErddapUrl + "/slidesorter.html", "") + "\n");

            //gather slide title, url, x, y
            int newSlide = 0;
            int maxY = 150; //guess at header height
            StringBuffer addToJavaScript = new StringBuffer();
            StringBuffer otherSetDhtml = new StringBuffer();
            for (int oldSlide = 0; oldSlide <= nSlides; oldSlide++) { //yes <=
                String tTitle = oldSlide == nSlides? "" : request.getParameter("title" + oldSlide);
                String tUrl   = oldSlide == nSlides? "" : request.getParameter("url" + oldSlide);
                int tX = String2.parseInt(request.getParameter("x" + oldSlide));
                int tY = String2.parseInt(request.getParameter("y" + oldSlide));
                int tSize = String2.parseInt(request.getParameter("size" + oldSlide));
                if (reallyVerbose) String2.log("  found oldSlide=" + oldSlide + 
                    " title=\"" + tTitle + "\" url=\"" + tUrl + "\" x=" + tX + " y=" + tY);
                tTitle = tTitle == null? "" : tTitle.trim();
                tUrl   = tUrl == null? "" : tUrl.trim();
                String lcUrl = tUrl.toLowerCase();
                if (lcUrl.length() > 0 &&
                    !lcUrl.startsWith("file://") &&
                    !lcUrl.startsWith("ftp://") &&
                    !lcUrl.startsWith("http://") &&
                    !lcUrl.startsWith("https://") &&  //??? will never work?
                    !lcUrl.startsWith("sftp://") &&   //??? will never work?
                    !lcUrl.startsWith("smb://")) {
                    tUrl = "http://" + tUrl;
                    lcUrl = tUrl.toLowerCase();
                }

                //delete this slide if it just has default info
                if (oldSlide < nSlides && 
                    tTitle.length() == 0 &&
                    tUrl.length() == 0)
                    continue;

                //clean up oldSlide's info
                //clean up tSize below
                if (tX < 0 || tX > 3000) tX = 0;
                if (tY < 0 || tY > 20000) tY = maxY;

                //pick apart tUrl
                int qPo = tUrl.indexOf('?');
                if (qPo < 0) qPo = tUrl.length();
                String preQ = tUrl.substring(0, qPo);
                String qAndPost = tUrl.substring(qPo);
                String lcQAndPost = qAndPost.toLowerCase();
                String ext = File2.getExtension(preQ);
                String lcExt = ext.toLowerCase();
                String preExt = preQ.substring(0, preQ.length() - ext.length());
                String pngExts[] = {".smallPng", ".png", ".largePng"};
                int pngSize = String2.indexOf(pngExts, ext);

                //create the slide's content
                String content = ""; //will be html
                int contentWidth = defaultContentWidth;  //default, in px    same size as erddap .png
                int contentHeight = 20;  //default (ok for 1 line of text) in px
                String contentCellStyle = "";

                String dataUrl = null;
                if (tUrl.length() == 0) {
                    if (tSize < 0 || tSize > 2) tSize = 1;
                    contentWidth = defaultContentWidth;
                    contentHeight = 20;                        
                    content = "ERROR: No URL has been specified.\n";

                } else if (lcUrl.startsWith("file://")) {
                    //local file on client's computer
                    //file:// doesn't work in iframe because of security restrictions:
                    //  so server can't get a user's local file, and send it back to server!                                
                    if (tSize < 0 || tSize > 2) tSize = 1;
                    contentWidth = defaultContentWidth;
                    contentHeight = 20;                        
                    content = "ERROR: 'file://' URL's aren't supported (for security reasons).\n";

                } else if ((tUrl.indexOf("/tabledap/") > 0 || tUrl.indexOf("/griddap/") > 0) &&
                    (ext.equals(".graph") || pngSize >= 0)) {
                    //Make A Graph
                    //change non-.png file type to .png
                    if (tSize < 0 || tSize > 2) {
                        //if size not specified, try to use pngSize; else tSize=1
                        tSize = pngSize >= 0? pngSize : 1;
                    }
                    ext = pngExts[tSize];
                    tUrl = preExt + pngExts[tSize] + qAndPost;

                    contentWidth  = EDStatic.imageWidths[tSize];
                    contentHeight = EDStatic.imageHeights[tSize];
                    content = "<img src=\"" + XML.encodeAsXML(tUrl) +  
                        "\" width=\"" + contentWidth + "\" height=\"" + contentHeight + 
                        "\" " + bePatientAlt + ">";
                    dataUrl = preExt + ".graph" + qAndPost;

                } else {
                    //all other types
                    if (tSize < 0 || tSize > 2) tSize = 1;
                    //is file info cached?
                    int info[] = (int[])slideInfo.get(tUrl);
                    try {

                        //get info[0]=contentType
                        if (info == null) {
                            if (tUrl.startsWith(EDStatic.baseUrl)) { //url on server
                                //treat as htmlContent
                                //There is trouble with getUrlConnInputStream(ourServer).
                                //I could use selfUrl (numeric), but I know content is htmlConent.
                                info = new int[]{htmlContent};
                            } else {
                                Object oar[] = SSR.getUrlConnInputStream(tUrl, connTimeout); 
                                URLConnection conn = (URLConnection)oar[0];
                                BufferedInputStream bis = new BufferedInputStream((InputStream)oar[1]);
                                String contentType = conn.getContentType();

                                //images
                                //String2.log("contentType=" + contentType);
                                if (contentType.indexOf("image/gif") >= 0   || contentType.indexOf("image/png") >= 0 ||
                                    contentType.indexOf("image/pjpeg") >= 0 || contentType.indexOf("image/jpeg") >= 0) {
                                    //While you can fool around with dynamically loading the image in the client's browser,
                                    //things are vastly easier if the images size is know here.
                                    //So download the image and cache its size.
                                    BufferedImage image = ImageIO.read(bis);
                                    if (image == null)
                                        throw new SimpleException("Error: " +
                                            "Unable to read image. contentType=\"" + contentType + "\".\n" +
                                            "url=" + tUrl);
                                    info = new int[]{imageContent, image.getWidth(), image.getHeight()};

                                //all other content type -> htmlContent
                                } else {
                                    info = new int[]{htmlContent};
                                }

                                bis.close();
                            }

                            slideInfo.put(tUrl, info);
                        }

                        //imageContent
                        if (info[0] == imageContent) {
                            contentWidth = info[1];
                            contentHeight = info[2];
                            if (Math.max(contentWidth, contentHeight) > 500) {
                                //sizes:  x/4, x/2, x
                                if (tSize == 0) { contentWidth /= 4; contentHeight /= 4; }
                                if (tSize == 1) { contentWidth /= 2; contentHeight /= 2; }
                            } else {
                                //sizes: x/2, x, 2*x
                                if (tSize == 0) { contentWidth /= 2; contentHeight /= 2; }
                                if (tSize == 2) { contentWidth *= 2; contentHeight *= 2; }
                            }
                            String encodedUrl = XML.encodeAsXML(tUrl);
                            content = "<img src=\"" + encodedUrl + "\" " +
                                "width=\"" + contentWidth + "\" height=\"" + contentHeight + "\" " +
                                bePatientAlt + ">";                                     

                        //htmlContent
                        } else {
                            //sizes: small, wide, wide&high
                            contentWidth = EDStatic.imageWidths[tSize == 0? 1 : 2];
                            contentHeight = EDStatic.imageWidths[tSize == 2? 2 : tSize] * 3 / 4; //yes, widths; make wide
                            String encodedUrl = XML.encodeAsXML(tUrl);
                            content = "<iframe src=\"" + encodedUrl + "\" " +
                                "width=\"" + contentWidth + "\" height=\"" + contentHeight + "\" " + 
                                "style=\"background:#FFFFFF\" " +
                                ">Your browser does not support inline frames.</iframe>";
                        }

                    } catch (Throwable t) {
                        //something went wrong
                        tSize = 1;
                        contentWidth = EDStatic.imageWidths[1];
                        contentHeight = contentWidth * 3 / 4;                        
                        String tError = MustBe.getShortErrorMessage(t);
                        String2.log(MustBe.throwableToString(t)); //log full message with stack trace
                        content = XML.encodeAsPreXML(tError, 50);
                    }
                }

                //write it all
                contentWidth = Math.max(150, contentWidth); //150 so enough for urlTextField+Submit
                int tW = contentWidth + 2 * border;
                writer.write(widgets.hidden("x" + newSlide, "" + tX));
                writer.write(widgets.hidden("y" + newSlide, "" + tY));
                writer.write(widgets.hidden("w" + newSlide, "" + tW));
                //writer.write(widgets.hidden("h" + newSlide, "" + tH));
                writer.write(widgets.hidden("size" + newSlide, "" + tSize));
                writer.write(

                    "<div id=\"div" + newSlide + "\" \n" +
                        "style=\"position:absolute; left:" + tX + "px; top:" + tY + "px; " + //no \n
                        "width:" + tW + "px; " +
                        "border:1px solid #555555; background:" + bgColor + "; overflow:hidden;\"> \n\n" +
                    //top border of gadget
                    "<table bgcolor=\"" + bgColor + "\" border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
                    "<tr><td width=" + borderPx + " height=" + borderPx + "></td>\n" +
                    "  <td align=\"right\">\n\n");

                if (oldSlide < nSlides) {
                    //table for buttons
                    writer.write(  //width=20 makes it as narrow as possible
                        "  <table width=\"20px\" border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
                        "  <tr>\n\n");

                    //data button
                    if (dataUrl != null) 
                        writer.write(
                        "   <td><img src=\"" + EDStatic.imageDirUrl + "data.gif\" alt=\"data\" \n" +
                        "      title=\"Edit the image or download the data in a new browser window.\" \n" +
                        "      style=\"cursor:default;\" \n" +  //cursor:hand doesn't work in Firefox
                        "      onClick=\"window.open('" + dataUrl + "');\" ></td>\n\n"); //open a new window 

                    //resize button
                    writer.write(
                        "   <td><img src=\"" + EDStatic.imageDirUrl + "resize.gif\" alt=\"s\" \n" +
                        "      title=\"Change between small/medium/large image sizes.\" \n" +
                        "      style=\"cursor:default;\" \n" +
                        "      onClick=\"" + dFormName + ".size" + newSlide + ".value='" + 
                               (tSize == 2? 0 : tSize + 1) + "'; \n" +
                        "        setHidden(); " + dFormName + ".submit();\"></td>\n\n");

                    //end button's table
                    writer.write(
                        "  </tr>\n" +
                        "  </table>\n\n");

                    //end slide top/center cell; start top/right 
                    writer.write(
                        "  </td>\n" +
                        "  <td width=" + borderPx + " height=" + borderPx + " align=\"right\">\n");
                }

                //delete button
                if (oldSlide < nSlides) 
                    writer.write(
                    "    <img src=\"" + EDStatic.imageDirUrl + "x.gif\" alt=\"x\" \n" +
                    "      title=\"Delete this slide.\" \n" +
                    "      style=\"cursor:default\" width=" + borderPx + " height=" + borderPx + "\n" +
                    "      onClick=\"if (confirm('Delete this slide?')) {\n" +
                    "        " + dFormName + ".title" + newSlide + ".value=''; " + 
                                 dFormName + ".url" + newSlide + ".value=''; \n" +
                    "        setHidden(); " + dFormName + ".submit();}\">\n\n");
                writer.write(
                    "  </td>\n" +
                    "</tr>\n\n");

                //Add a Slide
                if (oldSlide == nSlides) 
                    writer.write(
                    "<tr><td>&nbsp;</td>\n" +
                    "  <td align=\"left\" nowrap><b>Add a Slide</b></td>\n" +
                    "</tr>\n\n");

                //gap
                writer.write(
                    "<tr><td height=" + gapPx + "></td>\n" +
                    "  </tr>\n\n");

                //title textfield
                String tPrompt = oldSlide == nSlides? "Title: " : "";
                int tWidth = contentWidth - 7*tPrompt.length() - 6;  // /7px=avg char width   6=border
                writer.write(
                    "<tr><td>&nbsp;</td>\n" +
                    "  <td align=\"left\" nowrap>" + //no \n
                    "<b>" + tPrompt + "</b>");
                writer.write(widgets.textField("title" + newSlide, 
                    "Enter a title for the slide.", 
                    -1, //(contentWidth / 8) - tPrompt.length(),  // /8px=avg bold char width 
                    255, tTitle, 
                    "style=\"width:" + tWidth + "px; background:" + bgColor + "; font-weight:bold;\""));
                writer.write(
                    "</td>\n" + //no space before /td
                    "</tr>\n\n");

                //gap
                writer.write(
                    "<tr><td height=" + gapPx + "></td>\n" +
                    "  </tr>\n\n");

                //content cell
                if (oldSlide < nSlides)
                    writer.write(
                    "<tr><td>&nbsp;</td>\n" +
                    "  <td id=\"cell" + newSlide + "\" align=\"left\" valign=\"top\" " +
                        contentCellStyle +
                        "width=\"" + contentWidth + "\" height=\"" + contentHeight + "\" >" + //no \n
                    content +
                    "</td>\n" + //no space before /td
                    "</tr>\n\n");

                //gap
                if (oldSlide < nSlides)
                    writer.write(
                    "<tr><td height=" + gapPx + "></td>\n" +
                    "  </tr>\n\n");

                //url textfield
                tPrompt = oldSlide == nSlides? "URL:   " : ""; //3 sp make it's length() longer
                tWidth = contentWidth - 7*(tPrompt.length() + 10) - 6;  // /7px=avg char width   //10 for submit  //6=border
                writer.write(
                    "<tr><td>&nbsp;</td>\n" +
                    "  <td align=\"left\" nowrap>" + //no \n
                    "<b>" + tPrompt + "</b>");
                writer.write(widgets.textField("url" + newSlide, 
                    "Enter a URL for the slide from ERDDAP's Make-A-Graph (or any URL).", 
                    -1, //(contentWidth / 7) - (tPrompt.length()-10),  // /7px=avg char width   10 for submit
                    1000, tUrl, 
                    "style=\"width:" + tWidth + "px; background:" + bgColor + "\""));
                //submit button (same row as URL textfield)
                writer.write(widgets.button("button", "submit" + newSlide, 
                    "Click to submit the information on this page to the server.",
                    "Submit",  //button label
                    "style=\"cursor:default;\" onClick=\"setHidden(); " + dFormName + ".submit();\""));
                writer.write(
                    "</td>\n" + //no space before /td
                    "</tr>\n\n");

                //bottom border of gadget
                writer.write(
                    "<tr><td height=" + borderPx + " width=" + borderPx + "></td></tr>\n" +
                    "</table>\n" +
                    "</div> \n" +
                    "\n");

                maxY = Math.max(maxY, tY + contentHeight + 3 * gap + 6 * border);  //5= 2borders, 1 title, 1 url, 2 dbl gap
                newSlide++;
            }
            writer.write(widgets.hidden("nSlides", "" + newSlide));
            //not important to save scrollXY, but important to have a place for setHidden to store changes
            writer.write(widgets.hidden("scrollX", "" + scrollX)); 
            writer.write(widgets.hidden("scrollY", "" + scrollY));

            //JavaScript
            //setHidden is called by widgets before submit() so position info is stored
            writer.write(
                "<script type=\"text/javascript\">\n" +
                "<!--\n" +
                "function setHidden() { \n" 
                //+ "alert('x0='+ dd.elements.div0.x);"
                );
            for (int i = 0; i < newSlide; i++) 
                writer.write(
                    "try {" +
                    dFormName + ".x" + i + ".value=dd.elements.div" + i + ".x; " +    
                    dFormName + ".y" + i + ".value=dd.elements.div" + i + ".y; " +  
                    dFormName + ".w" + i + ".value=dd.elements.div" + i + ".w; " + 
                    //dFormName + ".h+ " + i + ".value=dd.elements.div" + i + ".h; " +
                    "\n} catch (ex) {if (typeof(console) != 'undefined') console.log(ex.toString());}\n");
            writer.write(
                "try {" +
                dFormName + ".scrollX.value=dd.getScrollX(); " +    
                dFormName + ".scrollY.value=dd.getScrollY(); " +
                "\n} catch (ex) {if (typeof(console) != 'undefined') console.log(ex.toString());}\n" +
                "}\n");
            writer.write(
                "//-->\n" +
                "</script> \n");  

            //make space in document for slides, before end matter
            int nP = (maxY + 30) / 30;  // /30px = avg height of <p>&nbsp;  +30=round up
            for (int i = 0; i < nP; i++) 
                writer.write("<p>&nbsp;\n");
            writer.write("<p>");
            writer.write(widgets.button("button", "submit" + newSlide, 
                "Click to submit the information on this page to the server.",
                "Submit",  //button label
                "style=\"cursor:default;\" onClick=\"setHidden(); " + dFormName + ".submit();\""));
            writer.write(HtmlWidgets.ifJavaScriptDisabled);
            writer.write("<a name=\"instructions\">&nbsp;</a><p>");
            writer.write(ssInstructionsHtml);

            //end form
            writer.write(widgets.endForm());        

            //write the end stuff / set up drag'n'drop
            writer.write(
                "<script type=\"text/javascript\">\n" +
                "<!--\n" +
                "SET_DHTML(CURSOR_MOVE"); //the default cursor for the div's
            for (int i = 0; i < newSlide; i++) 
                writer.write(",\"div" + i + "\""); 
            writer.write(otherSetDhtml.toString() + ");\n");
            for (int i = 0; i < newSlide; i++) 
                writer.write("dd.elements.div" + i + ".setZ(" + i + "); \n");
            writer.write(
                "window.scrollTo(" + scrollX + "," + scrollY + ");\n" +
                addToJavaScript.toString() +
                "//-->\n" +
                "</script>\n");

            //alternatives
            writer.write("\n<hr noshade>\n" +
                "<h2><a name=\"alternatives\">Alternatives to Slide Sorter</a></h2>\n" +
                "<ul>\n" +
                "<li>Web page authors can \n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/embed.html\">embed a graph with the latest data in a web page</a> \n" +
                "  using HTML &lt;img&gt; tags.\n" +
                "<li>Anyone can use or make \n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/gadgets/GoogleGadgets.html\">Google " +
                  "Gadgets</a> to display graphs of the latest data on their iGoogle home page.\n" +
                "</ul>\n" +
                "\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end
        endHtmlWriter(out, writer, tErddapUrl, false);
        
    }

    /**
     * This responds to a full text search request.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "search") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     */
    public void doSearch(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String protocol, int datasetIDStartsAt, String userQuery) throws Throwable {
        
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String fileTypeName = "";
        String searchFor = "";
        try {
            
            //first, always set the standard DAP response header info
            //standardDapHeader(response);

            //respond to search.html request
            String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
                requestUrl.substring(datasetIDStartsAt);

            //redirect to index.html
            if (endOfRequestUrl.equals("") ||
                endOfRequestUrl.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/" + protocol + "/index.html");
                return;
            }                    

            //get the 'searchFor' value
            searchFor = request.getParameter("searchFor");
            searchFor = searchFor == null? "" : searchFor.trim();

            fileTypeName = File2.getExtension(endOfRequestUrl); //eg ".html"
            boolean toHtml = fileTypeName.equals(".html");
            if (reallyVerbose) String2.log("  searchFor=" + searchFor + 
                "\n  fileTypeName=" + fileTypeName);
            EDStatic.tally.add("Search For (since startup)", searchFor);
            EDStatic.tally.add("Search For (last 24 hours)", searchFor);
            EDStatic.tally.add("Search File Type (since startup)", fileTypeName);
            EDStatic.tally.add("Search File Type (last 24 hours)", fileTypeName);

            if (endOfRequestUrl.equals("index.html")) {
                if (searchFor.length() == 0) throw new Exception("show index"); //show form below
                //else handle just below here
            } else if (endsWithPlainFileType(endOfRequestUrl, "index")) {
                if (searchFor == null) {
                    sendResourceNotFoundError(request, response, //or SC_NO_CONTENT error?
                        "An " + requestUrl + 
                        " request must include a query: \"?searchFor=search+words\".");
                    return;
                }
                //else handle just below here
            } else { //usually unsupported fileType
                sendResourceNotFoundError(request, response, "");
                return;
            }

            //do the search
            Table table = getSearchTable(loggedInAs, searchFor, toHtml, fileTypeName);

            //show the results as an .html file 
            if (fileTypeName.equals(".html")) { 
                //display start of web page
                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, "Search", out); 
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, protocol));

                    //display the search form
                    writeSearchFormHtml(tErddapUrl, writer, 2, searchFor);

                    //display datasets
                    writer.write("<h2>" + EDStatic.resultsOfSearchFor + " <font color=\"#0000FF\">" + 
                        //encodeAsXML(searchFor) is essential -- to prevent Cross-site-scripting security vulnerability
                        //(which allows hacker to insert his javascript into pages returned by server)
                        //See Tomcat (Definitive Guide) pg 147
                        XML.encodeAsXML(searchFor) + "</font></h2>\n");  
                    if (table.nRows() == 0) {
                         writer.write("<b>" + XML.encodeAsXML(EDStatic.THERE_IS_NO_DATA) + "</b>\n" +
                             (searchFor.length() > 0? "<br>" + EDStatic.searchSpelling + "\n" : "") +
                             (searchFor.indexOf(' ') >= 0? "<br>" + EDStatic.searchFewerWords + "\n" : ""));
                    } else {
                        writer.write(
                            //table.nRows() + " " + EDStatic.nDatasetsListed + " " + 
                            EDStatic.searchRelevantAreFirst + "\n<p>"
                            //was "Lower rating numbers indicate a better match.\n" +
                            //+ "<br>" + EDStatic.clickAccessHtml + "\n" +
                            //"<br>&nbsp;\n"
                            );
                        table.saveAsHtmlTable(writer, EDStatic.tableBGColor, 1, false, -1, 
                            false, false); //don't encodeAsXML the cell's contents, !allowWrap
                        writer.write("<p>" + table.nRows() + " " + EDStatic.nDatasetsListed + "\n");
                    }
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }

                //end of document
                endHtmlWriter(out, writer, tErddapUrl, false);
                return;
            }

            //show the results in other file types
            sendPlainTable(request, response, table, protocol, fileTypeName);
            return;

        } catch (Throwable t) {

            //deal with search error (or just need empty .html searchForm)
            OutputStream out = null;
            Writer writer = null;
            //catch errors after the response has begun
            if (neededToSendErrorCode(request, response, t))
                return;

            if (String2.indexOf(plainFileTypes, fileTypeName) >= 0) 
                //for plainFileTypes, rethrow the error
                throw t;

            //make html page with [error message and] search form
            String error = MustBe.getShortErrorMessage(t);
            out = getHtmlOutputStream(request, response);
            writer = getHtmlWriter(loggedInAs, "Search", out);
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, protocol));
                if (error.indexOf("show index") < 0) 
                    writeErrorHtml(writer, request, error);
                writeSearchFormHtml(tErddapUrl, writer, 2, searchFor);
            } catch (Throwable t2) {
                writer.write(EDStatic.htmlForException(t2));
            }
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }
    }

    /**
     * This generates a results table in response to a searchFor string.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param searchFor the Google-like string of search terms.
     * @param toHtml if true, this returns a table with values suited
     *    to display via HTML. If false, the table has plain text values
     * @param fileTypeName the file type name (e.g., .htmlTable) to be used
     *    for the info links.
     * @return a table with the results.
     *    It may have 0 rows.
     */
    public Table getSearchTable(String loggedInAs, 
        String searchFor, boolean toHtml, String fileTypeName) {

        //*** respond to search request
        StringArray searchWords = StringArray.wordsAndQuotedPhrases(searchFor);
        for (int i = 0; i < searchWords.size(); i++)
            searchWords.set(i, searchWords.get(i).toLowerCase());
        
        //gather the matching datasets
        Table table = new Table();
        IntArray rankPa = new IntArray();
        StringArray idPa = new StringArray();
        int rankCol = table.addColumn("rank", rankPa); 
        int idCol   = table.addColumn("id", idPa);

        //make the list of possible datasetIDs
        //specific dataset type? 
        StringArray tDatasetIDs; 
        int gpo = searchWords.indexOf("type=grid");  //see info in writeSearchFormHtml
        int tpo = searchWords.indexOf("type=table");
        if      (gpo >= 0 && tpo <  0) {tDatasetIDs = gridDatasetIDs(false);  searchWords.remove(gpo); if (searchWords.size() == 0) searchWords.add("");}
        else if (gpo < 0  && tpo >= 0) {tDatasetIDs = tableDatasetIDs(false); searchWords.remove(tpo); if (searchWords.size() == 0) searchWords.add("");}
        else if (gpo < 0  && tpo <  0) {tDatasetIDs = allDatasetIDs(false); }
        else {tDatasetIDs = new StringArray(); searchWords.clear();} //no datasets are grid and table

        //do the search; populate the results table 
        String roles[] = EDStatic.getRoles(loggedInAs);
        int nDatasetsSearched = 0;
        long tTime = System.currentTimeMillis();
        if (searchWords.size() > 0) {
            //prepare the byte[]s
            byte searchWordsB[][] = new byte[searchWords.size()][];
            int  jumpB[][]        = new int[ searchWords.size()][];
            for (int w = 0; w < searchWords.size(); w++) {
                searchWordsB[w] = String2.getUTF8Bytes(searchWords.get(w));
                jumpB[w] = String2.makeJumpTable(searchWordsB[w]);
            }

            //do the searches
            for (int i = 0; i < tDatasetIDs.size(); i++) {
                String tId = tDatasetIDs.get(i);
                EDD edd = (EDD)gridDatasetHashMap.get(tId);
                if (edd == null)
                    edd = (EDD)tableDatasetHashMap.get(tId);
                if (edd == null)  //just deleted?
                    continue;
                if (!EDStatic.listPrivateDatasets && !edd.isAccessibleTo(roles))
                    continue;
                nDatasetsSearched++;
                int rank = edd.searchRank(searchWordsB, jumpB);           
                if (rank < Integer.MAX_VALUE) {
                    rankPa.add(rank);
                    idPa.add(tId);
                }
            }
        }
        if (verbose) {
            tTime = System.currentTimeMillis() - tTime;
            String2.log("Erddap.search " +
                //"searchFor=" + searchFor + "\n" +
                //"searchWords=" + searchWords.toString() + "\n" +
                "nDatasetsSearched=" + nDatasetsSearched + 
                " nWords=" + searchWords.size() + " totalTime=" + tTime + "ms"); 
                //" avgTime=" + (tTime / Math.max(1, nDatasetsSearched*searchWords.size())));
        }
        table.sort(new int[]{rankCol, idCol}, new boolean[]{true,true});

        return toHtml? 
            makeHtmlDatasetTable(loggedInAs, idPa, false) :   //roles check is redundant for some cases, but not all
            makePlainDatasetTable(loggedInAs, idPa, false, fileTypeName);
    }

    /**
     * Process a generateDatasetsXml request:    erddap/generateDatasetsXml/{datasetType}.html
     * e.g., erddap/generateDatasetsXml/EDDGridFromDap.html
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "generateDatasetsXml") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    /* This works but is inactive. 
       ERDDAP admins should use command line generateDatasetXml instead,
       since it supports ALL dataset types (not just a few).
    public void doGenerateDatasetsXml(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, 
        String protocol, int datasetIDStartsAt, String userQuery) throws Throwable {
        
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs); 
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        if (endOfRequestUrl.length() == 0) {
            response.sendRedirect(EDStatic.erddapUrl + "/generateDatasetsXml/index.html"); //always non-https
            return;
        }

        EDStatic.tally.add("GenerateDatasetsXml (since startup)", endOfRequestUrl);
        EDStatic.tally.add("GenerateDatasetsXml (last 24 hours)", endOfRequestUrl);
        String baseUrl = EDStatic.erddapUrl + "/" + protocol + "/"; //always non-https
        String className = File2.getNameNoExtension(endOfRequestUrl);
        
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase

        String generalDescription =  
            "This is a web service for ERDDAP administrators.\n" +
            "<br>This web service generates a rough draft of a datasets.xml entry.\n" +
            "<br>The XML can then be edited by hand and added to the datasets.xml file.\n" +
            "<br>This service is a big help, but you still need to read all of\n" +
            "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html\">Working with the datasets.xml File</a>\n" +
            "<br>and make important decisions yourself.\n";

        //define classNames and classDescriptions
        String classNames[] = {
            "EDDGridFromDap",
            "EDDGridFromErddap",
            //"EDDGridSideBySide",
            "EDDTableFromDapSequence",
            "EDDTableFromErddap",
            //"EDDTableFromMWFS",
            //"EDDTableFromNOS",
            //"EDDTableFromOBIS",
            //"EDDTableFromSOS"
            };
        String classDescription[] = { //html
            " handles gridded data from DAP servers.", //EDDGridFromDap
            " handles gridded data from ERDDAP servers.", //EDDGridFromErddap
            //" aggregates two or more EDDGrid datasets side by side.", //EDDGridSideBySide
            " handles tabular data from DAP sequence servers.", //EDDTableFromDapSequence
            " handles tabular data from ERDDAP servers.", //EDDTableFromErddap
            //" handles tabular data from microWFS servers.", //EDDTableFromMWFS
            //" handles tabular data from NOS XML servers.", //EDDTableFromNOS
            //" handles tabular data from OBIS servers.", //EDDTableFromOBIS
            //" handles tabular data from SOS servers." //EDDTableFromSOS
            };

        //setup the parameters for the form
        String sourceUrlTT = "sourceUrl specifies the URL of the data source.";
        String eddDescription;
        String[] names, tooltips, defaults;   //tooltips is html
        if (endOfRequestUrl.equals("EDDGridFromDap.html")) {
            eddDescription = 
"<p>This particular service will generate the XML for one <b>EDDGridFromDap</b> dataset,\n" +
"<br>which handles multidimensional data from a <a href=\"http://opendap.org/\">DAP</a> server.\n";
            names = new String[]{"sourceUrl"};
            tooltips = new String[]{sourceUrlTT};
            defaults = new String[]{"http://thredds1.pfeg.noaa.gov/thredds/dodsC/satellite/BA/ssta/5day"};

        } else if (endOfRequestUrl.equals("EDDTableFromDapSequence.html")) {
            eddDescription = 
"<p>This particular service will generate the XML for one <b>EDDTableFromDapSequence</b> dataset,\n" +
"<br>which handles data variables within 1- and 2-level sequences from a\n" +
"  <a href=\"http://opendap.org/\">DAP</a> server such as\n" +
"  <a href=\"http://www.epic.noaa.gov/epic/software/dapper/\">DAPPER</a>.\n" +
"<br>A variable is in a DAP sequence if the .dds response (look at <i>sourceUrl</i>.dds in your browser)\n" +
"<br>indicates that the data structure holding the variable is a \"sequence\" (case insensitive).\n" +
"<br>In some cases, you will see a sequence within a sequence, a 2-level sequence -- \n" +
"<br>EDDTableFromDapSequence handles these, too.\n" +
"<br>The .dds response is also how you can find the outer (and inner) sequence name.\n";
            names = new String[]{"sourceUrl", "outerSequenceName", "innerSequenceName"};
            String fromDAS = "<br>You can find this by looking at [sourceUrl].das .";
            tooltips = new String[]{
                sourceUrlTT,
                "outerSequenceName specifies the name of the outer sequence for DAP sequence data." + fromDAS,
                "innerSequenceName specifies the name of the inner sequence for DAP sequence data.<br>Leave blank if none." + fromDAS};
            defaults = new String[]{"http://dapper.pmel.noaa.gov/dapper/epic/tao_time_series.cdp",
                "location",
                "time_series"};

        } else if (endOfRequestUrl.equals("EDDGridFromErddap.html")) {
            eddDescription = 
                "<p>This particular service will generate the <b>EDDGridFromErddap</b> XML\n" +
                "for <b>all</b> gridded datasets from an\n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/index.html\">ERDDAP</a> server.\n";
            names = new String[]{"sourceUrl"};
            tooltips = new String[]{sourceUrlTT};
            defaults = new String[]{"http://coastwatch.pfeg.noaa.gov/erddap"};

        } else if (endOfRequestUrl.equals("EDDTableFromErddap.html")) {
            eddDescription = 
                "<p>This particular service will generate the <b>EDDTableFromErddap</b> XML\n" +
                "for <b>all</b> tabular datasets from an\n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/index.html\">ERDDAP</a> server.\n";
            names = new String[]{"sourceUrl"};
            tooltips = new String[]{sourceUrlTT};
            defaults = new String[]{"http://coastwatch.pfeg.noaa.gov/erddap"};

        } else if (endOfRequestUrl.equals("index.html")) {
            //for index.html and non-class names, show list of classes, then exit
            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "Generate Datasets XML", out); 
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, protocol));
                writer.write(
                    generalDescription +
                    "<p><b>Please choose the data source class for which you would like to generate XML:</b>\n" +
                    "<br>(The service is not available for all dataset types.)\n" +
                    "<ul>\n");
                for (int i = 0; i < classNames.length; i++) 
                    writer.write(
                    "<li><a href=\"" + baseUrl + classNames[i] + ".html\">" + classNames[i] + "</a> \n" +
                    classDescription[i] + "\n");    
                writer.write(
                    "</ul>\n");
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            endHtmlWriter(out, writer, tErddapUrl, false);                
            return;

        } else {
            //unsupported dataset type
            sendResourceNotFoundError(request, response, "");
            return;
        }


        //the specific classname/datasetType is valid
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Generate Datasets XML", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol, className) +
                generalDescription + 
                eddDescription + "\n");

            //get the values 
            String values[] = new String[names.length];
            boolean queryMapHasValues = queryMap.size() > 0;
            for (int i = 0; i < names.length; i++) {  //queryMap has names.toLowerCase() so case insensitive
                String s = queryMapHasValues? queryMap.get(names[i].toLowerCase()) : defaults[i];
                values[i] = s == null? "" : s;
            }

            //display the appropriate form
            HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
            widgets.enterTextSubmitsForm = false;
            writer.write(
                widgets.beginForm("genXml", "GET", baseUrl + endOfRequestUrl, "") +
                "<p><b>Please fill out this form:</b>\n" +
                widgets.beginTable(0, 0, ""));  
            for (int i = 0; i < names.length; i++) {
                writer.write(
                    "<tr>\n" +
                    "  <td>" + names[i] + ":&nbsp;</td>\n" +
                    "  <td>" + widgets.textField(names[i], tooltips[i], 80, 512, values[i], "") + "</td>\n" +
                    "</tr>\n");
            }
            writer.write(
                "<tr>\n" +
                "  <td>" + widgets.button("submit", null, 
                    "Click to submit the form to the server.", "Submit", "") + "</td>\n" +
                "</tr>\n" +
                widgets.endTable() +  
                widgets.endForm());        

            //try to get and display the results
            if (queryMapHasValues) {
                writer.write(
                    "<hr noshade>\n" +
                    "<h2>Results</h2>\n" +
                    "<pre>\n");

                try {
                    String results = null;
                    if (endOfRequestUrl.equals("EDDGridFromDap.html")) 
                        results = EDDGridFromDap.generateDatasetsXml(values[0]);
                    else if (endOfRequestUrl.equals("EDDTableFromDapSequence.html")) 
                        results = EDDTableFromDapSequence.generateDatasetsXml(values[0], values[1], values[2], true);
                    else if (endOfRequestUrl.equals("EDDGridFromErddap.html")) 
                        results = EDDGridFromErddap.generateDatasetsXml(values[0]);
                    else if (endOfRequestUrl.equals("EDDTableFromErddap.html")) 
                        results = EDDTableFromErddap.generateDatasetsXml(values[0]);
                    
                    writer.write(XML.encodeAsXML(results));
                } catch (Throwable t) {
                    writer.write(XML.encodeAsXML(MustBe.getShortErrorMessage(t)));
                }
                writer.write(
                    "</pre>\n");
            }   
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        endHtmlWriter(out, writer, tErddapUrl, false);

    }
    */

    /**
     * Process a categorize request:    erddap/categorize/{attribute}/{categoryName}/index.html
     * e.g., erddap/categorize/ioos_category/Temperature/index.html
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "categorize") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doCategorize(HttpServletRequest request, HttpServletResponse response,
        String loggedInAs, String protocol, int datasetIDStartsAt, String userQuery) throws Throwable {
        
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String fileTypeName = "";
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
       
        //first, always set the standard DAP response header info
        //standardDapHeader(response);
        String parts[] = String2.split(endOfRequestUrl, '/');

        //*** attribute string should be e.g., ioos_category
        fileTypeName = File2.getExtension(endOfRequestUrl);
        String attribute = parts.length < 1? "" : parts[0];
        if (reallyVerbose) String2.log("  attribute=" + attribute);
        int whichAttribute = String2.indexOf(EDStatic.categoryAttributes, attribute);
        if (whichAttribute < 0) {
            //*** deal with invalid attribute string

            //redirect to index.html
            if (attribute.equals("") ||
                attribute.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/" + protocol + "/index.html");
                return;
            }   
            
            //return table of categoryAttributes
            if (String2.indexOf(plainFileTypes, fileTypeName) >= 0) {
                //plainFileType
                if (attribute.equals("index" + fileTypeName)) {
                    //respond to categorize/index.xxx
                    //display list of categoryAttributes in plainFileType file
                    Table table = categorizeOptionsTable(tErddapUrl, fileTypeName);
                    sendPlainTable(request, response, table, protocol, fileTypeName);
                } else {
                    sendResourceNotFoundError(request, response, "");
                    return;
                }
            } else { 
                //respond to categorize/index.html or errors: unknown attribute, unknown fileTypeName 
                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, "Categorizie", out); 
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, EDStatic.categoryTitleHtml)); //protocol));
                    if (!attribute.equals("index.html")) 
                        writeErrorHtml(writer, request, "categoryAttribute=\"" + attribute + "\" is not an option.");
                    writeCategorizeOptionsHtml1(tErddapUrl, writer, false);
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }
                endHtmlWriter(out, writer, tErddapUrl, false);
            }
            return;
        }   
        //attribute is valid
        if (reallyVerbose) String2.log("  attribute=" + attribute + " is valid.");

        //*** categoryName string should be e.g., Location
        //*** deal with index.xxx and invalid categoryName
        String categoryName = parts.length < 2? "" : parts[1];
        if (reallyVerbose) String2.log("  categoryName=" + categoryName);
        StringArray catDats = categoryInfo(attribute, categoryName); 
        if (catDats.size() == 0) {

            //redirect to index.html
            if (categoryName.equals("") ||
                categoryName.equals("index.htm")) {
                response.sendRedirect(tErddapUrl + "/" + protocol + "/" + 
                    attribute + "/index.html");
                return;
            }   
            
            //return table of categoryNames
            if (String2.indexOf(plainFileTypes, fileTypeName) >= 0) {
                //plainFileType
                if (categoryName.equals("index" + fileTypeName)) {
                    //respond to categorize/attribute/index.xxx
                    //display list of categoryNames in plainFileType file
                    sendCategoryPftOptionsTable(request, response, loggedInAs, attribute, fileTypeName);
                } else {
                    sendResourceNotFoundError(request, response, "");
                    return;
                }
            } else { 
                //respond to categorize/index.html or errors: unknown attribute, unknown fileTypeName 
                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, "Categorize", out); 
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, EDStatic.categoryTitleHtml)); //protocol, attribute));
                    if (!categoryName.equals("index.html")) {
                        writeErrorHtml(writer, request, 
                            "categoryName=\"" + categoryName + "\" is not an option when categoryAttribute=\"" + 
                            attribute + "\".");
                        writer.write("<hr noshade>\n");
                    }
                    writeCategorizeOptionsHtml1(tErddapUrl, writer, false);
                    writeCategoryOptionsHtml2(tErddapUrl, writer, attribute);
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }
                endHtmlWriter(out, writer, tErddapUrl, false);
            }
            return;
        }           
        //categoryName is valid
        if (reallyVerbose) String2.log("  categoryName=" + categoryName + " is valid.");

        //*** attribute (e.g., ioosCategory) and categoryName (e.g., Location) are valid
        //endOfRequestUrl3 should be index.xxx or {categoryName}.xxx
        String part2 = parts.length < 3? "" : parts[2];

        //redirect categorize/{attribute}/{categoryName}/index.htm request index.html
        if (part2.equals("") ||
            part2.equals("index.htm")) {
            response.sendRedirect(tErddapUrl + "/" + protocol + "/" + 
                attribute + "/" + categoryName + "/index.html");
            return;
        }   

        //*** respond to categorize/{attribute}/{categoryName}/index.fileTypeName request
        EDStatic.tally.add("Categorize Attribute (since startup)", attribute);
        EDStatic.tally.add("Categorize Attribute (last 24 hours)", attribute);
        EDStatic.tally.add("Categorize Attribute / Value (since startup)", attribute + " / " + categoryName);
        EDStatic.tally.add("Categorize Attribute / Value (last 24 hours)", attribute + " / " + categoryName);
        EDStatic.tally.add("Categorize File Type (since startup)", fileTypeName);
        EDStatic.tally.add("Categorize File Type (last 24 hours)", fileTypeName);
        if (endsWithPlainFileType(part2, "index")) {
            //show the results as plain file type
            Table table = makePlainDatasetTable(loggedInAs, catDats, true, fileTypeName);
            sendPlainTable(request, response, table, attribute + "_" + categoryName, fileTypeName);
            return;
        }

        //respond to categorize/{attribute}/{categoryName}/index.html request
        if (part2.equals("index.html")) {
            //make a table of the datasets
            Table table = makeHtmlDatasetTable(loggedInAs, catDats, true);

            //display start of web page
            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "Categorize", out); 
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, EDStatic.categoryTitleHtml)); //protocol, attribute, categoryName));

                //write categorizeOptions
                writeCategorizeOptionsHtml1(tErddapUrl, writer, false);

                //write categoryOptions
                writeCategoryOptionsHtml2(tErddapUrl, writer, attribute);

                //display datasets
                writer.write("<h3>3) " + EDStatic.resultsOfSearchFor + " " + attribute + 
                    " = <font color=\"#0000FF\">" + categoryName + "</font> &nbsp; " +
                    //EDStatic.htmlTooltipImage(
                        //table.nRows() + " " + EDStatic.nDatasetsListed + "\n" +                    
                        //"<br>" + EDStatic.clickAccessHtml + "\n") +
                    "</h3>\n" +
                    "<b>Pick a dataset:</b>\n<p>");
                table.saveAsHtmlTable(writer, EDStatic.tableBGColor, 1, false, -1, false, false);        
                writer.write("<p>" + table.nRows() + " " + EDStatic.nDatasetsListed + "\n");
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }

            //end of document
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }

        sendResourceNotFoundError(request, response, "");
    }

    /**
     * Process an info request: erddap/info/[{datasetID}/index.xxx]
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @throws Throwable if trouble
     */
    public void doInfo(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String protocol, int datasetIDStartsAt) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        String fileTypeName = File2.getExtension(endOfRequestUrl);

        String parts[] = String2.split(endOfRequestUrl, '/');
        int nParts = parts.length;
        if (nParts == 0 || !parts[nParts - 1].startsWith("index.")) {
            StringArray sa = new StringArray(parts);
            sa.add("index.html");
            parts = sa.toArray();
            nParts = parts.length;
            //now last part is "index...."
        }
        fileTypeName = File2.getExtension(endOfRequestUrl);        
        boolean endsWithPlainFileType = endsWithPlainFileType(parts[nParts - 1], "index");
        if (!endsWithPlainFileType && !fileTypeName.equals(".html")) {
            sendResourceNotFoundError(request, response, 
                "Unsupported fileExtension=" + fileTypeName);
            return;
        }
        EDStatic.tally.add("Info File Type (since startup)", fileTypeName);
        EDStatic.tally.add("Info File Type (last 24 hours)", fileTypeName);
        if (nParts < 2) {
            StringArray tIDs = allDatasetIDs(true);
            EDStatic.tally.add("Info (since startup)", "View All Datasets");
            EDStatic.tally.add("Info (last 24 hours)", "View All Datasets");

            if (fileTypeName.equals(".html")) {
                //make the table with the dataset list
                Table table = makeHtmlDatasetTable(loggedInAs, tIDs, true);

                //display start of web page
                OutputStream out = getHtmlOutputStream(request, response);
                Writer writer = getHtmlWriter(loggedInAs, "List of Datasets", out); 
                try {
                    writer.write(EDStatic.youAreHere(tErddapUrl, protocol));
                   
                    if (table.nRows() == 0) {
                        writer.write("<b>" + EDStatic.THERE_IS_NO_DATA + "</b>\n");
                    } else {
                        writer.write(
                            "<h2>Pick a Dataset</h2>\n"
                            //+ EDStatic.clickAccessHtml + "\n" +
                            //"<br>&nbsp;\n"
                            );
                        table.saveAsHtmlTable(writer, EDStatic.tableBGColor, 
                            1, false, -1, false, false);        
                        writer.write(
                            "<p>" + table.nRows() + " " + EDStatic.nDatasetsListed + "\n");
                    }
                } catch (Throwable t) {
                    writer.write(EDStatic.htmlForException(t));
                }

                //end of document
                endHtmlWriter(out, writer, tErddapUrl, false);
            } else {
                Table table = makePlainDatasetTable(loggedInAs, tIDs, true, fileTypeName);
                sendPlainTable(request, response, table, protocol, fileTypeName);
            }
            return;
        }
        if (nParts > 2) {
            sendResourceNotFoundError(request, response, 
                "erddap/info requests must be in the form erddap/info/<datasetID>/index<fileType> .");
            return;
        }
        String tID = parts[0];
        EDD edd = (EDD)gridDatasetHashMap.get(tID);
        if (edd == null)
            edd = (EDD)tableDatasetHashMap.get(tID);
        if (edd == null) { 
            sendResourceNotFoundError(request, response,
                EDStatic.noDatasetWith + " datasetID=" + tID + ".");
            return;
        }
        if (!edd.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
            EDStatic.redirectToLogin(loggedInAs, response, tID);
            return;
        }

        //request is valid -- make the table
        EDStatic.tally.add("Info (since startup)", tID);
        EDStatic.tally.add("Info (last 24 hours)", tID);
        Table table = new Table();
        StringArray rowTypeSA = new StringArray();
        StringArray variableNameSA = new StringArray();
        StringArray attributeNameSA = new StringArray();
        StringArray javaTypeSA = new StringArray();
        StringArray valueSA = new StringArray();
        table.addColumn("Row Type", rowTypeSA);
        table.addColumn("Variable Name", variableNameSA);
        table.addColumn("Attribute Name", attributeNameSA);
        table.addColumn("Data Type", javaTypeSA);
        table.addColumn("Value", valueSA);

        //global attribute rows
        Attributes atts = edd.combinedGlobalAttributes();
        String names[] = atts.getNames();
        int nAtts = names.length;
        for (int i = 0; i < nAtts; i++) {
            rowTypeSA.add("attribute");
            variableNameSA.add("NC_GLOBAL");
            attributeNameSA.add(names[i]);
            PrimitiveArray value = atts.get(names[i]);
            javaTypeSA.add(value.getElementTypeString());
            valueSA.add(Attributes.valueToNcString(value));
        }

        //dimensions
        String axisNamesCsv = "";
        if (edd instanceof EDDGrid) {
            EDDGrid eddGrid = (EDDGrid)edd;
            int nDims = eddGrid.axisVariables().length;
            axisNamesCsv = String2.toCSVString(eddGrid.axisVariableDestinationNames());
            for (int dim = 0; dim < nDims; dim++) {
                //dimension row
                EDVGridAxis edv = eddGrid.axisVariables()[dim];
                rowTypeSA.add("dimension");
                variableNameSA.add(edv.destinationName());
                attributeNameSA.add("");
                javaTypeSA.add(edv.destinationDataType());
                int tSize = edv.sourceValues().size();
                double avgSp = edv.averageSpacing();
                if (tSize == 1) {
                    double dValue = edv.firstDestinationValue();
                    valueSA.add(
                        "nValues=1, onlyValue=" + 
                        (Double.isNaN(dValue)? "NaN" : edv.destinationToString(dValue))); //want "NaN", not ""
                } else {
                    valueSA.add(
                        "nValues=" + tSize + 
                        ", evenlySpaced=" + (edv.isEvenlySpaced()? "true" : "false") +
                        ", averageSpacing=" + 
                        (edv instanceof EDVTimeGridAxis? 
                            Calendar2.elapsedTimeString(Math.rint(avgSp) * 1000) : 
                            avgSp)
                        );
                }

                //attribute rows
                atts = edv.combinedAttributes();
                names = atts.getNames();
                nAtts = names.length;
                for (int i = 0; i < nAtts; i++) {
                    rowTypeSA.add("attribute");
                    variableNameSA.add(edv.destinationName());
                    attributeNameSA.add(names[i]);
                    PrimitiveArray value = atts.get(names[i]);
                    javaTypeSA.add(value.getElementTypeString());
                    valueSA.add(Attributes.valueToNcString(value));
                }
            }
        }

        //data variables
        int nVars = edd.dataVariables().length;
        for (int var = 0; var < nVars; var++) {
            //data variable row
            EDV edv = edd.dataVariables()[var];
            rowTypeSA.add("variable");
            variableNameSA.add(edv.destinationName());
            attributeNameSA.add("");
            javaTypeSA.add(edv.destinationDataType());
            valueSA.add(axisNamesCsv);

            //attribute rows
            atts = edv.combinedAttributes();
            names = atts.getNames();
            nAtts = names.length;
            for (int i = 0; i < nAtts; i++) {
                rowTypeSA.add("attribute");
                variableNameSA.add(edv.destinationName());
                attributeNameSA.add(names[i]);
                PrimitiveArray value = atts.get(names[i]);
                javaTypeSA.add(value.getElementTypeString());
                valueSA.add(Attributes.valueToNcString(value));
            }
        }

        //write the file
        if (endsWithPlainFileType) {
            sendPlainTable(request, response, table, parts[0] + "_info", fileTypeName);
            return;
        }

        //respond to index.html request
        if (parts[1].equals("index.html")) {
            //display start of web page
            OutputStream out = getHtmlOutputStream(request, response);
            Writer writer = getHtmlWriter(loggedInAs, "Information about " + 
                XML.encodeAsXML(edd.title() + " from " + edd.institution()), out); 
            try {
                writer.write(EDStatic.youAreHere(tErddapUrl, protocol, parts[0]));

                //display a table with the one dataset
                //writer.write(EDStatic.clickAccessHtml + "\n" +
                //    "<br>&nbsp;\n");
                StringArray sa = new StringArray();
                sa.add(parts[0]);
                Table dsTable = makeHtmlDatasetTable(loggedInAs, sa, true);
                dsTable.saveAsHtmlTable(writer, EDStatic.tableBGColor, 1, false, -1, false, false);        

                //html format the valueSA values
                for (int i = 0; i < valueSA.size(); i++) 
                    valueSA.set(i, XML.encodeAsPreXML(valueSA.get(i), 100000));

                //display the info table
                writer.write("<h2>" + EDStatic.infoTableTitleHtml + "</h2>");
                //table.saveAsHtmlTable(writer, EDStatic.tableBGColor, 1, false, -1, 
                //    false, true);  //false=already html, true=allowWrap

                //******** custom table writer (to change color on "variable" rows)
                writer.write(
                    "<table class=\"erd\" bgcolor=\"" + EDStatic.tableBGColor + "\" cellspacing=\"0\">\n"); 
                    //"<table bgcolor=\"" + EDStatic.tableBGColor + "\" " +
                    //"border=\"1\" cellpadding=\"2\" cellspacing=\"0\">\n"); 

                //write the column names   
                writer.write("<tr>\n");
                int nColumns = table.nColumns();
                for (int col = 0; col < nColumns; col++) 
                    writer.write("<th>" + table.getColumnName(col) + "</th>\n");
                writer.write("</tr>\n");

                //write the data
                int nRows = table.nRows();
                for (int row = 0; row < nRows; row++) {
                    String s = table.getStringData(0, row);
                    if (s.equals("variable") || s.equals("dimension"))
                         writer.write("<tr bgcolor=\"" + EDStatic.tableHighlightColor+ "\">\n"); 
                    else writer.write("<tr>\n"); 
                    for (int col = 0; col < nColumns; col++) {
                        writer.write("<td>"); 
                        s = table.getStringData(col, row);
                        writer.write(s.length() == 0? "&nbsp;" : s); 
                        writer.write("</td>\n");
                    }
                    writer.write("</tr>\n");
                }

                //close the table
                writer.write("</table>\n");
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }

            //end of document
            endHtmlWriter(out, writer, tErddapUrl, false);
            return;
        }

        sendResourceNotFoundError(request, response, "");

    }

    /**
     * Process erddap/subscriptions/index.html
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param ipAddress the requestor's ipAddress
     * @param endOfRequest e.g., subscriptions/add.html
     * @param protocol is always subscriptions
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doSubscriptions(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String ipAddress,
        String endOfRequest, String protocol, int datasetIDStartsAt, String userQuery) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);

        if (endOfRequest.equals("subscriptions") ||
            endOfRequest.equals("subscriptions/")) {
            response.sendRedirect(tErddapUrl + "/" + Subscriptions.INDEX_HTML);
            return;
        }

        EDStatic.tally.add("Subscriptions (since startup)", endOfRequest);
        EDStatic.tally.add("Subscriptions (last 24 hours)", endOfRequest);

        if (endOfRequest.equals(Subscriptions.INDEX_HTML)) {
            //fall through
        } else if (!EDStatic.subscriptionSystemActive) {
            sendResourceNotFoundError(request, response, "");
            return;
        } else if (endOfRequest.equals(Subscriptions.ADD_HTML)) {
            doAddSubscription(request, response, loggedInAs, ipAddress, protocol, datasetIDStartsAt, userQuery);
            return;
        } else if (endOfRequest.equals(Subscriptions.LIST_HTML)) {
            doListSubscriptions(request, response, loggedInAs, ipAddress, protocol, datasetIDStartsAt, userQuery);
            return;
        } else if (endOfRequest.equals(Subscriptions.REMOVE_HTML)) {
            doRemoveSubscription(request, response, loggedInAs, protocol, datasetIDStartsAt, userQuery);
            return;
        } else if (endOfRequest.equals(Subscriptions.VALIDATE_HTML)) {
            doValidateSubscription(request, response, loggedInAs, protocol, datasetIDStartsAt, userQuery);
            return;
        } else {
            sendResourceNotFoundError(request, response, "");
            return;
        }

        //display start of web page
        if (reallyVerbose) String2.log("doSubscriptions");
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Subscriptions", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol) +
                EDStatic.subscriptionHtml(tErddapUrl) + "\n");
            if (EDStatic.subscriptionSystemActive) 
                writer.write(
                "<p><b>Options:</b>\n" +
                "<ul>\n" +
                "<li> <a href=\"" + tErddapUrl + "/" + Subscriptions.ADD_HTML      + "\">Add a new subscription</a>\n" +
                "<li> <a href=\"" + tErddapUrl + "/" + Subscriptions.VALIDATE_HTML + "\">Validate a subscription</a>\n" +
                "<li> <a href=\"" + tErddapUrl + "/" + Subscriptions.LIST_HTML     + "\">List your subscriptions</a>\n" +
                "<li> <a href=\"" + tErddapUrl + "/" + Subscriptions.REMOVE_HTML   + "\">Remove a subscription</a>\n" +
                "</ul>\n");
            else writer.write(
                "<p><b>Sorry.  The email/URL subscription system is not available at this ERDDAP installation.</b>\n" +
                "<br>Consider using the <a href=\"" + tErddapUrl + "/information.html#subscriptions\">RSS</a>\n" +
                "subscription service instead.\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }


    /** 
     * This html is used at the bottom of many doXxxSubscription web pages. 
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param tEmail  the user's email address (or "")
     */
    private String requestSubscriptionListHtml(String tErddapUrl, String tEmail) {
        return 
            "<br>&nbsp;\n" +
            "<p><b>Or, you can request an email with a\n" +
            "<a href=\"" + tErddapUrl + "/" + Subscriptions.LIST_HTML + 
            (tEmail.length() > 0? "?email=" + tEmail : "") +
            "\">list of your valid and pending subscriptions</a>.</b>\n";
    }

           
    /**
     * Process erddap/subscriptions/add.html.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param ipAddress the requestor's ip address
     * @param protocol is always subscriptions
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doAddSubscription(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String ipAddress, String protocol, int datasetIDStartsAt, 
        String userQuery) throws Throwable {

        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //parse the userQuery
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=lowercase keys
        String tDatasetID = queryMap.get("datasetid"); 
        String tEmail     = queryMap.get("email");
        String tAction    = queryMap.get("action");
        if (tDatasetID == null) tDatasetID = "";
        if (tEmail     == null) tEmail     = "";
        if (tAction    == null) tAction    = "";
        boolean tEmailIfAlreadyValid = String2.parseBoolean(queryMap.get("emailifalreadyvalid")); //default=true 
        boolean tShowErrors          = String2.parseBoolean(queryMap.get("showerrors"));          //default=true; 

        //validate params
        String trouble = "";
        if      (tDatasetID.length() == 0)                                trouble += "<li> The datasetID wasn't specified.\n";
        else if (tDatasetID.length() > Subscriptions.DATASETID_LENGTH)    trouble += "<li> The datasetID is too long.\n";
        else if (!String2.isFileNameSafe(tDatasetID))                     trouble += "<li> The datasetID isn't valid.\n";
        else {
            EDD edd = (EDD)gridDatasetHashMap.get(tDatasetID);
            if (edd == null) 
                edd = (EDD)tableDatasetHashMap.get(tDatasetID);
            if (edd == null)                                              trouble += "<li> That dataset isn't currently available.\n";
            else if (!edd.isAccessibleTo(EDStatic.getRoles(loggedInAs))) { //listPrivateDatasets doesn't apply
                EDStatic.redirectToLogin(loggedInAs, response, tDatasetID);
                return;
            }
        }
        if      (tEmail.length() == 0)                                    trouble += "<li> Your email address wasn't specified.\n";
        else if (tEmail.length() > Subscriptions.EMAIL_LENGTH)            trouble += "<li> Your email address is too long.\n";
        else if (!String2.isEmailAddress(tEmail))                         trouble += "<li> Your email address isn't valid.\n";
        if      (tAction.length() > Subscriptions.ACTION_LENGTH)          trouble += "<li> The URL/action is too long.\n";
        else if (!tAction.equals("") && 
            !(tAction.length() > 10 && tAction.startsWith("http://")))    trouble += 
            "<li> The URL/action isn't valid. If present, it must start with \"http://\" .\n";

        //display start of web page
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
        widgets.enterTextSubmitsForm = true; 
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Add a Subscription", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol, "add") +
                EDStatic.subscriptionHtml(tErddapUrl) + "\n" +
                "<p><b>IMPORTANT INFORMATION:</b>\n" +
                "<ul>\n" +
                "<li> ERDDAP will store the information that you provide, including your email address.\n" +
                "<li> ERDDAP will treat the information that you provide as confidential and will not give it to anyone else.\n" +
                "<li> These subscriptions have nothing to do with getting the actual data (see\n" + 
                    EDStatic.erddapHref(tErddapUrl) + ").\n" +
                "<li> ERDDAP is very good at detecting changes to gridded datasets because it can detect\n" +
                    "<br>when the axis values (e.g., the time values) change.\n" +
                "<li> ERDDAP is not very good at detecting changes to tabular datasets because there are usually\n" +
                    "<br>no changes to the metadata when new data is added.\n" +
                "<li> ERDDAP will detect if a dataset becomes unavailable (but perhaps not immediately).\n" +
                 "<br>ERDDAP will detect when that dataset becomes available again.\n" +
                "<li> ERDDAP makes no promises about the suitability or accuracy of this service (see the\n" +
                    "<a href=\"http://coastwatch.pfeg.noaa.gov/erddap/download/setup.html#disclaimers\">\n" +
                    "DISCLAIMER OF LIABILITY</a>).\n" +
                "</ul>\n");

            String another = "a";
            if (tDatasetID.length() > 0 || tEmail.length() > 0 || tAction.length() > 0) {
                if (trouble.length() > 0) {
                    if (tShowErrors) 
                        writer.write(
                        "<p><font color=\"#FF0000\"><b>" +
                        "Your request to add a subscription had one or more errors:</b></font>\n" +
                        "<ul>\n" +
                        trouble + "\n" +
                        "</ul>\n");
                } else {
                    //try to add 
                    try {
                        int row = EDStatic.subscriptions.add(tDatasetID, tEmail, tAction);
                        if (tEmailIfAlreadyValid || 
                            EDStatic.subscriptions.readStatus(row) == Subscriptions.STATUS_PENDING) {
                            String invitation = EDStatic.subscriptions.getInvitation(ipAddress, row);
                            String tError = EDStatic.email(tEmail, "Subscription Invitation", invitation);
                            if (tError.length() > 0)
                                throw new SimpleException(tError);

                            //tally
                            EDStatic.tally.add("Subscriptions (since startup)", "Add successful");
                            EDStatic.tally.add("Subscriptions (last 24 hours)", "Add successful");
                        }
                        writer.write(
                            "<p><font color=\"#008800\"><b>" +
                            "Your add-subscription request has been submitted successfully.</b></font>\n" +
                            "<br>You should get an email soon with instructions for validating the subscription.\n");
                        another = "another";
                    } catch (Throwable t) {
                        writer.write(
                            "<p><font color=\"#FF0000\"><b>Your add-subscription request had an error:</b></font>\n" +
                            "<br>" + XML.encodeAsXML(MustBe.getShortErrorMessage(t)) + "\n");
                        String2.log("Subscription Add Exception:\n" + MustBe.throwableToString(t)); //log stack trace, too

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "Add unsuccessful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "Add unsuccessful");
                    }
                }
            }

            //show the form
            String urlTT = 
                "<tt>URL/action</tt> is an optional URL that ERDDAP will contact instead of sending you an email.\n" +
                "<br>So if you want to receive an email whenever the dataset changes, leave <tt>URL/action</tt> blank.\n" +
                "<br>If you do supply a URL, ERDDAP will just contact the URL and ignore the response.\n" +
                "<br>So this can be used to trigger some action at some other web service.\n" +
                "<br>To use this feature, enter a percent-encoded URL, for example\n" +
                "<br><tt>http://www.yourWebSite.com?department=R%26D&action=rerunTheModel</tt>";
            writer.write(
                widgets.beginForm("addSub", "GET", tErddapUrl + "/" + Subscriptions.ADD_HTML, "") +
                "<p><b>To add " + another + " subscription, please fill out this form:</b>\n" + 
                "<br>The easiest way to use this form is to click on an envelope icon \n" +
                "<img alt=\"Subscribe\" src=\"" + EDStatic.imageDirUrl + "envelope.gif\">\n" + //no end tag
                "<br>on an ERDDAP web page with a list of datasets (at the far right),\n" +
                "<br>or on a dataset's Data Access Form or Make A Graph web page.\n" +
                "<br>Then, the datasetID will be filled in for you.\n" +
                widgets.beginTable(0, 0, "") +
                "<tr>\n" +
                "  <td>The datasetID:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("datasetID", 
                    "For example, " + EDStatic.EDDGridIdExample,
                    40, Subscriptions.DATASETID_LENGTH, tDatasetID, 
                    "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>Your email address:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("email", "", 
                    60, Subscriptions.EMAIL_LENGTH, tEmail, 
                    "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>The URL/action:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("action", urlTT,
                    80, Subscriptions.ACTION_LENGTH, tAction, "") + "\n" +
                "    " + EDStatic.htmlTooltipImage(urlTT) +
                "  </td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td colspan=\"2\">" + widgets.button("submit", null, 
                    "Click to submit the form to the server.", "Submit", "") + "\n" +
                "    <br>Then, ERDDAP will send you an email with a link to validate your request.\n" +
                "  </td>\n" +
                "</tr>\n" +
                widgets.endTable() +  
                widgets.endForm() +
                "<font color=\"#FF0000\">Don't abuse this system by entering someone else's email address!</font>\n" +
                "<br>If you do, you may be blocked from using ERDDAP.\n" +
                "<br>Your IP address will be included in the email with the subscription invitation.\n" +
                "<br>So if you enter someone else's email address,\n" +
                "<br>the email recipient (and the ERDDAP administrator) will be able to figure out that you did it.\n");

            //link to list of subscriptions
            writer.write(requestSubscriptionListHtml(tErddapUrl, tEmail));
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }

    /**
     * Process erddap/subscriptions/list.html
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param ipAddress the requestor's ip address
     * @param protocol is always subscriptions
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doListSubscriptions(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String ipAddress, String protocol, int datasetIDStartsAt, String userQuery) 
        throws Throwable {

        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //process the query
        String tEmail = queryMap.get("email");
        if (tEmail == null) tEmail = "";
        String trouble = "";
        if      (tEmail.length() == 0)                                  trouble += "<li> Your email address wasn't specified.\n";
        else if (tEmail.length() > Subscriptions.EMAIL_LENGTH)          trouble += "<li> Your email address is too long.\n";
        else if (!String2.isEmailAddress(tEmail))                       trouble += "<li> Your email address isn't valid.\n";

        //display start of web page
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
        widgets.enterTextSubmitsForm = true; 
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "List Subscriptions", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol, "list") +
                EDStatic.subscriptionHtml(tErddapUrl) + "\n");

            if (tEmail.length() > 0) {
                if (trouble.length() > 0) {
                    writer.write(
                        "<p><font color=\"#FF0000\"><b>" +
                        "Your request for an email with a list of your subscriptions had one or more errors:</b></font>\n" +
                        "<ul>\n" +
                        trouble + "\n" +
                        "</ul>\n");
                } else {
                    //try to list the subscriptions
                    try {
                        String tList = EDStatic.subscriptions.listSubscriptions(ipAddress, tEmail);
                        String tError = EDStatic.email(tEmail, "Subscriptions List", tList);
                        if (tError.length() > 0)
                            throw new SimpleException(tError);

                        writer.write(
                            "<p><font color=\"#008800\"><b>" +
                            "Your request for an email with a list of your subscriptions has been submitted successfully.</b></font>\n" +
                            "<br>You should get an email with the list soon.\n");
                        //end of document
                        endHtmlWriter(out, writer, tErddapUrl, false);

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "List successful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "List successful");
                        return;
                    } catch (Throwable t) {
                        writer.write(
                            "<p><font color=\"#FF0000\"><b>" +
                            "Your request for an email with a list of your subscriptions had an error:</b></font>\n" +
                            "<br>" + XML.encodeAsXML(MustBe.getShortErrorMessage(t)) + "\n");
                        String2.log("Subscription list Exception:\n" + MustBe.throwableToString(t)); //log the details

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "List unsuccessful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "List unsuccessful");
                    }
                }
            }

            //show the form
            writer.write(
                widgets.beginForm("listSub", "GET", tErddapUrl + "/" + Subscriptions.LIST_HTML, "") +
                "<p><b>To request an email with a list of your subscriptions, please fill out this form:</b>\n" + 
                widgets.beginTable(0, 0, "") +
                "<tr>\n" +
                "  <td>Your email address:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("email", "", 
                    60, Subscriptions.EMAIL_LENGTH, tEmail, 
                    "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>" + widgets.button("submit", null, 
                    "Click to submit the form to the server.", "Submit", "") + "</td>\n" +
                "</tr>\n" +
                widgets.endTable() +  
                widgets.endForm() +
                "<font color=\"#FF0000\">Don't abuse this system by entering someone else's email address!</font>\n" +
                "<br>If you do, you may be blocked from using ERDDAP.\n" +
                "<br>Your IP address will be included in the email with the subscription list.\n" +
                "<br>So if you enter someone else's email address,\n" +
                "<br>the email recipient (and the ERDDAP administrator) will be able to figure out that you did it.\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }

    /**
     * Process erddap/subscriptions/validate.html
     *
     * @param protocol is always subscriptions
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doValidateSubscription(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String protocol, int datasetIDStartsAt, String userQuery) throws Throwable {

        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //process the query
        String tSubscriptionID = queryMap.get("subscriptionid"); //lowercase since case insensitive
        String tKey            = queryMap.get("key");
        if (tSubscriptionID == null) tSubscriptionID = "";
        if (tKey            == null) tKey            = "";
        String trouble = "";
        if      (tSubscriptionID.length() == 0)            trouble += "<li> The datasetID wasn't specified.\n";
        else if (!tSubscriptionID.matches("[0-9]{1,10}"))  trouble += "<li> The datasetID isn't valid.\n";
        if      (tKey.length() == 0)                       trouble += "<li> The key wasn't specified.\n";
        else if (!tKey.matches("[0-9]{1,10}"))             trouble += "<li> The key isn't valid.\n";

        //display start of web page
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
        widgets.enterTextSubmitsForm = true; 
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Validate a Subscription", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol, "validate") +
                EDStatic.subscriptionHtml(tErddapUrl) + "\n");

            String another = "a";
            if (tSubscriptionID.length() > 0 || tKey.length() > 0) {
                if (trouble.length() > 0) {
                    writer.write(
                        "<p><font color=\"#FF0000\"><b>" +
                        "Your request to validate a subscription had one or more errors:</b></font>\n" +
                        "<ul>\n" +
                        trouble + "\n" +
                        "</ul>\n");
                } else {
                    //try to validate 
                    try {
                        String message = EDStatic.subscriptions.validate(
                            String2.parseInt(tSubscriptionID), String2.parseInt(tKey));
                        if (message.length() > 0) {
                            writer.write(
                                "<p><font color=\"#FF0000\"><b>" +
                                "Your request to validate a subscription had an error:</b></font><ul>\n" +
                                message + "\n");

                        } else {writer.write(
                            "<p><font color=\"#008800\"><b>" +
                            "Your request to validate a subscription was successful.</b></font>\n");

                            //tally
                            EDStatic.tally.add("Subscriptions (since startup)", "Validate successful");
                            EDStatic.tally.add("Subscriptions (last 24 hours)", "Validate successful");
                        }
                        another = "another";
                    } catch (Throwable t) {
                        writer.write(
                            "<p><font color=\"#FF0000\"><b>Your validate-subscription request had an error:</b></font>\n" +
                            "<br>" + XML.encodeAsXML(MustBe.getShortErrorMessage(t)) + "\n");
                        String2.log("Subscription validate Exception:\n" + MustBe.throwableToString(t));

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "Validate unsuccessful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "Validate unsuccessful");
                    }
                }
            }

            //show the form
            writer.write(
                widgets.beginForm("validateSub", "GET", tErddapUrl + "/" + Subscriptions.VALIDATE_HTML, "") +
                "<p><b>To validate " + another + " subscription, please fill out this form:</b>\n" + 
                "<br>Normally, you won't fill out this form by hand. \n" +
                "<br>You will click on a link in the email that ERDDAP sends you after you request to \n" +
                "  <a href=\"" + tErddapUrl + "/" + Subscriptions.ADD_HTML + "\">add a subscription</a>.\n" +
                widgets.beginTable(0, 0, "") +
                "<tr>\n" +
                "  <td>The subscriptionID:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("subscriptionID", "", 
                    15, 15, tSubscriptionID, "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>The key:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("key", "", 
                    15, 15, tKey, "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>" + widgets.button("submit", null, 
                    "Click to submit the form to the server.", "Submit", "") + "</td>\n" +
                "</tr>\n" +
                widgets.endTable() +  
                widgets.endForm());        

            //link to list of subscriptions
            writer.write(requestSubscriptionListHtml(tErddapUrl, ""));
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }


    /**
     * Process erddap/subscriptions/remove.html
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param protocol is always subscriptions
     * @param datasetIDStartsAt is the position right after the / at the end of the protocol
     *    (always "info") in the requestUrl
     * @param userQuery  post "?", still percentEncoded, may be null.
     * @throws Throwable if trouble
     */
    public void doRemoveSubscription(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String protocol, int datasetIDStartsAt, String userQuery) 
        throws Throwable {

        String requestUrl = request.getRequestURI();  //post EDStatic.baseUrl, pre "?"
        String endOfRequestUrl = datasetIDStartsAt >= requestUrl.length()? "" : 
            requestUrl.substring(datasetIDStartsAt);
        HashMap<String, String> queryMap = EDD.userQueryHashMap(userQuery, true); //true=names toLowerCase
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //process the query
        String tSubscriptionID = queryMap.get("subscriptionid"); //lowercase since case insensitive
        String tKey            = queryMap.get("key");
        if (tSubscriptionID == null) tSubscriptionID = "";
        if (tKey            == null) tKey            = "";
        String trouble = "";
        if      (tSubscriptionID.length() == 0)            trouble += "<li> The datasetID wasn't specified.\n";
        else if (!tSubscriptionID.matches("[0-9]{1,10}"))  trouble += "<li> The datasetID isn't valid.\n";
        if      (tKey.length() == 0)                       trouble += "<li> The key wasn't specified.\n";
        else if (!tKey.matches("[0-9]{1,10}"))             trouble += "<li> The key isn't valid.\n";

        //display start of web page
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
        widgets.enterTextSubmitsForm = true; 
        OutputStream out = getHtmlOutputStream(request, response);
        Writer writer = getHtmlWriter(loggedInAs, "Remove a Subscription", out); 
        try {
            writer.write(
                EDStatic.youAreHere(tErddapUrl, protocol, "remove") +
                EDStatic.subscriptionHtml(tErddapUrl) + "\n");

            String another = "a";
            if (tSubscriptionID.length() > 0 || tKey.length() > 0) {
                if (trouble.length() > 0) {
                    writer.write(
                        "<p><font color=\"#FF0000\"><b>" +
                        "Your request to remove a subscription had one or more errors:</b></font>\n" +
                        "<ul>\n" +
                        trouble + "\n" +
                        "</ul>\n");
                } else {
                    //try to remove 
                    try {
                        String message = EDStatic.subscriptions.remove(
                            String2.parseInt(tSubscriptionID), String2.parseInt(tKey));
                        if (message.length() > 0) 
                            writer.write(
                                "<p><font color=\"#FF0000\"><b>" +
                                "Your request to remove a subscription had an error:</b></font>\n" +
                                "<br>" + message + "\n");
                        else writer.write(
                            "<p><font color=\"#008800\"><b>" +
                            "Your request to remove a subscription was successful.</b></font>\n");
                        another = "another";

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "Remove successful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "Remove successful");
                    } catch (Throwable t) {
                        writer.write(
                            "<p><font color=\"#FF0000\"><b>Your remove-subscription request had an error:</b></font>\n" +
                            "<br>" + XML.encodeAsXML(MustBe.getShortErrorMessage(t)) + "\n");
                        String2.log("Subscription remove Exception:\n" + MustBe.throwableToString(t)); //log the details

                        //tally
                        EDStatic.tally.add("Subscriptions (since startup)", "Remove unsuccessful");
                        EDStatic.tally.add("Subscriptions (last 24 hours)", "Remove unsuccessful");
                    }
                }
            }

            //show the form
            writer.write(
                widgets.beginForm("removeSub", "GET", tErddapUrl + "/" + Subscriptions.REMOVE_HTML, "") +
                "<p><b>To remove " + another + " subscription, please fill out this form:</b>\n" + 
                "<br>Normally, you won't fill out this form by hand. \n" +
                "<br>You will click on the link in the email that ERDDAP sends you after you request a \n" +
                "  <a href=\"" + tErddapUrl + "/" + Subscriptions.LIST_HTML + "\">list of your subscriptions</a>.\n" +
                widgets.beginTable(0, 0, "") +
                "<tr>\n" +
                "  <td>The subscriptionID:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("subscriptionID", "", 
                    15, 15, tSubscriptionID, "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>The key:&nbsp;</td>\n" +
                "  <td>" + widgets.textField("key", "", 
                    15, 15, tKey, "") + "</td>\n" +
                "</tr>\n" +
                "<tr>\n" +
                "  <td>" + widgets.button("submit", null, 
                    "Click to submit the form to the server.", "Submit", "") + "</td>\n" +
                "</tr>\n" +
                widgets.endTable() +  
                widgets.endForm());        

            //link to list of subscriptions
            writer.write(requestSubscriptionListHtml(tErddapUrl, "") +
                "<br>It includes URLs to remove each of your subscriptions.\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        //end of document
        endHtmlWriter(out, writer, tErddapUrl, false);
    }


    /**
     * This indicates if the string 's' equals 'start' (e.g., "index") 
     * plus one of the plain file types.
     */
    protected static boolean endsWithPlainFileType(String s, String start) {
        for (int pft = 0; pft < plainFileTypes.length; pft++) { 
            if (s.equals(start + plainFileTypes[pft]))
                return true;
        }
        return false;
    }

    /**
     * Set the standard DAP header information. Call this before getting outputStream.
     *
     * @param response
     * @throws Throwable if trouble
     */
    public void standardDapHeader(HttpServletResponse response) throws Throwable {
        String rfc822date = Calendar2.getCurrentRFC822Zulu();
        response.setHeader("Date", rfc822date);             //DAP 2.0, 7.1.4.1
        response.setHeader("Last-Modified", rfc822date);    //DAP 2.0, 7.1.4.2   //this is not a good implementation
        //response.setHeader("Server", );                   //DAP 2.0, 7.1.4.3  optional
        response.setHeader("xdods-server", serverVersion);  //DAP 2.0, 7.1.7 (http header field names are case-insensitive)
        response.setHeader(EDStatic.programname + "-server", EDStatic.erddapVersion);  
    }

    
    /**
     * Get an outputStream for an html file
     *
     * @param request
     * @param response
     * @return an outputStream
     * @throws Throwable if trouble
     */
    public static OutputStream getHtmlOutputStream(HttpServletRequest request, HttpServletResponse response) 
        throws Throwable {

        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, "index", ".html", ".html");
        return outSource.outputStream("UTF-8");
    }

    /**
     * Get a writer for an html file and write up to and including the startHtmlBody
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param out
     * @return writer
     * @throws Throwable if trouble
     */
    Writer getHtmlWriter(String loggedInAs, String addToTitle, OutputStream out) throws Throwable {

        Writer writer = new OutputStreamWriter(out, "UTF-8");

        //write the information for this protocol (dataset list table and instructions)
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        writer.write(EDStatic.startHeadHtml(tErddapUrl, addToTitle));
        writer.write(EDStatic.standardHead);
        writer.write("\n</head>\n");
        writer.write(EDStatic.startBodyHtml(loggedInAs));
        writer.write("\n");
        writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl));
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
        return writer;
    }

    /**
     * Write the end of the standard html doc to writer.
     *
     * @param out
     * @param writer
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param forceWriteDiagnostics
     * @throws Throwable if trouble
     */
    void endHtmlWriter(OutputStream out, Writer writer, String tErddapUrl,
        boolean forceWriteDiagnostics) throws Throwable {

        writer.write("<br>&nbsp;\n" +
            "<br><font color=\"#444444\"><small>" + 
            EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small></font>\n");

        //add the diagnostic info  
        if (EDStatic.displayDiagnosticInfo || forceWriteDiagnostics) 
            EDStatic.writeDiagnosticInfoHtml(writer);

        //end of document
        writer.write(EDStatic.endBodyHtml(tErddapUrl));
        writer.write("\n</html>\n");

        //essential
        writer.flush();
        if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
        out.close();         
    }


    /**
     * This writes the error (if not null or "") to the html writer.
     *
     * @param writer
     * @param request
     * @param error plain text, will be html-encoded here
     * @throws Throwable if trouble
     */
    void writeErrorHtml(Writer writer, HttpServletRequest request, String error) throws Throwable {
        if (error == null || error.length() == 0) 
            return;
        int colonPo = error.indexOf(": ");
        if (colonPo >= 0 && colonPo < error.length() - 5)
            error = error.substring(colonPo + 2);
        String query = SSR.percentDecode(request.getQueryString()); //percentDecode returns "" instead of null
        String requestUrl = request.getRequestURI();
        if (requestUrl == null) 
            requestUrl = "";
        if (requestUrl.startsWith("/"))
            requestUrl = requestUrl.substring(1);
        //encodeAsPreXML(error) is essential -- to prevent Cross-site-scripting security vulnerability
        //(which allows hacker to insert his javascript into pages returned by server)
        //See Tomcat (Definitive Guide) pg 147
        error = XML.encodeAsPreXML(error, 110);
        int brPo = error.indexOf("<br> at ");
        if (brPo < 0) 
            brPo = error.indexOf("<br>at ");
        if (brPo < 0) 
            brPo = error.length();
        writer.write(
            "<h2>" + EDStatic.errorTitle + "</h2>\n" +
            "<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
            "<tr>\n" +
            "  <td nowrap>" + EDStatic.errorRequestUrl + "&nbsp;</td>\n" +
            //encodeAsXML(query) is essential -- to prevent Cross-site-scripting security vulnerability
            //(which allows hacker to insert his javascript into pages returned by server)
            //See Tomcat (Definitive Guide) pg 147
            "  <td nowrap>" + EDStatic.baseUrl + "/" + XML.encodeAsXML(requestUrl) + "</td>\n" +
            "</tr>\n" +
            "<tr>\n" +
            "  <td nowrap>" + EDStatic.errorRequestQuery + "&nbsp;</td>\n" +
            //encodeAsXML(query) is essential -- to prevent Cross-site-scripting security vulnerability
            //(which allows hacker to insert his javascript into pages returned by server)
            //See Tomcat (Definitive Guide) pg 147
            "  <td nowrap>" + (query.length() == 0? "&nbsp;" : XML.encodeAsXML(query)) + "</td>\n" +
            "</tr>\n" +
            "<tr>\n" +
            "  <td valign=\"top\" nowrap><b>" + EDStatic.errorTheError + "</b>&nbsp;</td>\n" +
            "  <td><b>" + error.substring(0, brPo) + "</b>" + 
                error.substring(brPo) + "</td>\n" + //not nowrap
            "</tr>\n" +
            "</table>\n" +
            "<br>&nbsp;");
        /* old versions
        writer.write(
            "<p><b>There was an error in your request:</b>\n" +
            "<br>&nbsp; &nbsp;Your request URL: " + EDStatic.baseUrl + request.getRequestURI() + "\n" +
            "<br>&nbsp; &nbsp;Your request query: " + (query == null || userQuery.length() == 0? "" : query) + "\n" +
            "<br>&nbsp; &nbsp;The error: <b>" + error + "</b>\n");
        writer.write(
            "<p><b>Your request URL:</b> " + EDStatic.baseUrl + request.getRequestURI() + "\n" +
            "<p><b>Your request query:</b> " + (query == null || userQuery.length() == 0? "" : query) + "\n" +
            "<p><b>There was an error in your request:</b>\n" +
            "<br>" + error + "\n");
        */
    }


    /**
     * This is the first step in handling an exception/error.
     * If this returns true or throws Throwable, that is all that can be done: caller should call return.
     * If this returns false, the caller can/should handle the exception (response.isCommitted() is false);
     *
     * @returns false if response !isCommitted() and caller needs to handle the error 
     *   (e.g., send the desired type of error message)
     *   (this logs the error to String2.log).
     *   This currently doesn't return true.
     * @throw Throwable if response isCommitted(), t was rethrown.
     */
    public static boolean neededToSendErrorCode(HttpServletRequest request, 
        HttpServletResponse response, Throwable t) throws Throwable {
            
        if (response.isCommitted()) {
            //rethrow exception (will be handled in doGet try/catch)
            throw t;
        }

        //just log it
        String q = request.getQueryString();
        String message = ERROR + " for " + request.getRequestURI() +  
            (q != null && q.length() > 0? "?" + q : "") + //not decoded
            "\n" + MustBe.throwableToString(t); //log the details
        String2.log(message);
        return false;
    }

    /**
     * This calls response.sendError(500 INTERNAL_SERVER_ERROR, MustBe.throwableToString(t)).
     * Return after calling this.
     */
    public static void sendErrorCode(HttpServletRequest request, 
        HttpServletResponse response, Throwable t) throws ServletException {
        
        String tError = MustBe.getShortErrorMessage(t);

        try {

            //log the error            
            String q = request.getQueryString();
            String2.log(
                "*** sendErrorCode ERROR for " +
                    request.getRequestURI() + (q != null && q.length() > 0? "?" + q : "") + //not decoded
                "\nisCommitted=" + response.isCommitted() +
                "\n" + MustBe.throwableToString(t));  //always log full stack trace

            //try to writeit to html page
            if (response.isCommitted()) 
                //nothing can be done
                return;

            //we can send the error code
            int errorNo = 
                HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
            response.sendError(errorNo, tError); 
            return;

        } catch (Throwable t2) {
            //an exception occurs if response is committed
            throw new ServletException(t2);
        }
    }

    /**
     * This sends the HTTP resource NOT_FOUND error.
     * This always also sends the error to String2.log.
     *
     * @param message  use "" if nothing specific.
     *    The requestURI will always be pre-pended to the message.
     */
    public static void sendResourceNotFoundError(HttpServletRequest request, 
        HttpServletResponse response, String message) throws Throwable {

        try {
            message = (message == null || message.length() == 0)?
                request.getRequestURI() :
                request.getRequestURI() + " (" + message + ")";
            String2.log("Calling response.sendError(404 - SC_NOT_FOUND):\n" + message);
            response.sendError(HttpServletResponse.SC_NOT_FOUND, 
                "Resource not available: " + message);
        } catch (Throwable t) {
            throw new SimpleException("Resource not available: " + message);
        }
    }


    /** 
     * This writes the html for a search form to the writer.
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param writer
     * @param headerLevel 2 or 3, to make h2 or h3 tag
     * @param searchFor the default text to be searched for
     * @throws Throwable if trouble
     */
    public static void writeSearchFormHtml(String tErddapUrl, Writer writer, 
        int headerLevel, String searchFor) throws Throwable {

        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl); //true=htmlTooltips
        widgets.enterTextSubmitsForm = true;
        writer.write("<h" + headerLevel + "><a name=\"FullTextSearch\">" + EDStatic.searchFullTextHtml + "</a></h" + headerLevel + ">\n");
        writer.write(widgets.beginForm("search", "GET", tErddapUrl + "/search/index.html", ""));
        if (searchFor == null)
            searchFor = "";
        widgets.htmlTooltips = false;
        writer.write(widgets.textField("searchFor", EDStatic.searchTip, 40, 255, searchFor, ""));
        widgets.htmlTooltips = true;
        writer.write(EDStatic.htmlTooltipImage(EDStatic.searchHintsHtml(tErddapUrl)));
        widgets.htmlTooltips = false;
        writer.write(widgets.button("submit", null, EDStatic.searchClickTip, EDStatic.searchButton, ""));
        widgets.htmlTooltips = true;
        //writer.write(EDStatic.searchHintsHtml(tErddapUrl));
        writer.write("\n");
        writer.write(widgets.endForm());        
    }


    /** 
     * This returns a table with categorize options.
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param fileTypeName .html or a plainFileType e.g., .htmlTable
     * @return a table with categorize options.
     * @throws Throwable if trouble
     */
    public Table categorizeOptionsTable(String tErddapUrl, String fileTypeName) throws Throwable {

        Table table = new Table();
        StringArray csa = new StringArray();
        table.addColumn("Categorize", csa);
        if (fileTypeName.equals(".html")) {
            //1 column: links
            for (int cat = 0; cat < EDStatic.categoryAttributes.length; cat++) {
                String s = tErddapUrl + "/categorize/" + EDStatic.categoryAttributes[cat] + "/index.html";
                csa.add("<a href=\"" + s + "\">" + EDStatic.categoryAttributes[cat] + "</a>");
            }
        } else {
            //2 columns: categorize, url
            StringArray usa = new StringArray();
            table.addColumn("URL", usa);
            for (int cat = 0; cat < EDStatic.categoryAttributes.length; cat++) {
                csa.add(EDStatic.categoryAttributes[cat]);
                usa.add(tErddapUrl + "/categorize/" + EDStatic.categoryAttributes[cat] + "/index" + fileTypeName);
            }
        }
        return table;
    }

    /**
     * This writes the categorize options table
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param writer
     * @param homePage
     */
    public void writeCategorizeOptionsHtml1(String tErddapUrl, Writer writer, 
        boolean homePage) throws Throwable {

        Table table = categorizeOptionsTable(tErddapUrl, ".html");
        int n = table.nRows();

        if (homePage) {
            writer.write(
                "<h3><a name=\"SearchByCategory\">" + EDStatic.categoryTitleHtml + 
                "</a></h3>\n" +
                EDStatic.category1Html + " (");
            for (int row = 0; row < n; row++) 
                writer.write(table.getStringData(0, row) + (row < n - 1? ", " : ""));
            writer.write(") " + EDStatic.category2Html + "\n");
            writer.write(EDStatic.category3Html + "\n");
            return;
        }

        //categorize page
        writer.write(
            //"<h2>" + EDStatic.categoryTitleHtml + "</h2>\n" +
            "<h3>1) Pick an attribute: &nbsp; " + 
            EDStatic.htmlTooltipImage(EDStatic.category1Html + EDStatic.category2Html) +
            "</h3>\n" +
            "<table class=\"erd\" bgcolor=\"" + EDStatic.tableBGColor + "\" cellspacing=\"0\">\n" +
            "  <tr>\n");
        for (int row = 0; row < n; row++) 
            writer.write("    <td>&nbsp;" + table.getStringData(0, row) + "&nbsp;</td>\n");
        writer.write(
            "  </tr>\n" +
            "</table>\n\n");
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better

    }

    /** 
     * This writes the html with the category options to the writer (in a table with lots of columns).
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param writer
     * @param categoryAttribute must be valid  (e.g., ioos_category)
     * @throws Throwable if trouble
     */
    public void writeCategoryOptionsHtml2(String tErddapUrl, Writer writer, 
        String categoryAttribute) throws Throwable {

        StringArray cats = categoryInfo(categoryAttribute);
        int nCats = cats.size();

        //find longest cat  and calculate nCols
        int max = Math.max(1, cats.maxStringLength());
        //table width never more than 150 chars.      e.g., 150/30 -> 5 cols;  150/31 -> 4 cols 
        int nCols = 150 / max; 
        nCols = Math.min(nCats, nCols); //never more cols than nCats
        nCols = Math.max(1, nCols);     //always at least 1 col
        //String2.log("  writeCategoryOptionsHtml max=" + max + " nCols=" + nCols);
        int nRows = Math2.hiDiv(nCats, nCols);

        //write the table
        String catTitle = EDStatic.categorySearchHtml;
        catTitle = String2.replaceAll(catTitle, "&category;", categoryAttribute);
        writer.write(
            "<h3>2) " + catTitle + ": &nbsp; " +
            EDStatic.htmlTooltipImage(EDStatic.categoryClickHtml) +
            "</h3>\n");
        writer.write(
            "<table class=\"erd\" bgcolor=\"" + EDStatic.tableBGColor + "\" cellspacing=\"0\">\n"); 
            //"<table border=\"1\" bgcolor=\"" + EDStatic.tableBGColor + "\"" + 
            //"  cellspacing=\"0\" cellpadding=\"2\">\n");

        //organized to be read top to bottom, then left to right
        //interesting case: nCats=7, nCols=6
        //   so nRows=2, then only need nCols=4; so modify nCols   
        nCols = Math2.hiDiv(nCats, Math.max(1, nRows));
        for (int row = 0; row < nRows; row++) {
            writer.write("  <tr>\n");
            for (int col = 0; col < nCols; col++) {
                writer.write("    <td nowrap>");
                int i = col * nRows + row;
                if (i < nCats) { 
                    String tc = cats.get(i); //e.g., Temperature
                    writer.write(
                        "&nbsp;<a href=\"" + tErddapUrl + "/categorize/" + categoryAttribute + "/" + 
                        tc + "/index.html\">" + tc + "</a>&nbsp;");
                } else {
                    writer.write("&nbsp;");
                }
                writer.write("</td>\n");
            }
            writer.write("  </tr>\n");
        }

        writer.write("</table>\n");
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
    }


    /** 
     * This sens a response: a table with two columns (Category, URL).
     *
     * @param request
     * @param response
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param categoryAttribute must be valid  (e.g., ioos_category)
     * @param fileTypeName a plainFileType, e.g., .htmlTable
     * @throws Throwable if trouble
     */
    public void sendCategoryPftOptionsTable(HttpServletRequest request, HttpServletResponse response, 
        String loggedInAs, String categoryAttribute, String fileTypeName) throws Throwable {

        //public String categoryNames[][];      //[eg. 0=ioos_category][e.g., 3=Temperature]
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        StringArray cats = categoryInfo(categoryAttribute); //already safe
        int nCats = cats.size();

        //make the table
        Table table = new Table();
        StringArray catCol = new StringArray();
        StringArray urlCol = new StringArray();
        table.addColumn("Category", catCol);
        table.addColumn("URL", urlCol);
        for (int i = 0; i < nCats; i++) {
            String cat = cats.get(i); //e.g., Temperature    already safe
            catCol.add(cat);
            urlCol.add(tErddapUrl + "/categorize/" + categoryAttribute + "/" + cat + "/index" + fileTypeName); 
        }

        //send it  
        sendPlainTable(request, response, table, categoryAttribute, fileTypeName);
    }

    /**
     * Given a list of datasetIDs, this makes a sorted table of the datasets info.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDs the id's of the datasets (e.g., "pmelTao") that should be put into the table
     * @param sortByTitle if true, rows will be sorted by title.
     *    If false, they are left in order of datasetIDs.
     * @param fileTypeName the file type name (e.g., ".htmlTable") to use for info links
     * @return table a table with plain text information about the datasets
     */
    public Table makePlainDatasetTable(String loggedInAs, 
        StringArray datasetIDs, boolean sortByTitle, String fileTypeName) {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String roles[] = EDStatic.getRoles(loggedInAs);
        boolean isLoggedIn = loggedInAs != null;
        Table table = new Table();
        StringArray gdCol = new StringArray();
        StringArray tdCol = new StringArray();
        StringArray magCol = new StringArray();
        StringArray wmsCol = new StringArray();
        StringArray accessCol = new StringArray();
        StringArray titleCol = new StringArray();
        StringArray summaryCol = new StringArray();
        StringArray infoCol = new StringArray();
        StringArray backgroundCol = new StringArray();
        StringArray rssCol = new StringArray();
        StringArray emailCol = new StringArray();
        StringArray institutionCol = new StringArray();
        StringArray idCol = new StringArray();  //useful for java programs
        table.addColumn("griddap", gdCol);  //just protocol name
        table.addColumn("tabledap", tdCol);
        table.addColumn("Make A Graph", magCol);
        table.addColumn("wms", wmsCol);
        if (EDStatic.authentication.length() > 0)
            table.addColumn("Accessible", accessCol);
        int sortOn = table.addColumn("Title", titleCol);
        table.addColumn("Summary", summaryCol);
        table.addColumn("Info", infoCol);
        table.addColumn("Background Info", backgroundCol);
        table.addColumn("RSS", rssCol);
        table.addColumn("Email", emailCol);
        table.addColumn("Institution", institutionCol);
        table.addColumn("Dataset ID", idCol);
        for (int i = 0; i < datasetIDs.size(); i++) {
            String tId = datasetIDs.get(i);
            EDD edd = (EDD)gridDatasetHashMap.get(tId);
            if (edd == null) 
                edd = (EDD)tableDatasetHashMap.get(tId);
            if (edd == null) //perhaps just deleted
                continue;
            boolean isAccessible = edd.isAccessibleTo(roles);
            if (!EDStatic.listPrivateDatasets && !isAccessible)
                continue;

            String daps = tErddapUrl + "/" + edd.dapProtocol() + "/" + tId; //without an extension, so easy to add
            gdCol.add(edd instanceof EDDGrid? daps : "");
            tdCol.add(edd instanceof EDDTable? daps : "");
            magCol.add(daps + ".graph");
            wmsCol.add(edd.accessibleViaWMS()? 
                tErddapUrl + "/wms/" + tId + "/request": "");
            accessCol.add(edd.getAccessibleTo() == null? "public" :
                !isLoggedIn? "log in" :
                isAccessible? "yes" : "no");
            titleCol.add(edd.title());
            summaryCol.add(edd.summary());
            infoCol.add(tErddapUrl + "/info/" + edd.datasetID() + "/index" + fileTypeName);
            backgroundCol.add(edd.infoUrl());
            rssCol.add(EDStatic.erddapUrl + "/rss/" + edd.datasetID()+ ".rss"); //never https url
            emailCol.add(EDStatic.subscriptionSystemActive?
                tErddapUrl + "/" + Subscriptions.ADD_HTML + 
                    "?datasetID=" + edd.datasetID()+ "&showErrors=false&email=" : 
                "");
            institutionCol.add(edd.institution());
            idCol.add(tId);
        }
        if (sortByTitle)
            table.sort(new int[]{sortOn}, new boolean[]{true});
        return table;
    }

    /**
     * Given a list of datasetIDs, this makes a sorted table of the datasets info.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in)
     * @param datasetIDs the id's of the datasets (e.g., "pmelTao") that should be put into the table
     * @param sortByTitle if true, rows will be sorted by title.
     *    If false, they are left in order of datasetIDs.
     * @return table a table with html-formatted information about the datasets
     */
    public Table makeHtmlDatasetTable(String loggedInAs,
        StringArray datasetIDs, boolean sortByTitle) {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String roles[] = EDStatic.getRoles(loggedInAs);
        boolean isLoggedIn = loggedInAs != null;
        Table table = new Table();
        StringArray gdCol = new StringArray();
        StringArray tdCol = new StringArray();
        StringArray magCol = new StringArray();
        StringArray wmsCol = new StringArray();
        StringArray accessCol = new StringArray();
        StringArray titleCol = new StringArray();
        StringArray summaryCol = new StringArray();
        StringArray infoCol = new StringArray();
        StringArray backgroundCol = new StringArray();
        StringArray rssCol = new StringArray();  
        StringArray emailCol = new StringArray(); 
        StringArray institutionCol = new StringArray();
        StringArray idCol = new StringArray();  //useful for java programs
        table.addColumn("Grid<br>DAP<br>Data", gdCol);
        table.addColumn("Table<br>DAP<br>Data", tdCol);
        table.addColumn("Make<br>A<br>Graph", magCol);
        table.addColumn("W<br>M<br>S", wmsCol);
        String accessTip = 
            "<b>Is this dataset accessible to you?</b>\n" +
            "<br>\"public\" = Yes.  It is accessible to anyone (logged in or not logged in).\n";
        if (isLoggedIn)
            accessTip += "<br>\"yes\" = Yes. You are logged in and have the appropriate role.";
        if (EDStatic.authentication.length() > 0 && EDStatic.listPrivateDatasets) //this erddap supports logging in
            accessTip += isLoggedIn?
                "<br>\"no\" = No. You are logged in but don't have the appropriate role." :
                "<br>\"log in\" = Currently, no. It may become accessible if you log in.";
        if (EDStatic.authentication.length() > 0)
            table.addColumn("Acces-<br>sible<br>" + EDStatic.htmlTooltipImage(accessTip),
                accessCol);
        String loginHref = EDStatic.authentication.length() == 0? "no" :
            "<a href=\"" + EDStatic.erddapHttpsUrl + "/login.html\" " +
            "title=\"Click to log in and try to make this dataset accessible.\">log in</a>";
        int sortOn = table.addColumn("Title", titleCol);
        table.addColumn("Summary", summaryCol);
        table.addColumn("Info", infoCol);
        table.addColumn("Background<br>Info", backgroundCol);
        table.addColumn("RSS", rssCol);
        table.addColumn("E<br>mail", emailCol);
        table.addColumn("Institution", institutionCol);
        table.addColumn("Dataset ID", idCol);
        for (int i = 0; i < datasetIDs.size(); i++) {
            String tId = datasetIDs.get(i);
            EDD edd = (EDD)gridDatasetHashMap.get(tId);
            if (edd == null)
                edd = (EDD)tableDatasetHashMap.get(tId);
            if (edd == null)  //if just deleted
                continue; 
            boolean isAccessible = edd.isAccessibleTo(roles);
            if (!EDStatic.listPrivateDatasets && !isAccessible)
                continue;

            String daps = "&nbsp;<a href=\"" + tErddapUrl + "/" + edd.dapProtocol() + 
                "/" + tId + ".html\" " +
                "title=\"Click to see a " + edd.dapProtocol() + 
                  " Data Access Form for this dataset so that you can request data.\" " +
                ">data</a>&nbsp;"; 
            gdCol.add(edd instanceof EDDGrid?  daps : "&nbsp;"); 
            tdCol.add(edd instanceof EDDTable? daps : "&nbsp;");
            magCol.add(" &nbsp;<a href=\"" + tErddapUrl + "/" + edd.dapProtocol() + 
                "/" + tId + ".graph\" " +
                "title=\"Click to see Make A Graph for this dataset.\" " +
                ">graph</a>");
            wmsCol.add(edd.accessibleViaWMS()? 
                "&nbsp;<a href=\"" + tErddapUrl + "/wms/" + tId + "/index.html\" " +
                    "title=\"Click to find out about making maps of this data via ERDDAP's Web Map Service.\" >" +
                    "M</a>&nbsp;" : 
                "&nbsp;");
            accessCol.add(edd.getAccessibleTo() == null? "public" :
                !isLoggedIn? loginHref :
                isAccessible? "yes" : "no");
            titleCol.add(XML.encodeAsXML(edd.title()));
            summaryCol.add(EDStatic.htmlTooltipImage(XML.encodeAsPreXML(edd.summary(), 100)));
            infoCol.add("<a href=\"" + tErddapUrl + "/info/" + edd.datasetID() + 
                "/index.html\" " + //here, always .html
                "title=\"" + EDStatic.clickInfo + "\" >info</a>");
            backgroundCol.add("<a href=\"" + edd.infoUrl() + "\" " +
                "title=\"" + EDStatic.clickBackgroundInfo + "\" >background</a>");
            rssCol.add(edd.rssHref());
            emailCol.add("&nbsp;" + edd.emailHref(tErddapUrl) + "&nbsp;");
            institutionCol.add(XML.encodeAsXML(edd.institution()));
            idCol.add(tId);
        }
        if (sortByTitle) 
            table.sort(new int[]{sortOn}, new boolean[]{true});
        return table;
    }

    /**
     * This writes the plain (non-html) table as a plainFileType response.
     *
     * @param fileName e.g., Time
     * @param fileTypeName e.g., .htmlTable
     */
    void sendPlainTable(HttpServletRequest request, HttpServletResponse response, 
        Table table, String fileName, String fileTypeName) throws Throwable {

        int po = String2.indexOf(EDDTable.dataFileTypeNames, fileTypeName);
        String fileTypeExtension = EDDTable.dataFileTypeExtensions[po];

        OutputStreamSource outSource = new OutputStreamFromHttpResponse(
            request, response, fileName, fileTypeName, fileTypeExtension); 

        if (fileTypeName.equals(".htmlTable")) 
            TableWriterHtmlTable.writeAllAndFinish(table, outSource, fileName, false,
                "", "", true, false); //pre, post, encodeAsXML, writeUnits

        else if (fileTypeName.equals(".json")) {
            //did query include &.jsonp= ?
            String parts[] = EDD.getUserQueryParts(request.getQueryString());
            String jsonp = String2.stringStartsWith(parts, ".jsonp="); //may be null
            if (jsonp != null) 
                jsonp = SSR.percentDecode(jsonp.substring(7));

            TableWriterJson.writeAllAndFinish(table, outSource, jsonp, false); //writeUnits

        } else if (fileTypeName.equals(".csv")) 
            TableWriterSeparatedValue.writeAllAndFinish(table, outSource,
                ", ", true, false); //separator, quoted, writeUnits

        else if (fileTypeName.equals(".mat")) 
            //??? use goofy standard structure name (nice that it's always the same);
            //  could use fileName but often long
            table.saveAsMatlab(outSource.outputStream(""), "response");  

        else if (fileTypeName.equals(".tsv")) 
            TableWriterSeparatedValue.writeAllAndFinish(table, outSource, 
                "\t", false, false); //separator, quoted, writeUnits

        else if (fileTypeName.equals(".xhtml")) 
            TableWriterHtmlTable.writeAllAndFinish(table, outSource, fileName, true,
                "", "", true, false); //pre, post, encodeAsXML, writeUnits

        else throw new SimpleException("Unsupported fileType=" + fileTypeName + " at the end of the requestUrl."); 

        //essential
        OutputStream out = outSource.outputStream(""); 
        if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
        out.close(); 

    }

    /** THIS IS NO LONGER ACTIVE. USE sendErrorCode() INSTEAD.
     * This sends a plain error message. 
     * 
     */
    /*void sendPlainError(HttpServletRequest request, HttpServletResponse response, 
        String fileTypeName, String error) throws Throwable {

        if (fileTypeName.equals(".json")) {
            OutputStream out = (new OutputStreamFromHttpResponse(request, response, ERROR, ".json", ".json")).outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8");
            writer.write(
                "{\n" +
                "  \"" + ERROR + "\": " + String2.toJson(error) + "\n" +
                "}\n");

            //essential
            writer.flush();
            if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
            out.close(); 
            return;
        }

        if (fileTypeName.equals(".csv") ||
            fileTypeName.equals(".tsv") ||
            fileTypeName.equals(".htmlTable") ||
//better error format for .htmlTable and .xhtml???
            fileTypeName.equals(".xhtml")) {  

            OutputStream out = (new OutputStreamFromHttpResponse(request, response, ERROR, ".txt", ".txt")).outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8");
            if (!error.startsWith(ERROR))
                writer.write(ERROR + ": ");
            writer.write(error + "\n");

            //essential
            writer.flush();
            if (out instanceof ZipOutputStream) ((ZipOutputStream)out).closeEntry();
            out.close(); 
            return;
        }

        throw new SimpleException("Unsupported fileType=" + fileTypeName + " at the end of the requestUrl."); 
    }*/

    /**
     * This makes a erddapContent.zip file with the [tomcat]/content/erddap files for distribution.
     *
     * @param removeDir e.g., "c:/programs/tomcat/samples/"     
     * @param destinationDir  e.g., "c:/backup/"
     */
    public static void makeErddapContentZip(String removeDir, String destinationDir) throws Throwable {
        String2.log("*** makeErddapContentZip dir=" + destinationDir);
        String baseDir = removeDir + "content/erddap/";
        SSR.zip(destinationDir + "erddapContent.zip", 
            new String[]{
                baseDir + "datasets.xml",
                baseDir + "messages.xml",
                baseDir + "setup.xml",
                baseDir + "images/nlogo.gif",
                baseDir + "images/noaa20.gif",
                baseDir + "images/noaa_simple.gif",
                baseDir + "images/noaab.png",
                baseDir + "images/QuestionMark.jpg"},
            10, removeDir);
    }

    /**
     * This is an attempt to assist Tomcat/Java in shutting down erddap.
     * Tomcat/Java will call this; no one else should.
     * Java calls this when an object is no longer used, just before garbage collection. 
     * 
     */
    protected void finalize() throws Throwable {
        try {  //extra assistance/insurance
            EDStatic.destroy();   //but Tomcat should call ERDDAP.destroy, which calls EDStatic.destroy().
        } catch (Throwable t) {
        }
        super.finalize();
    }

    /**
     * This is used by Bob to do simple tests of the basic Erddap services 
     * from the ERDDAP at EDStatic.erddapUrl. It assumes Bob's test datasets are available.
     *
     */
    public static void test() throws Throwable {
        Erddap.verbose = true;
        Erddap.reallyVerbose = true;
        EDD.testVerboseOn();
        String results, expected;
        String2.log("\n*** Erddap.test");
        int po;

        try {
            //home page
            results = SSR.getUrlResponseString(EDStatic.erddapUrl); //redirects to index.html
            expected = "The small effort to set up ERDDAP brings many benefits.";
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/"); //redirects to index.html
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.html"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            
            //test version info  (opendap spec section 7.2.5)
            //"version" instead of datasetID
            expected = 
                "Core Version: DAP/2.0\n" +
                "Server Version: dods/3.7\n";
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/version");
            Test.ensureEqual(results, expected, "results=" + results);
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/version");
            Test.ensureEqual(results, expected, "results=" + results);

            //"version.txt"
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/version.txt");
            Test.ensureEqual(results, expected, "results=" + results);
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/version.txt");
            Test.ensureEqual(results, expected, "results=" + results);

            //".ver"
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/etopo180.ver");
            Test.ensureEqual(results, expected, "results=" + results);
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/erdGlobecBottle.ver");
            Test.ensureEqual(results, expected, "results=" + results);


            //help
            expected = "griddap to Request Data and Graphs from Gridded Datasets";
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/help"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/documentation.html"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/erdMHchla8day.help"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);


            expected = "tabledap to Request Data and Graphs from Tabular Datasets";
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/help"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/documentation.html"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/erdGlobecBottle.help"); 
            Test.ensureTrue(results.indexOf(expected) >= 0, "results=" + results);

            //error 404
            results = "";
            try {
                SSR.getUrlResponseString(EDStatic.erddapUrl + "/gibberish"); 
            } catch (Throwable t) {
                results = t.toString();
            }
            Test.ensureTrue(results.indexOf("java.io.FileNotFoundException") >= 0, "results=" + results);

            //info    list all datasets
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/info/index.html"); 
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("GLOBEC NEP Rosette Bottle Data (2002)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("SST, Blended, Global, EXPERIMENTAL (5 Day Composite)") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/info/index.csv"); 
            Test.ensureTrue(results.indexOf("</html>") < 0, "results=" + results);
            Test.ensureTrue(results.indexOf("GLOBEC NEP Rosette Bottle Data (2002)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("SST, Blended, Global, EXPERIMENTAL (5 Day Composite)") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/info/erdGlobecBottle/index.html"); 
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("ioos_category") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Location") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("long_name") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Cast Number") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/info/erdGlobecBottle/index.tsv"); 
            Test.ensureTrue(results.indexOf("\t") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("ioos_category") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Location") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("long_name") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Cast Number") >= 0, "results=" + results);


            //search    
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/search/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Do a Full Text Search for Datasets") >= 0, "results=" + results);
            //index.otherFileType must have ?searchFor=...

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/search/index.html?searchFor=all");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Title</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">RSS</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)</td>") >= 0,
                "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">GLOBEC NEP Rosette Bottle Data (2002)</td>") >= 0,
                "results=" + results);            
           
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/search/index.htmlTable?searchFor=all");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Title</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">RSS</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)</td>") >= 0,
                "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">GLOBEC NEP Rosette Bottle Data (2002)</td>") >= 0,
                "results=" + results);            
           
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/search/index.html?searchFor=tao+pmel");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">TAO Array Data from the PMEL DAPPER Server<") > 0,
                "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/search/index.tsv?searchFor=tao+pmel");
            Test.ensureTrue(results.indexOf("\tTAO Array Data from the PMEL DAPPER Server\t") > 0,
                "results=" + results);


            //categorize
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">standard_name<") >= 0,
                "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/index.json");
            Test.ensureEqual(results, 
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"Categorize\", \"URL\"],\n" +
"    \"columnTypes\": [\"String\", \"String\"],\n" +
"    \"rows\": [\n" +
"      [\"institution\", \"http://127.0.0.1:8080/cwexperimental/categorize/institution/index.json\"],\n" +
"      [\"ioos_category\", \"http://127.0.0.1:8080/cwexperimental/categorize/ioos_category/index.json\"],\n" +
"      [\"long_name\", \"http://127.0.0.1:8080/cwexperimental/categorize/long_name/index.json\"],\n" +
"      [\"standard_name\", \"http://127.0.0.1:8080/cwexperimental/categorize/standard_name/index.json\"]\n" +
"    ]\n" +
"  }\n" +
"}\n", 
                "results=" + results);

            //json with jsonp 
            String jsonp = "Some encoded {}\n() ! text";
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/index.json?.jsonp=" + SSR.percentEncode(jsonp));
            Test.ensureEqual(results, 
jsonp + "(" +
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"Categorize\", \"URL\"],\n" +
"    \"columnTypes\": [\"String\", \"String\"],\n" +
"    \"rows\": [\n" +
"      [\"institution\", \"http://127.0.0.1:8080/cwexperimental/categorize/institution/index.json\"],\n" +
"      [\"ioos_category\", \"http://127.0.0.1:8080/cwexperimental/categorize/ioos_category/index.json\"],\n" +
"      [\"long_name\", \"http://127.0.0.1:8080/cwexperimental/categorize/long_name/index.json\"],\n" +
"      [\"standard_name\", \"http://127.0.0.1:8080/cwexperimental/categorize/standard_name/index.json\"]\n" +
"    ]\n" +
"  }\n" +
"}\n" +
")", 
                "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/standard_name/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">sea_water_temperature<") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/standard_name/index.json");
            Test.ensureTrue(results.indexOf("\"table\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"sea_water_temperature\"") >= 0, "results=" + results);

            results = String2.annotatedString(SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/institution/index.html"));
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">ioos_category<") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "http://127.0.0.1:8080/cwexperimental/categorize/institution/NOAA_CoastWatch_West_Coast_Node/index.html") >= 0, 
                "results=" + results);
            Test.ensureTrue(results.indexOf(
                "http://127.0.0.1:8080/cwexperimental/categorize/institution/NOAA_PMEL/index.html") >= 0, 
                "results=" + results);
            
            results = String2.annotatedString(SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/institution/index.tsv"));
            Test.ensureTrue(results.indexOf("Category[9]URL[10]") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "NOAA_CoastWatch_West_Coast_Node[9]http://127.0.0.1:8080/cwexperimental/categorize/institution/NOAA_CoastWatch_West_Coast_Node/index.tsv[10]") >= 0, 
                "results=" + results);
            Test.ensureTrue(results.indexOf(
                "NOAA_PMEL[9]http://127.0.0.1:8080/cwexperimental/categorize/institution/NOAA_PMEL/index.tsv[10]") >= 0, 
                "results=" + results);
            
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/standard_name/sea_water_temperature/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">erdGlobecBottle<") >= 0,
                "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/categorize/standard_name/sea_water_temperature/index.json");
            expected = 
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"griddap\", \"tabledap\", \"Make A Graph\", \"wms\", \"Accessible\", \"Title\", \"Summary\", \"Info\", \"Background Info\", \"RSS\", \"Email\", \"Institution\", \"Dataset ID\"],\n" +
"    \"columnTypes\": [\"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\", \"String\"],\n" +
"    \"rows\": [\n";
            Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);

            expected =             
"http://127.0.0.1:8080/cwexperimental/tabledap/erdGlobecBottle\", \"http://127.0.0.1:8080/cwexperimental/tabledap/erdGlobecBottle.graph\", \"\", \"public\", \"GLOBEC NEP Rosette Bottle Data (2002)\", \"GLOBEC (GLOBal Ocean ECosystems Dynamics) NEP (Northeast Pacific) \\nRosette Bottle Data from New Horizon Cruise (NH0207: 1-19 August 2002).\\nNotes: \\nPhysical data processed by Jane Fleischbein (OSU). \\nChlorophyll readings done by Leah Feinberg (OSU). \\nNutrient analysis done by Burke Hales (OSU). \\nSal00 - salinity calculated from primary sensors (C0,T0). \\nSal11 - salinity calc. from secondary sensors (C1,T1). \\nsecondary sensor pair was used in final processing of CTD data for \\nmost stations because the primary had more noise and spikes The \\nprimary pair were used for cast# 9,24,48,111 and 150 due to multiple \\nspikes or offsets in the secondary pair. \\nNutrient samples were collected from most bottles; all nutrient data \\ndeveloped from samples frozen during the cruise and analyzed ashore; \\ndata developed by Burke Hales (OSU). \\nOperation Detection Limits for Nutrient Concentrations \\nNutrient  Range         Mean    Variable         Units\\nPO4       0.003-0.004   0.004   Phosphate        micromoles per liter\\nN+N       0.04-0.08     0.06    Nitrate+Nitrite  micromoles per liter\\nSi        0.13-0.24     0.16    Silicate         micromoles per liter\\nNO2       0.003-0.004   0.003   Nitrite          micromoles per liter\\nDates and Times are UTC.\\n\\nFor more information, see http://cis.whoi.edu/science/bcodmo/dataset.cfm?id=10180&flag=view\\n\\nInquiries about how to access this data should be directed to Dr. Hal Batchelder (hbatchelder@coas.oregonstate.edu).\", \"http://127.0.0.1:8080/cwexperimental/info/erdGlobecBottle/index.json\", \"http://oceanwatch.pfeg.noaa.gov/thredds/PaCOOS/GLOBEC/catalog.html?dataset=GLOBEC_Bottle_data\", \"http://127.0.0.1:8080/cwexperimental/rss/erdGlobecBottle.rss\", \"http://127.0.0.1:8080/cwexperimental/subscriptions/add.html?datasetID=erdGlobecBottle&showErrors=false&email=\", \"GLOBEC\", \"erdGlobecBottle\"],";
            po = results.indexOf("http://127.0.0.1:8080/cwexperimental/tabledap/erdGlobecBottle");
            Test.ensureEqual(results.substring(po, po + expected.length()), expected, "results=" + results);


            //griddap
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/index.html");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Datasets Which Can Be Accessed via griddap") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Title</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">RSS</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                ">SST, Blended, Global, EXPERIMENTAL (5 Day Composite)</td>") >= 0,
                "results=" + results);
            Test.ensureTrue(results.indexOf(">erdMHchla8day<") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/index.json");
            Test.ensureTrue(results.indexOf("\"table\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"Title\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"RSS\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "\"SST, Blended, Global, EXPERIMENTAL (5 Day Composite)\"") >= 0,
                "results=" + results);
            Test.ensureTrue(results.indexOf("\"erdMHchla8day\"") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/erdMHchla8day.html");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Data Access Form") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Make A Graph") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("(Centered Time, UTC)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("chlorophyll") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Just generate the URL:") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/griddap/erdMHchla8day.graph");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Make A Graph") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Data Access Form") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("(UTC)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("chlorophyll") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Download the Data or an Image") >= 0, "results=" + results);


            //tabledap
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Datasets Which Can Be Accessed via tabledap") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Title</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">RSS</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">GLOBEC NEP Rosette Bottle Data (2002)</td>") >= 0,
                "results=" + results);            
            Test.ensureTrue(results.indexOf(">erdGlobecBottle<") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/index.json");
            Test.ensureTrue(results.indexOf("\"table\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"Title\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"RSS\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"GLOBEC NEP Rosette Bottle Data (2002)\"") >= 0,
                "results=" + results);            
            Test.ensureTrue(results.indexOf("\"erdGlobecBottle\"") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/erdGlobecBottle.html");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Data Access Form") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Make A Graph") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("(UTC)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("NO3") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Just generate the URL:") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/tabledap/erdGlobecBottle.graph");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Make A Graph") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Data Access Form") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("NO3") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Filled Square") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Download the Data or an Image") >= 0, "results=" + results);


            //wms
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/wms/index.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Datasets Which Can Be Accessed via WMS") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Title</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">RSS</th>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(">Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)</td>") >= 0,
                "results=" + results);            
            Test.ensureTrue(results.indexOf(">erdMHchla8day<") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/wms/index.json");
            Test.ensureTrue(results.indexOf("\"table\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"Title\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"RSS\"") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("\"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\"") >= 0,
                "results=" + results);            
            Test.ensureTrue(results.indexOf("\"erdMHchla8day\"") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/wms/documentation.html");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("display of registered and superimposed map-like views") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Three Ways to Make Maps with WMS") >= 0, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/wms/erdMHchla8day/index.html");            
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)") >= 0,
                "results=" + results);            
            Test.ensureTrue(results.indexOf("Data Access Form") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Make A Graph") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("on-the-fly by ERDDAP's Web Map Server (WMS)") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("altitude") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf("Three Ways to Make Maps with WMS") >= 0, "results=" + results);

//            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
//                "/categorize/standard_name/index.html");
//            Test.ensureTrue(results.indexOf(">sea_water_temperature<") >= 0,
//                "results=" + results);

            //validate the various GetCapabilities documents
/*            String s = "http://www.validome.org/xml/validate/?lang=en" +
                "&url=" + EDStatic.erddapUrl + "/wms/request%3fservice=WMS%26" +
                "request=GetCapabilities%26version=";
            SSR.displayInBrowser(s + "1.1.0");
            SSR.displayInBrowser(s + "1.1.1");
            SSR.displayInBrowser(s + "1.3.0");
*/

            //more information
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/information.html");
            Test.ensureTrue(results.indexOf(
                "ERDDAP a solution to everyone's data distribution / data access problems?") >= 0,
                "results=" + results);

            //subscriptions
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/subscriptions/index.html");
            if (EDStatic.subscriptionSystemActive) {
                Test.ensureTrue(results.indexOf("Add a new subscription") >= 0, "results=" + results);
                Test.ensureTrue(results.indexOf("Validate a subscription") >= 0, "results=" + results);
                Test.ensureTrue(results.indexOf("List your subscriptions") >= 0, "results=" + results);
                Test.ensureTrue(results.indexOf("Remove a subscription") >= 0, "results=" + results);

                results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                    "/subscriptions/add.html");
                Test.ensureTrue(results.indexOf(
                    "To add a subscription, please fill out this form:") >= 0, 
                    "results=" + results);

                results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                    "/subscriptions/validate.html");
                Test.ensureTrue(results.indexOf(
                    "To validate a subscription, please fill out this form:") >= 0, 
                    "results=" + results);

                results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                    "/subscriptions/list.html");
                Test.ensureTrue(results.indexOf(
                    "To request an email with a list of your subscriptions, please fill out this form:") >= 0, 
                    "results=" + results);

                results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                    "/subscriptions/remove.html");
                Test.ensureTrue(results.indexOf(
                    "To remove a subscription, please fill out this form:") >= 0, 
                    "results=" + results);
            }


            //slideSorter
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + 
                "/slidesorter.html");
            Test.ensureTrue(results.indexOf(
                "Your slides will be lost when you close this browser window, unless you:") >= 0, 
                "results=" + results);


            //google Gadgets (always at coastwatch)
            results = SSR.getUrlResponseString(
                "http://coastwatch.pfeg.noaa.gov/erddap/images/gadgets/GoogleGadgets.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "Google Gadgets with Oceanographic Data for Your iGoogle Home Page") >= 0, 
                "results=" + results);
            Test.ensureTrue(results.indexOf(
                "are self-contained chunks of web content") >= 0, 
                "results=" + results);


            //embed a graph  (always at coastwatch)
            results = SSR.getUrlResponseString(
                "http://coastwatch.pfeg.noaa.gov/erddap/images/embed.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "Embed a Graph in a Web Page") >= 0, 
                "results=" + results);

            //Computer Programs            
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/rest.html");
            Test.ensureTrue(results.indexOf("</html>") >= 0, "results=" + results);
            Test.ensureTrue(results.indexOf(
                "ERDDAP as a Web Service") >= 0,
                "results=" + results);

            //list of services
            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.csv");
            expected = 
"Resource, URL\n" +
"info, http://127.0.0.1:8080/cwexperimental/info/index.csv\n" +
"search, http://127.0.0.1:8080/cwexperimental/search/index.csv?searchFor=\n" +
"categorize, http://127.0.0.1:8080/cwexperimental/categorize/index.csv\n" +
"griddap, http://127.0.0.1:8080/cwexperimental/griddap/index.csv\n" +
"tabledap, http://127.0.0.1:8080/cwexperimental/tabledap/index.csv\n" +
"wms, http://127.0.0.1:8080/cwexperimental/wms/index.csv\n";
//subscriptions?
//generateDatasetsXml?
            Test.ensureEqual(results, expected, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.htmlTable");
            expected = 
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \n" +
"  \"http://www.w3.org/TR/html4/loose.dtd\"> \n" +
"<html>\n" +
"<head>\n" +
"  <title>Resources</title>\n" +
"\n" +
"<style type=\"text/CSS\"> <!--\n" +
"  table.erd {border-collapse:collapse; border:1px solid gray; }\n" +
"  table.erd th, table.erd td {padding:2px; border:1px solid gray; }\n" +
"--> </style>\n" +
"</head>\n" +
"<body style=\"color:black; background:white; font-family:Arial,Helvetica,sans-serif; font-size:85%;\">\n" +
"<table class=\"erd\" bgcolor=\"#FFFFCC\" cellspacing=\"0\">\n" +
"<tr>\n" +
"<th>Resource</th>\n" +
"<th>URL</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>info</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/info/index.htmlTable</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>search</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/search/index.htmlTable?searchFor=</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>categorize</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/categorize/index.htmlTable</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>griddap</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/griddap/index.htmlTable</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>tabledap</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/tabledap/index.htmlTable</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>wms</td>\n" +
"<td nowrap>http://127.0.0.1:8080/cwexperimental/wms/index.htmlTable</td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
            Test.ensureEqual(results, expected, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.json");
            expected = 
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"Resource\", \"URL\"],\n" +
"    \"columnTypes\": [\"String\", \"String\"],\n" +
"    \"rows\": [\n" +
"      [\"info\", \"http://127.0.0.1:8080/cwexperimental/info/index.json\"],\n" +
"      [\"search\", \"http://127.0.0.1:8080/cwexperimental/search/index.json?searchFor=\"],\n" +
"      [\"categorize\", \"http://127.0.0.1:8080/cwexperimental/categorize/index.json\"],\n" +
"      [\"griddap\", \"http://127.0.0.1:8080/cwexperimental/griddap/index.json\"],\n" +
"      [\"tabledap\", \"http://127.0.0.1:8080/cwexperimental/tabledap/index.json\"],\n" +
"      [\"wms\", \"http://127.0.0.1:8080/cwexperimental/wms/index.json\"]\n" +
//subscriptions?
//generateDatasetsXml?
"    ]\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "results=" + results);

            results = String2.hexDump(String2.toByteArray(SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.mat")));
            expected = 
"4d 41 54 4c 41 42 20 35   2e 30 20 4d 41 54 2d 66   MATLAB 5.0 MAT-f |\n" +
"69 6c 65 2c 20 43 72 65   61 74 65 64 20 62 79 3a   ile, Created by: |\n" +
"20 67 6f 76 2e 6e 6f 61   61 2e 70 66 65 6c 2e 63    gov.noaa.pfel.c |\n" +
"6f 61 73 74 77 61 74 63   68 2e 4d 61 74 6c 61 62   oastwatch.Matlab |\n" +
//"2c 20 43 72 65 61 74 65   64 20 6f 6e 3a 20 57 65   , Created on: We |\n" +
//"64 20 4d 61 72 20 34 20   31 35 3a 30 34 3a 31 33   d Mar 4 15:04:13 |\n" +
//"20 32 30 30 39 20 20 20   20 20 20 20 20 20 20 20    2009            |\n" +
"20 20 20 20 00 00 00 00   00 00 00 00 01 00 4d 49                 MI |\n" +
"00 00 00 0e 00 00 04 68   00 00 00 06 00 00 00 08          h         |\n" +
"00 00 00 02 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 08                    |\n" +
"72 65 73 70 6f 6e 73 65   00 04 00 05 00 00 00 20   response         |\n" +
"00 00 00 01 00 00 00 40   52 65 73 6f 75 72 63 65          @Resource |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 00 00 00 00 00   55 52 4c 00 00 00 00 00           URL      |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 00 00 00 00 00   00 00 00 0e 00 00 00 a8                    |\n" +
"00 00 00 06 00 00 00 08   00 00 00 04 00 00 00 00                    |\n" +
"00 00 00 05 00 00 00 08   00 00 00 06 00 00 00 0a                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 03 00 00 00 78                  x |\n" +
"00 69 00 73 00 63 00 67   00 74 00 77 00 6e 00 65    i s c g t w n e |\n" +
"00 61 00 72 00 61 00 6d   00 66 00 61 00 74 00 69    a r a m f a t i |\n" +
"00 62 00 73 00 6f 00 72   00 65 00 64 00 6c 00 20    b s o r e d l   |\n" +
"00 20 00 63 00 67 00 64   00 65 00 20 00 20 00 68      c g d e     h |\n" +
"00 6f 00 61 00 64 00 20   00 20 00 20 00 72 00 70    o a d       r p |\n" +
"00 61 00 20 00 20 00 20   00 69 00 20 00 70 00 20    a       i   p   |\n" +
"00 20 00 20 00 7a 00 20   00 20 00 20 00 20 00 20        z           |\n" +
"00 65 00 20 00 20 00 20   00 00 00 0e 00 00 03 30    e             0 |\n" +
"00 00 00 06 00 00 00 08   00 00 00 04 00 00 00 00                    |\n" +
"00 00 00 05 00 00 00 08   00 00 00 06 00 00 00 40                  @ |\n" +
"00 00 00 01 00 00 00 00   00 00 00 03 00 00 03 00                    |\n" +
"00 68 00 68 00 68 00 68   00 68 00 68 00 74 00 74    h h h h h h t t |\n" +
"00 74 00 74 00 74 00 74   00 74 00 74 00 74 00 74    t t t t t t t t |\n" +
"00 74 00 74 00 70 00 70   00 70 00 70 00 70 00 70    t t p p p p p p |\n" +
"00 3a 00 3a 00 3a 00 3a   00 3a 00 3a 00 2f 00 2f    : : : : : : / / |\n" +
"00 2f 00 2f 00 2f 00 2f   00 2f 00 2f 00 2f 00 2f    / / / / / / / / |\n" +
"00 2f 00 2f 00 31 00 31   00 31 00 31 00 31 00 31    / / 1 1 1 1 1 1 |\n" +
"00 32 00 32 00 32 00 32   00 32 00 32 00 37 00 37    2 2 2 2 2 2 7 7 |\n" +
"00 37 00 37 00 37 00 37   00 2e 00 2e 00 2e 00 2e    7 7 7 7 . . . . |\n" +
"00 2e 00 2e 00 30 00 30   00 30 00 30 00 30 00 30    . . 0 0 0 0 0 0 |\n" +
"00 2e 00 2e 00 2e 00 2e   00 2e 00 2e 00 30 00 30    . . . . . . 0 0 |\n" +
"00 30 00 30 00 30 00 30   00 2e 00 2e 00 2e 00 2e    0 0 0 0 . . . . |\n" +
"00 2e 00 2e 00 31 00 31   00 31 00 31 00 31 00 31    . . 1 1 1 1 1 1 |\n" +
"00 3a 00 3a 00 3a 00 3a   00 3a 00 3a 00 38 00 38    : : : : : : 8 8 |\n" +
"00 38 00 38 00 38 00 38   00 30 00 30 00 30 00 30    8 8 8 8 0 0 0 0 |\n" +
"00 30 00 30 00 38 00 38   00 38 00 38 00 38 00 38    0 0 8 8 8 8 8 8 |\n" +
"00 30 00 30 00 30 00 30   00 30 00 30 00 2f 00 2f    0 0 0 0 0 0 / / |\n" +
"00 2f 00 2f 00 2f 00 2f   00 63 00 63 00 63 00 63    / / / / c c c c |\n" +
"00 63 00 63 00 77 00 77   00 77 00 77 00 77 00 77    c c w w w w w w |\n" +
"00 65 00 65 00 65 00 65   00 65 00 65 00 78 00 78    e e e e e e x x |\n" +
"00 78 00 78 00 78 00 78   00 70 00 70 00 70 00 70    x x x x p p p p |\n" +
"00 70 00 70 00 65 00 65   00 65 00 65 00 65 00 65    p p e e e e e e |\n" +
"00 72 00 72 00 72 00 72   00 72 00 72 00 69 00 69    r r r r r r i i |\n" +
"00 69 00 69 00 69 00 69   00 6d 00 6d 00 6d 00 6d    i i i i m m m m |\n" +
"00 6d 00 6d 00 65 00 65   00 65 00 65 00 65 00 65    m m e e e e e e |\n" +
"00 6e 00 6e 00 6e 00 6e   00 6e 00 6e 00 74 00 74    n n n n n n t t |\n" +
"00 74 00 74 00 74 00 74   00 61 00 61 00 61 00 61    t t t t a a a a |\n" +
"00 61 00 61 00 6c 00 6c   00 6c 00 6c 00 6c 00 6c    a a l l l l l l |\n" +
"00 2f 00 2f 00 2f 00 2f   00 2f 00 2f 00 69 00 73    / / / / / / i s |\n" +
"00 63 00 67 00 74 00 77   00 6e 00 65 00 61 00 72    c g t w n e a r |\n" +
"00 61 00 6d 00 66 00 61   00 74 00 69 00 62 00 73    a m f a t i b s |\n" +
"00 6f 00 72 00 65 00 64   00 6c 00 2f 00 2f 00 63    o r e d l / / c |\n" +
"00 67 00 64 00 65 00 69   00 69 00 68 00 6f 00 61    g d e i i h o a |\n" +
"00 64 00 6e 00 6e 00 2f   00 72 00 70 00 61 00 64    d n n / r p a d |\n" +
"00 64 00 69 00 69 00 2f   00 70 00 65 00 65 00 6e    d i i / p e e n |\n" +
"00 7a 00 69 00 2f 00 78   00 78 00 64 00 65 00 6e    z i / x x d e n |\n" +
"00 69 00 2e 00 2e 00 65   00 2f 00 64 00 6e 00 6d    i . . e / d n m |\n" +
"00 6d 00 78 00 69 00 65   00 64 00 61 00 61 00 2e    m x i e d a a . |\n" +
"00 6e 00 78 00 65 00 74   00 74 00 6d 00 64 00 2e    n x e t t m d . |\n" +
"00 78 00 20 00 20 00 61   00 65 00 6d 00 2e 00 20    x     a e m .   |\n" +
"00 20 00 74 00 78 00 61   00 6d 00 20 00 20 00 3f      t x a m     ? |\n" +
"00 2e 00 74 00 61 00 20   00 20 00 73 00 6d 00 20    . t a     s m   |\n" +
"00 74 00 20 00 20 00 65   00 61 00 20 00 20 00 20    t     e a       |\n" +
"00 20 00 61 00 74 00 20   00 20 00 20 00 20 00 72      a t         r |\n" +
"00 20 00 20 00 20 00 20   00 20 00 63 00 20 00 20              c     |\n" +
"00 20 00 20 00 20 00 68   00 20 00 20 00 20 00 20          h         |\n" +
"00 20 00 46 00 20 00 20   00 20 00 20 00 20 00 6f      F           o |\n" +
"00 20 00 20 00 20 00 20   00 20 00 72 00 20 00 20              r     |\n" +
"00 20 00 20 00 20 00 3d   00 20 00 20 00 20 00 20          =         |\n";
            Test.ensureEqual(results.substring(0, 71 * 4) + results.substring(71 * 7), //remove the creation dateTime
                expected, "results=" + results);

            results = String2.annotatedString(SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.tsv"));
            expected = 
"Resource[9]URL[10]\n" +
"info[9]http://127.0.0.1:8080/cwexperimental/info/index.tsv[10]\n" +
"search[9]http://127.0.0.1:8080/cwexperimental/search/index.tsv?searchFor=[10]\n" +
"categorize[9]http://127.0.0.1:8080/cwexperimental/categorize/index.tsv[10]\n" +
"griddap[9]http://127.0.0.1:8080/cwexperimental/griddap/index.tsv[10]\n" +
"tabledap[9]http://127.0.0.1:8080/cwexperimental/tabledap/index.tsv[10]\n" +
"wms[9]http://127.0.0.1:8080/cwexperimental/wms/index.tsv[10]\n" +
"[end]";
            Test.ensureEqual(results, expected, "results=" + results);

            results = SSR.getUrlResponseString(EDStatic.erddapUrl + "/index.xhtml");
            expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" +
"  \"http://www.w3.org/TR/xhtml1/xhtml1-transitional.dtd\">\n" +
"<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" +
"<head>\n" +
"  <meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\" />\n" +
"  <title>Resources</title>\n" +
"\n" +
"<style type=\"text/CSS\"> <!--\n" +
"  table.erd {border-collapse:collapse; border:1px solid gray; }\n" +
"  table.erd th, table.erd td {padding:2px; border:1px solid gray; }\n" +
"--> </style>\n" +
"</head>\n" +
"<body style=\"color:black; background:white; font-family:Arial,Helvetica,sans-serif; font-size:85%;\">\n" +
"<table bgcolor=\"#FFFFCC\" border=\"1\" cellpadding=\"2\" cellspacing=\"0\">\n" +
"<tr>\n" +
"<th>Resource</th>\n" +
"<th>URL</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">info</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/info/index.xhtml</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">search</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/search/index.xhtml?searchFor=</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">categorize</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/categorize/index.xhtml</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">griddap</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/griddap/index.xhtml</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">tabledap</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/tabledap/index.xhtml</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">wms</td>\n" +
"<td nowrap=\"nowrap\">http://127.0.0.1:8080/cwexperimental/wms/index.xhtml</td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
            Test.ensureEqual(results, expected, "results=" + results);


        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing " + EDStatic.erddapUrl +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }


}



