/*
 * GenerateDatasetsXml Copyright 2009, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap;

import com.cohort.array.StringArray;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;

import gov.noaa.pfel.erddap.dataset.*;
import gov.noaa.pfel.erddap.util.EDStatic;

/**
 * This is a command line program to run GenerateDatasetsXml for the 
 * various EDD subclasses.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2009-06-04
 */
public class GenerateDatasetsXml {

    private static void println(String s) {
        String2.log(s);
    }

    /** This gets the i'th value from args, or prompts the user. */
    private static String get(String args[], int i, String prompt, String def) throws Throwable {
        if (args.length > i) {
            println(prompt + "? " + args[i]);
            return args[i];
        }
        String s = String2.getStringFromSystemIn(prompt + " (default=\"" + def + "\")? ");
        if (s == null)  //null if ^C
            return s;
        if (s.equals("\"\"")) 
            s = "";
        else if (s.length() == 0) 
            s = def;
        return s;
    }
    
    /**
     * This is used when called from within a program.
     * If args.length is 0, this loops; otherwise it returns when done.
     *
     * @param args if args has values, they are used to answer the questions.
     */
    public static void doIt(String args[], boolean loop) throws Throwable {
        EDStatic.returnLoggingToSystemOut(); //also, this forces instantiation when run TestAll

        if (args == null) 
            args = new String[0];
        String eddType = "EDDGridFromDap";
        String s1 = "", s2 = "", s3 = "", s4 = "", s5 = "", s6 = "", s7 = "", s8 = "", s9 = "";

        //look for -verbose (and remove it)
        boolean reallyVerbose = false;  
        int vi = String2.indexOf(args, "-verbose");
        if (vi >= 0) {
            String2.log("verbose=true");
            reallyVerbose = true;
            StringArray sa = new StringArray(args);
            sa.remove(vi);
            args = sa.toArray();
        }
        EDD.verbose = true;
        EDD.reallyVerbose = reallyVerbose;
        NcHelper.verbose = reallyVerbose;
        OpendapHelper.verbose = reallyVerbose;
        Table.verbose = reallyVerbose;

        do {
            //get the EDD type
            eddType = get(args, 0, 
                "\n*** GenerateDatasetsXml ***\n" +
                "Press ^C to exit at any time.\n" +
                "Type \"\" to change from a non-\"\" default back to \"\".\n\n" +
                "EDDType Options:\n" +
                "  EDDGridFromDap, EDDGridFromErddap, EDDGridFromNcFiles, \n" +
                "  EDDTableFromDapSequence, EDDTableFromDatabase, EDDTableFromErddap, \n" +
                "  EDDTableFromHyraxFiles, EDDTableFromNcFiles, EDDTableFromSOS\n" +
                "Which EDDType",
                eddType);
            if (eddType == null) return;

            try {
                //EDDGrid
                if (eddType.equals("EDDGridFromDap")) {
                    s1 = get(args, 1, "URL (without trailing .dds or .html)", s1);           if (s1 == null) return;
                    println(EDDGridFromDap.generateDatasetsXml(s1));

                } else if (eddType.equals("EDDGridFromErddap")) {
                    s1 = get(args, 1, "URL of remote ERDDAP (ending in (\"/erddap\")", s1);  if (s1 == null) return;
                    println(EDDGridFromErddap.generateDatasetsXml(s1));

                } else if (eddType.equals("EDDGridFromNcFiles")) {
                    s1 = get(args, 1, "Full file name of one file", s1);                     if (s1 == null) return;
                    println(EDDGridFromNcFiles.generateDatasetsXml(s1));

                //EDDTable
                //currently no EDDTableFromBMDE
                } else if (eddType.equals("EDDTableFromDapSequence")) {
                    s1 = get(args, 1, "URL (without trailing .dds or .html)", s1);           if (s1 == null) return;
                    s2 = get(args, 2, "Outer sequence name (see the .das)", s2);             if (s2 == null) return;
                    s3 = get(args, 3, "Inner Sequence name (use \"\" if none)", s3);         if (s3 == null) return;
                    s4 = get(args, 4, "Sort columns by name (true or false)", s4);           if (s4 == null) return;
                    println(EDDTableFromDapSequence.generateDatasetsXml(s1, s2, s3, String2.parseBoolean(s4)));

                } else if (eddType.equals("EDDTableFromDatabase")) {
                    s1 = get(args, 1, "URL", s1);                                            if (s1 == null) return;
                    s2 = get(args, 2, "Driver name", s2);                                    if (s2 == null) return;
                    s3 = get(args, 3, "Connection properties (format: name1|value1|name2|value2)", s3);  if (s3 == null) return;
                    s4 = get(args, 4, "Catalog name", s4);                                   if (s4 == null) return;
                    s5 = get(args, 5, "Schema name", s5);                                    if (s5 == null) return;
                    s6 = get(args, 6, "Table name", s6);                                     if (s6 == null) return;
                    s7 = get(args, 7, "Sort columns by name (true or false)", s7);           if (s7 == null) return;
                    s8 = get(args, 8, "Just print table info (true or false)", s8);          if (s8 == null) return;
                    String sa3[] = s3.length() == 0? new String[0] : String2.split(s3, '|');
                    println(EDDTableFromDatabase.generateDatasetsXml(s1, s2, sa3, 
                        s4, s5, s6, String2.parseBoolean(s7), String2.parseBoolean(s8)));

                } else if (eddType.equals("EDDTableFromErddap")) {
                    s1 = get(args, 1, "URL of remote ERDDAP (ending in (\"/erddap\")", s1);  if (s1 == null) return;
                    println(EDDTableFromErddap.generateDatasetsXml(s1));

                } else if (eddType.equals("EDDTableFromHyraxFiles")) {
                    s1 = get(args, 1, "Base Directory URL (with slash at end)", s1);         if (s1 == null) return;
                    s2 = get(args, 2, "DAP URL for one file (without trailing .dds or .html)", s2); if (s2 == null) return;
                    s3 = get(args, 3, "Sort columns by name (true or false)", s3);           if (s3 == null) return;
                    println(EDDTableFromHyraxFiles.generateDatasetsXml(s1, s2, String2.parseBoolean(s3)));

                } else if (eddType.equals("EDDTableFromNcFiles")) {
                    s1 = get(args, 1, "A sample full file name", s1);                        if (s1 == null) return;
                    s2 = get(args, 2, "Sort columns by name (true or false)", s2);           if (s2 == null) return;
                    println(EDDTableFromNcFiles.generateDatasetsXml(s1, String2.parseBoolean(s2)));

                } else if (eddType.equals("EDDTableFromSOS")) {
                    s1 = get(args, 1, "URL", s1);                                            if (s1 == null) return;
                    s2 = get(args, 2, "Sort columns by name (true or false)", s2);           if (s2 == null) return;
                    println(EDDTableFromSOS.generateDatasetsXml(s1, String2.parseBoolean(s2)));

                } else {
                    println("\n ERROR: eddType=" + eddType + " is not an option.");
                }
            } catch (Throwable t) {
                println(MustBe.throwableToString(t));
            }
        } while (loop && args.length == 0);
    }

    /**
     * This is used when called from the command line.
     * It explicitly calls System.exit(0) when done.
     *
     * @param args if args has values, they are used to answer the questions.
     */
    public static void main(String args[]) throws Throwable {
        doIt(args, true);
        System.exit(0);
    }

}