/* 
 * EDD Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.FileNameUtility;
import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.SgtGraph;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;
import gov.noaa.pfel.erddap.util.*;
import gov.noaa.pfel.erddap.variable.*;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.NoSuchElementException;
import javax.servlet.http.HttpServletRequest;

/** 
This class represents an ERDDAP Dataset (EDD) -- 
a gridded or tabular dataset 
(usually Longitude, Latitude, Altitude, and Time-referenced)
usable by ERDDAP (ERD's Distributed Access Program).

<p>Currently, ERDDAP serves these datasets via a superset of the DAP protocol
(http://www.opendap.org/), which is the recommended
IOOS DMAC (http://dmac.ocean.us/index.jsp) data transport mechanism. 
(DAP is great!)
Other ways of serving the data (e.g., WCS, WFS, and SOS) may be added in the future.
ERDDAP is structured for this and there don't seem to be any impediments.

<p>Main goal: make it easier for scientists (especially those in the IOOS realm)
to access geo- and time-referenced data from diverse remote datasources via common protocols
(e.g., DAP) and get the results in common file formats.
To achieve that, ERDDAP tries to reduce data to a few common
data structures (grids and tables) so that the data can be dealt with in a simple way.
Although DAP is great, it is a low level protocol. 
ERDDAP works as a DAP server, but it is also a higher level web service built
upon and compatible with DAP.
Some features are:
<ul>
<li> A few, simple data structures:
  Since it can be difficult for clients to deal with the infinite number of 
  dataset structures offered by DAP, 
  ERDDAP currently deals with two dataset structures: 
  gridded data (e.g., satellite data and model data) and 
  tabular data (e.g., in-situ station and trajectory data).
  Certainly, not all data can be expressed in these structures, but much of it can. 
  Tables, in particular, are very flexible data structures
  (look at the success of relational database programs).
  This makes it easier to serve the data in standard file types
  (which often just support simple data structures).   
  And this makes it easier to compare data from different sources.   
  Other data structures (e.g., projected grids) could be
  supported in the future if called for.
<li> Requests can be made in user units: 
  Although requests in ERDDAP can be made with array indices (as with DAP),
  requests can also be in user units (e.g., degrees east).
<li> Results are formatted to suit the user:
  The results can be returned in any of several common data file formats: 
  (e.g., ESRI .asc, Google Earth's .kml, .nc, .mat, comma-separated ASCII,
  and tab-separated ASCII) instead of just the original format
  or just the DAP transfer format (which has no standard file manifestation).
  These files are created on-the-fly. Since there are few internal 
  data structures, it is easy to add additional file-type drivers. 
<li> Local or remote data:
  Datasets in ERDDAP can be local (on the same computer) or remote
  (accessible via the web).
<li> Additional metadata:
  Many data sources have little or no metadata describing the data.
  ERDDAP lets (and encourages) the administrator to describe metadata which 
  will be added to datasets and their variables on-the-fly.
<li> Standardized variable names and units for longitude, latitude, altitude, and time:
  To facilitate comparisons of data from different datasets,
  the requests and results in ERDDAP use standardized space/time axis units:
  longitude is always in degrees_east; latitude is always in degrees_north;
  altitude is always in meters with positive=up; 
  time is always in seconds since 1970-01-01T00:00:00Z
  and, when formatted as a string, is formatted according to the ISO 8601 standard.
  This makes it easy to specify constraints in requests
  without having to worry about the altitude data format 
  (are positive values up or down? in meters or fathoms?)
  or time data format (a nightmarish realm of possible formats and time zones).
  This makes the results from different data sources easy to compare.
<li>Modular structure:
  ERDDAP is structured so that it is easy to add different components
  (e.g., a class to request data from a DAP 2-level sequence dataset and store
  it as a table).  The new component then gains all the features 
  and capabilities of the parent (e.g., support for DAP requests 
  and the ability to save the data in several common file formats).
<li>Data Flow:
  To save memory (it is a big issue) and make responses start sooner,
  ERDDAP processes data requests in chunks -- repeatedly getting a chunk of data
  and sending that to the client.
  For many datasources (e.g., SOS sources), this means that the first chunk of 
  data (e.g., from the first sensor) gets to the client in seconds
  instead of minutes (e.g., after data from the last sensor has been retrieved).
  From a memory standpoint, this allows numerous large requests 
  (each larger than available memory) to be handled simultaneously
  and efficiently.
<li>Is ERDDAP a solution to all our data distribution problems? Not even close.
  But hopefully ERDDAP fills some common needs that
  aren't being filled by other data servers.
</ul>  

 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-04
 */
public abstract class EDD { 


    /** "ERROR" is defined here (from String2.ERROR) so that it is consistent in log files. */
    public final static String ERROR = String2.ERROR; 

    /**
     * Set this to true (by calling verbose=true in your program, 
     * not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean verbose = false; 

    /**
     * Set this to true (by calling reallyVerbose=true in your program, 
     * not but changing the code here)
     * if you want lots of diagnostic messages sent to String2.log.
     */
    public static boolean reallyVerbose = false; 

    /**
     * Set this to true (by calling debugMode=true in your program, 
     * not but changing the code here)
     * if you want every possible diagnostic message sent to String2.log.
     */
    public static boolean debugMode = false; 


    /** The allowed cdm_data_type. */
    public final static String CDM_GRID = "Grid", 
        CDM_OTHER = "Other", //Bob added 
        CDM_POINT = "Point", //Bob added to standard   (isn't it in the standard?)
        CDM_STATION = "Station", 
        CDM_TRAJECTORY = "Trajectory";

    /** 
     * CDM_TYPES is an array in alphabetic order. 
     * Don't rely on the positions, since new types will be added in 
     * alphabetic order.) */
    public final static String[] CDM_TYPES = {
        CDM_GRID, CDM_OTHER, CDM_POINT, CDM_STATION, CDM_TRAJECTORY};


    public static int DEFAULT_RELOAD_EVERY_N_MINUTES = 10080; //1 week  //The value is mentioned in datasets.xml.


    //*********** END OF STATIC DECLARATIONS ***************************

    protected long creationTimeMillis = System.currentTimeMillis();

    /** The constructor must set all of these protected variables 
     * (see definitions below in their accessor methods). 
     */    
    protected String datasetID, className;

    /** 0 or more actions (starting with "http://" or "mailto:")
     * to be done whenever the dataset changes significantly (or null)
     */
    protected StringArray onChange;

    /** 
      sourceAtt are straight from the source.
      addAtt are specified by the admin and supercede sourceAttributes. 
      combinedAtt are made from sourceAtt and addAtt, then revised (e.g., remove "null" values) 
    */
    protected Attributes sourceGlobalAttributes, addGlobalAttributes, combinedGlobalAttributes;
    //dataVariables isn't a hashMap because it is nice to allow a specified order for the variables
    protected EDV[] dataVariables;
    private int reloadEveryNMinutes = DEFAULT_RELOAD_EVERY_N_MINUTES;
    /** accessibleTo is stored in sorted order.  null means accessible to anyone, even if not logged in. 
     *   length=0 means accessible to no one.
     */
    protected String[] accessibleTo = null; 

    /** These are created as needed from combinedGlobalAttributes. */
    protected String id, title, summary, institution, infoUrl, cdmDataType, sourceUrl; 
    protected byte[] searchString;
    /** These are created as needed from dataVariables[]. */
    protected String[] dataVariableSourceNames, dataVariableDestinationNames;
    



    /**
     * This constructs an EDDXxx based on the information in an .xml file.
     * This ignores the &lt;dataset active=.... &gt; setting.
     * All of the subclasses fromXml() methods ignore the &lt;dataset active=.... &gt; setting.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDXxx&gt; 
     *    having just been read.  
     * @return a 'type' subclass of EDD.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDD fromXml(String type, SimpleXMLReader xmlReader) throws Throwable {
        String startError = "datasets.xml error on or before line #";
        if (type == null) 
            throw new SimpleException(startError + xmlReader.lineNumber() + 
                ": Unexpected <dataset> type=" + type + ".");
        try {
            //future: classes could be added at runtime if I used reflection
            if (type.equals("EDDGridAggregateExistingDimension")) 
                return EDDGridAggregateExistingDimension.fromXml(xmlReader);
            if (type.equals("EDDGridCopy"))             return EDDGridCopy.fromXml(xmlReader);
            if (type.equals("EDDGridFromDap"))          return EDDGridFromDap.fromXml(xmlReader);
            if (type.equals("EDDGridFromErddap"))       return EDDGridFromErddap.fromXml(xmlReader);
            if (type.equals("EDDGridFromEtopo"))        return EDDGridFromEtopo.fromXml(xmlReader);
            if (type.equals("EDDGridFromNcFiles"))      return EDDGridFromNcFiles.fromXml(xmlReader);
            if (type.equals("EDDGridSideBySide"))       return EDDGridSideBySide.fromXml(xmlReader);

            if (type.equals("EDDTableCopy"))            return EDDTableCopy.fromXml(xmlReader);
            if (type.equals("EDDTableFromBMDE"))        return EDDTableFromBMDE.fromXml(xmlReader);
            if (type.equals("EDDTableFromDapSequence")) return EDDTableFromDapSequence.fromXml(xmlReader);
            if (type.equals("EDDTableFromDatabase"))    return EDDTableFromDatabase.fromXml(xmlReader);
            if (type.equals("EDDTableFromErddap"))      return EDDTableFromErddap.fromXml(xmlReader);
            //if (type.equals("EDDTableFromMWFS"))        return EDDTableFromMWFS.fromXml(xmlReader); //inactive as of 2009-01-14
            if (type.equals("EDDTableFromHyraxFiles"))  return EDDTableFromHyraxFiles.fromXml(xmlReader);
            if (type.equals("EDDTableFromNcFiles"))     return EDDTableFromNcFiles.fromXml(xmlReader);
            if (type.equals("EDDTableFromNOS"))         return EDDTableFromNOS.fromXml(xmlReader);
            if (type.equals("EDDTableFromOBIS"))        return EDDTableFromOBIS.fromXml(xmlReader);
            if (type.equals("EDDTableFromSOS"))         return EDDTableFromSOS.fromXml(xmlReader);
        } catch (Throwable t) {
            throw new RuntimeException(startError + xmlReader.lineNumber() + 
                ": " + MustBe.getShortErrorMessage(t), t);
        }
        throw new Exception(startError + xmlReader.lineNumber() + 
            ": Unexpected <dataset> type=" + type + ".");
    }

    /**
     * This is used to test the xmlReader constructor in each subclass.
     * Because this uses a simple method to convert the String to bytes,
     * the xml's encoding must be ISO-8859-1 (or just use low ASCII chars).
     * This ignores the &lt;dataset active=.... &gt; setting.
     *
     * @param type the name of the subclass, e.g., EDDGridFromDap
     * @param xml a complete xml file with the information for one dataset.
     * @return a 'type' subclass of EDD
     * @throws Throwable if trouble
     */
    public static EDD testXmlReader(String type, String xml) throws Throwable {
        String2.log("\nEDD.testXmlConstructor...");
        SimpleXMLReader xmlReader = new SimpleXMLReader(
            new ByteArrayInputStream(String2.toByteArray(xml)), "erddapDatasets");
        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            if      (tags.equals("</erddapDatasets>")) {
                xmlReader.close();
                throw new IllegalArgumentException("No <dataset> tag in xml.");
            } else if (tags.equals("<erddapDatasets><dataset>")) {
                EDD edd = fromXml(xmlReader.attributeValue("type"), xmlReader);
                xmlReader.close();
                return edd;
            } else {
                xmlReader.unexpectedTagException();
            }
        }
    }

    /**
     * This is used by various test procedures to get one of the datasets
     * specified in <tomcat>/content/erddap/datasets.xml.
     * This ignores the &lt;dataset active=.... &gt; setting.
     *
     * @param tDatasetID
     * @return an instance of a subclass of EDD
     * @throws Throwable if trouble
     */
    public static EDD oneFromDatasetXml(String tDatasetID) throws Throwable {
        String2.log("\nEDD.oneFromDatasetXml(" + tDatasetID + ")...");

        SimpleXMLReader xmlReader = new SimpleXMLReader(
            new FileInputStream(EDStatic.contentDirectory + 
                "datasets" + (EDStatic.developmentMode? "2" : "") + ".xml"), 
            "erddapDatasets");
        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            if      (tags.equals("</erddapDatasets>")) {
                xmlReader.close();
                throw new IllegalArgumentException(tDatasetID + " not found in datasets.xml.");
            } else if (tags.equals("<erddapDatasets><dataset>")) {
                if (xmlReader.attributeValue("datasetID").equals(tDatasetID)) {
                    EDD edd = EDD.fromXml(xmlReader.attributeValue("type"), xmlReader);
                    xmlReader.close();
                    return edd;
                } else {
                    //skip to </dataset> tag
                    while (!tags.equals("<erddapDatasets></dataset>")) {
                        xmlReader.nextTag();
                        tags = xmlReader.allTags();
                    }
                }
            } else if (tags.equals("<erddapDatasets><requestBlacklist>")) {
            } else if (tags.equals("<erddapDatasets></requestBlacklist>")) {
            } else if (tags.equals("<erddapDatasets><subscriptionEmailBlacklist>")) {
            } else if (tags.equals("<erddapDatasets></subscriptionEmailBlacklist>")) {
            } else if (tags.equals("<erddapDatasets><user>")) {
            } else if (tags.equals("<erddapDatasets></user>")) {
            } else {
                xmlReader.unexpectedTagException();
            }
        }
    }

    /**
     * This is commonly used by subclass constructors to set all the items
     * common to all EDDs.
     * Or, subclasses can just set these things directly.
     *
     * <p>sourceGlobalAttributes and/or addGlobalAttributes must include:
     *   <ul>
     *   <li> "title" - the short (&lt; 80 characters) description of the dataset 
     *   <li> "summary" - the longer description of the dataset 
     *   <li> "institution" - the source of the data 
     *      (best if &lt; 50 characters so it fits in a graph's legend).
     *   <li> "infoUrl" - the url with information about this data set 
     *   <li> "sourceUrl" - the url (for descriptive purposes only) of the source of the data,
     *      e.g., the basic opendap url.
     *   <li> "cdm_data_type" - one of the EDD.CDM_xxx options
     *   </ul>
     * Special case: value="null" causes that item to be removed from combinedGlobalAttributes.
     * Special case: if addGlobalAttributes name="license" value="[standard]",
     *    the EDStatic.standardLicense will be used.
     *
     */
    public void setup(String tDatasetID, 
        Attributes tSourceGlobalAttributes, Attributes tAddGlobalAttributes, 
        EDV[] tDataVariables,
        int tReloadEveryNMinutes) {

        //save the parameters
        datasetID = tDatasetID;
        sourceGlobalAttributes = tSourceGlobalAttributes;
        addGlobalAttributes = tAddGlobalAttributes;
        String tLicense = addGlobalAttributes.getString("license");
        if (tLicense != null && tLicense.equals("[standard]"))
            addGlobalAttributes.set("license", EDStatic.standardLicense);
        combinedGlobalAttributes = new Attributes(addGlobalAttributes, sourceGlobalAttributes); //order is important
        combinedGlobalAttributes.removeValue("null");

        dataVariables = tDataVariables;
        reloadEveryNMinutes = tReloadEveryNMinutes <= 0 || tReloadEveryNMinutes == Integer.MAX_VALUE?
            DEFAULT_RELOAD_EVERY_N_MINUTES : tReloadEveryNMinutes;

    }

    /**
     * This should be used by all subclass constructors to ensure that 
     * all of the items common to all EDDs are properly set.
     * This also does a few standard things.
     *
     * @throws Throwable if any required item isn't properly set
     */
    public void ensureValid() throws Throwable {
        //ensure valid
        String errorInMethod = "datasets.xml/EDD.ensureValid error for " + datasetID() + ":\n ";

        //test that required things are set
        Test.ensureFileNameSafe(datasetID, errorInMethod + "datasetID");
        File2.makeDirectory(EDStatic.fullCacheDirectory + datasetID);
        Test.ensureSomethingUtf8(sourceGlobalAttributes,  errorInMethod + "sourceGlobalAttributes");
        Test.ensureSomethingUtf8(addGlobalAttributes,     errorInMethod + "addGlobalAttributes");
        Test.ensureSomethingUtf8(combinedGlobalAttributes,errorInMethod + "combinedGlobalAttributes");
        Test.ensureSomethingUtf8(title(),                 errorInMethod + "title");
        Test.ensureSomethingUtf8(summary(),               errorInMethod + "summary");
        Test.ensureSomethingUtf8(institution(),           errorInMethod + "institution");
        Test.ensureSomethingUtf8(infoUrl(),               errorInMethod + "infoUrl");
        Test.ensureSomethingUtf8(sourceUrl(),             errorInMethod + "sourceUrl");
        Test.ensureSomethingUtf8(cdmDataType(),           errorInMethod + "cdm_data_type");
        Test.ensureSomethingUtf8(className(),             errorInMethod + "className");
        Test.ensureTrue(String2.indexOf(CDM_TYPES, cdmDataType()) >= 0,      
            errorInMethod + "cdm_data_type=" + cdmDataType + 
            " isn't one of the standard CDM types (" + String2.toCSVString(CDM_TYPES) + ").");
        Test.ensureTrue(dataVariables != null && dataVariables.length > 0, errorInMethod + "'dataVariables' wasn't set.");
        HashSet destNames = new HashSet();
        for (int i = 0; i < dataVariables.length; i++) {
            Test.ensureNotNull(dataVariables[i], errorInMethod + "'dataVariables[" + i + "]' wasn't set.");
            String tErrorInMethod = errorInMethod + 
                "for dataVariable #" + i + "=" + dataVariables[i].destinationName() + ":\n";
            dataVariables[i].ensureValid(tErrorInMethod);
            if (!destNames.add(dataVariables[i].destinationName()))
                throw new IllegalArgumentException(tErrorInMethod + "Two variables have destinationName=" + 
                    dataVariables[i].destinationName() + ".");
        }
        //String2.log("\n***** beginSearchString\n" + String2.utf8ToString(searchString()) + 
        //            "\n***** endSearchString\n");
        //reloadEveryNMinutes
    }


    /**
     * The string representation of this gridDataSet (for diagnostic purposes).
     *
     * @return the string representation of this EDD.
     */
    public String toString() {  
        //make this JSON format?
        StringBuffer sb = new StringBuffer();
        sb.append(datasetID() + ": " + 
            "\ntitle=" + title() +
            "\nsummary=" + summary() +
            "\ninstitution=" + institution() +
            "\ninfoUrl=" + infoUrl() +
            "\nsourceUrl=" + sourceUrl() +
            "\ncdm_data_type=" + cdmDataType() +
            "\nreloadEveryNMinutes=" + reloadEveryNMinutes +
            "\nonChange=" + onChange +
            "\nsourceGlobalAttributes=\n" + sourceGlobalAttributes + 
              "addGlobalAttributes=\n" + addGlobalAttributes);
        for (int i = 0; i < dataVariables.length; i++)
            sb.append("dataVariables[" + i + "]=" + dataVariables[i]);
        return sb.toString();
    }

    /**
     * This tests if the dataVariables of the other dataset are similar 
     *     (same destination data var names, same sourceDataType, same units, 
     *     same missing values).
     *
     * @param other
     * @return "" if similar (same axis and data var names,
     *    same units, same sourceDataType, same missing values) or a message if not.
     */
    public String similar(EDD other) {
        String msg = "EDD.similar: The other dataset has a different ";

        try {
            if (other == null) 
                return "EDD.similar: Previously, this dataset was (temporarily?) not available.  " +
                    "Perhaps ERDDAP was just restarted.";

            int nDv = dataVariables.length;
            if (nDv != other.dataVariables.length)
                return msg + "number of dataVariables (" + 
                    nDv + " != " + other.dataVariables.length + ")";

            for (int dv = 0; dv < nDv; dv++) {
                EDV dv1 = dataVariables[dv];
                EDV dv2 = other.dataVariables[dv];

                //destinationName
                String s1 = dv1.destinationName();
                String s2 = dv2.destinationName();
                String msg2 = " for dataVariable #" + dv + "=" + s1 + " (";
                if (!s1.equals(s2))
                    return msg + "destinationName" + msg2 +  s1 + " != " + s2 + ")";

                //sourceDataType
                s1 = dv1.sourceDataType();
                s2 = dv2.sourceDataType();
                if (!s1.equals(s2))
                    return msg + "sourceDataType" + msg2 +  s1 + " != " + s2 + ")";

                //destinationDataType
                s1 = dv1.destinationDataType();
                s2 = dv2.destinationDataType();
                if (!s1.equals(s2))
                    return msg + "destinationDataType" + msg2 +  s1 + " != " + s2 + ")";

                //units
                s1 = dv1.units();
                s2 = dv2.units();
                if (!s1.equals(s2))
                    return msg + "units" + msg2 +  s1 + " != " + s2 + ")";

                //sourceMissingValue
                double d1 = dv1.sourceMissingValue();
                double d2 = dv2.sourceMissingValue();
                if (!Test.equal(d1, d2)) //says NaN==NaN is true
                    return msg + "sourceMissingValue" + msg2 +  d1 + " != " + d2 + ")";

                //sourceFillValue
                d1 = dv1.sourceFillValue();
                d2 = dv2.sourceFillValue();
                if (!Test.equal(d1, d2)) //says NaN==NaN is true
                    return msg + "sourceFillValue" + msg2 +  d1 + " != " + d2 + ")";
            }

            //they are similar
            return "";
        } catch (Throwable t) {
            return t.toString();
        }
    }

    protected static String test1Changed(String msg, String diff) {
        return diff.length() == 0? "" : msg + "\n" + diff + "\n";
    }

    protected static String test2Changed(String msg, String oldS, String newS) {
        if (oldS.equals(newS))
            return "";
        return msg + 
            "\n  old=" + oldS + ",\n" +
              "  new=" + newS + ".\n";
    }

    /**
     * This tests if 'old' is different from this in any way.
     * <br>This test is from the view of a subscriber who wants to know
     *    when a dataset has changed in any way.
     * <br>So some things like onChange and reloadEveryNMinutes are not checked.
     * <br>This only lists the first change found.
     *
     * <p>EDDGrid overwrites this to also check the axis variables.
     *
     * @param old
     * @return "" if same or message if not.
     */
    public String changed(EDD old) {
        //future: perhaps it would be nice if EDDTable changed showed new data.
        //  so it would appear in email subscription and rss.
        //  but for many datasets (e.g., ndbc met) there are huge number of buoys. so not practical.
        if (old == null)
            return "Previously, this dataset was (temporarily?) not available.  Perhaps ERDDAP was just restarted.\n";

        //check most important things first
        int nDv = dataVariables.length;
        StringBuffer diff = new StringBuffer();
        diff.append(test2Changed("The number of dataVariables changed:",
            "" + old.dataVariables.length, 
            "" + nDv));
        if (diff.length() > 0) 
            return diff.toString(); //because tests below assume nDv are same

        for (int dv = 0; dv < nDv; dv++) { 
            EDV oldDV = old.dataVariables()[dv];
            EDV newDV =     dataVariables()[dv];             
            String newName = newDV.destinationName();

            diff.append(test2Changed(
                "The destinationName for dataVariable #" + dv + " changed:",
                oldDV.destinationName(), 
                newName));

            diff.append(test2Changed(
                "The destinationDataType for dataVariable #" + dv + "=" + newName + " changed:",
                oldDV.destinationDataType(), 
                newDV.destinationDataType()));

            diff.append(test1Changed(
                "A combinedAttribute for dataVariable #" + dv + "=" + newName + " changed:",
                String2.differentLine(
                    oldDV.combinedAttributes().toString(), 
                    newDV.combinedAttributes().toString())));
        }

        //check least important things last
        diff.append(test1Changed("A combinedGlobalAttribute changed:",
            String2.differentLine(
                old.combinedGlobalAttributes().toString(), 
                    combinedGlobalAttributes().toString())));

        return diff.toString();    
    }

    /** 
     * This returns the link tag for an HTML head section which advertises 
     * the RSS feed for this dataset.
     */
    public String rssHeadLink() {
        return 
            "<link rel=\"alternate\" type=\"application/rss+xml\" \n" +
            "  href=\"" + EDStatic.erddapUrl + //RSS always uses non-https link
                "/rss/" + datasetID() + ".rss\" \n" +
            "  title=\"ERDDAP: " + title() + "\">\n";
    }

    /** 
     * This returns the a/href tag which advertises the RSS feed for this dataset.
     */
    public String rssHref() {
        return 
            "<a type=\"application/rss+xml\" " +
            "  href=\"" + EDStatic.erddapUrl + //RSS always uses a non-https link
            "/rss/" + datasetID()+ ".rss\" \n" +
            "  title=\"\"><img alt=\"RSS\" align=\"top\" border=\"0\"\n" +
            "    title=\"" + EDStatic.subscriptionRSS + "\" \n" +
            "    src=\"" + EDStatic.imageDirUrl + "rss.gif\" ></a>"; //no img end tag
    }

    /** 
     * This returns the a/href tag which advertises the email subscription url for this dataset
     * (or "" if !EDStatic.subscriptionSystemActive).
     * This uses baseHttpsUrl if accessibleTo is defined; else baseUrl.
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     */
    public String emailHref(String tErddapUrl) {
        if (EDStatic.subscriptionSystemActive) 
            return 
            "<a href=\"" + tErddapUrl + "/" + Subscriptions.ADD_HTML + 
                "?datasetID=" + datasetID()+ "&amp;showErrors=false&amp;email=\" \n" +
            "  title=\"\"><img alt=\"Subscribe\" border=\"0\" align=\"top\" \n" +
            "    title=\"" + XML.encodeAsXML(EDStatic.subscriptionEmail) + "\" \n" +
            "    src=\"" + EDStatic.imageDirUrl + "envelope.gif\" ></a>";
        return "&nbsp;";
    }

    /**
     * This is used by EDDXxx.fromXml to get Attributes from the e.g., datasets.xml file.
     * 
     * @param xmlReader with the (e.g.,) ...<globalAttributes> having just been read.    
     *    The subsequent tags must all be &lt;att name=\"attName\" type=\"someType\"&gt;someValue&lt;/att&gt; .
     *    <br>someType can be:
     *    <br>for single values: boolean, unsignedShort, short, int, long, float, double, or string,
     *       (these are standard XML atomic data types),
     *    <br>or for space-separated values: 
     *       booleanList, unsignedShortList, shortList, intList, longList, floatList, doubleList, or stringList,
     *       (these could be defined in an XML schema as xml Lists: space separated lists of atomic types).
     *    <br>Or, type=\"someType\" can be omitted (interpreted as 'string').
     *    <br>If type='stringList', individual values with interior whitespace or commas
     *    must be completely enclosed in double quotes with interior double
     *    quotes converted to 2 double quotes. For String values without interior whitespace or commas,
     *    you don't have to double quote the whole value. 
     *    This doesn't match the xml definition of list as applied to strings,
     *    but a stringList type could be defined as string data which has special
     *    meaning to this application.
     * @return the Attributes based on the information in an .xml file.
     *    And xmlReader will have just read (e.g.,) ...</globalAttributes>
     * @throws Throwable if trouble
     */
    public static Attributes getAttributesFromXml(SimpleXMLReader xmlReader) throws Throwable {

        //process the tags
        if (reallyVerbose) String2.log("    getAttributesFromXml...");
        Attributes tAttributes = new Attributes();
        int startOfTagsN = xmlReader.stackSize();
        String tName = null, tType =  null;
        while (true) {
            xmlReader.nextTag();
            String topTag = xmlReader.topTag();
            if (xmlReader.stackSize() == startOfTagsN) {
                if (reallyVerbose) String2.log("      leaving getAttributesFromXml");
                return tAttributes; //the </attributes> tag
            }
            if (xmlReader.stackSize() > startOfTagsN + 1) 
                xmlReader.unexpectedTagException();

            if (topTag.equals("att")) {
                tName = xmlReader.attributeValue("name");
                tType = xmlReader.attributeValue("type");

            } else if (topTag.equals("/att")) {
                String content = xmlReader.content();
                if (tName == null)
                    throw new IllegalArgumentException("datasets.xml error on line #" + xmlReader.lineNumber() +
                        ": An <att> tag doesn't have a \"name\" attribute (content=" + content + ").");
                //if (reallyVerbose) 
                //    String2.log("      tags=" + xmlReader.allTags() + 
                //        " name=" + tName + " type=" + tType + " content=" + content);
                if (tType == null) 
                    tType = "string";
                PrimitiveArray pa;
                if (content.length() == 0) {
                    //content="" interpreted as want to delete that attribute
                    pa = new StringArray();  //always make it a StringArray (to hold "null")
                    pa.addString("null"); 
                } else if (tType.equals("string") || tType.equals("String")) { //spec requests "string"; support "String" to be nice?
                    //for "string", don't want to split at commas
                    pa = new StringArray(); 
                    pa.addString(content);
                } else {
                    //for all other types, csv designation is irrelevant
                    if (tType.endsWith("List")) 
                        tType = tType.substring(0, tType.length() - 4);
                    if (tType.equals("unsignedShort")) //the xml name
                        tType = "char"; //the PrimitiveArray name
                    else if (tType.equals("string")) //the xml name
                        tType = "String"; //the PrimitiveArray name
                    pa = PrimitiveArray.ssvFactory(PrimitiveArray.elementStringToType(tType), 
                        content); 
                }
                //if (tName.equals("_FillValue")) 
                //    String2.log("!!!!EDD attribute name=\"" + tName + "\" content=" + content + 
                //    "\n  type=" + pa.getElementTypeString() + " pa=" + pa.toString());
                tAttributes.add(tName, pa);
                //String2.log("????EDD _FillValue=" + tAttributes.get("_FillValue"));

            } else {
                xmlReader.unexpectedTagException();
            }
        }
    }

    /**
     * This is used by EDDXxx.fromXml to get the sourceName,destinationName,attributes information for an 
     * axisVariable or dataVariable from the e.g., datasets.xml file.
     * Unofficial: this is becoming the standard for &lt;axisVariable&gt;.
     * 
     * @param xmlReader with the ...&lt;axisVariable&gt; or ...&lt;dataVariable&gt; having just been read.    
     *    The allowed subtags are sourceName, destinationName, and addAttributes.
     * @return Object[4] [0]=sourceName, [1]=destinationName, [2]=addAttributes, 
     *    [3]=values PrimitiveArray.
     *    This doesn't check the validity of the objects. The objects may be null. 
     *    The xmlReader will have just read ...&lt;/axisVariable&gt; or ...&lt;/dataVariable&gt;.
     * @throws Throwable if trouble
     */
    public static Object[] getSDAVVariableFromXml(SimpleXMLReader xmlReader) throws Throwable {

        //process the tags
        if (reallyVerbose) String2.log("  getSDAVVariableFromXml...");
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();
        String tSourceName = null, tDestinationName = null;
        Attributes tAttributes = null;
        PrimitiveArray tValuesPA = null;
        while (true) {
            xmlReader.nextTag();
            String topTag = xmlReader.topTag();
            String content = xmlReader.content();
            //if (reallyVerbose) String2.log("    topTag=" + topTag + " content=" + content);
            if (xmlReader.stackSize() == startOfTagsN) { //the /variable tag
                if (reallyVerbose) String2.log("    leaving getSDAVVariableFromXml" +
                    " sourceName=" + tSourceName + " destName=" + tDestinationName);
                return new Object[]{tSourceName, tDestinationName, tAttributes, tValuesPA};
            }
            if (xmlReader.stackSize() > startOfTagsN + 1) 
                xmlReader.unexpectedTagException();

            if      (topTag.equals( "sourceName")) {}
            else if (topTag.equals("/sourceName")) tSourceName = content;
            else if (topTag.equals( "destinationName")) {}
            else if (topTag.equals("/destinationName")) tDestinationName = content;
            else if (topTag.equals( "addAttributes"))
                tAttributes = getAttributesFromXml(xmlReader);
            else if (topTag.equals( "values")) {
                //always make a PA 
                String type = xmlReader.attributeValue("type");
                if (type == null) 
                    type = "";
                if (type.endsWith("List"))
                    type = type.substring(0, type.length() - 4);
                Class elementType = PrimitiveArray.elementStringToType(type); //throws Throwable if trouble
                double start      = String2.parseDouble(xmlReader.attributeValue("start"));
                double increment  = String2.parseDouble(xmlReader.attributeValue("increment"));
                int n             = String2.parseInt(xmlReader.attributeValue("n"));
                if (!Double.isNaN(start) && 
                    increment > 0 && //this could change to !NaN and !0
                    n > 0 && n < Integer.MAX_VALUE) {
                    //make PA with 1+ evenly spaced values
                    tValuesPA = PrimitiveArray.factory(elementType, n, false);
                    for (int i = 0; i < n; i++) 
                        tValuesPA.addDouble(start + i * increment);
                } else {
                    //make PA with correct type, but size=0
                    tValuesPA = PrimitiveArray.factory(elementType, 0, "");
                }
            } else if (topTag.equals("/values")) {
                if (tValuesPA.size() == 0) {
                    //make a new PA from content values 
                    tValuesPA = PrimitiveArray.csvFactory(tValuesPA.getElementType(), content);         
                }
                if (reallyVerbose) String2.log("values for sourceName=" + tSourceName + "=" + tValuesPA.toString());

            } else xmlReader.unexpectedTagException();

        }
    }

    /**
     * This is used by EDDXxx.fromXml to get the 
     * sourceName,destinationName,attributes,dataType information for an 
     * axisVariable or dataVariable from the e.g., datasets.xml file.
     * Unofficial: this is becoming the standard for &lt;dataVariable&gt;.
     * 
     * @param xmlReader with the ...&lt;axisVariable&gt; or ...&lt;dataVariable&gt; having just been read.    
     *    The allowed subtags are sourceName, destinationName, addAttributes, and dataType. 
     * @return Object[4] 0=sourceName, 1=destinationName, 2=addAttributes, 3=dataType.
     *    This doesn't check the validity of the objects. The objects may be null. 
     *    The xmlReader will have just read ...&lt;/axisVariable&gt; or ...&lt;/dataVariable&gt;
     * @throws Throwable if trouble
     */
    public static Object[] getSDADVariableFromXml(SimpleXMLReader xmlReader) throws Throwable {

        //process the tags
        if (reallyVerbose) String2.log("  getSDADVVariableFromXml...");
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();
        String tSourceName = null, tDestinationName = null, tDataType = null;
        Attributes tAttributes = null;
        while (true) {
            xmlReader.nextTag();
            String topTag = xmlReader.topTag();
            String content = xmlReader.content();
            //if (reallyVerbose) String2.log("    topTag=" + topTag + " content=" + content);
            if (xmlReader.stackSize() == startOfTagsN) { //the /variable tag
                if (reallyVerbose) String2.log("    leaving getSDADVVariableFromXml" +
                    " sourceName=" + tSourceName + " destName=" + tDestinationName + " dataType=" + tDataType);
                return new Object[]{tSourceName, tDestinationName, tAttributes, tDataType};
            }
            if (xmlReader.stackSize() > startOfTagsN + 1) 
                xmlReader.unexpectedTagException();

            if      (topTag.equals( "sourceName")) {}
            else if (topTag.equals("/sourceName")) tSourceName = content;
            else if (topTag.equals( "destinationName")) {}
            else if (topTag.equals("/destinationName")) tDestinationName = content;
            else if (topTag.equals( "dataType")) {}
            else if (topTag.equals("/dataType")) tDataType = content;
            else if (topTag.equals( "addAttributes")) {
                tAttributes = getAttributesFromXml(xmlReader);
                //PrimitiveArray taa= tAttributes.get("_FillValue");
                //String2.log("getSDAD " + tSourceName + " _FillValue=" + taa);
            } else xmlReader.unexpectedTagException();
        }
    }

    /**
     * This sets accessibleTo.
     *
     * @param csvList a space separated value list.
     *    null indicates the dataset is accessible to anyone.
     *    "" means it is accessible to no one.
     */
    public void setAccessibleTo(String csvList) {
        if (csvList == null) {
            accessibleTo = null;  //accessible to all
            return;
        }
        
        accessibleTo = csvList.trim().length() == 0? new String[0] : //accessible to no one
            String2.split(csvList, ',');
        Arrays.sort(accessibleTo);
    }

    /**
     * This gets accessibleTo.
     *
     * @return accessibleTo 
     *    null indicates the dataset is accessible to anyone (i.e., it is public).
     *    length=0 means it is accessible to no one.
     *    length>0 means it is accessible to some roles.
     */
    public String[] getAccessibleTo() {
        return accessibleTo;
    }

    /**
     * Given a list of the current user's roles, this compares it to
     * accessibleTo to determine if this dataset is accessible to this user.
     *
     * @param roles a sorted list of the current user's roles, or null if not logged in.
     * @return true if the dataset is accessible to this user
     */
    public boolean isAccessibleTo(String roles[]) {
        String message = "";
        boolean showMessage = reallyVerbose;
        if (showMessage) message = datasetID + 
            " accessibleTo=" + String2.toSSVString(accessibleTo) + 
            "\n  user roles=" + String2.toSSVString(roles) +
            "\n  accessible="; 

        //dataset is accessible to all?
        if (accessibleTo == null) {
            if (showMessage) String2.log(message + "true");
            return true;
        }

        //i.e., user not logged in
        if (roles == null) {
            if (showMessage) String2.log(message + "false");
            return false;
        }

        //look for a match in the two sorted lists by walking along each list
        int accessibleToPo = 0;
        int rolesPo = 0;
        while (accessibleToPo < accessibleTo.length &&
               rolesPo < roles.length) {
            int diff = accessibleTo[accessibleToPo].compareTo(roles[rolesPo]);
            if (diff == 0) {
                if (showMessage) String2.log(message + "true");
                return true;
            }

            //advance the pointer for the lower string
            if (diff < 0) accessibleToPo++;
            else rolesPo++;
        }

        //we reached the end of one of the lists without finding a match
        if (showMessage) String2.log(message + "false");
        return false;
    }



    /** 
     * This indicates if the dataset is accessible via WMS.
     */
    public boolean accessibleViaWMS() {
        return false;
    }

    /**
     * This returns the dapProtocol for this dataset (e.g., griddap).
     *
     * @return the dapProtocol
     */
    public abstract String dapProtocol();

    /**
     * This returns an HTML description of the dapProtocol for this dataset.
     *
     * @return the dapDescription
     */
    public abstract String dapDescription();

    /**
     * This returns the standard long HTML description of the dapProtocol for this dataset.
     *
     * @return the standard long HTML dapDescription
     */
    public abstract String longDapDescriptionHtml();

    /** 
     * The datasetID is a very short string identifier 
     * (required: just safe characters: A-Z, a-z, 0-9, _, -, or .)
     * for this dataset, 
     * often the source of the dataset (e.g., "erd") and the source's
     * name for the dataset (e.g., "ATssta8day") combined (e.g., "erdATssta8day").  
     * <br>The datasetID must be unique, 
     *   as datasetID is used as the virtual directory for this dataset.
     * <br>This is for use in this program (it is part of the datasets name
     *   that is shown to the user) and shouldn't (if at all possible)
     *   change over time (whereas the 'title' might change). 
     * <br>This needn't match any external name for this dataset (e.g., the
     *   id used by the source, or close to it), but it is sometimes helpful for users if it does.
     * <br>It is usually &lt; 15 characters long.
     *
     * @return the datasetID
     */
    public String datasetID() {return datasetID; }

    /** 
     * The className is the name of the non-abstract subclass of this EDD, e.g., EDDTableFromDapSequence.
     * 
     * @return the className
     */
    public String className() {return className; }

    /** 
     * onChange is a list of 0 or more actions (starting with "http://" or "mailto:")
     * to be done whenever the dataset changes significantly.
     * onChange may be null.
     *
     * @return the internal onChange StringArray -- don't change it!
     */
    public StringArray onChange() {return onChange; }

    /** 
     * The title is a descriptive title for this dataset, 
     * e.g., "SST Anomaly, Pathfinder Ver 5.0, Day and Night, 0.05 degrees, Global, Science Quality".  
     * It is usually &lt; 80 characters long.
     * The information is often originally from the CF-1.0 global metadata for "title".
     *
     * @return the title
     */
    public String title() {
        if (title == null) 
            title = combinedGlobalAttributes.getString("title");
        return title;
    }

    /** 
     * The summary is a longer description for this dataset.  
     * It is usually &lt; 500 characters long.
     * It may have newline characters (usually at &lt;= 72 chars per line).
     * The information is often originally from the CF-1.0 global metadata for "summary".
     *
     * @return the summary
     */
    public String summary() {
        if (summary == null) 
            summary = combinedGlobalAttributes.getString("summary");
        return summary; 
    }

    /** 
     * The institution identifies the source of the data which should receive
     * credit for the data, suitable for "Data courtesy of " in the legend on a graph,
     * e.g., NOAA NESDIS OSDPD. 
     * It is usually &lt; 20 characters long.
     * The information is often originally from the CF-1.0 global metadata for "institution".
     * 
     * @return the institution
     */
    public String institution() {
        if (institution == null) 
            institution = combinedGlobalAttributes.getString(EDStatic.INSTITUTION);
        return institution; 
    }

    /** 
     * The infoUrl identifies a url with information about the dataset. 
     * The information was supplied by the constructor and is stored as 
     * global metadata for "infoUrl" (non-standard).
     * 
     * @return the infoUrl
     */
    public String infoUrl() {
        if (infoUrl == null) 
            infoUrl = combinedGlobalAttributes.getString("infoUrl");
        return infoUrl; 
    }

    /** 
     * The sourceUrl identifies the source (usually) url (for information purposes only)
     * for this dataset. 
     * The information was supplied by the constructor and is stored as 
     * global metadata for "sourceUrl" (non-standard).
     * 
     * @return the sourceUrl
     */
    public String sourceUrl() {
        if (sourceUrl == null) 
            sourceUrl = combinedGlobalAttributes.getString("sourceUrl");
        return sourceUrl; 
    }

    /** 
     * The cdm_data_type global attribute identifies the type of data according to the 
     * options in 
     * http://www.unidata.ucar.edu/software/netcdf-java/formats/DataDiscoveryAttConvention.html
     * for the cdm_data_type global metadata. 
     * It must be one of "Grid", "Station", "Trajectory", "Point" 
     * (or not yet used options: "Image", "Swath"). 
     * [What if some other type???]
     * 
     * @return the cdmDataType
     */
    public String cdmDataType() {
        if (cdmDataType == null) 
            cdmDataType = combinedGlobalAttributes.getString("cdm_data_type");
        return cdmDataType; 
    }

    /** 
     * The global attributes from the source.
     *
     * @return the global attributes from the source.
     */
    public Attributes sourceGlobalAttributes() {return sourceGlobalAttributes; }

    /**
     * The global attributes which will be added to (and take precedence over) 
     * the sourceGlobalAttributes when results files are created.
     *
     * @return the global attributes which will be added to (and take precedence over) 
     * the sourceGlobal attributes when results files are created.
     */
    public Attributes addGlobalAttributes() {return addGlobalAttributes; }

    /**
     * The source+add global attributes, then tweaked (e.g., remove "null" values).
     * 
     * @return the source+add global attributes.
     */
    public Attributes combinedGlobalAttributes() {return combinedGlobalAttributes;  }

    /** 
     * This returns the data variable which has the specified source name.
     *
     * @return the specified data variable sourceName
     * @throws Throwable if not found
     */
    public EDV findDataVariableBySourceName(String tSourceName) 
        throws Throwable {

        int which = String2.indexOf(dataVariableSourceNames(), tSourceName);
        if (which < 0) throw new SimpleException(
            "Error: source variable name='" + tSourceName + "' wasn't found.");
        return dataVariables[which];
    }

    /** 
     * This returns the data variable which has the specified destination name.
     *
     * @return the specified data variable destinationName
     * @throws Throwable if not found
     */
    public EDV findDataVariableByDestinationName(String tDestinationName) 
        throws Throwable {

        int which = String2.indexOf(dataVariableDestinationNames(), tDestinationName);
        if (which < 0) throw new SimpleException(
            "Error: destination variable name='" + tDestinationName + "' wasn't found.");
        return dataVariables[which];
    }

    /**
     * This returns a list of the dataVariables' source names.
     *
     * @return a list of the dataVariables' source names.
     *    This always returns the same internal array, so don't change it!
     */
    public String[] dataVariableSourceNames() {
        if (dataVariableSourceNames == null) {
            //do it this way to be a little more thread safe
            String tNames[] = new String[dataVariables.length];
            for (int i = 0; i < dataVariables.length; i++)
                tNames[i] = dataVariables[i].sourceName();
            dataVariableSourceNames = tNames;
        }
        return dataVariableSourceNames;
    }

    /**
     * This returns a list of the dataVariables' destination names.
     *
     * @return a list of the dataVariables' destination names.
     *    This always returns the same internal array, so don't change it!
     */
    public String[] dataVariableDestinationNames() {
        if (dataVariableDestinationNames == null) {
            //do it this way to be a little more thread safe
            String tNames[] = new String[dataVariables.length];
            for (int i = 0; i < dataVariables.length; i++)
                tNames[i] = dataVariables[i].destinationName();
            dataVariableDestinationNames = tNames;
        }
        return dataVariableDestinationNames;
    }

    /**
     * This returns the dataVariables.
     * This is the internal data structure, so don't change it.
     *
     * @return the dataVariables.
     */
    public EDV[] dataVariables() {return dataVariables; }

    /** 
     * This returns the axis or data variable which has the specified destination name.
     * This implementation only knows about data variables, so subclasses
     * like EDDGrid that have axis variables, too, override it.
     *
     * @return the specified axis or data variable destinationName
     * @throws Throwable if not found
     */
    public EDV findVariableByDestinationName(String tDestinationName) 
        throws Throwable {
        return findDataVariableByDestinationName(tDestinationName);
    }

    /** 
     * creationTimeMillis indicates when this dataset was created.
     * 
     * @return when this dataset was created
     */
    public long creationTimeMillis() {return creationTimeMillis; }


    /** 
     * reloadEveryNMinutes indicates how often this program should check
     * for new data for this dataset by recreating this EDD, e.g., 60. 
     * 
     * @return the suggested number of minutes between refreshes
     */
    public int getReloadEveryNMinutes() {return reloadEveryNMinutes; }

    /** 
     * This sets reloadEveryNMinutes.
     * 
     * @param minutes if &lt;=0 or == Integer.MAX_VALUE,
     *    this uses DEFAULT_RELOAD_EVERY_N_MINUTES.
     */
    public void setReloadEveryNMinutes(int tReloadEveryNMinutes) {
        reloadEveryNMinutes = 
            tReloadEveryNMinutes <= 0 || tReloadEveryNMinutes == Integer.MAX_VALUE?
            DEFAULT_RELOAD_EVERY_N_MINUTES : tReloadEveryNMinutes;
    }

    /**
     * This marks this dataset so that it will be reloaded soon 
     * (but not as fast as possible -- via requestReloadASAP) 
     * by setting the creationTime to 0, making it appear as if the 
     * dataset was created long, long ago.
     * In LoadDatasets, &lt;=0 is treated as special case to force reload
     * no matter what reloadEveryNMinutes is.
     */
    public void setCreationTimeTo0() {
        creationTimeMillis = 0;
    }

    /**
     * This creates a flag file in the EDStatic.fullResetFlagDirectory
     * to mark this dataset so that it will be reloaded as soon as possible.
     *
     * When an exception is thrown and this is called, EDStatic.waitThenTryAgain
     * is usually at the start of the exception message.
     */
    public void requestReloadASAP() {
        requestReloadASAP(datasetID);
    }

    public static void requestReloadASAP(String tDatasetID) {
        String2.log("EDD.requestReloadASAP " + tDatasetID);
        String2.writeToFile(EDStatic.fullResetFlagDirectory + tDatasetID, tDatasetID);
        EDStatic.tally.add("RequestReloadASAP (since startup)", tDatasetID);
        EDStatic.tally.add("RequestReloadASAP (last 24 hours)", tDatasetID);
    }



    /**
     * Given words from a text search query, this returns a ranking of this dataset.
     * Not all words need to be found, but not finding a word incurs a large 
     * penalty.
     * This is a case-insensitve search.
     * This uses a Boyer-Moore-like search (see String2.indexOf(byte[], byte[], int[])).
     *
     * @param words the words or phrases to be searched for (already lowercase)
     *    stored as byte[] via word.getBytes("UTF-8").
     * @param jump the jumpTables from String2.makeJumpTable(word).
     * @return a rating value for this dataset (lower numbers are better),
     *   or Integer.MAX_VALUE if words.length == 0 or 
     *   one of the words wasn't found.
     */
    public int searchRank(byte words[][], int jump[][]) {
        if (words.length == 0)
            return Integer.MAX_VALUE;
        int rank = 0;
        searchString(); //ensure it is available
        //int penalty = 10000;
        for (int w = 0; w < words.length; w++) {
            int po = String2.indexOf(searchString, words[w], jump[w]);

            //word not found
            if (po < 0)
                return Integer.MAX_VALUE;
            rank += po;
            //Exact penalty value doesn't really matter. Any large penalty will
            //  force rankings to be ranked by n words found first, 
            //  then the quality of those found.
            //rank += po < 0? penalty : po;  
        }
        return rank;

        //standardize to 0..1000
        //int rank = Math2.roundToInt((1000.0 * rank) / words.length * penalty);
        //if (rank >= 1000? Integer.MAX_VALUE : rank;
        //return rank;
        
        //return rank == words.length * penalty? Integer.MAX_VALUE : rank;
    }

    /**
     * This returns the flagKey (a String of digits) for the datasetID.
     *
     * @param datasetID
     * @return the flagKey (a String of digits)
     */
    public static String flagKey(String tDatasetID) {
        return Math2.reduceHashCode(
            EDStatic.erddapUrl.hashCode() ^ //always use non-https url
            tDatasetID.hashCode() ^ EDStatic.flagKeyKey.hashCode());
    }

    /**
     * This returns flag URL for the datasetID.
     *
     * @param datasetID
     * @return the url which will cause a flag to be set for a given dataset.
     */
    public static String flagUrl(String tDatasetID) {
        //see also Erddap.doSetDatasetFlag
        return EDStatic.erddapUrl + //always use non-https url
            "/setDatasetFlag.txt?datasetID=" + tDatasetID + 
            "&flagKey=" + flagKey(tDatasetID);
    }

    /**
     * This makes/returns the searchString that searchRank searches.
     * Subclasses may overwrite this.
     *
     * @return the searchString that searchRank searches.
     */
    public byte[] searchString() {
        //!!EDDGrid overwrites this
        if (searchString != null) 
            return searchString;

        //make a string to search through
        StringBuffer sb = new StringBuffer();
        sb.append("all\n");
        sb.append(title() + "\n");
        sb.append("datasetID=" + datasetID() + "\n");
        sb.append(dapProtocol() + "\n");
        if (accessibleViaWMS()) 
            sb.append("WMS\n");
        for (int dv = 0; dv < dataVariables.length; dv++) {
            sb.append(dataVariables[dv].destinationName() + "\n");
            if (!dataVariables[dv].sourceName().equalsIgnoreCase(dataVariables[dv].destinationName()))
                sb.append(dataVariables[dv].sourceName() + "\n");
            if (!dataVariables[dv].longName().equalsIgnoreCase(dataVariables[dv].destinationName()))
                sb.append(dataVariables[dv].longName() + "\n");
        }
        sb.append(combinedGlobalAttributes.toString() + "\n");
        for (int dv = 0; dv < dataVariables.length; dv++) sb.append(dataVariables[dv].combinedAttributes().toString() + "\n");
        sb.append("className=" + className + "\n");

        String2.replaceAll(sb, "\"", ""); //no double quotes (esp around attribute values)
        String2.replaceAll(sb, "\n    ", "\n"); //occurs for all attributes
        String tSearchString = sb.toString().toLowerCase();
        searchString = String2.getUTF8Bytes(tSearchString);
        return searchString;
    }

    /**
     * This returns the types of data files that this dataset can be returned as.
     * These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".nc". 
     *
     * @return the types of data files that this dataset can be returned as
     *   (e.g., ".nc"). 
     */
    public abstract String[] dataFileTypeNames();

    /**
     * This returns the file extensions corresponding to the dataFileTypes.
     * E.g., dataFileTypeName=".GoogleEarth" returns dataFileTypeExtension=".kml".
     *
     * @return the file extensions corresponding to the dataFileTypes 
     *   (e.g., ".nc").
     */
    public abstract String[] dataFileTypeExtensions();

    /**
     * This returns descriptions (up to 80 characters long, suitable for a tooltip)
     * corresponding to the dataFileTypes. 
     *
     * @return descriptions corresponding to the dataFileTypes.
     */
    public abstract String[] dataFileTypeDescriptions();

    /**
     * This returns an info URL corresponding to the dataFileTypes. 
     *
     * @return an info URL corresponding to the dataFileTypes.
     */
    public abstract String[] dataFileTypeInfo();

    /**
     * This returns the types of image files that this dataset can be returned 
     * as. These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".largePng". 
     *
     * @return the types of image files that this dataset can be returned as 
     *   (e.g., ".largePng").
     */
    public abstract String[] imageFileTypeNames();

    /**
     * This returns the file extensions corresponding to the imageFileTypes,
     * e.g., imageFileTypeNames=".largePng" returns imageFileTypeExtensions=".png".
     *
     * @return the file extensions corresponding to the imageFileTypes 
     *   (e.g., ".png").
     */
    public abstract String[] imageFileTypeExtensions();

    /**
     * This returns descriptions corresponding to the imageFileTypes 
     * (each is suitable for a tooltip).
     *
     * @return descriptions corresponding to the imageFileTypes.
     */
    public abstract String[] imageFileTypeDescriptions();

    /**
     * This returns an info URL corresponding to the imageFileTypes. 
     *
     * @return an info URL corresponding to the imageFileTypes.
     */
    public abstract String[] imageFileTypeInfo();

    /**
     * This returns the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     *
     * @return the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     */
    public abstract String[] allFileTypeOptions();
     
    /** 
     * This returns the file extension corresponding to a dataFileType
     * or imageFileType.
     *
     * @param fileTypeName (e.g., ".largePng")
     * @return the file extension corresponding to a dataFileType
     *   imageFileType (e.g., ".png").
     * @throws Throwable if not found
     */
    public String fileTypeExtension(String fileTypeName) throws Throwable {
        //if there is need for speed in the future: use hashmap
        int po = String2.indexOf(dataFileTypeNames(), fileTypeName);
        if (po >= 0)
            return dataFileTypeExtensions()[po];

        po = String2.indexOf(imageFileTypeNames(), fileTypeName);
        if (po >= 0)
            return imageFileTypeExtensions()[po];

        throw new SimpleException("Error: fileType=" + fileTypeName + 
                " is not supported by this dataset.");
    }

    /**
     * This returns a suggested fileName (no dir or extension).
     *
     * @param userDapQuery
     * @param fileTypeName
     * @return a suggested fileName (no dir or extension)
     */
    public String suggestFileName(String userDapQuery, String fileTypeName) {

        //include fileTypeName in hash so, e.g., different sized .png 
        //  have different file names
        return datasetID + "_" + 
            Math2.reduceHashCode((userDapQuery + fileTypeName).hashCode()); 
    }

    /**
     * This responds to a DAP-style query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userQuery the part of the user's request after the '?', still percentEncoded, may be null.
     * @param outputStreamSource  the source of an outputStream that receives the results,
     *    usually already buffered.
     * @param dir the directory to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName the fileTypeName for the new file.
     * @throws Throwable if trouble
     */
    public abstract void respondToDapQuery(HttpServletRequest request, 
        String loggedInAs, String requestUrl, String userQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable;

    /**
     * This responds to a graph query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userQuery the part of the user's request after the '?', still percentEncoded, may be null.
     * @param outputStreamSource  the source of an outputStream that receives the results,
     *    usually already buffered.
     * @param dir the directory to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName the fileTypeName for the new file.
     * @throws Throwable if trouble
     */
    public abstract void respondToGraphQuery(HttpServletRequest request, 
        String loggedInAs, String requestUrl, String userQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable;

    /**
     * This deletes the old file (if any) and makes a new actual file 
     * based on an OPeNDAP DAP-style query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param userDapQuery the part of the user's request after the '?'.
     * @param dir the directory that will hold the new file (with a trailing slash).
     * @param fileName the name for the file (no dir, no extension).
     * @param fileTypeName the fileTypeName for the new file.
     * @return fileName + fileExtension for the resulting file
     * @throws Throwable if trouble
     */
    public String makeNewFileForDapQuery(HttpServletRequest request, 
        String loggedInAs, String userDapQuery, 
        String dir, String fileName, String fileTypeName) throws Throwable {

        String fileTypeExtension = fileTypeExtension(fileTypeName);
        File2.delete(dir + fileName + fileTypeExtension);

        return lowMakeFileForDapQuery(request, loggedInAs, userDapQuery, 
            dir, fileName, fileTypeName);
    }

    /**
     * This reuses an existing file or makes a new actual file based on an 
     * OPeNDAP DAP-style query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param userDapQuery the part of the user's request after the '?'.
     * @param dir the directory that will hold the new file (with a trailing slash).
     * @param fileName the name for the file (no dir, no extension).
     * @param fileTypeName the fileTypeName for the new file.
     * @return fileName + fileExtension for the resulting file
     * @throws Throwable if trouble
     */
    public String reuseOrMakeFileForDapQuery(HttpServletRequest request, 
        String loggedInAs, String userDapQuery, 
        String dir, String fileName, String fileTypeName) throws Throwable {

        String fileTypeExtension = fileTypeExtension(fileTypeName);
        String fullName = dir + fileName + fileTypeExtension;
        if (File2.touch(fullName)) {
            if (verbose) String2.log(
                "EDD.makeFileForDapQuery reusing " + fileName + fileTypeExtension);
            return fileName + fileTypeExtension;
        }
        return lowMakeFileForDapQuery(request, loggedInAs, userDapQuery, 
            dir, fileName, fileTypeName);
    }

    /**
     * This makes an actual file based on an OPeNDAP DAP-style query
     * and returns its name (not including the dir, but with the extension).
     * This is mostly used for testing since Erddap uses respondToDapQuery directly.
     *
     * <p>This is a default implementation which calls respondToDapQuery.
     * Some classes overwrite this to have this be the main responder
     * (and have respondToDapQuery call this and then copy the file to outputStream).
     * But that approach isn't as good, because it requires all data be obtained and
     * then written to file before response to user can be started.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param userDapQuery the part of the user's request after the '?'.
     * @param dir the directory that will hold the new file (with a trailing slash).
     * @param fileName the name for the file (no dir, no extension).
     * @param fileTypeName the fileTypeName for the new file.
     * @return fileName + fileExtension
     * @throws Throwable if trouble
     */
    public String lowMakeFileForDapQuery(HttpServletRequest request, 
        String loggedInAs, String userDapQuery, 
        String dir, String fileName, String fileTypeName) throws Throwable {

        String fileTypeExtension = fileTypeExtension(fileTypeName);
        String fullName = dir + fileName + fileTypeExtension;
       
        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        OutputStreamSource outputStreamSource = new OutputStreamSourceSimple(
            (OutputStream)new BufferedOutputStream(new FileOutputStream(fullName + randomInt)));

        try {

            //send the data to the outputStream
            respondToDapQuery(request, loggedInAs,
                "/" + EDStatic.warName +
                (this instanceof EDDGrid? "/griddap/" :
                 this instanceof EDDTable? "/tabledap/" :
                 "/UNKNOWN/") + 
                 datasetID() + fileTypeName, 
                userDapQuery, outputStreamSource, dir, fileName, fileTypeName);

            //close the outputStream
            outputStreamSource.outputStream("").close();
        } catch (Throwable t) {
            try {
                outputStreamSource.outputStream("").close();
            } catch (Throwable t2) {
                //don't care
            }
            //delete the temporary file
            File2.delete(fullName + randomInt);
            throw t;
        }

        //rename the file to the specified name
        File2.rename(fullName + randomInt, fullName);
        return fileName + fileTypeExtension;
    }

    /**
     * This writes the dataset info (id, title, institution, infoUrl, summary)
     * to an html document.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param writer
     * @param showDafLink if true, a link is shown to this dataset's Data Access Form
     * @param showGraphLink if true, a link is shown to this dataset's Make A Graph form
     * @param userDapQuery  the part of the user's request after the '?', still percentEncoded, may be null.
     * @param otherRows
     * @throws Throwable if trouble
     */
    public void writeHtmlDatasetInfo(
        String loggedInAs, Writer writer, boolean showDafLink, boolean showGraphLink,
        String userDapQuery, String otherRows) 
        throws Throwable {
        //String type = this instanceof EDDGrid? "Gridded" :
        //   this instanceof EDDTable? "Tabular" : "(type???)";
        
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String tQuery = userDapQuery == null || userDapQuery.length() == 0? "" :
            //since this may be direct from user, I need to XML encode it 
            //to prevent HTML insertion security vulnerability
            //(which allows hacker to insert his javascript into pages returned by server)
            //See Tomcat (Definitive Guide) pg 147...
            "?" + XML.encodeAsXML(userDapQuery); 
        String dapLink = "", graphLink = "";
        if (showDafLink) 
            dapLink = 
                "     | <a title=\"Click to see a DAP Data Access Form for this data set.\" \n" +
                "         href=\"" + tErddapUrl + "/" + dapProtocol() + "/" + datasetID() + ".html" + 
                    tQuery + "\">Data Access Form</a>\n";
        if (showGraphLink) 
            graphLink = 
                "     | <a title=\"Click to see a Make A Graph form for this data set.\" \n" +
                "         href=\"" + tErddapUrl + "/" + dapProtocol() + "/" + datasetID() + ".graph" + 
                    tQuery + "\">Make A Graph</a>\n";
        String tSummary = summary(); 
        writer.write(
            //"<p><b>" + type + " Dataset:</b>\n" +
            "<p><table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
            "  <tr>\n" +
            "    <td nowrap>Dataset Title:&nbsp;</td>\n" +
            "    <td><b>" + XML.encodeAsXML(title()) + "</b></td>\n" +
            "  </tr>\n" +
            "  <tr>\n" +
            "    <td nowrap>Dataset ID:&nbsp;</td>\n" +
            "    <td>" + XML.encodeAsXML(datasetID()) + "\n" +
            "      " + rssHref() + "\n" +
            "      " + emailHref(tErddapUrl) + " </td>\n" +
            "  </tr>\n" +
            "  <tr>\n" +
            "    <td nowrap>Institution:&nbsp;</td>\n" +
            "    <td>" + XML.encodeAsXML(institution()) + "</td>\n" +
            "  </tr>\n" +
            otherRows +
            "  <tr>\n" +
            "    <td nowrap>Information:&nbsp;</td>\n" +
            "    <td>Summary " + 
                EDStatic.htmlTooltipImage(XML.encodeAsPreXML(tSummary, 100)) +  "\n" +
            "     | <a title=\"" + EDStatic.clickInfo + "\" \n" +
            "          href=\"" + tErddapUrl + "/info/" + datasetID() + "/index.html\">Variables</a>\n" +
            "     | <a title=\"" + EDStatic.clickBackgroundInfo + "\" \n" +
            "         href=\"" + infoUrl() + "\">Background</a>\n" +
                dapLink + "\n" +
                graphLink + "</td>\n" +
            "  </tr>\n" +
            "</table>\n");
    }

    /**
     * This returns the kml code for the screenOverlay (which is the KML code
     * which describes how/where to display the googleEarthLogoFile).
     * This is used by EDD subclasses when creating KML files.
     *
     * @return the kml code for the screenOverlay.
     */
    public String getKmlIconScreenOverlay() {
        return 
            "  <ScreenOverlay id=\"Logo\">\n" + //generic id
            "    <description>" + EDStatic.erddapUrl + //always use non-https url
                "</description>\n" +
            "    <name>Logo</name>\n" + //generic name
            "    <Icon>" +
                  "<href>" + EDStatic.imageDirUrl + EDStatic.googleEarthLogoFile + "</href>" +
                "</Icon>\n" +
            "    <overlayXY x=\"0.005\" y=\".04\" xunits=\"fraction\" yunits=\"fraction\"/>\n" +
            "    <screenXY x=\"0.005\" y=\".04\" xunits=\"fraction\" yunits=\"fraction\"/>\n" +
            "    <size x=\"0\" y=\"0\" xunits=\"pixels\" yunits=\"pixels\"/>\n" + //0=original size
            "  </ScreenOverlay>\n";
    }

    /**
     * This is used by subclass's generateDatasetsXml methods to make
     * sure that the global attributes includes at least place holders (dummy values)
     * for the required/common global attributes.
     *
     * @param gAtts
     * @param cdmDataType can be a specific type (e.g., "Grid") or null (for the default)
     */
    public static void addDummyRequiredGlobalAttributesForDatasetsXml(Attributes gAtts, 
        String cdmDataType) {
        if (gAtts.getString("cdm_data_type")            == null) 
            gAtts.add("cdm_data_type",            
                cdmDataType == null? "???Grid|Point|Station|Trajectory" : cdmDataType);
        gAtts.add("Conventions", 
            (gAtts.getString("Conventions") == null? "" : gAtts.getString("Conventions") + "???") +
            "???COARDS, CF-1.0, Unidata Dataset Discovery v1.0");
        if (gAtts.getString("infoUrl")                  == null) gAtts.add("infoUrl",                  "???");
        if (gAtts.getString("institution")              == null) gAtts.add("institution",              "???");
        if (gAtts.getString("license")                  == null) gAtts.add("license",                  "???[standard]");
        if (gAtts.getString("standard_name_vocabulary") == null) gAtts.add("standard_name_vocabulary", "???" + FileNameUtility.getStandardNameVocabulary());
        if (gAtts.getString("summary")                  == null) gAtts.add("summary",                  "???");
        if (gAtts.getString("title")                    == null) gAtts.add("title",                    "???");
    }

    /**
     * This is used by subclass's generateDatasetsXml methods to make
     * sure that a variable's attributes includes at least place holders (dummy values)
     * for the required/common attributes.
     *
     * @param atts
     * @param varName
     */
    public static void addDummyRequiredVariableAttributesForDatasetsXml(Attributes atts, 
        String varName, boolean addColorBarMinMax) {
        if (addColorBarMinMax) {
            if (atts.getString("colorBarMinimum") == null) atts.add("colorBarMinimum", Double.NaN);
            if (atts.getString("colorBarMaximum") == null) atts.add("colorBarMaximum", Double.NaN);
        }
        if (atts.getString("ioos_category")  == null) atts.add("ioos_category",  "???");
        if (atts.getString("long_name")      == null) atts.add("long_name", "???" + 
            EDV.suggestLongName(varName, atts.getString("standard_name")));
        if (atts.getString("standard_name")  == null) atts.add("standard_name",  "???");
        if (atts.getString("units")          == null) atts.add("units",          "???");
    }

    /**
     * This is used by subclass's generateDatasetsXml methods to write 
     * directions to datasets.xml file.
     *
     * @param writer
     * @throws Throwable if trouble
     */
    public static String directionsForDatasetsXml() throws Throwable {
        return 
"<!-- Directions:\n" +
" * Read about this type of dataset in\n" +
"   http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html .\n" +
" * Read http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html#addAttributes\n" +
"   so that you understand about sourceAttributes and addAttributes.\n" +
" * All of the content below that starts with \"???\" is a guess. It must be edited.\n" +
" * All of the other tags and their content are based on information from the source.\n" +
" * For the att tags, you should either:\n" +
"    * Delete the att tag (so ERDDAP will use the unchanged source attribute).\n" +   
"    * Change the att tag's value (because it isn't quite right).\n" +
"    * Or, remove the att tag's value, but leave the att tag\n" +
"      (so the source attribute will be removed by ERDDAP).\n" +
" * You can reorder data variables, but don't reorder axis variables.\n" +
" * The current IOOS category options are:\n" +
"      " + String2.noLongLines(String2.toCSVString(EDV.IOOS_CATEGORIES), 80, "      ") + "\n" +
" * For longitude, latitude, altitude (or depth), and time variables:\n" +
"   * If the sourceName isn't \"longitude\", \"latitude\", \"altitude\", or \"time\",\n" +
"     you need to specify \"longitude\", \"latitude\", \"altitude\", or \"time\"\n" +
"     with a destinationName tag.\n" +
"   * For EDDTable datasets: if possible, add an actual_range attribute\n" +
"     if one isn't already there.\n" +
"   * (Usually) remove all other attributes. They usually aren't needed.\n" +
"     Attributes will be added automatically.\n" +
"-->\n";
    }

    /**
     * This is used by subclass's generateDatasetsXml methods to write the
     * variables to the writer in the datasets.xml format.
     *
     * @param writer
     * @param table
     * @param variableType  e.g., axisVariable or dataVariable
     * @throws Throwable if trouble
     */
    public static String variablesForDatasetsXml(Table table,
        String variableType, boolean includeDataType) throws Throwable {

        String indent = "        ";
        StringBuffer sb = new StringBuffer();
        for (int col = 0; col < table.nColumns(); col++) {
            sb.append(
                indent + "<" + variableType + ">\n" +
                indent + "    <sourceName>" + table.getColumnName(col) + "</sourceName>\n" +
                indent + "    <destinationName>???" + table.getColumnName(col) + "</destinationName>\n");
            if (includeDataType) sb.append(
                indent + "    <dataType>" + table.getColumn(col).getElementTypeString() + "</dataType>\n");
            sb.append(attsForDatasetsXml(table.columnAttributes(col), indent + "    "));
            sb.append(
                indent + "</" + variableType + ">\n");
        }
        return sb.toString();
    }

    /**
     * This is used by subclass's generateDatasetsXml methods to write the
     * attributes to the writer in the datasets.xml format.
     *
     * @param writer
     * @param atts
     * @param indent a string of spaces
     * @throws Throwable if trouble
     */
    public static String attsForDatasetsXml(Attributes atts, String indent) throws Throwable {
        StringBuffer sb = new StringBuffer();
        sb.append(indent + "<addAttributes>\n");
        String names[] = atts.getNames();
        for (int att = 0; att < names.length; att++) {
            PrimitiveArray attPa = atts.get(names[att]);
            sb.append(indent + "    <att name=\"" + names[att] + "\"");
            if (attPa instanceof StringArray)
                sb.append(">" + XML.encodeAsXML(attPa.getString(0)) + "</att>\n");
            else sb.append(" type=\"" + attPa.getElementTypeString() + 
                (attPa.size() > 1? "List" : "") +
                "\">" + String2.replaceAll(attPa.toString(), ", ", " ") + "</att>\n");
        }
        sb.append(indent + "</addAttributes>\n");
        return sb.toString();
    }

    /**
     * This adds a line to the "history" attribute (which is created if it 
     * doesn't already exist).
     *
     * @param attributes (always a COPY of the dataset's global attributes,
     *    so you don't get multiple similar history lines of info)
     * @param text  usually one line of info
     */
    public static void addToHistory(Attributes attributes, String text) {
        String add = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10) +
            " " + text;
        String history = attributes.getString("history");
        if (history == null)
            history = add;
        else history += "\n" + add;
        attributes.set("history", history);
    }

    /**
     * This determines if a longName is substantially different from a destinationName
     * and should be shown on a Data Access Form.
     *
     * @param varName
     * @param longName
     * @return true if the longName is substantially different and should be shown.
     */
    public static boolean showLongName(String destinationName, String longName) {
        if (destinationName.length() >= 20)
            return false; //varName is already pretty long
        destinationName = String2.replaceAll(destinationName.toLowerCase(), " ", "");
        destinationName = String2.replaceAll(destinationName, "_", "");
        longName = String2.replaceAll(longName.toLowerCase(), " ", "");
        longName = String2.replaceAll(longName, "_", "");
        return !destinationName.equals(longName); //if not the same, show longName
    }

    /**
     * This returns a new, empty, badFileMap (a synchronized map).
     */
    public Map newEmptyBadFileMap() {
        return Collections.synchronizedMap(new HashMap());
    }

    /** The name of the badFileMap file. */
    public String badFileMapFileName() {
        return EDStatic.fullDatasetInfoDirectory + datasetID + ".bad.json";
    }

    /**
     * This reads a badFile table from disk and creates a synchronized map 
     * (key=dir#/fileName, value=Object[0=(Double)lastMod, 1=(String)reason]).
     * <br>If trouble, this won't throw an Exception, but will return null.
     *
     * @return a synchronized Map
     */
    public Map readBadFileMap() {
        Map badFilesMap = newEmptyBadFileMap();
        String fileName = badFileMapFileName();
        try {
            if (File2.isFile(fileName)) {
                Table badTable = new Table();
                badTable.readJson(fileName);  //it logs nRows=
                int nRows = badTable.nRows();
                int nColumns = badTable.nColumns();
                Test.ensureEqual(nColumns, 3, "Unexpected number of columns.");
                Test.ensureEqual(badTable.getColumnName(0), "fileName", "Unexpected column#0 name.");
                Test.ensureEqual(badTable.getColumnName(1), "lastMod",  "Unexpected column#1 name.");
                Test.ensureEqual(badTable.getColumnName(2), "reason",   "Unexpected column#2 name.");
                Test.ensureEqual(badTable.getColumn(0).getElementTypeString(), "String", "Unexpected column#0 type.");
                Test.ensureEqual(badTable.getColumn(1).getElementTypeString(), "double", "Unexpected column#1 type.");
                Test.ensureEqual(badTable.getColumn(2).getElementTypeString(), "String", "Unexpected column#2 type.");
                if (nRows == 0)
                    return badFilesMap;
                for (int row = 0; row < nRows; row++) 
                    badFilesMap.put(badTable.getStringData(0, row), 
                        new Object[]{new Double(badTable.getDoubleData(1, row)),
                                     badTable.getStringData(2, row)});
            }
            return badFilesMap;
        } catch (Throwable t) {
            EDStatic.email(EDStatic.emailEverythingTo, 
                "Error while reading table of badFiles",
                "Error while reading table of badFiles\n" + fileName + "\n" + 
                MustBe.throwableToString(t));  
            File2.delete(fileName);
            return null;
        }
    }

    /**
     * This makes a badFile table from a synchronized map  
     * (key=dir#/fileName, value=Object[0=(Double)lastMod, 1=(String)reason]).
     * and writes it to disk.
     * <br>This won't throw an Exception. If the file can't be written, nothing is done.
     *
     * @param badFilesMap
     * @return a synchronized Map
     * @throws Throwable if trouble
     */
    public void writeBadFileMap(String randomFileName, Map badFilesMap) throws Throwable {

        try {
            //gather the fileNames and reasons
            StringArray fileNames = new StringArray();
            DoubleArray lastMods  = new DoubleArray();
            StringArray reasons   = new StringArray();
            Object keys[] = badFilesMap.keySet().toArray();
            for (int k = 0; k < keys.length; k++) {
                Object o = badFilesMap.get(keys[k]);
                if (o != null) {
                    fileNames.add(keys[k].toString());
                    Object oar[] = (Object[])o;
                    lastMods.add(((Double)oar[0]).doubleValue());
                    reasons.add(oar[1].toString());
                }
            }

            //make and write the badFilesTable
            Table badTable = new Table();
            badTable.addColumn("fileName", fileNames);
            badTable.addColumn("lastMod",  lastMods);
            badTable.addColumn("reason",   reasons);
            badTable.saveAsJson(randomFileName, -1, false);
            if (verbose) String2.log("Table of badFiles successfully written. nRows=" + badTable.nRows() + "\n" +
                randomFileName);
        } catch (Throwable t) {
            String subject = "Error while writing table of badFiles";
            EDStatic.email(EDStatic.emailEverythingTo, 
                subject,
                subject + "\n" + randomFileName + "\n" + 
                MustBe.throwableToString(t));  
            File2.delete(randomFileName);
            throw t;
        }
    }

    /** 
     * This adds fileName, lastMod, and reason to a badFiles map. 
     *
     * @param badFileMap
     * @param dirIndex   
     * @param fileName   the fileName, for example  AG20090109.nc
     * @param lastMod   the lastModified time of the file
     * @param reason
     */
    public void addBadFile(Map badFileMap, int dirIndex, String fileName, double lastMod, String reason) {
        String2.log(datasetID + " addBadFile: " + fileName + "\n  reason=" + reason);
        badFileMap.put(dirIndex + "/" + fileName, new Object[]{new Double(lastMod), reason});
    }

    /** 
     * This reads the table of badFiles, adds fileName and reason, and writes the table of badFiles. 
     * This is used outside of the constructor, when a previously good file is found to be bad.
     * This won't throw an exception, just logs the message.
     *
     * @param dirIndex   
     * @param fileName   the fileName, for example  AG20090109.nc
     * @param lastMod   the lastModified time of the file
     * @param reason
     * @return an error string ("" if no error).
     */
    public String addBadFileToTableOnDisk(int dirIndex, String fileName, double lastMod, 
        String reason) {

        Map badFileMap = readBadFileMap();
        addBadFile(badFileMap, dirIndex, fileName, lastMod, reason);
        String badFileMapFileName = badFileMapFileName();
        int random = Math2.random(Integer.MAX_VALUE);
        try {
            writeBadFileMap(badFileMapFileName + random, badFileMap);
            File2.rename(badFileMapFileName + random, badFileMapFileName);
            return "";
        } catch (Throwable t) {
            File2.delete(badFileMapFileName + random);
            String msg = "Error: " + MustBe.throwableToString(t);
            String2.log(msg);
            return msg;
        }
    }

    /** 
     * This returns a string representation of the information in a badFileMap.
     * 
     * @param badFileMap
     * @param dirList
     * @return a string representation of the information in a badFileMap.
     *     If there are no badFiles, this returns "".
     */
    public String badFileMapToString(Map badFileMap, StringArray dirList) {

        Object keys[] = badFileMap.keySet().toArray();
        if (keys.length == 0) 
            return "";
        StringBuffer sb = new StringBuffer(
            "\n" +
            "********************************************\n" +
            "List of Bad Files for datasetID=" + datasetID + "\n\n");
        int nDir = dirList.size();
        Arrays.sort(keys);
        for (int k = 0; k < keys.length; k++) {
            Object o = badFileMap.get(keys[k]);
            String dir = File2.getDirectory(keys[k].toString());
            int dirI = dir.length() > 1 && dir.endsWith("/")?
                String2.parseInt(dir.substring(0, dir.length() - 1)) : -1;
            if (o != null && dirI >= 0 && dirI < nDir) { 
                Object oar[] = (Object[])o;
                sb.append(dirList.get(dirI) + File2.getNameAndExtension(keys[k].toString()) + "\n" +
                  oar[1].toString() + "\n\n"); //reason
            }
        }
        sb.append(
            "********************************************\n");
        return sb.toString();
    }


    /**
     * This returns list of &amp;-separated parts, in their original order, from a percent encoded userQuery.
     * This is like split(,'&amp;'), but smarter.
     * This accepts:
     * <ul>
     * <li>connecting &amp;'s already visible (within a part, 
     *     &amp;'s must be within double quotes or percent-encoded)
     * <li>connecting &amp;'s are percent encoded (within a part, 
     *     &amp;'s must be within double quotes).
     * </ul>
     *
     * @param userQuery the part after the '?', still percentEncoded, may be null.
     * @return a StringArray with the percentDecoded parts, in their original order.
     *   A null or "" userQuery will return String[1] with #0=""
     * @throws Throwable if trouble (e.g., invalid percentEncoding)
     */
    public static String[] getUserQueryParts(String userQuery) throws Throwable {
        if (userQuery == null || userQuery.length() == 0)
            return new String[]{""};

        boolean stillEncoded = true;
        if (userQuery.indexOf('&') < 0) {
            //perhaps user percentEncoded everything, even the connecting &'s, so decode everything right away
            userQuery = SSR.percentDecode(userQuery);
            stillEncoded = false;
        }
        //String2.log("userQuery=" + userQuery);

        //one way or another, connecting &'s should now be visible
        userQuery += "&"; //& triggers grabbing final part
        int userQueryLength = userQuery.length();
        int start = 0;
        boolean inQuotes = false;
        StringArray parts = new StringArray(); 
        for (int po = 0; po < userQueryLength; po++) {
            char ch = userQuery.charAt(po);
            //String2.log("ch=" + ch);
            if (ch == '"') {             //what about \" within "..."?
                inQuotes = !inQuotes;
            } else if (ch == '&' && !inQuotes) {
                String part = userQuery.substring(start, po);
                parts.add(stillEncoded? SSR.percentDecode(part) : part);
                //String2.log("part=" + parts.get(parts.size() - 1));
                start = po + 1;
            }
        }
        if (inQuotes)
            throw new SimpleException("Query error: A closing doublequote is missing.");
        return parts.toArray();
    }

    /**
     * This returns a HashMap with the variable=value entries from a userQuery.
     *
     * @param userQuery the part after the '?', still percentEncoded, may be null.
     * @param namesLC if true, the names are made toLowerCase.
     * @return HashMap<String, String>  
     *   <br>The keys and values will be percentDecoded.
     *   <br>A null or "" userQuery will return an empty hashMap.
     *   <br>If a part doesn't have '=', then it doesn't generate an entry in hashmap.
     * @throws Throwable if trouble (e.g., invalid percentEncoding)
     */
    public static HashMap<String, String> userQueryHashMap(String userQuery, boolean namesLC) throws Throwable {
        HashMap<String, String> queryHash = new HashMap<String, String>();
        if (userQuery != null) {
            String tParts[] = getUserQueryParts(userQuery); //userQuery="" returns String[1]  with #0=""
            for (int i = 0; i < tParts.length; i++) {
                int po = tParts[i].indexOf('=');
                if (po > 0) {
                    //if (reallyVerbose) String2.log(tParts[i]);
                    String name = tParts[i].substring(0, po);
                    if (namesLC)
                        name = name.toLowerCase();
                    queryHash.put(name, tParts[i].substring(po + 1));
                }
            }
        }
        return queryHash;
    }

    /**
     * This callse testDasDds(tDatasetID, true).
     */
    public static void testDasDds(String tDatasetID) throws Throwable {
        testDasDds(tDatasetID, true);
    }

    /**
     * Write a dataset's .das and .dds to String2.log
     * (usually for test purposes when setting up a dataset).
     */
    public static void testDasDds(String tDatasetID, boolean reallyVerbose) throws Throwable {
        verbose = true;
        reallyVerbose = reallyVerbose;
        Table.verbose = true;
        EDV.verbose = true;
        NcHelper.verbose = true;
        OpendapHelper.verbose = true;
        String2.log("\n*** DasDds " + tDatasetID);
        String tName, results, expected;

        EDD edd = oneFromDatasetXml(tDatasetID); 

        tName = edd.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            "EDD.testDasDds_" + tDatasetID, ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log("\n**************************** The .das for " + tDatasetID + " ****************************");
        String2.log(results);

        tName = edd.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            "EDD.testDasDds_" + tDatasetID, ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log("\n**************************** The .dds for " + tDatasetID + " ****************************");
        String2.log(results);

    }

    /**
     * This sets verbose=true and reallyVerbose=true for this class
     * and related clases, for tests.
     *
     * @throws Throwable if trouble
     */
    public static void testVerboseOn() {
        verbose = true;
        reallyVerbose = true;
        Calendar2.verbose = true;
        Calendar2.reallyVerbose = true;
        gov.noaa.pfel.coastwatch.pointdata.DigirHelper.verbose = true;
        gov.noaa.pfel.coastwatch.pointdata.DigirHelper.reallyVerbose = true;
        EDV.verbose = true;
        EDV.reallyVerbose = true;
        GridDataAccessor.verbose = true;
        GridDataAccessor.reallyVerbose = true;
        OpendapHelper.verbose = true;
        SgtGraph.verbose = true;
        SgtGraph.reallyVerbose = true;
        SgtMap.verbose = true;
        SgtMap.reallyVerbose = true;
        Table.verbose = true;
        TaskThread.verbose = true;
        TaskThread.reallyVerbose = true;
    }

}
