/* 
 * EDDGrid Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.NDimensionalIndex;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import dods.dap.*;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.griddata.Matlab;
import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.CompoundColorMap;
import gov.noaa.pfel.coastwatch.sgt.GraphDataLayer;
import gov.noaa.pfel.coastwatch.sgt.SgtGraph;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.sgt.SgtUtil;
import gov.noaa.pfel.coastwatch.util.SSR;
import gov.noaa.pfel.erddap.util.*;
import gov.noaa.pfel.erddap.variable.*;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.RenderingHints;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.NoSuchElementException;
import javax.servlet.http.HttpServletRequest;

import ucar.ma2.Array;
import ucar.nc2.Dimension;
import ucar.nc2.geotiff.GeotiffWriter;
import ucar.nc2.NetcdfFileWriteable;
import ucar.unidata.geoloc.LatLonRect;
import ucar.unidata.geoloc.LatLonPointImpl;

/** 
 * This class represents a dataset where the results can be represented as 
 * a grid -- one or more EDV variables
 * sharing the same EDVGridAxis variables (in the same order).
 * If present, the lon, lat, alt, and time axisVariables
 * allow queries to be made in standard units (alt in m above sea level and
 * time as seconds since 1970-01-01T00:00:00Z or as an ISO date/time).
 * 
 * <p>Note that all variables for a given EDDGrid use the same 
 * axis variables. If there are source datasets that
 * serve variables which use different sets of axes, you have
 * to separate them out and create separate EDDGrids (one per 
 * set of axes).
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-04
 */
public abstract class EDDGrid extends EDD { 

    public final static String dapProtocol = "griddap";

    /** The constructor must set these. */
    protected EDVGridAxis axisVariables[];

    /** These are needed for EDD-required methods of the same name. */
    public final static String[] dataFileTypeNames = {  //
        ".asc", ".csv", ".das", ".dds", ".dods", 
        ".esriAscii", //".grd", ".hdf", 
        ".graph", ".help", ".html", ".htmlTable",
        ".json", 
        ".mat", ".nc", ".ncHeader", ".tsv", ".xhtml"};
    public final static String[] dataFileTypeExtensions = {
        ".asc", ".csv", ".das", ".dds", ".dods", 
        ".asc", //".grd", ".hdf", 
        ".html", ".html", ".html", ".html",
        ".json", 
        ".mat", ".nc", ".txt", ".tsv", ".xhtml"};
    //These all used to have " (It may take a minute. Please be patient.)" at the end.
    public static String[] dataFileTypeDescriptions = {
        "View the data as OPeNDAP-style comma-separated ASCII text.",
        "Download the data as comma-separated ASCII text table (missing value = 'NaN'; times are ISO 8601 strings).",
        "View the data's metadata via an OPeNDAP Dataset Attribute Structure (DAS).",
        "View the data's structure via an OPeNDAP Dataset Descriptor Structure (DDS).",
        "OPeNDAP clients use this to download the data in the DODS binary format.",
        "Download an ESRI ASCII file (for lat lon data only; lon values can't be below and above 180).",
        //"Download a GMT-style NetCDF .grd file (for lat lon data only).",
        //"Download a Hierarchal Data Format Version 4 SDS file (for lat lon data only).",
        "View a Make A Graph web page.",
        "View a web page with a description of griddap.",
        "View an OPeNDAP-style HTML Data Access Form.",
        "View an HTML file with the data in a table (times are ISO 8601 strings).",
        "View the data as a JSON table (missing value = 'null'; times are ISO 8601 strings).",
        "Download a MATLAB binary file.",
        "Download a NetCDF binary file with COARDS/CF/THREDDS metadata.",
        "View the header (the metadata) for the NetCDF file.",
        "Download the data as tab-separated ASCII text table (missing value = 'NaN'; times are ISO 8601 strings).",
        "View an XHTML file with the data in a table (times are ISO 8601 strings)."
    };
    public static String[] dataFileTypeInfo = {
        "http://www.opendap.org/user/guide-html/guide_20.html#id4", //dap ascii
        "http://www.creativyst.com/Doc/Articles/CSV/CSV01.htm", //csv
        "http://www.opendap.org/user/guide-html/guide_66.html", //das
        "http://www.opendap.org/user/guide-html/guide_65.html", //dds
        "http://www.opendap.org", //dods
        "http://www.geotools.org/ArcInfo+ASCII+Grid+format", //esriAscii
        //"http://gmt.soest.hawaii.edu/gmt/doc/html/GMT_Docs/node60.html", //grd
        //"http://hdf.ncsa.uiuc.edu/hdf4.html", //hdf
        "http://coastwatch.pfeg.noaa.gov/erddap/griddap/index.html#GraphicsCommands", //GraphicsCommands
        "http://www.opendap.org/user/guide-html/guide_20.html#id8", //help
        "http://www.opendap.org/user/guide-html/guide_21.html", //html
        "http://www.w3schools.com/html/html_tables.asp", //htmlTable
        "http://www.json.org/", //json
        "http://www.mathworks.com/", //mat
        "http://www.unidata.ucar.edu/software/netcdf/", //nc
        "http://www.unidata.ucar.edu/software/netcdf/guide_12.html#SEC95", //ncHeader
        "http://www.cs.tut.fi/~jkorpela/TSV.html",  //tsv
        "http://www.w3schools.com/html/html_tables.asp" //xhtml
        //"http://www.tizag.com/htmlT/tables.php" //xhtml
    };

    public final static String[] imageFileTypeNames = {
        ".geotif", ".kml", 
        ".smallPdf", ".pdf", ".largePdf", ".smallPng", ".png", ".largePng", ".transparentPng"
        };
    public final static String[] imageFileTypeExtensions = {
        ".tif", ".kml", 
        ".pdf", ".pdf", ".pdf", ".png", ".png", ".png", ".png"
        };
    public static String[] imageFileTypeDescriptions = {
        "Download a georeferenced .tif (GeoTIFF) image file (for lat lon data only; lon values can't be below and above 180).",
        "Download a Google Earth .kml file (for lat, lon, [time] results only)",
        "Download a small .pdf image file with a graph/map of the data you selected.",
        "Download a standard, medium-sized, .pdf image file with a graph/map of the data you selected.",
        "Download a large .pdf image file with a graph/map of the data you selected.",
        "Download a small .png image file with a map of the data you selected.",
        "Download a standard, medium-sized .png image file with a map of the data you selected.",
        "Download a large .png image file with a map of the data you selected." ,
        "Download a .png image file with the data you selected (a geographic map without axes, landmask, or legend)."
        };  //.transparentPng: if lon and lat are evenly spaced, .png size will be 1:1; otherwise, 1:1 but morphed a little
    public static String[] imageFileTypeInfo = {
        "http://www.remotesensing.org/geotiff/geotiff.html", //geotiff
        "http://earth.google.com/", //kml
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.libpng.org/pub/png/", //png
        "http://www.libpng.org/pub/png/", //png
        "http://www.libpng.org/pub/png/", //png
        "http://www.libpng.org/pub/png/" //png
    };

    private static String[] allFileTypeOptions, allFileTypeNames;
    private static int defaultFileTypeOption = 0; //will be reset below
    
    //static constructor
    static {
        int nDFTN = dataFileTypeNames.length;
        int nIFTN = imageFileTypeNames.length;
        Test.ensureEqual(nDFTN, dataFileTypeDescriptions.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeDescriptions.length'.");                                     
        Test.ensureEqual(nDFTN, dataFileTypeExtensions.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeExtensions.length'.");                                     
        Test.ensureEqual(nDFTN, dataFileTypeInfo.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeInfo.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeDescriptions.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeDescriptions.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeExtensions.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeExtensions.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeInfo.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeInfo.length'.");                                     
        defaultFileTypeOption = String2.indexOf(dataFileTypeNames, ".htmlTable");

        //construct allFileTypeOptions
        allFileTypeOptions = new String[nDFTN + nIFTN];
        allFileTypeNames = new String[nDFTN + nIFTN];
        for (int i = 0; i < nDFTN; i++) {
            allFileTypeOptions[i] = dataFileTypeNames[i] + " - " + dataFileTypeDescriptions[i];
            allFileTypeNames[i] = dataFileTypeNames[i];
        }
        for (int i = 0; i < nIFTN; i++) {
            allFileTypeOptions[nDFTN + i] = imageFileTypeNames[i] + " - " + imageFileTypeDescriptions[i];
            allFileTypeNames[nDFTN + i] = imageFileTypeNames[i];
        }
    }

    //ensure org.jdom.Content is compiled -- 
    //GeotiffWriter needs it, but it isn't called directly so
    //it isn't automatically compiled.
    private static org.jdom.Content orgJdomContent;

    //*********** end of static declarations ***************************

    /** The constructor should set these to indicate where the 
     * lon,lat,alt,time variables are in axisVariables 
     * (or leave as -1 if not present).
     */
    protected int lonIndex = -1, latIndex = -1, altIndex = -1, timeIndex = -1;

    /** These are created as needed from axisVariables. */
    protected String[] axisVariableSourceNames, axisVariableDestinationNames;

    /**
     * This makes the searchString that searchRank searches.
     *
     * @return the searchString that searchRank searches.
     */
    public byte[] searchString() {
        if (searchString != null) 
            return searchString;

        //make a string to search through
        StringBuffer sb = new StringBuffer();
        sb.append("all\n");
        sb.append(title() + "\n");
        sb.append("datasetID=" + datasetID() + "\n");
        sb.append(dapProtocol() + "\n");
        if (accessibleViaWMS()) 
            sb.append("WMS\n");
        for (int dv = 0; dv < dataVariables.length; dv++) {
            sb.append(dataVariables[dv].destinationName() + "\n");
            if (!dataVariables[dv].sourceName().equalsIgnoreCase(dataVariables[dv].destinationName()))
                sb.append(dataVariables[dv].sourceName() + "\n");
            if (!dataVariables[dv].longName().equalsIgnoreCase(dataVariables[dv].destinationName()))
                sb.append(dataVariables[dv].longName() + "\n");
        }
        for (int av = 0; av < axisVariables.length; av++) {
            sb.append(axisVariables[av].destinationName() + "\n");
            if (!axisVariables[av].sourceName().equalsIgnoreCase(axisVariables[av].destinationName()))
                sb.append(axisVariables[av].sourceName() + "\n");
            if (!axisVariables[av].longName().equalsIgnoreCase(axisVariables[av].destinationName()))
                sb.append(axisVariables[av].longName() + "\n");
        }
        sb.append(combinedGlobalAttributes.toString() + "\n");
        for (int dv = 0; dv < dataVariables.length; dv++) sb.append(dataVariables[dv].combinedAttributes().toString() + "\n");
        for (int av = 0; av < axisVariables.length; av++) sb.append(axisVariables[av].combinedAttributes().toString() + "\n");
        sb.append("className=" + className + "\n");

        String2.replaceAll(sb, "\"", ""); //no double quotes (esp around attribute values)
        String2.replaceAll(sb, "\n    ", "\n"); //occurs for all attributes
        String tSearchString = sb.toString().toLowerCase();
        searchString = String2.getUTF8Bytes(tSearchString);
        return searchString;
    }

    /**
     * This returns the types of data files that this dataset can be returned as.
     * These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".nc". 
     *
     * @return the types of data files that this dataset can be returned as.
     */
    public String[] dataFileTypeNames() {return dataFileTypeNames; }

    /**
     * This returns the file extensions corresponding to the dataFileTypes.
     * E.g., dataFileTypeName=".htmlTable" returns dataFileTypeExtension=".html".
     *
     * @return the file extensions corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeExtensions() {return dataFileTypeExtensions; }

    /**
     * This returns descriptions (up to 80 characters long, suitable for a tooltip)
     * corresponding to the dataFileTypes. 
     *
     * @return descriptions corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeDescriptions() {return dataFileTypeDescriptions; }

    /**
     * This returns an info URL corresponding to the dataFileTypes. 
     *
     * @return an info URL corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeInfo() {return dataFileTypeInfo; }

    /**
     * This returns the types of image files that this dataset can be returned 
     * as. These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".largePng". 
     *
     * @return the types of image files that this dataset can be returned as.
     */
    public String[] imageFileTypeNames() {return imageFileTypeNames; }

    /**
     * This returns the file extensions corresponding to the imageFileTypes,
     * e.g., imageFileTypeNames=".largePng" returns imageFileTypeExtensions=".png".
     *
     * @return the file extensions corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeExtensions() {return imageFileTypeExtensions; }

    /**
     * This returns descriptions corresponding to the imageFileTypes 
     * (each is suitable for a tooltip).
     *
     * @return descriptions corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeDescriptions() {return imageFileTypeDescriptions; }

    /**
     * This returns an info URL corresponding to the imageFileTypes. 
     *
     * @return an info URL corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeInfo() {return imageFileTypeInfo; }
    
    /**
     * This returns the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     *
     * @return the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     */
    public String[] allFileTypeOptions() {return allFileTypeOptions; }
     
    /** 
     * This indicates if the dataset is accessible via WMS.
     * There used to be a lon +/-180 restriction, but no more.
     */
    public boolean accessibleViaWMS() {
        if (lonIndex < 0 || latIndex < 0)
            return false;
        EDVGridAxis lonVar = axisVariables[lonIndex];
        EDVGridAxis latVar = axisVariables[latIndex];
        if (lonVar.destinationMin() == lonVar.destinationMax() || //only 1 value
            latVar.destinationMin() == latVar.destinationMax() ||
            lonVar.destinationMin() >= 360 ||  //unlikely
            lonVar.destinationMax() <= -180)   //unlikely
            return false;
        //if (!lonVar.isEvenlySpaced() ||  //not necessary. map is drawn as appropriate.
        //    !latVar.isEvenlySpaced())
        //    return false;

        for (int dv = 0; dv < dataVariables.length; dv++)
            if (dataVariables[dv].hasColorBarMinMax())
                return true;
        return false;
    }

    /**
     * This returns the dapProtocol
     *
     * @return the dapProtocol
     */
    public String dapProtocol() {return dapProtocol; }

    /**
     * This returns the dapDescription 
     *
     * @return the dapDescription
     */
    public String dapDescription() {return EDStatic.EDDGridDapDescription; }

    /** It is useful to have a static field and a non-static method to access this. */
    public static String longDapDescriptionHtml =
        dapProtocol + " lets you request gridded data (for example, environmental data from satellites) and graphs of gridded data, via specially formed URLs.\n" +
        "<br>The URLs specify everything: the dataset, the desired file type for the response, and the subset of data that you want to receive.\n" +
        "<br>You can form these URLs by hand or with a computer program,\n" +
        "<br>or (much easier) you can use the dataset's \"Data Access Form\" (oriented to requests for data)\n" +
        "<br>or the dataset's \"Make A Graph\" form (oriented to requests for graphs and maps),\n" +
        "<br>both of which generate the URL when the form is submitted.\n" +
        "<br>" + dapProtocol + " is a superset of the \n" +
        "<a href=\"http://www.opendap.org\">OPeNDAP</a>\n " +
        "<a href=\"http://www.opendap.org/pdf/ESE-RFC-004v1.1.pdf\">DAP</a>\n" +
        "<a href=\"http://www.opendap.org/user/guide-html/guide_61.html#id5\">hyperslab protocol</a>.\n";

    /**
     * This returns the standard long HTML description of this DAP protocol.
     */
    public String longDapDescriptionHtml() {
        return longDapDescriptionHtml;
    }


    /**
     * This should be used by all subclass constructors to ensure that 
     * all of the items common to all EDDGrids are properly set.
     *
     * @throws Throwable if any required item isn't properly set
     */
    public void ensureValid() throws Throwable {
        super.ensureValid();
        String errorInMethod = "datasets.xml/EDDGrid.ensureValid error for " + datasetID() + ":\n ";

        for (int v = 0; v < axisVariables.length; v++) {
            Test.ensureTrue(axisVariables[v] != null, 
                errorInMethod + "axisVariable[" + v + "] is null.");
            String tErrorInMethod = errorInMethod + 
                "for axisVariable #" + v + "=" + axisVariables[v].destinationName() + ":\n";
            Test.ensureTrue(axisVariables[v] instanceof EDVGridAxis, 
                tErrorInMethod + "axisVariable[" + v + "] isn't an EDVGridAxis.");
            axisVariables[v].ensureValid(tErrorInMethod);
        }
        Test.ensureTrue(lonIndex < 0 || axisVariables[lonIndex] instanceof EDVLonGridAxis, 
            errorInMethod + "axisVariable[lonIndex=" + lonIndex + "] isn't an EDVLonGridAxis.");
        Test.ensureTrue(latIndex < 0 || axisVariables[latIndex] instanceof EDVLatGridAxis, 
            errorInMethod + "axisVariable[latIndex=" + latIndex + "] isn't an EDVLatGridAxis.");
        Test.ensureTrue(altIndex < 0 || axisVariables[altIndex] instanceof EDVAltGridAxis, 
            errorInMethod + "axisVariable[altIndex=" + altIndex + "] isn't an EDVAltGridAxis.");
        Test.ensureTrue(timeIndex < 0 || axisVariables[timeIndex] instanceof EDVTimeGridAxis, 
            errorInMethod + "axisVariable[timeIndex=" + timeIndex + "] isn't an EDVTimeGridAxis.");

    }

    /**
     * The string representation of this gridDataSet (for diagnostic purposes).
     *
     * @return the string representation of this gridDataSet.
     */
    public String toString() {  
        //make this JSON format?
        StringBuffer sb = new StringBuffer();
        sb.append("//** EDDGrid " + super.toString());
        for (int v = 0; v < axisVariables.length; v++)
            sb.append(axisVariables[v].toString());            
        sb.append(
            "\\**\n\n");
        return sb.toString();
    }

    
    /** 
     * This returns the axis or data variable which has the specified destination name.
     *
     * @return the specified axis or data variable destinationName
     * @throws Throwable if not found
     */
    public EDV findVariableByDestinationName(String tDestinationName) 
        throws Throwable {
        for (int v = 0; v < axisVariables.length; v++)
            if (axisVariables[v].destinationName().equals(tDestinationName))
                return (EDV)axisVariables[v];
        return (EDV)findDataVariableByDestinationName(tDestinationName);
    }

    /**
     * This returns the index of the lon axisVariable (or -1 if none).
     * @return the index of the lon axisVariable (or -1 if none).
     */
    public int lonIndex() {return lonIndex;}

    /**
     * This returns the index of the lat axisVariable (or -1 if none).
     * @return the index of the lat axisVariable (or -1 if none).
     */
    public int latIndex() {return latIndex;}

    /**
     * This returns the index of the altitude axisVariable (or -1 if none).
     * @return the index of the altitude axisVariable (or -1 if none).
     */
    public int altIndex() {return altIndex;}

    /**
     * This returns the index of the time axisVariable (or -1 if none).
     * @return the index of the time axisVariable (or -1 if none).
     */
    public int timeIndex() {return timeIndex;}

    
    /**
     * This returns the axisVariables.
     * This is the internal data structure, so don't change it.
     *
     * @return the axisVariables.
     */
    public EDVGridAxis[] axisVariables() {return axisVariables; }

    /** 
     * This returns the axis variable which has the specified source name.
     *
     * @return the specified axis variable sourceName
     * @throws Throwable if not found
     */
    public EDVGridAxis findAxisVariableBySourceName(String tSourceName) 
        throws Throwable {

        int which = String2.indexOf(axisVariableSourceNames(), tSourceName);
        if (which < 0) throw new SimpleException(
            "Error: source variable name='" + tSourceName + "' wasn't found.");
        return axisVariables[which];
    }

    /** 
     * This returns the axis variable which has the specified destination name.
     *
     * @return the specified axis variable destinationName
     * @throws Throwable if not found
     */
    public EDVGridAxis findAxisVariableByDestinationName(String tDestinationName) 
        throws Throwable {

        int which = String2.indexOf(axisVariableDestinationNames(), tDestinationName);
        if (which < 0) throw new SimpleException(
            "Error: destination variable name='" + tDestinationName + "' wasn't found.");
        return axisVariables[which];
    }

    /**
     * This returns a list of the axisVariables' source names.
     *
     * @return a list of the axisVariables' source names.
     *    This always returns the same internal array, so don't change it!
     */
    public String[] axisVariableSourceNames() {
        if (axisVariableSourceNames == null) {
            //do it this way to be a little more thread safe
            String tNames[] = new String[axisVariables.length];
            for (int i = 0; i < axisVariables.length; i++)
                tNames[i] = axisVariables[i].sourceName();
            axisVariableSourceNames = tNames;
        }
        return axisVariableSourceNames;
    }

    /**
     * This returns a list of the axisVariables' destination names.
     *
     * @return a list of the axisVariables' destination names.
     *    This always returns the same internal array, so don't change it!
     */
    public String[] axisVariableDestinationNames() {
        if (axisVariableDestinationNames == null) {
            //do it this way to be a little more thread safe
            String tNames[] = new String[axisVariables.length];
            for (int i = 0; i < axisVariables.length; i++)
                tNames[i] = axisVariables[i].destinationName();
            axisVariableDestinationNames = tNames;
        }
        return axisVariableDestinationNames;
    }

    /**
     * This indicates if userDapQuery is a request for one or more axis variables
     * (vs. a request for one or more data variables).
     * 
     * @param userDapQuery the part after the '?', still percentEncoded (may be null).
     */
    public boolean isAxisDapQuery(String userDapQuery) throws Throwable {
        if (userDapQuery == null) return false;

        //remove any &constraints; 
        String ampParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
        userDapQuery = ampParts[0];
        int qLength = userDapQuery.length();
        if (qLength == 0) return false;

        //basically, see if first thing is an axis destination name     
        int tPo = userDapQuery.indexOf('[');
        int po = tPo >= 0? tPo : qLength;
        tPo = userDapQuery.indexOf(',');
        if (tPo >= 0) po = Math.min(tPo, po);

        //or request uses gridName.axisName notation?
        String tName = userDapQuery.substring(0, po);
        int period = tName.indexOf('.');
        if (period > 0 && 
            String2.indexOf(dataVariableDestinationNames(), tName.substring(0, period)) >= 0)
            tName = tName.substring(period + 1);

        //is tName an axisName?
        int tAxis = String2.indexOf(axisVariableDestinationNames(), tName);
        return tAxis >= 0;
    }           

    /** 
     * This parses a OPeNDAP DAP-style grid-style query for grid data (not axis) variables, 
     *   e.g., var1,var2 or
     *   var1[start],var2[start] or
     *   var1[start:stop],var2[start:stop] or
     *   var1[start:stride:stop][start:stride:stop][].
     * <ul>
     * <li>An ERDDAP extension of the DAP standard: If within parentheses, 
     *   start and/or stop are assumed to be specified in destination units (not indices).
     * <li>If only two values are specified for a dimension (e.g., [a:b]),
     *   it is interpreted as [a:1:b].
     * <li>If only one value is specified for a dimension (e.g., [a]),
     *   it is interpreted as [a:1:a].
     * <li>If 0 values are specified for a dimension (e.g., []),
     *     it is interpreted as [0:1:max].
     * <li> Currently, if more than one variable is requested, all variables must
     *     have the same [] constraints.
     * <li> If userDapQuery is "", it is treated as a request for the entire dataset.
     * <li> The query may also have &amp; clauses at the end.
     *   Currently, they must all start with "." (for graphics commands).
     * </ul>
     *
     * @param userDapQuery the part of the user's request after the '?', still percentEncoded (shouldn't be null).
     * @param destinationNames will receive the list of requested destination variable names
     * @param constraints will receive the list of constraints,
     *    stored in axisVariables.length groups of 3 int's: 
     *    start0, stride0, stop0, start1, stride1, stop1, ...
     * @param repair if true, this method tries to do its best repair problems (guess at intent), 
     *     not to throw exceptions 
     * @throws Throwable if invalid query
     *     (0 resultsVariables is a valid query)
     */
    public void parseDataDapQuery(String userDapQuery, StringArray destinationNames,
        IntArray constraints, boolean repair) throws Throwable {

        destinationNames.clear();
        constraints.clear();
        if (reallyVerbose) String2.log("    EDDGrid.parseDataDapQuery: " + userDapQuery);

        //split userDapQuery at '&' and decode
        String ampParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")

        //ignore any &.cmd constraints
        for (int ap = 1; ap < ampParts.length; ap++) {
            if (!repair && !ampParts[ap].startsWith("."))
                throw new SimpleException("Query error: " +
                    "In a griddap query, '&' must be followed by a .graphicsCommand."); 
        }
        String query = ampParts[0]; //it has been percentDecoded

        //expand query="" into request for everything
        if (query.length() == 0) {
            if (reallyVerbose) String2.log("      query=\"\" is expanded to request entire dataset.");
            query = String2.toSVString(dataVariableDestinationNames(), ",", false);
        }

        //process queries with no [], just csv list of desired dataVariables
        if (query.indexOf('[') < 0) {
            for (int av = 0; av < axisVariables.length; av++) {
                constraints.add(0);
                constraints.add(1);
                constraints.add(axisVariables[av].sourceValues().size() - 1);
            }
            String destNames[] = String2.split(query, ',');
            for (int dv = 0; dv < destNames.length; dv++) {
                //if gridName.gridName notation, remove "gridName."
                //This isn't exactly correct: technically, the response shouldn't include the axis variables.
                String destName = destNames[dv];
                int period = destName.indexOf('.');
                if (period > 0) {
                    String shortName = destName.substring(0, period);
                    if (destName.equals(shortName + "." + shortName) &&
                        String2.indexOf(dataVariableDestinationNames(), shortName) >= 0)
                        destName = shortName;                        
                }

                //ensure destName is valid
                int tdi = String2.indexOf(dataVariableDestinationNames(), destName);
                if (tdi < 0) {
                    if (repair) destName = dataVariableDestinationNames()[0];
                    else {
                        if (String2.indexOf(axisVariableDestinationNames(), destName) >= 0)
                            throw new SimpleException("Query error: " + 
                                "A griddap data variable query can't include an axis variable (" + 
                                destName + ").");
                        findDataVariableByDestinationName(destName); //throws Throwable if trouble                
                    }
                }

                //ensure not duplicate destName
                tdi = destinationNames.indexOf(destName);
                if (tdi >= 0) {
                    if (!repair) 
                        throw new SimpleException("Query error: Variable name='" + destName + 
                            "' occurs twice.");
                } else {
                    destinationNames.add(destName);
                }
            }
            return;
        }


        //get the destinationNames
        int po = 0;
        while (po < query.length()) {
            //after first destinationName+constraints, "," should be next char
            if (po > 0) {
                if (query.charAt(po) != ',') {
                    if (repair) return; //this can only be trouble for second variable
                    else throw new SimpleException("Query error: ',' expected at position=" + po + ".");
                }
                po++;
            }

            //get the destinationName
            //find the '['               ??? Must I require "[" ???
            int leftPo = query.indexOf('[', po);
            if (leftPo < 0) {
                if (repair) return; //this can only be trouble for second variable
                else throw new SimpleException("Query error: '[' not found after position=" + po + ".");
            }
            String destinationName = query.substring(po, leftPo);

            //if gridName.gridName notation, remove "gridName."
            //This isn't exactly correct: technically, the response shouldn't include the axis variables.
            int period = destinationName.indexOf('.');
            if (period > 0) {
                String shortName = destinationName.substring(0, period); 
                if (destinationName.equals(shortName + "." + shortName) &&
                    String2.indexOf(dataVariableDestinationNames(), shortName) >= 0)
                    destinationName = shortName;
            }
            
            //ensure destinationName is valid
            if (reallyVerbose) String2.log("      destinationName=" + destinationName);
            int tdi = String2.indexOf(dataVariableDestinationNames(), destinationName);
            if (tdi < 0) {
                if (repair) destinationName = dataVariableDestinationNames()[0];
                else findDataVariableByDestinationName(destinationName); //throws Throwable if trouble
            }

            //ensure not duplicate destName
            tdi = destinationNames.indexOf(destinationName);
            if (tdi >= 0) {
                if (!repair) 
                    throw new SimpleException("Query error: variable name='" + destinationName + 
                        "' occurs twice.");
            } else {
                destinationNames.add(destinationName);
            }
            po = leftPo;

            //get the axis constraints
            for (int axis = 0; axis < axisVariables.length; axis++) {
                int sssp[] = parseAxisBrackets(query, destinationName, po, axis, repair);
                int startI  = sssp[0];
                int strideI = sssp[1];
                int stopI   = sssp[2];
                po          = sssp[3];

                if (destinationNames.size() == 1) {
                    //store convert sourceStart and sourceStop to indices
                    constraints.add(startI);
                    constraints.add(strideI);
                    constraints.add(stopI);
                    //if (reallyVerbose) String2.log("      axis=" + axis + 
                    //    " constraints: " + startI + " : " + strideI + " : " + stopI);
                } else {
                    //ensure start,stride,stop match first variable
                    if (startI  != constraints.get(axis * 3 + 0) ||
                        strideI != constraints.get(axis * 3 + 1) ||
                        stopI   != constraints.get(axis * 3 + 2)) {
                        if (!repair) throw new SimpleException("Query error: constraint(" +
                            startI + ":" + strideI + ":" + stopI + 
                            ") for variable=" + destinationName + " axis=" + axis + 
                            " is not identical to first variable's constraint(" +
                            constraints.get(axis * 3 + 0) + ":" + 
                            constraints.get(axis * 3 + 1) + ":" + 
                            constraints.get(axis * 3 + 2) + 
                            ").  If you need different subsets, make separate requests.");
                    }
                }
            }
        }        
    }

    /** 
     * This parses a OPeNDAP DAP-style grid-style query for one or more axis (not grid) variables, 
     *  (e.g., var1,var2, perhaps with [start:stride:stop] values).
     * !!!This should only be called if the userDapQuery starts with the name of 
     * an axis variable.
     * <ul>
     * <li>An ERDDAP extension of the DAP standard: If within parentheses, 
     *   start and/or stop are assumed to be specified in destination units (not indices).
     * <li>If only two values are specified for a dimension (e.g., [a:b]),
     *   it is interpreted as [a:1:b].
     * <li>If only one value is specified for a dimension (e.g., [a]),
     *   it is interpreted as [a:1:a].
     * <li>If 0 values are specified for a dimension (e.g., []),
     *     it is interpreted as [0:1:max].
     * <li> Currently, if more than one variable is requested, all variables must
     *     have the same [] constraints.
     * <li> If userDapQuery is varName, it is treated as a request for the entire variable.
     * </ul>
     *
     * @param userDapQuery the part of the user's request after the '?', still percentEncoded (shouldn't be null).
     * @param destinationNames will receive the list of requested destination axisVariable names
     * @param constraints will receive the list of constraints,
     *    stored as 3 int's (for for each destinationName): start, stride, stop.
     * @param repair if true, this method tries to do its best repair problems (guess at intent), 
     *     not to throw exceptions 
     * @throws Throwable if invalid query (and if !repair)     
     */
    public void parseAxisDapQuery(String userDapQuery, StringArray destinationNames,
        IntArray constraints, boolean repair) throws Throwable {

        destinationNames.clear();
        constraints.clear();
        if (reallyVerbose) String2.log("    EDDGrid.parseAxisDapQuery: " + userDapQuery);

        //split userDapQuery at '&' and decode
        String ampParts[] = getUserQueryParts(userDapQuery);  //always at least 1 part (may be "")

        //ensure not nothing (which is a data request)
        if (ampParts[0].length() == 0) 
            throw new SimpleException("Query error: " + 
                "Grid axis queries must specify at least one axis variable.");

        //ignore any &.cmd constraints
        for (int ap = 1; ap < ampParts.length; ap++)
            if (!repair && !ampParts[ap].startsWith("."))
                throw new SimpleException("Query error: " + 
                    "In a griddap query, '&' must be followed by a .graphicsCommand.");                
        userDapQuery = ampParts[0];

        //get the destinationNames
        int po = 0;
        int qLength = userDapQuery.length();
        while (po < qLength) {
            int commaPo = userDapQuery.indexOf(',', po);
            if (commaPo < 0)
                commaPo = qLength;
            int leftPo = userDapQuery.indexOf('[', po);
            boolean hasBrackets = leftPo >= 0 && leftPo < commaPo;

            //get the destinationName
            String destinationName = userDapQuery.substring(po, 
                hasBrackets? leftPo : commaPo);
            //if gridName.axisName notation, remove "gridName."
            int period = destinationName.indexOf('.');
            if (period > 0) {
                //ensure gridName is valid
                if (!repair && 
                    String2.indexOf(dataVariableDestinationNames(), destinationName.substring(0, period)) < 0)
                    throw new SimpleException("Query error: Unexpected data variable name=\"" + 
                        destinationName.substring(0, period) + "\".");
                destinationName = destinationName.substring(period + 1); 
            }
            
            //ensure destinationName is valid
            if (reallyVerbose) String2.log("      destinationName=" + destinationName);
            int axis = String2.indexOf(axisVariableDestinationNames(), destinationName);
            if (axis < 0) {
                if (repair) destinationName = axisVariableDestinationNames()[0];
                else {
                    if (String2.indexOf(dataVariableDestinationNames(), destinationName) >= 0)
                        throw new SimpleException("Query error: " + 
                            "A griddap axis variable query can't include a data variable (" + 
                            destinationName + ").");
                    findAxisVariableByDestinationName(destinationName); //throws Throwable if trouble
                }
            }

            //ensure not duplicate destName
            int tdi = destinationNames.indexOf(destinationName);
            if (tdi >= 0) {
                if (repair) return;
                else throw new SimpleException("Query error: variable name='" + 
                    destinationName + "' occurs twice.");
            } else {
                destinationNames.add(destinationName);
            }

            if (hasBrackets) {
                //get the axis constraints
                int sssp[] = parseAxisBrackets(userDapQuery, destinationName, leftPo, axis, repair);
                constraints.add(sssp[0]); //start
                constraints.add(sssp[1]); //stride
                constraints.add(sssp[2]); //stop
                po = sssp[3];
                if (po != commaPo && !repair)
                    throw new SimpleException("Query error: Unexpected character#" + (po + 1) + "='" + 
                        userDapQuery.charAt(po) + "'."); 
                //if (reallyVerbose) String2.log("      axis=" + axis + 
                //    " constraints: " + startI + " : " + strideI + " : " + stopI);
            } else {
                constraints.add(0); //start
                constraints.add(1); //stride
                constraints.add(axisVariables[axis].sourceValues().size() - 1); //stop
            }

            po = commaPo + 1;
        }        
    }

    /** 
     * Given a percentDecoded part of a userDapQuery, leftPo, and axis, this parses the contents of a [ ] 
     * in userDapQuery and returns the startI, strideI, stopI, and rightPo+1.
     *
     * @param deQuery a percentDecoded part of a userDapQuery
     * @param destinationName axis or grid variable name (for diagnostic purposes only)
     * @param leftPo the position of the "["
     * @param axis the axis number, 0..
     * @param repair if true, this tries to do its best not to throw an exception (guess at intent)
     * @return int[4], 0=startI, 1=strideI, 2=stopI, and 3=newPo (rightPo+1)
     * @throws Throwable if trouble
     */
    protected int[] parseAxisBrackets(String deQuery, String destinationName, 
        int leftPo, int axis, boolean repair) throws Throwable {

        EDVGridAxis av = axisVariables[axis];
        int nAvSourceValues = av.sourceValues().size();
        int precision = axis == timeIndex? 9 : 5;
        String diagnostic = "for variable=" + destinationName + ", axis=" + axis;
        if (reallyVerbose) String2.log("parseAxisBrackets " + diagnostic + ", leftPo=" + leftPo);
        int defaults[] = {0, 1, nAvSourceValues - 1, deQuery.length()};

        //leftPo must be '['
        int po = leftPo;
        if (po >= deQuery.length() || deQuery.charAt(leftPo) != '[') {
            if (repair) return defaults;
            else throw new SimpleException("Query error: '[' expected at position=" + 
                leftPo + " (" + diagnostic + ").");
        }

        //find the ']'    
        //It shouldn't occur within paren values, so a simple search is fine.
        int rightPo = deQuery.indexOf(']', leftPo + 1);
        if (rightPo < 0) {
            if (repair) return defaults;
            else throw new SimpleException("Query error: ']' not found after position=" + 
                leftPo + " (" + diagnostic + ").");
        }
        defaults[3] = rightPo;
        diagnostic += ", constraint=" +
            deQuery.substring(leftPo, rightPo + 1);
        po = rightPo + 1; //prepare for next axis constraint
    
        //is there anything between [ and ]?
        int startI, strideI, stopI;
        if (leftPo == rightPo - 1) {
            //[] -> 0:1:max
            startI = 0; //indices
            strideI = 1;
            stopI = nAvSourceValues - 1;
        } else {
            //find colon1
            int colon1 = -1;
            if (deQuery.charAt(leftPo + 1) == '(') {
                //seek closing )
                colon1 = deQuery.indexOf(')', leftPo + 2);
                if (colon1 >= rightPo) {
                    if (repair) return defaults;
                    else throw new SimpleException("Query error: " + 
                        "Close ')' not found after position=" + leftPo + 
                        " (" + diagnostic + ").");
                }
                colon1++;
            } else {
                //non-paren value
                colon1 = deQuery.indexOf(':', leftPo + 1);
            }
            if (colon1 < 0 || colon1 >= rightPo)
                //just one value inside []
                colon1 = -1;

            //find colon2
            int colon2 = -1;
            if (colon1 < 0) {
                //there is no colon1, so there is no colon2
            } else if (deQuery.charAt(colon1 + 1) == '(') {
                //seek closing "
                colon2 = deQuery.indexOf(')', colon1 + 2);
                if (colon2 >= rightPo) {
                    if (repair) return defaults;
                    else throw new SimpleException("Query error: " +
                        "close ')' not found after position=" + (colon2 + 2) + 
                        " (" + diagnostic + ").");
                } else {
                    //next char must be ']' or ':'
                    colon2++;
                    if (colon2 == rightPo) {
                        colon2 = -1;
                    } else if (deQuery.charAt(colon2) == ':') {
                        //colon2 is set correctly
                    } else { 
                        if (repair) return defaults;
                        else throw new SimpleException("Query error: " +
                            "':' expected at position=" + colon2 + 
                            " (" + diagnostic + ").");                    
                    }
                }
            } else {
                //non-paren value
                colon2 = deQuery.indexOf(':', colon1 + 1);
                if (colon2 > rightPo)
                    colon2 = -1;
            }
            //String2.log("      " + diagnostic + " colon1=" + colon1 + " colon2=" + colon2);

            //extract the string values
            String startS, stopS;
            if (colon1 < 0) {
                //[start]
                startS = deQuery.substring(leftPo + 1, rightPo);
                strideI = 1;
                stopS = startS; 
            } else if (colon2 < 0) {
                //[start:stop]
                startS = deQuery.substring(leftPo + 1, colon1);
                strideI = 1;
                stopS = deQuery.substring(colon1 + 1, rightPo);
            } else {
                //[start:stride:stop]
                startS = deQuery.substring(leftPo + 1, colon1);
                String strideS = deQuery.substring(colon1 + 1, colon2);
                strideI = String2.parseInt(strideS);
                stopS = deQuery.substring(colon2 + 1, rightPo);
                if (strideI < 1 || strideI == Integer.MAX_VALUE) {
                    if (repair) strideI = 1;
                    else throw new SimpleException("Query error: " + 
                        "Invalid stride=" + strideS + ".");
                }
            }
            startS = startS.trim();
            stopS = stopS.trim();
            //String2.log("      startS=" + startS + " strideI=" + strideI + " stopS=" + stopS);

            double sourceMin = av.sourceValues().getDouble(0);
            double sourceMax = av.sourceValues().getDouble(nAvSourceValues - 1);
            //if (startS.equals("last") || startS.equals("(last)")) {
            //    startI = av.sourceValues().size() - 1;
            //} else 
            if (startS.startsWith("last") || startS.startsWith("(last")) 
                startS = convertLast(av, "Start", startS);

            if (startS.startsWith("(")) {
                //convert paren startS
                startS = startS.substring(1, startS.length() - 1).trim(); //remove begin and end parens
                if (startS.length() == 0 && !repair) 
                    throw new SimpleException("Query error: " +
                        "A Start value inside \"()\" is missing.");
                double startDestD = av.destinationToDouble(startS);

                //since closest() below makes far out values valid, need to test validity
                if (repair && Double.isNaN(startDestD))
                    startDestD = av.destinationMin();

                if (Math2.greaterThanAE(precision, startDestD, av.destinationCoarseMin())) {
                } else {
                    if (repair) startDestD = av.firstDestinationValue();
                    else throw new SimpleException("Query error: " +
                        EDStatic.THERE_IS_NO_DATA +
                        " (in query " + diagnostic + ", the requestedStart=\"" + startDestD + 
                        "\" is less than " + av.destinationMin() + 
                        " (and even " + av.destinationCoarseMin() + "))");
                }

                if (Math2.lessThanAE(   precision, startDestD, av.destinationCoarseMax())) {
                } else {
                    if (repair) startDestD = av.lastDestinationValue();
                    else throw new SimpleException(EDStatic.THERE_IS_NO_DATA +
                        " (in query " + diagnostic + ", the requestedStart=\"" + startDestD + 
                        "\" is greater than " + av.destinationMax() + 
                        " (and even " + av.destinationCoarseMax() + "))");
                }

                startI = av.destinationToClosestSourceIndex(startDestD);
            } else {
                //it must be a >= 0 integer index
                if (!startS.matches("[0-9]+")) {
                    if (repair) startS = "0";
                    else throw new SimpleException("Query error: " + 
                        "Invalid requested axis start=\"" + startS + 
                        "\" isn't an integer >= 0 (" + diagnostic + ").");
                }

                startI = String2.parseInt(startS);
                if (startI < 0 || startI > nAvSourceValues - 1) {
                    if (repair) startI = 0;
                    else throw new SimpleException("Query error: " + 
                        "start=" + startS + " must be between 0 and " + 
                        (nAvSourceValues - 1) + " (" + diagnostic + ").");
                }
            }

            //if (startS.equals("last") || stopS.equals("(last)")) {
            //    stopI = av.sourceValues().size() - 1;
            //} else 
            if (stopS.startsWith("last") || stopS.startsWith("(last")) 
                stopS = convertLast(av, "Stop", stopS);

            if (stopS.startsWith("(")) {
                //convert paren stopS
                stopS = stopS.substring(1, stopS.length() - 1).trim(); //remove begin and end parens
                if (stopS.length() == 0 && !repair)
                    throw new SimpleException("Query error: " +
                        "A Stop value inside \"()\" is missing.");
                double stopDestD = av.destinationToDouble(stopS);

                //since closest() below makes far out values valid, need to test validity
                if (repair && Double.isNaN(stopDestD))
                    stopDestD = av.destinationMax();

                if (Math2.greaterThanAE(precision, stopDestD, av.destinationCoarseMin())) {
                } else {
                    if (repair) stopDestD = av.firstDestinationValue();
                    else throw new SimpleException(EDStatic.THERE_IS_NO_DATA +
                        " (in query " + diagnostic + ", the requestedStop=\"" + stopDestD + 
                        "\" is less than " + av.destinationMin() + 
                        " (and even " + av.destinationCoarseMin() + "))");
                }

                if (Math2.lessThanAE(   precision, stopDestD, av.destinationCoarseMax())) {
                } else {
                    if (repair) stopDestD = av.lastDestinationValue();
                    else throw new SimpleException(EDStatic.THERE_IS_NO_DATA +
                        " (in query " + diagnostic + ", the requestedStop=\"" + stopDestD + 
                        "\" is greater than " + av.destinationMax() + 
                        " (and even " + av.destinationCoarseMax() + "))");
                }

                stopI = av.destinationToClosestSourceIndex(stopDestD);
            } else {
                //it must be a >= 0 integer index
                stopS = stopS.trim();
                if (!stopS.matches("[0-9]+")) {
                    if (repair) stopS = "" + (nAvSourceValues - 1);
                    else throw new SimpleException("Query error: " + 
                        "Invalid requested axis stop=\"" + stopS + 
                        "\" in constraint isn't an integer >= 0 (" + 
                        diagnostic + ").");
                }
                stopI = String2.parseInt(stopS);
                if (stopI < 0 || stopI > nAvSourceValues - 1) {
                    if (repair) stopI = nAvSourceValues - 1;
                    else throw new SimpleException("Query error: " + 
                        "stop=" + stopS + 
                        " in constraint must be between 0 and " + 
                        (nAvSourceValues - 1) + " (" + diagnostic + ").");
                }
            }
        }

        //test for no data
        if (startI > stopI) {
            if (repair) {
                int ti = startI; startI = stopI; stopI = ti;
            } else throw new SimpleException("Query error: " +
                "requestStartIndex=" + startI + " is less than requestStopIndex=" + 
                stopI + " in constraint (" + diagnostic + ").");
        }

        //return
        return new int[]{startI, strideI, stopI, po};
    }

    /**
     * This converts a DAP Start or Stop value of "last[-n]" or "(last-x)"
     * into "index" or "(value)".
     * Without parentheses, n is an index number.
     * With parentheses, x is a numeric value
     * '+' is allowed instead of '-'.
     * Internal spaces are allowed.
     *
     * @param av an EDVGridAxis variable
     * @param name "Start" or "Stop"
     * @param ssValue the start or stop value
     * @return ssValue converted to "index" or a "(value)"
     * @throws Throwable if invalid format or n is too large
     */
    public static String convertLast(EDVGridAxis av, String name, String ssValue) throws Throwable {
        //remove parens
        String ossValue = ssValue;
        boolean hasParens = ssValue.startsWith("(");
        if (hasParens) {
            if (ssValue.endsWith(")"))
                ssValue = ssValue.substring(1, ssValue.length() - 1).trim();
            else
                throw new SimpleException("Query error: " +
                    name + "Value=" + ossValue + 
                    " starts with '(', but doesn't end with ')'.");
        }

        //remove "last"
        if (ssValue.startsWith("last"))
            ssValue = ssValue.substring(4).trim();
        else 
            throw new SimpleException("Query error: " +
                "'last' was expected at beginning of " + 
                name + "Value=" + ossValue + ".");

        //done?
        int lastIndex = av.sourceValues().size() - 1;
        if (ssValue.length() == 0) 
            return hasParens?  
                "(" + av.lastDestinationValue() + ")" : 
                "" + lastIndex;

        // +/-
        int pm = ssValue.startsWith("-")? -1 : 
                 ssValue.startsWith("+")? 1 : 0;
        if (pm == 0)
            throw new IllegalArgumentException ("Unexpected character after 'last' in " + 
                name + "Value=" + ossValue + ".");
        ssValue = ssValue.substring(1).trim();

        //parse the value
        if (hasParens) {
            double td = String2.parseDouble(ssValue);
            if (!Math2.isFinite(td))
                throw new IllegalArgumentException ("The +/-value in " + name + 
                    "Value=" + ossValue + " isn't valid.");               
            return "(" + (av.lastDestinationValue() + pm * td) + ")";
        } else {
            try {
                int ti = Integer.parseInt(ssValue); //be strict
                return "" + (lastIndex + pm * ti); 
            } catch (Throwable t) {
                throw new IllegalArgumentException ("The +/-index value in " + 
                    name + "Value=" + ossValue + " isn't an integer.");
            }
        }
    }

    /** 
     * This builds an OPeNDAP DAP-style grid-style query, 
     *   e.g., var1[start1:stop1][start2:stride2:stop2].
     * This is close to the opposite of parseDapQuery.
     *
     * @param destinationNames
     * @param constraints will receive the list of constraints,
     *    stored in axisVariables.length groups of 3 int's: 
     *    start0, stride0, stop0, start1, stride1, stop1, ...
     * @return the array part of an OPeNDAP DAP-style grid-style query, 
     *   e.g., [start1:stop1][start2:stride2:stop2].
     * @throws Throwable if invalid query
     *     (0 resultsVariables is a valid query)
     */
    public static String buildDapQuery(StringArray destinationNames, IntArray constraints) {
        String arrayQuery = buildDapArrayQuery(constraints);
        String names[] = destinationNames.toArray();        
        for (int i = 0; i < names.length; i++) 
            names[i] += arrayQuery;
        return String2.toSVString(names, ",", false);
    }

    /** 
     * This returns the array part of an OPeNDAP DAP-style grid-style query, 
     *   e.g., [start1:stop1][start2:stride2:stop2].
     * This is close to the opposite of parseDapQuery.
     *
     * @param constraints will receive the list of constraints,
     *    stored in axisVariables.length groups of 3 int's: 
     *    start0, stride0, stop0, start1, stride1, stop1, ...
     * @return the array part of an OPeNDAP DAP-style grid-style query, 
     *   e.g., [start1:stop1][start2:stride2:stop2].
     * @throws Throwable if invalid query
     *     (0 resultsVariables is a valid query)
     */
    public static String buildDapArrayQuery(IntArray constraints) {
        StringBuffer sb = new StringBuffer();
        int po = 0;
        while (po < constraints.size()) {
            int stride = constraints.get(po + 1);
            sb.append("[" + constraints.get(po) + ":" + 
                (stride == 1? "" : stride + ":") + 
                constraints.get(po +  2) + "]");
            po += 3;
        }
        return sb.toString();
    }

   /** 
     * This gets data (not yet standardized) from the data 
     * source for this EDDGrid.     
     * 
     * @param tDataVariables
     * @param tConstraints
     * @return a PrimitiveArray[] where the first axisVariables.length elements
     *   are the axisValues and the next tDataVariables.length elements
     *   are the dataValues.
     *   Both the axisValues and dataValues are straight from the source,
     *   not modified.
     * @throws Throwable if trouble
     */
    public abstract PrimitiveArray[] getSourceData(EDV tDataVariables[], IntArray tConstraints) 
        throws Throwable;

    /**
     * This makes a sibling dataset, based on the new sourceUrl.
     *
     * @param tSourceUrl
     * @param ensureAxisValuesAreEqual If Integer.MAX_VALUE, no axis sourceValue tests are performed. 
     *    If 0, this tests if sourceValues for axis-variable #0+ are same.
     *    If 1, this tests if sourceValues for axis-variable #1+ are same.
     *    (This is useful if the, for example, lat and lon values vary slightly and you 
     *    are willing to accept the initial values as the correct values.)
     *    Actually, the tests are always done but this determines whether
     *    the error is just logged or whether it throws an exception.
     * @param shareInfo if true, this ensures that the sibling's 
     *    axis and data variables are basically the same as this datasets,
     *    and then makes the new dataset point to the this instance's data structures
     *    to save memory. (AxisVariable #0 isn't duplicated.)
     * @return EDDGrid
     * @throws Throwable if trouble
     */
    public abstract EDDGrid sibling(String tSourceUrl, int ensureAxisValuesAreEqual,
        boolean shareInfo) throws Throwable;

    /**
     * This tests if the axisVariables and dataVariables of the other dataset are similar 
     *     (same destination data var names, same sourceDataType, same units, 
     *     same missing values).
     *
     * @param other   
     * @param ensureAxisValuesAreEqual If Integer.MAX_VALUE, no axis sourceValue 
     *    tests are performed. 
     *    If 0, this tests if sourceValues for axis-variable #0+ are same.
     *    If 1, this tests if sourceValues for axis-variable #1+ are same.
     *    (This is useful if the, for example, lat and lon values vary slightly and you 
     *    are willing to accept the initial values as the correct values.)
     *    Actually, the tests are always done but this determines whether
     *    the error is just logged or whether it throws an exception.
     * @param strict if !strict, this is less strict
     * @return "" if similar (same axis and data var names,
     *    same units, same sourceDataType, same missing values) 
     *    or a message if not (including if other is null).
     */
    public String similar(EDDGrid other, int ensureAxisValuesAreEqual, boolean strict) {
        try {
            if (other == null) 
                return "EDDGrid.similar: There is no 'other' dataset.  (Perhaps ERDDAP just restarted.)";
            if (reallyVerbose) String2.log("EDDGrid.similar ensureAxisValuesAreEqual=" + ensureAxisValuesAreEqual);
            String results = super.similar(other);
            if (results.length() > 0) 
                return results;

            return similarAxisVariables(other, ensureAxisValuesAreEqual, strict);
        } catch (Throwable t) {
            return t.toString();
        }
    }

    /**
     * This tests if 'old' is different from this in any way.
     * <br>This test is from the view of a subscriber who wants to know
     *    when a dataset has changed in any way.
     * <br>So some things like onChange and reloadEveryNMinutes are not checked.
     * <br>This only lists the first change found.
     *
     * <p>EDDGrid overwrites this to also check the axis variables.
     *
     * @param old
     * @return "" if same or message if not.
     */
    public String changed(EDD old) {
        if (old == null)
            return super.changed(old); //so message is consistent

        if (!(old instanceof EDDGrid)) 
            return "The new version is an EDDGrid.  The old version isn't!\n";

        EDDGrid oldG = (EDDGrid)old;

        //check most important things first
        int nAv = axisVariables.length;
        StringBuffer diff = new StringBuffer();
        diff.append(test2Changed("The number of axisVariables changed:",
            "" + oldG.axisVariables().length, 
            "" + nAv));
        if (diff.length() > 0) 
            return diff.toString(); //because tests below assume nAv are same

        for (int av = 0; av < nAv; av++) { 
            EDVGridAxis oldAV = oldG.axisVariables[av];
            EDVGridAxis newAV =      axisVariables[av];             
            String newName = newAV.destinationName();

            diff.append(test2Changed("The destinationName for axisVariable #" + av + " changed:",
                oldAV.destinationName(), newName));

            diff.append(test2Changed(
                "The destinationDataType for axisVariable #" + av + "=" + newName + " changed:",
                oldAV.destinationDataType(), 
                newAV.destinationDataType()));

            //most import case: new time value will be displayed as an iso time
            if (newAV.sourceValues().size() != oldAV.sourceValues().size()) 
                diff.append(
                "The number of axisVariable #" + av + "=" + newName + " values changed from " +
                oldAV.sourceValues().size() + " to " + newAV.sourceValues().size() + ".\n");
            int diffIndex = newAV.sourceValues().diffIndex(oldAV.sourceValues());
            if (diffIndex >= 0)
                diff.append(
                "The destinationValues for axisVariable #" + av + "=" + newName + " changed:" +
                "\n  old index #" + diffIndex + "=" + 
                    (diffIndex >= oldAV.sourceValues().size()? "(no value)" : 
                        oldAV.destinationToString(oldAV.destinationValue(diffIndex).getDouble(0))) +
                ",\n  new index #" + diffIndex + "=" + 
                    (diffIndex >= newAV.sourceValues().size()? "(no value)" : 
                        newAV.destinationToString(newAV.destinationValue(diffIndex).getDouble(0))) +
                ".\n");

            diff.append(test1Changed(
                "A combinedAttribute for axisVariable #" + av + "=" + newName + " changed:",
                String2.differentLine(
                    oldAV.combinedAttributes().toString(), 
                    newAV.combinedAttributes().toString())));
        }

        //check least important things last
        diff.append(super.changed(oldG));
        return diff.toString();
    }

    /**
     * This tests if the axisVariables of the other dataset are similar 
     *     (same destination data var names, same sourceDataType, same units, 
     *     same missing values).
     *
     * @param other 
     * @param ensureAxisValuesAreEqual If Integer.MAX_VALUE, no axis sourceValue 
     *        tests are performed. 
     *    If 0, this tests if sourceValues for axis-variable #0+ are same.
     *    If 1, this tests if sourceValues for axis-variable #1+ are same.
     *    (This is useful if the, for example, lat and lon values vary slightly and you 
     *    are willing to accept the initial values as the correct values.)
     *    Actually, the tests are always done but this determines whether
     *    the error is just logged or whether it throws an exception.
     * @param strict if !strict, this is less strict (including allowing different
     *    sourceDataTypes and destinationDataTypes)
     * @return "" if similar (same axis and data var names,
     *    same units, same sourceDataType, same missing values) or a message if not.
     */
    public String similarAxisVariables(EDDGrid other, int ensureAxisValuesAreEqual, 
            boolean strict) {
        if (reallyVerbose) String2.log("EDDGrid.similarAxisVariables ensureAxisValuesAreEqual=" + 
            ensureAxisValuesAreEqual);
        String msg = "EDDGrid.similar: The other dataset has a different ";
        int nAv = axisVariables.length;
        if (nAv != other.axisVariables.length)
            return msg + "number of axisVariables (" + 
                nAv + " != " + other.axisVariables.length + ")";

        for (int av = 0; av < nAv; av++) {
            EDVGridAxis av1 = axisVariables[av];
            EDVGridAxis av2 = other.axisVariables[av];

            //destinationName
            String s1 = av1.destinationName();
            String s2 = av2.destinationName();
            String msg2 = " for axisVariable #" + av + "=" + s1 + " (";
            if (!s1.equals(s2))
                return msg + "destinationName" + msg2 + s1 + " != " + s2 + ")";

            //sourceDataType 
            //if !strict, don't care e.g., if one is float and the other is double
            if (strict) {    
                s1 = av1.sourceDataType();
                s2 = av2.sourceDataType();
                if (!s1.equals(s2))
                    return msg + "sourceDataType" + msg2 +  s1 + " != " + s2 + ")";
            }

            //destinationDataType
            if (strict) {
                s1 = av1.destinationDataType();
                s2 = av2.destinationDataType();
                if (!s1.equals(s2))
                    return msg + "destinationDataType" + msg2 +  s1 + " != " + s2 + ")";
            }

            //units
            s1 = av1.units();
            s2 = av2.units();
            if (!s1.equals(s2))
                return msg + "units" + msg2 +  s1 + " != " + s2 + ")";

            //sourceMissingValue  (irrelevant, since shouldn't be any mv)
            double d1, d2;
            if (strict) {
                d1 = av1.sourceMissingValue();
                d2 = av2.sourceMissingValue();
                if (!Test.equal(d1, d2)) //says NaN==NaN is true
                    return msg + "sourceMissingValue" + msg2 +  d1 + " != " + d2 + ")";
            }

            //sourceFillValue  (irrelevant, since shouldn't be any mv)
            if (strict) {
                d1 = av1.sourceFillValue();
                d2 = av2.sourceFillValue();
                if (!Test.equal(d1, d2)) //says NaN==NaN is true
                    return msg + "sourceFillValue" + msg2 +  d1 + " != " + d2 + ")";
            }

            //test sourceValues  
            String results = av1.sourceValues().almostEqual(av2.sourceValues());
            if (results.length() > 0) {
                results = msg + "sourceValue" + msg2 + results + ")";
                if (av >= ensureAxisValuesAreEqual) 
                    return results; 
                else String2.log("WARNING: " + results);
            }
        }
        //they are similar
        return "";
    }



    /**
     * This responds to a DAP-style query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery the part of the user's request after the '?', still percentEncoded (shouldn't be null).
     * @param outputStreamSource  the source of an outputStream that receives the results,
     *    usually already buffered.
     * @param dir the directory (on this computer's hard drive) to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName the fileTypeName for the new file (e.g., .largePng).
     * @throws Throwable if trouble
     */
    public void respondToDapQuery(HttpServletRequest request, String loggedInAs,
        String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //save data to outputStream
        if (fileTypeName.equals(".asc")) {
            saveAsAsc(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".csv")) {
            saveAsCsv(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".das")) {
            saveAsDAS(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".dds")) {
            saveAsDDS(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".dods")) {
            saveAsDODS(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".esriAscii")) {
            saveAsEsriAscii(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".graph")) {
            respondToGraphQuery(request, loggedInAs, requestUrl, userDapQuery, outputStreamSource,
                dir, fileName, fileTypeName);
            return;
        }

        if (fileTypeName.equals(".html")) {
            //it is important that this use outputStreamSource so stream is compressed (if possible)
            //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible unicode
            OutputStream out = outputStreamSource.outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8"); 
            writer.write(EDStatic.startHeadHtml(tErddapUrl, "Data Access Form for " + 
                XML.encodeAsXML(title() + " from " + institution())));
            writer.write(EDStatic.standardHead);
            writer.write("\n" + rssHeadLink());
            writer.write("\n</head>\n");
            writer.write(EDStatic.startBodyHtml(loggedInAs));
            writer.write("\n");
            writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl)); //this is a link to a script
            writer.write(HtmlWidgets.dragDropScript(EDStatic.imageDirUrl));    //this is a link to a script
            writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
            try {
                writer.write(EDStatic.youAreHereWithHelp(tErddapUrl, dapProtocol, "Data Access Form", 
                    EDStatic.EDDGridDataAccessFormHtml + "<p>" + EDStatic.EDDGridDownloadDataHtml +
                    "</ol>\n" +
                    "This web page just simplifies the creation of griddap URLs. " +
                    "<br>If you want, you can create these URLs by hand or have a computer program do it." +
                    "<br>See the 'Just generate the URL' button below, or see the griddap documentation."));
                writeHtmlDatasetInfo(loggedInAs, writer, false, true, userDapQuery, "");
                writeDapHtmlForm(loggedInAs, userDapQuery, writer);
                writer.write("<hr noshade>\n");
                writer.write("<h2>The Dataset Attribute Structure (.das) for this Dataset</h2>\n" +
                    "<pre>\n");
                writeDAS(File2.forceExtension(requestUrl, ".das"), "", writer, true); //useful so search engines find all relevant words
                writer.write("</pre>\n");
                writer.write("<br>&nbsp;\n");
                writer.write("<hr noshade>\n");
                writeGeneralDapHtmlInstructions(tErddapUrl, writer, false); 
                writer.write("<br>&nbsp;\n" +
                    "<br><small>" + EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small>\n");
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            if (EDStatic.displayDiagnosticInfo) 
                EDStatic.writeDiagnosticInfoHtml(writer);
            writer.write(EDStatic.endBodyHtml(tErddapUrl));
            writer.write("\n</html>\n");
            writer.flush(); //essential
            return;
        }

        if (fileTypeName.equals(".htmlTable")) {
            saveAsHtmlTable(requestUrl, userDapQuery, outputStreamSource, fileName, 
                false, "", ""); 
            return;
        }

        if (fileTypeName.equals(".json")) {
            saveAsJson(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".mat")) {
            saveAsMatlab(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".tsv")) {
            saveAsTsv(requestUrl, userDapQuery, outputStreamSource);
            return;
        }

        if (fileTypeName.equals(".xhtml")) {
            saveAsHtmlTable(requestUrl, userDapQuery, outputStreamSource, fileName, true, "", ""); 
            return;
        }

        //*** make a file (then copy it to outputStream)
        //nc files are handled this way because .ncHeader needs to call
        //  NcHelper.dumpString(aRealFile, false). 
        String fileTypeExtension = fileTypeExtension(fileTypeName);
        String fullName = dir + fileName + fileTypeExtension;

        //does the file already exist?
        //if .ncHeader, make sure the .nc file exists (and it is the better file to cache)
        String cacheFullName = fileTypeName.equals(".ncHeader")?  //the only exception there will ever be
            dir + fileName + ".nc" : fullName;
        int random = Math2.random(Integer.MAX_VALUE);

        if (File2.isFile(cacheFullName)) { //don't 'touch()'; files for latest data will change
            if (verbose) String2.log("  reusing cached " + cacheFullName);

        } else if (fileTypeName.equals(".nc") || fileTypeName.equals(".ncHeader")) {
            //if .ncHeader, make sure the .nc file exists (and it is the better file to cache)
            saveAsNc(requestUrl, userDapQuery, dir + fileName + ".nc", true, 0); //it saves to temp random file first

        } else {
            //all other file types
            //create random file; and if error, only random file will be created
            FileOutputStream fos = new FileOutputStream(cacheFullName + random); 
            OutputStreamSourceSimple osss = new OutputStreamSourceSimple(fos);
            boolean ok;
            
            if (fileTypeName.equals(".geotif")) {
                ok = saveAsGeotiff(requestUrl, userDapQuery, osss, dir, fileName);

            } else if (fileTypeName.equals(".kml")) {
                ok = saveAsKml(loggedInAs, requestUrl, userDapQuery, osss);

            } else if (String2.indexOf(imageFileTypeNames, fileTypeName) >= 0) {
                //do pdf and png LAST, so kml caught above
                ok = saveAsImage(requestUrl, userDapQuery, osss, fileTypeName);

            } else {
                fos.close();
                File2.delete(cacheFullName + random);
                throw new SimpleException("Error: " +
                    "fileType=" + fileTypeName + " isn't supported by this dataset.");
            }

            fos.close();
            File2.renameIfNewDoesntExist(cacheFullName + random, cacheFullName); //make available in an instant
            if (!ok) //make eligible to be removed from cache in 5 minutes
                File2.touch(cacheFullName, 
                    Math.max(0, EDStatic.cacheMillis - 5 * Calendar2.MILLIS_PER_MINUTE));
        
        }

        //then handle .ncHeader
        if (fileTypeName.equals(".ncHeader")) {
            String error = String2.writeToFile(fullName, 
                NcHelper.dumpString(cacheFullName, false)); 
            if (error.length() != 0)
                throw new RuntimeException(error);
        }

        //copy file to outputStream
        //(I delayed getting actual outputStream as long as possible.)
        if (!File2.copy(fullName, outputStreamSource.outputStream(
            fileTypeName.equals(".ncHeader")? "UTF-8" : 
            fileTypeName.equals(".kml")? "UTF-8" : 
            ""))) {
            //outputStream contentType already set,
            //so I can't go back to html and display error message
            //note than the message is thrown if user cancels the transmission; so don't email to me
            String2.log("Error while transmitting " + fileName + fileTypeExtension);
        }



    }

    /**
     * This deals with requests for a Make A Graph (MakeAGraph) web page for this dataset. 
     *
     * @param request may be null. It is just used to determine if user is logged in.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery from the user (may be "" or null), still percentEncoded (shouldn't be null).
     *    If the query has missing or invalid parameters, defaults will be used.
     *    If the query has irrelevant parameters, they will be ignored.
     * @param outputStreamSource  the source of an outputStream that receives the results,
     *    usually already buffered.
     * @param dir the directory to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName must be .graph
     * @throws Throwable if trouble
     */
    public void respondToGraphQuery(HttpServletRequest request, String loggedInAs,
        String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable {

        if (reallyVerbose)
            String2.log("*** respondToGraphQuery");

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        String formName = "f1"; //change JavaScript below if this changes
        OutputStream out = outputStreamSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out, "UTF-8"); 
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl);

        //write the header
        writer.write(EDStatic.startHeadHtml(tErddapUrl, "Make A Graph for " + 
            XML.encodeAsXML(title() + " from " + institution())));
        writer.write(EDStatic.standardHead);
        writer.write("\n" + rssHeadLink());
        writer.write("\n</head>\n");
        writer.write(EDStatic.startBodyHtml(loggedInAs));
        writer.write("\n");
        writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl)); //this is a link to a script
        writer.write(HtmlWidgets.dragDropScript(EDStatic.imageDirUrl));    //this is a link to a script
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
        try {
            writer.write(EDStatic.youAreHereWithHelp(tErddapUrl, "griddap", 
                "Make a Graph", 
                "<b>To make a graph of data from this grid dataset, repeatedly:</b><ol>" +
                "<li>Change the 'Graph Type' and the variables for the graph's axes." +
                "<li>Change the 'Dimension Ranges' to specify a subset of the data." +
                "<li>Change the 'Graph Settings' as desired." +
                "<li>Press 'Redraw the Graph'." +
                "</ol>" +
                "This Make A Graph web page just simplifies the creation of griddap URLs with graphics commands. " +
                "<br>If you want, you can create these URLs by hand or have a computer program do it." +
                "<br>See the 'view the URL' textfield below, or see 'Make A Graph' in the griddap documentation."));
            writeHtmlDatasetInfo(loggedInAs, writer, true, false, userDapQuery, "");

            //make the big table
            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, ""));  //the big table
            writer.write("<tr><td align=\"left\" valign=\"top\">\n"); 

            //begin the form
            writer.write(widgets.beginForm(formName, "GET", "", ""));
         
            //parse the query so &-separated parts are handy
            String paramName, paramValue, partName, partValue, pParts[];
            String queryParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")

            //find the axisVariables (all are always numeric) with >1 value
            StringArray sa = new StringArray();
            for (int av = 0; av < axisVariables.length; av++) {
                if (axisVariables[av].sourceValues().size() > 1) 
                    sa.add(axisVariableDestinationNames()[av]);
            }
            if (sa.size() == 0)
                throw new SimpleException("Query error: " +
                    "Make A Graph requires at least one axis with more than one value.");
            String[] avNames = sa.toArray();

            //find the numeric dataVariables 
            sa = new StringArray();
            for (int dv = 0; dv < dataVariables.length; dv++) {
                if (dataVariables[dv].destinationDataTypeClass() != String.class) 
                    sa.add(dataVariables[dv].destinationName());
            }
            if (sa.size() == 0)
                throw new SimpleException("Query error: " +
                    "Make A Graph requires at least one numeric data variable.");
            String[] dvNames = sa.toArray();
            sa.add(0, "");
            String[] dvNames0 = sa.toArray();
            sa.remove(0);
            sa = null;
            //if you need advNames, you will need to modify the javascript below

            //parse the query to get preferredDV0 and constraints
            StringArray tDestNames = new StringArray();
            IntArray tConstraints = new IntArray();
            parseDataDapQuery(userDapQuery, tDestNames, tConstraints, true);
            String preferredDV0 = tDestNames.size() > 0? tDestNames.get(0) : dvNames[0];
            if (reallyVerbose) String2.log("preferredDV0=" + preferredDV0); 

            String gap = "&nbsp;&nbsp;&nbsp;";

            //*** set the Graph Type
            StringArray drawsSA = new StringArray();        
            //it is important for javascript below that first 3 options are the very similar (L L&M M)
            drawsSA.add("lines");  
            drawsSA.add("linesAndMarkers");   int defaultDraw = 1; 
            drawsSA.add("markers");  
            if (axisVariables.length >= 1 && dataVariables.length >= 2) 
                drawsSA.add("sticks");
            if (lonIndex >= 0 && latIndex >= 0) {//currently on if x=lon and y=lat 
                defaultDraw = drawsSA.size();
                drawsSA.add("surface");
            }
            if (lonIndex >= 0 && latIndex >= 0 && dataVariables.length >= 2) 
                drawsSA.add("vectors");
            String draws[] = drawsSA.toArray();
            boolean preferDefaultVars = true;
            int draw = defaultDraw;
            partValue = String2.stringStartsWith(queryParts, partName = ".draw=");
            if (partValue != null) {
                draw = String2.indexOf(draws, partValue.substring(partName.length()));
                if (draw >= 0) { // valid .draw was specified
                    preferDefaultVars = false;
                    //but check that it is possible
                    boolean trouble = false;
                    if ((draws[draw].equals("surface") || draws[draw].equals("vectors")) &&
                        (lonIndex < 0 || latIndex < 0)) 
                        trouble = true;
                    if ((draws[draw].equals("sticks") || draws[draw].equals("vectors")) && 
                        dvNames.length < 2)
                        trouble = true;
                    if (trouble) {
                        preferDefaultVars = true;
                        draw = String2.indexOf(draws, "linesAndMarkers"); //safest
                    }
                } else {
                    preferDefaultVars = true;
                    draw = defaultDraw;
                }
            }
            boolean drawLines = draws[draw].equals("lines");
            boolean drawLinesAndMarkers = draws[draw].equals("linesAndMarkers");
            boolean drawMarkers = draws[draw].equals("markers");
            boolean drawSticks  = draws[draw].equals("sticks");
            boolean drawSurface = draws[draw].equals("surface");
            boolean drawVectors = draws[draw].equals("vectors");
            if (reallyVerbose) String2.log("draw=" + draws[draw] + " preferDefaultVars=" + preferDefaultVars);
            writer.write(widgets.beginTable(0, 0, "")); //the Graph Type and vars table
            paramName = "draw";
            writer.write(
                "<tr>\n" +
                "  <td nowrap><b>Graph Type:&nbsp;</b>" + 
                "  </td>\n" +
                "  <td>\n");
            writer.write(widgets.select(paramName, "", 
                1, draws, draw, 
                "onChange='mySubmit(" +  //change->submit so form always reflects graph type
                (draw < 3? "f1.draw.selectedIndex<3" : //if old and new draw are <3, do send var names
                    "false") + //else don't send var names
                ");'")); 
            writer.write(
                EDStatic.htmlTooltipImage(
                    "<b>Graph Type</b>" +
                    "<br>Graph Type = 'lines' draws lines on a graph where X=a dimension" +
                    "<br>and Y=a data variable. " +

                    "<p>Graph Type = 'linesAndMarkers' draws lines and markers on a graph " +
                    "<br>where X=a dimension and Y=a data variable. " +
                    "<br>If a Color variable is specified, the markers are colored." +

                    "<p>Graph Type = 'markers' plots markers on a graph where X=a dimension" +
                    "<br>and Y=a data variable. " +
                    "<br>If a Color variable is specified, the markers are colored." +

                    "<p>Graph Type = 'sticks' is usually used to plot time on the x axis," +
                    "<br>with the sticks being draw from the x component and y component" +
                    "<br>of currents or wind data." +

                    "<p>Graph Type = 'surface' plots a longitude/latitude grid of data as a" +
                    "<br>colored surface on a map." +
                    "<br>Currently, 'surface' requires that X=longitude and Y=latitude." +

                    "<p>Graph Type = 'vectors' plots vectors on a map." +
                    "<br>Currently, this requires that X=longitude and Y=latitude." +
                    "<br>The other variables provide the vector's x component and y component." +
                    "<br>So it is often used for currents or wind data." +

                    "<p><b>Changing the Graph Type automatically submits this form.</b>" +
                    "<br>For the remainder of this form:" +
                    "<br>make changes, then press 'Redraw the Graph' below.") + 
                "  </td>\n" +
                "</tr>\n");
            //if (debugMode) String2.log("respondToGraphQuery 3");

            //find default stick or vector data vars  (adjacent, starting at dv=sameUnits1)
            //heuristic: look for two adjacent dv that have same units
            int sameUnits1 = 0; //default  dv
            String units1 = null;
            String units2 = findDataVariableByDestinationName(dvNames[0]).units(); 
            for (int sameUnits2 = 1; sameUnits2 < dvNames.length; sameUnits2++) {
                units1 = units2;
                units2 = findDataVariableByDestinationName(dvNames[sameUnits2]).units(); 
                if (units1 != null && units2 != null && units1.equals(units2)) {
                    sameUnits1 = sameUnits2 - 1;
                    break;
                }
            }

            //set draw-related things
            int nVars = -1, dvPo = 0;
            String varLabel[], varHelp[], varOptions[][];
            String varName[] = {"", "", "", ""};  //fill with defaults below, then from .vars      
            if (drawLines) {
                nVars = 2;
                varLabel = new String[]{"X Axis:", "Y Axis:"};
                varHelp  = new String[]{"graph's X Axis.", "graph's X Axis."};
                varOptions = new String[][]{avNames, dvNames};
                varName[0] = timeIndex >= 0? EDV.TIME_NAME : avNames[0];
                varName[1] = preferredDV0;
            } else if (drawLinesAndMarkers || drawMarkers) {
                nVars = 3;
                varLabel = new String[]{"X Axis:", "Y Axis:", "Color:"};
                varHelp  = new String[]{"graph's X Axis.", "graph's X Axis.", 
                    "marker's color via the Color Bar (or leave blank)."};
                varOptions = new String[][]{avNames, dvNames, dvNames0};
                varName[0] = timeIndex >= 0? EDV.TIME_NAME : avNames[0];
                varName[1] = preferredDV0;
            } else if (drawSticks) {
                nVars = 3;
                varLabel = new String[]{"X Axis:", "Stick X:", "Stick Y:"};
                varHelp  = new String[]{"graph's X Axis.", "stick's x-component.", "stick's y-component."};
                varOptions = new String[][]{avNames, dvNames, dvNames};
                varName[0] = timeIndex >= 0? EDV.TIME_NAME : avNames[0];
                varName[1] = dvNames[sameUnits1];
                varName[2] = dvNames[sameUnits1 + 1];
            } else if (drawSurface) {
                nVars = 3;
                varLabel = new String[]{"X Axis:", "Y Axis:", "Color:"};
                varHelp  = new String[]{"map's X Axis.", "map's Y Axis.", "surface's color via the Color Bar."};
                varOptions = new String[][]{new String[]{EDV.LON_NAME}, new String[]{EDV.LAT_NAME}, dvNames};
                varName[0] = EDV.LON_NAME;
                varName[1] = EDV.LAT_NAME;
                varName[2] = preferredDV0;
            } else if (drawVectors) {
                nVars = 4;
                varLabel = new String[]{"X Axis:", "Y Axis:", "Vector X:", "Vector Y:"};
                varHelp  = new String[]{"map's X Axis.", "map's Y Axis.", "vector's x-component.", "vector's y-component."};
                varOptions = new String[][]{new String[]{EDV.LON_NAME}, new String[]{EDV.LAT_NAME}, dvNames, dvNames};
                varName[0] = EDV.LON_NAME;
                varName[1] = EDV.LAT_NAME;
                varName[2] = dvNames[sameUnits1];
                varName[3] = dvNames[sameUnits1 + 1];
            } else throw new SimpleException("Query error: " +
                "'draw' was not set.");
            //if (debugMode) String2.log("respondToGraphQuery 4");

            //avoid lat lon reversed (which sgtMap will reverse)
            if (varName[0].equals("latitude") &&
                varName[1].equals("longitude")) {
                varName[0] = "longitude";
                varName[1] = "latitude";
            }

            //pick variables
            partValue = String2.stringStartsWith(queryParts, partName = ".vars=");
            if (partValue == null)
                 pParts = new String[0];
            else pParts = String2.split(partValue.substring(partName.length()), '|');
            for (int v = 0; v < nVars; v++) {
                String tDvNames[] = varOptions[v];
                int vi = -1;
                if (!preferDefaultVars && pParts.length > v) 
                    vi = String2.indexOf(tDvNames, pParts[v]);
                if (vi < 0) //use default
                    vi = String2.indexOf(tDvNames, varName[v]); 
                //avoid duplicate with previous var 
                //(there are never more than 2 axis or 2 data vars in a row)
                if (v >= 1 && varName[v-1].equals(tDvNames[vi]))
                    vi = vi == 0? 1 : 0; 
                varName[v] = tDvNames[vi];
                paramName = "var" + v;
                writer.write("<tr>\n" +
                    "  <td nowrap>" + varLabel[v] + "&nbsp;" +
                    "  </td>\n" +
                    "  <td>\n");
                writer.write(widgets.select(paramName, 
                    "Select the variable for the " + varHelp[v], 
                    1, tDvNames, vi, ""));
                writer.write(
                    "  </td>\n" +
                    "</tr>\n");
            }
            //find axisVar index (or -1 if not an axis var), not index in avNames
            int axisVarX = String2.indexOf(axisVariableDestinationNames(), varName[0]); 
            int axisVarY = String2.indexOf(axisVariableDestinationNames(), varName[1]);
            if (reallyVerbose) String2.log("varName[]=" + String2.toCSVString(varName));

            //end the Graph Type and vars table
            writer.write(widgets.endTable()); 

            //*** write the Dimension Constraints table
            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, "width=\"50%\"")); 
            String tRangeStartStopHtml = 
                "<tt>Start:Stop</tt> specify the subset of data that will be plotted on the graph.";
            writer.write(
                "<tr>\n" +
                "  <th nowrap align=\"left\">Dimension Ranges " + 
                    EDStatic.htmlTooltipImage(tRangeStartStopHtml) + 
                    "</th>\n" +
                "  <th nowrap align=\"center\">" + gap + EDStatic.EDDGridStart + " " + 
                    EDStatic.htmlTooltipImage(tRangeStartStopHtml + "<br>" + EDStatic.EDDGridStartHtml) + 
                    "</th>\n" +
                "  <th nowrap align=\"center\">" + gap + EDStatic.EDDGridStop + " " + 
                    EDStatic.htmlTooltipImage(tRangeStartStopHtml + "<br>" + EDStatic.EDDGridStopHtml) + 
                    "</th>\n" +
                "</tr>\n");

            //set dimensions' start and stop
            int nAv = axisVariables.length;
            String avStart[]  = new String[nAv];
            String avStop[]  = new String[nAv];
            StringBuffer constraints = new StringBuffer();
            String sliderFromNames[] = new String[nAv];
            String sliderToNames[] = new String[nAv];
            int sliderNThumbs[] = new int[nAv];
            String sliderUserValuesCsvs[] = new String[nAv];
            int sliderInitFromPositions[] = new int[nAv];
            int sliderInitToPositions[] = new int[nAv];
            for (int av = 0; av < nAv; av++) {
                EDVGridAxis edvga = axisVariables[av];
                EDVTimeGridAxis edvtga = av == timeIndex? (EDVTimeGridAxis)edvga : null;
                double defStart = av == timeIndex?  //note max vs first
                    Math.max(edvga.destinationMax() - 7 * Calendar2.SECONDS_PER_DAY, edvga.destinationMin()) :
                    edvga.firstDestinationValue();
                double defStop = av == timeIndex?
                    edvga.destinationMax():
                    edvga.lastDestinationValue();
                int sourceSize = edvga.sourceValues().size();
                String tDim = "[" + sourceSize + "]";
                String tUnits = av == timeIndex? "UTC" : edvga.units();
                int precision = av == timeIndex? 10 : 7;
                tUnits = tUnits == null? "" : "(" + tUnits + ") ";
                writer.write("<tr>\n" +
                    "  <td nowrap>" + edvga.destinationName() + " " +
                    tUnits +
                    EDStatic.htmlTooltipImage(edvga.destinationDataTypeClass(), 
                        edvga.destinationName() + tDim, edvga.combinedAttributes()) +           
                    "</td>\n");

                //find start and end
                double dStart = userDapQuery.length() == 0? defStart :
                    edvga.destinationValue(tConstraints.get(av * 3 + 0)).getNiceDouble(0);
                double dStop = userDapQuery.length() == 0? defStop :
                    edvga.destinationValue(tConstraints.get(av * 3 + 2)).getNiceDouble(0);

                //compare dStart and dStop to ensure valid
                boolean showStartAndStopFields = av == axisVarX || av == axisVarY;
                if (edvga.averageSpacing() > 0) { //looser test than isAscending
                    //ascending axis values
                    if (showStartAndStopFields) {
                        if (Math2.greaterThanAE(precision, dStart, dStop)) 
                            dStart = defStart; 
                        if (Math2.greaterThanAE(precision, dStart, dStop)) 
                            dStop = defStop; 
                    } else {
                        if (!Math2.lessThanAE(precision, dStart, dStop)) 
                            dStart = edvga.firstDestinationValue(); //not defStart; stop field is always visible, so change start
                        if (!Math2.lessThanAE(precision, dStart, dStop)) 
                            dStop = edvga.lastDestinationValue(); 
                    }
                } else {
                    //descending axis values
                    if (showStartAndStopFields) {
                        if (Math2.greaterThanAE(precision, dStop, dStart)) //stop start reversed from above 
                            dStart = defStart; 
                        if (Math2.greaterThanAE(precision, dStop, dStart)) 
                            dStop = defStop; 
                    } else {
                        if (!Math2.lessThanAE(precision, dStop, dStart)) 
                            dStart = edvga.firstDestinationValue(); //not defStart; stop field is always visible, so change start
                        if (!Math2.lessThanAE(precision, dStop, dStart)) 
                            dStop = edvga.lastDestinationValue(); 
                    }
                }
                
                //format
                avStart[av] = edvga.destinationToString(dStart);
                avStop[av]  = edvga.destinationToString(dStop);
                int startIndex = edvga.destinationToClosestSourceIndex(dStart);
                int stopIndex  = edvga.destinationToClosestSourceIndex(dStop);

                String tFirst = edvga.destinationToString(edvga.firstDestinationValue());
                String tLast  = edvga.destinationToString(edvga.lastDestinationValue());
                String edvgaTooltip = edvga.htmlRangeTooltip();
                for (int ss = 0; ss < 2; ss++) { //0=start, 1=stop
                    paramName = (ss == 0? "start" : "stop") + av;
                    int tIndex = ss == 0? startIndex : stopIndex;
                    writer.write("<td nowrap>"); //a cell in the dimensions table

                    if (ss == 1 || showStartAndStopFields) {
                        //show start or stop field, in a table with buttons

                        //generate the buttons
                        int fieldSize = 24;
                        StringBuffer buttons = new StringBuffer();

                        //show arrowLL?
                        if (tIndex >= 1 &&
                            (ss == 0 || 
                             (ss == 1 && !showStartAndStopFields))) {
                            buttons.append(
                                "<td nowrap>\n" +
                                HtmlWidgets.htmlTooltipImage(
                                    EDStatic.imageDirUrl + "arrowLL.gif", 
                                    "Click here to use the very first value in the list of values (" + 
                                        tFirst + ")<br>and redraw the graph.", 
                                    "onMouseUp='f1." + paramName + ".value=\"" + tFirst + 
                                        "\"; mySubmit(true);'") +
                                "</td>\n");
                            fieldSize -= 2;                            
                        } 
                        //show -?
                        if (tIndex >= 1 && 
                            (ss == 0 || 
                             (ss == 1 && (!showStartAndStopFields || startIndex + 1 < stopIndex)))) {
                            //bug: wrong direction if source alt values are ascending depth values
                            String ts = edvga.destinationToString(
                                edvga.destinationValue(tIndex - 1).getNiceDouble(0)); 
                            buttons.append(
                                "<td nowrap>\n" +
                                HtmlWidgets.htmlTooltipImage(
                                    EDStatic.imageDirUrl + "minus.gif", 
                                    "Click here to use the previous value in the list of values (" + 
                                        ts + ")<br>and redraw the graph.", 
                                    "onMouseUp='f1." + paramName + ".value=\"" + ts + 
                                        "\"; mySubmit(true);'") +
                                "</td>\n");
                            fieldSize -= 1;                            
                        } 
                        //show +?
                        if (tIndex < sourceSize - 1 && 
                            ((ss == 0 && startIndex + 1 < stopIndex) ||
                             ss == 1)) {
                            String ts = edvga.destinationToString(
                                edvga.destinationValue(tIndex + 1).getNiceDouble(0)); 
                            buttons.append(
                                "<td nowrap>\n" +
                                HtmlWidgets.htmlTooltipImage(
                                    EDStatic.imageDirUrl + "plus.gif", 
                                    "Click here to use the next value in the list of values (" + 
                                        ts + ")<br>and redraw the graph.", 
                                    "onMouseUp='f1." + paramName + ".value=\"" + ts + 
                                        "\"; mySubmit(true);'") +
                                "</td>\n");
                            fieldSize -= 1;                            
                        } 
                        //show arrowRR?
                        if (tIndex < sourceSize - 1 && ss == 1) {
                            buttons.append(
                                "<td nowrap>\n" +
                                HtmlWidgets.htmlTooltipImage(
                                    EDStatic.imageDirUrl + "arrowRR.gif", 
                                    "Click here to use the very last value in the list of values (" + 
                                        tLast + ")<br>and redraw the graph.", 
                                    "onMouseUp='f1." + paramName + ".value=\"" + tLast + 
                                        "\"; mySubmit(true);'") +
                                "</td>\n");
                            fieldSize -= 2;                            
                        } 

                        //show start or stop field
                        writer.write(widgets.beginTable(0, 0, "width=\"10%\"")); //keep it small
                        writer.write("<tr><td nowrap>" + gap);
                        writer.write(widgets.textField(paramName, edvgaTooltip, fieldSize, 255, 
                            ss == 0? avStart[av] : avStop[av], ""));
                        writer.write("</td>\n" + 
                            buttons + 
                            "</tr>\n");
                        writer.write(widgets.endTable()); 
                    } else { 
                        writer.write(gap + "<font color=\"gray\">&nbsp;specify just 1 value &rarr;</font>\n"); 
                        writer.write(widgets.hidden(paramName, "SeeStop"));
                    }
                    writer.write("  </td>\n");
                }

                //add avStart avStop to constraints
                if (showStartAndStopFields) 
                    constraints.append("[(" + avStart[av] + "):(" + avStop[av] + ")]");
                else 
                    constraints.append("[(" + avStop[av] + ")]");

                // *** and a slider for this axis    (Make A Graph)
                sliderFromNames[av] = formName + ".start" + av;
                sliderToNames[  av] = formName + ".stop"  + av;
                sliderUserValuesCsvs[av] = edvga.sliderCsvValues();
                int safeSourceSize1 = Math.max(1, sourceSize - 1);
                if (showStartAndStopFields) {
                    //2 thumbs
                    sliderNThumbs[av] = 2;
                    sliderInitFromPositions[av] = Math2.roundToInt((startIndex * (EDV.SLIDER_PIXELS - 1.0)) / safeSourceSize1);
                    sliderInitToPositions[av]   = sourceSize == 1? EDV.SLIDER_PIXELS - 1 :
                                                  Math2.roundToInt((stopIndex  * (EDV.SLIDER_PIXELS - 1.0)) / safeSourceSize1);
                    writer.write(
                        "<tr align=\"left\">\n" +
                        "  <td nowrap colspan=\"3\" align=\"left\">\n" +
                        widgets.dualSlider(av, EDV.SLIDER_PIXELS - 1, "align=\"left\"") +
                        "  </td>\n" +
                        "</tr>\n");
                } else {
                    //1 thumb
                    sliderNThumbs[av] = 1;
                    sliderFromNames[av] = formName + ".stop"  + av;  //change from default
                    sliderInitFromPositions[av] = Math2.roundToInt((stopIndex * (EDV.SLIDER_PIXELS - 1.0)) / safeSourceSize1);
                    sliderInitToPositions[av] = EDV.SLIDER_PIXELS - 1;
                    writer.write(
                        "<tr align=\"left\">\n" +
                        "  <td nowrap colspan=\"3\" align=\"left\">\n" +
                        widgets.slider(av, EDV.SLIDER_PIXELS - 1, "align=\"left\"") +
                        "  </td>\n" +
                        "</tr>\n");
                }

            }
            //end of dimensions constraints table
            writer.write(widgets.endTable());


            //*** make graphQuery
            StringBuffer graphQuery = new StringBuffer();
            //add data varNames and constraints
            for (int v = 1; v < nVars; v++) {
                if (String2.indexOf(dvNames, varName[v]) >= 0) {
                    if (graphQuery.length() > 0)
                        graphQuery.append(",");
                    graphQuery.append(varName[v] + constraints);
                }
            }

            //add .draw and .vars to graphQuery 
            graphQuery.append(
                "&.draw=" + draws[draw] +
                "&.vars=" + varName[0] + "|" + varName[1] + "|" + varName[2] + 
                    (nVars > 3? "|" + varName[3] : ""));

            //*** Graph Settings
            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write("  <tr><th align=\"left\" colspan=\"2\" nowrap>Graph Settings</th></tr>\n");
            if (drawLinesAndMarkers || drawMarkers) {
                //get Marker settings
                int mType = -1, mSize = -1;
                partValue = String2.stringStartsWith(queryParts, partName = ".marker=");
                if (partValue != null) {
                    pParts = String2.split(partValue.substring(partName.length()), '|');
                    if (pParts.length > 0) mType = String2.parseInt(pParts[0]);
                    if (pParts.length > 1) mSize = String2.parseInt(pParts[1]); //the literal, not the index
                    if (reallyVerbose)
                        String2.log(".marker type=" + mType + " size=" + mSize);
                }

                //markerType
                paramName = "mType";
                if (mType < 0 || mType >= GraphDataLayer.MARKER_TYPES.length)
                    mType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
                //if (!yIsAxisVar && varName[2].length() > 0 && 
                //    GraphDataLayer.MARKER_TYPES[mType].toLowerCase().indexOf("filled") < 0) 
                //    mType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE; //needs "filled" marker type
                writer.write("  <tr>\n" +
                             "    <td nowrap>Marker Type:&nbsp;</td>\n" +
                             "    <td nowrap>");
                writer.write(widgets.select(paramName, 
                    "Specify the marker type.", 
                    1, GraphDataLayer.MARKER_TYPES, mType, ""));

                //markerSize
                paramName = "mSize";
                String mSizes[] = {"3", "4", "5", "6", "7", "8", "9", "10", "11", 
                    "12", "13", "14", "15", "16", "17", "18"};
                mSize = String2.indexOf(mSizes, "" + mSize); //convert from literal 3.. to index in mSizes[0..]
                if (mSize < 0)
                    mSize = String2.indexOf(mSizes, "" + GraphDataLayer.MARKER_SIZE_SMALL);
                writer.write(gap + "Size: ");
                writer.write(widgets.select(paramName, 
                    "Specify the marker size.", 
                    1, mSizes, mSize, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery
                graphQuery.append("&.marker=" + mType + "|" + mSizes[mSize]);
            }

            String colors[] = HtmlWidgets.PALETTE17;
            if (drawLines || drawLinesAndMarkers || drawMarkers || drawSticks || drawVectors) {

                //color
                paramName = "colr"; //not color, to avoid possible conflict
                partValue = String2.stringStartsWith(queryParts, partName = ".color=0x");
                int colori = String2.indexOf(colors, 
                    partValue == null? "" : partValue.substring(partName.length()));
                if (colori < 0)
                    colori = String2.indexOf(colors, "000000");
                writer.write("  <tr>\n" +
                             "    <td nowrap>Color:&nbsp;</td>\n" +
                             "    <td nowrap>");
                writer.write(widgets.color17("", paramName, 
                    "Select a color.", 
                    colori, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery
                graphQuery.append("&.color=0x" + HtmlWidgets.PALETTE17[colori]);
            }

            if (drawLinesAndMarkers || drawMarkers || drawSurface) {
                //color bar
                partValue = String2.stringStartsWith(queryParts, partName = ".colorBar=");
                pParts = partValue == null? new String[0] : String2.split(partValue.substring(partName.length()), '|');
                if (reallyVerbose)
                    String2.log(".colorBar=" + String2.toCSVString(pParts));

                //find dataVariable relevant to colorBar
                //(force change in values if this var changes?  but how know, since no state?)
                int tDataVariablePo = String2.indexOf(dataVariableDestinationNames(), 
                    varName[2]); //currently, bothrelevant representation uses varName[2] for "Color"
                EDV tDataVariable = tDataVariablePo >= 0? dataVariables[tDataVariablePo] : null;

                paramName = "p";
                String defaultPalette = ""; 
                //String2.log("defaultPalette=" + defaultPalette + " pParts.length=" + pParts.length);
                int palette = Math.max(0, 
                    String2.indexOf(EDStatic.palettes0, pParts.length > 0? pParts[0] : defaultPalette));
                writer.write("  <tr>\n" +
                             "    <td colspan=\"2\" nowrap>Color Bar: ");
                writer.write(widgets.select(paramName, 
                    "Select a palette for the color bar (or leave blank to get the default).", 
                    1, EDStatic.palettes0, palette, ""));

                String conDis[] = new String[]{"", "Continuous", "Discrete"};
                paramName = "pc";
                int continuous = pParts.length > 1? (pParts[1].equals("D")? 2 : pParts[1].equals("C")? 1 : 0) : 0;
                writer.write(gap + "Continuity: ");
                writer.write(widgets.select(paramName, 
                    "Specify whether the colors should be continuous or discrete<br>(or leave blank for the default).", 
                    1, conDis, continuous, ""));

                paramName = "ps";
                String defaultScale = "";
                int scale = Math.max(0, String2.indexOf(EDV.VALID_SCALES0, pParts.length > 2? pParts[2] : defaultScale));
                writer.write(gap + "Scale: ");
                writer.write(widgets.select(paramName, 
                    "Select a scale for the color bar (or leave blank for the default).", 
                    1, EDV.VALID_SCALES0, scale, ""));
                writer.write(
                    "    </td>\n" +
                    "  </tr>\n");

                paramName = "pMin";
                String defaultMin = "";
                String palMin = pParts.length > 3? pParts[3] : defaultMin;
                writer.write(
                    "  <tr>\n" +
                    "    <td colspan=\"2\" nowrap> " + gap + gap + "Min: ");
                writer.write(widgets.textField(paramName, 
                    "Specify the low value for the color bar (or leave blank for the default).", 
                   10, 40, palMin, ""));

                paramName = "pMax";
                String defaultMax = "";
                String palMax = pParts.length > 4? pParts[4] : defaultMax;
                writer.write(gap + "Max: ");
                writer.write(widgets.textField(paramName, 
                    "Specify the high value for the color bar (or leave blank for the default).", 
                   10, 40, palMax, ""));

                paramName = "pSec";
                int pSections = Math.max(0, String2.indexOf(EDStatic.paletteSections, pParts.length > 5? pParts[5] : ""));
                writer.write(gap + "N Sections: ");
                writer.write(widgets.select(paramName, 
                    "Specify the number of sections for the color bar<br>(or leave blank for the default).", 
                    1, EDStatic.paletteSections, pSections, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery 
                graphQuery.append(
                    "&.colorBar=" + EDStatic.palettes0[palette] + "|" +
                    (conDis[continuous].length() == 0? "" : conDis[continuous].charAt(0)) + "|" +
                    EDV.VALID_SCALES0[scale] + "|" +
                    palMin + "|" + palMax + "|" + EDStatic.paletteSections[pSections]);
            }

            if (drawVectors) {

                //Vector Standard 
                paramName = "vec";
                String vec = String2.stringStartsWith(queryParts, partName = ".vec=");
                vec = vec == null? "" : vec.substring(partName.length());
                writer.write("  <tr>\n" +
                             "    <td nowrap>Vector Standard:&nbsp;</td>\n" +
                             "    <td nowrap>");
                writer.write(widgets.textField(paramName, 
                    "Specify the data vector length (in data units) to be " +
                    "<br>scaled to the size of the sample vector in the legend" +
                    "<br>(or leave blank for the default).", 
                   10, 20, vec, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery
                if (vec.length() > 0)
                    graphQuery.append("&.vec=" + vec);
            }

            if (drawSurface) {
                //Draw Land
                String landOptions[] = {"under the data", "over the data"};  //order also affects javascript below
                partValue = String2.stringStartsWith(queryParts, partName = ".land=");
                if (partValue != null) 
                    partValue = partValue.substring(6);
                int tLand = partValue == null? 
                    (EDStatic.drawLand.equals("under")? 0 : 1) :
                    (partValue.equals("under")? 0 : 1);
                writer.write(
                    "<tr>\n" +
                    "  <td colSpan=\"2\" nowrap>Draw the land mask: \n");
                writer.write(widgets.select("land", 
                    "Specify whether the land mask should be drawn under or over the data.\n" +
                    "<br>(Some data has already had a land mask applied, making this irrelevant.", 
                    1, landOptions, tLand, ""));
                writer.write(
                    "  </td>\n" +
                    "</tr>\n");

                //add to graphQuery
                graphQuery.append("&.land=" + (tLand == 0? "under" : "over"));
            }

            //*** end of form
            writer.write(widgets.endTable()); //end of Graph Settings table

            //make javascript function to generate query
            writer.write(
                "<script type=\"text/javascript\"> \n" +
                "function makeQuery(varsToo) { \n" +
                "  try { \n" +
                "    var d = document; \n" +
                "    var start, tv, c = \"\", q = \"\"; \n"); //c=constraint  q=query
            //gather constraints
            for (int av = 0; av < nAv; av++) 
                writer.write(
                    "    start = d.f1.start" + av + ".value; \n" +
                    "    c += \"[\"; \n" +
                    "    if (start != \"SeeStop\") c += \"(\" + start + \"):\"; \n" + //javascript uses !=, not !equals()
                    "    c += \"(\" + d.f1.stop" + av + ".value + \")]\"; \n");
            //var[constraints],var[constraints]
            for (int v = 1; v < nVars; v++) {
                if (varOptions[v] == dvNames || varOptions[v] == dvNames0) { //simpler because advNames isn't an option
                    writer.write(
                        "    tv = d.f1.var" + v + ".options[d.f1.var" + v + ".selectedIndex].text; \n" +
                        "    if (tv.length > 0) { \n" +
                        "      if (q.length > 0) q += \",\"; \n" +  //javascript uses length, not length()
                        "      q += tv + c; \n" +
                        "    } \n");
                }
            }
            //graph settings  
            writer.write(
                "    q += \"&.draw=\" + d.f1.draw.options[d.f1.draw.selectedIndex].text; \n");
            writer.write(                
                "    if (varsToo) { \n" +
                "      q += \"&.vars=\" + d.f1.var0.options[d.f1.var0.selectedIndex].text + \n" +
                "        \"|\" + d.f1.var1.options[d.f1.var1.selectedIndex].text; \n");
            if (nVars >= 3) writer.write(
                "      q += \"|\" + d.f1.var2.options[d.f1.var2.selectedIndex].text; \n");
            if (nVars >= 4) writer.write(
                "      q += \"|\" + d.f1.var3.options[d.f1.var3.selectedIndex].text; \n");
            writer.write(
                "    } \n");  
            if (drawLinesAndMarkers || drawMarkers) writer.write(
                "    q += \"&.marker=\" + d.f1.mType.selectedIndex + \"|\" + \n" +
                "      d.f1.mSize.options[d.f1.mSize.selectedIndex].text; \n");
            if (drawLines || drawLinesAndMarkers || drawMarkers || drawSticks || drawVectors) writer.write(
                "    q += \"&.color=0x\"; \n" +
                "    for (var rb = 0; rb < " + colors.length + "; rb++) \n" + 
                "      if (d.f1.colr[rb].checked) q += d.f1.colr[rb].value; \n"); //always: one will be checked
            if (drawLinesAndMarkers || drawMarkers || drawSurface) writer.write(
                "    var tpc = d.f1.pc.options[d.f1.pc.selectedIndex].text;\n" +
                "    q += \"&.colorBar=\" + d.f1.p.options[d.f1.p.selectedIndex].text + \"|\" + \n" +
                "      (tpc.length > 0? tpc.charAt(0) : \"\") + \"|\" + \n" +
                "      d.f1.ps.options[d.f1.ps.selectedIndex].text + \"|\" + \n" +
                "      d.f1.pMin.value + \"|\" + d.f1.pMax.value + \"|\" + \n" +
                "      d.f1.pSec.options[d.f1.pSec.selectedIndex].text; \n");
            if (drawVectors) writer.write(
                "    if (d.f1.vec.value.length > 0) q += \"&.vec=\" + d.f1.vec.value; \n");
            if (drawSurface) writer.write(
                "    q += \"&.land=\" + (d.f1.land.selectedIndex==0? \"under\" : \"over\"); \n");
            writer.write(
                "    return q; \n" +
                "  } catch (e) { \n" +
                "    alert(e); \n" +
                "    return \"\"; \n" +
                "  } \n" +
                "} \n" +
                "function mySubmit(varsToo) { \n" +
                "  var q = makeQuery(varsToo); \n" +
                "  if (q.length > 0) window.location=\"" + //javascript uses length, not length()
                    tErddapUrl + "/griddap/" + datasetID() + 
                    ".graph?\" + q;\n" + 
                "} \n" +
                "</script> \n");  

            //submit
            writer.write(HtmlWidgets.ifJavaScriptDisabled);
            writer.write("<p>\n"); 
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write(
                "<tr><td nowrap>");
            writer.write(widgets.htmlButton("button", "", "",
                "Redraw the graph based on the settings above. " +
                "<br>Please be patient; sometimes it takes a long time " +
                "<br>to get the data from the source.", 
                "<b>Redraw the Graph</b>", 
                "onMouseUp='mySubmit(true);'")); 
            writer.write(
                "</td></tr>\n");

            //Download the Data
            writer.write(
                "<tr><td nowrap>Then set the File Type:\n");
            paramName = "fType";
            writer.write(widgets.select(paramName, EDStatic.EDDSelectFileType, 1,
                allFileTypeNames, defaultFileTypeOption, 
                "onChange='f1.tUrl.value=\"" + tErddapUrl + "/griddap/" + datasetID() + 
                    "\" + f1.fType.options[f1.fType.selectedIndex].text + " + 
                    "\"?" + String2.replaceAll(graphQuery.toString(), "&", "&amp;") + "\";'"));
            writer.write(" and\n");
            writer.write(widgets.button("button", "", 
                String2.replaceAll(
                "Click here to download the data in the specified File Type. " +
                "<br>Please be patient; sometimes it takes a long time " +
                "<br>to get the data from the source.",
                "&protocolName;", dapProtocol),
                "Download the Data or an Image", 
                "onMouseUp='window.location=\"" + 
                    tErddapUrl + "/griddap/" + datasetID() + "\" + f1." + 
                    paramName + ".options[f1." + paramName + ".selectedIndex].text + \"?" + 
                    XML.encodeAsXML(graphQuery.toString()) +
                "\";'")); //or open a new window: window.open(result);\n" +
            writer.write(
                "</td></tr>\n");

            //view the url
            writer.write(
                "<tr><td nowrap>&nbsp;&nbsp;or view the URL: \n");
            writer.write(widgets.textField("tUrl",   
                String2.replaceAll(EDStatic.justGenerateAndViewGraphUrlHtml, 
                    "&protocolName;", dapProtocol), 
                55, 1000, 
                tErddapUrl + "/griddap/" + datasetID() + 
                    dataFileTypeNames[defaultFileTypeOption] + "?" + graphQuery.toString(), 
                ""));
            writer.write(
                "</td></tr>\n" +
                "</table>\n\n");

            //end form
            writer.write(widgets.endForm());

            //end of left half of big table
            writer.write("</td>\n" +
                "<td>" + gap + "</td>\n" + //gap in center
                "<td align=\"left\" valign=\"top\">\n"); //begin right half

            //display the graph
            if (verbose) 
                String2.log("graphQuery=" + graphQuery);
            writer.write("<img alt=\"The graph you specified. Please be patient. It may take a long time to get data from the source.\" " +
                "src=\"" + tErddapUrl + "/griddap/" + //don't use \n for the following lines
                datasetID()+ ".png?" + String2.replaceAll(graphQuery.toString(), "&", "&amp;") + "\">\n");

            //end of right half of big table
            writer.write("</td></tr></table>\n");


            //do things with graphs
            writer.write("<hr noshade>\n" +
                "<h2><a name=\"uses\">Things</a> You Can Do With Your Graphs</h2>\n" +
                "<ul>\n" +
                "<li>Web page authors can \n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/embed.html\">embed a graph of the latest data in a web page</a> \n" +
                "  using HTML &lt;img&gt; tags.\n" +
                "<li>Anyone can use <a href=\"" + tErddapUrl + "/slidesorter.html\">Slide Sorter</a> \n" +
                "  to build a personal web page that displays graphs of the latest data, each in its own, draggable slide.\n" +
                "<li>Anyone can use or make\n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/gadgets/GoogleGadgets.html\">Google Gadgets</a>\n" +
                "  to display images with the latest data on their iGoogle home page. \n" +
                "  <br>&nbsp;\n" +
                "</ul>\n" +
                "\n");


            //end of document
            writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
            writer.write("<hr noshade>\n");
            writer.write("<h2>The Dataset Attribute Structure (.das) for this Dataset</h2>\n" +
                "<pre>\n");
            writeDAS("/griddap/" + datasetID() + ".das", "", writer, true); //useful so search engines find all relevant words
            writer.write("</pre>\n");
            writer.write("<br>&nbsp;\n");
            writer.write("<hr noshade>\n");
            writeGeneralDapHtmlInstructions(tErddapUrl, writer, false); 
            writer.write("<br>&nbsp;\n" +
                "<br><small>" + EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small>\n");
            if (EDStatic.displayDiagnosticInfo) 
                EDStatic.writeDiagnosticInfoHtml(writer);

            //the javascript for the sliders
            writer.write(widgets.sliderScript(sliderFromNames, sliderToNames, 
                sliderNThumbs, sliderUserValuesCsvs, 
                sliderInitFromPositions, sliderInitToPositions, EDV.SLIDER_PIXELS - 1));
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }

        writer.write(EDStatic.endBodyHtml(tErddapUrl));
        writer.write("\n</html>\n");

        //essential
        writer.flush(); 
        out.close();        
    }


    /**
     * This gets the data for the userDapQuery and writes the grid data to the 
     * outputStream in the DODS ASCII data format, which is not defined in DAP 2.0,
     * but which is very close to saveAsDODS below.
     * This mimics http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day.asc?MHchla[1477][0][2080:2:2082][4940] .
     * 
     * @param requestUrl
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null).
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsAsc(String requestUrl, String userDapQuery, OutputStreamSource outputStreamSource) 
        throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsAsc"); 
        long time = System.currentTimeMillis();

        //handle axis request
        if (isAxisDapQuery(userDapQuery)) {
            //get AxisDataAccessor first, in case of error when parsing query
            AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
            int nRAV = ada.nRequestedAxisVariables();

            //write the dds    //DAP 2.0, 7.2.3
            saveAsDDS(requestUrl, userDapQuery, outputStreamSource);  

            //write the connector  //DAP 2.0, 7.2.3
            OutputStreamWriter writer = new OutputStreamWriter(
                outputStreamSource.outputStream("ISO-8859-1"),
                "ISO-8859-1"); //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit
            writer.write(
                "---------------------------------------------" + OpendapHelper.EOL + 
                "Data:" + OpendapHelper.EOL); //see EOL definition for comments

            //write the data  //DAP 2.0, 7.3.2.4
            for (int av = 0; av < nRAV; av++) {
                writer.write(ada.axisVariables(av).destinationName() +
                    "[" + ada.axisValues(av).size() + "]" + OpendapHelper.EOL); 
                writer.write(ada.axisValues(av).toString());
                writer.write(OpendapHelper.EOL);
            }

            writer.flush(); //essential

            //diagnostic
            if (reallyVerbose) String2.log("  EDDGrid.saveAsAsc axis done.\n");
            return;
        }

        //get full gridDataAccessor first, in case of error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, false);  //rowMajor, convertToNaN
        String arrayQuery = buildDapArrayQuery(gridDataAccessor.constraints());
        EDV tDataVariables[] = gridDataAccessor.dataVariables();
        boolean entireDataset = userDapQuery.trim().length() == 0;

        //get partial gridDataAccessor, to test for size error
        GridDataAccessor tedGrid = new GridDataAccessor(this, requestUrl,
            tDataVariables[0].destinationName() + arrayQuery, 
            true, false);   //rowMajor, convertToNaN
        long tSize = tedGrid.totalIndex().size();
        if (tSize >= Integer.MAX_VALUE)
            throw new SimpleException("Error: " +
                "Requested array size=" + tSize + " is greater than DAP limit.");

        //write the dds    //DAP 2.0, 7.2.3
        saveAsDDS(requestUrl, userDapQuery, outputStreamSource);  

        //write the connector  //DAP 2.0, 7.2.3
        OutputStreamWriter writer = new OutputStreamWriter(
            outputStreamSource.outputStream("ISO-8859-1"),
            "ISO-8859-1"); //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit
        writer.write("---------------------------------------------" + 
            OpendapHelper.EOL); //see EOL definition for comments 

        //write the axis variables
        int nAxisVariables = axisVariables.length;
        if (entireDataset) {
            //send the axis data
            int tShape[] = gridDataAccessor.totalIndex().shape();
            for (int av = 0; av < nAxisVariables; av++) {
                writer.write(axisVariables[av].destinationName() +
                    "[" + tShape[av] + "]" + OpendapHelper.EOL); //see EOL definition for comments
                writer.write(gridDataAccessor.axisValues[av].toString());
                writer.write(OpendapHelper.EOL); //see EOL definition for comments
            }
            writer.write(OpendapHelper.EOL); //see EOL definition for comments
        }

        //write the data  //DAP 2.0, 7.3.2.4
        //write elements of the array, in dds order
        int nDataVariables = tDataVariables.length;
        for (int dv = 0; dv < nDataVariables; dv++) {
            String dvDestName = tDataVariables[dv].destinationName();
            tedGrid = new GridDataAccessor(this, requestUrl, dvDestName + arrayQuery, 
                true, false);   //rowMajor, convertToNaN
            int shape[] = tedGrid.totalIndex().shape();
            int current[] = tedGrid.totalIndex().getCurrent();

            //identify the array
            writer.write(dvDestName + "." + dvDestName);
            int nAv = axisVariables.length;
            for (int av = 0; av < nAv; av++)
                writer.write("[" + shape[av] + "]");

            //send the array data
            while (tedGrid.increment()) {
                //if last dimension's value is 0, start a new row
                if (current[nAv - 1] == 0) {
                    writer.write(OpendapHelper.EOL); //see EOL definition for comments
                    for (int av = 0; av < nAv - 1; av++)
                        writer.write("[" + current[av] + "]");
                }
                writer.write(", " + tedGrid.getDataValueAsString(0));
            }

            //send the axis data
            for (int av = 0; av < nAxisVariables; av++) {
                writer.write(OpendapHelper.EOL + OpendapHelper.EOL + dvDestName + "." + axisVariables[av].destinationName() +
                    "[" + shape[av] + "]" + OpendapHelper.EOL); //see EOL definition for comments
                writer.write(tedGrid.axisValues[av].toString());
            }
            writer.write(OpendapHelper.EOL); //see EOL definition for comments
        }

        writer.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsAsc done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * This writes the dataset data attributes (DAS) to the outputStream.
     * It is always the same regardless of the userDapQuery.
     * (That's what THREDDS does -- DAP 2.0 7.2.1 is vague.
     *  THREDDs doesn't even object if userDapQuery is invalid.)
     * See writeDAS().
     *
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery the part of the user's request
     *    after the '?', still percentEncoded (shouldn't be null).
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsDAS(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsDAS"); 
        long time = System.currentTimeMillis();

        //get the modified outputStream
        Writer writer = new OutputStreamWriter(outputStreamSource.outputStream("ISO-8859-1"),
            "ISO-8859-1"); //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit

        //write the DAS
        writeDAS(File2.forceExtension(requestUrl, ".das"), "", writer, false);

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsDAS done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * This writes the dataset data attributes (DAS) to the outputStream.
     * It is always the same regardless of the userDapQuery (except for history).
     * (That's what THREDDS does -- DAP 2.0 7.2.1 is vague.
     *  THREDDs doesn't even object if userDapQuery is invalid.)
     * <p>
     * E.g., <pre>
Attributes {
    altitude {
        Float32 actual_range 0.0, 0.0;
        Int32 fraction_digits 0;
        String long_name "Altitude";
        String standard_name "altitude";
        String units "m";
        String axis "Z";
    }
    NC_GLOBAL {
        ....
    }
}
</pre> 
     *
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery the part of the user's request after the '?', 
     *    still percentEncoded (shouldn't be null).  (Affects history only.)
     * @param writer a Writer.
     *   At the end of this method the Writer is flushed, not closed.
     * @param encodeAsHtml if true, characters like &lt; are converted to their 
     *    character entities.
     * @throws Throwable  if trouble. 
     */
    public void writeDAS(String requestUrl, String userDapQuery, 
        Writer writer, boolean encodeAsHtml) throws Throwable {

        int nAxisVariables = axisVariables.length;
        int nDataVariables = dataVariables.length;
        writer.write("Attributes {" + OpendapHelper.EOL); //see EOL definition for comments
        for (int av = 0; av < nAxisVariables; av++) 
            OpendapHelper.writeToDAS(axisVariables[av].destinationName(),
                axisVariables[av].combinedAttributes(), writer, encodeAsHtml);
        for (int dv = 0; dv < nDataVariables; dv++) 
            OpendapHelper.writeToDAS(dataVariables[dv].destinationName(),
                dataVariables[dv].combinedAttributes(), writer, encodeAsHtml);

        //how do global attributes fit into opendap view of attributes?
        Attributes gAtts = new Attributes(combinedGlobalAttributes); //a copy

        //fix up global attributes  (always to a local COPY of global attributes)
        EDD.addToHistory(gAtts, sourceUrl());
        EDD.addToHistory(gAtts, EDStatic.baseUrl + requestUrl + 
            (userDapQuery == null || userDapQuery.length() == 0? "" : "?" + userDapQuery));

        OpendapHelper.writeToDAS(
            "NC_GLOBAL", //.nc files say NC_GLOBAL; ncBrowse and netcdf-java treat NC_GLOBAL as special case
            gAtts, writer, encodeAsHtml);
        writer.write("}" + OpendapHelper.EOL); //see EOL definition for comments
        writer.flush(); //essential

    }

    /**
     * This gets the data for the userDapQuery and writes the grid data 
     * structure (DDS) to the outputStream.
     * E.g. <pre>
  Dataset {
      Float64 lat[lat = 180];
      Float64 lon[lon = 360];
      Float64 time[time = 404];
      Grid {
       ARRAY:
          Int32 sst[time = 404][lat = 180][lon = 360];
       MAPS:
          Float64 time[time = 404];
          Float64 lat[lat = 180];
          Float64 lon[lon = 360];
      } sst;
  } weekly;
 </pre>
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null).
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsDDS(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsDDS"); 
        long time = System.currentTimeMillis();

        //handle axisDapQuery
        if (isAxisDapQuery(userDapQuery)) {
            //get axisDataAccessor first, in case of error when parsing query
            AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
            int nRAV = ada.nRequestedAxisVariables();

            //then get the modified outputStream
            Writer writer = new OutputStreamWriter(outputStreamSource.outputStream("ISO-8859-1"),
                "ISO-8859-1"); //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit

            writer.write("Dataset {" + OpendapHelper.EOL); //see EOL definition for comments
            for (int av = 0; av < nRAV; av++) {
                String destName = ada.axisVariables(av).destinationName();
                PrimitiveArray apa = ada.axisValues(av);
                writer.write("  " + //e.g., Float64 time[time = 404];
                    OpendapHelper.getAtomicType(apa.getElementType()) + " " + 
                    destName + "[" + destName + " = " + apa.size() + "];" + OpendapHelper.EOL); 
            }
            //Thredds recently started using urlEncoding the final name (and other names?). I don't.
            //Update: I think they undid this change.
            writer.write("} " + datasetID + ";" + OpendapHelper.EOL); 
            writer.flush(); //essential

            //diagnostic
            if (reallyVerbose) String2.log("  EDDGrid.saveAsDDS axis done.");
            return;
        }

        //get gridDataAccessor first, in case of error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, false);   //rowMajor, convertToNaN
        boolean entireDataset = userDapQuery == null || SSR.percentDecode(userDapQuery).trim().length() == 0;

        //then get the modified outputStream
        Writer writer = new OutputStreamWriter(outputStreamSource.outputStream("ISO-8859-1"),
            "ISO-8859-1"); //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit

        int nAxisVariables = axisVariables.length;
        int nDataVariables = gridDataAccessor.dataVariables().length;
        writer.write("Dataset {" + OpendapHelper.EOL); //see EOL definition for comments
        String arrayDims[] = new String[nAxisVariables]; //each e.g., [time = 404]
        String dims[] = new String[nAxisVariables];        //each e.g., Float64 time[time = 404];
        StringBuffer allArrayDims = new StringBuffer();
        for (int av = 0; av < nAxisVariables; av++) {
            PrimitiveArray apa = gridDataAccessor.axisValues(av);
            arrayDims[av] = "[" + axisVariables[av].destinationName() + " = " + 
                apa.size() + "]";
            dims[av] = OpendapHelper.getAtomicType(apa.getElementType()) +
                " " + axisVariables[av].destinationName() + arrayDims[av] + ";" + OpendapHelper.EOL;
            allArrayDims.append(arrayDims[av]);
            if (entireDataset) 
                writer.write("  " + dims[av]);
        }
        for (int dv = 0; dv < nDataVariables; dv++) {
            String dvName = gridDataAccessor.dataVariables()[dv].destinationName();
            writer.write("  GRID {" + OpendapHelper.EOL);
            writer.write("    ARRAY:" + OpendapHelper.EOL); 
            writer.write("      " + 
                OpendapHelper.getAtomicType(
                    gridDataAccessor.dataVariables()[dv].destinationDataTypeClass()) +
                " " + dvName + allArrayDims + ";" + OpendapHelper.EOL); 
            writer.write("    MAPS:" + OpendapHelper.EOL); 
            for (int av = 0; av < nAxisVariables; av++) 
                writer.write("      " + dims[av]);
            writer.write("  } " + dvName + ";" + OpendapHelper.EOL); 
        }

        //Thredds recently started using urlEncoding the final name (and other names?).
        //I don't (yet).
        writer.write("} " + datasetID + ";" + OpendapHelper.EOL);
        writer.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsDDS done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * This gets the data for the userDapQuery and writes the grid data to the 
     * outputStream in the DODS DataDDS format (DAP 2.0, 7.2.3).
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsDODS(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsDODS"); 
        long time = System.currentTimeMillis();

        //handle axisDapQuery
        if (isAxisDapQuery(userDapQuery)) {
            //get axisDataAccessor first, in case of error when parsing query
            AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
            int nRAV = ada.nRequestedAxisVariables();

            //write the dds    //DAP 2.0, 7.2.3
            saveAsDDS(requestUrl, userDapQuery, outputStreamSource);  

            //write the connector  //DAP 2.0, 7.2.3
            //see EOL definition for comments
            OutputStream outputStream = outputStreamSource.outputStream("");
            outputStream.write((OpendapHelper.EOL + "Data:" + OpendapHelper.EOL).getBytes());

            //write the data  //DAP 2.0, 7.3.2.4
            //write elements of the array, in dds order
            DataOutputStream dos = new DataOutputStream(outputStream);
            for (int av = 0; av < nRAV; av++)
                ada.axisValues(av).externalizeForDODS(dos);
            dos.flush(); //essential

            //diagnostic
            if (reallyVerbose) String2.log("  EDDGrid.saveAsDODS axis done.\n");
            return;
        }

        //get gridDataAccessor first, in case of error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, false);  //rowMajor, convertToNaN 
        String arrayQuery = buildDapArrayQuery(gridDataAccessor.constraints());
        EDV tDataVariables[] = gridDataAccessor.dataVariables();
        boolean entireDataset = userDapQuery == null || SSR.percentDecode(userDapQuery).trim().length() == 0;

        //get partial gridDataAccessor, in case of size error
        GridDataAccessor tedGrid = new GridDataAccessor(this, requestUrl, 
            tDataVariables[0].destinationName() + arrayQuery, true, false);
        long tSize = tedGrid.totalIndex().size();
        if (tSize >= Integer.MAX_VALUE)
            throw new SimpleException("Error: " +
                "Requested array size=" + tSize + " is greater than DAP limit.");

        //write the dds    //DAP 2.0, 7.2.3
        saveAsDDS(requestUrl, userDapQuery, outputStreamSource);  

        //write the connector  //DAP 2.0, 7.2.3
        //see EOL definition for comments
        OutputStream outputStream = outputStreamSource.outputStream("");
        outputStream.write((OpendapHelper.EOL + "Data:" + OpendapHelper.EOL).getBytes()); 

        //make the dataOutputStream
        DataOutputStream dos = new DataOutputStream(outputStream);

        //write the axis variables
        int nAxisVariables = axisVariables.length;
        if (entireDataset) {
            for (int av = 0; av < nAxisVariables; av++) 
                gridDataAccessor.axisValues[av].externalizeForDODS(dos);
        }

        //write the data  //DAP 2.0, 7.3.2.4
        //write elements of the array, in dds order
        int nDataVariables = tDataVariables.length;
        for (int dv = 0; dv < nDataVariables; dv++) {
            tedGrid = new GridDataAccessor(this, requestUrl, 
                tDataVariables[dv].destinationName() + arrayQuery, 
                true, false);   //rowMajor, convertToNaN
            tSize = tedGrid.totalIndex().size();

            //send the array size (twice)  //DAP 2.0, 7.3.2.1
            dos.writeInt((int)tSize);
            dos.writeInt((int)tSize);

            //send the array data
            Class type = tDataVariables[dv].destinationDataTypeClass();
            if        (type == byte.class  ) {while (tedGrid.increment()) dos.writeByte(tedGrid.getDataValueAsInt(0));
                //pad byte array to 4 byte boundary
                long tn = tedGrid.totalIndex().size();
                while (tn++ % 4 != 0) dos.writeByte(0);
            } else if (type == short.class ) {while (tedGrid.increment()) dos.writeInt(tedGrid.getDataValueAsInt(0));
            } else if (type == int.class   ) {while (tedGrid.increment()) dos.writeInt(tedGrid.getDataValueAsInt(0));
            } else if (type == float.class ) {while (tedGrid.increment()) dos.writeFloat(tedGrid.getDataValueAsFloat(0));
            } else if (type == double.class) {while (tedGrid.increment()) dos.writeDouble(tedGrid.getDataValueAsDouble(0));
            } else if (type == String.class) {while (tedGrid.increment()) StringArray.externalizeForDODS(dos, tedGrid.getDataValueAsString(0));
            } else {throw new RuntimeException("Internal error: unsupported source data type=" + 
                PrimitiveArray.elementTypeToString(type));
            }
            for (int av = 0; av < nAxisVariables; av++) 
                gridDataAccessor.axisValues[av].externalizeForDODS(dos);
        }

        dos.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsDODS done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }


    /**
     * This gets the data for the userDapQuery and writes the grid data to the 
     * outputStream in the ESRI ASCII data format.
     * For .esriAsci, dataVariable queries can specify multiple longitude and latitude
     * values, but just one value for other dimensions.
* Currently, the requested lon values can't be below and above 180  
* (below is fine; above is automatically shifted down).
* [future: do more extensive fixup].
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null).
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsEsriAscii(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsEsriAscii"); 
        long time = System.currentTimeMillis();

        //handle axis request
        if (isAxisDapQuery(userDapQuery)) 
            throw new SimpleException("Error: " +
                "The ESRI .asc format is for latitude longitude data requests only.");

        if (lonIndex < 0 || latIndex < 0) 
            throw new SimpleException("Error: " +
                "The ESRI .asc format is for latitude longitude data requests only.");

        //parse the userDapQuery and get the GridDataAccessor
        //this also tests for error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, true);  //rowMajor, convertToNaN
        if (gridDataAccessor.dataVariables().length > 1) 
            throw new SimpleException("Error: " +
                "The ESRI .asc format can only handle one data variable.");
        EDV edv = gridDataAccessor.dataVariables()[0];
        Class edvClass = edv.destinationDataTypeClass();
        boolean isIntType = !edvClass.equals(float.class) && !edvClass.equals(double.class) &&
            !edvClass.equals(String.class);
        boolean isFloatType = edvClass.equals(float.class);

        //check that request meets ESRI restrictions
        PrimitiveArray lonPa = null, latPa = null;
        for (int av = 0; av < axisVariables.length; av++) {
            PrimitiveArray avpa = gridDataAccessor.axisValues(av);
            if (av == lonIndex) {
                lonPa = avpa;
            } else if (av == latIndex) {
                latPa = avpa;
            } else {
                if (avpa.size() > 1)
                    throw new SimpleException("Error: " +
                        "For ESRI .asc requests, the " + 
                        axisVariables[av].destinationName() + " dimension's size must be 1."); 
            }
        }

        String error = lonPa.isEvenlySpaced();
        if (error.length() > 0) //size=1 is evenlySpaced
            throw new SimpleException("Error: " +
                "The dataset's longitude values aren't evenly spaced (as required by ESRI's .asc format):\n" + 
                error);
        error = latPa.isEvenlySpaced();
        if (error.length() > 0) //size=1 is evenlySpaced
            throw new SimpleException("Error: " +
                "The dataset's latitude values aren't evenly spaced (as required by ESRI's .asc format):\n" + 
                error);

        int nLon = lonPa.size();
        int nLat = latPa.size();
        double minX = lonPa.getDouble(0);
        double minY = latPa.getDouble(0);
        double maxX = lonPa.getDouble(nLon - 1);
        double maxY = latPa.getDouble(nLat - 1);
        boolean flipX = false;
        boolean flipY = false;
        if (minX > maxX) {flipX = true; double d = minX; minX = maxX; maxX = d; }
        if (minY > maxY) {flipY = true; double d = minY; minY = maxY; maxY = d; }
        double lonSpacing = lonPa.size() <= 1? Double.NaN : (maxX - minX) / (nLon - 1);
        double latSpacing = latPa.size() <= 1? Double.NaN : (maxY - minY) / (nLat - 1);
        if ( Double.isNaN(lonSpacing) && !Double.isNaN(latSpacing)) lonSpacing = latSpacing;
        if (!Double.isNaN(lonSpacing) &&  Double.isNaN(latSpacing)) latSpacing = lonSpacing;
        if ( Double.isNaN(lonSpacing) &&  Double.isNaN(latSpacing)) 
            throw new SimpleException("Error: " +
                "For ESRI .asc requests, the longitude or latitude dimension size must be greater than 1."); 

        //for almostEqual(3, lonSpacing, latSpacing) DON'T GO BELOW 3!!!
        //For example: PHssta has 4096 lon points so spacing is ~.0878
        //But .0878 * 4096 = 359.6   
        //and .0879 * 4096 = 360.0    (just beyond extreme test of 3 digit match)
        //That is unacceptable. So 2 would be abominable.  Even 3 is stretching the limits.
        if (!Math2.almostEqual(3, lonSpacing, latSpacing))
            throw new SimpleException("Error: " +
                "For ESRI .asc requests, the longitude spacing (" + 
                lonSpacing + ") must equal the latitude spacing (" + latSpacing + ").");
        if (minX < 180 && maxX > 180)
            throw new SimpleException("Error: " +
                "For ESRI .asc requests, the longitude values can't be below and above 180.");
        double lonAdjust = lonPa.getDouble(0) >= 180? -360 : 0;
        if (minX + lonAdjust < -180 || maxX + lonAdjust > 180)
            throw new SimpleException("Error: " +
                "For ESRI.asc requests, the adjusted longitude values (" + 
                (minX + lonAdjust) + " to " + (maxX + lonAdjust) + ") must be between -180 and 180.");

        //request is ok and compatible with ESRI .asc!

        //complications:
        //* lonIndex and latIndex can be in any position in axisVariables.
        //* ESRI .asc wants latMajor (that might be rowMajor or columnMajor), 
        //   and TOP row first!
        //The simplest solution is to save all data to temp file,
        //then read values as needed from file and write to writer.

        //make the GridDataRandomAccessor
        GridDataRandomAccessor gdra = new GridDataRandomAccessor(gridDataAccessor);
        int current[] = gridDataAccessor.totalIndex().getCurrent(); //the internal object that changes

        //then get the writer
        //???!!! ISO-8859-1 is a guess. I found no specification.
        OutputStreamWriter writer = new OutputStreamWriter(
            outputStreamSource.outputStream("ISO-8859-1"), "ISO-8859-1"); 

        //ESRI .asc doesn't like NaN
        double dmv = edv.safeDestinationMissingValue();
        String NaNString = Double.isNaN(dmv)? "-9999999" : //good for int and floating data types    
            dmv == Math2.roundToLong(dmv)? "" + Math2.roundToLong(dmv) :
            "" + dmv;

        //write the data
        writer.write("ncols " + nLon + "\n");
        writer.write("nrows " + nLat + "\n");
        //???!!! ERD always uses centered, but others might need was xllcorner yllcorner
        writer.write("xllcenter " + (minX + lonAdjust) + "\n"); 
        writer.write("yllcenter " + minY + "\n"); 
        //ArcGIS forces cellsize to be square; see test above
        writer.write("cellsize " + latSpacing + "\n"); 
        writer.write("nodata_value " + NaNString + "\n");

        //write values from row to row, top to bottom
        Arrays.fill(current, 0);  //manipulate indices in current[]
        for (int tLat = 0; tLat < nLat; tLat++) {
            current[latIndex] = flipY? tLat : nLat - tLat -1;
            for (int tLon = 0; tLon < nLon; tLon++) {
                current[lonIndex] = flipX? nLon - tLon - 1 : tLon;
                double d = gdra.getDataValueAsDouble(current, 0);
                if (Double.isNaN(d)) writer.write(NaNString);
                else if (isIntType)  writer.write("" + Math2.roundToLong(d));
                else if (isFloatType)writer.write("" + (float)d); //it isn't NaN
                else                 writer.write("" + d);
                writer.write(tLon == nLon - 1? '\n' : ' ');
            }
        }
        gdra.closeAndDelete();

        writer.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsEsriAscii done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }


    /**
     * This saves the requested data (must be lat- lon-based data only)  
     * as a grayscale GeoTIFF file.
     * For .geotiff, dataVariable queries can specify multiple longitude and latitude
     * values, but just one value for other dimensions.
     * GeotiffWriter requires that lons are +/-180 (because ESRI only accepts that range).
* Currently, the lons in the request can't be below and above 180  
* (below is fine; above is automatically shifted down).
* [future: do more extensive fixup].
     *
     * <p>javaDoc for the netcdf GeotiffWriter class isn't in standard javaDocs.
     * Try https://www.unidata.ucar.edu/software/netcdf-java/v2.2.20/javadocAll/ucar/nc2/geotiff/GeotiffWriter.html
     * or search Google for GeotiffWriter. 
     *
     * <p>Grayscale GeoTIFFs may not be very colorful, but they have an advantage
     * over color GeoTIFFs: the clear correspondence of the gray level of each pixel 
     * (0 - 255) to the original data allows programs to reconstruct the 
     * original data values, something that is not possible with color GeoTIFFS.
     *
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @param directory with a slash at the end in which to cache the file
     * @param fileName The file name with out the extension (e.g., myFile).
     *    The extension ".tif" will be added to create the output file name.
     * @return true of written ok; false if exception occurred (and written on image)
     * @throws Throwable 
     */
    public boolean saveAsGeotiff(String requestUrl, String userDapQuery, OutputStreamSource outputStreamSource,
        String directory, String fileName) throws Throwable {

        if (reallyVerbose) String2.log("Grid.saveAsGeotiff " + fileName);
        long time = System.currentTimeMillis();

        //handle axis request
        if (isAxisDapQuery(userDapQuery)) 
            throw new SimpleException("Error: " +
                "The GeoTIFF format is for latitude longitude data requests only.");

        //lon and lat are required; time is not required
        if (lonIndex < 0 || latIndex < 0) 
            throw new SimpleException("Error: " +
                "The GeoTIFF format is for latitude longitude data requests only.");

        //force to be pm180
//        makeLonPM180(true);

        //parse the userDapQuery and get the GridDataAccessor
        //this also tests for error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, true);  //rowMajor, convertToNaN
        if (gridDataAccessor.dataVariables().length > 1) 
            throw new SimpleException("Error: " +
                "The GeoTIFF format can only handle one data variable.");
        EDV edv = gridDataAccessor.dataVariables()[0];
        String dataName = edv.destinationName();

        //check that request meets ESRI restrictions

        //The geotiffWriter just throws non-helpful error messages if these requirements aren't met.
        PrimitiveArray lonPa = null, latPa = null;
        double minX = Double.NaN, maxX = Double.NaN, minY = Double.NaN, maxY = Double.NaN,
            lonAdjust = 0, lonSpacing = Double.NaN, latSpacing = Double.NaN;
        for (int av = 0; av < axisVariables.length; av++) {
            PrimitiveArray avpa = gridDataAccessor.axisValues(av);
            if (av == lonIndex) {
                lonPa = avpa;
                String error = lonPa.isEvenlySpaced();
                if (error.length() > 0) //size=1 is evenlySpaced
                    throw new SimpleException("Error: " +
                        "The dataset's longitude values aren't evenly spaced (as required by the GeoTIFF format):\n" + error);
                minX = lonPa.getNiceDouble(0);
                maxX = lonPa.getNiceDouble(lonPa.size() - 1);
                if (lonPa.size() > 1) //first calculate spacing  (may be negative)
                    lonSpacing = (maxX - minX) / (lonPa.size() - 1);
                if (minX > maxX) { //then deal with descending axis values
                    double d = minX; minX = maxX; maxX = d;}
                if (minX < 180 && maxX > 180)
                    throw new SimpleException("Error: " +
                        "For GeoTIFF requests, the dataset's longitude values can't be below and above 180.");
                if (minX >= 180) lonAdjust = -360;
                minX += lonAdjust;
                maxX += lonAdjust;
                if (minX < -180 || maxX > 180)
                    throw new SimpleException("Error: " +
                        "For GeoTIFF requests, the adjusted longitude values (" + 
                        minX + " to " + maxX + ") must be between -180 and 180.");
            } else if (av == latIndex) {
                latPa = avpa;
                String error = latPa.isEvenlySpaced();
                if (error.length() > 0) //size=1 is evenlySpaced
                    throw new SimpleException("Error: " +
                            "The dataset's latitude values aren't evenly spaced (as required by the GeoTIFF format):\n" + error);
                minY = latPa.getNiceDouble(0);
                maxY = latPa.getNiceDouble(latPa.size() - 1);
                if (latPa.size() > 1) //first calculate spacing (may be negative)
                    latSpacing = (maxY - minY) / (latPa.size() - 1);
                if (minY > maxY) { //then deal with descending axis values
                    double d = minY; minY = maxY; maxY = d;}
            } else {
                if (avpa.size() > 1)
                    throw new SimpleException("Error: " +
                        "For GeoTIFF requests, the " + 
                        axisVariables[av].destinationName() + " dimension's size must be 1."); 
            }
        }
        if ( Double.isNaN(lonSpacing) && !Double.isNaN(latSpacing)) lonSpacing = latSpacing;
        if (!Double.isNaN(lonSpacing) &&  Double.isNaN(latSpacing)) latSpacing = lonSpacing;
        if ( Double.isNaN(lonSpacing) &&  Double.isNaN(latSpacing)) 
            throw new SimpleException("Error: " +
                "For GeoTIFF requests, the latitude or longitude dimension size must be greater than 1."); 

        //lon and lat are ascending?    
//future: rearrange the data or get unidata to rearrange the data
        if (lonSpacing < 0 || latSpacing < 0) //see test in EDDGridFromDap.testPmelOscar
            throw new SimpleException("Error: " +
                "For GeoTIFF requests, the latitude and longitude values must be in ascending order."); 

//I don't think geotiff require lonSpacing=latSpacing.  Does it???
        //if (!Math2.almostEqual(3, lonSpacing, latSpacing))
        //    throw new SimpleException("Error: " +
        //        "For GeoTIFF requests, the longitude spacing (" + 
        //        lonSpacing + ") must equal the latitude spacing (" + latSpacing + ").");
        //request is ok and compatible with geotiffWriter!

        //save the data in a .nc file
//???I'm pretty sure the axis order can be lon,lat or lat,lon, but not certain.
//???The GeotiffWriter seems to be detecting lat and lon and reacting accordingly.
        String ncFullName = directory + fileName + "_tiff.nc";  //_tiff is needed because unused axes aren't saved
        if (!File2.isFile(ncFullName))
            saveAsNc(requestUrl, userDapQuery, ncFullName, 
                false, //this is necessary 
                lonAdjust);
//String2.log(NcHelper.dumpString(ncFullName, false));

        //attempt to create geotif via java netcdf libraries
        LatLonRect latLonRect = new LatLonRect(
            new LatLonPointImpl(minY, minX),
            new LatLonPointImpl(maxY, maxX));
        GeotiffWriter writer = new GeotiffWriter(directory + fileName + ".tif");
        writer.writeGrid(ncFullName, dataName, 0, 0, 
            true, //true=grayscale   color didn't work for me. and see javadocs above.
            latLonRect);
        writer.close();    

        //copy to outputStream
        if (!File2.copy(directory + fileName + ".tif", outputStreamSource.outputStream(""))) {
            //outputStream contentType already set,
            //so I can't go back to html and display error message
            //note than the message is thrown if user cancels the transmission; so don't email to me
            String2.log("Error while transmitting " + fileName + ".tif");
        }

        if (reallyVerbose) String2.log("  Grid.saveAsGeotiff done. TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
        return true;
    }

    /**
     * This writes the data to various types of images.
     * This requires a dataDapQuery (not an axisDapQuery) 
     * where just 1 or 2 of the dimensions be size &gt; 1.
     * One active dimension results in a graph.
     * Two active dimensions results in a map (one active data variable
     *   results in colored graph, two results in vector plot).
     *
     * <p>For transparentPng maps, the longitude and latitude dimensions must be evenly spaced.
     * (Only because the primary client, GoogleEarth assumes they are.)
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource
     * @param fileTypeName
     * @return true of written ok; false if exception occurred (and written on image)
     * @throws Throwable  if trouble. 
     */
    public boolean saveAsImage(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource, String fileTypeName) throws Throwable {
        if (reallyVerbose) String2.log("  EDDGrid.saveAsImage query=" + userDapQuery);
        long time = System.currentTimeMillis();

        //determine the image size
        int sizeIndex = 
            fileTypeName.startsWith(".small")? 0 :
            fileTypeName.startsWith(".medium")? 1 :
            fileTypeName.startsWith(".large")? 2 : 1;
        boolean pdf = fileTypeName.toLowerCase().endsWith("pdf");
        boolean png = fileTypeName.toLowerCase().endsWith("png");
        boolean transparentPng = fileTypeName.equals(".transparentPng");
        if (!pdf && !png) 
            throw new SimpleException("Error: " +
                "Unexpected image type=" + fileTypeName);
        int imageWidth, imageHeight;
        if (pdf) {
            imageWidth  = EDStatic.pdfWidths[ sizeIndex]; 
            imageHeight = EDStatic.pdfHeights[sizeIndex];
        } else {
            imageWidth  = EDStatic.imageWidths[sizeIndex]; 
            imageHeight = EDStatic.imageHeights[sizeIndex];
        }
        if (reallyVerbose) String2.log("  sizeIndex=" + sizeIndex + 
            " pdf=" + pdf + " imageWidth=" + imageWidth + " imageHeight=" + imageHeight);
        Object pdfInfo[] = null;
        BufferedImage bufferedImage = null;
        Graphics2D g2 = null;
        Color transparentColor = null;
        boolean ok = true;

        try {
            //reject axisDapQuery
            if (isAxisDapQuery(userDapQuery)) {
                AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
                throw new SimpleException("Error: " + 
                    "Requests for axis values can't be made into images.");
            }

            //modify the query to get no more data than needed
            StringArray reqDataNames = new StringArray();
            IntArray constraints     = new IntArray();
            parseDataDapQuery(userDapQuery, reqDataNames, constraints, false);

            //for now, just plot first 1 or 2 data variables
            int nDv = reqDataNames.size();
            EDV reqDataVars[] = new EDV[nDv];
            for (int dv = 0; dv < nDv; dv++) 
                reqDataVars[dv] = findDataVariableByDestinationName(reqDataNames.get(dv));

            //extract optional .graphicsSettings from userDapQuery
            //  xRange, yRange, color and colorbar information
            //  title2 -- a prettified constraint string 
            boolean drawLines = false, drawLinesAndMarkers = false, drawMarkers = false, 
                drawSticks = false, drawSurface = false, drawVectors = false; 
            Color color = Color.black;

            //for now, palette values are unset.
            String palette = "";
            String scale = "";
            double paletteMin = Double.NaN;
            double paletteMax = Double.NaN;
            String continuousS = "";
            int nSections = Integer.MAX_VALUE;

            double minX = Double.NaN, maxX = Double.NaN, minY = Double.NaN, maxY = Double.NaN;
            int nVars = 4;
            EDV vars[] = null; //set by .vars or lower
            int axisVarI[] = null, dataVarI[] = null; //set by .vars or lower
            String ampParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
            int markerType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
            int markerSize = GraphDataLayer.MARKER_SIZE_SMALL;
            double fontScale = 1, vectorStandard = Double.NaN;
            boolean drawLandAsMask = true;
            for (int ap = 0; ap < ampParts.length; ap++) {
                String ampPart = ampParts[ap];

                //.colorBar defaults: palette=""|continuous=C|scale=Linear|min=NaN|max=NaN|nSections=-1
                if (ampPart.startsWith(".colorBar=")) {
                    String pParts[] = String2.split(ampPart.substring(10), '|'); //subparts may be ""; won't be null
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0 && pParts[0].length() > 0) palette = pParts[0];  
                    if (pParts.length > 1 && pParts[1].length() > 0) continuousS = pParts[1].toLowerCase();
                    if (pParts.length > 2 && pParts[2].length() > 0) scale = pParts[2];
                    if (pParts.length > 3 && pParts[3].length() > 0) paletteMin = String2.parseDouble(pParts[3]);
                    if (pParts.length > 4 && pParts[4].length() > 0) paletteMax = String2.parseDouble(pParts[4]);
                    if (pParts.length > 5 && pParts[5].length() > 0) nSections  = String2.parseInt(pParts[5]);
                    if (reallyVerbose)
                        String2.log(".colorBar palette=" + palette + 
                            " continuousS=" + continuousS +
                            " scale=" + scale +                            
                            " min=" + paletteMin + " max=" + paletteMax + 
                            " nSections=" + nSections);                                

                //.color
                } else if (ampPart.startsWith(".color=")) {
                    int iColor = String2.parseInt(ampPart.substring(7));
                    if (iColor < Integer.MAX_VALUE) {
                        color = new Color(iColor);
                        if (reallyVerbose)
                            String2.log(".color=0x" + Integer.toHexString(iColor));
                    }

                //.draw 
                } else if (ampPart.startsWith(".draw=")) {
                    String gt = ampPart.substring(6);
                    //try to set an option to true
                    //ensure others are false in case of multiple .draw
                    drawLines = gt.equals("lines");
                    drawLinesAndMarkers = gt.equals("linesAndMarkers");
                    drawMarkers = gt.equals("markers");
                    drawSticks  = gt.equals("sticks"); 
                    drawSurface = gt.equals("surface");
                    drawVectors = gt.equals("vectors");

                //.font
                } else if (ampPart.startsWith(".font=")) {
                    String pParts[] = String2.split(ampPart.substring(6), '|'); //subparts may be ""; won't be null
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0) fontScale = String2.parseDouble(pParts[0]);
                    fontScale = Double.isNaN(fontScale)? 1 : fontScale < 0.1? 0.1 : fontScale > 10? 10 : fontScale;
                    if (reallyVerbose)
                        String2.log(".font= scale=" + fontScale);

                //.land 
                } else if (ampPart.startsWith(".land=")) {
                    String gt = ampPart.substring(6);
                    drawLandAsMask = !gt.equals("under");
                    if (reallyVerbose)
                        String2.log(".land= drawLandAsMask=" + drawLandAsMask);

                //.marker
                } else if (ampPart.startsWith(".marker=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|'); //subparts may be ""; won't be null
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0) markerType = String2.parseInt(pParts[0]);
                    if (pParts.length > 1) markerSize = String2.parseInt(pParts[1]);
                    if (markerType < 0 || markerType >= GraphDataLayer.MARKER_TYPES.length) 
                        markerType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
                    if (markerSize < 1 || markerSize > 50) markerSize = GraphDataLayer.MARKER_SIZE_SMALL;
                    if (reallyVerbose)
                        String2.log(".marker= type=" + markerType + " size=" + markerSize);

                //.vars    request should use this with values or don't use this; no defaults
                } else if (ampPart.startsWith(".vars=")) {
                    vars = new EDV[nVars];
                    axisVarI = new int[nVars]; Arrays.fill(axisVarI, -1);
                    dataVarI = new int[nVars]; Arrays.fill(dataVarI, -1);
                    String pParts[] = String2.split(ampPart.substring(6), '|');
                    for (int p = 0; p < nVars; p++) {
                        if (pParts.length > p && pParts[p].length() > 0) {
                            int ti = String2.indexOf(axisVariableDestinationNames(), pParts[p]);
                            if (ti >= 0) {
                                vars[p] = axisVariables[ti];
                                axisVarI[p] = ti;
                            } else if (reqDataNames.indexOf(pParts[p]) >= 0) {
                                ti = String2.indexOf(dataVariableDestinationNames(), pParts[p]);
                                vars[p] = dataVariables[ti];
                                dataVarI[p] = ti;
                            } else {
                                throw new SimpleException("Query error: " +
                                    ".var #" + p + "=" + pParts[p] + " isn't a valid variable name.");
                            }
                        }
                    }

                //.vec
                } else if (ampPart.startsWith(".vec=")) {
                    vectorStandard = String2.parseDouble(ampPart.substring(5));
                    if (reallyVerbose)
                        String2.log(".vec " + vectorStandard);

                //.xRange   (supported, but currently not created by the Make A Graph form)
                //  prefer set via xVar constratints
                } else if (ampPart.startsWith(".xRange=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|');
                    if (pParts.length > 0) minX = String2.parseDouble(pParts[0]);
                    if (pParts.length > 1) maxX = String2.parseDouble(pParts[1]);
                    if (reallyVerbose)
                        String2.log(".xRange min=" + minX + " max=" + maxX);

                //.yRange   (supported, but currently not created by the Make A Graph form)
                //  prefer set via yVar range
                } else if (ampPart.startsWith(".yRange=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|');
                    if (pParts.length > 0) minY = String2.parseDouble(pParts[0]);
                    if (pParts.length > 1) maxY = String2.parseDouble(pParts[1]);
                    if (reallyVerbose)
                        String2.log(".yRange min=" + minY + " max=" + maxY);

                //just to be clear: ignore any unrecognized .something 
                } else if (ampPart.startsWith(".")) {
                }
            }

            //figure out which axes are active (>1 value)
            IntArray activeAxes = new IntArray();
            for (int av = 0; av < axisVariables.length; av++)
                if (constraints.get(av * 3) < constraints.get(av * 3 + 2))
                    activeAxes.add(av);
            int nAAv = activeAxes.size();
            if (nAAv < 1 || nAAv > 2)
                throw new SimpleException("Query error: " +
                    "To draw a graph, either 1 or 2 axes must be active and have a range of values.");

            //figure out / validate graph set up
            //if .draw= was provided...
            int cAxisI = 0, cDataI = 0; //use them up as needed
            if (drawLines) {
                if (vars == null) {
                    vars = new EDV[nVars];
                    for (int v = 0; v < 2; v++) { //get 2 vars
                        if (nAAv > cAxisI)     vars[v] = axisVariables[activeAxes.get(cAxisI++)];
                        else if (nDv > cDataI) vars[v] = reqDataVars[cDataI++]; 
                        else throw new SimpleException("Query error: " +
                            "Too few active axes and/or data variables for .draw=lines.");
                    }
                } else {
                    //vars 0,1 must be valid (any type)
                    if (vars[0] == null) throw new SimpleException("Query error: " +
                        "For .draw=lines, .var #0 is required.");
                    if (vars[1] == null) throw new SimpleException("Query error: " +
                        "For .draw=lines, .var #1 is required.");
                }
                vars[2] = null;
                vars[3] = null;
            } else if (drawLinesAndMarkers || drawMarkers) {
                String what = drawLinesAndMarkers? "linesAndMarkers" : "markers";
                if (vars == null) {
                    vars = new EDV[nVars];
                    for (int v = 0; v < 3; v++) { //get 2 or 3 vars
                        if (nAAv > cAxisI)     vars[v] = axisVariables[activeAxes.get(cAxisI++)];
                        else if (nDv > cDataI) vars[v] = reqDataVars[cDataI++]; 
                        else if (v < 2) throw new SimpleException("Query error: " +
                            "Too few active axes and/or data variables for " +
                            ".draw=" + what + ".");
                    }
                } else {
                    //vars 0,1 must be valid (any type)
                    if (vars[0] == null) throw new SimpleException("Query error: " +
                        "For .draw=" + what + ", .var #0 is required.");
                    if (vars[1] == null) throw new SimpleException("Query error: " +
                        "For .draw=" + what + ", .var #1 is required.");
                }
                vars[3] = null;
            } else if (drawSticks) {
                if (vars == null) {
                    vars = new EDV[nVars];
                    //var0 must be axis
                    if (nAAv > 0) vars[0] = axisVariables[activeAxes.get(cAxisI++)];
                    else throw new SimpleException("Query error: " +
                        ".draw=sticks requires an active axis variable.");
                    //var 1,2 must be data
                    for (int v = 1; v <= 2; v++) { 
                        if (nDv > cDataI) vars[v] = reqDataVars[cDataI++]; 
                        else throw new SimpleException("Query error: " +
                            "Too few data variables to .draw=sticks.");
                    }
                } else {
                    //vars 0 must be axis, 1,2 must be data
                    if (axisVarI[0] < 0) throw new SimpleException("Query error: " +
                        "For .draw=sticks, .var #0 must be an axis variable.");
                    if (dataVarI[1] < 0) throw new SimpleException("Query error: " +
                        "For .draw=sticks, .var #1 must be a data variable.");
                    if (dataVarI[2] < 0) throw new SimpleException("Query error: " +
                        "For .draw=sticks, .var #2 must be a data variable.");
                }
                vars[3] = null;
            } else if (drawSurface) {
                if (vars == null) {
                    vars = new EDV[nVars];
                    //var0,1 must be axis  (currently must be lon,lat)
                    if (activeAxes.indexOf("" + lonIndex) >= 0 &&
                        activeAxes.indexOf("" + latIndex) >= 0) {
                        vars[0] = axisVariables[lonIndex];
                        vars[1] = axisVariables[latIndex];
                    } else throw new SimpleException("Query error: " +
                        ".draw=surface requires active longitude and latitude axes.");
                    //var 2 must be data
                    vars[2] = reqDataVars[cDataI++]; //at least one is valid
                } else {
                    //vars 0 must be axis, 1,2 must be data
                    if (axisVarI[0] != lonIndex || lonIndex < 0) 
                        throw new SimpleException("Query error: " +
                            "For .draw=surface, .var #0 must be longitude.");
                    if (axisVarI[1] != latIndex || latIndex < 0) 
                        throw new SimpleException("Query error: " +
                            "For .draw=surface, .var #1 must be latitude.");
                    if (dataVarI[2] < 0) 
                        throw new SimpleException("Query error: " +
                            "For .draw=surface, .var #2 must be a data variable.");
                }
                vars[3] = null;
            } else if (drawVectors) {
                if (vars == null) {
                    vars = new EDV[nVars];
                    //var0,1 must be axes
                    if (nAAv == 2) {
                        vars[0] = axisVariables[activeAxes.get(0)];
                        vars[1] = axisVariables[activeAxes.get(1)];
                    } else throw new SimpleException("Query error: " +
                        ".draw=vectors requires 2 active axis variables.");
                    //var2,3 must be data
                    if (nDv == 2) {
                        vars[2] = reqDataVars[0];
                        vars[3] = reqDataVars[1];
                    } else throw new SimpleException("Query error: " +
                        ".draw=vectors requires 2 data variables.");
                } else {
                    //vars 0,1 must be axes, 2,3 must be data
                    if (axisVarI[0] < 0) throw new SimpleException("Query error: " +
                        "For .draw=vectors, .var #0 must be an axis variable.");
                    if (axisVarI[1] < 0) throw new SimpleException("Query error: " +
                        "For .draw=vectors, .var #1 must be an axis variable.");
                    if (dataVarI[2] < 0) throw new SimpleException("Query error: " +
                        "For .draw=vectors, .var #2 must be a data variable.");
                    if (dataVarI[3] < 0) throw new SimpleException("Query error: " +
                        "For .draw=vectors, .var #3 must be a data variable.");
                }

            } else if (vars == null) {
                //neither .vars nor .draw were provided
                //detect from dap request  (favor linesAndMarkers)
                vars = new EDV[nVars];
                if (nAAv == 0) {
                    throw new SimpleException("Query error: " +
                        "At least 1 axis variable must be active and have a range of values.");
                } else if (nAAv == 1) {
                    drawLinesAndMarkers = true;
                    vars[0] = axisVariables[activeAxes.get(0)];
                    vars[1] = reqDataVars[0];
                    if (nDv > 1) vars[2] = reqDataVars[1];
                } else if (nAAv == 2) {  
                    //currently only if lon lat
                    if (lonIndex >= 0 && latIndex >= 0 &&
                        activeAxes.indexOf(lonIndex) >= 0 &&
                        activeAxes.indexOf(latIndex) >= 0) {
                        vars[0] = axisVariables[lonIndex];
                        vars[1] = axisVariables[latIndex];
                        vars[2] = reqDataVars[0];
                        if (reqDataVars.length >= 2) {
                            //draw vectors
                            drawVectors = true;
                            vars[3] = reqDataVars[1];
                        } else {
                            //draw surface
                            drawSurface = true;
                        }

                    } else throw new SimpleException("Query error: " +
                        "If 2 axes are active, they must be longitude and latitude.");
                } else {
                    throw new SimpleException("Query error: " +
                        "Either 1 or 2 axes must be active and have a range of values.");
                }
            } else {
                //.vars was provided, .draw wasn't
                //look for drawSurface
                if (axisVarI[0] >= 0 &&
                    axisVarI[1] >= 0 &&
                    dataVarI[2] >= 0 &&
                    dataVarI[3] >= 0) {
                    drawVectors = true;

                //look for drawVector(currently must have lon and lat)
                } else if (lonIndex >= 0 && latIndex >= 0 &&
                    activeAxes.indexOf(lonIndex) >= 0 && //lon or lat, in either order
                    activeAxes.indexOf(latIndex) >= 0 &&
                    dataVarI[2] >= 0) {
                    vars[0] = axisVariables[lonIndex]; //force lon 
                    vars[1] = axisVariables[latIndex]; //force lat
                    //vars[2] already set
                    drawSurface = true;
                    vars[3] = null;

                //drawMarker 
                } else {
                    //ensure marker compatible
                    if (axisVarI[0] < 0) 
                        throw new SimpleException("Query error: " +
                            ".var #0 must be an axis variable.");
                    if (axisVarI[1] < 0 && dataVarI[1] < 0) 
                        throw new SimpleException("Query error: " +
                            ".var #1 must be an axis or a data variable.");
                    axisVarI[1] = -1;
                    //var2 may be a dataVar or ""
                    vars[3] = null;
                    drawLinesAndMarkers = true;
                }
            }

            boolean isMap = vars[0] instanceof EDVLonGridAxis &&
                            vars[1] instanceof EDVLatGridAxis;
            boolean xIsTimeAxis = vars[0] instanceof EDVTimeGridAxis;
            boolean yIsTimeAxis = vars[1] instanceof EDVTimeGridAxis;
            int xAxisIndex = String2.indexOf(axisVariableDestinationNames(), vars[0].destinationName());
            int yAxisIndex = String2.indexOf(axisVariableDestinationNames(), vars[1].destinationName());

            if (transparentPng && !isMap)  //???future support drawSurface???
                throw new SimpleException("Error: " + 
                    "For .transparentPng, make the size of 2 axes greater than 1.");
            if (transparentPng && drawVectors) 
                throw new SimpleException("Error: " +
                    "For .transparentPng, just specify 1 data variable.");

            //if map or coloredSurface, modify the constraints so as to get only minimal amount of data
            //if 1D graph, no restriction
            int minXIndex = constraints.get(xAxisIndex * 3);
            int maxXIndex = constraints.get(xAxisIndex * 3 + 2);
            EDVGridAxis xAxisVar = axisVariables[xAxisIndex];
            if (Double.isNaN(minX)) minX = xAxisVar.destinationValue(minXIndex).getNiceDouble(0); 
            if (Double.isNaN(maxX)) maxX = xAxisVar.destinationValue(maxXIndex).getNiceDouble(0); 
            if (minX > maxX) {
                double d = minX; minX = maxX; maxX = d;}

            int minYIndex, maxYIndex;
            EDVGridAxis yAxisVar = yAxisIndex >= 0? axisVariables[yAxisIndex] : null;
            double minData = Double.NaN, maxData = Double.NaN;
             
            if (drawSurface || drawVectors) {  
                minYIndex = constraints.get(yAxisIndex * 3);
                maxYIndex = constraints.get(yAxisIndex * 3 + 2);
                if (Double.isNaN(minY)) minY = yAxisVar.destinationValue(minYIndex).getNiceDouble(0); 
                if (Double.isNaN(maxY)) maxY = yAxisVar.destinationValue(maxYIndex).getNiceDouble(0); 
                if (minY > maxY) {
                    double d = minY; minY = maxY; maxY = d;}

                if (transparentPng) {
                    int have = maxXIndex - minXIndex + 1;
                    int stride = constraints.get(xAxisIndex * 3 + 1);
                    imageWidth = DataHelper.strideWillFind(have, stride);
                    //protect against huge .png (and huge amount of data in memory)
                    if (imageWidth > 3601) {
                        stride = DataHelper.findStride(have, 3601);
                        imageWidth = DataHelper.strideWillFind(have, stride);
                        constraints.set(xAxisIndex * 3 + 1, stride);
                        if (reallyVerbose) String2.log("  xStride reduced to stride=" + stride);
                    }
                    have = maxYIndex - minYIndex + 1;
                    stride = constraints.get(yAxisIndex * 3 + 1);
                    imageHeight = DataHelper.strideWillFind(have, stride);
                    if (imageHeight > 1801) {
                        stride = DataHelper.findStride(have, 1801);
                        imageHeight = DataHelper.strideWillFind(have, stride);
                        constraints.set(yAxisIndex * 3 + 1, stride);
                        if (reallyVerbose) String2.log("  yStride reduced to stride=" + stride);
                    }
                } else {
                    //find size of map or graph
                    int activeWidth = imageWidth - 50; //decent guess for drawSurface
                    int activeHeight = imageHeight - 75;
                    if (drawVectors) {
                        double maxXY = Math.max(maxX - minX, maxY - minY);
                        double vecInc = SgtMap.suggestVectorIncrement(//e.g. 2 degrees
                            maxXY, Math.max(imageWidth, imageHeight), fontScale);

                        activeWidth  = Math.max(5, Math2.roundToInt((maxX - minX) / vecInc)); //e.g., 20 deg / 2 deg -> 10 
                        activeHeight = Math.max(5, Math2.roundToInt((maxY - minY) / vecInc));
                    } else { //currently drawSurface is always a map
                        int wh[] = SgtMap.predictGraphSize(fontScale, imageWidth, imageHeight,
                            minX, maxX, minY, maxY);
                        activeWidth = wh[0];
                        activeHeight = wh[1];
                    } 

                    //calculate/fix up stride so as to get enough data (but not too much)
                    int have = maxXIndex - minXIndex + 1;
                    int stride = DataHelper.findStride(have, activeWidth);
                    constraints.set(xAxisIndex * 3 + 1, stride);
                    if (reallyVerbose) 
                        String2.log("  xStride=" + stride + " activeHeight=" + activeHeight + 
                        " strideWillFind=" + DataHelper.strideWillFind(have, stride));

                    have = maxYIndex - minYIndex + 1;
                    stride = DataHelper.findStride(have, activeHeight);
                    constraints.set(yAxisIndex * 3 + 1, stride);
                    if (reallyVerbose)
                        String2.log("  yStride=" + stride + " activeHeight=" + activeHeight + 
                        " strideWillFind=" + DataHelper.strideWillFind(have, stride));
                }
            } 
         
            //units
            String xUnits = vars[0].units();
            String yUnits = vars[1].units();
            String zUnits = vars[2] == null? null : vars[2].units();
            String tUnits = vars[3] == null? null : vars[3].units();
            xUnits = xUnits == null? "" : " (" + xUnits + ")";
            yUnits = yUnits == null? "" : " (" + yUnits + ")";
            zUnits = zUnits == null? "" : " (" + zUnits + ")";
            tUnits = tUnits == null? "" : " (" + tUnits + ")";

            //get the desctiptive info for the axes with 1 value
            StringBuffer otherInfo = new StringBuffer();
            for (int av = 0; av < axisVariables.length; av++) {
                if (av != xAxisIndex && av != yAxisIndex) {
                    int ttIndex = constraints.get(av * 3);
                    EDVGridAxis axisVar = axisVariables[av];
                    if (otherInfo.length() > 0) 
                        otherInfo.append(", ");
                    double td = axisVar.destinationValue(ttIndex).getNiceDouble(0); 
                    if (av == lonIndex)
                        otherInfo.append(td + " E"); // didn't work
                    else if (av == latIndex) 
                        otherInfo.append(td + " N"); // didn't work
                    else if (av == timeIndex)
                        otherInfo.append(Calendar2.epochSecondsToIsoStringT(td) + "Z");
                    else 
                        otherInfo.append(axisVar.longName() + "=" + td + " " + axisVar.units());
                }
            }
            if (otherInfo.length() > 0) {
                otherInfo.insert(0, "(");
                otherInfo.append(")");
            }
                
            //get the data  
            StringArray newReqDataNames = new StringArray();
            int nBytesPerElement = 0;
            for (int v = 0; v < nVars; v++) {
                if (vars[v] != null && !(vars[v] instanceof EDVGridAxis)) {
                    newReqDataNames.add(vars[v].destinationName());
                    nBytesPerElement += drawSurface? 8: //grid always stores data in double[]
                        vars[v].destinationBytesPerElement();
                }
            }
            String newQuery = buildDapQuery(newReqDataNames, constraints);
            if (reallyVerbose) String2.log("  newQuery=" + newQuery);
            GridDataAccessor gda = new GridDataAccessor(this, requestUrl, newQuery, 
                yAxisVar == null, //Table needs row-major order, Grid needs column-major order
                true); //convertToNaN
            long requestNL = gda.totalIndex().size();
            if (requestNL * nBytesPerElement >= Integer.MAX_VALUE)
                throw new SimpleException("Error: " +
                    "The data request is too large.");
            int requestN = (int)requestNL;
            EDStatic.ensureMemoryAvailable(requestN * nBytesPerElement, datasetID); 
            Grid grid = null;
            Table table = null;
            GraphDataLayer graphDataLayer = null;
            ArrayList graphDataLayers = new ArrayList();
            String cptFullName = null;

            if (drawVectors) {
                //put the data in a Table   0=xAxisVar 1=yAxisVar 2=dataVar1 3=dataVar2
                table = new Table();
                PrimitiveArray xpa = PrimitiveArray.factory(vars[0].destinationDataTypeClass(), requestN, false);
                PrimitiveArray ypa = PrimitiveArray.factory(vars[1].destinationDataTypeClass(), requestN, false);
                PrimitiveArray zpa = PrimitiveArray.factory(vars[2].destinationDataTypeClass(), requestN, false);
                PrimitiveArray tpa = PrimitiveArray.factory(vars[3].destinationDataTypeClass(), requestN, false);
                table.addColumn(vars[0].destinationName(), xpa);
                table.addColumn(vars[1].destinationName(), ypa);
                table.addColumn(vars[2].destinationName(), zpa);
                table.addColumn(vars[3].destinationName(), tpa);
                while (gda.increment()) {
                    xpa.addDouble(gda.getAxisValueAsDouble(xAxisIndex));
                    ypa.addDouble(gda.getAxisValueAsDouble(yAxisIndex));
                    zpa.addDouble(gda.getDataValueAsDouble(0));
                    tpa.addDouble(gda.getDataValueAsDouble(1));
                }
                if (Double.isNaN(vectorStandard)) {
                    double stats1[] = zpa.calculateStats();
                    double stats2[] = tpa.calculateStats();
                    double lh[] = Math2.suggestLowHigh(0, Math.max(  //suggestLowHigh handles NaNs
                        Math.abs(stats1[PrimitiveArray.STATS_MAX]), 
                        Math.abs(stats2[PrimitiveArray.STATS_MAX])));
                    vectorStandard = lh[1];
                }

                String varInfo =  
                    vars[2].longName() + 
                    (zUnits.equals(tUnits)? "" : zUnits) +
                    ", " + 
                    vars[3].longName() + " (" + (float)vectorStandard +
                    (tUnits.length() == 0? "" : " " + vars[3].units()) + 
                    ")";

                //make a graphDataLayer with data  time series line
                graphDataLayer = new GraphDataLayer(
                    -1, //which pointScreen
                    0, 1, 2, 3, 1, //x,y,z1,z2,z3 column numbers
                    GraphDataLayer.DRAW_POINT_VECTORS,
                    xIsTimeAxis, yIsTimeAxis,
                    xAxisVar.longName() + xUnits, 
                    yAxisVar.longName() + yUnits, 
                    varInfo,
                    title(),             
                    otherInfo.toString(), 
                    "Data courtesy of " + institution(), 
                    table, null, null,
                    null, color, 
                    GraphDataLayer.MARKER_TYPE_NONE, 0,
                    vectorStandard,
                    GraphDataLayer.REGRESS_NONE);
                graphDataLayers.add(graphDataLayer);

            } else if (drawSticks) {
                //put the data in a Table   0=xAxisVar 1=uDataVar 2=vDataVar 
                table = new Table();
                PrimitiveArray xpa = PrimitiveArray.factory(vars[0].destinationDataTypeClass(), requestN, false);
                PrimitiveArray ypa = PrimitiveArray.factory(vars[1].destinationDataTypeClass(), requestN, false);
                PrimitiveArray zpa = PrimitiveArray.factory(vars[2].destinationDataTypeClass(), requestN, false);
                table.addColumn(vars[0].destinationName(), xpa);
                table.addColumn(vars[1].destinationName(), ypa);
                table.addColumn(vars[2].destinationName(), zpa);
                while (gda.increment()) {
                    xpa.addDouble(gda.getAxisValueAsDouble(xAxisIndex));
                    ypa.addDouble(gda.getDataValueAsDouble(0));
                    zpa.addDouble(gda.getDataValueAsDouble(1));
                }

                String varInfo =  
                    vars[1].longName() + 
                    (yUnits.equals(zUnits)? "" : yUnits) +
                    ", " + 
                    vars[2].longName() +  
                    (zUnits.length() == 0? "" : zUnits);

                //make a graphDataLayer with data  time series line
                graphDataLayer = new GraphDataLayer(
                    -1, //which pointScreen
                    0, 1, 2, 1, 1, //x,y,z1,z2,z3 column numbers
                    GraphDataLayer.DRAW_STICKS,
                    xIsTimeAxis, yIsTimeAxis,
                    xAxisVar.longName() + xUnits, 
                    varInfo, 
                    title(),
                    otherInfo.toString(), 
                    "",             
                    "Data courtesy of " + institution(), 
                    table, null, null,
                    null, color, 
                    GraphDataLayer.MARKER_TYPE_NONE, 0,
                    1,
                    GraphDataLayer.REGRESS_NONE);
                graphDataLayers.add(graphDataLayer);

            } else if (isMap || drawSurface) {
                //if .colorBar info didn't provide info, try to get defaults from vars[2] colorBarXxx attributes 
                if (vars[2] != null) { //it shouldn't be
                    Attributes colorVarAtts = vars[2].combinedAttributes();
                    if (palette.length() == 0)     palette     = colorVarAtts.getString("colorBarPalette");
                    if (scale.length() == 0)       scale       = colorVarAtts.getString("colorBarScale");
                    if (Double.isNaN(paletteMin))  paletteMin  = colorVarAtts.getDouble("colorBarMinimum");
                    if (Double.isNaN(paletteMax))  paletteMax  = colorVarAtts.getDouble("colorBarMaximum");
                    String ts = colorVarAtts.getString("colorBarContinuous");
                    if (continuousS.length() == 0 && ts != null) continuousS = String2.parseBoolean(ts)? "c" : "d"; //defaults to true
                }

                if (String2.indexOf(EDStatic.palettes, palette) < 0) palette   = "";
                if (String2.indexOf(EDV.VALID_SCALES, scale) < 0)    scale     = "Linear";
                if (nSections < 0 || nSections >= 100)               nSections = -1;
                boolean continuous = continuousS.startsWith("d")? false : true; 

                //put the data in a Grid, data in column-major order
                grid = new Grid();
                grid.data = new double[requestN];
                int po = 0;
                while (gda.increment()) 
                    grid.data[po++] = gda.getDataValueAsDouble(0);

                if (false) { //reallyVerbose) {
                    DoubleArray da = new DoubleArray(grid.data);
                    double stats[] = da.calculateStats();
                    String2.log("dataNTotal=" + da.size() + 
                        " dataN=" + stats[PrimitiveArray.STATS_N] +
                        " dataMin=" + stats[PrimitiveArray.STATS_MIN] +
                        " dataMax=" + stats[PrimitiveArray.STATS_MAX]);
                }

                //get the lon values
                PrimitiveArray tpa = gda.axisValues(xAxisIndex);
                int tn = tpa.size();
                grid.lon = new double[tn];
                for (int i = 0; i < tn; i++)
                    grid.lon[i] = tpa.getDouble(i);
                grid.lonSpacing = (grid.lon[tn - 1] - grid.lon[0]) / Math.max(1, tn - 1);

                //get the lat values
                tpa = gda.axisValues(yAxisIndex);
                tn = tpa.size();
                grid.lat = new double[tn];
                for (int i = 0; i < tn; i++)
                    grid.lat[i] = tpa.getDouble(i); 
                grid.latSpacing = (grid.lat[tn - 1] - grid.lat[0]) / Math.max(1, tn - 1);

                //cptFullName
                if (Double.isNaN(paletteMin) || Double.isNaN(paletteMax)) {
                    //if not specified, I have the right to change
                    DoubleArray da = new DoubleArray(grid.data);
                    double stats[] = da.calculateStats();
                    minData = stats[PrimitiveArray.STATS_MIN];
                    maxData = stats[PrimitiveArray.STATS_MAX];
                    if (maxData >= minData / -2 && 
                        maxData <= minData * -2) {
                        double td = Math.max(maxData, -minData);
                        minData = -td;
                        maxData = td;
                    }
                    double tRange[] = Math2.suggestLowHigh(minData, maxData);
                    minData = tRange[0];
                    maxData = tRange[1];
                    if (maxData >= minData / -2 && 
                        maxData <= minData * -2) {
                        double td = Math.max(maxData, -minData);
                        minData = -td;
                        maxData = td;
                    }
                    if (Double.isNaN(paletteMin)) paletteMin = minData;
                    if (Double.isNaN(paletteMax)) paletteMax = maxData;
                }
                if (paletteMin > paletteMax) {
                    double d = paletteMin; paletteMin = paletteMax; paletteMax = d;
                }
                if (paletteMin == paletteMax) {
                    double tRange[] = Math2.suggestLowHigh(paletteMin, paletteMax);
                    paletteMin = tRange[0];
                    paletteMax = tRange[1];
                }
                if (palette.length() == 0)
                    palette = Math2.almostEqual(3, -paletteMin, paletteMax)? "BlueWhiteRed" : "Rainbow";
                if (scale.length() == 0)       
                    scale = "Linear";
                cptFullName = CompoundColorMap.makeCPT(EDStatic.fullPaletteDirectory, 
                    palette, scale, paletteMin, paletteMax, nSections, continuous, 
                    EDStatic.fullCptCacheDirectory);

                //make a graphDataLayer with coloredSurface setup
                graphDataLayer = new GraphDataLayer(
                    -1, //which pointScreen
                    0, 1, 1, 1, 1, //x,y,z1,z2,z3 column numbers    irrelevant
                    GraphDataLayer.DRAW_COLORED_SURFACE, //AND_CONTOUR_LINE?
                    xIsTimeAxis, yIsTimeAxis,
                    xAxisVar.longName() + xUnits, //x,yAxisTitle  for now, always std units 
                    yAxisVar.longName() + yUnits, 
                    vars[2].longName() + zUnits, //boldTitle
                    title(),             
                    "",
                    "Data courtesy of " + institution(), 
                    null, grid, null,
                    new CompoundColorMap(cptFullName), color, //color is irrelevant 
                    -1, -1, //marker type, size
                    0, //vectorStandard
                    GraphDataLayer.REGRESS_NONE);
                graphDataLayers.add(graphDataLayer);

            } else {  //make graph with lines, linesAndMarkers, or markers
                //put the data in a Table   x,y,(z)
                table = new Table();
                PrimitiveArray xpa = PrimitiveArray.factory(vars[0].destinationDataTypeClass(), requestN, false);
                PrimitiveArray ypa = PrimitiveArray.factory(vars[1].destinationDataTypeClass(), requestN, false);
                PrimitiveArray zpa = vars[2] == null? null :
                                     PrimitiveArray.factory(vars[2].destinationDataTypeClass(), requestN, false);
                table.addColumn(vars[0].destinationName(), xpa);
                table.addColumn(vars[1].destinationName(), ypa);
                if (vars[2] != null) 
                    table.addColumn(vars[2].destinationName(), zpa);
                while (gda.increment()) {
                    xpa.addDouble(gda.getAxisValueAsDouble(xAxisIndex));
                    ypa.addDouble(yAxisIndex >= 0? gda.getAxisValueAsDouble(yAxisIndex) : 
                        gda.getDataValueAsDouble(0));
                    if (vars[2] != null) 
                        zpa.addDouble(gda.getDataValueAsDouble(yAxisIndex >= 0? 0 : 1)); //yAxisIndex>=0 is true if y is index
                }


                //make the colorbar
                CompoundColorMap colorMap = null;
                if (vars[2] != null) {
                    //if .colorBar info didn't provide info, try to get defaults from vars[2] colorBarXxx attributes 
                    Attributes colorVarAtts = vars[2].combinedAttributes();
                    if (palette.length() == 0)     palette     = colorVarAtts.getString("colorBarPalette");
                    if (scale.length() == 0)       scale       = colorVarAtts.getString("colorBarScale");
                    if (Double.isNaN(paletteMin))  paletteMin  = colorVarAtts.getDouble("colorBarMinimum");
                    if (Double.isNaN(paletteMax))  paletteMax  = colorVarAtts.getDouble("colorBarMaximum");
                    String ts = colorVarAtts.getString("colorBarContinuous");
                    if (continuousS.length() == 0 && ts != null) continuousS = String2.parseBoolean(ts)? "c" : "d"; //defaults to true

                    if (String2.indexOf(EDStatic.palettes, palette) < 0) palette   = "";
                    if (String2.indexOf(EDV.VALID_SCALES, scale) < 0)    scale     = "Linear";
                    if (nSections < 0 || nSections >= 100)               nSections = -1;
                    boolean continuous = continuousS.startsWith("d")? false : true; 

                    if (palette.length() == 0 || Double.isNaN(paletteMin) || Double.isNaN(paletteMax)) {
                        //set missing items based on z data
                        double zStats[] = table.getColumn(2).calculateStats();
                        if (zStats[PrimitiveArray.STATS_N] > 0) {
                            double minMax[];
                            if (vars[2] instanceof EDVTimeStamp) {
                                //???I think this is too crude. Smarter code elsewhere? Or handled by compoundColorMap?
                                double r20 = 
                                    (zStats[PrimitiveArray.STATS_MAX] -
                                     zStats[PrimitiveArray.STATS_MIN]) / 20;
                                minMax = new double[]{
                                    zStats[PrimitiveArray.STATS_MIN] - r20,
                                    zStats[PrimitiveArray.STATS_MAX] + r20};
                            } else {
                                minMax = Math2.suggestLowHigh(
                                    zStats[PrimitiveArray.STATS_MIN],
                                    zStats[PrimitiveArray.STATS_MAX]);
                            }

                            if (palette.length() == 0) {
                                if (minMax[1] >= minMax[0] / -2 && 
                                    minMax[1] <= minMax[0] * -2) {
                                    double td = Math.max(minMax[1], -minMax[0]);
                                    minMax[0] = -td;
                                    minMax[1] = td;
                                    palette = "BlueWhiteRed";
                                //} else if (minMax[0] >= 0 && minMax[0] < minMax[1] / 5) {
                                //    palette = "WhiteRedBlack";
                                } else {
                                    palette = "Rainbow";
                                }
                            }
                            if (Double.isNaN(paletteMin)) 
                                paletteMin = minMax[0];
                            if (Double.isNaN(paletteMax)) 
                                paletteMax = minMax[1];
                        }                                             
                    }
                    if (palette.length() == 0 || Double.isNaN(paletteMin) || Double.isNaN(paletteMax)) {
                        //don't create a colorMap
                        String2.log("Warning in EDDTable.saveAsImage: NaNs not allowed (zVar has no numeric data):" +
                            " palette=" + palette +
                            " paletteMin=" + paletteMin +
                            " paletteMax=" + paletteMax);
                    } else {
                        if (reallyVerbose)
                            String2.log("create colorBar palette=" + palette +
                                " continuous=" + continuous +
                                " scale=" + scale +
                                " min=" + paletteMin + " max=" + paletteMax +
                                " nSections=" + nSections);                                
                        if (vars[2] instanceof EDVTimeStamp)
                            colorMap = new CompoundColorMap(
                            EDStatic.fullPaletteDirectory, palette, false, //false= data is seconds
                            paletteMin, paletteMax, nSections, 
                            continuous, EDStatic.fullCptCacheDirectory);
                        else colorMap = new CompoundColorMap(
                            EDStatic.fullPaletteDirectory, palette, scale, 
                            paletteMin, paletteMax, nSections, 
                            continuous, EDStatic.fullCptCacheDirectory);
                    }
                }

                //make a graphDataLayer with data  time series line
                graphDataLayer = new GraphDataLayer(
                    -1, //which pointScreen
                    0, 1, vars[2] == null? 1 : 2, 1, 1, //x,y,z1,z2,z3 column numbers
                    drawLines? GraphDataLayer.DRAW_LINES :
                        drawMarkers? GraphDataLayer.DRAW_MARKERS :
                        GraphDataLayer.DRAW_MARKERS_AND_LINES,
                    xIsTimeAxis, yIsTimeAxis,
                    xAxisVar.longName() + xUnits, //x,yAxisTitle  for now, always std units 
                    vars[1].longName() + yUnits, 
                    vars[2] == null? title() : vars[2].longName() + zUnits,             
                    vars[2] == null? ""      : title(),
                    otherInfo.toString(), 
                    "Data courtesy of " + institution(), 
                    table, null, null,
                    colorMap, color, 
                    markerType, markerSize,
                    0, //vectorStandard
                    GraphDataLayer.REGRESS_NONE);
                graphDataLayers.add(graphDataLayer);
            } 

            //setup graphics2D
            String logoImageFile;
            //transparentPng will revise this below
            if (pdf) {
                fontScale *= 1.25 * fontScale; //SgtMap.PDF_FONTSCALE=1.5 is too big
                logoImageFile = EDStatic.highResLogoImageFile;
                pdfInfo = SgtUtil.createPdf(SgtUtil.PDF_PORTRAIT, 
                    imageWidth, imageHeight, outputStreamSource.outputStream("UTF-8"));
                g2 = (Graphics2D)pdfInfo[0];
            } else {
                fontScale *= sizeIndex <= 1? 1: 1.25;
                logoImageFile = sizeIndex <= 1? EDStatic.lowResLogoImageFile : EDStatic.highResLogoImageFile;
                bufferedImage = SgtUtil.getBufferedImage(imageWidth, imageHeight);
                g2 = (Graphics2D)bufferedImage.getGraphics();
            }

            if (drawSurface) {
                if (transparentPng) {
                    //fill with 128,128,128 --> later convert to transparent
                    //Not a great approach to the problem.
                    //Gray isn't used by any palette (except gray scale)
                    //and no data set has gray scale as default 
                    transparentColor = SgtMap.oceanColor;
                    g2.setColor(transparentColor);
                    g2.fillRect(0, 0, imageWidth, imageHeight);

                    //draw the map
                    SgtMap.makeCleanMap(minX, maxX, minY, maxY,
                        false,
                        grid, 1, 1, 0, //double gridScaleFactor, gridAltScaleFactor, gridAltOffset,
                        cptFullName, 
                        false, false, false,
                        g2, 0, 0, imageWidth, imageHeight); 
                } else {
                    SgtMap.makeMap(SgtUtil.LEGEND_BELOW,
                        EDStatic.legendTitle1, EDStatic.legendTitle2,
                        EDStatic.imageDir, logoImageFile,
                        minX, maxX, minY, maxY, 
                        drawLandAsMask,
                        true, //plotGridData 
                        grid, 1, 1, 0, //double gridScaleFactor, gridAltScaleFactor, gridAltOffset,
                        cptFullName,
                        vars[2].longName() + zUnits,
                        title(),
                        otherInfo.toString(),
                        "Data courtesy of " + institution(),
                        false, null, 1, 1, 1, "", null, "", "", "", "", "", //plot contour 
                        new ArrayList(),
                        g2, 0, 0, imageWidth, imageHeight,
                        0, //no boundaryResAdjust,
                        fontScale);
                }
            } else if (drawVectors || drawLines || drawLinesAndMarkers || drawMarkers || drawSticks) {
                if (isMap) 
                    SgtMap.makeMap(SgtUtil.LEGEND_BELOW,
                        EDStatic.legendTitle1, EDStatic.legendTitle2,
                        EDStatic.imageDir, logoImageFile,
                        minX, maxX, minY, maxY, 
                        drawLandAsMask,
                        false, null, 1, 1, 0, "", "", "", "", "", //plotGridData                
                        false, null, 1, 1, 1, "", null, "", "", "", "", "", //plot contour 
                        graphDataLayers,
                        g2, 0, 0, imageWidth, imageHeight,
                        0, //no boundaryResAdjust,
                        fontScale);
                else 
                    EDStatic.sgtGraph.makeGraph(
                        graphDataLayer.xAxisTitle, 
                        graphDataLayer.yAxisTitle, 
                        SgtUtil.LEGEND_BELOW, EDStatic.legendTitle1, EDStatic.legendTitle2,
                        EDStatic.imageDir, logoImageFile,
                        minX, maxX,
                        minY, maxY,
                        xIsTimeAxis, yIsTimeAxis, 
                        graphDataLayers,
                        g2, 0, 0, imageWidth, imageHeight,  1, //graph imageWidth/imageHeight
                        fontScale); 
            }
        } catch (Throwable t) {
            ok = false;
            try {
                //write exception info on image
                String msg = MustBe.getShortErrorMessage(t);
                String2.log(MustBe.throwableToString(t)); //log full message with stack trace
                double tFontScale = pdf? 1.25 : 1;
                int tHeight = Math2.roundToInt(tFontScale * 12);

                if (pdf) {
                    if (pdfInfo == null)
                        pdfInfo = SgtUtil.createPdf(SgtUtil.PDF_PORTRAIT, 
                            imageWidth, imageHeight, outputStreamSource.outputStream("UTF-8"));
                    if (g2 == null)
                        g2 = (Graphics2D)pdfInfo[0];
                } else {
                    //make a new image (I don't think pdf can work this way -- sent as created)
                    bufferedImage = SgtUtil.getBufferedImage(imageWidth, imageHeight);
                    g2 = (Graphics2D)bufferedImage.getGraphics();
                }
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                    RenderingHints.VALUE_ANTIALIAS_ON);
                g2.setClip(0, 0, imageWidth, imageHeight); //unset in case set by sgtGraph
                msg = String2.noLongLines(msg, (imageWidth * 10 / 6) / tHeight, "    ");
                String lines[] = msg.split("\\n"); //not String2.split which trims
                g2.setColor(Color.black);
                g2.setFont(new Font(EDStatic.fontFamily, Font.PLAIN, tHeight));
                int ty = tHeight * 2;
                for (int i = 0; i < lines.length; i++) {
                    g2.drawString(lines[i], tHeight, ty);
                    ty += tHeight + 2;
                }
            } catch (Throwable t2) {
                String2.log("ERROR2 while creating image:\n" + MustBe.throwableToString(t2));
                if (pdf) {
                    if (pdfInfo == null) throw t;
                } else {
                    if (bufferedImage == null) throw t;
                }
                //else fall through to close/save image below
            }
        }

        //save image
        if (pdf) {
            SgtUtil.closePdf(pdfInfo);
        } else {
            SgtUtil.saveAsTransparentPng(bufferedImage, transparentColor, 
                outputStreamSource.outputStream("")); 
        }

        outputStreamSource.outputStream("").flush(); //safety

        if (reallyVerbose) String2.log("  EDDGrid.saveAsImage done. TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
        return ok;
    }

    /**
     * This writes the axis or grid data to the outputStream in JSON 
     * (http://www.json.org/) format.
     * If no exception is thrown, the data was successfully written.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160].
     *   This method extracts the jsonp text to be prepended to the results (or null if none).
     *     See http://niryariv.wordpress.com/2009/05/05/jsonp-quickly/
     *     and http://bob.pythonmac.org/archives/2005/12/05/remote-json-jsonp/
     *     and http://www.insideria.com/2009/03/what-in-the-heck-is-jsonp-and.html .
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsJson(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

//currently, this writes a table. 
//Perhaps better to write nDimensional array?
        if (reallyVerbose) String2.log("  EDDGrid.saveAsJson"); 
        long time = System.currentTimeMillis();

        //did query include &.jsonp= ?
        String parts[] = getUserQueryParts(userDapQuery);
        String jsonp = String2.stringStartsWith(parts, ".jsonp="); //may be null
        if (jsonp != null) 
            jsonp = SSR.percentDecode(jsonp.substring(7));

        //get dataAccessor first, in case of error when parsing query
        boolean isAxisDapQuery = isAxisDapQuery(userDapQuery);
        AxisDataAccessor ada = null;
        GridDataAccessor gda = null;
        if (isAxisDapQuery) 
             ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
        else gda = new GridDataAccessor(this, requestUrl, userDapQuery, 
            true, false);   //rowMajor, convertToNaN (would be true, but TableWriterJson will do it)

        //write the data to the tableWriter
        TableWriter tw = new TableWriterJson(outputStreamSource, jsonp, true); //writeUnits 
        if (isAxisDapQuery) 
             saveAsTableWriter(ada, tw);
        else saveAsTableWriter(gda, tw);

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsJson done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");

    }

    /**
     * This writes grid data (not axis data) to the outputStream in Google Earth's 
     * .kml format (http://earth.google.com/).
     * If no exception is thrown, the data was successfully written.
     * For .kml, dataVariable queries can specify multiple longitude, latitude,
     * and time values, but just one value for other dimensions.
     * 
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160].
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @return true of written ok; false if exception occurred (and written on image)
     * @throws Throwable  if trouble. 
     */
    public boolean saveAsKml(
        String loggedInAs, String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) 
            String2.log("  EDDGrid.saveAsKml"); 
        long time = System.currentTimeMillis();
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //check that request meets .kml restrictions.
        //.transparentPng does some of these tests, but better to catch problems
        //  here than in GoogleEarth.

        //.kml not available for axis request
        //lon and lat are required; time is not required
        if (isAxisDapQuery(userDapQuery) || lonIndex < 0 || latIndex < 0) 
            throw new SimpleException("Error: " +
                "The .kml format is for latitude longitude data requests only.");

        //parse the userDapQuery
        //this also tests for error when parsing query
        StringArray tDestinationNames = new StringArray();
        IntArray tConstraints = new IntArray();
        parseDataDapQuery(userDapQuery, tDestinationNames, tConstraints, false);
        if (tDestinationNames.size() != 1) 
            throw new SimpleException("Query error: " +
                "The .kml format can only handle one data variable.");

        //find any &constraints (simplistic approach, but sufficient for here and hard to replace with getUserQueryParts)
        int ampPo = -1;
        if (userDapQuery != null) {
            ampPo = userDapQuery.indexOf('&');
            if (ampPo == -1) 
                ampPo = userDapQuery.indexOf("%26");  //shouldn't be.  but allow overly zealous percent encoding.
        }
        String percentEncodedAmpQuery = ampPo >= 0?   //so constraints can be used in reference urls in kml
            XML.encodeAsXML(userDapQuery.substring(ampPo)) : "";

        EDVTimeGridAxis timeEdv = null;
        PrimitiveArray timePa = null;
        double timeStartd = Double.NaN, timeStopd = Double.NaN;
        int nTimes = 0;
        for (int av = 0; av < axisVariables.length; av++) {
            if (av == lonIndex) {

            } else if (av == latIndex) {

            } else if (av == timeIndex) {
                timeEdv = (EDVTimeGridAxis)axisVariables[timeIndex];
                timePa = timeEdv.sourceValues().subset(tConstraints.get(av*3 + 0), 
                    tConstraints.get(av*3 + 1), tConstraints.get(av*3 + 2));
                timePa = timeEdv.toDestination(timePa);
                nTimes = timePa.size();
                timeStartd = timePa.getNiceDouble(0);
                timeStopd  = timePa.getNiceDouble(nTimes - 1);
                if (nTimes > 500) //arbitrary: prevents requests that would take too long to respond to
                    throw new SimpleException("Error: " +
                        "For .kml requests, the time dimension's size must be less than 500.");

            } else {
                if (tConstraints.get(av*3 + 0) != tConstraints.get(av*3 + 2))
                    throw new SimpleException("Error: " +
                        "For .kml requests, the " + 
                        axisVariables[av].destinationName() + " dimension's size must be 1."); 
            }
        }

        //lat lon info
        //lon and lat axis values don't have to be evenly spaced.
        //.transparentPng uses Sgt.makeCleanMap which projects data (even, e.g., Mercator)
        //so resulting .png will use a geographic projection.

        //although the Google docs say lon must be +-180, lon > 180 is sortof ok!
        EDVLonGridAxis lonEdv = (EDVLonGridAxis)axisVariables[lonIndex];
        EDVLatGridAxis latEdv = (EDVLatGridAxis)axisVariables[latIndex];

        int    totalNLon = lonEdv.sourceValues().size();
        int    lonStarti = tConstraints.get(lonIndex*3 + 0);
        int    lonStopi  = tConstraints.get(lonIndex*3 + 2);
        double lonStartd = lonEdv.destinationValue(lonStarti).getNiceDouble(0);
        double lonStopd  = lonEdv.destinationValue(lonStopi).getNiceDouble(0);
        if (lonStopd  <= -180 || lonStartd >= 360)
            throw new SimpleException("Error: " +
                "For .kml requests, there must be some longitude values must be between -180 and 360.");
        if (lonStartd < -180) {
            lonStarti = lonEdv.destinationToClosestSourceIndex(-180);
            lonStartd = lonEdv.destinationValue(lonStarti).getNiceDouble(0);
        }
        if (lonStopd > Math.min(lonStartd + 360, 360)) {
            lonStopi = lonEdv.destinationToClosestSourceIndex(Math.min(lonStartd + 360, 360));
            lonStopd = lonEdv.destinationValue(lonStopi).getNiceDouble(0);
        }
        int    lonMidi   = (lonStarti + lonStopi) / 2;
        double lonMidd   = lonEdv.destinationValue(lonMidi).getNiceDouble(0);
        double lonAverageSpacing = Math.abs(lonEdv.averageSpacing());

        int    totalNLat = latEdv.sourceValues().size();
        int    latStarti = tConstraints.get(latIndex*3 + 0);
        int    latStopi  = tConstraints.get(latIndex*3 + 2);
        double latStartd = latEdv.destinationValue(latStarti).getNiceDouble(0);
        double latStopd  = latEdv.destinationValue(latStopi).getNiceDouble(0);
        if (latStartd < -90 || latStopd > 90)
            throw new SimpleException("Error: " +
                "For .kml requests, the latitude values must be between -90 and 90.");
        int    latMidi   = (latStarti + latStopi) / 2;
        double latMidd   = latEdv.destinationValue(latMidi).getNiceDouble(0);
        double latAverageSpacing = Math.abs(latEdv.averageSpacing());

        if (lonStarti == lonStopi || latStarti == latStopi) 
            throw new SimpleException("Error: " +
                "For .kml requests, the lon and lat dimension sizes must be greater than 1."); 
        //request is ok and compatible with .kml request!

        String datasetUrl = tErddapUrl + "/" + dapProtocol + "/" + datasetID();
        String timeString = "";
        if (nTimes >= 1) timeString += Calendar2.epochSecondsToIsoStringT(Math.min(timeStartd, timeStopd)) + "Z";
        if (nTimes >= 2) 
            throw new SimpleException("Error: " +
                "For .kml requests, the time dimension size must be 1."); 
            //timeString += " through " +
            //Calendar2.epochSecondsToIsoStringT(Math.max(timeStartd, timeStopd)) + "Z";
        String brTimeString = timeString.length() == 0? "" : "Time: " + timeString + "<br />\n"; 

        //calculate doMax and get drawOrder
        int drawOrder = 1;
        int doMax = 1;  //max value of drawOrder for this dataset
        double tnLon = totalNLon;
        double tnLat = totalNLat;
        int txPo = Math2.roundToInt(lonStarti / tnLon); //at this level, the txPo'th x tile 
        int tyPo = Math2.roundToInt(latStarti / tnLat);
        while (Math.min(tnLon, tnLat) > 512) { //256 led to lots of artifacts and gaps at seams
            //This determines size of all tiles.
            //512 leads to smallest tile edge being >256.
            //256 here relates to minLodPixels 256 below (although Google example used 128 below)
            //and Google example uses tile sizes of 256x256.

            //go to next level
            tnLon /= 2;
            tnLat /= 2;
            doMax++;

            //if user requested lat lon range < this level, drawOrder is at least this level
            //!!!THIS IS TRICKY if user starts at some wierd subset (not full image).
            if (reallyVerbose) 
                String2.log("doMax=" + doMax + 
                "; cLon=" + (lonStopi - lonStarti + 1) + " <= 1.5*tnLon=" + (1.5*tnLon) + 
                "; cLat=" + (latStopi - latStarti + 1) + " <= 1.5*tnLat=" + (1.5*tnLat)); 
            if (lonStopi - lonStarti + 1 <= 1.5 * tnLon &&  //1.5 ~rounds to nearest drawOrder
                latStopi - latStarti + 1 <= 1.5 * tnLat) { 
                drawOrder++;
                txPo = Math2.roundToInt(lonStarti / tnLon); //at this level, this is the txPo'th x tile 
                tyPo = Math2.roundToInt(latStarti / tnLat);
                if (reallyVerbose)
                    String2.log("    drawOrder=" + drawOrder +  
                        " txPo=" + lonStarti + "/" + tnLon + "+" + txPo + 
                        " tyPo=" + latStarti + "/" + tnLat + "+" + tyPo);
            }
        }

        //calculate lonLatStride: 1 for doMax, 2 for doMax-1
        int lonLatStride = 1;
        for (int i = drawOrder; i < doMax; i++)
            lonLatStride *= 2;
        if (reallyVerbose) 
            String2.log("    final drawOrder=" + drawOrder +  
            " txPo=" + txPo + " tyPo=" + tyPo +
            " doMax=" + doMax + " lonLatStride=" + lonLatStride);

        //Based on http://code.google.com/apis/kml/documentation/kml_21tutorial.html#superoverlays
        //Was based on quirky example (but lots of useful info):
        //http://161.55.17.243/cgi-bin/pydap.cgi/AG/ssta/3day/AG2006001_2006003_ssta.nc.kml?LAYERS=AGssta
        //kml docs: http://earth.google.com/kml/kml_tags.html
        //CDATA is necessary for url's with queries
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
            outputStreamSource.outputStream("UTF-8"), "UTF-8"));
        writer.write(
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
            "<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n" +
            "<Document>\n" +
            //human-friendly, but descriptive, <name>
            //name is used as link title -- leads to <description> 
            "  <name>");
        if (drawOrder == 1) writer.write(
            XML.encodeAsXML(title()) + "</name>\n" +
            //<description appears in help balloon
            //<br /> is what kml/description documentation recommends
            "  <description><![CDATA[" + 
            brTimeString +
            "Data courtesy of: " + XML.encodeAsXML(institution()) + "<br />\n" +
            //link to download data
            "<a href=\"" + 
                datasetUrl + ".html?" + 
                   SSR.minimalPercentEncode(tDestinationNames.get(0)) + //XML.encodeAsXML doesn't work 
                "\">Download data from this dataset.</a><br />\n" +
            "    ]]></description>\n");
        else writer.write(drawOrder + "_" + txPo + "_" + tyPo + "</name>\n");

        //GoogleEarth says it just takes lon +/-180, but it does ok (not perfect) with 180.. 360.
        //If minLon>=180, it is easy to adjust the lon value references in the kml,
        //  but leave the userDapQuery for the .transparentPng unchanged.
        //lonAdjust is ESSENTIAL for proper work with lon > 180.
        //GoogleEarth doesn't select correct drawOrder region if lon > 180.
        double lonAdjust = Math.min(lonStartd, lonStopd) >= 180? -360 : 0;
        String llBox = 
            "      <west>"  + (Math.min(lonStartd, lonStopd) + lonAdjust) + "</west>\n" +
            "      <east>"  + (Math.max(lonStartd, lonStopd) + lonAdjust) + "</east>\n" +
            "      <south>" +  Math.min(latStartd, latStopd) + "</south>\n" +
            "      <north>" +  Math.max(latStartd, latStopd) + "</north>\n";


        //is nTimes <= 1?
        StringBuffer tQuery; 
        if (nTimes <= 1) {
            //the Region
            writer.write(
                //min Level Of Detail: minimum size (initially while zooming in) at which this region is made visible
                //see http://code.google.com/apis/kml/documentation/kmlreference.html#lod
                "  <Region>\n" +
                "    <Lod><minLodPixels>" + (drawOrder == 1? 2 : 256) + "</minLodPixels>" +
                         //"<maxLodPixels>" + (drawOrder == 1? -1 : 1024) + "</maxLodPixels>" + //doesn't work as expected
                   "</Lod>\n" +
                "    <LatLonAltBox>\n" +
                    llBox +
                "    </LatLonAltBox>\n" +
                "  </Region>\n");

            if (drawOrder < doMax) {
                //NetworkLinks to subregions (quadrant)
                tQuery = new StringBuffer(tDestinationNames.get(0)); //limited chars, no need to URLEncode
                for (int nl = 0; nl < 4; nl++) {
                    double tLonStartd   = nl < 2? lonStartd : lonMidd;
                    double tLonStopd    = nl < 2? lonMidd   : lonStopd;
                    int ttxPo = txPo*2 + (nl < 2? 0 : 1);
                    double tLatStartd   = Math2.odd(nl)? latMidd : latStartd;
                    double tLatStopd    = Math2.odd(nl)? latStopd : latMidd;
                    int ttyPo = tyPo*2 + (Math2.odd(nl)? 1 : 0);                    
                    double tLonAdjust = Math.min(tLonStartd, tLonStopd) >= 180? -360 : 0; //see comments for lonAdjust above

                    tQuery = new StringBuffer(tDestinationNames.get(0)); //limited chars, no need to URLEncode
                    for (int av = 0; av < axisVariables.length; av++) {
                        if (av == lonIndex)
                            tQuery.append("[(" + tLonStartd + "):(" + tLonStopd + ")]");
                        else if (av == latIndex)
                            tQuery.append("[(" + tLatStartd + "):(" + tLatStopd + ")]");
                        else if (av == timeIndex)
                            tQuery.append("[(" + timeString + ")]");
                        else tQuery.append("[" + tConstraints.get(av*3 + 0) + "]");
                    }                

                    writer.write(
                    "  <NetworkLink>\n" +
                    "    <name>" + drawOrder + "_" + txPo + "_" + tyPo + "_" + nl + "</name>\n" +
                    "    <Region>\n" +
                    "      <Lod><minLodPixels>256</minLodPixels>" +
                           //"<maxLodPixels>1024</maxLodPixels>" + //doesn't work as expected.
                          "</Lod>\n" +
                    "      <LatLonAltBox>\n" +
                    "        <west>"  + (Math.min(tLonStartd, tLonStopd) + tLonAdjust) + "</west>\n" +
                    "        <east>"  + (Math.max(tLonStartd, tLonStopd) + tLonAdjust) + "</east>\n" +
                    "        <south>" +  Math.min(tLatStartd, tLatStopd) + "</south>\n" +
                    "        <north>" +  Math.max(tLatStartd, tLatStopd) + "</north>\n" +
                    "      </LatLonAltBox>\n" +
                    "    </Region>\n" +
                    "    <Link>\n" +
                    "      <href>" + datasetUrl + ".kml?" + 
                              SSR.minimalPercentEncode(tQuery.toString()) + //XML.encodeAsXML doesn't work 
                              percentEncodedAmpQuery + 
                          "</href>\n" +
                    "      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
                    "    </Link>\n" +
                    "  </NetworkLink>\n");
                }
            }

            //the GroundOverlay which shows the current image
            tQuery = new StringBuffer(tDestinationNames.get(0)); //limited chars, no need to URLEncode
            for (int av = 0; av < axisVariables.length; av++) {
                if (av == lonIndex)
                    tQuery.append("[(" + lonStartd + "):" + lonLatStride + ":(" + lonStopd + ")]");
                else if (av == latIndex)
                    tQuery.append("[(" + latStartd + "):" + lonLatStride + ":(" + latStopd + ")]");
                else if (av == timeIndex)
                    tQuery.append("[(" + timeString + ")]");
                else tQuery.append("[" + tConstraints.get(av*3 + 0) + "]");
            }                
            writer.write(
                "  <GroundOverlay>\n" +
                //"    <name>" + XML.encodeAsXML(title()) + 
                //    (timeString.length() > 0? ", " + timeString : "") +
                //    "</name>\n" +
                "    <drawOrder>" + drawOrder + "</drawOrder>\n" +
                "    <Icon>\n" +
                "      <href>" + datasetUrl + ".transparentPng?" + 
                    SSR.minimalPercentEncode(tQuery.toString()) + //XML.encodeAsXML doesn't work 
                    percentEncodedAmpQuery + 
                     "</href>\n" +
                "    </Icon>\n" +
                "    <LatLonBox>\n" +
                        llBox +
                "    </LatLonBox>\n" +
                //"    <visibility>1</visibility>\n" +
                "  </GroundOverlay>\n");
        } /*else { 
            //nTimes >= 2, so make a timeline in Google Earth
            //Problem: I don't know what time range each image represents.
            //  Because I don't know what the timePeriod is for the dataset (e.g., 8day).
            //  And I don't know if the images overlap (e.g., 8day composites, every day)
            //  And if the stride>1, it is further unknown.
            //Solution (crummy): assume an image represents -1/2 time to previous image until 1/2 time till next image

            //get all the .dotConstraints
            String parts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
            StringBuffer dotConstraintsSB = new StringBuffer();
            for (int i = 0; i < parts.length; i++) {
                if (parts[i].startsWith(".")) {
                    if (dotConstraintsSB.size() > 0)
                        dotConstraintsSB.append("&");
                    dotConstraintsSB.append(parts[i]);
                }
            }        
            String dotConstraints = dotConstraintsSB.toString();

            IntArray tConstraints = (IntArray)gridDataAccessor.constraints().clone();
            int startTimeIndex = tConstraints.get(timeIndex * 3);
            int timeStride     = tConstraints.get(timeIndex * 3 + 1);
            int stopTimeIndex  = tConstraints.get(timeIndex * 3 + 2); 
            double preTime = Double.NaN;
            double nextTime = allTimeDestPa.getDouble(startTimeIndex);
            double currentTime = nextTime - (allTimeDestPa.getDouble(startTimeIndex + timeStride) - nextTime);
            for (int tIndex = startTimeIndex; tIndex <= stopTimeIndex; tIndex += timeStride) {
                preTime = currentTime;
                currentTime = nextTime;
                nextTime = tIndex + timeStride > stopTimeIndex? 
                    currentTime + (currentTime - preTime) :
                    allTimeDestPa.getDouble(tIndex + timeStride);
                //String2.log("  tIndex=" + tIndex + " preT=" + preTime + " curT=" + currentTime + " nextT=" + nextTime);
                //just change the time constraints; leave all others unchanged
                tConstraints.set(timeIndex * 3, tIndex);
                tConstraints.set(timeIndex * 3 + 1, 1);
                tConstraints.set(timeIndex * 3 + 2, tIndex); 
                String tDapQuery = buildDapQuery(tDestinationNames, tConstraints) + dotConstraints;
                writer.write(
                    //the kml link to the data 
                    "  <GroundOverlay>\n" +
                    "    <name>" + Calendar2.epochSecondsToIsoStringT(currentTime) + "Z" + "</name>\n" +
                    "    <Icon>\n" +
                    "      <href>" + 
                        datasetUrl + ".transparentPng?" + I changed this: was minimalPercentEncode()... tDapQuery + //XML.encodeAsXML isn't ok
                        "</href>\n" +
                    "    </Icon>\n" +
                    "    <LatLonBox>\n" +
                    "      <west>" + west + "</west>\n" +
                    "      <east>" + east + "</east>\n" +
                    "      <south>" + south + "</south>\n" +
                    "      <north>" + north + "</north>\n" +
                    "    </LatLonBox>\n" +
                    "    <TimeSpan>\n" +
                    "      <begin>" + Calendar2.epochSecondsToIsoStringT((preTime + currentTime)  / 2.0) + "Z</begin>\n" +
                    "      <end>"   + Calendar2.epochSecondsToIsoStringT((currentTime + nextTime) / 2.0) + "Z</end>\n" +
                    "    </TimeSpan>\n" +
                    "    <visibility>1</visibility>\n" +
                    "  </GroundOverlay>\n");
            }
        }*/
        if (drawOrder == 1) 
            writer.write(getKmlIconScreenOverlay());
        writer.write(
            "</Document>\n" +
            "</kml>\n");
        writer.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsKml done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
        return true;
    }

    /*
    public void saveAsKml(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsKml"); 
        long time = System.currentTimeMillis();

        //handle axis request
        if (isAxisDapQuery(userDapQuery)) 
            throw new SimpleException("Error: " +
                "The .kml format is for latitude longitude data requests only.");

        //lon and lat are required; time is not required
        if (lonIndex < 0 || latIndex < 0) 
            throw new SimpleException("Error: " +
                "The .kml format is for latitude longitude data requests only.");

        //parse the userDapQuery and get the GridDataAccessor
        //this also tests for error when parsing query
        GridDataAccessor gridDataAccessor = new GridDataAccessor(this, 
            requestUrl, userDapQuery, true, true);  //rowMajor, convertToNaN
        if (gridDataAccessor.dataVariables().length != 1) 
            throw new SimpleException("Error: " +
                "The .kml format can only handle one data variable.");
        StringArray tDestinationNames = new StringArray();
        tDestinationNames.add(gridDataAccessor.dataVariables()[0].destinationName());

        //check that request meets .kml restrictions.
        //.transparentPng does some of these tests, but better to catch problems
        //here than in GoogleEarth.
        int nTimes = 0;
        double firstTime = Double.NaN, lastTime = Double.NaN, timeSpacing = Double.NaN;
        PrimitiveArray lonPa = null, latPa = null, timePa = null, allTimeDestPa = null;
        EDVTimeGridAxis timeEdv = null;
        double lonAdjust = 0;
        for (int av = 0; av < axisVariables.length; av++) {
            PrimitiveArray avpa = gridDataAccessor.axisValues(av);
            if (av == lonIndex) {
                lonPa = avpa;

                //lon and lat axis values don't have to be evenly spaced.
                //.transparentPng uses Sgt.makeCleanMap which projects data (even, e.g., Mercator)
                //so resulting .png will use a geographic projection.

                //although the Google docs say lon must be +-180, lon > 180 is ok!
                //if (lonPa.getDouble(0) < 180 && lonPa.getDouble(lonPa.size() - 1) > 180)
                //    throw new SimpleException("Error: " +
                //    "For .kml requests, the longitude values can't be below and above 180.");

                //But if minLon>=180, it is easy to adjust the lon value references in the kml,
                //but leave the userDapQuery for the .transparentPng unchanged.
                if (lonPa.getDouble(0) >= 180)
                    lonAdjust = -360;
            } else if (av == latIndex) {
                latPa = avpa;
            } else if (av == timeIndex) {
                timeEdv = (EDVTimeGridAxis)axisVariables[timeIndex];
                allTimeDestPa = timeEdv.destinationValues();
                timePa = avpa;
                nTimes = timePa.size();
                if (nTimes > 500) //arbitrary: prevents requests that would take too long to respond to
                    throw new SimpleException("Error: " +
                        "For .kml requests, the time dimension's size must be less than 500.");
                firstTime = timePa.getDouble(0);
                lastTime = timePa.getDouble(nTimes - 1);
                if (nTimes > 1) 
                    timeSpacing = (lastTime - firstTime) / (nTimes - 1);
            } else {
                if (avpa.size() > 1)
                    throw new SimpleException("Error: " +
                        "For .kml requests, the " + 
                        axisVariables[av].destinationName() + " dimension's size must be 1."); 
            }
        }
        if (lonPa == null || latPa == null || lonPa.size() < 2 || latPa.size() < 2) 
            throw new SimpleException("Error: " +
                "For .kml requests, the lon and lat dimension sizes must be greater than 1."); 
        //request is ok and compatible with .kml request!

        //based on quirky example (but lots of useful info):
        //http://161.55.17.243/cgi-bin/pydap.cgi/AG/ssta/3day/AG2006001_2006003_ssta.nc.kml?LAYERS=AGssta
        //kml docs: http://earth.google.com/kml/kml_tags.html
        //CDATA is necessary for url's with queries
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
            outputStreamSource.outputStream("UTF-8"), "UTF-8"));
        double dWest = lonPa.getNiceDouble(0) + lonAdjust;
        double dEast = lonPa.getNiceDouble(lonPa.size() - 1) + lonAdjust;
        if (dWest > dEast) {  //it happens if axis is in descending order
            double td = dWest; dWest = dEast; dEast = td;}
        String west  = String2.genEFormat10(dWest);
        String east  = String2.genEFormat10(dEast);

        double dSouth = latPa.getNiceDouble(0);
        double dNorth = latPa.getNiceDouble(latPa.size() - 1);
        if (dSouth > dNorth) { //it happens if axis is in descending order
            double td = dSouth; dSouth = dNorth; dNorth = td; }        
        String south  = String2.genEFormat10(dSouth);
        String north  = String2.genEFormat10(dNorth);
        String datasetUrl = tErddapUrl + "/" + dapProtocol + "/" + datasetID();
        String timeString = nTimes == 0? "" :
            nTimes == 1? Calendar2.epochSecondsToIsoStringT(firstTime) :
              Calendar2.epochSecondsToIsoStringT(firstTime) + " through " + 
              Calendar2.epochSecondsToIsoStringT(lastTime);
        String brTimeString = timeString.length() == 0? "" : "Time: " + timeString + "<br />\n"; 
        writer.write(
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
            "<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n" +
            "<Document>\n" +
            //human-friendly, but descriptive, <name>
            //name is used as link title -- leads to <description> 
            "  <name>" + XML.encodeAsXML(title()) + "</name>\n" +
            //<description appears in help balloon
            //<br /> is what kml/description documentation recommends
            "  <description><![CDATA[" + 
            brTimeString +
            "Data courtesy of: " + XML.encodeAsXML(institution()) + "<br />\n" +
            //link to download data
            "<a href=\"" + datasetUrl + ".html?" + SSR.minimalPercentEncode(userDapQuery) + //XML.encodeAsXML isn't ok
                "\">Download data from this dataset.</a><br />\n" +
            "    ]]></description>\n");

        //is nTimes <= 1?
        if (nTimes <= 1) {
            //no timeline in Google Earth
            writer.write(
                //the kml link to the data 
                "  <GroundOverlay>\n" +
                "    <name>" + title() + 
                    (timeString.length() > 0? ", " + timeString : "") +
                    "</name>\n" +
                "    <Icon>\n" +
                "      <href>" + 
                    datasetUrl + ".transparentPng?" + SSR.minimalPercentEncode(userDapQuery) + //XML.encodeAsXML isn't ok
                    "</href>\n" +
                "    </Icon>\n" +
                "    <LatLonBox>\n" +
                "      <west>" + west + "</west>\n" +
                "      <east>" + east + "</east>\n" +
                "      <south>" + south + "</south>\n" +
                "      <north>" + north + "</north>\n" +
                "    </LatLonBox>\n" +
                "    <visibility>1</visibility>\n" +
                "  </GroundOverlay>\n");
        } else { 
            //nTimes >= 2, so make a timeline in Google Earth
            //Problem: I don't know what time range each image represents.
            //  Because I don't know what the timePeriod is for the dataset (e.g., 8day).
            //  And I don't know if the images overlap (e.g., 8day composites, every day)
            //  And if the stride>1, it is further unknown.
            //Solution (crummy): assume an image represents -1/2 time to previous image until 1/2 time till next image

            //get all the .dotConstraints
            String parts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
            StringBuffer dotConstraintsSB = new StringBuffer();
            for (int i = 0; i < parts.length; i++) {
                if (parts[i].startsWith(".")) {
                    if (dotConstraintsSB.size() > 0)
                        dotConstraintsSB.append("&");
                    dotConstraintsSB.append(parts[i]);
                }
            }        
            String dotConstraints = dotConstraintsSB.toString();

            IntArray tConstraints = (IntArray)gridDataAccessor.constraints().clone();
            int startTimeIndex = tConstraints.get(timeIndex * 3);
            int timeStride     = tConstraints.get(timeIndex * 3 + 1);
            int stopTimeIndex  = tConstraints.get(timeIndex * 3 + 2); 
            double preTime = Double.NaN;
            double nextTime = allTimeDestPa.getDouble(startTimeIndex);
            double currentTime = nextTime - (allTimeDestPa.getDouble(startTimeIndex + timeStride) - nextTime);
            for (int tIndex = startTimeIndex; tIndex <= stopTimeIndex; tIndex += timeStride) {
                preTime = currentTime;
                currentTime = nextTime;
                nextTime = tIndex + timeStride > stopTimeIndex? 
                    currentTime + (currentTime - preTime) :
                    allTimeDestPa.getDouble(tIndex + timeStride);
                //String2.log("  tIndex=" + tIndex + " preT=" + preTime + " curT=" + currentTime + " nextT=" + nextTime);
                //just change the time constraints; leave all others unchanged
                tConstraints.set(timeIndex * 3, tIndex);
                tConstraints.set(timeIndex * 3 + 1, 1);
                tConstraints.set(timeIndex * 3 + 2, tIndex); 
                String tDapQuery = buildDapQuery(tDestinationNames, tConstraints) + dotConstraints;
                writer.write(
                    //the kml link to the data 
                    "  <GroundOverlay>\n" +
                    "    <name>" + Calendar2.epochSecondsToIsoStringT(currentTime) + "Z" + "</name>\n" +
                    "    <Icon>\n" +
                    "      <href>" + 
                        datasetUrl + ".transparentPng?" + SSR.minimalPercentEncode(tDapQuery) + //XML.encodeAsXML isn't ok
                        "</href>\n" +
                    "    </Icon>\n" +
                    "    <LatLonBox>\n" +
                    "      <west>" + west + "</west>\n" +
                    "      <east>" + east + "</east>\n" +
                    "      <south>" + south + "</south>\n" +
                    "      <north>" + north + "</north>\n" +
                    "    </LatLonBox>\n" +
                    "    <TimeSpan>\n" +
                    "      <begin>" + Calendar2.epochSecondsToIsoStringT((preTime + currentTime)  / 2.0) + "Z</begin>\n" +
                    "      <end>"   + Calendar2.epochSecondsToIsoStringT((currentTime + nextTime) / 2.0) + "Z</end>\n" +
                    "    </TimeSpan>\n" +
                    "    <visibility>1</visibility>\n" +
                    "  </GroundOverlay>\n");
            }
        }
        writer.write(
            getKmlIconScreenOverlay() +
            "</Document>\n" +
            "</kml>\n");
        writer.flush(); //essential

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsKml done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }
    */
    /**
     * This writes the grid from this dataset to the outputStream in 
     * Matlab .mat format.
     * This writes the lon values as they are currently in this grid
     *    (e.g., +-180 or 0..360).
     * If no exception is thrown, the data was successfully written.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percent-encoded (shouldn't be null),
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable 
     */
    public void saveAsMatlab(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsMatlab");
        long time = System.currentTimeMillis();

        //handle axisDapQuery
        if (isAxisDapQuery(userDapQuery)) {
            //this doesn't write attributes because .mat files don't store attributes
            //get axisDataAccessor first, in case of error when parsing query
            AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);

            //make the table
            Table table = new Table();
            int nRAV = ada.nRequestedAxisVariables();
            for (int av = 0; av < nRAV; av++)
                table.addColumn( 
                    ada.axisVariables(av).destinationName(),
                    ada.axisValues(av));
            //don't call table.ensureColumnsAreSameSize();  leave them different lengths

            //then get the modified outputStream
            DataOutputStream dos = new DataOutputStream(outputStreamSource.outputStream(""));
            table.saveAsMatlab(dos, datasetID());
            dos.flush(); //essential

            if (reallyVerbose) String2.log("  EDDGrid.saveAsMatlab axis done.\n");
            return;
        }

        //get gridDataAccessor first, in case of error when parsing query
        GridDataAccessor mainGda = new GridDataAccessor(this, requestUrl, userDapQuery, 
            false, //Matlab is one of the few drivers that needs column-major order
            true); //convertToNaN

        //Make sure no String data and that gridsize isn't > Integer.MAX_VALUE bytes (Matlab's limit)
        EDV tDataVariables[] = mainGda.dataVariables();
        int nAv = axisVariables.length;
        int ntDv = tDataVariables.length;
        byte structureNameInfo[] = Matlab.nameInfo(datasetID); //structure name
        int largest = 1; //find the largest data item nBytesPerElement
        int cumSize = //see 1-32
            16 + //for array flags
            16 + //my structure is always 2 dimensions 
            structureNameInfo.length +
            8 + //field name length (for all fields)
            8 + (nAv + ntDv) * 32; //field names

        PrimitiveArray avPa[] = new PrimitiveArray[nAv];
        NDimensionalIndex avNDIndex[] = new NDimensionalIndex[nAv];
        for (int av = 0; av < nAv; av++) {
            avPa[av] = mainGda.axisValues[av];
            avNDIndex[av] = Matlab.make2DNDIndex(avPa[av].size());
            cumSize += 8 + Matlab.sizeOfNDimensionalArray("", //names are done separately
                avPa[av].getElementType(), avNDIndex[av]);
        }

        GridDataAccessor tGda[] = new GridDataAccessor[ntDv];
        NDimensionalIndex dvNDIndex[] = new NDimensionalIndex[ntDv];
        String arrayQuery = buildDapArrayQuery(mainGda.constraints());
        for (int dv = 0; dv < ntDv; dv++) {
            if (tDataVariables[dv].destinationDataTypeClass() == String.class) 
                //can't do String data because you need random access to all values
                //that could be a memory nightmare
                //so just don't allow it
                throw new SimpleException("Error: " +
                    "Matlab files can't have String data.");
            largest = Math.max(largest, 
                tDataVariables[dv].destinationBytesPerElement());

            //make a GridDataAccessor for this dataVariable
            String tUserDapQuery = tDataVariables[dv].destinationName() + arrayQuery;
            tGda[dv] = new GridDataAccessor(this, requestUrl, tUserDapQuery, 
                false, //Matlab is one of the few drivers that needs column-major order
                true); //convertToNaN
            dvNDIndex[dv] = tGda[dv].totalIndex();
            if (dvNDIndex[dv].nDimensions() == 1)
                dvNDIndex[dv] = Matlab.make2DNDIndex(dvNDIndex[dv].shape()[0]);
            cumSize += 8 + Matlab.sizeOfNDimensionalArray("", //names are done separately
                tDataVariables[dv].destinationDataTypeClass(), dvNDIndex[dv]);
        }
        if (cumSize >= Integer.MAX_VALUE)
            throw new SimpleException("Error: " +
                "The requested data (" + 
                (cumSize / 1048576) + 
                " MB) is greater than Matlab's limit (" +
                (Integer.MAX_VALUE / 1048576) + " MB)."); 

        //then get the modified outputStream
        DataOutputStream stream = new DataOutputStream(outputStreamSource.outputStream(""));

        //write the header
        Matlab.writeMatlabHeader(stream);

        //write the miMatrix dataType and nBytes
        stream.writeInt(Matlab.miMATRIX);        //dataType
        stream.writeInt(cumSize); 

        //write array flags 
        stream.writeInt(Matlab.miUINT32); //dataType
        stream.writeInt(8);  //fixed nBytes of data
        stream.writeInt(Matlab.mxSTRUCT_CLASS); //array flags  
        stream.writeInt(0); //reserved; ends on 8 byte boundary

        //write structure's dimension array  
        stream.writeInt(Matlab.miINT32); //dataType
        stream.writeInt(2 * 4);  //nBytes
        //matlab docs have 2,1, octave has 1,1. 
        //Think of structure as one row of a table, where elements are entire arrays:  e.g., sst.lon sst.lat sst.sst.
        //Having multidimensions (e.g., 2 here) lets you have additional rows, e.g., sst(2).lon sst(2).lat sst(2).sst.
        //So 1,1 makes sense.
        stream.writeInt(1);  
        stream.writeInt(1);
         
        //write structure name 
        stream.write(structureNameInfo, 0, structureNameInfo.length);

        //write length for all field names (always 32)  (short form)
        stream.writeShort(4);                //nBytes
        stream.writeShort(Matlab.miINT32);    //dataType
        stream.writeInt(32);                 //nBytes per field name

        //write the structure's field names (each 32 bytes)
        stream.writeInt(Matlab.miINT8);    //dataType
        stream.writeInt((nAv + ntDv) * 32);      //nBytes per field name
        String nulls = String2.makeString('\u0000', 32);
        for (int av = 0; av < nAv; av++)
            stream.write(String2.toByteArray(
                String2.noLongerThan(axisVariables[av].destinationName(), 31) + nulls), 0, 32);
        for (int dv = 0; dv < ntDv; dv++) 
            stream.write(String2.toByteArray(
                String2.noLongerThan(tDataVariables[dv].destinationName(), 31) + nulls), 0, 32);

        //write the axis miMatrix
        for (int av = 0; av < nAv; av++)
            Matlab.writeNDimensionalArray(stream, "", //name is written above
                avPa[av], avNDIndex[av]);

        //make the data miMatrix
        for (int dv = 0; dv < ntDv; dv++) 
            writeNDimensionalMatlabArray(stream, "",  //name is written above 
                tGda[dv], dvNDIndex[dv]); 

        //this doesn't write attributes because .mat files don't store attributes
        stream.flush(); //essential

        if (reallyVerbose) String2.log("  EDDGrid.saveAsMatlab done. TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * This writes gda's dataVariable[0] as a "matrix" to the Matlab stream.
     *
     * @param stream the stream for the Matlab file
     * @param name usually the destinationName. But inside a Structure, use "".
     * @param gda provides access to just *one* of the data variables. gda must be column-major.
     * @param ndIndex the 2+ dimension NDimensionalIndex
     * @throws Throwable if trouble
     */
    public void writeNDimensionalMatlabArray(DataOutputStream stream, String name, 
        GridDataAccessor gda, NDimensionalIndex ndIndex) throws Throwable {

        if (gda.rowMajor())
            throw new SimpleException("Internal error in " +
                "EDDGrid.writeNDimensionalMatlabArray: the GridDataAccessor must be column-major.");

        //do the first part
        EDV edv = gda.dataVariables()[0];
        Class elementType = edv.destinationDataTypeClass();
        if (elementType == String.class) 
            //can't do String data because you need random access to all values
            //that could be a memory nightmare
            //so just don't allow it
            throw new SimpleException("Error: " +
                "Matlab files can't have String data.");
        int nDataBytes = Matlab.writeNDimensionalArray1(stream, name, 
            elementType, ndIndex);

        //do the second part here 
        //note:  calling increment() on column-major gda returns data in column-major order
        if      (elementType == double.class) while (gda.increment()) stream.writeDouble(gda.getDataValueAsDouble(0)); 
        else if (elementType == float.class)  while (gda.increment()) stream.writeFloat( gda.getDataValueAsFloat(0));
        else if (elementType == long.class)   while (gda.increment()) stream.writeDouble(gda.getDataValueAsDouble(0)); 
        else if (elementType == int.class)    while (gda.increment()) stream.writeInt(   gda.getDataValueAsInt(0));
        else if (elementType == short.class)  while (gda.increment()) stream.writeShort( gda.getDataValueAsInt(0));
        else if (elementType == byte.class)   while (gda.increment()) stream.writeByte(  gda.getDataValueAsInt(0));
        else if (elementType == char.class)   while (gda.increment()) stream.writeChar(  gda.getDataValueAsInt(0));
        //else if (elementType == String.class) ...

        //pad data to 8 byte boundary
        int i = nDataBytes % 8;
        while ((i++ % 8) != 0)
            stream.write(0); //0 padded to 8 byte boundary
    }

    /**
     * Save the grid data in a netCDF .nc file.
     * This overwrites any existing file of the specified name.
     * This makes an effort not to create a partial file if there is an error.
     * If no exception is thrown, the file was successfully created.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param fullFileName the name for the file (including directory and extension)
     * @param keepUnusedAxes if true, axes with size=1 will be stored in the file.
     *    If false, axes with size=1 will not be stored in the file (geotiff needs this).
     * @param lonAdjust the value to be added to all lon values (e.g., 0 or -360).
     * @throws Throwable 
     */
    public void saveAsNc(String requestUrl, String userDapQuery, 
        String fullFileName, boolean keepUnusedAxes,
        double lonAdjust) throws Throwable {
        if (reallyVerbose) String2.log("  EDDGrid.saveAsNc"); 
        long time = System.currentTimeMillis();

        //delete any existing file
        File2.delete(fullFileName);

        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //handle axisDapQuery
        if (isAxisDapQuery(userDapQuery)) {
            //get axisDataAccessor first, in case of error when parsing query
            AxisDataAccessor ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
            int nRAV = ada.nRequestedAxisVariables();

            //write the data
            //items determined by looking at a .nc file; items written in that order 
            NetcdfFileWriteable nc = NetcdfFileWriteable.createNew(fullFileName + randomInt,
                false); //false says: create a new file and don't fill with missing_values
            try {

                //define the dimensions
                Array axisArrays[] = new Array[nRAV];
                for (int av = 0; av < nRAV; av++) {
                    String destName = ada.axisVariables(av).destinationName();
                    PrimitiveArray pa = ada.axisValues(av);
                    Dimension dimension = nc.addDimension(destName, pa.size());
                    axisArrays[av] = Array.factory(
                        pa.getElementType(),
                        new int[]{pa.size()},
                        pa.toObjectArray());
                    nc.addVariable(destName, 
                        NcHelper.getDataType(pa.getElementType()), 
                        new Dimension[]{dimension}); 

                    //write axis attributes
                    Attributes at = ada.axisAttributes(av);
                    String names[] = at.getNames();
                    for (int i = 0; i < names.length; i++) {
                        nc.addVariableAttribute(destName,  
                            NcHelper.getAttribute(names[i], at.get(names[i])));
                    }
                }

                //write global attributes
                String names[] = ada.globalAttributes().getNames();
                for (int i = 0; i < names.length; i++) { 
                    nc.addGlobalAttribute(
                        NcHelper.getAttribute(names[i], ada.globalAttributes().get(names[i])));
                }

                //leave "define" mode
                nc.create();

                //write the axis values
                for (int av = 0; av < nRAV; av++) 
                    nc.write(ada.axisVariables(av).destinationName(), axisArrays[av]);

                //if close throws Throwable, it is trouble
                nc.close(); //it calls flush() and doesn't like flush called separately

                //rename the file to the specified name
                File2.rename(fullFileName + randomInt, fullFileName);

                //diagnostic
                if (reallyVerbose) String2.log("  EDDGrid.saveAsNc axis done\n");
                //String2.log(NcHelper.dumpString(directory + name + ext, false));

            } catch (Throwable t) {
                //try to close the file
                try {
                    nc.close(); //it calls flush() and doesn't like flush called separately
                } catch (Throwable t2) {
                    //don't care
                }

                //delete the partial file
                File2.delete(fullFileName);

                throw t;
            }
            return;
        }

        //get gridDataAccessor first, in case of error when parsing query
        GridDataAccessor mainGda = new GridDataAccessor(this, requestUrl, userDapQuery, 
            true, false);  //rowMajor, convertToNaN 
        EDV tDataVariables[] = mainGda.dataVariables();

        //ensure enough memory to get each variable
        long dvRequestSize[] = new long[tDataVariables.length];
        long tMaxMemory = 0;
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            dvRequestSize[dv] = mainGda.totalIndex().size() * tDataVariables[dv].destinationBytesPerElement();
            tMaxMemory = Math.max(tMaxMemory, dvRequestSize[dv]);
        }
        EDStatic.ensureMemoryAvailable(tMaxMemory, datasetID);

        //write the data
        //items determined by looking at a .nc file; items written in that order 
        NetcdfFileWriteable nc = NetcdfFileWriteable.createNew(fullFileName + randomInt,
            false); //false says: create a new file and don't fill with missing_values
        try {

            //find active axes
            IntArray activeAxes = new IntArray();
            for (int av = 0; av < axisVariables.length; av++) {
                if (keepUnusedAxes || mainGda.axisValues(av).size() > 1)
                    activeAxes.add(av);
            }

            //define the dimensions
            int nActiveAxes = activeAxes.size();
            Dimension dimensions[] = new Dimension[nActiveAxes];
            Array axisArrays[] = new Array[nActiveAxes];
            for (int a = 0; a < nActiveAxes; a++) {
                int av = activeAxes.get(a);
                String avName = axisVariables[av].destinationName();
                PrimitiveArray pa = mainGda.axisValues(av);
                if (reallyVerbose) String2.log(" create dim=" + avName + " size=" + pa.size());
                dimensions[a] = nc.addDimension(avName, pa.size());
                if (av == lonIndex)
                    pa.scaleAddOffset(1, lonAdjust);
                axisArrays[a] = Array.factory(
                    mainGda.axisValues(av).getElementType(),
                    new int[]{pa.size()},
                    pa.toObjectArray());
                if (reallyVerbose) String2.log(" create var=" + avName);
                nc.addVariable(avName, 
                    NcHelper.getDataType(pa.getElementType()), 
                    new Dimension[]{dimensions[a]});
            }            

            //define the data variables
            Array dataArrays[] = new Array[tDataVariables.length]; 
            for (int dv = 0; dv < tDataVariables.length; dv++) {
                if (reallyVerbose) String2.log(" create var=" + tDataVariables[dv].destinationName());
                nc.addVariable(tDataVariables[dv].destinationName(),
                    NcHelper.getDataType(tDataVariables[dv].destinationDataTypeClass()), 
                    dimensions);
            }

            //write global attributes
            String names[] = mainGda.globalAttributes.getNames();
            for (int i = 0; i < names.length; i++) { 
                PrimitiveArray attPa = mainGda.globalAttributes.get(names[i]);
                if (reallyVerbose) String2.log(" create gatt " + names[i] + "=" + attPa);
                nc.addGlobalAttribute(NcHelper.getAttribute(names[i], attPa));
            }

            //write axis attributes
            for (int a = 0; a < nActiveAxes; a++) {
                int av = activeAxes.get(a);
                String destName = axisVariables[av].destinationName();
                if (reallyVerbose) String2.log(" dataAtt for " + destName);
                Attributes at = mainGda.axisAttributes[av];
                names  = at.getNames();
                for (int i = 0; i < names.length; i++) {
                    PrimitiveArray attPa = at.get(names[i]);
                    if (reallyVerbose) String2.log("   create axAtt " + names[i] + "=" + attPa);
                    nc.addVariableAttribute(destName, NcHelper.getAttribute(names[i], attPa));
                } 
            }

            //write data attributes
            for (int dv = 0; dv < tDataVariables.length; dv++) {
                String destName = tDataVariables[dv].destinationName();
                if (reallyVerbose) String2.log(" dataAtt for " + destName);
                Attributes at = mainGda.dataAttributes[dv];
                names  = at.getNames();
                for (int i = 0; i < names.length; i++) {
                    PrimitiveArray attPa = at.get(names[i]);
                    if (reallyVerbose) String2.log("   create dAtt " + names[i] + "=" + attPa);
                    nc.addVariableAttribute(destName, NcHelper.getAttribute(names[i], attPa));
                }
            }

            //leave "define" mode
            nc.create();

            //write the axis variables
            for (int a = 0; a < nActiveAxes; a++) {
                int av = activeAxes.get(a);
                nc.write(axisVariables[av].destinationName(), axisArrays[a]);
            }

            //write the data variables
            for (int dv = 0; dv < tDataVariables.length; dv++) {
                //EEEK! nc requires that I assemble the entire array then write it!
                //So just get all the data (don't use GridDataAccessor). 
                //(test for enough memory above)
                //Test here: enough data available right now.
                EDStatic.ensureMemoryAvailable(dvRequestSize[dv], datasetID);
                //get data right after ensuring memory available
                PrimitiveArray tpa[] = getSourceData(new EDV[]{tDataVariables[dv]}, 
                    mainGda.constraints());
//!!!I should verify that axis values are as expected
                int shape[] = new int[nActiveAxes];
                for (int a = 0; a < nActiveAxes; a++) 
                    shape[a] = tpa[activeAxes.get(a)].size();
                PrimitiveArray dataPa = tpa[axisVariables.length];
                dataPa = tDataVariables[dv].toDestination(dataPa); 
                Array array = Array.factory(
                    tDataVariables[dv].destinationDataTypeClass(),
                    shape, //was mainGda.totalIndex().shape(),
                    dataPa.toObjectArray()); //data is in PA after axisVariables' PAs
                nc.write(tDataVariables[dv].destinationName(), array);
            }

            //if close throws Throwable, it is trouble
            nc.close(); //it calls flush() and doesn't like flush called separately

            //rename the file to the specified name
            File2.rename(fullFileName + randomInt, fullFileName);

            //diagnostic
            if (reallyVerbose) String2.log("  EDDGrid.saveAsNc done.  TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
            //String2.log(NcHelper.dumpString(directory + name + ext, false));

        } catch (Throwable t) {
            //try to close the file
            try {
                nc.close(); //it calls flush() and doesn't like flush called separately
            } catch (Throwable t2) {
                //don't care
            }

            //delete the partial file
            File2.delete(fullFileName);

            throw t;
        }

    }
 
    /**
     * This writes the grid data to the outputStream in comma-separated-value 
     * ASCII format.
     * If no exception is thrown, the data was successfully written.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsCsv(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        saveAsSeparatedAscii(requestUrl, userDapQuery, outputStreamSource, ", ", true); //true=quoted
    }

    /**
     * This writes the grid data to the outputStream in tab-separated-value 
     * ASCII format.
     * If no exception is thrown, the data was successfully written.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @throws Throwable  if trouble. 
     */
    public void saveAsTsv(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource) throws Throwable {

        saveAsSeparatedAscii(requestUrl, userDapQuery, outputStreamSource, "\t", false); //false = !quoted
    }

    /**
     * This writes the axis or grid data to the outputStream in a separated-value 
     * ASCII format.
     * If no exception is thrown, the data was successfully written.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @param separator
     * @param quoted if true, String values are enclosed in double quotes
     *   and internal double quotes are converted to 2 double quotes.
     * @throws Throwable  if trouble. 
     */
    public void saveAsSeparatedAscii(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource,
        String separator, boolean quoted) throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsSeparatedAscii separator=\"" + 
            String2.annotatedString(separator) + "\""); 
        long time = System.currentTimeMillis();

        //get dataAccessor first, in case of error when parsing query
        boolean isAxisDapQuery = isAxisDapQuery(userDapQuery);
        AxisDataAccessor ada = null;
        GridDataAccessor gda = null;
        if (isAxisDapQuery) 
             ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
        else gda = new GridDataAccessor(this, requestUrl, userDapQuery, 
            true, false);   //rowMajor, convertToNaN (would be true, but TableWriterSeparatedValue will do it)

        //write the data to the tableWriter
        TableWriter tw = new TableWriterSeparatedValue(outputStreamSource, 
            separator, quoted, true); //writeUnits
        if (isAxisDapQuery) 
             saveAsTableWriter(ada, tw);
        else saveAsTableWriter(gda, tw);

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsSeparatedAscii done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");

    }

    /**
     * This gets the data for the userDapQuery and writes the data 
     * to the outputStream as an html or xhtml table.
     * See TableWriterHtml for details.
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     * @param userDapQuery an OPeNDAP DAP-style query string, still percentEncoded (shouldn't be null). 
     *   for a axis data, e.g., time[40:45], or 
     *   for a grid data, e.g., ATssta[45:1:45][0:1:0][120:10:140][130:10:160]
     * @param outputStreamSource the source of an outputStream (usually already 
     *   buffered) to receive the results.
     *   At the end of this method the outputStream is flushed, not closed.
     * @param fileName (no extension) used for the document title
     * @param xhtmlMode if true, the table is stored as an XHTML table.
     *   If false, it is stored as an HTML table.
     * @param preTableHtml is html or xhtml text to be inserted at the start of the 
     *   body of the document, before the table tag
     *   (or "" if none).
     * @param postTableHtml is html or xhtml text to be inserted at the end of the 
     *   body of the document, after the table tag
     *   (or "" if none).
     * @throws Throwable  if trouble. 
     */
    public void saveAsHtmlTable(String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource, 
        String fileName, boolean xhtmlMode, String preTableHtml, String postTableHtml) 
        throws Throwable {

        if (reallyVerbose) String2.log("  EDDGrid.saveAsHtmlTable"); 
        long time = System.currentTimeMillis();

        //get dataAccessor first, in case of error when parsing query
        boolean isAxisDapQuery = isAxisDapQuery(userDapQuery);
        AxisDataAccessor ada = null;
        GridDataAccessor gda = null;
        if (isAxisDapQuery) 
             ada = new AxisDataAccessor(this, requestUrl, userDapQuery);
        else gda = new GridDataAccessor(this, requestUrl, userDapQuery, 
            true, false);   //rowMajor, convertToNaN  (would be true, but TableWriterHtmlTable will do it)

        //write the data to the tableWriter
        TableWriter tw = new TableWriterHtmlTable(outputStreamSource,
            fileName, xhtmlMode, preTableHtml, postTableHtml, 
            true, true); //tEncodeAsXML, tWriteUnits 
        if (isAxisDapQuery) 
             saveAsTableWriter(ada, tw);
        else saveAsTableWriter(gda, tw);

        //diagnostic
        if (reallyVerbose)
            String2.log("  EDDGrid.saveAsHtmlTable done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }


    /**
     * This gets the axis data for the userDapQuery and writes the data to 
     * a tableWriter.
     * This writes the lon values as they are in the source
     *    (e.g., +-180 or 0..360). 
     * If no exception is thrown, the data was successfully written.
     * 
     * @param ada The source of data to be written to the tableWriter.
     * @param tw  This calls tw.finish() at the end.
     * @throws Throwable  if trouble. 
     */
    public void saveAsTableWriter(AxisDataAccessor ada, TableWriter tw) 
        throws Throwable {

        //make the table
        //note that TableWriter expects time values as doubles, and (sometimes) displays them as ISO 8601 strings
        Table table = new Table();
        int nRAV = ada.nRequestedAxisVariables();
        for (int av = 0; av < nRAV; av++) {
            table.addColumn(ada.axisVariables(av).destinationName(), ada.axisValues(av));
            String tUnits = ada.axisVariables(av).units(); //ok if null
            if (tUnits != null) 
                table.columnAttributes(av).set("units", tUnits);
        }
        table.ensureColumnsAreSameSize();

        //write the table
        tw.writeAllAndFinish(table);
    }

    /**
     * This gets the data for the userDapQuery and writes the data to 
     * a TableWriter.
     * This writes the lon values as they are in the source
     *    (e.g., +-180 or 0..360). 
     *    //note that TableWriter expects time values as doubles, and displays them as ISO 8601 strings
     * If no exception is thrown, the data was successfully written.
     * 
     * @param gridDataAccessor The source of data to be written to the tableWriter.
     *    Missing values should be as they are in the source.
     *    Some tableWriter's convert them to other values (e.g., NaN).
     * @param tw  This calls tw.finish() at the end.
     * @throws Throwable  if trouble. 
     */
    public void saveAsTableWriter(GridDataAccessor gridDataAccessor, 
        TableWriter tw) throws Throwable {

        //create the table (with one dummy row of data)
        Table table = new Table();
        int nAv = axisVariables.length;
        EDV queryDataVariables[] = gridDataAccessor.dataVariables();
        int nDv = queryDataVariables.length;
        PrimitiveArray avPa[] = new PrimitiveArray[nAv];
        boolean isDoubleAv[] = new boolean[nAv];
        boolean isFloatAv[]  = new boolean[nAv];
        PrimitiveArray dvPa[] = new PrimitiveArray[nDv];
        boolean isStringDv[] = new boolean[nDv];
        boolean isDoubleDv[] = new boolean[nDv];
        boolean isFloatDv[]  = new boolean[nDv];
        int nBufferRows = 1000;
        for (int av = 0; av < nAv; av++) {
            EDV edv = axisVariables[av];
            Class tClass = edv.destinationDataTypeClass();
            isDoubleAv[av] = tClass == double.class || tClass == long.class;
            isFloatAv[av]  = tClass == float.class;
            avPa[av] = PrimitiveArray.factory(tClass, nBufferRows, false);
            //???need to remove file-specific metadata (e.g., actual_range) from Attributes clone?
            table.addColumn(av, edv.destinationName(), avPa[av], 
                (Attributes)edv.combinedAttributes().clone());
        }
        for (int dv = 0; dv < nDv; dv++) {
            EDV edv = queryDataVariables[dv];
            Class tClass = edv.destinationDataTypeClass();
            isStringDv[dv] = tClass == String.class;
            isDoubleDv[dv] = tClass == double.class || tClass == long.class;
            isFloatDv[dv]  = tClass == float.class;
            dvPa[dv] = PrimitiveArray.factory(tClass, nBufferRows, false);
            //???need to remove file-specific metadata (e.g., actual_range) from Attributes clone?
            table.addColumn(nAv + dv, edv.destinationName(), dvPa[dv], 
                (Attributes)edv.combinedAttributes().clone());
        }

        //write the data
        int tRows = 0;
        while (gridDataAccessor.increment()) {
            //put the data in row one of the table
            for (int av = 0; av < nAv; av++) {
                if      (isDoubleAv[av]) avPa[av].addDouble(gridDataAccessor.getAxisValueAsDouble(av));
                else if (isFloatAv[av])  avPa[av].addFloat( gridDataAccessor.getAxisValueAsFloat(av));
                else                     avPa[av].addInt(   gridDataAccessor.getAxisValueAsInt(av));
            }

            for (int dv = 0; dv < nDv; dv++) {
                if      (isStringDv[dv]) dvPa[dv].addString(gridDataAccessor.getDataValueAsString(dv));
                else if (isDoubleDv[dv]) dvPa[dv].addDouble(gridDataAccessor.getDataValueAsDouble(dv));
                else if (isFloatDv[dv])  dvPa[dv].addFloat( gridDataAccessor.getDataValueAsFloat(dv));
                else                     dvPa[dv].addInt(   gridDataAccessor.getDataValueAsInt(dv));
            }
            tRows++;

            //write the table 
            if (tRows >= nBufferRows) {
                tw.writeSome(table);
                table.removeAllRows();
                tRows = 0;
            }
        }
        if (tRows > 0) 
            tw.writeSome(table);
        tw.finish();

    }

    /**
     * This writes an HTML form requesting info from this dataset (like the DAP Data Access forms).
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param userDapQuery the part after the '?', still percentEncoded (shouldn't be null).
     * @param writer
     * @throws Throwable if trouble
     */
    public void writeDapHtmlForm(String loggedInAs,
        String userDapQuery, Writer writer) throws Throwable {

        //parse userDapQuery 
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        if (userDapQuery == null)
            userDapQuery = "";
        userDapQuery = userDapQuery.trim();
        StringArray destinationNames = new StringArray();
        IntArray constraints = new IntArray();
        boolean isAxisDapQuery = false; //only true if userDapQuery.length() > 0 and is axisDapQuery
        if (userDapQuery.length() > 0) {
            try {
                isAxisDapQuery = isAxisDapQuery(userDapQuery);
                if (isAxisDapQuery) 
                     parseAxisDapQuery(userDapQuery, destinationNames, constraints, true); 
                else parseDataDapQuery(userDapQuery, destinationNames, constraints, true); 
            } catch (Throwable t) {
                String2.log(MustBe.throwableToString(t));
                userDapQuery = ""; //as if no userDapQuery
            }
        }


        //beginning of form   ("form1" is used in javascript below")
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl);
        String formName = "form1";
        EDVTimeGridAxis timeVar = timeIndex >= 0? (EDVTimeGridAxis)axisVariables[timeIndex] : null;
        String liClickSubmit = "\n" +
            "  <li> " + EDStatic.EDDClickOnSubmitHtml + "\n" +
            "  </ol>\n";
        writer.write("<p>");
        writer.write(widgets.beginForm(formName, "GET", "", ""));

        //begin table  ("width=60%" doesn't keep table tight; I wish it did)
        writer.write(widgets.beginTable(0, 0, "")); 

        //write the table's column names   
        String dimHelp = EDStatic.EDDGridDimensionHtml + "\n<br>";
        String sss = dimHelp +
            EDStatic.EDDGridSSSHtml + "\n<br>";
        String startTooltip  = sss + EDStatic.EDDGridStartHtml;
        String stopTooltip   = sss + EDStatic.EDDGridStopHtml;
        String strideTooltip = sss + EDStatic.EDDGridStrideHtml;
        String downloadTooltip = EDStatic.EDDGridDownloadTooltipHtml;
        String gap = "&nbsp;&nbsp;&nbsp;";
        writer.write(
            "<tr>\n" +
            "  <th nowrap align=\"left\">" + EDStatic.EDDGridDimension + " " + 
                EDStatic.htmlTooltipImage(dimHelp + 
                    EDStatic.EDDGridVarHasDimHtml) + " </th>\n" +
            "  <th nowrap align=\"left\">" + EDStatic.EDDGridStart  + " " + EDStatic.htmlTooltipImage(startTooltip)  + " </th>\n" +
            "  <th nowrap align=\"left\">" + EDStatic.EDDGridStride + " " + EDStatic.htmlTooltipImage(strideTooltip) + " </th>\n" +
            "  <th nowrap align=\"left\">" + EDStatic.EDDGridStop   + " " + EDStatic.htmlTooltipImage(stopTooltip)   + " </th>\n" +
            //"  <th nowrap align=\"left\">&nbsp;" + EDStatic.EDDGridFirst + " " + 
            //    EDStatic.htmlTooltipImage(EDStatic.EDDGridDimensionFirstHtml) + "</th>\n" +
            "  <th nowrap align=\"left\">&nbsp;" + EDStatic.EDDGridNValues + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDGridNValuesHtml) + "</th>\n" +
            "  <th nowrap align=\"left\">" + gap + EDStatic.EDDGridSpacing + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDGridSpacingHtml) + "</th>\n" +
            //"  <th nowrap align=\"left\">" + gap + EDStatic.EDDGridLast + " " +
            //    EDStatic.htmlTooltipImage(EDStatic.EDDGridDimensionLastHtml) + "</th>\n" +
            "</tr>\n");

        //a row for each axisVariable
        StringBuffer allDim = new StringBuffer();
        int nAv = axisVariables.length;
        String sliderFromNames[] = new String[nAv];
        String sliderToNames[] = new String[nAv];
        int sliderNThumbs[] = new int[nAv];
        String sliderUserValuesCsvs[] = new String[nAv];
        int sliderInitFromPositions[] = new int[nAv];
        int sliderInitToPositions[] = new int[nAv];
        for (int av = 0; av < nAv; av++) {
            EDVGridAxis edvga = axisVariables[av];
            int sourceSize = edvga.sourceValues().size();
            writer.write("<tr>\n");
            
            //get the extra info   
            String extra = edvga.units();
            if (av == timeIndex)
                extra = "UTC"; //no longer true: "seconds since 1970-01-01..."
            if (extra == null) 
                extra = "";
            if (showLongName(edvga.destinationName(), edvga.longName()))
                extra = edvga.longName() + (extra.length() == 0? "" : ", " + extra);
            if (extra.length() > 0) 
                extra = " (" + extra + ")";

            //variables: checkbox destName (longName, extra) 
            writer.write("  <td nowrap>\n");
            writer.write(widgets.checkbox("avar" + av, downloadTooltip,  
                userDapQuery.length() > 0 && isAxisDapQuery? 
                    destinationNames.indexOf(edvga.destinationName()) >= 0 : 
                    true, 
                edvga.destinationName(), edvga.destinationName(), ""));

            writer.write(extra + " ");
            String tDim = "[" + sourceSize + "]";
            allDim.append(tDim);
            writer.write(EDStatic.htmlTooltipImage(edvga.destinationDataTypeClass(), 
                edvga.destinationName() + tDim, edvga.combinedAttributes()));           
            writer.write("&nbsp;</td>\n");

            //set default start, stride, stop                       
            int tStarti = av == timeIndex? sourceSize - 1 : 0;
            int tStopi  = sourceSize - 1;
            double tdv = av == timeIndex? edvga.destinationMax() : //yes, time max, to limit time
                                          edvga.firstDestinationValue();
            String tStart = edvga.destinationToString(tdv);
            String tStride = "1";
            tdv = av == timeIndex? edvga.destinationMax() : 
                                   edvga.lastDestinationValue();
            String tStop = edvga.destinationToString(tdv);
 
            //if possible, override defaults via userDapQuery
            if (userDapQuery.length() > 0) {
                int tAv = isAxisDapQuery?
                    destinationNames.indexOf(edvga.destinationName()) :
                    av;
                if (tAv >= 0) {
                    tStarti = constraints.get(3*tAv + 0);
                    tStopi  = constraints.get(3*tAv + 2);
                    tStart  = edvga.destinationToString(edvga.destinationValue(tStarti).getNiceDouble(0));
                    tStride = "" + constraints.get(3*tAv + 1);
                    tStop   = edvga.destinationToString(edvga.destinationValue(tStopi).getNiceDouble(0));
                }
            }

            //start
            String edvgaTooltip = edvga.htmlRangeTooltip();
            writer.write("  <td nowrap>");
            //new style: textfield
            writer.write(widgets.textField("start" + av, edvgaTooltip, 19, 30,  tStart, ""));
            //old style: select; very slow in IE 7
            //PrimitiveArray destValues = edvga.destinationStringValues();
            ////StringArray destValues = new StringArray(tdestValues);
            ////int maxLength = destValues.maxStringLength();
            ////String tStyle = ""; //"style=\"width:" + (maxLength*0.6) + "em\"";
            //writer.write(widgets.select("start" + av, startTooltip, 
            //    HtmlWidgets.BUTTONS_0n + HtmlWidgets.BUTTONS_1000,
            //    destValues, 
            //    av == timeIndex? destValues.size() - 1 : 0, tStyle));

            writer.write("  </td>\n");

            //stride
            writer.write("  <td nowrap>"); // + gap);
            writer.write(widgets.textField("stride" + av, edvgaTooltip, 7, 10, tStride, ""));
            writer.write("  " + //gap + 
                "</td>\n");

            //stop
            writer.write("  <td nowrap>");
            //new style: textfield
            writer.write(widgets.textField("stop" + av, edvgaTooltip, 19, 30, tStop, ""));
            //old style: select; very slow in IE 7
            //writer.write(widgets.select("stop" + av, stopTooltip, 
            //    HtmlWidgets.BUTTONS_0n + HtmlWidgets.BUTTONS_1000,
            //    destValues, 
            //    av == latIndex || av == lonIndex || av == timeIndex? destValues.size() - 1 : 0, 
            //    tStyle));
            writer.write("  </td>\n");

            //first
            //writer.write("  <td nowrap>&nbsp;" + 
            //    edvga.destinationToString(edvga.firstDestinationValue()) + "</td>\n");

            //n Values
            writer.write("  <td nowrap>" + gap + sourceSize + "</td>\n");

            //spacing
            writer.write("  <td nowrap>" + gap + edvga.spacingDescription() + "</td>\n");

            //last
            //writer.write("  <td nowrap>" + gap +
            //    edvga.destinationToString(edvga.lastDestinationValue()) +
            //    "</td>\n");

            //end of row
            writer.write("</tr>\n");

            // *** and a slider for this axis  (Data Access Form)
            sliderFromNames[av] = formName + ".start" + av;
            sliderToNames[  av] = formName + ".stop"  + av;
            int safeSourceSize1 = Math.max(1, sourceSize - 1);
            //if (sourceSize == 1) {
            //    sliderNThumbs[av] = 0;
            //    sliderUserValuesCsvs[av] = "";
            //    sliderInitFromPositions[av] = 0;
            //    sliderInitToPositions[av] = 0;
            //} else {
                sliderNThumbs[av] = 2;
                sliderUserValuesCsvs[av] = edvga.sliderCsvValues();
                sliderInitFromPositions[av] = Math2.roundToInt((tStarti * (EDV.SLIDER_PIXELS - 1.0)) / safeSourceSize1);
                sliderInitToPositions[  av] = sourceSize == 1? EDV.SLIDER_PIXELS - 1 :
                                              Math2.roundToInt((tStopi  * (EDV.SLIDER_PIXELS - 1.0)) / safeSourceSize1);
                writer.write(
                    "<tr align=\"left\">\n" +
                    "  <td nowrap colspan=\"6\" align=\"left\">\n" +
                    widgets.spacer(10, 1, "align=\"left\"") +
                    widgets.dualSlider(av, EDV.SLIDER_PIXELS - 1, "align=\"left\"") +
                    "  </td>\n" +
                    "</tr>\n");
            //}
        }

        writer.write(
            "<tr>\n" +
            "  <td colspan=\"4\" align=\"left\">&nbsp;<br>" + EDStatic.EDDGridGridVariableHtml + "</td>\n" +
            "</tr>\n");

        //a row for each dataVariable
        for (int dv = 0; dv < dataVariables.length; dv++) {
            EDV edv = dataVariables[dv];
            writer.write("<tr>\n");
            
            //get the extra info   
            String extra = edv.units();
            if (extra == null) 
                extra = "";
            if (showLongName(edv.destinationName(), edv.longName()))
                extra = edv.longName() + (extra.length() == 0? "" : ", " + extra);
            if (extra.length() > 0) 
                extra = " (" + extra + ")";

            //variables: checkbox destName (longName, units)
            writer.write("  <td nowrap colspan=\"4\">\n");
            writer.write(widgets.checkbox("dvar" + dv, downloadTooltip,  
                userDapQuery.length() > 0 && isAxisDapQuery? false :
                    userDapQuery.length() > 0? destinationNames.indexOf(edv.destinationName()) >= 0 : 
                    true,  
                edv.destinationName(), edv.destinationName(), ""));

            writer.write(extra + " ");
            writer.write(EDStatic.htmlTooltipImage(edv.destinationDataTypeClass(), 
                edv.destinationName() + allDim.toString(), 
                edv.combinedAttributes()));           
            writer.write("</td>\n");

            //end of row
            writer.write("</tr>\n");
        }

        //end of table
        writer.write(widgets.endTable());

        //fileType
        writer.write("<p><b>" + EDStatic.EDDFileType + "</b>\n");
        writer.write(widgets.select("fileType", EDStatic.EDDSelectFileType, 1,
            allFileTypeOptions, defaultFileTypeOption, ""));

        //generate the javaScript
        String javaScript = 
            "var result = \"\";\n" +
            "try {\n" +
            "  var ft = form1.fileType.options[form1.fileType.selectedIndex].text;\n" +
            "  var start = \"" + tErddapUrl + "/griddap/" + datasetID() + 
              "\" + ft.substring(0, ft.indexOf(\" - \")) + \"?\";\n" +
            "  var sss = new Array(); var cum = \"\"; var done = false;\n" +

            //gather startStrideStop and cumulative sss
            "  for (var av = 0; av < " + nAv + "; av++) {\n" +
            "    sss[av] = \"[(\" + " + 
                   "eval(\"form1.start\"  + av + \".value\") + \"):\" + " +                   
                   //"eval(\"form1.start\"  + av + \".options[form1.start\"  + av + \".selectedIndex].text\") + \"):\" + " +
                   "eval(\"form1.stride\" + av + \".value\") + \":(\" + " +
                   "eval(\"form1.stop\"   + av + \".value\") + \")]\";\n" +                   
                   //"eval(\"form1.stop\"   + av + \".options[form1.stop\"   + av + \".selectedIndex].text\") + \")]\";\n" +
            "    cum += sss[av];\n" +
            "  }\n" +

            //see if any dataVars were selected
            "  for (var dv = 0; dv < " + dataVariables.length + "; dv++) {\n" +
            "    if (eval(\"form1.dvar\" + dv + \".checked\")) {\n" +
            "      if (result.length > 0) result += \",\";\n" +
            "      result += (eval(\"form1.dvar\" + dv + \".value\")) + cum; }\n" +
            "  }\n" +

            //else find selected axes
            "  if (result.length > 0) {\n" +
            "    result = start + result;\n" +
            "  } else {\n" +
            "    result = start;\n" +
            "    for (var av = 0; av < " + nAv + "; av++) {\n" +
            "      if (eval(\"form1.avar\" + av + \".checked\")) { \n" +
            "        if (result.length > start.length) result += \",\";\n" +
            "        result += (eval(\"form1.avar\" + av + \".value\")) + sss[av];\n" +
            "      }\n" +
            "    }\n" +
            "    if (result.length == start.length) { result = \"\";\n" +
            "      throw new Error(\"Please select (check) one of the variables.\"); }\n" +
            "  }\n" +
            "} catch (e) {alert(e);}\n" +  //result is in 'result'  (or "" if error or nothing selected)
            "form1.tUrl.value = result;\n";

        //just generate URL
        writer.write("<br>"); 
        writer.write(widgets.button("button", "getUrl", 
            String2.replaceAll(EDStatic.justGenerateAndViewHtml, "&protocolName;", dapProtocol),
            EDStatic.justGenerateAndView, 
            "onclick='" + javaScript + "'"));
        writer.write(widgets.textField("tUrl", 
            EDStatic.justGenerateAndViewUrl,
            95, 1000, "", ""));

        //submit
        writer.write(HtmlWidgets.ifJavaScriptDisabled);
        writer.write("<br>"); 
        writer.write(widgets.button("button", "submit1", 
            EDStatic.submitTooltip, EDStatic.submit, 
            "onclick='" + javaScript +
            "if (result.length > 0) window.location=result;\n" + //or open a new window: window.open(result);\n" +
            "'"));

        //end of form
        writer.write(widgets.endForm());
        writer.write("<br>&nbsp;\n");

        //the javascript for the sliders
        writer.write(widgets.sliderScript(sliderFromNames, sliderToNames, 
            sliderNThumbs, sliderUserValuesCsvs, 
            sliderInitFromPositions, sliderInitToPositions, EDV.SLIDER_PIXELS - 1));

        writer.flush(); //be nice   

    }

    /**
     * This writes HTML info on forming OPeNDAP DAP-style requests for this type of dataset.
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param writer to which will be written HTML info on forming OPeNDAP DAP-style 
     *  requests for this type of dataset.
     * @param complete if false, this just writes a paragraph and shows a link
     *    to [protocol]/documentation.html
     * @throws Throwable if trouble
     */
    public static void writeGeneralDapHtmlInstructions(String tErddapUrl, 
        Writer writer, boolean complete) throws Throwable {

        String dapBase = tErddapUrl + "/" + dapProtocol + "/";
        String datasetBase = dapBase + EDStatic.EDDGridIdExample;
        String ddsExample           = datasetBase + ".dds";
        String dds1VarExample       = datasetBase + ".dds?" + EDStatic.EDDGridNoHyperExample;
        //all of the fullXxx examples are pre-encoded
        String fullDimensionExample = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridDimensionExample);
        String fullIndexExample     = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridDataIndexExample);
        String fullValueExample     = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridDataValueExample);
        String fullTimeExample      = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridDataTimeExample);
        String fullGraphExample     = XML.encodeAsXML(datasetBase + ".png?"       + EDStatic.EDDGridGraphExample);
        String fullGraphMAGExample  = XML.encodeAsXML(datasetBase + ".graph?"     + EDStatic.EDDGridGraphExample);
        String fullGraphDataExample = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridGraphExample);
        String fullMapExample       = XML.encodeAsXML(datasetBase + ".png?"       + EDStatic.EDDGridMapExample);
        String fullMapMAGExample    = XML.encodeAsXML(datasetBase + ".graph?"     + EDStatic.EDDGridMapExample);
        String fullMapDataExample   = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDGridMapExample);
        String fullMatExample       = XML.encodeAsXML(datasetBase + ".mat?" + EDStatic.EDDGridMapExample);

        writer.write(
            "<h2><a name=\"instructions\">Using</a> griddap to Request Data and Graphs from Gridded Datasets</h2>\n" +
            longDapDescriptionHtml +
            "<p>griddap request URLs must be in the form\n" +
            "<br>&nbsp;&nbsp;&nbsp;" + dapBase + " " + 
                                        //these links don't work in firefox
                "<i>datasetID</i> " +   //"<i><a href=\"#datasetID\">datasetID</a></i> " + 
                "<i>fileType</i> {?" +  //"<i><a href=\"#fileType\">fileType</a></i> {?" + 
                "<i>query</i>}\n" +     //"<i><a href=\"#query\">query</a></i>}\n" +
            "<br>For example,\n" +
            "<br>&nbsp;&nbsp;&nbsp;<a href=\"" + fullTimeExample + "\">" + 
                                                 fullTimeExample + "</a>\n" +
            "\n");

        if (!complete) {
            writer.write(
            "<p>For details, see the <a href=\"" + dapBase + 
                "documentation.html\">" + dapProtocol + " Documentation</a>.\n");
            return;
        }
         
        //details
        writer.write(
            "<p><b>Details:</b><ul>\n" +
            "<li>Requests must not have any internal spaces.\n"+
            "<li>Requests are case sensitive.\n" +
            "<li>{} is notation to denote an optional part of the request.\n" + 
            "  <br>&nbsp;\n" +

            //datasetID
            "<li><a name=\"datasetID\"><i><b>datasetID</b></i></a> identifies the name that ERDDAP\n" +
            "      assigned to the dataset (for example, \"" + EDStatic.EDDGridIdExample + "\"). \n" +
            "      <br>You can see a list of " +
            "<a href=\"" + tErddapUrl + "/" + dapProtocol + 
                "/index.html\">datasetID options available via griddap</a>.\n" +
            "   <br>&nbsp;\n" +

            //fileType
            "<li><a name=\"fileType\"><i><b>fileType</b></i></a> specifies the type of grid data file that you want to download\n" +
            "     (for example, \".htmlTable\").\n" +
            "   <br>The actual extension of the resulting file may be slightly different than the fileType\n" +
            "     (for example, \".smallPdf\" returns a small .pdf file).\n" +
            "   <br>The fileType options for downloading gridded data are:\n" +
            "   <br>&nbsp;\n" +
            "  <table class=\"erd\" cellspacing=\"0\">\n" + 
            //"  <table border=\"1\" cellpadding=\"2\" cellspacing=\"0\" >\n" +
            "    <tr><th>Data<br>fileTypes</th><th>Description</th><th>Info</th><th>Example</th></tr>\n");
        for (int i = 0; i < dataFileTypeNames.length; i++) 
            writer.write(
                "    <tr>\n" +
                "      <td>" + dataFileTypeNames[i] + "</td>\n" +
                "      <td>" + dataFileTypeDescriptions[i] + "</td>\n" +
                "      <td>" + 
                      (dataFileTypeInfo[i] == null || dataFileTypeInfo[i].equals("")? 
                          "&nbsp;" : "<a href=\"" +  dataFileTypeInfo[i] + "\">info</a>") + 
                      "</td>\n" +
                "      <td><a href=\"" +  datasetBase + dataFileTypeNames[i] + "?" + 
                    XML.encodeAsXML(EDStatic.EDDGridDataTimeExample) + "\">example</a></td>\n" +
                "    </tr>\n");
        writer.write(
            "   </table>\n" +
            "   <br>For example, here is a request to download data:\n" +
            "   <br><a href=\"" + fullTimeExample + "\">" + 
                                  fullTimeExample + "</a>\n" +
            "   \n" +
            "   <p><a name=\"matlab\"><b>MATLAB</b></a> users can download data from within MATLAB.\n" +
            "     Here is a one line example:<pre>\n" +
                 "load(urlwrite('" + fullMatExample + "', 'test.mat'));</pre>\n" +
            "     The data will be in a MATLAB structure. \n" +
                 "The structure's name will be the datasetID (for example, \"" + EDStatic.EDDGridIdExample + "\"). \n" +
            "     <br>The structure's internal variables will have the same names as in ERDDAP \n" +
                 "(for example, use \"<tt>fieldnames(" + EDStatic.EDDGridIdExample + ")</tt>\" ). \n" +
            "     <br>If you download a 2D matrix of data (as in the example above), you can plot it with (for example):<pre>\n" +
                  EDStatic.EDDGridMatlabPlotExample + "</pre>\n" +
            "     The numbers at the end of the first line specify the range for the color mapping. \n" +
                 "The 'set' command flips the map to make it upright.\n" +
            "\n" +
            "   <p><a name=\"jsonp\"><b>JSONP</b></a> - Requests for .json files may now include an optional\n" +
            "     <a href=\"http://niryariv.wordpress.com/2009/05/05/jsonp-quickly/\">jsonp</a> request\n" +
            "     <br>by adding \"&amp;.jsonp=<i>functionName</i>\" to the end of the query.\n" +
            "     <br>Basically, this just tells ERDDAP to add \"<i>functionName</i>(\" to the beginning of the response \n" +
            "     <br>and \")\" to the end of the response.\n" +
            "     <br>If originally there was no query, leave off the \"&amp;\" in your query.\n"); 

        //imageFile Types, graphs and maps
        writer.write(
            "   <p><a name=\"imageFileTypes\"><b>Making an Image with a Graph or Map of Gridded Data</b></a>\n" +
            "   <br>If a griddap request URL specifies a subset of data which is suitable for making a graph or a map,\n" +
            "   <br>and the fileType is an image fileType, griddap will return an image with a graph or map. \n" +
            "   <br>griddap request URLs can include optional <a href=\"#GraphicsCommands\">graphics commands</a> which let you customize the graph or map.\n" +
            "   <br>As with other griddap request URLs, you can create these URLs by hand or have a computer program do it. \n" +
            "   <br>Or, you can use the Make A Graph web pages, which simplify creating these URLs (see the \"graph\" links in the table of <a href=\"" + 
                dapBase + "index.html\">griddap datasets</a>). \n" +
            "\n" +
            "   <p>The fileType options for downloading images of graphs and maps of grid data are:\n" +
            "  <table class=\"erd\" cellspacing=\"0\">\n" + 
            //"  <table border=\"1\" cellpadding=\"2\" cellspacing=\"0\" >\n" +
            "    <tr><th>Image<br>fileTypes</th><th>Description</th><th>Info</th><th>Example</th></tr>\n");
        for (int i = 0; i < imageFileTypeNames.length; i++) 
            writer.write(
                "    <tr>\n" +
                "      <td>" + imageFileTypeNames[i] + "</td>\n" +
                "      <td>" + imageFileTypeDescriptions[i] + "</td>\n" +
                "      <td>" + 
                      (imageFileTypeInfo[i] == null || imageFileTypeInfo[i].equals("")? 
                          "&nbsp;" : "<a href=\"" +  imageFileTypeInfo[i] + "\">info</a>") + 
                      "</td>\n" +   //must be mapExample below because kml doesn't work with graphExample
                "      <td><a href=\"" +  datasetBase + imageFileTypeNames[i] + "?" + 
                    XML.encodeAsXML(EDStatic.EDDGridMapExample) + "\">example</a></td>\n" +
                "    </tr>\n");
        writer.write(
            "   </table>\n" +
            "   <br>&nbsp;\n"); //linebreak

        //query
        writer.write(
            "<li><a name=\"query\"><i><b>query</b></i></a> is the part of the request after the \"?\". \n" +
            "       It specifies the subset of data that you want to receive.\n" +
            "  <br>In griddap, it is an optional\n" +
            "  <a href=\"http://www.opendap.org\">OPeNDAP</a>\n " +
            "  <a href=\"http://www.opendap.org/pdf/ESE-RFC-004v1.1.pdf\">DAP</a>-style\n" +
            "  <a href=\"http://www.opendap.org/user/guide-html/guide_61.html#id5\">hyperslab query</a>\n" +
            "  which can request:\n" +
            "   <ul>\n" +
            "   <li>One or more dimension (axis) variables, for example\n " +
            "       <br><a href=\"" + fullDimensionExample + "\">" + 
                                      fullDimensionExample + "</a> .\n" +
            "   <li>One or more data variables, for example\n" +
            "       <br><a href=\"" + fullIndexExample + "\">" + 
                                      fullIndexExample + "</a> .\n" +
            "       <br>To request more than one data variable, separate the desired data variable names by commas.\n" +
            "       <br>If you do request more than one data variable,\n" +
            "          the requested subset for each variable must be identical (see below).\n" +
            "       <br>(In griddap, all data variables within a grid dataset share the same dimensions.)\n" +
            "   <li>The entire dataset.\n" +
            "         Omitting the entire query is the same as requesting all of the data for all of the variables. \n" +
            "     <br>Because of the large size of most datasets, this is usually only appropriate for\n" +
            "        fileTypes that don't return actual data values (e.g., .das, .dds, and .html). For example,\n" +
            "     <br><a href=\"" + XML.encodeAsXML(ddsExample) + "\">" + 
                                    XML.encodeAsXML(ddsExample) + "</a>\n" +
            "     <br>griddap is designed to handle requests of any size but\n" +
            "        trying to download all of the data with one request will usually fail\n" +
            "        (for example, downloads that last days usually fail at some point).\n" +
            "     <br>If you need all of the data, consider breaking your big request into several smaller requests.\n" +
            "     <br>If you just need a sample of the data, use the largest acceptable stride values (see below).\n" +
            "   </ul>\n" +
            "   \n" +
            "   <p><a name=\"StartStrideStop\">Using</a> [start:stride:stop]\n" +
            "   <br>When requesting dimension (axis) variables or data variables, the query may specify a subset\n" +
            "      of a given dimension by identifying the [start{{:stride}:stop}] indices for that dimension.\n" +
            "   <ul>\n" +
            "   <li><tt>start</tt> is the index of the first desired value. Indices are 0-based. (0 is the first index. 1 is the second index. ...) \n" +
            "   <li><tt>stride</tt> indicates how many intervening values to get: 1=get every value, 2=get every other value, 3=get every third value, ...\n" +
            "     <br>Stride values are in index units (not the units of the dimension).\n" +
            "   <li><tt>stop</tt> is the index of the last desired value. \n" +             
            "   <li>Specifying only two values for a dimension (i.e., [start:stop]) is interpreted as [start:1:stop].\n" +
            "   <li>Specifying only one value for a dimension (i.e., [start]) is interpreted as [start:1:start].\n" +
            "   <li>Specifying no values for a dimension (i.e., []) is interpreted as [0:1:max].\n" +
            "   <li>Omitting all of the [start:stride:stop] values \n" +
            "       (that is, requesting the variable without the subset constraint)\n" +
            "       is equivalent to requesting the entire variable.\n" +
            "     <br>For dimension variables (for example, longitude, latitude, and time)\n" +
            "       and for fileTypes that don't download actual data (notably, .das, .dds, .html,\n" +
            "       and all of the graph and map fileTypes) this is fine. For example,\n" +
            "     <br><a href=\"" + XML.encodeAsXML(dds1VarExample) + "\">" + 
                                    XML.encodeAsXML(dds1VarExample) + "</a>\n" +
            "     <br>For other fileTypes (for example, .csv, .mat, .nc, .tsv), the resulting data may be very large.\n" +
            "       <br>griddap is designed to handle requests of any size.\n" +
            "       But if you try to download all of the data with one request,\n" +
            "       the request will often fail for other reasons\n" +
            "       (for example, downloads that last for days usually fail at some point).\n" +
            "     <br>If you need all of the data, consider breaking your big request into several smaller requests.\n" +
            "     <br>If you just need a sample of the data, use the largest acceptable stride values.\n" +
            "   <li><a name=\"last\"><tt>last</tt></a> - ERDDAP extends the DAP standard by interpreting \n" +
            "       a <tt>start</tt> or <tt>stop</tt> value of \"<tt>last</tt>\" as the last available index value.\n" + 
            "     <br>You can also use the notation \"<tt>last-<i>n</i></tt>\" (e.g., \"<tt>last-10</tt>\")\n" +
            "       to specify the last index minus some number of indices.\n" + 
            "     <br>You can use '+' in place of '-'. The number of indices can be negative.\n" +
            "   <li><a name=\"parentheses\">griddap</a> extends the standard DAP subset syntax by allowing the start and/or stop \n" +
            "        values to be actual dimension values (for example, longitude values in degrees_east) \n" +
            "        within parentheses, instead of array indices.  \n" +
            "     <br>This example with " + EDStatic.EDDGridDimNamesExample + " dimension values \n" +
            "     <br><a href=\"" + fullValueExample + "\">" + 
                                    fullValueExample + "</a>\n" +
            "     <br>is (at least at the time of writing this) equivalent to this example with dimension indices\n" +
            "     <br><a href=\"" + fullIndexExample + "\">" + 
                                    fullIndexExample + "</a>\n" +
            "     <br>The value in parentheses must be within the range of values for the dimension. \n" +
            "     <br>If the value in parentheses doesn't exactly equal one of the \n" +
            "        dimension values, the closest dimension value will be used.\n" +
            "   <li><a name=\"strideParentheses\">griddap</a> does not allow parentheses around stride values.  The reasoning is...\n" +
            "     <br>With the start and stop values, it is easy to convert the value in parentheses into the appropriate index value\n" +
            "     <br>by finding the nearest dimension value. This works if the dimension values are evenly spaced or not.\n" +
            "     <br>If the dimension values were always evenly spaced, it would be easy to use a similar technique to convert a stride\n" +
            "     <br>value in parentheses into a stride index value. But dimension values often aren't evenly spaced.\n" +
            "     <br>So for now, ERDDAP doesn't support the parentheses notation for stride values.\n" +
            "   <li>griddap always stores date/time values as numeric values in\n" +
            "       seconds since 1970-01-01T00:00:00Z.\n" +
            "     <br>Here is an example of a query which includes date/time numeric values:\n" +
            "     <br><a href=\"" + fullValueExample + "\">" + 
                                    fullValueExample + "</a>\n" +
            "     <br>Some fileTypes (notably, .csv, .tsv, .htmlTable, and .xhtml)\n" +
            "        display date/time values as \n" +
            "        <a href=\"http://www.iso.org/iso/date_and_time_format\">ISO 8601:2004 \"extended\" date/time strings</a>\n" +
            "        (e.g., 2002-08-03T12:30:00Z).\n" +
            "   <li>For the time dimension, griddap extends the DAP standard by allowing you\n" +
            "       to specify an ISO 8601 date/time values in parentheses, \n" +
            "       which griddap then converts to the internal numeric value (in seconds since 1970-01-01T00:00:00Z) \n" +
            "       and then to the appropriate array index.\n" +
            "     <br>The ISO date/time value should be in the form: <i>YYYY-MM-DD</i>T<i>hh:mm:ssZ</i>, where Z is 'Z' or a &plusmn;hh:mm offset from UTC.\n" +
            "     <br>If you omit Z (or the &plusmn;hh:mm offset), :ssZ, :mm:ssZ, or Thh:mm:ssZ from the ISO date/time that you specify, \n" +
            "       the missing fields are assumed to be 0.\n" +
            "     <br>The example below is equivalent (at least at the time of writing this) to the examples above:\n" +
            "     <br><a href=\"" + fullTimeExample + "\">" + 
                                    fullTimeExample + "</a>\n" +
            "   <li><a name=\"lastInParentheses\"><tt>(last)</tt></a> - ERDDAP interprets \n" +
            "       a <tt>start</tt> or <tt>stop</tt> value of \"<tt>(last)</tt>\" as the last available index.\n" + 
            "     <br>You can also use the notation \"<tt>(last-<i>d</i>)</tt>\"  (e.g., \"<tt>last-10.5</tt>\") \n" +
            "       to specify the last index's value minus some numeric value, which is then converted to the nearest index.\n" + 
            "     <br>You can use '+' in place of '-'. The numeric value can be negative.\n" +
            "     <br>For a time axis, the numeric value is interpreted as some number of seconds.\n" +

            //Graphics Commands
            "   <li><a name=\"GraphicsCommands\"><b>Graphics Commands</b></a> - <a name=\"MakeAGraph\">griddap</a> extends the DAP standard by allowing graphics commands in the query. \n" +
            "      <br>The Make A Graph web pages simplify the creation of URLs with these graphics commands (see the \"graph\" links in the table of <a href=\"" + 
                dapBase + "index.html\">griddap datasets</a>). \n" +
            "      <br>So we recommend using the Make A Graph web pages to generate URLs, and then, when needed, using the information here to modify the URLs for special purposes.\n" +
            "      <br>These commands are optional. If present, they must occur after the data request part of the query. \n" +
            "      <br>These commands are used by griddap if you request an <a href=\"#imageFileTypes\">image fileType</a> (e.g., .png) and are ignored if you request a data file (e.g., .asc). \n" +
            "      <br>If relevant commands are not included in your request, griddap uses the defaults and tries its best to generate a reasonable graph or map. \n" +
            "      <br>All of the commands are in the form &amp;.<i>commandName</i>=<i>value</i> . \n" +
            "      <br>If the value has sub-values, they are separated by the '|' character. \n" +
            "      <br>In most cases, sub-values can be omitted, causing the default value to be used.\n" +
            "      <br>In most cases, if sub-values are omitted from the end of the list, you can just truncate the list \n" +
            "      (for example, \"a||c\" can be used instead of \"a||c||||\").\n" +
            "      <br>The commands are:\n" +
            "      <ul>\n" +
            "      <li><tt>&amp;.colorBar=</tt> specifies the settings for a color bar.  The sub-values are:\n" +
            "        <ul>\n" +
            "        <li>palette - see a Make A Graph web page for options (e.g., \"Rainbow\"). \n" +
            "           <br>The default varies based on min and max: if -1*min ~= max, the default is \"BlueWhiteRed\";\n" +
            "           <br>otherwise, the default is \"Rainbow\".\n" +
            "        <li>continuous - must be either no value (the default), 'C' (for Continuous), or 'D' (for Discrete).\n" +
            "          The default is different for different datasets.\n" +
            "        <li>scale - must be either no value (the default), \"Linear\", or \"Log\".\n" +
            "          The default is different for different datasets.\n" +
            "        <li>min - The minimum value for the color bar. \n" +
            "          The default is different for different datasets.\n" +
            "        <li>max - The maximum value for the color bar. \n" +
            "          The default is different for different datasets.\n" +
            "        <li>nSections - The preferred number of sections (for Log color bars, this is a minimum value). \n" +
            "          The default is different for different datasets.\n" +
            "        </ul>\n" +
            "      <li><tt>&amp;.color=</tt> specifies the color for lines, markers, vectors, etc. The value must be specified as \n" +
            "        an 0xRRGGBB value (e.g., 0xFF0000 is red, 0x00FF00 is green). The default is 0x000000 (black).\n" +
            "      <li><tt>&amp;.draw=</tt> specifies how the data will be drawn, as \"lines\", \"linesAndMarkers\", \"markers\", \"sticks\", \"surface\", or \"vectors\". \n" +
            "      <li><tt>&amp;.font=</tt> specifies a scale factor for the font (e.g., 1.5 would make the font 1.5 times as big as normal).\n" +
            "      <li><tt>&amp;.land=</tt> specifies whether the land should be drawn \"under\" or \"over\" (the default) the data.\n" +
            "        <br>Terrestrial researchers usually prefer \"under\". Oceanographers often prefer \"over\". \n" +
            "      <li><tt>&amp;.marker=</tt> specifies markerType|markerSize . \n" +
            "        <br>markerType is an integer: 0=None, 1=Plus, 2=X, 3=Dot, 4=Square, 5=Filled Square (default), 6=Circle, \n" +
            "           7=Filled Circle, 8=Up Triangle, 9=Filled Up Triangle.\n" +
            "        <br>markerSize is an integer from 3 to 50 (default=5)\n" +
            "      <li><tt>&amp;.vars=</tt> is a '|'-separated list of variables names. Defaults are hard to predict.\n" +
            "        <br>The meaning associated with each position varies with the <tt>&amp;.draw</tt> value:\n" +
            "        <ul>\n" +
            "        <li>for lines: xAxis|yAxis\n" +
            "        <li>for linesAndMarkers: xAxis|yAxis|Color\n" +
            "        <li>for markers: xAxis|yAxis|Color\n" +
            "        <li>for sticks: xAxis|uComponent|vComponent\n" +
            "        <li>for surface: xAxis|yAxis|Color\n" +
            "        <li>for vectors: xAxis|yAxis|uComponent|vComponent\n" +
            "        </ul>\n" +
            "        If xAxis=longitude and yAxis=latitude, you get a map; otherwise, you get a graph.\n " +
            "      <li><tt>&amp;.vec=</tt> specifies the data vector length (in data units) to be scaled to the size of \n" + 
            "        the sample vector in the legend. The default varies based on the data.\n" +
            "      <li><tt>&amp;.xRange=</tt> specifies the min|max for the X axis. The default varies based on the data.\n" +
            "      <li><tt>&amp;.yRange=</tt> specifies the min|max for the Y axis. The default varies based on the data.\n" +
            "      </ul>\n" +
            "    <br>A sample graph URL is \n" +
            "    <br><a href=\"" + fullGraphExample + "\">" + 
                                   fullGraphExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to .graph, you can see a Make A Graph web page with that request loaded:\n" +
            "    <br><a href=\"" + fullGraphMAGExample + "\">" + 
                                   fullGraphMAGExample + "</a>\n" +
            "    <br>That makes it easy for humans to modify an image request to make a similar graph or map.\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to a data fileType (e.g., .htmlTable), you can download the data that was graphed:\n" +
            "    <br><a href=\"" + fullGraphDataExample + "\">" + 
                                   fullGraphDataExample + "</a>\n" +
            "\n" +
            "    <p>A sample map URL is \n" +
            "    <br><a href=\"" + fullMapExample + "\">" + 
                                   fullMapExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to .graph, you can see a Make A Graph web page with that request loaded:\n" +
            "    <br><a href=\"" + fullMapMAGExample + "\">" + 
                                   fullMapMAGExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to a data fileType (e.g., .htmlTable), you can download the data that was mapped:\n" +
            "    <br><a href=\"" + fullMapDataExample + "\">" + 
                                   fullMapDataExample + "</a>\n" +
            "  </ul>\n" +
            "</ul>\n" +

            //other info
            "<a name=\"otherInformation\"><b>Other Information</b></a>\n" +
            "<ul>\n" +
            "<li><a name=\"dataModel\"><b>Data Model</b></a> - Each griddap dataset can be represented as:\n" + 
            "  <ul>\n" +
            "  <li>An ordered list of one or more 1-dimensional axis variables.\n" +
            "      <br>Each axis variable has data of one specific type.\n" +
            "      <br>If the data source has a dimension with a size but no values, ERDDAP uses the values 0, 1, 2, ...\n" +
            "      <br>The supported types are int8, uint16, int16, int32, int64, float32, and float64.\n" +
            "      <br>Missing values are not allowed.\n" +
            "      <br>The values MUST be sorted in either ascending (recommended) or descending order.\n" +
            "      <br>Unsorted values are not allowed because <tt>[(start):(stop)]</tt> requests must translate into a contiguous range of indices.\n" +
            "      <br>Tied values are not allowed because requests for a single <tt>[(value)]</tt> must translate unambiguously to one index.\n" +
            "      <br>Each axis variable has a name composed of a letter (A-Z, a-z) and then 0 or more characters (A-Z, a-z, 0-9, _).\n" +
            "      <br>Each axis variable has a set of metadata expressed as Key=Value pairs.\n" +
            "  <li>A set of one or more n-dimensional data variables.\n" +
            "      <br>All data variables use all of the axis variables, in order, as their dimensions.\n" +
            "      <br>Each data variable has data of one specific type.\n" +
            "      <br>The supported types are (int8, uint16, int16, int32, int64, float32, float64, String of any length).\n" +
            "      <br>Missing values are allowed.\n" +
            "      <br>Each data variable has a name composed of a letter (A-Z, a-z) and then 0 or more characters (A-Z, a-z, 0-9, _).\n" +
            "      <br>Each data variable has a set of metadata expressed as Key=Value pairs.\n" +
            "  <li>Global metadata expressed as Key=Value pairs.\n" +
            "      <br>Each metadata value is either one or more numeric values (of one type), or one or more Strings\n" +
            "      <br>of any length (using \\n as the separator).\n" +
            "  </ul>\n" +
            "<li><a name=\"specialVariables\"><b>Special Variables</b></a>\n" +
            "  <ul>\n" +
            "  <li>In griddap, a longitude axis variable (if present) always has the name \"" + EDV.LON_NAME + 
                 "\" and the units \"" + EDV.LON_UNITS + "\".\n" +
            "  <li>In griddap, a latitude axis variable (if present) always has the name \"" + EDV.LAT_NAME + 
                 "\" and the units \"" + EDV.LAT_UNITS + "\".\n" +
            "  <li>In griddap, an altitude axis variable (if present) always has the name \"" + EDV.ALT_NAME + 
                 "\" and the units \"" + EDV.ALT_UNITS + "\" above sea level.\n" +
            "     Locations below sea level have negative altitude values.\n" +
            "  <li>In griddap, a time axis variable (if present) always has the name \"" + EDV.TIME_NAME + 
                 "\" and the units \"" + EDV.TIME_UNITS + "\".\n" +
            "    <br>If you request data and specify a start and/or stop value for the time axis,\n" +
            "      you can specify the time as a numeric value (seconds since 1970-01-01T00:00:00Z)\n" +
            "      or as a String value (e.g., \"2002-12-25T07:00:00\" in the GMT/Zulu time zone).\n" +
            "  <li>Because the longitude, latitude, altitude, and time axis variables are specifically\n" +
                "recognized, ERDDAP is aware of the geo/temporal features of each dataset.\n" +
                "This is useful when making images with maps or time-series,\n" +
                "and when saving data in geo-referenced file types (e.g., .esriAscii and .kml).\n" +
            "  </ul>\n" +
            "<li><a name=\"incompatibilities\"><b>Incompatibilities</b></a>\n" +
            "  <ul>\n" +
            "  <li>File Types - Some results file types have restrictions.\n" +
            "    <br>For example, .kml is only appropriate for results with a range of longitude and latitude values.\n" +
            "    <br>If a given request is incompatible with the requested file type, griddap throws an error.\n" + 
            "  </ul>\n" +
            "<li>" + OutputStreamFromHttpResponse.acceptEncodingHtml +
            "</ul>\n" +
            "<br>&nbsp;\n");
    }                   

    /**
     * Get colorBarMinimum and Maximum for all grid variables in erddap.
     * Currently, this is just set up for Bob's use.
     */
    public static void suggestGraphMinMax() throws Throwable {
        String tDir = "c:/temp/griddap/";
        String tName = "datasets.tsv";

        while (true) {
            String dsName = String2.getStringFromSystemIn("Grid datasetID? "); 
            if (dsName.length() == 0) 
                dsName = "erdBAssta5day"; //hycomPacS";

            Table info = new Table();
            SSR.downloadFile("http://coastwatch.pfeg.noaa.gov/erddap/info/" + dsName + "/index.tsv",
                tDir + tName, true);

            String response[] = String2.readFromFile(tDir + tName);
            Test.ensureTrue(response[0].length() == 0, response[0]);
            String2.log("Dataset info (500 chars):\n" + response[1].substring(0, Math.min(response[1].length(), 500)));

            info.readASCII(tDir + tName, 0, 1, null, null, null, null);
            //String2.log(info.toString());

            //generate request for data for range of lat and lon  and middle one of other axes
            StringBuffer subset = new StringBuffer();
            StringArray dataVars = new StringArray();
            int nDim = 0;
            for (int row = 0; row < info.nRows(); row++) {
                String type = info.getStringData(0, row);
                String varName = info.getStringData(1, row);
                if (type.equals("variable")) {
                    dataVars.add(varName);
                    continue;
                }

                if (!type.equals("dimension"))
                    continue;

                //deal with dimensions
                nDim++;
                String s4[] = String2.split(info.getStringData(4, row), ',');
                int nValues = String2.parseInt(String2.split(s4[0], '=')[1]);
                String2.log(varName + " " + nValues);
                if (varName.equals("longitude")) 
                    subset.append("[0:" + (nValues / 36) + ":" + (nValues - 1) + "]");
                else if (varName.equals("latitude")) 
                    subset.append("[0:" + (nValues / 18) + ":" + (nValues - 1) + "]");
                else 
                    subset.append("[" + (nValues / 2) + "]");
            }
            String2.log("subset=" + subset.toString() +
                "\nnDim=" + nDim + " vars=" + dataVars.toString());

            //get suggested range for each dataVariable
            Table data = new Table();
            int ndv = dataVars.size();
            for (int v = 0; v < ndv; v++) {
                try {
                    String varName = dataVars.get(v);
                    SSR.downloadFile("http://coastwatch.pfeg.noaa.gov/erddap/griddap/" + dsName + ".tsv?" +
                        varName + subset.toString(), tDir + tName, true);

                    response = String2.readFromFile(tDir + tName);
                    Test.ensureTrue(response[0].length() == 0, response[0]);
                    if (response[1].startsWith("<!DOCTYPE HTML")) {
                        int start = response[1].indexOf("The error:");
                        int stop = response[1].length();
                        if (start >= 0) {
                            start = response[1].indexOf("Your request URL:");
                            stop = response[1].indexOf("</tr>", start);
                            stop = response[1].indexOf("</tr>", stop);
                            stop = response[1].indexOf("</tr>", stop);
                        }
                        if (start < 0) {
                            start = 0;
                            stop = response[1].length();
                        }
                        String2.log("Response for varName=" + varName + ":\n" + 
                            String2.replaceAll(response[1].substring(start, stop),"<br>", "\n<br>"));
                    }

                    data.readASCII(tDir + tName, 0, 1, null, null, null, null);
                    PrimitiveArray pa = data.getColumn(data.nColumns() - 1);
                    double stats[] = pa.calculateStats();
                    double tMin = stats[PrimitiveArray.STATS_MIN];
                    double tMax = stats[PrimitiveArray.STATS_MAX];
                    double range = tMax - tMin; 
                    double loHi[] = Math2.suggestLowHigh(tMin + range/10, tMax - range/10); //interior range
                    String2.log(
                        "varName=" + varName + " min=" + tMin + " max=" + tMax + "\n" +
                        "                <att name=\"colorBarMinimum\" type=\"double\">" + loHi[0] + "</att>\n" +
                        "                <att name=\"colorBarMaximum\" type=\"double\">" + loHi[1] + "</att>\n");

                } catch (Throwable t) {
                    String2.log("\n" + MustBe.throwableToString(t));
                }
            }

        }
    }

}
